"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const coin_selection_1 = require("@fivebinaries/coin-selection");
const schema_utils_1 = require("@trezor/schema-utils");
const constants_1 = require("../../../constants");
const AbstractMethod_1 = require("../../../core/AbstractMethod");
const coinInfo_1 = require("../../../data/coinInfo");
const cardano_1 = require("../../../types/api/cardano");
const pathUtils_1 = require("../../../utils/pathUtils");
const paramsValidator_1 = require("../../common/paramsValidator");
const cardanoAuxiliaryData_1 = require("../cardanoAuxiliaryData");
const cardanoCertificate_1 = require("../cardanoCertificate");
const cardanoInputs_1 = require("../cardanoInputs");
const cardanoOutputs_1 = require("../cardanoOutputs");
const cardanoTokenBundle_1 = require("../cardanoTokenBundle");
const cardanoWitnesses_1 = require("../cardanoWitnesses");
const CardanoSignTransactionFeatures = Object.freeze({
    Conway: ['0', '2.7.1'],
});
class CardanoSignTransaction extends AbstractMethod_1.AbstractMethod {
    init() {
        this.requiredPermissions = ['read', 'write'];
        this.requiredDeviceCapabilities = ['Capability_Cardano'];
        this.firmwareRange = (0, paramsValidator_1.getFirmwareRange)(this.name, (0, coinInfo_1.getMiscNetwork)('Cardano'), this.firmwareRange);
        const { payload } = this;
        if (payload.metadata) {
            throw constants_1.ERRORS.TypedError('Method_InvalidParameter', 'Metadata field has been replaced by auxiliaryData.');
        }
        if (payload.auxiliaryData && payload.auxiliaryData.blob) {
            throw constants_1.ERRORS.TypedError('Method_InvalidParameter', 'Auxiliary data can now only be sent as a hash.');
        }
        if (payload.auxiliaryData && payload.auxiliaryData.catalystRegistrationParameters) {
            console.warn('Please use cVoteRegistrationParameters instead of catalystRegistrationParameters.');
            payload.auxiliaryData.cVoteRegistrationParameters =
                payload.auxiliaryData.catalystRegistrationParameters;
        }
        if (payload.auxiliaryData && payload.auxiliaryData.governanceRegistrationParameters) {
            console.warn('Please use cVoteRegistrationParameters instead of governanceRegistrationParameters.');
            payload.auxiliaryData.cVoteRegistrationParameters =
                payload.auxiliaryData.governanceRegistrationParameters;
        }
        (0, schema_utils_1.AssertWeak)(schema_utils_1.Type.Union([cardano_1.CardanoSignTransaction, cardano_1.CardanoSignTransactionExtended]), payload);
        const inputsWithPath = payload.inputs.map(cardanoInputs_1.transformInput);
        const outputsWithData = payload.outputs.map(cardanoOutputs_1.transformOutput);
        let certificatesWithPoolOwnersAndRelays = [];
        if (payload.certificates) {
            certificatesWithPoolOwnersAndRelays = payload.certificates.map(cardanoCertificate_1.transformCertificate);
        }
        let withdrawals = [];
        if (payload.withdrawals) {
            withdrawals = payload.withdrawals.map(withdrawal => ({
                path: withdrawal.path ? (0, pathUtils_1.validatePath)(withdrawal.path, 5) : undefined,
                amount: withdrawal.amount,
                script_hash: withdrawal.scriptHash,
                key_hash: withdrawal.keyHash,
            }));
        }
        let mint = [];
        if (payload.mint) {
            mint = (0, cardanoTokenBundle_1.tokenBundleToProto)(payload.mint);
        }
        let auxiliaryData;
        if (payload.auxiliaryData) {
            auxiliaryData = (0, cardanoAuxiliaryData_1.transformAuxiliaryData)(payload.auxiliaryData);
        }
        let additionalWitnessRequests = [];
        if (payload.additionalWitnessRequests) {
            additionalWitnessRequests = payload.additionalWitnessRequests.map(witnessRequest => (0, pathUtils_1.validatePath)(witnessRequest, 3));
        }
        let collateralInputsWithPath = [];
        if (payload.collateralInputs) {
            collateralInputsWithPath = payload.collateralInputs.map(cardanoInputs_1.transformCollateralInput);
        }
        let requiredSigners = [];
        if (payload.requiredSigners) {
            requiredSigners = payload.requiredSigners.map(requiredSigner => ({
                key_path: requiredSigner.keyPath
                    ? (0, pathUtils_1.validatePath)(requiredSigner.keyPath, 3)
                    : undefined,
                key_hash: requiredSigner.keyHash,
            }));
        }
        const collateralReturnWithData = payload.collateralReturn
            ? (0, cardanoOutputs_1.transformOutput)(payload.collateralReturn)
            : undefined;
        let referenceInputs = [];
        if (payload.referenceInputs) {
            referenceInputs = payload.referenceInputs.map(cardanoInputs_1.transformReferenceInput);
        }
        this.params = {
            signingMode: payload.signingMode,
            inputsWithPath,
            outputsWithData,
            fee: payload.fee,
            ttl: payload.ttl,
            certificatesWithPoolOwnersAndRelays,
            withdrawals,
            mint,
            auxiliaryData,
            validityIntervalStart: payload.validityIntervalStart,
            scriptDataHash: payload.scriptDataHash,
            collateralInputsWithPath,
            requiredSigners,
            collateralReturnWithData,
            totalCollateral: payload.totalCollateral,
            referenceInputs,
            protocolMagic: payload.protocolMagic,
            networkId: payload.networkId,
            witnessPaths: (0, cardanoWitnesses_1.gatherWitnessPaths)(inputsWithPath, certificatesWithPoolOwnersAndRelays, withdrawals, collateralInputsWithPath, requiredSigners, additionalWitnessRequests, payload.signingMode),
            additionalWitnessRequests,
            derivationType: typeof payload.derivationType !== 'undefined'
                ? payload.derivationType
                : constants_1.PROTO.CardanoDerivationType.ICARUS_TREZOR,
            includeNetworkId: payload.includeNetworkId,
            tagCborSets: payload.tagCborSets,
            unsignedTx: 'unsignedTx' in payload ? payload.unsignedTx : undefined,
            testnet: 'testnet' in payload ? payload.testnet : undefined,
            chunkify: typeof payload.chunkify === 'boolean' ? payload.chunkify : false,
        };
    }
    get info() {
        return 'Sign Cardano transaction';
    }
    _isFeatureSupported(feature) {
        return this.device.atLeast(CardanoSignTransactionFeatures[feature]);
    }
    _ensureFeatureIsSupported(feature) {
        if (!this._isFeatureSupported(feature)) {
            throw constants_1.ERRORS.TypedError('Method_InvalidParameter', `Feature ${feature} not supported by device firmware`);
        }
    }
    _ensureFirmwareSupportsParams() {
        const { params } = this;
        params.certificatesWithPoolOwnersAndRelays.forEach(({ certificate }) => {
            if (certificate.type === constants_1.PROTO.CardanoCertificateType.STAKE_REGISTRATION_CONWAY ||
                certificate.type === constants_1.PROTO.CardanoCertificateType.STAKE_DEREGISTRATION_CONWAY ||
                certificate.type === constants_1.PROTO.CardanoCertificateType.VOTE_DELEGATION) {
                this._ensureFeatureIsSupported('Conway');
            }
        });
        if (params.tagCborSets) {
            this._ensureFeatureIsSupported('Conway');
        }
    }
    async _sign_tx() {
        const { typedCall } = this.device.getCommands();
        const hasAuxiliaryData = !!this.params.auxiliaryData;
        const signTxInitMessage = {
            signing_mode: this.params.signingMode,
            protocol_magic: this.params.protocolMagic,
            network_id: this.params.networkId,
            inputs_count: this.params.inputsWithPath.length,
            outputs_count: this.params.outputsWithData.length,
            fee: this.params.fee,
            ttl: this.params.ttl,
            certificates_count: this.params.certificatesWithPoolOwnersAndRelays.length,
            withdrawals_count: this.params.withdrawals.length,
            has_auxiliary_data: hasAuxiliaryData,
            validity_interval_start: this.params.validityIntervalStart,
            witness_requests_count: this.params.witnessPaths.length,
            minting_asset_groups_count: this.params.mint.length,
            script_data_hash: this.params.scriptDataHash,
            collateral_inputs_count: this.params.collateralInputsWithPath.length,
            required_signers_count: this.params.requiredSigners.length,
            has_collateral_return: this.params.collateralReturnWithData != null,
            total_collateral: this.params.totalCollateral,
            reference_inputs_count: this.params.referenceInputs.length,
            derivation_type: this.params.derivationType,
            include_network_id: this.params.includeNetworkId,
            chunkify: this.params.chunkify,
            tag_cbor_sets: this.params.tagCborSets,
        };
        await typedCall('CardanoSignTxInit', 'CardanoTxItemAck', signTxInitMessage);
        for (const { input } of this.params.inputsWithPath) {
            await typedCall('CardanoTxInput', 'CardanoTxItemAck', input);
        }
        for (const outputWithData of this.params.outputsWithData) {
            await (0, cardanoOutputs_1.sendOutput)(typedCall, outputWithData);
        }
        for (const { certificate, poolOwners, poolRelays } of this.params
            .certificatesWithPoolOwnersAndRelays) {
            await typedCall('CardanoTxCertificate', 'CardanoTxItemAck', certificate);
            for (const poolOwner of poolOwners) {
                await typedCall('CardanoPoolOwner', 'CardanoTxItemAck', poolOwner);
            }
            for (const poolRelay of poolRelays) {
                await typedCall('CardanoPoolRelayParameters', 'CardanoTxItemAck', poolRelay);
            }
        }
        for (const withdrawal of this.params.withdrawals) {
            await typedCall('CardanoTxWithdrawal', 'CardanoTxItemAck', withdrawal);
        }
        let auxiliaryDataSupplement;
        if (this.params.auxiliaryData) {
            const { cvote_registration_parameters } = this.params.auxiliaryData;
            if (cvote_registration_parameters) {
                this.params.auxiliaryData = (0, cardanoAuxiliaryData_1.modifyAuxiliaryDataForBackwardsCompatibility)(this.params.auxiliaryData);
            }
            const { message } = await typedCall('CardanoTxAuxiliaryData', 'CardanoTxAuxiliaryDataSupplement', this.params.auxiliaryData);
            const auxiliaryDataType = constants_1.PROTO.CardanoTxAuxiliaryDataSupplementType[message.type];
            if (auxiliaryDataType !== constants_1.PROTO.CardanoTxAuxiliaryDataSupplementType.NONE) {
                auxiliaryDataSupplement = {
                    type: auxiliaryDataType,
                    auxiliaryDataHash: message.auxiliary_data_hash,
                    cVoteRegistrationSignature: message.cvote_registration_signature,
                    catalystSignature: message.cvote_registration_signature,
                    governanceSignature: message.cvote_registration_signature,
                };
            }
            await typedCall('CardanoTxHostAck', 'CardanoTxItemAck');
        }
        if (this.params.mint.length > 0) {
            await typedCall('CardanoTxMint', 'CardanoTxItemAck', {
                asset_groups_count: this.params.mint.length,
            });
            for (const assetGroup of this.params.mint) {
                await typedCall('CardanoAssetGroup', 'CardanoTxItemAck', {
                    policy_id: assetGroup.policyId,
                    tokens_count: assetGroup.tokens.length,
                });
                for (const token of assetGroup.tokens) {
                    await typedCall('CardanoToken', 'CardanoTxItemAck', token);
                }
            }
        }
        for (const { collateralInput } of this.params.collateralInputsWithPath) {
            await typedCall('CardanoTxCollateralInput', 'CardanoTxItemAck', collateralInput);
        }
        for (const requiredSigner of this.params.requiredSigners) {
            await typedCall('CardanoTxRequiredSigner', 'CardanoTxItemAck', requiredSigner);
        }
        if (this.params.collateralReturnWithData) {
            await (0, cardanoOutputs_1.sendOutput)(typedCall, this.params.collateralReturnWithData);
        }
        for (const referenceInput of this.params.referenceInputs) {
            await typedCall('CardanoTxReferenceInput', 'CardanoTxItemAck', referenceInput);
        }
        const witnesses = [];
        for (const path of this.params.witnessPaths) {
            const { message } = await typedCall('CardanoTxWitnessRequest', 'CardanoTxWitnessResponse', { path });
            witnesses.push({
                type: constants_1.PROTO.CardanoTxWitnessType[message.type],
                pubKey: message.pub_key,
                signature: message.signature,
                chainCode: message.chain_code,
            });
        }
        const { message: txBodyHashMessage } = await typedCall('CardanoTxHostAck', 'CardanoTxBodyHash');
        await typedCall('CardanoTxHostAck', 'CardanoSignTxFinished');
        return { hash: txBodyHashMessage.tx_hash, witnesses, auxiliaryDataSupplement };
    }
    async run() {
        this._ensureFirmwareSupportsParams();
        const result = await this._sign_tx();
        if (!this.params.unsignedTx)
            return result;
        const { unsignedTx, testnet } = this.params;
        const { hash, witnesses } = result;
        if (hash !== unsignedTx.hash) {
            throw constants_1.ERRORS.TypedError('Runtime', "Constructed transaction doesn't match the hash returned by the device.");
        }
        const serializedTx = coin_selection_1.trezorUtils.signTransaction(unsignedTx.body, witnesses, { testnet });
        return {
            ...result,
            serializedTx,
        };
    }
}
exports.default = CardanoSignTransaction;
//# sourceMappingURL=cardanoSignTransaction.js.map