"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.verifyTx = exports.deriveOutputScript = void 0;
const utxo_lib_1 = require("@trezor/utxo-lib");
const constants_1 = require("../../constants");
const derivePubKeyHash = async (getHDNode, address_n, network) => {
    if (address_n.length === 5) {
        const response = await getHDNode(address_n.slice(0, 4));
        const node = utxo_lib_1.bip32.fromBase58(response.xpub, network);
        return node.derive(address_n[address_n.length - 1]);
    }
    const response = await getHDNode(address_n);
    return utxo_lib_1.bip32.fromBase58(response.xpub, network);
};
const deriveOutputScript = async (getHDNode, output, network) => {
    if ('multisig' in output)
        return;
    if ('op_return_data' in output) {
        return utxo_lib_1.payments.embed({ data: [Buffer.from(output.op_return_data, 'hex')] })
            .output;
    }
    if (output.address) {
        return utxo_lib_1.address.toOutputScript(output.address, network);
    }
    if (!output.address_n) {
        throw constants_1.ERRORS.TypedError('Runtime', 'deriveOutputScript: Neither address or address_n is set');
    }
    const node = await derivePubKeyHash(getHDNode, output.address_n, network);
    const payment = { hash: node.identifier, network };
    if (output.script_type === 'PAYTOADDRESS') {
        return utxo_lib_1.payments.p2pkh(payment).output;
    }
    if (output.script_type === 'PAYTOSCRIPTHASH') {
        return utxo_lib_1.payments.p2sh(payment).output;
    }
    if (output.script_type === 'PAYTOP2SHWITNESS') {
        return utxo_lib_1.payments.p2sh({
            redeem: utxo_lib_1.payments.p2wpkh(payment),
        }).output;
    }
    if (output.script_type === 'PAYTOWITNESS') {
        return utxo_lib_1.payments.p2wpkh(payment).output;
    }
    if (output.script_type === 'PAYTOTAPROOT') {
        return utxo_lib_1.payments.p2tr({
            pubkey: node.publicKey,
            network,
        }).output;
    }
    if (output.script_type === 'PAYTOMULTISIG') {
        throw constants_1.ERRORS.TypedError('Runtime', `deriveOutputScript: script_type PAYTOMULTISIG not expected without multisig defined`);
    }
    throw constants_1.ERRORS.TypedError('Runtime', `deriveOutputScript: Unknown script type ${output.script_type}`);
};
exports.deriveOutputScript = deriveOutputScript;
const verifyTx = (serializedTx, params) => {
    const { inputs, outputs, outputScripts, network } = params;
    const bitcoinTx = utxo_lib_1.Transaction.fromHex(serializedTx, { network });
    if (inputs.length !== bitcoinTx.ins.length) {
        throw constants_1.ERRORS.TypedError('Runtime', 'verifyTx: Signed transaction inputs invalid length');
    }
    if (outputs.length !== bitcoinTx.outs.length) {
        throw constants_1.ERRORS.TypedError('Runtime', 'verifyTx: Signed transaction outputs invalid length');
    }
    outputs.forEach((output, i) => {
        if (output.amount) {
            if (output.amount.toString() !== bitcoinTx.outs[i].value) {
                throw constants_1.ERRORS.TypedError('Runtime', `verifyTx: Wrong output amount at output ${i}. Requested: ${output.amount}, signed: ${bitcoinTx.outs[i].value}`);
            }
        }
    });
    for (let i = 0; i < outputs.length; i++) {
        const scriptB = bitcoinTx.outs[i].script;
        const scriptA = outputScripts[i];
        if (scriptA && scriptA.compare(scriptB) !== 0) {
            throw constants_1.ERRORS.TypedError('Runtime', `verifyTx: Output ${i} scripts differ`);
        }
    }
    return bitcoinTx;
};
exports.verifyTx = verifyTx;
//# sourceMappingURL=signtxVerify.js.map