"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CachingElectrumClient = void 0;
const electrum_1 = require("./electrum");
class CachingElectrumClient extends electrum_1.ElectrumClient {
    cache = {};
    statuses = {};
    cached = 0;
    total = 0;
    logTimer;
    constructor() {
        super();
        this.logTimer = setInterval(() => {
            this.log(`Caching effectiveness: ${this.cached}/${this.total}`);
            this.log('Subscription count: ', Object.keys(this.statuses).length);
        }, 60000);
    }
    async cacheRequest(status, method, params) {
        const descriptor = [method, ...params].join(':');
        const cached = this.cache[descriptor];
        if (cached) {
            const [cachedStatus, cachedResponse] = cached;
            if (cachedStatus === status) {
                this.cached++;
                return cachedResponse;
            }
        }
        const response = await super.request(method, ...params);
        this.cache[descriptor] = [status, response];
        return response;
    }
    async trySubscribe(scripthash) {
        const status = this.statuses[scripthash];
        if (status !== undefined) {
            return status;
        }
        const newStatus = await super.request('blockchain.scripthash.subscribe', scripthash);
        this.statuses[scripthash] = newStatus;
        return newStatus;
    }
    async request(method, ...params) {
        this.total++;
        switch (method) {
            case 'blockchain.scripthash.get_history':
            case 'blockchain.scripthash.get_balance':
            case 'blockchain.scripthash.listunspent': {
                const [scripthash] = params;
                const status = await this.trySubscribe(scripthash);
                return this.cacheRequest(status, method, params);
            }
            case 'blockchain.transaction.get': {
                const curBlock = this.lastBlock?.hex;
                if (curBlock === undefined)
                    break;
                return this.cacheRequest(curBlock, method, params);
            }
            case 'blockchain.scripthash.subscribe': {
                const [scripthash] = params;
                return this.trySubscribe(scripthash);
            }
            case 'blockchain.scripthash.unsubscribe': {
                const [scripthash] = params;
                delete this.statuses[scripthash];
                return super.request(method, ...params);
            }
            default:
                break;
        }
        return super.request(method, ...params);
    }
    response(response) {
        const { method, params } = response;
        switch (method) {
            case 'blockchain.scripthash.subscribe': {
                const [scripthash, status] = params;
                this.statuses[scripthash] = status;
                break;
            }
            default:
                break;
        }
        super.response(response);
    }
    onClose() {
        super.onClose();
        clearInterval(this.logTimer);
    }
}
exports.CachingElectrumClient = CachingElectrumClient;
//# sourceMappingURL=caching.js.map