"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getTokenMetadata = exports.buildSendTransaction = exports.transformTransaction = exports.toStroops = exports.STELLAR_DECIMALS = void 0;
const stellar_sdk_1 = require("@stellar/stellar-sdk");
const env_utils_1 = require("@trezor/env-utils");
const bigNumber_1 = require("@trezor/utils/lib/bigNumber");
exports.STELLAR_DECIMALS = 7;
const toStroops = (value) => {
    const multiplier = new bigNumber_1.BigNumber(10).pow(exports.STELLAR_DECIMALS);
    const amount = new bigNumber_1.BigNumber(value).times(multiplier);
    return amount;
};
exports.toStroops = toStroops;
const isoToTimestamp = (isoDate) => {
    const timestamp = Date.parse(isoDate);
    if (isNaN(timestamp)) {
        throw new Error('Invalid ISO date string');
    }
    return Math.floor(timestamp / 1000);
};
const convertMemo = (memo) => {
    switch (memo.type) {
        case 'text':
        case 'id':
            return memo.value?.toString();
        case 'hash':
        case 'return':
            return memo.value?.toString('hex');
        default:
            return undefined;
    }
};
const transformTransaction = (rawTx, descriptor) => {
    const parsedTx = new stellar_sdk_1.Transaction(rawTx.envelope_xdr, stellar_sdk_1.Networks.PUBLIC);
    const tx = {
        type: 'unknown',
        txid: rawTx.hash,
        amount: '0',
        fee: rawTx.fee_charged.toString(),
        blockTime: isoToTimestamp(rawTx.created_at),
        blockHeight: rawTx.ledger_attr,
        targets: [],
        tokens: [],
        internalTransfers: [],
        feeRate: undefined,
        details: {
            vin: [],
            vout: [],
            size: 0,
            totalInput: '0',
            totalOutput: '0',
        },
        stellarSpecific: {
            memo: convertMemo(parsedTx.memo),
            feeSource: (0, stellar_sdk_1.extractBaseAddress)(rawTx.source_account),
        },
    };
    if (!rawTx.successful) {
        return { ...tx, type: 'failed' };
    }
    if (parsedTx.operations.length !== 1) {
        return tx;
    }
    const rawOp = parsedTx.operations[0];
    const opSource = rawOp.source || rawTx.source_account;
    const params = {
        fromAddress: (0, stellar_sdk_1.extractBaseAddress)(opSource),
        toAddress: '',
        amount: '',
    };
    switch (rawOp.type) {
        case 'createAccount': {
            params.toAddress = (0, stellar_sdk_1.extractBaseAddress)(rawOp.destination);
            params.amount = (0, exports.toStroops)(rawOp.startingBalance).toString();
            break;
        }
        case 'payment': {
            if (!rawOp.asset.isNative()) {
                return tx;
            }
            params.toAddress = (0, stellar_sdk_1.extractBaseAddress)(rawOp.destination);
            params.amount = (0, exports.toStroops)(rawOp.amount).toString();
            break;
        }
        default: {
            return tx;
        }
    }
    if (descriptor !== params.fromAddress && descriptor !== params.toAddress) {
        return tx;
    }
    const targets = [
        {
            n: 0,
            addresses: [params.toAddress],
            isAddress: true,
            amount: params.amount,
        },
    ];
    const details = {
        vin: [
            {
                n: 0,
                addresses: [params.fromAddress],
                isAddress: true,
                value: params.amount,
            },
        ],
        vout: [
            {
                n: 0,
                addresses: [params.toAddress],
                isAddress: true,
                value: params.amount,
            },
        ],
        size: 0,
        totalInput: params.amount,
        totalOutput: params.amount,
    };
    const type = descriptor === params.fromAddress ? 'sent' : 'recv';
    return { ...tx, amount: params.amount, details, targets, type };
};
exports.transformTransaction = transformTransaction;
const buildSendTransaction = (descriptor, sequence, fee, destinationActivated, destination, amount, asset, destinationTag, isTestnet = false) => {
    const source = new stellar_sdk_1.Account(descriptor, sequence);
    const txBuilder = new stellar_sdk_1.TransactionBuilder(source, {
        fee,
        networkPassphrase: isTestnet ? stellar_sdk_1.Networks.TESTNET : stellar_sdk_1.Networks.PUBLIC,
    }).setTimebounds(0, 0);
    if (destinationTag) {
        txBuilder.addMemo(stellar_sdk_1.Memo.text(destinationTag));
    }
    if (destinationActivated) {
        txBuilder.addOperation(stellar_sdk_1.Operation.payment({
            destination,
            amount,
            asset: new stellar_sdk_1.Asset(asset.code || 'XLM', asset.issuer),
        }));
    }
    else {
        txBuilder.addOperation(stellar_sdk_1.Operation.createAccount({
            destination,
            startingBalance: amount,
        }));
    }
    return txBuilder.build();
};
exports.buildSendTransaction = buildSendTransaction;
const getTokenMetadata = async () => {
    const env = (0, env_utils_1.isCodesignBuild)() ? 'stable' : 'develop';
    const response = await fetch(`https://data.trezor.io/suite/definitions/${env}/stellar.advanced.coin.definitions.v1.json`);
    if (!response.ok) {
        throw Error(`Failed to fetch token metadata: ${response.statusText}`);
    }
    const data = await response.json();
    return data;
};
exports.getTokenMetadata = getTokenMetadata;
//# sourceMappingURL=stellar.js.map