import type { ReadonlyUint8Array } from '@solana/codecs-core';
import { Brand } from '@solana/nominal-types';
import { Address } from './address';
/**
 * A tuple representing a program derived address (derived from the address of some program and a
 * set of seeds) and the associated bump seed used to ensure that the address, as derived, does not
 * fall on the Ed25519 curve.
 *
 * Whenever you need to validate an arbitrary tuple as one that represents a program derived
 * address, use the {@link assertIsProgramDerivedAddress} or {@link isProgramDerivedAddress}
 * functions in this package.
 */
export type ProgramDerivedAddress<TAddress extends string = string> = Readonly<[
    Address<TAddress>,
    ProgramDerivedAddressBump
]>;
/**
 * Represents an integer in the range [0,255] used in the derivation of a program derived address to
 * ensure that it does not fall on the Ed25519 curve.
 */
export type ProgramDerivedAddressBump = Brand<number, 'ProgramDerivedAddressBump'>;
/**
 * A type guard that returns `true` if the input tuple conforms to the {@link ProgramDerivedAddress}
 * type, and refines its type for use in your program.
 *
 * @see The {@link isAddress} function for an example of how to use a type guard.
 */
export declare function isProgramDerivedAddress<TAddress extends string = string>(value: unknown): value is ProgramDerivedAddress<TAddress>;
/**
 * In the event that you receive an address/bump-seed tuple from some untrusted source, use this
 * function to assert that it conforms to the {@link ProgramDerivedAddress} interface.
 *
 * @see The {@link assertIsAddress} function for an example of how to use an assertion function.
 */
export declare function assertIsProgramDerivedAddress<TAddress extends string = string>(value: unknown): asserts value is ProgramDerivedAddress<TAddress>;
type ProgramDerivedAddressInput = Readonly<{
    programAddress: Address;
    seeds: Seed[];
}>;
type SeedInput = Readonly<{
    baseAddress: Address;
    programAddress: Address;
    seed: Seed;
}>;
type Seed = ReadonlyUint8Array | string;
/**
 * Given a program's {@link Address} and up to 16 {@link Seed | Seeds}, this method will return the
 * program derived address (PDA) associated with each.
 *
 * @example
 * ```ts
 * import { getAddressEncoder, getProgramDerivedAddress } from '@solana/addresses';
 *
 * const addressEncoder = getAddressEncoder();
 * const [pda, bumpSeed] = await getProgramDerivedAddress({
 *     programAddress: 'ATokenGPvbdGVxr1b2hvZbsiqW5xWH25efTNsLJA8knL' as Address,
 *     seeds: [
 *         // Owner
 *         addressEncoder.encode('9fYLFVoVqwH37C3dyPi6cpeobfbQ2jtLpN5HgAYDDdkm' as Address),
 *         // Token program
 *         addressEncoder.encode('TokenkegQfeZyiNwAJbNbGKPFXCWuBvf9Ss623VQ5DA' as Address),
 *         // Mint
 *         addressEncoder.encode('EPjFWdd5AufqSSqeM2qN1xzybapC8G4wEGGkZwyTDt1v' as Address),
 *     ],
 * });
 * ```
 */
export declare function getProgramDerivedAddress({ programAddress, seeds, }: ProgramDerivedAddressInput): Promise<ProgramDerivedAddress>;
/**
 * Returns a base58-encoded address derived from some base address, some program address, and a seed
 * string or byte array.
 *
 * @example
 * ```ts
 * import { createAddressWithSeed } from '@solana/addresses';
 *
 * const derivedAddress = await createAddressWithSeed({
 *     // The private key associated with this address will be able to sign for `derivedAddress`.
 *     baseAddress: 'B9Lf9z5BfNPT4d5KMeaBFx8x1G4CULZYR1jA2kmxRDka' as Address,
 *     // Only this program will be able to write data to this account.
 *     programAddress: '445erYq578p2aERrGW9mn9KiYe3fuG6uHdcJ2LPPShGw' as Address,
 *     seed: 'data-account',
 * });
 * ```
 */
export declare function createAddressWithSeed({ baseAddress, programAddress, seed }: SeedInput): Promise<Address>;
export {};
//# sourceMappingURL=program-derived-address.d.ts.map