"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.mapBleErrorToHwTransportError = exports.bleErrorToHwTransportError = exports.decoratePromiseErrors = exports.rethrowError = exports.remapError = void 0;
const errors_1 = require("@ledgerhq/errors");
const react_native_ble_plx_1 = require("react-native-ble-plx");
const remapError = (error) => {
    if (!error || !error.message)
        return error;
    if (error instanceof react_native_ble_plx_1.BleError) {
        if (
        // eslint-disable-next-line @typescript-eslint/ban-ts-comment
        // @ts-expect-error
        error.iosErrorCode === react_native_ble_plx_1.BleATTErrorCode.UnlikelyError ||
            error.reason === "Peer removed pairing information") {
            return new errors_1.PeerRemovedPairing();
            // eslint-disable-next-line @typescript-eslint/ban-ts-comment
            // @ts-ignore It's not documented but seems to match a refusal on Android pairing
        }
        else if (error?.attErrorCode === 22) {
            return new errors_1.PairingFailed();
        }
    }
    if (error.message.includes("was disconnected") || error.message.includes("not found")) {
        return new errors_1.DisconnectedDevice();
    }
    return error;
};
exports.remapError = remapError;
const rethrowError = (e) => {
    throw (0, exports.remapError)(e);
};
exports.rethrowError = rethrowError;
const decoratePromiseErrors = (promise) => promise.catch(exports.rethrowError);
exports.decoratePromiseErrors = decoratePromiseErrors;
exports.bleErrorToHwTransportError = new Map([
    [react_native_ble_plx_1.BleErrorCode.ScanStartFailed, errors_1.HwTransportErrorType.BluetoothScanStartFailed],
    [react_native_ble_plx_1.BleErrorCode.LocationServicesDisabled, errors_1.HwTransportErrorType.LocationServicesDisabled],
    [
        // BluetoothUnauthorized actually represents a location service unauthorized error
        react_native_ble_plx_1.BleErrorCode.BluetoothUnauthorized,
        errors_1.HwTransportErrorType.LocationServicesUnauthorized,
    ],
]);
const mapBleErrorToHwTransportError = (bleError) => {
    const message = `${bleError.message}. Origin: ${bleError.errorCode}`;
    const inferedType = exports.bleErrorToHwTransportError.get(bleError.errorCode);
    const type = !inferedType ? errors_1.HwTransportErrorType.Unknown : inferedType;
    return new errors_1.HwTransportError(type, message);
};
exports.mapBleErrorToHwTransportError = mapBleErrorToHwTransportError;
//# sourceMappingURL=remapErrors.js.map