import { openTransportReplayer, RecordStore } from "@ledgerhq/hw-transport-mocker";
import Btc from "../src/Btc";

describe("splitTransaction", () => {
  test("zcash", async () => {
    const transport = await openTransportReplayer(RecordStore.fromString(""));
    const btc = new Btc({ transport, currency: "zcash" });
    /*

splitTransaction 050000800a27a726b4d0d6c20000000000000000011d73f1a467297aab205ee7a4ed506f28ea558056401b4f6d308016c1b58d27f4010000006a4730440220337050efe67689fdbdccd2058f6f7b7fe3b13070d91cd0d7ecb1f84e622a220b02201356d33259d64db1095879cfce666b016771cf4e239376497b7f82efedd9c54a01210396fcfd94e1bfb2949e0acbab934583c11ad29d14105d25528aff75673c50650c000000000250c30000000000001976a914de3542c396924ada3c5850225770f6dd3e2249b988ac3df2c202000000001976a914fc0da061ca85923e01d97ac276aa8dc890a28efa88ac000000:
TX version 05000080 locktime 00000000 timestamp  nVersionGroupId 0a27a726 nExpiryHeight 00000000 extraData 
input 0: prevout 1d73f1a467297aab205ee7a4ed506f28ea558056401b4f6d308016c1b58d27f401000000 script 4730440220337050efe67689fdbdccd2058f6f7b7fe3b13070d91cd0d7ecb1f84e622a220b02201356d33259d64db1095879cfce666b016771cf4e239376497b7f82efedd9c54a01210396fcfd94e1bfb2949e0acbab934583c11ad29d14105d25528aff75673c50650c sequence 00000000
output 0: amount 50c3000000000000 script 76a914de3542c396924ada3c5850225770f6dd3e2249b988ac
output 1: amount 3df2c20200000000 script 76a914fc0da061ca85923e01d97ac276aa8dc890a28efa88ac
      */
    const tx1 = btc.splitTransaction(
      "050000800a27a726b4d0d6c20000000000000000011d73f1a467297aab205ee7a4ed506f28ea558056401b4f6d308016c1b58d27f4010000006a4730440220337050efe67689fdbdccd2058f6f7b7fe3b13070d91cd0d7ecb1f84e622a220b02201356d33259d64db1095879cfce666b016771cf4e239376497b7f82efedd9c54a01210396fcfd94e1bfb2949e0acbab934583c11ad29d14105d25528aff75673c50650c000000000250c30000000000001976a914de3542c396924ada3c5850225770f6dd3e2249b988ac3df2c202000000001976a914fc0da061ca85923e01d97ac276aa8dc890a28efa88ac000000",
      true,
      true,
      ["zcash", "sapling"],
    );
    /*
       splitTransaction 050000800a27a7265510e7c80000000000000000018c63f70704d9987dafffc5481b171dee900e9b6d71261fee880543bc96c41d11000000006b48304502210098a92ce696ff51d46233885e5ea7d0bc0bcd04621c6d79e4230e579f9b13f1480220772d04b65133859ef7fb146a41080b1187335fed9daf62a237b6bd54657f555d0121039402a22682e936ab3c1e2f649859ba13b39a59bd74212ac903a42b5aea5032790000000002404b4c00000000001976a914c59ace9b52af703379f3f89ebbc8ec1813ca50ec88ac0e532a00000000001976a9144cd6509f71020b6a9e890bef43c4d5e61f9c0dad88ac000000:
TX version 05000080 locktime 00000000 timestamp  nVersionGroupId 0a27a726 nExpiryHeight 00000000 extraData 
input 0: prevout 8c63f70704d9987dafffc5481b171dee900e9b6d71261fee880543bc96c41d1100000000 script 48304502210098a92ce696ff51d46233885e5ea7d0bc0bcd04621c6d79e4230e579f9b13f1480220772d04b65133859ef7fb146a41080b1187335fed9daf62a237b6bd54657f555d0121039402a22682e936ab3c1e2f649859ba13b39a59bd74212ac903a42b5aea503279 sequence 00000000
output 0: amount 404b4c0000000000 script 76a914c59ace9b52af703379f3f89ebbc8ec1813ca50ec88ac
output 1: amount 0e532a0000000000 script 76a9144cd6509f71020b6a9e890bef43c4d5e61f9c0dad88ac
      */
    const tx2 = btc.splitTransaction(
      "050000800a27a7265510e7c80000000000000000018c63f70704d9987dafffc5481b171dee900e9b6d71261fee880543bc96c41d11000000006b48304502210098a92ce696ff51d46233885e5ea7d0bc0bcd04621c6d79e4230e579f9b13f1480220772d04b65133859ef7fb146a41080b1187335fed9daf62a237b6bd54657f555d0121039402a22682e936ab3c1e2f649859ba13b39a59bd74212ac903a42b5aea5032790000000002404b4c00000000001976a914c59ace9b52af703379f3f89ebbc8ec1813ca50ec88ac0e532a00000000001976a9144cd6509f71020b6a9e890bef43c4d5e61f9c0dad88ac000000",
      true,
      true,
      ["zcash", "sapling"],
    );
  });
  test("Zcash Sapling transaction (v4)", async () => {
    const transport = await openTransportReplayer(RecordStore.fromString(""));
    const btc = new Btc({ transport, currency: "zcash" });

    const hex =
      "0400008085202f890177507ef339c27d8d453723c568361fb93671f521f1ba2c42a0f136650939aaa5010000006b48304502210099a9fa0817083a1ce6f96404ed7366d9200f5533a9ccfcd4eddb50be4646c8a102205a6cccc8965f1ea45d6f32d96dda89a3cbb0422fad2a0e05b40fa51c0e51322a0121029f7331870af5630f14fe86e10b6ef696ee152bffb34e71396a4ce82ef64aa23effffffff0240781501000000001976a9144cf48844c49a77ba86e48b070f06151b712c862988ace39e0f02000000001976a91445110888402e6fd0c86329d9eda36c7a3fa354a588ac00000000000000000000000000000000000000";

    const tx = btc.splitTransaction(hex, true, true, ["zcash", "sapling"]);

    expect(tx?.version.toString("hex")).toBe("04000080");
    // expect(tx.nVersionGroupId).toBeDefined();
    expect(tx.nVersionGroupId?.toString("hex")).toBe("85202f89");
    expect(tx.locktime?.toString("hex")).toBe("00000000");
    expect(tx.nExpiryHeight?.toString("hex")).toBe("00000000");

    expect(tx.inputs.length).toBe(1);
    expect(tx.outputs?.length).toBe(2);

    expect(tx.outputs![0].amount.toString("hex")).toBe("4078150100000000");
    expect(tx.outputs![0].script.toString("hex")).toBe(
      "76a9144cf48844c49a77ba86e48b070f06151b712c862988ac",
    );

    expect(tx.extraData).toBeDefined();
    expect(tx.extraData?.length).toBe(11); // valueBalance (8) + shieldedSpend (1) + shieldedOutput (1) + joinSplit (1)
    expect(tx.extraData!.toString("hex")).toBe("0000000000000000" + "00" + "00" + "00"); // empty shielded stuff
  });

  test("Zcash NU5 transaction (v5)", async () => {
    const transport = await openTransportReplayer(RecordStore.fromString(""));
    const btc = new Btc({ transport, currency: "zcash" });

    const hex =
      "050000800a27a7265510e7c8000000000000000001b5c51aa7f90bd40671eb4887f0022613f5c773f8a30e0c38ff9fc933b754a218000000006b483045022100b4b7b664f7ac6e78026f81f04f9c6fbd7ccbee532ba53e4150ccb6a7c0bd21510220277b4cfdf44683e77a4211e88a3052be00d1c678c36f357db935450c13f2f33701210223b8ffaccab6cc90d2164bfc4361bb058b030217e7cccf677347f075beeef3bb0000000001c0a79500000000001976a914168bb00f59a2d1a059d7e60fcc709cd5a979992988ac000000";

    const tx = btc.splitTransaction(hex, true, true, ["zcash", "sapling"]);

    expect(tx.version.toString("hex")).toBe("05000080");
    expect(tx.nVersionGroupId?.toString("hex")).toBe("0a27a726");
    expect(tx.locktime?.toString("hex")).toBe("00000000");
    expect(tx.nExpiryHeight?.toString("hex")).toBe("00000000");

    expect(tx.inputs.length).toBe(1);
    expect(tx.outputs?.length).toBe(1);

    expect(tx.outputs![0].amount.toString("hex")).toBe("c0a7950000000000");
    expect(tx.outputs![0].script.toString("hex")).toBe(
      "76a914168bb00f59a2d1a059d7e60fcc709cd5a979992988ac",
    );
    expect(tx.extraData).toBeDefined();
    // Overwinter : use nJoinSplit (1)
    expect(tx.extraData!.length).toBe(0); // no extraData for pure NU5 transparent tx
  });

  test("Zcash NU5 transaction Orchard (v5)", async () => {
    const transport = await openTransportReplayer(RecordStore.fromString(""));
    const btc = new Btc({ transport, currency: "zcash" });

    const hex =
      "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";

    const tx = btc.splitTransaction(hex, true, true, ["zcash", "orchard"]);

    expect(tx.version.toString("hex")).toBe("05000080");
    expect(tx.nVersionGroupId?.toString("hex")).toBe("0a27a726");
    expect(tx.locktime?.toString("hex")).toBe("00000000");
    expect(tx.nExpiryHeight?.toString("hex")).toBe("6fe32c00");

    expect(tx.inputs.length).toBe(0);
    expect(tx.outputs?.length).toBe(1);

    expect(tx.outputs![0].amount.toString("hex")).toBe("488e100000000000");
    expect(tx.outputs![0].script.toString("hex")).toBe(
      "76a914e58749ee655c0e39ae3ce063a33fb9edc86d23dd88ac",
    );

    expect(tx.orchard).toBeDefined();
    expect(tx.orchard?.vActions.length).toBe(2);

    expect(tx.orchard?.vActions[0].cmx.toString("hex")).toBe(
      "a9856f1ec074084e8a91d7008cbed27a00a4e4a816f66e2772c5e04b3f876c08",
    );
    expect(tx.orchard?.vActions[0].cv.toString("hex")).toBe(
      "bd730701f6e2e1feffc2021707919aaced34d5e43c13a5010695cd9d333c19bf",
    );
    expect(tx.orchard?.vActions[0].encCiphertext.toString("hex")).toBe(
      "ff76ee84fae0912e178422fff4dc220610589d344a461acaf9487f231d5091575a7deec69f9cdccec92a537b5c996e60a6c16f8502f1e4c873bc88dd2eb2036ba2bfa9cc7b515d721844ce251df61955f78c4e84bff6fdc62404cfbd3bd360b5020bae564e47594ef6cbb4687065749579ab477623d2752f38f30eda3d2231a513b0238b4f45101b2378b50acf8877b182bce2b994ff166cb95f6d319e2b50a0eb432ef7fc556a0e66752fe8ebb0b4184c38820c733fccbeee6e78f3b72f8d61838721f689c4a6a6b1cf8f0fb6bfa1bbdf860ac1631567de443b9cdde8ccdb458b12413f1a1cc90966b40d34249e60eb49d24d9476c89e19721a2d32ce4dbf125a6ffc0a3a9bf3423328ae97e7a2227a08da8e4d1d5031e9d2a6ee04c8d800b1ce42e223d6a16afb88657a46663d7256288e6949b751b4df845a9f544a2366e7976f9ff491e4779dc41431393c7ad99747b2470f24302b28df856506a16ec7b88e77e2c37e2a7a7580d0d6122c63e2f34b63de3011526e6ce4b16bfe43eed07a5364f230d0689c7cc97a2945a7a7e8e8535fa147e665e20bce1d8370e96be7629f34aa72d2a9031c90faf92b8383ed3c51065378b01e143fa64a2eb8322fcf01207c740437b9a9470a67815e541ee1cc3da71349c9092bc296e2d20da040120345717b18df4bb6559caca87349389d9b5567267c640f8289f6caf7c701dc11d0decd9996c47227cc6418c3b13bedd9f4086c7b874b96f0e15bbd6793ba29bd74a4f969f75697e1a817ddba599b2878885724727a32624758df64f86e09d78667e26b4ada",
    );
    expect(tx.orchard?.vActions[0].ephemeralKey.toString("hex")).toBe(
      "671af32842982bcc7f344410d8eb50a2b166e7b00f6a4e7ff64cb368d84fa789",
    );
    expect(tx.orchard?.vActions[0].nullifier.toString("hex")).toBe(
      "310746abef4d8a77fdd55fc303a306f3953f794a95bb8f14d9c94b59d63f792b",
    );
    expect(tx.orchard?.vActions[0].outCiphertext.toString("hex")).toBe(
      "d05f7d8a539a15e7e19eff71fd29a71c6f58bf920456d8a1348f1225fc1d0b1abbc25bce80a645839d272389faf7a0d4367dcff461a64922b85186de7534bdac9739c57bd8dccd9c0b543a2c1cb1a2ed",
    );
    expect(tx.orchard?.vActions[0].rk.toString("hex")).toBe(
      "9e6f9133be659f9cc8b53fc1a26f2e1a9e4e132dd0e78df56be5d06688f103b2",
    );

    expect(tx.orchard?.vActions[1].cmx.toString("hex")).toBe(
      "d3bec60fed5d37c0d46a2384516ecb02520a8dc06f83f747504078569c55a029",
    );
    expect(tx.orchard?.vActions[1].cv.toString("hex")).toBe(
      "bdbcc7541c03dab9665eda1bc07682ce3dcf4f2293e1bba1aabdbfeb2464739a",
    );
    expect(tx.orchard?.vActions[1].encCiphertext.toString("hex")).toBe(
      "4ee493445fdb428c95de6016203993655595c7785d3a0c912bfd6af29601429fb773514ed6eb27f376f56384f52c65f9090fa4627cf17b3c4c577685b1aa7f19ddb6c0596f48447d7aa8253d8faca7bf9bfdb76217500e6a9f1fb794c2c5dba36a4d7efafdbc74d28b7b6f36234312a1e3a99bb412bbe14ebc9e94287994e7846e14ac765809eef7e87289665d0089ed4f3399d547f9156ed34a0598852d52a4daff4096ebd3998e748de36a262e718514577baf7ad16d6be978d382ff83fceefd37cc18940c9f9d7f88e1797023fdd4a017721e876088c86121667780e9a32ed92c749bb018e06157b1c7dcd81d7d35de08e3886338b1c6309b15b099c98281ebb79662526753596f9c5784ef6306c04cadf30f3fa650c2ce49bea66bbbe67580585f7e21cc4d04a16b0f2a3729881307020ee35efac01ae6a79a209dfdc911d4119a57ee9184ac38ccba8475d39c40010d2ae5fbb61fa0b0846cdff83afb9f35e63331743fe8b7098da2e58e0ce97593e9cab30bf362537a49ebe30469fc08aad3e75c9ccdd067d73afa042656a1395d15197f98d3ed1034e0552d852202e4eb34b89b801b10d0402a9d698897e6f9d650f685064794d16c7f93755da6db19cd37de31a41746bbb43199e832f0e7f5f675470e9df5d3afc38716873cb0ade0c4b0e1b5e6275d3adf97e260b10fb74fc379407ccb329c4e06645d80c4ab188a7d952a3e631dc16f1ec82e2c5cafd39005006737d047c276aa7ef03325d4405c9e2bd86ad616d9d343dc8cf2b4543a437be4de0d544dd47d91d7e218a16ee75597137e5f",
    );
    expect(tx.orchard?.vActions[1].ephemeralKey.toString("hex")).toBe(
      "0ea78f4a3b765cd00d69a3f60ad695b42d652b285caf3aea639aadb694e24f37",
    );
    expect(tx.orchard?.vActions[1].nullifier.toString("hex")).toBe(
      "e16a84f5840e498b935bce15d2900a592d33446c14e9b9bfd8d05b4e24d0142a",
    );
    expect(tx.orchard?.vActions[1].outCiphertext.toString("hex")).toBe(
      "4d95e868a9a0939cc41f3d36bae7c2e769f5a6c5607b6f314d667d1a7dd951ae5351c98c4595cce67cec410a6609a15472670b6841cc12719e645b97aad11afce304cb7ffe866cd9297717de348d25c6",
    );
    expect(tx.orchard?.vActions[1].rk.toString("hex")).toBe(
      "9027b8a15f11e25d9efd7886e96a214344bbd44c046b223bd1ee5f0b636c5e07",
    );

    expect(tx.orchard?.anchor.toString("hex")).toBe(
      "6cfe8df49885ea5ba8005b075db957a0f11d524a1f226a2da39e912e990ca93e",
    );

    expect(tx.orchard?.vSpendsAuthSigs.toString("hex")).toBe(
      "c89d60b899bc3e54079da17c5cc487ccc555ab83ebd36d44b401f772169dea21bd14563007373e0e55ca1a084d35fa557ef0c66a57f708912193a6cb1f924d28037c98e42a44153fe28ddb7023aa8bc77a60806c4a0d8be104a42004dda77aa056165106b0b13510dd97514e3bd5f4059b2f9b525b565bc798020e4582c60602",
    );

    expect(tx.orchard?.bindingSig.toString("hex")).toBe(
      "48c8075628cd39d663497ade707a4be75a347b27ef86fb9d73f1fec0d2c877b42f28b999406f533b158ed8c2b05180966c375922eaffb8a096c4ae9eed2b073d",
    );

    expect(tx.extraData).toBeDefined();
    expect(tx.extraData?.length).toBe(0);
  });

  test("Zcash NU5 transaction Sapling (v5)", async () => {
    const transport = await openTransportReplayer(RecordStore.fromString(""));
    const btc = new Btc({ transport, currency: "zcash" });

    const hex =
      "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";

    const tx = btc.splitTransaction(hex, true, true, ["zcash", "sapling"]);

    expect(tx.version.toString("hex")).toBe("05000080");
    expect(tx.nVersionGroupId?.toString("hex")).toBe("0a27a726");
    expect(tx.locktime?.toString("hex")).toBe("00000000");
    expect(tx.nExpiryHeight?.toString("hex")).toBe("94e02c00");

    expect(tx.inputs.length).toBe(0);
    expect(tx.outputs?.length).toBe(1);

    expect(tx.outputs![0].amount.toString("hex")).toBe("20d6130000000000");
    expect(tx.outputs![0].script.toString("hex")).toBe(
      "76a914040ac822bd91e60f709174ac94bee1fb1aaadf2a88ac",
    );

    expect(tx.sapling).toBeDefined();
    expect(tx.sapling?.vSpendsSapling.length).toBe(1);

    expect(tx.sapling?.vSpendsSapling[0].cv.toString("hex")).toBe(
      "2b88b763bbed1f015f89aca3786acd2702a5ec69482404fc90cb4e6f597827a1",
    );
    expect(tx.sapling?.vSpendsSapling[0].nullifier.toString("hex")).toBe(
      "884a2be02987a69f1461b3d479661407c8f31f5f26140c964c7e28315e429115",
    );
    expect(tx.sapling?.vSpendsSapling[0].rk.toString("hex")).toBe(
      "c534b33f896e0320f630a4c4418b25db0c2ce8412a386f29be29a138d496a260",
    );

    expect(tx.sapling?.vOutputSapling.length).toBe(2);

    expect(tx.sapling?.vOutputSapling[0].cv.toString("hex")).toBe(
      "d50845284903cd9e2f3da8b5d716217e82aa6021acf7e402a5918dd39602710c",
    );
    expect(tx.sapling?.vOutputSapling[0].cmu.toString("hex")).toBe(
      "bfd33cf9757e6da7374d12598bdfee4a821f915254a1fed7189a0acecf52aa40",
    );
    expect(tx.sapling?.vOutputSapling[0].ephemeralKey.toString("hex")).toBe(
      "87d61a8797497ba66dffeec0d3451b01c16b4dcbcc27f6896cdf59f07d4d535f",
    );
    expect(tx.sapling?.vOutputSapling[0].encCiphertext.toString("hex")).toBe(
      "b3b9f3b7a013d2b69aee28b9d4bdee0eaab836b776e6419324d5690dc1050b0b10063bead8c37ea1ebf03e390be41f313d370e709953f91cfa0923a220f031b58e536e503cc161a046d7eefe1194f7fe3d879417b3ad8ed4e04620ecf771cf503939eb441fd2586d2711a1d1228292e66bc09986d06e332b697153228efa383094ecb27eca5288771f6a721185ec59bfeb7695b66692de9207a638386304b2fef517c1002158b7a340230d79be282d56764e78c77a4c09bb70874e0888a9c4b54df76e92621ec8ce3981e3413e0ce4576caa99a2fbeee2cbaa9d4448fa22c27fa588455bdb6ea9cf2515c8edb475a6da460a5e0072150f5d71568bf875a6cc0be52944aaa360dd425722ef70ff6e785263b45d0ce6053ea2a48360ad66432b84550e1634d6eb40a99e8c44e229d9de18306325d64f1595c928df3bece71461c281ed948fb54ad4c059cd78e0bd50f39a38cbeeb68977582b90ec18cc799ae9c63fbb028d884d4c32067cd6e73809b42b473c1c5031de16e524fe242c2840d8cb2d554544525e4d2e48655ed17c5d8f98a5f0b30d72a4be22fe250ea5d4b28e51421ecfa3d6c146edafdc90c6950f8f86c1c5db73b53e8323a1c59347de9f6a420aab8a1ea3245f0eac5c420e8d2333c25846289515ccfe15288670d870c2de8ff6dc737b5bb15b6d8c59366d59a15b3361475f7de91fabf0972e323356ecd6ee3bbe8129eac8c8a55dbd4945ad6c02b774dd986503ea7dbd1ef4d0fe00e88efbf09cc2ce2754963e085076cbda51a43fe04ffbc5a57e7862ccbfa5b7ebf4f999c2450108",
    );
    expect(tx.sapling?.vOutputSapling[0].outCiphertext.toString("hex")).toBe(
      "589143fbcd257fade54537bd6b808f0f59a020033cc09166790f7eae8eb2c236bbcfa0515846939b10f8d675a0c39d5dadf8f8df6039d7c56965255de8b22f43bf40749c65ff1c6b9c939d394e165be0",
    );

    expect(tx.sapling?.vOutputSapling[1].cv.toString("hex")).toBe(
      "66bc1edf06b465f42ffaabddc059f238e02879ade2a08041314665b38ab2e13b",
    );
    expect(tx.sapling?.vOutputSapling[1].cmu.toString("hex")).toBe(
      "ef91cb7052c730740ca3fdb5082ca9316fd7fbe4ce63ad66fc32bdfe7106bb34",
    );
    expect(tx.sapling?.vOutputSapling[1].ephemeralKey.toString("hex")).toBe(
      "f6cf09d89739dff6205051189a6d1bd7aa4f7518e2cf1872272f9a90baa053c9",
    );
    expect(tx.sapling?.vOutputSapling[1].encCiphertext.toString("hex")).toBe(
      "32622ebd195de4ce00c7c3fdbdde46c11fd616fee703cf770caa03e9f6ffa1f32b77e21fffee6fa03badeb99dadaafdfd20ad05a98c04020e8a852c105a2b82a543e8e1e3b24646e196fbf1d75fe4bd77a82990a4cf54935d02b33b2853504567b9f8b3c2d056c489d991b16543cfbaf6e69a42cc356b799705085b7b58347272a5ff0c454dc5ba733dc6127d573844e4f7461f57de2ec04eb97bd94dc29f9bd700cd4f7e0f0c5144e1e24c83528735e17ce585d4ca2ca7318d44f5fe00b069cc27806c557c5dba69f3e8ab910ba3509f25ec05af5a0f0e7e86eaec4ec1264124c02d7c35bb90ef500cc813d1f1f37ead8eba59e3c7c346042e4bf29055a823c820bb47000d8e0aa07707f33f0ba4254538532066f0e59ac60a64547acf69ba15b4cd61bc6b7e7d4fa111fcbe7eed9cd125f57358c74d1a0db663487ea4f88ae079460c09199e93315defab6ace9df089954beb09dde24549b5ec3e6e0c6d95580ea1a25516016f87b52ecb902c9bc87f9386e668867f920a6c0da05ff9da0718fb6dff84d3f381f1c40da6e1bc75b6045a54296b759f99071e145d99b9772616280d04c278bff7c3688070fc7ac4ff6e0bdb3294224af07e1efe10cf57f6898a39b63ddd9792c4effd933414a1cbc23b20ccf1f3624a187b4faab8687ad58b18a4dc5753343ddff7477370f8e766d74e7c5db381910c768270cb62096b110dd9268ad97149fc13c5513024dfafb4093fd6d31c942d9bfb0630f98957d9782dac478f39c6d74affcdf031a56e63e090437d1670afc2a1e75cbd359667fd002f83c70aa35",
    );
    expect(tx.sapling?.vOutputSapling[1].outCiphertext.toString("hex")).toBe(
      "685314e8c9a8e00d3e653a9d90ccff47b02cd6621ee5ddd94e3e3a822c52bbbfce64b8a55bfe6d3da9bfb1b1f8948c3a7fdfee92bf369b43bb432954273d98f251d344aa710d812c8b84b8ab4f694080",
    );

    expect(tx.sapling?.valueBalanceSapling.toString("hex")).toBe("b810140000000000");
    expect(tx.sapling?.anchorSapling.toString("hex")).toBe(
      "ec191fd9bc3a13624b396dcdef5d13559ced5bcc2058b98f55b2250b9b98b658",
    );
    expect(tx.sapling?.vSpendProofsSapling.toString("hex")).toBe(
      "8ac98d92427f8008c4b292c3b75d1c36a92f1fa92d5c3581993230b572fa0324334ce2b7379cbc0dd5bc13e66cb27bc1a87c4cabe06c49c246a1bacb9745f3cf64f626cf9d7d0483a8049efa634490ca4c93d7fd5f7d2fbfbdcc61adef53e7ee09652156e3d9a5564d3db8ac040346fe5b133fe6839a67ac05876dbf3c14152da4efff436c1b593208f30a3d61c232bbae6caf5a916f8a3b402bb60d2f07a38976445eef9c9cf097b7bb08c81d1a84dcdbffd0af6fb144ef1da6320d97e4370e",
    );
    expect(tx.sapling?.vOutputProofsSapling.toString("hex")).toBe(
      "a9fba81b3f6270c485ef960fed6e94f95c130f3e82407700b37f195436bf54b2f7fa679420d98c7ce817432617bc2c0fa3cd3699a93b282e565bdbaeb0c79f686efe2ffa7210e55c0197441b2316f744dfd63715090558fa80c68f453a1d54f314dd356cc57c8978c625695cccca179f03105571a2011068727b471d019824edbc6f15edfb241adc0e4ce0dfd2bae8188a295745ae69027f0781127116d6ed72afcc49584d26cb063d6cee4adf4900e82d2173202cd905a9db2ea254e7126a5786ea4eede15bb2b57f0b218faba033aeb9e6ae9a57c3ce635638f7d40364624a516c5dca0b8df23f1564eba8fd9cae02920a6a7e20a904efcb5f049545c95625289f8c3f1d3772ba556272d7eec84592d1838cbba94b44ef68a74a11da7cc92c0cf2230a01bd6c7724ee6bf9893af8b75574e785614c0d0ed207dbf9f6b469ac449de4c4bbb2c48036a86580c4fa36ccb51d617dba9bf9281eddf1ec1c417e5ba2282cae6f071052ed783de396ba2d4c2cf3699c46aad4b4d678e4a314553c6e",
    );
    expect(tx.sapling?.bindingSigSapling.toString("hex")).toBe(
      "f5488e3905fc621c3c5281194a4a941a922b6a4cf323db5cd9a5ae1cf1582c808d0165b9cc46af1a13c7eb30737a74464daf551415cfd91b33ae98e350e56a07",
    );

    expect(tx.extraData).toBeDefined();
    expect(tx.extraData?.length).toBe(0);
  });
});
