"use strict";
// Copyright (c) 2023 Jose-Luis Landabaso - https://bitcoinerlab.com
// Distributed under the MIT software license
Object.defineProperty(exports, "__esModule", { value: true });
exports.signLedger = exports.signInputLedger = exports.signBIP32 = exports.signInputBIP32 = exports.signECPair = exports.signInputECPair = void 0;
const ledger_1 = require("./ledger");
function signInputECPair({ psbt, index, ecpair }) {
    psbt.signInput(index, ecpair);
}
exports.signInputECPair = signInputECPair;
function signECPair({ psbt, ecpair }) {
    psbt.signAllInputs(ecpair);
}
exports.signECPair = signECPair;
function signInputBIP32({ psbt, index, node }) {
    psbt.signInputHD(index, node);
}
exports.signInputBIP32 = signInputBIP32;
function signBIP32({ psbt, masterNode }) {
    psbt.signAllInputsHD(masterNode);
}
exports.signBIP32 = signBIP32;
const ledgerSignaturesForInputIndex = (index, ledgerSignatures) => ledgerSignatures
    .filter(([i]) => i === index)
    .map(([_i, partialSignature]) => ({
    pubkey: partialSignature.pubkey,
    signature: partialSignature.signature
}));
async function signInputLedger({ psbt, index, descriptor, ledgerClient, ledgerState }) {
    const { PsbtV2, DefaultWalletPolicy, WalletPolicy, AppClient } = (await (0, ledger_1.importAndValidateLedgerBitcoin)(ledgerClient));
    if (!(ledgerClient instanceof AppClient))
        throw new Error(`Error: pass a valid ledgerClient`);
    const result = await (0, ledger_1.descriptorToLedgerFormat)({
        descriptor,
        ledgerClient,
        ledgerState
    });
    if (!result)
        throw new Error(`Error: descriptor does not have a ledger input`);
    const { ledgerTemplate, keyRoots } = result;
    let ledgerSignatures;
    const standardPolicy = await (0, ledger_1.ledgerPolicyFromStandard)({
        descriptor,
        ledgerClient,
        ledgerState
    });
    if (standardPolicy) {
        ledgerSignatures = await ledgerClient.signPsbt(new PsbtV2().fromBitcoinJS(psbt), new DefaultWalletPolicy(ledgerTemplate, keyRoots[0]), null);
    }
    else {
        const policy = await (0, ledger_1.ledgerPolicyFromState)({
            descriptor,
            ledgerClient,
            ledgerState
        });
        if (!policy || !policy.policyName || !policy.policyHmac)
            throw new Error(`Error: the descriptor's policy is not registered`);
        const walletPolicy = new WalletPolicy(policy.policyName, ledgerTemplate, keyRoots);
        ledgerSignatures = await ledgerClient.signPsbt(new PsbtV2().fromBitcoinJS(psbt), walletPolicy, policy.policyHmac);
    }
    //Add the signatures to the Psbt object using PartialSig format:
    psbt.updateInput(index, {
        partialSig: ledgerSignaturesForInputIndex(index, ledgerSignatures)
    });
}
exports.signInputLedger = signInputLedger;
//signLedger is able to sign several inputs of the same wallet policy since it
//it clusters together wallet policy types before signing
//it throws if it cannot sign any input.
async function signLedger({ psbt, descriptors, ledgerClient, ledgerState }) {
    const { PsbtV2, DefaultWalletPolicy, WalletPolicy, AppClient } = (await (0, ledger_1.importAndValidateLedgerBitcoin)(ledgerClient));
    if (!(ledgerClient instanceof AppClient))
        throw new Error(`Error: pass a valid ledgerClient`);
    const ledgerPolicies = [];
    for (const descriptor of descriptors) {
        const policy = (await (0, ledger_1.ledgerPolicyFromState)({
            descriptor,
            ledgerClient,
            ledgerState
        })) ||
            (await (0, ledger_1.ledgerPolicyFromStandard)({
                descriptor,
                ledgerClient,
                ledgerState
            }));
        if (policy)
            ledgerPolicies.push(policy);
    }
    if (ledgerPolicies.length === 0)
        throw new Error(`Error: there are no inputs which could be signed`);
    //cluster unique LedgerPolicies
    const uniquePolicies = [];
    for (const policy of ledgerPolicies) {
        if (!uniquePolicies.find((uniquePolicy) => (0, ledger_1.comparePolicies)(uniquePolicy, policy)))
            uniquePolicies.push(policy);
    }
    for (const uniquePolicy of uniquePolicies) {
        let ledgerSignatures;
        if (uniquePolicy.policyName &&
            uniquePolicy.policyHmac &&
            uniquePolicy.policyId) {
            //non-standard policy
            const walletPolicy = new WalletPolicy(uniquePolicy.policyName, uniquePolicy.ledgerTemplate, uniquePolicy.keyRoots);
            ledgerSignatures = await ledgerClient.signPsbt(new PsbtV2().fromBitcoinJS(psbt), walletPolicy, uniquePolicy.policyHmac);
        }
        else {
            //standard policy
            ledgerSignatures = await ledgerClient.signPsbt(new PsbtV2().fromBitcoinJS(psbt), new DefaultWalletPolicy(uniquePolicy.ledgerTemplate, uniquePolicy.keyRoots[0]), null);
        }
        for (const [index, ,] of ledgerSignatures) {
            psbt.updateInput(index, {
                partialSig: ledgerSignaturesForInputIndex(index, ledgerSignatures)
            });
        }
    }
}
exports.signLedger = signLedger;
