/// <reference types="node" />
import type { DescriptorInstance } from './descriptors';
/**
 * Dynamically imports the 'ledger-bitcoin' module and, if provided, checks if `ledgerClient` is an instance of `AppClient`.
 *
 * @async
 * @param {unknown} ledgerClient - An optional parameter that, if provided, is checked to see if it's an instance of `AppClient`.
 * @throws {Error} Throws an error if `ledgerClient` is provided but is not an instance of `AppClient`.
 * @throws {Error} Throws an error if the 'ledger-bitcoin' module cannot be imported. This typically indicates that the 'ledger-bitcoin' peer dependency is not installed.
 * @returns {Promise<unknown>} Returns a promise that resolves with the entire 'ledger-bitcoin' module if it can be successfully imported. We force it to return an unknown type so that the declaration of this function won't break projects that don't use ledger-bitcoin as dependency
 *
 * @example
 *
 * importAndValidateLedgerBitcoin(ledgerClient)
 *   .then((module) => {
 *     const { AppClient, PsbtV2, DefaultWalletPolicy, WalletPolicy, DefaultDescriptorTemplate, PartialSignature } = module;
 *     // Use the imported objects...
 *   })
 *   .catch((error) => console.error(error));
 */
export declare function importAndValidateLedgerBitcoin(ledgerClient?: unknown): Promise<unknown>;
export declare function assertLedgerApp({ transport, name, minVersion }: {
    transport: any;
    name: string;
    minVersion: string;
}): Promise<void>;
export type LedgerPolicy = {
    policyName?: string;
    ledgerTemplate: string;
    keyRoots: string[];
    policyId?: Buffer;
    policyHmac?: Buffer;
};
export type LedgerState = {
    masterFingerprint?: Buffer;
    policies?: LedgerPolicy[];
    xpubs?: {
        [key: string]: string;
    };
};
export declare function getLedgerMasterFingerPrint({ ledgerClient, ledgerState }: {
    ledgerClient: unknown;
    ledgerState: LedgerState;
}): Promise<Buffer>;
export declare function getLedgerXpub({ originPath, ledgerClient, ledgerState }: {
    originPath: string;
    ledgerClient: unknown;
    ledgerState: LedgerState;
}): Promise<string>;
/**
 * Takes a descriptor and gets its Ledger Wallet Policy, that is, its keyRoots and template.
 * keyRoots and template follow Ledger's specifications:
 * https://github.com/LedgerHQ/app-bitcoin-new/blob/develop/doc/wallet.md
 *
 * keyRoots and template are a generalization of a descriptor and serve to
 * describe internal and external addresses and any index.
 *
 * So, this function starts from a descriptor and obtains generalized Ledger
 * wallet policy.
 *
 * keyRoots is an array of strings, encoding xpub-type key expressions up to the origin.
 * F.ex.: [76223a6e/48'/1'/0'/2']tpubDE7NQymr4AFtewpAsWtnreyq9ghkzQBXpCZjWLFVRAvnbf7vya2eMTvT2fPapNqL8SuVvLQdbUbMfWLVDCZKnsEBqp6UK93QEzL8Ck23AwF
 *
 * Template encodes the descriptor script expression, where its key
 * expressions are represented using variables for each keyRoot and finished with "/**"
 * (for change 1 or 0 and any index). F.ex.:
 * wsh(sortedmulti(2,@0/**,@1/**)), where @0 corresponds the first element in the keyRoots array.
 *
 * If this descriptor does not contain any key that can be signed with the ledger
 * (non-matching masterFingerprint), then this function returns null.
 *
 * This function takes into account all the considerations regarding Ledger
 * policy implementation details expressed in the header of this file.
 */
export declare function descriptorToLedgerFormat({ descriptor, ledgerClient, ledgerState }: {
    descriptor: DescriptorInstance;
    ledgerClient: unknown;
    ledgerState: LedgerState;
}): Promise<{
    ledgerTemplate: string;
    keyRoots: string[];
} | null>;
/**
 * It registers a policy based on a descriptor. It stores it in ledgerState.
 *
 * If the policy was already registered, it does not register it.
 * If the policy is standard, it does not register it.
 *
 **/
export declare function registerLedgerWallet({ descriptor, ledgerClient, ledgerState, policyName }: {
    descriptor: DescriptorInstance;
    ledgerClient: unknown;
    ledgerState: LedgerState;
    policyName: string;
}): Promise<void>;
/**
 * Retrieve a standard ledger policy or null if it does correspond.
 **/
export declare function ledgerPolicyFromStandard({ descriptor, ledgerClient, ledgerState }: {
    descriptor: DescriptorInstance;
    ledgerClient: unknown;
    ledgerState: LedgerState;
}): Promise<LedgerPolicy | null>;
export declare function comparePolicies(policyA: LedgerPolicy, policyB: LedgerPolicy): boolean;
/**
 * Retrieve a ledger policy from ledgerState or null if it does not exist yet.
 **/
export declare function ledgerPolicyFromState({ descriptor, ledgerClient, ledgerState }: {
    descriptor: DescriptorInstance;
    ledgerClient: unknown;
    ledgerState: LedgerState;
}): Promise<LedgerPolicy | null>;
