"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getMiscFeeLevels = exports.getEthereumFeeLevels = exports.getBitcoinFeeLevels = exports.DEFAULT_BITCOIN_LONGTERM_FEE_RATE = void 0;
const utils_1 = require("@trezor/utils");
const BLOCKS_FOR_FEE_LEVEL = {
    btc: {
        high: 1,
        normal: 3,
        economy: 12,
        low: 36,
    },
};
const DEFAULT_BLOCK_FOR_FEE_LEVEL = 1;
const getDefaultBlocksForFeeLevel = (shortcut, label) => BLOCKS_FOR_FEE_LEVEL[shortcut] && BLOCKS_FOR_FEE_LEVEL[shortcut][label]
    ? BLOCKS_FOR_FEE_LEVEL[shortcut][label]
    : DEFAULT_BLOCK_FOR_FEE_LEVEL;
exports.DEFAULT_BITCOIN_LONGTERM_FEE_RATE = '1';
const EVM_GAS_PRICE_PER_CHAIN_IN_GWEI = {
    eth: { min: 0.1, max: 10000, defaultGas: 10, minPriorityFee: 0 },
    pol: { min: 0.1, max: 10000000, defaultGas: 200, minPriorityFee: 30 },
    bsc: { min: 0.1, max: 100000, defaultGas: 1, minPriorityFee: 0 },
    base: { min: 0.0000001, max: 1000, defaultGas: 0.01, minPriorityFee: 0 },
    arb: { min: 0.001, max: 1000, defaultGas: 0.01, minPriorityFee: 0 },
    op: { min: 0.000000001, max: 1000, defaultGas: 0.01, minPriorityFee: 0 },
};
const getEvmChainGweiGasPrice = (chain) => EVM_GAS_PRICE_PER_CHAIN_IN_GWEI[chain] ?? {
    min: 0.000000001,
    max: 10000,
    defaultGas: 1,
    minPriorityFee: 0,
};
const getBitcoinFeeLevels = (coin) => {
    const defaultFees = coin.default_fee_b;
    const shortcut = coin.shortcut.toLowerCase();
    const keys = (0, utils_1.typedObjectKeys)(defaultFees);
    const levels = keys
        .sort((levelA, levelB) => defaultFees[levelB] - defaultFees[levelA])
        .map(level => {
        const label = level.toLowerCase();
        return {
            label,
            feePerUnit: defaultFees[level].toString(),
            blocks: getDefaultBlocksForFeeLevel(shortcut, label),
        };
    });
    const minFee = coin.shortcut === 'BTC' ? coin.minfee_kb / 1000 : Math.round(coin.minfee_kb / 1000);
    return {
        blockTime: Math.max(1, Math.round(coin.blocktime_seconds / 60)),
        dustLimit: coin.dust_limit,
        maxFee: Math.round(coin.maxfee_kb / 1000),
        minFee,
        minPriorityFee: -1,
        defaultFees: levels,
    };
};
exports.getBitcoinFeeLevels = getBitcoinFeeLevels;
const getEthereumFeeLevels = (network) => {
    const { min, max, defaultGas, minPriorityFee } = getEvmChainGweiGasPrice(network.chain);
    return {
        blockTime: Math.max(0.1, Math.round(network.blocktime_seconds)),
        defaultFees: [
            {
                label: 'normal',
                feePerUnit: new utils_1.BigNumber(defaultGas).multipliedBy('1e+9').toString(),
                feeLimit: '21000',
                blocks: -1,
            },
        ],
        minFee: min,
        maxFee: max,
        minPriorityFee,
        dustLimit: -1,
    };
};
exports.getEthereumFeeLevels = getEthereumFeeLevels;
const RIPPLE_FEE_INFO = {
    blockTime: -1,
    defaultFees: [{ label: 'normal', feePerUnit: '12', blocks: -1 }],
    minFee: 10,
    maxFee: 10000,
    minPriorityFee: -1,
    dustLimit: -1,
};
const CARDANO_FEE_INFO = {
    blockTime: -1,
    defaultFees: [{ label: 'normal', feePerUnit: '44', blocks: -1 }],
    minFee: 44,
    maxFee: 16384 * 44 + 155381,
    minPriorityFee: -1,
    dustLimit: -1,
};
const SOLANA_FEE_INFO = {
    blockTime: -1,
    defaultFees: [
        {
            label: 'normal',
            feePerUnit: '100000',
            feeLimit: '200000',
            feePerTx: '25000',
            blocks: -1,
        },
    ],
    minFee: 5000,
    maxFee: 1000000000,
    minPriorityFee: -1,
    dustLimit: -1,
};
const STELLAR_FEE_INFO = {
    blockTime: -1,
    defaultFees: [{ label: 'normal', feePerUnit: '100', blocks: -1 }],
    minFee: 100,
    maxFee: 10000000,
    minPriorityFee: -1,
    dustLimit: -1,
};
const MISC_FEE_LEVELS = {
    xrp: RIPPLE_FEE_INFO,
    txrp: RIPPLE_FEE_INFO,
    ada: CARDANO_FEE_INFO,
    tada: CARDANO_FEE_INFO,
    sol: SOLANA_FEE_INFO,
    dsol: SOLANA_FEE_INFO,
    xlm: STELLAR_FEE_INFO,
    txlm: STELLAR_FEE_INFO,
};
const getMiscFeeLevels = (data) => {
    const shortcut = data.shortcut.toLowerCase();
    return (MISC_FEE_LEVELS[shortcut] || {
        blockTime: -1,
        minFee: -1,
        maxFee: -1,
        defaultFees: [{ label: 'normal', feePerUnit: '-1', blocks: -1 }],
        minPriorityFee: -1,
        dustLimit: -1,
    });
};
exports.getMiscFeeLevels = getMiscFeeLevels;
//# sourceMappingURL=defaultFeeLevels.js.map