"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const schema_utils_1 = require("@trezor/schema-utils");
const AbstractMethod_1 = require("../core/AbstractMethod");
const events_1 = require("../events");
const paramsValidator_1 = require("./common/paramsValidator");
const cipherKeyValue_1 = require("../types/api/cipherKeyValue");
const params_1 = require("../types/params");
const pathUtils_1 = require("../utils/pathUtils");
class CipherKeyValue extends AbstractMethod_1.AbstractMethod {
    hasBundle;
    init() {
        this.requiredPermissions = ['read', 'write'];
        this.firmwareRange = (0, paramsValidator_1.getFirmwareRange)(this.name, null, this.firmwareRange);
        this.hasBundle = !!this.payload.bundle;
        const payload = !this.payload.bundle
            ? { ...this.payload, bundle: [this.payload] }
            : this.payload;
        this.useEmptyPassphrase =
            typeof payload.useEmptyPassphrase === 'boolean' ? payload.useEmptyPassphrase : true;
        (0, schema_utils_1.Assert)((0, params_1.Bundle)(cipherKeyValue_1.CipherKeyValue), payload);
        this.params = payload.bundle.map(batch => ({
            address_n: (0, pathUtils_1.validatePath)(batch.path),
            key: batch.key,
            value: batch.value instanceof Buffer
                ? batch.value.toString('hex')
                : batch.value,
            encrypt: batch.encrypt,
            ask_on_encrypt: batch.askOnEncrypt,
            ask_on_decrypt: batch.askOnDecrypt,
            iv: batch.iv instanceof Buffer ? batch.iv.toString('hex') : batch.iv,
        }));
    }
    get info() {
        return 'Cipher key value';
    }
    async run() {
        const responses = [];
        const cmd = this.device.getCommands();
        for (let i = 0; i < this.params.length; i++) {
            const response = await cmd.typedCall('CipherKeyValue', 'CipheredKeyValue', this.params[i]);
            responses.push(response.message);
            if (this.hasBundle) {
                this.postMessage((0, events_1.createUiMessage)(events_1.UI.BUNDLE_PROGRESS, {
                    total: this.params.length,
                    progress: i,
                    response,
                }));
            }
        }
        return this.hasBundle ? responses : responses[0];
    }
}
exports.default = CipherKeyValue;
//# sourceMappingURL=cipherKeyValue.js.map