"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.inputToTrezor = exports.enhanceTrezorInputs = exports.validateTrezorInputs = void 0;
const hdnodeUtils_1 = require("../../utils/hdnodeUtils");
const pathUtils_1 = require("../../utils/pathUtils");
const paramsValidator_1 = require("../common/paramsValidator");
const validateTrezorInputs = (inputs, coinInfo) => inputs
    .map(i => (0, pathUtils_1.fixPath)(i))
    .map(i => (0, hdnodeUtils_1.convertMultisigPubKey)(coinInfo.network, i))
    .map(input => {
    const useAmount = input.script_type === 'EXTERNAL' || (0, pathUtils_1.isSegwitPath)(input.address_n);
    (0, paramsValidator_1.validateParams)(input, [
        { name: 'prev_hash', type: 'string', required: true },
        { name: 'prev_index', type: 'number', required: true },
        { name: 'amount', type: 'uint', required: useAmount },
        { name: 'script_type', type: 'string' },
        { name: 'sequence', type: 'number' },
        { name: 'multisig', type: 'object' },
        { name: 'coinjoin_flags', type: 'number' },
    ]);
    if (input.script_type === 'EXTERNAL') {
        (0, paramsValidator_1.validateParams)(input, [
            { name: 'script_pubkey', type: 'string', required: true },
            { name: 'commitment_data', type: 'string' },
            { name: 'ownership_proof', type: 'string' },
            { name: 'script_sig', type: 'string' },
            { name: 'witness', type: 'string' },
        ]);
    }
    else {
        (0, pathUtils_1.validatePath)(input.address_n);
    }
    return input;
});
exports.validateTrezorInputs = validateTrezorInputs;
const enhanceTrezorInputs = (inputs, rawTxs) => {
    inputs.forEach(input => {
        if (!input.amount) {
            console.warn('TrezorConnect.signTransaction deprecation: missing input amount.');
            const refTx = rawTxs.find(t => t.getId() === input.prev_hash);
            if (refTx && refTx.outs[input.prev_index]) {
                input.amount = refTx.outs[input.prev_index].value;
            }
        }
    });
};
exports.enhanceTrezorInputs = enhanceTrezorInputs;
const inputToTrezor = (input, sequence = 0xffffffff) => {
    const address_n = (0, pathUtils_1.getHDPath)(input.path);
    return {
        address_n,
        prev_index: input.vout,
        prev_hash: input.txid,
        script_type: (0, pathUtils_1.getScriptType)(address_n),
        amount: input.amount,
        sequence,
    };
};
exports.inputToTrezor = inputToTrezor;
//# sourceMappingURL=inputs.js.map