"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.sumAddressValues = void 0;
const blockchain_link_utils_1 = require("@trezor/blockchain-link-utils");
const blockbook_1 = require("@trezor/blockchain-link-utils/lib/blockbook");
const utxo_lib_1 = require("@trezor/utxo-lib");
const utils_1 = require("../utils");
const PAGE_SIZE_DEFAULT = 25;
const getBalances = (client) => (addresses) => Promise.all(addresses.map(async ({ address, path, history, scripthash, empty }) => {
    const { confirmed, unconfirmed } = history.length
        ? await client.request('blockchain.scripthash.get_balance', scripthash)
        : {
            confirmed: 0,
            unconfirmed: 0,
        };
    return {
        address,
        path,
        history,
        empty,
        confirmed,
        unconfirmed,
    };
}));
const getPagination = (perPage, txs) => ({
    index: 1,
    size: perPage,
    total: Math.ceil(txs.length / perPage),
});
const sumAddressValues = (transactions, address, getVinVouts) => transactions
    .flatMap(tx => getVinVouts(tx)
    .filter(({ addresses }) => addresses?.includes(address))
    .map(({ value }) => (value ? Number.parseFloat(value) : 0)))
    .reduce((a, b) => a + b, 0);
exports.sumAddressValues = sumAddressValues;
const getAccountInfo = async (client, payload) => {
    const { descriptor, details = 'basic', pageSize = PAGE_SIZE_DEFAULT } = payload;
    const network = client.getInfo()?.network;
    const parsed = (0, utils_1.tryGetScripthash)(descriptor, network);
    if (parsed.valid) {
        const { confirmed, unconfirmed, history } = await Promise.all([
            client.request('blockchain.scripthash.get_balance', parsed.scripthash),
            client.request('blockchain.scripthash.get_history', parsed.scripthash),
        ]).then(([{ confirmed, unconfirmed }, history]) => ({
            confirmed,
            unconfirmed,
            history,
        }));
        const historyUnconfirmed = history.filter(r => r.height <= 0).length;
        const transactions = details === 'txs'
            ? await (0, utils_1.getTransactions)(client, history)
                .then(txs => txs.map(tx => (0, blockbook_1.transformTransaction)(tx, descriptor)))
                .then(blockchain_link_utils_1.sortTxsFromLatest)
            : undefined;
        return {
            descriptor,
            balance: confirmed.toString(),
            availableBalance: (confirmed + unconfirmed).toString(),
            empty: !history.length,
            history: {
                total: history.length - historyUnconfirmed,
                unconfirmed: historyUnconfirmed,
                transactions,
            },
            page: details === 'txs' ? getPagination(pageSize, transactions ?? []) : undefined,
        };
    }
    const discover = (0, utils_1.discoverAddress)(client);
    const receive = await (0, utxo_lib_1.discovery)(discover, descriptor, 'receive', network).then(getBalances(client));
    const change = await (0, utxo_lib_1.discovery)(discover, descriptor, 'change', network).then(getBalances(client));
    const batch = receive.concat(change);
    const [confirmed, unconfirmed] = batch.reduce(([c, u], { confirmed, unconfirmed }) => [c + confirmed, u + unconfirmed], [0, 0]);
    const history = batch.flatMap(({ history }) => history);
    const historyUnconfirmed = history.filter(r => r.height <= 0).length;
    const transformAddressInfo = ({ address, path, history, confirmed }) => ({
        address,
        path,
        transfers: history.length,
        balance: confirmed.toString(),
        sent: '0',
        received: '0',
    });
    const addresses = {
        change: change.map(transformAddressInfo),
        unused: receive.filter(recv => !recv.history.length).map(transformAddressInfo),
        used: receive.filter(recv => recv.history.length).map(transformAddressInfo),
    };
    const transactions = ['tokenBalances', 'txids', 'txs'].includes(details)
        ? await (0, utils_1.getTransactions)(client, history)
            .then(txs => txs.map(tx => (0, blockbook_1.transformTransaction)(tx, addresses)))
            .then(blockchain_link_utils_1.sortTxsFromLatest)
        : [];
    const extendAddressInfo = ({ address, path, transfers, balance }) => ({
        address,
        path,
        transfers,
        balance: '0',
        sent: '0',
        received: '0',
        ...(['tokenBalances', 'txids', 'txs'].includes(details) && transfers
            ? {
                balance,
                sent: (0, exports.sumAddressValues)(transactions, address, tx => tx.details.vin).toString(),
                received: (0, exports.sumAddressValues)(transactions, address, tx => tx.details.vout).toString(),
            }
            : {}),
    });
    return {
        descriptor,
        balance: confirmed.toString(),
        availableBalance: (confirmed + unconfirmed).toString(),
        empty: !history.length,
        history: {
            total: history.length - historyUnconfirmed,
            unconfirmed: historyUnconfirmed,
            transactions: details === 'txs' ? transactions : undefined,
        },
        addresses: details !== 'basic'
            ? {
                change: addresses.change.map(extendAddressInfo),
                unused: addresses.unused.map(extendAddressInfo),
                used: addresses.used.map(extendAddressInfo),
            }
            : undefined,
        page: details === 'txs' ? getPagination(pageSize, transactions) : undefined,
    };
};
exports.default = getAccountInfo;
//# sourceMappingURL=getAccountInfo.js.map