"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.transformAccountInfo = exports.transformTransaction = exports.filterTokenTransfers = exports.transformInputOutput = exports.transformTokenInfo = exports.transformToken = exports.parseAsset = exports.transformUtxos = void 0;
const bigNumber_1 = require("@trezor/utils/lib/bigNumber");
const utils_1 = require("./utils");
const transformUtxos = (utxos) => {
    const result = [];
    utxos.forEach(utxo => utxo.utxoData.amount.forEach(u => {
        result.push({
            address: utxo.address,
            txid: utxo.utxoData.tx_hash,
            confirmations: utxo.blockInfo.confirmations,
            blockHeight: utxo.blockInfo.height || 0,
            amount: u.quantity,
            vout: utxo.utxoData.output_index,
            path: utxo.path,
            cardanoSpecific: {
                unit: u.unit,
            },
        });
    }));
    return result;
};
exports.transformUtxos = transformUtxos;
const hexToString = (input) => {
    let str = '';
    for (let n = 0; n < input.length; n += 2) {
        str += String.fromCharCode(parseInt(input.substring(n, n + 2), 16));
    }
    return str;
};
const getSubtype = (tx) => {
    const withdrawal = tx.txData.withdrawal_count > 0;
    if (withdrawal) {
        return 'withdrawal';
    }
    const registrations = tx.txData.stake_cert_count;
    const delegations = tx.txData.delegation_count;
    if (registrations === 0 && delegations === 0)
        return;
    if (registrations > 0) {
        if (new bigNumber_1.BigNumber(tx.txData.deposit).gt(0)) {
            return 'stake_registration';
        }
        return 'stake_deregistration';
    }
    if (delegations > 0) {
        return 'stake_delegation';
    }
};
const parseAsset = (hex) => {
    const policyIdSize = 56;
    const policyId = hex.slice(0, policyIdSize);
    const assetNameInHex = hex.slice(policyIdSize);
    const assetName = hexToString(assetNameInHex);
    return {
        policyId,
        assetName,
    };
};
exports.parseAsset = parseAsset;
const transformToken = (token) => {
    const { policyId, assetName } = (0, exports.parseAsset)(token.unit);
    const symbol = token.ticker || assetName || (0, utils_1.formatTokenSymbol)(token.fingerprint);
    return {
        name: token.name || symbol,
        contract: token.unit,
        symbol,
        decimals: token.decimals,
        fingerprint: token.fingerprint,
        policyId,
    };
};
exports.transformToken = transformToken;
const transformTokenInfo = (tokens) => {
    if (!tokens || !Array.isArray(tokens)) {
        return undefined;
    }
    const info = tokens.map(token => ({
        type: 'BLOCKFROST',
        balance: token.quantity,
        standard: 'BLOCKFROST',
        ...(0, exports.transformToken)(token),
    }));
    return info.length > 0 ? info : undefined;
};
exports.transformTokenInfo = transformTokenInfo;
const transformInputOutput = (data, asset = 'lovelace') => data.map(utxo => ({
    n: utxo.output_index,
    addresses: [utxo.address],
    isAddress: true,
    value: utxo.amount.find(a => a.unit === asset)?.quantity ?? '0',
}));
exports.transformInputOutput = transformInputOutput;
const filterTokenTransfers = (accountAddress, tx, type) => {
    const transfers = [];
    const myNonChangeAddresses = accountAddress.used.concat(accountAddress.unused);
    const myAddresses = accountAddress.change.concat(myNonChangeAddresses);
    tx.txUtxos.outputs.forEach(output => {
        output.amount
            .filter(a => a.unit !== 'lovelace')
            .forEach(asset => {
            const tokenUnit = asset.unit;
            const inputs = (0, exports.transformInputOutput)(tx.txUtxos.inputs, tokenUnit);
            const outputs = (0, exports.transformInputOutput)(tx.txUtxos.outputs, tokenUnit);
            const outgoing = (0, utils_1.filterTargets)(myAddresses, inputs);
            const incoming = (0, utils_1.filterTargets)(myAddresses, outputs);
            const isChange = accountAddress.change.find(a => a.address === output.address);
            if (incoming.length === 0 && outgoing.length === 0)
                return null;
            const incomingForOutput = (0, utils_1.filterTargets)(myNonChangeAddresses, (0, exports.transformInputOutput)([output], tokenUnit));
            let amount = '0';
            if (type === 'sent') {
                amount = isChange ? '0' : asset.quantity;
            }
            else if (type === 'recv') {
                amount = incomingForOutput.reduce(utils_1.sumVinVout, 0).toString();
            }
            else if (type === 'self' && !isChange) {
                amount = incomingForOutput.reduce(utils_1.sumVinVout, 0).toString();
            }
            if (amount === '0' || !asset.fingerprint)
                return null;
            transfers.push({
                ...(0, exports.transformToken)(asset),
                type,
                amount: amount.toString(),
                from: type === 'sent' || type === 'self'
                    ? tx.address
                    : tx.txUtxos.inputs.find(i => i.amount.find(a => a.unit === tokenUnit))
                        ?.address || '',
                to: type === 'recv' ? tx.address : output.address,
                standard: 'BLOCKFROST',
            });
        });
    });
    return transfers.filter(t => !!t);
};
exports.filterTokenTransfers = filterTokenTransfers;
const transformTransaction = (blockfrostTxData, addressesOrDescriptor) => {
    const fullData = ((data) => 'txUtxos' in data)(blockfrostTxData);
    const [accountAddress, descriptor] = typeof addressesOrDescriptor === 'object'
        ? [addressesOrDescriptor, undefined]
        : [undefined, addressesOrDescriptor];
    const myAddresses = accountAddress
        ? accountAddress.change
            .concat(accountAddress.used, accountAddress.unused)
            .map(a => a.address)
        : (descriptor && [descriptor]) || [];
    let type;
    let targets = [];
    let amount = blockfrostTxData.txData.output_amount.find(b => b.unit === 'lovelace')?.quantity || '0';
    const fee = blockfrostTxData.txData.fees;
    let withdrawal;
    let deposit;
    const inputs = fullData ? (0, exports.transformInputOutput)(blockfrostTxData.txUtxos.inputs) : [];
    const outputs = fullData ? (0, exports.transformInputOutput)(blockfrostTxData.txUtxos.outputs) : [];
    const vinLength = Array.isArray(inputs) ? inputs.length : 0;
    const voutLength = Array.isArray(outputs) ? outputs.length : 0;
    const outgoing = (0, utils_1.filterTargets)(myAddresses, inputs);
    const incoming = (0, utils_1.filterTargets)(myAddresses, outputs);
    const internal = accountAddress ? (0, utils_1.filterTargets)(accountAddress.change, outputs) : [];
    const totalInput = inputs.reduce(utils_1.sumVinVout, 0);
    const totalOutput = outputs.reduce(utils_1.sumVinVout, 0);
    if (outgoing.length === 0 && incoming.length === 0) {
        type = 'unknown';
    }
    else if (vinLength > 0 &&
        voutLength > 0 &&
        outgoing.length === vinLength &&
        incoming.length === voutLength) {
        type = 'self';
        targets = outputs.filter(o => internal.indexOf(o) < 0);
        amount = blockfrostTxData.txData.fees;
        const depositBn = new bigNumber_1.BigNumber(blockfrostTxData.txData.deposit || 0);
        if (!depositBn.isZero()) {
            deposit = depositBn.abs().toString();
        }
        if (blockfrostTxData.txData.withdrawal_count > 0) {
            const extra = new bigNumber_1.BigNumber(totalOutput)
                .plus(blockfrostTxData.txData.fees || 0)
                .minus(totalInput);
            const withdrawalBn = depositBn.isNegative() ? extra.minus(depositBn) : extra;
            withdrawal = withdrawalBn.abs().toString();
        }
    }
    else if (outgoing.length === 0 && incoming.length > 0) {
        type = 'recv';
        amount = '0';
        if (incoming.length > 0) {
            targets = incoming;
            amount = incoming.reduce(utils_1.sumVinVout, 0);
        }
    }
    else {
        type = 'sent';
        targets = outputs.filter(o => internal.indexOf(o) < 0);
        if (voutLength) {
            const myInputsSum = outgoing.reduce(utils_1.sumVinVout, 0);
            const totalSpent = incoming.reduce(utils_1.sumVinVout, 0);
            amount = new bigNumber_1.BigNumber(myInputsSum).minus(totalSpent).minus(fee ?? '0');
        }
    }
    const tokens = accountAddress && fullData
        ? (0, exports.filterTokenTransfers)(accountAddress, blockfrostTxData, type)
        : [];
    return {
        type,
        txid: blockfrostTxData.txData.hash,
        blockTime: blockfrostTxData.txData.block_time,
        blockHeight: blockfrostTxData.txData.block_height || undefined,
        blockHash: blockfrostTxData.txData.block,
        amount: amount?.toString(),
        fee,
        targets: targets.map(t => (0, utils_1.transformTarget)(t, incoming)),
        tokens,
        internalTransfers: [],
        cardanoSpecific: {
            subtype: getSubtype(blockfrostTxData),
            withdrawal,
            deposit,
        },
        feeRate: undefined,
        details: {
            vin: inputs.map((0, utils_1.enhanceVinVout)(myAddresses)),
            vout: outputs.map((0, utils_1.enhanceVinVout)(myAddresses)),
            size: blockfrostTxData.txData.size,
            totalInput: totalInput.toString(),
            totalOutput: totalOutput.toString(),
        },
    };
};
exports.transformTransaction = transformTransaction;
const transformAccountInfo = (info) => {
    const blockfrostTxs = info.history.transactions;
    const result = {
        ...info,
        tokens: (0, exports.transformTokenInfo)(info.tokens),
        history: {
            ...info.history,
            transactions: !blockfrostTxs
                ? []
                : blockfrostTxs?.map(tx => (0, exports.transformTransaction)(tx, info.addresses ?? info.descriptor)),
        },
    };
    return result;
};
exports.transformAccountInfo = transformAccountInfo;
//# sourceMappingURL=blockfrost.js.map