"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.signP2SHTransaction = void 0;
const getTrustedInput_1 = require("./getTrustedInput");
const startUntrustedHashTransactionInput_1 = require("./startUntrustedHashTransactionInput");
const getTrustedInputBIP143_1 = require("./getTrustedInputBIP143");
const signTransaction_1 = require("./signTransaction");
const finalizeInput_1 = require("./finalizeInput");
const constants_1 = require("./constants");
const defaultArg = {
    lockTime: constants_1.DEFAULT_LOCKTIME,
    sigHashType: constants_1.SIGHASH_ALL,
    segwit: false,
    transactionVersion: constants_1.DEFAULT_VERSION,
};
async function signP2SHTransaction(transport, arg) {
    const { inputs, associatedKeysets, outputScriptHex, lockTime, sigHashType, segwit, transactionVersion, } = { ...defaultArg, ...arg };
    // Inputs are provided as arrays of [transaction, output_index, redeem script, optional sequence]
    // associatedKeysets are provided as arrays of [path]
    const nullScript = Buffer.alloc(0);
    const nullPrevout = Buffer.alloc(0);
    const defaultVersion = Buffer.alloc(4);
    defaultVersion.writeUInt32LE(transactionVersion, 0);
    const trustedInputs = [];
    const regularOutputs = [];
    const signatures = [];
    let firstRun = true;
    const resuming = false;
    const targetTransaction = {
        inputs: [],
        version: defaultVersion,
    };
    const getTrustedInputCall = segwit ? getTrustedInputBIP143_1.getTrustedInputBIP143 : getTrustedInput_1.getTrustedInput;
    const outputScript = Buffer.from(outputScriptHex, "hex");
    for (const input of inputs) {
        if (!resuming) {
            const trustedInput = await getTrustedInputCall(transport, input[1], input[0]);
            const sequence = Buffer.alloc(4);
            sequence.writeUInt32LE(input.length >= 4 && typeof input[3] === "number" ? input[3] : constants_1.DEFAULT_SEQUENCE, 0);
            trustedInputs.push({
                trustedInput: false,
                value: segwit
                    ? Buffer.from(trustedInput, "hex")
                    : Buffer.from(trustedInput, "hex").slice(4, 4 + 0x24),
                sequence,
            });
        }
        const { outputs } = input[0];
        const index = input[1];
        if (outputs && index <= outputs.length - 1) {
            regularOutputs.push(outputs[index]);
        }
    }
    // Pre-build the target transaction
    for (let i = 0; i < inputs.length; i++) {
        const sequence = Buffer.alloc(4);
        sequence.writeUInt32LE(inputs[i].length >= 4 && typeof inputs[i][3] === "number"
            ? inputs[i][3]
            : constants_1.DEFAULT_SEQUENCE, 0);
        targetTransaction.inputs.push({
            script: nullScript,
            prevout: nullPrevout,
            sequence,
        });
    }
    if (segwit) {
        await (0, startUntrustedHashTransactionInput_1.startUntrustedHashTransactionInput)(transport, true, targetTransaction, trustedInputs, true);
        await (0, finalizeInput_1.hashOutputFull)(transport, outputScript);
    }
    for (let i = 0; i < inputs.length; i++) {
        const input = inputs[i];
        const script = inputs[i].length >= 3 && typeof input[2] === "string"
            ? Buffer.from(input[2], "hex")
            : regularOutputs[i].script;
        const pseudoTX = Object.assign({}, targetTransaction);
        const pseudoTrustedInputs = segwit ? [trustedInputs[i]] : trustedInputs;
        if (segwit) {
            pseudoTX.inputs = [{ ...pseudoTX.inputs[i], script }];
        }
        else {
            pseudoTX.inputs[i].script = script;
        }
        await (0, startUntrustedHashTransactionInput_1.startUntrustedHashTransactionInput)(transport, !segwit && firstRun, pseudoTX, pseudoTrustedInputs, segwit);
        if (!segwit) {
            await (0, finalizeInput_1.hashOutputFull)(transport, outputScript);
        }
        const signature = await (0, signTransaction_1.signTransaction)(transport, associatedKeysets[i], lockTime, sigHashType);
        signatures.push(segwit ? signature.toString("hex") : signature.slice(0, signature.length - 1).toString("hex"));
        targetTransaction.inputs[i].script = nullScript;
        if (firstRun) {
            firstRun = false;
        }
    }
    return signatures;
}
exports.signP2SHTransaction = signP2SHTransaction;
//# sourceMappingURL=signP2SHTransaction.js.map