"use strict";
/**
 * @file bip32.ts
 * @description BIP32 Path Handling for Bitcoin Wallets
 *
 * This file provides utility functions to handle BIP32 paths,
 * which are commonly used in hierarchical deterministic (HD) wallets.
 * It includes functions to convert BIP32 paths to and from different formats,
 * extract components from extended public keys (xpubs), and manipulate path elements.
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.hardenedPathOf = exports.getXpubComponents = exports.pubkeyFromXpub = exports.pathStringToArray = exports.pathArrayToString = exports.bip32asBuffer = exports.pathElementsToBuffer = void 0;
const bip32_path_1 = __importDefault(require("bip32-path"));
const bs58check_1 = __importDefault(require("bs58check"));
function pathElementsToBuffer(paths) {
    const buffer = Buffer.alloc(1 + paths.length * 4);
    buffer[0] = paths.length;
    paths.forEach((element, index) => {
        buffer.writeUInt32BE(element, 1 + 4 * index);
    });
    return buffer;
}
exports.pathElementsToBuffer = pathElementsToBuffer;
function bip32asBuffer(path) {
    const pathElements = !path ? [] : pathStringToArray(path);
    return pathElementsToBuffer(pathElements);
}
exports.bip32asBuffer = bip32asBuffer;
function pathArrayToString(pathElements) {
    // Limitation: bippath can't handle and empty path. It shouldn't affect us
    // right now, but might in the future.
    // TODO: Fix support for empty path.
    return bip32_path_1.default.fromPathArray(pathElements).toString();
}
exports.pathArrayToString = pathArrayToString;
function pathStringToArray(path) {
    return bip32_path_1.default.fromString(path).toPathArray();
}
exports.pathStringToArray = pathStringToArray;
function pubkeyFromXpub(xpub) {
    const xpubBuf = bs58check_1.default.decode(xpub);
    return xpubBuf.slice(xpubBuf.length - 33);
}
exports.pubkeyFromXpub = pubkeyFromXpub;
function getXpubComponents(xpub) {
    const xpubBuf = bs58check_1.default.decode(xpub);
    return {
        chaincode: xpubBuf.slice(13, 13 + 32),
        pubkey: xpubBuf.slice(xpubBuf.length - 33),
        version: xpubBuf.readUInt32BE(0),
    };
}
exports.getXpubComponents = getXpubComponents;
function hardenedPathOf(pathElements) {
    for (let i = pathElements.length - 1; i >= 0; i--) {
        if (pathElements[i] >= 0x80000000) {
            return pathElements.slice(0, i + 1);
        }
    }
    return [];
}
exports.hardenedPathOf = hardenedPathOf;
//# sourceMappingURL=bip32.js.map