import type { CreateTransactionArg } from "./createTransaction";
import type { AddressFormat } from "./getWalletPublicKey";
import { AppClient as Client } from "./newops/appClient";
/**
 * @class BtcNew
 * @description This class implements the same interface as BtcOld (formerly
 * named Btc), but interacts with Bitcoin hardware app version 2.1.0+
 * which uses a totally new APDU protocol. This new
 * protocol is documented at
 * https://github.com/LedgerHQ/app-bitcoin-new/blob/master/doc/bitcoin.md
 *
 * Since the interface must remain compatible with BtcOld, the methods
 * of this class are quite clunky, because it needs to adapt legacy
 * input data into the PSBT process. In the future, a new interface should
 * be developed that exposes PSBT to the outer world, which would render
 * a much cleaner implementation.
 *
 */
export default class BtcNew {
    private client;
    constructor(client: Client);
    /**
     * This is a new method that allow users to get an xpub at a standard path.
     * Standard paths are described at
     * https://github.com/LedgerHQ/app-bitcoin-new/blob/master/doc/bitcoin.md#description
     *
     * This boils down to paths (N=0 for Bitcoin, N=1 for Testnet):
     * M/44'/N'/x'/**
     * M/48'/N'/x'/y'/**
     * M/49'/N'/x'/**
     * M/84'/N'/x'/**
     * M/86'/N'/x'/**
     *
     * The method was added because of added security in the hardware app v2+. The
     * new hardware app will allow export of any xpub up to and including the
     * deepest hardened key of standard derivation paths, whereas the old app
     * would allow export of any key.
     *
     * This caused an issue for callers of this class, who only had
     * getWalletPublicKey() to call which means they have to constuct xpub
     * themselves:
     *
     * Suppose a user of this class wants to create an account xpub on a standard
     * path, M/44'/0'/Z'. The user must get the parent key fingerprint (see BIP32)
     * by requesting the parent key M/44'/0'. The new app won't allow that, because
     * it only allows exporting deepest level hardened path. So the options are to
     * allow requesting M/44'/0' from the app, or to add a new function
     * "getWalletXpub".
     *
     * We opted for adding a new function, which can greatly simplify client code.
     */
    getWalletXpub({ path, xpubVersion, }: {
        path: string;
        xpubVersion: number;
    }): Promise<string>;
    /**
     * This method returns a public key, a bitcoin address, and and a chaincode
     * for a specific derivation path.
     *
     * Limitation: If the path is not a leaf node of a standard path, the address
     * will be the empty string "", see this.getWalletAddress() for details.
     */
    getWalletPublicKey(path: string, opts?: {
        verify?: boolean;
        format?: AddressFormat;
    }): Promise<{
        publicKey: string;
        bitcoinAddress: string;
        chainCode: string;
    }>;
    /**
     * Get an address for the specified path.
     *
     * If display is true, we must get the address from the device, which would require
     * us to determine WalletPolicy. This requires two *extra* queries to the device, one
     * for the account xpub and one for master key fingerprint.
     *
     * If display is false we *could* generate the address ourselves, but chose to
     * get it from the device to save development time. However, it shouldn't take
     * too much time to implement local address generation.
     *
     * Moreover, if the path is not for a leaf, ie accountPath+/X/Y, there is no
     * way to get the address from the device. In this case we have to create it
     * ourselves, but we don't at this time, and instead return an empty ("") address.
     */
    private getWalletAddress;
    /**
     * Build and sign a transaction. See Btc.createPaymentTransaction for
     * details on how to use this method.
     *
     * This method will convert the legacy arguments, CreateTransactionArg, into
     * a psbt which is finally signed and finalized, and the extracted fully signed
     * transaction is returned.
     */
    createPaymentTransaction(arg: CreateTransactionArg): Promise<string>;
    /**
     * Signs an arbitrary hex-formatted message with the private key at
     * the provided derivation path according to the Bitcoin Signature format
     * and returns v, r, s.
     */
    signMessage({ path, messageHex }: {
        path: string;
        messageHex: string;
    }): Promise<{
        v: number;
        r: string;
        s: string;
    }>;
    /**
     * Calculates an output script along with public key and possible redeemScript
     * from a path and accountType. The accountPath must be a prefix of path.
     *
     * @returns an object with output script (property "script"), redeemScript (if
     * wrapped p2wpkh), and pubkey at provided path. The values of these three
     * properties depend on the accountType used.
     */
    private outputScriptAt;
    /**
     * Adds relevant data about an input to the psbt. This includes sequence,
     * previous txid, output index, spent UTXO, redeem script for wrapped p2wpkh,
     * public key and its derivation path.
     */
    private setInput;
    /**
     * This implements the "Signer" role of the BIP370 transaction signing
     * process.
     *
     * It ssks the hardware device to sign the a psbt using the specified wallet
     * policy. This method assumes BIP32 derived keys are used for all inputs, see
     * comment in-line. The signatures returned from the hardware device is added
     * to the appropriate input fields of the PSBT.
     */
    private signPsbt;
}
export declare function isPathNormal(path: string): boolean;
//# sourceMappingURL=BtcNew.d.ts.map