type Marker = {
    name: string;
    time: number;
    duration: number;
};

/**
 * Represents the different types of events that can be dispatched.
 */
type EventType = 'complete' | 'frame' | 'load' | 'loadError' | 'renderError' | 'loop' | 'pause' | 'play' | 'stop' | 'destroy' | 'freeze' | 'unfreeze' | 'render' | 'ready' | 'stateMachineStart' | 'stateMachineStop' | 'stateMachineTransition' | 'stateMachineStateEntered' | 'stateMachineStateExit' | 'stateMachineCustomEvent' | 'stateMachineError' | 'stateMachineBooleanInputValueChange' | 'stateMachineNumericInputValueChange' | 'stateMachineStringInputValueChange' | 'stateMachineInputFired' | 'stateMachineInternalMessage';
/**
 * Maps an event type string to its respective event interface.
 */
type EventByType<T extends EventType> = Extract<Event, {
    type: T;
}>;
/**
 * Base interface for all events.
 */
interface BaseEvent {
    type: EventType;
}
interface RenderEvent extends BaseEvent {
    currentFrame: number;
    type: 'render';
}
interface FreezeEvent extends BaseEvent {
    type: 'freeze';
}
interface UnfreezeEvent extends BaseEvent {
    type: 'unfreeze';
}
interface DestroyEvent extends BaseEvent {
    type: 'destroy';
}
/**
 * Event fired when a loop action occurs.
 */
interface LoopEvent extends BaseEvent {
    loopCount: number;
    type: 'loop';
}
/**
 * Event fired during frame changes.
 */
interface FrameEvent extends BaseEvent {
    currentFrame: number;
    type: 'frame';
}
/**
 * Event fired when a load action occurs.
 */
interface LoadEvent extends BaseEvent {
    type: 'load';
}
/**
 * Event fired when a loading error occurs.
 */
interface LoadErrorEvent extends BaseEvent {
    error: Error;
    type: 'loadError';
}
/**
 * Event fired when a loading error occurs.
 */
interface RenderErrorEvent extends BaseEvent {
    error: Error;
    type: 'renderError';
}
/**
 * Event fired when a completion action occurs.
 */
interface CompleteEvent extends BaseEvent {
    type: 'complete';
}
/**
 * Event fired when a pause action occurs.
 */
interface PauseEvent extends BaseEvent {
    type: 'pause';
}
/**
 * Event fired when a play action occurs.
 */
interface PlayEvent extends BaseEvent {
    type: 'play';
}
/**
 * Event fired when a stop action occurs.
 */
interface StopEvent extends BaseEvent {
    type: 'stop';
}
/**
 * Event fired when a WASM module is initialized and ready.
 */
interface ReadyEvent extends BaseEvent {
    type: 'ready';
}
interface StateMachineStartEvent extends BaseEvent {
    type: 'stateMachineStart';
}
interface StateMachineStopEvent extends BaseEvent {
    type: 'stateMachineStop';
}
interface StateMachineTransitionEvent extends BaseEvent {
    fromState: string;
    toState: string;
    type: 'stateMachineTransition';
}
interface StateMachineStateEnteredEvent extends BaseEvent {
    state: string;
    type: 'stateMachineStateEntered';
}
interface StateMachineStateExitEvent extends BaseEvent {
    state: string;
    type: 'stateMachineStateExit';
}
interface StateMachineCustomEvent extends BaseEvent {
    eventName: string;
    type: 'stateMachineCustomEvent';
}
interface StateMachineErrorEvent extends BaseEvent {
    error: string;
    type: 'stateMachineError';
}
interface StateMachineBooleanInputValueChangeEvent extends BaseEvent {
    inputName: string;
    newValue: boolean;
    oldValue: boolean;
    type: 'stateMachineBooleanInputValueChange';
}
interface StateMachineNumericInputValueChangeEvent extends BaseEvent {
    inputName: string;
    newValue: number;
    oldValue: number;
    type: 'stateMachineNumericInputValueChange';
}
interface StateMachineStringInputValueChangeEvent extends BaseEvent {
    inputName: string;
    newValue: string;
    oldValue: string;
    type: 'stateMachineStringInputValueChange';
}
interface StateMachineInputFiredEvent extends BaseEvent {
    inputName: string;
    type: 'stateMachineInputFired';
}
interface StateMachineInternalMessage extends BaseEvent {
    message: string;
    type: 'stateMachineInternalMessage';
}
/**
 * Type representing all possible event types.
 */
type Event = LoopEvent | FrameEvent | LoadEvent | LoadErrorEvent | RenderErrorEvent | CompleteEvent | PauseEvent | PlayEvent | StopEvent | DestroyEvent | FreezeEvent | UnfreezeEvent | RenderEvent | ReadyEvent | StateMachineStartEvent | StateMachineStopEvent | StateMachineTransitionEvent | StateMachineStateEnteredEvent | StateMachineStateExitEvent | StateMachineCustomEvent | StateMachineErrorEvent | StateMachineBooleanInputValueChangeEvent | StateMachineNumericInputValueChangeEvent | StateMachineStringInputValueChangeEvent | StateMachineInputFiredEvent | StateMachineInternalMessage;
interface EventListener<T extends EventType> {
    (event: EventByType<T>): void;
}
/**
 * Manages registration and dispatching of event listeners.
 */
declare class EventManager {
    private readonly _eventListeners;
    addEventListener<T extends EventType>(type: T, listener: EventListener<T>): void;
    removeEventListener<T extends EventType>(type: T, listener?: EventListener<T>): void;
    dispatch<T extends EventType>(event: EventByType<T>): void;
    removeAllEventListeners(): void;
}

type Transform = [number, number, number, number, number, number, number, number, number];
interface RenderConfig {
    autoResize?: boolean;
    devicePixelRatio?: number;
    freezeOnOffscreen?: boolean;
}
type Mode = 'forward' | 'reverse' | 'bounce' | 'reverse-bounce';
type Data = string | ArrayBuffer | Record<string, unknown>;
type Fit = 'contain' | 'cover' | 'fill' | 'none' | 'fit-width' | 'fit-height';
interface Layout {
    align?: [number, number];
    fit?: Fit;
}
interface Config {
    animationId?: string;
    autoplay?: boolean;
    backgroundColor?: string;
    canvas: HTMLCanvasElement | OffscreenCanvas;
    data?: Data;
    layout?: Layout;
    loop?: boolean;
    loopCount?: number;
    marker?: string;
    mode?: Mode;
    renderConfig?: RenderConfig;
    segment?: [number, number];
    speed?: number;
    src?: string;
    stateMachineConfig?: StateMachineConfig;
    stateMachineId?: string;
    themeId?: string;
    useFrameInterpolation?: boolean;
}
interface StateMachineConfig {
    /**
     * Controls whether and which URLs can be opened by a state machine.
     *
     * - requireUserInteraction: When true, URLs open only after an explicit user action
     *   (e.g., click/pointer down) on the animation.
     * - whitelist: List of allowed URL patterns. An empty list blocks all URLs. Use
     *   "*" to allow all URLs. Wildcards are supported in host and path (e.g.,
     *   "*.example.com/*").
     *
     * @example
     * ```typescript
     * // Require user interaction before opening any URL
     * openUrlPolicy: { requireUserInteraction: true, whitelist: ["*"] }
     *
     * // Block all URLs
     * openUrlPolicy: { whitelist: [] }
     *
     * // Allow all URLs
     * openUrlPolicy: { whitelist: ["*"] }
     *
     * // Allow a specific domain only
     * openUrlPolicy: { whitelist: ["https://example.com"] }
     *
     * // Allow subdomains and any path under lottiefiles.com
     * openUrlPolicy: { whitelist: ["*.lottiefiles.com/*"] }
     * ```
     *
     * By default, URLs are denied and require user interaction.
     */
    openUrlPolicy?: {
        requireUserInteraction?: boolean;
        whitelist?: string[];
    };
}
interface Manifest {
    animations: Array<{
        background?: string;
        id: string;
        initialTheme?: string;
        themes?: string[];
    }>;
    generator?: string;
    stateMachines?: Array<{
        id: string;
    }>;
    themes?: Array<{
        id: string;
    }>;
    version?: string;
}

interface RenderSurface {
    height: number;
    width: number;
}
declare class DotLottie {
    private readonly _canvas;
    private _context;
    private readonly _eventManager;
    private _animationFrameId;
    private readonly _frameManager;
    private _dotLottieCore;
    private _stateMachineId;
    private _stateMachineConfig;
    private _isStateMachineRunning;
    private _stateMachineObserverHandle;
    private _stateMachineInternalMessageObserver;
    private _dotLottieObserverHandle;
    private static _wasmModule;
    private _renderConfig;
    private _isFrozen;
    private _backgroundColor;
    private _boundOnClick;
    private _boundOnPointerUp;
    private _boundOnPointerDown;
    private _boundOnPointerMove;
    private _boundOnPointerEnter;
    private _boundOnPointerLeave;
    constructor(config: Omit<Config, 'canvas'> & {
        canvas: HTMLCanvasElement | OffscreenCanvas | RenderSurface;
    });
    private _dispatchError;
    private _fetchData;
    private _loadFromData;
    private _loadFromSrc;
    get buffer(): Uint8Array | null;
    get activeAnimationId(): string | undefined;
    get activeThemeId(): string | undefined;
    get layout(): Layout | undefined;
    get marker(): string | undefined;
    get manifest(): Manifest | null;
    get renderConfig(): RenderConfig;
    get segment(): [number, number] | undefined;
    get loop(): boolean;
    get mode(): Mode;
    get isFrozen(): boolean;
    get isStateMachineRunning(): boolean;
    get backgroundColor(): string;
    get autoplay(): boolean;
    get useFrameInterpolation(): boolean;
    get speed(): number;
    get isReady(): boolean;
    get isLoaded(): boolean;
    get isPlaying(): boolean;
    get isPaused(): boolean;
    get isStopped(): boolean;
    get currentFrame(): number;
    get loopCount(): number;
    get totalFrames(): number;
    get duration(): number;
    get segmentDuration(): number;
    get canvas(): HTMLCanvasElement | OffscreenCanvas | RenderSurface;
    load(config: Omit<Config, 'canvas'>): void;
    private _draw;
    private _stopAnimationLoop;
    private _startAnimationLoop;
    private _animationLoop;
    play(): void;
    pause(): void;
    stop(): void;
    setFrame(frame: number): void;
    setSpeed(speed: number): void;
    setBackgroundColor(color: string): void;
    setLoop(loop: boolean): void;
    setLoopCount(loopCount: number): void;
    setUseFrameInterpolation(useFrameInterpolation: boolean): void;
    addEventListener<T extends EventType>(type: T, listener: EventListener<T>): void;
    removeEventListener<T extends EventType>(type: T, listener?: EventListener<T>): void;
    destroy(): void;
    freeze(): void;
    unfreeze(): void;
    resize(): void;
    setTransform(transform: Transform): boolean;
    getTransform(): Transform | undefined;
    setSegment(startFrame: number, endFrame: number): void;
    setMode(mode: Mode): void;
    setRenderConfig(config: RenderConfig): void;
    loadAnimation(animationId: string): void;
    setMarker(marker: string): void;
    markers(): Marker[];
    setTheme(themeId: string): boolean;
    resetTheme(): boolean;
    setThemeData(themeData: string): boolean;
    setSlots(slots: string): void;
    setLayout(layout: Layout): void;
    setViewport(x: number, y: number, width: number, height: number): boolean;
    static setWasmUrl(url: string): void;
    /**
     * @experimental
     * Start a tween animation between two frame values with custom easing
     * @param frame - Starting frame value
     * @param duration - Duration of the tween in seconds
     * @returns true if tween was started successfully
     */
    tween(frame: number, duration: number): boolean;
    /**
     * @experimental
     * Start a tween animation to a specific marker
     * @param marker - The marker name to tween to
     * @returns true if tween was started successfully
     */
    tweenToMarker(marker: string, duration: number): boolean;
    animationSize(): {
        height: number;
        width: number;
    };
    /**
     * Get the Oriented Bounding Box (OBB) points of a layer by its name
     * @param layerName - The name of the layer
     * @returns An array of 8 numbers representing 4 points (x,y) of the OBB in clockwise order starting from top-left
     *          [x0, y0, x1, y1, x2, y2, x3, y3]
     *
     * @example
     * ```typescript
     * // Draw a polygon around the layer 'Layer 1'
     * dotLottie.addEventListener('render', () => {
     *   const obbPoints = dotLottie.getLayerBoundingBox('Layer 1');
     *
     *   if (obbPoints) {
     *     context.beginPath();
     *     context.moveTo(obbPoints[0], obbPoints[1]); // First point
     *     context.lineTo(obbPoints[2], obbPoints[3]); // Second point
     *     context.lineTo(obbPoints[4], obbPoints[5]); // Third point
     *     context.lineTo(obbPoints[6], obbPoints[7]); // Fourth point
     *     context.closePath();
     *     context.stroke();
     *   }
     * });
     * ```
     */
    getLayerBoundingBox(layerName: string): number[] | undefined;
    static transformThemeToLottieSlots(theme: string, slots: string): string;
    private _setupStateMachineObservers;
    private _cleanupStateMachineObservers;
    /**
     * @experimental
     * Load a state machine by ID
     * @param stateMachineId - The ID of the state machine to load
     * @returns true if the state machine was loaded successfully
     */
    stateMachineLoad(stateMachineId: string): boolean;
    /**
     * @experimental
     * Load a state machine from data string
     * @param stateMachineData - The state machine data as a string
     * @returns true if the state machine was loaded successfully
     */
    stateMachineLoadData(stateMachineData: string): boolean;
    /**
     * @experimental
     * Set the state machine config
     * @param config - The state machine config
     */
    stateMachineSetConfig(config: StateMachineConfig | null): void;
    /**
     * @experimental
     * Start the state machine
     * @returns true if the state machine was started successfully
     */
    stateMachineStart(): boolean;
    /**
     * @experimental
     * Stop the state machine
     * @returns true if the state machine was stopped successfully
     */
    stateMachineStop(): boolean;
    /**
     * @experimental
     * Get the current status of the state machine
     * @returns The current status of the state machine as a string
     */
    stateMachineGetStatus(): string;
    /**
     * @experimental
     * Get the current state of the state machine
     * @returns The current state of the state machine as a string
     */
    stateMachineGetCurrentState(): string;
    /**
     * @experimental
     * Get the active state machine ID
     * @returns The active state machine ID as a string
     */
    stateMachineGetActiveId(): string;
    /**
     * @experimental
     * Override the current state of the state machine
     * @param state - The state to override to
     * @param immediate - Whether to immediately transition to the state
     * @returns true if the state override was successful
     */
    stateMachineOverrideState(state: string, immediate?: boolean): boolean;
    /**
     * @experimental
     * Get a specific state machine by ID
     * @param stateMachineId - The ID of the state machine to get
     * @returns The state machine data as a string
     */
    stateMachineGet(stateMachineId: string): string;
    /**
     * @experimental
     * Get the list of state machine listeners
     * @returns Array of listener names
     */
    stateMachineGetListeners(): string[];
    /**
     * @experimental
     * Set a boolean input value for the state machine
     * @param name - The name of the boolean input
     * @param value - The boolean value to set
     */
    stateMachineSetBooleanInput(name: string, value: boolean): boolean;
    /**
     * @experimental
     * Set a numeric input value for the state machine
     * @param name - The name of the numeric input
     * @param value - The numeric value to set
     */
    stateMachineSetNumericInput(name: string, value: number): boolean;
    /**
     * @experimental
     * Set a string input value for the state machine
     * @param name - The name of the string input
     * @param value - The string value to set
     */
    stateMachineSetStringInput(name: string, value: string): boolean;
    /**
     * @experimental
     * Get a boolean input value from the state machine
     * @param name - The name of the boolean input
     * @returns The boolean value or undefined if not found
     */
    stateMachineGetBooleanInput(name: string): boolean | undefined;
    /**
     * @experimental
     * Get a numeric input value from the state machine
     * @param name - The name of the numeric input
     * @returns The numeric value or undefined if not found
     */
    stateMachineGetNumericInput(name: string): number | undefined;
    /**
     * @experimental
     * Get a string input value from the state machine
     * @param name - The name of the string input
     * @returns The string value or undefined if not found
     */
    stateMachineGetStringInput(name: string): string | undefined;
    /**
     * @experimental
     * Fire an event in the state machine
     * @param name - The name of the event to fire
     */
    stateMachineFireEvent(name: string): void;
    /**
     * @experimental
     * Post a click event to the state machine
     * @param x - The x coordinate of the click
     * @param y - The y coordinate of the click
     * @returns The event result or undefined
     */
    stateMachinePostClickEvent(x: number, y: number): void;
    /**
     * @experimental
     * Post a pointer up event to the state machine
     * @param x - The x coordinate of the pointer
     * @param y - The y coordinate of the pointer
     * @returns The event result or undefined
     */
    stateMachinePostPointerUpEvent(x: number, y: number): void;
    /**
     * @experimental
     * Post a pointer down event to the state machine
     * @param x - The x coordinate of the pointer
     * @param y - The y coordinate of the pointer
     * @returns The event result or undefined
     */
    stateMachinePostPointerDownEvent(x: number, y: number): void;
    /**
     * @experimental
     * Post a pointer move event to the state machine
     * @param x - The x coordinate of the pointer
     * @param y - The y coordinate of the pointer
     * @returns The event result or undefined
     */
    stateMachinePostPointerMoveEvent(x: number, y: number): void;
    /**
     * @experimental
     * Post a pointer enter event to the state machine
     * @param x - The x coordinate of the pointer
     * @param y - The y coordinate of the pointer
     * @returns The event result or undefined
     */
    stateMachinePostPointerEnterEvent(x: number, y: number): void;
    /**
     * @experimental
     * Post a pointer exit event to the state machine
     * @param x - The x coordinate of the pointer
     * @param y - The y coordinate of the pointer
     * @returns The event result or undefined
     */
    stateMachinePostPointerExitEvent(x: number, y: number): void;
    private _onClick;
    private _onPointerUp;
    private _onPointerDown;
    private _onPointerMove;
    private _onPointerEnter;
    private _onPointerLeave;
    private _setupStateMachineListeners;
    private _cleanupStateMachineListeners;
}

interface DotLottieInstanceState {
    activeAnimationId: string | undefined;
    activeThemeId: string | undefined;
    autoplay: boolean;
    backgroundColor: string;
    currentFrame: number;
    duration: number;
    isFrozen: boolean;
    isLoaded: boolean;
    isPaused: boolean;
    isPlaying: boolean;
    isReady: boolean;
    isStopped: boolean;
    layout: Layout | undefined;
    loop: boolean;
    loopCount: number;
    manifest: Manifest | null;
    marker: string | undefined;
    markers: Marker[];
    mode: Mode;
    renderConfig: RenderConfig;
    segment: [number, number] | undefined;
    segmentDuration: number;
    speed: number;
    totalFrames: number;
    useFrameInterpolation: boolean;
}
declare class DotLottieWorker {
    private static readonly _workerManager;
    private readonly _eventManager;
    private readonly _id;
    private readonly _worker;
    private readonly _canvas;
    private _dotLottieInstanceState;
    private static _wasmUrl;
    private _created;
    private _boundOnClick;
    private _boundOnPointerUp;
    private _boundOnPointerDown;
    private _boundOnPointerMove;
    private _boundOnPointerEnter;
    private _boundOnPointerLeave;
    constructor(config: Config & {
        workerId?: string;
    });
    private _handleWorkerEvent;
    private _create;
    get loopCount(): number;
    get isLoaded(): boolean;
    get isPlaying(): boolean;
    get isPaused(): boolean;
    get isStopped(): boolean;
    get currentFrame(): number;
    get isFrozen(): boolean;
    get segmentDuration(): number;
    get totalFrames(): number;
    get segment(): [number, number] | undefined;
    get speed(): number;
    get duration(): number;
    get isReady(): boolean;
    get mode(): Mode;
    get canvas(): HTMLCanvasElement | OffscreenCanvas | null;
    get autoplay(): boolean;
    get backgroundColor(): string;
    get loop(): boolean;
    get useFrameInterpolation(): boolean;
    get renderConfig(): RenderConfig;
    get manifest(): Manifest | null;
    get activeAnimationId(): string | undefined;
    get marker(): string | undefined;
    get activeThemeId(): string | undefined;
    get layout(): Layout | undefined;
    play(): Promise<void>;
    pause(): Promise<void>;
    stop(): Promise<void>;
    setSpeed(speed: number): Promise<void>;
    setMode(mode: Mode): Promise<void>;
    setFrame(frame: number): Promise<void>;
    setSegment(start: number, end: number): Promise<void>;
    setRenderConfig(renderConfig: RenderConfig): Promise<void>;
    setUseFrameInterpolation(useFrameInterpolation: boolean): Promise<void>;
    setTheme(themeId: string): Promise<boolean>;
    load(config: Omit<Config, 'canvas'>): Promise<void>;
    setLoop(loop: boolean): Promise<void>;
    setLoopCount(loopCount: number): Promise<void>;
    resize(): Promise<void>;
    destroy(): Promise<void>;
    freeze(): Promise<void>;
    unfreeze(): Promise<void>;
    setBackgroundColor(backgroundColor: string): Promise<void>;
    loadAnimation(animationId: string): Promise<void>;
    setLayout(layout: Layout): Promise<void>;
    private _updateDotLottieInstanceState;
    markers(): Marker[];
    setMarker(marker: string): Promise<void>;
    setThemeData(themeData: string): Promise<boolean>;
    setViewport(x: number, y: number, width: number, height: number): Promise<boolean>;
    animationSize(): Promise<{
        height: number;
        width: number;
    }>;
    /**
     * @experimental
     * Start a tween animation between two frame values with custom easing
     * @param frame - Starting frame value
     * @param duration - Duration of the tween in seconds
     * @returns true if tween was started successfully
     */
    tween(frame: number, duration: number): Promise<boolean>;
    /**
     * @experimental
     * Start a tween animation to a specific marker
     * @param marker - The marker name to tween to
     * @param duration - Duration of the tween in seconds
     * @returns true if tween was started successfully
     */
    tweenToMarker(marker: string, duration: number): Promise<boolean>;
    setTransform(transform: Transform): Promise<boolean>;
    getTransform(): Promise<Transform | undefined>;
    private _sendMessage;
    addEventListener<T extends EventType>(type: T, listener: EventListener<T>): void;
    removeEventListener<T extends EventType>(type: T, listener?: EventListener<T>): void;
    static setWasmUrl(url: string): void;
    /**
     * @experimental
     * Load a state machine by ID
     * @param stateMachineId - The ID of the state machine to load
     * @returns true if the state machine was loaded successfully
     */
    stateMachineLoad(stateMachineId: string): Promise<boolean>;
    /**
     * @experimental
     * Load a state machine from data string
     * @param stateMachineData - The state machine data as a string
     * @returns true if the state machine was loaded successfully
     */
    stateMachineLoadData(stateMachineData: string): Promise<boolean>;
    /**
     * @experimental
     * Start the state machine
     * @returns true if the state machine was started successfully
     */
    stateMachineStart(): Promise<boolean>;
    /**
     * @experimental
     * Stop the state machine
     * @returns true if the state machine was stopped successfully
     */
    stateMachineStop(): Promise<boolean>;
    /**
     * @experimental
     * Set a numeric input value for the state machine
     * @param name - The name of the numeric input
     * @param value - The numeric value to set
     * @returns true if the input was set successfully
     */
    stateMachineSetNumericInput(name: string, value: number): Promise<boolean>;
    /**
     * @experimental
     * Set a boolean input value for the state machine
     * @param name - The name of the boolean input
     * @param value - The boolean value to set
     * @returns true if the input was set successfully
     */
    stateMachineSetBooleanInput(name: string, value: boolean): Promise<boolean>;
    /**
     * @experimental
     * Set the state machine config
     * @param config - The state machine config
     */
    stateMachineSetConfig(config: StateMachineConfig | null): Promise<void>;
    /**
     * @experimental
     * Set a string input value for the state machine
     * @param name - The name of the string input
     * @param value - The string value to set
     * @returns true if the input was set successfully
     */
    stateMachineSetStringInput(name: string, value: string): Promise<boolean>;
    /**
     * @experimental
     * Get a numeric input value from the state machine
     * @param name - The name of the numeric input
     * @returns The numeric value or undefined if not found
     */
    stateMachineGetNumericInput(name: string): Promise<number | undefined>;
    /**
     * @experimental
     * Get a boolean input value from the state machine
     * @param name - The name of the boolean input
     * @returns The boolean value or undefined if not found
     */
    stateMachineGetBooleanInput(name: string): Promise<boolean | undefined>;
    /**
     * @experimental
     * Get a string input value from the state machine
     * @param name - The name of the string input
     * @returns The string value or undefined if not found
     */
    stateMachineGetStringInput(name: string): Promise<string | undefined>;
    /**
     * @experimental
     * Fire an event in the state machine
     * @param name - The name of the event to fire
     */
    stateMachineFireEvent(name: string): Promise<void>;
    /**
     * @experimental
     * Get the current status of the state machine
     * @returns The current status of the state machine as a string
     */
    stateMachineGetStatus(): Promise<string>;
    /**
     * @experimental
     * Get the current state of the state machine
     * @returns The current state of the state machine as a string
     */
    stateMachineGetCurrentState(): Promise<string>;
    /**
     * @experimental
     * Get the active state machine ID
     * @returns The active state machine ID as a string
     */
    stateMachineGetActiveId(): Promise<string>;
    /**
     * @experimental
     * Override the current state of the state machine
     * @param state - The state to override to
     * @param immediate - Whether to immediately transition to the state
     * @returns true if the state override was successful
     */
    stateMachineOverrideState(state: string, immediate?: boolean): Promise<boolean>;
    /**
     * @experimental
     * Get a specific state machine by ID
     * @param stateMachineId - The ID of the state machine to get
     * @returns The state machine data as a string
     */
    stateMachineGet(stateMachineId: string): Promise<string>;
    /**
     * @experimental
     * Get the list of state machine listeners
     * @returns Array of listener names
     */
    stateMachineGetListeners(): Promise<string[]>;
    /**
     * @experimental
     * Post a click event to the state machine
     * @param x - The x coordinate of the click
     * @param y - The y coordinate of the click
     * @returns The event result or undefined
     */
    stateMachinePostClickEvent(x: number, y: number): Promise<void>;
    /**
     * @experimental
     * Post a pointer up event to the state machine
     * @param x - The x coordinate of the pointer
     * @param y - The y coordinate of the pointer
     * @returns The event result or undefined
     */
    stateMachinePostPointerUpEvent(x: number, y: number): Promise<void>;
    /**
     * @experimental
     * Post a pointer down event to the state machine
     * @param x - The x coordinate of the pointer
     * @param y - The y coordinate of the pointer
     * @returns The event result or undefined
     */
    stateMachinePostPointerDownEvent(x: number, y: number): Promise<void>;
    /**
     * @experimental
     * Post a pointer move event to the state machine
     * @param x - The x coordinate of the pointer
     * @param y - The y coordinate of the pointer
     * @returns The event result or undefined
     */
    stateMachinePostPointerMoveEvent(x: number, y: number): Promise<void>;
    /**
     * @experimental
     * Post a pointer enter event to the state machine
     * @param x - The x coordinate of the pointer
     * @param y - The y coordinate of the pointer
     * @returns The event result or undefined
     */
    stateMachinePostPointerEnterEvent(x: number, y: number): Promise<void>;
    /**
     * @experimental
     * Post a pointer exit event to the state machine
     * @param x - The x coordinate of the pointer
     * @param y - The y coordinate of the pointer
     * @returns The event result or undefined
     */
    stateMachinePostPointerExitEvent(x: number, y: number): Promise<void>;
    private _onClick;
    private _onPointerUp;
    private _onPointerDown;
    private _onPointerMove;
    private _onPointerEnter;
    private _onPointerLeave;
    private _setupStateMachineListeners;
    private _cleanupStateMachineListeners;
}

export { type BaseEvent, type CompleteEvent, type Config, type Data, type DestroyEvent, DotLottie, type DotLottieInstanceState, DotLottieWorker, type Event, type EventListener, EventManager, type EventType, type Fit, type FrameEvent, type FreezeEvent, type Layout, type LoadErrorEvent, type LoadEvent, type LoopEvent, type Manifest, type Mode, type PauseEvent, type PlayEvent, type ReadyEvent, type RenderConfig, type RenderErrorEvent, type RenderEvent, type StateMachineBooleanInputValueChangeEvent, type StateMachineConfig, type StateMachineCustomEvent, type StateMachineErrorEvent, type StateMachineInputFiredEvent, type StateMachineInternalMessage, type StateMachineNumericInputValueChangeEvent, type StateMachineStartEvent, type StateMachineStateEnteredEvent, type StateMachineStateExitEvent, type StateMachineStopEvent, type StateMachineStringInputValueChangeEvent, type StateMachineTransitionEvent, type StopEvent, type Transform, type UnfreezeEvent };
