/// <reference types="node" />
import { Network, Payment, Psbt } from 'bitcoinjs-lib';
import type { PartialSig } from 'bip174/src/lib/interfaces';
import { BIP32API } from 'bip32';
import { ECPairAPI } from 'ecpair';
import type { TinySecp256k1Interface, Preimage, TimeConstraints, ExpansionMap, Expand, ParseKeyExpression } from './types';
/**
 * Builds the functions needed to operate with descriptors using an external elliptic curve (ecc) library.
 * @param {Object} ecc - an object containing elliptic curve operations, such as [tiny-secp256k1](https://github.com/bitcoinjs/tiny-secp256k1) or [@bitcoinerlab/secp256k1](https://github.com/bitcoinerlab/secp256k1).
 */
export declare function DescriptorsFactory(ecc: TinySecp256k1Interface): {
    Descriptor: {
        new ({ expression, index, checksumRequired, allowMiniscriptInP2SH, network, preimages, signersPubKeys }: {
            /**
             * The descriptor string in ASCII format. It may include a "*" to denote an arbitrary index.
             */
            expression: string;
            /**
             * The descriptor's index in the case of a range descriptor (must be an integer >=0).
             */
            index?: number;
            /**
             * A flag indicating whether the descriptor is required to include a checksum.
             * @defaultValue false
             */
            checksumRequired?: boolean;
            /**
             * A flag indicating whether this instance can parse and generate script satisfactions for sh(miniscript) top-level expressions of miniscripts. This is not recommended.
             * @defaultValue false
             */
            allowMiniscriptInP2SH?: boolean;
            /**
             * One of bitcoinjs-lib [`networks`](https://github.com/bitcoinjs/bitcoinjs-lib/blob/master/src/networks.js) (or another one following the same interface).
             * @defaultValue networks.bitcoin
             */
            network?: Network;
            /**
             * An array of preimages. This info is necessary to finalize Psbts.
             * @defaultValue `[]`
             */
            preimages?: Preimage[];
            /**
             * An array of the public keys used for signing the transaction when spending the output associated with this descriptor. This parameter is only used if the descriptor object is being used to finalize a transaction. It is necessary to specify the spending path when working with miniscript-based expressions that have multiple spending paths. Set this parameter to an array containing the public keys involved in the desired spending path. Leave it `undefined` if you only need to generate the `scriptPubKey` or `address` for a descriptor, or if all the public keys involved in the descriptor will sign the transaction. In the latter case, the satisfier will automatically choose the most optimal spending path (if more than one is available).
             */
            signersPubKeys?: Buffer[];
        }): {
            readonly "__#1@#payment": Payment;
            readonly "__#1@#preimages": Preimage[];
            readonly "__#1@#signersPubKeys": Buffer[];
            readonly "__#1@#miniscript"?: string;
            readonly "__#1@#witnessScript"?: Buffer;
            readonly "__#1@#redeemScript"?: Buffer;
            readonly "__#1@#isSegwit"?: boolean;
            readonly "__#1@#expandedExpression"?: string;
            readonly "__#1@#expandedMiniscript"?: string;
            readonly "__#1@#expansionMap"?: ExpansionMap;
            readonly "__#1@#network": Network;
            /**
             * Gets the TimeConstraints (nSequence and nLockTime) of the miniscript
             * descriptor as passed in the constructor, just using the expression,
             * the signersPubKeys and preimages.
             *
             * We just need to know which will be the signatures that will be
             * used (signersPubKeys) but final signatures are not necessary for
             * obtaning nLockTime and nSequence.
             *
             * Remember: nSequence and nLockTime are part of the hash that is signed.
             * Thus, they must not change after computing the signatures.
             * When running getScriptSatisfaction, using the final signatures,
             * satisfyMiniscript verifies that the time constraints did not change.
             */
            "__#1@#getTimeConstraints"(): TimeConstraints | undefined;
            getPayment(): Payment;
            /**
             * Returns the Bitcoin Address
             */
            getAddress(): string;
            getScriptPubKey(): Buffer;
            /**
             * Returns the compiled script satisfaction
             * @param {PartialSig[]} signatures An array of signatures using this format: `interface PartialSig { pubkey: Buffer; signature: Buffer; }`
             * @returns {Buffer}
             */
            getScriptSatisfaction(signatures: PartialSig[]): Buffer;
            getSequence(): number | undefined;
            getLockTime(): number | undefined;
            getWitnessScript(): Buffer | undefined;
            getRedeemScript(): Buffer | undefined;
            getNetwork(): Network;
            isSegwit(): boolean | undefined;
            /**
             * Updates a Psbt where the descriptor describes an utxo.
             * The txHex (nonWitnessUtxo) and vout of the utxo must be passed.
             *
             * updatePsbt adds an input to the psbt and updates the tx locktime if needed.
             * It also adds a new input to the Psbt based on txHex
             * It returns the number of the input that is added.
             * psbt and vout are mandatory. Also pass txHex.
             *
             * The following is not recommended but, alternatively, ONLY for Segwit inputs,
             * you can pass txId and value, instead of txHex.
             * If you do so, it is your responsibility to make sure that `value` is
             * correct to avoid possible fee vulnerability attacks:
             * https://github.com/bitcoinjs/bitcoinjs-lib/issues/1625
             * Note that HW wallets require the full txHex also for Segwit anyways:
             * https://blog.trezor.io/details-of-firmware-updates-for-trezor-one-version-1-9-1-and-trezor-model-t-version-2-3-1-1eba8f60f2dd
             *
             * In doubt, simply pass txHex (and you can skip passing txId and value) and
             * you shall be fine.
             */
            updatePsbt({ psbt, txHex, txId, value, vout }: {
                psbt: Psbt;
                txHex?: string;
                txId?: string;
                value?: number;
                vout: number;
            }): number;
            "__#1@#assertPsbtInput"({ psbt, index }: {
                psbt: Psbt;
                index: number;
            }): void;
            finalizePsbtInput({ index, psbt, validate }: {
                index: number;
                psbt: Psbt;
                validate?: boolean | undefined;
            }): void;
            expand(): {
                expansionMap?: ExpansionMap;
                expandedMiniscript?: string;
                miniscript?: string;
                expandedExpression?: string;
            };
        };
    };
    parseKeyExpression: ParseKeyExpression;
    expand: Expand;
    ECPair: ECPairAPI;
    BIP32: BIP32API;
};
/**
 * The {@link DescriptorsFactory | `DescriptorsFactory`} function internally creates and returns the {@link _Internal_.Descriptor | `Descriptor`} class.
 * This class is specialized for the provided `TinySecp256k1Interface`.
 * Use `DescriptorInstance` to declare instances for this class: `const: DescriptorInstance = new Descriptor();`
 *
 * See the {@link _Internal_.Descriptor | documentation for the internal Descriptor class} for a complete list of available methods.
 */
type DescriptorConstructor = ReturnType<typeof DescriptorsFactory>['Descriptor'];
type DescriptorInstance = InstanceType<DescriptorConstructor>;
export { DescriptorInstance, DescriptorConstructor };
