"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DefaultWalletPolicy = exports.WalletPolicy = void 0;
const bitcoinjs_lib_1 = require("bitcoinjs-lib");
const buffertools_1 = require("./buffertools");
const merkle_1 = require("./merkle");
const WALLET_POLICY_V2 = 2;
/**
 * The Bitcon hardware app uses a descriptors-like thing to describe
 * how to construct output scripts from keys. A "Wallet Policy" consists
 * of a "Descriptor Template" and a list of "keys". A key is basically
 * a serialized BIP32 extended public key with some added derivation path
 * information. This is documented at
 * https://github.com/LedgerHQ/app-bitcoin-new/blob/master/doc/wallet.md
 */
class WalletPolicy {
    /**
     * Creates and instance of a wallet policy.
     * @param name an ascii string, up to 16 bytes long; it must be an empty string for default wallet policies
     * @param descriptorTemplate the wallet policy template
     * @param keys and array of the keys, with the key derivation information
     */
    constructor(name, descriptorTemplate, keys) {
        this.name = name;
        this.descriptorTemplate = descriptorTemplate;
        this.keys = keys;
    }
    /**
     * Returns the unique 32-bytes id of this wallet policy.
     */
    getId() {
        return bitcoinjs_lib_1.crypto.sha256(this.serialize());
    }
    /**
     * Serializes the wallet policy for transmission via the hardware wallet protocol.
     * @returns the serialized wallet policy
     */
    serialize() {
        const keyBuffers = this.keys.map((k) => {
            return Buffer.from(k, 'ascii');
        });
        const m = new merkle_1.Merkle(keyBuffers.map((k) => (0, merkle_1.hashLeaf)(k)));
        const buf = new buffertools_1.BufferWriter();
        buf.writeUInt8(WALLET_POLICY_V2); // wallet version
        // length of wallet name, and wallet name
        buf.writeVarSlice(Buffer.from(this.name, 'ascii'));
        // length of descriptor template
        buf.writeVarInt(this.descriptorTemplate.length);
        // sha256 hash of descriptor template
        buf.writeSlice(bitcoinjs_lib_1.crypto.sha256(Buffer.from(this.descriptorTemplate)));
        // number of keys
        buf.writeVarInt(this.keys.length);
        // root of Merkle tree of keys
        buf.writeSlice(m.getRoot());
        return buf.buffer();
    }
}
exports.WalletPolicy = WalletPolicy;
/**
 * Simplified class to handle default wallet policies that can be used without policy registration.
 */
class DefaultWalletPolicy extends WalletPolicy {
    constructor(descriptorTemplate, key) {
        super('', descriptorTemplate, [key]);
    }
}
exports.DefaultWalletPolicy = DefaultWalletPolicy;
//# sourceMappingURL=data:application/json;base64,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