/// <reference types="node" />
import Transport from '@ledgerhq/hw-transport';
import { WalletPolicy } from './policy';
import { PsbtV2 } from './psbtv2';
/**
 * This class represents a partial signature produced by the app during signing.
 * It always contains the `signature` and the corresponding `pubkey` whose private key
 * was used for signing; in the case of taproot script paths, it also contains the
 * tapleaf hash.
 */
export declare class PartialSignature {
    readonly pubkey: Buffer;
    readonly signature: Buffer;
    readonly tapleafHash?: Buffer;
    constructor(pubkey: Buffer, signature: Buffer, tapleafHash?: Buffer);
}
/**
 * This class encapsulates the APDU protocol documented at
 * https://github.com/LedgerHQ/app-bitcoin-new/blob/master/doc/bitcoin.md
 */
export declare class AppClient {
    readonly transport: Transport;
    constructor(transport: Transport);
    private makeRequest;
    /**
     * Returns an object containing the currently running app's name, version and the device status flags.
     *
     * @returns an object with app name, version and device status flags.
     */
    getAppAndVersion(): Promise<{
        name: string;
        version: string;
        flags: number | Buffer;
    }>;
    /**
     * Requests the BIP-32 extended pubkey to the hardware wallet.
     * If `display` is `false`, only standard paths will be accepted; an error is returned if an unusual path is
     * requested.
     * If `display` is `true`, the requested path is shown on screen for user verification; unusual paths can be
     * requested, and a warning is shown to the user in that case.
     *
     * @param path the requested BIP-32 path as a string
     * @param display `false` to silently retrieve a pubkey for a standard path, `true` to display the path on screen
     * @returns the base58-encoded serialized extended pubkey (xpub)
     */
    getExtendedPubkey(path: string, display?: boolean): Promise<string>;
    /**
     * Registers a `WalletPolicy`, after interactive verification from the user.
     * On success, after user's approval, this function returns the id (which is the same that can be computed with
     * `walletPolicy.getid()`), followed by the 32-byte hmac. The client should store the hmac to use it for future
     * requests to `getWalletAddress` or `signPsbt` using this `WalletPolicy`.
     *
     * @param walletPolicy the `WalletPolicy` to register
     * @returns a pair of two 32-byte arrays: the id of the Wallet Policy, followed by the policy hmac
     */
    registerWallet(walletPolicy: WalletPolicy): Promise<readonly [Buffer, Buffer]>;
    /**
     * Returns the address of `walletPolicy` for the given `change` and `addressIndex`.
     *
     * @param walletPolicy the `WalletPolicy` to use
     * @param walletHMAC the 32-byte hmac returned during wallet registration for a registered policy; otherwise
     * `null` for a standard policy
     * @param change `0` for a normal receive address, `1` for a change address
     * @param addressIndex the address index to retrieve
     * @param display `True` to show the address on screen, `False` to retrieve it silently
     * @returns the address, as an ascii string.
     */
    getWalletAddress(walletPolicy: WalletPolicy, walletHMAC: Buffer | null, change: number, addressIndex: number, display: boolean): Promise<string>;
    /**
     * Signs a psbt using a (standard or registered) `WalletPolicy`. This is an interactive command, as user validation
     * is necessary using the device's secure screen.
     * On success, a map of input indexes and signatures is returned.
     * @param psbt a base64-encoded string, or a psbt in a binary Buffer. Using the `PsbtV2` type is deprecated.
     * @param walletPolicy the `WalletPolicy` to use for signing
     * @param walletHMAC the 32-byte hmac obtained during wallet policy registration, or `null` for a standard policy
     * @param progressCallback optionally, a callback that will be called every time a signature is produced during
     * the signing process. The callback does not receive any argument, but can be used to track progress.
     * @returns an array of of tuples with 2 elements containing:
     *    - the index of the input being signed;
     *    - an instance of PartialSignature
     */
    signPsbt(psbt: PsbtV2 | string | Buffer, walletPolicy: WalletPolicy, walletHMAC: Buffer | null, progressCallback?: () => void): Promise<[number, PartialSignature][]>;
    /**
     * Returns the fingerprint of the master public key, as per BIP-32 standard.
     * @returns the master key fingerprint as a string of 8 hexadecimal digits.
     */
    getMasterFingerprint(): Promise<string>;
    /**
     * Signs a message using the legacy Bitcoin Message Signing standard. The signed message is
     * the double-sha256 hash of the concatenation of:
     * - "\x18Bitcoin Signed Message:\n";
     * - the length of `message`, encoded as a Bitcoin-style variable length integer;
     * - `message`.
     *
     * @param message the serialized message to sign
     * @param path the BIP-32 path of the key used to sign the message
     * @returns base64-encoded signature of the message.
     */
    signMessage(message: Buffer, path: string): Promise<string>;
    private validatePolicy;
}
export default AppClient;
