"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.envUtils = exports.getFirmwareReleaseJwsPublicKey = exports.getJWSPublicKey = exports.getEnvironment = exports.isNative = exports.isDesktop = exports.isWeb = void 0;
const ua_parser_js_1 = require("ua-parser-js");
const jws_1 = require("./jws");
const isWeb = () => process.env.SUITE_TYPE === 'web';
exports.isWeb = isWeb;
const isDesktop = () => process.env.SUITE_TYPE === 'desktop';
exports.isDesktop = isDesktop;
const isNative = () => false;
exports.isNative = isNative;
const getEnvironment = () => {
    if ((0, exports.isWeb)())
        return 'web';
    return 'desktop';
};
exports.getEnvironment = getEnvironment;
let userAgentParser;
const getUserAgent = () => window.navigator.userAgent;
const getUserAgentParser = () => {
    if (!userAgentParser) {
        const ua = getUserAgent();
        userAgentParser = new ua_parser_js_1.UAParser(ua);
    }
    return userAgentParser;
};
const isAndroid = () => /Android/.test(getUserAgent());
const isChromeOs = () => /CrOS/.test(getUserAgent());
const getBrowserVersion = () => getUserAgentParser().getBrowser().version || '';
const getCommitHash = () => process.env.COMMITHASH || '';
const getOsVersion = async () => {
    const { version } = await getUserAgentParser().getOS().withClientHints();
    return version ?? '';
};
const getCpuArch = async () => {
    const { architecture } = await getUserAgentParser().getCPU().withClientHints();
    return architecture ?? '';
};
const getSuiteVersion = () => process.env.VERSION || '';
const getBrowserName = () => {
    const browserName = getUserAgentParser().getBrowser().name?.replace(' ', '');
    return browserName?.toLowerCase() || '';
};
const isFirefox = () => getBrowserName() === 'firefox';
const getPlatform = () => window.navigator.platform;
const getPlatformLanguages = () => window.navigator.languages;
const getScreenWidth = () => window.screen.width;
const getScreenHeight = () => window.screen.height;
const getWindowWidth = () => window.innerWidth;
const getWindowHeight = () => window.innerHeight;
const getLocationOrigin = () => window.location.origin;
const getLocationHostname = () => window.location.hostname;
const getProcessPlatform = () => (typeof process !== 'undefined' ? process.platform : '');
const isMacOs = () => {
    if (getProcessPlatform() === 'darwin')
        return true;
    if (typeof window === 'undefined')
        return false;
    return getPlatform().toLowerCase().startsWith('mac');
};
const isWindows = () => {
    if (getProcessPlatform() === 'win32')
        return true;
    if (typeof window === 'undefined')
        return false;
    return getPlatform().toLowerCase().startsWith('win');
};
const isIOs = () => ['iPhone', 'iPad', 'iPod'].includes(getPlatform());
const isLinux = () => {
    if (getProcessPlatform() === 'linux')
        return true;
    if (typeof window === 'undefined')
        return false;
    if (isAndroid() || isChromeOs())
        return false;
    return getPlatform().toLowerCase().startsWith('linux');
};
const isCodesignBuild = () => process.env.IS_CODESIGN_BUILD === 'true';
const getOsName = () => {
    if (isWindows())
        return 'windows';
    if (isMacOs())
        return 'macos';
    if (isAndroid())
        return 'android';
    if (isChromeOs())
        return 'chromeos';
    if (isLinux())
        return 'linux';
    if (isIOs())
        return 'ios';
    return '';
};
const getOsNameWeb = () => getUserAgentParser().getOS().name?.replaceAll(' ', '');
const getOsFamily = () => {
    const osName = getUserAgentParser().getOS().name?.toLowerCase().replaceAll(' ', '');
    if (osName === 'windows') {
        return 'Windows';
    }
    if (osName === 'macos') {
        return 'MacOS';
    }
    return 'Linux';
};
const getDeviceType = () => getUserAgentParser().getDevice().type;
const getJWSPublicKey = () => (isCodesignBuild() ? jws_1.publicKey.codesign : jws_1.publicKey.dev);
exports.getJWSPublicKey = getJWSPublicKey;
const getFirmwareReleaseJwsPublicKey = (useCodeSignKey) => useCodeSignKey ? jws_1.firmwareConfigPublicKey.codesign : jws_1.firmwareConfigPublicKey.dev;
exports.getFirmwareReleaseJwsPublicKey = getFirmwareReleaseJwsPublicKey;
exports.envUtils = {
    isWeb: exports.isWeb,
    isDesktop: exports.isDesktop,
    isNative: exports.isNative,
    getEnvironment: exports.getEnvironment,
    getUserAgent,
    isAndroid,
    isChromeOs,
    getOsVersion,
    getCpuArch,
    getBrowserName,
    getBrowserVersion,
    getCommitHash,
    getDeviceType,
    getSuiteVersion,
    isFirefox,
    getPlatform,
    getPlatformLanguages,
    getScreenWidth,
    getScreenHeight,
    getWindowWidth,
    getWindowHeight,
    getLocationOrigin,
    getLocationHostname,
    getProcessPlatform,
    isMacOs,
    isWindows,
    isIOs,
    isLinux,
    isCodesignBuild,
    getOsName,
    getOsNameWeb,
    getOsFamily,
    getJWSPublicKey: exports.getJWSPublicKey,
    getFirmwareReleaseJwsPublicKey: exports.getFirmwareReleaseJwsPublicKey,
};
//# sourceMappingURL=envUtils.js.map