"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const solana_1 = require("@trezor/blockchain-link-utils/lib/solana");
const schema_utils_1 = require("@trezor/schema-utils");
const BlockchainLink_1 = require("../../../backend/BlockchainLink");
const constants_1 = require("../../../constants");
const AbstractMethod_1 = require("../../../core/AbstractMethod");
const coinInfo_1 = require("../../../data/coinInfo");
const solana_2 = require("../../../types/api/solana");
const solanaUtils_1 = require("../solanaUtils");
class SolanaComposeTransaction extends AbstractMethod_1.AbstractMethod {
    init() {
        this.useDevice = false;
        this.useUi = false;
        const { payload } = this;
        (0, schema_utils_1.Assert)(solana_2.SolanaComposeTransaction, payload);
        const coinInfo = (0, coinInfo_1.getCoinInfo)(payload.coin || 'sol');
        if (!coinInfo) {
            throw constants_1.ERRORS.TypedError('Method_UnknownCoin');
        }
        (0, BlockchainLink_1.isBackendSupported)(coinInfo);
        this.params = {
            coinInfo,
            ...payload,
        };
    }
    get info() {
        return 'Compose Solana transaction';
    }
    async run() {
        const backend = await (0, BlockchainLink_1.initBlockchain)(this.params.coinInfo, this.postMessage, this.params.identity);
        const [recipientAccountOwner, recipientTokenAccounts] = this.params.token
            ? await (0, solanaUtils_1.fetchAccountOwnerAndTokenInfoForAddress)(backend, this.params.toAddress, this.params.token.mint, this.params.token.program)
            : [undefined, undefined];
        const tokenTransferTxAndDestinationAddress = this.params.token && this.params.token.accounts
            ? await (0, solanaUtils_1.buildTokenTransferTransaction)(this.params.fromAddress, this.params.toAddress, recipientAccountOwner || solana_1.SYSTEM_PROGRAM_PUBLIC_KEY, this.params.token.mint, this.params.amount || '0', this.params.token.decimals, this.params.token.accounts, recipientTokenAccounts, this.params.blockHash, this.params.lastValidBlockHeight, this.params.priorityFees || solanaUtils_1.dummyPriorityFeesForFeeEstimation, this.params.token.program)
            : undefined;
        if (this.params.token && !tokenTransferTxAndDestinationAddress)
            throw constants_1.ERRORS.TypedError('Method_InvalidParameter', 'Token accounts not found');
        const tx = tokenTransferTxAndDestinationAddress
            ? tokenTransferTxAndDestinationAddress.transaction
            : await (0, solanaUtils_1.buildTransferTransaction)(this.params.fromAddress, this.params.toAddress, this.params.amount, this.params.blockHash, this.params.lastValidBlockHeight, this.params.priorityFees || solanaUtils_1.dummyPriorityFeesForFeeEstimation);
        const isCreatingAccount = this.params.token &&
            recipientTokenAccounts === undefined &&
            (recipientAccountOwner === solana_1.SYSTEM_PROGRAM_PUBLIC_KEY || recipientAccountOwner == null);
        const newAccountProgramName = isCreatingAccount ? this.params.token?.program : undefined;
        return {
            serializedTx: tx.serialize(),
            additionalInfo: {
                newAccountProgramName,
                tokenAccountInfo: tokenTransferTxAndDestinationAddress?.tokenAccountInfo,
            },
        };
    }
}
exports.default = SolanaComposeTransaction;
//# sourceMappingURL=solanaComposeTransaction.js.map