"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.modifyAuxiliaryDataForBackwardsCompatibility = exports.transformAuxiliaryData = void 0;
const schema_utils_1 = require("@trezor/schema-utils");
const cardanoAddressParameters_1 = require("./cardanoAddressParameters");
const constants_1 = require("../../constants");
const cardano_1 = require("../../types/api/cardano");
const pathUtils_1 = require("../../utils/pathUtils");
const MAX_DELEGATION_COUNT = 32;
const transformDelegation = (delegation) => {
    if (delegation.votingPublicKey) {
        console.warn('Please use votePublicKey instead of votingPublicKey.');
        delegation.votePublicKey = delegation.votingPublicKey;
    }
    (0, schema_utils_1.Assert)(cardano_1.CardanoCVoteRegistrationDelegation, delegation);
    return {
        vote_public_key: delegation.votePublicKey,
        weight: delegation.weight,
    };
};
const transformCvoteRegistrationParameters = (cVoteRegistrationParameters) => {
    if (cVoteRegistrationParameters.votingPublicKey) {
        console.warn('Please use votePublicKey instead of votingPublicKey.');
        cVoteRegistrationParameters.votePublicKey = cVoteRegistrationParameters.votingPublicKey;
    }
    if (cVoteRegistrationParameters.rewardAddressParameters) {
        console.warn('Please use paymentAddressParameters instead of rewardAddressParameters.');
        cVoteRegistrationParameters.paymentAddressParameters =
            cVoteRegistrationParameters.rewardAddressParameters;
    }
    (0, schema_utils_1.Assert)(cardano_1.CardanoCVoteRegistrationParameters, cVoteRegistrationParameters);
    const { paymentAddressParameters } = cVoteRegistrationParameters;
    if (paymentAddressParameters) {
        (0, cardanoAddressParameters_1.validateAddressParameters)(paymentAddressParameters);
    }
    const { delegations } = cVoteRegistrationParameters;
    if (delegations && delegations.length > MAX_DELEGATION_COUNT) {
        throw constants_1.ERRORS.TypedError('Method_InvalidParameter', `At most ${MAX_DELEGATION_COUNT} delegations are allowed in a CIP-36 registration`);
    }
    return {
        vote_public_key: cVoteRegistrationParameters.votePublicKey,
        staking_path: (0, pathUtils_1.validatePath)(cVoteRegistrationParameters.stakingPath, 3),
        payment_address_parameters: paymentAddressParameters
            ? (0, cardanoAddressParameters_1.addressParametersToProto)(paymentAddressParameters)
            : undefined,
        nonce: cVoteRegistrationParameters.nonce,
        format: cVoteRegistrationParameters.format,
        delegations: delegations?.map(transformDelegation),
        voting_purpose: cVoteRegistrationParameters.votingPurpose,
        payment_address: cVoteRegistrationParameters.paymentAddress,
    };
};
const transformAuxiliaryData = (auxiliaryData) => {
    (0, schema_utils_1.Assert)(cardano_1.CardanoAuxiliaryData, auxiliaryData);
    let cVoteRegistrationParameters;
    if (auxiliaryData.cVoteRegistrationParameters) {
        cVoteRegistrationParameters = transformCvoteRegistrationParameters(auxiliaryData.cVoteRegistrationParameters);
    }
    return {
        hash: auxiliaryData.hash,
        cvote_registration_parameters: cVoteRegistrationParameters,
    };
};
exports.transformAuxiliaryData = transformAuxiliaryData;
const modifyAuxiliaryDataForBackwardsCompatibility = (auxiliary_data) => {
    const { cvote_registration_parameters } = auxiliary_data;
    if (cvote_registration_parameters?.payment_address_parameters) {
        cvote_registration_parameters.payment_address_parameters =
            (0, cardanoAddressParameters_1.modifyAddressParametersForBackwardsCompatibility)(cvote_registration_parameters.payment_address_parameters);
        return {
            ...auxiliary_data,
            cvote_registration_parameters,
        };
    }
    return auxiliary_data;
};
exports.modifyAuxiliaryDataForBackwardsCompatibility = modifyAuxiliaryDataForBackwardsCompatibility;
//# sourceMappingURL=cardanoAuxiliaryData.js.map