"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.addressParametersFromProto = exports.addressParametersToProto = exports.modifyAddressParametersForBackwardsCompatibility = exports.validateAddressParameters = void 0;
const schema_utils_1 = require("@trezor/schema-utils");
const constants_1 = require("../../constants");
const cardano_1 = require("../../types/api/cardano");
const pathUtils_1 = require("../../utils/pathUtils");
const validateAddressParameters = (addressParameters) => {
    (0, schema_utils_1.Assert)(cardano_1.CardanoAddressParameters, addressParameters);
    if (addressParameters.path) {
        (0, pathUtils_1.validatePath)(addressParameters.path);
    }
    if (addressParameters.stakingPath) {
        (0, pathUtils_1.validatePath)(addressParameters.stakingPath);
    }
};
exports.validateAddressParameters = validateAddressParameters;
const modifyAddressParametersForBackwardsCompatibility = (address_parameters) => {
    if (address_parameters.address_type === constants_1.PROTO.CardanoAddressType.REWARD) {
        let { address_n, address_n_staking } = address_parameters;
        if (address_n.length > 0 && address_n_staking.length > 0) {
            throw constants_1.ERRORS.TypedError('Method_InvalidParameter', `Only stakingPath is allowed for CardanoAddressType.REWARD`);
        }
        if (address_n.length > 0) {
            address_n_staking = address_n;
            address_n = [];
        }
        return {
            ...address_parameters,
            address_n,
            address_n_staking,
        };
    }
    return address_parameters;
};
exports.modifyAddressParametersForBackwardsCompatibility = modifyAddressParametersForBackwardsCompatibility;
const addressParametersToProto = (addressParameters) => {
    let path = [];
    if (addressParameters.path) {
        path = (0, pathUtils_1.validatePath)(addressParameters.path, 3);
    }
    let stakingPath = [];
    if (addressParameters.stakingPath) {
        stakingPath = (0, pathUtils_1.validatePath)(addressParameters.stakingPath, 3);
    }
    let certificatePointer;
    if (addressParameters.certificatePointer) {
        certificatePointer = {
            block_index: addressParameters.certificatePointer.blockIndex,
            tx_index: addressParameters.certificatePointer.txIndex,
            certificate_index: addressParameters.certificatePointer.certificateIndex,
        };
    }
    return {
        address_type: addressParameters.addressType,
        address_n: path,
        address_n_staking: stakingPath,
        staking_key_hash: addressParameters.stakingKeyHash,
        certificate_pointer: certificatePointer,
        script_payment_hash: addressParameters.paymentScriptHash,
        script_staking_hash: addressParameters.stakingScriptHash,
    };
};
exports.addressParametersToProto = addressParametersToProto;
const addressParametersFromProto = (addressParameters) => {
    let certificatePointer;
    if (addressParameters.certificate_pointer) {
        certificatePointer = {
            blockIndex: addressParameters.certificate_pointer.block_index,
            txIndex: addressParameters.certificate_pointer.tx_index,
            certificateIndex: addressParameters.certificate_pointer.certificate_index,
        };
    }
    return {
        addressType: addressParameters.address_type,
        path: addressParameters.address_n,
        stakingPath: addressParameters.address_n_staking,
        stakingKeyHash: addressParameters.staking_key_hash,
        certificatePointer,
    };
};
exports.addressParametersFromProto = addressParametersFromProto;
//# sourceMappingURL=cardanoAddressParameters.js.map