"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const schema_utils_1 = require("@trezor/schema-utils");
const constants_1 = require("../../../constants");
const AbstractMethod_1 = require("../../../core/AbstractMethod");
const coinInfo_1 = require("../../../data/coinInfo");
const events_1 = require("../../../events");
const types_1 = require("../../../types");
const cardano_1 = require("../../../types/api/cardano");
const pathUtils_1 = require("../../../utils/pathUtils");
const paramsValidator_1 = require("../../common/paramsValidator");
const cardanoAddressParameters_1 = require("../cardanoAddressParameters");
class CardanoGetAddress extends AbstractMethod_1.AbstractMethod {
    hasBundle;
    progress = 0;
    init() {
        this.noBackupConfirmationMode = 'always';
        this.requiredPermissions = ['read'];
        this.requiredDeviceCapabilities = ['Capability_Cardano'];
        this.firmwareRange = (0, paramsValidator_1.getFirmwareRange)(this.name, (0, coinInfo_1.getMiscNetwork)('Cardano'), this.firmwareRange);
        this.hasBundle = !!this.payload.bundle;
        const payload = !this.payload.bundle
            ? { ...this.payload, bundle: [this.payload] }
            : this.payload;
        (0, schema_utils_1.Assert)((0, types_1.Bundle)(cardano_1.CardanoGetAddress), payload);
        this.params = payload.bundle.map(batch => {
            (0, cardanoAddressParameters_1.validateAddressParameters)(batch.addressParameters);
            return {
                address_parameters: (0, cardanoAddressParameters_1.addressParametersToProto)(batch.addressParameters),
                address: batch.address,
                protocol_magic: batch.protocolMagic,
                network_id: batch.networkId,
                derivation_type: typeof batch.derivationType !== 'undefined'
                    ? batch.derivationType
                    : constants_1.PROTO.CardanoDerivationType.ICARUS_TREZOR,
                show_display: typeof batch.showOnTrezor === 'boolean' ? batch.showOnTrezor : true,
                chunkify: typeof batch.chunkify === 'boolean' ? batch.chunkify : false,
            };
        });
        const useEventListener = payload.useEventListener &&
            this.params.length === 1 &&
            typeof this.params[0].address === 'string' &&
            this.params[0].show_display;
        this.useUi = !useEventListener;
    }
    get info() {
        if (this.params.length === 1) {
            return `Export Cardano address for account #${(0, pathUtils_1.fromHardened)(this.params[0].address_parameters.address_n[2]) + 1}`;
        }
        return 'Export multiple Cardano addresses';
    }
    getButtonRequestData(code) {
        if (code === 'ButtonRequest_Address') {
            return {
                type: 'address',
                serializedPath: (0, pathUtils_1.getSerializedPath)(this.params[this.progress].address_parameters.address_n),
                address: this.params[this.progress].address || 'not-set',
            };
        }
    }
    get confirmation() {
        return !this.useUi
            ? undefined
            : {
                view: 'export-address',
                label: this.info,
            };
    }
    async _call({ address_parameters, protocol_magic, network_id, derivation_type, show_display, chunkify, }) {
        const cmd = this.device.getCommands();
        const response = await cmd.typedCall('CardanoGetAddress', 'CardanoAddress', {
            address_parameters,
            protocol_magic,
            network_id,
            derivation_type,
            show_display,
            chunkify,
        });
        return response.message;
    }
    async run() {
        const responses = [];
        for (let i = 0; i < this.params.length; i++) {
            const batch = this.params[i];
            batch.address_parameters = (0, cardanoAddressParameters_1.modifyAddressParametersForBackwardsCompatibility)(batch.address_parameters);
            if (batch.show_display) {
                const silent = await this._call({
                    ...batch,
                    show_display: false,
                });
                if (typeof batch.address === 'string') {
                    if (batch.address !== silent.address) {
                        throw constants_1.ERRORS.TypedError('Method_AddressNotMatch');
                    }
                }
                else {
                    batch.address = silent.address;
                }
            }
            const response = await this._call(batch);
            responses.push({
                addressParameters: (0, cardanoAddressParameters_1.addressParametersFromProto)(batch.address_parameters),
                protocolMagic: batch.protocol_magic,
                networkId: batch.network_id,
                serializedPath: (0, pathUtils_1.getSerializedPath)(batch.address_parameters.address_n),
                serializedStakingPath: (0, pathUtils_1.getSerializedPath)(batch.address_parameters.address_n_staking),
                address: response.address,
            });
            if (this.hasBundle) {
                this.postMessage((0, events_1.createUiMessage)(events_1.UI.BUNDLE_PROGRESS, {
                    total: this.params.length,
                    progress: i,
                    response,
                }));
            }
            this.progress++;
        }
        return this.hasBundle ? responses : responses[0];
    }
}
exports.default = CardanoGetAddress;
//# sourceMappingURL=cardanoGetAddress.js.map