"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.outputToTrezor = exports.validateHDOutput = exports.validateTrezorOutputs = void 0;
const constants_1 = require("../../constants");
const addressUtils_1 = require("../../utils/addressUtils");
const hdnodeUtils_1 = require("../../utils/hdnodeUtils");
const pathUtils_1 = require("../../utils/pathUtils");
const paramsValidator_1 = require("../common/paramsValidator");
const validateTrezorOutputs = (outputs, coinInfo) => {
    const trezorOutputs = outputs
        .map(o => (0, pathUtils_1.fixPath)(o))
        .map(o => (0, hdnodeUtils_1.convertMultisigPubKey)(coinInfo.network, o));
    trezorOutputs.forEach(output => {
        (0, paramsValidator_1.validateParams)(output, [
            { name: 'address_n', type: 'array' },
            { name: 'address', type: 'string' },
            { name: 'amount', type: 'uint' },
            { name: 'op_return_data', type: 'string' },
            { name: 'multisig', type: 'object' },
        ]);
        if (Object.prototype.hasOwnProperty.call(output, 'address_n') &&
            Object.prototype.hasOwnProperty.call(output, 'address')) {
            throw constants_1.ERRORS.TypedError('Method_InvalidParameter', 'Cannot use address and address_n in one output');
        }
        if (output.address_n && !output.script_type) {
            output.script_type = (0, pathUtils_1.getOutputScriptType)(output.address_n);
        }
        if ('address' in output &&
            typeof output.address === 'string' &&
            !(0, addressUtils_1.isValidAddress)(output.address, coinInfo)) {
            throw constants_1.ERRORS.TypedError('Method_InvalidParameter', `Invalid ${coinInfo.label} output address ${output.address}`);
        }
    });
    return trezorOutputs;
};
exports.validateTrezorOutputs = validateTrezorOutputs;
const validateHDOutput = (output, coinInfo) => {
    const validateAddress = (address) => {
        if (!address || !(0, addressUtils_1.isValidAddress)(address, coinInfo)) {
            throw constants_1.ERRORS.TypedError('Method_InvalidParameter', `Invalid ${coinInfo.label} output address format`);
        }
    };
    switch (output.type) {
        case 'opreturn':
            (0, paramsValidator_1.validateParams)(output, [{ name: 'dataHex', type: 'string' }]);
            return {
                type: 'opreturn',
                dataHex: output.dataHex,
            };
        case 'send-max':
            (0, paramsValidator_1.validateParams)(output, [{ name: 'address', type: 'string', required: true }]);
            validateAddress(output.address);
            return {
                type: 'send-max',
                address: output.address,
            };
        case 'payment-noaddress':
            (0, paramsValidator_1.validateParams)(output, [{ name: 'amount', type: 'uint', required: true }]);
            return {
                type: 'payment-noaddress',
                amount: output.amount,
            };
        case 'send-max-noaddress':
            return {
                type: 'send-max-noaddress',
            };
        default:
            (0, paramsValidator_1.validateParams)(output, [
                { name: 'amount', type: 'uint', required: true },
                { name: 'address', type: 'string', required: true },
            ]);
            validateAddress(output.address);
            return {
                type: 'payment',
                address: output.address,
                amount: output.amount,
            };
    }
};
exports.validateHDOutput = validateHDOutput;
const outputToTrezor = (output) => {
    if (output.type === 'opreturn') {
        return {
            amount: '0',
            op_return_data: output.dataHex,
            script_type: 'PAYTOOPRETURN',
        };
    }
    if (output.type === 'change') {
        const address_n = (0, pathUtils_1.getHDPath)(output.path);
        return {
            address_n,
            amount: output.amount,
            script_type: (0, pathUtils_1.getOutputScriptType)(address_n),
        };
    }
    return {
        address: output.address,
        amount: output.amount,
        script_type: 'PAYTOADDRESS',
    };
};
exports.outputToTrezor = outputToTrezor;
//# sourceMappingURL=outputs.js.map