"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.default = Blockfrost;
const constants_1 = require("@trezor/blockchain-link-types/lib/constants");
const errors_1 = require("@trezor/blockchain-link-types/lib/constants/errors");
const blockfrost_1 = require("@trezor/blockchain-link-utils/lib/blockfrost");
const baseWorker_1 = require("../baseWorker");
const websocket_1 = require("./websocket");
const getInfo = async (request) => {
    const api = await request.connect();
    const info = await api.getServerInfo();
    return {
        type: constants_1.RESPONSES.GET_INFO,
        payload: {
            url: api.options.url,
            network: info.shortcut,
            ...info,
        },
    };
};
const getBlockHash = async (request) => {
    const api = await request.connect();
    const blockMessage = await api.getBlockHash(request.payload);
    return {
        type: constants_1.RESPONSES.GET_BLOCK_HASH,
        payload: blockMessage.hash,
    };
};
const getAccountBalanceHistory = async (request) => {
    const socket = await request.connect();
    const history = await socket.getAccountBalanceHistory(request.payload);
    return {
        type: constants_1.RESPONSES.GET_ACCOUNT_BALANCE_HISTORY,
        payload: history,
    };
};
const getTransaction = async (request) => {
    const api = await request.connect();
    const txData = await api.getTransaction(request.payload);
    const tx = (0, blockfrost_1.transformTransaction)({ txData });
    return {
        type: constants_1.RESPONSES.GET_TRANSACTION,
        payload: tx,
    };
};
const estimateFee = async (request) => {
    const api = await request.connect();
    const resp = await api.estimateFee(request.payload);
    const feeOptions = [];
    feeOptions.push({ feePerUnit: resp.lovelacePerByte.toString() });
    return {
        type: constants_1.RESPONSES.ESTIMATE_FEE,
        payload: feeOptions,
    };
};
const pushTransaction = async (request) => {
    const api = await request.connect();
    const payload = await api.pushTransaction(request.payload.hex);
    return {
        type: constants_1.RESPONSES.PUSH_TRANSACTION,
        payload,
    };
};
const getAccountInfo = async (request) => {
    const api = await request.connect();
    const { details = 'basic', ...rest } = request.payload;
    const info = await api.getAccountInfo({ details, ...rest });
    return {
        type: constants_1.RESPONSES.GET_ACCOUNT_INFO,
        payload: (0, blockfrost_1.transformAccountInfo)(info),
    };
};
const getAccountUtxo = async (request) => {
    const api = await request.connect();
    const utxos = await api.getAccountUtxo(request.payload);
    return {
        type: constants_1.RESPONSES.GET_ACCOUNT_UTXO,
        payload: (0, blockfrost_1.transformUtxos)(utxos),
    };
};
const onNewBlock = ({ post }, event) => {
    post({
        id: -1,
        type: constants_1.RESPONSES.NOTIFICATION,
        payload: {
            type: 'block',
            payload: {
                blockHeight: event.height || 0,
                blockHash: event.hash,
            },
        },
    });
};
const onTransaction = ({ state, post }, event) => {
    const descriptor = event.address;
    const account = state.getAccount(descriptor);
    post({
        id: -1,
        type: constants_1.RESPONSES.NOTIFICATION,
        payload: {
            type: 'notification',
            payload: {
                descriptor: account ? account.descriptor : descriptor,
                tx: account
                    ? (0, blockfrost_1.transformTransaction)(event, account.addresses ?? account.descriptor)
                    : (0, blockfrost_1.transformTransaction)(event, descriptor),
            },
        },
    });
};
const subscribeBlock = async (ctx) => {
    if (ctx.state.getSubscription('block'))
        return { subscribed: true };
    const api = await ctx.connect();
    ctx.state.addSubscription('block');
    api.on('block', ev => onNewBlock(ctx, ev));
    return api.subscribeBlock();
};
const subscribeAccounts = async (ctx, accounts) => {
    const api = await ctx.connect();
    const { state } = ctx;
    state.addAccounts(accounts);
    if (!state.getSubscription('notification')) {
        api.on('notification', ev => onTransaction(ctx, ev));
        state.addSubscription('notification');
    }
    return api.subscribeAddresses(state.getAddresses());
};
const subscribeAddresses = async (ctx, addresses) => {
    const api = await ctx.connect();
    const { state } = ctx;
    state.addAddresses(addresses);
    if (!state.getSubscription('notification')) {
        api.on('notification', ev => onTransaction(ctx, ev));
        state.addSubscription('notification');
    }
    return api.subscribeAddresses(state.getAddresses());
};
const subscribe = async (request) => {
    const { payload } = request;
    let response;
    if (payload.type === 'accounts') {
        response = await subscribeAccounts(request, payload.accounts);
    }
    else if (payload.type === 'addresses') {
        response = await subscribeAddresses(request, payload.addresses);
    }
    else if (payload.type === 'block') {
        response = await subscribeBlock(request);
    }
    else {
        throw new errors_1.CustomError('invalid_param', '+type');
    }
    return {
        type: constants_1.RESPONSES.SUBSCRIBE,
        payload: response,
    };
};
const unsubscribeBlock = async ({ state, connect }) => {
    if (!state.getSubscription('block'))
        return { subscribed: false };
    const api = await connect();
    api.removeAllListeners('block');
    state.removeSubscription('block');
    return api.unsubscribeBlock();
};
const unsubscribeAccounts = async ({ state, connect }, accounts) => {
    state.removeAccounts(accounts || state.getAccounts());
    const api = await connect();
    const subscribed = state.getAddresses();
    if (subscribed.length < 1) {
        api.removeAllListeners('notification');
        state.removeSubscription('notification');
        return api.unsubscribeAddresses();
    }
    return api.subscribeAddresses(subscribed);
};
const unsubscribeAddresses = async ({ state, connect }, addresses) => {
    const socket = await connect();
    if (!addresses) {
        state.removeAccounts(state.getAccounts());
    }
    const subscribed = state.removeAddresses(addresses || state.getAddresses());
    if (subscribed.length < 1) {
        socket.removeAllListeners('notification');
        state.removeSubscription('notification');
        return socket.unsubscribeAddresses();
    }
    return socket.subscribeAddresses(subscribed);
};
const unsubscribe = async (request) => {
    const { payload } = request;
    let response;
    if (payload.type === 'accounts') {
        response = await unsubscribeAccounts(request, payload.accounts);
    }
    else if (payload.type === 'addresses') {
        response = await unsubscribeAddresses(request, payload.addresses);
    }
    else if (payload.type === 'block') {
        response = await unsubscribeBlock(request);
    }
    else {
        throw new errors_1.CustomError('invalid_param', '+type');
    }
    return {
        type: constants_1.RESPONSES.UNSUBSCRIBE,
        payload: response,
    };
};
const onRequest = (request) => {
    switch (request.type) {
        case constants_1.MESSAGES.GET_INFO:
            return getInfo(request);
        case constants_1.MESSAGES.GET_BLOCK_HASH:
            return getBlockHash(request);
        case constants_1.MESSAGES.GET_ACCOUNT_BALANCE_HISTORY:
            return getAccountBalanceHistory(request);
        case constants_1.MESSAGES.GET_ACCOUNT_INFO:
            return getAccountInfo(request);
        case constants_1.MESSAGES.GET_ACCOUNT_UTXO:
            return getAccountUtxo(request);
        case constants_1.MESSAGES.GET_TRANSACTION:
            return getTransaction(request);
        case constants_1.MESSAGES.ESTIMATE_FEE:
            return estimateFee(request);
        case constants_1.MESSAGES.PUSH_TRANSACTION:
            return pushTransaction(request);
        case constants_1.MESSAGES.SUBSCRIBE:
            return subscribe(request);
        case constants_1.MESSAGES.UNSUBSCRIBE:
            return unsubscribe(request);
        default:
            throw new errors_1.CustomError('worker_unknown_request', `+${request.type}`);
    }
};
class BlockfrostWorker extends baseWorker_1.BaseWorker {
    cleanup() {
        if (this.api) {
            this.api.dispose();
            this.api.removeAllListeners();
        }
        super.cleanup();
    }
    isConnected(api) {
        return api?.isConnected() ?? false;
    }
    async tryConnect(url) {
        const { timeout, pingTimeout, keepAlive } = this.settings;
        const api = new websocket_1.BlockfrostAPI({
            url,
            timeout,
            pingTimeout,
            keepAlive,
            agent: this.proxyAgent,
        });
        await api.connect();
        api.on('disconnected', () => {
            this.post({ id: -1, type: constants_1.RESPONSES.DISCONNECTED, payload: true });
            this.cleanup();
        });
        this.post({
            id: -1,
            type: constants_1.RESPONSES.CONNECTED,
        });
        return api;
    }
    disconnect() {
        if (this.api) {
            this.api.disconnect();
        }
    }
    async messageHandler(event) {
        try {
            if (await super.messageHandler(event))
                return true;
            const request = {
                ...event.data,
                connect: () => this.connect(),
                post: (data) => this.post(data),
                state: this.state,
            };
            const response = await onRequest(request);
            this.post({ id: event.data.id, ...response });
        }
        catch (error) {
            this.errorResponse(event.data.id, error);
        }
    }
}
function Blockfrost() {
    return new BlockfrostWorker();
}
if (baseWorker_1.CONTEXT === 'worker') {
    const module = new BlockfrostWorker();
    onmessage = module.messageHandler.bind(module);
}
//# sourceMappingURL=index.js.map