import type tls from 'tls';
import type { BaseCurrencyCode } from '@trezor/blockchain-link-types';
import type { Transaction as BlockbookTransaction, VinVout } from './blockbook';
import type { AddressAlias, TokenTransfer as BlockbookTokenTransfer, ContractInfo, MultiTokenValue, StakingPool } from './blockbook-api';
import type { SolanaStakingAccount } from './solana';
type AgentOptions = {
    timeout?: number | undefined;
};
interface BaseSocksProxyAgentOptions {
    host?: string | null;
    port?: string | number | null;
    username?: string | null;
    tls?: tls.ConnectionOptions | null;
    ipaddress?: string;
    type: 4 | 5;
    userId?: string;
    password?: string;
}
interface SocksProxyAgentOptions extends AgentOptions, BaseSocksProxyAgentOptions {
}
export interface BlockchainSettings {
    name: string;
    worker: string | (() => any);
    server: string[];
    proxy?: string | SocksProxyAgentOptions;
    debug?: boolean;
    timeout?: number;
    pingTimeout?: number;
    keepAlive?: boolean;
    throttleBlockEvent?: number;
}
export interface ServerInfo {
    url: string;
    name: string;
    shortcut: string;
    testnet: boolean;
    version: string;
    decimals: number;
    blockHeight: number;
    blockHash: string;
    consensusBranchId?: number;
    network: string;
}
export type TokenStandard = 'ERC20' | 'BEP20' | 'BEP721' | 'ERC721' | 'ERC1155' | 'BEP1155' | 'SPL' | 'SPL-2022' | 'BLOCKFROST' | 'STELLAR-CLASSIC';
export type TransferType = 'sent' | 'recv' | 'self' | 'unknown';
export type TokenTransfer = Omit<BlockbookTokenTransfer, 'value' | 'type' | 'standard'> & {
    type: TransferType;
    standard?: TokenStandard;
    amount: string;
};
export interface InternalTransfer {
    type: TransferType | 'external';
    amount: string;
    from: string;
    to: string;
}
export interface Target {
    n: number;
    addresses?: string[];
    isAddress: boolean;
    amount?: string;
    coinbase?: string;
    isAccountTarget?: boolean;
}
export type EnhancedVinVout = VinVout & {
    isAccountOwned?: boolean;
};
export type TransactionDetail = {
    vin: EnhancedVinVout[];
    vout: EnhancedVinVout[];
    size: number;
    totalInput: string;
    totalOutput: string;
};
export type FiatRatesBySymbol = {
    [K in BaseCurrencyCode]?: number | undefined;
};
export interface AccountBalanceHistory {
    time: number;
    txs: number;
    received: string;
    sent: string;
    sentToSelf?: string;
    rates: FiatRatesBySymbol;
}
export interface Transaction {
    type: 'sent' | 'recv' | 'self' | 'joint' | 'contract' | 'failed' | 'unknown';
    txid: string;
    hex?: string;
    blockTime?: number;
    blockHeight?: number;
    blockHash?: string;
    lockTime?: number;
    amount: string;
    fee: string;
    targets: Target[];
    tokens: TokenTransfer[];
    rbf?: boolean;
    ethereumSpecific?: BlockbookTransaction['ethereumSpecific'];
    internalTransfers: InternalTransfer[];
    cardanoSpecific?: {
        subtype?: 'withdrawal' | 'stake_delegation' | 'stake_registration' | 'stake_deregistration';
        withdrawal?: string;
        deposit?: string;
    };
    solanaSpecific?: {
        status: 'confirmed';
        stakeOperation?: {
            type: StakeType;
            amount: string;
        };
    };
    details: TransactionDetail;
    vsize?: number;
    feeRate?: string;
    rippleSpecific?: {
        destinationTag?: number;
    };
    stellarSpecific?: {
        memo?: string;
        feeSource: string;
    };
}
export type AnonymitySet = Record<string, number | undefined>;
export interface Address {
    address: string;
    path: string;
    transfers: number;
    balance: string;
    sent: string;
    received: string;
}
export interface AccountAddresses {
    change: Address[];
    used: Address[];
    unused: Address[];
    anonymitySet?: AnonymitySet;
}
export interface Utxo {
    txid: string;
    vout: number;
    amount: string;
    blockHeight: number;
    address: string;
    path: string;
    confirmations: number;
    coinbase?: boolean;
    cardanoSpecific?: {
        unit: string;
    };
}
export interface TokenAccount {
    publicKey: string;
    balance: string;
}
export interface TokenInfo {
    type: string;
    standard: string;
    contract: string;
    balance?: string;
    name?: string;
    symbol?: string;
    decimals: number;
    accounts?: TokenAccount[];
    policyId?: string;
    fingerprint?: string;
    multiTokenValues?: MultiTokenValue[];
    ids?: string[];
    totalReceived?: string;
    totalSent?: string;
}
export interface AccountInfo {
    descriptor: string;
    balance: string;
    availableBalance: string;
    empty: boolean;
    tokens?: TokenInfo[];
    addresses?: AccountAddresses;
    history: {
        total: number;
        tokens?: number;
        unconfirmed: number;
        transactions?: Transaction[];
        txids?: string[];
        addrTxCount?: number;
    };
    misc?: {
        nonce?: string;
        contractInfo?: ContractInfo;
        stakingPools?: StakingPool[];
        addressAliases?: {
            [key: string]: AddressAlias;
        };
        sequence?: number;
        stellarSequence?: string;
        reserve?: string;
        rewards?: string;
        staking?: {
            address: string;
            isActive: boolean;
            rewards: string;
            poolId: string | null;
            drep: {
                drep_id: string;
                hex: string;
                amount: string;
                active: boolean;
                active_epoch: number | null;
                has_script: boolean;
            } | null;
        };
        owner?: string;
        rent?: number;
        solStakingAccounts?: SolanaStakingAccount[];
        solEpoch?: number;
    };
    page?: {
        index: number;
        size: number;
        total: number;
    };
    marker?: {
        ledger: number;
        seq: number;
    };
    stellarCursor?: string;
}
export interface SubscriptionAccountInfo {
    descriptor: string;
    addresses?: AccountAddresses;
    subscriptionId?: number;
    tokens?: TokenInfo[];
}
export type ChannelMessage<T> = T & {
    id: number;
};
export type StakeType = 'stake' | 'unstake' | 'claim';
export type TokenDetailByMint = {
    [mint: string]: {
        name: string;
        symbol: string;
    };
};
export {};
//# sourceMappingURL=common.d.ts.map