"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.TrezorConnectDynamic = void 0;
const utils_1 = require("@trezor/utils");
const constants_1 = require("../constants");
const events_1 = require("../events");
const proxy_event_emitter_1 = require("../utils/proxy-event-emitter");
class TrezorConnectDynamic {
    eventEmitter;
    currentTarget;
    implementations;
    getInitTarget;
    handleBeforeCall;
    handleErrorFallback;
    lastSettings;
    callPending = 0;
    beforeCallSynchronize = (0, utils_1.getSynchronize)();
    constructor({ implementations, getInitTarget, handleBeforeCall, handleErrorFallback, }) {
        this.implementations = implementations;
        this.currentTarget = this.implementations[0].type;
        this.getInitTarget = getInitTarget;
        this.handleBeforeCall = handleBeforeCall;
        this.handleErrorFallback = handleErrorFallback;
        this.eventEmitter = new proxy_event_emitter_1.ProxyEventEmitter(this.implementations.map(impl => impl.impl.eventEmitter));
    }
    getTarget() {
        return this.implementations.find(impl => impl.type === this.currentTarget).impl;
    }
    getTargetType() {
        return this.currentTarget;
    }
    async switchTarget(target) {
        if (this.currentTarget === target) {
            return;
        }
        if (!this.lastSettings) {
            throw constants_1.ERRORS.TypedError('Init_ManifestMissing');
        }
        const oldTargetType = this.getTargetType();
        const oldTarget = this.getTarget();
        try {
            this.currentTarget = target;
            await this.getTarget().init(this.lastSettings);
            await oldTarget.dispose();
        }
        catch {
            this.currentTarget = oldTargetType;
        }
    }
    manifest(manifest) {
        this.lastSettings = { ...this.lastSettings, manifest };
        this.getTarget().manifest(manifest);
    }
    async init(settings) {
        if (!settings?.manifest) {
            throw constants_1.ERRORS.TypedError('Init_ManifestMissing');
        }
        this.lastSettings = settings;
        this.currentTarget = this.getInitTarget(settings);
        this.callPending = 0;
        try {
            return await this.getTarget().init(this.lastSettings);
        }
        catch (error) {
            if (await this.handleErrorFallback(error.code)) {
                return;
            }
            throw error;
        }
    }
    setTransports({ transports }) {
        this.lastSettings = { ...this.lastSettings, transports };
        this.getTarget().setTransports({ transports });
    }
    async call(params) {
        try {
            if (this.callPending === 0) {
                await this.beforeCallSynchronize(async () => {
                    this.callPending++;
                    await this.handleBeforeCall();
                });
            }
            const response = await this.getTarget().call(params);
            if (!response.success) {
                if (await this.handleErrorFallback(response.payload.code)) {
                    return await this.getTarget().call(params);
                }
            }
            return response;
        }
        catch (error) {
            return (0, events_1.createErrorMessage)(error);
        }
        finally {
            this.callPending--;
        }
    }
    requestLogin(params) {
        return this.getTarget().requestLogin(params);
    }
    uiResponse(params) {
        return this.getTarget().uiResponse(params);
    }
    cancel(error) {
        return this.getTarget().cancel(error);
    }
    dispose() {
        this.eventEmitter.removeAllListeners();
        this.callPending = 0;
        return this.getTarget().dispose();
    }
}
exports.TrezorConnectDynamic = TrezorConnectDynamic;
//# sourceMappingURL=dynamic.js.map