"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseConnectSettings = exports.corsValidator = exports.parseLocalFirmwares = exports.DEFAULT_PRIORITY = void 0;
const thpSettings_1 = require("./thpSettings");
const version_1 = require("./version");
exports.DEFAULT_PRIORITY = 2;
const initialSettings = {
    configSrc: './data/config.json',
    version: version_1.VERSION,
    debug: false,
    priority: exports.DEFAULT_PRIORITY,
    trustedHost: true,
    connectSrc: version_1.DEFAULT_DOMAIN,
    iframeSrc: `${version_1.DEFAULT_DOMAIN}iframe.html`,
    popup: false,
    popupSrc: `${version_1.DEFAULT_DOMAIN}popup.html`,
    webusbSrc: `${version_1.DEFAULT_DOMAIN}webusb.html`,
    transports: undefined,
    pendingTransportEvent: true,
    env: 'node',
    lazyLoad: false,
    timestamp: new Date().getTime(),
    interactionTimeout: 1200,
    sharedLogger: true,
    deeplinkUrl: `${version_1.DEFAULT_DOMAIN}deeplink/${version_1.DEEPLINK_VERSION}/`,
    transportReconnect: true,
};
const parseManifest = (manifest) => {
    if (!manifest)
        return;
    if (typeof manifest.email !== 'string')
        return;
    if (typeof manifest.appUrl !== 'string')
        return;
    if (typeof manifest.appName !== 'undefined' && typeof manifest.appName !== 'string')
        return;
    if (typeof manifest.appIcon !== 'undefined' && typeof manifest.appIcon !== 'string')
        return;
    return {
        email: manifest.email,
        appUrl: manifest.appUrl,
        appName: manifest.appName,
        appIcon: manifest.appIcon,
    };
};
const parseLocalFirmwares = (localFirmwares) => {
    if (!localFirmwares)
        return;
    if (typeof localFirmwares.firmwareDir !== 'string')
        return;
    if (!Array.isArray(localFirmwares.firmwareList))
        return;
    return {
        firmwareDir: localFirmwares.firmwareDir,
        firmwareList: localFirmwares.firmwareList,
    };
};
exports.parseLocalFirmwares = parseLocalFirmwares;
const corsValidator = (url) => {
    if (typeof url !== 'string')
        return;
    if (url === '../')
        return url;
    if (url.match(/^https:\/\/([A-Za-z0-9\-_]+\.)*trezor\.io\//))
        return url;
    if (url.match(/^https?:\/\/localhost:[58][0-9]{3}\//))
        return url;
    if (url.match(/^https:\/\/([A-Za-z0-9\-_]+\.)*sldev\.cz\//))
        return url;
    if (url.match(/^https?:\/\/([A-Za-z0-9\-_]+\.)*trezoriovpjcahpzkrewelclulmszwbqpzmzgub37gbcjlvluxtruqad\.onion\//))
        return url;
};
exports.corsValidator = corsValidator;
const parseConnectSettings = (input = {}) => {
    const settings = { ...initialSettings };
    if ('debug' in input) {
        if (typeof input.debug === 'boolean') {
            settings.debug = input.debug;
        }
        else if (typeof input.debug === 'string') {
            settings.debug = input.debug === 'true';
        }
    }
    if (input.trustedHost === false) {
        settings.trustedHost = input.trustedHost;
    }
    if (typeof input.connectSrc === 'string') {
        settings.connectSrc = (0, exports.corsValidator)(input.connectSrc);
    }
    else if (settings.trustedHost) {
        settings.connectSrc = input.connectSrc;
    }
    const src = settings.connectSrc || version_1.DEFAULT_DOMAIN;
    settings.iframeSrc = `${src}iframe.html`;
    settings.popupSrc = `${src}popup.html`;
    settings.webusbSrc = `${src}webusb.html`;
    if (typeof input.transportReconnect === 'boolean') {
        settings.transportReconnect = input.transportReconnect;
    }
    if (typeof input.localFirmwares === 'object') {
        settings.localFirmwares = (0, exports.parseLocalFirmwares)(input.localFirmwares);
    }
    if (typeof input.firmwareUpdateSource === 'string') {
        settings.firmwareUpdateSource = input.firmwareUpdateSource;
    }
    if (Array.isArray(input.transports)) {
        settings.transports = input.transports;
    }
    if (typeof input.popup === 'boolean') {
        settings.popup = input.popup;
    }
    if (typeof input.lazyLoad === 'boolean') {
        settings.lazyLoad = input.lazyLoad;
    }
    if (typeof input.pendingTransportEvent === 'boolean') {
        settings.pendingTransportEvent = input.pendingTransportEvent;
    }
    if (typeof input.extension === 'string') {
        settings.extension = input.extension;
    }
    if (typeof input.env === 'string') {
        settings.env = input.env;
    }
    if (typeof input.timestamp === 'number') {
        settings.timestamp = input.timestamp;
    }
    if (typeof input.interactionTimeout === 'number') {
        settings.interactionTimeout = input.interactionTimeout;
    }
    if (typeof input.manifest === 'object') {
        settings.manifest = parseManifest(input.manifest);
    }
    if (typeof input.sharedLogger === 'boolean') {
        settings.sharedLogger = input.sharedLogger;
    }
    if (typeof input.coreMode === 'string' &&
        ['auto', 'popup', 'iframe', 'suite-desktop'].includes(input.coreMode)) {
        settings.coreMode = input.coreMode;
    }
    if (typeof input._extendWebextensionLifetime === 'boolean') {
        settings._extendWebextensionLifetime = input._extendWebextensionLifetime;
    }
    if (typeof input.binFilesBaseUrl === 'string') {
        settings.binFilesBaseUrl = input.binFilesBaseUrl;
    }
    if (typeof input.enableFirmwareHashCheck === 'boolean') {
        settings.enableFirmwareHashCheck = Boolean(input.enableFirmwareHashCheck);
    }
    if (typeof input.firmwareHashCheckTimeouts === 'object' &&
        input.firmwareHashCheckTimeouts !== null) {
        settings.firmwareHashCheckTimeouts = input.firmwareHashCheckTimeouts;
    }
    if (typeof input.npmVersion === 'string') {
        settings.npmVersion = input.npmVersion;
    }
    settings.thp = (0, thpSettings_1.parseThpSettings)(input);
    return settings;
};
exports.parseConnectSettings = parseConnectSettings;
//# sourceMappingURL=connectSettings.js.map