"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const schema_utils_1 = require("@trezor/schema-utils");
const AbstractMethod_1 = require("../../../core/AbstractMethod");
const coinInfo_1 = require("../../../data/coinInfo");
const events_1 = require("../../../events");
const types_1 = require("../../../types");
const pathUtils_1 = require("../../../utils/pathUtils");
const paramsValidator_1 = require("../../common/paramsValidator");
class TezosGetPublicKey extends AbstractMethod_1.AbstractMethod {
    hasBundle;
    init() {
        this.requiredPermissions = ['read'];
        this.requiredDeviceCapabilities = ['Capability_Tezos'];
        this.firmwareRange = (0, paramsValidator_1.getFirmwareRange)(this.name, (0, coinInfo_1.getMiscNetwork)('Tezos'), this.firmwareRange);
        this.hasBundle = !!this.payload.bundle;
        const payload = !this.payload.bundle
            ? { ...this.payload, bundle: [this.payload] }
            : this.payload;
        (0, schema_utils_1.Assert)((0, types_1.Bundle)(types_1.GetPublicKey), payload);
        this.params = payload.bundle.map(batch => {
            const path = (0, pathUtils_1.validatePath)(batch.path, 3);
            return {
                address_n: path,
                show_display: typeof batch.showOnTrezor === 'boolean' ? batch.showOnTrezor : true,
                chunkify: typeof batch.chunkify === 'boolean' ? batch.chunkify : false,
            };
        });
    }
    get info() {
        return 'Export Tezos public key';
    }
    get confirmation() {
        return {
            view: 'export-address',
            label: this.params.length > 1
                ? 'Export multiple Tezos public keys'
                : `Export Tezos public key for account #${(0, pathUtils_1.fromHardened)(this.params[0].address_n[2]) + 1}`,
        };
    }
    async run() {
        const responses = [];
        const cmd = this.device.getCommands();
        for (let i = 0; i < this.params.length; i++) {
            const batch = this.params[i];
            const { message } = await cmd.typedCall('TezosGetPublicKey', 'TezosPublicKey', batch);
            responses.push({
                path: batch.address_n,
                serializedPath: (0, pathUtils_1.getSerializedPath)(batch.address_n),
                publicKey: message.public_key,
            });
            if (this.hasBundle) {
                this.postMessage((0, events_1.createUiMessage)(events_1.UI.BUNDLE_PROGRESS, {
                    total: this.params.length,
                    progress: i,
                    response: message,
                }));
            }
        }
        return this.hasBundle ? responses : responses[0];
    }
}
exports.default = TezosGetPublicKey;
//# sourceMappingURL=tezosGetPublicKey.js.map