"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const schema_utils_1 = require("@trezor/schema-utils");
const constants_1 = require("../../../constants");
const AbstractMethod_1 = require("../../../core/AbstractMethod");
const coinInfo_1 = require("../../../data/coinInfo");
const stellar_1 = require("../../../types/api/stellar");
const pathUtils_1 = require("../../../utils/pathUtils");
const paramsValidator_1 = require("../../common/paramsValidator");
const helper = tslib_1.__importStar(require("../stellarSignTx"));
const StellarSignTransactionFeatures = Object.freeze({
    manageBuyOffer: ['1.10.4', '2.4.3'],
    pathPaymentStrictSend: ['1.10.4', '2.4.3'],
});
class StellarSignTransaction extends AbstractMethod_1.AbstractMethod {
    init() {
        this.requiredPermissions = ['read', 'write'];
        this.requiredDeviceCapabilities = ['Capability_Stellar'];
        this.firmwareRange = (0, paramsValidator_1.getFirmwareRange)(this.name, (0, coinInfo_1.getMiscNetwork)('Stellar'), this.firmwareRange);
        const { payload } = this;
        (0, schema_utils_1.AssertWeak)(stellar_1.StellarSignTransaction, payload);
        const path = (0, pathUtils_1.validatePath)(payload.path, 3);
        const { transaction } = payload;
        this.params = {
            path,
            networkPassphrase: payload.networkPassphrase,
            transaction,
        };
    }
    get info() {
        return 'Sign Stellar transaction';
    }
    _isFeatureSupported(feature) {
        return this.device.atLeast(StellarSignTransactionFeatures[feature]);
    }
    _ensureFeatureIsSupported(feature) {
        if (!this._isFeatureSupported(feature)) {
            throw constants_1.ERRORS.TypedError('Method_InvalidParameter', `Feature ${feature} not supported by device firmware`);
        }
    }
    _ensureFirmwareSupportsParams() {
        const { params } = this;
        if (params.transaction.operations &&
            params.transaction.operations.find(o => o.type === 'manageBuyOffer')) {
            this._ensureFeatureIsSupported('manageBuyOffer');
        }
        if (params.transaction.operations &&
            params.transaction.operations.find(o => o.type === 'pathPaymentStrictSend')) {
            this._ensureFeatureIsSupported('pathPaymentStrictSend');
        }
    }
    async run() {
        this._ensureFirmwareSupportsParams();
        const response = await helper.stellarSignTx(this.device.getCommands().typedCall, this.params.path, this.params.networkPassphrase, this.params.transaction);
        return {
            publicKey: response.public_key,
            signature: response.signature,
        };
    }
}
exports.default = StellarSignTransaction;
//# sourceMappingURL=stellarSignTransaction.js.map