"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const schema_utils_1 = require("@trezor/schema-utils");
const AbstractMethod_1 = require("../core/AbstractMethod");
const types_1 = require("../types");
const paramsValidator_1 = require("./common/paramsValidator");
const coinInfo_1 = require("../data/coinInfo");
const formatUtils_1 = require("../utils/formatUtils");
const pathUtils_1 = require("../utils/pathUtils");
class SignMessage extends AbstractMethod_1.AbstractMethod {
    coinInfo;
    init() {
        this.requiredPermissions = ['read', 'write'];
        const { payload } = this;
        (0, schema_utils_1.Assert)(types_1.SignMessage, payload);
        const path = (0, pathUtils_1.validatePath)(payload.path);
        if (payload.coin) {
            this.coinInfo = (0, coinInfo_1.getBitcoinNetwork)(payload.coin);
            (0, paramsValidator_1.validateCoinPath)(path, this.coinInfo);
        }
        else {
            this.coinInfo = (0, coinInfo_1.getBitcoinNetwork)(path);
        }
        this.firmwareRange = (0, paramsValidator_1.getFirmwareRange)(payload.no_script_type ? 'signMessageNoScriptType' : this.name, this.coinInfo, this.firmwareRange);
        const messageHex = payload.hex
            ? (0, formatUtils_1.messageToHex)(payload.message)
            : Buffer.from(payload.message, 'utf8').toString('hex');
        const scriptType = (0, pathUtils_1.getScriptType)(path);
        this.params = {
            address_n: path,
            message: messageHex,
            coin_name: this.coinInfo ? this.coinInfo.name : undefined,
            script_type: scriptType && scriptType !== 'SPENDMULTISIG' ? scriptType : 'SPENDADDRESS',
            no_script_type: payload.no_script_type,
        };
    }
    get info() {
        const coinInfo = (0, coinInfo_1.getBitcoinNetwork)(this.payload.coin ?? this.params.address_n);
        return (0, pathUtils_1.getLabel)('Sign #NETWORK message', coinInfo);
    }
    getButtonRequestData(code, name) {
        if (code === 'ButtonRequest_Other' && name === 'sign_message') {
            return {
                type: 'message',
                serializedPath: (0, pathUtils_1.getSerializedPath)(this.params.address_n),
                coin: this.coinInfo?.shortcut ?? 'BTC',
                message: this.payload.hex ? (0, formatUtils_1.hexToText)(this.payload.message) : this.payload.message,
            };
        }
    }
    async run() {
        const cmd = this.device.getCommands();
        const { message } = await cmd.typedCall('SignMessage', 'MessageSignature', this.params);
        const signatureBuffer = Buffer.from(message.signature, 'hex');
        message.signature = signatureBuffer.toString('base64');
        return message;
    }
}
exports.default = SignMessage;
//# sourceMappingURL=signMessage.js.map