"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const schema_utils_1 = require("@trezor/schema-utils");
const AbstractMethod_1 = require("../../../core/AbstractMethod");
const coinInfo_1 = require("../../../data/coinInfo");
const types_1 = require("../../../types");
const ethereumUtils_1 = require("../../../utils/ethereumUtils");
const formatUtils_1 = require("../../../utils/formatUtils");
const pathUtils_1 = require("../../../utils/pathUtils");
const paramsValidator_1 = require("../../common/paramsValidator");
const ethereumDefinitions_1 = require("../ethereumDefinitions");
class EthereumSignMessage extends AbstractMethod_1.AbstractMethod {
    init() {
        this.requiredPermissions = ['read', 'write'];
        this.requiredDeviceCapabilities = ['Capability_Ethereum'];
        const { payload } = this;
        (0, schema_utils_1.Assert)(types_1.EthereumSignMessage, payload);
        const path = (0, pathUtils_1.validatePath)(payload.path, 3);
        const network = (0, coinInfo_1.getEthereumNetwork)(path);
        this.firmwareRange = (0, paramsValidator_1.getFirmwareRange)(this.name, network, this.firmwareRange);
        const messageHex = payload.hex
            ? (0, formatUtils_1.messageToHex)(payload.message)
            : Buffer.from(payload.message, 'utf8').toString('hex');
        this.params = {
            address_n: path,
            message: messageHex,
        };
    }
    async initAsync() {
        if (this.params.network)
            return;
        const { address_n } = this.params;
        const slip44 = (0, pathUtils_1.getSlip44ByPath)(address_n);
        this.params.definitions = await (0, ethereumDefinitions_1.getEthereumDefinitions)({
            slip44,
        });
    }
    get info() {
        return (0, ethereumUtils_1.getNetworkLabel)('Sign #NETWORK message', (0, coinInfo_1.getEthereumNetwork)(this.params.address_n));
    }
    getButtonRequestData(code, name) {
        if (code === 'ButtonRequest_Other' && name === 'sign_message') {
            return {
                type: 'message',
                coin: this.params.network?.shortcut ?? 'ETH',
                serializedPath: (0, pathUtils_1.getSerializedPath)(this.params.address_n),
                message: this.payload.hex ? (0, formatUtils_1.hexToText)(this.payload.message) : this.payload.message,
            };
        }
    }
    async run() {
        const cmd = this.device.getCommands();
        const { address_n, message } = this.params;
        const response = await cmd.typedCall('EthereumSignMessage', 'EthereumMessageSignature', {
            encoded_network: this.params.definitions?.encoded_network,
            address_n,
            message,
        });
        return response.message;
    }
}
exports.default = EthereumSignMessage;
//# sourceMappingURL=ethereumSignMessage.js.map