"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const schema_utils_1 = require("@trezor/schema-utils");
const AbstractMethod_1 = require("../../../core/AbstractMethod");
const coinInfo_1 = require("../../../data/coinInfo");
const events_1 = require("../../../events");
const types_1 = require("../../../types");
const ethereumUtils_1 = require("../../../utils/ethereumUtils");
const pathUtils_1 = require("../../../utils/pathUtils");
const paramsValidator_1 = require("../../common/paramsValidator");
class EthereumGetPublicKey extends AbstractMethod_1.AbstractMethod {
    hasBundle;
    init() {
        this.requiredPermissions = ['read'];
        this.requiredDeviceCapabilities = ['Capability_Ethereum'];
        this.hasBundle = !!this.payload.bundle;
        const payload = !this.payload.bundle
            ? { ...this.payload, bundle: [this.payload] }
            : this.payload;
        (0, schema_utils_1.Assert)((0, types_1.Bundle)(types_1.GetPublicKey), payload);
        this.params = payload.bundle.map(batch => {
            const path = (0, pathUtils_1.validatePath)(batch.path, 3);
            const network = (0, coinInfo_1.getEthereumNetwork)(path);
            this.firmwareRange = (0, paramsValidator_1.getFirmwareRange)(this.name, network, this.firmwareRange);
            return {
                address_n: path,
                show_display: typeof batch.showOnTrezor === 'boolean' ? batch.showOnTrezor : false,
                network,
            };
        });
    }
    get info() {
        if (this.params.length === 1) {
            return (0, ethereumUtils_1.getNetworkLabel)('Export #NETWORK public key', this.params[0].network);
        }
        const requestedNetworks = this.params.map(b => b.network);
        const uniqNetworks = (0, coinInfo_1.getUniqueNetworks)(requestedNetworks);
        if (uniqNetworks.length === 1 && uniqNetworks[0]) {
            return (0, ethereumUtils_1.getNetworkLabel)('Export multiple #NETWORK public keys', uniqNetworks[0]);
        }
        return 'Export multiple public keys';
    }
    get confirmation() {
        return {
            view: 'export-xpub',
            label: this.info,
        };
    }
    async run() {
        const responses = [];
        const cmd = this.device.getCommands();
        for (let i = 0; i < this.params.length; i++) {
            const { address_n, show_display } = this.params[i];
            const publicKey = await cmd.ethereumGetPublicKey({ address_n, show_display });
            const response = {
                path: address_n,
                serializedPath: (0, pathUtils_1.getSerializedPath)(address_n),
                childNum: publicKey.node.child_num,
                xpub: publicKey.xpub,
                chainCode: publicKey.node.chain_code,
                publicKey: publicKey.node.public_key,
                fingerprint: publicKey.node.fingerprint,
                depth: publicKey.node.depth,
            };
            responses.push(response);
            if (this.hasBundle) {
                this.postMessage((0, events_1.createUiMessage)(events_1.UI.BUNDLE_PROGRESS, {
                    total: this.params.length,
                    progress: i,
                    response,
                }));
            }
        }
        return this.hasBundle ? responses : responses[0];
    }
}
exports.default = EthereumGetPublicKey;
//# sourceMappingURL=ethereumGetPublicKey.js.map