"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const schema_utils_1 = require("@trezor/schema-utils");
const constants_1 = require("../../../constants");
const AbstractMethod_1 = require("../../../core/AbstractMethod");
const coinInfo_1 = require("../../../data/coinInfo");
const events_1 = require("../../../events");
const types_1 = require("../../../types");
const getAddress_1 = require("../../../types/api/getAddress");
const ethereumUtils_1 = require("../../../utils/ethereumUtils");
const formatUtils_1 = require("../../../utils/formatUtils");
const pathUtils_1 = require("../../../utils/pathUtils");
const paramsValidator_1 = require("../../common/paramsValidator");
const ethereumDefinitions_1 = require("../ethereumDefinitions");
class EthereumGetAddress extends AbstractMethod_1.AbstractMethod {
    hasBundle;
    progress = 0;
    init() {
        this.noBackupConfirmationMode = 'always';
        this.requiredPermissions = ['read'];
        this.requiredDeviceCapabilities = ['Capability_Ethereum'];
        this.hasBundle = !!this.payload.bundle;
        const payload = !this.payload.bundle
            ? { ...this.payload, bundle: [this.payload] }
            : this.payload;
        (0, schema_utils_1.Assert)((0, types_1.Bundle)(getAddress_1.GetAddress), payload);
        this.params = payload.bundle.map(batch => {
            const path = (0, pathUtils_1.validatePath)(batch.path, 3);
            const network = (0, coinInfo_1.getEthereumNetwork)(path);
            this.firmwareRange = (0, paramsValidator_1.getFirmwareRange)(this.name, network, this.firmwareRange);
            return {
                address_n: path,
                show_display: typeof batch.showOnTrezor === 'boolean' ? batch.showOnTrezor : true,
                address: batch.address,
                network,
                chunkify: typeof batch.chunkify === 'boolean' ? batch.chunkify : false,
            };
        });
        const useEventListener = payload.useEventListener &&
            this.params.length === 1 &&
            typeof this.params[0].address === 'string' &&
            this.params[0].show_display;
        this.useUi = !useEventListener;
    }
    async initAsync() {
        for (let i = 0; i < this.params.length; i++) {
            if (!this.params[i].network) {
                const slip44 = (0, pathUtils_1.getSlip44ByPath)(this.params[i].address_n);
                const definitions = await (0, ethereumDefinitions_1.getEthereumDefinitions)({
                    slip44,
                });
                const decoded = (0, ethereumDefinitions_1.decodeEthereumDefinition)(definitions);
                if (decoded.network) {
                    this.params[i].network = (0, ethereumDefinitions_1.ethereumNetworkInfoFromDefinition)(decoded.network);
                }
                if (definitions.encoded_network) {
                    this.params[i].encoded_network = definitions.encoded_network;
                }
            }
        }
    }
    get info() {
        if (this.params.length === 1) {
            return (0, ethereumUtils_1.getNetworkLabel)('Export #NETWORK address', this.params[0].network);
        }
        const requestedNetworks = this.params.map(b => b.network);
        const uniqNetworks = (0, coinInfo_1.getUniqueNetworks)(requestedNetworks);
        if (uniqNetworks.length === 1 && uniqNetworks[0]) {
            return (0, ethereumUtils_1.getNetworkLabel)('Export multiple #NETWORK addresses', uniqNetworks[0]);
        }
        return 'Export multiple addresses';
    }
    getButtonRequestData(code) {
        if (code === 'ButtonRequest_Address') {
            return {
                type: 'address',
                serializedPath: (0, pathUtils_1.getSerializedPath)(this.params[this.progress].address_n),
                address: this.params[this.progress].address || 'not-set',
            };
        }
    }
    get confirmation() {
        return {
            view: 'export-address',
            label: this.info,
        };
    }
    async _call(params) {
        const response = await this.device.getCommands().ethereumGetAddress(params);
        return {
            path: params.address_n,
            serializedPath: (0, pathUtils_1.getSerializedPath)(params.address_n),
            address: response.address,
            mac: response.mac,
        };
    }
    async run() {
        const responses = [];
        for (let i = 0; i < this.params.length; i++) {
            const batch = this.params[i];
            if (batch.show_display) {
                const silent = await this._call({
                    ...batch,
                    show_display: false,
                });
                if (typeof batch.address === 'string') {
                    if ((0, formatUtils_1.stripHexPrefix)(batch.address).toLowerCase() !==
                        (0, formatUtils_1.stripHexPrefix)(silent.address).toLowerCase()) {
                        throw constants_1.ERRORS.TypedError('Method_AddressNotMatch');
                    }
                }
                else {
                    batch.address = silent.address;
                }
            }
            const response = await this._call({
                ...batch,
            });
            responses.push(response);
            if (this.hasBundle) {
                this.postMessage((0, events_1.createUiMessage)(events_1.UI.BUNDLE_PROGRESS, {
                    total: this.params.length,
                    progress: i,
                    response,
                }));
            }
            this.progress++;
        }
        return this.hasBundle ? responses : responses[0];
    }
}
exports.default = EthereumGetAddress;
//# sourceMappingURL=ethereumGetAddress.js.map