"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.sendChunkedHexString = exports.hexStringByteLength = exports.composeTxPlan = exports.getTtl = exports.prepareCertificates = exports.transformUtxos = void 0;
const coin_selection_1 = require("@fivebinaries/coin-selection");
const constants_1 = require("../../constants");
const CARDANO_DEFAULT_TTL_OFFSET = 7200;
const transformUtxos = (utxos) => {
    const result = [];
    utxos?.forEach(utxo => {
        const foundItem = result.find(res => res.txHash === utxo.txid && res.outputIndex === utxo.vout);
        if (utxo.cardanoSpecific) {
            if (!foundItem) {
                result.push({
                    address: utxo.address,
                    txHash: utxo.txid,
                    outputIndex: utxo.vout,
                    amount: [{ quantity: utxo.amount, unit: utxo.cardanoSpecific.unit }],
                });
            }
            else {
                foundItem.amount.push({ quantity: utxo.amount, unit: utxo.cardanoSpecific.unit });
            }
        }
    });
    return result;
};
exports.transformUtxos = transformUtxos;
const prepareCertificates = (certs) => {
    const convertedCerts = [];
    certs.forEach(cert => {
        switch (cert.type) {
            case constants_1.PROTO.CardanoCertificateType.STAKE_DELEGATION:
                convertedCerts.push({
                    type: cert.type,
                    pool: cert.pool,
                });
                break;
            case constants_1.PROTO.CardanoCertificateType.STAKE_REGISTRATION:
            case constants_1.PROTO.CardanoCertificateType.STAKE_DEREGISTRATION:
                convertedCerts.push({
                    type: cert.type,
                });
                break;
            case constants_1.PROTO.CardanoCertificateType.VOTE_DELEGATION:
                if (cert.dRep?.type === constants_1.PROTO.CardanoDRepType.ABSTAIN ||
                    cert.dRep?.type === constants_1.PROTO.CardanoDRepType.NO_CONFIDENCE) {
                    convertedCerts.push({
                        type: cert.type,
                        dRep: {
                            type: cert.dRep.type,
                        },
                    });
                }
                else if (cert.dRep?.type === constants_1.PROTO.CardanoDRepType.KEY_HASH) {
                    convertedCerts.push({
                        type: cert.type,
                        dRep: {
                            type: cert.dRep.type,
                            keyHash: cert.dRep.keyHash,
                        },
                    });
                }
                else if (cert.dRep?.type === constants_1.PROTO.CardanoDRepType.SCRIPT_HASH) {
                    convertedCerts.push({
                        type: cert.type,
                        dRep: {
                            type: cert.dRep.type,
                            scriptHash: cert.dRep.scriptHash,
                        },
                    });
                }
                break;
        }
    });
    return convertedCerts;
};
exports.prepareCertificates = prepareCertificates;
const getTtl = (testnet) => {
    const shelleySlot = testnet ? 6192449 : 4924800;
    const shelleyTimestamp = testnet ? 1672848449 : 1596491091;
    const currentTimestamp = Math.floor(new Date().getTime() / 1000);
    const currentSlot = shelleySlot + currentTimestamp - shelleyTimestamp;
    return currentSlot + CARDANO_DEFAULT_TTL_OFFSET;
};
exports.getTtl = getTtl;
const composeTxPlan = (descriptor, utxo, outputs, certificates, withdrawals, changeAddress, isTestnet, options) => (0, coin_selection_1.coinSelection)({
    utxos: (0, exports.transformUtxos)(utxo),
    outputs,
    changeAddress,
    certificates: (0, exports.prepareCertificates)(certificates),
    withdrawals,
    accountPubKey: descriptor,
    ttl: (0, exports.getTtl)(isTestnet),
}, options);
exports.composeTxPlan = composeTxPlan;
const hexStringByteLength = (s) => s.length / 2;
exports.hexStringByteLength = hexStringByteLength;
const sendChunkedHexString = async (typedCall, data, chunkSize, messageType, responseType = 'CardanoTxItemAck') => {
    let processedSize = 0;
    while (processedSize < data.length) {
        const chunk = data.slice(processedSize, processedSize + chunkSize);
        await typedCall(messageType, responseType, {
            data: chunk,
        });
        processedSize += chunkSize;
    }
};
exports.sendChunkedHexString = sendChunkedHexString;
//# sourceMappingURL=cardanoUtils.js.map