"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.signTxLegacy = void 0;
const constants_1 = require("../../constants");
const requestPrevTxInfo = ({ txRequest: { request_type, details }, refTxs, }) => {
    const { tx_hash } = details;
    if (!tx_hash) {
        throw constants_1.ERRORS.TypedError('Runtime', 'requestPrevTxInfo: unknown details.tx_hash');
    }
    const tx = refTxs[tx_hash.toLowerCase()];
    if (!tx) {
        throw constants_1.ERRORS.TypedError('Runtime', `requestPrevTxInfo: Requested unknown tx: ${tx_hash}`);
    }
    if (!tx.bin_outputs) {
        throw constants_1.ERRORS.TypedError('Runtime', `requestPrevTxInfo: bin_outputs not set tx: ${tx_hash}`);
    }
    if (request_type === 'TXINPUT') {
        return { inputs: [tx.inputs[details.request_index]] };
    }
    if (request_type === 'TXOUTPUT') {
        return { bin_outputs: [tx.bin_outputs[details.request_index]] };
    }
    if (request_type === 'TXEXTRADATA') {
        if (typeof details.extra_data_len !== 'number') {
            throw constants_1.ERRORS.TypedError('Runtime', 'requestPrevTxInfo: Missing extra_data_len');
        }
        if (typeof details.extra_data_offset !== 'number') {
            throw constants_1.ERRORS.TypedError('Runtime', 'requestPrevTxInfo: Missing extra_data_offset');
        }
        if (typeof tx.extra_data !== 'string') {
            throw constants_1.ERRORS.TypedError('Runtime', `requestPrevTxInfo: No extra data for transaction ${tx.hash}`);
        }
        const data = tx.extra_data;
        const dataLen = details.extra_data_len;
        const dataOffset = details.extra_data_offset;
        const extra_data = data.substring(dataOffset * 2, (dataOffset + dataLen) * 2);
        return { extra_data };
    }
    if (request_type === 'TXMETA') {
        const data = tx.extra_data;
        const meta = {
            version: tx.version,
            lock_time: tx.lock_time,
            inputs_cnt: tx.inputs.length,
            outputs_cnt: tx.bin_outputs.length,
            timestamp: tx.timestamp,
            version_group_id: tx.version_group_id,
            expiry: tx.expiry,
            branch_id: tx.branch_id,
        };
        if (typeof data === 'string' && data.length !== 0) {
            return {
                ...meta,
                extra_data_len: data.length / 2,
            };
        }
        return meta;
    }
    throw constants_1.ERRORS.TypedError('Runtime', `requestPrevTxInfo: Unknown request type: ${request_type}`);
};
const requestSignedTxInfo = ({ txRequest: { request_type, details }, inputs, outputs, }) => {
    if (request_type === 'TXINPUT') {
        return { inputs: [inputs[details.request_index]] };
    }
    if (request_type === 'TXOUTPUT') {
        return { outputs: [outputs[details.request_index]] };
    }
    if (request_type === 'TXMETA') {
        throw constants_1.ERRORS.TypedError('Runtime', 'requestSignedTxInfo: Cannot read TXMETA from signed transaction');
    }
    if (request_type === 'TXEXTRADATA') {
        throw constants_1.ERRORS.TypedError('Runtime', 'requestSignedTxInfo: Cannot read TXEXTRADATA from signed transaction');
    }
    throw constants_1.ERRORS.TypedError('Runtime', `requestSignedTxInfo: Unknown request type: ${request_type}`);
};
const requestTxAck = (props) => {
    const { tx_hash } = props.txRequest.details;
    if (tx_hash) {
        return requestPrevTxInfo(props);
    }
    return requestSignedTxInfo(props);
};
const saveTxSignatures = (txRequest, serializedTx, signatures) => {
    if (!txRequest)
        return;
    const { signature_index, signature, serialized_tx } = txRequest;
    if (serialized_tx) {
        serializedTx.push(serialized_tx);
    }
    if (typeof signature_index === 'number') {
        if (!signature) {
            throw constants_1.ERRORS.TypedError('Runtime', 'saveTxSignatures: Unexpected null in trezor:TxRequestSerialized signature.');
        }
        signatures[signature_index] = signature;
    }
};
const processTxRequest = async (props) => {
    const { typedCall, txRequest, serializedTx, signatures } = props;
    if (txRequest.serialized)
        saveTxSignatures(txRequest.serialized, serializedTx, signatures);
    if (txRequest.request_type === 'TXFINISHED') {
        return Promise.resolve({
            signatures,
            serializedTx: serializedTx.join(''),
        });
    }
    const txAck = requestTxAck(props);
    const { message } = await typedCall('TxAck', 'TxRequest', { tx: txAck });
    return processTxRequest({
        ...props,
        txRequest: message,
    });
};
const signTxLegacy = async ({ typedCall, inputs, outputs, refTxs, options, coinInfo, }) => {
    const { message } = await typedCall('SignTx', 'TxRequest', {
        ...options,
        version: options.version === undefined && coinInfo.isBitcoin ? 1 : options.version,
        inputs_count: inputs.length,
        outputs_count: outputs.length,
        coin_name: coinInfo.name,
    });
    return processTxRequest({
        typedCall,
        txRequest: message,
        refTxs: refTxs.reduce((record, tx) => ({
            ...record,
            [tx.hash.toLowerCase()]: tx,
        }), {}),
        inputs,
        outputs,
        paymentRequests: [],
        serializedTx: [],
        signatures: [],
    });
};
exports.signTxLegacy = signTxLegacy;
//# sourceMappingURL=signtxLegacy.js.map