"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getSolanaStakingData = exports.getDelegations = exports.stakeAccountState = exports.isStake = exports.FILTER_OFFSET = exports.FILTER_DATA_SIZE = exports.STAKE_ACCOUNT_V2_SIZE = void 0;
const kit_1 = require("@solana/kit");
const stake_1 = require("@solana-program/stake");
const solana_1 = require("@trezor/blockchain-link-types/lib/solana");
exports.STAKE_ACCOUNT_V2_SIZE = 200;
exports.FILTER_DATA_SIZE = 200n;
exports.FILTER_OFFSET = 44n;
const EVERSTAKE_VOTER_PUBKEYS = [
    '9QU2QSxhb24FUX3Tu2FpczXjpK3VYrvRudywSZaM29mF',
    'GkqYQysEGmuL6V2AJoNnWZUz2ZBGWhzQXsJiXm2CLKAN',
];
const isStake = (state) => state.__kind === 'Stake';
exports.isStake = isStake;
const stakeAccountState = (account, currentEpoch) => {
    const { state } = account;
    if (!(0, exports.isStake)(state)) {
        return solana_1.StakeState.Inactive;
    }
    const { activationEpoch, deactivationEpoch } = state.fields[1].delegation;
    if (activationEpoch > currentEpoch) {
        return solana_1.StakeState.Inactive;
    }
    if (activationEpoch === currentEpoch) {
        if (deactivationEpoch === activationEpoch)
            return solana_1.StakeState.Inactive;
        return solana_1.StakeState.Activating;
    }
    if (deactivationEpoch > currentEpoch)
        return solana_1.StakeState.Active;
    if (deactivationEpoch === currentEpoch)
        return solana_1.StakeState.Deactivating;
    return solana_1.StakeState.Deactivated;
};
exports.stakeAccountState = stakeAccountState;
const getDelegations = async (rpc, descriptor) => {
    try {
        const accounts = await rpc
            .getProgramAccounts(stake_1.STAKE_PROGRAM_ADDRESS, {
            encoding: 'base64',
            filters: [
                {
                    dataSize: exports.FILTER_DATA_SIZE,
                },
                {
                    memcmp: {
                        bytes: descriptor,
                        encoding: 'base58',
                        offset: exports.FILTER_OFFSET,
                    },
                },
            ],
        })
            .send();
        return accounts.map(account => {
            const parsedAccount = (0, kit_1.parseBase64RpcAccount)(account.pubkey, account.account);
            return (0, stake_1.decodeStakeStateAccount)(parsedAccount);
        });
    }
    catch {
        throw new Error('Failed to fetch delegations');
    }
};
exports.getDelegations = getDelegations;
const getSolanaStakingData = async (rpc, descriptor, epoch) => {
    const stakingAccounts = await (0, exports.getDelegations)(rpc, descriptor);
    return stakingAccounts
        .map(account => {
        const stakeAccount = account?.data;
        if (!stakeAccount)
            return;
        const stakeState = (0, exports.stakeAccountState)(stakeAccount, BigInt(epoch));
        const { state } = account?.data ?? {};
        if (!(0, exports.isStake)(state))
            return;
        if (state && 'fields' in state) {
            const { fields } = state;
            const voterPubkey = fields[1]?.delegation?.voterPubkey;
            if (!EVERSTAKE_VOTER_PUBKEYS.includes(voterPubkey))
                return;
            return {
                rentExemptReserve: fields[0]?.rentExemptReserve.toString(),
                stake: fields[1]?.delegation?.stake.toString(),
                status: stakeState,
            };
        }
    })
        .filter(account => account !== undefined);
};
exports.getSolanaStakingData = getSolanaStakingData;
//# sourceMappingURL=stakingAccounts.js.map