"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.default = Solana;
const kit_1 = require("@solana/kit");
const token_1 = require("@solana-program/token");
const token_2022_1 = require("@solana-program/token-2022");
const constants_1 = require("@trezor/blockchain-link-types/lib/constants");
const errors_1 = require("@trezor/blockchain-link-types/lib/constants/errors");
const blockchain_link_utils_1 = require("@trezor/blockchain-link-utils");
const solana_1 = require("@trezor/blockchain-link-utils/lib/solana");
const env_utils_1 = require("@trezor/env-utils");
const utils_1 = require("@trezor/utils");
const baseWorker_1 = require("../baseWorker");
const fee_1 = require("./utils/fee");
const getThrottledTransport_1 = require("./utils/getThrottledTransport");
const stakingAccounts_1 = require("./utils/stakingAccounts");
const THROTTLE_OPTIONS = {
    maxRps: 4,
    interval: 100,
};
function nonNullable(value) {
    return value !== null && value !== undefined;
}
const getAllSignatures = async (api, descriptor, fullHistory = false) => {
    let lastSignature;
    let keepFetching = true;
    let allSignatures = [];
    const defaultValueLimit = 100;
    while (keepFetching) {
        const signaturesInfos = await api.rpc
            .getSignaturesForAddress((0, kit_1.address)(descriptor), {
            before: lastSignature?.signature,
            limit: defaultValueLimit,
        })
            .send();
        const signatures = signaturesInfos.map(info => ({
            signature: info.signature,
            slot: info.slot,
        }));
        lastSignature = signatures[signatures.length - 1];
        keepFetching = signatures.length === defaultValueLimit && fullHistory;
        allSignatures = [...allSignatures, ...signatures];
    }
    return allSignatures;
};
const fetchTransactionPage = async (api, signatures) => (await Promise.all(signatures.map(signature => api.rpc
    .getTransaction(signature, {
    encoding: 'jsonParsed',
    maxSupportedTransactionVersion: 0,
    commitment: 'confirmed',
})
    .send()))).filter(nonNullable);
const isValidTransaction = (tx) => !!(tx && tx.meta && tx.transaction && tx.blockTime);
const pushTransaction = async (request) => {
    const rawTx = request.payload.hex.startsWith('0x')
        ? request.payload.hex.slice(2)
        : request.payload.hex;
    const api = await request.connect();
    const txByteArray = (0, kit_1.getBase16Encoder)().encode(rawTx);
    const transaction = (0, kit_1.getTransactionDecoder)().decode(txByteArray);
    (0, kit_1.assertTransactionIsFullySigned)(transaction);
    const compiledMessage = (0, kit_1.getCompiledTransactionMessageDecoder)().decode(transaction.messageBytes);
    const message = await (0, kit_1.decompileTransactionMessageFetchingLookupTables)(compiledMessage, api.rpc);
    if ((0, kit_1.isDurableNonceTransaction)(message)) {
        throw new Error('Unimplemented: Confirming durable nonce transactions');
    }
    let transactionWithBlockhashLifetime = transaction;
    if (message.lifetimeConstraint === undefined) {
        const { value: { blockhash, lastValidBlockHeight }, } = await api.rpc.getLatestBlockhash({ commitment: 'confirmed' }).send();
        transactionWithBlockhashLifetime = {
            ...transactionWithBlockhashLifetime,
            lifetimeConstraint: { blockhash, lastValidBlockHeight },
        };
    }
    else {
        transactionWithBlockhashLifetime = {
            ...transactionWithBlockhashLifetime,
            lifetimeConstraint: message.lifetimeConstraint,
        };
    }
    try {
        const signature = (0, kit_1.getSignatureFromTransaction)(transaction);
        const sendAndConfirmTransaction = (0, kit_1.sendAndConfirmTransactionFactory)(api);
        await sendAndConfirmTransaction(transactionWithBlockhashLifetime, {
            commitment: 'confirmed',
            skipPreflight: false,
        });
        return {
            type: constants_1.RESPONSES.PUSH_TRANSACTION,
            payload: signature,
        };
    }
    catch (error) {
        if ((0, kit_1.isSolanaError)(error, kit_1.SOLANA_ERROR__BLOCK_HEIGHT_EXCEEDED)) {
            throw new Error('Please make sure that you submit the transaction within 1 minute after signing.');
        }
        if ((0, kit_1.isSolanaError)(error, kit_1.SOLANA_ERROR__RPC_SUBSCRIPTIONS__CHANNEL_FAILED_TO_CONNECT) ||
            (0, kit_1.isSolanaError)(error, kit_1.SOLANA_ERROR__RPC_SUBSCRIPTIONS__CHANNEL_CONNECTION_CLOSED) ||
            (0, kit_1.isSolanaError)(error, kit_1.SOLANA_ERROR__RPC__TRANSPORT_HTTP_ERROR)) {
            throw new Error('Solana backend connection failure. The backend might be inaccessible or the connection is unstable.');
        }
        if ((0, kit_1.isSolanaError)(error, kit_1.SOLANA_ERROR__JSON_RPC__SERVER_ERROR_SEND_TRANSACTION_PREFLIGHT_FAILURE) &&
            (0, kit_1.isSolanaError)(error.cause, kit_1.SOLANA_ERROR__TRANSACTION_ERROR__BLOCKHASH_NOT_FOUND)) {
            throw new Error('The transaction has expired because too much time passed between signing and sending. Please try again.');
        }
        if ((0, kit_1.isSolanaError)(error)) {
            throw new Error(`Solana error code: ${error.context.__code}. Please try again or contact support.`);
        }
        throw error;
    }
};
const getAccountInfo = async (request) => {
    const { payload } = request;
    const { details = 'basic' } = payload;
    const api = await request.connect();
    const publicKey = (0, kit_1.address)(payload.descriptor);
    const { value: accountInfo } = await api.rpc
        .getAccountInfo(publicKey, { encoding: 'base64' })
        .send();
    const tokenMetadata = await request.getTokenMetadata();
    const getAllTxIds = async (tokenAccountPubkeys) => {
        const sortedTokenAccountPubkeys = tokenAccountPubkeys.sort();
        const allAccounts = [payload.descriptor, ...sortedTokenAccountPubkeys];
        const allTxIds = details === 'txs' || details === 'txids'
            ? Array.from(new Set((await Promise.all(allAccounts.map(account => getAllSignatures(api, account))))
                .flat()
                .sort((a, b) => Number(b.slot - a.slot))
                .map(it => it.signature)))
            : [];
        return allTxIds;
    };
    const getEpoch = async () => {
        const cachedEpoch = await request.state.cache.get('epoch');
        if (cachedEpoch) {
            return cachedEpoch;
        }
        const deferred = (0, utils_1.createDeferred)();
        request.state.cache.set('epoch', deferred.promise, 3_600_000);
        const { epoch } = await api.rpc.getEpochInfo().send();
        deferred.resolve(Number(epoch));
        return deferred.promise;
    };
    if (details === 'txids') {
        const txids = await getAllTxIds(request.payload.tokenAccountsPubKeys || []);
        const solEpoch = await getEpoch();
        const account = {
            descriptor: payload.descriptor,
            balance: '0',
            availableBalance: '0',
            empty: txids.length === 0,
            history: {
                total: txids.length,
                unconfirmed: 0,
                txids,
            },
            misc: { solEpoch },
        };
        return {
            type: constants_1.RESPONSES.GET_ACCOUNT_INFO,
            payload: account,
        };
    }
    const getATAOwnerAddress = async (address) => {
        const { value: accountInfo } = await api.rpc
            .getAccountInfo(address, {
            encoding: 'jsonParsed',
        })
            .send();
        if (!accountInfo?.data || 'parsed' in accountInfo.data === false) {
            return address;
        }
        return accountInfo.data.parsed?.info?.owner ?? address;
    };
    const getTransactionPage = async (txIds, tokenAccountsInfos) => {
        if (txIds.length === 0) {
            return [];
        }
        const transactionsPage = await fetchTransactionPage(api, txIds);
        const page = transactionsPage
            .filter(isValidTransaction)
            .map(tx => blockchain_link_utils_1.solanaUtils.transformTransaction(tx, payload.descriptor, tokenAccountsInfos, tokenMetadata))
            .filter((tx) => !!tx);
        const transactions = await Promise.all(page.map(async (tx) => {
            const tokens = await Promise.all(tx.tokens.map(async (transfer) => {
                const from = transfer.from !== payload.descriptor
                    ? await getATAOwnerAddress((0, kit_1.address)(transfer.from))
                    : transfer.from;
                const to = transfer.to !== payload.descriptor
                    ? await getATAOwnerAddress((0, kit_1.address)(transfer.to))
                    : transfer.to;
                return {
                    ...transfer,
                    from,
                    to,
                };
            }));
            return { ...tx, tokens };
        }));
        return transactions;
    };
    const getTokenAccountsForProgram = (programPublicKey) => api.rpc
        .getTokenAccountsByOwner(publicKey, { programId: (0, kit_1.address)(programPublicKey) }, {
        encoding: 'jsonParsed',
    })
        .send();
    const tokenAccounts = (await Promise.all(Object.values(solana_1.tokenProgramsInfo).map(programInfo => getTokenAccountsForProgram(programInfo.publicKey))))
        .map(res => res.value)
        .flat();
    const recognisedWithBalance = tokenAccounts.filter(acc => {
        const info = acc.account.data.parsed?.info;
        const mint = info?.mint;
        const amount = info?.tokenAmount?.amount;
        return mint && tokenMetadata[mint] && amount !== '0';
    });
    const allAccounts = tokenAccounts.length > 100
        ? recognisedWithBalance.map(a => a.pubkey)
        : [payload.descriptor, ...recognisedWithBalance.map(a => a.pubkey)];
    const allTxIds = await getAllTxIds(allAccounts);
    const pageNumber = payload.page ? payload.page - 1 : 0;
    const pageSize = payload.pageSize || 5;
    const pageStartIndex = pageNumber * pageSize;
    const pageEndIndex = Math.min(pageStartIndex + pageSize, allTxIds.length);
    const txIdPage = allTxIds.slice(pageStartIndex, pageEndIndex);
    const tokenAccountsInfos = tokenAccounts.map(a => ({
        address: a.pubkey,
        mint: a.account.data.parsed?.info?.mint,
        decimals: a.account.data.parsed?.info?.tokenAmount?.decimals,
    }));
    const transactionPage = details === 'txs' ? await getTransactionPage(txIdPage, tokenAccountsInfos) : undefined;
    let tokens = [];
    if (tokenAccounts.length > 0) {
        tokens = (0, solana_1.transformTokenInfo)(tokenAccounts, tokenMetadata);
    }
    const { value: balance } = await api.rpc.getBalance(publicKey).send();
    let misc;
    if (!['basic', 'tokens'].includes(details)) {
        const solEpoch = await getEpoch();
        const solStakingAccounts = await (0, stakingAccounts_1.getSolanaStakingData)(api?.rpc, publicKey, solEpoch);
        misc = {
            solStakingAccounts,
            solEpoch,
        };
        if (accountInfo) {
            const [accountDataEncoded] = accountInfo.data;
            const accountDataBytes = (0, kit_1.getBase64Encoder)().encode(accountDataEncoded);
            const accountDataLength = BigInt(accountDataBytes.byteLength);
            const rent = await api.rpc.getMinimumBalanceForRentExemption(accountDataLength).send();
            misc.rent = Number(rent);
        }
    }
    const isAccountEmpty = !(allTxIds.length || balance || tokens.length);
    const account = {
        descriptor: payload.descriptor,
        balance: balance.toString(),
        availableBalance: balance.toString(),
        empty: isAccountEmpty,
        history: {
            total: allTxIds.length,
            unconfirmed: 0,
            transactions: transactionPage,
            txids: txIdPage,
        },
        page: transactionPage
            ? {
                total: allTxIds.length,
                index: pageNumber,
                size: transactionPage.length,
            }
            : undefined,
        tokens,
        misc: { ...misc, owner: accountInfo?.owner },
    };
    const workerAccount = request.state.getAccount(payload.descriptor);
    if (workerAccount) {
        request.state.addAccounts([{ ...workerAccount, tokens }]);
    }
    return {
        type: constants_1.RESPONSES.GET_ACCOUNT_INFO,
        payload: account,
    };
};
const getInfo = async (request, isTestnet) => {
    const api = await request.connect();
    const { value: { blockhash: blockHash, lastValidBlockHeight: blockHeight }, } = await api.rpc.getLatestBlockhash({ commitment: 'confirmed' }).send();
    const serverInfo = {
        testnet: isTestnet,
        blockHeight: Number(blockHeight),
        blockHash,
        shortcut: isTestnet ? 'dsol' : 'sol',
        network: isTestnet ? 'dsol' : 'sol',
        url: api.clusterUrl,
        name: 'Solana',
        version: '1',
        decimals: 9,
    };
    return {
        type: constants_1.RESPONSES.GET_INFO,
        payload: { ...serverInfo },
    };
};
const getAccountProgramSize = (programName) => ({
    staking: stakingAccounts_1.STAKE_ACCOUNT_V2_SIZE,
    'spl-token': (0, token_1.getTokenSize)(),
    'spl-token-2022': (0, token_2022_1.getTokenSize)(),
})[programName];
const estimateFee = async (request) => {
    const api = await request.connect();
    const { data: messageHex, newAccountProgramName } = request.payload.specific ?? {};
    if (messageHex == null) {
        throw new Error('Could not estimate fee for transaction.');
    }
    const transaction = (0, kit_1.pipe)(messageHex, (0, kit_1.getBase16Encoder)().encode, (0, kit_1.getTransactionDecoder)().decode);
    const message = (0, kit_1.pipe)(transaction.messageBytes, (0, kit_1.getCompiledTransactionMessageDecoder)().decode);
    const decompiledTransactionMessage = await (0, kit_1.decompileTransactionMessageFetchingLookupTables)(message, api.rpc);
    const priorityFee = await (0, fee_1.getPriorityFee)(api.rpc, decompiledTransactionMessage, message, transaction.signatures);
    const baseFee = await (0, fee_1.getBaseFee)(api.rpc, message);
    const accountCreationFee = newAccountProgramName
        ? await api.rpc
            .getMinimumBalanceForRentExemption(BigInt(getAccountProgramSize(newAccountProgramName)))
            .send()
        : BigInt(0);
    const payload = [
        {
            feePerTx: new utils_1.BigNumber(baseFee.toString())
                .plus(priorityFee.fee)
                .plus(accountCreationFee.toString())
                .toString(10),
            feePerUnit: priorityFee.computeUnitPrice,
            feeLimit: priorityFee.computeUnitLimit,
            feePayer: decompiledTransactionMessage.feePayer.address,
        },
    ];
    return {
        type: constants_1.RESPONSES.ESTIMATE_FEE,
        payload,
    };
};
const BLOCK_SUBSCRIBE_INTERVAL_MS = 50000;
const subscribeBlock = async ({ state, connect, post }) => {
    if (state.getSubscription('block'))
        return { subscribed: true };
    const api = await connect();
    const fetchBlock = async () => {
        const { value: { blockhash: blockHash, lastValidBlockHeight: blockHeight }, } = await api.rpc.getLatestBlockhash({ commitment: 'confirmed' }).send();
        if (blockHeight) {
            post({
                id: -1,
                type: constants_1.RESPONSES.NOTIFICATION,
                payload: {
                    type: 'block',
                    payload: {
                        blockHeight: Number(blockHeight),
                        blockHash,
                    },
                },
            });
        }
    };
    fetchBlock();
    const interval = setInterval(fetchBlock, BLOCK_SUBSCRIBE_INTERVAL_MS);
    state.addSubscription('block', interval);
    return { subscribed: true };
};
const unsubscribeBlock = ({ state }) => {
    if (!state.getSubscription('block'))
        return;
    const interval = state.getSubscription('block');
    clearInterval(interval);
    state.removeSubscription('block');
};
const extractTokenAccounts = (accounts, tokenMetadata) => accounts.flatMap(account => account.tokens?.flatMap(token => token.accounts
    ?.filter(tokenAccount => tokenAccount.balance !== '0' && tokenMetadata[token.contract])
    .map(tokenAccount => ({ descriptor: tokenAccount.publicKey })) || []) || []);
const findTokenAccountOwner = (accounts, accountDescriptor) => accounts.find(account => account.tokens?.find(token => token.accounts?.find(tokenAccount => tokenAccount.publicKey === accountDescriptor)));
let NEXT_ACCOUNT_SUBSCRIPTION_ID = 0;
const ACCOUNT_SUBSCRIPTION_ABORT_CONTROLLERS = new Map();
function abortSubscription(id) {
    const abortController = ACCOUNT_SUBSCRIPTION_ABORT_CONTROLLERS.get(id);
    ACCOUNT_SUBSCRIPTION_ABORT_CONTROLLERS.delete(id);
    abortController?.abort();
}
const handleAccountNotification = async (context, accountNotifications, account) => {
    const { connect, state, post, getTokenMetadata } = context;
    try {
        for await (const _ of accountNotifications) {
            const api = await connect();
            const [lastSignatureResponse] = await api.rpc
                .getSignaturesForAddress((0, kit_1.address)(account.descriptor), {
                limit: 1,
            })
                .send();
            const lastSignature = lastSignatureResponse?.signature;
            if (!lastSignature)
                return;
            const lastTx = await api.rpc
                .getTransaction(lastSignature, {
                encoding: 'jsonParsed',
                maxSupportedTransactionVersion: 0,
                commitment: 'confirmed',
            })
                .send();
            if (!lastTx || !isValidTransaction(lastTx)) {
                return;
            }
            const tokenMetadata = await getTokenMetadata();
            const tx = blockchain_link_utils_1.solanaUtils.transformTransaction(lastTx, account.descriptor, [], tokenMetadata);
            const descriptor = findTokenAccountOwner(state.getAccounts(), account.descriptor)?.descriptor ||
                account.descriptor;
            post({
                id: -1,
                type: constants_1.RESPONSES.NOTIFICATION,
                payload: {
                    type: 'notification',
                    payload: {
                        descriptor,
                        tx,
                    },
                },
            });
        }
    }
    catch (error) {
        if ((0, kit_1.isSolanaError)(error, kit_1.SOLANA_ERROR__RPC_SUBSCRIPTIONS__CHANNEL_CONNECTION_CLOSED)) {
            if (account.subscriptionId)
                abortSubscription(account.subscriptionId);
            state.removeAccounts([account]);
            context.onNetworkDisconnect();
        }
    }
};
const subscribeAccounts = async (context, accounts) => {
    const { connect, state } = context;
    const api = await connect();
    const subscribedAccounts = state.getAccounts();
    const tokenMetadata = await context.getTokenMetadata();
    const tokenAccounts = extractTokenAccounts(accounts, tokenMetadata);
    const newAccounts = [...accounts, ...tokenAccounts].filter(account => !subscribedAccounts.some(subscribedAccount => account.descriptor === subscribedAccount.descriptor));
    await Promise.all(newAccounts.map(async (a) => {
        const abortController = new AbortController();
        const accountNotifications = await api.rpcSubscriptions
            .accountNotifications((0, kit_1.address)(a.descriptor), { commitment: 'confirmed' })
            .subscribe({ abortSignal: abortController.signal });
        const subscriptionId = NEXT_ACCOUNT_SUBSCRIPTION_ID++;
        ACCOUNT_SUBSCRIPTION_ABORT_CONTROLLERS.set(subscriptionId, abortController);
        const account = {
            ...a,
            subscriptionId,
        };
        state.addAccounts([account]);
        handleAccountNotification(context, accountNotifications, account);
    }));
    return { subscribed: newAccounts.length > 0 };
};
const unsubscribeAccounts = ({ state }, accounts = []) => {
    const subscribedAccounts = state.getAccounts();
    accounts.forEach(a => {
        if (a.subscriptionId != null) {
            abortSubscription(a.subscriptionId);
            state.removeAccounts([a]);
        }
        a.tokens?.forEach(t => {
            t.accounts?.forEach(ta => {
                const tokenAccount = subscribedAccounts.find(sa => sa.descriptor === ta.publicKey);
                if (tokenAccount?.subscriptionId != null) {
                    abortSubscription(tokenAccount.subscriptionId);
                    state.removeAccounts([tokenAccount]);
                }
            });
        });
    });
};
const subscribe = async (request) => {
    let response;
    switch (request.payload.type) {
        case 'block':
            response = await subscribeBlock(request);
            break;
        case 'accounts':
            response = await subscribeAccounts(request, request.payload.accounts);
            break;
        default:
            throw new errors_1.CustomError('worker_unknown_request', `+${request.type}`);
    }
    return {
        type: constants_1.RESPONSES.SUBSCRIBE,
        payload: response,
    };
};
const unsubscribe = (request) => {
    switch (request.payload.type) {
        case 'block':
            unsubscribeBlock(request);
            break;
        case 'accounts': {
            unsubscribeAccounts(request, request.payload.accounts);
            break;
        }
        default:
            throw new errors_1.CustomError('worker_unknown_request', `+${request.type}`);
    }
    return {
        type: constants_1.RESPONSES.UNSUBSCRIBE,
        payload: { subscribed: request.state.getAccounts().length > 0 },
    };
};
const onRequest = (request, isTestnet) => {
    switch (request.type) {
        case constants_1.MESSAGES.GET_ACCOUNT_INFO:
            return getAccountInfo(request);
        case constants_1.MESSAGES.GET_INFO:
            return getInfo(request, isTestnet);
        case constants_1.MESSAGES.PUSH_TRANSACTION:
            return pushTransaction(request);
        case constants_1.MESSAGES.ESTIMATE_FEE:
            return estimateFee(request);
        case constants_1.MESSAGES.SUBSCRIBE:
            return subscribe(request);
        case constants_1.MESSAGES.UNSUBSCRIBE:
            return unsubscribe(request);
        default:
            throw new errors_1.CustomError('worker_unknown_request', `+${request.type}`);
    }
};
class SolanaWorker extends baseWorker_1.BaseWorker {
    isConnected(api) {
        return !!api;
    }
    lazyTokens = (0, utils_1.createLazy)(() => blockchain_link_utils_1.solanaUtils.getTokenMetadata());
    isTestnet = false;
    async tryConnect(url) {
        const clusterUrl = (0, kit_1.mainnet)(url);
        const transport = (0, kit_1.createDefaultRpcTransport)({
            url: clusterUrl,
            headers: {
                'User-Agent': `Trezor Suite ${(0, env_utils_1.getSuiteVersion)()}`,
            },
        });
        const throttledTransport = (0, getThrottledTransport_1.getThrottledTransport)(transport, THROTTLE_OPTIONS);
        const throttledRpc = (0, kit_1.createSolanaRpcFromTransport)(throttledTransport);
        const api = {
            clusterUrl,
            rpc: throttledRpc,
            rpcSubscriptions: (0, kit_1.createSolanaRpcSubscriptions)((0, kit_1.mainnet)(url.replace('http', 'ws'))),
        };
        this.isTestnet =
            (await api.rpc.getGenesisHash().send()) !==
                '5eykt4UsFv8P8NJdTREpY1vzqKqZKvdpKuc147dw2N9d';
        this.post({ id: -1, type: constants_1.RESPONSES.CONNECTED });
        return api;
    }
    async messageHandler(event) {
        try {
            if (await super.messageHandler(event))
                return true;
            const request = {
                ...event.data,
                connect: () => this.connect(),
                onNetworkDisconnect: () => {
                    if (this.api) {
                        this.post({
                            id: -1,
                            type: constants_1.RESPONSES.DISCONNECTED,
                            payload: true,
                        });
                    }
                    this.disconnect();
                },
                post: (data) => this.post(data),
                state: this.state,
                getTokenMetadata: this.lazyTokens.getOrInit,
            };
            const response = await onRequest(request, this.isTestnet);
            this.post({ id: event.data.id, ...response });
        }
        catch (error) {
            this.errorResponse(event.data.id, error);
        }
    }
    disconnect() {
        if (!this.api) {
            return;
        }
        this.state.getAccounts().forEach(a => {
            if (a.subscriptionId != null) {
                abortSubscription(a.subscriptionId);
            }
        });
        if (this.state.getSubscription('block')) {
            const interval = this.state.getSubscription('block');
            clearInterval(interval);
            this.state.removeSubscription('block');
        }
        this.api = undefined;
    }
}
function Solana() {
    return new SolanaWorker();
}
if (baseWorker_1.CONTEXT === 'worker') {
    const module = new SolanaWorker();
    onmessage = module.messageHandler.bind(module);
}
//# sourceMappingURL=index.js.map