"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.default = Ripple;
const tslib_1 = require("tslib");
const xrpl_1 = require("xrpl");
const constants_1 = require("@trezor/blockchain-link-types/lib/constants");
const errors_1 = require("@trezor/blockchain-link-types/lib/constants/errors");
const utils = tslib_1.__importStar(require("@trezor/blockchain-link-utils/lib/ripple"));
const env_utils_1 = require("@trezor/env-utils");
const bigNumber_1 = require("@trezor/utils/lib/bigNumber");
const baseWorker_1 = require("../baseWorker");
const DEFAULT_TIMEOUT = 20 * 1000;
const DEFAULT_PING_TIMEOUT = 3 * 60 * 1000;
const RESERVE = {
    BASE: '10000000',
    OWNER: '2000000',
};
const transformError = (error) => {
    if (error instanceof xrpl_1.XrplError) {
        const code = error.name === 'TimeoutError' ? 'websocket_timeout' : 'websocket_error_message';
        if (error.data) {
            const errorMessageData = error.data.error_message;
            const errorMessage = `${error.name} ${errorMessageData}`;
            return new errors_1.CustomError(code, errorMessage);
        }
        return new errors_1.CustomError(code, error.toString());
    }
    return error;
};
const getInfo = async (request) => {
    const client = await request.connect();
    const response = await client.request({
        command: 'server_info',
    });
    if (response.result.info.validated_ledger != null) {
        RESERVE.BASE = (0, xrpl_1.xrpToDrops)(response.result.info.validated_ledger.reserve_base_xrp);
        RESERVE.OWNER = (0, xrpl_1.xrpToDrops)(response.result.info.validated_ledger.reserve_inc_xrp);
    }
    return {
        type: constants_1.RESPONSES.GET_INFO,
        payload: {
            url: client.connection.getUrl(),
            ...utils.transformServerInfo(response),
        },
    };
};
const getMempoolAccountInfo = async (client, account) => {
    const response = await client.request({
        command: 'account_info',
        account,
        ledger_index: 'current',
        queue: true,
    });
    return {
        xrpBalance: response.result.account_data.Balance,
        sequence: response.result.account_data.Sequence,
        txs: response.result.queue_data ? response.result.queue_data.txn_count : 0,
    };
};
const getAccountInfo = async (request) => {
    const { payload } = request;
    const account = {
        descriptor: payload.descriptor,
        balance: '0',
        availableBalance: '0',
        empty: true,
        history: {
            total: -1,
            unconfirmed: 0,
            transactions: undefined,
        },
        misc: {
            sequence: 0,
            reserve: RESERVE.BASE,
        },
    };
    try {
        const client = await request.connect();
        const info = await client.request({
            command: 'account_info',
            account: payload.descriptor,
            ledger_index: 'validated',
        });
        const ownersReserve = info.result.account_data.OwnerCount > 0
            ? new bigNumber_1.BigNumber(info.result.account_data.OwnerCount).times(RESERVE.OWNER).toString()
            : '0';
        const reserve = new bigNumber_1.BigNumber(RESERVE.BASE).plus(ownersReserve).toString();
        const misc = {
            sequence: info.result.account_data.Sequence,
            reserve,
        };
        account.misc = misc;
        account.balance = info.result.account_data.Balance;
        account.availableBalance = new bigNumber_1.BigNumber(account.balance).minus(reserve).toString();
        account.empty = false;
    }
    catch (error) {
        if (error instanceof xrpl_1.XrplError && error.data?.error === 'actNotFound') {
            return {
                type: constants_1.RESPONSES.GET_ACCOUNT_INFO,
                payload: account,
            };
        }
        throw error;
    }
    try {
        const client = await request.connect();
        const mempoolInfo = await getMempoolAccountInfo(client, payload.descriptor);
        const { misc } = account;
        const reserve = misc && typeof misc.reserve === 'string' ? misc.reserve : RESERVE.BASE;
        account.availableBalance = new bigNumber_1.BigNumber(mempoolInfo.xrpBalance).minus(reserve).toString();
        account.misc.sequence = mempoolInfo.sequence;
        account.history.unconfirmed = mempoolInfo.txs;
    }
    catch {
    }
    if (payload.details !== 'txs') {
        return {
            type: constants_1.RESPONSES.GET_ACCOUNT_INFO,
            payload: account,
        };
    }
    const client = await request.connect();
    const response = await client.request({
        command: 'account_tx',
        account: payload.descriptor,
        ledger_index_min: payload.from ? payload.from : undefined,
        ledger_index_max: payload.to ? payload.to : undefined,
        limit: payload.pageSize || 25,
        marker: payload.marker,
        api_version: 2,
    });
    account.history.transactions = response.result.transactions.flatMap(raw => raw.tx_json != null
        ? [utils.transformTransaction(raw.hash, raw.tx_json, raw.meta, payload.descriptor)]
        : []);
    return {
        type: constants_1.RESPONSES.GET_ACCOUNT_INFO,
        payload: {
            ...account,
            marker: response.result.marker,
        },
    };
};
const getTransaction = async ({ connect, payload }) => {
    const client = await connect();
    const rawTx = await client.request({
        command: 'tx',
        transaction: payload,
        binary: false,
    });
    const tx = utils.transformTransaction(rawTx.result.hash, rawTx.result.tx_json, rawTx.result.meta);
    return {
        type: constants_1.RESPONSES.GET_TRANSACTION,
        payload: tx,
    };
};
const pushTransaction = async ({ connect, payload }) => {
    const client = await connect();
    const info = await client.submit(payload.hex.toUpperCase());
    if (info.result.engine_result === 'tesSUCCESS' && info.result.tx_json.hash) {
        return {
            type: constants_1.RESPONSES.PUSH_TRANSACTION,
            payload: info.result.tx_json.hash,
        };
    }
    throw new Error(info.result.engine_result_message);
};
const estimateFee = async (request) => {
    const client = await request.connect();
    const fee = await client.request({
        command: 'fee',
    });
    const drops = fee.result.drops.base_fee;
    const payload = request.payload && Array.isArray(request.payload.blocks)
        ? request.payload.blocks.map(() => ({ feePerUnit: drops }))
        : [{ feePerUnit: drops }];
    return {
        type: constants_1.RESPONSES.ESTIMATE_FEE,
        payload,
    };
};
const onNewBlock = ({ post }, event) => {
    post({
        id: -1,
        type: constants_1.RESPONSES.NOTIFICATION,
        payload: {
            type: 'block',
            payload: {
                blockHeight: event.ledger_index,
                blockHash: event.ledger_hash,
            },
        },
    });
};
const onTransaction = ({ state, post }, event) => {
    if (event.type !== 'transaction')
        return;
    if (event.tx_json?.TransactionType !== 'Payment')
        return;
    const { tx_json, hash } = event;
    const notify = (descriptor) => {
        if (!tx_json || !hash)
            return;
        post({
            id: -1,
            type: constants_1.RESPONSES.NOTIFICATION,
            payload: {
                type: 'notification',
                payload: {
                    descriptor,
                    tx: utils.transformTransaction(hash, tx_json, undefined, descriptor),
                },
            },
        });
    };
    const subscribed = state.getAddresses();
    const sent = subscribed.find(a => a === tx_json.Account);
    if (sent)
        notify(sent);
    const recv = subscribed.find(a => a === tx_json.Destination);
    if (recv)
        notify(recv);
};
const subscribeAccounts = async (ctx, accounts) => {
    const api = await ctx.connect();
    const { state } = ctx;
    const prevAddresses = state.getAddresses();
    state.addAccounts(accounts);
    const uniqueAddresses = state.getAddresses().filter(a => prevAddresses.indexOf(a) < 0);
    if (uniqueAddresses.length > 0) {
        if (!state.getSubscription('notification')) {
            api.on('transaction', ev => onTransaction(ctx, ev));
            state.addSubscription('notification');
        }
        await api.request({
            command: 'subscribe',
            accounts_proposed: uniqueAddresses,
        });
    }
    return { subscribed: state.getAddresses().length > 0 };
};
const subscribeAddresses = async (ctx, addresses) => {
    const api = await ctx.connect();
    const { state } = ctx;
    const uniqueAddresses = state.addAddresses(addresses);
    if (uniqueAddresses.length > 0) {
        if (!state.getSubscription('transaction')) {
            api.on('transaction', ev => onTransaction(ctx, ev));
            state.addSubscription('transaction');
        }
        await api.request({
            command: 'subscribe',
            accounts_proposed: uniqueAddresses,
        });
    }
    return { subscribed: state.getAddresses().length > 0 };
};
const subscribeBlock = async (ctx) => {
    if (!ctx.state.getSubscription('ledger')) {
        const api = await ctx.connect();
        api.on('ledgerClosed', ev => onNewBlock(ctx, ev));
        ctx.state.addSubscription('ledger');
    }
    return { subscribed: true };
};
const subscribe = async (request) => {
    const { payload } = request;
    let response;
    if (payload.type === 'accounts') {
        response = await subscribeAccounts(request, payload.accounts);
    }
    else if (payload.type === 'addresses') {
        response = await subscribeAddresses(request, payload.addresses);
    }
    else if (payload.type === 'block') {
        response = await subscribeBlock(request);
    }
    else {
        throw new errors_1.CustomError('invalid_param', '+type');
    }
    return {
        type: constants_1.RESPONSES.SUBSCRIBE,
        payload: response,
    };
};
const unsubscribeAddresses = async ({ state, connect }, addresses) => {
    const api = await connect();
    if (!addresses) {
        const all = state.getAddresses();
        state.removeAccounts(state.getAccounts());
        state.removeAddresses(all);
        await api.request({
            command: 'unsubscribe',
            accounts_proposed: all,
        });
    }
    else {
        state.removeAddresses(addresses);
        await api.request({
            command: 'unsubscribe',
            accounts_proposed: addresses,
        });
    }
    if (state.getAccounts().length < 1) {
        api.connection.removeAllListeners('transaction');
        state.removeSubscription('transaction');
    }
};
const unsubscribeAccounts = async (ctx, accounts) => {
    const { state } = ctx;
    const prevAddresses = state.getAddresses();
    state.removeAccounts(accounts || state.getAccounts());
    const addresses = state.getAddresses();
    const uniqueAddresses = prevAddresses.filter(a => addresses.indexOf(a) < 0);
    await unsubscribeAddresses(ctx, uniqueAddresses);
};
const unsubscribeBlock = async ({ state, connect }) => {
    if (!state.getSubscription('ledger'))
        return;
    const client = await connect();
    client.removeAllListeners('ledgerClosed');
    state.removeSubscription('ledger');
};
const unsubscribe = async (request) => {
    const { payload } = request;
    if (payload.type === 'accounts') {
        await unsubscribeAccounts(request, payload.accounts);
    }
    else if (payload.type === 'addresses') {
        await unsubscribeAddresses(request, payload.addresses);
    }
    else if (payload.type === 'block') {
        await unsubscribeBlock(request);
    }
    return {
        type: constants_1.RESPONSES.UNSUBSCRIBE,
        payload: { subscribed: request.state.getAddresses().length > 0 },
    };
};
const onRequest = (request) => {
    switch (request.type) {
        case constants_1.MESSAGES.GET_INFO:
            return getInfo(request);
        case constants_1.MESSAGES.GET_ACCOUNT_INFO:
            return getAccountInfo(request);
        case constants_1.MESSAGES.GET_TRANSACTION:
            return getTransaction(request);
        case constants_1.MESSAGES.ESTIMATE_FEE:
            return estimateFee(request);
        case constants_1.MESSAGES.PUSH_TRANSACTION:
            return pushTransaction(request);
        case constants_1.MESSAGES.SUBSCRIBE:
            return subscribe(request);
        case constants_1.MESSAGES.UNSUBSCRIBE:
            return unsubscribe(request);
        default:
            throw new errors_1.CustomError('worker_unknown_request', `+${request.type}`);
    }
};
class RippleWorker extends baseWorker_1.BaseWorker {
    pingTimeout;
    cleanup() {
        if (this.pingTimeout) {
            clearTimeout(this.pingTimeout);
        }
        if (this.api) {
            this.api.removeAllListeners();
        }
        super.cleanup();
    }
    isConnected(client) {
        return client?.isConnected() ?? false;
    }
    async tryConnect(url) {
        const options = {
            headers: {
                'User-Agent': `Trezor Suite ${(0, env_utils_1.getSuiteVersion)()}`,
            },
            timeout: this.settings.timeout || DEFAULT_TIMEOUT,
            connectionTimeout: this.settings.timeout || DEFAULT_TIMEOUT,
            ...(this.proxyAgent && { agent: this.proxyAgent }),
        };
        const client = new xrpl_1.Client(url, options);
        await client.connect();
        client.on('ledgerClosed', ledger => {
            RESERVE.BASE = ledger.reserve_base.toString();
            RESERVE.OWNER = ledger.reserve_inc.toString();
        });
        client.on('disconnected', () => {
            this.post({ id: -1, type: constants_1.RESPONSES.DISCONNECTED, payload: true });
            this.cleanup();
        });
        this.post({ id: -1, type: constants_1.RESPONSES.CONNECTED });
        client.request({
            command: 'subscribe',
            streams: ['ledger'],
        });
        return client;
    }
    async disconnect() {
        try {
            if (this.api?.isConnected()) {
                await this.api.disconnect();
            }
        }
        catch (error) {
            this.debug('Disconnect failed', error);
        }
        finally {
            this.cleanup();
        }
    }
    async messageHandler(event) {
        try {
            if (await super.messageHandler(event))
                return true;
            const request = {
                ...event.data,
                connect: () => this.connect(),
                post: (data) => this.post(data),
                state: this.state,
            };
            const response = await onRequest(request);
            this.post({ id: event.data.id, ...response });
        }
        catch (error) {
            this.errorResponse(event.data.id, transformError(error));
        }
        finally {
            if (event.data.type !== constants_1.MESSAGES.DISCONNECT) {
                this.setPingTimeout();
            }
        }
    }
    setPingTimeout() {
        if (this.pingTimeout) {
            clearTimeout(this.pingTimeout);
        }
        this.pingTimeout = this.api?.isConnected()
            ? setTimeout(() => this.onPing(), this.settings.pingTimeout || DEFAULT_PING_TIMEOUT)
            : undefined;
    }
    async onPing() {
        if (!this.api || !this.api.isConnected())
            return;
        if (this.state.hasSubscriptions() || this.settings.keepAlive) {
            try {
                await this.api.request({ command: 'ping' });
            }
            catch (error) {
                this.debug(`Error in timeout ping request: ${error}`);
            }
            this.setPingTimeout();
        }
        else {
            this.api.disconnect();
        }
    }
}
function Ripple() {
    return new RippleWorker();
}
if (baseWorker_1.CONTEXT === 'worker') {
    const module = new RippleWorker();
    onmessage = module.messageHandler.bind(module);
}
//# sourceMappingURL=index.js.map