"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.default = Blockbook;
const tslib_1 = require("tslib");
const constants_1 = require("@trezor/blockchain-link-types/lib/constants");
const errors_1 = require("@trezor/blockchain-link-types/lib/constants/errors");
const utils = tslib_1.__importStar(require("@trezor/blockchain-link-utils/lib/blockbook"));
const baseWorker_1 = require("../baseWorker");
const websocket_1 = require("./websocket");
const getInfo = async (request) => {
    const api = await request.connect();
    const info = await api.getServerInfo();
    return {
        type: constants_1.RESPONSES.GET_INFO,
        payload: {
            url: api.options.url,
            ...utils.transformServerInfo(info),
        },
    };
};
const getBlockHash = async (request) => {
    const api = await request.connect();
    const info = await api.getBlockHash(request.payload);
    return {
        type: constants_1.RESPONSES.GET_BLOCK_HASH,
        payload: info.hash,
    };
};
const getBlock = async (request) => {
    const api = await request.connect();
    const info = await api.getBlock(request.payload);
    return {
        type: constants_1.RESPONSES.GET_BLOCK,
        payload: info,
    };
};
const getAccountInfo = async (request) => {
    const { payload } = request;
    const api = await request.connect();
    const info = await api.getAccountInfo(payload);
    return {
        type: constants_1.RESPONSES.GET_ACCOUNT_INFO,
        payload: utils.transformAccountInfo(info),
    };
};
const getAccountUtxo = async (request) => {
    const { payload } = request;
    const api = await request.connect();
    const utxos = await api.getAccountUtxo(payload);
    return {
        type: constants_1.RESPONSES.GET_ACCOUNT_UTXO,
        payload: utils.transformAccountUtxo(utxos),
    };
};
const getAccountBalanceHistory = async (request) => {
    const { payload } = request;
    const api = await request.connect();
    const history = await api.getAccountBalanceHistory(payload);
    return {
        type: constants_1.RESPONSES.GET_ACCOUNT_BALANCE_HISTORY,
        payload: history,
    };
};
const getCurrentFiatRates = async (request) => {
    const { payload } = request;
    const api = await request.connect();
    const fiatRates = await api.getCurrentFiatRates(payload);
    return {
        type: constants_1.RESPONSES.GET_CURRENT_FIAT_RATES,
        payload: fiatRates,
    };
};
const getFiatRatesForTimestamps = async (request) => {
    const { payload } = request;
    const api = await request.connect();
    const { tickers } = await api.getFiatRatesForTimestamps(payload);
    return {
        type: constants_1.RESPONSES.GET_FIAT_RATES_FOR_TIMESTAMPS,
        payload: { tickers },
    };
};
const getFiatRatesTickersList = async (request) => {
    const { payload } = request;
    const api = await request.connect();
    const tickers = await api.getFiatRatesTickersList(payload);
    return {
        type: constants_1.RESPONSES.GET_FIAT_RATES_TICKERS_LIST,
        payload: {
            ts: tickers.ts,
            availableCurrencies: tickers.available_currencies,
        },
    };
};
const getTransaction = async (request) => {
    const api = await request.connect();
    const rawtx = await api.getTransaction(request.payload);
    const tx = utils.transformTransaction(rawtx);
    return {
        type: constants_1.RESPONSES.GET_TRANSACTION,
        payload: tx,
    };
};
const getTransactionHex = async (request) => {
    const api = await request.connect();
    const { hex } = await api.getTransaction(request.payload);
    if (!hex)
        throw new errors_1.CustomError(`Missing hex of ${request.payload}`);
    return {
        type: constants_1.RESPONSES.GET_TRANSACTION_HEX,
        payload: hex,
    };
};
const pushTransaction = async (request) => {
    const api = await request.connect();
    const { hex, disableAlternativeRPC } = request.payload;
    const resp = await api.pushTransaction(hex, disableAlternativeRPC);
    return {
        type: constants_1.RESPONSES.PUSH_TRANSACTION,
        payload: resp.result,
    };
};
const estimateFee = async (request) => {
    const api = await request.connect();
    const resp = await api.estimateFee(request.payload);
    return {
        type: constants_1.RESPONSES.ESTIMATE_FEE,
        payload: resp,
    };
};
const rpcCall = async (request) => {
    const api = await request.connect();
    const resp = await api.rpcCall(request.payload);
    return {
        type: constants_1.RESPONSES.RPC_CALL,
        payload: resp,
    };
};
const onNewBlock = ({ post }, event) => {
    post({
        id: -1,
        type: constants_1.RESPONSES.NOTIFICATION,
        payload: {
            type: 'block',
            payload: {
                blockHeight: event.height,
                blockHash: event.hash,
            },
        },
    });
};
const onMempoolTx = ({ post }, payload) => {
    post({
        id: -1,
        type: constants_1.RESPONSES.NOTIFICATION,
        payload: {
            type: 'mempool',
            payload,
        },
    });
};
const onTransaction = ({ state, post }, event) => {
    if (!event.tx)
        return;
    const descriptor = event.address;
    const account = state.getAccount(descriptor);
    post({
        id: -1,
        type: constants_1.RESPONSES.NOTIFICATION,
        payload: {
            type: 'notification',
            payload: {
                descriptor: account ? account.descriptor : descriptor,
                tx: account
                    ? utils.transformTransaction(event.tx, account.addresses ?? account.descriptor)
                    : utils.transformTransaction(event.tx, descriptor),
            },
        },
    });
};
const onNewFiatRates = ({ post }, event) => {
    post({
        id: -1,
        type: constants_1.RESPONSES.NOTIFICATION,
        payload: {
            type: 'fiatRates',
            payload: {
                rates: event.rates,
            },
        },
    });
};
const subscribeAccounts = async (ctx, accounts) => {
    const api = await ctx.connect();
    const { state } = ctx;
    state.addAccounts(accounts);
    if (!state.getSubscription('notification')) {
        api.on('notification', ev => onTransaction(ctx, ev));
        state.addSubscription('notification');
    }
    return api.subscribeAddresses(state.getAddresses());
};
const subscribeAddresses = async (ctx, addresses) => {
    const api = await ctx.connect();
    const { state } = ctx;
    state.addAddresses(addresses);
    if (!state.getSubscription('notification')) {
        api.on('notification', ev => onTransaction(ctx, ev));
        state.addSubscription('notification');
    }
    return api.subscribeAddresses(state.getAddresses());
};
const subscribeBlock = async (ctx) => {
    if (ctx.state.getSubscription('block'))
        return { subscribed: true };
    ctx.state.addSubscription('block');
    const api = await ctx.connect();
    api.on('block', ev => onNewBlock(ctx, ev));
    return api.subscribeBlock();
};
const subscribeFiatRates = async (ctx, currency) => {
    const api = await ctx.connect();
    if (!ctx.state.getSubscription('fiatRates')) {
        ctx.state.addSubscription('fiatRates');
        api.on('fiatRates', ev => onNewFiatRates(ctx, ev));
    }
    return api.subscribeFiatRates(currency);
};
const subscribeMempool = async (ctx) => {
    const api = await ctx.connect();
    if (!ctx.state.getSubscription('mempool')) {
        ctx.state.addSubscription('mempool');
        api.on('mempool', ev => onMempoolTx(ctx, ev));
    }
    return api.subscribeMempool();
};
const subscribe = async (request) => {
    const { payload } = request;
    let response;
    if (payload.type === 'accounts') {
        response = await subscribeAccounts(request, payload.accounts);
    }
    else if (payload.type === 'addresses') {
        response = await subscribeAddresses(request, payload.addresses);
    }
    else if (payload.type === 'block') {
        response = await subscribeBlock(request);
    }
    else if (payload.type === 'fiatRates') {
        response = await subscribeFiatRates(request, payload.currency);
    }
    else if (payload.type === 'mempool') {
        response = await subscribeMempool(request);
    }
    else {
        throw new errors_1.CustomError('invalid_param', '+type');
    }
    return {
        type: constants_1.RESPONSES.SUBSCRIBE,
        payload: response,
    };
};
const unsubscribeAccounts = async ({ state, connect }, accounts) => {
    state.removeAccounts(accounts || state.getAccounts());
    const api = await connect();
    const subscribed = state.getAddresses();
    if (subscribed.length < 1) {
        api.removeAllListeners('notification');
        state.removeSubscription('notification');
        return api.unsubscribeAddresses();
    }
    return api.subscribeAddresses(subscribed);
};
const unsubscribeAddresses = async ({ state, connect }, addresses) => {
    const api = await connect();
    if (!addresses) {
        state.removeAccounts(state.getAccounts());
    }
    const subscribed = state.removeAddresses(addresses || state.getAddresses());
    if (subscribed.length < 1) {
        api.removeAllListeners('notification');
        state.removeSubscription('notification');
        return api.unsubscribeAddresses();
    }
    return api.subscribeAddresses(subscribed);
};
const unsubscribeBlock = async ({ state, connect }) => {
    if (!state.getSubscription('block'))
        return { subscribed: false };
    const api = await connect();
    api.removeAllListeners('block');
    state.removeSubscription('block');
    return api.unsubscribeBlock();
};
const unsubscribeFiatRates = async ({ state, connect }) => {
    if (!state.getSubscription('fiatRates'))
        return { subscribed: false };
    const api = await connect();
    api.removeAllListeners('fiatRates');
    state.removeSubscription('fiatRates');
    return api.unsubscribeFiatRates();
};
const unsubscribeMempool = async ({ state, connect }) => {
    if (!state.getSubscription('mempool'))
        return { subscribed: false };
    const api = await connect();
    api.removeAllListeners('mempool');
    state.removeSubscription('mempool');
    return api.unsubscribeMempool();
};
const unsubscribe = async (request) => {
    const { payload } = request;
    let response;
    if (payload.type === 'accounts') {
        response = await unsubscribeAccounts(request, payload.accounts);
    }
    else if (payload.type === 'addresses') {
        response = await unsubscribeAddresses(request, payload.addresses);
    }
    else if (payload.type === 'block') {
        response = await unsubscribeBlock(request);
    }
    else if (payload.type === 'fiatRates') {
        response = await unsubscribeFiatRates(request);
    }
    else if (payload.type === 'mempool') {
        response = await unsubscribeMempool(request);
    }
    else {
        throw new errors_1.CustomError('invalid_param', '+type');
    }
    return {
        type: constants_1.RESPONSES.UNSUBSCRIBE,
        payload: response,
    };
};
const onRequest = (request) => {
    switch (request.type) {
        case constants_1.MESSAGES.GET_INFO:
            return getInfo(request);
        case constants_1.MESSAGES.GET_BLOCK_HASH:
            return getBlockHash(request);
        case constants_1.MESSAGES.GET_BLOCK:
            return getBlock(request);
        case constants_1.MESSAGES.GET_ACCOUNT_INFO:
            return getAccountInfo(request);
        case constants_1.MESSAGES.GET_ACCOUNT_UTXO:
            return getAccountUtxo(request);
        case constants_1.MESSAGES.GET_TRANSACTION:
            return getTransaction(request);
        case constants_1.MESSAGES.GET_TRANSACTION_HEX:
            return getTransactionHex(request);
        case constants_1.MESSAGES.GET_ACCOUNT_BALANCE_HISTORY:
            return getAccountBalanceHistory(request);
        case constants_1.MESSAGES.GET_CURRENT_FIAT_RATES:
            return getCurrentFiatRates(request);
        case constants_1.MESSAGES.GET_FIAT_RATES_FOR_TIMESTAMPS:
            return getFiatRatesForTimestamps(request);
        case constants_1.MESSAGES.GET_FIAT_RATES_TICKERS_LIST:
            return getFiatRatesTickersList(request);
        case constants_1.MESSAGES.ESTIMATE_FEE:
            return estimateFee(request);
        case constants_1.MESSAGES.RPC_CALL:
            return rpcCall(request);
        case constants_1.MESSAGES.PUSH_TRANSACTION:
            return pushTransaction(request);
        case constants_1.MESSAGES.SUBSCRIBE:
            return subscribe(request);
        case constants_1.MESSAGES.UNSUBSCRIBE:
            return unsubscribe(request);
        default:
            throw new errors_1.CustomError('worker_unknown_request', `+${request.type}`);
    }
};
class BlockbookWorker extends baseWorker_1.BaseWorker {
    cleanup() {
        if (this.api) {
            this.api.dispose();
            this.api.removeAllListeners();
        }
        super.cleanup();
    }
    isConnected(api) {
        return api?.isConnected() ?? false;
    }
    async tryConnect(url) {
        const { timeout, pingTimeout, keepAlive } = this.settings;
        const api = new websocket_1.BlockbookAPI({
            url,
            timeout,
            pingTimeout,
            keepAlive,
            agent: this.proxyAgent,
        });
        await api.connect();
        api.on('disconnected', () => {
            this.post({ id: -1, type: constants_1.RESPONSES.DISCONNECTED, payload: true });
            this.cleanup();
        });
        this.post({
            id: -1,
            type: constants_1.RESPONSES.CONNECTED,
        });
        return api;
    }
    disconnect() {
        if (this.api) {
            this.api.disconnect();
        }
    }
    async messageHandler(event) {
        try {
            if (await super.messageHandler(event))
                return true;
            const request = {
                ...event.data,
                connect: () => this.connect(),
                post: (data) => this.post(data),
                state: this.state,
            };
            const response = await onRequest(request);
            this.post({ id: event.data.id, ...response });
        }
        catch (error) {
            this.errorResponse(event.data.id, error);
        }
    }
}
function Blockbook() {
    return new BlockbookWorker();
}
if (baseWorker_1.CONTEXT === 'worker') {
    const module = new BlockbookWorker();
    onmessage = module.messageHandler.bind(module);
}
//# sourceMappingURL=index.js.map