import { Brand } from '@solana/nominal-types';
import { CompilableTransactionMessage } from '@solana/transaction-messages';
import { TransactionMessageWithSigners } from './account-signer-meta';
/**
 * Defines a transaction message with exactly one {@link TransactionSendingSigner}.
 *
 * This type is used to narrow the type of transaction messages that have been
 * checked to have exactly one sending signer.
 *
 * @example
 * ```ts
 * import { assertIsTransactionMessageWithSingleSendingSigner } from '@solana/signers';
 *
 * assertIsTransactionMessageWithSingleSendingSigner(transactionMessage);
 * transactionMessage satisfies TransactionMessageWithSingleSendingSigner;
 * ```
 *
 * @see {@link isTransactionMessageWithSingleSendingSigner}
 * @see {@link assertIsTransactionMessageWithSingleSendingSigner}
 */
export type TransactionMessageWithSingleSendingSigner = Brand<TransactionMessageWithSigners, 'TransactionMessageWithSingleSendingSigner'>;
/**
 * Checks whether the provided transaction has exactly one {@link TransactionSendingSigner}.
 *
 * This can be useful when using {@link signAndSendTransactionMessageWithSigners} to provide
 * a fallback strategy in case the transaction message cannot be send using this function.
 *
 * @typeParam TTransactionMessage - The inferred type of the transaction message provided.
 *
 * @example
 * ```ts
 * import {
 *     isTransactionMessageWithSingleSendingSigner,
 *     signAndSendTransactionMessageWithSigners,
 *     signTransactionMessageWithSigners,
 * } from '@solana/signers';
 * import { getBase64EncodedWireTransaction } from '@solana/transactions';
 *
 * let transactionSignature: SignatureBytes;
 * if (isTransactionMessageWithSingleSendingSigner(transactionMessage)) {
 *     transactionSignature = await signAndSendTransactionMessageWithSigners(transactionMessage);
 * } else {
 *     const signedTransaction = await signTransactionMessageWithSigners(transactionMessage);
 *     const encodedTransaction = getBase64EncodedWireTransaction(signedTransaction);
 *     transactionSignature = await rpc.sendTransaction(encodedTransaction).send();
 * }
 * ```
 *
 * @see {@link signAndSendTransactionMessageWithSigners}
 * @see {@link assertIsTransactionMessageWithSingleSendingSigner}
 */
export declare function isTransactionMessageWithSingleSendingSigner<TTransactionMessage extends CompilableTransactionMessage>(transaction: TTransactionMessage): transaction is TransactionMessageWithSingleSendingSigner & TTransactionMessage;
/**
 * Asserts that the provided transaction message has exactly one {@link TransactionSendingSigner}.
 *
 * This can be useful when using the {@link signAndSendTransactionMessageWithSigners} function
 * to ensure it will be able to select the correct signer to send the transaction.
 *
 * @typeParam TTransactionMessage - The inferred type of the transaction message provided.
 *
 * @example
 * ```ts
 * import {
 *     assertIsTransactionMessageWithSingleSendingSigner,
 *     signAndSendTransactionMessageWithSigners
 * } from '@solana/signers';
 *
 * assertIsTransactionMessageWithSingleSendingSigner(transactionMessage);
 * const transactionSignature = await signAndSendTransactionMessageWithSigners(transactionMessage);
 * ```
 *
 * @see {@link signAndSendTransactionMessageWithSigners}
 * @see {@link isTransactionMessageWithSingleSendingSigner}
 */
export declare function assertIsTransactionMessageWithSingleSendingSigner<TTransactionMessage extends CompilableTransactionMessage>(transaction: TTransactionMessage): asserts transaction is TransactionMessageWithSingleSendingSigner & TTransactionMessage;
//# sourceMappingURL=transaction-with-single-sending-signer.d.ts.map