"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BlePlxManager = void 0;
const react_native_ble_plx_1 = require("react-native-ble-plx");
const awaitsBleOn_1 = require("./awaitsBleOn");
const devices_1 = require("@ledgerhq/devices");
class BlePlxManager {
    /**
     * Returns the instance of the Bluetooth Low Energy Manager. It initializes it only
     * when it's first needed, preventing the permission prompt happening prematurely.
     * Important: Do NOT access the _bleManager variable directly.
     * Use this function instead.
     * @returns {BleManager} - The instance of the BleManager.
     */
    static _instance;
    static get instance() {
        if (!this._instance) {
            this._instance = new react_native_ble_plx_1.BleManager();
        }
        return this._instance;
    }
    static waitOn() {
        return (0, awaitsBleOn_1.awaitsBleOn)(BlePlxManager.instance);
    }
    static async getKnownDevice(identifier) {
        const devices = await this.instance.devices([identifier]);
        return devices[0];
    }
    static getConnectedDevices() {
        return this.instance.connectedDevices((0, devices_1.getBluetoothServiceUuids)());
    }
    static connect(identifier, options = {}) {
        return this.instance.connectToDevice(identifier, options);
    }
    /**
     * Exposed method from the ble-plx library
     * Sets new log level for native module's logging mechanism.
     * @param logLevel
     */
    static async setLogLevel(logLevel) {
        if (Object.values(react_native_ble_plx_1.LogLevel).includes(logLevel)) {
            await this.instance.setLogLevel(logLevel);
        }
        else {
            throw new Error(`${logLevel} is not a valid LogLevel`);
        }
    }
    static onStateChange(listener, emitCurrentState) {
        return this.instance.onStateChange(listener, emitCurrentState);
    }
    static async startScan(callback) {
        await this.instance.startDeviceScan((0, devices_1.getBluetoothServiceUuids)(), null, (error, device) => {
            callback(error, device);
        });
    }
    static async stopScan() {
        await this.instance.stopDeviceScan();
    }
    static async disconnectDevice(deviceIdentifier) {
        await this.instance.cancelDeviceConnection(deviceIdentifier);
    }
    static async cancelTransaction(transactionId) {
        await this.instance.cancelTransaction(transactionId);
    }
}
exports.BlePlxManager = BlePlxManager;
//# sourceMappingURL=BlePlxManager.js.map