/// <reference types="node" />
/// <reference types="node" />
import type { Observer as TransportObserver, Subscription as TransportSubscription } from "@ledgerhq/hw-transport";
import Transport from "@ledgerhq/hw-transport";
import { Characteristic, Device, DeviceId, Subscription } from "react-native-ble-plx";
import type { DeviceModel } from "@ledgerhq/devices";
import { LocalTracer, TraceContext } from "@ledgerhq/logs";
import { Observable, Observer, SchedulerLike } from "rxjs";
import { HwTransportError } from "@ledgerhq/errors";
import { ReconnectionConfig } from "./types";
import { BlePlxManager } from "./BlePlxManager";
export declare const setReconnectionConfig: (config: ReconnectionConfig | null | undefined) => void;
/**
 * react-native bluetooth BLE implementation
 * @example
 * import BleTransport from "@ledgerhq/react-native-hw-transport-ble";
 */
export default class BleTransport extends Transport {
    static disconnectTimeoutMs: number;
    /**
     *
     */
    static isSupported: () => Promise<boolean>;
    /**
     *
     */
    static list: () => Promise<void[]>;
    static setLogLevel: typeof BlePlxManager.setLogLevel;
    /**
     * Listen to state changes on the BlePlxManagerInstance and notify the
     * specified observer.
     * @param observer
     * @returns TransportSubscription
     */
    static observeState(observer: Observer<{
        type: string;
        available: boolean;
    }>): TransportSubscription;
    static safeRemove: (sub: Subscription, tracer: LocalTracer) => void;
    /**
     * Scan for bluetooth Ledger devices
     * @param observer Device is partial in order to avoid the live-common/this dep
     * @param context
     * @returns TransportSubscription
     */
    static listen(observer: TransportObserver<any, HwTransportError>, context?: TraceContext): TransportSubscription;
    /**
     * Opens a BLE transport
     *
     * @param {Device | string} deviceOrId
     * @param timeoutMs Applied when trying to connect to a device
     * @param context An optional context object for log/tracing strategy
     * @param injectedDependencies Contains optional injected dependencies used by the transport implementation
     *  - rxjsScheduler: dependency injected RxJS scheduler to control time. Default AsyncScheduler.
     */
    static open(deviceOrId: Device | string, timeoutMs?: number, context?: TraceContext, { rxjsScheduler }?: {
        rxjsScheduler?: SchedulerLike;
    }): Promise<BleTransport>;
    /**
     * Exposes method from the ble-plx library to disconnect a device
     *
     * Disconnects from {@link Device} if it's connected or cancels pending connection.
     * A "disconnect" event will normally be emitted by the ble-plx lib once the device is disconnected.
     * Errors are logged but silenced.
     */
    static disconnectDevice: (id: DeviceId, context?: TraceContext) => Promise<void>;
    device: Device;
    deviceModel: DeviceModel;
    disconnectTimeout: null | ReturnType<typeof setTimeout>;
    id: string;
    isConnected: boolean;
    mtuSize: number;
    notifyObservable: Observable<Buffer | Error>;
    notYetDisconnected: boolean;
    writableWithResponseCharacteristic: Characteristic;
    writableWithoutResponseCharacteristic: Characteristic | undefined;
    rxjsScheduler?: SchedulerLike;
    currentTransactionIds: Array<string>;
    /**
     * The static `open` function is used to handle BleTransport instantiation
     *
     * @param device
     * @param writableWithResponseCharacteristic A BLE characteristic that we can write on,
     *   and that will be acknowledged in response from the device when it receives the written value.
     * @param writableWithoutResponseCharacteristic A BLE characteristic that we can write on,
     *   and that will not be acknowledged in response from the device
     * @param notifyObservable A multicast observable that emits messages received from the device
     * @param deviceModel
     * @param params Contains optional options and injected dependencies used by the transport implementation
     *  - abortTimeoutMs: stop the exchange after a given timeout. Another timeout exists
     *    to detect unresponsive device (see `unresponsiveTimeout`). This timeout aborts the exchange.
     *  - rxjsScheduler: dependency injected RxJS scheduler to control time. Default: AsyncScheduler.
     */
    constructor(device: Device, writableWithResponseCharacteristic: Characteristic, writableWithoutResponseCharacteristic: Characteristic | undefined, notifyObservable: Observable<Buffer | Error>, deviceModel: DeviceModel, { context, rxjsScheduler }?: {
        context?: TraceContext;
        rxjsScheduler?: SchedulerLike;
    });
    /**
     * A message exchange (APDU request <-> response) with the device that can be aborted
     *
     * The message will be BLE-encoded/framed before being sent, and the response will be BLE-decoded.
     *
     * @param message A buffer (u8 array) of a none BLE-encoded message (an APDU for ex) to be sent to the device
     *   as a request
     * @param options Contains optional options for the exchange function
     *  - abortTimeoutMs: stop the exchange after a given timeout. Another timeout exists
     *    to detect unresponsive device (see `unresponsiveTimeout`). This timeout aborts the exchange.
     * @returns A promise that resolves with the response data from the device.
     */
    exchange: (message: Buffer, { abortTimeoutMs }?: {
        abortTimeoutMs?: number | undefined;
    }) => Promise<Buffer>;
    /**
     * Tries to cancel all operations that have a recorded transaction and are pending
     *
     * Cancelling transaction which doesn't exist is ignored.
     *
     * Note: cancelling `writableWithoutResponseCharacteristic.write...` will throw a `BleError` with code `OperationCancelled`
     * but this error should be ignored. (In `exchange` our observable is unsubscribed before `cancelPendingOperations`
     * is called so the error is ignored)
     */
    private cancelPendingOperations;
    /**
     * Sets the collection of current transaction ids to an empty array
     */
    private clearCurrentTransactionIds;
    /**
     * Negotiate with the device the maximum transfer unit for the ble frames
     * @returns Promise<number>
     */
    inferMTU(): Promise<number>;
    /**
     * Exposed method from the ble-plx library
     * Request the connection priority for the given device.
     * @param {"Balanced" | "High" | "LowPower"} connectionPriority: Connection priority.
     * @returns {Promise<Device>} Connected device.
     */
    requestConnectionPriority(connectionPriority: "Balanced" | "High" | "LowPower"): Promise<Device>;
    /**
     * Do not call this directly unless you know what you're doing. Communication
     * with a Ledger device should be through the {@link exchange} method.
     *
     * For each call a transaction id is added to the current stack of transaction ids.
     * With this transaction id, a pending BLE communication operations can be cancelled.
     * Note: each frame/packet of a longer BLE-encoded message to be sent should have their unique transaction id.
     *
     * @param buffer BLE-encoded packet to send to the device
     * @param frameId Frame id to make `write` aware of a bigger message that this frame/packet is part of.
     *  Helps creating related a collection of transaction ids
     */
    write: (buffer: Buffer) => Promise<void>;
    /**
     * We intentionally do not immediately close a transport connection.
     * Instead, we queue the disconnect and wait for a future connection to dismiss the event.
     * This approach prevents unnecessary disconnects and reconnects. We use the isConnected
     * flag to ensure that we do not trigger a disconnect if the current cached transport has
     * already been disconnected.
     * @returns {Promise<void>}
     */
    close(): Promise<void>;
}
//# sourceMappingURL=BleTransport.d.ts.map