'use strict';

var bip68 = require('bip68');

// Copyright (c) 2022 Jose-Luis Landabaso - https://bitcoinerlab.com

const ABSOLUTE = 'ABSOLUTE';
const RELATIVE = 'RELATIVE';

/**
 * Calculates the maximum lock time value between two lock time values (a and b)
 * using the specified lock type (ABSOLUTE or RELATIVE).
 * It asserts that there are no timeLock mixings.
 * It asserts that a or b are correctly encoded in bip68 format if they are
 * RELATIVE.
 * Either both a and b are block based or time based.
 * https://medium.com/blockstream/dont-mix-your-timelocks-d9939b665094
 *
 * If only a or b is undefined it asserts the value and returns it.
 * If none are defined it returns undefined.
 * @param {string|number} [a] - The first lock time value to compare.
 * @param {string|number} [b] - The second lock time value to compare.
 * @param {string} lockType - The type of lock time to use. Can be either "ABSOLUTE" or "RELATIVE".
 * @return {number} - The maximum lock time value between a and b.
 */

function maxLock(a, b, lockType) {
  if (typeof a === 'undefined' && typeof b === 'undefined') {
    return undefined;
  }
  if (typeof lockType === 'undefined')
    throw new Error('lockType must be specified');
  // Check that lockType is either "ABSOLUTE" or "RELATIVE"
  if (lockType !== ABSOLUTE && lockType !== RELATIVE)
    throw new Error('lockType must be either "ABSOLUTE" or "RELATIVE"');

  function isInteger(number) {
    const isNumeric = !isNaN(number) && !isNaN(parseFloat(number));
    if (isNumeric && Number.isInteger(Number(number))) return true;
    else return false;
  }
  if (typeof a !== 'undefined') {
    if (isInteger(a) === false)
      throw new Error('nSequence/nLockTime must be an integer: ' + a);
    a = Number(a);
    if (
      lockType === RELATIVE &&
      !bip68.decode(a).hasOwnProperty('seconds') &&
      !bip68.decode(a).hasOwnProperty('blocks')
    )
      throw new Error('Invalid bip68 encoded a value: ' + a);
  }
  if (typeof b !== 'undefined') {
    if (isInteger(b) === false)
      throw new Error('nSequence/nLockTime must be an integer: ' + b);
    b = Number(b);
    if (
      lockType === RELATIVE &&
      !bip68.decode(b).hasOwnProperty('seconds') &&
      !bip68.decode(b).hasOwnProperty('blocks')
    )
      throw new Error('Invalid bip68 encoded b value: ' + b);
  }

  if (typeof a !== 'undefined' && typeof b !== 'undefined') {
    if (lockType === ABSOLUTE) {
      // Both a and b must be either below 500000000 or both above or equal 500000000
      if (
        (a < 500000000 && b >= 500000000) ||
        (a >= 500000000 && b < 500000000)
      ) {
        throw new Error(
          'nLockTime values must be either below 500000000 or both above or equal 500000000'
        );
      }
    } else {
      const decodedA = bip68.decode(a);
      const decodedB = bip68.decode(b);

      if (
        decodedA.hasOwnProperty('seconds') !==
        decodedB.hasOwnProperty('seconds')
      ) {
        throw new Error(
          'a and b must both be either represent seconds or block height'
        );
      }
    }
    return Math.max(a, b);
  }

  if (typeof a !== 'undefined') return a;
  if (typeof b !== 'undefined') return b;
  return undefined;
}

// Copyright (c) 2022 Jose-Luis Landabaso - https://bitcoinerlab.com

/**
 * Given a `solutionTemplate`, such as "0 dsat(X) sat(Y) 1 sat(Z)", and given the
 * sat/dissatisfactions for X, Y, Z,... (comprised in `satisfactionsMap`) it
 * computes all the possible combination of solutions returned in an array
 * of {@link Solution}.
 *
 * Terminology:
 *
 * A solution of type {@link Solution} is an object
 * that contains an unlocking witness for a
 * miniscript fragment.
 *
 * When matching an unlocking witness with a locking
 * script, they can produce True or False. These solutions are called sats and
 * dsats, respectively.
 *
 * For example, take this solution: `sol = {asm:"sig key"}`.
 *
 * When matched with `DUP HASH160 <HASH160(key)> EQUALVERIFY CHECKSIG` it
 * produces `True`, that is, a valid sat solution: `<sig> <key> DUP HASH160 <HASH160(key)> EQUALVERIFY CHECKSIG`.
 *
 * A solution object can also contain other associated pieces of information: `solution: {asm, nLockTime, nSequence}`:
 *   -  nLockTime, nSequence: (number/str) interlock attached to this transaction
 *
 * A `solutionTemplate` describes a solution using sats and dsats of subexpressions.
 *
 * F.ex., this is a `solutionTemplate = "0 dsat(X) sat(Y) 1 sat(Z)"`.
 *
 * satisfactions is an object of type {@link Satisfactions}
 * that comprises both sat and dsat solutions for a
 * miniscript fragment:
 *
 *   `satisfactions = {sats: [sol_s1, sol_s2, ...], dsats: [sol_d1, ...]}`
 *
 * satisfactionsMap is a group of satisfactions:
 * ```javascript
 *   satisfactionsMap = { X: satisfactions_X, Y: satisfactions_Y, ...} =
 *   satisfactionsMap = {
 *                     X: {
 *                         sats: [sol_s1, sol_s2, ...],
 *                         dsats:[...]
 *                     },
 *                     Y: {sats: [...], dsats: [...]},
 *                     ...}
 * ```
 *
 * @param {string} solutionTemplate - a string containing sat or dsat expressions such as: "0 dsat(X) sat(Y) 1 sat(Z)"
 * @param {Object} satisfactionsMap - an object mapping the arguments for the sat and dsat expressions
 * in `solutionTemplate` (f.ex: `X, Y, Z`) to their {@link Satisfactions}.
 * @param {Satisfactions} [satisfactionsMap.X] - The satisfactions for `X`
 * @param {Satisfactions} [satisfactionsMap.Y] - The satisfactions for `Y`
 * @param {Satisfactions} [satisfactionsMap.Z] - The satisfactions for `Z`
 * @param {Satisfactions} [satisfactionsMap....] - The satisfactions for `...`
 *
 * @returns {Solution[]} an array of solutions, containing the resulting witness, nSequence and nLockTime values, and whether the solution has a HASSIG marker or should be marked as "DONTUSE".
 */

function combine(solutionTemplate, satisfactionsMap) {
  //First, break solutionTemplate into 3 parts:
  //  pre + curr + post,
  //where curr is the first sat(...)/dsat(...) in solutionTemplate.
  //
  //For example, in "0 dsat(X) sat(Y) 1 sat(Z)":
  //  pre= "0 "; curr = "dsat(X)"; post: " sat(Y) sat(Z)"
  //
  //Recursively call combine(post) until there are no further
  //sat()/dsat() in *post*

  //regExp that matches the first dsat(...) or sat(...):
  const reCurr = /d?sat\(([^\(]*)\)/;
  const currMatch = solutionTemplate.match(reCurr);

  if (currMatch && currMatch.length) {
    //The array of solutions to be computed and returned.
    const solutions = [];

    //curr is the first d?sat() matched in solutionTemplate:
    const curr = currMatch[0];
    //pre is whatever was before the first d?sat():
    const pre = solutionTemplate.split(curr)[0];
    //post is whatever was after the first d?sat():
    const post = solutionTemplate.slice(curr.length + pre.length);

    //the argument for curr: d?sat( -> argument <- ):
    //This will match the string "X" in the example above:
    //currArg = "X" for solutionTemplate "0 dsat(X) sat(Y) 1 sat(Z)"
    const currArg = currMatch[1];
    //currKey = "sats" or "dsats". "dsats" for the example above.
    const currKey = curr[0] === 'd' ? 'dsats' : 'sats';

    if (typeof satisfactionsMap[currArg] !== 'object')
      throw new Error(
        `satisfactionsMap does not provide sats/dsats solutions for argument ${currArg}, evaluating: ${solutionTemplate}`
      );
    const currSolutions = satisfactionsMap[currArg][currKey] || [];
    for (const currSolution of currSolutions) {
      //Does *post* contain further sat() or dsat() expressions?
      if (post.match(reCurr)) {
        //There are more sat/dsat, do a recursive call:
        const postSolutions = combine(post, satisfactionsMap);
        for (const postSolution of postSolutions) {
          //if ((currSolution.nLockTime && postSolution.nLockTime)) return [];
          solutions.push({
            nSequence: maxLock(
              currSolution.nSequence,
              postSolution.nSequence,
              RELATIVE
            ),
            nLockTime: maxLock(
              currSolution.nLockTime,
              postSolution.nLockTime,
              ABSOLUTE
            ),
            asm: `${pre}${currSolution.asm}${postSolution.asm}`
          });
        }
      } else {
        //This was the last instance of combine where there are no *post*
        //sat/dsats
        solutions.push({
          ...currSolution,
          asm: `${pre}${currSolution.asm}${post}`
        });
      }
    }
    //markDontUseSolutions(solutions);

    return solutions;
  }
  throw new Error('Invalid solutionTemplate: ' + solutionTemplate);
}

/**
 * An object containing functions for generating Basic satisfaction and dissatisfaction sets for miniscripts.
 *
 * @see {@link https://bitcoin.sipa.be/miniscript/}
 * @typedef {Object} SatisfactionsMaker
 *
 */
const satisfactionsMaker = {
  0: () => ({
    dsats: [{ asm: `` }]
  }),
  1: () => ({
    sats: [{ asm: `` }]
  }),
  pk_k: key => ({
    dsats: [{ asm: `0` }],
    sats: [{ asm: `<sig(${key})>` }]
  }),
  pk_h: key => ({
    dsats: [{ asm: `0 <${key}>` }],
    sats: [{ asm: `<sig(${key})> <${key}>` }]
  }),
  older: n => ({
    sats: [{ asm: ``, nSequence: n }]
  }),
  after: n => ({
    sats: [{ asm: ``, nLockTime: n }]
  }),
  sha256: h => ({
    sats: [{ asm: `<sha256_preimage(${h})>` }],
    dsats: [{ asm: `<random_preimage()>` }]
  }),
  ripemd160: h => ({
    sats: [{ asm: `<ripemd160_preimage(${h})>` }],
    dsats: [{ asm: `<random_preimage()>` }]
  }),
  hash256: h => ({
    sats: [{ asm: `<hash256_preimage(${h})>` }],
    dsats: [{ asm: `<random_preimage()>` }]
  }),
  hash160: h => ({
    sats: [{ asm: `<hash160_preimage(${h})>` }],
    dsats: [{ asm: `<random_preimage()>` }]
  }),
  andor: (X, Y, Z) => ({
    dsats: [
      ...combine(`dsat(Z) dsat(X)`, { X, Y, Z }),
      ...combine(`dsat(Y) sat(X)`, { X, Y, Z })
    ],
    sats: [
      ...combine('sat(Y) sat(X)', { X, Y, Z }),
      ...combine('sat(Z) dsat(X)', { X, Y, Z })
    ]
  }),
  and_v: (X, Y) => ({
    dsats: [...combine(`dsat(Y) sat(X)`, { X, Y })],
    sats: [...combine(`sat(Y) sat(X)`, { X, Y })]
  }),
  and_b: (X, Y) => ({
    dsats: [
      ...combine(`dsat(Y) dsat(X)`, { X, Y }),
      //https://bitcoin.sipa.be/miniscript/
      //The non-canonical options for and_b, or_b, and thresh are always
      //overcomplete (reason 3), so instead use DONTUSE there
      ...combine(`sat(Y) dsat(X)`, { X, Y }),
      ...combine(`dsat(Y) sat(X)`, { X, Y })
    ],
    sats: [...combine(`sat(Y) sat(X)`, { Y, X })]
  }),
  or_b: (X, Z) => ({
    dsats: [...combine(`dsat(Z) dsat(X)`, { X, Z })],
    sats: [
      ...combine(`dsat(Z) sat(X)`, { X, Z }),
      ...combine(`sat(Z) dsat(X)`, { X, Z }),
      //https://bitcoin.sipa.be/miniscript/
      //The non-canonical options for and_b, or_b, and thresh are always
      //overcomplete (reason 3), so instead use DONTUSE there
      ...combine(`sat(Z) sat(X)`, { X, Z })
    ]
  }),
  or_c: (X, Z) => ({
    sats: [
      ...combine(`sat(X)`, { X, Z }),
      ...combine(`sat(Z) dsat(X)`, { X, Z })
    ]
  }),
  or_d: (X, Z) => ({
    dsats: [...combine(`dsat(Z) dsat(X)`, { X, Z })],
    sats: [
      ...combine(`sat(X)`, { X, Z }),
      ...combine(`sat(Z) dsat(X)`, { X, Z })
    ]
  }),
  or_i: (X, Z) => ({
    dsats: [
      ...combine(`dsat(X) 1`, { X, Z }),
      ...combine(`dsat(Z) 0`, { X, Z })
    ],
    sats: [...combine(`sat(X) 1`, { X, Z }), ...combine(`sat(Z) 0`, { X, Z })]
  }),
  /*
   * This is not entirely trivial from the docs
   * (https://bitcoin.sipa.be/miniscript/),
   *  but solution templates for thresh must be written in reverse order.
   *  For example the first dsat, which is "All dsats", corresponds to:
   *  "<DSAT_N> <DSAT_N-1> ... <DSAT_1>" (note the reverse order in N)
   *  While this is not entirely trivial by reading the document
   *  it can be deduced by analyzing the script:
   *  thresh(k,X1,...,Xn)	[X1] [X2] ADD ... [Xn] ADD ... <k> EQUAL
   */
  thresh: (k, ...satisfactionsArray) => {
    if (Number.isInteger(Number(k)) && Number(k) > 0) k = Number(k);
    else throw new Error(`k must be positive number: ${k}`);

    //First, convert input satisfactions (which are dynamic for thresh
    //and multi) into an object.
    //For example, if input was, thresh(k, X, Y, Z), then
    //create an object like this: satisfactionsMap = {X, Y, Z};
    const satisfactionsMap = {};
    const N = satisfactionsArray.length;
    satisfactionsArray.map((satisfactions, index) => {
      satisfactionsMap[index] = satisfactions;
    });

    const dsats = [];
    const sats = [];
    //Push the canonical dsat (All dsats):
    //"<DSAT_N> <DSAT_N-1> ... <DSAT_1>" (note the reverse order)
    dsats.push(
      ...combine(
        Object.keys(satisfactionsMap)
          .map(mapKeyName => `dsat(${mapKeyName})`)
          .reverse()
          .join(' '),
        satisfactionsMap
      )
    );

    const dsatsNonCanSolutionTemplates = []; //Sats/dsats with 1 ≤ #(sats) ≠ k
    const satsSolutionTemplates = []; //Sats/dsats with #(sats) = k
    for (let i = 1; i < 1 << N; i++) {
      const c = [];
      let totalSatisfactions = 0;
      for (let j = 0; j < N; j++) {
        if (i & (1 << j)) totalSatisfactions++;
        c.push(i & (1 << j) ? `sat(${j})` : `dsat(${j})`);
      }
      if (totalSatisfactions !== k)
        dsatsNonCanSolutionTemplates.push(c.reverse().join(' '));
      else satsSolutionTemplates.push(c.reverse().join(' '));
    }

    //Push the non canonical dsats:
    for (const solutionTemplate of dsatsNonCanSolutionTemplates) {
      //https://bitcoin.sipa.be/miniscript/
      //The non-canonical options for and_b, or_b, and thresh are always
      //overcomplete (reason 3), so instead use DONTUSE there
      dsats.push(...combine(solutionTemplate, satisfactionsMap));
    }

    //Push the sats (which all are canonical):
    for (const solutionTemplate of satsSolutionTemplates) {
      sats.push(...combine(solutionTemplate, satisfactionsMap));
    }

    return { dsats, sats };
  },
  multi: (k, ...keys) => {
    if (Number.isInteger(Number(k)) && Number(k) > 0) k = Number(k);
    else throw new Error(`k must be positive number: ${k}`);

    //Example of a multi-sig locking script:
    //OP_3 <pubKey1> <pubKey2> <pubKey3> <pubKey4> OP_4 OP_CHECKMULTISIG
    //unlockingScript: OP_0 <sig1> <sig2> <sig4>
    //OP_0 is a dummy OP, needed because of a bug in Bitcoin
    if (typeof k !== 'number') throw new Error('k must be a number:' + k);
    if (k === 0) throw new Error('k cannot be 0');
    const dsats = [{ asm: '0 '.repeat(k + 1).trim() }];

    // Create all combinations of k signatures
    function keyCombinations(keys, k) {
      if (k === 0) {
        return [[]];
      }

      const combinations = [];

      for (let i = 0; i < keys.length; i++) {
        const remainingKeys = keys.slice(i + 1);
        const subCombinations = keyCombinations(remainingKeys, k - 1);
        subCombinations.forEach(combination => {
          combinations.push([keys[i], ...combination]);
        });
      }

      return combinations;
    }

    const asms = [];
    // Create asms from keyCombination
    keyCombinations(keys, k).forEach(combination => {
      let asm = '0';
      combination.forEach(key => {
        asm += ` <sig(${key})>`;
      });
      asms.push(asm);
    });

    let asm = '0';
    for (let i = 0; i < k; i++) {
      asm += ` <sig(${keys[i]})>`;
    }
    const sats = asms.map(asm => ({ asm }));

    return { sats, dsats };
  },
  a: X => ({
    dsats: [...combine(`dsat(X)`, { X })],
    sats: [...combine(`sat(X)`, { X })]
  }),
  s: X => ({
    dsats: [...combine(`dsat(X)`, { X })],
    sats: [...combine(`sat(X)`, { X })]
  }),
  c: X => ({
    dsats: [...combine(`dsat(X)`, { X })],
    sats: [...combine(`sat(X)`, { X })]
  }),
  d: X => ({
    dsats: [{ asm: `0` }],
    sats: [...combine(`sat(X) 1`, { X })]
  }),
  v: X => ({
    sats: [...combine(`sat(X)`, { X })]
  }),

  /**
   * j:X corresponds to SIZE 0NOTEQUAL IF [X] ENDIF
   *
   * By definition, this: "dsat(X) X" produces a zero.
   *
   *
   * A dsat needs to make this "dsat(X) X" return zero.

   * The easiest dsat for j:X is using 0:
   * 0 SIZE 0NOTEQUAL IF [X] ENDIF
   * 0 0 0NOTEQUAL IF [X] ENDIF
   * 0 0 IF [X] ENDIF
   * 0

   * Now let's do the case where dsat(X) finishes with a 0: "... 0"
   * ... 0 SIZE 0NOTEQUAL IF [X] ENDIF
   * ... 0 0 0NOTEQUAL IF [X] ENDIF
   * ... 0 0 IF [X] ENDIF 
   * ... 0
   * DSAT(X)
   * The final expression is "DSAT(X)".
   * It should have either been "0" or "DSAT(X) X"
   * Thus, this is not a valid dsat.

   * Now let's do the case where dsat(X) is this: "... <nonzero>"
   * ... <nonzero> SIZE 0NOTEQUAL IF [X] ENDIF
   * ... <nonzero> <length_is_nonzero> 0NOTEQUAL IF [X] ENDIF
   * ... <nonzero> 1 IF [X] ENDIF
   * ... <nonzero> [X]
   * DSAT(X) [X]

   * DSAT(X) X is a good dsat
   */
  j: X => {
    const dsats = [];

    dsats.push({ asm: `0` });

    //Filter the dsats of X with Non Zero Top Stack (nztp).
    const dsats_nzts = X.dsats.filter(
      //The top stack corresponds to the last element pushed to the stack,
      //that is, the last element in the produced witness
      solution => solution.asm.trim().split(' ').pop() !== '0'
    );
    dsats.push(...dsats_nzts);

    return { dsats, sats: [...combine(`sat(X)`, { X })] };
  },
  n: X => ({
    dsats: [...combine(`dsat(X)`, { X })],
    sats: [...combine(`sat(X)`, { X })]
  })
};

function createCommonjsModule(fn, module) {
	return module = { exports: {} }, fn(module, module.exports), module.exports;
}

var bindings = createCommonjsModule(function (module, exports) {
var Module = (() => {
  var _scriptDir = typeof document !== 'undefined' && document.currentScript ? document.currentScript.src : undefined;
  
  return (
function(moduleArg = {}) {

var Module=moduleArg;var readyPromiseResolve,readyPromiseReject;Module["ready"]=new Promise((resolve,reject)=>{readyPromiseResolve=resolve;readyPromiseReject=reject;});var moduleOverrides=Object.assign({},Module);var thisProgram="./this.program";var scriptDirectory="";function locateFile(path){if(Module["locateFile"]){return Module["locateFile"](path,scriptDirectory)}return scriptDirectory+path}{if(typeof document!="undefined"&&document.currentScript){scriptDirectory=document.currentScript.src;}if(_scriptDir){scriptDirectory=_scriptDir;}if(scriptDirectory.indexOf("blob:")!==0){scriptDirectory=scriptDirectory.substr(0,scriptDirectory.replace(/[?#].*/,"").lastIndexOf("/")+1);}else {scriptDirectory="";}}Module["print"]||console.log.bind(console);var err=Module["printErr"]||console.error.bind(console);Object.assign(Module,moduleOverrides);moduleOverrides=null;if(Module["arguments"])Module["arguments"];if(Module["thisProgram"])thisProgram=Module["thisProgram"];if(Module["quit"])Module["quit"];var wasmBinary;if(Module["wasmBinary"])wasmBinary=Module["wasmBinary"];Module["noExitRuntime"]||true;var WebAssembly={Memory:function(opts){this.buffer=new ArrayBuffer(opts["initial"]*65536);},Module:function(binary){},Instance:function(module,info){this.exports=(
// EMSCRIPTEN_START_ASM
function instantiate(_a){function c(d){d.set=function(a,b){this[a]=b;};d.get=function(a){return this[a]};return d}var e;var f=new Uint8Array(123);for(var a=25;a>=0;--a){f[48+a]=52+a;f[65+a]=a;f[97+a]=26+a;}f[43]=62;f[47]=63;function l(m,n,o){var g,h,a=0,i=n,j=o.length,k=n+(j*3>>2)-(o[j-2]=="=")-(o[j-1]=="=");for(;a<j;a+=4){g=f[o.charCodeAt(a+1)];h=f[o.charCodeAt(a+2)];m[i++]=f[o.charCodeAt(a)]<<2|g>>4;if(i<k)m[i++]=g<<4|h>>2;if(i<k)m[i++]=h<<6|f[o.charCodeAt(a+3)];}}function p(q){l(e,1024,"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");l(e,5584,"////////////////////////////////////////////////////////////////AAECAwQFBgcICf////////8KCwwNDg///////////////////////////////////woLDA0OD/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////4QAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADEDwAAgw8AAFcPAADcDwAAtw0AAFQQAACtDwAAdA8AAGUPAABNDwAAQg8AABMPAAALDwAA+w4AALIQAADZDwAAoA8AAGcPAABkDwAATA8AAEEPAADFCwAAfAsAAB8NAAAlDQAAOg0AAC4NAABeDQAAQw0AADUKAADvCwAA0gwAAOAMAAC8CwAArAsAAKQLAACLCwAA/woAANsLAACbCwAA8AwAALQLAACUCwAAzAsAAIMLAADKDAAAWQwAAPgKAADTCwAAwgwAADMLAABKCwAAKwsAAA4LAABMDQAA7goAAKUNAABUCwAAWgsAAGgMAAA/CgAAtw8AAHYPAADKDQAA3w0AADYMAADmCgAAVA0AADoLAAAHCwAAgAwAAMMNAADYDQAALwwAAN8KAACeDQAAIQsAABcLAACsDQAAcgsAALYMAABOCgAAcQwAAA4MAAAaDAAAjQwAAKAMAAD5CwAA0goAAAAMAABzEAAA0Q8AADcPAABoEAAALA8AAGELAAD+DAAAYAoAAAoNAAByCgAArw8AAIkKAACgCgAATw8AAEQPAAAVDwAADQ8AAP0OAAC2DgAArBAAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAOQLAADkCwAA5AsAAI0NAAAEXwAABF8AAARfAAAEXwAABF8AAARfAAAEXwAABF8AAARfAAAEXwAAFF8AACRfAAA0XwAARF8AAFRfAABkXwAAdF8AAIRfAACUXwAApF8AAMRfAAC0XwAA1F8AAORfAAAEXwAABF8=");l(e,6976,"MDEyMzQ1Njc4OWFiY2RlZgAAAAAAAAAAJAEAACUBAAAmAQAAJwEAACgBAAAAAAAAGQAKABkZGQAAAAAFAAAAAAAACQAAAAALAAAAAAAAAAAZABEKGRkZAwoHAAEACQsYAAAJBgsAAAsABhkAAAAZGRk=");l(e,7105,"DgAAAAAAAAAAGQAKDRkZGQANAAACAAkOAAAACQAOAAAO");l(e,7163,"DA==");l(e,7175,"EwAAAAATAAAAAAkMAAAAAAAMAAAM");l(e,7221,"EA==");l(e,7233,"DwAAAAQPAAAAAAkQAAAAAAAQAAAQ");l(e,7279,"Eg==");l(e,7291,"EQAAAAARAAAAAAkSAAAAAAASAAASAAAaAAAAGhoa");l(e,7346,"GgAAABoaGgAAAAAAAAk=");l(e,7395,"FA==");l(e,7407,"FwAAAAAXAAAAAAkUAAAAAAAUAAAU");l(e,7453,"Fg==");l(e,7465,"FQAAAAAVAAAAAAkWAAAAAAAWAAAWAAAwMTIzNDU2Nzg5QUJDREVGAAAAAAIAAAADAAAABQAAAAcAAAALAAAADQAAABEAAAATAAAAFwAAAB0AAAAfAAAAJQAAACkAAAArAAAALwAAADUAAAA7AAAAPQAAAEMAAABHAAAASQAAAE8AAABTAAAAWQAAAGEAAABlAAAAZwAAAGsAAABtAAAAcQAAAH8AAACDAAAAiQAAAIsAAACVAAAAlwAAAJ0AAACjAAAApwAAAK0AAACzAAAAtQAAAL8AAADBAAAAxQAAAMcAAADTAAAAAQAAAAsAAAANAAAAEQAAABMAAAAXAAAAHQAAAB8AAAAlAAAAKQAAACsAAAAvAAAANQAAADsAAAA9AAAAQwAAAEcAAABJAAAATwAAAFMAAABZAAAAYQAAAGUAAABnAAAAawAAAG0AAABxAAAAeQAAAH8AAACDAAAAiQAAAIsAAACPAAAAlQAAAJcAAACdAAAAowAAAKcAAACpAAAArQAAALMAAAC1AAAAuwAAAL8AAADBAAAAxQAAAMcAAADRAAAAAAAAALwfAAApAQAAKgEAACsBAAAsAQAALQEAAC4BAAAvAQAAMAEAADEBAAAyAQAAMwEAADQBAAA1AQAANgEAAAAAAABQIAAANwEAADgBAAArAQAALAEAADkBAAA6AQAALwEAADABAAAxAQAAOwEAADMBAAA8AQAANQEAAD0BAABOU3QzX18yOWJhc2ljX2lvc0ljTlNfMTFjaGFyX3RyYWl0c0ljRUVFRQAAACRZAABQHwAAXCEAAE5TdDNfXzIxNWJhc2ljX3N0cmVhbWJ1ZkljTlNfMTFjaGFyX3RyYWl0c0ljRUVFRQAAAADkWAAAiB8AAE5TdDNfXzIxM2Jhc2ljX29zdHJlYW1JY05TXzExY2hhcl90cmFpdHNJY0VFRUUAAABXAADEHwAAAAAAAAEAAAB8HwAAA/T//05TdDNfXzIxNWJhc2ljX3N0cmluZ2J1ZkljTlNfMTFjaGFyX3RyYWl0c0ljRUVOU185YWxsb2NhdG9ySWNFRUVFAAAAJFkAAAwgAAC8HwAAOAAAAAAAAAD0IAAAPgEAAD8BAADI////yP////QgAABAAQAAQQEAADgAAAAAAAAA9B8AAEIBAABDAQAAyP///8j////0HwAARAEAAEUBAABOU3QzX18yMTliYXNpY19vc3RyaW5nc3RyZWFtSWNOU18xMWNoYXJfdHJhaXRzSWNFRU5TXzlhbGxvY2F0b3JJY0VFRUUAAAAkWQAArCAAAPQfAAAAAAAArCEAAEYBAABHAQAASAEAAEkBAABKAQAASwEAAEwBAAAAAAAAgCEAALEAAABNAQAATgEAAAAAAABcIQAATwEAAFABAABOU3QzX18yOGlvc19iYXNlRQAAAORYAABIIQAATlN0M19fMjhpb3NfYmFzZTdmYWlsdXJlRQAAACRZAABkIQAAlFUAAE5TdDNfXzIxOV9faW9zdHJlYW1fY2F0ZWdvcnlFAAAAJFkAAIwhAADkVQ==");l(e,8640,"/////////////////////////////////////////////////////////////////wABAgMEBQYHCAn/////////CgsMDQ4PEBESExQVFhcYGRobHB0eHyAhIiP///////8KCwwNDg8QERITFBUWFxgZGhscHR4fICEiI/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////8AAQIEBwMGBQAAAAAAAADRdJ4AV529KoBwUg///z4nCgAAAGQAAADoAwAAECcAAKCGAQBAQg8AgJaYAADh9QUYAAAANQAAAHEAAABr////zvv//5K///8AAAAAAAAAAN4SBJUAAAAA////////////////");l(e,9024,"TENfQ1RZUEUAAAAATENfTlVNRVJJQwAATENfVElNRQAAAAAATENfQ09MTEFURQAATENfTU9ORVRBUlkATENfTUVTU0FHRVMAICMAABQAAABDLlVURi04");l(e,9652,"AQAAAAIAAAADAAAABAAAAAUAAAAGAAAABwAAAAgAAAAJAAAACgAAAAsAAAAMAAAADQAAAA4AAAAPAAAAEAAAABEAAAASAAAAEwAAABQAAAAVAAAAFgAAABcAAAAYAAAAGQAAABoAAAAbAAAAHAAAAB0AAAAeAAAAHwAAACAAAAAhAAAAIgAAACMAAAAkAAAAJQAAACYAAAAnAAAAKAAAACkAAAAqAAAAKwAAACwAAAAtAAAALgAAAC8AAAAwAAAAMQAAADIAAAAzAAAANAAAADUAAAA2AAAANwAAADgAAAA5AAAAOgAAADsAAAA8AAAAPQAAAD4AAAA/AAAAQAAAAGEAAABiAAAAYwAAAGQAAABlAAAAZgAAAGcAAABoAAAAaQAAAGoAAABrAAAAbAAAAG0AAABuAAAAbwAAAHAAAABxAAAAcgAAAHMAAAB0AAAAdQAAAHYAAAB3AAAAeAAAAHkAAAB6AAAAWwAAAFwAAABdAAAAXgAAAF8AAABgAAAAYQAAAGIAAABjAAAAZAAAAGUAAABmAAAAZwAAAGgAAABpAAAAagAAAGsAAABsAAAAbQAAAG4AAABvAAAAcAAAAHEAAAByAAAAcwAAAHQAAAB1AAAAdgAAAHcAAAB4AAAAeQAAAHoAAAB7AAAAfAAAAH0AAAB+AAAAfw==");l(e,11188,"AQAAAAIAAAADAAAABAAAAAUAAAAGAAAABwAAAAgAAAAJAAAACgAAAAsAAAAMAAAADQAAAA4AAAAPAAAAEAAAABEAAAASAAAAEwAAABQAAAAVAAAAFgAAABcAAAAYAAAAGQAAABoAAAAbAAAAHAAAAB0AAAAeAAAAHwAAACAAAAAhAAAAIgAAACMAAAAkAAAAJQAAACYAAAAnAAAAKAAAACkAAAAqAAAAKwAAACwAAAAtAAAALgAAAC8AAAAwAAAAMQAAADIAAAAzAAAANAAAADUAAAA2AAAANwAAADgAAAA5AAAAOgAAADsAAAA8AAAAPQAAAD4AAAA/AAAAQAAAAEEAAABCAAAAQwAAAEQAAABFAAAARgAAAEcAAABIAAAASQAAAEoAAABLAAAATAAAAE0AAABOAAAATwAAAFAAAABRAAAAUgAAAFMAAABUAAAAVQAAAFYAAABXAAAAWAAAAFkAAABaAAAAWwAAAFwAAABdAAAAXgAAAF8AAABgAAAAQQAAAEIAAABDAAAARAAAAEUAAABGAAAARwAAAEgAAABJAAAASgAAAEsAAABMAAAATQAAAE4AAABPAAAAUAAAAFEAAABSAAAAUwAAAFQAAABVAAAAVgAAAFcAAABYAAAAWQAAAFoAAAB7AAAAfAAAAH0AAAB+AAAAfw==");l(e,12208,"AgAAwAMAAMAEAADABQAAwAYAAMAHAADACAAAwAkAAMAKAADACwAAwAwAAMANAADADgAAwA8AAMAQAADAEQAAwBIAAMATAADAFAAAwBUAAMAWAADAFwAAwBgAAMAZAADAGgAAwBsAAMAcAADAHQAAwB4AAMAfAADAAAAAswEAAMMCAADDAwAAwwQAAMMFAADDBgAAwwcAAMMIAADDCQAAwwoAAMMLAADDDAAAww0AANMOAADDDwAAwwAADLsBAAzDAgAMwwMADMMEAAzbAAAAADAxMjM0NTY3ODlhYmNkZWZBQkNERUZ4WCstcFBpSW5OACVJOiVNOiVTICVwJUg6JU0=");l(e,12480,"JQAAAG0AAAAvAAAAJQAAAGQAAAAvAAAAJQAAAHkAAAAlAAAAWQAAAC0AAAAlAAAAbQAAAC0AAAAlAAAAZAAAACUAAABJAAAAOgAAACUAAABNAAAAOgAAACUAAABTAAAAIAAAACUAAABwAAAAAAAAACUAAABIAAAAOgAAACUAAABN");l(e,12624,"JQAAAEgAAAA6AAAAJQAAAE0AAAA6AAAAJQAAAFMAAAAAAAAAxDoAAFEBAABSAQAAUwEAAAAAAAAkOwAAVAEAAFUBAABTAQAAVgEAAFcBAABYAQAAWQEAAFoBAABbAQAAXAEAAF0B");l(e,12736,"BAAAAAQAAAAEAAAABAAAAAQAAAAEAAAABAAAAAQAAAAEAAAABQIAAAUAAAAFAAAABQAAAAUAAAAEAAAABAAAAAQAAAAEAAAABAAAAAQAAAAEAAAABAAAAAQAAAAEAAAABAAAAAQAAAAEAAAABAAAAAQAAAAEAAAABAAAAAQAAAADAgAAggAAAIIAAACCAAAAggAAAIIAAACCAAAAggAAAIIAAACCAAAAggAAAIIAAACCAAAAggAAAIIAAACCAAAAQgEAAEIBAABCAQAAQgEAAEIBAABCAQAAQgEAAEIBAABCAQAAQgEAAIIAAACCAAAAggAAAIIAAACCAAAAggAAAIIAAAAqAQAAKgEAACoBAAAqAQAAKgEAACoBAAAqAAAAKgAAACoAAAAqAAAAKgAAACoAAAAqAAAAKgAAACoAAAAqAAAAKgAAACoAAAAqAAAAKgAAACoAAAAqAAAAKgAAACoAAAAqAAAAKgAAAIIAAACCAAAAggAAAIIAAACCAAAAggAAADIBAAAyAQAAMgEAADIBAAAyAQAAMgEAADIAAAAyAAAAMgAAADIAAAAyAAAAMgAAADIAAAAyAAAAMgAAADIAAAAyAAAAMgAAADIAAAAyAAAAMgAAADIAAAAyAAAAMgAAADIAAAAyAAAAggAAAIIAAACCAAAAggAAAAQ=");l(e,13764,"jDoAAF4BAABfAQAAUwEAAGABAABhAQAAYgEAAGMBAABkAQAAZQEAAGYBAAAAAAAAXDsAAGcBAABoAQAAUwEAAGkBAABqAQAAawEAAGwBAABtAQAAAAAAAIA7AABuAQAAbwEAAFMBAABwAQAAcQEAAHIBAABzAQAAdAEAAHQAAAByAAAAdQAAAGUAAAAAAAAAZgAAAGEAAABsAAAAcwAAAGUAAAAAAAAAJQAAAG0AAAAvAAAAJQAAAGQAAAAvAAAAJQAAAHkAAAAAAAAAJQAAAEgAAAA6AAAAJQAAAE0AAAA6AAAAJQAAAFMAAAAAAAAAJQAAAGEAAAAgAAAAJQAAAGIAAAAgAAAAJQAAAGQAAAAgAAAAJQAAAEgAAAA6AAAAJQAAAE0AAAA6AAAAJQAAAFMAAAAgAAAAJQAAAFkAAAAAAAAAJQAAAEkAAAA6AAAAJQAAAE0AAAA6AAAAJQAAAFMAAAAgAAAAJQAAAHA=");l(e,14140,"ZDcAAHUBAAB2AQAAUwEAAE5TdDNfXzI2bG9jYWxlNWZhY2V0RQAAACRZAABMNwAA3FUAAAAAAADkNwAAdQEAAHcBAABTAQAAeAEAAHkBAAB6AQAAewEAAHwBAAB9AQAAfgEAAH8BAACAAQAAgQEAAIIBAACDAQAATlN0M19fMjVjdHlwZUl3RUUATlN0M19fMjEwY3R5cGVfYmFzZUUAAORYAADGNwAAAFcAALQ3AAAAAAAAAgAAAGQ3AAACAAAA3DcAAAIAAAAAAAAAeDgAAHUBAACEAQAAUwEAAIUBAACGAQAAhwEAAIgBAACJAQAAigEAAIsBAABOU3QzX18yN2NvZGVjdnRJY2MxMV9fbWJzdGF0ZV90RUUATlN0M19fMjEyY29kZWN2dF9iYXNlRQAAAADkWAAAVjgAAABXAAA0OAAAAAAAAAIAAABkNwAAAgAAAHA4AAACAAAAAAAAAOw4AAB1AQAAjAEAAFMBAACNAQAAjgEAAI8BAACQAQAAkQEAAJIBAACTAQAATlN0M19fMjdjb2RlY3Z0SURzYzExX19tYnN0YXRlX3RFRQAAAFcAAMg4AAAAAAAAAgAAAGQ3AAACAAAAcDgAAAIAAAAAAAAAYDkAAHUBAACUAQAAUwEAAJUBAACWAQAAlwEAAJgBAACZAQAAmgEAAJsBAABOU3QzX18yN2NvZGVjdnRJRHNEdTExX19tYnN0YXRlX3RFRQAAVwAAPDkAAAAAAAACAAAAZDcAAAIAAABwOAAAAgAAAAAAAADUOQAAdQEAAJwBAABTAQAAnQEAAJ4BAACfAQAAoAEAAKEBAACiAQAAowEAAE5TdDNfXzI3Y29kZWN2dElEaWMxMV9fbWJzdGF0ZV90RUUAAABXAACwOQAAAAAAAAIAAABkNwAAAgAAAHA4AAACAAAAAAAAAEg6AAB1AQAApAEAAFMBAAClAQAApgEAAKcBAACoAQAAqQEAAKoBAACrAQAATlN0M19fMjdjb2RlY3Z0SURpRHUxMV9fbWJzdGF0ZV90RUUAAFcAACQ6AAAAAAAAAgAAAGQ3AAACAAAAcDgAAAIAAABOU3QzX18yN2NvZGVjdnRJd2MxMV9fbWJzdGF0ZV90RUUAAAAAVwAAaDoAAAAAAAACAAAAZDcAAAIAAABwOAAAAgAAAE5TdDNfXzI2bG9jYWxlNV9faW1wRQAAACRZAACsOgAAZDcAAE5TdDNfXzI3Y29sbGF0ZUljRUUAJFkAANA6AABkNwAATlN0M19fMjdjb2xsYXRlSXdFRQAkWQAA8DoAAGQ3AABOU3QzX18yNWN0eXBlSWNFRQAAAABXAAAQOwAAAAAAAAIAAABkNwAAAgAAANw3AAACAAAATlN0M19fMjhudW1wdW5jdEljRUUAAAAAJFkAAEQ7AABkNwAATlN0M19fMjhudW1wdW5jdEl3RUUAAAAAJFkAAGg7AABkNwAAAAAAAOQ6AACsAQAArQEAAFMBAACuAQAArwEAALABAAAAAAAABDsAALEBAACyAQAAUwEAALMBAAC0AQAAtQEAAAAAAACgPAAAdQEAALYBAABTAQAAtwEAALgBAAC5AQAAugEAALsBAAC8AQAAvQEAAL4BAAC/AQAAwAEAAMEBAABOU3QzX18yN251bV9nZXRJY05TXzE5aXN0cmVhbWJ1Zl9pdGVyYXRvckljTlNfMTFjaGFyX3RyYWl0c0ljRUVFRUVFAE5TdDNfXzI5X19udW1fZ2V0SWNFRQBOU3QzX18yMTRfX251bV9nZXRfYmFzZUUAAORYAABmPAAAAFcAAFA8AAAAAAAAAQAAAIA8AAAAAAAAAFcAAAw8AAAAAAAAAgAAAGQ3AAACAAAAiDw=");l(e,15556,"dD0AAHUBAADCAQAAUwEAAMMBAADEAQAAxQEAAMYBAADHAQAAyAEAAMkBAADKAQAAywEAAMwBAADNAQAATlN0M19fMjdudW1fZ2V0SXdOU18xOWlzdHJlYW1idWZfaXRlcmF0b3JJd05TXzExY2hhcl90cmFpdHNJd0VFRUVFRQBOU3QzX18yOV9fbnVtX2dldEl3RUUAAAAAVwAARD0AAAAAAAABAAAAgDwAAAAAAAAAVwAAAD0AAAAAAAACAAAAZDcAAAIAAABcPQ==");l(e,15768,"XD4AAHUBAADOAQAAUwEAAM8BAADQAQAA0QEAANIBAADTAQAA1AEAANUBAADWAQAATlN0M19fMjdudW1fcHV0SWNOU18xOW9zdHJlYW1idWZfaXRlcmF0b3JJY05TXzExY2hhcl90cmFpdHNJY0VFRUVFRQBOU3QzX18yOV9fbnVtX3B1dEljRUUATlN0M19fMjE0X19udW1fcHV0X2Jhc2VFAADkWAAAIj4AAABXAAAMPgAAAAAAAAEAAAA8PgAAAAAAAABXAADIPQAAAAAAAAIAAABkNwAAAgAAAEQ+");l(e,16e3,"JD8AAHUBAADXAQAAUwEAANgBAADZAQAA2gEAANsBAADcAQAA3QEAAN4BAADfAQAATlN0M19fMjdudW1fcHV0SXdOU18xOW9zdHJlYW1idWZfaXRlcmF0b3JJd05TXzExY2hhcl90cmFpdHNJd0VFRUVFRQBOU3QzX18yOV9fbnVtX3B1dEl3RUUAAAAAVwAA9D4AAAAAAAABAAAAPD4AAAAAAAAAVwAAsD4AAAAAAAACAAAAZDcAAAIAAAAMPw==");l(e,16200,"JEAAAOABAADhAQAAUwEAAOIBAADjAQAA5AEAAOUBAADmAQAA5wEAAOgBAAD4////JEAAAOkBAADqAQAA6wEAAOwBAADtAQAA7gEAAO8BAABOU3QzX18yOHRpbWVfZ2V0SWNOU18xOWlzdHJlYW1idWZfaXRlcmF0b3JJY05TXzExY2hhcl90cmFpdHNJY0VFRUVFRQBOU3QzX18yOXRpbWVfYmFzZUUA5FgAAN0/AABOU3QzX18yMjBfX3RpbWVfZ2V0X2Nfc3RvcmFnZUljRUUAAADkWAAA+D8AAABXAACYPwAAAAAAAAMAAABkNwAAAgAAAPA/AAACAAAAHEAAAAAIAAAAAAAAEEEAAPABAADxAQAAUwEAAPIBAADzAQAA9AEAAPUBAAD2AQAA9wEAAPgBAAD4////EEEAAPkBAAD6AQAA+wEAAPwBAAD9AQAA/gEAAP8BAABOU3QzX18yOHRpbWVfZ2V0SXdOU18xOWlzdHJlYW1idWZfaXRlcmF0b3JJd05TXzExY2hhcl90cmFpdHNJd0VFRUVFRQBOU3QzX18yMjBfX3RpbWVfZ2V0X2Nfc3RvcmFnZUl3RUUAAORYAADlQAAAAFcAAKBAAAAAAAAAAwAAAGQ3AAACAAAA8D8AAAIAAAAIQQAAAAgAAAAAAAC0QQAAAAIAAAECAABTAQAAAgIAAE5TdDNfXzI4dGltZV9wdXRJY05TXzE5b3N0cmVhbWJ1Zl9pdGVyYXRvckljTlNfMTFjaGFyX3RyYWl0c0ljRUVFRUVFAE5TdDNfXzIxMF9fdGltZV9wdXRFAAAA5FgAAJVBAAAAVwAAUEEAAAAAAAACAAAAZDcAAAIAAACsQQAAAAgAAAAAAAA0QgAAAwIAAAQCAABTAQAABQIAAE5TdDNfXzI4dGltZV9wdXRJd05TXzE5b3N0cmVhbWJ1Zl9pdGVyYXRvckl3TlNfMTFjaGFyX3RyYWl0c0l3RUVFRUVFAAAAAABXAADsQQAAAAAAAAIAAABkNwAAAgAAAKxBAAAACAAAAAAAAMhCAAB1AQAABgIAAFMBAAAHAgAACAIAAAkCAAAKAgAACwIAAAwCAAANAgAADgIAAA8CAABOU3QzX18yMTBtb25leXB1bmN0SWNMYjBFRUUATlN0M19fMjEwbW9uZXlfYmFzZUUAAAAA5FgAAKhCAAAAVwAAjEIAAAAAAAACAAAAZDcAAAIAAADAQgAAAgAAAAAAAAA8QwAAdQEAABACAABTAQAAEQIAABICAAATAgAAFAIAABUCAAAWAgAAFwIAABgCAAAZAgAATlN0M19fMjEwbW9uZXlwdW5jdEljTGIxRUVFAABXAAAgQwAAAAAAAAIAAABkNwAAAgAAAMBCAAACAAAAAAAAALBDAAB1AQAAGgIAAFMBAAAbAgAAHAIAAB0CAAAeAgAAHwIAACACAAAhAgAAIgIAACMCAABOU3QzX18yMTBtb25leXB1bmN0SXdMYjBFRUUAAFcAAJRDAAAAAAAAAgAAAGQ3AAACAAAAwEIAAAIAAAAAAAAAJEQAAHUBAAAkAgAAUwEAACUCAAAmAgAAJwIAACgCAAApAgAAKgIAACsCAAAsAgAALQIAAE5TdDNfXzIxMG1vbmV5cHVuY3RJd0xiMUVFRQAAVwAACEQAAAAAAAACAAAAZDcAAAIAAADAQgAAAgAAAAAAAADIRAAAdQEAAC4CAABTAQAALwIAADACAABOU3QzX18yOW1vbmV5X2dldEljTlNfMTlpc3RyZWFtYnVmX2l0ZXJhdG9ySWNOU18xMWNoYXJfdHJhaXRzSWNFRUVFRUUATlN0M19fMjExX19tb25leV9nZXRJY0VFAADkWAAApkQAAABXAABgRAAAAAAAAAIAAABkNwAAAgAAAMBE");l(e,17644,"bEUAAHUBAAAxAgAAUwEAADICAAAzAgAATlN0M19fMjltb25leV9nZXRJd05TXzE5aXN0cmVhbWJ1Zl9pdGVyYXRvckl3TlNfMTFjaGFyX3RyYWl0c0l3RUVFRUVFAE5TdDNfXzIxMV9fbW9uZXlfZ2V0SXdFRQAA5FgAAEpFAAAAVwAABEUAAAAAAAACAAAAZDcAAAIAAABkRQ==");l(e,17808,"EEYAAHUBAAA0AgAAUwEAADUCAAA2AgAATlN0M19fMjltb25leV9wdXRJY05TXzE5b3N0cmVhbWJ1Zl9pdGVyYXRvckljTlNfMTFjaGFyX3RyYWl0c0ljRUVFRUVFAE5TdDNfXzIxMV9fbW9uZXlfcHV0SWNFRQAA5FgAAO5FAAAAVwAAqEUAAAAAAAACAAAAZDcAAAIAAAAIRg==");l(e,17972,"tEYAAHUBAAA3AgAAUwEAADgCAAA5AgAATlN0M19fMjltb25leV9wdXRJd05TXzE5b3N0cmVhbWJ1Zl9pdGVyYXRvckl3TlNfMTFjaGFyX3RyYWl0c0l3RUVFRUVFAE5TdDNfXzIxMV9fbW9uZXlfcHV0SXdFRQAA5FgAAJJGAAAAVwAATEYAAAAAAAACAAAAZDcAAAIAAACsRg==");l(e,18136,"LEcAAHUBAAA6AgAAUwEAADsCAAA8AgAAPQIAAE5TdDNfXzI4bWVzc2FnZXNJY0VFAE5TdDNfXzIxM21lc3NhZ2VzX2Jhc2VFAAAAAORYAAAJRwAAAFcAAPRGAAAAAAAAAgAAAGQ3AAACAAAAJEcAAAIAAAAAAAAAhEcAAHUBAAA+AgAAUwEAAD8CAABAAgAAQQIAAE5TdDNfXzI4bWVzc2FnZXNJd0VFAAAAAABXAABsRwAAAAAAAAIAAABkNwAAAgAAACRHAAACAAAAUwAAAHUAAABuAAAAZAAAAGEAAAB5AAAAAAAAAE0AAABvAAAAbgAAAGQAAABhAAAAeQAAAAAAAABUAAAAdQAAAGUAAABzAAAAZAAAAGEAAAB5AAAAAAAAAFcAAABlAAAAZAAAAG4AAABlAAAAcwAAAGQAAABhAAAAeQAAAAAAAABUAAAAaAAAAHUAAAByAAAAcwAAAGQAAABhAAAAeQAAAAAAAABGAAAAcgAAAGkAAABkAAAAYQAAAHkAAAAAAAAAUwAAAGEAAAB0AAAAdQAAAHIAAABkAAAAYQAAAHkAAAAAAAAAUwAAAHUAAABuAAAAAAAAAE0AAABvAAAAbgAAAAAAAABUAAAAdQAAAGUAAAAAAAAAVwAAAGUAAABkAAAAAAAAAFQAAABoAAAAdQAAAAAAAABGAAAAcgAAAGkAAAAAAAAAUwAAAGEAAAB0AAAAAAAAAEoAAABhAAAAbgAAAHUAAABhAAAAcgAAAHkAAAAAAAAARgAAAGUAAABiAAAAcgAAAHUAAABhAAAAcgAAAHkAAAAAAAAATQAAAGEAAAByAAAAYwAAAGgAAAAAAAAAQQAAAHAAAAByAAAAaQAAAGwAAAAAAAAATQAAAGEAAAB5AAAAAAAAAEoAAAB1AAAAbgAAAGUAAAAAAAAASgAAAHUAAABsAAAAeQAAAAAAAABBAAAAdQAAAGcAAAB1AAAAcwAAAHQAAAAAAAAAUwAAAGUAAABwAAAAdAAAAGUAAABtAAAAYgAAAGUAAAByAAAAAAAAAE8AAABjAAAAdAAAAG8AAABiAAAAZQAAAHIAAAAAAAAATgAAAG8AAAB2AAAAZQAAAG0AAABiAAAAZQAAAHIAAAAAAAAARAAAAGUAAABjAAAAZQAAAG0AAABiAAAAZQAAAHIAAAAAAAAASgAAAGEAAABuAAAAAAAAAEYAAABlAAAAYgAAAAAAAABNAAAAYQAAAHIAAAAAAAAAQQAAAHAAAAByAAAAAAAAAEoAAAB1AAAAbgAAAAAAAABKAAAAdQAAAGwAAAAAAAAAQQAAAHUAAABnAAAAAAAAAFMAAABlAAAAcAAAAAAAAABPAAAAYwAAAHQAAAAAAAAATgAAAG8AAAB2AAAAAAAAAEQAAABlAAAAYwAAAAAAAABBAAAATQAAAAAAAABQAAAATQ==");l(e,19228,"HEAAAOkBAADqAQAA6wEAAOwBAADtAQAA7gEAAO8BAAAAAAAACEEAAPkBAAD6AQAA+wEAAPwBAAD9AQAA/gEAAP8BAABOU3QzX18yMTRfX3NoYXJlZF9jb3VudEU=");l(e,19348,"iCM=");l(e,19372,"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");l(e,21298,"pQJbAPABtQWMBSUBgwYdA5QE/wDHAzEDCwa8AY8BfwPKBCsA2gavAEIDTgPcAQ4EFQChBg0BlAILAjgGZAK8Av8CXQPnBAsHzwLLBe8F2wXhAh4GRQKFAIICbANvBPEA8wMYBdkA2gNMBlQCewGdA70EAABRABUCuwCzA20A/wGFBC8F+QQ4AGUBRgGfALcGqAFzAlMB");l(e,21496,"IQQAAAAAAAAAAC8C");l(e,21528,"NQRHBFYE");l(e,21550,"oAQ=");l(e,21570,"RgVgBW4FYQYAAM8BAAAAAAAAAADJBukG+QYeBzkHSQdeBw==");l(e,21620,"CgAAAGQAAADoAwAAECcAAKCGAQBAQg8AgJaYAADh9QUAypo7AAAAAAAAAAAwMDAxMDIwMzA0MDUwNjA3MDgwOTEwMTExMjEzMTQxNTE2MTcxODE5MjAyMTIyMjMyNDI1MjYyNzI4MjkzMDMxMzIzMzM0MzUzNjM3MzgzOTQwNDE0MjQzNDQ0NTQ2NDc0ODQ5NTA1MTUyNTM1NDU1NTY1NzU4NTk2MDYxNjI2MzY0NjU2NjY3Njg2OTcwNzE3MjczNzQ3NTc2Nzc3ODc5ODA4MTgyODM4NDg1ODY4Nzg4ODk5MDkxOTI5Mzk0OTU5Njk3OTg5OQAAAACUVQAAQgIAAEMCAABOAQAATlN0M19fMjEyc3lzdGVtX2Vycm9yRQAAJFkAAHxVAACIWAAATlN0M19fMjE0ZXJyb3JfY2F0ZWdvcnlFAAAAAORYAACgVQAATlN0M19fMjEyX19kb19tZXNzYWdlRQAA5FgAAGBLAAAkWQAAxFUAALxVAAAAAAAA3FUAAEQCAABFAgAARgIAAE4xMF9fY3h4YWJpdjExNl9fc2hpbV90eXBlX2luZm9FAAAAACRZAAAEVgAAFFkAAE4xMF9fY3h4YWJpdjExN19fY2xhc3NfdHlwZV9pbmZvRQAAACRZAAA0VgAAKFYAAE4xMF9fY3h4YWJpdjExN19fcGJhc2VfdHlwZV9pbmZvRQAAACRZAABkVgAAKFYAAE4xMF9fY3h4YWJpdjExOV9fcG9pbnRlcl90eXBlX2luZm9FACRZAACUVgAAiFYAAE4xMF9fY3h4YWJpdjEyMF9fc2lfY2xhc3NfdHlwZV9pbmZvRQAAAAAkWQAAxFYAAFhWAAAAAAAASFcAAEgCAABJAgAASgIAAEsCAABMAgAATQIAAE4CAABPAgAATjEwX19jeHhhYml2MTIxX192bWlfY2xhc3NfdHlwZV9pbmZvRQAAACRZAAAgVwAAWFYAAAAAAACwVwAABQAAAFACAABRAgAAAAAAALxXAAAFAAAAUgIAAFMCAABTdDlleGNlcHRpb24AU3QyMGJhZF9hcnJheV9uZXdfbGVuZ3RoAFN0OWJhZF9hbGxvYwAAJFkAAKJXAABQWQAAJFkAAIlXAACwVwAAAAAAAABYAAAHAAAAVAIAAFUCAAAAAAAAiFgAAFwAAABWAgAATgEAAFN0MTFsb2dpY19lcnJvcgAkWQAA8FcAAFBZAAAAAAAANFgAAAcAAABXAgAAVQIAAFN0MTJsZW5ndGhfZXJyb3IAAAAAJFkAACBYAAAAWAAAAAAAAGhYAAAHAAAAWAIAAFUCAABTdDEyb3V0X29mX3JhbmdlAAAAACRZAABUWAAAAFgAAFN0MTNydW50aW1lX2Vycm9yAAAAJFkAAHRYAABQWQAAAAAAALxYAABcAAAAWQIAAE4BAABTdDE0b3ZlcmZsb3dfZXJyb3IAACRZAACoWAAAiFgAAAAAAABYWQAArQAAAFoCAABbAgAAAAAAAFhWAABIAgAAXAIAAEoCAABLAgAATAIAAF0CAABeAgAAXwIAAFN0OXR5cGVfaW5mbwAAAADkWAAABFkAAAAAAADsVgAASAIAAGACAABKAgAASwIAAEwCAABhAgAAYgIAAGMCAABTdDhiYWRfY2FzdADkWAAAfFcAACRZAABEWQAAUFkAAAAAAABQWQAABQAAAGQCAABlAg==");l(e,22904,"0GoBAAghAABHAg==");}var r=new ArrayBuffer(16);var s=new Int32Array(r);var t=new Float32Array(r);var u=new Float64Array(r);function v(w){return s[w]}function x(w,y){s[w]=y;}function z(){return u[0]}function A(y){u[0]=y;}function B(){throw new Error("abort")}function C(){return t[2]}function D(y){t[2]=y;}function Za(q){var E=q.a;var F=E.a;var G=F.buffer;var H=new Int8Array(G);var I=new Int16Array(G);var J=new Int32Array(G);var K=new Uint8Array(G);var L=new Uint16Array(G);var M=new Uint32Array(G);var N=new Float32Array(G);var O=new Float64Array(G);var P=Math.imul;var Q=Math.fround;var R=Math.abs;var S=Math.clz32;var W=Math.ceil;var Z=E.b;var _=E.c;var $=E.d;var aa=E.e;var ba=E.f;var ca=E.g;var da=E.h;var ea=E.i;var fa=E.j;var ga=E.k;var ha=E.l;var ia=E.m;var ja=E.n;var ka=E.o;var la=E.p;var ma=E.q;var na=E.r;var oa=E.s;var pa=E.t;var qa=E.u;var ra=E.v;var sa=E.w;var ta=E.x;var ua=E.y;var va=E.z;var wa=E.A;var xa=E.B;var ya=E.C;var za=E.D;var Aa=E.E;var Ba=E.F;var Ca=E.G;var Da=E.H;var Ea=E.I;var Fa=E.J;var Ga=E.K;var Ha=E.L;var Ia=E.M;var Ja=E.N;var Ka=E.O;var La=E.P;var Ma=E.Q;var Na=E.R;var Oa=E.S;var Pa=E.T;var Qa=E.U;var Ra=E.V;var Sa=E.W;var Ta=E.X;var Ua=92880;var Va=0;var Wa=0;
// EMSCRIPTEN_START_FUNCS
function Vm(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0;e=Ua-352|0;Ua=e;J[e+328>>2]=0;J[e+320>>2]=0;J[e+324>>2]=0;J[5996]=0;J[e+316>>2]=0;J[e+308>>2]=0;J[e+312>>2]=0;f=$(3,24)|0;d=J[5996];J[5996]=0;a:{b:{c:{if((d|0)!=1){J[f+8>>2]=-1;J[f+12>>2]=-1;J[f>>2]=0;J[f+16>>2]=-1;J[f+20>>2]=-1;c=f+24|0;J[e+328>>2]=c;J[e+320>>2]=f;m=e+296|0;d:{e:{f:{g:{h:{i:{j:{k:{l:{while(1){d=c-24|0;g=J[d+20>>2];J[m>>2]=J[d+16>>2];J[m+4>>2]=g;g=J[d+12>>2];J[e+288>>2]=J[d+8>>2];J[e+292>>2]=g;h=J[d+4>>2];g=J[d>>2];J[e+280>>2]=g;J[e+284>>2]=h;J[e+324>>2]=d;l=18;m:{n:{o:{p:{q:{r:{s:{t:{u:{v:{w:{x:{y:{z:{switch(g|0){case 0:c=J[b+4>>2];A:{if((c|0)<2){f=-1;break A}g=J[b>>2];f=1;while(1){h=K[f+g|0];if((h|0)==58){l=0;B:{C:{D:{E:{while(1){F:{G:{switch(K[J[b>>2]+l|0]-97|0){case 0:d=J[e+324>>2];h=J[e+328>>2];if(d>>>0<h>>>0){J[d+8>>2]=-1;J[d+12>>2]=-1;J[d>>2]=3;J[d+16>>2]=-1;J[d+20>>2]=-1;J[e+324>>2]=d+24;break F}H:{I:{J:{g=J[e+320>>2];j=(d-g|0)/24|0;c=j+1|0;K:{if(c>>>0>=178956971){J[5996]=0;ca(142);a=J[5996];J[5996]=0;if((a|0)==1){break K}break a}h=(h-g|0)/24|0;i=h<<1;h=h>>>0>=89478485?178956970:c>>>0<i>>>0?i:c;if(!h){i=0;break I}if(h>>>0<178956971){break J}J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}}b=Z()|0;break b}J[5996]=0;i=$(3,P(h,24)|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break H}}c=P(j,24)+i|0;J[c+8>>2]=-1;J[c+12>>2]=-1;J[c>>2]=3;J[c+16>>2]=-1;J[c+20>>2]=-1;j=c+24|0;if((d|0)!=(g|0)){while(1){d=d-24|0;k=J[d+4>>2];c=c-24|0;J[c>>2]=J[d>>2];J[c+4>>2]=k;k=J[d+20>>2];J[c+16>>2]=J[d+16>>2];J[c+20>>2]=k;k=J[d+12>>2];J[c+8>>2]=J[d+8>>2];J[c+12>>2]=k;if((d|0)!=(g|0)){continue}break}d=g;}J[e+328>>2]=P(h,24)+i;J[e+324>>2]=j;J[e+320>>2]=c;if(!d){break F}$a(d);break F}b=Z()|0;break b;case 18:d=J[e+324>>2];h=J[e+328>>2];if(d>>>0<h>>>0){J[d+8>>2]=-1;J[d+12>>2]=-1;J[d>>2]=2;J[d+16>>2]=-1;J[d+20>>2]=-1;J[e+324>>2]=d+24;break F}L:{M:{N:{g=J[e+320>>2];j=(d-g|0)/24|0;c=j+1|0;O:{if(c>>>0>=178956971){J[5996]=0;ca(142);break O}h=(h-g|0)/24|0;i=h<<1;h=h>>>0>=89478485?178956970:c>>>0<i>>>0?i:c;if(!h){i=0;break M}if(h>>>0<178956971){break N}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break a}b=Z()|0;break b}J[5996]=0;i=$(3,P(h,24)|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break L}}c=P(j,24)+i|0;J[c+8>>2]=-1;J[c+12>>2]=-1;J[c>>2]=2;J[c+16>>2]=-1;J[c+20>>2]=-1;j=c+24|0;if((d|0)!=(g|0)){while(1){d=d-24|0;k=J[d+4>>2];c=c-24|0;J[c>>2]=J[d>>2];J[c+4>>2]=k;k=J[d+20>>2];J[c+16>>2]=J[d+16>>2];J[c+20>>2]=k;k=J[d+12>>2];J[c+8>>2]=J[d+8>>2];J[c+12>>2]=k;if((d|0)!=(g|0)){continue}break}d=g;}J[e+328>>2]=P(h,24)+i;J[e+324>>2]=j;J[e+320>>2]=c;if(!d){break F}$a(d);break F}b=Z()|0;break b;case 2:d=J[e+324>>2];h=J[e+328>>2];if(d>>>0<h>>>0){J[d+8>>2]=-1;J[d+12>>2]=-1;J[d>>2]=4;J[d+16>>2]=-1;J[d+20>>2]=-1;J[e+324>>2]=d+24;break F}P:{Q:{R:{g=J[e+320>>2];j=(d-g|0)/24|0;c=j+1|0;S:{if(c>>>0>=178956971){J[5996]=0;ca(142);break S}h=(h-g|0)/24|0;i=h<<1;h=h>>>0>=89478485?178956970:c>>>0<i>>>0?i:c;if(!h){i=0;break Q}if(h>>>0<178956971){break R}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break a}b=Z()|0;break b}J[5996]=0;i=$(3,P(h,24)|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break P}}c=P(j,24)+i|0;J[c+8>>2]=-1;J[c+12>>2]=-1;J[c>>2]=4;J[c+16>>2]=-1;J[c+20>>2]=-1;j=c+24|0;if((d|0)!=(g|0)){while(1){d=d-24|0;k=J[d+4>>2];c=c-24|0;J[c>>2]=J[d>>2];J[c+4>>2]=k;k=J[d+20>>2];J[c+16>>2]=J[d+16>>2];J[c+20>>2]=k;k=J[d+12>>2];J[c+8>>2]=J[d+8>>2];J[c+12>>2]=k;if((d|0)!=(g|0)){continue}break}d=g;}J[e+328>>2]=P(h,24)+i;J[e+324>>2]=j;J[e+320>>2]=c;if(!d){break F}$a(d);break F}b=Z()|0;break b;case 3:d=J[e+324>>2];h=J[e+328>>2];if(d>>>0<h>>>0){J[d+8>>2]=-1;J[d+12>>2]=-1;J[d>>2]=5;J[d+16>>2]=-1;J[d+20>>2]=-1;J[e+324>>2]=d+24;break F}T:{U:{V:{g=J[e+320>>2];j=(d-g|0)/24|0;c=j+1|0;W:{if(c>>>0>=178956971){J[5996]=0;ca(142);break W}h=(h-g|0)/24|0;i=h<<1;h=h>>>0>=89478485?178956970:c>>>0<i>>>0?i:c;if(!h){i=0;break U}if(h>>>0<178956971){break V}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break a}b=Z()|0;break b}J[5996]=0;i=$(3,P(h,24)|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break T}}c=P(j,24)+i|0;J[c+8>>2]=-1;J[c+12>>2]=-1;J[c>>2]=5;J[c+16>>2]=-1;J[c+20>>2]=-1;j=c+24|0;if((d|0)!=(g|0)){while(1){d=d-24|0;k=J[d+4>>2];c=c-24|0;J[c>>2]=J[d>>2];J[c+4>>2]=k;k=J[d+20>>2];J[c+16>>2]=J[d+16>>2];J[c+20>>2]=k;k=J[d+12>>2];J[c+8>>2]=J[d+8>>2];J[c+12>>2]=k;if((d|0)!=(g|0)){continue}break}d=g;}J[e+328>>2]=P(h,24)+i;J[e+324>>2]=j;J[e+320>>2]=c;if(!d){break F}$a(d);break F}b=Z()|0;break b;case 9:d=J[e+324>>2];h=J[e+328>>2];if(d>>>0<h>>>0){J[d+8>>2]=-1;J[d+12>>2]=-1;J[d>>2]=7;J[d+16>>2]=-1;J[d+20>>2]=-1;J[e+324>>2]=d+24;break F}X:{Y:{Z:{g=J[e+320>>2];j=(d-g|0)/24|0;c=j+1|0;_:{if(c>>>0>=178956971){J[5996]=0;ca(142);break _}h=(h-g|0)/24|0;i=h<<1;h=h>>>0>=89478485?178956970:c>>>0<i>>>0?i:c;if(!h){i=0;break Y}if(h>>>0<178956971){break Z}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break a}b=Z()|0;break b}J[5996]=0;i=$(3,P(h,24)|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break X}}c=P(j,24)+i|0;J[c+8>>2]=-1;J[c+12>>2]=-1;J[c>>2]=7;J[c+16>>2]=-1;J[c+20>>2]=-1;j=c+24|0;if((d|0)!=(g|0)){while(1){d=d-24|0;k=J[d+4>>2];c=c-24|0;J[c>>2]=J[d>>2];J[c+4>>2]=k;k=J[d+20>>2];J[c+16>>2]=J[d+16>>2];J[c+20>>2]=k;k=J[d+12>>2];J[c+8>>2]=J[d+8>>2];J[c+12>>2]=k;if((d|0)!=(g|0)){continue}break}d=g;}J[e+328>>2]=P(h,24)+i;J[e+324>>2]=j;J[e+320>>2]=c;if(!d){break F}$a(d);break F}b=Z()|0;break b;case 13:d=J[e+324>>2];h=J[e+328>>2];if(d>>>0<h>>>0){J[d+8>>2]=-1;J[d+12>>2]=-1;J[d>>2]=8;J[d+16>>2]=-1;J[d+20>>2]=-1;J[e+324>>2]=d+24;break F}$:{aa:{ba:{g=J[e+320>>2];j=(d-g|0)/24|0;c=j+1|0;ca:{if(c>>>0>=178956971){J[5996]=0;ca(142);break ca}h=(h-g|0)/24|0;i=h<<1;h=h>>>0>=89478485?178956970:c>>>0<i>>>0?i:c;if(!h){i=0;break aa}if(h>>>0<178956971){break ba}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break a}b=Z()|0;break b}J[5996]=0;i=$(3,P(h,24)|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break $}}c=P(j,24)+i|0;J[c+8>>2]=-1;J[c+12>>2]=-1;J[c>>2]=8;J[c+16>>2]=-1;J[c+20>>2]=-1;j=c+24|0;if((d|0)!=(g|0)){while(1){d=d-24|0;k=J[d+4>>2];c=c-24|0;J[c>>2]=J[d>>2];J[c+4>>2]=k;k=J[d+20>>2];J[c+16>>2]=J[d+16>>2];J[c+20>>2]=k;k=J[d+12>>2];J[c+8>>2]=J[d+8>>2];J[c+12>>2]=k;if((d|0)!=(g|0)){continue}break}d=g;}J[e+328>>2]=P(h,24)+i;J[e+324>>2]=j;J[e+320>>2]=c;if(!d){break F}$a(d);break F}b=Z()|0;break b;case 21:d=J[e+324>>2];h=J[e+328>>2];if(d>>>0<h>>>0){J[d+8>>2]=-1;J[d+12>>2]=-1;J[d>>2]=6;J[d+16>>2]=-1;J[d+20>>2]=-1;J[e+324>>2]=d+24;break F}da:{ea:{fa:{g=J[e+320>>2];j=(d-g|0)/24|0;c=j+1|0;ga:{if(c>>>0>=178956971){J[5996]=0;ca(142);break ga}h=(h-g|0)/24|0;i=h<<1;h=h>>>0>=89478485?178956970:c>>>0<i>>>0?i:c;if(!h){i=0;break ea}if(h>>>0<178956971){break fa}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break a}b=Z()|0;break b}J[5996]=0;i=$(3,P(h,24)|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break da}}c=P(j,24)+i|0;J[c+8>>2]=-1;J[c+12>>2]=-1;J[c>>2]=6;J[c+16>>2]=-1;J[c+20>>2]=-1;j=c+24|0;if((d|0)!=(g|0)){while(1){d=d-24|0;k=J[d+4>>2];c=c-24|0;J[c>>2]=J[d>>2];J[c+4>>2]=k;k=J[d+20>>2];J[c+16>>2]=J[d+16>>2];J[c+20>>2]=k;k=J[d+12>>2];J[c+8>>2]=J[d+8>>2];J[c+12>>2]=k;if((d|0)!=(g|0)){continue}break}d=g;}J[e+328>>2]=P(h,24)+i;J[e+324>>2]=j;J[e+320>>2]=c;if(!d){break F}$a(d);break F}b=Z()|0;break b;case 20:d=J[e+324>>2];h=J[e+328>>2];if(d>>>0<h>>>0){J[d+8>>2]=-1;J[d+12>>2]=-1;J[d>>2]=9;J[d+16>>2]=-1;J[d+20>>2]=-1;J[e+324>>2]=d+24;break F}ha:{ia:{ja:{g=J[e+320>>2];j=(d-g|0)/24|0;c=j+1|0;ka:{if(c>>>0>=178956971){J[5996]=0;ca(142);break ka}h=(h-g|0)/24|0;i=h<<1;h=h>>>0>=89478485?178956970:c>>>0<i>>>0?i:c;if(!h){i=0;break ia}if(h>>>0<178956971){break ja}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break a}b=Z()|0;break b}J[5996]=0;i=$(3,P(h,24)|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break ha}}c=P(j,24)+i|0;J[c+8>>2]=-1;J[c+12>>2]=-1;J[c>>2]=9;J[c+16>>2]=-1;J[c+20>>2]=-1;j=c+24|0;if((d|0)!=(g|0)){while(1){d=d-24|0;k=J[d+4>>2];c=c-24|0;J[c>>2]=J[d>>2];J[c+4>>2]=k;k=J[d+20>>2];J[c+16>>2]=J[d+16>>2];J[c+20>>2]=k;k=J[d+12>>2];J[c+8>>2]=J[d+8>>2];J[c+12>>2]=k;if((d|0)!=(g|0)){continue}break}d=g;}J[e+328>>2]=P(h,24)+i;J[e+324>>2]=j;J[e+320>>2]=c;if(!d){break F}$a(d);break F}b=Z()|0;break b;case 19:d=J[e+324>>2];h=J[e+328>>2];if(d>>>0<h>>>0){J[d+8>>2]=-1;J[d+12>>2]=-1;J[d>>2]=10;J[d+16>>2]=-1;J[d+20>>2]=-1;J[e+324>>2]=d+24;break F}la:{ma:{na:{g=J[e+320>>2];j=(d-g|0)/24|0;c=j+1|0;oa:{if(c>>>0>=178956971){J[5996]=0;ca(142);break oa}h=(h-g|0)/24|0;i=h<<1;h=h>>>0>=89478485?178956970:c>>>0<i>>>0?i:c;if(!h){i=0;break ma}if(h>>>0<178956971){break na}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break a}b=Z()|0;break b}J[5996]=0;i=$(3,P(h,24)|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break la}}c=P(j,24)+i|0;J[c+8>>2]=-1;J[c+12>>2]=-1;J[c>>2]=10;J[c+16>>2]=-1;J[c+20>>2]=-1;j=c+24|0;if((d|0)!=(g|0)){while(1){d=d-24|0;k=J[d+4>>2];c=c-24|0;J[c>>2]=J[d>>2];J[c+4>>2]=k;k=J[d+20>>2];J[c+16>>2]=J[d+16>>2];J[c+20>>2]=k;k=J[d+12>>2];J[c+8>>2]=J[d+8>>2];J[c+12>>2]=k;if((d|0)!=(g|0)){continue}break}d=g;}J[e+328>>2]=P(h,24)+i;J[e+324>>2]=j;J[e+320>>2]=c;if(!d){break F}$a(d);break F}b=Z()|0;break b;case 11:break G;default:break B}}J[5996]=0;d=$(3,104)|0;c=J[5996];J[5996]=0;if((c|0)==1){break E}J[d>>2]=7e3;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;g=ba(143,d+12|0,0,0)|0;c=J[5996];J[5996]=0;if((c|0)==1){b=Z()|0;$a(d);break b}J[e+244>>2]=d;J[e+240>>2]=g;c=J[e+312>>2];pa:{if(c>>>0<M[e+316>>2]){J[c+4>>2]=d;J[c>>2]=g;J[e+312>>2]=c+8;break pa}J[5996]=0;aa(144,e+308|0,e+240|0);d=J[5996];J[5996]=0;if((d|0)==1){break D}d=J[e+244>>2];if(!d){break pa}c=J[d+4>>2];J[d+4>>2]=c-1;if(c){break pa}Xa[J[J[d>>2]+8>>2]](d);bb(d);}d=J[e+324>>2];h=J[e+328>>2];if(d>>>0<h>>>0){J[d+8>>2]=-1;J[d+12>>2]=-1;J[d>>2]=18;J[d+16>>2]=-1;J[d+20>>2]=-1;J[e+324>>2]=d+24;break F}qa:{ra:{g=J[e+320>>2];j=(d-g|0)/24|0;c=j+1|0;sa:{if(c>>>0>=178956971){J[5996]=0;ca(142);break sa}h=(h-g|0)/24|0;i=h<<1;h=h>>>0>=89478485?178956970:c>>>0<i>>>0?i:c;if(!h){i=0;break qa}if(h>>>0<178956971){break ra}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break a}b=Z()|0;break b}J[5996]=0;i=$(3,P(h,24)|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break C}}c=P(j,24)+i|0;J[c+8>>2]=-1;J[c+12>>2]=-1;J[c>>2]=18;J[c+16>>2]=-1;J[c+20>>2]=-1;j=c+24|0;if((d|0)!=(g|0)){while(1){d=d-24|0;k=J[d+4>>2];c=c-24|0;J[c>>2]=J[d>>2];J[c+4>>2]=k;k=J[d+20>>2];J[c+16>>2]=J[d+16>>2];J[c+20>>2]=k;k=J[d+12>>2];J[c+8>>2]=J[d+8>>2];J[c+12>>2]=k;if((d|0)!=(g|0)){continue}break}d=g;}J[e+328>>2]=P(h,24)+i;J[e+324>>2]=j;J[e+320>>2]=c;if(!d){break F}$a(d);}l=l+1|0;if((l|0)!=(f|0)){continue}break}d=J[e+324>>2];break A}b=Z()|0;break b}b=Z()|0;mb(e+240|0);break b}b=Z()|0;break b}J[a>>2]=0;J[a+4>>2]=0;break l}if((h-97&255)>>>0>25){f=-1;break A}f=f+1|0;if((c|0)>(f|0)){continue}break}f=-1;}ta:{h=J[e+328>>2];ua:{if(h>>>0>d>>>0){J[d+8>>2]=-1;J[d+12>>2]=-1;J[d>>2]=1;J[d+16>>2]=-1;J[d+20>>2]=-1;J[e+324>>2]=d+24;break ua}va:{wa:{g=J[e+320>>2];l=(d-g|0)/24|0;c=l+1|0;xa:{if(c>>>0>=178956971){J[5996]=0;ca(142);break xa}h=(h-g|0)/24|0;i=h<<1;h=h>>>0>=89478485?178956970:c>>>0<i>>>0?i:c;if(!h){i=0;break va}if(h>>>0<178956971){break wa}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break a}b=Z()|0;break b}J[5996]=0;i=$(3,P(h,24)|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break ta}}c=P(l,24)+i|0;J[c+8>>2]=-1;J[c+12>>2]=-1;J[c>>2]=1;J[c+16>>2]=-1;J[c+20>>2]=-1;l=c+24|0;if((d|0)!=(g|0)){while(1){d=d-24|0;j=J[d+4>>2];c=c-24|0;J[c>>2]=J[d>>2];J[c+4>>2]=j;j=J[d+20>>2];J[c+16>>2]=J[d+16>>2];J[c+20>>2]=j;j=J[d+12>>2];J[c+8>>2]=J[d+8>>2];J[c+12>>2]=j;if((d|0)!=(g|0)){continue}break}d=g;}J[e+328>>2]=P(h,24)+i;J[e+324>>2]=l;J[e+320>>2]=c;if(!d){break ua}$a(d);}d=f+1|0;J[b+4>>2]=J[b+4>>2]-d;J[b>>2]=d+J[b>>2];break m}b=Z()|0;break b;case 21:ya:{d=J[b+4>>2];if(d){c=J[b>>2];if(K[c|0]==41){break ya}}J[a>>2]=0;J[a+4>>2]=0;break l}J[b+4>>2]=d-1;J[b>>2]=c+1;break m;case 20:za:{d=J[b+4>>2];if(d){c=J[b>>2];if(K[c|0]==44){break za}}J[a>>2]=0;J[a+4>>2]=0;break l}J[b+4>>2]=d-1;J[b>>2]=c+1;break m;case 19:g=J[b+4>>2];if(!g){J[a>>2]=0;J[a+4>>2]=0;break l}Aa:{Ba:{Ca:{Da:{Ea:{Fa:{h=J[b>>2];switch(K[h|0]-41|0){case 0:break Ea;case 3:break Fa;default:break Da}}J[b+4>>2]=g-1;J[b>>2]=h+1;g=J[e+292>>2];h=J[e+288>>2]+1|0;g=h?g:g+1|0;Ga:{Ha:{i=J[e+328>>2];Ia:{if(i>>>0>d>>>0){J[d>>2]=19;J[d+8>>2]=h;J[d+12>>2]=g;f=J[e+300>>2];J[d+16>>2]=J[e+296>>2];J[d+20>>2]=f;J[e+324>>2]=c;break Ia}Ja:{Ka:{l=(d-f|0)/24|0;c=l+1|0;La:{if(c>>>0>=178956971){J[5996]=0;ca(142);break La}i=(i-f|0)/24|0;j=i<<1;c=i>>>0>=89478485?178956970:c>>>0<j>>>0?j:c;if(!c){i=0;break Ja}if(c>>>0<178956971){break Ka}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break a}b=Z()|0;break b}J[5996]=0;i=$(3,P(c,24)|0)|0;j=J[5996];J[5996]=0;if((j|0)==1){break Ha}}l=P(l,24)+i|0;J[l+8>>2]=h;J[l+12>>2]=g;J[l>>2]=19;g=J[e+300>>2];J[l+16>>2]=J[e+296>>2];J[l+20>>2]=g;i=P(c,24)+i|0;c=l+24|0;Ma:{if((d|0)==(f|0)){J[e+328>>2]=i;J[e+324>>2]=c;J[e+320>>2]=l;break Ma}while(1){d=d-24|0;h=J[d+4>>2];l=l-24|0;g=l;J[g>>2]=J[d>>2];J[g+4>>2]=h;g=J[d+20>>2];J[l+16>>2]=J[d+16>>2];J[l+20>>2]=g;g=J[d+12>>2];J[l+8>>2]=J[d+8>>2];J[l+12>>2]=g;if((d|0)!=(f|0)){continue}break}J[e+328>>2]=i;J[e+324>>2]=c;J[e+320>>2]=l;if(!f){break Ia}}$a(f);i=J[e+328>>2];c=J[e+324>>2];}if(c>>>0<i>>>0){J[c+8>>2]=-1;J[c+12>>2]=-1;J[c>>2]=0;J[c+16>>2]=-1;J[c+20>>2]=-1;J[e+324>>2]=c+24;break m}Na:{Oa:{f=J[e+320>>2];h=(c-f|0)/24|0;d=h+1|0;Pa:{if(d>>>0>=178956971){J[5996]=0;ca(142);break Pa}g=(i-f|0)/24|0;i=g<<1;g=g>>>0>=89478485?178956970:d>>>0<i>>>0?i:d;if(!g){i=0;break Na}if(g>>>0<178956971){break Oa}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break a}b=Z()|0;break b}J[5996]=0;i=$(3,P(g,24)|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Ga}}d=P(h,24)+i|0;J[d+8>>2]=-1;J[d+12>>2]=-1;J[d>>2]=0;J[d+16>>2]=-1;J[d+20>>2]=-1;h=d+24|0;if((c|0)!=(f|0)){while(1){c=c-24|0;l=J[c+4>>2];d=d-24|0;J[d>>2]=J[c>>2];J[d+4>>2]=l;l=J[c+20>>2];J[d+16>>2]=J[c+16>>2];J[d+20>>2]=l;l=J[c+12>>2];J[d+8>>2]=J[c+8>>2];J[d+12>>2]=l;if((c|0)!=(f|0)){continue}break}c=f;}J[e+328>>2]=P(g,24)+i;J[e+324>>2]=h;J[e+320>>2]=d;if(!c){break m}$a(c);break m}b=Z()|0;break b}b=Z()|0;break b}d=J[e+288>>2];c=J[e+292>>2];f=J[e+300>>2];if(d>>>0<M[e+296>>2]&(c|0)<=(f|0)|(c|0)<(f|0)){J[a>>2]=0;J[a+4>>2]=0;break l}J[b+4>>2]=g-1;J[b>>2]=h+1;J[e+340>>2]=0;J[e+332>>2]=0;J[e+336>>2]=0;if(!d&(c|0)<=0|(c|0)<0){break Ba}f=0;c=0;d=J[e+312>>2];while(1){Qa:{g=d-8|0;h=J[e+336>>2];Ra:{if(h>>>0<M[e+340>>2]){J[h>>2]=J[g>>2];i=d-4|0;J[h+4>>2]=J[i>>2];J[g>>2]=0;J[i>>2]=0;J[e+336>>2]=h+8;break Ra}J[5996]=0;aa(144,e+332|0,g|0);d=J[5996];J[5996]=0;if((d|0)==1){break Qa}d=J[e+312>>2];g=J[d-4>>2];if(!g){break Ra}h=J[g+4>>2];J[g+4>>2]=h-1;if(h){break Ra}Xa[J[J[g>>2]+8>>2]](g);bb(g);}d=d-8|0;J[e+312>>2]=d;f=f+1|0;c=f?c:c+1|0;g=J[e+292>>2];if(M[e+288>>2]>f>>>0&(g|0)>=(c|0)|(c|0)<(g|0)){continue}break Ca}break}b=Z()|0;break Aa}J[a>>2]=0;J[a+4>>2]=0;break l}d=J[e+332>>2];f=J[e+336>>2];if((d|0)==(f|0)){break Ba}c=f-8|0;if(c>>>0<=d>>>0){break Ba}while(1){g=J[d>>2];J[d>>2]=J[c>>2];J[c>>2]=g;g=J[d+4>>2];f=f-4|0;J[d+4>>2]=J[f>>2];J[f>>2]=g;f=c;d=d+8|0;c=c-8|0;if(d>>>0<c>>>0){continue}break}}J[5996]=0;f=$(3,104)|0;d=J[5996];J[5996]=0;Sa:{Ta:{if((d|0)!=1){J[f>>2]=7e3;J[f+4>>2]=0;J[f+8>>2]=0;J[5996]=0;J[e+240>>2]=J[e+332>>2];d=J[e+336>>2];c=J[e+340>>2];J[e+340>>2]=0;J[e+244>>2]=d;J[e+248>>2]=c;J[e+332>>2]=0;J[e+336>>2]=0;h=ia(64,f+12|0,24,e+240|0,J[e+296>>2])|0;d=J[5996];J[5996]=0;if((d|0)==1){break Sa}c=J[e+240>>2];if(c){d=J[e+244>>2];if((c|0)!=(d|0)){while(1){d=d-8|0;g=J[d+4>>2];Ua:{if(!g){break Ua}i=J[g+4>>2];J[g+4>>2]=i-1;if(i){break Ua}Xa[J[J[g>>2]+8>>2]](g);bb(g);}if((c|0)!=(d|0)){continue}break}c=J[e+240>>2];}$a(c);}J[e+348>>2]=f;J[e+344>>2]=h;d=J[e+312>>2];Va:{if(d>>>0<M[e+316>>2]){J[d+4>>2]=f;J[d>>2]=h;J[e+312>>2]=d+8;break Va}J[5996]=0;aa(144,e+308|0,e+344|0);d=J[5996];J[5996]=0;if((d|0)==1){break Ta}d=J[e+348>>2];if(!d){break Va}c=J[d+4>>2];J[d+4>>2]=c-1;if(c){break Va}Xa[J[J[d>>2]+8>>2]](d);bb(d);}c=J[e+332>>2];if(!c){break m}d=J[e+336>>2];if((c|0)!=(d|0)){while(1){d=d-8|0;f=J[d+4>>2];Wa:{if(!f){break Wa}g=J[f+4>>2];J[f+4>>2]=g-1;if(g){break Wa}Xa[J[J[f>>2]+8>>2]](f);bb(f);}if((c|0)!=(d|0)){continue}break}c=J[e+332>>2];}$a(c);break m}b=Z()|0;break Aa}b=Z()|0;mb(e+344|0);break Aa}b=Z()|0;lb(e+240|0);$a(f);}lb(e+332|0);break b;case 14:c=J[e+312>>2];d=c-8|0;J[e+332>>2]=J[d>>2];J[e+336>>2]=J[d+4>>2];J[d>>2]=0;J[d+4>>2]=0;d=c-16|0;J[e+344>>2]=J[d>>2];J[e+348>>2]=J[d+4>>2];J[d>>2]=0;J[d+4>>2]=0;J[e+312>>2]=d;J[5996]=0;f=e+240|0;ga(145,f|0,c-24|0,e+344|0,e+332|0);d=J[5996];J[5996]=0;Xa:{Ya:{Za:{if((d|0)!=1){J[5996]=0;d=$(3,104)|0;c=J[5996];J[5996]=0;if((c|0)==1){break Za}J[5996]=0;c=ba(146,d|0,23,f|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){b=Z()|0;$a(d);break Ya}f=J[e+312>>2]-8|0;J[f>>2]=c+12;d=J[f+4>>2];J[f+4>>2]=c;_a:{if(!d){break _a}c=J[d+4>>2];J[d+4>>2]=c-1;if(c){break _a}Xa[J[J[d>>2]+8>>2]](d);bb(d);}f=J[e+240>>2];if(f){d=J[e+244>>2];c=f;if((d|0)!=(c|0)){while(1){d=d-8|0;c=J[d+4>>2];$a:{if(!c){break $a}g=J[c+4>>2];J[c+4>>2]=g-1;if(g){break $a}Xa[J[J[c>>2]+8>>2]](c);bb(c);}if((d|0)!=(f|0)){continue}break}c=J[e+240>>2];}J[e+244>>2]=f;$a(c);}d=J[e+348>>2];ab:{if(!d){break ab}c=J[d+4>>2];J[d+4>>2]=c-1;if(c){break ab}Xa[J[J[d>>2]+8>>2]](d);bb(d);}d=J[e+336>>2];if(!d){break m}c=J[d+4>>2];J[d+4>>2]=c-1;if(c){break m}Xa[J[J[d>>2]+8>>2]](d);bb(d);break m}b=Z()|0;break Xa}b=Z()|0;}lb(e+240|0);}mb(e+344|0);mb(e+332|0);break b;case 18:l=22;break y;case 17:l=21;break y;case 16:l=20;break y;case 15:l=19;break y;case 1:break n;case 3:break o;case 2:break p;case 4:break q;case 5:break r;case 7:break s;case 8:break t;case 6:break u;case 9:break v;case 10:break w;case 11:break x;case 13:break y;case 12:break z;default:break m}}l=17;}f=J[e+312>>2];c=f-8|0;J[e+344>>2]=J[c>>2];J[e+348>>2]=J[c+4>>2];J[c>>2]=0;J[c+4>>2]=0;J[e+312>>2]=c;J[5996]=0;J[e+340>>2]=0;J[e+332>>2]=0;J[e+336>>2]=0;d=$(3,16)|0;g=J[5996];J[5996]=0;bb:{if((g|0)==1){b=Z()|0;break bb}J[e+332>>2]=d;g=d+16|0;J[e+340>>2]=g;f=f-16|0;J[d>>2]=J[f>>2];J[d+4>>2]=J[f+4>>2];J[f>>2]=0;J[f+4>>2]=0;J[d+8>>2]=J[e+344>>2];J[d+12>>2]=J[e+348>>2];J[e+336>>2]=g;J[e+344>>2]=0;J[e+348>>2]=0;J[5996]=0;f=$(3,104)|0;h=J[5996];J[5996]=0;if((h|0)!=1){J[f>>2]=7e3;J[f+4>>2]=0;J[f+8>>2]=0;J[e+244>>2]=g;J[e+240>>2]=d;J[e+332>>2]=0;J[e+336>>2]=0;J[5996]=0;d=J[e+340>>2];J[e+340>>2]=0;J[e+248>>2]=d;g=ia(64,f+12|0,l|0,e+240|0,0)|0;d=J[5996];J[5996]=0;if((d|0)!=1){l=J[e+240>>2];if(l){d=J[e+244>>2];if((l|0)!=(d|0)){while(1){d=d-8|0;c=J[d+4>>2];cb:{if(!c){break cb}h=J[c+4>>2];J[c+4>>2]=h-1;if(h){break cb}Xa[J[J[c>>2]+8>>2]](c);bb(c);}if((d|0)!=(l|0)){continue}break}l=J[e+240>>2];c=J[e+312>>2];}$a(l);}c=c-8|0;J[c>>2]=g;d=J[c+4>>2];J[c+4>>2]=f;db:{if(!d){break db}c=J[d+4>>2];J[d+4>>2]=c-1;if(c){break db}Xa[J[J[d>>2]+8>>2]](d);bb(d);}c=J[e+332>>2];if(c){d=J[e+336>>2];if((c|0)!=(d|0)){while(1){d=d-8|0;f=J[d+4>>2];eb:{if(!f){break eb}g=J[f+4>>2];J[f+4>>2]=g-1;if(g){break eb}Xa[J[J[f>>2]+8>>2]](f);bb(f);}if((c|0)!=(d|0)){continue}break}c=J[e+332>>2];}$a(c);}d=J[e+348>>2];if(!d){break m}c=J[d+4>>2];J[d+4>>2]=c-1;if(c){break m}Xa[J[J[d>>2]+8>>2]](d);bb(d);break m}b=Z()|0;lb(e+240|0);$a(f);break bb}b=Z()|0;}lb(e+332|0);mb(e+344|0);break b}c=J[e+312>>2];d=c-8|0;J[e+332>>2]=J[d>>2];J[e+336>>2]=J[d+4>>2];J[d>>2]=0;J[d+4>>2]=0;J[e+312>>2]=d;J[5996]=0;d=$(3,104)|0;f=J[5996];J[5996]=0;fb:{gb:{hb:{ib:{jb:{if((f|0)!=1){J[d>>2]=7e3;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;f=ba(143,d+12|0,0,0)|0;g=J[5996];J[5996]=0;if((g|0)==1){b=Z()|0;$a(d);break fb}J[e+348>>2]=d;J[e+344>>2]=f;J[5996]=0;f=e+240|0;ga(145,f|0,c-16|0,e+332|0,e+344|0);d=J[5996];J[5996]=0;if((d|0)==1){break jb}J[5996]=0;d=$(3,104)|0;c=J[5996];J[5996]=0;if((c|0)==1){break ib}J[5996]=0;c=ba(146,d|0,23,f|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){b=Z()|0;$a(d);break hb}f=J[e+312>>2]-8|0;J[f>>2]=c+12;d=J[f+4>>2];J[f+4>>2]=c;kb:{if(!d){break kb}c=J[d+4>>2];J[d+4>>2]=c-1;if(c){break kb}Xa[J[J[d>>2]+8>>2]](d);bb(d);}f=J[e+240>>2];if(f){d=J[e+244>>2];c=f;if((d|0)!=(c|0)){while(1){d=d-8|0;c=J[d+4>>2];lb:{if(!c){break lb}g=J[c+4>>2];J[c+4>>2]=g-1;if(g){break lb}Xa[J[J[c>>2]+8>>2]](c);bb(c);}if((d|0)!=(f|0)){continue}break}c=J[e+240>>2];}J[e+244>>2]=f;$a(c);}d=J[e+348>>2];mb:{if(!d){break mb}c=J[d+4>>2];J[d+4>>2]=c-1;if(c){break mb}Xa[J[J[d>>2]+8>>2]](d);bb(d);}d=J[e+336>>2];if(!d){break m}c=J[d+4>>2];J[d+4>>2]=c-1;if(c){break m}Xa[J[J[d>>2]+8>>2]](d);bb(d);break m}b=Z()|0;break fb}b=Z()|0;break gb}b=Z()|0;}lb(e+240|0);}mb(e+344|0);}mb(e+332|0);break b}J[5996]=0;c=J[e+312>>2];d=$(3,104)|0;f=J[5996];J[5996]=0;nb:{ob:{if((f|0)!=1){J[d>>2]=7e3;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;f=ba(143,d+12|0,1,0)|0;g=J[5996];J[5996]=0;if((g|0)==1){b=Z()|0;$a(d);break b}J[e+336>>2]=d;J[e+332>>2]=f;J[e+248>>2]=0;J[e+240>>2]=0;J[e+244>>2]=0;J[5996]=0;d=$(3,16)|0;f=J[5996];J[5996]=0;if((f|0)==1){b=Z()|0;break nb}J[e+240>>2]=d;g=d+16|0;J[e+248>>2]=g;f=c-8|0;J[d>>2]=J[f>>2];c=c-4|0;J[d+4>>2]=J[c>>2];J[f>>2]=0;J[c>>2]=0;J[d+8>>2]=J[e+332>>2];J[d+12>>2]=J[e+336>>2];J[e+244>>2]=g;J[e+332>>2]=0;J[e+336>>2]=0;J[5996]=0;d=$(3,104)|0;g=J[5996];J[5996]=0;if((g|0)==1){break ob}J[5996]=0;g=ba(146,d|0,17,e+240|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){b=Z()|0;$a(d);break nb}J[f>>2]=g+12;d=J[c>>2];J[c>>2]=g;pb:{if(!d){break pb}c=J[d+4>>2];J[d+4>>2]=c-1;if(c){break pb}Xa[J[J[d>>2]+8>>2]](d);bb(d);}f=J[e+240>>2];if(f){d=J[e+244>>2];c=f;if((d|0)!=(c|0)){while(1){d=d-8|0;c=J[d+4>>2];qb:{if(!c){break qb}g=J[c+4>>2];J[c+4>>2]=g-1;if(g){break qb}Xa[J[J[c>>2]+8>>2]](c);bb(c);}if((d|0)!=(f|0)){continue}break}c=J[e+240>>2];}J[e+244>>2]=f;$a(c);}d=J[e+336>>2];if(!d){break m}c=J[d+4>>2];J[d+4>>2]=c-1;if(c){break m}Xa[J[J[d>>2]+8>>2]](d);bb(d);break m}b=Z()|0;break b}b=Z()|0;}lb(e+240|0);mb(e+332|0);break b}J[5996]=0;c=J[e+312>>2];d=$(3,104)|0;f=J[5996];J[5996]=0;rb:{sb:{if((f|0)!=1){J[d>>2]=7e3;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;f=ba(143,d+12|0,0,0)|0;g=J[5996];J[5996]=0;if((g|0)==1){b=Z()|0;$a(d);break b}J[e+336>>2]=d;J[e+332>>2]=f;J[e+248>>2]=0;J[e+240>>2]=0;J[e+244>>2]=0;J[5996]=0;d=$(3,16)|0;f=J[5996];J[5996]=0;if((f|0)==1){b=Z()|0;break rb}J[e+240>>2]=d;g=d+16|0;J[e+248>>2]=g;f=c-8|0;J[d>>2]=J[f>>2];c=c-4|0;J[d+4>>2]=J[c>>2];J[f>>2]=0;J[c>>2]=0;J[d+8>>2]=J[e+332>>2];J[d+12>>2]=J[e+336>>2];J[e+244>>2]=g;J[e+332>>2]=0;J[e+336>>2]=0;J[5996]=0;d=$(3,104)|0;g=J[5996];J[5996]=0;if((g|0)==1){break sb}J[5996]=0;g=ba(146,d|0,22,e+240|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){b=Z()|0;$a(d);break rb}J[f>>2]=g+12;d=J[c>>2];J[c>>2]=g;tb:{if(!d){break tb}c=J[d+4>>2];J[d+4>>2]=c-1;if(c){break tb}Xa[J[J[d>>2]+8>>2]](d);bb(d);}f=J[e+240>>2];if(f){d=J[e+244>>2];c=f;if((d|0)!=(c|0)){while(1){d=d-8|0;c=J[d+4>>2];ub:{if(!c){break ub}g=J[c+4>>2];J[c+4>>2]=g-1;if(g){break ub}Xa[J[J[c>>2]+8>>2]](c);bb(c);}if((d|0)!=(f|0)){continue}break}c=J[e+240>>2];}J[e+244>>2]=f;$a(c);}d=J[e+336>>2];if(!d){break m}c=J[d+4>>2];J[d+4>>2]=c-1;if(c){break m}Xa[J[J[d>>2]+8>>2]](d);bb(d);break m}b=Z()|0;break b}b=Z()|0;}lb(e+240|0);mb(e+332|0);break b}J[e+248>>2]=0;J[e+240>>2]=0;J[e+244>>2]=0;J[5996]=0;d=J[e+312>>2];c=$(3,8)|0;f=J[5996];J[5996]=0;vb:{if((f|0)!=1){J[e+240>>2]=c;f=c+8|0;J[e+248>>2]=f;d=d-8|0;J[c>>2]=J[d>>2];J[c+4>>2]=J[d+4>>2];J[d>>2]=0;J[d+4>>2]=0;J[5996]=0;J[e+244>>2]=f;c=$(3,104)|0;f=J[5996];J[5996]=0;if((f|0)!=1){J[5996]=0;f=ba(146,c|0,14,e+240|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){b=Z()|0;$a(c);break vb}J[d>>2]=f+12;c=J[d+4>>2];J[d+4>>2]=f;wb:{if(!c){break wb}d=J[c+4>>2];J[c+4>>2]=d-1;if(d){break wb}Xa[J[J[c>>2]+8>>2]](c);bb(c);}c=J[e+240>>2];if(!c){break m}d=J[e+244>>2];if((c|0)!=(d|0)){while(1){d=d-8|0;f=J[d+4>>2];xb:{if(!f){break xb}g=J[f+4>>2];J[f+4>>2]=g-1;if(g){break xb}Xa[J[J[f>>2]+8>>2]](f);bb(f);}if((c|0)!=(d|0)){continue}break}c=J[e+240>>2];}$a(c);break m}b=Z()|0;break vb}b=Z()|0;}lb(e+240|0);break b}J[e+248>>2]=0;J[e+240>>2]=0;J[e+244>>2]=0;J[5996]=0;d=J[e+312>>2];c=$(3,8)|0;f=J[5996];J[5996]=0;yb:{if((f|0)!=1){J[e+240>>2]=c;f=c+8|0;J[e+248>>2]=f;d=d-8|0;J[c>>2]=J[d>>2];J[c+4>>2]=J[d+4>>2];J[d>>2]=0;J[d+4>>2]=0;J[5996]=0;J[e+244>>2]=f;c=$(3,104)|0;f=J[5996];J[5996]=0;if((f|0)!=1){J[5996]=0;f=ba(146,c|0,16,e+240|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){b=Z()|0;$a(c);break yb}J[d>>2]=f+12;c=J[d+4>>2];J[d+4>>2]=f;zb:{if(!c){break zb}d=J[c+4>>2];J[c+4>>2]=d-1;if(d){break zb}Xa[J[J[c>>2]+8>>2]](c);bb(c);}c=J[e+240>>2];if(!c){break m}d=J[e+244>>2];if((c|0)!=(d|0)){while(1){d=d-8|0;f=J[d+4>>2];Ab:{if(!f){break Ab}g=J[f+4>>2];J[f+4>>2]=g-1;if(g){break Ab}Xa[J[J[f>>2]+8>>2]](f);bb(f);}if((c|0)!=(d|0)){continue}break}c=J[e+240>>2];}$a(c);break m}b=Z()|0;break yb}b=Z()|0;}lb(e+240|0);break b}J[e+248>>2]=0;J[e+240>>2]=0;J[e+244>>2]=0;J[5996]=0;d=J[e+312>>2];c=$(3,8)|0;f=J[5996];J[5996]=0;Bb:{if((f|0)!=1){J[e+240>>2]=c;f=c+8|0;J[e+248>>2]=f;d=d-8|0;J[c>>2]=J[d>>2];J[c+4>>2]=J[d+4>>2];J[d>>2]=0;J[d+4>>2]=0;J[5996]=0;J[e+244>>2]=f;c=$(3,104)|0;f=J[5996];J[5996]=0;if((f|0)!=1){J[5996]=0;f=ba(146,c|0,15,e+240|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){b=Z()|0;$a(c);break Bb}J[d>>2]=f+12;c=J[d+4>>2];J[d+4>>2]=f;Cb:{if(!c){break Cb}d=J[c+4>>2];J[c+4>>2]=d-1;if(d){break Cb}Xa[J[J[c>>2]+8>>2]](c);bb(c);}c=J[e+240>>2];if(!c){break m}d=J[e+244>>2];if((c|0)!=(d|0)){while(1){d=d-8|0;f=J[d+4>>2];Db:{if(!f){break Db}g=J[f+4>>2];J[f+4>>2]=g-1;if(g){break Db}Xa[J[J[f>>2]+8>>2]](f);bb(f);}if((c|0)!=(d|0)){continue}break}c=J[e+240>>2];}$a(c);break m}b=Z()|0;break Bb}b=Z()|0;}lb(e+240|0);break b}J[e+248>>2]=0;J[e+240>>2]=0;J[e+244>>2]=0;J[5996]=0;d=J[e+312>>2];c=$(3,8)|0;f=J[5996];J[5996]=0;Eb:{if((f|0)!=1){J[e+240>>2]=c;f=c+8|0;J[e+248>>2]=f;d=d-8|0;J[c>>2]=J[d>>2];J[c+4>>2]=J[d+4>>2];J[d>>2]=0;J[d+4>>2]=0;J[5996]=0;J[e+244>>2]=f;c=$(3,104)|0;f=J[5996];J[5996]=0;if((f|0)!=1){J[5996]=0;f=ba(146,c|0,13,e+240|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){b=Z()|0;$a(c);break Eb}J[d>>2]=f+12;c=J[d+4>>2];J[d+4>>2]=f;Fb:{if(!c){break Fb}d=J[c+4>>2];J[c+4>>2]=d-1;if(d){break Fb}Xa[J[J[c>>2]+8>>2]](c);bb(c);}c=J[e+240>>2];if(!c){break m}d=J[e+244>>2];if((c|0)!=(d|0)){while(1){d=d-8|0;f=J[d+4>>2];Gb:{if(!f){break Gb}g=J[f+4>>2];J[f+4>>2]=g-1;if(g){break Gb}Xa[J[J[f>>2]+8>>2]](f);bb(f);}if((c|0)!=(d|0)){continue}break}c=J[e+240>>2];}$a(c);break m}b=Z()|0;break Eb}b=Z()|0;}lb(e+240|0);break b}J[e+248>>2]=0;J[e+240>>2]=0;J[e+244>>2]=0;J[5996]=0;d=J[e+312>>2];c=$(3,8)|0;f=J[5996];J[5996]=0;Hb:{if((f|0)!=1){J[e+240>>2]=c;f=c+8|0;J[e+248>>2]=f;d=d-8|0;J[c>>2]=J[d>>2];J[c+4>>2]=J[d+4>>2];J[d>>2]=0;J[d+4>>2]=0;J[5996]=0;J[e+244>>2]=f;c=$(3,104)|0;f=J[5996];J[5996]=0;if((f|0)!=1){J[5996]=0;f=ba(146,c|0,12,e+240|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){b=Z()|0;$a(c);break Hb}J[d>>2]=f+12;c=J[d+4>>2];J[d+4>>2]=f;Ib:{if(!c){break Ib}d=J[c+4>>2];J[c+4>>2]=d-1;if(d){break Ib}Xa[J[J[c>>2]+8>>2]](c);bb(c);}c=J[e+240>>2];if(!c){break m}d=J[e+244>>2];if((c|0)!=(d|0)){while(1){d=d-8|0;f=J[d+4>>2];Jb:{if(!f){break Jb}g=J[f+4>>2];J[f+4>>2]=g-1;if(g){break Jb}Xa[J[J[f>>2]+8>>2]](f);bb(f);}if((c|0)!=(d|0)){continue}break}c=J[e+240>>2];}$a(c);break m}b=Z()|0;break Hb}b=Z()|0;}lb(e+240|0);break b}J[e+248>>2]=0;J[e+240>>2]=0;J[e+244>>2]=0;J[5996]=0;d=J[e+312>>2];c=$(3,8)|0;f=J[5996];J[5996]=0;Kb:{if((f|0)!=1){J[e+240>>2]=c;f=c+8|0;J[e+248>>2]=f;d=d-8|0;J[c>>2]=J[d>>2];J[c+4>>2]=J[d+4>>2];J[d>>2]=0;J[d+4>>2]=0;J[5996]=0;J[e+244>>2]=f;c=$(3,104)|0;f=J[5996];J[5996]=0;if((f|0)!=1){J[5996]=0;f=ba(146,c|0,11,e+240|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){b=Z()|0;$a(c);break Kb}J[d>>2]=f+12;c=J[d+4>>2];J[d+4>>2]=f;Lb:{if(!c){break Lb}d=J[c+4>>2];J[c+4>>2]=d-1;if(d){break Lb}Xa[J[J[c>>2]+8>>2]](c);bb(c);}c=J[e+240>>2];if(!c){break m}d=J[e+244>>2];if((c|0)!=(d|0)){while(1){d=d-8|0;f=J[d+4>>2];Mb:{if(!f){break Mb}g=J[f+4>>2];J[f+4>>2]=g-1;if(g){break Mb}Xa[J[J[f>>2]+8>>2]](f);bb(f);}if((c|0)!=(d|0)){continue}break}c=J[e+240>>2];}$a(c);break m}b=Z()|0;break Kb}b=Z()|0;}lb(e+240|0);break b}J[e+248>>2]=0;J[e+240>>2]=0;J[e+244>>2]=0;J[5996]=0;d=J[e+312>>2];c=$(3,8)|0;f=J[5996];J[5996]=0;Nb:{if((f|0)!=1){J[e+240>>2]=c;f=c+8|0;J[e+248>>2]=f;d=d-8|0;J[c>>2]=J[d>>2];J[c+4>>2]=J[d+4>>2];J[d>>2]=0;J[d+4>>2]=0;J[5996]=0;J[e+244>>2]=f;c=$(3,104)|0;f=J[5996];J[5996]=0;if((f|0)!=1){J[5996]=0;f=ba(146,c|0,10,e+240|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){b=Z()|0;$a(c);break Nb}J[d>>2]=f+12;c=J[d+4>>2];J[d+4>>2]=f;Ob:{if(!c){break Ob}d=J[c+4>>2];J[c+4>>2]=d-1;if(d){break Ob}Xa[J[J[c>>2]+8>>2]](c);bb(c);}c=J[e+240>>2];if(!c){break m}d=J[e+244>>2];if((c|0)!=(d|0)){while(1){d=d-8|0;f=J[d+4>>2];Pb:{if(!f){break Pb}g=J[f+4>>2];J[f+4>>2]=g-1;if(g){break Pb}Xa[J[J[f>>2]+8>>2]](f);bb(f);}if((c|0)!=(d|0)){continue}break}c=J[e+240>>2];}$a(c);break m}b=Z()|0;break Nb}b=Z()|0;}lb(e+240|0);break b}I[e+240>>1]=48;H[e+251|0]=1;Qb:{Rb:{Sb:{Tb:{d=J[b+4>>2];if(!d){H[e+251|0]=3;H[e+243|0]=0;I[e+240>>1]=K[5279]|K[5280]<<8;H[e+242|0]=K[5281];break Tb}c=J[b>>2];if(K[e+240|0]==K[c|0]){J[b+4>>2]=d-1;J[b>>2]=c+1;J[5996]=0;d=$(3,104)|0;c=J[5996];J[5996]=0;Ub:{if((c|0)!=1){J[d>>2]=7e3;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;f=ba(143,d+12|0,0,0)|0;c=J[5996];J[5996]=0;if((c|0)==1){b=Z()|0;$a(d);break b}J[e+244>>2]=d;J[e+240>>2]=f;c=J[e+312>>2];if(c>>>0<M[e+316>>2]){J[c+4>>2]=d;J[c>>2]=f;J[e+312>>2]=c+8;break m}J[5996]=0;aa(144,e+308|0,e+240|0);d=J[5996];J[5996]=0;if((d|0)==1){break Ub}d=J[e+244>>2];if(!d){break m}c=J[d+4>>2];J[d+4>>2]=c-1;if(c){break m}Xa[J[J[d>>2]+8>>2]](d);bb(d);break m}b=Z()|0;break b}b=Z()|0;mb(e+240|0);break b}H[e+251|0]=1;I[e+240>>1]=49;c=J[b>>2];if(K[c|0]==49){J[b+4>>2]=d-1;J[b>>2]=c+1;J[5996]=0;d=$(3,104)|0;c=J[5996];J[5996]=0;Vb:{if((c|0)!=1){J[d>>2]=7e3;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;f=ba(143,d+12|0,1,0)|0;c=J[5996];J[5996]=0;if((c|0)==1){b=Z()|0;$a(d);break b}J[e+244>>2]=d;J[e+240>>2]=f;c=J[e+312>>2];if(c>>>0<M[e+316>>2]){J[c+4>>2]=d;J[c>>2]=f;J[e+312>>2]=c+8;break m}J[5996]=0;aa(144,e+308|0,e+240|0);d=J[5996];J[5996]=0;if((d|0)==1){break Vb}d=J[e+244>>2];if(!d){break m}c=J[d+4>>2];J[d+4>>2]=c-1;if(c){break m}Xa[J[J[d>>2]+8>>2]](d);bb(d);break m}b=Z()|0;break b}b=Z()|0;mb(e+240|0);break b}H[e+243|0]=0;I[e+240>>1]=K[5279]|K[5280]<<8;H[e+242|0]=K[5281];H[e+251|0]=3;if(d>>>0<3){break Tb}Wb:{Xb:{Yb:{Zb:{_b:{$b:{ac:{bc:{cc:{dc:{ec:{fc:{gc:{c=J[b>>2];if(!(K[e+240|0]!=K[c|0]|K[e+241|0]!=K[c+1|0]|K[e+242|0]!=K[c+2|0])){g=d-3|0;J[b+4>>2]=g;J[b>>2]=c+3;if((g|0)<=0){break Wb}f=J[b>>2];c=J[b>>2];d=0;while(1){if(K[c+d|0]!=41){d=d+1|0;if((g|0)!=(d|0)){continue}break Wb}break}if(d-18>>>0<4294967279){break Wb}hc:{if(d>>>0<=10){H[e+251|0]=d;c=e+240|0;break hc}J[5996]=0;g=(d|15)+1|0;c=$(3,g|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break gc}J[e+240>>2]=c;J[e+244>>2]=d;J[e+248>>2]=g|-2147483648;}H[fb(c,f,d)+d|0]=0;J[e+272>>2]=J[e+244>>2];c=K[e+247|0]|K[e+248|0]<<8|(K[e+249|0]<<16|K[e+250|0]<<24);H[e+275|0]=c;H[e+276|0]=c>>>8;H[e+277|0]=c>>>16;H[e+278|0]=c>>>24;J[5996]=0;f=J[e+240>>2];g=H[e+251|0];J[e+340>>2]=0;J[e+332>>2]=0;J[e+336>>2]=0;c=$(3,12)|0;h=J[5996];J[5996]=0;if((h|0)!=1){J[e+332>>2]=c;h=c+12|0;J[e+340>>2]=h;J[c>>2]=f;J[c+4>>2]=J[e+272>>2];f=K[e+275|0]|K[e+276|0]<<8|(K[e+277|0]<<16|K[e+278|0]<<24);H[c+7|0]=f;H[c+8|0]=f>>>8;H[c+9|0]=f>>>16;H[c+10|0]=f>>>24;H[c+11|0]=g;H[e+275|0]=0;H[e+276|0]=0;H[e+277|0]=0;H[e+278|0]=0;J[e+272>>2]=0;J[e+336>>2]=h;J[5996]=0;c=$(3,104)|0;f=J[5996];J[5996]=0;ic:{jc:{if((f|0)!=1){J[5996]=0;f=ba(147,c|0,2,e+332|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){b=Z()|0;$a(c);break Xb}J[e+248>>2]=0;J[e+240>>2]=0;J[e+244>>2]=0;J[5996]=0;J[e+268>>2]=f;g=f+12|0;J[e+264>>2]=g;c=$(3,8)|0;h=J[5996];J[5996]=0;if((h|0)==1){break fc}J[e+240>>2]=c;h=c+8|0;J[e+248>>2]=h;J[c+4>>2]=f;J[c>>2]=g;J[e+244>>2]=h;J[e+264>>2]=0;J[e+268>>2]=0;J[5996]=0;f=$(3,104)|0;c=J[5996];J[5996]=0;if((c|0)==1){break jc}J[5996]=0;c=ba(146,f|0,12,e+240|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){b=Z()|0;$a(f);break Yb}J[e+348>>2]=c;g=c+12|0;J[e+344>>2]=g;f=J[e+312>>2];kc:{if(f>>>0<M[e+316>>2]){J[f+4>>2]=c;J[f>>2]=g;J[e+312>>2]=f+8;J[e+344>>2]=0;J[e+348>>2]=0;break kc}J[5996]=0;aa(144,e+308|0,e+344|0);c=J[5996];J[5996]=0;if((c|0)==1){break ic}c=J[e+348>>2];if(!c){break kc}f=J[c+4>>2];J[c+4>>2]=f-1;if(f){break kc}Xa[J[J[c>>2]+8>>2]](c);bb(c);}g=J[e+240>>2];if(g){c=J[e+244>>2];f=g;if((c|0)!=(f|0)){while(1){c=c-8|0;f=J[c+4>>2];lc:{if(!f){break lc}h=J[f+4>>2];J[f+4>>2]=h-1;if(h){break lc}Xa[J[J[f>>2]+8>>2]](f);bb(f);}if((c|0)!=(g|0)){continue}break}f=J[e+240>>2];}J[e+244>>2]=g;$a(f);}c=J[e+268>>2];mc:{if(!c){break mc}f=J[c+4>>2];J[c+4>>2]=f-1;if(f){break mc}Xa[J[J[c>>2]+8>>2]](c);bb(c);}g=J[e+332>>2];if(g){f=J[e+336>>2];if((g|0)!=(f|0)){while(1){c=f-12|0;if(H[f-1|0]<0){$a(J[c>>2]);}f=c;if((c|0)!=(g|0)){continue}break}}J[e+336>>2]=g;$a(g);}d=d+1|0;J[b+4>>2]=J[b+4>>2]-d;J[b>>2]=d+J[b>>2];break m}b=Z()|0;break Xb}b=Z()|0;break Yb}b=Z()|0;mb(e+344|0);break Yb}b=Z()|0;rb(e+332|0);if((g|0)>=0){break b}$a(f);break b}H[e+244|0]=0;J[e+240>>2]=677931888;H[e+251|0]=4;if(d>>>0<4){break Sb}c=J[b>>2];if(!(K[e+240|0]!=K[c|0]|K[e+241|0]!=K[c+1|0]|(K[e+242|0]!=K[c+2|0]|K[e+243|0]!=K[c+3|0]))){g=d-4|0;J[b+4>>2]=g;J[b>>2]=c+4;if((g|0)<=0){break Zb}f=J[b>>2];c=J[b>>2];d=0;while(1){if(K[c+d|0]!=41){d=d+1|0;if((g|0)!=(d|0)){continue}break Zb}break}if(d-18>>>0<4294967279){break Zb}nc:{if(d>>>0<=10){H[e+251|0]=d;c=e+240|0;break nc}J[5996]=0;g=(d|15)+1|0;c=$(3,g|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break ec}J[e+240>>2]=c;J[e+244>>2]=d;J[e+248>>2]=g|-2147483648;}H[fb(c,f,d)+d|0]=0;J[e+272>>2]=J[e+244>>2];c=K[e+247|0]|K[e+248|0]<<8|(K[e+249|0]<<16|K[e+250|0]<<24);H[e+275|0]=c;H[e+276|0]=c>>>8;H[e+277|0]=c>>>16;H[e+278|0]=c>>>24;J[5996]=0;f=J[e+240>>2];g=H[e+251|0];J[e+340>>2]=0;J[e+332>>2]=0;J[e+336>>2]=0;c=$(3,12)|0;h=J[5996];J[5996]=0;if((h|0)!=1){J[e+332>>2]=c;h=c+12|0;J[e+340>>2]=h;J[c>>2]=f;J[c+4>>2]=J[e+272>>2];f=K[e+275|0]|K[e+276|0]<<8|(K[e+277|0]<<16|K[e+278|0]<<24);H[c+7|0]=f;H[c+8|0]=f>>>8;H[c+9|0]=f>>>16;H[c+10|0]=f>>>24;H[c+11|0]=g;H[e+275|0]=0;H[e+276|0]=0;H[e+277|0]=0;H[e+278|0]=0;J[e+272>>2]=0;J[e+336>>2]=h;J[5996]=0;c=$(3,104)|0;f=J[5996];J[5996]=0;oc:{pc:{if((f|0)!=1){J[5996]=0;f=ba(147,c|0,3,e+332|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){b=Z()|0;$a(c);break _b}J[e+248>>2]=0;J[e+240>>2]=0;J[e+244>>2]=0;J[5996]=0;J[e+268>>2]=f;g=f+12|0;J[e+264>>2]=g;c=$(3,8)|0;h=J[5996];J[5996]=0;if((h|0)==1){break dc}J[e+240>>2]=c;h=c+8|0;J[e+248>>2]=h;J[c+4>>2]=f;J[c>>2]=g;J[e+244>>2]=h;J[e+264>>2]=0;J[e+268>>2]=0;J[5996]=0;f=$(3,104)|0;c=J[5996];J[5996]=0;if((c|0)==1){break pc}J[5996]=0;c=ba(146,f|0,12,e+240|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){b=Z()|0;$a(f);break $b}J[e+348>>2]=c;g=c+12|0;J[e+344>>2]=g;f=J[e+312>>2];qc:{if(f>>>0<M[e+316>>2]){J[f+4>>2]=c;J[f>>2]=g;J[e+312>>2]=f+8;J[e+344>>2]=0;J[e+348>>2]=0;break qc}J[5996]=0;aa(144,e+308|0,e+344|0);c=J[5996];J[5996]=0;if((c|0)==1){break oc}c=J[e+348>>2];if(!c){break qc}f=J[c+4>>2];J[c+4>>2]=f-1;if(f){break qc}Xa[J[J[c>>2]+8>>2]](c);bb(c);}g=J[e+240>>2];if(g){c=J[e+244>>2];f=g;if((c|0)!=(f|0)){while(1){c=c-8|0;f=J[c+4>>2];rc:{if(!f){break rc}h=J[f+4>>2];J[f+4>>2]=h-1;if(h){break rc}Xa[J[J[f>>2]+8>>2]](f);bb(f);}if((c|0)!=(g|0)){continue}break}f=J[e+240>>2];}J[e+244>>2]=g;$a(f);}c=J[e+268>>2];sc:{if(!c){break sc}f=J[c+4>>2];J[c+4>>2]=f-1;if(f){break sc}Xa[J[J[c>>2]+8>>2]](c);bb(c);}g=J[e+332>>2];if(g){f=J[e+336>>2];if((g|0)!=(f|0)){while(1){c=f-12|0;if(H[f-1|0]<0){$a(J[c>>2]);}f=c;if((c|0)!=(g|0)){continue}break}}J[e+336>>2]=g;$a(g);}d=d+1|0;J[b+4>>2]=J[b+4>>2]-d;J[b>>2]=d+J[b>>2];break m}b=Z()|0;break _b}b=Z()|0;break $b}b=Z()|0;mb(e+344|0);break $b}b=Z()|0;rb(e+332|0);if((g|0)>=0){break b}$a(f);break b}H[e+245|0]=0;H[e+244|0]=K[5287];J[e+240>>2]=K[5283]|K[5284]<<8|(K[5285]<<16|K[5286]<<24);H[e+251|0]=5;if(d>>>0<5){break Rb}tc:{c=J[b>>2];if(K[e+240|0]!=K[c|0]|K[e+241|0]!=K[c+1|0]|(K[e+242|0]!=K[c+2|0]|K[e+243|0]!=K[c+3|0])){break tc}if(K[e+244|0]!=K[c+4|0]){break tc}g=d-5|0;J[b+4>>2]=g;J[b>>2]=c+5;if((g|0)<=0){break ac}f=J[b>>2];c=J[b>>2];d=0;while(1){if(K[c+d|0]!=41){d=d+1|0;if((g|0)!=(d|0)){continue}break ac}break}if(d-18>>>0<4294967279){break ac}uc:{if(d>>>0<=10){H[e+251|0]=d;c=e+240|0;break uc}J[5996]=0;g=(d|15)+1|0;c=$(3,g|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break cc}J[e+240>>2]=c;J[e+244>>2]=d;J[e+248>>2]=g|-2147483648;}H[fb(c,f,d)+d|0]=0;J[e+344>>2]=J[e+244>>2];c=K[e+247|0]|K[e+248|0]<<8|(K[e+249|0]<<16|K[e+250|0]<<24);H[e+347|0]=c;H[e+348|0]=c>>>8;H[e+349|0]=c>>>16;H[e+350|0]=c>>>24;J[5996]=0;f=J[e+240>>2];g=H[e+251|0];J[e+248>>2]=0;J[e+240>>2]=0;J[e+244>>2]=0;c=$(3,12)|0;h=J[5996];J[5996]=0;if((h|0)!=1){J[e+240>>2]=c;h=c+12|0;J[e+248>>2]=h;J[c>>2]=f;J[c+4>>2]=J[e+344>>2];f=K[e+347|0]|K[e+348|0]<<8|(K[e+349|0]<<16|K[e+350|0]<<24);H[c+7|0]=f;H[c+8|0]=f>>>8;H[c+9|0]=f>>>16;H[c+10|0]=f>>>24;H[c+11|0]=g;H[e+347|0]=0;H[e+348|0]=0;H[e+349|0]=0;H[e+350|0]=0;J[e+344>>2]=0;J[e+244>>2]=h;J[5996]=0;f=$(3,104)|0;c=J[5996];J[5996]=0;vc:{wc:{if((c|0)!=1){J[5996]=0;c=ba(147,f|0,2,e+240|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){b=Z()|0;$a(f);break vc}J[e+336>>2]=c;g=c+12|0;J[e+332>>2]=g;f=J[e+312>>2];xc:{if(f>>>0<M[e+316>>2]){J[f+4>>2]=c;J[f>>2]=g;J[e+312>>2]=f+8;J[e+332>>2]=0;J[e+336>>2]=0;break xc}J[5996]=0;aa(144,e+308|0,e+332|0);c=J[5996];J[5996]=0;if((c|0)==1){break wc}c=J[e+336>>2];if(!c){break xc}f=J[c+4>>2];J[c+4>>2]=f-1;if(f){break xc}Xa[J[J[c>>2]+8>>2]](c);bb(c);}g=J[e+240>>2];if(g){f=J[e+244>>2];if((g|0)!=(f|0)){while(1){c=f-12|0;if(H[f-1|0]<0){$a(J[c>>2]);}f=c;if((c|0)!=(g|0)){continue}break}}J[e+244>>2]=g;$a(g);}d=d+1|0;J[b+4>>2]=J[b+4>>2]-d;J[b>>2]=d+J[b>>2];break m}b=Z()|0;break vc}b=Z()|0;mb(e+332|0);}rb(e+240|0);break b}b=Z()|0;rb(e+240|0);if((g|0)>=0){break b}$a(f);break b}H[e+245|0]=0;H[e+244|0]=K[5319];H[e+251|0]=5;f=K[5315]|K[5316]<<8|(K[5317]<<16|K[5318]<<24);J[e+240>>2]=f;c=J[b>>2];if(K[c|0]!=(f&255)|K[e+241|0]!=K[c+1|0]|(K[e+242|0]!=K[c+2|0]|K[e+243|0]!=K[c+3|0])){break Qb}if(K[e+244|0]!=K[c+4|0]){break Qb}J[b+4>>2]=d-5;J[b>>2]=c+5;d=J[b>>2];c=J[b+4>>2];J[5996]=0;f=e+240|0;Na(148,f|0,d|0,c|0);d=J[5996];J[5996]=0;if((d|0)==1){break bc}yc:{zc:{Ac:{Bc:{Cc:{if(K[e+256|0]){J[5996]=0;d=e+332|0;aa(149,d|0,f|0);c=J[5996];J[5996]=0;if((c|0)==1){break Cc}J[5996]=0;c=e+344|0;ea(150,c|0,3,d|0);f=J[5996];J[5996]=0;if((f|0)==1){break Bc}J[5996]=0;aa(151,e+308|0,c|0);f=J[5996];J[5996]=0;if((f|0)==1){break Ac}mb(c);rb(d);d=J[e+252>>2]+1|0;J[b+4>>2]=J[b+4>>2]-d;J[b>>2]=d+J[b>>2];if(!K[e+256|0]|H[e+251|0]>=0){break m}$a(J[e+240>>2]);break m}J[a>>2]=0;J[a+4>>2]=0;break l}b=Z()|0;break yc}b=Z()|0;break zc}b=Z()|0;mb(e+344|0);}rb(e+332|0);}if(!K[e+256|0]|H[e+251|0]>=0){break b}$a(J[e+240>>2]);break b}b=Z()|0;break b}b=Z()|0;break Yb}b=Z()|0;break b}b=Z()|0;break $b}b=Z()|0;break b}b=Z()|0;break b}J[a>>2]=0;J[a+4>>2]=0;break l}lb(e+240|0);mb(e+264|0);}rb(e+332|0);break b}J[a>>2]=0;J[a+4>>2]=0;break l}lb(e+240|0);mb(e+264|0);}rb(e+332|0);break b}J[a>>2]=0;J[a+4>>2]=0;break l}H[e+244|0]=0;J[e+240>>2]=677931888;H[e+251|0]=4;}H[e+245|0]=0;H[e+244|0]=K[5287];H[e+251|0]=5;J[e+240>>2]=K[5283]|K[5284]<<8|(K[5285]<<16|K[5286]<<24);}H[e+245|0]=0;H[e+244|0]=K[5319];H[e+251|0]=5;J[e+240>>2]=K[5315]|K[5316]<<8|(K[5317]<<16|K[5318]<<24);}J[5996]=0;d=_(30,e+228|0,5355)|0;c=J[5996];J[5996]=0;Dc:{Ec:{Fc:{if((c|0)!=1){c=vb(d,b);if(H[e+239|0]<0){$a(J[d>>2]);}if(!c){break Dc}d=J[b>>2];c=J[b+4>>2];J[5996]=0;bd(e+240|0,d,c,32);d=J[5996];J[5996]=0;if((d|0)==1){break Fc}if(K[e+256|0]){break Ec}J[a>>2]=0;J[a+4>>2]=0;break l}b=Z()|0;break b}b=Z()|0;break b}J[5996]=0;d=e+332|0;ea(153,d|0,6,e+240|0);c=J[5996];J[5996]=0;Gc:{Hc:{if((c|0)!=1){J[5996]=0;aa(151,e+308|0,d|0);c=J[5996];J[5996]=0;if((c|0)==1){break Hc}mb(d);d=J[e+252>>2]+1|0;J[b+4>>2]=J[b+4>>2]-d;J[b>>2]=d+J[b>>2];if(!K[e+256|0]){break m}d=J[e+240>>2];if(!d){break m}$a(d);break m}b=Z()|0;break Gc}b=Z()|0;mb(e+332|0);}if(!K[e+256|0]){break b}a=J[e+240>>2];if(!a){break b}$a(a);break b}J[5996]=0;d=_(30,e+216|0,5372)|0;c=J[5996];J[5996]=0;Ic:{Jc:{Kc:{if((c|0)!=1){c=vb(d,b);if(H[e+227|0]<0){$a(J[d>>2]);}if(!c){break Ic}d=J[b>>2];c=J[b+4>>2];J[5996]=0;bd(e+240|0,d,c,20);d=J[5996];J[5996]=0;if((d|0)==1){break Kc}if(K[e+256|0]){break Jc}J[a>>2]=0;J[a+4>>2]=0;break l}b=Z()|0;break b}b=Z()|0;break b}J[5996]=0;d=e+332|0;ea(153,d|0,8,e+240|0);c=J[5996];J[5996]=0;Lc:{Mc:{if((c|0)!=1){J[5996]=0;aa(151,e+308|0,d|0);c=J[5996];J[5996]=0;if((c|0)==1){break Mc}mb(d);d=J[e+252>>2]+1|0;J[b+4>>2]=J[b+4>>2]-d;J[b>>2]=d+J[b>>2];if(!K[e+256|0]){break m}d=J[e+240>>2];if(!d){break m}$a(d);break m}b=Z()|0;break Lc}b=Z()|0;mb(e+332|0);}if(!K[e+256|0]){break b}a=J[e+240>>2];if(!a){break b}$a(a);break b}J[5996]=0;d=_(30,e+204|0,5346)|0;c=J[5996];J[5996]=0;Nc:{Oc:{Pc:{if((c|0)!=1){c=vb(d,b);if(H[e+215|0]<0){$a(J[d>>2]);}if(!c){break Nc}d=J[b>>2];c=J[b+4>>2];J[5996]=0;bd(e+240|0,d,c,32);d=J[5996];J[5996]=0;if((d|0)==1){break Pc}if(K[e+256|0]){break Oc}J[a>>2]=0;J[a+4>>2]=0;break l}b=Z()|0;break b}b=Z()|0;break b}J[5996]=0;d=e+332|0;ea(153,d|0,7,e+240|0);c=J[5996];J[5996]=0;Qc:{Rc:{if((c|0)!=1){J[5996]=0;aa(151,e+308|0,d|0);c=J[5996];J[5996]=0;if((c|0)==1){break Rc}mb(d);d=J[e+252>>2]+1|0;J[b+4>>2]=J[b+4>>2]-d;J[b>>2]=d+J[b>>2];if(!K[e+256|0]){break m}d=J[e+240>>2];if(!d){break m}$a(d);break m}b=Z()|0;break Qc}b=Z()|0;mb(e+332|0);}if(!K[e+256|0]){break b}a=J[e+240>>2];if(!a){break b}$a(a);break b}J[5996]=0;d=_(30,e+192|0,5363)|0;c=J[5996];J[5996]=0;Sc:{Tc:{Uc:{if((c|0)!=1){c=vb(d,b);if(H[e+203|0]<0){$a(J[d>>2]);}if(!c){break Sc}d=J[b>>2];c=J[b+4>>2];J[5996]=0;bd(e+240|0,d,c,20);d=J[5996];J[5996]=0;if((d|0)==1){break Uc}if(K[e+256|0]){break Tc}J[a>>2]=0;J[a+4>>2]=0;break l}b=Z()|0;break b}b=Z()|0;break b}J[5996]=0;d=e+332|0;ea(153,d|0,9,e+240|0);c=J[5996];J[5996]=0;Vc:{Wc:{if((c|0)!=1){J[5996]=0;aa(151,e+308|0,d|0);c=J[5996];J[5996]=0;if((c|0)==1){break Wc}mb(d);d=J[e+252>>2]+1|0;J[b+4>>2]=J[b+4>>2]-d;J[b>>2]=d+J[b>>2];if(!K[e+256|0]){break m}d=J[e+240>>2];if(!d){break m}$a(d);break m}b=Z()|0;break Vc}b=Z()|0;mb(e+332|0);}if(!K[e+256|0]){break b}a=J[e+240>>2];if(!a){break b}$a(a);break b}J[5996]=0;d=_(30,e+180|0,5258)|0;c=J[5996];J[5996]=0;Xc:{Yc:{Zc:{_c:{$c:{if((c|0)!=1){c=vb(d,b);if(H[e+191|0]<0){$a(J[d>>2]);}if(!c){break _c}f=J[b+4>>2];ad:{if((f|0)<=0){break ad}c=J[b>>2];d=0;while(1){g=c+d|0;if(K[g|0]!=41){d=d+1|0;if((f|0)!=(d|0)){continue}break ad}break}if(d){break $c}}J[a>>2]=0;J[a+4>>2]=0;break l}b=Z()|0;break b}J[5996]=0;c=ba(154,e+168|0,c|0,g|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){break Yc}f=hd(c,e+240|0);if(H[e+179|0]<0){$a(J[c>>2]);}if(!f){break Zc}f=J[e+244>>2];g=f;c=J[e+240>>2];h=c- -2147483648|0;f=f-(c>>>0<2147483648)|0;if((f|0)==-1&h>>>0<2147483649|(f|0)!=-1){break Zc}J[5996]=0;f=e+332|0;Ae(f,5,c,g);c=J[5996];J[5996]=0;bd:{if((c|0)!=1){J[5996]=0;aa(151,e+308|0,f|0);c=J[5996];J[5996]=0;if((c|0)==1){break bd}mb(f);d=d+1|0;J[b+4>>2]=J[b+4>>2]-d;J[b>>2]=d+J[b>>2];break m}b=Z()|0;break b}b=Z()|0;mb(e+332|0);break b}J[5996]=0;d=_(30,e+156|0,5265)|0;c=J[5996];J[5996]=0;cd:{dd:{ed:{fd:{if((c|0)!=1){c=vb(d,b);if(H[e+167|0]<0){$a(J[d>>2]);}if(!c){break ed}f=J[b+4>>2];gd:{if((f|0)<=0){break gd}c=J[b>>2];d=0;while(1){g=c+d|0;if(K[g|0]!=41){d=d+1|0;if((f|0)!=(d|0)){continue}break gd}break}if(d){break fd}}J[a>>2]=0;J[a+4>>2]=0;break l}b=Z()|0;break b}J[5996]=0;c=ba(154,e+144|0,c|0,g|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){break cd}f=hd(c,e+240|0);if(H[e+155|0]<0){$a(J[c>>2]);}if(!f){break dd}f=J[e+244>>2];g=f;c=J[e+240>>2];h=c- -2147483648|0;f=f-(c>>>0<2147483648)|0;if((f|0)==-1&h>>>0<2147483649|(f|0)!=-1){break dd}J[5996]=0;f=e+332|0;Ae(f,4,c,g);c=J[5996];J[5996]=0;hd:{if((c|0)!=1){J[5996]=0;aa(151,e+308|0,f|0);c=J[5996];J[5996]=0;if((c|0)==1){break hd}mb(f);d=d+1|0;J[b+4>>2]=J[b+4>>2]-d;J[b>>2]=d+J[b>>2];break m}b=Z()|0;break b}b=Z()|0;mb(e+332|0);break b}J[5996]=0;d=_(30,e+132|0,5289)|0;c=J[5996];J[5996]=0;id:{if((c|0)!=1){c=vb(d,b);if(H[e+143|0]<0){$a(J[d>>2]);}if(!c){break Xc}f=J[b+4>>2];jd:{if((f|0)<=0){break jd}c=J[b>>2];d=0;while(1){g=c+d|0;h=K[g|0];if((h|0)!=44){if((h|0)==41){break jd}d=d+1|0;if((f|0)>(d|0)){continue}break jd}break}if(d){break id}}J[a>>2]=0;J[a+4>>2]=0;break l}b=Z()|0;break b}J[5996]=0;c=ba(154,e+120|0,c|0,g|0)|0;f=J[5996];J[5996]=0;kd:{if((f|0)!=1){f=hd(c,m);if(H[e+131|0]<0){$a(J[c>>2]);}if(f){break kd}J[a>>2]=0;J[a+4>>2]=0;break l}b=Z()|0;break b}d=d+1|0;l=J[b+4>>2]-d|0;J[b+4>>2]=l;f=d+J[b>>2]|0;J[b>>2]=f;J[e+340>>2]=0;J[e+332>>2]=0;J[e+336>>2]=0;ld:{md:{nd:{od:{pd:{qd:{while(1){d=0;g=(l|0)<=0;if(g){break od}rd:{while(1){c=K[d+f|0];if((c|0)==44){c=d;break rd}if((c|0)!=41){d=d+1|0;if((l|0)>(d|0)){continue}}break}c=0;if(g){break od}while(1){if(K[c+f|0]==41){d=-1;break rd}c=c+1|0;if((l|0)!=(c|0)){continue}break}break od}if((c|0)<=0){break od}J[5996]=0;ea(156,e+240|0,f|0,c+f|0);f=J[5996];J[5996]=0;if((f|0)==1){break ld}if(!K[e+252|0]){break od}f=J[e+336>>2];sd:{if(f>>>0<M[e+340>>2]){g=J[e+244>>2];J[f>>2]=J[e+240>>2];J[f+4>>2]=g;J[f+8>>2]=J[e+248>>2];J[e+248>>2]=0;J[e+240>>2]=0;J[e+244>>2]=0;J[e+336>>2]=f+12;l=1;g=0;break sd}J[5996]=0;aa(157,e+332|0,e+240|0);f=J[5996];J[5996]=0;if((f|0)==1){break qd}l=K[e+252|0];g=K[e+251|0];}c=c+1|0;J[b+4>>2]=J[b+4>>2]-c;f=c+J[b>>2]|0;J[b>>2]=f;if(!(!(l&255)|g<<24>>24>=0)){$a(J[e+240>>2]);f=J[b>>2];}if((d|0)!=-1){l=J[b+4>>2];continue}break}d=J[e+336>>2];c=J[e+332>>2];f=(d-c|0)/12|0;if((c|0)!=(d|0)&f>>>0<21){break pd}break od}b=Z()|0;if(!K[e+252|0]|H[e+251|0]>=0){break c}$a(J[e+240>>2]);break c}d=J[e+296>>2];c=J[e+300>>2];if(!((!!d&(c|0)>=0|(c|0)>0)&(!c&d>>>0<=f>>>0))){break od}J[5996]=0;f=e+240|0;g=e+332|0;Ka(158,f|0,25,g|0,d|0,c|0);d=J[5996];J[5996]=0;if((d|0)==1){break nd}J[5996]=0;aa(151,e+308|0,f|0);d=J[5996];J[5996]=0;if((d|0)==1){break md}mb(f);rb(g);break m}J[a>>2]=0;J[a+4>>2]=0;rb(e+332|0);break l}b=Z()|0;break c}b=Z()|0;mb(e+240|0);break c}b=Z()|0;break c}J[a>>2]=0;J[a+4>>2]=0;break l}b=Z()|0;break b}J[a>>2]=0;J[a+4>>2]=0;break l}b=Z()|0;break b}J[5996]=0;d=_(30,e+108|0,5302)|0;c=J[5996];J[5996]=0;td:{ud:{if((c|0)!=1){c=vb(d,b);if(H[e+119|0]<0){$a(J[d>>2]);}if(!c){break td}f=J[b+4>>2];vd:{if((f|0)<=0){break vd}c=J[b>>2];d=0;while(1){g=c+d|0;h=K[g|0];if((h|0)!=44){if((h|0)==41){break vd}d=d+1|0;if((f|0)>(d|0)){continue}break vd}break}if(d){break ud}}J[a>>2]=0;J[a+4>>2]=0;break l}b=Z()|0;break b}J[5996]=0;c=ba(154,e+96|0,c|0,g|0)|0;f=J[5996];J[5996]=0;wd:{if((f|0)!=1){f=hd(c,m);if(H[e+107|0]<0){$a(J[c>>2]);}if(f){break wd}J[a>>2]=0;J[a+4>>2]=0;break l}b=Z()|0;break b}c=J[e+300>>2];if(!J[e+296>>2]&(c|0)<=0|(c|0)<0){J[a>>2]=0;J[a+4>>2]=0;break l}d=d+1|0;J[b+4>>2]=J[b+4>>2]-d;J[b>>2]=d+J[b>>2];J[5996]=0;d=e+320|0;ga(159,d|0,19,1,m|0);c=J[5996];J[5996]=0;if((c|0)==1){break k}J[5996]=0;ga(160,d|0,0,-1,-1);d=J[5996];J[5996]=0;if((d|0)!=1){break m}b=Z()|0;break b}J[5996]=0;d=_(30,e+84|0,5251)|0;c=J[5996];J[5996]=0;xd:{yd:{zd:{Ad:{Bd:{Cd:{Dd:{if((c|0)!=1){c=vb(d,b);if(H[e+95|0]<0){$a(J[d>>2]);}if(!c){break xd}J[5996]=0;d=e+320|0;ga(160,d|0,14,-1,-1);c=J[5996];J[5996]=0;if((c|0)==1){break Dd}J[5996]=0;ga(160,d|0,21,-1,-1);c=J[5996];J[5996]=0;if((c|0)==1){break Cd}J[5996]=0;ga(160,d|0,0,-1,-1);c=J[5996];J[5996]=0;if((c|0)==1){break Bd}J[5996]=0;ga(160,d|0,20,-1,-1);c=J[5996];J[5996]=0;if((c|0)==1){break Ad}J[5996]=0;ga(160,d|0,0,-1,-1);c=J[5996];J[5996]=0;if((c|0)==1){break zd}J[5996]=0;ga(160,d|0,20,-1,-1);c=J[5996];J[5996]=0;if((c|0)==1){break yd}J[5996]=0;ga(160,d|0,0,-1,-1);d=J[5996];J[5996]=0;if((d|0)!=1){break m}b=Z()|0;break b}b=Z()|0;break b}b=Z()|0;break b}b=Z()|0;break b}b=Z()|0;break b}b=Z()|0;break b}b=Z()|0;break b}b=Z()|0;break b}J[5996]=0;d=_(30,e+72|0,5272)|0;c=J[5996];J[5996]=0;if((c|0)==1){break j}c=vb(d,b);if(H[e+83|0]<0){$a(J[d>>2]);}Ed:{if(c){J[5996]=0;ga(160,e+320|0,11,-1,-1);d=J[5996];J[5996]=0;if((d|0)!=1){break Ed}b=Z()|0;break b}J[5996]=0;d=_(30,e+60|0,5339)|0;c=J[5996];J[5996]=0;if((c|0)==1){break i}c=vb(d,b);if(H[e+71|0]<0){$a(J[d>>2]);}if(c){J[5996]=0;ga(160,e+320|0,13,-1,-1);d=J[5996];J[5996]=0;if((d|0)!=1){break Ed}b=Z()|0;break b}J[5996]=0;d=_(30,e+48|0,5244)|0;c=J[5996];J[5996]=0;if((c|0)==1){break h}c=vb(d,b);if(H[e+59|0]<0){$a(J[d>>2]);}if(c){J[5996]=0;ga(160,e+320|0,12,-1,-1);d=J[5996];J[5996]=0;if((d|0)!=1){break Ed}b=Z()|0;break b}J[5996]=0;d=_(30,e+36|0,5333)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}c=vb(d,b);if(H[e+47|0]<0){$a(J[d>>2]);}if(c){J[5996]=0;ga(160,e+320|0,15,-1,-1);d=J[5996];J[5996]=0;if((d|0)!=1){break Ed}b=Z()|0;break b}J[5996]=0;d=_(30,e+24|0,5327)|0;c=J[5996];J[5996]=0;if((c|0)==1){break f}c=vb(d,b);if(H[e+35|0]<0){$a(J[d>>2]);}if(c){J[5996]=0;ga(160,e+320|0,16,-1,-1);d=J[5996];J[5996]=0;if((d|0)!=1){break Ed}b=Z()|0;break b}J[5996]=0;d=_(30,e+12|0,5321)|0;c=J[5996];J[5996]=0;if((c|0)==1){break e}c=vb(d,b);if(H[e+23|0]<0){$a(J[d>>2]);}if(c){J[5996]=0;ga(160,e+320|0,17,-1,-1);d=J[5996];J[5996]=0;if((d|0)!=1){break Ed}b=Z()|0;break b}J[5996]=0;d=_(30,e|0,5296)|0;c=J[5996];J[5996]=0;if((c|0)==1){break d}c=vb(d,b);if(H[e+11|0]<0){$a(J[d>>2]);}if(c){J[5996]=0;ga(160,e+320|0,18,-1,-1);d=J[5996];J[5996]=0;if((d|0)!=1){break Ed}b=Z()|0;break b}J[a>>2]=0;J[a+4>>2]=0;break l}J[5996]=0;d=e+320|0;ga(160,d|0,21,-1,-1);c=J[5996];J[5996]=0;Fd:{Gd:{if((c|0)!=1){J[5996]=0;ga(160,d|0,0,-1,-1);c=J[5996];J[5996]=0;if((c|0)==1){break Gd}J[5996]=0;ga(160,d|0,20,-1,-1);c=J[5996];J[5996]=0;if((c|0)==1){break Fd}J[5996]=0;ga(160,d|0,0,-1,-1);d=J[5996];J[5996]=0;if((d|0)!=1){break m}b=Z()|0;break b}b=Z()|0;break b}b=Z()|0;break b}b=Z()|0;break b}c=J[e+324>>2];f=J[e+320>>2];if((c|0)!=(f|0)){continue}break}d=J[e+308>>2];if((J[e+312>>2]-d|0)!=8){J[5996]=0;ga(44,4122,2210,1561,2344);a=J[5996];J[5996]=0;if((a|0)!=1){break a}b=Z()|0;break b}if(J[b+4>>2]){J[a>>2]=0;J[a+4>>2]=0;break l}J[a>>2]=J[d>>2];J[a+4>>2]=J[d+4>>2];J[d>>2]=0;J[d+4>>2]=0;}a=J[e+308>>2];if(a){d=J[e+312>>2];if((a|0)!=(d|0)){while(1){d=d-8|0;b=J[d+4>>2];Hd:{if(!b){break Hd}c=J[b+4>>2];J[b+4>>2]=c-1;if(c){break Hd}Xa[J[J[b>>2]+8>>2]](b);bb(b);}if((a|0)!=(d|0)){continue}break}a=J[e+308>>2];}$a(a);}a=J[e+320>>2];if(a){J[e+324>>2]=a;$a(a);}Ua=e+352|0;return}b=Z()|0;break b}b=Z()|0;break b}b=Z()|0;break b}b=Z()|0;break b}b=Z()|0;break b}b=Z()|0;break b}b=Z()|0;break b}b=Z()|0;break b}b=Z()|0;break b}rb(e+332|0);}lb(e+308|0);a=J[e+320>>2];if(a){J[e+324>>2]=a;$a(a);}da(b|0);B();}B();}function hn(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;var f=0,g=0,h=0,i=0,j=0,k=0;f=Ua-128|0;Ua=f;H[f+127|0]=b;g=f+116|0;fb(g,b?3764:5575,b);H[b|g]=0;a:{b:{c:{d:{e:{f:{g:{h:{i:{j:{k:{l:{m:{n:{o:{p:{q:{r:{s:{t:{u:{v:{w:{x:{y:{z:{A:{B:{C:{D:{switch(J[c>>2]){case 10:J[5996]=0;b=ba(99,d|0,0,2553)|0;c=J[5996];J[5996]=0;if((c|0)==1){break u}d=J[b>>2];J[f+96>>2]=J[b+4>>2];c=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+99|0]=c;H[f+100|0]=c>>>8;H[f+101|0]=c>>>16;H[f+102|0]=c>>>24;J[b>>2]=0;J[b+4>>2]=0;c=K[b+11|0];J[b+8>>2]=0;J[a>>2]=d;J[a+4>>2]=J[f+96>>2];b=K[f+99|0]|K[f+100|0]<<8|(K[f+101|0]<<16|K[f+102|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=c;break b;case 11:J[5996]=0;b=ba(99,d|0,0,1681)|0;c=J[5996];J[5996]=0;if((c|0)!=1){d=J[b>>2];J[f+96>>2]=J[b+4>>2];c=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+99|0]=c;H[f+100|0]=c>>>8;H[f+101|0]=c>>>16;H[f+102|0]=c>>>24;J[b>>2]=0;J[b+4>>2]=0;c=K[b+11|0];J[b+8>>2]=0;J[a>>2]=d;J[a+4>>2]=J[f+96>>2];b=K[f+99|0]|K[f+100|0]<<8|(K[f+101|0]<<16|K[f+102|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=c;break b}a=Z()|0;break a;case 12:E:{F:{G:{b=J[J[c+32>>2]>>2];switch(J[b>>2]-2|0){case 1:break F;case 0:break G;default:break E}}b=J[b+8>>2];if(H[b+11|0]>=0){J[f+104>>2]=J[b+8>>2];c=J[b+4>>2];J[f+96>>2]=J[b>>2];J[f+100>>2]=c;break c}c=J[b+4>>2];b=J[b>>2];J[5996]=0;ea(40,f+96|0,b|0,c|0);b=J[5996];J[5996]=0;if((b|0)!=1){break c}a=Z()|0;break a}b=J[b+8>>2];if(H[b+11|0]>=0){J[f+104>>2]=J[b+8>>2];c=J[b+4>>2];J[f+96>>2]=J[b>>2];J[f+100>>2]=c;break d}c=J[b+4>>2];b=J[b>>2];J[5996]=0;ea(40,f+96|0,b|0,c|0);b=J[5996];J[5996]=0;if((b|0)!=1){break d}a=Z()|0;break a}J[5996]=0;b=ba(99,d|0,0,2541)|0;c=J[5996];J[5996]=0;if((c|0)!=1){d=J[b>>2];J[f+96>>2]=J[b+4>>2];c=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+99|0]=c;H[f+100|0]=c>>>8;H[f+101|0]=c>>>16;H[f+102|0]=c>>>24;J[b>>2]=0;J[b+4>>2]=0;c=K[b+11|0];J[b+8>>2]=0;J[a>>2]=d;J[a+4>>2]=J[f+96>>2];b=K[f+99|0]|K[f+100|0]<<8|(K[f+101|0]<<16|K[f+102|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=c;break b}a=Z()|0;break a;case 13:J[5996]=0;b=ba(99,d|0,0,2522)|0;c=J[5996];J[5996]=0;if((c|0)!=1){d=J[b>>2];J[f+96>>2]=J[b+4>>2];c=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+99|0]=c;H[f+100|0]=c>>>8;H[f+101|0]=c>>>16;H[f+102|0]=c>>>24;J[b>>2]=0;J[b+4>>2]=0;c=K[b+11|0];J[b+8>>2]=0;J[a>>2]=d;J[a+4>>2]=J[f+96>>2];b=K[f+99|0]|K[f+100|0]<<8|(K[f+101|0]<<16|K[f+102|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=c;break b}a=Z()|0;break a;case 14:J[5996]=0;b=ba(99,d|0,0,1271)|0;c=J[5996];J[5996]=0;if((c|0)!=1){d=J[b>>2];J[f+96>>2]=J[b+4>>2];c=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+99|0]=c;H[f+100|0]=c>>>8;H[f+101|0]=c>>>16;H[f+102|0]=c>>>24;J[b>>2]=0;J[b+4>>2]=0;c=K[b+11|0];J[b+8>>2]=0;J[a>>2]=d;J[a+4>>2]=J[f+96>>2];b=K[f+99|0]|K[f+100|0]<<8|(K[f+101|0]<<16|K[f+102|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=c;break b}a=Z()|0;break a;case 15:J[5996]=0;b=ba(99,d|0,0,2170)|0;c=J[5996];J[5996]=0;if((c|0)!=1){d=J[b>>2];J[f+96>>2]=J[b+4>>2];c=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+99|0]=c;H[f+100|0]=c>>>8;H[f+101|0]=c>>>16;H[f+102|0]=c>>>24;J[b>>2]=0;J[b+4>>2]=0;c=K[b+11|0];J[b+8>>2]=0;J[a>>2]=d;J[a+4>>2]=J[f+96>>2];b=K[f+99|0]|K[f+100|0]<<8|(K[f+101|0]<<16|K[f+102|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=c;break b}a=Z()|0;break a;case 16:J[5996]=0;b=ba(99,d|0,0,2029)|0;c=J[5996];J[5996]=0;if((c|0)!=1){d=J[b>>2];J[f+96>>2]=J[b+4>>2];c=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+99|0]=c;H[f+100|0]=c>>>8;H[f+101|0]=c>>>16;H[f+102|0]=c>>>24;J[b>>2]=0;J[b+4>>2]=0;c=K[b+11|0];J[b+8>>2]=0;J[a>>2]=d;J[a+4>>2]=J[f+96>>2];b=K[f+99|0]|K[f+100|0]<<8|(K[f+101|0]<<16|K[f+102|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=c;break b}a=Z()|0;break a;case 17:if(J[J[J[c+32>>2]+8>>2]>>2]!=1){break C}J[5996]=0;b=ba(99,d|0,0,1392)|0;c=J[5996];J[5996]=0;if((c|0)!=1){d=J[b>>2];J[f+96>>2]=J[b+4>>2];c=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+99|0]=c;H[f+100|0]=c>>>8;H[f+101|0]=c>>>16;H[f+102|0]=c>>>24;J[b>>2]=0;J[b+4>>2]=0;c=K[b+11|0];J[b+8>>2]=0;J[a>>2]=d;J[a+4>>2]=J[f+96>>2];b=K[f+99|0]|K[f+100|0]<<8|(K[f+101|0]<<16|K[f+102|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=c;break b}a=Z()|0;break a;case 22:b=J[c+32>>2];if(!J[J[b>>2]>>2]){J[5996]=0;b=ba(99,d+12|0,0,2096)|0;c=J[5996];J[5996]=0;if((c|0)!=1){e=J[b>>2];c=K[d+19|0]|K[d+20|0]<<8|(K[d+21|0]<<16|K[d+22|0]<<24);H[f+99|0]=c;H[f+100|0]=c>>>8;H[f+101|0]=c>>>16;H[f+102|0]=c>>>24;J[f+96>>2]=J[d+16>>2];c=K[d+23|0];J[b+8>>2]=0;J[b>>2]=0;J[b+4>>2]=0;J[a>>2]=e;J[a+4>>2]=J[f+96>>2];b=K[f+99|0]|K[f+100|0]<<8|(K[f+101|0]<<16|K[f+102|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=c;break b}a=Z()|0;break a}if(J[J[b+8>>2]>>2]){break x}J[5996]=0;b=ba(99,d|0,0,1295)|0;c=J[5996];J[5996]=0;if((c|0)!=1){d=J[b>>2];J[f+96>>2]=J[b+4>>2];c=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+99|0]=c;H[f+100|0]=c>>>8;H[f+101|0]=c>>>16;H[f+102|0]=c>>>24;J[b>>2]=0;J[b+4>>2]=0;c=K[b+11|0];J[b+8>>2]=0;J[a>>2]=d;J[a+4>>2]=J[f+96>>2];b=K[f+99|0]|K[f+100|0]<<8|(K[f+101|0]<<16|K[f+102|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=c;break b}a=Z()|0;break a;case 2:b=J[c+8>>2];if(H[b+11|0]>=0){J[f+104>>2]=J[b+8>>2];c=J[b+4>>2];J[f+96>>2]=J[b>>2];J[f+100>>2]=c;break e}c=J[b+4>>2];b=J[b>>2];J[5996]=0;ea(40,f+96|0,b|0,c|0);b=J[5996];J[5996]=0;if((b|0)!=1){break e}a=Z()|0;break a;case 3:b=J[c+8>>2];if(H[b+11|0]>=0){J[f+104>>2]=J[b+8>>2];c=J[b+4>>2];J[f+96>>2]=J[b>>2];J[f+100>>2]=c;break f}c=J[b+4>>2];b=J[b>>2];J[5996]=0;ea(40,f+96|0,b|0,c|0);b=J[5996];J[5996]=0;if((b|0)!=1){break f}a=Z()|0;break a;case 5:J[5996]=0;b=_(97,f+116|0,5258)|0;d=J[5996];J[5996]=0;H:{I:{J:{K:{L:{if((d|0)!=1){J[f+88>>2]=J[b+8>>2];d=J[b+4>>2];J[f+80>>2]=J[b>>2];J[f+84>>2]=d;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;J[5996]=0;b=f- -64|0;aa(119,b|0,c+4|0);c=J[5996];J[5996]=0;if((c|0)==1){break L}J[5996]=0;d=b;b=K[f+75|0];c=b<<24>>24<0;c=ba(100,f+80|0,(c?J[f+64>>2]:d)|0,(c?J[f+68>>2]:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break K}J[f+104>>2]=J[c+8>>2];b=J[c+4>>2];J[f+96>>2]=J[c>>2];J[f+100>>2]=b;J[c>>2]=0;J[c+4>>2]=0;J[c+8>>2]=0;J[5996]=0;b=_(97,f+96|0,5242)|0;d=J[5996];J[5996]=0;if((d|0)==1){break J}e=J[b>>2];J[f+48>>2]=J[b+4>>2];d=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+51|0]=d;H[f+52|0]=d>>>8;H[f+53|0]=d>>>16;H[f+54|0]=d>>>24;J[b>>2]=0;J[b+4>>2]=0;d=K[b+11|0];J[b+8>>2]=0;J[a>>2]=e;J[a+4>>2]=J[f+48>>2];b=K[f+51|0]|K[f+52|0]<<8|(K[f+53|0]<<16|K[f+54|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=d;if(H[f+75|0]<0){$a(J[f+64>>2]);}if(H[c+11|0]>=0){break b}$a(J[c>>2]);break b}a=Z()|0;break a}a=Z()|0;break H}a=Z()|0;break I}a=Z()|0;if(H[f+107|0]>=0){break I}$a(J[f+96>>2]);}if(H[f+75|0]>=0){break H}$a(J[f+64>>2]);}if(H[f+91|0]>=0){break a}$a(J[f+80>>2]);break a;case 4:J[5996]=0;b=_(97,f+116|0,5265)|0;d=J[5996];J[5996]=0;M:{N:{O:{P:{Q:{if((d|0)!=1){J[f+88>>2]=J[b+8>>2];d=J[b+4>>2];J[f+80>>2]=J[b>>2];J[f+84>>2]=d;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;J[5996]=0;b=f- -64|0;aa(119,b|0,c+4|0);c=J[5996];J[5996]=0;if((c|0)==1){break Q}J[5996]=0;d=b;b=K[f+75|0];c=b<<24>>24<0;c=ba(100,f+80|0,(c?J[f+64>>2]:d)|0,(c?J[f+68>>2]:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break P}J[f+104>>2]=J[c+8>>2];b=J[c+4>>2];J[f+96>>2]=J[c>>2];J[f+100>>2]=b;J[c>>2]=0;J[c+4>>2]=0;J[c+8>>2]=0;J[5996]=0;b=_(97,f+96|0,5242)|0;d=J[5996];J[5996]=0;if((d|0)==1){break O}e=J[b>>2];J[f+48>>2]=J[b+4>>2];d=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+51|0]=d;H[f+52|0]=d>>>8;H[f+53|0]=d>>>16;H[f+54|0]=d>>>24;J[b>>2]=0;J[b+4>>2]=0;d=K[b+11|0];J[b+8>>2]=0;J[a>>2]=e;J[a+4>>2]=J[f+48>>2];b=K[f+51|0]|K[f+52|0]<<8|(K[f+53|0]<<16|K[f+54|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=d;if(H[f+75|0]<0){$a(J[f+64>>2]);}if(H[c+11|0]>=0){break b}$a(J[c>>2]);break b}a=Z()|0;break a}a=Z()|0;break M}a=Z()|0;break N}a=Z()|0;if(H[f+107|0]>=0){break N}$a(J[f+96>>2]);}if(H[f+75|0]>=0){break M}$a(J[f+64>>2]);}if(H[f+91|0]>=0){break a}$a(J[f+80>>2]);break a;case 7:J[5996]=0;b=_(97,f+116|0,5346)|0;d=J[5996];J[5996]=0;R:{S:{T:{if((d|0)!=1){J[f+88>>2]=J[b+8>>2];d=J[b+4>>2];J[f+80>>2]=J[b>>2];J[f+84>>2]=d;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;d=J[c+24>>2];b=J[c+20>>2];J[f+72>>2]=0;J[f+64>>2]=0;J[f+68>>2]=0;J[5996]=0;c=d-b|0;aa(101,f- -64|0,c<<1);e=J[5996];J[5996]=0;if((e|0)==1){break h}U:{if(b>>>0<d>>>0){d=b+c|0;while(1){c=K[b|0];J[5996]=0;e=f- -64|0;aa(98,e|0,H[(c>>>4|0)+6976|0]);g=J[5996];J[5996]=0;if((g|0)==1){break U}J[5996]=0;aa(98,e|0,H[(c&15)+6976|0]);c=J[5996];J[5996]=0;if((c|0)==1){break U}b=b+1|0;if((d|0)!=(b|0)){continue}break}}J[5996]=0;b=K[f+75|0];c=b<<24>>24<0;c=ba(100,f+80|0,(c?J[f+64>>2]:f- -64|0)|0,(c?J[f+68>>2]:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break T}J[f+104>>2]=J[c+8>>2];b=J[c+4>>2];J[f+96>>2]=J[c>>2];J[f+100>>2]=b;J[c>>2]=0;J[c+4>>2]=0;J[c+8>>2]=0;J[5996]=0;b=_(97,f+96|0,5242)|0;d=J[5996];J[5996]=0;if((d|0)==1){break S}e=J[b>>2];J[f+48>>2]=J[b+4>>2];d=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+51|0]=d;H[f+52|0]=d>>>8;H[f+53|0]=d>>>16;H[f+54|0]=d>>>24;J[b>>2]=0;J[b+4>>2]=0;d=K[b+11|0];J[b+8>>2]=0;J[a>>2]=e;J[a+4>>2]=J[f+48>>2];b=K[f+51|0]|K[f+52|0]<<8|(K[f+53|0]<<16|K[f+54|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=d;if(H[f+75|0]<0){$a(J[f+64>>2]);}if(H[c+11|0]>=0){break b}$a(J[c>>2]);break b}break h}a=Z()|0;break a}a=Z()|0;break R}a=Z()|0;if(H[f+107|0]>=0){break R}$a(J[f+96>>2]);}break g;case 9:J[5996]=0;b=_(97,f+116|0,5363)|0;d=J[5996];J[5996]=0;V:{W:{X:{if((d|0)!=1){J[f+88>>2]=J[b+8>>2];d=J[b+4>>2];J[f+80>>2]=J[b>>2];J[f+84>>2]=d;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;d=J[c+24>>2];b=J[c+20>>2];J[f+72>>2]=0;J[f+64>>2]=0;J[f+68>>2]=0;J[5996]=0;c=d-b|0;aa(101,f- -64|0,c<<1);e=J[5996];J[5996]=0;if((e|0)==1){break j}Y:{if(b>>>0<d>>>0){d=b+c|0;while(1){c=K[b|0];J[5996]=0;e=f- -64|0;aa(98,e|0,H[(c>>>4|0)+6976|0]);g=J[5996];J[5996]=0;if((g|0)==1){break Y}J[5996]=0;aa(98,e|0,H[(c&15)+6976|0]);c=J[5996];J[5996]=0;if((c|0)==1){break Y}b=b+1|0;if((d|0)!=(b|0)){continue}break}}J[5996]=0;b=K[f+75|0];c=b<<24>>24<0;c=ba(100,f+80|0,(c?J[f+64>>2]:f- -64|0)|0,(c?J[f+68>>2]:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break X}J[f+104>>2]=J[c+8>>2];b=J[c+4>>2];J[f+96>>2]=J[c>>2];J[f+100>>2]=b;J[c>>2]=0;J[c+4>>2]=0;J[c+8>>2]=0;J[5996]=0;b=_(97,f+96|0,5242)|0;d=J[5996];J[5996]=0;if((d|0)==1){break W}e=J[b>>2];J[f+48>>2]=J[b+4>>2];d=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+51|0]=d;H[f+52|0]=d>>>8;H[f+53|0]=d>>>16;H[f+54|0]=d>>>24;J[b>>2]=0;J[b+4>>2]=0;d=K[b+11|0];J[b+8>>2]=0;J[a>>2]=e;J[a+4>>2]=J[f+48>>2];b=K[f+51|0]|K[f+52|0]<<8|(K[f+53|0]<<16|K[f+54|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=d;if(H[f+75|0]<0){$a(J[f+64>>2]);}if(H[c+11|0]>=0){break b}$a(J[c>>2]);break b}break j}a=Z()|0;break a}a=Z()|0;break V}a=Z()|0;if(H[f+107|0]>=0){break V}$a(J[f+96>>2]);}break i;case 6:J[5996]=0;b=_(97,f+116|0,5355)|0;d=J[5996];J[5996]=0;Z:{_:{$:{if((d|0)!=1){J[f+88>>2]=J[b+8>>2];d=J[b+4>>2];J[f+80>>2]=J[b>>2];J[f+84>>2]=d;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;d=J[c+24>>2];b=J[c+20>>2];J[f+72>>2]=0;J[f+64>>2]=0;J[f+68>>2]=0;J[5996]=0;c=d-b|0;aa(101,f- -64|0,c<<1);e=J[5996];J[5996]=0;if((e|0)==1){break l}aa:{if(b>>>0<d>>>0){d=b+c|0;while(1){c=K[b|0];J[5996]=0;e=f- -64|0;aa(98,e|0,H[(c>>>4|0)+6976|0]);g=J[5996];J[5996]=0;if((g|0)==1){break aa}J[5996]=0;aa(98,e|0,H[(c&15)+6976|0]);c=J[5996];J[5996]=0;if((c|0)==1){break aa}b=b+1|0;if((d|0)!=(b|0)){continue}break}}J[5996]=0;b=K[f+75|0];c=b<<24>>24<0;c=ba(100,f+80|0,(c?J[f+64>>2]:f- -64|0)|0,(c?J[f+68>>2]:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break $}J[f+104>>2]=J[c+8>>2];b=J[c+4>>2];J[f+96>>2]=J[c>>2];J[f+100>>2]=b;J[c>>2]=0;J[c+4>>2]=0;J[c+8>>2]=0;J[5996]=0;b=_(97,f+96|0,5242)|0;d=J[5996];J[5996]=0;if((d|0)==1){break _}e=J[b>>2];J[f+48>>2]=J[b+4>>2];d=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+51|0]=d;H[f+52|0]=d>>>8;H[f+53|0]=d>>>16;H[f+54|0]=d>>>24;J[b>>2]=0;J[b+4>>2]=0;d=K[b+11|0];J[b+8>>2]=0;J[a>>2]=e;J[a+4>>2]=J[f+48>>2];b=K[f+51|0]|K[f+52|0]<<8|(K[f+53|0]<<16|K[f+54|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=d;if(H[f+75|0]<0){$a(J[f+64>>2]);}if(H[c+11|0]>=0){break b}$a(J[c>>2]);break b}break l}a=Z()|0;break a}a=Z()|0;break Z}a=Z()|0;if(H[f+107|0]>=0){break Z}$a(J[f+96>>2]);}break k;case 8:J[5996]=0;b=_(97,f+116|0,5372)|0;d=J[5996];J[5996]=0;ba:{ca:{da:{if((d|0)!=1){J[f+88>>2]=J[b+8>>2];d=J[b+4>>2];J[f+80>>2]=J[b>>2];J[f+84>>2]=d;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;d=J[c+24>>2];b=J[c+20>>2];J[f+72>>2]=0;J[f+64>>2]=0;J[f+68>>2]=0;J[5996]=0;c=d-b|0;aa(101,f- -64|0,c<<1);e=J[5996];J[5996]=0;if((e|0)==1){break n}ea:{if(b>>>0<d>>>0){d=b+c|0;while(1){c=K[b|0];J[5996]=0;e=f- -64|0;aa(98,e|0,H[(c>>>4|0)+6976|0]);g=J[5996];J[5996]=0;if((g|0)==1){break ea}J[5996]=0;aa(98,e|0,H[(c&15)+6976|0]);c=J[5996];J[5996]=0;if((c|0)==1){break ea}b=b+1|0;if((d|0)!=(b|0)){continue}break}}J[5996]=0;b=K[f+75|0];c=b<<24>>24<0;c=ba(100,f+80|0,(c?J[f+64>>2]:f- -64|0)|0,(c?J[f+68>>2]:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break da}J[f+104>>2]=J[c+8>>2];b=J[c+4>>2];J[f+96>>2]=J[c>>2];J[f+100>>2]=b;J[c>>2]=0;J[c+4>>2]=0;J[c+8>>2]=0;J[5996]=0;b=_(97,f+96|0,5242)|0;d=J[5996];J[5996]=0;if((d|0)==1){break ca}e=J[b>>2];J[f+48>>2]=J[b+4>>2];d=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+51|0]=d;H[f+52|0]=d>>>8;H[f+53|0]=d>>>16;H[f+54|0]=d>>>24;J[b>>2]=0;J[b+4>>2]=0;d=K[b+11|0];J[b+8>>2]=0;J[a>>2]=e;J[a+4>>2]=J[f+48>>2];b=K[f+51|0]|K[f+52|0]<<8|(K[f+53|0]<<16|K[f+54|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=d;if(H[f+75|0]<0){$a(J[f+64>>2]);}if(H[c+11|0]>=0){break b}$a(J[c>>2]);break b}break n}a=Z()|0;break a}a=Z()|0;break ba}a=Z()|0;if(H[f+107|0]>=0){break ba}$a(J[f+96>>2]);}break m;case 1:J[5996]=0;b=_(97,f+116|0,4180)|0;c=J[5996];J[5996]=0;if((c|0)!=1){d=J[b>>2];J[f+96>>2]=J[b+4>>2];c=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+99|0]=c;H[f+100|0]=c>>>8;H[f+101|0]=c>>>16;H[f+102|0]=c>>>24;c=K[f+127|0];J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;J[a>>2]=d;J[a+4>>2]=J[f+96>>2];b=K[f+99|0]|K[f+100|0]<<8|(K[f+101|0]<<16|K[f+102|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=c;break b}a=Z()|0;break a;case 24:break s;case 25:break v;case 23:break w;case 20:break y;case 21:break z;case 19:break A;case 18:break B;case 0:break D;default:break t}}J[5996]=0;b=_(97,f+116|0,4350)|0;c=J[5996];J[5996]=0;if((c|0)!=1){d=J[b>>2];J[f+96>>2]=J[b+4>>2];c=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+99|0]=c;H[f+100|0]=c>>>8;H[f+101|0]=c>>>16;H[f+102|0]=c>>>24;c=K[f+127|0];J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;J[a>>2]=d;J[a+4>>2]=J[f+96>>2];b=K[f+99|0]|K[f+100|0]<<8|(K[f+101|0]<<16|K[f+102|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=c;break b}a=Z()|0;break a}J[5996]=0;b=_(97,f+116|0,5244)|0;c=J[5996];J[5996]=0;fa:{ga:{ha:{ia:{ja:{ka:{la:{if((c|0)!=1){J[f+56>>2]=J[b+8>>2];c=J[b+4>>2];J[f+48>>2]=J[b>>2];J[f+52>>2]=c;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;c=J[d>>2];e=J[d+4>>2];b=K[d+11|0];J[5996]=0;h=c;c=b<<24>>24<0;c=ba(100,f+48|0,(c?h:d)|0,(c?e:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break la}J[f+72>>2]=J[c+8>>2];b=J[c+4>>2];J[f+64>>2]=J[c>>2];J[f+68>>2]=b;J[c>>2]=0;J[c+4>>2]=0;J[c+8>>2]=0;J[5996]=0;e=_(97,f- -64|0,4354)|0;b=J[5996];J[5996]=0;if((b|0)==1){break ka}J[f+88>>2]=J[e+8>>2];b=J[e+4>>2];J[f+80>>2]=J[e>>2];J[f+84>>2]=b;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;g=J[d+16>>2];h=J[d+12>>2];b=K[d+23|0];J[5996]=0;i=d+12|0;d=b<<24>>24<0;d=ba(100,f+80|0,(d?h:i)|0,(d?g:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break ja}J[f+104>>2]=J[d+8>>2];b=J[d+4>>2];J[f+96>>2]=J[d>>2];J[f+100>>2]=b;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;b=_(97,f+96|0,5242)|0;g=J[5996];J[5996]=0;if((g|0)==1){break ia}h=J[b>>2];J[f+24>>2]=J[b+4>>2];g=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+27|0]=g;H[f+28|0]=g>>>8;H[f+29|0]=g>>>16;H[f+30|0]=g>>>24;J[b>>2]=0;J[b+4>>2]=0;g=K[b+11|0];J[b+8>>2]=0;J[a>>2]=h;J[a+4>>2]=J[f+24>>2];b=K[f+27|0]|K[f+28|0]<<8|(K[f+29|0]<<16|K[f+30|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=g;if(H[d+11|0]<0){$a(J[d>>2]);}if(H[e+11|0]<0){$a(J[e>>2]);}if(H[c+11|0]>=0){break b}$a(J[c>>2]);break b}a=Z()|0;break a}a=Z()|0;break fa}a=Z()|0;break ga}a=Z()|0;break ha}a=Z()|0;if(H[f+107|0]>=0){break ha}$a(J[f+96>>2]);}if(H[f+91|0]>=0){break ga}$a(J[f+80>>2]);}if(H[f+75|0]>=0){break fa}$a(J[f+64>>2]);}if(H[f+59|0]>=0){break a}$a(J[f+48>>2]);break a}J[5996]=0;b=_(97,f+116|0,5339)|0;c=J[5996];J[5996]=0;ma:{na:{oa:{pa:{qa:{ra:{sa:{if((c|0)!=1){J[f+56>>2]=J[b+8>>2];c=J[b+4>>2];J[f+48>>2]=J[b>>2];J[f+52>>2]=c;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;c=J[d>>2];e=J[d+4>>2];b=K[d+11|0];J[5996]=0;h=c;c=b<<24>>24<0;c=ba(100,f+48|0,(c?h:d)|0,(c?e:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break sa}J[f+72>>2]=J[c+8>>2];b=J[c+4>>2];J[f+64>>2]=J[c>>2];J[f+68>>2]=b;J[c>>2]=0;J[c+4>>2]=0;J[c+8>>2]=0;J[5996]=0;e=_(97,f- -64|0,4354)|0;b=J[5996];J[5996]=0;if((b|0)==1){break ra}J[f+88>>2]=J[e+8>>2];b=J[e+4>>2];J[f+80>>2]=J[e>>2];J[f+84>>2]=b;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;g=J[d+16>>2];h=J[d+12>>2];b=K[d+23|0];J[5996]=0;i=d+12|0;d=b<<24>>24<0;d=ba(100,f+80|0,(d?h:i)|0,(d?g:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break qa}J[f+104>>2]=J[d+8>>2];b=J[d+4>>2];J[f+96>>2]=J[d>>2];J[f+100>>2]=b;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;b=_(97,f+96|0,5242)|0;g=J[5996];J[5996]=0;if((g|0)==1){break pa}h=J[b>>2];J[f+24>>2]=J[b+4>>2];g=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+27|0]=g;H[f+28|0]=g>>>8;H[f+29|0]=g>>>16;H[f+30|0]=g>>>24;J[b>>2]=0;J[b+4>>2]=0;g=K[b+11|0];J[b+8>>2]=0;J[a>>2]=h;J[a+4>>2]=J[f+24>>2];b=K[f+27|0]|K[f+28|0]<<8|(K[f+29|0]<<16|K[f+30|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=g;if(H[d+11|0]<0){$a(J[d>>2]);}if(H[e+11|0]<0){$a(J[e>>2]);}if(H[c+11|0]>=0){break b}$a(J[c>>2]);break b}a=Z()|0;break a}a=Z()|0;break ma}a=Z()|0;break na}a=Z()|0;break oa}a=Z()|0;if(H[f+107|0]>=0){break oa}$a(J[f+96>>2]);}if(H[f+91|0]>=0){break na}$a(J[f+80>>2]);}if(H[f+75|0]>=0){break ma}$a(J[f+64>>2]);}if(H[f+59|0]>=0){break a}$a(J[f+48>>2]);break a}J[5996]=0;b=_(97,f+116|0,5333)|0;c=J[5996];J[5996]=0;ta:{ua:{va:{wa:{xa:{ya:{za:{if((c|0)!=1){J[f+56>>2]=J[b+8>>2];c=J[b+4>>2];J[f+48>>2]=J[b>>2];J[f+52>>2]=c;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;c=J[d>>2];e=J[d+4>>2];b=K[d+11|0];J[5996]=0;h=c;c=b<<24>>24<0;c=ba(100,f+48|0,(c?h:d)|0,(c?e:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break za}J[f+72>>2]=J[c+8>>2];b=J[c+4>>2];J[f+64>>2]=J[c>>2];J[f+68>>2]=b;J[c>>2]=0;J[c+4>>2]=0;J[c+8>>2]=0;J[5996]=0;e=_(97,f- -64|0,4354)|0;b=J[5996];J[5996]=0;if((b|0)==1){break ya}J[f+88>>2]=J[e+8>>2];b=J[e+4>>2];J[f+80>>2]=J[e>>2];J[f+84>>2]=b;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;g=J[d+16>>2];h=J[d+12>>2];b=K[d+23|0];J[5996]=0;i=d+12|0;d=b<<24>>24<0;d=ba(100,f+80|0,(d?h:i)|0,(d?g:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break xa}J[f+104>>2]=J[d+8>>2];b=J[d+4>>2];J[f+96>>2]=J[d>>2];J[f+100>>2]=b;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;b=_(97,f+96|0,5242)|0;g=J[5996];J[5996]=0;if((g|0)==1){break wa}h=J[b>>2];J[f+24>>2]=J[b+4>>2];g=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+27|0]=g;H[f+28|0]=g>>>8;H[f+29|0]=g>>>16;H[f+30|0]=g>>>24;J[b>>2]=0;J[b+4>>2]=0;g=K[b+11|0];J[b+8>>2]=0;J[a>>2]=h;J[a+4>>2]=J[f+24>>2];b=K[f+27|0]|K[f+28|0]<<8|(K[f+29|0]<<16|K[f+30|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=g;if(H[d+11|0]<0){$a(J[d>>2]);}if(H[e+11|0]<0){$a(J[e>>2]);}if(H[c+11|0]>=0){break b}$a(J[c>>2]);break b}a=Z()|0;break a}a=Z()|0;break ta}a=Z()|0;break ua}a=Z()|0;break va}a=Z()|0;if(H[f+107|0]>=0){break va}$a(J[f+96>>2]);}if(H[f+91|0]>=0){break ua}$a(J[f+80>>2]);}if(H[f+75|0]>=0){break ta}$a(J[f+64>>2]);}if(H[f+59|0]>=0){break a}$a(J[f+48>>2]);break a}J[5996]=0;b=_(97,f+116|0,5321)|0;c=J[5996];J[5996]=0;Aa:{Ba:{Ca:{Da:{Ea:{Fa:{Ga:{if((c|0)!=1){J[f+56>>2]=J[b+8>>2];c=J[b+4>>2];J[f+48>>2]=J[b>>2];J[f+52>>2]=c;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;c=J[d>>2];e=J[d+4>>2];b=K[d+11|0];J[5996]=0;h=c;c=b<<24>>24<0;c=ba(100,f+48|0,(c?h:d)|0,(c?e:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break Ga}J[f+72>>2]=J[c+8>>2];b=J[c+4>>2];J[f+64>>2]=J[c>>2];J[f+68>>2]=b;J[c>>2]=0;J[c+4>>2]=0;J[c+8>>2]=0;J[5996]=0;e=_(97,f- -64|0,4354)|0;b=J[5996];J[5996]=0;if((b|0)==1){break Fa}J[f+88>>2]=J[e+8>>2];b=J[e+4>>2];J[f+80>>2]=J[e>>2];J[f+84>>2]=b;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;g=J[d+16>>2];h=J[d+12>>2];b=K[d+23|0];J[5996]=0;i=d+12|0;d=b<<24>>24<0;d=ba(100,f+80|0,(d?h:i)|0,(d?g:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break Ea}J[f+104>>2]=J[d+8>>2];b=J[d+4>>2];J[f+96>>2]=J[d>>2];J[f+100>>2]=b;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;b=_(97,f+96|0,5242)|0;g=J[5996];J[5996]=0;if((g|0)==1){break Da}h=J[b>>2];J[f+24>>2]=J[b+4>>2];g=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+27|0]=g;H[f+28|0]=g>>>8;H[f+29|0]=g>>>16;H[f+30|0]=g>>>24;J[b>>2]=0;J[b+4>>2]=0;g=K[b+11|0];J[b+8>>2]=0;J[a>>2]=h;J[a+4>>2]=J[f+24>>2];b=K[f+27|0]|K[f+28|0]<<8|(K[f+29|0]<<16|K[f+30|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=g;if(H[d+11|0]<0){$a(J[d>>2]);}if(H[e+11|0]<0){$a(J[e>>2]);}if(H[c+11|0]>=0){break b}$a(J[c>>2]);break b}a=Z()|0;break a}a=Z()|0;break Aa}a=Z()|0;break Ba}a=Z()|0;break Ca}a=Z()|0;if(H[f+107|0]>=0){break Ca}$a(J[f+96>>2]);}if(H[f+91|0]>=0){break Ba}$a(J[f+80>>2]);}if(H[f+75|0]>=0){break Aa}$a(J[f+64>>2]);}if(H[f+59|0]>=0){break a}$a(J[f+48>>2]);break a}J[5996]=0;b=_(97,f+116|0,5327)|0;c=J[5996];J[5996]=0;Ha:{Ia:{Ja:{Ka:{La:{Ma:{Na:{if((c|0)!=1){J[f+56>>2]=J[b+8>>2];c=J[b+4>>2];J[f+48>>2]=J[b>>2];J[f+52>>2]=c;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;c=J[d>>2];e=J[d+4>>2];b=K[d+11|0];J[5996]=0;h=c;c=b<<24>>24<0;c=ba(100,f+48|0,(c?h:d)|0,(c?e:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break Na}J[f+72>>2]=J[c+8>>2];b=J[c+4>>2];J[f+64>>2]=J[c>>2];J[f+68>>2]=b;J[c>>2]=0;J[c+4>>2]=0;J[c+8>>2]=0;J[5996]=0;e=_(97,f- -64|0,4354)|0;b=J[5996];J[5996]=0;if((b|0)==1){break Ma}J[f+88>>2]=J[e+8>>2];b=J[e+4>>2];J[f+80>>2]=J[e>>2];J[f+84>>2]=b;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;g=J[d+16>>2];h=J[d+12>>2];b=K[d+23|0];J[5996]=0;i=d+12|0;d=b<<24>>24<0;d=ba(100,f+80|0,(d?h:i)|0,(d?g:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break La}J[f+104>>2]=J[d+8>>2];b=J[d+4>>2];J[f+96>>2]=J[d>>2];J[f+100>>2]=b;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;b=_(97,f+96|0,5242)|0;g=J[5996];J[5996]=0;if((g|0)==1){break Ka}h=J[b>>2];J[f+24>>2]=J[b+4>>2];g=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+27|0]=g;H[f+28|0]=g>>>8;H[f+29|0]=g>>>16;H[f+30|0]=g>>>24;J[b>>2]=0;J[b+4>>2]=0;g=K[b+11|0];J[b+8>>2]=0;J[a>>2]=h;J[a+4>>2]=J[f+24>>2];b=K[f+27|0]|K[f+28|0]<<8|(K[f+29|0]<<16|K[f+30|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=g;if(H[d+11|0]<0){$a(J[d>>2]);}if(H[e+11|0]<0){$a(J[e>>2]);}if(H[c+11|0]>=0){break b}$a(J[c>>2]);break b}a=Z()|0;break a}a=Z()|0;break Ha}a=Z()|0;break Ia}a=Z()|0;break Ja}a=Z()|0;if(H[f+107|0]>=0){break Ja}$a(J[f+96>>2]);}if(H[f+91|0]>=0){break Ia}$a(J[f+80>>2]);}if(H[f+75|0]>=0){break Ha}$a(J[f+64>>2]);}if(H[f+59|0]>=0){break a}$a(J[f+48>>2]);break a}J[5996]=0;b=_(97,f+116|0,5296)|0;c=J[5996];J[5996]=0;Oa:{Pa:{Qa:{Ra:{Sa:{Ta:{Ua:{if((c|0)!=1){J[f+56>>2]=J[b+8>>2];c=J[b+4>>2];J[f+48>>2]=J[b>>2];J[f+52>>2]=c;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;c=J[d>>2];e=J[d+4>>2];b=K[d+11|0];J[5996]=0;h=c;c=b<<24>>24<0;c=ba(100,f+48|0,(c?h:d)|0,(c?e:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break Ua}J[f+72>>2]=J[c+8>>2];b=J[c+4>>2];J[f+64>>2]=J[c>>2];J[f+68>>2]=b;J[c>>2]=0;J[c+4>>2]=0;J[c+8>>2]=0;J[5996]=0;e=_(97,f- -64|0,4354)|0;b=J[5996];J[5996]=0;if((b|0)==1){break Ta}J[f+88>>2]=J[e+8>>2];b=J[e+4>>2];J[f+80>>2]=J[e>>2];J[f+84>>2]=b;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;g=J[d+16>>2];h=J[d+12>>2];b=K[d+23|0];J[5996]=0;i=d+12|0;d=b<<24>>24<0;d=ba(100,f+80|0,(d?h:i)|0,(d?g:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break Sa}J[f+104>>2]=J[d+8>>2];b=J[d+4>>2];J[f+96>>2]=J[d>>2];J[f+100>>2]=b;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;b=_(97,f+96|0,5242)|0;g=J[5996];J[5996]=0;if((g|0)==1){break Ra}h=J[b>>2];J[f+24>>2]=J[b+4>>2];g=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+27|0]=g;H[f+28|0]=g>>>8;H[f+29|0]=g>>>16;H[f+30|0]=g>>>24;J[b>>2]=0;J[b+4>>2]=0;g=K[b+11|0];J[b+8>>2]=0;J[a>>2]=h;J[a+4>>2]=J[f+24>>2];b=K[f+27|0]|K[f+28|0]<<8|(K[f+29|0]<<16|K[f+30|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=g;if(H[d+11|0]<0){$a(J[d>>2]);}if(H[e+11|0]<0){$a(J[e>>2]);}if(H[c+11|0]>=0){break b}$a(J[c>>2]);break b}a=Z()|0;break a}a=Z()|0;break Oa}a=Z()|0;break Pa}a=Z()|0;break Qa}a=Z()|0;if(H[f+107|0]>=0){break Qa}$a(J[f+96>>2]);}if(H[f+91|0]>=0){break Pa}$a(J[f+80>>2]);}if(H[f+75|0]>=0){break Oa}$a(J[f+64>>2]);}if(H[f+59|0]>=0){break a}$a(J[f+48>>2]);break a}if(!J[J[J[c+32>>2]+16>>2]>>2]){J[5996]=0;b=_(97,f+116|0,5272)|0;c=J[5996];J[5996]=0;Va:{Wa:{Xa:{Ya:{Za:{_a:{$a:{if((c|0)!=1){J[f+56>>2]=J[b+8>>2];c=J[b+4>>2];J[f+48>>2]=J[b>>2];J[f+52>>2]=c;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;c=J[d>>2];e=J[d+4>>2];b=K[d+11|0];J[5996]=0;h=c;c=b<<24>>24<0;c=ba(100,f+48|0,(c?h:d)|0,(c?e:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break $a}J[f+72>>2]=J[c+8>>2];b=J[c+4>>2];J[f+64>>2]=J[c>>2];J[f+68>>2]=b;J[c>>2]=0;J[c+4>>2]=0;J[c+8>>2]=0;J[5996]=0;e=_(97,f- -64|0,4354)|0;b=J[5996];J[5996]=0;if((b|0)==1){break _a}J[f+88>>2]=J[e+8>>2];b=J[e+4>>2];J[f+80>>2]=J[e>>2];J[f+84>>2]=b;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;g=J[d+16>>2];h=J[d+12>>2];b=K[d+23|0];J[5996]=0;i=d+12|0;d=b<<24>>24<0;d=ba(100,f+80|0,(d?h:i)|0,(d?g:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break Za}J[f+104>>2]=J[d+8>>2];b=J[d+4>>2];J[f+96>>2]=J[d>>2];J[f+100>>2]=b;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;b=_(97,f+96|0,5242)|0;g=J[5996];J[5996]=0;if((g|0)==1){break Ya}h=J[b>>2];J[f+24>>2]=J[b+4>>2];g=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+27|0]=g;H[f+28|0]=g>>>8;H[f+29|0]=g>>>16;H[f+30|0]=g>>>24;J[b>>2]=0;J[b+4>>2]=0;g=K[b+11|0];J[b+8>>2]=0;J[a>>2]=h;J[a+4>>2]=J[f+24>>2];b=K[f+27|0]|K[f+28|0]<<8|(K[f+29|0]<<16|K[f+30|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=g;if(H[d+11|0]<0){$a(J[d>>2]);}if(H[e+11|0]<0){$a(J[e>>2]);}if(H[c+11|0]>=0){break b}$a(J[c>>2]);break b}a=Z()|0;break a}a=Z()|0;break Va}a=Z()|0;break Wa}a=Z()|0;break Xa}a=Z()|0;if(H[f+107|0]>=0){break Xa}$a(J[f+96>>2]);}if(H[f+91|0]>=0){break Wa}$a(J[f+80>>2]);}if(H[f+75|0]>=0){break Va}$a(J[f+64>>2]);}if(H[f+59|0]>=0){break a}$a(J[f+48>>2]);break a}J[5996]=0;b=_(97,f+116|0,5251)|0;c=J[5996];J[5996]=0;ab:{bb:{cb:{db:{eb:{fb:{gb:{hb:{ib:{jb:{kb:{if((c|0)!=1){J[f+16>>2]=J[b+8>>2];c=J[b+4>>2];J[f+8>>2]=J[b>>2];J[f+12>>2]=c;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;c=J[d>>2];e=J[d+4>>2];b=K[d+11|0];J[5996]=0;h=c;c=b<<24>>24<0;c=ba(100,f+8|0,(c?h:d)|0,(c?e:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break kb}J[f+32>>2]=J[c+8>>2];b=J[c+4>>2];J[f+24>>2]=J[c>>2];J[f+28>>2]=b;J[c>>2]=0;J[c+4>>2]=0;J[c+8>>2]=0;J[5996]=0;e=_(97,f+24|0,4354)|0;b=J[5996];J[5996]=0;if((b|0)==1){break jb}J[f+56>>2]=J[e+8>>2];b=J[e+4>>2];J[f+48>>2]=J[e>>2];J[f+52>>2]=b;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;g=J[d+16>>2];h=J[d+12>>2];b=K[d+23|0];J[5996]=0;i=h;h=b<<24>>24<0;g=ba(100,f+48|0,(h?i:d+12|0)|0,(h?g:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break ib}J[f+72>>2]=J[g+8>>2];b=J[g+4>>2];J[f+64>>2]=J[g>>2];J[f+68>>2]=b;J[g>>2]=0;J[g+4>>2]=0;J[g+8>>2]=0;J[5996]=0;h=_(97,f- -64|0,4354)|0;b=J[5996];J[5996]=0;if((b|0)==1){break hb}J[f+88>>2]=J[h+8>>2];b=J[h+4>>2];J[f+80>>2]=J[h>>2];J[f+84>>2]=b;J[h>>2]=0;J[h+4>>2]=0;J[h+8>>2]=0;i=J[d+28>>2];j=J[d+24>>2];b=K[d+35|0];J[5996]=0;k=d+24|0;d=b<<24>>24<0;d=ba(100,f+80|0,(d?j:k)|0,(d?i:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break gb}J[f+104>>2]=J[d+8>>2];b=J[d+4>>2];J[f+96>>2]=J[d>>2];J[f+100>>2]=b;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;b=_(97,f+96|0,5242)|0;i=J[5996];J[5996]=0;if((i|0)==1){break fb}j=J[b>>2];J[f+40>>2]=J[b+4>>2];i=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+43|0]=i;H[f+44|0]=i>>>8;H[f+45|0]=i>>>16;H[f+46|0]=i>>>24;J[b>>2]=0;J[b+4>>2]=0;i=K[b+11|0];J[b+8>>2]=0;J[a>>2]=j;J[a+4>>2]=J[f+40>>2];b=K[f+43|0]|K[f+44|0]<<8|(K[f+45|0]<<16|K[f+46|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=i;if(H[d+11|0]<0){$a(J[d>>2]);}if(H[h+11|0]<0){$a(J[h>>2]);}if(H[g+11|0]<0){$a(J[g>>2]);}if(H[e+11|0]<0){$a(J[e>>2]);}if(H[c+11|0]>=0){break b}$a(J[c>>2]);break b}a=Z()|0;break a}a=Z()|0;break ab}a=Z()|0;break bb}a=Z()|0;break cb}a=Z()|0;break db}a=Z()|0;break eb}a=Z()|0;if(H[f+107|0]>=0){break eb}$a(J[f+96>>2]);}if(H[f+91|0]>=0){break db}$a(J[f+80>>2]);}if(H[f+75|0]>=0){break cb}$a(J[f+64>>2]);}if(H[f+59|0]>=0){break bb}$a(J[f+48>>2]);}if(H[f+35|0]>=0){break ab}$a(J[f+24>>2]);}if(H[f+19|0]>=0){break a}$a(J[f+8>>2]);break a}J[5996]=0;b=_(97,f+116|0,5289)|0;d=J[5996];J[5996]=0;if((d|0)==1){break o}J[f+72>>2]=J[b+8>>2];d=J[b+4>>2];J[f+64>>2]=J[b>>2];J[f+68>>2]=d;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;J[5996]=0;b=f+96|0;aa(119,b|0,c+4|0);d=J[5996];J[5996]=0;if((d|0)==1){break q}J[5996]=0;e=b;b=K[f+107|0];d=b<<24>>24<0;b=ba(100,f- -64|0,(d?J[f+96>>2]:e)|0,(d?J[f+100>>2]:b)|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break r}J[f+88>>2]=J[b+8>>2];d=J[b+4>>2];J[f+80>>2]=J[b>>2];J[f+84>>2]=d;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;lb:{if(H[f+107|0]>=0){break lb}$a(J[f+96>>2]);if(H[b+11|0]>=0){break lb}$a(J[b>>2]);}mb:{nb:{ob:{pb:{qb:{b=J[c+8>>2];d=J[c+12>>2];if((b|0)!=(d|0)){while(1){rb:{if(H[b+11|0]>=0){J[f+104>>2]=J[b+8>>2];c=J[b+4>>2];J[f+96>>2]=J[b>>2];J[f+100>>2]=c;break rb}c=J[b+4>>2];e=J[b>>2];J[5996]=0;ea(40,f+96|0,e|0,c|0);c=J[5996];J[5996]=0;if((c|0)!=1){break rb}a=Z()|0;break nb}J[5996]=0;H[f+108|0]=1;c=ba(99,f+96|0,0,4354)|0;e=J[5996];J[5996]=0;if((e|0)==1){break qb}J[f+56>>2]=J[c+8>>2];e=J[c+4>>2];J[f+48>>2]=J[c>>2];J[f+52>>2]=e;J[c>>2]=0;J[c+4>>2]=0;J[c+8>>2]=0;J[5996]=0;e=K[f+59|0];g=e<<24>>24<0;ba(100,f+80|0,(g?J[f+48>>2]:f+48|0)|0,(g?J[f+52>>2]:e)|0)|0;e=J[5996];J[5996]=0;if((e|0)==1){break pb}if(H[f+59|0]<0){$a(J[f+48>>2]);}if(!(!K[f+108|0]|H[f+107|0]>=0)){$a(J[c>>2]);}b=b+12|0;if((d|0)!=(b|0)){continue}break}}J[5996]=0;b=_(97,f+80|0,5242)|0;c=J[5996];J[5996]=0;if((c|0)!=1){break mb}a=Z()|0;break nb}a=Z()|0;break ob}a=Z()|0;if(H[f+59|0]>=0){break ob}$a(J[f+48>>2]);}if(!K[f+108|0]|H[f+107|0]>=0){break nb}$a(J[f+96>>2]);}if(H[f+91|0]>=0){break a}$a(J[f+80>>2]);break a}d=J[b>>2];J[f+48>>2]=J[b+4>>2];c=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+51|0]=c;H[f+52|0]=c>>>8;H[f+53|0]=c>>>16;H[f+54|0]=c>>>24;J[b>>2]=0;J[b+4>>2]=0;c=K[b+11|0];J[b+8>>2]=0;J[a>>2]=d;J[a+4>>2]=J[f+48>>2];b=K[f+51|0]|K[f+52|0]<<8|(K[f+53|0]<<16|K[f+54|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=c;break b}a=Z()|0;break a}J[5996]=0;ga(44,2350,2210,725,5107);a=J[5996];J[5996]=0;if((a|0)==1){a=Z()|0;break a}B();}J[5996]=0;b=_(97,f+116|0,5302)|0;g=J[5996];J[5996]=0;sb:{tb:{ub:{vb:{wb:{xb:{yb:{if((g|0)!=1){J[f+88>>2]=J[b+8>>2];g=J[b+4>>2];J[f+80>>2]=J[b>>2];J[f+84>>2]=g;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;J[5996]=0;b=f- -64|0;aa(119,b|0,c+4|0);c=J[5996];J[5996]=0;if((c|0)==1){break yb}J[5996]=0;h=b;b=K[f+75|0];c=b<<24>>24<0;b=ba(100,f+80|0,(c?J[f+64>>2]:h)|0,(c?J[f+68>>2]:b)|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break xb}J[f+104>>2]=J[b+8>>2];c=J[b+4>>2];J[f+96>>2]=J[b>>2];J[f+100>>2]=c;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;zb:{if(H[f+75|0]>=0){break zb}$a(J[f+64>>2]);if(H[b+11|0]>=0){break zb}$a(J[b>>2]);}if(e){c=P(e,12)+d|0;while(1){J[5996]=0;b=ba(99,d|0,0,4354)|0;d=J[5996];J[5996]=0;if((d|0)==1){break vb}J[f+72>>2]=J[b+8>>2];d=J[b+4>>2];J[f+64>>2]=J[b>>2];J[f+68>>2]=d;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;J[5996]=0;d=K[f+75|0];e=d<<24>>24<0;ba(100,f+96|0,(e?J[f+64>>2]:f- -64|0)|0,(e?J[f+68>>2]:d)|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break ub}if(H[f+75|0]<0){$a(J[f+64>>2]);}d=b+12|0;if((c|0)!=(d|0)){continue}break}}J[5996]=0;b=_(97,f+96|0,5242)|0;c=J[5996];J[5996]=0;if((c|0)==1){break tb}d=J[b>>2];J[f+64>>2]=J[b+4>>2];c=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+67|0]=c;H[f+68|0]=c>>>8;H[f+69|0]=c>>>16;H[f+70|0]=c>>>24;J[b>>2]=0;J[b+4>>2]=0;c=K[b+11|0];J[b+8>>2]=0;J[a>>2]=d;J[a+4>>2]=J[f+64>>2];b=K[f+67|0]|K[f+68|0]<<8|(K[f+69|0]<<16|K[f+70|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=c;break b}a=Z()|0;break a}a=Z()|0;break wb}a=Z()|0;if(H[f+75|0]>=0){break wb}$a(J[f+64>>2]);}if(H[f+91|0]>=0){break a}$a(J[f+80>>2]);break a}a=Z()|0;break sb}a=Z()|0;if(H[f+75|0]>=0){break sb}$a(J[f+64>>2]);break sb}a=Z()|0;}if(H[f+107|0]>=0){break a}$a(J[f+96>>2]);break a}a=Z()|0;if(H[f+107|0]>=0){break p}$a(J[f+96>>2]);break p}a=Z()|0;}if(H[f+75|0]>=0){break a}$a(J[f+64>>2]);break a}a=Z()|0;break a}a=Z()|0;}if(H[f+75|0]<0){$a(J[f+64>>2]);}if(H[f+91|0]>=0){break a}$a(J[f+80>>2]);break a}a=Z()|0;}if(H[f+75|0]<0){$a(J[f+64>>2]);}if(H[f+91|0]>=0){break a}$a(J[f+80>>2]);break a}a=Z()|0;}if(H[f+75|0]<0){$a(J[f+64>>2]);}if(H[f+91|0]>=0){break a}$a(J[f+80>>2]);break a}a=Z()|0;}if(H[f+75|0]<0){$a(J[f+64>>2]);}if(H[f+91|0]>=0){break a}$a(J[f+80>>2]);break a}J[5996]=0;H[f+108|0]=1;b=_(97,f+116|0,5315)|0;c=J[5996];J[5996]=0;Ab:{Bb:{Cb:{Db:{if((c|0)!=1){J[f+72>>2]=J[b+8>>2];c=J[b+4>>2];J[f+64>>2]=J[b>>2];J[f+68>>2]=c;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;J[5996]=0;b=K[f+107|0];c=b<<24>>24<0;c=ba(100,f- -64|0,(c?J[f+96>>2]:f+96|0)|0,(c?J[f+100>>2]:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break Db}J[f+88>>2]=J[c+8>>2];b=J[c+4>>2];J[f+80>>2]=J[c>>2];J[f+84>>2]=b;J[c>>2]=0;J[c+4>>2]=0;J[c+8>>2]=0;J[5996]=0;b=_(97,f+80|0,5242)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Cb}e=J[b>>2];J[f+48>>2]=J[b+4>>2];d=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+51|0]=d;H[f+52|0]=d>>>8;H[f+53|0]=d>>>16;H[f+54|0]=d>>>24;J[b>>2]=0;J[b+4>>2]=0;d=K[b+11|0];J[b+8>>2]=0;J[a>>2]=e;J[a+4>>2]=J[f+48>>2];b=K[f+51|0]|K[f+52|0]<<8|(K[f+53|0]<<16|K[f+54|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=d;if(H[c+11|0]<0){$a(J[c>>2]);}if(!K[f+108|0]|H[f+107|0]>=0){break b}$a(J[f+96>>2]);break b}a=Z()|0;break Ab}a=Z()|0;break Bb}a=Z()|0;if(H[f+91|0]>=0){break Bb}$a(J[f+80>>2]);}if(H[f+75|0]>=0){break Ab}$a(J[f+64>>2]);}if(!K[f+108|0]|H[f+107|0]>=0){break a}$a(J[f+96>>2]);break a}J[5996]=0;H[f+108|0]=1;b=_(97,f+116|0,5283)|0;c=J[5996];J[5996]=0;Eb:{Fb:{Gb:{Hb:{if((c|0)!=1){J[f+72>>2]=J[b+8>>2];c=J[b+4>>2];J[f+64>>2]=J[b>>2];J[f+68>>2]=c;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;J[5996]=0;b=K[f+107|0];c=b<<24>>24<0;c=ba(100,f- -64|0,(c?J[f+96>>2]:f+96|0)|0,(c?J[f+100>>2]:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break Hb}J[f+88>>2]=J[c+8>>2];b=J[c+4>>2];J[f+80>>2]=J[c>>2];J[f+84>>2]=b;J[c>>2]=0;J[c+4>>2]=0;J[c+8>>2]=0;J[5996]=0;b=_(97,f+80|0,5242)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Gb}e=J[b>>2];J[f+48>>2]=J[b+4>>2];d=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+51|0]=d;H[f+52|0]=d>>>8;H[f+53|0]=d>>>16;H[f+54|0]=d>>>24;J[b>>2]=0;J[b+4>>2]=0;d=K[b+11|0];J[b+8>>2]=0;J[a>>2]=e;J[a+4>>2]=J[f+48>>2];b=K[f+51|0]|K[f+52|0]<<8|(K[f+53|0]<<16|K[f+54|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=d;if(H[c+11|0]<0){$a(J[c>>2]);}if(!K[f+108|0]|H[f+107|0]>=0){break b}$a(J[f+96>>2]);break b}a=Z()|0;break Eb}a=Z()|0;break Fb}a=Z()|0;if(H[f+91|0]>=0){break Fb}$a(J[f+80>>2]);}if(H[f+75|0]>=0){break Eb}$a(J[f+64>>2]);}if(!K[f+108|0]|H[f+107|0]>=0){break a}$a(J[f+96>>2]);break a}J[5996]=0;H[f+108|0]=1;b=_(97,f+116|0,5310)|0;c=J[5996];J[5996]=0;Ib:{Jb:{Kb:{Lb:{if((c|0)!=1){J[f+72>>2]=J[b+8>>2];c=J[b+4>>2];J[f+64>>2]=J[b>>2];J[f+68>>2]=c;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;J[5996]=0;b=K[f+107|0];c=b<<24>>24<0;c=ba(100,f- -64|0,(c?J[f+96>>2]:f+96|0)|0,(c?J[f+100>>2]:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break Lb}J[f+88>>2]=J[c+8>>2];b=J[c+4>>2];J[f+80>>2]=J[c>>2];J[f+84>>2]=b;J[c>>2]=0;J[c+4>>2]=0;J[c+8>>2]=0;J[5996]=0;b=_(97,f+80|0,5242)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Kb}e=J[b>>2];J[f+48>>2]=J[b+4>>2];d=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+51|0]=d;H[f+52|0]=d>>>8;H[f+53|0]=d>>>16;H[f+54|0]=d>>>24;J[b>>2]=0;J[b+4>>2]=0;d=K[b+11|0];J[b+8>>2]=0;J[a>>2]=e;J[a+4>>2]=J[f+48>>2];b=K[f+51|0]|K[f+52|0]<<8|(K[f+53|0]<<16|K[f+54|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=d;if(H[c+11|0]<0){$a(J[c>>2]);}if(!K[f+108|0]|H[f+107|0]>=0){break b}$a(J[f+96>>2]);break b}a=Z()|0;break Ib}a=Z()|0;break Jb}a=Z()|0;if(H[f+91|0]>=0){break Jb}$a(J[f+80>>2]);}if(H[f+75|0]>=0){break Ib}$a(J[f+64>>2]);}if(!K[f+108|0]|H[f+107|0]>=0){break a}$a(J[f+96>>2]);break a}J[5996]=0;H[f+108|0]=1;b=_(97,f+116|0,5279)|0;c=J[5996];J[5996]=0;Mb:{Nb:{Ob:{Pb:{if((c|0)!=1){J[f+72>>2]=J[b+8>>2];c=J[b+4>>2];J[f+64>>2]=J[b>>2];J[f+68>>2]=c;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;J[5996]=0;b=K[f+107|0];c=b<<24>>24<0;c=ba(100,f- -64|0,(c?J[f+96>>2]:f+96|0)|0,(c?J[f+100>>2]:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break Pb}J[f+88>>2]=J[c+8>>2];b=J[c+4>>2];J[f+80>>2]=J[c>>2];J[f+84>>2]=b;J[c>>2]=0;J[c+4>>2]=0;J[c+8>>2]=0;J[5996]=0;b=_(97,f+80|0,5242)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Ob}e=J[b>>2];J[f+48>>2]=J[b+4>>2];d=K[b+7|0]|K[b+8|0]<<8|(K[b+9|0]<<16|K[b+10|0]<<24);H[f+51|0]=d;H[f+52|0]=d>>>8;H[f+53|0]=d>>>16;H[f+54|0]=d>>>24;J[b>>2]=0;J[b+4>>2]=0;d=K[b+11|0];J[b+8>>2]=0;J[a>>2]=e;J[a+4>>2]=J[f+48>>2];b=K[f+51|0]|K[f+52|0]<<8|(K[f+53|0]<<16|K[f+54|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+12|0]=1;H[a+11|0]=d;if(H[c+11|0]<0){$a(J[c>>2]);}if(!K[f+108|0]|H[f+107|0]>=0){break b}$a(J[f+96>>2]);break b}a=Z()|0;break Mb}a=Z()|0;break Nb}a=Z()|0;if(H[f+91|0]>=0){break Nb}$a(J[f+80>>2]);}if(H[f+75|0]>=0){break Mb}$a(J[f+64>>2]);}if(!K[f+108|0]|H[f+107|0]>=0){break a}$a(J[f+96>>2]);break a}if(H[f+127|0]<0){$a(J[f+116>>2]);}Ua=f+128|0;return}if(H[f+127|0]<0){$a(J[f+116>>2]);}da(a|0);B();}function dn(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;var f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0;h=Ua-112|0;Ua=h;a:{b:{c:{d:{e:{f:{g:{h:{i:{j:{switch(J[c>>2]){case 2:f=J[c+8>>2];c=cb(4);J[h+48>>2]=c;b=c+4|0;J[h+56>>2]=b;H[c|0]=2;H[c+1|0]=80;H[c+2|0]=75;H[c+3|0]=98;J[h+52>>2]=b;J[5996]=0;H[h+16|0]=0;d=h+48|0;ea(126,d|0,29,h+16|0);b=J[5996];J[5996]=0;if((b|0)==1){break c}e=J[h+48>>2];c=K[f+11|0];b=c<<24>>24<0;eb(e+4|0,b?J[f>>2]:f,b?J[f+4>>2]:c);H[a+24|0]=0;H[a+25|0]=0;H[a+26|0]=0;H[a+27|0]=0;H[a+28|0]=0;H[a+29|0]=0;H[a+30|0]=0;H[a+31|0]=0;H[a+16|0]=0;H[a+17|0]=0;H[a+18|0]=0;H[a+19|0]=0;H[a+20|0]=0;H[a+21|0]=0;H[a+22|0]=0;H[a+23|0]=0;H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;H[a+12|0]=0;H[a+13|0]=0;H[a+14|0]=0;H[a+15|0]=0;H[a|0]=0;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;H[a+4|0]=0;H[a+5|0]=0;H[a+6|0]=0;H[a+7|0]=0;J[5996]=0;_(19,a|0,d|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){g=Z()|0;if((K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))>>>0>=29){$a(K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24));H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;}if(!e){break a}$a(e);break a}if(!e){break d}$a(e);break d;case 3:d=J[c+8>>2];c=cb(3);J[h+48>>2]=c;b=c+3|0;J[h+56>>2]=b;H[c+2|0]=104;H[c|0]=80;H[c+1|0]=75;J[h+52>>2]=b;J[5996]=0;H[h+16|0]=0;ea(126,h+48|0,17,h+16|0);b=J[5996];J[5996]=0;if((b|0)==1){g=Z()|0;a=J[h+48>>2];if(!a){break a}$a(a);break a}f=J[h+48>>2];c=K[d+11|0];b=c<<24>>24<0;eb(f+3|0,b?J[d>>2]:d,b?J[d+4>>2]:c);H[a+5|0]=0;H[a+6|0]=0;H[a+7|0]=0;H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;H[a+12|0]=0;H[a+28|0]=0;H[a+29|0]=0;H[a+30|0]=0;H[a+31|0]=0;H[a+21|0]=0;H[a+22|0]=0;H[a+23|0]=0;H[a+24|0]=0;H[a+25|0]=0;H[a+26|0]=0;H[a+27|0]=0;H[a+28|0]=0;H[a+13|0]=0;H[a+14|0]=0;H[a+15|0]=0;H[a+16|0]=0;H[a+17|0]=0;H[a+18|0]=0;H[a+19|0]=0;H[a+20|0]=0;H[a|0]=2;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;H[a+4|0]=118;H[a+5|0]=169;J[5996]=0;i=_(19,a|0,h+48|0)|0;b=J[5996];J[5996]=0;if((b|0)!=1){k:{g=K[i|0]|K[i+1|0]<<8|(K[i+2|0]<<16|K[i+3|0]<<24);e=g-29|0;c=g>>>0<29?g:e;b=c+1|0;d=a+4|0;if(b>>>0>(g>>>0<=28?28:K[d|0]|K[d+1|0]<<8|(K[d+2|0]<<16|K[d+3|0]<<24))>>>0){J[5996]=0;aa(124,i|0,b+(b>>>1|0)|0);b=J[5996];J[5996]=0;if((b|0)==1){break k}g=K[i|0]|K[i+1|0]<<8|(K[i+2|0]<<16|K[i+3|0]<<24);e=g-29|0;}b=K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24);a=g>>>0<29;b=c+(a?d:b)|0;eb(b+1|0,b,(a?g:e)-c|0);a=(K[i|0]|K[i+1|0]<<8|(K[i+2|0]<<16|K[i+3|0]<<24))+1|0;H[i|0]=a;H[i+1|0]=a>>>8;H[i+2|0]=a>>>16;H[i+3|0]=a>>>24;H[b|0]=136;if(!f){break d}$a(f);break d}}g=Z()|0;if((K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))>>>0>=29){$a(K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24));H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;}if(!f){break a}$a(f);break a;case 4:H[a|0]=0;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;H[a+4|0]=0;H[a+5|0]=0;H[a+6|0]=0;H[a+7|0]=0;H[a+24|0]=0;H[a+25|0]=0;H[a+26|0]=0;H[a+27|0]=0;H[a+28|0]=0;H[a+29|0]=0;H[a+30|0]=0;H[a+31|0]=0;H[a+16|0]=0;H[a+17|0]=0;H[a+18|0]=0;H[a+19|0]=0;H[a+20|0]=0;H[a+21|0]=0;H[a+22|0]=0;H[a+23|0]=0;H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;H[a+12|0]=0;H[a+13|0]=0;H[a+14|0]=0;H[a+15|0]=0;b=J[c+4>>2];J[5996]=0;e=nc(a,b,0);b=J[5996];J[5996]=0;if((b|0)!=1){l:{f=K[e|0]|K[e+1|0]<<8|(K[e+2|0]<<16|K[e+3|0]<<24);b=f-29|0;d=f>>>0<29?f:b;c=d+1|0;if(c>>>0>(f>>>0<=28?28:K[e+4|0]|K[e+5|0]<<8|(K[e+6|0]<<16|K[e+7|0]<<24))>>>0){J[5996]=0;aa(124,e|0,c+(c>>>1|0)|0);b=J[5996];J[5996]=0;if((b|0)==1){break l}f=K[e|0]|K[e+1|0]<<8|(K[e+2|0]<<16|K[e+3|0]<<24);b=f-29|0;}a=f>>>0<29;c=d+(a?e+4|0:K[e+8|0]|K[e+9|0]<<8|(K[e+10|0]<<16|K[e+11|0]<<24))|0;eb(c+1|0,c,(a?f:b)-d|0);a=(K[e|0]|K[e+1|0]<<8|(K[e+2|0]<<16|K[e+3|0]<<24))+1|0;H[e|0]=a;H[e+1|0]=a>>>8;H[e+2|0]=a>>>16;H[e+3|0]=a>>>24;H[c|0]=178;break d}}break b;case 5:H[a|0]=0;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;H[a+4|0]=0;H[a+5|0]=0;H[a+6|0]=0;H[a+7|0]=0;H[a+24|0]=0;H[a+25|0]=0;H[a+26|0]=0;H[a+27|0]=0;H[a+28|0]=0;H[a+29|0]=0;H[a+30|0]=0;H[a+31|0]=0;H[a+16|0]=0;H[a+17|0]=0;H[a+18|0]=0;H[a+19|0]=0;H[a+20|0]=0;H[a+21|0]=0;H[a+22|0]=0;H[a+23|0]=0;H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;H[a+12|0]=0;H[a+13|0]=0;H[a+14|0]=0;H[a+15|0]=0;b=J[c+4>>2];J[5996]=0;e=nc(a,b,0);b=J[5996];J[5996]=0;if((b|0)!=1){m:{f=K[e|0]|K[e+1|0]<<8|(K[e+2|0]<<16|K[e+3|0]<<24);b=f-29|0;d=f>>>0<29?f:b;c=d+1|0;if(c>>>0>(f>>>0<=28?28:K[e+4|0]|K[e+5|0]<<8|(K[e+6|0]<<16|K[e+7|0]<<24))>>>0){J[5996]=0;aa(124,e|0,c+(c>>>1|0)|0);b=J[5996];J[5996]=0;if((b|0)==1){break m}f=K[e|0]|K[e+1|0]<<8|(K[e+2|0]<<16|K[e+3|0]<<24);b=f-29|0;}a=f>>>0<29;c=d+(a?e+4|0:K[e+8|0]|K[e+9|0]<<8|(K[e+10|0]<<16|K[e+11|0]<<24))|0;eb(c+1|0,c,(a?f:b)-d|0);a=(K[e|0]|K[e+1|0]<<8|(K[e+2|0]<<16|K[e+3|0]<<24))+1|0;H[e|0]=a;H[e+1|0]=a>>>8;H[e+2|0]=a>>>16;H[e+3|0]=a>>>24;H[c|0]=177;break d}}break b;case 8:J[h+48>>2]=32;J[h+16>>2]=136;J[h+80>>2]=166;J[h+4>>2]=b?136:135;vd(a,h+48|0,h+16|0,h+80|0,c+20|0,h+4|0);break d;case 7:J[h+48>>2]=32;J[h+16>>2]=136;J[h+80>>2]=170;J[h+4>>2]=b?136:135;vd(a,h+48|0,h+16|0,h+80|0,c+20|0,h+4|0);break d;case 9:J[h+48>>2]=32;J[h+16>>2]=136;J[h+80>>2]=169;J[h+4>>2]=b?136:135;vd(a,h+48|0,h+16|0,h+80|0,c+20|0,h+4|0);break d;case 10:H[a+5|0]=0;H[a+6|0]=0;H[a+7|0]=0;H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;H[a+12|0]=0;H[a+4|0]=107;H[a|0]=1;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;H[a+28|0]=0;H[a+29|0]=0;H[a+30|0]=0;H[a+31|0]=0;H[a+21|0]=0;H[a+22|0]=0;H[a+23|0]=0;H[a+24|0]=0;H[a+25|0]=0;H[a+26|0]=0;H[a+27|0]=0;H[a+28|0]=0;H[a+13|0]=0;H[a+14|0]=0;H[a+15|0]=0;H[a+16|0]=0;H[a+17|0]=0;H[a+18|0]=0;H[a+19|0]=0;H[a+20|0]=0;c=K[d+8|0]|K[d+9|0]<<8|(K[d+10|0]<<16|K[d+11|0]<<24);e=-28;f=1;b=K[d|0]|K[d+1|0]<<8|(K[d+2|0]<<16|K[d+3|0]<<24);l=b>>>0<29;n:{j=l?b:b-29|0;b=j+1|0;if(b>>>0>=29){J[5996]=0;aa(124,a|0,b+(b>>>1|0)|0);b=J[5996];J[5996]=0;if((b|0)==1){break n}n=K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24);f=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);e=f-29|0;}i=a+4|0;b=f>>>0<29;g=(b?i:n)+1|0;eb(j+g|0,g,(b?f:e)-1|0);b=j+(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))|0;H[a|0]=b;H[a+1|0]=b>>>8;H[a+2|0]=b>>>16;H[a+3|0]=b>>>24;o:{if(!j){break o}b=l?d+4|0:c;c=j&7;p:{if(!c){f=b;break p}d=0;f=b;while(1){H[g|0]=K[f|0];f=f+1|0;g=g+1|0;d=d+1|0;if((c|0)!=(d|0)){continue}break}}if(j>>>0<8){break o}b=b+j|0;while(1){H[g|0]=K[f|0];H[g+1|0]=K[f+1|0];H[g+2|0]=K[f+2|0];H[g+3|0]=K[f+3|0];H[g+4|0]=K[f+4|0];H[g+5|0]=K[f+5|0];H[g+6|0]=K[f+6|0];H[g+7|0]=K[f+7|0];g=g+8|0;f=f+8|0;if((b|0)!=(f|0)){continue}break}}g=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);n=g-29|0;d=g>>>0<29?g:n;b=d+1|0;if(b>>>0>(g>>>0<=28?28:K[a+4|0]|K[a+5|0]<<8|(K[a+6|0]<<16|K[a+7|0]<<24))>>>0){J[5996]=0;aa(124,a|0,b+(b>>>1|0)|0);b=J[5996];J[5996]=0;if((b|0)==1){break n}g=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);n=g-29|0;}b=g>>>0<29;c=d+(b?i:K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24))|0;eb(c+1|0,c,(b?g:n)-d|0);b=(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))+1|0;H[a|0]=b;H[a+1|0]=b>>>8;H[a+2|0]=b>>>16;H[a+3|0]=b>>>24;H[c|0]=108;break d}break b;case 11:H[a+5|0]=0;H[a+6|0]=0;H[a+7|0]=0;H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;H[a+12|0]=0;H[a+4|0]=124;H[a|0]=1;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;H[a+28|0]=0;H[a+29|0]=0;H[a+30|0]=0;H[a+31|0]=0;H[a+21|0]=0;H[a+22|0]=0;H[a+23|0]=0;H[a+24|0]=0;H[a+25|0]=0;H[a+26|0]=0;H[a+27|0]=0;H[a+28|0]=0;H[a+13|0]=0;H[a+14|0]=0;H[a+15|0]=0;H[a+16|0]=0;H[a+17|0]=0;H[a+18|0]=0;H[a+19|0]=0;H[a+20|0]=0;c=K[d+8|0]|K[d+9|0]<<8|(K[d+10|0]<<16|K[d+11|0]<<24);e=-28;f=1;b=K[d|0]|K[d+1|0]<<8|(K[d+2|0]<<16|K[d+3|0]<<24);i=b>>>0<29;q:{l=i?b:b-29|0;b=l+1|0;if(b>>>0>=29){J[5996]=0;aa(124,a|0,b+(b>>>1|0)|0);b=J[5996];J[5996]=0;if((b|0)==1){break q}n=K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24);f=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);e=f-29|0;}b=f>>>0<29;g=(b?a+4|0:n)+1|0;eb(l+g|0,g,(b?f:e)-1|0);b=l+(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))|0;H[a|0]=b;H[a+1|0]=b>>>8;H[a+2|0]=b>>>16;H[a+3|0]=b>>>24;if(!l){break d}a=i?d+4|0:c;b=l&7;r:{if(!b){f=a;break r}d=0;f=a;while(1){H[g|0]=K[f|0];f=f+1|0;g=g+1|0;d=d+1|0;if((b|0)!=(d|0)){continue}break}}if(l>>>0<8){break d}a=a+l|0;while(1){H[g|0]=K[f|0];H[g+1|0]=K[f+1|0];H[g+2|0]=K[f+2|0];H[g+3|0]=K[f+3|0];H[g+4|0]=K[f+4|0];H[g+5|0]=K[f+5|0];H[g+6|0]=K[f+6|0];H[g+7|0]=K[f+7|0];g=g+8|0;f=f+8|0;if((a|0)!=(f|0)){continue}break}break d}break b;case 12:J[h+48>>2]=b?173:172;te(a,d,h+48|0);break d;case 13:H[a+5|0]=0;H[a+6|0]=0;H[a+7|0]=0;H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;H[a+12|0]=0;H[a|0]=2;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;H[a+4|0]=118;H[a+28|0]=0;H[a+29|0]=0;H[a+30|0]=0;H[a+31|0]=0;H[a+21|0]=0;H[a+22|0]=0;H[a+23|0]=0;H[a+24|0]=0;H[a+25|0]=0;H[a+26|0]=0;H[a+27|0]=0;H[a+28|0]=0;H[a+13|0]=0;H[a+14|0]=0;H[a+15|0]=0;H[a+16|0]=0;H[a+17|0]=0;H[a+18|0]=0;H[a+19|0]=0;H[a+20|0]=0;H[a+5|0]=99;c=K[d+8|0]|K[d+9|0]<<8|(K[d+10|0]<<16|K[d+11|0]<<24);e=-27;f=2;b=K[d|0]|K[d+1|0]<<8|(K[d+2|0]<<16|K[d+3|0]<<24);l=b>>>0<29;s:{j=l?b:b-29|0;b=j+2|0;if(b>>>0>=29){J[5996]=0;aa(124,a|0,b+(b>>>1|0)|0);b=J[5996];J[5996]=0;if((b|0)==1){break s}n=K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24);f=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);e=f-29|0;}i=a+4|0;b=f>>>0<29;g=(b?i:n)+2|0;eb(j+g|0,g,(b?f:e)-2|0);b=j+(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))|0;H[a|0]=b;H[a+1|0]=b>>>8;H[a+2|0]=b>>>16;H[a+3|0]=b>>>24;t:{if(!j){break t}b=l?d+4|0:c;c=j&7;u:{if(!c){f=b;break u}d=0;f=b;while(1){H[g|0]=K[f|0];f=f+1|0;g=g+1|0;d=d+1|0;if((c|0)!=(d|0)){continue}break}}if(j>>>0<8){break t}b=b+j|0;while(1){H[g|0]=K[f|0];H[g+1|0]=K[f+1|0];H[g+2|0]=K[f+2|0];H[g+3|0]=K[f+3|0];H[g+4|0]=K[f+4|0];H[g+5|0]=K[f+5|0];H[g+6|0]=K[f+6|0];H[g+7|0]=K[f+7|0];g=g+8|0;f=f+8|0;if((b|0)!=(f|0)){continue}break}}g=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);n=g-29|0;d=g>>>0<29?g:n;b=d+1|0;if(b>>>0>(g>>>0<=28?28:K[a+4|0]|K[a+5|0]<<8|(K[a+6|0]<<16|K[a+7|0]<<24))>>>0){J[5996]=0;aa(124,a|0,b+(b>>>1|0)|0);b=J[5996];J[5996]=0;if((b|0)==1){break s}g=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);n=g-29|0;}b=g>>>0<29;c=d+(b?i:K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24))|0;eb(c+1|0,c,(b?g:n)-d|0);b=(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))+1|0;H[a|0]=b;H[a+1|0]=b>>>8;H[a+2|0]=b>>>16;H[a+3|0]=b>>>24;H[c|0]=104;break d}break b;case 14:b=J[J[J[c+32>>2]>>2]+80>>2];if(!(Wb(1239,1)&(b^-1))){J[h+48>>2]=105;te(a,d,h+48|0);break d}H[a|0]=0;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;H[a+4|0]=0;H[a+5|0]=0;H[a+6|0]=0;H[a+7|0]=0;H[a+24|0]=0;H[a+25|0]=0;H[a+26|0]=0;H[a+27|0]=0;H[a+28|0]=0;H[a+29|0]=0;H[a+30|0]=0;H[a+31|0]=0;H[a+16|0]=0;H[a+17|0]=0;H[a+18|0]=0;H[a+19|0]=0;H[a+20|0]=0;H[a+21|0]=0;H[a+22|0]=0;H[a+23|0]=0;H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;H[a+12|0]=0;H[a+13|0]=0;H[a+14|0]=0;H[a+15|0]=0;J[h+72>>2]=K[a+28|0]|K[a+29|0]<<8|(K[a+30|0]<<16|K[a+31|0]<<24);b=K[a+24|0]|K[a+25|0]<<8|(K[a+26|0]<<16|K[a+27|0]<<24);c=h- -64|0;J[c>>2]=K[a+20|0]|K[a+21|0]<<8|(K[a+22|0]<<16|K[a+23|0]<<24);J[c+4>>2]=b;b=K[a+16|0]|K[a+17|0]<<8|(K[a+18|0]<<16|K[a+19|0]<<24);J[h+56>>2]=K[a+12|0]|K[a+13|0]<<8|(K[a+14|0]<<16|K[a+15|0]<<24);J[h+60>>2]=b;b=K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24);J[h+48>>2]=K[a+4|0]|K[a+5|0]<<8|(K[a+6|0]<<16|K[a+7|0]<<24);J[h+52>>2]=b;b=K[d+28|0]|K[d+29|0]<<8|(K[d+30|0]<<16|K[d+31|0]<<24);H[a+28|0]=b;H[a+29|0]=b>>>8;H[a+30|0]=b>>>16;H[a+31|0]=b>>>24;e=K[d+24|0]|K[d+25|0]<<8|(K[d+26|0]<<16|K[d+27|0]<<24);b=K[d+20|0]|K[d+21|0]<<8|(K[d+22|0]<<16|K[d+23|0]<<24);H[a+20|0]=b;H[a+21|0]=b>>>8;H[a+22|0]=b>>>16;H[a+23|0]=b>>>24;H[a+24|0]=e;H[a+25|0]=e>>>8;H[a+26|0]=e>>>16;H[a+27|0]=e>>>24;e=K[d+16|0]|K[d+17|0]<<8|(K[d+18|0]<<16|K[d+19|0]<<24);b=K[d+12|0]|K[d+13|0]<<8|(K[d+14|0]<<16|K[d+15|0]<<24);H[a+12|0]=b;H[a+13|0]=b>>>8;H[a+14|0]=b>>>16;H[a+15|0]=b>>>24;H[a+16|0]=e;H[a+17|0]=e>>>8;H[a+18|0]=e>>>16;H[a+19|0]=e>>>24;e=K[d+8|0]|K[d+9|0]<<8|(K[d+10|0]<<16|K[d+11|0]<<24);b=K[d+4|0]|K[d+5|0]<<8|(K[d+6|0]<<16|K[d+7|0]<<24);H[a+4|0]=b;H[a+5|0]=b>>>8;H[a+6|0]=b>>>16;H[a+7|0]=b>>>24;H[a+8|0]=e;H[a+9|0]=e>>>8;H[a+10|0]=e>>>16;H[a+11|0]=e>>>24;b=J[h+72>>2];H[d+28|0]=b;H[d+29|0]=b>>>8;H[d+30|0]=b>>>16;H[d+31|0]=b>>>24;e=J[c+4>>2];b=J[c>>2];H[d+20|0]=b;H[d+21|0]=b>>>8;H[d+22|0]=b>>>16;H[d+23|0]=b>>>24;H[d+24|0]=e;H[d+25|0]=e>>>8;H[d+26|0]=e>>>16;H[d+27|0]=e>>>24;c=J[h+60>>2];b=J[h+56>>2];H[d+12|0]=b;H[d+13|0]=b>>>8;H[d+14|0]=b>>>16;H[d+15|0]=b>>>24;H[d+16|0]=c;H[d+17|0]=c>>>8;H[d+18|0]=c>>>16;H[d+19|0]=c>>>24;c=J[h+52>>2];b=J[h+48>>2];H[d+4|0]=b;H[d+5|0]=b>>>8;H[d+6|0]=b>>>16;H[d+7|0]=b>>>24;H[d+8|0]=c;H[d+9|0]=c>>>8;H[d+10|0]=c>>>16;H[d+11|0]=c>>>24;J[a>>2]=J[d>>2];J[d>>2]=0;break d;case 15:H[a|0]=0;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;H[a+4|0]=0;H[a+5|0]=0;H[a+6|0]=0;H[a+7|0]=0;H[a+24|0]=0;H[a+25|0]=0;H[a+26|0]=0;H[a+27|0]=0;H[a+28|0]=0;H[a+29|0]=0;H[a+30|0]=0;H[a+31|0]=0;H[a+16|0]=0;H[a+17|0]=0;H[a+18|0]=0;H[a+19|0]=0;H[a+20|0]=0;H[a+21|0]=0;H[a+22|0]=0;H[a+23|0]=0;H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;H[a+12|0]=0;H[a+13|0]=0;H[a+14|0]=0;H[a+15|0]=0;J[5996]=0;c=_(127,a|0,130)|0;b=J[5996];J[5996]=0;if((b|0)!=1){v:{J[5996]=0;c=_(127,c|0,146)|0;b=J[5996];J[5996]=0;if((b|0)==1){break v}J[5996]=0;k=_(127,c|0,99)|0;b=J[5996];J[5996]=0;if((b|0)==1){break v}e=K[d+8|0]|K[d+9|0]<<8|(K[d+10|0]<<16|K[d+11|0]<<24);c=K[d|0]|K[d+1|0]<<8|(K[d+2|0]<<16|K[d+3|0]<<24);l=c>>>0<29;f=K[k|0]|K[k+1|0]<<8|(K[k+2|0]<<16|K[k+3|0]<<24);b=f-29|0;j=l?c:c-29|0;i=f>>>0<29?f:b;c=j+i|0;if(c>>>0>(f>>>0<=28?28:K[k+4|0]|K[k+5|0]<<8|(K[k+6|0]<<16|K[k+7|0]<<24))>>>0){J[5996]=0;aa(124,k|0,c+(c>>>1|0)|0);b=J[5996];J[5996]=0;if((b|0)==1){break v}f=K[k|0]|K[k+1|0]<<8|(K[k+2|0]<<16|K[k+3|0]<<24);b=f-29|0;}c=f>>>0<29;g=i+(c?k+4|0:K[k+8|0]|K[k+9|0]<<8|(K[k+10|0]<<16|K[k+11|0]<<24))|0;eb(j+g|0,g,(c?f:b)-i|0);b=j+(K[k|0]|K[k+1|0]<<8|(K[k+2|0]<<16|K[k+3|0]<<24))|0;H[k|0]=b;H[k+1|0]=b>>>8;H[k+2|0]=b>>>16;H[k+3|0]=b>>>24;w:{if(!j){break w}b=l?d+4|0:e;c=j&7;x:{if(!c){f=b;break x}d=0;f=b;while(1){H[g|0]=K[f|0];f=f+1|0;g=g+1|0;d=d+1|0;if((c|0)!=(d|0)){continue}break}}if(j>>>0<8){break w}b=b+j|0;while(1){H[g|0]=K[f|0];H[g+1|0]=K[f+1|0];H[g+2|0]=K[f+2|0];H[g+3|0]=K[f+3|0];H[g+4|0]=K[f+4|0];H[g+5|0]=K[f+5|0];H[g+6|0]=K[f+6|0];H[g+7|0]=K[f+7|0];g=g+8|0;f=f+8|0;if((b|0)!=(f|0)){continue}break}}J[5996]=0;_(127,k|0,104)|0;b=J[5996];J[5996]=0;if((b|0)!=1){break d}}}break b;case 16:J[h+48>>2]=146;te(a,d,h+48|0);break d;case 1:H[a+5|0]=0;H[a+6|0]=0;H[a+7|0]=0;H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;H[a+12|0]=0;H[a+4|0]=81;H[a|0]=1;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;H[a+28|0]=0;H[a+29|0]=0;H[a+30|0]=0;H[a+31|0]=0;H[a+21|0]=0;H[a+22|0]=0;H[a+23|0]=0;H[a+24|0]=0;H[a+25|0]=0;H[a+26|0]=0;H[a+27|0]=0;H[a+28|0]=0;H[a+13|0]=0;H[a+14|0]=0;H[a+15|0]=0;H[a+16|0]=0;H[a+17|0]=0;H[a+18|0]=0;H[a+19|0]=0;H[a+20|0]=0;break d;case 0:H[a+5|0]=0;H[a+6|0]=0;H[a+7|0]=0;H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;H[a+12|0]=0;H[a+4|0]=0;H[a|0]=1;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;H[a+28|0]=0;H[a+29|0]=0;H[a+30|0]=0;H[a+31|0]=0;H[a+21|0]=0;H[a+22|0]=0;H[a+23|0]=0;H[a+24|0]=0;H[a+25|0]=0;H[a+26|0]=0;H[a+27|0]=0;H[a+28|0]=0;H[a+13|0]=0;H[a+14|0]=0;H[a+15|0]=0;H[a+16|0]=0;H[a+17|0]=0;H[a+18|0]=0;H[a+19|0]=0;H[a+20|0]=0;break d;case 17:H[a|0]=0;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;H[a+4|0]=0;H[a+5|0]=0;H[a+6|0]=0;H[a+7|0]=0;H[a+24|0]=0;H[a+25|0]=0;H[a+26|0]=0;H[a+27|0]=0;H[a+28|0]=0;H[a+29|0]=0;H[a+30|0]=0;H[a+31|0]=0;H[a+16|0]=0;H[a+17|0]=0;H[a+18|0]=0;H[a+19|0]=0;H[a+20|0]=0;H[a+21|0]=0;H[a+22|0]=0;H[a+23|0]=0;H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;H[a+12|0]=0;H[a+13|0]=0;H[a+14|0]=0;H[a+15|0]=0;c=K[d+8|0]|K[d+9|0]<<8|(K[d+10|0]<<16|K[d+11|0]<<24);n=-29;b=0;i=K[d|0]|K[d+1|0]<<8|(K[d+2|0]<<16|K[d+3|0]<<24);e=i>>>0<29;y:{l=e?i:i-29|0;if(l>>>0>=29){J[5996]=0;aa(124,a|0,l+(l>>>1|0)|0);b=J[5996];J[5996]=0;if((b|0)==1){break y}f=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);n=f-29|0;b=K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24);}i=a+4|0;k=b;b=f>>>0<29;g=b?i:k;eb(l+g|0,g,b?f:n);b=l+(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))|0;H[a|0]=b;H[a+1|0]=b>>>8;H[a+2|0]=b>>>16;H[a+3|0]=b>>>24;z:{if(!l){break z}b=e?d+4|0:c;c=l&7;A:{if(!c){f=b;break A}n=0;f=b;while(1){H[g|0]=K[f|0];f=f+1|0;g=g+1|0;n=n+1|0;if((c|0)!=(n|0)){continue}break}}if(l>>>0<8){break z}b=b+l|0;while(1){H[g|0]=K[f|0];H[g+1|0]=K[f+1|0];H[g+2|0]=K[f+2|0];H[g+3|0]=K[f+3|0];H[g+4|0]=K[f+4|0];H[g+5|0]=K[f+5|0];H[g+6|0]=K[f+6|0];H[g+7|0]=K[f+7|0];g=g+8|0;f=f+8|0;if((b|0)!=(f|0)){continue}break}}c=K[d+40|0]|K[d+41|0]<<8|(K[d+42|0]<<16|K[d+43|0]<<24);b=K[d+32|0]|K[d+33|0]<<8|(K[d+34|0]<<16|K[d+35|0]<<24);j=b>>>0<29;f=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);e=f-29|0;k=j?b:b-29|0;l=f>>>0<29?f:e;b=k+l|0;if(b>>>0>(f>>>0<=28?28:K[a+4|0]|K[a+5|0]<<8|(K[a+6|0]<<16|K[a+7|0]<<24))>>>0){J[5996]=0;aa(124,a|0,b+(b>>>1|0)|0);b=J[5996];J[5996]=0;if((b|0)==1){break y}f=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);e=f-29|0;}b=f>>>0<29;g=l+(b?i:K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24))|0;eb(k+g|0,g,(b?f:e)-l|0);b=k+(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))|0;H[a|0]=b;H[a+1|0]=b>>>8;H[a+2|0]=b>>>16;H[a+3|0]=b>>>24;if(!k){break d}a=j?d+36|0:c;b=k&7;B:{if(!b){f=a;break B}d=0;f=a;while(1){H[g|0]=K[f|0];f=f+1|0;g=g+1|0;d=d+1|0;if((b|0)!=(d|0)){continue}break}}if(k>>>0<8){break d}a=a+k|0;while(1){H[g|0]=K[f|0];H[g+1|0]=K[f+1|0];H[g+2|0]=K[f+2|0];H[g+3|0]=K[f+3|0];H[g+4|0]=K[f+4|0];H[g+5|0]=K[f+5|0];H[g+6|0]=K[f+6|0];H[g+7|0]=K[f+7|0];g=g+8|0;f=f+8|0;if((a|0)!=(f|0)){continue}break}break d}break b;case 18:J[h+48>>2]=154;se(a,d,d+32|0,h+48|0);break d;case 19:J[h+48>>2]=155;se(a,d,d+32|0,h+48|0);break d;case 21:H[a|0]=0;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;H[a+4|0]=0;H[a+5|0]=0;H[a+6|0]=0;H[a+7|0]=0;H[a+24|0]=0;H[a+25|0]=0;H[a+26|0]=0;H[a+27|0]=0;H[a+28|0]=0;H[a+29|0]=0;H[a+30|0]=0;H[a+31|0]=0;H[a+16|0]=0;H[a+17|0]=0;H[a+18|0]=0;H[a+19|0]=0;H[a+20|0]=0;H[a+21|0]=0;H[a+22|0]=0;H[a+23|0]=0;H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;H[a+12|0]=0;H[a+13|0]=0;H[a+14|0]=0;H[a+15|0]=0;c=K[d+8|0]|K[d+9|0]<<8|(K[d+10|0]<<16|K[d+11|0]<<24);n=-29;b=0;i=K[d|0]|K[d+1|0]<<8|(K[d+2|0]<<16|K[d+3|0]<<24);e=i>>>0<29;i=e?i:i-29|0;C:{if(i>>>0>=29){J[5996]=0;aa(124,a|0,i+(i>>>1|0)|0);b=J[5996];J[5996]=0;if((b|0)==1){break C}f=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);n=f-29|0;b=K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24);}j=a+4|0;k=b;b=f>>>0<29;g=b?j:k;eb(i+g|0,g,b?f:n);b=i+(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))|0;H[a|0]=b;H[a+1|0]=b>>>8;H[a+2|0]=b>>>16;H[a+3|0]=b>>>24;D:{if(!i){break D}b=e?d+4|0:c;c=i&7;E:{if(!c){f=b;break E}n=0;f=b;while(1){H[g|0]=K[f|0];f=f+1|0;g=g+1|0;n=n+1|0;if((c|0)!=(n|0)){continue}break}}if(i>>>0<8){break D}b=b+i|0;while(1){H[g|0]=K[f|0];H[g+1|0]=K[f+1|0];H[g+2|0]=K[f+2|0];H[g+3|0]=K[f+3|0];H[g+4|0]=K[f+4|0];H[g+5|0]=K[f+5|0];H[g+6|0]=K[f+6|0];H[g+7|0]=K[f+7|0];g=g+8|0;f=f+8|0;if((b|0)!=(f|0)){continue}break}}J[5996]=0;c=_(127,a|0,115)|0;b=J[5996];J[5996]=0;if((b|0)==1){break C}J[5996]=0;m=_(127,c|0,100)|0;b=J[5996];J[5996]=0;if((b|0)==1){break C}e=K[d+40|0]|K[d+41|0]<<8|(K[d+42|0]<<16|K[d+43|0]<<24);c=K[d+32|0]|K[d+33|0]<<8|(K[d+34|0]<<16|K[d+35|0]<<24);l=c>>>0<29;f=K[m|0]|K[m+1|0]<<8|(K[m+2|0]<<16|K[m+3|0]<<24);b=f-29|0;k=l?c:c-29|0;i=f>>>0<29?f:b;c=k+i|0;if(c>>>0>(f>>>0<=28?28:K[j|0]|K[j+1|0]<<8|(K[j+2|0]<<16|K[j+3|0]<<24))>>>0){J[5996]=0;aa(124,m|0,c+(c>>>1|0)|0);b=J[5996];J[5996]=0;if((b|0)==1){break C}f=K[m|0]|K[m+1|0]<<8|(K[m+2|0]<<16|K[m+3|0]<<24);b=f-29|0;}c=f>>>0<29;g=i+(c?j:K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24))|0;eb(k+g|0,g,(c?f:b)-i|0);b=k+(K[m|0]|K[m+1|0]<<8|(K[m+2|0]<<16|K[m+3|0]<<24))|0;H[m|0]=b;H[m+1|0]=b>>>8;H[m+2|0]=b>>>16;H[m+3|0]=b>>>24;F:{if(!k){break F}b=l?d+36|0:e;c=k&7;G:{if(!c){f=b;break G}d=0;f=b;while(1){H[g|0]=K[f|0];f=f+1|0;g=g+1|0;d=d+1|0;if((c|0)!=(d|0)){continue}break}}if(k>>>0<8){break F}b=b+k|0;while(1){H[g|0]=K[f|0];H[g+1|0]=K[f+1|0];H[g+2|0]=K[f+2|0];H[g+3|0]=K[f+3|0];H[g+4|0]=K[f+4|0];H[g+5|0]=K[f+5|0];H[g+6|0]=K[f+6|0];H[g+7|0]=K[f+7|0];g=g+8|0;f=f+8|0;if((b|0)!=(f|0)){continue}break}}J[5996]=0;_(127,m|0,104)|0;b=J[5996];J[5996]=0;if((b|0)!=1){break d}}break b;case 20:H[a|0]=0;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;H[a+4|0]=0;H[a+5|0]=0;H[a+6|0]=0;H[a+7|0]=0;H[a+24|0]=0;H[a+25|0]=0;H[a+26|0]=0;H[a+27|0]=0;H[a+28|0]=0;H[a+29|0]=0;H[a+30|0]=0;H[a+31|0]=0;H[a+16|0]=0;H[a+17|0]=0;H[a+18|0]=0;H[a+19|0]=0;H[a+20|0]=0;H[a+21|0]=0;H[a+22|0]=0;H[a+23|0]=0;H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;H[a+12|0]=0;H[a+13|0]=0;H[a+14|0]=0;H[a+15|0]=0;c=K[d+8|0]|K[d+9|0]<<8|(K[d+10|0]<<16|K[d+11|0]<<24);n=-29;b=0;i=K[d|0]|K[d+1|0]<<8|(K[d+2|0]<<16|K[d+3|0]<<24);e=i>>>0<29;i=e?i:i-29|0;H:{if(i>>>0>=29){J[5996]=0;aa(124,a|0,i+(i>>>1|0)|0);b=J[5996];J[5996]=0;if((b|0)==1){break H}f=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);n=f-29|0;b=K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24);}j=a+4|0;k=b;b=f>>>0<29;g=b?j:k;eb(i+g|0,g,b?f:n);b=i+(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))|0;H[a|0]=b;H[a+1|0]=b>>>8;H[a+2|0]=b>>>16;H[a+3|0]=b>>>24;I:{if(!i){break I}b=e?d+4|0:c;c=i&7;J:{if(!c){f=b;break J}n=0;f=b;while(1){H[g|0]=K[f|0];f=f+1|0;g=g+1|0;n=n+1|0;if((c|0)!=(n|0)){continue}break}}if(i>>>0<8){break I}b=b+i|0;while(1){H[g|0]=K[f|0];H[g+1|0]=K[f+1|0];H[g+2|0]=K[f+2|0];H[g+3|0]=K[f+3|0];H[g+4|0]=K[f+4|0];H[g+5|0]=K[f+5|0];H[g+6|0]=K[f+6|0];H[g+7|0]=K[f+7|0];g=g+8|0;f=f+8|0;if((b|0)!=(f|0)){continue}break}}J[5996]=0;m=_(127,a|0,100)|0;b=J[5996];J[5996]=0;if((b|0)==1){break H}e=K[d+40|0]|K[d+41|0]<<8|(K[d+42|0]<<16|K[d+43|0]<<24);c=K[d+32|0]|K[d+33|0]<<8|(K[d+34|0]<<16|K[d+35|0]<<24);l=c>>>0<29;f=K[m|0]|K[m+1|0]<<8|(K[m+2|0]<<16|K[m+3|0]<<24);b=f-29|0;k=l?c:c-29|0;i=f>>>0<29?f:b;c=k+i|0;if(c>>>0>(f>>>0<=28?28:K[j|0]|K[j+1|0]<<8|(K[j+2|0]<<16|K[j+3|0]<<24))>>>0){J[5996]=0;aa(124,m|0,c+(c>>>1|0)|0);b=J[5996];J[5996]=0;if((b|0)==1){break H}f=K[m|0]|K[m+1|0]<<8|(K[m+2|0]<<16|K[m+3|0]<<24);b=f-29|0;}c=f>>>0<29;g=i+(c?j:K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24))|0;eb(k+g|0,g,(c?f:b)-i|0);b=k+(K[m|0]|K[m+1|0]<<8|(K[m+2|0]<<16|K[m+3|0]<<24))|0;H[m|0]=b;H[m+1|0]=b>>>8;H[m+2|0]=b>>>16;H[m+3|0]=b>>>24;K:{if(!k){break K}b=l?d+36|0:e;c=k&7;L:{if(!c){f=b;break L}d=0;f=b;while(1){H[g|0]=K[f|0];f=f+1|0;g=g+1|0;d=d+1|0;if((c|0)!=(d|0)){continue}break}}if(k>>>0<8){break K}b=b+k|0;while(1){H[g|0]=K[f|0];H[g+1|0]=K[f+1|0];H[g+2|0]=K[f+2|0];H[g+3|0]=K[f+3|0];H[g+4|0]=K[f+4|0];H[g+5|0]=K[f+5|0];H[g+6|0]=K[f+6|0];H[g+7|0]=K[f+7|0];g=g+8|0;f=f+8|0;if((b|0)!=(f|0)){continue}break}}J[5996]=0;_(127,m|0,104)|0;b=J[5996];J[5996]=0;if((b|0)!=1){break d}}break b;case 22:H[a|0]=0;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;H[a+4|0]=0;H[a+5|0]=0;H[a+6|0]=0;H[a+7|0]=0;H[a+24|0]=0;H[a+25|0]=0;H[a+26|0]=0;H[a+27|0]=0;H[a+28|0]=0;H[a+29|0]=0;H[a+30|0]=0;H[a+31|0]=0;H[a+16|0]=0;H[a+17|0]=0;H[a+18|0]=0;H[a+19|0]=0;H[a+20|0]=0;H[a+21|0]=0;H[a+22|0]=0;H[a+23|0]=0;H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;H[a+12|0]=0;H[a+13|0]=0;H[a+14|0]=0;H[a+15|0]=0;J[5996]=0;k=_(127,a|0,99)|0;b=J[5996];J[5996]=0;if((b|0)!=1){M:{e=K[d+8|0]|K[d+9|0]<<8|(K[d+10|0]<<16|K[d+11|0]<<24);c=K[d|0]|K[d+1|0]<<8|(K[d+2|0]<<16|K[d+3|0]<<24);l=c>>>0<29;f=K[k|0]|K[k+1|0]<<8|(K[k+2|0]<<16|K[k+3|0]<<24);b=f-29|0;j=l?c:c-29|0;i=f>>>0<29?f:b;c=j+i|0;if(c>>>0>(f>>>0<=28?28:K[k+4|0]|K[k+5|0]<<8|(K[k+6|0]<<16|K[k+7|0]<<24))>>>0){J[5996]=0;aa(124,k|0,c+(c>>>1|0)|0);b=J[5996];J[5996]=0;if((b|0)==1){break M}f=K[k|0]|K[k+1|0]<<8|(K[k+2|0]<<16|K[k+3|0]<<24);b=f-29|0;}c=f>>>0<29;g=i+(c?k+4|0:K[k+8|0]|K[k+9|0]<<8|(K[k+10|0]<<16|K[k+11|0]<<24))|0;eb(j+g|0,g,(c?f:b)-i|0);b=j+(K[k|0]|K[k+1|0]<<8|(K[k+2|0]<<16|K[k+3|0]<<24))|0;H[k|0]=b;H[k+1|0]=b>>>8;H[k+2|0]=b>>>16;H[k+3|0]=b>>>24;N:{if(!j){break N}b=l?d+4|0:e;c=j&7;O:{if(!c){f=b;break O}f=b;while(1){H[g|0]=K[f|0];f=f+1|0;g=g+1|0;n=n+1|0;if((c|0)!=(n|0)){continue}break}}if(j>>>0<8){break N}b=b+j|0;while(1){H[g|0]=K[f|0];H[g+1|0]=K[f+1|0];H[g+2|0]=K[f+2|0];H[g+3|0]=K[f+3|0];H[g+4|0]=K[f+4|0];H[g+5|0]=K[f+5|0];H[g+6|0]=K[f+6|0];H[g+7|0]=K[f+7|0];g=g+8|0;f=f+8|0;if((b|0)!=(f|0)){continue}break}}J[5996]=0;k=_(127,k|0,103)|0;b=J[5996];J[5996]=0;if((b|0)==1){break M}e=K[d+40|0]|K[d+41|0]<<8|(K[d+42|0]<<16|K[d+43|0]<<24);c=K[d+32|0]|K[d+33|0]<<8|(K[d+34|0]<<16|K[d+35|0]<<24);l=c>>>0<29;f=K[k|0]|K[k+1|0]<<8|(K[k+2|0]<<16|K[k+3|0]<<24);b=f-29|0;j=l?c:c-29|0;i=f>>>0<29?f:b;c=j+i|0;if(c>>>0>(f>>>0<=28?28:K[k+4|0]|K[k+5|0]<<8|(K[k+6|0]<<16|K[k+7|0]<<24))>>>0){J[5996]=0;aa(124,k|0,c+(c>>>1|0)|0);b=J[5996];J[5996]=0;if((b|0)==1){break M}f=K[k|0]|K[k+1|0]<<8|(K[k+2|0]<<16|K[k+3|0]<<24);b=f-29|0;}c=f>>>0<29;g=i+(c?k+4|0:K[k+8|0]|K[k+9|0]<<8|(K[k+10|0]<<16|K[k+11|0]<<24))|0;eb(j+g|0,g,(c?f:b)-i|0);b=j+(K[k|0]|K[k+1|0]<<8|(K[k+2|0]<<16|K[k+3|0]<<24))|0;H[k|0]=b;H[k+1|0]=b>>>8;H[k+2|0]=b>>>16;H[k+3|0]=b>>>24;P:{if(!j){break P}b=l?d+36|0:e;c=j&7;Q:{if(!c){f=b;break Q}d=0;f=b;while(1){H[g|0]=K[f|0];f=f+1|0;g=g+1|0;d=d+1|0;if((c|0)!=(d|0)){continue}break}}if(j>>>0<8){break P}b=b+j|0;while(1){H[g|0]=K[f|0];H[g+1|0]=K[f+1|0];H[g+2|0]=K[f+2|0];H[g+3|0]=K[f+3|0];H[g+4|0]=K[f+4|0];H[g+5|0]=K[f+5|0];H[g+6|0]=K[f+6|0];H[g+7|0]=K[f+7|0];g=g+8|0;f=f+8|0;if((b|0)!=(f|0)){continue}break}}J[5996]=0;_(127,k|0,104)|0;b=J[5996];J[5996]=0;if((b|0)!=1){break d}}}break b;case 23:H[a|0]=0;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;H[a+4|0]=0;H[a+5|0]=0;H[a+6|0]=0;H[a+7|0]=0;H[a+24|0]=0;H[a+25|0]=0;H[a+26|0]=0;H[a+27|0]=0;H[a+28|0]=0;H[a+29|0]=0;H[a+30|0]=0;H[a+31|0]=0;H[a+16|0]=0;H[a+17|0]=0;H[a+18|0]=0;H[a+19|0]=0;H[a+20|0]=0;H[a+21|0]=0;H[a+22|0]=0;H[a+23|0]=0;H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;H[a+12|0]=0;H[a+13|0]=0;H[a+14|0]=0;H[a+15|0]=0;c=K[d+8|0]|K[d+9|0]<<8|(K[d+10|0]<<16|K[d+11|0]<<24);n=-29;b=0;i=K[d|0]|K[d+1|0]<<8|(K[d+2|0]<<16|K[d+3|0]<<24);e=i>>>0<29;i=e?i:i-29|0;R:{if(i>>>0>=29){J[5996]=0;aa(124,a|0,i+(i>>>1|0)|0);b=J[5996];J[5996]=0;if((b|0)==1){break R}f=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);n=f-29|0;b=K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24);}k=a+4|0;m=b;b=f>>>0<29;g=b?k:m;eb(i+g|0,g,b?f:n);b=i+(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))|0;H[a|0]=b;H[a+1|0]=b>>>8;H[a+2|0]=b>>>16;H[a+3|0]=b>>>24;S:{if(!i){break S}b=e?d+4|0:c;c=i&7;T:{if(!c){f=b;break T}n=0;f=b;while(1){H[g|0]=K[f|0];f=f+1|0;g=g+1|0;n=n+1|0;if((c|0)!=(n|0)){continue}break}}if(i>>>0<8){break S}b=b+i|0;while(1){H[g|0]=K[f|0];H[g+1|0]=K[f+1|0];H[g+2|0]=K[f+2|0];H[g+3|0]=K[f+3|0];H[g+4|0]=K[f+4|0];H[g+5|0]=K[f+5|0];H[g+6|0]=K[f+6|0];H[g+7|0]=K[f+7|0];g=g+8|0;f=f+8|0;if((b|0)!=(f|0)){continue}break}}J[5996]=0;m=_(127,a|0,100)|0;b=J[5996];J[5996]=0;if((b|0)==1){break R}e=K[d+72|0]|K[d+73|0]<<8|(K[d+74|0]<<16|K[d+75|0]<<24);c=K[d+64|0]|K[d+65|0]<<8|(K[d+66|0]<<16|K[d+67|0]<<24);l=c>>>0<29;f=K[m|0]|K[m+1|0]<<8|(K[m+2|0]<<16|K[m+3|0]<<24);b=f-29|0;j=l?c:c-29|0;i=f>>>0<29?f:b;c=j+i|0;if(c>>>0>(f>>>0<=28?28:K[k|0]|K[k+1|0]<<8|(K[k+2|0]<<16|K[k+3|0]<<24))>>>0){J[5996]=0;aa(124,m|0,c+(c>>>1|0)|0);b=J[5996];J[5996]=0;if((b|0)==1){break R}f=K[m|0]|K[m+1|0]<<8|(K[m+2|0]<<16|K[m+3|0]<<24);b=f-29|0;}c=f>>>0<29;g=i+(c?k:K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24))|0;eb(j+g|0,g,(c?f:b)-i|0);b=j+(K[m|0]|K[m+1|0]<<8|(K[m+2|0]<<16|K[m+3|0]<<24))|0;H[m|0]=b;H[m+1|0]=b>>>8;H[m+2|0]=b>>>16;H[m+3|0]=b>>>24;U:{if(!j){break U}b=l?d+68|0:e;c=j&7;V:{if(!c){f=b;break V}n=0;f=b;while(1){H[g|0]=K[f|0];f=f+1|0;g=g+1|0;n=n+1|0;if((c|0)!=(n|0)){continue}break}}if(j>>>0<8){break U}b=b+j|0;while(1){H[g|0]=K[f|0];H[g+1|0]=K[f+1|0];H[g+2|0]=K[f+2|0];H[g+3|0]=K[f+3|0];H[g+4|0]=K[f+4|0];H[g+5|0]=K[f+5|0];H[g+6|0]=K[f+6|0];H[g+7|0]=K[f+7|0];g=g+8|0;f=f+8|0;if((b|0)!=(f|0)){continue}break}}J[5996]=0;m=_(127,m|0,103)|0;b=J[5996];J[5996]=0;if((b|0)==1){break R}e=K[d+40|0]|K[d+41|0]<<8|(K[d+42|0]<<16|K[d+43|0]<<24);c=K[d+32|0]|K[d+33|0]<<8|(K[d+34|0]<<16|K[d+35|0]<<24);l=c>>>0<29;f=K[m|0]|K[m+1|0]<<8|(K[m+2|0]<<16|K[m+3|0]<<24);b=f-29|0;j=l?c:c-29|0;i=f>>>0<29?f:b;c=j+i|0;if(c>>>0>(f>>>0<=28?28:K[k|0]|K[k+1|0]<<8|(K[k+2|0]<<16|K[k+3|0]<<24))>>>0){J[5996]=0;aa(124,m|0,c+(c>>>1|0)|0);b=J[5996];J[5996]=0;if((b|0)==1){break R}f=K[m|0]|K[m+1|0]<<8|(K[m+2|0]<<16|K[m+3|0]<<24);b=f-29|0;}c=f>>>0<29;g=i+(c?k:K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24))|0;eb(j+g|0,g,(c?f:b)-i|0);b=j+(K[m|0]|K[m+1|0]<<8|(K[m+2|0]<<16|K[m+3|0]<<24))|0;H[m|0]=b;H[m+1|0]=b>>>8;H[m+2|0]=b>>>16;H[m+3|0]=b>>>24;W:{if(!j){break W}b=l?d+36|0:e;c=j&7;X:{if(!c){f=b;break X}d=0;f=b;while(1){H[g|0]=K[f|0];f=f+1|0;g=g+1|0;d=d+1|0;if((c|0)!=(d|0)){continue}break}}if(j>>>0<8){break W}b=b+j|0;while(1){H[g|0]=K[f|0];H[g+1|0]=K[f+1|0];H[g+2|0]=K[f+2|0];H[g+3|0]=K[f+3|0];H[g+4|0]=K[f+4|0];H[g+5|0]=K[f+5|0];H[g+6|0]=K[f+6|0];H[g+7|0]=K[f+7|0];g=g+8|0;f=f+8|0;if((b|0)!=(f|0)){continue}break}}J[5996]=0;_(127,m|0,104)|0;b=J[5996];J[5996]=0;if((b|0)!=1){break d}}break b;case 25:J[h+72>>2]=0;J[h+76>>2]=0;d=h- -64|0;J[d>>2]=0;J[d+4>>2]=0;J[h+56>>2]=0;J[h+60>>2]=0;J[h+48>>2]=0;J[h+52>>2]=0;d=J[c+4>>2];J[5996]=0;o=nc(h+48|0,d,0);d=J[5996];J[5996]=0;if((d|0)==1){g=Z()|0;if(M[h+48>>2]<29){break a}$a(J[h+56>>2]);break a}e=J[c+8>>2];m=J[c+12>>2];if((e|0)==(m|0)){c=e;break g}l=o+4|0;i=h+20|0;while(1){Y:{J[5996]=0;f=$(3,4)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Y}J[h+4>>2]=f;d=f+4|0;J[h+12>>2]=d;H[f|0]=2;H[f+1|0]=80;H[f+2|0]=75;H[f+3|0]=98;J[h+8>>2]=d;J[5996]=0;H[h+80|0]=0;ea(126,h+4|0,29,h+80|0);d=J[5996];J[5996]=0;if((d|0)==1){g=Z()|0;a=J[h+4>>2];if(!a){break f}$a(a);break f}d=0;j=K[e+11|0];f=j<<24>>24<0;eb(J[h+4>>2]+4|0,f?J[e>>2]:e,f?J[e+4>>2]:j);J[h+40>>2]=0;J[h+44>>2]=0;J[h+32>>2]=0;J[h+36>>2]=0;J[h+24>>2]=0;J[h+28>>2]=0;J[h+16>>2]=0;J[h+20>>2]=0;n=-29;j=J[o+8>>2];f=0;p=J[o>>2];k=p>>>0<29;p=k?p:p-29|0;if(p>>>0>=29){J[5996]=0;aa(124,h+16|0,p+(p>>>1|0)|0);d=J[5996];J[5996]=0;if((d|0)==1){break i}f=J[h+16>>2];n=f-29|0;g=J[h+24>>2];}else {g=d;}d=f>>>0<29;g=d?i:g;eb(g+p|0,g,d?f:n);J[h+16>>2]=p+J[h+16>>2];Z:{if(!p){break Z}d=0;k=k?l:j;f=k;j=p&7;if(j){while(1){H[g|0]=K[f|0];f=f+1|0;g=g+1|0;d=d+1|0;if((j|0)!=(d|0)){continue}break}}if(p>>>0<8){break Z}d=k+p|0;while(1){H[g|0]=K[f|0];H[g+1|0]=K[f+1|0];H[g+2|0]=K[f+2|0];H[g+3|0]=K[f+3|0];H[g+4|0]=K[f+4|0];H[g+5|0]=K[f+5|0];H[g+6|0]=K[f+6|0];H[g+7|0]=K[f+7|0];g=g+8|0;f=f+8|0;if((d|0)!=(f|0)){continue}break}}J[5996]=0;f=_(19,h+16|0,h+4|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break i}J[h+104>>2]=K[l+24|0]|K[l+25|0]<<8|(K[l+26|0]<<16|K[l+27|0]<<24);d=K[l+20|0]|K[l+21|0]<<8|(K[l+22|0]<<16|K[l+23|0]<<24);J[h+96>>2]=K[l+16|0]|K[l+17|0]<<8|(K[l+18|0]<<16|K[l+19|0]<<24);J[h+100>>2]=d;d=K[l+12|0]|K[l+13|0]<<8|(K[l+14|0]<<16|K[l+15|0]<<24);J[h+88>>2]=K[l+8|0]|K[l+9|0]<<8|(K[l+10|0]<<16|K[l+11|0]<<24);J[h+92>>2]=d;d=K[l+4|0]|K[l+5|0]<<8|(K[l+6|0]<<16|K[l+7|0]<<24);J[h+80>>2]=K[l|0]|K[l+1|0]<<8|(K[l+2|0]<<16|K[l+3|0]<<24);J[h+84>>2]=d;J[l+24>>2]=J[i+24>>2];d=J[i+20>>2];J[l+16>>2]=J[i+16>>2];J[l+20>>2]=d;d=J[i+12>>2];J[l+8>>2]=J[i+8>>2];J[l+12>>2]=d;d=J[i+4>>2];J[l>>2]=J[i>>2];J[l+4>>2]=d;d=J[h+104>>2];H[i+24|0]=d;H[i+25|0]=d>>>8;H[i+26|0]=d>>>16;H[i+27|0]=d>>>24;j=J[h+100>>2];d=J[h+96>>2];H[i+16|0]=d;H[i+17|0]=d>>>8;H[i+18|0]=d>>>16;H[i+19|0]=d>>>24;H[i+20|0]=j;H[i+21|0]=j>>>8;H[i+22|0]=j>>>16;H[i+23|0]=j>>>24;j=J[h+92>>2];d=J[h+88>>2];H[i+8|0]=d;H[i+9|0]=d>>>8;H[i+10|0]=d>>>16;H[i+11|0]=d>>>24;H[i+12|0]=j;H[i+13|0]=j>>>8;H[i+14|0]=j>>>16;H[i+15|0]=j>>>24;j=J[h+84>>2];d=J[h+80>>2];H[i|0]=d;H[i+1|0]=d>>>8;H[i+2|0]=d>>>16;H[i+3|0]=d>>>24;H[i+4|0]=j;H[i+5|0]=j>>>8;H[i+6|0]=j>>>16;H[i+7|0]=j>>>24;d=J[o>>2];J[o>>2]=J[f>>2];J[f>>2]=d;if(d>>>0>=29){$a(J[h+24>>2]);J[h+24>>2]=0;}d=J[h+4>>2];if(d){$a(d);}e=e+12|0;if((m|0)!=(e|0)){continue}break h}break}g=Z()|0;break f;case 24:J[h+72>>2]=0;J[h+76>>2]=0;f=h- -64|0;J[f>>2]=0;J[f+4>>2]=0;J[h+56>>2]=0;J[h+60>>2]=0;J[h+48>>2]=0;J[h+52>>2]=0;n=J[h+76>>2];J[h+40>>2]=n;k=J[h+72>>2];f=J[h+68>>2];m=f;J[h+32>>2]=f;J[h+36>>2]=k;j=J[h+64>>2];i=J[h+60>>2];J[h+24>>2]=i;J[h+28>>2]=j;l=J[h+56>>2];f=J[h+52>>2];J[h+16>>2]=f;J[h+20>>2]=l;J[h+76>>2]=K[d+28|0]|K[d+29|0]<<8|(K[d+30|0]<<16|K[d+31|0]<<24);o=K[d+24|0]|K[d+25|0]<<8|(K[d+26|0]<<16|K[d+27|0]<<24);J[h+68>>2]=K[d+20|0]|K[d+21|0]<<8|(K[d+22|0]<<16|K[d+23|0]<<24);J[h+72>>2]=o;o=K[d+16|0]|K[d+17|0]<<8|(K[d+18|0]<<16|K[d+19|0]<<24);J[h+60>>2]=K[d+12|0]|K[d+13|0]<<8|(K[d+14|0]<<16|K[d+15|0]<<24);J[h+64>>2]=o;o=K[d+8|0]|K[d+9|0]<<8|(K[d+10|0]<<16|K[d+11|0]<<24);J[h+52>>2]=K[d+4|0]|K[d+5|0]<<8|(K[d+6|0]<<16|K[d+7|0]<<24);J[h+56>>2]=o;H[d+28|0]=n;H[d+29|0]=n>>>8;H[d+30|0]=n>>>16;H[d+31|0]=n>>>24;H[d+20|0]=m;H[d+21|0]=m>>>8;H[d+22|0]=m>>>16;H[d+23|0]=m>>>24;H[d+24|0]=k;H[d+25|0]=k>>>8;H[d+26|0]=k>>>16;H[d+27|0]=k>>>24;H[d+12|0]=i;H[d+13|0]=i>>>8;H[d+14|0]=i>>>16;H[d+15|0]=i>>>24;H[d+16|0]=j;H[d+17|0]=j>>>8;H[d+18|0]=j>>>16;H[d+19|0]=j>>>24;H[d+4|0]=f;H[d+5|0]=f>>>8;H[d+6|0]=f>>>16;H[d+7|0]=f>>>24;H[d+8|0]=l;H[d+9|0]=l>>>8;H[d+10|0]=l>>>16;H[d+11|0]=l>>>24;g=J[d>>2];J[h+48>>2]=g;J[d>>2]=0;j=h+48|4;_:{$:{if(e>>>0>=2){n=1;while(1){J[5996]=0;J[h+4>>2]=147;ga(128,h+16|0,h+48|0,(n<<5)+d|0,h+4|0);f=J[5996];J[5996]=0;if((f|0)==1){break $}J[h+104>>2]=K[j+24|0]|K[j+25|0]<<8|(K[j+26|0]<<16|K[j+27|0]<<24);f=K[j+20|0]|K[j+21|0]<<8|(K[j+22|0]<<16|K[j+23|0]<<24);J[h+96>>2]=K[j+16|0]|K[j+17|0]<<8|(K[j+18|0]<<16|K[j+19|0]<<24);J[h+100>>2]=f;f=K[j+12|0]|K[j+13|0]<<8|(K[j+14|0]<<16|K[j+15|0]<<24);J[h+88>>2]=K[j+8|0]|K[j+9|0]<<8|(K[j+10|0]<<16|K[j+11|0]<<24);J[h+92>>2]=f;f=K[j+4|0]|K[j+5|0]<<8|(K[j+6|0]<<16|K[j+7|0]<<24);J[h+80>>2]=K[j|0]|K[j+1|0]<<8|(K[j+2|0]<<16|K[j+3|0]<<24);J[h+84>>2]=f;J[j+24>>2]=J[h+44>>2];f=J[h+40>>2];J[j+16>>2]=J[h+36>>2];J[j+20>>2]=f;f=J[h+32>>2];J[j+8>>2]=J[h+28>>2];J[j+12>>2]=f;f=J[h+24>>2];J[j>>2]=J[h+20>>2];J[j+4>>2]=f;f=J[h+104>>2];H[h+44|0]=f;H[h+45|0]=f>>>8;H[h+46|0]=f>>>16;H[h+47|0]=f>>>24;i=J[h+100>>2];f=J[h+96>>2];H[h+36|0]=f;H[h+37|0]=f>>>8;H[h+38|0]=f>>>16;H[h+39|0]=f>>>24;H[h+40|0]=i;H[h+41|0]=i>>>8;H[h+42|0]=i>>>16;H[h+43|0]=i>>>24;i=J[h+92>>2];f=J[h+88>>2];H[h+28|0]=f;H[h+29|0]=f>>>8;H[h+30|0]=f>>>16;H[h+31|0]=f>>>24;H[h+32|0]=i;H[h+33|0]=i>>>8;H[h+34|0]=i>>>16;H[h+35|0]=i>>>24;i=J[h+84>>2];f=J[h+80>>2];H[h+20|0]=f;H[h+21|0]=f>>>8;H[h+22|0]=f>>>16;H[h+23|0]=f>>>24;H[h+24|0]=i;H[h+25|0]=i>>>8;H[h+26|0]=i>>>16;H[h+27|0]=i>>>24;f=J[h+48>>2];J[h+48>>2]=J[h+16>>2];J[h+16>>2]=f;if(f>>>0>=29){$a(J[h+24>>2]);}n=n+1|0;if((n|0)!=(e|0)){continue}break}g=J[h+48>>2];}H[a|0]=0;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;H[a+4|0]=0;H[a+5|0]=0;H[a+6|0]=0;H[a+7|0]=0;H[a+24|0]=0;H[a+25|0]=0;H[a+26|0]=0;H[a+27|0]=0;H[a+28|0]=0;H[a+29|0]=0;H[a+30|0]=0;H[a+31|0]=0;H[a+16|0]=0;H[a+17|0]=0;H[a+18|0]=0;H[a+19|0]=0;H[a+20|0]=0;H[a+21|0]=0;H[a+22|0]=0;H[a+23|0]=0;H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;H[a+12|0]=0;H[a+13|0]=0;H[a+14|0]=0;H[a+15|0]=0;n=-29;i=J[h+56>>2];d=0;aa:{k=a+4|0;f=g>>>0<29;l=f?g:g-29|0;ba:{if(l>>>0>=29){J[5996]=0;aa(124,a|0,l+(l>>>1|0)|0);d=J[5996];J[5996]=0;if((d|0)==1){break ba}d=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);n=d-29|0;p=K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24);}e=d>>>0<29;g=e?k:p;eb(g+l|0,g,e?d:n);d=l+(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))|0;H[a|0]=d;H[a+1|0]=d>>>8;H[a+2|0]=d>>>16;H[a+3|0]=d>>>24;ca:{if(!l){break ca}e=f?j:i;i=l&7;da:{if(!i){f=e;break da}d=0;f=e;while(1){H[g|0]=K[f|0];f=f+1|0;g=g+1|0;d=d+1|0;if((i|0)!=(d|0)){continue}break}}if(l>>>0<8){break ca}d=e+l|0;while(1){H[g|0]=K[f|0];H[g+1|0]=K[f+1|0];H[g+2|0]=K[f+2|0];H[g+3|0]=K[f+3|0];H[g+4|0]=K[f+4|0];H[g+5|0]=K[f+5|0];H[g+6|0]=K[f+6|0];H[g+7|0]=K[f+7|0];g=g+8|0;f=f+8|0;if((d|0)!=(f|0)){continue}break}}c=J[c+4>>2];J[5996]=0;d=nc(a,c,0);c=J[5996];J[5996]=0;if((c|0)==1){break ba}J[5996]=0;_(127,d|0,(b?136:135)|0)|0;b=J[5996];J[5996]=0;if((b|0)!=1){break aa}}g=Z()|0;if((K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))>>>0<29){break _}$a(K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24));H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;break _}if(M[h+48>>2]<29){break d}$a(J[h+56>>2]);break d}g=Z()|0;}if(M[h+48>>2]<29){break a}$a(J[h+56>>2]);break a;default:la(2350,2210,624,5107);B();case 6:break j}}J[h+48>>2]=32;J[h+16>>2]=136;J[h+80>>2]=168;J[h+4>>2]=b?136:135;vd(a,h+48|0,h+16|0,h+80|0,c+20|0,h+4|0);break d}g=Z()|0;if(M[h+16>>2]>=29){$a(J[h+24>>2]);J[h+24>>2]=0;}a=J[h+4>>2];if(!a){break f}$a(a);break f}e=J[c+8>>2];c=J[c+12>>2];}H[a|0]=0;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;H[a+4|0]=0;H[a+5|0]=0;H[a+6|0]=0;H[a+7|0]=0;H[a+24|0]=0;H[a+25|0]=0;H[a+26|0]=0;H[a+27|0]=0;H[a+28|0]=0;H[a+29|0]=0;H[a+30|0]=0;H[a+31|0]=0;H[a+16|0]=0;H[a+17|0]=0;H[a+18|0]=0;H[a+19|0]=0;H[a+20|0]=0;H[a+21|0]=0;H[a+22|0]=0;H[a+23|0]=0;H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;H[a+12|0]=0;H[a+13|0]=0;H[a+14|0]=0;H[a+15|0]=0;l=J[o+8>>2];d=-29;i=(c-e|0)/12|0;n=0;f=0;c=J[o>>2];e=c>>>0<29;j=e?c:c-29|0;ea:{if(j>>>0>=29){J[5996]=0;aa(124,a|0,j+(j>>>1|0)|0);c=J[5996];J[5996]=0;if((c|0)==1){break ea}n=K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24);f=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);d=f-29|0;}c=f>>>0<29;g=c?a+4|0:n;eb(j+g|0,g,c?f:d);c=j+(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))|0;H[a|0]=c;H[a+1|0]=c>>>8;H[a+2|0]=c>>>16;H[a+3|0]=c>>>24;fa:{if(!j){break fa}c=e?o+4|0:l;e=j&7;ga:{if(!e){f=c;break ga}d=0;f=c;while(1){H[g|0]=K[f|0];f=f+1|0;g=g+1|0;d=d+1|0;if((e|0)!=(d|0)){continue}break}}if(j>>>0<8){break fa}c=c+j|0;while(1){H[g|0]=K[f|0];H[g+1|0]=K[f+1|0];H[g+2|0]=K[f+2|0];H[g+3|0]=K[f+3|0];H[g+4|0]=K[f+4|0];H[g+5|0]=K[f+5|0];H[g+6|0]=K[f+6|0];H[g+7|0]=K[f+7|0];g=g+8|0;f=f+8|0;if((c|0)!=(f|0)){continue}break}}J[5996]=0;d=nc(a,i,0);c=J[5996];J[5996]=0;if((c|0)==1){break ea}J[5996]=0;_(127,d|0,(b?175:174)|0)|0;b=J[5996];J[5996]=0;if((b|0)!=1){break e}}g=Z()|0;if((K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))>>>0<29){break f}$a(K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24));H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;}if(M[o>>2]<29){break a}$a(J[o+8>>2]);break a}if(M[o>>2]<29){break d}$a(J[o+8>>2]);}Ua=h+112|0;return}g=Z()|0;a=J[h+48>>2];if(!a){break a}$a(a);break a}g=Z()|0;if((K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))>>>0<29){break a}$a(K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24));H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;}da(g|0);B();}function Um(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0,h=0,i=0,j=0;c=Ua-496|0;Ua=c;a:{b:{c:{d:{e:{f:{g:{h:{i:{j:{k:{l:{m:{n:{o:{p:{q:{r:{s:{t:{u:{v:{w:{x:{y:{z:{A:{B:{C:{D:{E:{F:{G:{H:{I:{J:{K:{L:{M:{N:{O:{P:{Q:{R:{S:{T:{U:{V:{W:{X:{Y:{e=J[b>>2];switch(J[e>>2]-1|0){case 23:break z;case 22:break A;case 21:break B;case 20:break C;case 19:break D;case 18:break E;case 17:break F;case 16:break G;case 14:break H;case 15:break I;case 13:break J;case 12:break K;case 11:break L;case 10:break M;case 9:break N;case 0:break O;case 8:break Q;case 6:break R;case 7:break S;case 5:break T;case 3:break U;case 4:break V;case 24:break W;case 2:break X;case 1:break Y;default:break P}}e=J[e+8>>2];Z:{if(H[e+11|0]>=0){J[c+456>>2]=J[e+8>>2];d=J[e+4>>2];J[c+448>>2]=J[e>>2];J[c+452>>2]=d;break Z}Zc(c+448|0,J[e>>2],J[e+4>>2]);}J[5996]=0;H[c+460|0]=1;e=ba(99,c+448|0,0,5283)|0;d=J[5996];J[5996]=0;_:{$:{aa:{ba:{if((d|0)!=1){J[c+472>>2]=J[e+8>>2];d=J[e+4>>2];J[c+464>>2]=J[e>>2];J[c+468>>2]=d;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;J[5996]=0;d=_(97,c+464|0,5242)|0;f=J[5996];J[5996]=0;if((f|0)==1){break ba}J[c+488>>2]=J[d+8>>2];f=J[d+4>>2];J[c+480>>2]=J[d>>2];J[c+484>>2]=f;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;ea(161,a|0,b|0,c+480|0);a=J[5996];J[5996]=0;if((a|0)==1){break aa}if(H[c+491|0]<0){$a(J[c+480>>2]);}if(H[d+11|0]<0){$a(J[d>>2]);}if(!K[c+460|0]|H[e+11|0]>=0){break y}$a(J[e>>2]);break y}b=Z()|0;break _}b=Z()|0;break $}b=Z()|0;if(H[c+491|0]>=0){break $}$a(J[c+480>>2]);}if(H[c+475|0]>=0){break _}$a(J[c+464>>2]);}break c}e=J[e+8>>2];ca:{if(H[e+11|0]>=0){J[c+456>>2]=J[e+8>>2];d=J[e+4>>2];J[c+448>>2]=J[e>>2];J[c+452>>2]=d;break ca}Zc(c+448|0,J[e>>2],J[e+4>>2]);}J[5996]=0;H[c+460|0]=1;e=ba(99,c+448|0,0,5315)|0;d=J[5996];J[5996]=0;da:{ea:{fa:{ga:{if((d|0)!=1){J[c+472>>2]=J[e+8>>2];d=J[e+4>>2];J[c+464>>2]=J[e>>2];J[c+468>>2]=d;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;J[5996]=0;d=_(97,c+464|0,5242)|0;f=J[5996];J[5996]=0;if((f|0)==1){break ga}J[c+440>>2]=J[d+8>>2];f=J[d+4>>2];J[c+432>>2]=J[d>>2];J[c+436>>2]=f;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;ea(161,a|0,b|0,c+432|0);a=J[5996];J[5996]=0;if((a|0)==1){break fa}if(H[c+443|0]<0){$a(J[c+432>>2]);}if(H[d+11|0]<0){$a(J[d>>2]);}if(!K[c+460|0]|H[e+11|0]>=0){break y}$a(J[e>>2]);break y}b=Z()|0;break da}b=Z()|0;break ea}b=Z()|0;if(H[c+443|0]>=0){break ea}$a(J[c+432>>2]);}if(H[c+475|0]>=0){break da}$a(J[c+464>>2]);}break c}d=c+384|0;Kc(d,J[e+4>>2]);J[5996]=0;e=ba(99,d|0,0,5289)|0;d=J[5996];J[5996]=0;ha:{ia:{ja:{ka:{la:{ma:{na:{oa:{pa:{qa:{if((d|0)!=1){J[c+408>>2]=J[e+8>>2];d=J[e+4>>2];J[c+400>>2]=J[e>>2];J[c+404>>2]=d;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;J[5996]=0;d=_(97,c+400|0,5417)|0;f=J[5996];J[5996]=0;if((f|0)==1){break qa}J[c+472>>2]=J[d+8>>2];f=J[d+4>>2];J[c+464>>2]=J[d>>2];J[c+468>>2]=f;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;f=J[b>>2];g=J[f+12>>2];f=J[f+8>>2];J[5996]=0;h=c+368|0;aa(115,h|0,(g-f|0)/12|0);f=J[5996];J[5996]=0;if((f|0)==1){break pa}J[5996]=0;f=K[c+379|0];g=f<<24>>24<0;f=ba(100,c+464|0,(g?J[c+368>>2]:h)|0,(g?J[c+372>>2]:f)|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break oa}J[c+456>>2]=J[f+8>>2];g=J[f+4>>2];J[c+448>>2]=J[f>>2];J[c+452>>2]=g;J[f>>2]=0;J[f+4>>2]=0;J[f+8>>2]=0;J[5996]=0;g=_(97,c+448|0,5242)|0;h=J[5996];J[5996]=0;if((h|0)==1){break na}J[c+424>>2]=J[g+8>>2];h=J[g+4>>2];J[c+416>>2]=J[g>>2];J[c+420>>2]=h;J[g>>2]=0;J[g+4>>2]=0;J[g+8>>2]=0;J[5996]=0;ea(161,a|0,b|0,c+416|0);a=J[5996];J[5996]=0;if((a|0)==1){break ma}if(H[c+427|0]<0){$a(J[c+416>>2]);}if(H[g+11|0]<0){$a(J[g>>2]);}if(H[c+379|0]<0){$a(J[c+368>>2]);}if(H[f+11|0]<0){$a(J[f>>2]);}if(H[d+11|0]<0){$a(J[d>>2]);}if(H[e+11|0]>=0){break y}$a(J[e>>2]);break y}b=Z()|0;break ha}b=Z()|0;break ia}b=Z()|0;break ja}b=Z()|0;break ka}b=Z()|0;break la}b=Z()|0;if(H[c+427|0]>=0){break la}$a(J[c+416>>2]);}if(H[c+459|0]>=0){break ka}$a(J[c+448>>2]);}if(H[c+379|0]>=0){break ja}$a(J[c+368>>2]);}if(H[c+475|0]>=0){break ia}$a(J[c+464>>2]);}if(H[c+411|0]>=0){break ha}$a(J[c+400>>2]);}if(H[c+395|0]>=0){break a}$a(J[c+384>>2]);break a}d=c+464|0;Kc(d,J[e+4>>2]);J[5996]=0;e=ba(99,d|0,0,5258)|0;d=J[5996];J[5996]=0;ra:{sa:{ta:{ua:{if((d|0)!=1){J[c+456>>2]=J[e+8>>2];d=J[e+4>>2];J[c+448>>2]=J[e>>2];J[c+452>>2]=d;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;J[5996]=0;d=_(97,c+448|0,5242)|0;f=J[5996];J[5996]=0;if((f|0)==1){break ua}J[c+360>>2]=J[d+8>>2];f=J[d+4>>2];J[c+352>>2]=J[d>>2];J[c+356>>2]=f;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;ea(161,a|0,b|0,c+352|0);a=J[5996];J[5996]=0;if((a|0)==1){break ta}if(H[c+363|0]<0){$a(J[c+352>>2]);}if(H[d+11|0]<0){$a(J[d>>2]);}if(H[e+11|0]>=0){break y}$a(J[e>>2]);break y}b=Z()|0;break ra}b=Z()|0;break sa}b=Z()|0;if(H[c+363|0]>=0){break sa}$a(J[c+352>>2]);}if(H[c+459|0]>=0){break ra}$a(J[c+448>>2]);}break b}d=c+464|0;Kc(d,J[e+4>>2]);J[5996]=0;e=ba(99,d|0,0,5265)|0;d=J[5996];J[5996]=0;va:{wa:{xa:{ya:{if((d|0)!=1){J[c+456>>2]=J[e+8>>2];d=J[e+4>>2];J[c+448>>2]=J[e>>2];J[c+452>>2]=d;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;J[5996]=0;d=_(97,c+448|0,5242)|0;f=J[5996];J[5996]=0;if((f|0)==1){break ya}J[c+344>>2]=J[d+8>>2];f=J[d+4>>2];J[c+336>>2]=J[d>>2];J[c+340>>2]=f;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;ea(161,a|0,b|0,c+336|0);a=J[5996];J[5996]=0;if((a|0)==1){break xa}if(H[c+347|0]<0){$a(J[c+336>>2]);}if(H[d+11|0]<0){$a(J[d>>2]);}if(H[e+11|0]>=0){break y}$a(J[e>>2]);break y}b=Z()|0;break va}b=Z()|0;break wa}b=Z()|0;if(H[c+347|0]>=0){break wa}$a(J[c+336>>2]);}if(H[c+459|0]>=0){break va}$a(J[c+448>>2]);}break b}H[c+328|0]=0;J[c+320>>2]=845244531;J[c+324>>2]=690501173;H[c+331|0]=8;J[5996]=0;ea(161,a|0,b|0,c+320|0);a=J[5996];J[5996]=0;if((a|0)!=1){if(H[c+331|0]>=0){break y}$a(J[c+320>>2]);break y}b=Z()|0;if(H[c+331|0]>=0){break a}$a(J[c+320>>2]);break a}e=cb(16);J[c+308>>2]=e;J[c+312>>2]=11;J[c+316>>2]=-2147483632;d=K[5239]|K[5240]<<8|(K[5241]<<16|K[5242]<<24);H[e+7|0]=d;H[e+8|0]=d>>>8;H[e+9|0]=d>>>16;H[e+10|0]=d>>>24;d=K[5236]|K[5237]<<8|(K[5238]<<16|K[5239]<<24);f=K[5232]|K[5233]<<8|(K[5234]<<16|K[5235]<<24);H[e|0]=f;H[e+1|0]=f>>>8;H[e+2|0]=f>>>16;H[e+3|0]=f>>>24;H[e+4|0]=d;H[e+5|0]=d>>>8;H[e+6|0]=d>>>16;H[e+7|0]=d>>>24;H[e+11|0]=0;J[5996]=0;ea(161,a|0,b|0,c+308|0);a=J[5996];J[5996]=0;if((a|0)!=1){if(H[c+319|0]>=0){break y}$a(J[c+308>>2]);break y}b=Z()|0;if(H[c+319|0]>=0){break a}$a(J[c+308>>2]);break a}H[c+304|0]=K[5220];H[c+307|0]=9;H[c+305|0]=0;J[5996]=0;e=K[5216]|K[5217]<<8|(K[5218]<<16|K[5219]<<24);J[c+296>>2]=K[5212]|K[5213]<<8|(K[5214]<<16|K[5215]<<24);J[c+300>>2]=e;ea(161,a|0,b|0,c+296|0);a=J[5996];J[5996]=0;if((a|0)!=1){if(H[c+307|0]>=0){break y}$a(J[c+296>>2]);break y}b=Z()|0;if(H[c+307|0]>=0){break a}$a(J[c+296>>2]);break a}H[c+288|0]=K[5230];H[c+291|0]=9;H[c+289|0]=0;J[5996]=0;e=K[5226]|K[5227]<<8|(K[5228]<<16|K[5229]<<24);J[c+280>>2]=K[5222]|K[5223]<<8|(K[5224]<<16|K[5225]<<24);J[c+284>>2]=e;ea(161,a|0,b|0,c+280|0);a=J[5996];J[5996]=0;if((a|0)!=1){if(H[c+291|0]>=0){break y}$a(J[c+280>>2]);break y}b=Z()|0;if(H[c+291|0]>=0){break a}$a(J[c+280>>2]);break a}H[c+272|0]=K[2354];H[c+279|0]=5;H[c+273|0]=0;J[5996]=0;J[c+268>>2]=K[2350]|K[2351]<<8|(K[2352]<<16|K[2353]<<24);ea(161,a|0,b|0,c+268|0);a=J[5996];J[5996]=0;if((a|0)!=1){if(H[c+279|0]>=0){break y}$a(J[c+268>>2]);break y}b=Z()|0;if(H[c+279|0]>=0){break a}$a(J[c+268>>2]);break a}H[c+260|0]=0;J[c+256>>2]=1702195828;H[c+267|0]=4;J[5996]=0;ea(161,a|0,b|0,c+256|0);a=J[5996];J[5996]=0;if((a|0)!=1){if(H[c+267|0]>=0){break y}$a(J[c+256>>2]);break y}b=Z()|0;if(H[c+267|0]>=0){break a}$a(J[c+256>>2]);break a}H[c+255|0]=2;H[c+246|0]=0;I[c+244>>1]=14945;J[5996]=0;e=c+464|0;ea(161,e|0,b|0,c+244|0);d=J[5996];J[5996]=0;za:{Aa:{Ba:{Ca:{Da:{Ea:{if((d|0)!=1){J[5996]=0;e=_(97,e|0,5574)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Ea}J[c+456>>2]=J[e+8>>2];d=J[e+4>>2];J[c+448>>2]=J[e>>2];J[c+452>>2]=d;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;b=J[J[b>>2]+32>>2];J[5996]=0;d=c+400|0;aa(141,d|0,b|0);b=J[5996];J[5996]=0;if((b|0)==1){break Da}J[5996]=0;f=d;b=K[c+411|0];d=b<<24>>24<0;b=ba(100,c+448|0,(d?J[c+400>>2]:f)|0,(d?J[c+404>>2]:b)|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Ca}d=J[b+4>>2];J[a>>2]=J[b>>2];J[a+4>>2]=d;J[a+8>>2]=J[b+8>>2];J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;Fa:{if(H[c+411|0]>=0){break Fa}$a(J[c+400>>2]);if(H[b+11|0]>=0){break Fa}$a(J[b>>2]);}if(H[e+11|0]<0){$a(J[e>>2]);}if(H[c+255|0]>=0){break y}$a(J[c+244>>2]);break y}b=Z()|0;break za}b=Z()|0;break Aa}b=Z()|0;break Ba}b=Z()|0;if(H[c+411|0]>=0){break Ba}$a(J[c+400>>2]);}if(H[c+459|0]>=0){break Aa}$a(J[c+448>>2]);}if(H[c+475|0]>=0){break za}$a(J[c+464>>2]);}if(H[c+255|0]>=0){break a}$a(J[c+244>>2]);break a}H[c+243|0]=2;H[c+234|0]=0;I[c+232>>1]=14963;J[5996]=0;e=c+464|0;ea(161,e|0,b|0,c+232|0);d=J[5996];J[5996]=0;Ga:{Ha:{Ia:{Ja:{Ka:{La:{if((d|0)!=1){J[5996]=0;e=_(97,e|0,5574)|0;d=J[5996];J[5996]=0;if((d|0)==1){break La}J[c+456>>2]=J[e+8>>2];d=J[e+4>>2];J[c+448>>2]=J[e>>2];J[c+452>>2]=d;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;b=J[J[b>>2]+32>>2];J[5996]=0;d=c+400|0;aa(141,d|0,b|0);b=J[5996];J[5996]=0;if((b|0)==1){break Ka}J[5996]=0;f=d;b=K[c+411|0];d=b<<24>>24<0;b=ba(100,c+448|0,(d?J[c+400>>2]:f)|0,(d?J[c+404>>2]:b)|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Ja}d=J[b+4>>2];J[a>>2]=J[b>>2];J[a+4>>2]=d;J[a+8>>2]=J[b+8>>2];J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;Ma:{if(H[c+411|0]>=0){break Ma}$a(J[c+400>>2]);if(H[b+11|0]>=0){break Ma}$a(J[b>>2]);}if(H[e+11|0]<0){$a(J[e>>2]);}if(H[c+243|0]>=0){break y}$a(J[c+232>>2]);break y}b=Z()|0;break Ga}b=Z()|0;break Ha}b=Z()|0;break Ia}b=Z()|0;if(H[c+411|0]>=0){break Ia}$a(J[c+400>>2]);}if(H[c+459|0]>=0){break Ha}$a(J[c+448>>2]);}if(H[c+475|0]>=0){break Ga}$a(J[c+464>>2]);}if(H[c+243|0]>=0){break a}$a(J[c+232>>2]);break a}H[c+231|0]=2;H[c+222|0]=0;I[c+220>>1]=14947;J[5996]=0;e=c+464|0;ea(161,e|0,b|0,c+220|0);d=J[5996];J[5996]=0;Na:{Oa:{Pa:{Qa:{Ra:{Sa:{if((d|0)!=1){J[5996]=0;e=_(97,e|0,5574)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Sa}J[c+456>>2]=J[e+8>>2];d=J[e+4>>2];J[c+448>>2]=J[e>>2];J[c+452>>2]=d;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;b=J[J[b>>2]+32>>2];J[5996]=0;d=c+400|0;aa(141,d|0,b|0);b=J[5996];J[5996]=0;if((b|0)==1){break Ra}J[5996]=0;f=d;b=K[c+411|0];d=b<<24>>24<0;b=ba(100,c+448|0,(d?J[c+400>>2]:f)|0,(d?J[c+404>>2]:b)|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Qa}d=J[b+4>>2];J[a>>2]=J[b>>2];J[a+4>>2]=d;J[a+8>>2]=J[b+8>>2];J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;Ta:{if(H[c+411|0]>=0){break Ta}$a(J[c+400>>2]);if(H[b+11|0]>=0){break Ta}$a(J[b>>2]);}if(H[e+11|0]<0){$a(J[e>>2]);}if(H[c+231|0]>=0){break y}$a(J[c+220>>2]);break y}b=Z()|0;break Na}b=Z()|0;break Oa}b=Z()|0;break Pa}b=Z()|0;if(H[c+411|0]>=0){break Pa}$a(J[c+400>>2]);}if(H[c+459|0]>=0){break Oa}$a(J[c+448>>2]);}if(H[c+475|0]>=0){break Na}$a(J[c+464>>2]);}if(H[c+231|0]>=0){break a}$a(J[c+220>>2]);break a}H[c+219|0]=2;H[c+210|0]=0;I[c+208>>1]=14948;J[5996]=0;e=c+464|0;ea(161,e|0,b|0,c+208|0);d=J[5996];J[5996]=0;Ua:{Va:{Wa:{Xa:{Ya:{Za:{if((d|0)!=1){J[5996]=0;e=_(97,e|0,5574)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Za}J[c+456>>2]=J[e+8>>2];d=J[e+4>>2];J[c+448>>2]=J[e>>2];J[c+452>>2]=d;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;b=J[J[b>>2]+32>>2];J[5996]=0;d=c+400|0;aa(141,d|0,b|0);b=J[5996];J[5996]=0;if((b|0)==1){break Ya}J[5996]=0;f=d;b=K[c+411|0];d=b<<24>>24<0;b=ba(100,c+448|0,(d?J[c+400>>2]:f)|0,(d?J[c+404>>2]:b)|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Xa}d=J[b+4>>2];J[a>>2]=J[b>>2];J[a+4>>2]=d;J[a+8>>2]=J[b+8>>2];J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;_a:{if(H[c+411|0]>=0){break _a}$a(J[c+400>>2]);if(H[b+11|0]>=0){break _a}$a(J[b>>2]);}if(H[e+11|0]<0){$a(J[e>>2]);}if(H[c+219|0]>=0){break y}$a(J[c+208>>2]);break y}b=Z()|0;break Ua}b=Z()|0;break Va}b=Z()|0;break Wa}b=Z()|0;if(H[c+411|0]>=0){break Wa}$a(J[c+400>>2]);}if(H[c+459|0]>=0){break Va}$a(J[c+448>>2]);}if(H[c+475|0]>=0){break Ua}$a(J[c+464>>2]);}if(H[c+219|0]>=0){break a}$a(J[c+208>>2]);break a}H[c+207|0]=2;H[c+198|0]=0;I[c+196>>1]=14966;J[5996]=0;e=c+464|0;ea(161,e|0,b|0,c+196|0);d=J[5996];J[5996]=0;$a:{ab:{bb:{cb:{db:{eb:{if((d|0)!=1){J[5996]=0;e=_(97,e|0,5574)|0;d=J[5996];J[5996]=0;if((d|0)==1){break eb}J[c+456>>2]=J[e+8>>2];d=J[e+4>>2];J[c+448>>2]=J[e>>2];J[c+452>>2]=d;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;b=J[J[b>>2]+32>>2];J[5996]=0;d=c+400|0;aa(141,d|0,b|0);b=J[5996];J[5996]=0;if((b|0)==1){break db}J[5996]=0;f=d;b=K[c+411|0];d=b<<24>>24<0;b=ba(100,c+448|0,(d?J[c+400>>2]:f)|0,(d?J[c+404>>2]:b)|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break cb}d=J[b+4>>2];J[a>>2]=J[b>>2];J[a+4>>2]=d;J[a+8>>2]=J[b+8>>2];J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;fb:{if(H[c+411|0]>=0){break fb}$a(J[c+400>>2]);if(H[b+11|0]>=0){break fb}$a(J[b>>2]);}if(H[e+11|0]<0){$a(J[e>>2]);}if(H[c+207|0]>=0){break y}$a(J[c+196>>2]);break y}b=Z()|0;break $a}b=Z()|0;break ab}b=Z()|0;break bb}b=Z()|0;if(H[c+411|0]>=0){break bb}$a(J[c+400>>2]);}if(H[c+459|0]>=0){break ab}$a(J[c+448>>2]);}if(H[c+475|0]>=0){break $a}$a(J[c+464>>2]);}if(H[c+207|0]>=0){break a}$a(J[c+196>>2]);break a}H[c+195|0]=2;H[c+186|0]=0;I[c+184>>1]=14958;J[5996]=0;e=c+464|0;ea(161,e|0,b|0,c+184|0);d=J[5996];J[5996]=0;gb:{hb:{ib:{jb:{kb:{lb:{if((d|0)!=1){J[5996]=0;e=_(97,e|0,5574)|0;d=J[5996];J[5996]=0;if((d|0)==1){break lb}J[c+456>>2]=J[e+8>>2];d=J[e+4>>2];J[c+448>>2]=J[e>>2];J[c+452>>2]=d;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;b=J[J[b>>2]+32>>2];J[5996]=0;d=c+400|0;aa(141,d|0,b|0);b=J[5996];J[5996]=0;if((b|0)==1){break kb}J[5996]=0;f=d;b=K[c+411|0];d=b<<24>>24<0;b=ba(100,c+448|0,(d?J[c+400>>2]:f)|0,(d?J[c+404>>2]:b)|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break jb}d=J[b+4>>2];J[a>>2]=J[b>>2];J[a+4>>2]=d;J[a+8>>2]=J[b+8>>2];J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;mb:{if(H[c+411|0]>=0){break mb}$a(J[c+400>>2]);if(H[b+11|0]>=0){break mb}$a(J[b>>2]);}if(H[e+11|0]<0){$a(J[e>>2]);}if(H[c+195|0]>=0){break y}$a(J[c+184>>2]);break y}b=Z()|0;break gb}b=Z()|0;break hb}b=Z()|0;break ib}b=Z()|0;if(H[c+411|0]>=0){break ib}$a(J[c+400>>2]);}if(H[c+459|0]>=0){break hb}$a(J[c+448>>2]);}if(H[c+475|0]>=0){break gb}$a(J[c+464>>2]);}if(H[c+195|0]>=0){break a}$a(J[c+184>>2]);break a}H[c+183|0]=2;H[c+174|0]=0;I[c+172>>1]=14954;J[5996]=0;e=c+464|0;ea(161,e|0,b|0,c+172|0);d=J[5996];J[5996]=0;nb:{ob:{pb:{qb:{rb:{sb:{if((d|0)!=1){J[5996]=0;e=_(97,e|0,5574)|0;d=J[5996];J[5996]=0;if((d|0)==1){break sb}J[c+456>>2]=J[e+8>>2];d=J[e+4>>2];J[c+448>>2]=J[e>>2];J[c+452>>2]=d;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;b=J[J[b>>2]+32>>2];J[5996]=0;d=c+400|0;aa(141,d|0,b|0);b=J[5996];J[5996]=0;if((b|0)==1){break rb}J[5996]=0;f=d;b=K[c+411|0];d=b<<24>>24<0;b=ba(100,c+448|0,(d?J[c+400>>2]:f)|0,(d?J[c+404>>2]:b)|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break qb}d=J[b+4>>2];J[a>>2]=J[b>>2];J[a+4>>2]=d;J[a+8>>2]=J[b+8>>2];J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;tb:{if(H[c+411|0]>=0){break tb}$a(J[c+400>>2]);if(H[b+11|0]>=0){break tb}$a(J[b>>2]);}if(H[e+11|0]<0){$a(J[e>>2]);}if(H[c+183|0]>=0){break y}$a(J[c+172>>2]);break y}b=Z()|0;break nb}b=Z()|0;break ob}b=Z()|0;break pb}b=Z()|0;if(H[c+411|0]>=0){break pb}$a(J[c+400>>2]);}if(H[c+459|0]>=0){break ob}$a(J[c+448>>2]);}if(H[c+475|0]>=0){break nb}$a(J[c+464>>2]);}if(H[c+183|0]>=0){break a}$a(J[c+172>>2]);break a}H[c+164|0]=K[1271];H[c+171|0]=5;H[c+165|0]=0;J[5996]=0;J[c+160>>2]=K[1267]|K[1268]<<8|(K[1269]<<16|K[1270]<<24);e=c+368|0;ea(161,e|0,b|0,c+160|0);d=J[5996];J[5996]=0;ub:{vb:{wb:{xb:{yb:{zb:{Ab:{Bb:{Cb:{Db:{Eb:{Fb:{Gb:{Hb:{if((d|0)!=1){J[5996]=0;e=_(97,e|0,3673)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Hb}J[c+392>>2]=J[e+8>>2];d=J[e+4>>2];J[c+384>>2]=J[e>>2];J[c+388>>2]=d;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;d=J[J[b>>2]+32>>2];J[5996]=0;f=c+144|0;aa(141,f|0,d|0);d=J[5996];J[5996]=0;if((d|0)==1){break Gb}J[5996]=0;g=f;d=K[c+155|0];f=d<<24>>24<0;d=ba(100,c+384|0,(f?J[c+144>>2]:g)|0,(f?J[c+148>>2]:d)|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){break Fb}J[c+408>>2]=J[d+8>>2];f=J[d+4>>2];J[c+400>>2]=J[d>>2];J[c+404>>2]=f;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;f=_(97,c+400|0,3613)|0;g=J[5996];J[5996]=0;if((g|0)==1){break Eb}J[c+472>>2]=J[f+8>>2];g=J[f+4>>2];J[c+464>>2]=J[f>>2];J[c+468>>2]=g;J[f>>2]=0;J[f+4>>2]=0;J[f+8>>2]=0;b=J[J[b>>2]+32>>2];J[5996]=0;g=c+128|0;aa(141,g|0,b+8|0);b=J[5996];J[5996]=0;if((b|0)==1){break Db}J[5996]=0;h=g;b=K[c+139|0];g=b<<24>>24<0;g=ba(100,c+464|0,(g?J[c+128>>2]:h)|0,(g?J[c+132>>2]:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break Cb}J[c+456>>2]=J[g+8>>2];b=J[g+4>>2];J[c+448>>2]=J[g>>2];J[c+452>>2]=b;J[g>>2]=0;J[g+4>>2]=0;J[g+8>>2]=0;J[5996]=0;b=_(97,c+448|0,3570)|0;h=J[5996];J[5996]=0;if((h|0)==1){break Bb}h=J[b+4>>2];J[a>>2]=J[b>>2];J[a+4>>2]=h;J[a+8>>2]=J[b+8>>2];J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;if(H[c+139|0]<0){$a(J[c+128>>2]);}if(H[g+11|0]<0){$a(J[g>>2]);}if(H[f+11|0]<0){$a(J[f>>2]);}if(H[c+155|0]<0){$a(J[c+144>>2]);}if(H[d+11|0]<0){$a(J[d>>2]);}if(H[e+11|0]<0){$a(J[e>>2]);}if(H[c+171|0]>=0){break y}$a(J[c+160>>2]);break y}b=Z()|0;break ub}b=Z()|0;break vb}b=Z()|0;break wb}b=Z()|0;break xb}b=Z()|0;break yb}b=Z()|0;break zb}b=Z()|0;break Ab}b=Z()|0;if(H[c+459|0]>=0){break Ab}$a(J[c+448>>2]);}if(H[c+139|0]>=0){break zb}$a(J[c+128>>2]);}if(H[c+475|0]>=0){break yb}$a(J[c+464>>2]);}if(H[c+411|0]>=0){break xb}$a(J[c+400>>2]);}if(H[c+155|0]>=0){break wb}$a(J[c+144>>2]);}if(H[c+395|0]>=0){break vb}$a(J[c+384>>2]);}if(H[c+379|0]>=0){break ub}$a(J[c+368>>2]);}if(H[c+171|0]>=0){break a}$a(J[c+160>>2]);break a}H[c+120|0]=K[2551];H[c+127|0]=5;H[c+121|0]=0;J[5996]=0;J[c+116>>2]=K[2547]|K[2548]<<8|(K[2549]<<16|K[2550]<<24);e=c+368|0;ea(161,e|0,b|0,c+116|0);d=J[5996];J[5996]=0;Ib:{Jb:{Kb:{Lb:{Mb:{Nb:{Ob:{Pb:{Qb:{Rb:{Sb:{Tb:{Ub:{Vb:{if((d|0)!=1){J[5996]=0;e=_(97,e|0,3673)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Vb}J[c+392>>2]=J[e+8>>2];d=J[e+4>>2];J[c+384>>2]=J[e>>2];J[c+388>>2]=d;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;d=J[J[b>>2]+32>>2];J[5996]=0;f=c+144|0;aa(141,f|0,d|0);d=J[5996];J[5996]=0;if((d|0)==1){break Ub}J[5996]=0;g=f;d=K[c+155|0];f=d<<24>>24<0;d=ba(100,c+384|0,(f?J[c+144>>2]:g)|0,(f?J[c+148>>2]:d)|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){break Tb}J[c+408>>2]=J[d+8>>2];f=J[d+4>>2];J[c+400>>2]=J[d>>2];J[c+404>>2]=f;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;f=_(97,c+400|0,3613)|0;g=J[5996];J[5996]=0;if((g|0)==1){break Sb}J[c+472>>2]=J[f+8>>2];g=J[f+4>>2];J[c+464>>2]=J[f>>2];J[c+468>>2]=g;J[f>>2]=0;J[f+4>>2]=0;J[f+8>>2]=0;b=J[J[b>>2]+32>>2];J[5996]=0;g=c+128|0;aa(141,g|0,b+8|0);b=J[5996];J[5996]=0;if((b|0)==1){break Rb}J[5996]=0;h=g;b=K[c+139|0];g=b<<24>>24<0;g=ba(100,c+464|0,(g?J[c+128>>2]:h)|0,(g?J[c+132>>2]:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break Qb}J[c+456>>2]=J[g+8>>2];b=J[g+4>>2];J[c+448>>2]=J[g>>2];J[c+452>>2]=b;J[g>>2]=0;J[g+4>>2]=0;J[g+8>>2]=0;J[5996]=0;b=_(97,c+448|0,3570)|0;h=J[5996];J[5996]=0;if((h|0)==1){break Pb}h=J[b+4>>2];J[a>>2]=J[b>>2];J[a+4>>2]=h;J[a+8>>2]=J[b+8>>2];J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;if(H[c+139|0]<0){$a(J[c+128>>2]);}if(H[g+11|0]<0){$a(J[g>>2]);}if(H[f+11|0]<0){$a(J[f>>2]);}if(H[c+155|0]<0){$a(J[c+144>>2]);}if(H[d+11|0]<0){$a(J[d>>2]);}if(H[e+11|0]<0){$a(J[e>>2]);}if(H[c+127|0]>=0){break y}$a(J[c+116>>2]);break y}b=Z()|0;break Ib}b=Z()|0;break Jb}b=Z()|0;break Kb}b=Z()|0;break Lb}b=Z()|0;break Mb}b=Z()|0;break Nb}b=Z()|0;break Ob}b=Z()|0;if(H[c+459|0]>=0){break Ob}$a(J[c+448>>2]);}if(H[c+139|0]>=0){break Nb}$a(J[c+128>>2]);}if(H[c+475|0]>=0){break Mb}$a(J[c+464>>2]);}if(H[c+411|0]>=0){break Lb}$a(J[c+400>>2]);}if(H[c+155|0]>=0){break Kb}$a(J[c+144>>2]);}if(H[c+395|0]>=0){break Jb}$a(J[c+384>>2]);}if(H[c+379|0]>=0){break Ib}$a(J[c+368>>2]);}if(H[c+127|0]>=0){break a}$a(J[c+116>>2]);break a}H[c+115|0]=4;H[c+108|0]=0;J[c+104>>2]=1650422383;J[5996]=0;e=c+368|0;ea(161,e|0,b|0,c+104|0);d=J[5996];J[5996]=0;Wb:{Xb:{Yb:{Zb:{_b:{$b:{ac:{bc:{cc:{dc:{ec:{fc:{gc:{hc:{if((d|0)!=1){J[5996]=0;e=_(97,e|0,3673)|0;d=J[5996];J[5996]=0;if((d|0)==1){break hc}J[c+392>>2]=J[e+8>>2];d=J[e+4>>2];J[c+384>>2]=J[e>>2];J[c+388>>2]=d;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;d=J[J[b>>2]+32>>2];J[5996]=0;f=c+144|0;aa(141,f|0,d|0);d=J[5996];J[5996]=0;if((d|0)==1){break gc}J[5996]=0;g=f;d=K[c+155|0];f=d<<24>>24<0;d=ba(100,c+384|0,(f?J[c+144>>2]:g)|0,(f?J[c+148>>2]:d)|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){break fc}J[c+408>>2]=J[d+8>>2];f=J[d+4>>2];J[c+400>>2]=J[d>>2];J[c+404>>2]=f;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;f=_(97,c+400|0,3613)|0;g=J[5996];J[5996]=0;if((g|0)==1){break ec}J[c+472>>2]=J[f+8>>2];g=J[f+4>>2];J[c+464>>2]=J[f>>2];J[c+468>>2]=g;J[f>>2]=0;J[f+4>>2]=0;J[f+8>>2]=0;b=J[J[b>>2]+32>>2];J[5996]=0;g=c+128|0;aa(141,g|0,b+8|0);b=J[5996];J[5996]=0;if((b|0)==1){break dc}J[5996]=0;h=g;b=K[c+139|0];g=b<<24>>24<0;g=ba(100,c+464|0,(g?J[c+128>>2]:h)|0,(g?J[c+132>>2]:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break cc}J[c+456>>2]=J[g+8>>2];b=J[g+4>>2];J[c+448>>2]=J[g>>2];J[c+452>>2]=b;J[g>>2]=0;J[g+4>>2]=0;J[g+8>>2]=0;J[5996]=0;b=_(97,c+448|0,3570)|0;h=J[5996];J[5996]=0;if((h|0)==1){break bc}h=J[b+4>>2];J[a>>2]=J[b>>2];J[a+4>>2]=h;J[a+8>>2]=J[b+8>>2];J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;if(H[c+139|0]<0){$a(J[c+128>>2]);}if(H[g+11|0]<0){$a(J[g>>2]);}if(H[f+11|0]<0){$a(J[f>>2]);}if(H[c+155|0]<0){$a(J[c+144>>2]);}if(H[d+11|0]<0){$a(J[d>>2]);}if(H[e+11|0]<0){$a(J[e>>2]);}if(H[c+115|0]>=0){break y}$a(J[c+104>>2]);break y}b=Z()|0;break Wb}b=Z()|0;break Xb}b=Z()|0;break Yb}b=Z()|0;break Zb}b=Z()|0;break _b}b=Z()|0;break $b}b=Z()|0;break ac}b=Z()|0;if(H[c+459|0]>=0){break ac}$a(J[c+448>>2]);}if(H[c+139|0]>=0){break $b}$a(J[c+128>>2]);}if(H[c+475|0]>=0){break _b}$a(J[c+464>>2]);}if(H[c+411|0]>=0){break Zb}$a(J[c+400>>2]);}if(H[c+155|0]>=0){break Yb}$a(J[c+144>>2]);}if(H[c+395|0]>=0){break Xb}$a(J[c+384>>2]);}if(H[c+379|0]>=0){break Wb}$a(J[c+368>>2]);}if(H[c+115|0]>=0){break a}$a(J[c+104>>2]);break a}H[c+103|0]=4;H[c+96|0]=0;J[c+92>>2]=1667199599;J[5996]=0;e=c+368|0;ea(161,e|0,b|0,c+92|0);d=J[5996];J[5996]=0;ic:{jc:{kc:{lc:{mc:{nc:{oc:{pc:{qc:{rc:{sc:{tc:{uc:{vc:{if((d|0)!=1){J[5996]=0;e=_(97,e|0,3673)|0;d=J[5996];J[5996]=0;if((d|0)==1){break vc}J[c+392>>2]=J[e+8>>2];d=J[e+4>>2];J[c+384>>2]=J[e>>2];J[c+388>>2]=d;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;d=J[J[b>>2]+32>>2];J[5996]=0;f=c+144|0;aa(141,f|0,d|0);d=J[5996];J[5996]=0;if((d|0)==1){break uc}J[5996]=0;g=f;d=K[c+155|0];f=d<<24>>24<0;d=ba(100,c+384|0,(f?J[c+144>>2]:g)|0,(f?J[c+148>>2]:d)|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){break tc}J[c+408>>2]=J[d+8>>2];f=J[d+4>>2];J[c+400>>2]=J[d>>2];J[c+404>>2]=f;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;f=_(97,c+400|0,3613)|0;g=J[5996];J[5996]=0;if((g|0)==1){break sc}J[c+472>>2]=J[f+8>>2];g=J[f+4>>2];J[c+464>>2]=J[f>>2];J[c+468>>2]=g;J[f>>2]=0;J[f+4>>2]=0;J[f+8>>2]=0;b=J[J[b>>2]+32>>2];J[5996]=0;g=c+128|0;aa(141,g|0,b+8|0);b=J[5996];J[5996]=0;if((b|0)==1){break rc}J[5996]=0;h=g;b=K[c+139|0];g=b<<24>>24<0;g=ba(100,c+464|0,(g?J[c+128>>2]:h)|0,(g?J[c+132>>2]:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break qc}J[c+456>>2]=J[g+8>>2];b=J[g+4>>2];J[c+448>>2]=J[g>>2];J[c+452>>2]=b;J[g>>2]=0;J[g+4>>2]=0;J[g+8>>2]=0;J[5996]=0;b=_(97,c+448|0,3570)|0;h=J[5996];J[5996]=0;if((h|0)==1){break pc}h=J[b+4>>2];J[a>>2]=J[b>>2];J[a+4>>2]=h;J[a+8>>2]=J[b+8>>2];J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;if(H[c+139|0]<0){$a(J[c+128>>2]);}if(H[g+11|0]<0){$a(J[g>>2]);}if(H[f+11|0]<0){$a(J[f>>2]);}if(H[c+155|0]<0){$a(J[c+144>>2]);}if(H[d+11|0]<0){$a(J[d>>2]);}if(H[e+11|0]<0){$a(J[e>>2]);}if(H[c+103|0]>=0){break y}$a(J[c+92>>2]);break y}b=Z()|0;break ic}b=Z()|0;break jc}b=Z()|0;break kc}b=Z()|0;break lc}b=Z()|0;break mc}b=Z()|0;break nc}b=Z()|0;break oc}b=Z()|0;if(H[c+459|0]>=0){break oc}$a(J[c+448>>2]);}if(H[c+139|0]>=0){break nc}$a(J[c+128>>2]);}if(H[c+475|0]>=0){break mc}$a(J[c+464>>2]);}if(H[c+411|0]>=0){break lc}$a(J[c+400>>2]);}if(H[c+155|0]>=0){break kc}$a(J[c+144>>2]);}if(H[c+395|0]>=0){break jc}$a(J[c+384>>2]);}if(H[c+379|0]>=0){break ic}$a(J[c+368>>2]);}if(H[c+103|0]>=0){break a}$a(J[c+92>>2]);break a}H[c+91|0]=4;H[c+84|0]=0;J[c+80>>2]=1683976815;J[5996]=0;e=c+368|0;ea(161,e|0,b|0,c+80|0);d=J[5996];J[5996]=0;wc:{xc:{yc:{zc:{Ac:{Bc:{Cc:{Dc:{Ec:{Fc:{Gc:{Hc:{Ic:{Jc:{if((d|0)!=1){J[5996]=0;e=_(97,e|0,3673)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Jc}J[c+392>>2]=J[e+8>>2];d=J[e+4>>2];J[c+384>>2]=J[e>>2];J[c+388>>2]=d;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;d=J[J[b>>2]+32>>2];J[5996]=0;f=c+144|0;aa(141,f|0,d|0);d=J[5996];J[5996]=0;if((d|0)==1){break Ic}J[5996]=0;g=f;d=K[c+155|0];f=d<<24>>24<0;d=ba(100,c+384|0,(f?J[c+144>>2]:g)|0,(f?J[c+148>>2]:d)|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){break Hc}J[c+408>>2]=J[d+8>>2];f=J[d+4>>2];J[c+400>>2]=J[d>>2];J[c+404>>2]=f;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;f=_(97,c+400|0,3613)|0;g=J[5996];J[5996]=0;if((g|0)==1){break Gc}J[c+472>>2]=J[f+8>>2];g=J[f+4>>2];J[c+464>>2]=J[f>>2];J[c+468>>2]=g;J[f>>2]=0;J[f+4>>2]=0;J[f+8>>2]=0;b=J[J[b>>2]+32>>2];J[5996]=0;g=c+128|0;aa(141,g|0,b+8|0);b=J[5996];J[5996]=0;if((b|0)==1){break Fc}J[5996]=0;h=g;b=K[c+139|0];g=b<<24>>24<0;g=ba(100,c+464|0,(g?J[c+128>>2]:h)|0,(g?J[c+132>>2]:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break Ec}J[c+456>>2]=J[g+8>>2];b=J[g+4>>2];J[c+448>>2]=J[g>>2];J[c+452>>2]=b;J[g>>2]=0;J[g+4>>2]=0;J[g+8>>2]=0;J[5996]=0;b=_(97,c+448|0,3570)|0;h=J[5996];J[5996]=0;if((h|0)==1){break Dc}h=J[b+4>>2];J[a>>2]=J[b>>2];J[a+4>>2]=h;J[a+8>>2]=J[b+8>>2];J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;if(H[c+139|0]<0){$a(J[c+128>>2]);}if(H[g+11|0]<0){$a(J[g>>2]);}if(H[f+11|0]<0){$a(J[f>>2]);}if(H[c+155|0]<0){$a(J[c+144>>2]);}if(H[d+11|0]<0){$a(J[d>>2]);}if(H[e+11|0]<0){$a(J[e>>2]);}if(H[c+91|0]>=0){break y}$a(J[c+80>>2]);break y}b=Z()|0;break wc}b=Z()|0;break xc}b=Z()|0;break yc}b=Z()|0;break zc}b=Z()|0;break Ac}b=Z()|0;break Bc}b=Z()|0;break Cc}b=Z()|0;if(H[c+459|0]>=0){break Cc}$a(J[c+448>>2]);}if(H[c+139|0]>=0){break Bc}$a(J[c+128>>2]);}if(H[c+475|0]>=0){break Ac}$a(J[c+464>>2]);}if(H[c+411|0]>=0){break zc}$a(J[c+400>>2]);}if(H[c+155|0]>=0){break yc}$a(J[c+144>>2]);}if(H[c+395|0]>=0){break xc}$a(J[c+384>>2]);}if(H[c+379|0]>=0){break wc}$a(J[c+368>>2]);}if(H[c+91|0]>=0){break a}$a(J[c+80>>2]);break a}H[c+79|0]=4;H[c+72|0]=0;J[c+68>>2]=1767862895;J[5996]=0;e=c+368|0;ea(161,e|0,b|0,c+68|0);d=J[5996];J[5996]=0;Kc:{Lc:{Mc:{Nc:{Oc:{Pc:{Qc:{Rc:{Sc:{Tc:{Uc:{Vc:{Wc:{Xc:{if((d|0)!=1){J[5996]=0;e=_(97,e|0,3673)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Xc}J[c+392>>2]=J[e+8>>2];d=J[e+4>>2];J[c+384>>2]=J[e>>2];J[c+388>>2]=d;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;d=J[J[b>>2]+32>>2];J[5996]=0;f=c+144|0;aa(141,f|0,d|0);d=J[5996];J[5996]=0;if((d|0)==1){break Wc}J[5996]=0;g=f;d=K[c+155|0];f=d<<24>>24<0;d=ba(100,c+384|0,(f?J[c+144>>2]:g)|0,(f?J[c+148>>2]:d)|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){break Vc}J[c+408>>2]=J[d+8>>2];f=J[d+4>>2];J[c+400>>2]=J[d>>2];J[c+404>>2]=f;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;f=_(97,c+400|0,3613)|0;g=J[5996];J[5996]=0;if((g|0)==1){break Uc}J[c+472>>2]=J[f+8>>2];g=J[f+4>>2];J[c+464>>2]=J[f>>2];J[c+468>>2]=g;J[f>>2]=0;J[f+4>>2]=0;J[f+8>>2]=0;b=J[J[b>>2]+32>>2];J[5996]=0;g=c+128|0;aa(141,g|0,b+8|0);b=J[5996];J[5996]=0;if((b|0)==1){break Tc}J[5996]=0;h=g;b=K[c+139|0];g=b<<24>>24<0;g=ba(100,c+464|0,(g?J[c+128>>2]:h)|0,(g?J[c+132>>2]:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break Sc}J[c+456>>2]=J[g+8>>2];b=J[g+4>>2];J[c+448>>2]=J[g>>2];J[c+452>>2]=b;J[g>>2]=0;J[g+4>>2]=0;J[g+8>>2]=0;J[5996]=0;b=_(97,c+448|0,3570)|0;h=J[5996];J[5996]=0;if((h|0)==1){break Rc}h=J[b+4>>2];J[a>>2]=J[b>>2];J[a+4>>2]=h;J[a+8>>2]=J[b+8>>2];J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;if(H[c+139|0]<0){$a(J[c+128>>2]);}if(H[g+11|0]<0){$a(J[g>>2]);}if(H[f+11|0]<0){$a(J[f>>2]);}if(H[c+155|0]<0){$a(J[c+144>>2]);}if(H[d+11|0]<0){$a(J[d>>2]);}if(H[e+11|0]<0){$a(J[e>>2]);}if(H[c+79|0]>=0){break y}$a(J[c+68>>2]);break y}b=Z()|0;break Kc}b=Z()|0;break Lc}b=Z()|0;break Mc}b=Z()|0;break Nc}b=Z()|0;break Oc}b=Z()|0;break Pc}b=Z()|0;break Qc}b=Z()|0;if(H[c+459|0]>=0){break Qc}$a(J[c+448>>2]);}if(H[c+139|0]>=0){break Pc}$a(J[c+128>>2]);}if(H[c+475|0]>=0){break Oc}$a(J[c+464>>2]);}if(H[c+411|0]>=0){break Nc}$a(J[c+400>>2]);}if(H[c+155|0]>=0){break Mc}$a(J[c+144>>2]);}if(H[c+395|0]>=0){break Lc}$a(J[c+384>>2]);}if(H[c+379|0]>=0){break Kc}$a(J[c+368>>2]);}if(H[c+79|0]>=0){break a}$a(J[c+68>>2]);break a}I[c- -64>>1]=K[2600]|K[2601]<<8;I[c+66>>1]=2560;J[5996]=0;e=K[2596]|K[2597]<<8|(K[2598]<<16|K[2599]<<24);J[c+56>>2]=K[2592]|K[2593]<<8|(K[2594]<<16|K[2595]<<24);J[c+60>>2]=e;e=c+128|0;ea(161,e|0,b|0,c+56|0);d=J[5996];J[5996]=0;Yc:{Zc:{_c:{$c:{ad:{bd:{cd:{dd:{ed:{fd:{gd:{hd:{id:{jd:{kd:{ld:{md:{nd:{od:{pd:{if((d|0)!=1){J[5996]=0;e=_(97,e|0,3623)|0;d=J[5996];J[5996]=0;if((d|0)==1){break pd}J[c+152>>2]=J[e+8>>2];d=J[e+4>>2];J[c+144>>2]=J[e>>2];J[c+148>>2]=d;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;d=J[J[b>>2]+32>>2];J[5996]=0;f=c+40|0;aa(141,f|0,d|0);d=J[5996];J[5996]=0;if((d|0)==1){break od}J[5996]=0;g=f;d=K[c+51|0];f=d<<24>>24<0;d=ba(100,c+144|0,(f?J[c+40>>2]:g)|0,(f?J[c+44>>2]:d)|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){break nd}J[c+376>>2]=J[d+8>>2];f=J[d+4>>2];J[c+368>>2]=J[d>>2];J[c+372>>2]=f;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;f=_(97,c+368|0,3613)|0;g=J[5996];J[5996]=0;if((g|0)==1){break md}J[c+392>>2]=J[f+8>>2];g=J[f+4>>2];J[c+384>>2]=J[f>>2];J[c+388>>2]=g;J[f>>2]=0;J[f+4>>2]=0;J[f+8>>2]=0;g=J[J[b>>2]+32>>2];J[5996]=0;h=c+28|0;aa(141,h|0,g+8|0);g=J[5996];J[5996]=0;if((g|0)==1){break ld}J[5996]=0;i=h;g=K[c+39|0];h=g<<24>>24<0;g=ba(100,c+384|0,(h?J[c+28>>2]:i)|0,(h?J[c+32>>2]:g)|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break kd}J[c+408>>2]=J[g+8>>2];h=J[g+4>>2];J[c+400>>2]=J[g>>2];J[c+404>>2]=h;J[g>>2]=0;J[g+4>>2]=0;J[g+8>>2]=0;J[5996]=0;h=_(97,c+400|0,3603)|0;i=J[5996];J[5996]=0;if((i|0)==1){break jd}J[c+472>>2]=J[h+8>>2];i=J[h+4>>2];J[c+464>>2]=J[h>>2];J[c+468>>2]=i;J[h>>2]=0;J[h+4>>2]=0;J[h+8>>2]=0;b=J[J[b>>2]+32>>2];J[5996]=0;i=c+16|0;aa(141,i|0,b+16|0);b=J[5996];J[5996]=0;if((b|0)==1){break id}J[5996]=0;j=i;b=K[c+27|0];i=b<<24>>24<0;i=ba(100,c+464|0,(i?J[c+16>>2]:j)|0,(i?J[c+20>>2]:b)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break hd}J[c+456>>2]=J[i+8>>2];b=J[i+4>>2];J[c+448>>2]=J[i>>2];J[c+452>>2]=b;J[i>>2]=0;J[i+4>>2]=0;J[i+8>>2]=0;J[5996]=0;b=_(97,c+448|0,3570)|0;j=J[5996];J[5996]=0;if((j|0)==1){break gd}j=J[b+4>>2];J[a>>2]=J[b>>2];J[a+4>>2]=j;J[a+8>>2]=J[b+8>>2];J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;if(H[c+27|0]<0){$a(J[c+16>>2]);}if(H[i+11|0]<0){$a(J[i>>2]);}if(H[h+11|0]<0){$a(J[h>>2]);}if(H[c+39|0]<0){$a(J[c+28>>2]);}if(H[g+11|0]<0){$a(J[g>>2]);}if(H[f+11|0]<0){$a(J[f>>2]);}if(H[c+51|0]<0){$a(J[c+40>>2]);}if(H[d+11|0]<0){$a(J[d>>2]);}if(H[e+11|0]<0){$a(J[e>>2]);}if(H[c+67|0]>=0){break y}$a(J[c+56>>2]);break y}b=Z()|0;break Yc}b=Z()|0;break Zc}b=Z()|0;break _c}b=Z()|0;break $c}b=Z()|0;break ad}b=Z()|0;break bd}b=Z()|0;break cd}b=Z()|0;break dd}b=Z()|0;break ed}b=Z()|0;break fd}b=Z()|0;if(H[c+459|0]>=0){break fd}$a(J[c+448>>2]);}if(H[c+27|0]>=0){break ed}$a(J[c+16>>2]);}if(H[c+475|0]>=0){break dd}$a(J[c+464>>2]);}if(H[c+411|0]>=0){break cd}$a(J[c+400>>2]);}if(H[c+39|0]>=0){break bd}$a(J[c+28>>2]);}if(H[c+395|0]>=0){break ad}$a(J[c+384>>2]);}if(H[c+379|0]>=0){break $c}$a(J[c+368>>2]);}if(H[c+51|0]>=0){break _c}$a(J[c+40>>2]);}if(H[c+155|0]>=0){break Zc}$a(J[c+144>>2]);}if(H[c+139|0]>=0){break Yc}$a(J[c+128>>2]);}if(H[c+67|0]>=0){break a}$a(J[c+56>>2]);break a}d=c+144|0;Kc(d,J[e+4>>2]);J[5996]=0;e=ba(99,d|0,0,5302)|0;d=J[5996];J[5996]=0;if((d|0)==1){break x}J[c+376>>2]=J[e+8>>2];d=J[e+4>>2];J[c+368>>2]=J[e>>2];J[c+372>>2]=d;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;J[5996]=0;d=_(97,c+368|0,5417)|0;f=J[5996];J[5996]=0;if((f|0)==1){break w}J[c+392>>2]=J[d+8>>2];f=J[d+4>>2];J[c+384>>2]=J[d>>2];J[c+388>>2]=f;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;f=J[b>>2];g=J[f+36>>2];f=J[f+32>>2];J[5996]=0;h=c+128|0;aa(115,h|0,g-f>>3);f=J[5996];J[5996]=0;if((f|0)==1){break v}J[5996]=0;f=K[c+139|0];g=f<<24>>24<0;f=ba(100,c+384|0,(g?J[c+128>>2]:h)|0,(g?J[c+132>>2]:f)|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break u}J[c+408>>2]=J[f+8>>2];g=J[f+4>>2];J[c+400>>2]=J[f>>2];J[c+404>>2]=g;J[f>>2]=0;J[f+4>>2]=0;J[f+8>>2]=0;J[5996]=0;g=_(97,c+400|0,5242)|0;h=J[5996];J[5996]=0;if((h|0)==1){break t}J[c+8>>2]=J[g+8>>2];h=J[g+4>>2];J[c>>2]=J[g>>2];J[c+4>>2]=h;J[g>>2]=0;J[g+4>>2]=0;J[g+8>>2]=0;J[5996]=0;h=c+464|0;ea(161,h|0,b|0,c|0);i=J[5996];J[5996]=0;if((i|0)==1){break s}J[5996]=0;h=_(97,h|0,3726)|0;i=J[5996];J[5996]=0;if((i|0)==1){break r}J[c+456>>2]=J[h+8>>2];i=J[h+4>>2];J[c+448>>2]=J[h>>2];J[c+452>>2]=i;J[h>>2]=0;J[h+4>>2]=0;J[h+8>>2]=0;if(H[c+11|0]<0){$a(J[c>>2]);}if(H[g+11|0]<0){$a(J[g>>2]);}if(H[c+139|0]<0){$a(J[c+128>>2]);}if(H[f+11|0]<0){$a(J[f>>2]);}if(H[d+11|0]<0){$a(J[d>>2]);}if(H[e+11|0]<0){$a(J[e>>2]);}b=J[b>>2];d=J[b+32>>2];f=J[b+36>>2];if((d|0)!=(f|0)){while(1){J[5996]=0;b=c+28|0;aa(141,b|0,d|0);e=J[5996];J[5996]=0;if((e|0)==1){break k}J[5996]=0;b=ba(99,b|0,0,3708)|0;e=J[5996];J[5996]=0;if((e|0)==1){break j}J[c+48>>2]=J[b+8>>2];e=J[b+4>>2];J[c+40>>2]=J[b>>2];J[c+44>>2]=e;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;J[5996]=0;e=_(97,c+40|0,3713)|0;g=J[5996];J[5996]=0;if((g|0)==1){break i}J[c+136>>2]=J[e+8>>2];g=J[e+4>>2];J[c+128>>2]=J[e>>2];J[c+132>>2]=g;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;J[5996]=0;g=K[c+139|0];h=g<<24>>24<0;ba(100,c+448|0,(h?J[c+128>>2]:c+128|0)|0,(h?J[c+132>>2]:g)|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break h}if(H[c+139|0]<0){$a(J[c+128>>2]);}if(H[c+51|0]<0){$a(J[e>>2]);}if(H[c+39|0]<0){$a(J[b>>2]);}d=d+8|0;if((f|0)!=(d|0)){continue}break}}J[5996]=0;b=_(97,c+448|0,3575)|0;e=J[5996];J[5996]=0;if((e|0)==1){break e}e=J[b+4>>2];J[a>>2]=J[b>>2];J[a+4>>2]=e;J[a+8>>2]=J[b+8>>2];J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;}Ua=c+496|0;return}b=Z()|0;break l}b=Z()|0;break m}b=Z()|0;break n}b=Z()|0;break o}b=Z()|0;break p}b=Z()|0;break q}b=Z()|0;if(H[c+475|0]>=0){break q}$a(J[c+464>>2]);}if(H[c+11|0]>=0){break p}$a(J[c>>2]);}if(H[c+411|0]>=0){break o}$a(J[c+400>>2]);}if(H[c+139|0]>=0){break n}$a(J[c+128>>2]);}if(H[c+395|0]>=0){break m}$a(J[c+384>>2]);}if(H[c+379|0]>=0){break l}$a(J[c+368>>2]);}if(H[c+155|0]>=0){break a}$a(J[c+144>>2]);break a}b=Z()|0;break d}b=Z()|0;break f}b=Z()|0;break g}b=Z()|0;if(H[c+139|0]>=0){break g}$a(J[c+128>>2]);}if(H[c+51|0]>=0){break f}$a(J[c+40>>2]);}if(H[c+39|0]>=0){break d}$a(J[c+28>>2]);break d}b=Z()|0;}if(H[c+459|0]>=0){break a}$a(J[c+448>>2]);break a}if(!K[c+460|0]|H[c+459|0]>=0){break a}$a(J[c+448>>2]);break a}if(H[c+475|0]>=0){break a}$a(J[c+464>>2]);}da(b|0);B();}function ze(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0;h=Ua-48|0;Ua=h;J[h+24>>2]=0;J[h+16>>2]=0;J[h+20>>2]=0;a:{b:{c:{d:{e:{f:{g:{h:{i:{j:{k:{l:{m:{n:{o:{p:{q:{r:{s:{t:{u:{v:{switch(J[a>>2]){case 1:J[5996]=0;b=$(3,56)|0;d=J[5996];J[5996]=0;if((d|0)==1){break d}J[5996]=0;a=_(32,h+28|0,a+28|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break u}J[b+4>>2]=0;J[b+8>>2]=0;J[b>>2]=2;J[b+12>>2]=0;J[b+16>>2]=J[a>>2];J[b+20>>2]=J[a+4>>2];J[b+24>>2]=J[a+8>>2];J[a+8>>2]=0;J[a>>2]=0;J[a+4>>2]=0;J[b+44>>2]=0;J[b+36>>2]=0;J[b+40>>2]=0;J[b+28>>2]=0;J[b+32>>2]=0;J[5996]=0;aa(33,c|0,b|0);a=J[5996];J[5996]=0;if((a|0)==1){break d}d=J[h+20>>2];f=J[h+24>>2];if(d>>>0<f>>>0){J[d>>2]=b;J[h+20>>2]=d+4;break o}a=J[h+16>>2];e=d-a>>2;g=e+1|0;if(g>>>0>=1073741824){J[5996]=0;ca(34);a=J[5996];J[5996]=0;if((a|0)==1){break d}break a}f=f-a|0;i=f>>1;g=f>>>0>=2147483644?1073741823:g>>>0<i>>>0?i:g;if(g){if(g>>>0>=1073741824){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break d}J[5996]=0;j=$(3,g<<2)|0;f=J[5996];J[5996]=0;if((f|0)==1){break d}}e=(e<<2)+j|0;J[e>>2]=b;f=e+4|0;if((a|0)!=(d|0)){while(1){e=e-4|0;d=d-4|0;J[e>>2]=J[d>>2];if((a|0)!=(d|0)){continue}break}d=J[h+16>>2];}J[h+24>>2]=(g<<2)+j;J[h+20>>2]=f;J[h+16>>2]=e;if(!d){break n}$a(d);break o;case 2:a=J[a+52>>2];J[5996]=0;b=$(3,56)|0;d=J[5996];J[5996]=0;if((d|0)==1){break e}J[b+4>>2]=0;J[b+8>>2]=0;J[b>>2]=4;J[b+40>>2]=a;J[b+44>>2]=0;J[b+12>>2]=0;J[b+16>>2]=0;J[b+20>>2]=0;J[b+24>>2]=0;J[b+28>>2]=0;J[b+32>>2]=0;J[b+36>>2]=0;J[5996]=0;aa(33,c|0,b|0);a=J[5996];J[5996]=0;if((a|0)==1){break e}d=J[h+20>>2];f=J[h+24>>2];if(d>>>0<f>>>0){J[d>>2]=b;J[h+20>>2]=d+4;break o}a=J[h+16>>2];e=d-a>>2;g=e+1|0;if(g>>>0>=1073741824){J[5996]=0;ca(34);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break e}f=f-a|0;i=f>>1;g=f>>>0>=2147483644?1073741823:g>>>0<i>>>0?i:g;if(g){if(g>>>0>=1073741824){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break e}J[5996]=0;j=$(3,g<<2)|0;f=J[5996];J[5996]=0;if((f|0)==1){break e}}e=(e<<2)+j|0;J[e>>2]=b;f=e+4|0;if((a|0)!=(d|0)){while(1){e=e-4|0;d=d-4|0;J[e>>2]=J[d>>2];if((a|0)!=(d|0)){continue}break}d=J[h+16>>2];}J[h+24>>2]=(g<<2)+j;J[h+20>>2]=f;J[h+16>>2]=e;if(!d){break n}$a(d);break o;case 3:a=J[a+52>>2];J[5996]=0;b=$(3,56)|0;d=J[5996];J[5996]=0;if((d|0)==1){break f}J[b+4>>2]=0;J[b+8>>2]=0;J[b>>2]=5;J[b+40>>2]=a;J[b+44>>2]=0;J[b+12>>2]=0;J[b+16>>2]=0;J[b+20>>2]=0;J[b+24>>2]=0;J[b+28>>2]=0;J[b+32>>2]=0;J[b+36>>2]=0;J[5996]=0;aa(33,c|0,b|0);a=J[5996];J[5996]=0;if((a|0)==1){break f}d=J[h+20>>2];f=J[h+24>>2];if(d>>>0<f>>>0){J[d>>2]=b;J[h+20>>2]=d+4;break o}a=J[h+16>>2];e=d-a>>2;g=e+1|0;if(g>>>0>=1073741824){J[5996]=0;ca(34);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break f}f=f-a|0;i=f>>1;g=f>>>0>=2147483644?1073741823:g>>>0<i>>>0?i:g;if(g){if(g>>>0>=1073741824){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break f}J[5996]=0;j=$(3,g<<2)|0;f=J[5996];J[5996]=0;if((f|0)==1){break f}}e=(e<<2)+j|0;J[e>>2]=b;f=e+4|0;if((a|0)!=(d|0)){while(1){e=e-4|0;d=d-4|0;J[e>>2]=J[d>>2];if((a|0)!=(d|0)){continue}break}d=J[h+16>>2];}J[h+24>>2]=(g<<2)+j;J[h+20>>2]=f;J[h+16>>2]=e;if(!d){break n}$a(d);break o;case 5:b=J[a+20>>2];a=J[a+16>>2];J[5996]=0;g=ia(36,c|0,7,a|0,b|0)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}d=J[h+20>>2];f=J[h+24>>2];if(d>>>0<f>>>0){J[d>>2]=g;J[h+20>>2]=d+4;break o}a=J[h+16>>2];e=d-a>>2;b=e+1|0;if(b>>>0>=1073741824){J[5996]=0;ca(34);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break g}f=f-a|0;i=f>>1;b=f>>>0>=2147483644?1073741823:b>>>0<i>>>0?i:b;if(b){if(b>>>0>=1073741824){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break g}J[5996]=0;j=$(3,b<<2)|0;f=J[5996];J[5996]=0;if((f|0)==1){break g}}e=(e<<2)+j|0;J[e>>2]=g;f=e+4|0;if((a|0)!=(d|0)){while(1){e=e-4|0;d=d-4|0;J[e>>2]=J[d>>2];if((a|0)!=(d|0)){continue}break}d=J[h+16>>2];}J[h+24>>2]=(b<<2)+j;J[h+20>>2]=f;J[h+16>>2]=e;if(!d){break n}$a(d);break o;case 4:b=J[a+20>>2];a=J[a+16>>2];J[5996]=0;g=ia(36,c|0,6,a|0,b|0)|0;a=J[5996];J[5996]=0;if((a|0)==1){break h}d=J[h+20>>2];f=J[h+24>>2];if(d>>>0<f>>>0){J[d>>2]=g;J[h+20>>2]=d+4;break o}a=J[h+16>>2];e=d-a>>2;b=e+1|0;if(b>>>0>=1073741824){J[5996]=0;ca(34);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break h}f=f-a|0;i=f>>1;b=f>>>0>=2147483644?1073741823:b>>>0<i>>>0?i:b;if(b){if(b>>>0>=1073741824){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break h}J[5996]=0;j=$(3,b<<2)|0;f=J[5996];J[5996]=0;if((f|0)==1){break h}}e=(e<<2)+j|0;J[e>>2]=g;f=e+4|0;if((a|0)!=(d|0)){while(1){e=e-4|0;d=d-4|0;J[e>>2]=J[d>>2];if((a|0)!=(d|0)){continue}break}d=J[h+16>>2];}J[h+24>>2]=(b<<2)+j;J[h+20>>2]=f;J[h+16>>2]=e;if(!d){break n}$a(d);break o;case 7:b=J[a+20>>2];a=J[a+16>>2];J[5996]=0;g=ia(36,c|0,8,a|0,b|0)|0;a=J[5996];J[5996]=0;if((a|0)==1){break i}d=J[h+20>>2];f=J[h+24>>2];if(d>>>0<f>>>0){J[d>>2]=g;J[h+20>>2]=d+4;break o}a=J[h+16>>2];e=d-a>>2;b=e+1|0;if(b>>>0>=1073741824){J[5996]=0;ca(34);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break i}f=f-a|0;i=f>>1;b=f>>>0>=2147483644?1073741823:b>>>0<i>>>0?i:b;if(b){if(b>>>0>=1073741824){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break i}J[5996]=0;j=$(3,b<<2)|0;f=J[5996];J[5996]=0;if((f|0)==1){break i}}e=(e<<2)+j|0;J[e>>2]=g;f=e+4|0;if((a|0)!=(d|0)){while(1){e=e-4|0;d=d-4|0;J[e>>2]=J[d>>2];if((a|0)!=(d|0)){continue}break}d=J[h+16>>2];}J[h+24>>2]=(b<<2)+j;J[h+20>>2]=f;J[h+16>>2]=e;if(!d){break n}$a(d);break o;case 6:b=J[a+20>>2];a=J[a+16>>2];J[5996]=0;g=ia(36,c|0,9,a|0,b|0)|0;a=J[5996];J[5996]=0;if((a|0)==1){break j}d=J[h+20>>2];f=J[h+24>>2];if(d>>>0<f>>>0){J[d>>2]=g;J[h+20>>2]=d+4;break o}a=J[h+16>>2];e=d-a>>2;b=e+1|0;if(b>>>0>=1073741824){J[5996]=0;ca(34);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break j}f=f-a|0;i=f>>1;b=f>>>0>=2147483644?1073741823:b>>>0<i>>>0?i:b;if(b){if(b>>>0>=1073741824){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break j}J[5996]=0;j=$(3,b<<2)|0;f=J[5996];J[5996]=0;if((f|0)==1){break j}}e=(e<<2)+j|0;J[e>>2]=g;f=e+4|0;if((a|0)!=(d|0)){while(1){e=e-4|0;d=d-4|0;J[e>>2]=J[d>>2];if((a|0)!=(d|0)){continue}break}d=J[h+16>>2];}J[h+24>>2]=(b<<2)+j;J[h+20>>2]=f;J[h+16>>2]=e;if(!d){break n}$a(d);break o;case 8:d=J[a+4>>2];if((J[a+8>>2]-d|0)!=112){break m}J[5996]=0;g=ba(37,d|0,b|0,c|0)|0;d=J[5996];J[5996]=0;w:{x:{y:{z:{A:{B:{C:{if((d|0)!=1){a=J[a+4>>2];J[5996]=0;f=ba(37,a+56|0,b|0,c|0)|0;a=J[5996];J[5996]=0;if((a|0)==1){break C}if(!g|!f){break m}J[5996]=0;a=$(3,8)|0;b=J[5996];J[5996]=0;if((b|0)==1){break z}J[a+4>>2]=f;J[a>>2]=g;J[5996]=0;b=$(3,56)|0;d=J[5996];J[5996]=0;if((d|0)==1){break y}J[b+16>>2]=0;J[b+20>>2]=0;J[b+4>>2]=a;J[b>>2]=10;a=a+8|0;J[b+12>>2]=a;J[b+8>>2]=a;J[b+24>>2]=0;J[b+28>>2]=0;J[b+32>>2]=0;J[b+36>>2]=0;J[b+40>>2]=0;J[b+44>>2]=0;J[5996]=0;aa(33,c|0,b|0);a=J[5996];J[5996]=0;if((a|0)==1){break k}e=J[h+20>>2];d=J[h+24>>2];if(e>>>0>=d>>>0){break B}J[e>>2]=b;J[h+20>>2]=e+4;break A}c=Z()|0;break c}c=Z()|0;break c}i=J[h+16>>2];k=e-i>>2;a=k+1|0;if(a>>>0>=1073741824){J[5996]=0;ca(34);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break k}d=d-i|0;j=d>>1;j=d>>>0>=2147483644?1073741823:a>>>0<j>>>0?j:a;D:{if(!j){a=0;break D}if(j>>>0>=1073741824){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break k}J[5996]=0;a=$(3,j<<2)|0;d=J[5996];J[5996]=0;if((d|0)==1){break k}}d=(k<<2)+a|0;J[d>>2]=b;b=d+4|0;if((e|0)!=(i|0)){while(1){d=d-4|0;e=e-4|0;J[d>>2]=J[e>>2];if((e|0)!=(i|0)){continue}break}e=J[h+16>>2];}J[h+24>>2]=(j<<2)+a;J[h+20>>2]=b;J[h+16>>2]=d;if(!e){break A}$a(e);}J[5996]=0;b=$(3,12)|0;a=J[5996];J[5996]=0;if((a|0)==1){break x}J[b+4>>2]=f;J[b>>2]=g;J[b+8>>2]=J[6079];J[5996]=0;a=$(3,56)|0;d=J[5996];J[5996]=0;if((d|0)==1){break w}J[a+16>>2]=0;J[a+20>>2]=0;J[a+4>>2]=b;J[a>>2]=12;J[a+48>>2]=0;J[a+52>>2]=1072693248;b=b+12|0;J[a+12>>2]=b;J[a+8>>2]=b;J[a+24>>2]=0;J[a+28>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[5996]=0;aa(33,c|0,a|0);b=J[5996];J[5996]=0;if((b|0)==1){break l}d=J[h+20>>2];f=J[h+24>>2];if(d>>>0<f>>>0){J[d>>2]=a;J[h+20>>2]=d+4;break o}b=J[h+16>>2];e=d-b>>2;g=e+1|0;if(g>>>0>=1073741824){J[5996]=0;ca(34);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break l}f=f-b|0;i=f>>1;g=f>>>0>=2147483644?1073741823:g>>>0<i>>>0?i:g;E:{if(!g){j=0;break E}if(g>>>0>=1073741824){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break l}J[5996]=0;j=$(3,g<<2)|0;f=J[5996];J[5996]=0;if((f|0)==1){break l}}e=(e<<2)+j|0;J[e>>2]=a;f=e+4|0;if((b|0)!=(d|0)){while(1){e=e-4|0;d=d-4|0;J[e>>2]=J[d>>2];if((b|0)!=(d|0)){continue}break}d=J[h+16>>2];}J[h+24>>2]=(g<<2)+j;J[h+20>>2]=f;J[h+16>>2]=e;if(!d){break n}$a(d);break o}c=Z()|0;break c}c=Z()|0;$a(a);break c}c=Z()|0;break c}c=Z()|0;$a(b);break c;case 9:d=J[a+4>>2];if((J[a+8>>2]-d|0)!=112){break m}g=J[a+40>>2];f=J[g+4>>2];i=J[g>>2];j=i+f|0;if(f>>>0>j>>>0){break m}J[5996]=0;g=ba(37,d|0,b|0,c|0)|0;d=J[5996];J[5996]=0;F:{G:{if((d|0)!=1){d=J[a+4>>2];J[5996]=0;f=ba(37,d+56|0,b|0,c|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break G}if(!g|!f){break m}o=+(i>>>0)/+(j>>>0);e=J[a+4>>2];if(J[e>>2]!=8){break p}d=J[e+4>>2];if((J[e+8>>2]-d|0)==112){break F}break p}c=Z()|0;break c}c=Z()|0;break c}J[5996]=0;i=ba(37,d|0,b|0,c|0)|0;d=J[5996];J[5996]=0;H:{I:{J:{if((d|0)!=1){d=J[J[a+4>>2]+4>>2];J[5996]=0;j=ba(37,d+56|0,b|0,c|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break J}e=0;if(!i|!j){break m}J[5996]=0;d=$(3,12)|0;e=J[5996];J[5996]=0;if((e|0)==1){break I}J[h+28>>2]=d;e=d+12|0;J[h+36>>2]=e;J[d+8>>2]=f;J[d+4>>2]=j;J[d>>2]=i;J[5996]=0;J[h+32>>2]=e;d=ya(38,c|0,12,h+28|0,+o)|0;e=J[5996];J[5996]=0;if((e|0)==1){break H}J[5996]=0;aa(39,h+16|0,d|0);d=J[5996];J[5996]=0;if((d|0)==1){break H}d=J[h+28>>2];if(d){$a(d);}e=J[a+4>>2];break p}c=Z()|0;break c}c=Z()|0;break c}c=Z()|0;break c}c=Z()|0;a=J[h+28>>2];if(!a){break c}$a(a);break c;case 0:break m;case 10:break v;default:break o}}g=J[a+4>>2];l=J[a+8>>2];if((g|0)==(l|0)){break s}while(1){K:{J[5996]=0;d=ba(37,g|0,b|0,c|0)|0;i=J[5996];J[5996]=0;if((i|0)==1){break K}L:{if(e>>>0<j>>>0){J[e>>2]=d;e=e+4|0;break L}n=e-f>>2;i=n+1|0;if(i>>>0>=1073741824){J[5996]=0;ca(34);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break r}j=j-f|0;k=j>>1;k=j>>>0>=2147483644?1073741823:i>>>0<k>>>0?k:i;M:{if(!k){j=0;break M}if(k>>>0>=1073741824){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break r}J[5996]=0;j=$(3,k<<2)|0;i=J[5996];J[5996]=0;if((i|0)==1){break K}}i=(n<<2)+j|0;J[i>>2]=d;d=i;if((e|0)!=(f|0)){while(1){d=d-4|0;e=e-4|0;J[d>>2]=J[e>>2];if((e|0)!=(f|0)){continue}break}}j=(k<<2)+j|0;e=i+4|0;if(f){$a(f);}f=d;}g=g+56|0;if((l|0)!=(g|0)){continue}break t}break}c=Z()|0;break q}c=Z()|0;$a(b);break c}if((e|0)==(f|0)){f=e;break s}d=f;while(1){if(J[d>>2]){d=d+4|0;if((e|0)!=(d|0)){continue}break s}break}e=0;if(!f){break m}$a(f);break m}N:{O:{n=J[a+8>>2];i=J[a+4>>2];P:{if((n-i|0)/56>>>0>20){break P}Q:{R:{if((i|0)!=(n|0)){d=i;while(1){if(J[d>>2]!=1){break P}d=d+56|0;if((n|0)!=(d|0)){continue}break}break R}J[h+12>>2]=0;J[h+4>>2]=0;J[h+8>>2]=0;break Q}J[h+12>>2]=0;J[h+4>>2]=0;J[h+8>>2]=0;q=h+12|0;while(1){b=J[i+28>>2];S:{T:{d=J[h+8>>2];if((d|0)!=J[h+12>>2]){if(H[b+11|0]>=0){g=J[b+4>>2];J[d>>2]=J[b>>2];J[d+4>>2]=g;J[d+8>>2]=J[b+8>>2];break T}g=J[b+4>>2];b=J[b>>2];J[5996]=0;ea(40,d|0,b|0,g|0);b=J[5996];J[5996]=0;if((b|0)!=1){break T}c=Z()|0;J[h+8>>2]=d;break N}U:{V:{W:{X:{g=J[h+4>>2];k=(d-g|0)/12|0;l=k+1|0;Y:{if(l>>>0>=357913942){J[5996]=0;ca(41);break Y}J[h+44>>2]=q;m=k<<1;m=k>>>0>=178956970?357913941:l>>>0<m>>>0?m:l;if(!m){l=0;break W}if(m>>>0<357913942){break X}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break a}c=Z()|0;break N}J[5996]=0;l=$(3,P(m,12)|0)|0;p=J[5996];J[5996]=0;if((p|0)==1){break V}}J[h+28>>2]=l;k=P(k,12)+l|0;J[h+36>>2]=k;l=P(m,12)+l|0;J[h+40>>2]=l;J[h+32>>2]=k;Z:{if(H[b+11|0]>=0){m=J[b+4>>2];J[k>>2]=J[b>>2];J[k+4>>2]=m;J[k+8>>2]=J[b+8>>2];break Z}d=J[b+4>>2];b=J[b>>2];J[5996]=0;ea(40,k|0,b|0,d|0);b=J[5996];J[5996]=0;if((b|0)==1){break U}g=J[h+4>>2];d=J[h+8>>2];}m=k+12|0;if((d|0)==(g|0)){b=g;}else {while(1){d=d-12|0;p=J[d+4>>2];k=k-12|0;b=k;J[b>>2]=J[d>>2];J[b+4>>2]=p;J[b+8>>2]=J[d+8>>2];J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;if((d|0)!=(g|0)){continue}break}g=J[h+8>>2];b=J[h+4>>2];}J[h+32>>2]=b;J[h+4>>2]=k;J[h+8>>2]=m;d=J[h+12>>2];J[h+12>>2]=l;J[h+40>>2]=d;J[h+28>>2]=b;if((b|0)!=(g|0)){while(1){d=g-12|0;J[h+36>>2]=d;if(H[g-1|0]<0){$a(J[d>>2]);}g=d;if((d|0)!=(b|0)){continue}break}g=b;}if(!g){break S}$a(g);break S}c=Z()|0;break N}c=Z()|0;a=J[h+36>>2];d=J[h+32>>2];if((a|0)!=(d|0)){while(1){b=a-12|0;J[h+36>>2]=b;if(H[a-1|0]>=0){a=b;}else {$a(J[b>>2]);a=J[h+36>>2];}if((d|0)!=(a|0)){continue}break}}a=J[h+28>>2];if(a){$a(a);}break N}J[h+8>>2]=d+12;}i=i+56|0;if((n|0)!=(i|0)){continue}break}}b=J[a+52>>2];J[5996]=0;i=$(3,56)|0;d=J[5996];J[5996]=0;if((d|0)==1){break O}d=J[h+12>>2];J[h+12>>2]=0;g=J[h+4>>2];k=J[h+8>>2];J[h+4>>2]=0;J[h+8>>2]=0;J[i+4>>2]=0;J[i+8>>2]=0;J[i>>2]=3;J[i+12>>2]=0;J[i+40>>2]=b;J[i+44>>2]=0;J[i+36>>2]=0;J[i+28>>2]=0;J[i+32>>2]=0;J[i+24>>2]=d;J[i+16>>2]=g;J[i+20>>2]=k;J[5996]=0;aa(33,c|0,i|0);b=J[5996];J[5996]=0;if((b|0)==1){break O}d=J[h+20>>2];k=J[h+24>>2];_:{if(d>>>0<k>>>0){J[d>>2]=i;J[h+20>>2]=d+4;break _}b=J[h+16>>2];n=d-b>>2;g=n+1|0;if(g>>>0>=1073741824){J[5996]=0;ca(34);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break O}k=k-b|0;l=k>>1;k=k>>>0>=2147483644?1073741823:g>>>0<l>>>0?l:g;$:{if(!k){l=0;break $}if(k>>>0>=1073741824){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break O}J[5996]=0;l=$(3,k<<2)|0;g=J[5996];J[5996]=0;if((g|0)==1){break O}}g=(n<<2)+l|0;J[g>>2]=i;i=g+4|0;if((b|0)!=(d|0)){while(1){g=g-4|0;d=d-4|0;J[g>>2]=J[d>>2];if((b|0)!=(d|0)){continue}break}d=J[h+16>>2];}J[h+24>>2]=(k<<2)+l;J[h+20>>2]=i;J[h+16>>2]=g;if(!d){break _}$a(d);}b=J[h+4>>2];if(!b){break P}d=b;g=J[h+8>>2];if((d|0)!=(g|0)){while(1){d=g-12|0;if(H[g-1|0]<0){$a(J[d>>2]);}g=d;if((d|0)!=(b|0)){continue}break}d=J[h+4>>2];}J[h+8>>2]=b;$a(d);}b=J[a+52>>2];aa:{if((b|0)!=1&(b|0)!=((J[a+8>>2]-J[a+4>>2]|0)/56|0)){break aa}ba:{ca:{da:{ea:{fa:{ga:{b=e-f|0;if(b>>>0>=5){while(1){k=J[a+52>>2];J[5996]=0;i=$(3,8)|0;d=J[5996];J[5996]=0;if((d|0)==1){break ga}d=e-8|0;J[i>>2]=J[d>>2];e=e-4|0;J[i+4>>2]=J[e>>2];J[5996]=0;g=$(3,56)|0;l=J[5996];J[5996]=0;if((l|0)==1){break fa}J[g+16>>2]=0;J[g+20>>2]=0;J[g+4>>2]=i;J[g>>2]=(k|0)==1?11:10;i=i+8|0;J[g+12>>2]=i;J[g+8>>2]=i;J[g+24>>2]=0;J[g+28>>2]=0;J[g+32>>2]=0;J[g+36>>2]=0;J[g+40>>2]=0;J[g+44>>2]=0;O[g+48>>3]=1/+(b>>2>>>0);J[5996]=0;aa(33,c|0,g|0);b=J[5996];J[5996]=0;if((b|0)==1){c=Z()|0;break q}J[5996]=0;i=$(3,4)|0;b=J[5996];J[5996]=0;if((b|0)==1){break ea}J[i>>2]=g;J[5996]=0;b=$(3,56)|0;g=J[5996];J[5996]=0;if((g|0)==1){break ca}J[b+16>>2]=0;J[b+20>>2]=0;J[b+4>>2]=i;J[b>>2]=21;g=i+4|0;J[b+12>>2]=g;J[b+8>>2]=g;J[b+24>>2]=0;J[b+28>>2]=0;J[b+32>>2]=0;J[b+36>>2]=0;J[b+40>>2]=0;J[b+44>>2]=0;J[5996]=0;aa(33,c|0,b|0);g=J[5996];J[5996]=0;if((g|0)==1){break da}ha:{if(d>>>0<j>>>0){J[d>>2]=b;break ha}ia:{ja:{i=d-f>>2;g=i+1|0;ka:{if(g>>>0>=1073741824){J[5996]=0;ca(34);break ka}e=j-f|0;j=e>>1;e=e>>>0>=2147483644?1073741823:g>>>0<j>>>0?j:g;if(!e){j=0;break ia}if(e>>>0<1073741824){break ja}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break a}c=Z()|0;break q}J[5996]=0;j=$(3,e<<2)|0;g=J[5996];J[5996]=0;if((g|0)==1){break da}}g=(i<<2)+j|0;J[g>>2]=b;j=(e<<2)+j|0;e=g+4|0;la:{if((d|0)!=(f|0)){while(1){g=g-4|0;d=d-4|0;J[g>>2]=J[d>>2];if((d|0)!=(f|0)){continue}break}if(!f){break la}}$a(f);}f=g;}b=e-f|0;if(b>>>0>4){continue}break}}j=J[f>>2];d=J[h+20>>2];if((d|0)==J[h+24>>2]){break ba}J[d>>2]=j;J[h+20>>2]=d+4;break aa}c=Z()|0;break q}c=Z()|0;$a(i);break q}c=Z()|0;break q}c=Z()|0;break q}c=Z()|0;$a(i);break q}k=J[h+16>>2];b=d-k|0;l=b>>2;g=l+1|0;if(g>>>0>=1073741824){J[5996]=0;ca(34);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break r}i=b>>1;b=b>>>0>=2147483644?1073741823:g>>>0<i>>>0?i:g;ma:{if(!b){i=0;break ma}if(b>>>0>=1073741824){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break r}J[5996]=0;i=$(3,b<<2)|0;g=J[5996];J[5996]=0;if((g|0)==1){break r}}g=(l<<2)+i|0;J[g>>2]=j;j=g+4|0;if((d|0)!=(k|0)){while(1){g=g-4|0;d=d-4|0;J[g>>2]=J[d>>2];if((d|0)!=(k|0)){continue}break}d=J[h+16>>2];}J[h+24>>2]=(b<<2)+i;J[h+20>>2]=j;J[h+16>>2]=g;if(!d){break aa}$a(d);}j=J[a+52>>2];b=0;J[5996]=0;a=$(3,56)|0;d=J[5996];J[5996]=0;na:{oa:{if((d|0)==1){break oa}g=e-f|0;k=g>>2;i=0;d=0;if((e|0)!=(f|0)){if((g|0)<0){J[5996]=0;ca(34);b=J[5996];J[5996]=0;if((b|0)!=1){break a}break na}J[5996]=0;d=$(3,g|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break na}g=(g-4&-4)+4|0;e=fb(d,f,g);b=e+(k<<2)|0;i=e+g|0;}J[a+16>>2]=0;J[a+20>>2]=0;J[a+4>>2]=d;J[a>>2]=13;O[a+48>>3]=+(j>>>0)/+(k>>>0);J[a+12>>2]=b;J[a+8>>2]=i;J[a+24>>2]=0;J[a+28>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+40>>2]=j;J[a+44>>2]=0;J[5996]=0;aa(33,c|0,a|0);b=J[5996];J[5996]=0;if((b|0)==1){break oa}e=J[h+20>>2];d=J[h+24>>2];pa:{if(e>>>0<d>>>0){J[e>>2]=a;J[h+20>>2]=e+4;break pa}g=J[h+16>>2];j=e-g>>2;b=j+1|0;if(b>>>0>=1073741824){J[5996]=0;ca(34);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break oa}d=d-g|0;i=d>>1;i=d>>>0>=2147483644?1073741823:b>>>0<i>>>0?i:b;qa:{if(!i){b=0;break qa}if(i>>>0>=1073741824){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break oa}J[5996]=0;b=$(3,i<<2)|0;d=J[5996];J[5996]=0;if((d|0)==1){break oa}}d=(j<<2)+b|0;J[d>>2]=a;a=d+4|0;if((e|0)!=(g|0)){while(1){d=d-4|0;e=e-4|0;J[d>>2]=J[e>>2];if((e|0)!=(g|0)){continue}break}e=J[h+16>>2];}J[h+24>>2]=(i<<2)+b;J[h+20>>2]=a;J[h+16>>2]=d;if(!e){break pa}$a(e);}if(!f){break o}$a(f);break o}c=Z()|0;break q}c=Z()|0;$a(a);break q}c=Z()|0;}rb(h+4|0);break q}c=Z()|0;}if(!f){break c}$a(f);break c}ra:{sa:{ta:{ua:{va:{wa:{xa:{ya:{za:{if(J[e+56>>2]!=8){break za}d=J[e+60>>2];if((J[e- -64>>2]-d|0)!=112){break za}J[5996]=0;d=ba(37,d|0,b|0,c|0)|0;e=J[5996];J[5996]=0;if((e|0)==1){break ya}a=J[J[a+4>>2]+60>>2];J[5996]=0;b=ba(37,a+56|0,b|0,c|0)|0;a=J[5996];J[5996]=0;if((a|0)==1){break xa}e=0;if(!d|!b){break m}J[5996]=0;a=$(3,12)|0;e=J[5996];J[5996]=0;if((e|0)==1){break wa}J[h+28>>2]=a;e=a+12|0;J[h+36>>2]=e;J[a+8>>2]=g;J[a+4>>2]=b;J[a>>2]=d;J[5996]=0;J[h+32>>2]=e;a=Sa(42,c|0,h+28|0,+(1-o))|0;b=J[5996];J[5996]=0;if((b|0)==1){break va}J[5996]=0;aa(39,h+16|0,a|0);a=J[5996];J[5996]=0;if((a|0)==1){break va}a=J[h+28>>2];if(!a){break za}$a(a);}J[5996]=0;a=$(3,12)|0;b=J[5996];J[5996]=0;if((b|0)==1){break ua}J[h+28>>2]=a;b=a+12|0;J[h+36>>2]=b;J[a>>2]=g;d=J[6080];J[a+8>>2]=f;J[a+4>>2]=d;J[5996]=0;J[h+32>>2]=b;a=ya(38,c|0,12,h+28|0,+o)|0;b=J[5996];J[5996]=0;if((b|0)==1){break ta}J[5996]=0;aa(39,h+16|0,a|0);a=J[5996];J[5996]=0;if((a|0)==1){break ta}a=J[h+28>>2];if(a){$a(a);}J[5996]=0;a=$(3,8)|0;b=J[5996];J[5996]=0;if((b|0)==1){break sa}J[h+28>>2]=a;b=a+8|0;J[h+36>>2]=b;J[a+4>>2]=f;J[a>>2]=g;J[5996]=0;J[h+32>>2]=b;a=ya(38,c|0,11,h+28|0,+o)|0;b=J[5996];J[5996]=0;if((b|0)==1){break ra}J[5996]=0;aa(39,h+16|0,a|0);a=J[5996];J[5996]=0;if((a|0)==1){break ra}a=J[h+28>>2];if(!a){break o}$a(a);break o}c=Z()|0;break c}c=Z()|0;break c}c=Z()|0;break c}c=Z()|0;a=J[h+28>>2];if(!a){break c}$a(a);break c}c=Z()|0;break c}c=Z()|0;a=J[h+28>>2];if(!a){break c}$a(a);break c}c=Z()|0;break c}c=Z()|0;a=J[h+28>>2];if(!a){break c}$a(a);break c}f=J[h+20>>2];e=J[h+16>>2];}Aa:{Ba:{Ca:{Da:{if((f-e|0)==4){break Da}b=J[h+24>>2];J[h+24>>2]=0;J[h+16>>2]=0;J[h+20>>2]=0;J[5996]=0;a=$(3,56)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Ca}J[a+16>>2]=0;J[a+20>>2]=0;J[a+4>>2]=e;J[a>>2]=20;J[a+12>>2]=b;J[a+8>>2]=f;J[a+24>>2]=0;J[a+28>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[5996]=0;aa(33,c|0,a|0);b=J[5996];J[5996]=0;if((b|0)==1){break Aa}e=J[h+20>>2];g=J[h+24>>2];if(e>>>0<g>>>0){J[e>>2]=a;J[h+20>>2]=e+4;break Da}b=J[h+16>>2];f=e-b>>2;d=f+1|0;if(d>>>0>=1073741824){J[5996]=0;ca(34);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break Aa}g=g-b|0;i=g>>1;g=g>>>0>=2147483644?1073741823:d>>>0<i>>>0?i:d;Ea:{if(!g){j=0;break Ea}if(g>>>0>=1073741824){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break Aa}J[5996]=0;j=$(3,g<<2)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Aa}}d=(f<<2)+j|0;J[d>>2]=a;a=d+4|0;if((b|0)!=(e|0)){while(1){d=d-4|0;e=e-4|0;J[d>>2]=J[e>>2];if((b|0)!=(e|0)){continue}break}e=J[h+16>>2];}J[h+24>>2]=(g<<2)+j;J[h+20>>2]=a;J[h+16>>2]=d;if(!e){break Da}$a(e);}J[5996]=0;e=$(3,56)|0;a=J[5996];J[5996]=0;if((a|0)!=1){a=J[h+24>>2];J[h+24>>2]=0;b=J[h+16>>2];d=J[h+20>>2];J[h+16>>2]=0;J[h+20>>2]=0;J[e+16>>2]=0;J[e+20>>2]=0;J[e+12>>2]=a;J[e+4>>2]=b;J[e+8>>2]=d;J[e>>2]=21;J[e+24>>2]=0;J[e+28>>2]=0;J[e+32>>2]=0;J[e+36>>2]=0;J[e+40>>2]=0;J[e+44>>2]=0;J[5996]=0;aa(33,c|0,e|0);a=J[5996];J[5996]=0;if((a|0)!=1){break Ba}}c=Z()|0;break c}c=Z()|0;if(!e){break b}$a(e);break c}J[5996]=0;b=$(3,4)|0;a=J[5996];J[5996]=0;if((a|0)==1){c=Z()|0;break c}J[b>>2]=e;J[5996]=0;a=$(3,56)|0;d=J[5996];J[5996]=0;Fa:{Ga:{Ha:{Ia:{Ja:{Ka:{La:{Ma:{Na:{Oa:{Pa:{Qa:{Ra:{Sa:{Ta:{if((d|0)!=1){J[a+16>>2]=0;J[a+20>>2]=0;J[a+4>>2]=b;J[a>>2]=15;b=b+4|0;J[a+12>>2]=b;J[a+8>>2]=b;J[a+24>>2]=0;J[a+28>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[5996]=0;aa(33,c|0,a|0);b=J[5996];J[5996]=0;if((b|0)==1){break Fa}d=J[e+8>>2];f=J[e+12>>2];Ua:{if(d>>>0<f>>>0){J[d>>2]=a;J[e+8>>2]=d+4;break Ua}g=J[e+4>>2];j=d-g>>2;b=j+1|0;if(b>>>0>=1073741824){J[5996]=0;ca(34);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break Fa}f=f-g|0;i=f>>1;i=f>>>0>=2147483644?1073741823:b>>>0<i>>>0?i:b;Va:{if(!i){b=0;break Va}if(i>>>0>=1073741824){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break Fa}J[5996]=0;b=$(3,i<<2)|0;f=J[5996];J[5996]=0;if((f|0)==1){break Fa}}f=(j<<2)+b|0;J[f>>2]=a;a=f+4|0;if((d|0)!=(g|0)){while(1){f=f-4|0;d=d-4|0;J[f>>2]=J[d>>2];if((d|0)!=(g|0)){continue}break}d=J[e+4>>2];}J[e+12>>2]=(i<<2)+b;J[e+8>>2]=a;J[e+4>>2]=f;if(!d){break Ua}$a(d);}J[5996]=0;b=$(3,4)|0;a=J[5996];J[5996]=0;if((a|0)==1){c=Z()|0;break c}J[b>>2]=e;J[5996]=0;a=$(3,56)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Ta}J[a+16>>2]=0;J[a+20>>2]=0;J[a+4>>2]=b;J[a>>2]=17;b=b+4|0;J[a+12>>2]=b;J[a+8>>2]=b;J[a+24>>2]=0;J[a+28>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[5996]=0;aa(33,c|0,a|0);b=J[5996];J[5996]=0;if((b|0)==1){break Ga}d=J[e+8>>2];f=J[e+12>>2];Wa:{if(d>>>0<f>>>0){J[d>>2]=a;J[e+8>>2]=d+4;break Wa}g=J[e+4>>2];j=d-g>>2;b=j+1|0;if(b>>>0>=1073741824){J[5996]=0;ca(34);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break Ga}f=f-g|0;i=f>>1;i=f>>>0>=2147483644?1073741823:b>>>0<i>>>0?i:b;Xa:{if(!i){b=0;break Xa}if(i>>>0>=1073741824){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break Ga}J[5996]=0;b=$(3,i<<2)|0;f=J[5996];J[5996]=0;if((f|0)==1){break Ga}}f=(j<<2)+b|0;J[f>>2]=a;a=f+4|0;if((d|0)!=(g|0)){while(1){f=f-4|0;d=d-4|0;J[f>>2]=J[d>>2];if((d|0)!=(g|0)){continue}break}d=J[e+4>>2];}J[e+12>>2]=(i<<2)+b;J[e+8>>2]=a;J[e+4>>2]=f;if(!d){break Wa}$a(d);}J[5996]=0;a=J[6080];b=$(3,8)|0;d=J[5996];J[5996]=0;if((d|0)==1){c=Z()|0;break c}J[b+4>>2]=a;J[b>>2]=e;J[5996]=0;a=$(3,56)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Sa}J[a+16>>2]=0;J[a+20>>2]=0;J[a+4>>2]=b;J[a>>2]=10;b=b+8|0;J[a+12>>2]=b;J[a+8>>2]=b;J[a+24>>2]=0;J[a+28>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[5996]=0;aa(33,c|0,a|0);b=J[5996];J[5996]=0;if((b|0)==1){break Ha}d=J[e+8>>2];f=J[e+12>>2];Ya:{if(d>>>0<f>>>0){J[d>>2]=a;J[e+8>>2]=d+4;break Ya}g=J[e+4>>2];j=d-g>>2;b=j+1|0;if(b>>>0>=1073741824){J[5996]=0;ca(34);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break Ha}f=f-g|0;i=f>>1;i=f>>>0>=2147483644?1073741823:b>>>0<i>>>0?i:b;Za:{if(!i){b=0;break Za}if(i>>>0>=1073741824){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break Ha}J[5996]=0;b=$(3,i<<2)|0;f=J[5996];J[5996]=0;if((f|0)==1){break Ha}}f=(j<<2)+b|0;J[f>>2]=a;a=f+4|0;if((d|0)!=(g|0)){while(1){f=f-4|0;d=d-4|0;J[f>>2]=J[d>>2];if((d|0)!=(g|0)){continue}break}d=J[e+4>>2];}J[e+12>>2]=(i<<2)+b;J[e+8>>2]=a;J[e+4>>2]=f;if(!d){break Ya}$a(d);}J[5996]=0;b=$(3,4)|0;a=J[5996];J[5996]=0;if((a|0)==1){c=Z()|0;break c}J[b>>2]=e;J[5996]=0;a=$(3,56)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Ra}J[a+16>>2]=0;J[a+20>>2]=0;J[a+4>>2]=b;J[a>>2]=19;b=b+4|0;J[a+12>>2]=b;J[a+8>>2]=b;J[a+24>>2]=0;J[a+28>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[5996]=0;aa(33,c|0,a|0);b=J[5996];J[5996]=0;if((b|0)==1){break Ia}d=J[e+8>>2];f=J[e+12>>2];_a:{if(d>>>0<f>>>0){J[d>>2]=a;J[e+8>>2]=d+4;break _a}g=J[e+4>>2];j=d-g>>2;b=j+1|0;if(b>>>0>=1073741824){J[5996]=0;ca(34);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break Ia}f=f-g|0;i=f>>1;i=f>>>0>=2147483644?1073741823:b>>>0<i>>>0?i:b;$a:{if(!i){b=0;break $a}if(i>>>0>=1073741824){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break Ia}J[5996]=0;b=$(3,i<<2)|0;f=J[5996];J[5996]=0;if((f|0)==1){break Ia}}f=(j<<2)+b|0;J[f>>2]=a;a=f+4|0;if((d|0)!=(g|0)){while(1){f=f-4|0;d=d-4|0;J[f>>2]=J[d>>2];if((d|0)!=(g|0)){continue}break}d=J[e+4>>2];}J[e+12>>2]=(i<<2)+b;J[e+8>>2]=a;J[e+4>>2]=f;if(!d){break _a}$a(d);}J[5996]=0;b=$(3,4)|0;a=J[5996];J[5996]=0;if((a|0)==1){c=Z()|0;break c}J[b>>2]=e;J[5996]=0;a=$(3,56)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Qa}J[a+16>>2]=0;J[a+20>>2]=0;J[a+4>>2]=b;J[a>>2]=16;b=b+4|0;J[a+12>>2]=b;J[a+8>>2]=b;J[a+24>>2]=0;J[a+28>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[5996]=0;aa(33,c|0,a|0);b=J[5996];J[5996]=0;if((b|0)==1){break Ja}d=J[e+8>>2];f=J[e+12>>2];ab:{if(d>>>0<f>>>0){J[d>>2]=a;J[e+8>>2]=d+4;break ab}g=J[e+4>>2];j=d-g>>2;b=j+1|0;if(b>>>0>=1073741824){J[5996]=0;ca(34);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break Ja}f=f-g|0;i=f>>1;i=f>>>0>=2147483644?1073741823:b>>>0<i>>>0?i:b;bb:{if(!i){b=0;break bb}if(i>>>0>=1073741824){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break Ja}J[5996]=0;b=$(3,i<<2)|0;f=J[5996];J[5996]=0;if((f|0)==1){break Ja}}f=(j<<2)+b|0;J[f>>2]=a;a=f+4|0;if((d|0)!=(g|0)){while(1){f=f-4|0;d=d-4|0;J[f>>2]=J[d>>2];if((d|0)!=(g|0)){continue}break}d=J[e+4>>2];}J[e+12>>2]=(i<<2)+b;J[e+8>>2]=a;J[e+4>>2]=f;if(!d){break ab}$a(d);}J[5996]=0;b=$(3,4)|0;a=J[5996];J[5996]=0;if((a|0)==1){c=Z()|0;break c}J[b>>2]=e;J[5996]=0;a=$(3,56)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Pa}J[a+16>>2]=0;J[a+20>>2]=0;J[a+4>>2]=b;J[a>>2]=18;b=b+4|0;J[a+12>>2]=b;J[a+8>>2]=b;J[a+24>>2]=0;J[a+28>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[5996]=0;aa(33,c|0,a|0);b=J[5996];J[5996]=0;if((b|0)==1){break Ka}d=J[e+8>>2];f=J[e+12>>2];cb:{if(d>>>0<f>>>0){J[d>>2]=a;J[e+8>>2]=d+4;break cb}g=J[e+4>>2];j=d-g>>2;b=j+1|0;if(b>>>0>=1073741824){J[5996]=0;ca(34);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break Ka}f=f-g|0;i=f>>1;i=f>>>0>=2147483644?1073741823:b>>>0<i>>>0?i:b;db:{if(!i){b=0;break db}if(i>>>0>=1073741824){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break Ka}J[5996]=0;b=$(3,i<<2)|0;f=J[5996];J[5996]=0;if((f|0)==1){break Ka}}f=(j<<2)+b|0;J[f>>2]=a;a=f+4|0;if((d|0)!=(g|0)){while(1){f=f-4|0;d=d-4|0;J[f>>2]=J[d>>2];if((d|0)!=(g|0)){continue}break}d=J[e+4>>2];}J[e+12>>2]=(i<<2)+b;J[e+8>>2]=a;J[e+4>>2]=f;if(!d){break cb}$a(d);}J[5996]=0;a=J[6079];b=$(3,8)|0;d=J[5996];J[5996]=0;if((d|0)==1){c=Z()|0;break c}J[b+4>>2]=a;J[b>>2]=e;J[5996]=0;a=$(3,56)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Oa}J[a+16>>2]=0;J[a+20>>2]=0;J[a+4>>2]=b;J[a>>2]=11;J[a+48>>2]=0;J[a+52>>2]=1072693248;b=b+8|0;J[a+12>>2]=b;J[a+8>>2]=b;J[a+24>>2]=0;J[a+28>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[5996]=0;aa(33,c|0,a|0);b=J[5996];J[5996]=0;if((b|0)==1){break La}d=J[e+8>>2];f=J[e+12>>2];eb:{if(d>>>0<f>>>0){J[d>>2]=a;J[e+8>>2]=d+4;break eb}g=J[e+4>>2];j=d-g>>2;b=j+1|0;if(b>>>0>=1073741824){J[5996]=0;ca(34);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break La}f=f-g|0;i=f>>1;i=f>>>0>=2147483644?1073741823:b>>>0<i>>>0?i:b;fb:{if(!i){b=0;break fb}if(i>>>0>=1073741824){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break La}J[5996]=0;b=$(3,i<<2)|0;f=J[5996];J[5996]=0;if((f|0)==1){break La}}f=(j<<2)+b|0;J[f>>2]=a;a=f+4|0;if((d|0)!=(g|0)){while(1){f=f-4|0;d=d-4|0;J[f>>2]=J[d>>2];if((d|0)!=(g|0)){continue}break}d=J[e+4>>2];}J[e+12>>2]=(i<<2)+b;J[e+8>>2]=a;J[e+4>>2]=f;if(!d){break eb}$a(d);}J[5996]=0;b=$(3,4)|0;a=J[5996];J[5996]=0;if((a|0)==1){c=Z()|0;break c}J[b>>2]=e;J[5996]=0;a=$(3,56)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Na}J[a+16>>2]=0;J[a+20>>2]=0;J[a+4>>2]=b;J[a>>2]=14;b=b+4|0;J[a+12>>2]=b;J[a+8>>2]=b;J[a+24>>2]=0;J[a+28>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[5996]=0;aa(33,c|0,a|0);b=J[5996];J[5996]=0;if((b|0)==1){break Ma}d=J[e+8>>2];g=J[e+12>>2];if(d>>>0<g>>>0){J[d>>2]=a;J[e+8>>2]=d+4;break m}b=J[e+4>>2];f=d-b>>2;c=f+1|0;if(c>>>0>=1073741824){J[5996]=0;ca(34);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break Ma}g=g-b|0;i=g>>1;c=g>>>0>=2147483644?1073741823:c>>>0<i>>>0?i:c;gb:{if(!c){j=0;break gb}if(c>>>0>=1073741824){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break Ma}J[5996]=0;j=$(3,c<<2)|0;g=J[5996];J[5996]=0;if((g|0)==1){break Ma}}f=(f<<2)+j|0;J[f>>2]=a;a=f+4|0;if((b|0)!=(d|0)){while(1){f=f-4|0;d=d-4|0;J[f>>2]=J[d>>2];if((b|0)!=(d|0)){continue}break}d=J[e+4>>2];}J[e+12>>2]=(c<<2)+j;J[e+8>>2]=a;J[e+4>>2]=f;if(!d){break m}$a(d);break m}c=Z()|0;$a(b);break c}c=Z()|0;$a(b);break c}c=Z()|0;$a(b);break c}c=Z()|0;$a(b);break c}c=Z()|0;$a(b);break c}c=Z()|0;$a(b);break c}c=Z()|0;$a(b);break c}c=Z()|0;$a(b);break c}c=Z()|0;break c}c=Z()|0;break c}c=Z()|0;break c}c=Z()|0;break c}c=Z()|0;break c}c=Z()|0;break c}c=Z()|0;break c}c=Z()|0;break c}c=Z()|0;break c}a=J[h+16>>2];if(a){J[h+20>>2]=a;$a(a);}Ua=h+48|0;return e|0}c=Z()|0;break c}c=Z()|0;break c}c=Z()|0;break c}c=Z()|0;break c}c=Z()|0;break c}c=Z()|0;break c}c=Z()|0;break c}c=Z()|0;break c}c=Z()|0;}a=J[h+16>>2];if(!a){break b}J[h+20>>2]=a;$a(a);}da(c|0);B();}B();}function yn(a){a=a|0;var b=0,c=0,d=0,e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0;a:{b:{c:{if(J[a>>2]!=24){break c}c=J[a+32>>2];g=J[a+36>>2];if((c|0)==(g|0)){break c}while(1){j=J[J[c>>2]+80>>2];d:{if(f>>>0>i>>>0){J[i>>2]=j;i=i+4|0;break d}h=i-b>>2;d=h+1|0;if(d>>>0>=1073741824){J[5996]=0;ca(94);a=J[5996];J[5996]=0;if((a|0)==1){break b}break a}e=f-b|0;f=e>>1;f=e>>>0>=2147483644?1073741823:d>>>0<f>>>0?f:d;e:{if(!f){e=0;break e}if(f>>>0>=1073741824){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break b}J[5996]=0;e=$(3,f<<2)|0;d=J[5996];J[5996]=0;if((d|0)==1){break b}}d=(h<<2)+e|0;J[d>>2]=j;v=d;if((b|0)!=(i|0)){while(1){v=v-4|0;i=i-4|0;J[v>>2]=J[i>>2];if((b|0)!=(i|0)){continue}break}}f=(f<<2)+e|0;i=d+4|0;if(b){$a(b);}b=v;}c=c+8|0;if((g|0)!=(c|0)){continue}break}}j=J[a+36>>2];f=J[a+32>>2];c=j-f|0;f:{if((f|0)==(j|0)){d=0;h=0;break f}h=J[J[f>>2]+80>>2];if(c>>>0<9){d=0;break f}d=J[J[f+8>>2]+80>>2];if(c>>>0<17){break f}m=J[J[f+16>>2]+80>>2];}s=c>>3;k=J[a+24>>2];o=J[a+20>>2];e=k-o|0;u=J[a+12>>2];w=J[a+8>>2];x=u-w|0;g=(x|0)/12|0;n=J[a+4>>2];g:{h:{i:{j:{k:{l:{m:{n:{o:{p:{q:{r:{s:{t:{u:{b=J[a>>2];a=b&-2;switch(a-6|0){case 2:break t;case 0:break u;default:break s}}if((e|0)==32){break r}e=2369;a=39;g=3984;break h}if((e|0)==20){break r}e=2369;a=41;g=4224;break h}e=2369;if((k|0)!=(o|0)){a=43;g=4337;break h}if((a|0)!=4){break r}if((n|0)>0){break q}a=47;g=3134;break h}v:{switch(b-24|0){case 1:if(g>>>0>n-1>>>0){break p}e=2369;a=49;g=1397;break h;case 0:if(s>>>0>n-1>>>0){break n}e=2369;a=51;g=1654;break h;default:break v}}e=2369;if(n){a=53;g=4330;break h}if(b-17>>>0>5){break q}if((c|0)==16){break o}a=58;g=4003;break h}if(b-10>>>0>=7){if((b|0)!=23){break p}if((c|0)==24){break n}a=60;g=3946;break h}if((c|0)==8){break n}a=64;g=4090;break h}if((b|0)==24|(f|0)==(j|0)){break o}e=2369;a=66;g=4311;break h}if((a|0)==2){if((x|0)==12){break m}e=2369;a=70;g=4063;break h}if((b|0)!=25){break n}if(g-1>>>0<20){break l}e=2369;a=72;g=4240;break h}if((u|0)==(w|0)){break m}e=2369;a=74;g=4299;break h}c=10;f=2108;e=2369;a=247;g=2350;w:{switch(b|0){case 3:c=9;f=2119;break k;case 4:J[5996]=0;a=_(54,2295,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}J[5996]=0;b=_(54,2278,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}J[5996]=0;c=_(54,2129,6)|0;d=J[5996];J[5996]=0;if((d|0)==1){break g}b=c|(n&4194304?a:b);break i;case 5:J[5996]=0;a=_(54,2174,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}J[5996]=0;b=_(54,2170,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}J[5996]=0;c=_(54,2129,6)|0;d=J[5996];J[5996]=0;if((d|0)==1){break g}b=c|((n>>>0<5e8?b:0)|(n>>>0>499999999?a:0));break i;case 6:case 7:case 8:case 9:c=7;f=2153;break k;case 1:c=7;f=2136;break k;case 0:c=9;f=2098;break k;case 10:J[5996]=0;a=_(54,2777,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}J[5996]=0;b=_(54,3557,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}J[5996]=0;c=_(54,2161,5)|0;d=J[5996];J[5996]=0;if((d|0)==1){break g}J[5996]=0;d=_(54,1600,6)|0;e=J[5996];J[5996]=0;if((e|0)==1){break g}J[5996]=0;e=_(54,1239,1)|0;f=J[5996];J[5996]=0;if((f|0)==1){break g}b=(c|d)&h|(e|(b&(h^-1)?0:a));break i;case 11:J[5996]=0;a=_(54,2777,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}J[5996]=0;b=_(54,1958,2)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}J[5996]=0;c=_(54,2161,5)|0;d=J[5996];J[5996]=0;if((d|0)==1){break g}J[5996]=0;d=_(54,1197,7)|0;e=J[5996];J[5996]=0;if((e|0)==1){break g}b=(c|d)&h|(b&(h^-1)?0:a);break i;case 12:J[5996]=0;a=_(54,3557,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}J[5996]=0;b=_(54,3310,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}J[5996]=0;c=_(54,2161,5)|0;d=J[5996];J[5996]=0;if((d|0)==1){break g}J[5996]=0;d=_(54,2035,6)|0;e=J[5996];J[5996]=0;if((e|0)==1){break g}J[5996]=0;e=_(54,1419,2)|0;f=J[5996];J[5996]=0;if((f|0)==1){break g}b=(c|d)&h|(e|(b&(h^-1)?0:a));break i;case 13:J[5996]=0;b=_(54,3557,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;c=_(54,1028,2)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;a=_(54,1959,1)|0;d=J[5996];J[5996]=0;if((d|0)==1){break g}J[5996]=0;d=_(54,1029,1)|0;e=J[5996];J[5996]=0;if((e|0)==1){break g}J[5996]=0;e=_(54,2451,1)|0;f=J[5996];J[5996]=0;if((f|0)==1){break g}J[5996]=0;f=_(54,2315,1)|0;j=J[5996];J[5996]=0;if((j|0)==1){break g}J[5996]=0;j=_(54,2161,5)|0;g=J[5996];J[5996]=0;if((g|0)==1){break g}J[5996]=0;g=_(54,1640,2)|0;i=J[5996];J[5996]=0;if((i|0)==1){break g}J[5996]=0;i=_(54,1208,3)|0;n=J[5996];J[5996]=0;if((n|0)==1){break g}k=a;a=h^-1;b=(g|j)&h|(i|((a&d?0:k)|(a&c?0:b)|(a&f?0:e)));break i;case 14:J[5996]=0;a=_(54,2796,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}J[5996]=0;b=_(54,3557,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}J[5996]=0;c=_(54,2161,5)|0;d=J[5996];J[5996]=0;if((d|0)==1){break g}J[5996]=0;d=_(54,1486,5)|0;e=J[5996];J[5996]=0;if((e|0)==1){break g}J[5996]=0;e=_(54,1205,2)|0;f=J[5996];J[5996]=0;if((f|0)==1){break g}b=(c|d)&h|(e|(b&(h^-1)?0:a));break i;case 15:J[5996]=0;a=_(54,3557,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}J[5996]=0;b=_(54,2028,2)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}J[5996]=0;c=_(54,2451,1)|0;d=J[5996];J[5996]=0;if((d|0)==1){break g}J[5996]=0;d=_(54,2315,1)|0;e=J[5996];J[5996]=0;if((e|0)==1){break g}J[5996]=0;e=_(54,2161,5)|0;f=J[5996];J[5996]=0;if((f|0)==1){break g}J[5996]=0;f=_(54,1430,4)|0;j=J[5996];J[5996]=0;if((j|0)==1){break g}J[5996]=0;j=_(54,1208,3)|0;g=J[5996];J[5996]=0;if((g|0)==1){break g}g=c;c=h^-1;b=(e|f)&h|(j|((c&d?0:g)|(b&c?0:a)));break i;case 16:J[5996]=0;a=_(54,2161,5)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}J[5996]=0;b=_(54,1607,9)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}J[5996]=0;c=_(54,1149,2)|0;d=J[5996];J[5996]=0;if((d|0)==1){break g}b=c|(a|b)&h;break i;case 17:J[5996]=0;j=_(54,3540,3)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;g=_(54,2796,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;i=_(54,2029,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;n=_(54,2029,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;m=_(54,1029,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;s=_(54,1959,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;k=_(54,1029,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;o=_(54,1024,3)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;u=_(54,1681,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;a=_(54,2315,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}J[5996]=0;b=_(54,2315,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}e=h^-1;f=d^-1;x:{if(f&b){c=0;J[5996]=0;b=_(54,1681,1)|0;w=J[5996];J[5996]=0;if((w|0)==1){break g}if(b&e){break x}}c=a;}J[5996]=0;w=_(54,1149,2)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;x=_(54,2167,4)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;a=_(54,2165,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}J[5996]=0;q=_(54,2165,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}b=0;p=d&h;y:{if(q&(p^-1)){break y}J[5996]=0;b=_(54,2295,1)|0;q=J[5996];J[5996]=0;if((q|0)==1){break g}if(!(b&e)){b=0;J[5996]=0;q=_(54,2278,1)|0;r=J[5996];J[5996]=0;if((r|0)==1){break g}if(!(f&q)){break y}}J[5996]=0;b=_(54,2278,1)|0;q=J[5996];J[5996]=0;if((q|0)==1){break g}if(!(b&e)){b=0;J[5996]=0;q=_(54,2295,1)|0;r=J[5996];J[5996]=0;if((r|0)==1){break g}if(!(f&q)){break y}}J[5996]=0;b=_(54,2174,1)|0;q=J[5996];J[5996]=0;if((q|0)==1){break g}if(!(b&e)){b=0;J[5996]=0;q=_(54,2170,1)|0;r=J[5996];J[5996]=0;if((r|0)==1){break g}if(!(f&q)){break y}}J[5996]=0;b=_(54,2170,1)|0;q=J[5996];J[5996]=0;if((q|0)==1){break g}if(!(b&e)){b=0;J[5996]=0;q=_(54,2174,1)|0;r=J[5996];J[5996]=0;if((r|0)==1){break g}if(!(f&q)){break y}}b=a;}a=d|h;b=d&w|(o&p|(h&i|(e&g?0:d&j)|(e&m?0:d&n))|(k&(a^-1)?0:a&s)|a&u|c)|a&x|b;break i;case 18:J[5996]=0;i=_(54,3557,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;n=_(54,2777,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;m=_(54,1959,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;s=_(54,1029,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;k=_(54,2029,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;o=_(54,2029,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;u=_(54,1029,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;w=_(54,2451,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;x=_(54,1681,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;q=_(54,2031,3)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;a=_(54,2315,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}J[5996]=0;e=_(54,2315,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}b=d^-1;c=h^-1;f=d&h;g=f^-1;z:{A:{if(!(g&e)){break A}J[5996]=0;e=_(54,2297,2)|0;j=J[5996];J[5996]=0;if((j|0)==1){break g}if(!(c&e)){break A}j=0;J[5996]=0;e=_(54,2297,2)|0;p=J[5996];J[5996]=0;if((p|0)==1){break g}if(b&e){break z}}j=a;}J[5996]=0;p=_(54,1681,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;r=_(54,1149,2)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;y=_(54,2167,4)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;a=_(54,2165,1)|0;e=J[5996];J[5996]=0;if((e|0)==1){break g}e=0;J[5996]=0;l=_(54,2165,1)|0;t=J[5996];J[5996]=0;if((t|0)==1){break g}B:{if(g&l){break B}J[5996]=0;e=_(54,2295,1)|0;l=J[5996];J[5996]=0;if((l|0)==1){break g}if(!(c&e)){e=0;J[5996]=0;l=_(54,2278,1)|0;t=J[5996];J[5996]=0;if((t|0)==1){break g}if(!(b&l)){break B}}J[5996]=0;e=_(54,2278,1)|0;l=J[5996];J[5996]=0;if((l|0)==1){break g}if(!(c&e)){e=0;J[5996]=0;l=_(54,2295,1)|0;t=J[5996];J[5996]=0;if((t|0)==1){break g}if(!(b&l)){break B}}J[5996]=0;e=_(54,2174,1)|0;l=J[5996];J[5996]=0;if((l|0)==1){break g}if(!(c&e)){e=0;J[5996]=0;l=_(54,2170,1)|0;t=J[5996];J[5996]=0;if((t|0)==1){break g}if(!(b&l)){break B}}J[5996]=0;e=_(54,2170,1)|0;l=J[5996];J[5996]=0;if((l|0)==1){break g}if(!(c&e)){e=0;J[5996]=0;l=_(54,2174,1)|0;t=J[5996];J[5996]=0;if((t|0)==1){break g}if(!(b&l)){break B}}e=a;}a=d|h;b=r|(f&q|(h&k|(b&n?0:h&i)|(s&(a^-1)?0:a&m)|(c&u?0:d&o))|(g&x?0:f&w)|j|a&p)|a&y|e;break i;case 19:J[5996]=0;a=_(54,3557,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}J[5996]=0;b=_(54,2504,2)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}c=0;if(!((h^-1)&b)){J[5996]=0;b=_(54,2501,2)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}c=b&(d^-1)?0:a;}J[5996]=0;e=_(54,1959,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;j=_(54,1029,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;g=_(54,2049,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}f=0;J[5996]=0;b=_(54,1681,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}a=d&h;k=b;b=d|h;d=b^-1;if(!(k&d)){J[5996]=0;f=_(54,2451,1)|0;h=J[5996];J[5996]=0;if((h|0)==1){break g}f=f&(a^-1)?0:a&g;}J[5996]=0;g=_(54,2340,3)|0;h=J[5996];J[5996]=0;if((h|0)==1){break g}J[5996]=0;h=_(54,1148,3)|0;i=J[5996];J[5996]=0;if((i|0)==1){break g}J[5996]=0;i=_(54,2167,4)|0;n=J[5996];J[5996]=0;if((n|0)==1){break g}J[5996]=0;n=_(54,2165,1)|0;m=J[5996];J[5996]=0;if((m|0)==1){break g}b=a&n|(b&i|(h|(a&g|((d&j?0:b&e)|c|f))));break i;case 21:J[5996]=0;e=_(54,3557,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;f=_(54,1293,3)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;j=_(54,1959,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;g=_(54,1029,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;a=_(54,2049,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}J[5996]=0;c=_(54,2451,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}b=d&h;i=h^-1;C:{if(i&c){a=d|h;c=0;break C}J[5996]=0;c=_(54,1681,1)|0;n=J[5996];J[5996]=0;if((n|0)==1){break g}k=a&b;a=d|h;c=c&(a^-1)?0:k;}J[5996]=0;n=_(54,1650,3)|0;m=J[5996];J[5996]=0;if((m|0)==1){break g}J[5996]=0;m=_(54,2468,3)|0;s=J[5996];J[5996]=0;if((s|0)==1){break g}J[5996]=0;s=_(54,1239,1)|0;k=J[5996];J[5996]=0;if((k|0)==1){break g}J[5996]=0;k=_(54,2167,4)|0;o=J[5996];J[5996]=0;if((o|0)==1){break g}J[5996]=0;o=_(54,2165,1)|0;u=J[5996];J[5996]=0;if((u|0)==1){break g}b=b&o|(a&k|(s|(d&m|(b&n|(c|((g&(d^-1)?0:h&j)|(f&i?0:d&e)))))));break i;case 20:J[5996]=0;e=_(54,2796,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;f=_(54,1293,3)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;j=_(54,1959,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;g=_(54,1029,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;a=_(54,2049,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}J[5996]=0;c=_(54,2451,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}b=d&h;i=h^-1;D:{if(i&c){a=d|h;c=0;break D}J[5996]=0;c=_(54,1681,1)|0;n=J[5996];J[5996]=0;if((n|0)==1){break g}k=a&b;a=d|h;c=c&(a^-1)?0:k;}J[5996]=0;n=_(54,1394,2)|0;m=J[5996];J[5996]=0;if((m|0)==1){break g}J[5996]=0;m=_(54,1205,2)|0;s=J[5996];J[5996]=0;if((s|0)==1){break g}J[5996]=0;s=_(54,2167,4)|0;k=J[5996];J[5996]=0;if((k|0)==1){break g}J[5996]=0;k=_(54,2165,1)|0;o=J[5996];J[5996]=0;if((o|0)==1){break g}b=b&k|(a&s|(m|(b&n|(c|((g&(d^-1)?0:h&j)|(f&i?0:d&e))))));break i;case 22:J[5996]=0;e=_(54,1643,6)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;b=_(54,1959,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;f=_(54,1029,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;j=_(54,2451,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;g=_(54,2315,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;i=_(54,2049,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;n=_(54,1681,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;m=_(54,2522,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;s=_(54,1239,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;k=_(54,2167,4)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;o=_(54,2165,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}a=d|h;c=a^-1;l=b;b=d&h;b=a&k|(s|(a&m|((f&(b^-1)?0:l)|b&e|(c&g?0:a&j)))|(c&n?0:b&i))|b&o;break i;case 24:break j;case 2:break k;case 25:break l;case 23:break w;default:break h}}J[5996]=0;s=_(54,2779,3)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;k=_(54,1293,3)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;o=_(54,1029,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;u=_(54,1959,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;w=_(54,1029,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;x=_(54,1295,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;a=_(54,2315,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}J[5996]=0;b=_(54,1681,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}c=h^-1;E:{if(c&b){e=0;J[5996]=0;b=_(54,2315,1)|0;f=J[5996];J[5996]=0;if((f|0)==1){break g}if((d^-1)&b){break E}}e=a&m;}J[5996]=0;q=_(54,2522,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;a=_(54,2451,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}J[5996]=0;b=_(54,1681,1)|0;f=J[5996];J[5996]=0;if((f|0)==1){break g}F:{if(b&c){j=0;J[5996]=0;b=_(54,2315,1)|0;f=J[5996];J[5996]=0;if((f|0)==1){break g}if((d^-1)&b){break F}}j=a&h&m;}J[5996]=0;a=_(54,2049,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}J[5996]=0;b=_(54,2451,1)|0;f=J[5996];J[5996]=0;if((f|0)==1){break g}n=d&h&m;G:{if(b&c){g=d|h;i=m|g;a=0;break G}J[5996]=0;b=_(54,1681,1)|0;f=J[5996];J[5996]=0;if((f|0)==1){break g}g=d|h;i=m|g;a=b&(i^-1)?0:a&n;}J[5996]=0;p=_(54,1681,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}J[5996]=0;r=_(54,1239,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}J[5996]=0;y=_(54,2167,4)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}J[5996]=0;b=_(54,2165,1)|0;f=J[5996];J[5996]=0;if((f|0)==1){break g}f=0;J[5996]=0;l=_(54,2165,1)|0;t=J[5996];J[5996]=0;if((t|0)==1){break g}H:{if(l&(n^-1)){break H}J[5996]=0;f=_(54,2295,1)|0;l=J[5996];J[5996]=0;if((l|0)==1){break g}if(!(c&f)){f=0;J[5996]=0;l=_(54,2278,1)|0;t=J[5996];J[5996]=0;if((t|0)==1){break g}if(!((d^-1)&l)){break H}}J[5996]=0;f=_(54,2278,1)|0;l=J[5996];J[5996]=0;if((l|0)==1){break g}if(!(c&f)){f=0;J[5996]=0;l=_(54,2295,1)|0;t=J[5996];J[5996]=0;if((t|0)==1){break g}if(!((d^-1)&l)){break H}}J[5996]=0;f=_(54,2174,1)|0;l=J[5996];J[5996]=0;if((l|0)==1){break g}if(!(c&f)){f=0;J[5996]=0;l=_(54,2170,1)|0;t=J[5996];J[5996]=0;if((t|0)==1){break g}if(!((d^-1)&l)){break H}}J[5996]=0;f=_(54,2170,1)|0;l=J[5996];J[5996]=0;if((l|0)==1){break g}if(!(c&f)){f=0;J[5996]=0;l=_(54,2174,1)|0;t=J[5996];J[5996]=0;if((t|0)==1){break g}if(!((d^-1)&l)){break H}}f=b;}b=d&m;d=b|h;b=i&y|(g&p&m|(r|(a|(m&q|(b&x|(n&o|(c&k?0:b&s))|(w&(d^-1)?0:d&u)|e)|j))))|f;break i}c=8;f=2144;}J[5996]=0;b=_(54,f|0,c|0)|0;a=J[5996];J[5996]=0;if((a|0)!=1){break i}break g}f=0;J[5996]=0;j=1;c=_(54,2165,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}I:{if((i|0)==(v|0)){a=1;h=0;break I}e=J[v>>2];b=0;J[5996]=0;a=_(54,1293,3)|0;d=J[5996];J[5996]=0;if((d|0)==1){break g}d=e^-1;if(d&a){break i}J[5996]=0;g=_(54,2451,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;h=_(54,2049,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;m=_(54,1681,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;a=_(54,1029,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}if(a&d){J[5996]=0;a=_(54,1959,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}f=a&d?2:1;}J[5996]=0;k=_(54,2167,4)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;a=_(54,2165,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}J[5996]=0;j=_(54,2165,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}b=0;J:{if(j&(c&e^-1)){break J}K:{if(n>>>0<2){break K}J[5996]=0;b=_(54,2295,1)|0;j=J[5996];J[5996]=0;if((j|0)==1){break g}j=c^-1;if(!(j&b)){b=0;J[5996]=0;o=_(54,2278,1)|0;u=J[5996];J[5996]=0;if((u|0)==1){break g}if(!(d&o)){break J}}J[5996]=0;b=_(54,2278,1)|0;o=J[5996];J[5996]=0;if((o|0)==1){break g}if(!(b&j)){b=0;J[5996]=0;o=_(54,2295,1)|0;u=J[5996];J[5996]=0;if((u|0)==1){break g}if(!(d&o)){break J}}J[5996]=0;b=_(54,2174,1)|0;o=J[5996];J[5996]=0;if((o|0)==1){break g}if(!(b&j)){b=0;J[5996]=0;o=_(54,2170,1)|0;u=J[5996];J[5996]=0;if((u|0)==1){break g}if(!(d&o)){break J}}J[5996]=0;b=_(54,2170,1)|0;o=J[5996];J[5996]=0;if((o|0)==1){break g}if(b&j){break K}b=0;J[5996]=0;j=_(54,2174,1)|0;o=J[5996];J[5996]=0;if((o|0)==1){break g}if(!(d&j)){break J}}b=a;}a=!(d&g);j=!(d&h);h=!(d&m);c=k&(c|e)|b;b=i-v|0;if(b>>>0<5){break I}b=b>>2;o=b>>>0<=2?2:b;e=1;while(1){m=J[(e<<2)+v>>2];J[5996]=0;b=_(54,1289,3)|0;d=J[5996];J[5996]=0;if((d|0)==1){break b}i=m^-1;if(i&b){b=0;break i}J[5996]=0;u=_(54,2451,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break b}J[5996]=0;w=_(54,2049,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break b}J[5996]=0;x=_(54,1681,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break b}J[5996]=0;b=_(54,1029,1)|0;d=J[5996];J[5996]=0;if((d|0)==1){break b}if(b&i){J[5996]=0;b=_(54,1959,1)|0;d=J[5996];J[5996]=0;if((d|0)==1){break b}b=b&i?2:1;}else {b=0;}J[5996]=0;q=_(54,2167,4)|0;d=J[5996];J[5996]=0;if((d|0)==1){break b}J[5996]=0;g=_(54,2165,1)|0;d=J[5996];J[5996]=0;if((d|0)==1){break b}J[5996]=0;k=_(54,2165,1)|0;d=J[5996];J[5996]=0;if((d|0)==1){break b}d=0;L:{if(k&(c&m^-1)){break L}M:{if(n>>>0<2){break M}J[5996]=0;k=_(54,2295,1)|0;p=J[5996];J[5996]=0;if((p|0)==1){break b}l=k;k=c^-1;if(!(l&k)){J[5996]=0;p=_(54,2278,1)|0;r=J[5996];J[5996]=0;if((r|0)==1){break b}if(!(i&p)){break L}}J[5996]=0;p=_(54,2278,1)|0;r=J[5996];J[5996]=0;if((r|0)==1){break b}if(!(k&p)){J[5996]=0;p=_(54,2295,1)|0;r=J[5996];J[5996]=0;if((r|0)==1){break b}if(!(i&p)){break L}}J[5996]=0;p=_(54,2174,1)|0;r=J[5996];J[5996]=0;if((r|0)==1){break b}if(!(k&p)){J[5996]=0;p=_(54,2170,1)|0;r=J[5996];J[5996]=0;if((r|0)==1){break b}if(!(i&p)){break L}}J[5996]=0;p=_(54,2170,1)|0;r=J[5996];J[5996]=0;if((r|0)==1){break b}if(k&p){break M}J[5996]=0;k=_(54,2174,1)|0;p=J[5996];J[5996]=0;if((p|0)==1){break b}if(!(i&k)){break L}}d=g;}a=!(i&u)&a;j=!(i&w)&j;h=!(i&x)+h|0;f=b+f|0;c=q&(c|m)|d;e=e+1|0;if((o|0)!=(e|0)){continue}break}}J[5996]=0;b=_(54,1293,3)|0;d=J[5996];J[5996]=0;if((d|0)==1){break g}J[5996]=0;d=_(54,1029,1)|0;e=J[5996];J[5996]=0;if((e|0)==1){break g}J[5996]=0;e=_(54,1959,1)|0;g=J[5996];J[5996]=0;if((g|0)==1){break g}J[5996]=0;g=_(54,2451,1)|0;i=J[5996];J[5996]=0;if((i|0)==1){break g}J[5996]=0;i=_(54,2049,1)|0;m=J[5996];J[5996]=0;if((m|0)==1){break g}J[5996]=0;m=_(54,1681,1)|0;k=J[5996];J[5996]=0;if((k|0)==1){break g}b=b|c|(f?0:d)|((f|0)==1?e:0)|(a?(h|0)==(s|0)?g:0:0);c=a&j?i:0;a=s-n|0;b=b|(a>>>0<=h>>>0?c:0)|(a+1>>>0<=h>>>0?m:0);}J[5996]=0;c=_(54,3310,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;d=_(54,2796,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;f=_(54,3557,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}J[5996]=0;j=_(54,2777,1)|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}e=2356;a=19;g=4075;N:{O:{k=d;d=b^-1;c=((!(k&d)+!(c&d)|0)+!(d&f)|0)+!(d&j)|0;switch(c|0){case 0:break N;case 1:break O;default:break h}}J[5996]=0;a=_(54,1029,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}P:{if(a&d){break P}J[5996]=0;a=_(54,1959,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}if(a&d){break P}a=20;g=4494;break h}J[5996]=0;a=_(54,2029,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}Q:{if(a&d){break Q}J[5996]=0;a=_(54,1029,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}if(a&d){break Q}a=21;g=4356;break h}J[5996]=0;a=_(54,2029,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}R:{if(a&d){break R}J[5996]=0;a=_(54,2777,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}if(a&d){break R}a=22;g=4771;break h}J[5996]=0;a=_(54,2796,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}S:{if(a&d){break S}J[5996]=0;a=_(54,2522,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}if(a&d){break S}a=23;g=4702;break h}J[5996]=0;a=_(54,3310,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}T:{if(a&d){break T}J[5996]=0;a=_(54,1295,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}if(!(a&d)){break T}a=24;g=4426;break h}J[5996]=0;a=_(54,2796,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}U:{if(a&d){break U}J[5996]=0;a=_(54,1295,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}if(a&d){break U}a=25;g=4391;break h}J[5996]=0;a=_(54,2451,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}V:{if(a&d){break V}J[5996]=0;a=_(54,2315,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}if(a&d){break V}a=26;g=4563;break h}J[5996]=0;a=_(54,2451,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}W:{if(a&d){break W}J[5996]=0;a=_(54,2522,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}if(!(a&d)){break W}a=27;g=4737;break h}J[5996]=0;a=_(54,2796,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}X:{if(a&d){break X}J[5996]=0;a=_(54,2451,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}if(a&d){break X}a=28;g=4667;break h}J[5996]=0;a=_(54,2522,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}Y:{if(a&d){break Y}J[5996]=0;a=_(54,2315,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}if(a&d){break Y}a=29;g=4598;break h}J[5996]=0;a=_(54,2796,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}Z:{if(a&d){break Z}J[5996]=0;a=_(54,2315,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}if(!(a&d)){break Z}a=30;g=4633;break h}J[5996]=0;a=_(54,3310,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}_:{if(a&d){break _}J[5996]=0;a=_(54,1681,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}if(!(a&d)){break _}a=31;g=4460;break h}J[5996]=0;a=_(54,1029,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}if(a&d){c=b;break N}J[5996]=0;a=_(54,2049,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break g}c=b;if(!(a&d)){break N}a=32;g=4529;break h}if(v){$a(v);}return c|0}J[5996]=0;ga(44,g|0,1859,a|0,e|0);a=J[5996];J[5996]=0;if((a|0)!=1){break a}}}a=Z()|0;if(v){$a(v);}da(a|0);B();}B();}function jn(a,b,c,d,e,f,g,h,i,j,k){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;i=i|0;j=j|0;k=k|0;var l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0;l=Ua-304|0;Ua=l;a:{b:{c:{d:{e:{f:{g:{h:{i:{j:{k:{l:{m:{n:{o:{p:{q:{r:{s:{m=Gb(a);if(m>>>0>=2147483632){J[5996]=0;ca(2);a=J[5996];J[5996]=0;if((a|0)==1){break s}break a}t:{if(m>>>0<=10){H[l+155|0]=m;n=l+144|0;break t}J[5996]=0;o=(m|15)+1|0;n=$(3,o|0)|0;p=J[5996];J[5996]=0;if((p|0)==1){break s}J[l+144>>2]=n;J[l+148>>2]=m;J[l+152>>2]=o|-2147483648;}H[fb(n,a,m)+m|0]=0;q=J[l+144>>2];a=K[l+155|0];r=a<<24>>24;m=(r|0)<0;p=m?q:l+144|0;s=m?J[l+148>>2]:a;m=p+s|0;n=-1;u:{while(1){if((m|0)==(p|0)){break u}m=m-1|0;o=K[m|0];v=o>>>0<=63;a=o&31;if((o&63)>>>0>=32){o=1<<a;t=0;}else {t=1<<a;o=t-1&1>>>32-a;}a=t&9728;if(v&(o&1|(a|0)!=0)){continue}break}n=m-p|0;}a=n+1|0;if(a>>>0>s>>>0){J[5996]=0;ca(108);a=J[5996];J[5996]=0;if((a|0)!=1){break a}m=fa(22864)|0;a=Va;break m}v:{if((r|0)<0){J[l+148>>2]=a;break v}H[l+155|0]=a;q=l+144|0;}H[a+q|0]=0;J[l+12>>2]=0;J[l+16>>2]=0;w:{if(H[l+155|0]>=0){J[l+120>>2]=J[l+152>>2];a=J[l+148>>2];J[l+112>>2]=J[l+144>>2];J[l+116>>2]=a;break w}J[5996]=0;ea(40,l+112|0,J[l+144>>2],J[l+148>>2]);a=J[5996];J[5996]=0;if((a|0)==1){break r}}J[5996]=0;m=l+128|0;aa(109,m|0,l+112|0);a=J[5996];J[5996]=0;if((a|0)==1){break q}n=H[l+139|0];a=(n|0)<0;J[l+180>>2]=a?J[l+132>>2]:n&255;J[l+176>>2]=a?J[l+128>>2]:m;J[5996]=0;aa(29,l+248|0,l+176|0);a=J[5996];J[5996]=0;x:{y:{z:{A:{if((a|0)!=1){if(!J[l+180>>2]){break A}J[l+240>>2]=0;J[l+244>>2]=0;J[l+232>>2]=0;J[l+236>>2]=0;J[l+224>>2]=0;J[l+228>>2]=0;J[l+216>>2]=0;J[l+220>>2]=0;J[l+208>>2]=0;J[l+212>>2]=0;J[l+200>>2]=0;J[l+204>>2]=0;J[l+192>>2]=0;J[l+196>>2]=0;break z}m=Ra(22528,22864)|0;a=Va;if((a|0)!=(wa(22528)|0)){break x}pa(m|0)|0;J[l+240>>2]=0;J[l+244>>2]=0;J[l+232>>2]=0;J[l+236>>2]=0;J[l+224>>2]=0;J[l+228>>2]=0;J[l+216>>2]=0;J[l+220>>2]=0;J[l+208>>2]=0;J[l+212>>2]=0;J[l+200>>2]=0;J[l+204>>2]=0;J[l+192>>2]=0;J[l+196>>2]=0;J[5996]=0;ca(110);a=J[5996];J[5996]=0;if((a|0)!=1){break y}m=fa(22864)|0;a=Va;break x}a=J[l+260>>2];m=J[l+264>>2];J[l+260>>2]=0;J[l+264>>2]=0;J[l+204>>2]=a;J[l+208>>2]=m;a=J[l+268>>2];m=J[l+272>>2];J[l+268>>2]=0;J[l+272>>2]=0;J[l+212>>2]=a;J[l+216>>2]=m;J[l+228>>2]=J[l+284>>2];a=J[l+288>>2];J[l+284>>2]=0;J[l+288>>2]=0;J[l+192>>2]=J[l+248>>2];m=J[l+252>>2];n=J[l+256>>2];J[l+252>>2]=0;J[l+256>>2]=0;J[l+196>>2]=m;J[l+200>>2]=n;m=J[l+280>>2];J[l+220>>2]=J[l+276>>2];J[l+224>>2]=m;J[l+276>>2]=0;J[l+280>>2]=0;J[l+232>>2]=a;J[l+236>>2]=J[l+292>>2];J[l+292>>2]=0;a=J[l+296>>2];J[l+296>>2]=0;J[l+240>>2]=a;J[l+244>>2]=J[l+300>>2];}Fb(l+248|0);}if(!J[l+192>>2]){n=0;break o}J[l+184>>2]=0;J[l+176>>2]=0;J[l+180>>2]=0;J[l+256>>2]=0;J[l+260>>2]=0;J[5996]=0;J[l+248>>2]=0;J[l+252>>2]=0;J[l+264>>2]=1065353216;o=ba(111,l+192|0,l+248|0,l+176|0)|0;a=J[5996];J[5996]=0;B:{if((a|0)!=1){m=J[l+256>>2];if(m){while(1){a=J[m>>2];$a(m);m=a;if(a){continue}break}}a=J[l+248>>2];n=0;J[l+248>>2]=0;if(a){$a(a);}C:{D:{if(o){J[l+252>>2]=0;J[l+256>>2]=0;J[5996]=0;J[l+248>>2]=l+252;a=qa(50,o|0,1,0,l+248|0)|0;m=J[5996];J[5996]=0;if((m|0)==1){break D}J[5996]=0;m=_(54,1639,3)|0;n=J[5996];J[5996]=0;E:{F:{G:{H:{if((n|0)==1){break H}n=J[a>>2];a=J[a+4>>2];J[5996]=0;oa(51,l+160|0,n|0,a|0,m|0,0);a=J[5996];J[5996]=0;if((a|0)==1){break H}m=J[l+164>>2];n=J[l+160>>2];o=m-n|0;if((o|0)!=32){break G}m=J[n>>2];p=J[n+4>>2];J[n>>2]=0;J[n+4>>2]=0;a=J[l+16>>2];J[l+12>>2]=m;J[l+16>>2]=p;I:{if(!a){break I}m=J[a+4>>2];J[a+4>>2]=m-1;if(m){break I}Xa[J[J[a>>2]+8>>2]](a);bb(a);}n=J[l+160>>2];u=O[n+8>>3];m=J[l+164>>2];break F}m=fa(22864)|0;break C}if(!n){break E}}if((m|0)!=(n|0)){while(1){m=m-32|0;a=J[m+4>>2];J:{if(!a){break J}p=J[a+4>>2];J[a+4>>2]=p-1;if(p){break J}Xa[J[J[a>>2]+8>>2]](a);bb(a);}if((m|0)!=(n|0)){continue}break}m=J[l+160>>2];}J[l+164>>2]=n;$a(m);}xd(J[l+252>>2]);n=(o|0)==32;}a=J[l+176>>2];if(!a){break o}m=J[l+180>>2];if((a|0)!=(m|0)){while(1){m=m-4|0;ad(m);if((a|0)!=(m|0)){continue}break}a=J[l+176>>2];}$a(a);break o}m=fa(22864)|0;}a=Va;xd(J[l+252>>2]);break B}m=fa(22864)|0;a=Va;h=J[l+256>>2];if(h){while(1){i=J[h>>2];$a(h);h=i;if(h){continue}break}}h=J[l+248>>2];J[l+248>>2]=0;if(h){$a(h);}}h=J[l+176>>2];if(h){j=J[l+180>>2];i=h;if((j|0)!=(h|0)){while(1){j=j-4|0;ad(j);if((h|0)!=(j|0)){continue}break}i=J[l+176>>2];}J[l+180>>2]=h;$a(i);}Fb(l+192|0);}if(H[l+139|0]>=0){break p}$a(J[l+128>>2]);break p}m=fa(22864)|0;a=Va;break l}m=fa(22864)|0;a=Va;break n}m=fa(22864)|0;a=Va;}if(H[l+123|0]>=0){break n}$a(J[l+112>>2]);break n}Fb(l+192|0);if(H[l+139|0]<0){$a(J[l+128>>2]);}if(H[l+123|0]<0){$a(J[l+112>>2]);}K:{L:{M:{N:{O:{P:{Q:{R:{if(!n){J[5996]=0;a=$(3,16)|0;h=J[5996];J[5996]=0;S:{T:{if((h|0)!=1){H[a+15|0]=0;h=K[2587]|K[2588]<<8|(K[2589]<<16|K[2590]<<24);i=K[2583]|K[2584]<<8|(K[2585]<<16|K[2586]<<24);H[a+7|0]=i;H[a+8|0]=i>>>8;H[a+9|0]=i>>>16;H[a+10|0]=i>>>24;H[a+11|0]=h;H[a+12|0]=h>>>8;H[a+13|0]=h>>>16;H[a+14|0]=h>>>24;h=K[2580]|K[2581]<<8|(K[2582]<<16|K[2583]<<24);i=K[2576]|K[2577]<<8|(K[2578]<<16|K[2579]<<24);H[a|0]=i;H[a+1|0]=i>>>8;H[a+2|0]=i>>>16;H[a+3|0]=i>>>24;H[a+4|0]=h;H[a+5|0]=h>>>8;H[a+6|0]=h>>>16;H[a+7|0]=h>>>24;h=((c|0)>=16?16:c)-1|0;H[fb(b,a,h)+h|0]=0;$a(a);J[5996]=0;k=$(3,16)|0;a=J[5996];J[5996]=0;if((a|0)==1){break T}H[k+15|0]=0;h=K[2587]|K[2588]<<8|(K[2589]<<16|K[2590]<<24);i=h;a=K[2583]|K[2584]<<8|(K[2585]<<16|K[2586]<<24);H[k+7|0]=a;n=a>>>8|0;H[k+8|0]=n;o=a>>>16|0;H[k+9|0]=o;p=a>>>24|0;H[k+10|0]=p;H[k+11|0]=h;H[k+12|0]=h>>>8;H[k+13|0]=h>>>16;H[k+14|0]=h>>>24;j=K[2580]|K[2581]<<8|(K[2582]<<16|K[2583]<<24);h=K[2576]|K[2577]<<8|(K[2578]<<16|K[2579]<<24);H[k|0]=h;H[k+1|0]=h>>>8;H[k+2|0]=h>>>16;H[k+3|0]=h>>>24;H[k+4|0]=j;H[k+5|0]=j>>>8;H[k+6|0]=j>>>16;H[k+7|0]=j>>>24;m=((e|0)>=16?16:e)-1|0;H[fb(d,k,m)+m|0]=0;$a(k);J[5996]=0;k=$(3,16)|0;m=J[5996];J[5996]=0;if((m|0)==1){break S}H[k+15|0]=0;H[k+7|0]=a;H[k+8|0]=n;H[k+9|0]=o;H[k+10|0]=p;a=i;H[k+11|0]=a;H[k+12|0]=a>>>8;H[k+13|0]=a>>>16;H[k+14|0]=a>>>24;H[k|0]=h;H[k+1|0]=h>>>8;H[k+2|0]=h>>>16;H[k+3|0]=h>>>24;a=j;H[k+4|0]=a;H[k+5|0]=a>>>8;H[k+6|0]=a>>>16;H[k+7|0]=a>>>24;a=((g|0)>=16?16:g)-1|0;H[fb(f,k,a)+a|0]=0;$a(k);break R}m=fa(22864)|0;a=Va;break n}m=fa(22864)|0;a=Va;break n}m=fa(22864)|0;a=Va;break n}J[5996]=0;a=J[l+12>>2];J[l+200>>2]=0;J[l+192>>2]=0;J[l+196>>2]=0;p=$(3,12)|0;m=J[5996];J[5996]=0;if((m|0)==1){break K}H[p+8|0]=0;J[p+4>>2]=0;J[p>>2]=a;m=p+12|0;t=m;U:{V:{W:{while(1){X:{r=m-12|0;o=J[r+4>>2];a=J[r>>2];n=J[a+32>>2];s=J[a+36>>2]-n>>3;Y:{if(s>>>0>o>>>0){s=1;J[r+4>>2]=o+1;a=J[a>>2];Z:{if(a-10>>>0<7){break Z}_:{switch(a-17|0){case 0:s=J[J[n+8>>2]>>2]==1;break Z;case 5:if(!J[J[n>>2]>>2]){break Z}s=!J[J[n+8>>2]>>2];break Z;default:break _}}s=0;}a=J[n+(o<<3)>>2];if(m>>>0<t>>>0){H[m+8|0]=s;J[m+4>>2]=0;J[m>>2]=a;m=m+12|0;break Y}$:{r=(m-p|0)/12|0;n=r+1|0;aa:{if(n>>>0>=357913942){J[5996]=0;ca(112);break aa}o=(t-p|0)/12|0;q=o<<1;n=o>>>0>=178956970?357913941:n>>>0<q>>>0?q:n;if(n>>>0<357913942){break $}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break a}m=fa(22864)|0;break M}J[5996]=0;q=P(n,12);o=$(3,q|0)|0;n=J[5996];J[5996]=0;if((n|0)==1){break N}n=o+P(r,12)|0;H[n+8|0]=s;J[n+4>>2]=0;J[n>>2]=a;a=n;if((m|0)!=(p|0)){while(1){m=m-12|0;r=J[m+4>>2];a=a-12|0;J[a>>2]=J[m>>2];J[a+4>>2]=r;J[a+8>>2]=J[m+8>>2];if((m|0)!=(p|0)){continue}break}}t=o+q|0;m=n+12|0;if(p){$a(p);}p=a;break Y}q=J[l+192>>2];n=(J[l+196>>2]-q|0)/12|0;if(n>>>0<s>>>0){J[5996]=0;ga(44,5135,2210,439,2439);a=J[5996];J[5996]=0;if((a|0)!=1){break a}m=fa(22864)|0;break M}m=K[m-4|0];J[5996]=0;oa(113,l+248|0,m|0,a|0,(P(n,12)+q|0)+P(s,-12)|0,s|0);m=J[5996];J[5996]=0;ba:{if((m|0)!=1){if(K[l+260|0]){break ba}s=0;break U}m=fa(22864)|0;break M}n=J[l+196>>2];m=J[a+36>>2]-J[a+32>>2]>>3;a=J[l+192>>2];o=((n+P(m,-12)|0)-a|0)/12|0;ca:{if(!P(m,-12)){m=n;break ca}o=a+P(o,12)|0;a=o;m=a+P((P(m,12)|0)/12|0,12)|0;if((n|0)!=(m|0)){while(1){if(H[a+11|0]<0){$a(J[a>>2]);}q=J[m+4>>2];J[a>>2]=J[m>>2];J[a+4>>2]=q;J[a+8>>2]=J[m+8>>2];H[m+11|0]=0;H[m|0]=0;a=a+12|0;m=m+12|0;if((n|0)!=(m|0)){continue}break}n=J[l+196>>2];}m=o+P((a-o|0)/12|0,12)|0;if((n|0)!=(m|0)){while(1){a=n-12|0;if(H[n-1|0]<0){$a(J[a>>2]);}n=a;if((a|0)!=(m|0)){continue}break}}J[l+196>>2]=m;}o=J[l+200>>2];da:{if(o>>>0>m>>>0){a=J[l+252>>2];J[m>>2]=J[l+248>>2];J[m+4>>2]=a;J[m+8>>2]=J[l+256>>2];J[l+256>>2]=0;J[l+248>>2]=0;J[l+252>>2]=0;J[l+196>>2]=m+12;break da}ea:{fa:{n=J[l+192>>2];s=(m-n|0)/12|0;a=s+1|0;ga:{if(a>>>0>=357913942){J[5996]=0;ca(41);break ga}o=(o-n|0)/12|0;q=o<<1;q=o>>>0>=178956970?357913941:a>>>0<q>>>0?q:a;if(!q){o=0;break ea}if(q>>>0<357913942){break fa}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break a}m=fa(22864)|0;break W}J[5996]=0;o=$(3,P(q,12)|0)|0;a=J[5996];J[5996]=0;if((a|0)==1){break X}}a=P(s,12)+o|0;s=J[l+252>>2];J[a>>2]=J[l+248>>2];J[a+4>>2]=s;J[a+8>>2]=J[l+256>>2];J[l+256>>2]=0;J[l+248>>2]=0;J[l+252>>2]=0;o=P(q,12)+o|0;q=a+12|0;ha:{if((m|0)==(n|0)){J[l+200>>2]=o;J[l+196>>2]=q;J[l+192>>2]=a;break ha}while(1){m=m-12|0;s=J[m+4>>2];a=a-12|0;J[a>>2]=J[m>>2];J[a+4>>2]=s;J[a+8>>2]=J[m+8>>2];J[m>>2]=0;J[m+4>>2]=0;J[m+8>>2]=0;if((m|0)!=(n|0)){continue}break}J[l+200>>2]=o;n=J[l+196>>2];J[l+196>>2]=q;m=J[l+192>>2];J[l+192>>2]=a;if((m|0)==(n|0)){break ha}while(1){a=n-12|0;if(H[n-1|0]<0){$a(J[a>>2]);}n=a;if((a|0)!=(m|0)){continue}break}}if(!m){break da}$a(m);}if(!(!K[l+260|0]|H[l+259|0]>=0)){$a(J[l+248>>2]);}m=r;}if((m|0)!=(p|0)){continue}break V}break}m=fa(22864)|0;}a=Va;if(!K[l+260|0]|H[l+259|0]>=0){break L}$a(J[l+248>>2]);break L}a=J[l+192>>2];if((J[l+196>>2]-a|0)!=12){J[5996]=0;ga(44,4102,2210,450,2439);a=J[5996];J[5996]=0;if((a|0)!=1){break a}m=fa(22864)|0;break M}s=J[a>>2];J[l+160>>2]=J[a+4>>2];m=K[a+7|0]|K[a+8|0]<<8|(K[a+9|0]<<16|K[a+10|0]<<24);H[l+163|0]=m;H[l+164|0]=m>>>8;H[l+165|0]=m>>>16;H[l+166|0]=m>>>24;J[a>>2]=0;J[a+4>>2]=0;q=K[a+11|0];J[a+8>>2]=0;}n=J[l+192>>2];if(n){a=J[l+196>>2];m=n;if((a|0)!=(m|0)){while(1){m=a-12|0;if(H[a-1|0]<0){$a(J[m>>2]);}a=m;if((n|0)!=(a|0)){continue}break}m=J[l+192>>2];}J[l+196>>2]=n;$a(m);}if(p){$a(p);}a=K[l+163|0]|K[l+164|0]<<8|(K[l+165|0]<<16|K[l+166|0]<<24);H[l+103|0]=a;H[l+104|0]=a>>>8;H[l+105|0]=a>>>16;H[l+106|0]=a>>>24;J[l+100>>2]=J[l+160>>2];J[l+96>>2]=s;J[l+160>>2]=0;H[l+163|0]=0;H[l+164|0]=0;H[l+165|0]=0;H[l+166|0]=0;H[l+107|0]=q;ia:{ja:{ka:{la:{ma:{na:{oa:{pa:{qa:{ra:{sa:{ta:{ua:{va:{wa:{xa:{ya:{za:{Aa:{Ba:{Ca:{Da:{p=J[l+100>>2];a=q<<24>>24<0;m=a?p:q&255;Ea:{if((m|0)<72){break Ea}o=a?s:l+96|0;n=o+m|0;while(1){a=o;while(1){a=Zb(a,115,m-71|0);if(!a){break Ea}if(Eb(a,5034,72)){a=a+1|0;m=n-a|0;if((m|0)>=72){continue}break Ea}break}if((a|0)==(n|0)){break Ea}a=a-o|0;if((a|0)==-1){break Ea}J[5996]=0;m=l+96|0;ga(114,m|0,a|0,72,4824);a=J[5996];J[5996]=0;if((a|0)==1){break Da}s=J[l+96>>2];n=m;a=K[l+107|0];q=a<<24>>24;m=(q|0)<0;o=m?s:n;p=J[l+100>>2];m=m?p:a;n=o+m|0;if((m|0)>71){continue}break}}Fa:{a=q<<24>>24;m=(a|0)<0?p:q&255;Ga:{if((m|0)<73){break Ga}o=(a|0)<0?s:l+96|0;n=o+m|0;while(1){a=o;while(1){a=Zb(a,104,m-72|0);if(!a){break Ga}if(Eb(a,4960,73)){a=a+1|0;m=n-a|0;if((m|0)>=73){continue}break Ga}break}if((a|0)==(n|0)){break Ga}a=a-o|0;if((a|0)==-1){break Ga}J[5996]=0;m=l+96|0;ga(114,m|0,a|0,73,4813);a=J[5996];J[5996]=0;if((a|0)==1){break Fa}s=J[l+96>>2];n=m;a=K[l+107|0];q=a<<24>>24;m=(q|0)<0;o=m?s:n;p=J[l+100>>2];m=m?p:a;n=o+m|0;if((m|0)>72){continue}break}}Ha:{a=q<<24>>24<0;m=a?p:q&255;Ia:{if((m|0)<51){break Ia}o=a?s:l+96|0;n=o+m|0;while(1){a=o;while(1){a=Zb(a,114,m-50|0);if(!a){break Ia}if(Eb(a,4908,51)){a=a+1|0;m=n-a|0;if((m|0)>=51){continue}break Ia}break}if((a|0)==(n|0)){break Ia}a=a-o|0;if((a|0)==-1){break Ia}J[5996]=0;m=l+96|0;ga(114,m|0,a|0,51,4845);a=J[5996];J[5996]=0;if((a|0)==1){break Ha}s=J[l+96>>2];n=m;a=K[l+107|0];q=a<<24>>24;m=(q|0)<0;o=m?s:n;p=J[l+100>>2];m=m?p:a;n=o+m|0;if((m|0)>50){continue}break}}Ja:{a=q<<24>>24;m=(a|0)<0?p:q&255;Ka:{if((m|0)<49){break Ka}p=(a|0)<0?s:l+96|0;n=p+m|0;while(1){a=p;while(1){a=Zb(a,104,m-48|0);if(!a){break Ka}if(Eb(a,4858,49)){a=a+1|0;m=n-a|0;if((m|0)>=49){continue}break Ka}break}if((a|0)==(n|0)){break Ka}a=a-p|0;if((a|0)==-1){break Ka}J[5996]=0;m=l+96|0;ga(114,m|0,a|0,49,4834);a=J[5996];J[5996]=0;if((a|0)==1){break Ja}n=m;a=K[l+107|0];m=a<<24>>24<0;p=m?J[l+96>>2]:n;m=m?J[l+100>>2]:a;n=p+m|0;if((m|0)>48){continue}break}}J[l+184>>2]=J[l+104>>2];J[l+104>>2]=0;m=J[l+100>>2];a=J[l+96>>2];J[l+176>>2]=a;J[l+180>>2]=m;J[l+96>>2]=0;J[l+100>>2]=0;m=K[l+187|0];n=m<<24>>24;o=(n|0)<0;p=o?a:l+176|0;m=o?J[l+180>>2]:m;o=c-1|0;m=(m|0)<(o|0)?m:o;H[fb(b,p,m)+m|0]=0;La:{if((n|0)>=0){break La}$a(a);if(H[l+107|0]>=0){break La}$a(J[l+96>>2]);}a=J[J[l+12>>2]+84>>2];J[5996]=0;m=l+36|0;aa(115,m|0,a|0);a=J[5996];J[5996]=0;if((a|0)==1){break Ba}J[5996]=0;a=ba(99,m|0,0,5442)|0;m=J[5996];J[5996]=0;if((m|0)==1){break Aa}J[l+56>>2]=J[a+8>>2];m=J[a+4>>2];J[l+48>>2]=J[a>>2];J[l+52>>2]=m;J[a>>2]=0;J[a+4>>2]=0;J[a+8>>2]=0;J[5996]=0;m=_(97,l+48|0,5422)|0;n=J[5996];J[5996]=0;if((n|0)==1){break za}J[l+72>>2]=J[m+8>>2];n=J[m+4>>2];J[l+64>>2]=J[m>>2];J[l+68>>2]=n;J[m>>2]=0;J[m+4>>2]=0;J[m+8>>2]=0;J[5996]=0;n=l+248|0;Da(116,n|0,+u);o=J[5996];J[5996]=0;if((o|0)==1){break ya}J[5996]=0;p=n;n=K[l+259|0];o=n<<24>>24<0;n=ba(100,l- -64|0,(o?J[l+248>>2]:p)|0,(o?J[l+252>>2]:n)|0)|0;o=J[5996];J[5996]=0;if((o|0)==1){break xa}J[l+88>>2]=J[n+8>>2];o=J[n+4>>2];J[l+80>>2]=J[n>>2];J[l+84>>2]=o;J[n>>2]=0;J[n+4>>2]=0;J[n+8>>2]=0;J[5996]=0;o=_(97,l+80|0,5504)|0;p=J[5996];J[5996]=0;if((p|0)==1){break wa}J[l+136>>2]=J[o+8>>2];p=J[o+4>>2];J[l+128>>2]=J[o>>2];J[l+132>>2]=p;J[o>>2]=0;J[o+4>>2]=0;J[o+8>>2]=0;p=J[J[l+12>>2]+84>>2];J[5996]=0;r=l+192|0;Da(116,r|0,+(u+ +(p>>>0)));p=J[5996];J[5996]=0;if((p|0)==1){break va}J[5996]=0;t=r;p=K[l+203|0];r=p<<24>>24<0;p=ba(100,l+128|0,(r?J[l+192>>2]:t)|0,(r?J[l+196>>2]:p)|0)|0;r=J[5996];J[5996]=0;if((r|0)==1){break ua}J[l+168>>2]=J[p+8>>2];r=J[p+4>>2];J[l+160>>2]=J[p>>2];J[l+164>>2]=r;J[p>>2]=0;J[p+4>>2]=0;J[p+8>>2]=0;J[5996]=0;r=_(97,l+160|0,3567)|0;q=J[5996];J[5996]=0;if((q|0)==1){break ta}J[l+184>>2]=J[r+8>>2];q=J[r+4>>2];J[l+176>>2]=J[r>>2];J[l+180>>2]=q;J[r>>2]=0;J[r+4>>2]=0;J[r+8>>2]=0;if(H[l+203|0]<0){$a(J[l+192>>2]);}if(H[p+11|0]<0){$a(J[p>>2]);}if(H[o+11|0]<0){$a(J[o>>2]);}if(H[l+259|0]<0){$a(J[l+248>>2]);}if(H[n+11|0]<0){$a(J[n>>2]);}if(H[m+11|0]<0){$a(J[m>>2]);}if(H[a+11|0]<0){$a(J[a>>2]);}n=J[l+176>>2];a=K[l+187|0];o=a<<24>>24;m=(o|0)<0;p=m?n:l+176|0;a=m?J[l+180>>2]:a;m=e-1|0;a=(a|0)<(m|0)?a:m;H[fb(d,p,a)+a|0]=0;J[5996]=0;m=l+248|0;aa(117,m|0,J[l+12>>2]);a=J[5996];J[5996]=0;if((a|0)==1){break la}J[l+220>>2]=J[l+276>>2];a=J[l+272>>2];J[l+212>>2]=J[l+268>>2];J[l+216>>2]=a;a=J[l+264>>2];J[l+204>>2]=J[l+260>>2];J[l+208>>2]=a;a=J[l+256>>2];J[l+196>>2]=J[l+252>>2];J[l+200>>2]=a;a=J[l+248>>2];J[l+192>>2]=a;J[5996]=0;p=a>>>0<29;r=p?l+196|0:J[l+200>>2];J[l+248>>2]=r;q=l+24|0;ga(102,q|0,m|0,r+(p?a:a-29|0)|0,0);a=J[5996];J[5996]=0;if((a|0)==1){break ka}a=J[l+24>>2];m=K[l+35|0];p=m<<24>>24;r=(p|0)<0;t=r?a:q;m=r?J[l+28>>2]:m;r=g-1|0;m=(m|0)<(r|0)?m:r;H[fb(f,t,m)+m|0]=0;if((p|0)<0){$a(a);}if(M[l+192>>2]>=29){$a(J[l+200>>2]);}a=J[l+12>>2];m=J[a+80>>2];if(!m|M[a+84>>2]>3600|(J[a+52>>2]+J[a+44>>2]>>>0>201|M[a+68>>2]>100)){break ja}J[5996]=0;p=_(54,2049,1)|0;r=J[5996];J[5996]=0;if((r|0)==1){break Q}if((m^-1)&p){break ja}m=J[a+80>>2];J[5996]=0;p=_(54,2165,1)|0;r=J[5996];J[5996]=0;if((r|0)==1){break Q}if(K[a+88|0]|(m^-1)&p){break ja}H[l+252|0]=0;J[l+248>>2]=1702195828;H[l+259|0]=4;a=((k|0)>=5?5:k)-1|0;H[fb(j,l+248|0,a)+a|0]=0;break ia}m=fa(22864)|0;break Ca}m=fa(22864)|0;break Ca}m=fa(22864)|0;break Ca}m=fa(22864)|0;}a=Va;if(H[l+107|0]>=0){break n}$a(J[l+96>>2]);break n}m=fa(22864)|0;a=Va;break n}m=fa(22864)|0;a=Va;break ma}m=fa(22864)|0;a=Va;break na}m=fa(22864)|0;a=Va;break oa}m=fa(22864)|0;a=Va;break pa}m=fa(22864)|0;a=Va;break qa}m=fa(22864)|0;a=Va;break ra}m=fa(22864)|0;a=Va;break sa}m=fa(22864)|0;a=Va;if(H[l+171|0]>=0){break sa}$a(J[l+160>>2]);}if(H[l+203|0]>=0){break ra}$a(J[l+192>>2]);}if(H[l+139|0]>=0){break qa}$a(J[l+128>>2]);}if(H[l+91|0]>=0){break pa}$a(J[l+80>>2]);}if(H[l+259|0]>=0){break oa}$a(J[l+248>>2]);}if(H[l+75|0]>=0){break na}$a(J[l+64>>2]);}if(H[l+59|0]>=0){break ma}$a(J[l+48>>2]);}if(H[l+47|0]>=0){break n}$a(J[l+36>>2]);break n}m=fa(22864)|0;break P}m=fa(22864)|0;a=Va;if(M[l+192>>2]<29){break O}$a(J[l+200>>2]);break O}H[l+252|0]=K[2354];H[l+259|0]=5;H[l+253|0]=0;J[l+248>>2]=K[2350]|K[2351]<<8|(K[2352]<<16|K[2353]<<24);a=((k|0)>=6?6:k)-1|0;H[fb(j,l+248|0,a)+a|0]=0;}J[5996]=0;a=$(118,J[l+12>>2])|0;j=J[5996];J[5996]=0;if((j|0)==1){break Q}Ma:{if(a){H[l+252|0]=0;J[l+248>>2]=1702195828;H[l+259|0]=4;a=((i|0)>=5?5:i)-1|0;H[fb(h,l+248|0,a)+a|0]=0;break Ma}H[l+252|0]=K[2354];H[l+259|0]=5;H[l+253|0]=0;J[l+248>>2]=K[2350]|K[2351]<<8|(K[2352]<<16|K[2353]<<24);a=((i|0)>=6?6:i)-1|0;H[fb(h,l+248|0,a)+a|0]=0;}if((o|0)>=0){break R}$a(n);}a=J[l+16>>2];Na:{if(!a){break Na}b=J[a+4>>2];J[a+4>>2]=b-1;if(b){break Na}Xa[J[J[a>>2]+8>>2]](a);bb(a);}if(H[l+155|0]>=0){break k}$a(J[l+144>>2]);break k}m=fa(22864)|0;}a=Va;}if((o|0)>=0){break n}$a(n);break n}m=fa(22864)|0;}a=Va;}rb(l+192|0);if(!p){break n}$a(p);break n}m=fa(22864)|0;a=Va;rb(l+192|0);}mb(l+12|0);}if(H[l+155|0]>=0){break l}$a(J[l+144>>2]);}if((wa(22864)|0)!=(a|0)){break c}a=pa(m|0)|0;a=Xa[J[J[a>>2]+8>>2]](a)|0;J[5996]=0;i=_(30,l+24|0,a|0)|0;a=J[5996];J[5996]=0;if((a|0)==1){break j}J[5996]=0;a=ba(99,i|0,0,5474)|0;h=J[5996];J[5996]=0;if((h|0)==1){break i}J[l+152>>2]=J[a+8>>2];h=J[a+4>>2];J[l+144>>2]=J[a>>2];J[l+148>>2]=h;J[a>>2]=0;J[a+4>>2]=0;J[a+8>>2]=0;J[5996]=0;h=_(97,l+144|0,2611)|0;j=J[5996];J[5996]=0;if((j|0)==1){break h}J[l+184>>2]=J[h+8>>2];i=J[h+4>>2];J[l+176>>2]=J[h>>2];J[l+180>>2]=i;J[h>>2]=0;J[h+4>>2]=0;J[h+8>>2]=0;m=b;b=J[l+176>>2];i=K[l+187|0];j=i<<24>>24;k=(j|0)<0;i=k?J[l+180>>2]:i;c=c-1|0;c=(c|0)>(i|0)?i:c;H[fb(m,k?b:l+176|0,c)+c|0]=0;Oa:{if((j|0)>=0){break Oa}$a(b);if(H[h+11|0]>=0){break Oa}$a(J[h>>2]);}if(H[a+11|0]<0){$a(J[a>>2]);}J[5996]=0;a=_(30,l+176|0,5575)|0;b=J[5996];J[5996]=0;if((b|0)==1){break f}i=d;b=J[a>>2];c=K[a+11|0];d=c<<24>>24;h=(d|0)<0;j=h?b:a;a=h?J[a+4>>2]:c;c=e-1|0;a=(a|0)<(c|0)?a:c;H[fb(i,j,a)+a|0]=0;if((d|0)<0){$a(b);}J[5996]=0;a=_(30,l+12|0,5575)|0;b=J[5996];J[5996]=0;if((b|0)==1){break e}b=J[a>>2];c=K[a+11|0];d=c<<24>>24;e=(d|0)<0;h=e?b:a;a=e?J[a+4>>2]:c;c=g-1|0;a=(a|0)<(c|0)?a:c;H[fb(f,h,a)+a|0]=0;if((d|0)<0){$a(b);}ra();}Ua=l+304|0;return}m=Z()|0;break d}m=Z()|0;break g}m=Z()|0;if(H[l+155|0]>=0){break g}$a(J[l+144>>2]);}if(H[i+11|0]>=0){break d}$a(J[i>>2]);break d}m=Z()|0;break d}m=Z()|0;}J[5996]=0;ca(110);a=J[5996];J[5996]=0;if((a|0)==1){break b}}da(m|0);B();}fa(0)|0;Vb();B();}B();}function _g(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0;f=Ua-144|0;Ua=f;l=O[b+24>>3];m=O[b+16>>3];a:{b:{c:{d:{e:{f:{g:{h:{i:{j:{k:{l:{m:{n:{o:{p:{q:{r:{s:{t:{u:{v:{w:{x:{y:{z:{A:{B:{e=J[a>>2];switch(e|0){case 9:break j;case 8:break k;case 2:break l;case 3:break m;case 14:break n;case 15:break o;case 16:break p;case 19:break q;case 18:break r;case 17:break s;case 10:break t;case 11:break u;case 12:break v;case 13:break w;case 0:break x;case 4:case 5:break y;case 1:break z;case 21:break A;case 20:break B;case 6:break h;case 7:break i;default:break g}}d=J[a+4>>2];a=J[a+8>>2];if((d|0)==(a|0)){break g}while(1){_g(J[d>>2],b,c);d=d+4|0;if((a|0)!=(d|0)){continue}break}break g}a=$g(a,m,l,c);d=J[a>>2];c=J[a+4>>2];if((d|0)==(c|0)){break g}while(1){J[f+56>>2]=J[d>>2];a=J[d+4>>2];J[f+60>>2]=a;if(a){J[a+4>>2]=J[a+4>>2]+1;}a=J[f+60>>2];J[f>>2]=J[f+56>>2];J[f+4>>2]=a;Zg(b,d+8|0,f);d=d+32|0;if((c|0)!=(d|0)){continue}break}break g}hc(b,c,1,a+4|0,0,0);break g}n=(e|0)==4?4:5;Yc(f+56|0,n,m,l,0,0);J[5996]=0;d=$(46,n|0)|0;e=J[5996];J[5996]=0;if((e|0)==1){break f}J[f+88>>2]=0;J[f+80>>2]=0;J[f+84>>2]=0;H[f+136|0]=0;J[f+132>>2]=f+80;e=J[d+4>>2];h=J[d>>2];k=e-h|0;g=(k|0)/12|0;if((e|0)!=(h|0)){C:{D:{E:{if(g>>>0>=357913942){J[5996]=0;ca(47);a=J[5996];J[5996]=0;if((a|0)==1){break E}break d}J[5996]=0;e=$(3,k|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break E}J[f+84>>2]=e;J[f+80>>2]=e;J[f+88>>2]=e+P(g,12);g=J[d+4>>2];d=J[d>>2];J[5996]=0;d=ia(48,f+88|0,d|0,g|0,e|0)|0;g=J[5996];J[5996]=0;if((g|0)!=1){break C}d=Z()|0;J[f+84>>2]=e;break D}d=Z()|0;}Nc(f+132|0);break c}J[f+84>>2]=d;}J[f+140>>2]=0;J[f+132>>2]=0;J[f+136>>2]=0;e=J[a+8>>2];d=J[a+4>>2];J[5996]=0;aa(49,f+132|0,e-d>>2);e=J[5996];J[5996]=0;if((e|0)!=1){d=J[f+84>>2];e=J[f+80>>2];if((d|0)!=(e|0)){u=a+40|0;d=e;while(1){J[f+128>>2]=0;J[f+120>>2]=0;J[f+124>>2]=0;F:{G:{H:{I:{g=J[a+8>>2];e=J[a+4>>2];q=P(s,12);d=q+d|0;if(g-e>>2==J[d+4>>2]-J[d>>2]>>3){h=0;o=1;j=0;if((e|0)!=(g|0)){break I}break H}J[5996]=0;ga(44,5170,1900,642,2497);a=J[5996];J[5996]=0;if((a|0)!=1){break d}d=Z()|0;break G}J:{K:{while(1){L:{d=j<<3;l=O[d+J[f+68>>2]>>3];m=O[d+J[f+56>>2]>>3];e=J[(j<<2)+e>>2];J[5996]=0;e=qa(50,e|0,+m,+l,c|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break L}d=d+J[q+J[f+80>>2]>>2]|0;g=J[d>>2];d=J[d+4>>2];k=J[e+4>>2];e=J[e>>2];J[5996]=0;oa(51,f+100|0,e|0,k|0,g|0,d|0);e=J[5996];J[5996]=0;if((e|0)==1){break K}g=J[f+128>>2];M:{if(g>>>0>h>>>0){J[h+8>>2]=0;J[h>>2]=0;J[h+4>>2]=0;J[h>>2]=J[f+100>>2];J[h+4>>2]=J[f+104>>2];J[h+8>>2]=J[f+108>>2];J[f+124>>2]=h+12;break M}N:{e=J[f+120>>2];k=(h-e|0)/12|0;d=k+1|0;O:{if(d>>>0>=357913942){J[5996]=0;ca(52);break O}g=(g-e|0)/12|0;i=g<<1;d=g>>>0>=178956970?357913941:d>>>0<i>>>0?i:d;if(d>>>0<357913942){break N}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break d}break J}J[5996]=0;i=P(d,12);g=$(3,i|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break J}d=g+P(k,12)|0;J[d>>2]=J[f+100>>2];J[d+4>>2]=J[f+104>>2];J[d+8>>2]=J[f+108>>2];J[f+108>>2]=0;J[f+100>>2]=0;J[f+104>>2]=0;g=g+i|0;k=d+12|0;P:{if((e|0)==(h|0)){J[f+128>>2]=g;J[f+124>>2]=k;J[f+120>>2]=d;break P}while(1){d=d-12|0;h=h-12|0;J[d>>2]=J[h>>2];J[d+4>>2]=J[h+4>>2];J[d+8>>2]=J[h+8>>2];J[h+8>>2]=0;J[h>>2]=0;J[h+4>>2]=0;if((e|0)!=(h|0)){continue}break}J[f+128>>2]=g;e=J[f+124>>2];J[f+124>>2]=k;h=J[f+120>>2];J[f+120>>2]=d;if((e|0)==(h|0)){break P}while(1){i=e-12|0;k=J[i>>2];if(k){r=e-8|0;d=J[r>>2];g=k;if((d|0)!=(g|0)){while(1){d=d-32|0;e=J[d+4>>2];Q:{if(!e){break Q}g=J[e+4>>2];J[e+4>>2]=g-1;if(g){break Q}Xa[J[J[e>>2]+8>>2]](e);bb(e);}if((d|0)!=(k|0)){continue}break}g=J[i>>2];}J[r>>2]=k;$a(g);}e=i;if((h|0)!=(e|0)){continue}break}}if(h){$a(h);}g=J[f+100>>2];if(!g){break M}e=g;d=J[f+104>>2];if((e|0)!=(d|0)){while(1){d=d-32|0;e=J[d+4>>2];R:{if(!e){break R}h=J[e+4>>2];J[e+4>>2]=h-1;if(h){break R}Xa[J[J[e>>2]+8>>2]](e);bb(e);}if((d|0)!=(g|0)){continue}break}e=J[f+100>>2];}J[f+104>>2]=g;$a(e);}h=J[f+124>>2];e=h-12|0;o=P(J[e+4>>2]-J[e>>2]>>5,o);j=j+1|0;e=J[a+4>>2];if(j>>>0<J[a+8>>2]-e>>2>>>0){continue}break H}break}d=Z()|0;break G}d=Z()|0;break G}d=Z()|0;_b(f+100|0);break G}h=0;k=J[f+132>>2];while(1){if((h|0)==(o|0)){break F}if(J[a+8>>2]!=J[a+4>>2]){d=0;g=J[f+120>>2];e=h;while(1){i=P(d,12);j=i+g|0;g=J[j>>2];J[k+(d<<2)>>2]=g+((e>>>0)%(J[j+4>>2]-g>>5>>>0)<<5);j=e;g=J[f+120>>2];e=i+g|0;e=(j>>>0)/(J[e+4>>2]-J[e>>2]>>5>>>0)|0;d=d+1|0;if(d>>>0<J[a+8>>2]-J[a+4>>2]>>2>>>0){continue}break}}J[5996]=0;Ea(53,b|0,n|0,f+132|0,0,u|0);e=J[5996];J[5996]=0;h=h+1|0;if((e|0)!=1){continue}break}d=Z()|0;}Xc(f+120|0);break e}g=J[f+120>>2];if(g){d=J[f+124>>2];if((g|0)!=(d|0)){while(1){k=d-12|0;h=J[k>>2];if(h){e=h;i=d-8|0;d=J[i>>2];if((e|0)!=(d|0)){while(1){d=d-32|0;e=J[d+4>>2];S:{if(!e){break S}j=J[e+4>>2];J[e+4>>2]=j-1;if(j){break S}Xa[J[J[e>>2]+8>>2]](e);bb(e);}if((d|0)!=(h|0)){continue}break}e=J[k>>2];}J[i>>2]=h;$a(e);}d=k;if((g|0)!=(d|0)){continue}break}g=J[f+120>>2];}$a(g);}s=s+1|0;d=J[f+80>>2];if(s>>>0<(J[f+84>>2]-d|0)/12>>>0){continue}break}}a=J[f+132>>2];if(a){$a(a);d=J[f+80>>2];}if(d){e=d;g=J[f+84>>2];if((e|0)!=(g|0)){while(1){a=g-12|0;b=J[a>>2];if(b){J[g-8>>2]=b;$a(b);}g=a;if((a|0)!=(d|0)){continue}break}e=J[f+80>>2];}J[f+84>>2]=d;$a(e);}a=J[f+68>>2];if(a){J[f+72>>2]=a;$a(a);}a=J[f+56>>2];if(!a){break g}J[f+60>>2]=a;$a(a);break g}d=Z()|0;break e}hc(b,c,0,a+4|0,0,0);break g}Yc(f+56|0,24,m,l,O[a+48>>3],J[a+8>>2]-J[a+4>>2]>>2);J[f+88>>2]=0;J[f+80>>2]=0;J[f+84>>2]=0;J[f+140>>2]=0;J[f+132>>2]=0;J[f+136>>2]=0;T:{U:{V:{W:{X:{Y:{Z:{_:{$:{g=J[a+4>>2];aa:{if((g|0)==J[a+8>>2]){k=-1;e=g;d=0;break aa}l=-1;e=g;k=-1;ba:{while(1){d=h<<3;m=O[d+J[f+68>>2]>>3];p=O[d+J[f+56>>2]>>3];e=J[(h<<2)+e>>2];J[5996]=0;e=qa(50,e|0,+p,+m,c|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break $}J[5996]=0;d=_(54,1283,5)|0;g=J[5996];J[5996]=0;ca:{da:{ea:{fa:{ga:{if((g|0)==1){break ga}g=J[e>>2];i=J[e+4>>2];J[5996]=0;oa(51,f+120|0,g|0,i|0,d|0,0);d=J[5996];J[5996]=0;if((d|0)==1){break ga}i=1;g=J[f+120>>2];switch(J[f+124>>2]-g>>5){case 0:break da;case 1:break ea;default:break fa}}d=Z()|0;break T}J[5996]=0;ga(44,4164,1900,807,2386);a=J[5996];J[5996]=0;if((a|0)!=1){break d}d=Z()|0;break U}J[5996]=0;aa(55,f+80|0,g|0);d=J[5996];J[5996]=0;if((d|0)==1){break _}J[5996]=0;d=_(54,1277,5)|0;g=J[5996];J[5996]=0;if((g|0)==1){break ca}g=J[e>>2];e=J[e+4>>2];J[5996]=0;oa(51,f+100|0,g|0,e|0,d|0,0);e=J[5996];J[5996]=0;if((e|0)==1){break ca}ha:{ia:{ja:{e=J[f+100>>2];switch(J[f+104>>2]-e>>5){case 0:break ha;case 1:break ia;default:break ja}}J[5996]=0;ga(44,4146,1900,811,2386);a=J[5996];J[5996]=0;if((a|0)!=1){break d}break V}J[5996]=0;aa(55,f+132|0,e|0);e=J[5996];J[5996]=0;if((e|0)==1){break V}i=0;m=O[J[f+136>>2]-8>>3]-O[J[f+84>>2]-8>>3];if(!(m>l)){break ha}k=h;l=m;}g=J[f+100>>2];if(g){e=g;d=J[f+104>>2];if((e|0)!=(d|0)){while(1){d=d-32|0;e=J[d+4>>2];ka:{if(!e){break ka}j=J[e+4>>2];J[e+4>>2]=j-1;if(j){break ka}Xa[J[J[e>>2]+8>>2]](e);bb(e);}if((d|0)!=(g|0)){continue}break}e=J[f+100>>2];}J[f+104>>2]=g;$a(e);}g=J[f+120>>2];}if(g){e=g;d=J[f+124>>2];if((e|0)!=(d|0)){while(1){d=d-32|0;e=J[d+4>>2];la:{if(!e){break la}j=J[e+4>>2];J[e+4>>2]=j-1;if(j){break la}Xa[J[J[e>>2]+8>>2]](e);bb(e);}if((d|0)!=(g|0)){continue}break}e=J[f+120>>2];}J[f+124>>2]=g;$a(e);}if(i){break W}h=h+1|0;g=J[a+8>>2];e=J[a+4>>2];if(h>>>0>=g-e>>2>>>0){break ba}continue}break}d=Z()|0;break U}d=J[f+80>>2];}J[5996]=0;c=$(3,4)|0;h=J[5996];J[5996]=0;if((h|0)==1){break Z}J[c>>2]=d+(k<<5);d=c+4|0;J[f+128>>2]=d;J[f+124>>2]=d;J[f+120>>2]=c;if((e|0)!=(g|0)){h=0;while(1){ma:{if((h|0)==(k|0)){break ma}c=J[f+132>>2]+(h<<5)|0;i=J[f+128>>2];if(i>>>0>d>>>0){J[d>>2]=c;d=d+4|0;J[f+124>>2]=d;break ma}na:{oa:{g=J[f+120>>2];o=d-g>>2;e=o+1|0;pa:{if(e>>>0>=1073741824){J[5996]=0;ca(56);break pa}i=i-g|0;j=i>>1;i=i>>>0>=2147483644?1073741823:e>>>0<j>>>0?j:e;if(!i){j=0;break na}if(i>>>0<1073741824){break oa}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break d}d=Z()|0;break X}J[5996]=0;j=$(3,i<<2)|0;e=J[5996];J[5996]=0;if((e|0)==1){break Y}}e=(o<<2)+j|0;J[e>>2]=c;c=e+4|0;if((d|0)!=(g|0)){while(1){e=e-4|0;d=d-4|0;J[e>>2]=J[d>>2];if((d|0)!=(g|0)){continue}break}d=J[f+120>>2];}J[f+128>>2]=(i<<2)+j;J[f+124>>2]=c;J[f+120>>2]=e;if(d){$a(d);}d=c;}h=h+1|0;if(h>>>0<J[a+8>>2]-J[a+4>>2]>>2>>>0){continue}break}}l=O[a+48>>3];J[5996]=0;Ea(53,b|0,24,f+120|0,+l,a+40|0);a=J[5996];J[5996]=0;if((a|0)!=1){a=J[f+120>>2];if(!a){break W}$a(a);break W}d=Z()|0;g=J[f+120>>2];break X}d=Z()|0;break T}d=Z()|0;break U}d=Z()|0;break T}d=Z()|0;}if(!g){break T}$a(g);break T}a=J[f+132>>2];if(a){d=J[f+136>>2];if((a|0)!=(d|0)){while(1){d=d-32|0;b=J[d+4>>2];qa:{if(!b){break qa}c=J[b+4>>2];J[b+4>>2]=c-1;if(c){break qa}Xa[J[J[b>>2]+8>>2]](b);bb(b);}if((a|0)!=(d|0)){continue}break}a=J[f+132>>2];}$a(a);}a=J[f+80>>2];if(a){d=J[f+84>>2];if((a|0)!=(d|0)){while(1){d=d-32|0;b=J[d+4>>2];ra:{if(!b){break ra}c=J[b+4>>2];J[b+4>>2]=c-1;if(c){break ra}Xa[J[J[b>>2]+8>>2]](b);bb(b);}if((a|0)!=(d|0)){continue}break}a=J[f+80>>2];}$a(a);}a=J[f+68>>2];if(a){J[f+72>>2]=a;$a(a);}a=J[f+56>>2];if(!a){break g}J[f+60>>2]=a;$a(a);break g}d=Z()|0;_b(f+100|0);}_b(f+120|0);}_b(f+132|0);_b(f+80|0);break c}e=J[a+4>>2];d=J[e+4>>2];g=J[e>>2];h=J[e+8>>2];e=cb(12);J[f+56>>2]=e;k=e+12|0;J[f+64>>2]=k;J[e+8>>2]=h;J[e+4>>2]=g;J[e>>2]=d;J[f+60>>2]=k;l=O[a+48>>3];J[5996]=0;ja(57,b|0,c|0,23,a+4|0,+l,0);a=J[5996];J[5996]=0;sa:{if((a|0)==1){break sa}J[5996]=0;ja(57,b|0,c|0,23,f+56|0,+l,0);a=J[5996];J[5996]=0;if((a|0)==1){break sa}$a(e);break g}break a}e=J[a+4>>2];d=J[e>>2];g=J[e+4>>2];e=cb(8);J[f+56>>2]=e;h=e+8|0;J[f+64>>2]=h;J[e>>2]=go(d,g,32);J[e+4>>2]=Wa;J[f+60>>2]=h;d=a+4|0;m=O[a+48>>3];p=1-m;ta:{if(l==0){J[5996]=0;ja(57,b|0,c|0,20,d|0,+m,0);a=J[5996];J[5996]=0;if((a|0)==1){break ta}J[5996]=0;ja(57,b|0,c|0,20,f+56|0,+p,0);a=J[5996];J[5996]=0;if((a|0)==1){break ta}}J[5996]=0;ja(57,b|0,c|0,19,d|0,+m,0);a=J[5996];J[5996]=0;if((a|0)==1){break ta}J[5996]=0;a=f+56|0;ja(57,b|0,c|0,19,a|0,+p,0);g=J[5996];J[5996]=0;if((g|0)==1){break ta}J[5996]=0;ja(57,b|0,c|0,21,d|0,+m,0);g=J[5996];J[5996]=0;if((g|0)==1){break ta}J[5996]=0;ja(57,b|0,c|0,21,a|0,+p,0);g=J[5996];J[5996]=0;if((g|0)==1){break ta}J[5996]=0;ja(57,b|0,c|0,22,d|0,+m,0);g=J[5996];J[5996]=0;if((g|0)==1){break ta}J[5996]=0;ja(57,b|0,c|0,22,a|0,+p,0);g=J[5996];J[5996]=0;if((g|0)==1){break ta}J[5996]=0;ja(57,b|0,c|0,22,d|0,+m,1);d=J[5996];J[5996]=0;if((d|0)==1){break ta}J[5996]=0;ja(57,b|0,c|0,22,a|0,+p,1);a=J[5996];J[5996]=0;if((a|0)==1){break ta}$a(e);break g}break a}e=J[a+4>>2];d=J[e>>2];g=J[e+4>>2];e=cb(8);J[f+56>>2]=e;h=e+8|0;J[f+64>>2]=h;J[e>>2]=go(d,g,32);J[e+4>>2]=Wa;J[f+60>>2]=h;a=a+4|0;ua:{if(l==0){J[5996]=0;ja(57,b|0,c|0,17,a|0,0,0);d=J[5996];J[5996]=0;if((d|0)==1){break ua}J[5996]=0;ja(57,b|0,c|0,17,f+56|0,0,0);d=J[5996];J[5996]=0;if((d|0)==1){break ua}}J[5996]=0;ja(57,b|0,c|0,18,a|0,0,0);a=J[5996];J[5996]=0;if((a|0)==1){break ua}J[5996]=0;ja(57,b|0,c|0,18,f+56|0,0,0);a=J[5996];J[5996]=0;if((a|0)==1){break ua}$a(e);break g}break a}hc(b,c,14,a+4|0,0,0);break g}hc(b,c,15,a+4|0,0,0);break g}hc(b,c,16,a+4|0,0,0);break g}hc(b,c,13,a+4|0,0,0);break g}hc(b,c,12,a+4|0,0,0);break g}a=a+4|0;hc(b,c,10,a,0,0);hc(b,c,11,a,0,0);break g}e=J[a+40>>2];k=J[a+44>>2];if(!K[24588]){J[6145]=0;J[6146]=0;J[6143]=0;J[6144]=0;J[6141]=0;J[6142]=0;H[24588]=1;}n=Yg(f+56|0);J[5996]=0;d=$(46,25)|0;g=J[5996];J[5996]=0;va:{wa:{if((g|0)!=1){J[f+52>>2]=0;J[f+44>>2]=0;J[f+48>>2]=0;H[f+84|0]=0;J[f+80>>2]=f+44;g=J[d+4>>2];i=J[d>>2];j=g-i|0;h=(j|0)/12|0;if((g|0)!=(i|0)){if(h>>>0>=357913942){J[5996]=0;ca(47);a=J[5996];J[5996]=0;if((a|0)!=1){break d}break wa}J[5996]=0;g=$(3,j|0)|0;i=J[5996];J[5996]=0;if((i|0)==1){break wa}J[f+48>>2]=g;J[f+44>>2]=g;J[f+52>>2]=g+P(h,12);h=J[d+4>>2];d=J[d>>2];J[5996]=0;d=ia(48,f+52|0,d|0,h|0,g|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){d=Z()|0;J[f+48>>2]=g;break va}J[f+48>>2]=d;}J[f+40>>2]=0;J[f+32>>2]=0;J[f+36>>2]=0;d=J[a+8>>2];g=J[a+4>>2];J[5996]=0;aa(49,f+32|0,d-g>>2);d=J[5996];J[5996]=0;xa:{if((d|0)!=1){ya:{za:{d=J[f+48>>2];g=J[f+44>>2];if((d|0)!=(g|0)){u=a+16|0;l=+(e>>>0)+ +(k|0)*4294967296;m=l+1;l=l*73+1;d=g;while(1){J[f+28>>2]=0;J[f+20>>2]=0;J[f+24>>2]=0;g=J[a+8>>2];e=J[a+4>>2];q=P(s,12);d=q+d|0;if(g-e>>2!=J[d+4>>2]-J[d>>2]>>3){break za}h=0;o=1;j=0;Aa:{if((e|0)!=(g|0)){Ba:{Ca:{Da:{while(1){d=j<<3;p=O[d+J[n+12>>2]>>3];v=O[d+J[n>>2]>>3];e=J[(j<<2)+e>>2];J[5996]=0;e=qa(50,e|0,+v,+p,c|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break Ca}d=d+J[q+J[f+44>>2]>>2]|0;g=J[d>>2];d=J[d+4>>2];k=J[e+4>>2];e=J[e>>2];J[5996]=0;oa(51,f+80|0,e|0,k|0,g|0,d|0);e=J[5996];J[5996]=0;if((e|0)!=1){g=J[f+28>>2];Ea:{if(g>>>0>h>>>0){J[h+8>>2]=0;J[h>>2]=0;J[h+4>>2]=0;J[h>>2]=J[f+80>>2];J[h+4>>2]=J[f+84>>2];J[h+8>>2]=J[f+88>>2];J[f+24>>2]=h+12;break Ea}Fa:{e=J[f+20>>2];k=(h-e|0)/12|0;d=k+1|0;Ga:{if(d>>>0>=357913942){J[5996]=0;ca(52);break Ga}g=(g-e|0)/12|0;i=g<<1;d=g>>>0>=178956970?357913941:d>>>0<i>>>0?i:d;if(d>>>0<357913942){break Fa}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break d}break Da}J[5996]=0;i=P(d,12);g=$(3,i|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break Da}d=g+P(k,12)|0;J[d>>2]=J[f+80>>2];J[d+4>>2]=J[f+84>>2];J[d+8>>2]=J[f+88>>2];J[f+88>>2]=0;J[f+80>>2]=0;J[f+84>>2]=0;g=g+i|0;k=d+12|0;Ha:{if((e|0)==(h|0)){J[f+28>>2]=g;J[f+24>>2]=k;J[f+20>>2]=d;break Ha}while(1){d=d-12|0;h=h-12|0;J[d>>2]=J[h>>2];J[d+4>>2]=J[h+4>>2];J[d+8>>2]=J[h+8>>2];J[h+8>>2]=0;J[h>>2]=0;J[h+4>>2]=0;if((e|0)!=(h|0)){continue}break}J[f+28>>2]=g;e=J[f+24>>2];J[f+24>>2]=k;h=J[f+20>>2];J[f+20>>2]=d;if((e|0)==(h|0)){break Ha}while(1){i=e-12|0;k=J[i>>2];if(k){r=e-8|0;d=J[r>>2];g=k;if((d|0)!=(g|0)){while(1){d=d-32|0;e=J[d+4>>2];Ia:{if(!e){break Ia}g=J[e+4>>2];J[e+4>>2]=g-1;if(g){break Ia}Xa[J[J[e>>2]+8>>2]](e);bb(e);}if((d|0)!=(k|0)){continue}break}g=J[i>>2];}J[r>>2]=k;$a(g);}e=i;if((h|0)!=(e|0)){continue}break}}if(h){$a(h);}g=J[f+80>>2];if(!g){break Ea}e=g;d=J[f+84>>2];if((e|0)!=(d|0)){while(1){d=d-32|0;e=J[d+4>>2];Ja:{if(!e){break Ja}h=J[e+4>>2];J[e+4>>2]=h-1;if(h){break Ja}Xa[J[J[e>>2]+8>>2]](e);bb(e);}if((d|0)!=(g|0)){continue}break}e=J[f+80>>2];}J[f+84>>2]=g;$a(e);}h=J[f+24>>2];e=h-12|0;o=P(J[e+4>>2]-J[e>>2]>>5,o);j=j+1|0;e=J[a+4>>2];if(j>>>0>=J[a+8>>2]-e>>2>>>0){break Ba}continue}break}d=Z()|0;break ya}d=Z()|0;_b(f+80|0);break ya}d=Z()|0;break ya}if(!o){break Aa}}k=0;Ka:{La:{Ma:{Na:{while(1){h=J[f+32>>2];if(J[a+8>>2]!=J[a+4>>2]){d=0;g=J[f+20>>2];e=k;while(1){i=P(d,12);j=i+g|0;g=J[j>>2];J[(d<<2)+h>>2]=g+((e>>>0)%(J[j+4>>2]-g>>5>>>0)<<5);j=e;g=J[f+20>>2];e=i+g|0;e=(j>>>0)/(J[e+4>>2]-J[e>>2]>>5>>>0)|0;d=d+1|0;if(d>>>0<J[a+8>>2]-J[a+4>>2]>>2>>>0){continue}break}}J[f+108>>2]=0;J[f+100>>2]=0;J[f+104>>2]=0;Oa:{Pa:{q=J[f+36>>2];if((q|0)!=(h|0)){while(1){i=J[h>>2];d=J[f+104>>2];Qa:{if((d|0)!=J[f+108>>2]){J[d>>2]=J[i>>2];e=J[i+4>>2];J[d+4>>2]=e;if(e){J[e+4>>2]=J[e+4>>2]+1;}J[f+104>>2]=d+8;break Qa}Ra:{g=J[f+100>>2];e=d-g|0;r=e>>3;j=r+1|0;Sa:{if(j>>>0>=536870912){J[5996]=0;ca(59);break Sa}t=e>>2;e=e>>>0>=2147483640?536870911:j>>>0<t>>>0?t:j;if(e>>>0<536870912){break Ra}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break d}d=Z()|0;break Ka}J[5996]=0;t=e<<3;j=$(3,t|0)|0;e=J[5996];J[5996]=0;if((e|0)==1){break Pa}e=j+(r<<3)|0;J[e>>2]=J[i>>2];i=J[i+4>>2];J[e+4>>2]=i;if(i){J[i+4>>2]=J[i+4>>2]+1;d=J[f+104>>2];}i=j+t|0;j=e+8|0;Ta:{if((d|0)==(g|0)){J[f+108>>2]=i;J[f+104>>2]=j;J[f+100>>2]=e;break Ta}while(1){e=e-8|0;d=d-8|0;J[e>>2]=J[d>>2];J[e+4>>2]=J[d+4>>2];J[d>>2]=0;J[d+4>>2]=0;if((d|0)!=(g|0)){continue}break}J[f+108>>2]=i;d=J[f+104>>2];J[f+104>>2]=j;g=J[f+100>>2];J[f+100>>2]=e;if((d|0)==(g|0)){break Ta}while(1){d=d-8|0;e=J[d+4>>2];Ua:{if(!e){break Ua}i=J[e+4>>2];J[e+4>>2]=i-1;if(i){break Ua}Xa[J[J[e>>2]+8>>2]](e);bb(e);}if((d|0)!=(g|0)){continue}break}}if(!g){break Qa}$a(g);}h=h+4|0;if((q|0)!=(h|0)){continue}break}}O[f+88>>3]=m;O[f+80>>3]=l;J[5996]=0;h=$(3,104)|0;e=J[5996];J[5996]=0;if((e|0)!=1){break Oa}break La}d=Z()|0;break Ka}J[h>>2]=7e3;J[h+4>>2]=0;J[h+8>>2]=0;J[5996]=0;J[f+132>>2]=J[f+100>>2];e=J[f+104>>2];d=J[f+108>>2];J[f+108>>2]=0;J[f+136>>2]=e;J[f+140>>2]=d;J[f+100>>2]=0;J[f+104>>2]=0;i=_(32,f+120|0,u|0)|0;e=J[5996];J[5996]=0;if((e|0)!=1){e=J[a+40>>2];J[5996]=0;j=xa(60,h+12|0,25,f+132|0,i|0,e|0)|0;e=J[5996];J[5996]=0;if((e|0)==1){break Na}g=J[i>>2];if(g){e=J[f+124>>2];d=g;if((e|0)!=(d|0)){while(1){d=e-12|0;if(H[e-1|0]<0){$a(J[d>>2]);}e=d;if((e|0)!=(g|0)){continue}break}d=J[i>>2];}J[f+124>>2]=g;$a(d);}e=J[f+132>>2];if(e){d=J[f+136>>2];if((e|0)!=(d|0)){while(1){d=d-8|0;g=J[d+4>>2];Va:{if(!g){break Va}i=J[g+4>>2];J[g+4>>2]=i-1;if(i){break Va}Xa[J[J[g>>2]+8>>2]](g);bb(g);}if((d|0)!=(e|0)){continue}break}e=J[f+132>>2];}$a(e);}J[f+116>>2]=h;J[f+112>>2]=j;J[5996]=0;e=J[f+116>>2];J[f+8>>2]=J[f+112>>2];J[f+12>>2]=e;ea(61,b|0,f+80|0,f+8|0);e=J[5996];J[5996]=0;if((e|0)==1){break La}e=J[f+100>>2];if(e){d=J[f+104>>2];if((e|0)!=(d|0)){while(1){d=d-8|0;g=J[d+4>>2];Wa:{if(!g){break Wa}h=J[g+4>>2];J[g+4>>2]=h-1;if(h){break Wa}Xa[J[J[g>>2]+8>>2]](g);bb(g);}if((d|0)!=(e|0)){continue}break}e=J[f+100>>2];}$a(e);}k=k+1|0;if((o|0)!=(k|0)){continue}break Aa}break}d=Z()|0;break Ma}d=Z()|0;rb(i);}lb(f+132|0);$a(h);break Ka}d=Z()|0;}lb(f+100|0);break ya}g=J[f+20>>2];if(g){d=J[f+24>>2];if((g|0)!=(d|0)){while(1){k=d-12|0;h=J[k>>2];if(h){e=h;i=d-8|0;d=J[i>>2];if((e|0)!=(d|0)){while(1){d=d-32|0;e=J[d+4>>2];Xa:{if(!e){break Xa}j=J[e+4>>2];J[e+4>>2]=j-1;if(j){break Xa}Xa[J[J[e>>2]+8>>2]](e);bb(e);}if((d|0)!=(h|0)){continue}break}e=J[k>>2];}J[i>>2]=h;$a(e);}d=k;if((g|0)!=(d|0)){continue}break}g=J[f+20>>2];}$a(g);}s=s+1|0;d=J[f+44>>2];if(s>>>0<(J[f+48>>2]-d|0)/12>>>0){continue}break}}a=J[f+32>>2];if(a){$a(a);d=J[f+44>>2];}if(d){e=d;g=J[f+48>>2];if((e|0)!=(g|0)){while(1){a=g-12|0;b=J[a>>2];if(b){J[g-8>>2]=b;$a(b);}g=a;if((a|0)!=(d|0)){continue}break}e=J[f+44>>2];}J[f+48>>2]=d;$a(e);}a=J[n+12>>2];if(a){J[n+16>>2]=a;$a(a);}a=J[n>>2];if(!a){break g}J[n+4>>2]=a;$a(a);break g}J[5996]=0;ga(44,5170,1900,642,2497);a=J[5996];J[5996]=0;if((a|0)!=1){break d}d=Z()|0;}Xc(f+20|0);break xa}d=Z()|0;}a=J[f+32>>2];if(a){$a(a);}Wc(f+44|0);break b}a=Z()|0;Mc(n);da(a|0);B();}d=Z()|0;}Nc(f+80|0);break b}e=a+4|0;a=a+16|0;Xg(b,c,2,e,a);Xg(b,c,3,e,a);break g}wd(b,c,6,a+4|0,a+28|0);break g}wd(b,c,8,a+4|0,a+28|0);break g}wd(b,c,7,a+4|0,a+28|0);break g}wd(b,c,9,a+4|0,a+28|0);}Ua=f+144|0;return}d=Z()|0;break c}a=J[f+132>>2];if(a){$a(a);}Wc(f+80|0);break c}B();}Mc(f+56|0);da(d|0);B();}Mc(n);da(d|0);B();}a=Z()|0;$a(e);da(a|0);B();}function ig(a,b,c,d){var e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,w=0,x=0,y=0,z=0,A=0,B=0,C=0,E=0,F=0,G=0;q=Ua-48|0;Ua=q;a:{if(c>>>0<=2){c=c<<2;A=J[c+8972>>2];B=J[c+8960>>2];while(1){c=J[b+4>>2];b:{if((c|0)!=J[b+104>>2]){J[b+4>>2]=c+1;c=K[c|0];break b}c=gb(b);}if((c|0)==32|c-9>>>0<5){continue}break}t=1;c:{d:{switch(c-43|0){case 0:case 2:break d;default:break c}}t=(c|0)==45?-1:1;c=J[b+4>>2];if((c|0)!=J[b+104>>2]){J[b+4>>2]=c+1;c=K[c|0];break c}c=gb(b);}e:{f:{while(1){if(H[g+1031|0]==(c|32)){g:{if(g>>>0>6){break g}c=J[b+4>>2];if((c|0)!=J[b+104>>2]){J[b+4>>2]=c+1;c=K[c|0];break g}c=gb(b);}g=g+1|0;if((g|0)!=8){continue}break f}break}if((g|0)!=3){m=(g|0)==8;if(m){break f}if(!d|g>>>0<4){break e}if(m){break f}}c=J[b+116>>2];if((c|0)>0|(c|0)>=0){J[b+4>>2]=J[b+4>>2]-1;}if(!d|g>>>0<4){break f}c=(c|0)<0;while(1){if(!c){J[b+4>>2]=J[b+4>>2]-1;}g=g-1|0;if(g>>>0>3){continue}break}}l=Ua-16|0;Ua=l;h=(D(Q(Q(t|0)*Q(Infinity))),v(2));b=h&2147483647;h:{if(b-8388608>>>0<=2130706431){c=b;b=b>>>7|0;c=c<<25;d=b+1065353216|0;break h}c=h<<25;d=h>>>7|2147418112;if(b>>>0>=2139095040){break h}c=0;d=0;if(!b){break h}c=b;b=S(b);Cb(l,c,0,0,0,b+81|0);j=J[l>>2];i=J[l+4>>2];c=J[l+8>>2];d=J[l+12>>2]^65536|16265-b<<16;}J[q>>2]=j;J[q+4>>2]=i;J[q+8>>2]=c;J[q+12>>2]=h&-2147483648|d;Ua=l+16|0;j=J[q+8>>2];i=J[q+12>>2];h=J[q>>2];k=J[q+4>>2];break a}i:{j:{k:{if(g){break k}g=0;while(1){if(H[g+2020|0]!=(c|32)){break k}l:{if(g>>>0>1){break l}c=J[b+4>>2];if((c|0)!=J[b+104>>2]){J[b+4>>2]=c+1;c=K[c|0];break l}c=gb(b);}g=g+1|0;if((g|0)!=3){continue}break}break j}m:{switch(g|0){case 0:n:{if((c|0)!=48){break n}g=J[b+4>>2];o:{if((g|0)!=J[b+104>>2]){J[b+4>>2]=g+1;g=K[g|0];break o}g=gb(b);}if((g&-33)==88){m=0;f=Ua-432|0;Ua=f;c=J[b+4>>2];p:{if((c|0)!=J[b+104>>2]){J[b+4>>2]=c+1;g=K[c|0];break p}g=gb(b);}q:{r:{while(1){if((g|0)!=48){s:{if((g|0)!=46){break q}c=J[b+4>>2];if((c|0)==J[b+104>>2]){break s}J[b+4>>2]=c+1;g=K[c|0];break r}}else {c=J[b+4>>2];if((c|0)!=J[b+104>>2]){e=1;J[b+4>>2]=c+1;g=K[c|0];}else {e=1;g=gb(b);}continue}break}g=gb(b);}p=1;if((g|0)!=48){break q}while(1){c=r;r=c-1|0;s=s-!c|0;c=J[b+4>>2];t:{if((c|0)!=J[b+104>>2]){J[b+4>>2]=c+1;g=K[c|0];break t}g=gb(b);}if((g|0)==48){continue}break}e=1;}k=1073676288;u:{while(1){v:{c=g|32;w:{x:{E=g-48|0;if(E>>>0<10){break x}F=(g|0)!=46;if(F&c-97>>>0>5){break u}if(F){break x}if(p){break v}p=1;r=j;s=i;break w}c=(g|0)>57?c-87|0:E;y:{if((i|0)<=0&j>>>0<=7|(i|0)<0){m=c+(m<<4)|0;break y}if(!i&j>>>0<=28){Nb(f+48|0,c);ob(f+32|0,y,z,h,k,0,0,0,1073414144);y=J[f+32>>2];z=J[f+36>>2];h=J[f+40>>2];k=J[f+44>>2];ob(f+16|0,J[f+48>>2],J[f+52>>2],J[f+56>>2],J[f+60>>2],y,z,h,k);Hb(f,J[f+16>>2],J[f+20>>2],J[f+24>>2],J[f+28>>2],l,o,w,x);w=J[f+8>>2];x=J[f+12>>2];l=J[f>>2];o=J[f+4>>2];break y}if(C|!c){break y}ob(f+80|0,y,z,h,k,0,0,0,1073610752);Hb(f- -64|0,J[f+80>>2],J[f+84>>2],J[f+88>>2],J[f+92>>2],l,o,w,x);w=J[f+72>>2];x=J[f+76>>2];C=1;l=J[f+64>>2];o=J[f+68>>2];}j=j+1|0;i=j?i:i+1|0;e=1;}c=J[b+4>>2];if((c|0)!=J[b+104>>2]){J[b+4>>2]=c+1;g=K[c|0];}else {g=gb(b);}continue}break}g=46;}z:{if(!e){c=J[b+116>>2];A:{B:{if((c|0)>0|(c|0)>=0){c=J[b+4>>2];J[b+4>>2]=c-1;if(!d){break B}J[b+4>>2]=c-2;if(!p){break A}J[b+4>>2]=c-3;break A}if(d){break A}}Yb(b,0,0);}fc(f+96|0,+(t|0)*0);l=J[f+96>>2];o=J[f+100>>2];c=J[f+108>>2];b=J[f+104>>2];break z}if((i|0)<=0&j>>>0<=7|(i|0)<0){h=j;k=i;while(1){m=m<<4;h=h+1|0;k=h?k:k+1|0;if((h|0)!=8|k){continue}break}}C:{D:{E:{if((g&-33)==80){h=hg(b,d);c=Wa;k=c;if(h|(c|0)!=-2147483648){break C}if(d){c=J[b+116>>2];if((c|0)>0|(c|0)>=0){break E}break D}l=0;o=0;Yb(b,0,0);c=0;b=0;break z}h=0;k=0;if(J[b+116>>2]<0){break C}}J[b+4>>2]=J[b+4>>2]-1;}h=0;k=0;}if(!m){fc(f+112|0,+(t|0)*0);l=J[f+112>>2];o=J[f+116>>2];c=J[f+124>>2];b=J[f+120>>2];break z}b=p?r:j;i=(p?s:i)<<2|b>>>30;c=h+(b<<2)|0;b=i+k|0;j=c-32|0;i=(c>>>0<h>>>0?b+1|0:b)-(c>>>0<32)|0;b=i;if(j>>>0>0-A>>>0&(b|0)>=0|(b|0)>0){J[5732]=68;Nb(f+160|0,t);ob(f+144|0,J[f+160>>2],J[f+164>>2],J[f+168>>2],J[f+172>>2],-1,-1,-1,2147418111);ob(f+128|0,J[f+144>>2],J[f+148>>2],J[f+152>>2],J[f+156>>2],-1,-1,-1,2147418111);l=J[f+128>>2];o=J[f+132>>2];c=J[f+140>>2];b=J[f+136>>2];break z}b=A-226|0;c=b>>31;if((i|0)>=(c|0)&b>>>0<=j>>>0|(c|0)<(i|0)){if((m|0)>=0){while(1){Hb(f+416|0,l,o,w,x,0,0,0,-1073807360);b=Rg(l,o,w,x,1073610752);c=(b|0)>=0;b=c;Hb(f+400|0,l,o,w,x,b?J[f+416>>2]:l,b?J[f+420>>2]:o,b?J[f+424>>2]:w,b?J[f+428>>2]:x);b=j;j=b-1|0;i=i-!b|0;w=J[f+408>>2];x=J[f+412>>2];l=J[f+400>>2];o=J[f+404>>2];m=c|m<<1;if((m|0)>=0){continue}break}}b=i-((A>>31)+(j>>>0<A>>>0)|0)|0;c=(j-A|0)+32|0;b=c>>>0<32?b+1|0:b;c=c>>>0<B>>>0&(b|0)<=0|(b|0)<0?(c|0)>0?c:0:B;F:{if((c|0)>=113){Nb(f+384|0,t);r=J[f+392>>2];s=J[f+396>>2];y=J[f+384>>2];z=J[f+388>>2];h=0;b=0;break F}fc(f+352|0,xe(144-c|0));Nb(f+336|0,t);y=J[f+336>>2];z=J[f+340>>2];r=J[f+344>>2];s=J[f+348>>2];kg(f+368|0,J[f+352>>2],J[f+356>>2],J[f+360>>2],J[f+364>>2],y,z,r,s);n=J[f+376>>2];u=J[f+380>>2];h=J[f+372>>2];b=J[f+368>>2];}d=!(m&1)&((yc(l,o,w,x,0,0,0,0)|0)!=0&(c|0)<32);Ec(f+320|0,d+m|0);ob(f+304|0,y,z,r,s,J[f+320>>2],J[f+324>>2],J[f+328>>2],J[f+332>>2]);c=b;Hb(f+272|0,J[f+304>>2],J[f+308>>2],J[f+312>>2],J[f+316>>2],b,h,n,u);b=d;ob(f+288|0,y,z,r,s,b?0:l,b?0:o,b?0:w,b?0:x);Hb(f+256|0,J[f+288>>2],J[f+292>>2],J[f+296>>2],J[f+300>>2],J[f+272>>2],J[f+276>>2],J[f+280>>2],J[f+284>>2]);he(f+240|0,J[f+256>>2],J[f+260>>2],J[f+264>>2],J[f+268>>2],c,h,n,u);b=J[f+240>>2];d=J[f+244>>2];c=J[f+248>>2];h=J[f+252>>2];if(!yc(b,d,c,h,0,0,0,0)){J[5732]=68;}jg(f+224|0,b,d,c,h,j);l=J[f+224>>2];o=J[f+228>>2];c=J[f+236>>2];b=J[f+232>>2];break z}J[5732]=68;Nb(f+208|0,t);ob(f+192|0,J[f+208>>2],J[f+212>>2],J[f+216>>2],J[f+220>>2],0,0,0,65536);ob(f+176|0,J[f+192>>2],J[f+196>>2],J[f+200>>2],J[f+204>>2],0,0,0,65536);l=J[f+176>>2];o=J[f+180>>2];c=J[f+188>>2];b=J[f+184>>2];}J[q+16>>2]=l;J[q+20>>2]=o;J[q+24>>2]=b;J[q+28>>2]=c;Ua=f+432|0;j=J[q+24>>2];i=J[q+28>>2];h=J[q+16>>2];k=J[q+20>>2];break a}if(J[b+116>>2]<0){break n}J[b+4>>2]=J[b+4>>2]-1;}g=b;m=c;f=d;d=0;e=Ua-8976|0;Ua=e;F=0-A|0;E=F-B|0;G:{H:{while(1){if((m|0)!=48){I:{if((m|0)!=46){break G}b=J[g+4>>2];if((b|0)==J[g+104>>2]){break I}J[g+4>>2]=b+1;m=K[b|0];break H}}else {b=J[g+4>>2];if((b|0)!=J[g+104>>2]){J[g+4>>2]=b+1;m=K[b|0];}else {m=gb(g);}d=1;continue}break}m=gb(g);}C=1;if((m|0)!=48){break G}while(1){b=j;j=b-1|0;i=i-!b|0;b=J[g+4>>2];J:{if((b|0)!=J[g+104>>2]){J[g+4>>2]=b+1;m=K[b|0];break J}m=gb(g);}if((m|0)==48){continue}break}d=1;}J[e+784>>2]=0;K:{L:{b=(m|0)==46;c=m-48|0;M:{N:{O:{if(b|c>>>0<=9){while(1){P:{if(b&1){if(!C){j=h;i=k;C=1;break P}b=!d;break O}h=h+1|0;k=h?k:k+1|0;if((p|0)<=2044){u=(m|0)==48?u:h;b=(e+784|0)+(p<<2)|0;if(n){c=(P(J[b>>2],10)+m|0)-48|0;}J[b>>2]=c;d=1;c=n+1|0;b=(c|0)==9;n=b?0:c;p=b+p|0;break P}if((m|0)==48){break P}J[e+8960>>2]=J[e+8960>>2]|1;u=18396;}b=J[g+4>>2];Q:{if((b|0)!=J[g+104>>2]){J[g+4>>2]=b+1;m=K[b|0];break Q}m=gb(g);}b=(m|0)==46;c=m-48|0;if(b|c>>>0<10){continue}break}}j=C?j:h;i=C?i:k;if(!(!d|(m&-33)!=69)){l=hg(g,f);b=Wa;o=b;R:{if(l|(b|0)!=-2147483648){break R}if(!f){break M}l=0;o=0;if(J[g+116>>2]<0){break R}J[g+4>>2]=J[g+4>>2]-1;}i=i+o|0;j=j+l|0;i=j>>>0<l>>>0?i+1|0:i;break L}b=!d;if((m|0)<0){break N}}if(J[g+116>>2]<0){break N}J[g+4>>2]=J[g+4>>2]-1;}if(!b){break L}J[5732]=28;}h=0;k=0;Yb(g,0,0);c=0;b=0;break K}b=J[e+784>>2];if(!b){fc(e,+(t|0)*0);h=J[e>>2];k=J[e+4>>2];c=J[e+12>>2];b=J[e+8>>2];break K}if(!(h>>>0>9&(k|0)>=0|(k|0)>0|((h|0)!=(j|0)|(i|0)!=(k|0))|(b>>>B|0?(B|0)<=30:0))){Nb(e+48|0,t);Ec(e+32|0,b);ob(e+16|0,J[e+48>>2],J[e+52>>2],J[e+56>>2],J[e+60>>2],J[e+32>>2],J[e+36>>2],J[e+40>>2],J[e+44>>2]);h=J[e+16>>2];k=J[e+20>>2];c=J[e+28>>2];b=J[e+24>>2];break K}if(j>>>0>F>>>1>>>0&(i|0)>=0|(i|0)>0){J[5732]=68;Nb(e+96|0,t);ob(e+80|0,J[e+96>>2],J[e+100>>2],J[e+104>>2],J[e+108>>2],-1,-1,-1,2147418111);ob(e- -64|0,J[e+80>>2],J[e+84>>2],J[e+88>>2],J[e+92>>2],-1,-1,-1,2147418111);h=J[e+64>>2];k=J[e+68>>2];c=J[e+76>>2];b=J[e+72>>2];break K}b=A-226|0;c=j>>>0<b>>>0;b=b>>31;if(c&(i|0)<=(b|0)|(b|0)>(i|0)){J[5732]=68;Nb(e+144|0,t);ob(e+128|0,J[e+144>>2],J[e+148>>2],J[e+152>>2],J[e+156>>2],0,0,0,65536);ob(e+112|0,J[e+128>>2],J[e+132>>2],J[e+136>>2],J[e+140>>2],0,0,0,65536);h=J[e+112>>2];k=J[e+116>>2];c=J[e+124>>2];b=J[e+120>>2];break K}if(n){if((n|0)<=8){b=(e+784|0)+(p<<2)|0;g=J[b>>2];while(1){g=P(g,10);n=n+1|0;if((n|0)!=9){continue}break}J[b>>2]=g;}p=p+1|0;}n=j;S:{if((j|0)<(u|0)|(u|0)>=9|(j|0)>17){break S}if((j|0)==9){Nb(e+192|0,t);Ec(e+176|0,J[e+784>>2]);ob(e+160|0,J[e+192>>2],J[e+196>>2],J[e+200>>2],J[e+204>>2],J[e+176>>2],J[e+180>>2],J[e+184>>2],J[e+188>>2]);h=J[e+160>>2];k=J[e+164>>2];c=J[e+172>>2];b=J[e+168>>2];break K}if((n|0)<=8){Nb(e+272|0,t);Ec(e+256|0,J[e+784>>2]);ob(e+240|0,J[e+272>>2],J[e+276>>2],J[e+280>>2],J[e+284>>2],J[e+256>>2],J[e+260>>2],J[e+264>>2],J[e+268>>2]);Nb(e+224|0,J[(0-n<<2)+8960>>2]);Ng(e+208|0,J[e+240>>2],J[e+244>>2],J[e+248>>2],J[e+252>>2],J[e+224>>2],J[e+228>>2],J[e+232>>2],J[e+236>>2]);h=J[e+208>>2];k=J[e+212>>2];c=J[e+220>>2];b=J[e+216>>2];break K}b=(P(n,-3)+B|0)+27|0;c=J[e+784>>2];if(c>>>b|0?(b|0)<=30:0){break S}Nb(e+352|0,t);Ec(e+336|0,c);ob(e+320|0,J[e+352>>2],J[e+356>>2],J[e+360>>2],J[e+364>>2],J[e+336>>2],J[e+340>>2],J[e+344>>2],J[e+348>>2]);Nb(e+304|0,J[(n<<2)+8888>>2]);ob(e+288|0,J[e+320>>2],J[e+324>>2],J[e+328>>2],J[e+332>>2],J[e+304>>2],J[e+308>>2],J[e+312>>2],J[e+316>>2]);h=J[e+288>>2];k=J[e+292>>2];c=J[e+300>>2];b=J[e+296>>2];break K}while(1){b=p;p=b-1|0;if(!J[(e+784|0)+(p<<2)>>2]){continue}break}u=0;d=(n|0)%9|0;T:{if(!d){c=0;break T}c=0;d=(n|0)<0?d+9|0:d;U:{if(!b){b=0;break U}j=J[(0-d<<2)+8960>>2];l=1e9/(j|0)|0;m=0;g=0;while(1){h=m;i=e+784|0;k=i+(g<<2)|0;m=J[k>>2];p=(m>>>0)/(j>>>0)|0;h=h+p|0;J[k>>2]=h;h=!h&(c|0)==(g|0);c=h?c+1&2047:c;n=h?n-9|0:n;m=P(l,m-P(j,p)|0);g=g+1|0;if((g|0)!=(b|0)){continue}break}if(!m){break U}J[i+(b<<2)>>2]=m;b=b+1|0;}n=(n-d|0)+9|0;}while(1){g=(e+784|0)+(c<<2)|0;V:{while(1){W:{if((n|0)<36){break W}if((n|0)!=36){break V}if(M[g>>2]<=10384592){break W}n=36;break V}p=b+2047|0;d=0;while(1){h=b;l=p&2047;m=(e+784|0)+(l<<2)|0;b=J[m>>2];k=b>>>3|0;j=d;d=b<<29;j=j+d|0;b=k;i=d>>>0>j>>>0?b+1|0:b;if(!i&j>>>0<1000000001){d=0;}else {d=eo(j,i,1e9);b=co(d,Wa,-1e9,-1);j=b+j|0;}J[m>>2]=j;b=(c|0)==(l|0)?h:j?h:l;j=h-1&2047;b=(j|0)!=(l|0)?h:b;p=l-1|0;if((c|0)!=(l|0)){continue}break}u=u-29|0;b=h;if(!d){continue}break}c=c-1&2047;if((c|0)==(b|0)){g=(b+2046&2047)<<2;b=e+784|0;h=g+b|0;J[h>>2]=J[h>>2]|J[b+(j<<2)>>2];b=j;}n=n+9|0;J[(e+784|0)+(c<<2)>>2]=d;continue}break}X:{Y:while(1){j=b+1&2047;l=(e+784|0)+((b-1&2047)<<2)|0;while(1){h=(n|0)>45?9:1;Z:{while(1){d=c;g=0;_:{while(1){$:{c=d+g&2047;if((c|0)==(b|0)){break $}c=J[(e+784|0)+(c<<2)>>2];i=J[(g<<2)+8912>>2];if(c>>>0<i>>>0){break $}if(c>>>0>i>>>0){break _}g=g+1|0;if((g|0)!=4){continue}}break}if((n|0)!=36){break _}g=0;j=0;i=0;h=0;k=0;while(1){c=d+g&2047;if((c|0)==(b|0)){b=b+1&2047;J[(e+(b<<2)|0)+780>>2]=0;}Ec(e+768|0,J[(e+784|0)+(c<<2)>>2]);ob(e+752|0,j,i,h,k,0,0,1342177280,1075633366);Hb(e+736|0,J[e+752>>2],J[e+756>>2],J[e+760>>2],J[e+764>>2],J[e+768>>2],J[e+772>>2],J[e+776>>2],J[e+780>>2]);h=J[e+744>>2];k=J[e+748>>2];j=J[e+736>>2];i=J[e+740>>2];g=g+1|0;if((g|0)!=4){continue}break}Nb(e+720|0,t);ob(e+704|0,j,i,h,k,J[e+720>>2],J[e+724>>2],J[e+728>>2],J[e+732>>2]);h=J[e+712>>2];k=J[e+716>>2];j=0;i=0;l=J[e+704>>2];o=J[e+708>>2];p=u+113|0;g=p-A|0;m=(g|0)<(B|0);c=m?(g|0)>0?g:0:B;if((c|0)<=112){break Z}break X}u=h+u|0;c=b;if((b|0)==(d|0)){continue}break}k=1e9>>>h|0;m=-1<<h^-1;g=0;c=d;while(1){i=g;p=e+784|0;g=p+(d<<2)|0;f=J[g>>2];i=i+(f>>>h|0)|0;J[g>>2]=i;i=!i&(c|0)==(d|0);c=i?c+1&2047:c;n=i?n-9|0:n;g=P(k,f&m);d=d+1&2047;if((d|0)!=(b|0)){continue}break}if(!g){continue}if((c|0)!=(j|0)){J[p+(b<<2)>>2]=g;b=j;continue Y}J[l>>2]=J[l>>2]|1;continue}break}break}fc(e+656|0,xe(225-c|0));kg(e+688|0,J[e+656>>2],J[e+660>>2],J[e+664>>2],J[e+668>>2],l,o,h,k);y=J[e+696>>2];z=J[e+700>>2];w=J[e+688>>2];x=J[e+692>>2];fc(e+640|0,xe(113-c|0));Hg(e+672|0,l,o,h,k,J[e+640>>2],J[e+644>>2],J[e+648>>2],J[e+652>>2]);j=J[e+672>>2];i=J[e+676>>2];r=J[e+680>>2];s=J[e+684>>2];he(e+624|0,l,o,h,k,j,i,r,s);Hb(e+608|0,w,x,y,z,J[e+624>>2],J[e+628>>2],J[e+632>>2],J[e+636>>2]);h=J[e+616>>2];k=J[e+620>>2];l=J[e+608>>2];o=J[e+612>>2];}n=d+4&2047;aa:{if((n|0)==(b|0)){break aa}n=J[(e+784|0)+(n<<2)>>2];ba:{if(n>>>0<=499999999){if(!n&(d+5&2047)==(b|0)){break ba}fc(e+496|0,+(t|0)*.25);Hb(e+480|0,j,i,r,s,J[e+496>>2],J[e+500>>2],J[e+504>>2],J[e+508>>2]);r=J[e+488>>2];s=J[e+492>>2];j=J[e+480>>2];i=J[e+484>>2];break ba}if((n|0)!=5e8){fc(e+592|0,+(t|0)*.75);Hb(e+576|0,j,i,r,s,J[e+592>>2],J[e+596>>2],J[e+600>>2],J[e+604>>2]);r=J[e+584>>2];s=J[e+588>>2];j=J[e+576>>2];i=J[e+580>>2];break ba}G=+(t|0);if((d+5&2047)==(b|0)){fc(e+528|0,G*.5);Hb(e+512|0,j,i,r,s,J[e+528>>2],J[e+532>>2],J[e+536>>2],J[e+540>>2]);r=J[e+520>>2];s=J[e+524>>2];j=J[e+512>>2];i=J[e+516>>2];break ba}fc(e+560|0,G*.75);Hb(e+544|0,j,i,r,s,J[e+560>>2],J[e+564>>2],J[e+568>>2],J[e+572>>2]);r=J[e+552>>2];s=J[e+556>>2];j=J[e+544>>2];i=J[e+548>>2];}if((c|0)>111){break aa}Hg(e+464|0,j,i,r,s,0,0,0,1073676288);if(yc(J[e+464>>2],J[e+468>>2],J[e+472>>2],J[e+476>>2],0,0,0,0)){break aa}Hb(e+448|0,j,i,r,s,0,0,0,1073676288);r=J[e+456>>2];s=J[e+460>>2];j=J[e+448>>2];i=J[e+452>>2];}Hb(e+432|0,l,o,h,k,j,i,r,s);he(e+416|0,J[e+432>>2],J[e+436>>2],J[e+440>>2],J[e+444>>2],w,x,y,z);h=J[e+424>>2];k=J[e+428>>2];l=J[e+416>>2];o=J[e+420>>2];ca:{if((E-2|0)>=(p&2147483647)){break ca}J[e+408>>2]=h;J[e+412>>2]=k&2147483647;J[e+400>>2]=l;J[e+404>>2]=o;ob(e+384|0,l,o,h,k,0,0,0,1073610752);d=Rg(J[e+400>>2],J[e+404>>2],J[e+408>>2],J[e+412>>2],1081081856);b=(d|0)>=0;h=b?J[e+392>>2]:h;k=b?J[e+396>>2]:k;l=b?J[e+384>>2]:l;o=b?J[e+388>>2]:o;u=b+u|0;if(!(m&((c|0)!=(g|0)|(d|0)<0)&(yc(j,i,r,s,0,0,0,0)|0)!=0)&(u+110|0)<=(E|0)){break ca}J[5732]=68;}jg(e+368|0,l,o,h,k,u);h=J[e+368>>2];k=J[e+372>>2];c=J[e+380>>2];b=J[e+376>>2];}J[q+40>>2]=b;J[q+44>>2]=c;J[q+32>>2]=h;J[q+36>>2]=k;Ua=e+8976|0;j=J[q+40>>2];i=J[q+44>>2];h=J[q+32>>2];k=J[q+36>>2];break a;case 3:break j;default:break m}}c=J[b+116>>2];if((c|0)>0|(c|0)>=0){J[b+4>>2]=J[b+4>>2]-1;}break i}da:{c=J[b+4>>2];ea:{if((c|0)!=J[b+104>>2]){J[b+4>>2]=c+1;c=K[c|0];break ea}c=gb(b);}if((c|0)==40){g=1;break da}i=2147450880;if(J[b+116>>2]<0){break a}J[b+4>>2]=J[b+4>>2]-1;break a}while(1){fa:{c=J[b+4>>2];ga:{if((c|0)!=J[b+104>>2]){J[b+4>>2]=c+1;c=K[c|0];break ga}c=gb(b);}if(!(c-48>>>0<10|c-65>>>0<26|(c|0)==95)){if(c-97>>>0>=26){break fa}}g=g+1|0;continue}break}i=2147450880;if((c|0)==41){break a}c=J[b+116>>2];if((c|0)>0|(c|0)>=0){J[b+4>>2]=J[b+4>>2]-1;}ha:{if(d){if(g){break ha}break a}break i}while(1){if((c|0)>0|(c|0)>=0){J[b+4>>2]=J[b+4>>2]-1;}g=g-1|0;if(g){continue}break}break a}J[5732]=28;Yb(b,0,0);}i=0;}J[a>>2]=h;J[a+4>>2]=k;J[a+8>>2]=j;J[a+12>>2]=i;Ua=q+48|0;}function Bd(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0,h=0,i=0,j=0;c=Ua-112|0;Ua=c;f=J[b>>2];e=f;i=J[b+4>>2];a:{if(!i){break a}d=f+i|0;while(1){g=K[e|0];b:{if(!((g|0)!=123&(g|0)!=40)){h=h+1|0;break b}if(h){if((g|0)!=125&(g|0)!=41){break b}h=h-1|0;break b}h=0;c:{switch(g-41|0){case 0:case 3:break a;case 1:case 2:break b;default:break c}}if((g|0)==125){break a}}e=e+1|0;if((d|0)!=(e|0)){continue}break}e=d;}J[c+104>>2]=f;d=e-f|0;J[c+108>>2]=d;J[b+4>>2]=i-d;J[b>>2]=e;H[c+18|0]=0;I[c+16>>1]=27504;H[c+27|0]=2;d:{e:{f:{g:{h:{i:{j:{k:{if(K[f+2|0]!=40|d>>>0<4|(K[(d+f|0)-1|0]!=41|K[c+16|0]!=K[f|0])){break k}if(K[c+17|0]!=K[f+1|0]){break k}b=d-4|0;J[c+108>>2]=b;d=f+3|0;J[c+104>>2]=d;if(!(!b|(b|0)>17)){if(b>>>0>=2147483632){break g}l:{if(b>>>0<=10){H[c+27|0]=b;h=c+16|0;break l}e=(b|15)+1|0;h=cb(e);J[c+24>>2]=e|-2147483648;J[c+16>>2]=h;J[c+20>>2]=b;}H[fb(h,d,b)+b|0]=0;J[c+88>>2]=J[c+20>>2];b=K[c+23|0]|K[c+24|0]<<8|(K[c+25|0]<<16|K[c+26|0]<<24);H[c+91|0]=b;H[c+92|0]=b>>>8;H[c+93|0]=b>>>16;H[c+94|0]=b>>>24;J[5996]=0;d=J[c+16>>2];f=H[c+27|0];J[c+24>>2]=0;J[c+16>>2]=0;J[c+20>>2]=0;b=$(3,12)|0;e=J[5996];J[5996]=0;if((e|0)!=1){J[b>>2]=d;J[b+4>>2]=J[c+88>>2];d=K[c+91|0]|K[c+92|0]<<8|(K[c+93|0]<<16|K[c+94|0]<<24);H[b+7|0]=d;H[b+8|0]=d>>>8;H[b+9|0]=d>>>16;H[b+10|0]=d>>>24;H[b+11|0]=f;J[a+4>>2]=0;J[a+8>>2]=0;J[a>>2]=1;J[a+12>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+24>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;d=b+12|0;J[a+36>>2]=d;J[a+32>>2]=d;J[a+28>>2]=b;J[a+48>>2]=0;J[a+52>>2]=0;break h}e=Z()|0;rb(c+16|0);if((f|0)>=0){break j}$a(d);da(e|0);B();}J[a>>2]=0;J[a+4>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+8>>2]=0;J[a+12>>2]=0;break h}H[c+21|0]=0;H[c+20|0]=K[1777];H[c+27|0]=5;J[c+16>>2]=K[1773]|K[1774]<<8|(K[1775]<<16|K[1776]<<24);m:{b=d>>>0<7;if(b|K[f+5|0]!=40|(K[(d+f|0)-1|0]!=41|K[c+16|0]!=K[f|0])){break m}if(K[c+17|0]!=K[f+1|0]|K[c+18|0]!=K[f+2|0]|(K[c+19|0]!=K[f+3|0]|K[c+20|0]!=K[f+4|0])){break m}g=d-7|0;J[c+108>>2]=g;d=f+6|0;J[c+104>>2]=d;if(g>>>0>=2147483632){break g}n:{o:{if(g>>>0>=11){b=(g|15)+1|0;e=cb(b);J[c+24>>2]=b|-2147483648;J[c+16>>2]=e;J[c+20>>2]=g;break o}H[c+27|0]=g;e=c+16|0;if(!g){break n}}f=g-1|0;i=g&7;p:{if(!i){b=d;break p}h=0;b=d;while(1){H[e|0]=K[b|0];e=e+1|0;b=b+1|0;h=h+1|0;if((i|0)!=(h|0)){continue}break}}if(f>>>0<7){break n}d=d+g|0;while(1){H[e|0]=K[b|0];H[e+1|0]=K[b+1|0];H[e+2|0]=K[b+2|0];H[e+3|0]=K[b+3|0];H[e+4|0]=K[b+4|0];H[e+5|0]=K[b+5|0];H[e+6|0]=K[b+6|0];H[e+7|0]=K[b+7|0];e=e+8|0;b=b+8|0;if((d|0)!=(b|0)){continue}break}}H[e|0]=0;b=jf(c+16|0,c+88|0);if(H[c+27|0]<0){$a(J[c+16>>2]);}if(!b){J[a>>2]=0;J[a+4>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+8>>2]=0;J[a+12>>2]=0;break h}b=J[c+88>>2];d=b-1|0;if(J[c+92>>2]==(!b|0)&d>>>0<=2147483646){J[a+4>>2]=0;J[a+8>>2]=0;J[a>>2]=3;J[a+52>>2]=b;J[a+12>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+48>>2]=0;break h}J[a>>2]=0;J[a+4>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+8>>2]=0;J[a+12>>2]=0;break h}H[c+21|0]=0;H[c+20|0]=K[1783];H[c+27|0]=5;J[c+16>>2]=K[1779]|K[1780]<<8|(K[1781]<<16|K[1782]<<24);q:{if(K[f+5|0]!=40|b|(K[(d+f|0)-1|0]!=41|K[c+16|0]!=K[f|0])){break q}if(K[c+17|0]!=K[f+1|0]|K[c+18|0]!=K[f+2|0]|(K[c+19|0]!=K[f+3|0]|K[c+20|0]!=K[f+4|0])){break q}g=d-7|0;J[c+108>>2]=g;d=f+6|0;J[c+104>>2]=d;if(g>>>0>=2147483632){break g}r:{s:{if(g>>>0>=11){b=(g|15)+1|0;e=cb(b);J[c+24>>2]=b|-2147483648;J[c+16>>2]=e;J[c+20>>2]=g;break s}H[c+27|0]=g;e=c+16|0;if(!g){break r}}f=g-1|0;i=g&7;t:{if(!i){b=d;break t}h=0;b=d;while(1){H[e|0]=K[b|0];e=e+1|0;b=b+1|0;h=h+1|0;if((i|0)!=(h|0)){continue}break}}if(f>>>0<7){break r}d=d+g|0;while(1){H[e|0]=K[b|0];H[e+1|0]=K[b+1|0];H[e+2|0]=K[b+2|0];H[e+3|0]=K[b+3|0];H[e+4|0]=K[b+4|0];H[e+5|0]=K[b+5|0];H[e+6|0]=K[b+6|0];H[e+7|0]=K[b+7|0];e=e+8|0;b=b+8|0;if((d|0)!=(b|0)){continue}break}}H[e|0]=0;b=jf(c+16|0,c+88|0);if(H[c+27|0]<0){$a(J[c+16>>2]);}if(!b){J[a>>2]=0;J[a+4>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+8>>2]=0;J[a+12>>2]=0;break h}b=J[c+88>>2];d=b-1|0;if(J[c+92>>2]==(!b|0)&d>>>0<=2147483646){J[a+4>>2]=0;J[a+8>>2]=0;J[a>>2]=2;J[a+52>>2]=b;J[a+12>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+48>>2]=0;break h}J[a>>2]=0;J[a+4>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+8>>2]=0;J[a+12>>2]=0;break h}H[c+22|0]=0;I[c+20>>1]=K[3881]|K[3882]<<8;H[c+27|0]=6;J[c+16>>2]=K[3877]|K[3878]<<8|(K[3879]<<16|K[3880]<<24);u:{if(K[f+6|0]!=40|d>>>0<8|(K[(d+f|0)-1|0]!=41|K[c+16|0]!=K[f|0])){break u}if(K[c+17|0]!=K[f+1|0]|K[c+18|0]!=K[f+2|0]|(K[c+19|0]!=K[f+3|0]|K[c+20|0]!=K[f+4|0])){break u}if(K[c+21|0]!=K[f+5|0]){break u}b=d-8|0;J[c+108>>2]=b;d=f+7|0;J[c+104>>2]=d;$c(c+16|0,d,b,32);b=J[c+20>>2];d=J[c+16>>2];if((b|0)!=(d|0)){J[a+4>>2]=0;J[a+8>>2]=0;J[a>>2]=7;J[a+16>>2]=d;J[a+12>>2]=0;J[a+20>>2]=b;b=J[c+24>>2];J[a+28>>2]=0;J[a+32>>2]=0;J[a+24>>2]=b;J[a+36>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;break h}J[a>>2]=0;J[a+4>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+8>>2]=0;J[a+12>>2]=0;if(!b){break h}$a(b);break h}H[c+25|0]=0;H[c+24|0]=K[4198];H[c+27|0]=9;b=K[4194]|K[4195]<<8|(K[4196]<<16|K[4197]<<24);J[c+16>>2]=K[4190]|K[4191]<<8|(K[4192]<<16|K[4193]<<24);J[c+20>>2]=b;v:{if(K[f+9|0]!=40|d>>>0<11|(K[(d+f|0)-1|0]!=41|K[c+16|0]!=K[f|0])){break v}if(K[c+17|0]!=K[f+1|0]|K[c+18|0]!=K[f+2|0]|(K[c+19|0]!=K[f+3|0]|K[c+20|0]!=K[f+4|0])){break v}if(K[c+21|0]!=K[f+5|0]|K[c+22|0]!=K[f+6|0]|(K[c+23|0]!=K[f+7|0]|K[c+24|0]!=K[f+8|0])){break v}b=d-11|0;J[c+108>>2]=b;d=f+10|0;J[c+104>>2]=d;$c(c+16|0,d,b,20);b=J[c+20>>2];d=J[c+16>>2];if((b|0)!=(d|0)){J[a+4>>2]=0;J[a+8>>2]=0;J[a>>2]=6;J[a+16>>2]=d;J[a+12>>2]=0;J[a+20>>2]=b;b=J[c+24>>2];J[a+28>>2]=0;J[a+32>>2]=0;J[a+24>>2]=b;J[a+36>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;break h}J[a>>2]=0;J[a+4>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+8>>2]=0;J[a+12>>2]=0;if(!b){break h}$a(b);break h}H[c+27|0]=7;H[c+23|0]=0;J[c+16>>2]=K[3869]|K[3870]<<8|(K[3871]<<16|K[3872]<<24);b=K[3872]|K[3873]<<8|(K[3874]<<16|K[3875]<<24);H[c+19|0]=b;H[c+20|0]=b>>>8;H[c+21|0]=b>>>16;H[c+22|0]=b>>>24;w:{if(K[f+7|0]!=40|d>>>0<9|(K[(d+f|0)-1|0]!=41|K[c+16|0]!=K[f|0])){break w}if(K[c+17|0]!=K[f+1|0]|K[c+18|0]!=K[f+2|0]|(K[c+19|0]!=K[f+3|0]|K[c+20|0]!=K[f+4|0])){break w}if(K[c+21|0]!=K[f+5|0]|K[c+22|0]!=K[f+6|0]){break w}b=d-9|0;J[c+108>>2]=b;d=f+8|0;J[c+104>>2]=d;$c(c+16|0,d,b,32);b=J[c+20>>2];d=J[c+16>>2];if((b|0)!=(d|0)){J[a+4>>2]=0;J[a+8>>2]=0;J[a>>2]=5;J[a+16>>2]=d;J[a+12>>2]=0;J[a+20>>2]=b;b=J[c+24>>2];J[a+28>>2]=0;J[a+32>>2]=0;J[a+24>>2]=b;J[a+36>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;break h}J[a>>2]=0;J[a+4>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+8>>2]=0;J[a+12>>2]=0;if(!b){break h}$a(b);break h}b=Kb(c+16|0,4182);d=fd(b,c+104|0);if(H[b+11|0]<0){$a(J[b>>2]);}if(d){$c(c+16|0,J[c+104>>2],J[c+108>>2],20);b=J[c+20>>2];d=J[c+16>>2];if((b|0)!=(d|0)){J[a+4>>2]=0;J[a+8>>2]=0;J[a>>2]=4;J[a+16>>2]=d;J[a+12>>2]=0;J[a+20>>2]=b;b=J[c+24>>2];J[a+28>>2]=0;J[a+32>>2]=0;J[a+24>>2]=b;J[a+36>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;break h}J[a>>2]=0;J[a+4>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+8>>2]=0;J[a+12>>2]=0;if(!b){break h}$a(b);break h}b=Kb(c+16|0,1743);d=fd(b,c+104|0);if(H[b+11|0]<0){$a(J[b>>2]);}if(d){J[c+96>>2]=0;J[c+88>>2]=0;J[c+92>>2]=0;J[c+84>>2]=0;J[c+76>>2]=0;J[c+80>>2]=0;J[5996]=0;b=c+16|0;ea(26,b|0,c+104|0,c+8|0);d=J[5996];J[5996]=0;if((d|0)!=1){J[5996]=0;aa(27,c+88|0,b|0);d=J[5996];J[5996]=0;x:{y:{if((d|0)!=1){Fb(b);b=J[c+92>>2];if(J[b-56>>2]){break y}J[a>>2]=0;J[a+4>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+8>>2]=0;J[a+12>>2]=0;break x}e=Z()|0;Fb(c+16|0);break e}J[5996]=0;aa(28,c+76|0,J[c+8>>2]);d=J[5996];J[5996]=0;if((d|0)==1){break f}z:{A:{if(!J[c+108>>2]){break A}B:{C:{while(1){I[c+16>>1]=44;H[c+27|0]=1;b=c+16|0;d=c+104|0;if(!vb(b,d)){break z}J[5996]=0;ea(26,b|0,d|0,c+8|0);d=J[5996];J[5996]=0;if((d|0)==1){break C}J[5996]=0;aa(27,c+88|0,b|0);d=J[5996];J[5996]=0;if((d|0)==1){break B}Fb(b);b=J[c+92>>2];if(!J[b-56>>2]){break z}J[5996]=0;aa(28,c+76|0,J[c+8>>2]);d=J[5996];J[5996]=0;if((d|0)!=1){if(!J[c+108>>2]){break A}continue}break}e=Z()|0;break e}e=Z()|0;break e}e=Z()|0;Fb(c+16|0);break e}J[a>>2]=9;d=J[c+88>>2];J[a+8>>2]=b;J[a+4>>2]=d;J[a+12>>2]=J[c+96>>2];J[c+96>>2]=0;J[c+88>>2]=0;J[c+92>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+40>>2]=J[c+76>>2];J[a+44>>2]=J[c+80>>2];b=J[c+84>>2];J[a+52>>2]=0;J[a+48>>2]=b;break x}b=J[c+76>>2];J[a>>2]=0;J[a+4>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+8>>2]=0;J[a+12>>2]=0;if(!b){break x}$a(b);}Oc(c+88|0);break h}e=Z()|0;break e}b=Kb(c+16|0,2464);d=fd(b,c+104|0);if(H[b+11|0]<0){$a(J[b>>2]);}if(d){J[c+96>>2]=0;J[c+88>>2]=0;J[c+92>>2]=0;J[5996]=0;b=c+16|0;aa(29,b|0,c+104|0);d=J[5996];J[5996]=0;D:{if((d|0)!=1){J[5996]=0;aa(27,c+88|0,b|0);d=J[5996];J[5996]=0;E:{F:{G:{if((d|0)!=1){Fb(b);e=J[c+92>>2];if(J[e-56>>2]){if(!J[c+108>>2]){break F}break G}J[a>>2]=0;J[a+4>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+8>>2]=0;J[a+12>>2]=0;break E}e=Z()|0;Fb(c+16|0);break D}H:{I:{while(1){J[5996]=0;b=_(30,c+76|0,4354)|0;d=J[5996];J[5996]=0;J:{if((d|0)!=1){d=vb(b,c+104|0);if(H[c+87|0]<0){$a(J[b>>2]);}if(d){break J}J[a>>2]=0;J[a+4>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+8>>2]=0;J[a+12>>2]=0;break E}e=Z()|0;break D}J[5996]=0;b=c+16|0;aa(29,b|0,c+104|0);d=J[5996];J[5996]=0;if((d|0)==1){break I}J[5996]=0;aa(27,c+88|0,b|0);d=J[5996];J[5996]=0;if((d|0)==1){break H}Fb(b);e=J[c+92>>2];if(J[e-56>>2]){if(!J[c+108>>2]){break F}continue}break}J[a>>2]=0;J[a+4>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+8>>2]=0;J[a+12>>2]=0;break E}e=Z()|0;break D}e=Z()|0;Fb(c+16|0);break D}J[a>>2]=8;b=J[c+88>>2];J[a+8>>2]=e;J[a+4>>2]=b;J[a+12>>2]=J[c+96>>2];J[c+96>>2]=0;J[c+88>>2]=0;J[c+92>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;}Oc(c+88|0);break h}e=Z()|0;}break d}b=Kb(c+16|0,2197);d=fd(b,c+104|0);if(H[b+11|0]<0){$a(J[b>>2]);}if(!d){break i}e=0;h=J[c+104>>2];b=h;g=J[c+108>>2];K:{if(!g){break K}d=b+g|0;while(1){f=K[b|0];L:{if(!((f|0)!=123&(f|0)!=40)){e=e+1|0;break L}if(e){if((f|0)!=125&(f|0)!=41){break L}e=e-1|0;break L}e=0;M:{switch(f-41|0){case 0:case 3:break K;case 1:case 2:break L;default:break M}}if((f|0)==125){break K}}b=b+1|0;if((d|0)!=(b|0)){continue}break}b=d;}J[c+8>>2]=h;d=b-h|0;J[c+12>>2]=d;J[c+108>>2]=g-d;J[c+104>>2]=b;b=J[c+8>>2];d=Ad(c+16|0,b,b+J[c+12>>2]|0);f=c+4|0;b=0;e=Ua-16|0;Ua=e;g=K[d+11|0];h=g<<24>>24;h=(h|0)<0;g=h?J[d+4>>2]:g;N:{if(!g){break N}h=h?J[d>>2]:d;i=K[h|0];j=i-9&255;if(j>>>0<24&(8388639>>>j&1)){break N}j=K[(g+h|0)-1|0]-9|0;if((j&255)>>>0<24&(8388639>>>j&1)){break N}if((i|0)==45|(Gb(h)|0)!=(g|0)){break N}J[e+12>>2]=0;J[5732]=0;h=qe(H[d+11|0]<0?J[d>>2]:d,e+12|0,10,-1,0);if(f){J[f>>2]=h;}f=J[e+12>>2];if(!f|K[f|0]){break N}b=!J[5732];}Ua=e+16|0;if(H[d+11|0]<0){$a(J[d>>2]);}if(!b){J[a>>2]=0;J[a+4>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+8>>2]=0;J[a+12>>2]=0;break h}if(!J[c+4>>2]){J[a>>2]=0;J[a+4>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+8>>2]=0;J[a+12>>2]=0;break h}b=0;J[c+96>>2]=0;J[c+88>>2]=0;J[c+92>>2]=0;e=0;O:{P:{Q:{if(J[c+108>>2]){R:{S:{T:{while(1){J[5996]=0;b=_(30,c+76|0,4354)|0;d=J[5996];J[5996]=0;if((d|0)==1){break P}d=vb(b,c+104|0);if(H[c+87|0]<0){$a(J[b>>2]);}if(!d){J[a>>2]=0;J[a+4>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+8>>2]=0;J[a+12>>2]=0;break Q}J[5996]=0;b=c+16|0;aa(29,b|0,c+104|0);d=J[5996];J[5996]=0;if((d|0)==1){break T}J[5996]=0;aa(27,c+88|0,b|0);d=J[5996];J[5996]=0;if((d|0)==1){break S}Fb(b);e=J[c+92>>2];if(J[e-56>>2]){if(!J[c+108>>2]){break R}continue}break}J[a>>2]=0;J[a+4>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+8>>2]=0;J[a+12>>2]=0;break Q}e=Z()|0;break O}e=Z()|0;Fb(c+16|0);break O}b=J[c+88>>2];}U:{d=(e-b|0)/56|0;if(d>>>0<=100){f=d;d=J[c+4>>2];if(f>>>0>=d>>>0){break U}}J[a>>2]=0;J[a+4>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+8>>2]=0;J[a+12>>2]=0;break Q}J[a+4>>2]=b;J[a>>2]=10;J[a+8>>2]=e;J[a+12>>2]=J[c+96>>2];J[c+96>>2]=0;J[c+88>>2]=0;J[c+92>>2]=0;J[a+48>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+52>>2]=d;}Oc(c+88|0);break h}e=Z()|0;}Oc(c+88|0);}da(e|0);B();}J[a>>2]=0;J[a+4>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+8>>2]=0;J[a+12>>2]=0;}Ua=c+112|0;return}ub();B();}e=Z()|0;}a=J[c+76>>2];if(a){$a(a);}}Oc(c+88|0);da(e|0);B();}function sn(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;var e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0;f=Ua-160|0;Ua=f;J[f+152>>2]=0;J[f+144>>2]=0;J[f+148>>2]=0;o=d+1|0;l=1;a:{b:{c:{d:{e:{f:{g:{h:{i:{j:{while(1){e=K[b|0]|K[b+1|0]<<8|(K[b+2|0]<<16|K[b+3|0]<<24);if((e|0)==(c|0)){break j}k:{l:{m:{n:{o:{p:{q:{r:{s:{t:{u:{v:{w:{x:{y:{z:{A:{if((c-e|0)<=0|c>>>0<=e>>>0){break A}h=e+1|0;g=0;k=K[e|0];B:{if(k>>>0>78){i=0;break B}i=k;C:{if(i>>>0<76){break C}D:{switch(i-76|0){case 0:if((c-h|0)<=0){break A}h=e+2|0;i=K[e+1|0];break C;case 1:if((c-h|0)<2){break A}h=e+3|0;i=K[e+1|0]|K[e+2|0]<<8;break C;default:break D}}if((c-h|0)<4){break A}h=e+5|0;i=K[e+1|0]|K[e+2|0]<<8|(K[e+3|0]<<16|K[e+4|0]<<24);}g=c-h|0;if((g|0)<0){break A}e=i;if(g>>>0<e>>>0){break A}E:{if(!e){i=0;g=0;break E}if((e|0)<0){J[5996]=0;ca(8);a=J[5996];J[5996]=0;if((a|0)==1){break z}break a}J[5996]=0;i=$(3,e|0)|0;j=J[5996];J[5996]=0;g=0;if((j|0)==1){break f}g=fb(i,h,e)+e|0;}h=e+h|0;}e=3;if(k-103>>>0<2){break k}H[b|0]=h;H[b+1|0]=h>>>8;H[b+2|0]=h>>>16;H[b+3|0]=h>>>24;if(!l){break y}e=0;if((d|0)<=0){break x}while(1){J[5996]=0;_(97,f+144|0,5573)|0;h=J[5996];J[5996]=0;if((h|0)==1){break w}e=e+1|0;if((e|0)!=(d|0)){continue}break}break x}b=K[f+155|0];d=b<<24>>24;i=(d|0)<0?J[f+148>>2]:b;c=i+8|0;if(c>>>0>=2147483632){J[5996]=0;ca(2);a=J[5996];J[5996]=0;if((a|0)!=1){break a}e=Z()|0;break d}F:{if(c>>>0<=10){J[a+8>>2]=0;J[a>>2]=0;J[a+4>>2]=0;H[a+11|0]=c;break F}J[5996]=0;e=(c|15)+1|0;b=$(3,e|0)|0;h=J[5996];J[5996]=0;g=0;if((h|0)==1){break f}J[a+4>>2]=c;J[a>>2]=b;J[a+8>>2]=e|-2147483648;a=b;}a=eb(a,(d|0)<0?J[f+144>>2]:f+144|0,i)+i|0;H[a+8|0]=0;H[a|0]=32;H[a+1|0]=91;H[a+2|0]=101;H[a+3|0]=114;H[a+4|0]=114;H[a+5|0]=111;H[a+6|0]=114;H[a+7|0]=93;break c}e=Z()|0;break d}J[5996]=0;aa(98,f+144|0,32);e=J[5996];J[5996]=0;if((e|0)==1){break g}e=K[f+155|0];p=(e<<24>>24<0?J[f+148>>2]:e)-1|0;}G:{H:{I:{J:{K:{m=g-i|0;switch(m|0){case 0:break G;case 33:break I;case 32:break J;case 20:break K;default:break H}}J[5996]=0;j=$(3,20)|0;e=J[5996];J[5996]=0;if((e|0)==1){e=Z()|0;break h}e=j;H[e|0]=153;H[e+1|0]=153;H[e+2|0]=153;H[e+3|0]=153;H[e+4|0]=153;H[e+5|0]=153;H[e+6|0]=153;H[e+7|0]=153;H[e+16|0]=153;H[e+17|0]=153;H[e+18|0]=153;H[e+19|0]=153;H[e+8|0]=153;H[e+9|0]=153;H[e+10|0]=153;H[e+11|0]=153;H[e+12|0]=153;H[e+13|0]=153;H[e+14|0]=153;H[e+15|0]=153;h=i;if((g|0)==(h|0)){break p}while(1){if(K[h|0]==K[e|0]){e=e+1|0;h=h+1|0;if((g|0)!=(h|0)){continue}break p}break}$a(j);if(!(K[i|0]!=80|K[i+1|0]!=75|K[i+2|0]!=104)){L:{M:{while(1){e=g-1|0;if(!K[e|0]){g=e;if((e|0)!=(i|0)){continue}break M}break}g=(g-i|0)-3|0;if(g>>>0<2147483632){break L}}J[5996]=0;ca(2);a=J[5996];J[5996]=0;if((a|0)!=1){break a}e=Z()|0;break h}N:{O:{P:{Q:{R:{S:{T:{if(g>>>0<=10){H[f+111|0]=g;h=f+100|0;break T}J[5996]=0;e=(g|15)+1|0;h=$(3,e|0)|0;j=J[5996];J[5996]=0;if((j|0)==1){break S}J[f+100>>2]=h;J[f+104>>2]=g;J[f+108>>2]=e|-2147483648;}H[eb(h,i+3|0,g)+g|0]=0;J[5996]=0;g=ba(99,f+100|0,0,5383)|0;e=J[5996];J[5996]=0;if((e|0)==1){break R}J[f+120>>2]=J[g+8>>2];e=J[g+4>>2];J[f+112>>2]=J[g>>2];J[f+116>>2]=e;J[g>>2]=0;J[g+4>>2]=0;J[g+8>>2]=0;J[5996]=0;e=_(97,f+112|0,3723)|0;h=J[5996];J[5996]=0;if((h|0)==1){break Q}J[f+136>>2]=J[e+8>>2];h=J[e+4>>2];J[f+128>>2]=J[e>>2];J[f+132>>2]=h;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;J[5996]=0;h=K[f+139|0];j=h<<24>>24<0;ba(100,f+144|0,(j?J[f+128>>2]:f+128|0)|0,(j?J[f+132>>2]:h)|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break P}if(H[f+139|0]<0){$a(J[f+128>>2]);}if(H[f+123|0]<0){$a(J[e>>2]);}if(H[f+111|0]<0){$a(J[g>>2]);}if(!l){break m}break l}e=Z()|0;break h}e=Z()|0;break N}e=Z()|0;break O}e=Z()|0;if(H[f+139|0]>=0){break O}$a(J[f+128>>2]);}if(H[f+123|0]>=0){break N}$a(J[f+112>>2]);}if(H[f+111|0]>=0){break h}$a(J[f+100>>2]);break h}J[f+80>>2]=0;J[5996]=0;J[f+72>>2]=0;J[f+76>>2]=0;aa(101,f+72|0,40);e=J[5996];J[5996]=0;if((e|0)==1){break r}if(g>>>0>i>>>0){h=i+20|0;e=i;while(1){g=K[e|0];J[5996]=0;j=f+72|0;aa(98,j|0,H[(g>>>4|0)+6976|0]);k=J[5996];J[5996]=0;if((k|0)==1){break r}J[5996]=0;aa(98,j|0,H[(g&15)+6976|0]);g=J[5996];J[5996]=0;if((g|0)==1){break r}e=e+1|0;if((h|0)!=(e|0)){continue}break}}J[5996]=0;g=ba(99,f+72|0,0,3762)|0;e=J[5996];J[5996]=0;U:{V:{W:{X:{if((e|0)!=1){J[f+96>>2]=J[g+8>>2];e=J[g+4>>2];J[f+88>>2]=J[g>>2];J[f+92>>2]=e;J[g>>2]=0;J[g+4>>2]=0;J[g+8>>2]=0;J[5996]=0;e=_(97,f+88|0,3760)|0;h=J[5996];J[5996]=0;if((h|0)==1){break X}J[f+136>>2]=J[e+8>>2];h=J[e+4>>2];J[f+128>>2]=J[e>>2];J[f+132>>2]=h;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;J[5996]=0;h=K[f+139|0];j=h<<24>>24<0;ba(100,f+144|0,(j?J[f+128>>2]:f+128|0)|0,(j?J[f+132>>2]:h)|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break W}if(H[f+139|0]<0){$a(J[f+128>>2]);}if(H[f+99|0]<0){$a(J[e>>2]);}if(H[f+83|0]<0){$a(J[g>>2]);}if(!l){break m}break l}e=Z()|0;break U}e=Z()|0;break V}e=Z()|0;if(H[f+139|0]>=0){break V}$a(J[f+128>>2]);}if(H[f+99|0]>=0){break U}$a(J[f+88>>2]);}break q}J[5996]=0;j=$(3,32)|0;e=J[5996];J[5996]=0;if((e|0)==1){e=Z()|0;break h}e=j;H[e|0]=136;H[e+1|0]=136;H[e+2|0]=136;H[e+3|0]=136;H[e+4|0]=136;H[e+5|0]=136;H[e+6|0]=136;H[e+7|0]=136;H[e+24|0]=136;H[e+25|0]=136;H[e+26|0]=136;H[e+27|0]=136;H[e+28|0]=136;H[e+29|0]=136;H[e+30|0]=136;H[e+31|0]=136;H[e+16|0]=136;H[e+17|0]=136;H[e+18|0]=136;H[e+19|0]=136;H[e+20|0]=136;H[e+21|0]=136;H[e+22|0]=136;H[e+23|0]=136;H[e+8|0]=136;H[e+9|0]=136;H[e+10|0]=136;H[e+11|0]=136;H[e+12|0]=136;H[e+13|0]=136;H[e+14|0]=136;H[e+15|0]=136;h=i;if((g|0)==(h|0)){break p}while(1){q=K[h|0];r=K[e|0];if((q|0)==(r|0)){e=e+1|0;h=h+1|0;if((g|0)!=(h|0)){continue}}break}$a(j);if((q|0)==(r|0)){break o}if(!m){break G}if((m|0)!=33){break H}}if(K[i|0]!=2|K[i+1|0]!=80|(K[i+2|0]!=75|K[i+3|0]!=98)){break H}Y:{if((g|0)!=(i|0)){Z:{while(1){e=g-1|0;if(!K[e|0]){g=e;if((e|0)!=(i|0)){continue}break Z}break}g=(g-i|0)-4|0;if(g>>>0<2147483632){break Y}}g=i;}J[5996]=0;ca(2);a=J[5996];J[5996]=0;if((a|0)!=1){break a}e=Z()|0;break e}_:{$:{aa:{ba:{ca:{da:{ea:{if(g>>>0<=10){H[f+55|0]=g;h=f+44|0;break ea}J[5996]=0;e=(g|15)+1|0;h=$(3,e|0)|0;j=J[5996];J[5996]=0;if((j|0)==1){break da}J[f+44>>2]=h;J[f+48>>2]=g;J[f+52>>2]=e|-2147483648;}H[eb(h,i+4|0,g)+g|0]=0;J[5996]=0;g=ba(99,f+44|0,0,3762)|0;e=J[5996];J[5996]=0;if((e|0)==1){break ca}J[f- -64>>2]=J[g+8>>2];e=J[g+4>>2];J[f+56>>2]=J[g>>2];J[f+60>>2]=e;J[g>>2]=0;J[g+4>>2]=0;J[g+8>>2]=0;J[5996]=0;e=_(97,f+56|0,3760)|0;h=J[5996];J[5996]=0;if((h|0)==1){break ba}J[f+136>>2]=J[e+8>>2];h=J[e+4>>2];J[f+128>>2]=J[e>>2];J[f+132>>2]=h;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;J[5996]=0;h=K[f+139|0];j=h<<24>>24<0;ba(100,f+144|0,(j?J[f+128>>2]:f+128|0)|0,(j?J[f+132>>2]:h)|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break aa}if(H[f+139|0]<0){$a(J[f+128>>2]);}if(H[f+67|0]<0){$a(J[e>>2]);}if(H[f+55|0]<0){$a(J[g>>2]);}if(l){break l}break m}e=Z()|0;break h}e=Z()|0;break _}e=Z()|0;break $}e=Z()|0;if(H[f+139|0]>=0){break $}$a(J[f+128>>2]);}if(H[f+67|0]>=0){break _}$a(J[f+56>>2]);}if(H[f+55|0]>=0){break h}$a(J[f+44>>2]);break h}J[f+24>>2]=0;J[5996]=0;J[f+16>>2]=0;J[f+20>>2]=0;aa(101,f+16|0,m<<1);e=J[5996];J[5996]=0;if((e|0)==1){break t}if(g>>>0>i>>>0){h=i+m|0;e=i;while(1){g=K[e|0];J[5996]=0;j=f+16|0;aa(98,j|0,H[(g>>>4|0)+6976|0]);k=J[5996];J[5996]=0;if((k|0)==1){break t}J[5996]=0;aa(98,j|0,H[(g&15)+6976|0]);g=J[5996];J[5996]=0;if((g|0)==1){break t}e=e+1|0;if((h|0)!=(e|0)){continue}break}}J[5996]=0;g=ba(99,f+16|0,0,3762)|0;e=J[5996];J[5996]=0;fa:{ga:{ha:{ia:{if((e|0)!=1){J[f+40>>2]=J[g+8>>2];e=J[g+4>>2];J[f+32>>2]=J[g>>2];J[f+36>>2]=e;J[g>>2]=0;J[g+4>>2]=0;J[g+8>>2]=0;J[5996]=0;e=_(97,f+32|0,3760)|0;h=J[5996];J[5996]=0;if((h|0)==1){break ia}J[f+136>>2]=J[e+8>>2];h=J[e+4>>2];J[f+128>>2]=J[e>>2];J[f+132>>2]=h;J[e>>2]=0;J[e+4>>2]=0;J[e+8>>2]=0;J[5996]=0;h=K[f+139|0];j=h<<24>>24<0;ba(100,f+144|0,(j?J[f+128>>2]:f+128|0)|0,(j?J[f+132>>2]:h)|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break ha}if(H[f+139|0]<0){$a(J[f+128>>2]);}if(H[f+43|0]<0){$a(J[e>>2]);}if(H[f+27|0]<0){$a(J[g>>2]);}if(!l){break m}break l}e=Z()|0;break fa}e=Z()|0;break ga}e=Z()|0;if(H[f+139|0]>=0){break ga}$a(J[f+128>>2]);}if(H[f+43|0]>=0){break fa}$a(J[f+32>>2]);}break s}e=J[(k<<2)+5840>>2];g=Gb(e);if(g>>>0>=2147483632){J[5996]=0;ca(2);a=J[5996];J[5996]=0;if((a|0)!=1){break a}e=Z()|0;break h}ja:{ka:{la:{ma:{na:{oa:{pa:{qa:{ra:{sa:{ta:{ua:{va:{wa:{xa:{ya:{za:{Aa:{Ba:{if(g>>>0<=10){H[f+139|0]=g;h=f+128|0;break Ba}J[5996]=0;j=(g|15)+1|0;h=$(3,j|0)|0;m=J[5996];J[5996]=0;if((m|0)==1){break Aa}J[f+128>>2]=h;J[f+132>>2]=g;J[f+136>>2]=j|-2147483648;}H[eb(h,e,g)+g|0]=0;J[5996]=0;g=K[f+139|0];e=g<<24>>24<0;g=ba(100,f+144|0,(e?J[f+128>>2]:f+128|0)|0,(e?J[f+132>>2]:g)|0)|0;e=J[5996];J[5996]=0;if((e|0)==1){break za}if(H[f+139|0]<0){$a(J[f+128>>2]);}if(k-99>>>0>1){break n}J[5996]=0;aa(98,g|0,10);e=J[5996];J[5996]=0;if((e|0)==1){break g}J[5996]=0;e=f+128|0;ga(102,e|0,b|0,c|0,o|0);h=J[5996];J[5996]=0;if((h|0)==1){break ya}J[5996]=0;j=g;h=e;g=K[f+139|0];e=g<<24>>24<0;j=ba(100,j|0,(e?J[f+128>>2]:h)|0,(e?J[f+132>>2]:g)|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break xa}if(H[f+139|0]<0){$a(J[f+128>>2]);}h=K[b|0]|K[b+1|0]<<8|(K[b+2|0]<<16|K[b+3|0]<<24);if((h|0)==(c|0)){break na}if(K[h|0]!=103){break va}e=0;if((d|0)>0){while(1){J[5996]=0;_(97,j|0,5573)|0;g=J[5996];J[5996]=0;if((g|0)==1){break v}e=e+1|0;if((e|0)!=(d|0)){continue}break}h=K[b|0]|K[b+1|0]<<8|(K[b+2|0]<<16|K[b+3|0]<<24);}g=h+1|0;H[b|0]=g;H[b+1|0]=g>>>8;H[b+2|0]=g>>>16;H[b+3|0]=g>>>24;e=J[(K[h|0]<<2)+5840>>2];g=Gb(e);if(g>>>0<2147483632){break wa}J[5996]=0;ca(2);a=J[5996];J[5996]=0;if((a|0)!=1){break a}e=Z()|0;break h}e=Z()|0;break h}e=Z()|0;if(H[f+139|0]>=0){break h}$a(J[f+128>>2]);break h}e=Z()|0;break h}e=Z()|0;if(H[f+139|0]>=0){break h}$a(J[f+128>>2]);break h}Ca:{if(g>>>0<=10){H[f+11|0]=g;h=f;break Ca}J[5996]=0;k=(g|15)+1|0;h=$(3,k|0)|0;l=J[5996];J[5996]=0;if((l|0)==1){break ua}J[f>>2]=h;J[f+4>>2]=g;J[f+8>>2]=k|-2147483648;}H[eb(h,e,g)+g|0]=0;J[5996]=0;aa(98,f|0,10);g=J[5996];J[5996]=0;if((g|0)==1){break ta}J[f+136>>2]=J[f+8>>2];J[f+8>>2]=0;e=J[f+4>>2];g=J[f>>2];J[f+128>>2]=g;J[f+132>>2]=e;J[f>>2]=0;J[f+4>>2]=0;J[5996]=0;h=g;g=H[f+139|0];e=(g|0)<0;g=ba(100,j|0,(e?h:f+128|0)|0,(e?J[f+132>>2]:g&255)|0)|0;e=J[5996];J[5996]=0;if((e|0)==1){break sa}if(H[f+139|0]<0){$a(J[f+128>>2]);}if(H[f+11|0]<0){$a(J[f>>2]);}J[5996]=0;e=f+128|0;ga(102,e|0,b|0,c|0,o|0);h=J[5996];J[5996]=0;if((h|0)==1){break qa}J[5996]=0;k=g;h=e;g=K[f+139|0];e=g<<24>>24<0;ba(100,k|0,(e?J[f+128>>2]:h)|0,(e?J[f+132>>2]:g)|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break pa}if(H[f+139|0]<0){$a(J[f+128>>2]);}h=K[b|0]|K[b+1|0]<<8|(K[b+2|0]<<16|K[b+3|0]<<24);}if((c|0)==(h|0)|K[h|0]!=104){break na}e=0;if((d|0)>0){while(1){J[5996]=0;_(97,j|0,5573)|0;g=J[5996];J[5996]=0;if((g|0)==1){break u}e=e+1|0;if((e|0)!=(d|0)){continue}break}h=K[b|0]|K[b+1|0]<<8|(K[b+2|0]<<16|K[b+3|0]<<24);}g=h+1|0;H[b|0]=g;H[b+1|0]=g>>>8;H[b+2|0]=g>>>16;H[b+3|0]=g>>>24;e=J[(K[h|0]<<2)+5840>>2];g=Gb(e);if(g>>>0<2147483632){break oa}J[5996]=0;ca(2);a=J[5996];J[5996]=0;if((a|0)!=1){break a}e=Z()|0;break h}e=Z()|0;break h}e=Z()|0;break ra}e=Z()|0;if(H[f+139|0]>=0){break ra}$a(J[f+128>>2]);}if(H[f+11|0]>=0){break h}$a(J[f>>2]);break h}e=Z()|0;break h}e=Z()|0;if(H[f+139|0]>=0){break h}$a(J[f+128>>2]);break h}Da:{if(g>>>0<=10){H[f+11|0]=g;h=f;break Da}J[5996]=0;k=(g|15)+1|0;h=$(3,k|0)|0;l=J[5996];J[5996]=0;if((l|0)==1){break ma}J[f>>2]=h;J[f+4>>2]=g;J[f+8>>2]=k|-2147483648;}H[eb(h,e,g)+g|0]=0;J[5996]=0;aa(98,f|0,10);g=J[5996];J[5996]=0;if((g|0)==1){break la}J[f+136>>2]=J[f+8>>2];J[f+8>>2]=0;e=J[f+4>>2];g=J[f>>2];J[f+128>>2]=g;J[f+132>>2]=e;J[f>>2]=0;J[f+4>>2]=0;J[5996]=0;h=g;g=H[f+139|0];e=(g|0)<0;ba(100,j|0,(e?h:f+128|0)|0,(e?J[f+132>>2]:g&255)|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break ka}if(H[f+139|0]<0){$a(J[f+128>>2]);}if(H[f+11|0]>=0){break na}$a(J[f>>2]);}g=K[f+155|0];n=g<<24>>24<0?J[f+148>>2]:g;break l}e=Z()|0;break h}e=Z()|0;break ja}e=Z()|0;if(H[f+139|0]>=0){break ja}$a(J[f+128>>2]);}if(H[f+11|0]>=0){break h}$a(J[f>>2]);break h}e=Z()|0;break h}e=Z()|0;break h}e=Z()|0;break h}e=Z()|0;}if(H[f+27|0]>=0){break h}$a(J[f+16>>2]);break h}e=Z()|0;}if(H[f+83|0]>=0){break h}$a(J[f+72>>2]);break h}$a(j);}J[5996]=0;_(97,f+144|0,3719)|0;g=J[5996];J[5996]=0;if((g|0)==1){break g}}if(l){break l}}e=K[f+155|0];g=e<<24>>24;h=(g|0)<0;if((h?J[f+148>>2]:e)-n>>>0<81){break l}e=0;H[(h?J[f+144>>2]:f+144|0)+p|0]=10;n=p+1|0;if((d|0)<=0){break l}while(1){J[5996]=0;ba(99,f+144|0,n|0,5573)|0;g=J[5996];J[5996]=0;if((g|0)==1){break i}e=e+1|0;if((e|0)!=(d|0)){continue}break}}g=K[f+155|0];l=((g<<24>>24<0?J[f+148>>2]:g)|0)==(n|0);e=0;}if(i){$a(i);}if(!e){continue}break}if((e|0)!=3){break c}}Ea:{if(!l){J[5996]=0;aa(98,f+144|0,10);b=J[5996];J[5996]=0;if((b|0)==1){break Ea}}b=J[f+148>>2];J[a>>2]=J[f+144>>2];J[a+4>>2]=b;J[a+8>>2]=J[f+152>>2];break b}e=Z()|0;break d}e=Z()|0;}g=i;break e}g=i;}e=Z()|0;}if(!g){break d}$a(g);}if(H[f+155|0]<0){$a(J[f+144>>2]);}da(e|0);B();}if(H[f+155|0]>=0){break b}$a(J[f+144>>2]);}Ua=f+160|0;return}B();}function ah(){var a=0,b=0,c=0,d=0,e=0,f=0,g=0;while(1){b=a<<4;c=b+22944|0;J[b+22948>>2]=c;J[b+22952>>2]=c;a=a+1|0;if((a|0)!=64){continue}break}Cd(48);a=Ua-16|0;Ua=a;a:{if(Ga(a+12|0,a+8|0)|0){break a}b=sb((J[a+12>>2]<<2)+4|0);J[6181]=b;if(!b){break a}b=sb(J[a+8>>2]);if(b){c=J[6181];J[c+(J[a+12>>2]<<2)>>2]=0;if(!(Fa(c|0,b|0)|0)){break a}}J[6181]=0;}Ua=a+16|0;a=cb(64);J[5998]=a;J[5999]=62;J[6e3]=-2147483584;b=K[3832]|K[3833]<<8|(K[3834]<<16|K[3835]<<24);c=K[3828]|K[3829]<<8|(K[3830]<<16|K[3831]<<24);H[a+54|0]=c;H[a+55|0]=c>>>8;H[a+56|0]=c>>>16;H[a+57|0]=c>>>24;H[a+58|0]=b;H[a+59|0]=b>>>8;H[a+60|0]=b>>>16;H[a+61|0]=b>>>24;b=K[3826]|K[3827]<<8|(K[3828]<<16|K[3829]<<24);c=K[3822]|K[3823]<<8|(K[3824]<<16|K[3825]<<24);H[a+48|0]=c;H[a+49|0]=c>>>8;H[a+50|0]=c>>>16;H[a+51|0]=c>>>24;H[a+52|0]=b;H[a+53|0]=b>>>8;H[a+54|0]=b>>>16;H[a+55|0]=b>>>24;b=K[3818]|K[3819]<<8|(K[3820]<<16|K[3821]<<24);c=K[3814]|K[3815]<<8|(K[3816]<<16|K[3817]<<24);H[a+40|0]=c;H[a+41|0]=c>>>8;H[a+42|0]=c>>>16;H[a+43|0]=c>>>24;H[a+44|0]=b;H[a+45|0]=b>>>8;H[a+46|0]=b>>>16;H[a+47|0]=b>>>24;b=K[3810]|K[3811]<<8|(K[3812]<<16|K[3813]<<24);c=K[3806]|K[3807]<<8|(K[3808]<<16|K[3809]<<24);H[a+32|0]=c;H[a+33|0]=c>>>8;H[a+34|0]=c>>>16;H[a+35|0]=c>>>24;H[a+36|0]=b;H[a+37|0]=b>>>8;H[a+38|0]=b>>>16;H[a+39|0]=b>>>24;b=K[3802]|K[3803]<<8|(K[3804]<<16|K[3805]<<24);c=K[3798]|K[3799]<<8|(K[3800]<<16|K[3801]<<24);H[a+24|0]=c;H[a+25|0]=c>>>8;H[a+26|0]=c>>>16;H[a+27|0]=c>>>24;H[a+28|0]=b;H[a+29|0]=b>>>8;H[a+30|0]=b>>>16;H[a+31|0]=b>>>24;b=K[3794]|K[3795]<<8|(K[3796]<<16|K[3797]<<24);c=K[3790]|K[3791]<<8|(K[3792]<<16|K[3793]<<24);H[a+16|0]=c;H[a+17|0]=c>>>8;H[a+18|0]=c>>>16;H[a+19|0]=c>>>24;H[a+20|0]=b;H[a+21|0]=b>>>8;H[a+22|0]=b>>>16;H[a+23|0]=b>>>24;b=K[3786]|K[3787]<<8|(K[3788]<<16|K[3789]<<24);c=K[3782]|K[3783]<<8|(K[3784]<<16|K[3785]<<24);H[a+8|0]=c;H[a+9|0]=c>>>8;H[a+10|0]=c>>>16;H[a+11|0]=c>>>24;H[a+12|0]=b;H[a+13|0]=b>>>8;H[a+14|0]=b>>>16;H[a+15|0]=b>>>24;b=K[3778]|K[3779]<<8|(K[3780]<<16|K[3781]<<24);c=K[3774]|K[3775]<<8|(K[3776]<<16|K[3777]<<24);H[a|0]=c;H[a+1|0]=c>>>8;H[a+2|0]=c>>>16;H[a+3|0]=c>>>24;H[a+4|0]=b;H[a+5|0]=b>>>8;H[a+6|0]=b>>>16;H[a+7|0]=b>>>24;H[a+62|0]=0;a=K[24003];b=a<<24>>24;b:{c:{d=(b|0)<0?J[5999]:a;a=d+12|0;if(a>>>0<2147483632){d:{if(a>>>0<=10){J[6004]=0;J[6005]=0;J[6006]=0;H[24027]=a;c=24016;break d}e=(a|15)+1|0;c=cb(e);J[6005]=a;J[6004]=c;J[6006]=e|-2147483648;}a=c;c=(b|0)>=0?23992:J[5998];a=eb(a,c,d)+d|0;H[a+12|0]=0;b=K[5207]|K[5208]<<8|(K[5209]<<16|K[5210]<<24);H[a+8|0]=b;H[a+9|0]=b>>>8;H[a+10|0]=b>>>16;H[a+11|0]=b>>>24;b=K[5203]|K[5204]<<8|(K[5205]<<16|K[5206]<<24);e=K[5199]|K[5200]<<8|(K[5201]<<16|K[5202]<<24);H[a|0]=e;H[a+1|0]=e>>>8;H[a+2|0]=e>>>16;H[a+3|0]=e>>>24;H[a+4|0]=b;H[a+5|0]=b>>>8;H[a+6|0]=b>>>16;H[a+7|0]=b>>>24;e:{a=d+8|0;f:{if(a>>>0>=2147483632){J[5996]=0;ca(2);b=J[5996];J[5996]=0;a=24028;if((b|0)==1){break f}break c}g:{if(a>>>0<=10){J[6007]=0;J[6008]=0;J[6009]=0;H[24039]=a;b=24028;break g}J[5996]=0;e=(a|15)+1|0;b=$(3,e|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){a=24028;break f}J[6008]=a;J[6007]=b;J[6009]=e|-2147483648;}a=eb(b,c,d)+d|0;H[a+8|0]=0;H[a|0]=32;H[a+1|0]=46;H[a+2|0]=44;H[a+3|0]=59;H[a+4|0]=45;H[a+5|0]=95;H[a+6|0]=63;H[a+7|0]=64;a=d+3|0;if(a>>>0>=2147483632){J[5996]=0;ca(2);b=J[5996];J[5996]=0;a=24040;if((b|0)!=1){break c}break f}h:{if(a>>>0<=10){J[6010]=0;J[6011]=0;J[6012]=0;H[24051]=a;b=24040;break h}J[5996]=0;e=(a|15)+1|0;b=$(3,e|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){a=24040;break f}J[6011]=a;J[6010]=b;J[6012]=e|-2147483648;}a=eb(b,c,d)+d|0;H[a+3|0]=0;H[a+2|0]=K[2557];b=K[2555]|K[2556]<<8;H[a|0]=b;H[a+1|0]=b>>>8;a=d+23|0;if(a>>>0>=2147483632){J[5996]=0;ca(2);b=J[5996];J[5996]=0;a=24052;if((b|0)!=1){break c}break f}J[5996]=0;e=(a|15)+1|0;b=$(3,e|0)|0;f=J[5996];J[5996]=0;if((f|0)!=1){break e}a=24052;}c=Z()|0;while(1){b=a-12|0;if(H[a-1|0]<0){$a(J[b>>2]);}a=b;if((a|0)!=24016){continue}break}da(c|0);B();}J[6014]=a;J[6013]=b;J[6015]=e|-2147483648;a=eb(b,c,d)+d|0;H[a+23|0]=0;b=K[5412]|K[5413]<<8|(K[5414]<<16|K[5415]<<24);c=K[5408]|K[5409]<<8|(K[5410]<<16|K[5411]<<24);H[a+15|0]=c;H[a+16|0]=c>>>8;H[a+17|0]=c>>>16;H[a+18|0]=c>>>24;H[a+19|0]=b;H[a+20|0]=b>>>8;H[a+21|0]=b>>>16;H[a+22|0]=b>>>24;b=K[5405]|K[5406]<<8|(K[5407]<<16|K[5408]<<24);c=K[5401]|K[5402]<<8|(K[5403]<<16|K[5404]<<24);H[a+8|0]=c;H[a+9|0]=c>>>8;H[a+10|0]=c>>>16;H[a+11|0]=c>>>24;H[a+12|0]=b;H[a+13|0]=b>>>8;H[a+14|0]=b>>>16;H[a+15|0]=b>>>24;b=K[5397]|K[5398]<<8|(K[5399]<<16|K[5400]<<24);c=K[5393]|K[5394]<<8|(K[5395]<<16|K[5396]<<24);H[a|0]=c;H[a+1|0]=c>>>8;H[a+2|0]=c>>>16;H[a+3|0]=c>>>24;H[a+4|0]=b;H[a+5|0]=b>>>8;H[a+6|0]=b>>>16;H[a+7|0]=b>>>24;break b}ub();B();}B();}d=Ua-48|0;Ua=d;J[d+20>>2]=0;J[d+12>>2]=0;J[d+16>>2]=0;J[5996]=0;_(9,24064,d+12|0)|0;a=J[5996];J[5996]=0;i:{j:{k:{l:{m:{n:{o:{p:{q:{r:{if((a|0)!=1){a=J[d+12>>2];if(a){$a(a);}a=cb(32);J[d+36>>2]=a;b=a+32|0;J[d+44>>2]=b;H[a+24|0]=0;H[a+25|0]=0;H[a+26|0]=0;H[a+27|0]=0;H[a+28|0]=0;H[a+29|0]=0;H[a+30|0]=0;H[a+31|0]=0;H[a+16|0]=0;H[a+17|0]=0;H[a+18|0]=0;H[a+19|0]=0;H[a+20|0]=0;H[a+21|0]=0;H[a+22|0]=0;H[a+23|0]=0;H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;H[a+12|0]=0;H[a+13|0]=0;H[a+14|0]=0;H[a+15|0]=0;H[a|0]=0;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;H[a+4|0]=0;H[a+5|0]=0;H[a+6|0]=0;H[a+7|0]=0;J[5996]=0;J[d+40>>2]=b;f=_(9,d+12|0,d+36|0)|0;a=J[5996];J[5996]=0;if((a|0)==1){break m}H[f+5|0]=1;J[6024]=J[f+8>>2];a=J[f+4>>2];J[6022]=J[f>>2];J[6023]=a;J[6025]=0;J[6026]=0;J[6027]=0;c=J[f+16>>2];b=J[f+12>>2];a=c-b|0;e=(a|0)/12|0;if((b|0)==(c|0)){break n}if(e>>>0<357913942){break r}J[5996]=0;ca(11);a=J[5996];J[5996]=0;if((a|0)==1){break q}B();}e=Z()|0;a=J[d+12>>2];if(!a){break i}$a(a);break i}J[5996]=0;a=$(3,a|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break q}J[6026]=a;J[6025]=a;J[6027]=a+P(e,12);J[5996]=0;b=ia(12,24108,b|0,c|0,a|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){break o}e=Z()|0;J[6026]=a;break p}e=Z()|0;}c=J[6025];if(c){a=J[6026];b=c;if((a|0)!=(b|0)){while(1){b=a-12|0;g=J[b>>2];if(g){J[a-8>>2]=g;$a(g);}a=b;if((c|0)!=(a|0)){continue}break}b=J[6025];}J[6026]=c;$a(b);}c=J[f+12>>2];if(!c){break j}a=J[f+16>>2];b=c;if((a|0)!=(b|0)){while(1){b=a-12|0;g=J[b>>2];if(g){J[a-8>>2]=g;$a(g);}a=b;if((c|0)!=(a|0)){continue}break}b=J[f+12>>2];}J[f+16>>2]=c;$a(b);break j}J[6026]=b;c=J[f+12>>2];}if(c){a=J[f+16>>2];b=c;if((a|0)!=(b|0)){while(1){b=a-12|0;e=J[b>>2];if(e){J[a-8>>2]=e;$a(e);}a=b;if((c|0)!=(a|0)){continue}break}b=J[f+12>>2];}J[f+16>>2]=c;$a(b);}a=J[d+36>>2];if(a){$a(a);}a=cb(1);J[d+36>>2]=a;b=a+1|0;J[d+44>>2]=b;H[a|0]=1;J[5996]=0;J[d+40>>2]=b;_(9,24112,d+36|0)|0;a=J[5996];J[5996]=0;if((a|0)==1){break l}a=J[d+36>>2];if(a){$a(a);}J[6036]=0;J[6037]=0;J[6034]=1;J[6038]=0;J[6039]=0;H[24142]=0;I[12070]=0;J[6044]=0;J[6045]=0;J[6042]=-1;J[6043]=0;H[24163]=0;H[24164]=0;H[24165]=0;H[24166]=0;J[6040]=0;Ua=d+48|0;break k}e=Z()|0;break j}e=Z()|0;break j}d=Ua-48|0;Ua=d;J[d+20>>2]=0;J[d+12>>2]=0;J[d+16>>2]=0;J[5996]=0;_(9,24184,d+12|0)|0;a=J[5996];J[5996]=0;s:{t:{u:{v:{w:{x:{y:{z:{A:{B:{if((a|0)!=1){a=J[d+12>>2];if(a){$a(a);}a=cb(32);J[d+36>>2]=a;b=a+32|0;J[d+44>>2]=b;H[a+24|0]=0;H[a+25|0]=0;H[a+26|0]=0;H[a+27|0]=0;H[a+28|0]=0;H[a+29|0]=0;H[a+30|0]=0;H[a+31|0]=0;H[a+16|0]=0;H[a+17|0]=0;H[a+18|0]=0;H[a+19|0]=0;H[a+20|0]=0;H[a+21|0]=0;H[a+22|0]=0;H[a+23|0]=0;H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;H[a+12|0]=0;H[a+13|0]=0;H[a+14|0]=0;H[a+15|0]=0;H[a|0]=0;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;H[a+4|0]=0;H[a+5|0]=0;H[a+6|0]=0;H[a+7|0]=0;J[5996]=0;J[d+40>>2]=b;f=_(9,d+12|0,d+36|0)|0;a=J[5996];J[5996]=0;if((a|0)==1){break w}H[f+5|0]=1;J[6054]=J[f+8>>2];a=J[f+4>>2];J[6052]=J[f>>2];J[6053]=a;J[6055]=0;J[6056]=0;J[6057]=0;c=J[f+16>>2];b=J[f+12>>2];a=c-b|0;e=(a|0)/12|0;if((b|0)==(c|0)){break x}if(e>>>0<357913942){break B}J[5996]=0;ca(11);a=J[5996];J[5996]=0;if((a|0)==1){break A}B();}e=Z()|0;a=J[d+12>>2];if(!a){break i}$a(a);break i}J[5996]=0;a=$(3,a|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break A}J[6056]=a;J[6055]=a;J[6057]=a+P(e,12);J[5996]=0;b=ia(12,24228,b|0,c|0,a|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){break y}e=Z()|0;J[6056]=a;break z}e=Z()|0;}c=J[6055];if(c){a=c;b=J[6056];if((a|0)!=(b|0)){while(1){a=b-12|0;g=J[a>>2];if(g){J[b-8>>2]=g;$a(g);}b=a;if((a|0)!=(c|0)){continue}break}a=J[6055];}J[6056]=c;$a(a);}c=J[f+12>>2];if(!c){break j}a=c;b=J[f+16>>2];if((a|0)!=(b|0)){while(1){a=b-12|0;g=J[a>>2];if(g){J[b-8>>2]=g;$a(g);}b=a;if((a|0)!=(c|0)){continue}break}a=J[f+12>>2];}J[f+16>>2]=c;$a(a);break j}J[6056]=b;c=J[f+12>>2];}if(c){a=c;b=J[f+16>>2];if((a|0)!=(b|0)){while(1){a=b-12|0;e=J[a>>2];if(e){J[b-8>>2]=e;$a(e);}b=a;if((a|0)!=(c|0)){continue}break}a=J[f+12>>2];}J[f+16>>2]=c;$a(a);}a=J[d+36>>2];if(a){$a(a);}a=cb(1);J[d+36>>2]=a;b=a+1|0;J[d+44>>2]=b;H[a|0]=1;J[5996]=0;J[d+40>>2]=b;_(9,24232,d+36|0)|0;a=J[5996];J[5996]=0;if((a|0)==1){break v}a=J[d+36>>2];if(a){$a(a);}J[6066]=0;J[6067]=0;J[6064]=1;J[6068]=0;J[6069]=0;H[24262]=0;I[12130]=0;J[6074]=0;J[6075]=0;J[6072]=-1;J[6073]=0;H[24283]=0;H[24284]=0;H[24285]=0;H[24286]=0;J[6070]=0;J[6076]=0;J[6077]=0;J[6078]=0;b=cb(56);J[b>>2]=0;J[b+4>>2]=0;J[b+40>>2]=0;J[b+44>>2]=0;J[b+32>>2]=0;J[b+36>>2]=0;J[b+24>>2]=0;J[b+28>>2]=0;J[b+16>>2]=0;J[b+20>>2]=0;J[b+8>>2]=0;J[b+12>>2]=0;mc(24304,b);a=cb(4);J[a>>2]=b;J[5996]=0;b=$(3,56)|0;c=J[5996];J[5996]=0;if((c|0)!=1){break u}break t}e=Z()|0;break j}e=Z()|0;break j}J[b+16>>2]=0;J[b+20>>2]=0;J[b+4>>2]=a;J[b>>2]=21;a=a+4|0;J[b+12>>2]=a;J[b+8>>2]=a;J[b+24>>2]=0;J[b+28>>2]=0;J[b+32>>2]=0;J[b+36>>2]=0;J[b+40>>2]=0;J[b+44>>2]=0;mc(24304,b);J[6079]=b;b=cb(56);J[b+4>>2]=0;J[b+8>>2]=0;J[b>>2]=1;J[b+12>>2]=0;J[b+16>>2]=0;J[b+20>>2]=0;J[b+24>>2]=0;J[b+28>>2]=0;J[b+32>>2]=0;J[b+36>>2]=0;J[b+40>>2]=0;J[b+44>>2]=0;mc(24304,b);a=cb(4);J[a>>2]=b;J[5996]=0;b=$(3,56)|0;c=J[5996];J[5996]=0;if((c|0)==1){break t}J[b+16>>2]=0;J[b+20>>2]=0;J[b+4>>2]=a;J[b>>2]=21;a=a+4|0;J[b+12>>2]=a;J[b+8>>2]=a;J[b+24>>2]=0;J[b+28>>2]=0;J[b+32>>2]=0;J[b+36>>2]=0;J[b+40>>2]=0;J[b+44>>2]=0;mc(24304,b);J[6080]=b;Ua=d+48|0;break s}b=Z()|0;$a(a);da(b|0);B();}d=Ua-48|0;Ua=d;J[d+20>>2]=0;J[d+12>>2]=0;J[d+16>>2]=0;J[5996]=0;_(9,24592,d+12|0)|0;a=J[5996];J[5996]=0;C:{D:{E:{F:{G:{H:{I:{J:{if((a|0)!=1){a=J[d+12>>2];if(a){$a(a);}a=cb(32);J[d+36>>2]=a;b=a+32|0;J[d+44>>2]=b;H[a+24|0]=0;H[a+25|0]=0;H[a+26|0]=0;H[a+27|0]=0;H[a+28|0]=0;H[a+29|0]=0;H[a+30|0]=0;H[a+31|0]=0;H[a+16|0]=0;H[a+17|0]=0;H[a+18|0]=0;H[a+19|0]=0;H[a+20|0]=0;H[a+21|0]=0;H[a+22|0]=0;H[a+23|0]=0;H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;H[a+12|0]=0;H[a+13|0]=0;H[a+14|0]=0;H[a+15|0]=0;H[a|0]=0;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;H[a+4|0]=0;H[a+5|0]=0;H[a+6|0]=0;H[a+7|0]=0;J[5996]=0;J[d+40>>2]=b;f=_(9,d+12|0,d+36|0)|0;a=J[5996];J[5996]=0;if((a|0)==1){break E}H[f+5|0]=1;J[6156]=J[f+8>>2];a=J[f+4>>2];J[6154]=J[f>>2];J[6155]=a;J[6157]=0;J[6158]=0;J[6159]=0;c=J[f+16>>2];b=J[f+12>>2];a=c-b|0;e=(a|0)/12|0;if((b|0)==(c|0)){break F}if(e>>>0<357913942){break J}J[5996]=0;ca(11);a=J[5996];J[5996]=0;if((a|0)==1){break I}B();}e=Z()|0;a=J[d+12>>2];if(!a){break i}$a(a);break i}J[5996]=0;a=$(3,a|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break I}J[6158]=a;J[6157]=a;J[6159]=a+P(e,12);J[5996]=0;b=ia(12,24636,b|0,c|0,a|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){break G}e=Z()|0;J[6158]=a;break H}e=Z()|0;}c=J[6157];if(c){a=J[6158];b=c;if((a|0)!=(b|0)){while(1){b=a-12|0;g=J[b>>2];if(g){J[a-8>>2]=g;$a(g);}a=b;if((c|0)!=(a|0)){continue}break}b=J[6157];}J[6158]=c;$a(b);}c=J[f+12>>2];if(!c){break j}a=J[f+16>>2];b=c;if((a|0)!=(b|0)){while(1){b=a-12|0;g=J[b>>2];if(g){J[a-8>>2]=g;$a(g);}a=b;if((c|0)!=(a|0)){continue}break}b=J[f+12>>2];}J[f+16>>2]=c;$a(b);break j}J[6158]=b;c=J[f+12>>2];}if(c){a=J[f+16>>2];b=c;if((a|0)!=(b|0)){while(1){b=a-12|0;e=J[b>>2];if(e){J[a-8>>2]=e;$a(e);}a=b;if((c|0)!=(a|0)){continue}break}b=J[f+12>>2];}J[f+16>>2]=c;$a(b);}a=J[d+36>>2];if(a){$a(a);}a=cb(1);J[d+36>>2]=a;b=a+1|0;J[d+44>>2]=b;H[a|0]=1;J[5996]=0;J[d+40>>2]=b;_(9,24640,d+36|0)|0;a=J[5996];J[5996]=0;if((a|0)==1){break D}a=J[d+36>>2];if(a){$a(a);}J[6168]=0;J[6169]=0;J[6166]=1;J[6170]=0;J[6171]=0;H[24670]=0;I[12334]=0;J[6176]=0;J[6177]=0;J[6174]=-1;J[6175]=0;H[24691]=0;H[24692]=0;H[24693]=0;H[24694]=0;J[6172]=0;Ua=d+48|0;break C}e=Z()|0;break j}e=Z()|0;break j}J[6178]=24816;return}a=J[d+36>>2];if(!a){break i}$a(a);}da(e|0);B();}function fn(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0;g=Ua-96|0;Ua=g;J[g+36>>2]=0;J[g+28>>2]=0;J[g+32>>2]=0;J[5996]=0;k=$(3,12)|0;c=J[5996];J[5996]=0;a:{if((c|0)!=1){r=g+36|0;H[k+8|0]=0;J[k+4>>2]=0;J[k>>2]=b;f=k+12|0;q=f;b:{while(1){i=f-12|0;c=i;j=J[c+4>>2];n=J[c>>2];d=J[n+32>>2];c:{d:{e:{h=J[n+36>>2]-d>>3;if(h>>>0>j>>>0){b=1;J[c+4>>2]=j+1;e=K[f-4|0];f:{g:{h:{c=J[n>>2];switch(c-11|0){case 0:break g;case 3:break f;default:break h}}b=0;if((j|0)!=1|(c|0)!=17){break f}}b=(e|0)!=0;}j=J[d+(j<<3)>>2];if(f>>>0<q>>>0){H[f+8|0]=b;J[f+4>>2]=0;J[f>>2]=j;f=f+12|0;break c}i:{d=(f-k|0)/12|0;i=d+1|0;j:{if(i>>>0>=357913942){J[5996]=0;ca(121);a=J[5996];J[5996]=0;if((a|0)==1){break j}break a}e=(q-k|0)/12|0;c=e<<1;c=e>>>0>=178956970?357913941:c>>>0>i>>>0?c:i;if(c>>>0<357913942){break i}J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}}l=Z()|0;break b}J[5996]=0;i=P(c,12);h=$(3,i|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break e}c=h+P(d,12)|0;H[c+8|0]=b;J[c+4>>2]=0;J[c>>2]=j;b=c;if((f|0)!=(k|0)){while(1){f=f-12|0;d=f;e=J[d+4>>2];b=b-12|0;J[b>>2]=J[d>>2];J[b+4>>2]=e;J[b+8>>2]=J[d+8>>2];if((d|0)!=(k|0)){continue}break}}q=h+i|0;f=c+12|0;if(k){$a(k);}k=b;break c}e=J[g+28>>2];c=J[g+32>>2]-e>>5;if(c>>>0<h>>>0){J[5996]=0;ga(44,5135,2210,439,2439);a=J[5996];J[5996]=0;if((a|0)!=1){break a}l=Z()|0;break b}b=K[f-4|0];J[5996]=0;oa(122,g+40|0,b|0,n|0,(e+(c<<5)|0)-(h<<5)|0,h|0);b=J[5996];J[5996]=0;if((b|0)==1){l=Z()|0;break b}b=J[g+64>>2];J[g+16>>2]=J[g+60>>2];J[g+20>>2]=b;J[g+24>>2]=J[g+68>>2];b=J[g+56>>2];J[g+8>>2]=J[g+52>>2];J[g+12>>2]=b;o=J[g+48>>2];m=J[g+44>>2];j=J[g+40>>2];b=J[g+32>>2];e=J[n+36>>2];c=J[n+32>>2];k:{if((e|0)==(c|0)){c=b;break k}e=e-c<<2;c=b-e|0;f=c+(e&-32)|0;if((f|0)!=(b|0)){while(1){d=g- -64|0;J[d>>2]=K[c+28|0]|K[c+29|0]<<8|(K[c+30|0]<<16|K[c+31|0]<<24);e=K[c+24|0]|K[c+25|0]<<8|(K[c+26|0]<<16|K[c+27|0]<<24);J[g+56>>2]=K[c+20|0]|K[c+21|0]<<8|(K[c+22|0]<<16|K[c+23|0]<<24);J[g+60>>2]=e;e=K[c+16|0]|K[c+17|0]<<8|(K[c+18|0]<<16|K[c+19|0]<<24);J[g+48>>2]=K[c+12|0]|K[c+13|0]<<8|(K[c+14|0]<<16|K[c+15|0]<<24);J[g+52>>2]=e;e=K[c+8|0]|K[c+9|0]<<8|(K[c+10|0]<<16|K[c+11|0]<<24);J[g+40>>2]=K[c+4|0]|K[c+5|0]<<8|(K[c+6|0]<<16|K[c+7|0]<<24);J[g+44>>2]=e;e=K[f+28|0]|K[f+29|0]<<8|(K[f+30|0]<<16|K[f+31|0]<<24);H[c+28|0]=e;H[c+29|0]=e>>>8;H[c+30|0]=e>>>16;H[c+31|0]=e>>>24;h=K[f+24|0]|K[f+25|0]<<8|(K[f+26|0]<<16|K[f+27|0]<<24);e=K[f+20|0]|K[f+21|0]<<8|(K[f+22|0]<<16|K[f+23|0]<<24);H[c+20|0]=e;H[c+21|0]=e>>>8;H[c+22|0]=e>>>16;H[c+23|0]=e>>>24;H[c+24|0]=h;H[c+25|0]=h>>>8;H[c+26|0]=h>>>16;H[c+27|0]=h>>>24;h=K[f+16|0]|K[f+17|0]<<8|(K[f+18|0]<<16|K[f+19|0]<<24);e=K[f+12|0]|K[f+13|0]<<8|(K[f+14|0]<<16|K[f+15|0]<<24);H[c+12|0]=e;H[c+13|0]=e>>>8;H[c+14|0]=e>>>16;H[c+15|0]=e>>>24;H[c+16|0]=h;H[c+17|0]=h>>>8;H[c+18|0]=h>>>16;H[c+19|0]=h>>>24;h=K[f+8|0]|K[f+9|0]<<8|(K[f+10|0]<<16|K[f+11|0]<<24);e=K[f+4|0]|K[f+5|0]<<8|(K[f+6|0]<<16|K[f+7|0]<<24);H[c+4|0]=e;H[c+5|0]=e>>>8;H[c+6|0]=e>>>16;H[c+7|0]=e>>>24;H[c+8|0]=h;H[c+9|0]=h>>>8;H[c+10|0]=h>>>16;H[c+11|0]=h>>>24;e=J[d>>2];H[f+28|0]=e;H[f+29|0]=e>>>8;H[f+30|0]=e>>>16;H[f+31|0]=e>>>24;d=J[g+60>>2];e=J[g+56>>2];H[f+20|0]=e;H[f+21|0]=e>>>8;H[f+22|0]=e>>>16;H[f+23|0]=e>>>24;H[f+24|0]=d;H[f+25|0]=d>>>8;H[f+26|0]=d>>>16;H[f+27|0]=d>>>24;d=J[g+52>>2];e=J[g+48>>2];H[f+12|0]=e;H[f+13|0]=e>>>8;H[f+14|0]=e>>>16;H[f+15|0]=e>>>24;H[f+16|0]=d;H[f+17|0]=d>>>8;H[f+18|0]=d>>>16;H[f+19|0]=d>>>24;d=J[g+44>>2];e=J[g+40>>2];H[f+4|0]=e;H[f+5|0]=e>>>8;H[f+6|0]=e>>>16;H[f+7|0]=e>>>24;H[f+8|0]=d;H[f+9|0]=d>>>8;H[f+10|0]=d>>>16;H[f+11|0]=d>>>24;e=J[c>>2];J[c>>2]=J[f>>2];J[f>>2]=e;c=c+32|0;f=f+32|0;if((f|0)!=(b|0)){continue}break}}if((b|0)!=(c|0)){while(1){e=b-32|0;if((K[e|0]|K[e+1|0]<<8|(K[e+2|0]<<16|K[e+3|0]<<24))>>>0>=29){b=b-24|0;$a(K[b|0]|K[b+1|0]<<8|(K[b+2|0]<<16|K[b+3|0]<<24));H[b|0]=0;H[b+1|0]=0;H[b+2|0]=0;H[b+3|0]=0;}b=e;if((c|0)!=(b|0)){continue}break}}J[g+32>>2]=c;}f=J[g+36>>2];if(f>>>0>c>>>0){H[c+4|0]=0;H[c+5|0]=0;H[c+6|0]=0;H[c+7|0]=0;H[c+8|0]=0;H[c+9|0]=0;H[c+10|0]=0;H[c+11|0]=0;H[c+28|0]=0;H[c+29|0]=0;H[c+30|0]=0;H[c+31|0]=0;H[c+20|0]=0;H[c+21|0]=0;H[c+22|0]=0;H[c+23|0]=0;H[c+24|0]=0;H[c+25|0]=0;H[c+26|0]=0;H[c+27|0]=0;H[c+12|0]=0;H[c+13|0]=0;H[c+14|0]=0;H[c+15|0]=0;H[c+16|0]=0;H[c+17|0]=0;H[c+18|0]=0;H[c+19|0]=0;H[c+8|0]=o;H[c+9|0]=o>>>8;H[c+10|0]=o>>>16;H[c+11|0]=o>>>24;H[c+4|0]=m;H[c+5|0]=m>>>8;H[c+6|0]=m>>>16;H[c+7|0]=m>>>24;e=J[g+12>>2];b=J[g+8>>2];H[c+12|0]=b;H[c+13|0]=b>>>8;H[c+14|0]=b>>>16;H[c+15|0]=b>>>24;H[c+16|0]=e;H[c+17|0]=e>>>8;H[c+18|0]=e>>>16;H[c+19|0]=e>>>24;e=J[g+20>>2];b=J[g+16>>2];H[c+20|0]=b;H[c+21|0]=b>>>8;H[c+22|0]=b>>>16;H[c+23|0]=b>>>24;H[c+24|0]=e;H[c+25|0]=e>>>8;H[c+26|0]=e>>>16;H[c+27|0]=e>>>24;b=J[g+24>>2];H[c+28|0]=b;H[c+29|0]=b>>>8;H[c+30|0]=b>>>16;H[c+31|0]=b>>>24;J[c>>2]=j;J[g+32>>2]=c+32;break d}l:{m:{n:{p=J[g+28>>2];d=c-p>>5;h=d+1|0;o:{if(h>>>0>=134217728){J[5996]=0;ca(123);break o}J[g+92>>2]=r;e=f-p|0;b=e>>4;h=e>>>0>=2147483616?134217727:b>>>0>h>>>0?b:h;if(!h){b=0;break m}if(h>>>0<134217728){break n}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break a}break l}J[5996]=0;b=$(3,h<<5)|0;e=J[5996];J[5996]=0;if((e|0)==1){break l}}J[g+76>>2]=b;l=(h<<5)+b|0;J[g+88>>2]=l;h=(d<<5)+b|0;J[g+80>>2]=h;d=h;H[d+24|0]=0;H[d+25|0]=0;H[d+26|0]=0;H[d+27|0]=0;H[d+28|0]=0;H[d+29|0]=0;H[d+30|0]=0;H[d+31|0]=0;H[d+16|0]=0;H[d+17|0]=0;H[d+18|0]=0;H[d+19|0]=0;H[d+20|0]=0;H[d+21|0]=0;H[d+22|0]=0;H[d+23|0]=0;H[d+8|0]=0;H[d+9|0]=0;H[d+10|0]=0;H[d+11|0]=0;H[d+12|0]=0;H[d+13|0]=0;H[d+14|0]=0;H[d+15|0]=0;H[d|0]=0;H[d+1|0]=0;H[d+2|0]=0;H[d+3|0]=0;H[d+4|0]=0;H[d+5|0]=0;H[d+6|0]=0;H[d+7|0]=0;J[g+56>>2]=K[d+28|0]|K[d+29|0]<<8|(K[d+30|0]<<16|K[d+31|0]<<24);b=K[d+24|0]|K[d+25|0]<<8|(K[d+26|0]<<16|K[d+27|0]<<24);J[g+48>>2]=K[d+20|0]|K[d+21|0]<<8|(K[d+22|0]<<16|K[d+23|0]<<24);J[g+52>>2]=b;b=K[d+16|0]|K[d+17|0]<<8|(K[d+18|0]<<16|K[d+19|0]<<24);J[g+40>>2]=K[d+12|0]|K[d+13|0]<<8|(K[d+14|0]<<16|K[d+15|0]<<24);J[g+44>>2]=b;H[d+8|0]=o;H[d+9|0]=o>>>8;H[d+10|0]=o>>>16;H[d+11|0]=o>>>24;H[d+4|0]=m;H[d+5|0]=m>>>8;H[d+6|0]=m>>>16;H[d+7|0]=m>>>24;b=J[g+24>>2];H[d+28|0]=b;H[d+29|0]=b>>>8;H[d+30|0]=b>>>16;H[d+31|0]=b>>>24;e=J[g+20>>2];b=J[g+16>>2];H[d+20|0]=b;H[d+21|0]=b>>>8;H[d+22|0]=b>>>16;H[d+23|0]=b>>>24;H[d+24|0]=e;H[d+25|0]=e>>>8;H[d+26|0]=e>>>16;H[d+27|0]=e>>>24;e=J[g+12>>2];b=J[g+8>>2];H[d+12|0]=b;H[d+13|0]=b>>>8;H[d+14|0]=b>>>16;H[d+15|0]=b>>>24;H[d+16|0]=e;H[d+17|0]=e>>>8;H[d+18|0]=e>>>16;H[d+19|0]=e>>>24;b=J[g+52>>2];J[g+16>>2]=J[g+48>>2];J[g+20>>2]=b;J[g+24>>2]=J[g+56>>2];b=J[g+44>>2];J[g+8>>2]=J[g+40>>2];J[g+12>>2]=b;J[d>>2]=j;b=d+32|0;J[g+84>>2]=b;p:{if((c|0)==(p|0)){e=c;}else {while(1){b=d;d=d-32|0;H[d|0]=0;H[d+1|0]=0;H[d+2|0]=0;H[d+3|0]=0;H[d+4|0]=0;H[d+5|0]=0;H[d+6|0]=0;H[d+7|0]=0;H[d+24|0]=0;H[d+25|0]=0;H[d+26|0]=0;H[d+27|0]=0;H[d+28|0]=0;H[d+29|0]=0;H[d+30|0]=0;H[d+31|0]=0;H[d+16|0]=0;H[d+17|0]=0;H[d+18|0]=0;H[d+19|0]=0;H[d+20|0]=0;H[d+21|0]=0;H[d+22|0]=0;H[d+23|0]=0;H[d+8|0]=0;H[d+9|0]=0;H[d+10|0]=0;H[d+11|0]=0;H[d+12|0]=0;H[d+13|0]=0;H[d+14|0]=0;H[d+15|0]=0;e=c;c=c-32|0;f=K[c|0]|K[c+1|0]<<8|(K[c+2|0]<<16|K[c+3|0]<<24);J[5996]=0;j=f>>>0<29?f:f-29|0;aa(124,d|0,j|0);f=J[5996];J[5996]=0;if((f|0)==1){break p}m=j+(K[d|0]|K[d+1|0]<<8|(K[d+2|0]<<16|K[d+3|0]<<24))|0;H[d|0]=m;H[d+1|0]=m>>>8;H[d+2|0]=m>>>16;H[d+3|0]=m>>>24;f=K[c|0]|K[c+1|0]<<8|(K[c+2|0]<<16|K[c+3|0]<<24);j=f>>>0<29;n=j?f:f-29|0;q:{if(!n){break q}f=b-28|0;b=b-24|0;f=m>>>0<29?f:K[b|0]|K[b+1|0]<<8|(K[b+2|0]<<16|K[b+3|0]<<24);l=0;b=e-24|0;j=j?e-28|0:K[b|0]|K[b+1|0]<<8|(K[b+2|0]<<16|K[b+3|0]<<24);b=j;e=n&7;if(e){while(1){H[f|0]=K[b|0];b=b+1|0;f=f+1|0;l=l+1|0;if((e|0)!=(l|0)){continue}break}}if(n>>>0<8){break q}e=j+n|0;while(1){H[f|0]=K[b|0];H[f+1|0]=K[b+1|0];H[f+2|0]=K[b+2|0];H[f+3|0]=K[b+3|0];H[f+4|0]=K[b+4|0];H[f+5|0]=K[b+5|0];H[f+6|0]=K[b+6|0];H[f+7|0]=K[b+7|0];f=f+8|0;b=b+8|0;if((e|0)!=(b|0)){continue}break}}if((c|0)!=(p|0)){continue}break}l=J[g+88>>2];f=J[g+36>>2];b=J[g+84>>2];c=J[g+32>>2];e=J[g+28>>2];}J[g+80>>2]=e;J[g+28>>2]=d;J[g+32>>2]=b;J[g+36>>2]=l;J[g+88>>2]=f;J[g+76>>2]=e;J[g+84>>2]=c;if((c|0)!=(e|0)){while(1){b=c-32|0;J[g+84>>2]=b;if((K[b|0]|K[b+1|0]<<8|(K[b+2|0]<<16|K[b+3|0]<<24))>>>0>=29){b=c-24|0;$a(K[b|0]|K[b+1|0]<<8|(K[b+2|0]<<16|K[b+3|0]<<24));H[b|0]=0;H[b+1|0]=0;H[b+2|0]=0;H[b+3|0]=0;b=J[g+84>>2];}c=b;if((e|0)!=(c|0)){continue}break}c=J[g+76>>2];}if(!c){break d}$a(c);f=i;break c}l=Z()|0;if((b|0)!=(h|0)){while(1){if((K[b|0]|K[b+1|0]<<8|(K[b+2|0]<<16|K[b+3|0]<<24))>>>0>=29){$a(K[b+8|0]|K[b+9|0]<<8|(K[b+10|0]<<16|K[b+11|0]<<24));H[b+8|0]=0;H[b+9|0]=0;H[b+10|0]=0;H[b+11|0]=0;}b=b+32|0;if((h|0)!=(b|0)){continue}break}}b=J[g+84>>2];c=J[g+80>>2];if((b|0)!=(c|0)){while(1){a=b-32|0;J[g+84>>2]=a;if((K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))>>>0>=29){a=b-24|0;$a(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24));H[a|0]=0;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;a=J[g+84>>2];}b=a;if((c|0)!=(b|0)){continue}break}}a=J[g+76>>2];if(a){$a(a);}break b}l=Z()|0;if(j>>>0<29){break b}$a(o);break b}l=Z()|0;break b}f=i;}if((f|0)!=(k|0)){continue}break}b=J[g+32>>2];c=J[g+28>>2];if((b-c|0)!=32){J[5996]=0;ga(44,4102,2210,450,2439);a=J[5996];J[5996]=0;if((a|0)!=1){break a}l=Z()|0;break b}H[a|0]=0;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;H[a+4|0]=0;H[a+5|0]=0;H[a+6|0]=0;H[a+7|0]=0;H[a+24|0]=0;H[a+25|0]=0;H[a+26|0]=0;H[a+27|0]=0;H[a+28|0]=0;H[a+29|0]=0;H[a+30|0]=0;H[a+31|0]=0;H[a+16|0]=0;H[a+17|0]=0;H[a+18|0]=0;H[a+19|0]=0;H[a+20|0]=0;H[a+21|0]=0;H[a+22|0]=0;H[a+23|0]=0;H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;H[a+12|0]=0;H[a+13|0]=0;H[a+14|0]=0;H[a+15|0]=0;i=g- -64|0;J[i>>2]=K[a+28|0]|K[a+29|0]<<8|(K[a+30|0]<<16|K[a+31|0]<<24);e=K[a+24|0]|K[a+25|0]<<8|(K[a+26|0]<<16|K[a+27|0]<<24);J[g+56>>2]=K[a+20|0]|K[a+21|0]<<8|(K[a+22|0]<<16|K[a+23|0]<<24);J[g+60>>2]=e;e=K[a+16|0]|K[a+17|0]<<8|(K[a+18|0]<<16|K[a+19|0]<<24);J[g+48>>2]=K[a+12|0]|K[a+13|0]<<8|(K[a+14|0]<<16|K[a+15|0]<<24);J[g+52>>2]=e;e=K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24);J[g+40>>2]=K[a+4|0]|K[a+5|0]<<8|(K[a+6|0]<<16|K[a+7|0]<<24);J[g+44>>2]=e;e=K[c+28|0]|K[c+29|0]<<8|(K[c+30|0]<<16|K[c+31|0]<<24);H[a+28|0]=e;H[a+29|0]=e>>>8;H[a+30|0]=e>>>16;H[a+31|0]=e>>>24;d=K[c+24|0]|K[c+25|0]<<8|(K[c+26|0]<<16|K[c+27|0]<<24);e=K[c+20|0]|K[c+21|0]<<8|(K[c+22|0]<<16|K[c+23|0]<<24);H[a+20|0]=e;H[a+21|0]=e>>>8;H[a+22|0]=e>>>16;H[a+23|0]=e>>>24;H[a+24|0]=d;H[a+25|0]=d>>>8;H[a+26|0]=d>>>16;H[a+27|0]=d>>>24;d=K[c+16|0]|K[c+17|0]<<8|(K[c+18|0]<<16|K[c+19|0]<<24);e=K[c+12|0]|K[c+13|0]<<8|(K[c+14|0]<<16|K[c+15|0]<<24);H[a+12|0]=e;H[a+13|0]=e>>>8;H[a+14|0]=e>>>16;H[a+15|0]=e>>>24;H[a+16|0]=d;H[a+17|0]=d>>>8;H[a+18|0]=d>>>16;H[a+19|0]=d>>>24;d=K[c+8|0]|K[c+9|0]<<8|(K[c+10|0]<<16|K[c+11|0]<<24);e=K[c+4|0]|K[c+5|0]<<8|(K[c+6|0]<<16|K[c+7|0]<<24);H[a+4|0]=e;H[a+5|0]=e>>>8;H[a+6|0]=e>>>16;H[a+7|0]=e>>>24;H[a+8|0]=d;H[a+9|0]=d>>>8;H[a+10|0]=d>>>16;H[a+11|0]=d>>>24;e=J[i>>2];H[c+28|0]=e;H[c+29|0]=e>>>8;H[c+30|0]=e>>>16;H[c+31|0]=e>>>24;i=J[g+60>>2];e=J[g+56>>2];H[c+20|0]=e;H[c+21|0]=e>>>8;H[c+22|0]=e>>>16;H[c+23|0]=e>>>24;H[c+24|0]=i;H[c+25|0]=i>>>8;H[c+26|0]=i>>>16;H[c+27|0]=i>>>24;i=J[g+52>>2];e=J[g+48>>2];H[c+12|0]=e;H[c+13|0]=e>>>8;H[c+14|0]=e>>>16;H[c+15|0]=e>>>24;H[c+16|0]=i;H[c+17|0]=i>>>8;H[c+18|0]=i>>>16;H[c+19|0]=i>>>24;i=J[g+44>>2];e=J[g+40>>2];H[c+4|0]=e;H[c+5|0]=e>>>8;H[c+6|0]=e>>>16;H[c+7|0]=e>>>24;H[c+8|0]=i;H[c+9|0]=i>>>8;H[c+10|0]=i>>>16;H[c+11|0]=i>>>24;J[a>>2]=J[c>>2];J[c>>2]=0;H[a+32|0]=1;if((b|0)!=(c|0)){while(1){a=b-32|0;if((K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))>>>0>=29){b=b-24|0;$a(K[b|0]|K[b+1|0]<<8|(K[b+2|0]<<16|K[b+3|0]<<24));H[b|0]=0;H[b+1|0]=0;H[b+2|0]=0;H[b+3|0]=0;}b=a;if((c|0)!=(b|0)){continue}break}b=J[g+28>>2];}J[g+32>>2]=c;$a(b);if(k){$a(k);}Ua=g+96|0;return}Qg(g+28|0);if(k){$a(k);}da(l|0);B();}a=Z()|0;Qg(g+28|0);da(a|0);B();}B();}function xn(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0,A=0,C=0,D=0;g=Ua-48|0;Ua=g;J[g+32>>2]=0;J[g+24>>2]=0;J[g+28>>2]=0;J[5996]=0;o=$(3,12)|0;d=J[5996];J[5996]=0;a:{if((d|0)!=1){J[o+4>>2]=0;J[o>>2]=b;r=g+40|0;x=g+12|0;y=o+12|0;f=y;b:{c:{d:{while(1){e:{D=f-12|0;b=D;d=J[b+4>>2];f:{g:{h:{i:{j:{s=J[b>>2];n=J[s+36>>2];z=J[s+32>>2];A=n-z>>3;if(d>>>0<A>>>0){J[b+4>>2]=d+1;i=J[(d<<3)+z>>2];if(f>>>0<y>>>0){J[f+4>>2]=0;J[f>>2]=i;f=f+12|0;break f}k:{e=(f-o|0)/12|0;h=e+1|0;l:{if(h>>>0>=357913942){J[5996]=0;ca(95);a=J[5996];J[5996]=0;if((a|0)==1){break l}break a}d=(y-o|0)/12|0;b=d<<1;b=d>>>0>=178956970?357913941:b>>>0>h>>>0?b:h;if(b>>>0<357913942){break k}J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}}b=Z()|0;break b}J[5996]=0;h=P(b,12);j=$(3,h|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break j}d=j+P(e,12)|0;J[d+4>>2]=0;J[d>>2]=i;b=d;if((f|0)!=(o|0)){while(1){f=f-12|0;e=J[f+4>>2];b=b-12|0;J[b>>2]=J[f>>2];J[b+4>>2]=e;J[b+8>>2]=J[f+8>>2];if((f|0)!=(o|0)){continue}break}}y=h+j|0;f=d+12|0;if(o){$a(o);}o=b;break f}k=J[g+24>>2];j=(J[g+28>>2]-k|0)/12|0;if(j>>>0<A>>>0){J[5996]=0;ga(44,5135,2210,439,2439);a=J[5996];J[5996]=0;if((a|0)!=1){break a}b=Z()|0;break b}if(K[s+88|0]?J[c>>2]!=(s|0):0){break h}C=J[s+12>>2];l=J[s+8>>2];J[g+36>>2]=r;J[g+40>>2]=0;J[g+44>>2]=0;v=(C-l|0)/12|0;m=0;b=r;e=0;if((l|0)==(C|0)){break i}while(1){m:{n:{o:{p:{f=r;q:{if((f|0)==(b|0)){break q}d=f;b=e;r:{if(e){while(1){f=b;b=J[b+4>>2];if(b){continue}break r}}while(1){f=J[d+8>>2];b=J[f>>2]==(d|0);d=f;if(b){continue}break}}w=J[l>>2];p=K[l+11|0];h=p<<24>>24;t=(h|0)<0;b=K[f+27|0];u=b<<24>>24<0;s:{d=J[l+4>>2];q=t?d:p;i=u?J[f+20>>2]:b;b=i>>>0>q>>>0?q:i;if(b){b=Eb(u?J[f+16>>2]:f+16|0,t?w:l,b);if(b){break s}}if(i>>>0<q>>>0){break q}break p}if((b|0)>=0){break p}}b=e?f:r;i=e?f+4|0:r;break o}b=r;i=b;if(!e){break o}b=(h|0)>=0;t=b?l:w;w=b?p:d;d=e;while(1){b=d;f=K[d+27|0];u=f<<24>>24<0;p=u?J[d+20>>2]:f;q=p>>>0<w>>>0;t:{u:{v:{w:{h=q?p:w;x:{if(h){f=u?J[d+16>>2]:d+16|0;d=Eb(t,f,h);if(!d){if(p>>>0>w>>>0){break x}break w}if((d|0)>=0){break w}break x}if(p>>>0<=w>>>0){break v}}i=b;d=J[b>>2];if(d){continue}break n}d=Eb(f,t,h);if(d){break u}}if(!q){break o}break t}if((d|0)>=0){break o}}i=b+4|0;d=J[b+4>>2];if(d){continue}break}break n}if(J[i>>2]){break m}}J[5996]=0;f=$(3,28)|0;d=J[5996];J[5996]=0;y:{if((d|0)!=1){h=f+16|0;if(H[l+11|0]>=0){d=J[l+4>>2];J[h>>2]=J[l>>2];J[h+4>>2]=d;J[h+8>>2]=J[l+8>>2];break y}e=J[l+4>>2];d=J[l>>2];J[5996]=0;ea(40,h|0,d|0,e|0);d=J[5996];J[5996]=0;if((d|0)!=1){break y}b=Z()|0;$a(f);Db(J[g+40>>2]);break b}b=Z()|0;Db(e);break b}J[f+8>>2]=b;J[f>>2]=0;J[f+4>>2]=0;J[i>>2]=f;b=J[J[g+36>>2]>>2];if(b){J[g+36>>2]=b;f=J[i>>2];}zd(J[g+40>>2],f);m=J[g+44>>2]+1|0;J[g+44>>2]=m;}l=l+12|0;if((C|0)==(l|0)){break i}b=J[g+36>>2];e=J[g+40>>2];continue}}b=Z()|0;break b}z:{A:{if((m|0)!=(v|0)){break A}if((n|0)!=(z|0)){t=k+P(j,12)|0;m=t+P(A,-12)|0;while(1){d=J[m+8>>2];if(d>>>0>v>>>0){b=J[g+36>>2];J[g+36>>2]=J[m>>2];J[m>>2]=b;b=J[g+40>>2];J[g+40>>2]=J[m+4>>2];J[m+4>>2]=b;b=J[g+44>>2];J[g+44>>2]=d;J[m+8>>2]=b;J[(J[g+44>>2]?J[g+40>>2]+8|0:g+36|0)>>2]=r;J[(b?J[m+4>>2]+8|0:m)>>2]=m+4;}v=d+v|0;j=J[m>>2];u=m+4|0;if((j|0)!=(u|0)){while(1){h=0;d=J[g+40>>2];B:{if(!d){f=r;l=f;break B}e=K[j+27|0];b=e<<24>>24>=0;n=b?j+16|0:J[j+16>>2];p=b?e:J[j+20>>2];l=r;while(1){f=d;b=K[d+27|0];k=b<<24>>24<0;q=k?J[d+20>>2]:b;i=q>>>0<p>>>0;C:{D:{E:{F:{e=i?q:p;G:{if(e){d=k?J[d+16>>2]:d+16|0;b=Eb(n,d,e);if(!b){if(p>>>0<q>>>0){break G}break F}if((b|0)>=0){break F}break G}if(p>>>0>=q>>>0){break E}}l=f;d=J[f>>2];if(d){continue}break B}b=Eb(d,n,e);if(b){break D}}if(i){break C}h=f;break B}if((b|0)<0){break C}h=f;break B}l=f+4|0;d=J[f+4>>2];if(d){continue}break}}e=j;i=J[e+4>>2];d=i;H:{if(d){while(1){b=d;d=J[d>>2];if(d){continue}break H}}while(1){b=J[e+8>>2];d=J[b>>2]!=(e|0);e=b;if(d){continue}break}}if(!h){e=j;I:{if(i){while(1){d=i;i=J[d>>2];if(i){continue}break I}}while(1){d=J[e+8>>2];h=J[d>>2]!=(e|0);e=d;if(h){continue}break}}if(J[m>>2]==(j|0)){J[m>>2]=d;}J[m+8>>2]=J[m+8>>2]-1;i=J[m+4>>2];J:{K:{e=j;h=J[e>>2];if(h){d=J[e+4>>2];if(!d){break K}while(1){e=d;d=J[d>>2];if(d){continue}break}}h=J[e+4>>2];if(h){break K}h=0;k=1;break J}J[h+8>>2]=J[e+8>>2];k=0;}n=J[e+8>>2];d=J[n>>2];L:{if((e|0)==(d|0)){J[n>>2]=h;if((e|0)==(i|0)){d=0;i=h;break L}d=J[n+4>>2];break L}J[n+4>>2]=h;}p=!K[e+12|0];if((e|0)!=(j|0)){n=J[j+8>>2];J[e+8>>2]=n;J[n+((J[J[j+8>>2]>>2]!=(j|0))<<2)>>2]=e;n=J[j>>2];J[e>>2]=n;J[n+8>>2]=e;n=J[j+4>>2];J[e+4>>2]=n;if(n){J[n+8>>2]=e;}H[e+12|0]=K[j+12|0];i=(i|0)==(j|0)?e:i;}M:{if(p|!i){break M}if(!k){H[h+12|0]=1;break M}N:{O:{while(1){P:{e=K[d+12|0];k=J[d+8>>2];Q:{if(J[k>>2]!=(d|0)){if(!e){H[d+12|0]=1;H[k+12|0]=0;h=J[k+4>>2];e=J[h>>2];J[k+4>>2]=e;if(e){J[e+8>>2]=k;}J[h+8>>2]=J[k+8>>2];e=J[k+8>>2];J[(((k|0)!=J[e>>2])<<2)+e>>2]=h;J[h>>2]=k;J[k+8>>2]=h;e=d;d=J[d>>2];i=(d|0)==(i|0)?e:i;d=J[d+4>>2];}e=J[d>>2];R:{if(!(K[e+12|0]?0:e)){h=J[d+4>>2];if(!(!h|K[h+12|0])){e=d;break R}H[d+12|0]=0;d=J[d+8>>2];if((i|0)==(d|0)){H[i+12|0]=1;break M}if(K[d+12|0]){break Q}H[d+12|0]=1;break M}h=J[d+4>>2];if(!(!h|K[h+12|0])){e=d;break R}H[e+12|0]=1;H[d+12|0]=0;h=J[e+4>>2];J[d>>2]=h;if(h){J[h+8>>2]=d;}J[e+8>>2]=J[d+8>>2];h=J[d+8>>2];J[((J[h>>2]!=(d|0))<<2)+h>>2]=e;J[e+4>>2]=d;J[d+8>>2]=e;h=d;}i=J[e+8>>2];H[e+12|0]=K[i+12|0];H[i+12|0]=1;H[h+12|0]=1;e=J[i+4>>2];d=J[e>>2];J[i+4>>2]=d;if(d){J[d+8>>2]=i;}J[e+8>>2]=J[i+8>>2];d=J[i+8>>2];J[(((i|0)!=J[d>>2])<<2)+d>>2]=e;J[e>>2]=i;J[i+8>>2]=e;break M}if(!e){H[d+12|0]=1;H[k+12|0]=0;e=J[d+4>>2];J[k>>2]=e;if(e){J[e+8>>2]=k;}J[d+8>>2]=J[k+8>>2];e=J[k+8>>2];J[(((k|0)!=J[e>>2])<<2)+e>>2]=d;J[d+4>>2]=k;J[k+8>>2]=d;i=(i|0)==(k|0)?d:i;d=J[k>>2];}h=J[d>>2];if(!(!h|K[h+12|0])){e=d;break N}e=J[d+4>>2];if(K[e+12|0]?0:e){break O}H[d+12|0]=0;d=J[d+8>>2];if(!K[d+12|0]|(d|0)==(i|0)){break P}}e=d;d=J[d+8>>2];d=J[(((e|0)==J[d>>2])<<2)+d>>2];continue}break}H[d+12|0]=1;break M}if(!(K[h+12|0]|!h)){e=d;break N}H[e+12|0]=1;H[d+12|0]=0;h=J[e>>2];J[d+4>>2]=h;if(h){J[h+8>>2]=d;}J[e+8>>2]=J[d+8>>2];h=J[d+8>>2];J[((J[h>>2]!=(d|0))<<2)+h>>2]=e;J[e>>2]=d;J[d+8>>2]=e;h=d;}i=J[e+8>>2];H[e+12|0]=K[i+12|0];H[i+12|0]=1;H[h+12|0]=1;e=J[i>>2];d=J[e+4>>2];J[i>>2]=d;if(d){J[d+8>>2]=i;}J[e+8>>2]=J[i+8>>2];d=J[i+8>>2];J[(((i|0)!=J[d>>2])<<2)+d>>2]=e;J[e+4>>2]=i;J[i+8>>2]=e;}J[j+8>>2]=f;J[j>>2]=0;J[j+4>>2]=0;J[l>>2]=j;d=J[J[g+36>>2]>>2];if(d){J[g+36>>2]=d;j=J[l>>2];}zd(J[g+40>>2],j);J[g+44>>2]=J[g+44>>2]+1;}j=b;if((b|0)!=(u|0)){continue}break}}if(J[g+44>>2]!=(v|0)){break A}m=m+12|0;if((t|0)!=(m|0)){continue}break}}J[g+8>>2]=J[g+36>>2];J[g+16>>2]=v;b=J[g+40>>2];J[g+12>>2]=b;S:{if(!v){J[g+8>>2]=x;break S}J[b+8>>2]=x;J[g+40>>2]=0;J[g+44>>2]=0;J[g+36>>2]=r;}b=1;break z}H[g+8|0]=0;b=0;}H[g+20|0]=b;Db(J[g+40>>2]);if(K[g+20|0]){break g}}H[a+12|0]=0;H[a|0]=0;break c}d=J[g+28>>2];h=J[s+36>>2]-J[s+32>>2]>>3;e=J[g+24>>2];b=((d+P(h,-12)|0)-e|0)/12|0;T:{if(!P(h,-12)){f=d;break T}i=e+P(b,12)|0;f=i;b=f+P((P(h,12)|0)/12|0,12)|0;if((d|0)!=(b|0)){while(1){Db(J[f+4>>2]);J[f>>2]=J[b>>2];h=J[b+4>>2];J[f+4>>2]=h;e=J[b+8>>2];J[f+8>>2]=e;j=f+4|0;U:{if(!e){J[f>>2]=j;break U}J[h+8>>2]=j;J[b+4>>2]=0;J[b+8>>2]=0;J[b>>2]=b+4;}f=f+12|0;b=b+12|0;if((d|0)!=(b|0)){continue}break}d=J[g+28>>2];}f=i+P((f-i|0)/12|0,12)|0;if((f|0)!=(d|0)){while(1){d=d-12|0;Db(J[d+4>>2]);if((d|0)!=(f|0)){continue}break}}J[g+28>>2]=f;}b=J[g+32>>2];V:{if(b>>>0>f>>>0){J[f>>2]=J[g+8>>2];d=J[g+12>>2];J[f+4>>2]=d;b=J[g+16>>2];J[f+8>>2]=b;e=f+4|0;W:{if(!b){J[f>>2]=e;break W}J[d+8>>2]=e;J[g+12>>2]=0;J[g+16>>2]=0;J[g+8>>2]=x;}J[g+28>>2]=f+12;break V}X:{Y:{k=J[g+24>>2];h=(f-k|0)/12|0;e=h+1|0;Z:{if(e>>>0>=357913942){J[5996]=0;ca(96);break Z}d=(b-k|0)/12|0;b=d<<1;d=d>>>0>=178956970?357913941:b>>>0>e>>>0?b:e;if(!d){e=0;break X}if(d>>>0<357913942){break Y}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break a}break e}J[5996]=0;e=$(3,P(d,12)|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break e}}b=P(h,12)+e|0;J[b>>2]=J[g+8>>2];j=J[g+12>>2];J[b+4>>2]=j;h=J[g+16>>2];J[b+8>>2]=h;d=P(d,12);i=b+4|0;_:{if(!h){J[b>>2]=i;break _}J[j+8>>2]=i;J[g+12>>2]=0;J[g+16>>2]=0;J[g+8>>2]=x;}l=d+e|0;i=b+12|0;$:{if((f|0)==(k|0)){J[g+32>>2]=l;J[g+28>>2]=i;J[g+24>>2]=b;break $}while(1){b=b-12|0;f=f-12|0;J[b>>2]=J[f>>2];j=b+4|0;h=f+4|0;e=J[h>>2];J[j>>2]=e;d=J[f+8>>2];J[b+8>>2]=d;aa:{if(!d){J[b>>2]=j;break aa}J[e+8>>2]=j;J[f>>2]=h;J[f+4>>2]=0;J[f+8>>2]=0;}if((f|0)!=(k|0)){continue}break}J[g+32>>2]=l;d=J[g+28>>2];J[g+28>>2]=i;f=J[g+24>>2];J[g+24>>2]=b;if((d|0)==(f|0)){break $}while(1){d=d-12|0;Db(J[d+4>>2]);if((d|0)!=(f|0)){continue}break}}if(!f){break V}$a(f);}if(K[g+20|0]){Db(J[g+12>>2]);}f=D;}if((f|0)!=(o|0)){continue}break d}break}b=Z()|0;if(!K[g+20|0]){break b}Db(J[g+12>>2]);break b}e=J[g+24>>2];if((J[g+28>>2]-e|0)!=12){J[5996]=0;ga(44,4102,2210,450,2439);a=J[5996];J[5996]=0;if((a|0)!=1){break a}b=Z()|0;break b}J[a>>2]=J[e>>2];c=J[e+4>>2];J[a+4>>2]=c;b=J[e+8>>2];J[a+8>>2]=b;d=a+4|0;ba:{if(!b){J[a>>2]=d;break ba}J[c+8>>2]=d;J[e+4>>2]=0;J[e+8>>2]=0;J[e>>2]=e+4;}H[a+12|0]=1;}a=J[g+24>>2];if(a){f=J[g+28>>2];if((a|0)!=(f|0)){while(1){f=f-12|0;Db(J[f+4>>2]);if((a|0)!=(f|0)){continue}break}}$a(a);}if(o){$a(o);}Ua=g+48|0;return}Ug(g+24|0);if(o){$a(o);}da(b|0);B();}a=Z()|0;Ug(g+24|0);da(a|0);B();}B();}function fg(a,b,c){var d=0,e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0;e=Ua-144|0;Ua=e;d=Sb(e,0,144);J[d+76>>2]=-1;J[d+44>>2]=a;J[d+32>>2]=178;J[d+84>>2]=a;s=c;a=0;j=Ua-304|0;Ua=j;a:{b:{c:{d:{if(J[d+4>>2]){break d}Dg(d);if(J[d+4>>2]){break d}break c}h=K[b|0];if(!h){break a}e:{f:{g:{h:{while(1){i:{c=h&255;j:{if((c|0)==32|c-9>>>0<5){while(1){h=b;b=b+1|0;c=K[h+1|0];if((c|0)==32|c-9>>>0<5){continue}break}Yb(d,0,0);while(1){b=J[d+4>>2];k:{if((b|0)!=J[d+104>>2]){J[d+4>>2]=b+1;b=K[b|0];break k}b=gb(d);}if((b|0)==32|b-9>>>0<5){continue}break}b=J[d+4>>2];c=J[d+116>>2];if((c|0)>0|(c|0)>=0){b=b-1|0;J[d+4>>2]=b;}b=b-J[d+44>>2]|0;e=b;i=o+J[d+124>>2]|0;c=b>>31;b=m+J[d+120>>2]|0;i=c+(b>>>0<m>>>0?i+1|0:i)|0;m=b+e|0;o=m>>>0<b>>>0?i+1|0:i;break j}l:{m:{n:{if((c|0)==37){c=K[b+1|0];if((c|0)==42){break n}if((c|0)!=37){break m}}Yb(d,0,0);o:{if(K[b|0]==37){while(1){c=J[d+4>>2];p:{if((c|0)!=J[d+104>>2]){J[d+4>>2]=c+1;h=K[c|0];break p}h=gb(d);}if((h|0)==32|h-9>>>0<5){continue}break}b=b+1|0;break o}c=J[d+4>>2];if((c|0)!=J[d+104>>2]){J[d+4>>2]=c+1;h=K[c|0];break o}h=gb(d);}if(K[b|0]!=(h|0)){b=J[d+116>>2];if((b|0)>0|(b|0)>=0){J[d+4>>2]=J[d+4>>2]-1;}if((h|0)>=0){break a}g=0;if(t){break a}break c}c=J[d+4>>2]-J[d+44>>2]|0;h=c;f=o+J[d+124>>2]|0;e=c>>31;c=m+J[d+120>>2]|0;g=e+(c>>>0<m>>>0?f+1|0:f)|0;m=c+h|0;o=m>>>0<c>>>0?g+1|0:g;h=b;break j}l=0;b=b+2|0;break l}e=c-48|0;if(!(K[b+2|0]!=36|e>>>0>=10)){c=Ua-16|0;J[c+12>>2]=s;f=c;c=(e>>>0>1?(e<<2)-4|0:0)+s|0;J[f+8>>2]=c+4;l=J[c>>2];b=b+3|0;break l}l=J[s>>2];s=s+4|0;b=b+1|0;}c=0;while(1){f=K[b|0];if(f-48>>>0<10){b=b+1|0;c=(P(c,10)+f|0)-48|0;continue}break}r=0;if((f|0)==109){p=0;r=(l|0)!=0;f=K[b+1|0];b=b+1|0;a=0;}h=b+1|0;k=3;g=r;q:{r:{switch((f&255)-65|0){case 39:e=b+2|0;b=K[b+1|0]==104;h=b?e:h;k=b?-2:-1;break q;case 43:e=b+2|0;b=K[b+1|0]==108;h=b?e:h;k=b?3:1;break q;case 51:case 57:k=1;break q;case 11:k=2;break q;case 41:break q;case 0:case 2:case 4:case 5:case 6:case 18:case 23:case 26:case 32:case 34:case 35:case 36:case 37:case 38:case 40:case 45:case 46:case 47:case 50:case 52:case 55:break r;default:break e}}k=0;h=b;}e=K[h|0];b=(e&47)==3;u=b?1:k;q=b?e|32:e;s:{if((q|0)==91){break s}t:{if((q|0)!=110){if((q|0)!=99){break t}c=(c|0)<=1?1:c;break s}gg(l,u,m,o);break j}Yb(d,0,0);while(1){b=J[d+4>>2];u:{if((b|0)!=J[d+104>>2]){J[d+4>>2]=b+1;b=K[b|0];break u}b=gb(d);}if((b|0)==32|b-9>>>0<5){continue}break}b=J[d+4>>2];e=J[d+116>>2];if((e|0)>0|(e|0)>=0){b=b-1|0;J[d+4>>2]=b;}b=b-J[d+44>>2]|0;f=b;i=o+J[d+124>>2]|0;e=b>>31;b=m+J[d+120>>2]|0;o=e+(b>>>0<m>>>0?i+1|0:i)|0;m=b+f|0;o=m>>>0<b>>>0?o+1|0:o;}i=c;n=c>>31;Yb(d,c,n);b=J[d+4>>2];v:{if((b|0)!=J[d+104>>2]){J[d+4>>2]=b+1;break v}if((gb(d)|0)<0){break f}}b=J[d+116>>2];if((b|0)>0|(b|0)>=0){J[d+4>>2]=J[d+4>>2]-1;}b=16;w:{x:{y:{z:{A:{switch(q-88|0){default:b=q-65|0;if(b>>>0>6|!(1<<b&113)){break w}case 9:case 13:case 14:case 15:ig(j+8|0,d,u,0);c=J[d+4>>2]-J[d+44>>2]|0;if(J[d+120>>2]!=(0-c|0)|J[d+124>>2]!=(0-((c>>31)+((c|0)!=0)|0)|0)){break y}break g;case 3:case 11:case 27:if((q|16)==115){Sb(j+32|0,-1,257);H[j+32|0]=0;if((q|0)!=115){break x}H[j+65|0]=0;H[j+46|0]=0;I[j+42>>1]=0;I[j+44>>1]=0;break x}f=K[h+1|0];e=(f|0)==94;Sb(j+32|0,e,257);H[j+32|0]=0;b=e?h+2|0:h+1|0;B:{C:{D:{e=K[(e?2:1)+h|0];if((e|0)!=45){if((e|0)==93){break D}k=(f|0)!=94;break B}k=(f|0)!=94;H[j+78|0]=k;break C}k=(f|0)!=94;H[j+126|0]=k;}b=b+1|0;}h=b;while(1){f=K[h|0];E:{if((f|0)!=45){if(!f){break f}if((f|0)==93){break x}break E}f=45;g=K[h+1|0];if(!g|(g|0)==93){break E}e=h+1|0;b=K[h-1|0];F:{if(g>>>0<=b>>>0){f=g;break F}while(1){b=b+1|0;H[b+(j+32|0)|0]=k;f=K[e|0];if(f>>>0>b>>>0){continue}break}}h=e;}H[(f+j|0)+33|0]=k;h=h+1|0;continue}case 23:b=8;break z;case 12:case 29:b=10;break z;case 1:case 2:case 4:case 5:case 6:case 7:case 8:case 10:case 16:case 18:case 19:case 20:case 21:case 22:case 25:case 26:case 28:case 30:case 31:break w;case 0:case 24:case 32:break z;case 17:break A}}b=0;}e=0;i=0;f=0;g=0;w=0;v=Ua-16|0;Ua=v;G:{if((b|0)==1){J[5732]=28;break G}while(1){c=J[d+4>>2];H:{if((c|0)!=J[d+104>>2]){J[d+4>>2]=c+1;c=K[c|0];break H}c=gb(d);}if((c|0)==32|c-9>>>0<5){continue}break}I:{J:{switch(c-43|0){case 0:case 2:break J;default:break I}}w=(c|0)==45?-1:0;c=J[d+4>>2];if((c|0)!=J[d+104>>2]){J[d+4>>2]=c+1;c=K[c|0];break I}c=gb(d);}K:{L:{M:{N:{if(!((b|0)!=0&(b|0)!=16|(c|0)!=48)){c=J[d+4>>2];O:{if((c|0)!=J[d+104>>2]){J[d+4>>2]=c+1;c=K[c|0];break O}c=gb(d);}if((c&-33)==88){b=16;c=J[d+4>>2];P:{if((c|0)!=J[d+104>>2]){J[d+4>>2]=c+1;c=K[c|0];break P}c=gb(d);}if(K[c+8641|0]<16){break M}b=J[d+116>>2];if((b|0)>0|(b|0)>=0){J[d+4>>2]=J[d+4>>2]-1;}Yb(d,0,0);break G}if(b){break N}b=8;break M}b=b?b:10;if(b>>>0>K[c+8641|0]){break N}b=J[d+116>>2];if((b|0)>0|(b|0)>=0){J[d+4>>2]=J[d+4>>2]-1;}Yb(d,0,0);J[5732]=28;break G}if((b|0)!=10){break M}f=c-48|0;if(f>>>0<=9){c=0;while(1){c=P(c,10)+f|0;g=c>>>0<429496729;e=J[d+4>>2];Q:{if((e|0)!=J[d+104>>2]){J[d+4>>2]=e+1;b=K[e|0];break Q}b=gb(d);}f=b-48|0;if(g&f>>>0<=9){continue}break}e=c;}if(f>>>0>9){break K}b=co(e,0,10,0);g=Wa;while(1){e=b+f|0;i=e>>>0<f>>>0?g+1|0:g;b=(i|0)==429496729&e>>>0<2576980378|i>>>0<429496729;c=J[d+4>>2];R:{if((c|0)!=J[d+104>>2]){J[d+4>>2]=c+1;c=K[c|0];break R}c=gb(d);}k=c-48|0;if(!(b&k>>>0<=9)){b=10;if(k>>>0<=9){break L}break K}b=co(e,i,10,0);g=Wa;f=k;if((g|0)==-1&(f^-1)>>>0>=b>>>0|(g|0)!=-1){continue}break}b=10;break L}if(b-1&b){g=K[c+8641|0];if(g>>>0<b>>>0){while(1){f=P(b,f)+g|0;e=f>>>0<119304647;c=J[d+4>>2];S:{if((c|0)!=J[d+104>>2]){J[d+4>>2]=c+1;c=K[c|0];break S}c=gb(d);}g=K[c+8641|0];if(e&g>>>0<b>>>0){continue}break}e=f;}if(b>>>0<=g>>>0){break L}n=b;while(1){f=co(e,i,n,0);k=Wa;g=g&255;if((k|0)==-1&(g^-1)>>>0<f>>>0){break L}i=k;e=f+g|0;i=e>>>0<g>>>0?i+1|0:i;c=J[d+4>>2];T:{if((c|0)!=J[d+104>>2]){J[d+4>>2]=c+1;c=K[c|0];break T}c=gb(d);}g=K[c+8641|0];if(b>>>0<=g>>>0){break L}xb(v,n,0,0,0,e,i,0,0);if(!(J[v+8>>2]|J[v+12>>2])){continue}break}break L}k=H[(P(b,23)>>>5&7)+8897|0];f=K[c+8641|0];if(f>>>0<b>>>0){while(1){g=g<<k|f;e=g>>>0<134217728;c=J[d+4>>2];U:{if((c|0)!=J[d+104>>2]){J[d+4>>2]=c+1;c=K[c|0];break U}c=gb(d);}f=K[c+8641|0];if(e&f>>>0<b>>>0){continue}break}e=g;}if(b>>>0<=f>>>0){break L}n=k&31;if((k&63)>>>0>=32){g=0;n=-1>>>n|0;}else {g=-1>>>n|0;n=g|(1<<n)-1<<32-n;}if(!g&e>>>0>n>>>0){break L}while(1){x=f&255;c=e;f=k&31;if((k&63)>>>0>=32){i=c<<f;c=0;}else {i=(1<<f)-1&c>>>32-f|i<<f;c=c<<f;}e=x|c;c=J[d+4>>2];V:{if((c|0)!=J[d+104>>2]){J[d+4>>2]=c+1;c=K[c|0];break V}c=gb(d);}f=K[c+8641|0];if(b>>>0<=f>>>0){break L}if((g|0)==(i|0)&e>>>0<=n>>>0|g>>>0>i>>>0){continue}break}}if(K[c+8641|0]>=b>>>0){break K}while(1){c=J[d+4>>2];W:{if((c|0)!=J[d+104>>2]){J[d+4>>2]=c+1;c=K[c|0];break W}c=gb(d);}if(K[c+8641|0]<b>>>0){continue}break}J[5732]=68;e=-1;i=-1;w=0;}b=J[d+116>>2];if((b|0)>0|(b|0)>=0){J[d+4>>2]=J[d+4>>2]-1;}b=w;f=b^e;e=f-b|0;c=b>>31;i=(c^i)-((b>>>0>f>>>0)+c|0)|0;}Ua=v+16|0;c=J[d+4>>2]-J[d+44>>2]|0;if(J[d+120>>2]==(0-c|0)&J[d+124>>2]==(0-((c>>31)+((c|0)!=0)|0)|0)){break g}if(!(!l|(q|0)!=112)){J[l>>2]=e;break w}gg(l,u,e,i);break w}if(!l){break w}c=J[j+16>>2];f=J[j+20>>2];b=J[j+8>>2];e=J[j+12>>2];X:{switch(u|0){case 0:N[l>>2]=Vf(b,e,c,f);break w;case 1:O[l>>3]=ge(b,e,c,f);break w;case 2:break X;default:break w}}J[l>>2]=b;J[l+4>>2]=e;J[l+8>>2]=c;J[l+12>>2]=f;break w}k=(q|0)!=99;c=k?31:c+1|0;Y:{if((u|0)==1){f=l;if(r){f=sb(c<<2);if(!f){break h}}J[j+296>>2]=0;J[j+300>>2]=0;b=0;while(1){a=f;Z:{while(1){e=J[d+4>>2];_:{if((e|0)!=J[d+104>>2]){J[d+4>>2]=e+1;e=K[e|0];break _}e=gb(d);}if(!K[(e+j|0)+33|0]){break Z}H[j+27|0]=e;e=qd(j+28|0,j+27|0,1,j+296|0);if((e|0)==-2){continue}p=0;if((e|0)==-1){break f}if(a){J[(b<<2)+a>>2]=J[j+28>>2];b=b+1|0;}if(!r|(b|0)!=(c|0)){continue}break}g=1;b=c;c=b<<1|1;f=Vc(a,c<<2);if(f){continue}break e}break}p=0;c=a;if(j+296|0?J[j+296>>2]:0){break f}break Y}if(r){b=0;f=sb(c);if(!f){break h}while(1){a=f;while(1){e=J[d+4>>2];$:{if((e|0)!=J[d+104>>2]){J[d+4>>2]=e+1;e=K[e|0];break $}e=gb(d);}if(!K[(e+j|0)+33|0]){c=0;p=a;break Y}H[a+b|0]=e;b=b+1|0;if((c|0)!=(b|0)){continue}break}g=1;b=c;c=b<<1|1;f=Vc(a,c);if(f){continue}break}p=a;a=0;break e}b=0;if(l){while(1){a=J[d+4>>2];aa:{if((a|0)!=J[d+104>>2]){J[d+4>>2]=a+1;a=K[a|0];break aa}a=gb(d);}if(K[(a+j|0)+33|0]){H[b+l|0]=a;b=b+1|0;continue}else {c=0;a=l;p=a;break Y}}}while(1){a=J[d+4>>2];ba:{if((a|0)!=J[d+104>>2]){J[d+4>>2]=a+1;a=K[a|0];break ba}a=gb(d);}if(K[(a+j|0)+33|0]){continue}break}a=0;p=0;c=0;}f=J[d+4>>2];e=J[d+116>>2];if((e|0)>0|(e|0)>=0){f=f-1|0;J[d+4>>2]=f;}e=f-J[d+44>>2]|0;f=e+J[d+120>>2]|0;g=J[d+124>>2]+(e>>31)|0;g=f>>>0<e>>>0?g+1|0:g;if(!(g|f)|!(k|(f|0)==(i|0)&(g|0)==(n|0))){break i}if(r){J[l>>2]=a;}ca:{if((q|0)==99){break ca}if(c){J[(b<<2)+c>>2]=0;}if(!p){p=0;break ca}H[b+p|0]=0;}a=c;}b=J[d+4>>2]-J[d+44>>2]|0;e=b;f=o+J[d+124>>2]|0;c=b>>31;b=m+J[d+120>>2]|0;i=c+(b>>>0<m>>>0?f+1|0:f)|0;m=b+e|0;o=m>>>0<b>>>0?i+1|0:i;t=((l|0)!=0)+t|0;}b=h+1|0;h=K[h+1|0];if(h){continue}break a}break}a=c;break g}g=1;p=0;a=0;break e}g=r;break b}g=r;}if(t){break b}}t=-1;}if(!g){break a}$a(p);$a(a);}Ua=j+304|0;Ua=d+144|0;return t}function ao(a){a=a|0;var b=0,c=0,d=0,e=0,f=0,g=0;d=Ua-320|0;Ua=d;a:{b:{c:{d:{e:{f:{g:{h:{i:{j:{k:{l:{m:{n:{o:{p:{q:{r:{s:{t:{u:{v:{w:{if(!K[24336]){J[d+208>>2]=0;J[d+200>>2]=0;J[d+204>>2]=0;J[5996]=0;ba(66,24324,d+200|0,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break w}b=J[d+200>>2];if(b){J[d+204>>2]=b;$a(b);}H[24336]=1;}if(!K[24352]){J[5996]=0;b=d+200|0;ea(68,b|0,1598,8);c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;b=ba(69,d+308|0,b|0,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;ba(66,24340,b|0,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break v}c=J[b>>2];if(c){J[b+4>>2]=c;$a(c);}H[24352]=1;}if(!K[24368]){J[5996]=0;b=d+200|0;ea(68,b|0,1194,10);c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;b=ba(69,d+296|0,b|0,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;ba(66,24356,b|0,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break u}c=J[b>>2];if(c){J[b+4>>2]=c;$a(c);}H[24368]=1;}if(!K[24384]){J[5996]=0;b=d+200|0;ea(68,b|0,2432,6);c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;b=ba(69,d+284|0,b|0,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;ba(66,24372,b|0,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break t}c=J[b>>2];if(c){J[b+4>>2]=c;$a(c);}H[24384]=1;}if(!K[24400]){J[5996]=0;b=d+200|0;ea(68,b|0,1492,6);c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;b=ba(69,d+272|0,b|0,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;ba(66,24388,b|0,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break s}c=J[b>>2];if(c){J[b+4>>2]=c;$a(c);}H[24400]=1;}if(!K[24416]){J[5996]=0;b=d+200|0;ea(68,b|0,1152,8);c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;b=ba(69,d+260|0,b|0,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;ba(66,24404,b|0,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break r}c=J[b>>2];if(c){J[b+4>>2]=c;$a(c);}H[24416]=1;}if(!K[24432]){J[5996]=0;b=d+200|0;ea(68,b|0,1526,8);c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;b=ba(69,d+248|0,b|0,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;ba(66,24420,b|0,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break q}c=J[b>>2];if(c){J[b+4>>2]=c;$a(c);}H[24432]=1;}if(!K[24448]){J[5996]=0;b=d+200|0;ea(68,b|0,1628,10);c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;b=ba(69,d+236|0,b|0,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;ba(66,24436,b|0,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break p}c=J[b>>2];if(c){J[b+4>>2]=c;$a(c);}H[24448]=1;}if(!K[24464]){J[5996]=0;b=d+116|0;ea(68,b|0,1435,7);c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;ea(68,d+124|0,1183,10);c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;c=ba(69,d+200|0,b|0,2)|0;b=J[5996];J[5996]=0;if((b|0)==1){break c}J[5996]=0;e=d+80|0;ea(68,e|0,1478,7);f=J[5996];J[5996]=0;b=c+12|0;if((f|0)==1){break n}J[5996]=0;ea(68,d+88|0,1172,10);f=J[5996];J[5996]=0;if((f|0)==1){break n}J[5996]=0;f=ba(69,b|0,e|0,2)|0;e=J[5996];J[5996]=0;if((e|0)==1){break n}J[5996]=0;e=d+56|0;ea(68,e|0,1435,7);g=J[5996];J[5996]=0;b=c+24|0;if((g|0)==1){break n}J[5996]=0;ea(68,d- -64|0,1161,10);g=J[5996];J[5996]=0;if((g|0)==1){break n}J[5996]=0;g=ba(69,b|0,e|0,2)|0;e=J[5996];J[5996]=0;if((e|0)==1){break n}J[5996]=0;ba(66,24452,c|0,3)|0;b=J[5996];J[5996]=0;if((b|0)==1){break o}b=J[g>>2];if(b){J[c+28>>2]=b;$a(b);}b=J[f>>2];if(b){J[c+16>>2]=b;$a(b);}b=J[c>>2];if(b){J[c+4>>2]=b;$a(b);}H[24464]=1;}if(!K[24480]){J[5996]=0;b=d+116|0;ea(68,b|0,1628,10);c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;ea(68,d+124|0,1617,10);c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;b=ba(69,d+188|0,b|0,2)|0;c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;ba(66,24468,b|0,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break m}c=J[b>>2];if(c){J[b+4>>2]=c;$a(c);}H[24480]=1;}if(!K[24496]){J[5996]=0;b=d+116|0;ea(68,b|0,1462,8);c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;ea(68,d+124|0,1453,8);c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;b=ba(69,d+176|0,b|0,2)|0;c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;ba(66,24484,b|0,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break l}c=J[b>>2];if(c){J[b+4>>2]=c;$a(c);}H[24496]=1;}if(!K[24512]){J[5996]=0;b=d+116|0;ea(68,b|0,1443,9);c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;ea(68,d+124|0,1587,10);c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;b=ba(69,d+164|0,b|0,2)|0;c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;ba(66,24500,b|0,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break k}c=J[b>>2];if(c){J[b+4>>2]=c;$a(c);}H[24512]=1;}if(!K[24528]){J[5996]=0;b=d+116|0;ea(68,b|0,1443,9);c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;ea(68,d+124|0,1471,6);c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;b=ba(69,d+152|0,b|0,2)|0;c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;ba(66,24516,b|0,1)|0;c=J[5996];J[5996]=0;if((c|0)==1){break j}c=J[b>>2];if(c){J[b+4>>2]=c;$a(c);}H[24528]=1;}if(!K[24544]){J[5996]=0;b=d+80|0;ea(68,b|0,1535,9);c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;ea(68,d+88|0,1535,9);c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;c=ba(69,d+116|0,b|0,2)|0;b=J[5996];J[5996]=0;if((b|0)==1){break c}J[5996]=0;e=d+56|0;ea(68,e|0,1555,9);f=J[5996];J[5996]=0;b=c+12|0;if((f|0)==1){break h}J[5996]=0;ea(68,d- -64|0,1555,9);f=J[5996];J[5996]=0;if((f|0)==1){break h}J[5996]=0;f=ba(69,b|0,e|0,2)|0;e=J[5996];J[5996]=0;if((e|0)==1){break h}J[5996]=0;e=d+32|0;ea(68,e|0,1545,9);g=J[5996];J[5996]=0;b=c+24|0;if((g|0)==1){break h}J[5996]=0;ea(68,d+40|0,1545,9);g=J[5996];J[5996]=0;if((g|0)==1){break h}J[5996]=0;g=ba(69,b|0,e|0,2)|0;e=J[5996];J[5996]=0;if((e|0)==1){break h}J[5996]=0;ba(66,24532,c|0,3)|0;b=J[5996];J[5996]=0;if((b|0)==1){break i}b=J[g>>2];if(b){J[c+28>>2]=b;$a(b);}b=J[f>>2];if(b){J[c+16>>2]=b;$a(b);}b=J[c>>2];if(b){J[c+4>>2]=b;$a(b);}H[24544]=1;}if(!K[24560]){J[5996]=0;b=d+56|0;ea(68,b|0,1443,9);c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;ea(68,d- -64|0,1517,8);c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;ea(68,d+72|0,1587,10);c=J[5996];J[5996]=0;if((c|0)==1){break c}J[5996]=0;c=ba(69,d+80|0,b|0,3)|0;b=J[5996];J[5996]=0;if((b|0)==1){break c}J[5996]=0;e=d+32|0;ea(68,e|0,1443,9);f=J[5996];J[5996]=0;b=c+12|0;if((f|0)==1){break f}J[5996]=0;ea(68,d+40|0,1508,8);f=J[5996];J[5996]=0;if((f|0)==1){break f}J[5996]=0;ea(68,d+48|0,1576,10);f=J[5996];J[5996]=0;if((f|0)==1){break f}J[5996]=0;f=ba(69,b|0,e|0,3)|0;e=J[5996];J[5996]=0;if((e|0)==1){break f}J[5996]=0;e=d+8|0;ea(68,e|0,1443,9);g=J[5996];J[5996]=0;b=c+24|0;if((g|0)==1){break f}J[5996]=0;ea(68,d+16|0,1499,8);g=J[5996];J[5996]=0;if((g|0)==1){break f}J[5996]=0;ea(68,d+24|0,1565,10);g=J[5996];J[5996]=0;if((g|0)==1){break f}J[5996]=0;g=ba(69,b|0,e|0,3)|0;e=J[5996];J[5996]=0;if((e|0)==1){break f}J[5996]=0;ba(66,24548,c|0,3)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}b=J[g>>2];if(b){J[c+28>>2]=b;$a(b);}b=J[f>>2];if(b){J[c+16>>2]=b;$a(b);}b=J[c>>2];if(b){J[c+4>>2]=b;$a(b);}H[24560]=1;}if(!(50331647>>>a&1)){break e}Ua=d+320|0;return J[(a<<2)+6864>>2]}e=Z()|0;a=J[d+200>>2];if(!a){break d}J[d+204>>2]=a;$a(a);break d}e=Z()|0;a=J[b>>2];if(!a){break d}break b}e=Z()|0;a=J[b>>2];if(!a){break d}break b}e=Z()|0;a=J[b>>2];if(!a){break d}break b}e=Z()|0;a=J[b>>2];if(!a){break d}break b}e=Z()|0;a=J[b>>2];if(!a){break d}break b}e=Z()|0;a=J[b>>2];if(!a){break d}break b}e=Z()|0;a=J[b>>2];if(!a){break d}break b}e=Z()|0;a=J[g>>2];if(a){J[c+28>>2]=a;$a(a);}a=J[f>>2];if(a){J[c+16>>2]=a;$a(a);}a=J[c>>2];if(!a){break d}break a}e=Z()|0;while(1){a=b-12|0;d=J[a>>2];if(d){J[b-8>>2]=d;$a(d);}b=a;if((b|0)!=(c|0)){continue}break}break d}e=Z()|0;a=J[b>>2];if(!a){break d}break b}e=Z()|0;a=J[b>>2];if(!a){break d}break b}e=Z()|0;a=J[b>>2];if(!a){break d}break b}e=Z()|0;a=J[b>>2];if(!a){break d}break b}e=Z()|0;a=J[g>>2];if(a){J[c+28>>2]=a;$a(a);}a=J[f>>2];if(a){J[c+16>>2]=a;$a(a);}a=J[c>>2];if(!a){break d}break a}e=Z()|0;while(1){a=b-12|0;d=J[a>>2];if(d){J[b-8>>2]=d;$a(d);}b=a;if((b|0)!=(c|0)){continue}break}break d}e=Z()|0;a=J[g>>2];if(a){J[c+28>>2]=a;$a(a);}a=J[f>>2];if(a){J[c+16>>2]=a;$a(a);}a=J[c>>2];if(!a){break d}break a}e=Z()|0;while(1){a=b-12|0;d=J[a>>2];if(d){J[b-8>>2]=d;$a(d);}b=a;if((b|0)!=(c|0)){continue}break}break d}la(2350,1900,622,1759);B();}da(e|0);B();}da(Z()|0);B();}J[b+4>>2]=a;$a(a);da(e|0);B();}J[c+4>>2]=a;$a(a);da(e|0);B();}function wd(a,b,c,d,e){var f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0;h=Ua-144|0;Ua=h;Yc(h+56|0,c,O[a+16>>3],O[a+24>>3],0,0);J[5996]=0;f=$(46,c|0)|0;g=J[5996];J[5996]=0;a:{b:{c:{if((g|0)!=1){J[h+52>>2]=0;J[h+44>>2]=0;J[h+48>>2]=0;H[h+128|0]=0;J[h+124>>2]=h+44;g=J[f+4>>2];j=J[f>>2];k=g-j|0;i=(k|0)/12|0;if((g|0)!=(j|0)){d:{e:{f:{if(i>>>0>=357913942){J[5996]=0;ca(47);a=J[5996];J[5996]=0;if((a|0)==1){break f}break a}J[5996]=0;g=$(3,k|0)|0;j=J[5996];J[5996]=0;if((j|0)==1){break f}J[h+48>>2]=g;J[h+44>>2]=g;J[h+52>>2]=g+P(i,12);i=J[f+4>>2];f=J[f>>2];J[5996]=0;f=ia(48,h+52|0,f|0,i|0,g|0)|0;i=J[5996];J[5996]=0;if((i|0)!=1){break d}f=Z()|0;J[h+48>>2]=g;break e}f=Z()|0;}Nc(h+124|0);break c}J[h+48>>2]=f;}J[h+40>>2]=0;J[h+32>>2]=0;J[h+36>>2]=0;f=J[d+4>>2];g=J[d>>2];J[5996]=0;aa(49,h+32|0,f-g>>2);f=J[5996];J[5996]=0;g:{if((f|0)!=1){f=J[h+48>>2];g=J[h+44>>2];if((f|0)==(g|0)){break b}f=g;h:{while(1){i:{J[h+28>>2]=0;J[h+20>>2]=0;J[h+24>>2]=0;i=J[d+4>>2];g=J[d>>2];n=P(s,12);f=n+f|0;if(i-g>>2!=J[f+4>>2]-J[f>>2]>>3){break i}j=0;r=1;k=0;j:{if((g|0)!=(i|0)){k:{l:{m:{while(1){f=k<<3;v=O[f+J[h+68>>2]>>3];w=O[f+J[h+56>>2]>>3];g=J[(k<<2)+g>>2];J[5996]=0;g=qa(50,g|0,+w,+v,b|0)|0;i=J[5996];J[5996]=0;if((i|0)==1){break l}f=f+J[n+J[h+44>>2]>>2]|0;i=J[f>>2];f=J[f+4>>2];m=J[g+4>>2];g=J[g>>2];J[5996]=0;oa(51,h+124|0,g|0,m|0,i|0,f|0);f=J[5996];J[5996]=0;if((f|0)!=1){i=J[h+28>>2];n:{if(i>>>0>j>>>0){J[j+8>>2]=0;J[j>>2]=0;J[j+4>>2]=0;J[j>>2]=J[h+124>>2];J[j+4>>2]=J[h+128>>2];J[j+8>>2]=J[h+132>>2];J[h+24>>2]=j+12;break n}o:{g=J[h+20>>2];m=(j-g|0)/12|0;f=m+1|0;p:{if(f>>>0>=357913942){J[5996]=0;ca(52);break p}i=(i-g|0)/12|0;l=i<<1;f=i>>>0>=178956970?357913941:f>>>0<l>>>0?l:f;if(f>>>0<357913942){break o}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break a}break m}J[5996]=0;l=P(f,12);i=$(3,l|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){break m}f=i+P(m,12)|0;J[f>>2]=J[h+124>>2];J[f+4>>2]=J[h+128>>2];J[f+8>>2]=J[h+132>>2];J[h+132>>2]=0;J[h+124>>2]=0;J[h+128>>2]=0;i=i+l|0;m=f+12|0;q:{if((g|0)==(j|0)){J[h+28>>2]=i;J[h+24>>2]=m;J[h+20>>2]=f;break q}while(1){f=f-12|0;j=j-12|0;J[f>>2]=J[j>>2];J[f+4>>2]=J[j+4>>2];J[f+8>>2]=J[j+8>>2];J[j+8>>2]=0;J[j>>2]=0;J[j+4>>2]=0;if((g|0)!=(j|0)){continue}break}J[h+28>>2]=i;g=J[h+24>>2];J[h+24>>2]=m;j=J[h+20>>2];J[h+20>>2]=f;if((g|0)==(j|0)){break q}while(1){l=g-12|0;m=J[l>>2];if(m){o=g-8|0;f=J[o>>2];i=m;if((f|0)!=(i|0)){while(1){f=f-32|0;g=J[f+4>>2];r:{if(!g){break r}i=J[g+4>>2];J[g+4>>2]=i-1;if(i){break r}Xa[J[J[g>>2]+8>>2]](g);bb(g);}if((f|0)!=(m|0)){continue}break}i=J[l>>2];}J[o>>2]=m;$a(i);}g=l;if((j|0)!=(g|0)){continue}break}}if(j){$a(j);}i=J[h+124>>2];if(!i){break n}f=J[h+128>>2];g=i;if((f|0)!=(g|0)){while(1){f=f-32|0;g=J[f+4>>2];s:{if(!g){break s}j=J[g+4>>2];J[g+4>>2]=j-1;if(j){break s}Xa[J[J[g>>2]+8>>2]](g);bb(g);}if((f|0)!=(i|0)){continue}break}g=J[h+124>>2];}J[h+128>>2]=i;$a(g);}j=J[h+24>>2];f=j-12|0;r=P(J[f+4>>2]-J[f>>2]>>5,r);k=k+1|0;g=J[d>>2];if(k>>>0>=J[d+4>>2]-g>>2>>>0){break k}continue}break}f=Z()|0;break h}f=Z()|0;_b(h+124|0);break h}f=Z()|0;break h}if(!r){break j}}m=0;t:{u:{v:{w:{while(1){j=J[h+32>>2];if(J[d+4>>2]!=J[d>>2]){f=0;i=J[h+20>>2];g=m;while(1){k=P(f,12);l=k+i|0;i=J[l>>2];J[(f<<2)+j>>2]=i+((g>>>0)%(J[l+4>>2]-i>>5>>>0)<<5);l=g;i=J[h+20>>2];g=k+i|0;g=(l>>>0)/(J[g+4>>2]-J[g>>2]>>5>>>0)|0;f=f+1|0;if(f>>>0<J[d+4>>2]-J[d>>2]>>2>>>0){continue}break}}J[h+108>>2]=0;J[h+100>>2]=0;J[h+104>>2]=0;f=j;n=J[h+36>>2];if((f|0)!=(n|0)){x:{while(1){y:{k=J[j>>2];f=J[h+104>>2];z:{if((f|0)!=J[h+108>>2]){J[f>>2]=J[k>>2];g=J[k+4>>2];J[f+4>>2]=g;if(g){J[g+4>>2]=J[g+4>>2]+1;}J[h+104>>2]=f+8;break z}A:{i=J[h+100>>2];g=f-i|0;o=g>>3;l=o+1|0;B:{if(l>>>0>=536870912){J[5996]=0;ca(59);break B}p=g>>2;g=g>>>0>=2147483640?536870911:l>>>0<p>>>0?p:l;if(g>>>0<536870912){break A}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break a}f=Z()|0;break t}J[5996]=0;p=g<<3;l=$(3,p|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break y}g=l+(o<<3)|0;J[g>>2]=J[k>>2];k=J[k+4>>2];J[g+4>>2]=k;if(k){J[k+4>>2]=J[k+4>>2]+1;f=J[h+104>>2];}k=l+p|0;l=g+8|0;C:{if((f|0)==(i|0)){J[h+108>>2]=k;J[h+104>>2]=l;J[h+100>>2]=g;break C}while(1){g=g-8|0;f=f-8|0;J[g>>2]=J[f>>2];J[g+4>>2]=J[f+4>>2];J[f>>2]=0;J[f+4>>2]=0;if((f|0)!=(i|0)){continue}break}J[h+108>>2]=k;f=J[h+104>>2];J[h+104>>2]=l;i=J[h+100>>2];J[h+100>>2]=g;if((f|0)==(i|0)){break C}while(1){f=f-8|0;g=J[f+4>>2];D:{if(!g){break D}k=J[g+4>>2];J[g+4>>2]=k-1;if(k){break D}Xa[J[J[g>>2]+8>>2]](g);bb(g);}if((f|0)!=(i|0)){continue}break}}if(!i){break z}$a(i);}j=j+4|0;if((n|0)!=(j|0)){continue}break x}break}f=Z()|0;break t}j=J[h+36>>2];f=J[h+32>>2];}J[5996]=0;va(63,h+80|0,c|0,f|0,j|0,0);f=J[5996];J[5996]=0;if((f|0)==1){break u}J[5996]=0;j=$(3,104)|0;f=J[5996];J[5996]=0;if((f|0)==1){break u}J[j>>2]=7e3;J[j+4>>2]=0;J[j+8>>2]=0;p=J[h+100>>2];J[h+112>>2]=p;q=J[h+104>>2];J[h+116>>2]=q;l=J[h+108>>2];f=0;J[h+108>>2]=0;J[h+120>>2]=l;J[h+100>>2]=0;J[h+104>>2]=0;g=J[e>>2];n=J[e+4>>2];E:{if((g|0)==(n|0)){k=0;i=0;break E}o=n-g|0;if((o|0)<0){J[5996]=0;ca(8);a=J[5996];J[5996]=0;if((a|0)!=1){break a}f=Z()|0;break v}J[5996]=0;k=$(3,o|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){break w}t=n+(g^-1)|0;i=0;f=k;u=o&7;if(u){while(1){H[f|0]=K[g|0];f=f+1|0;g=g+1|0;i=i+1|0;if((u|0)!=(i|0)){continue}break}}i=k+o|0;if(t>>>0<7){break E}while(1){H[f|0]=K[g|0];H[f+1|0]=K[g+1|0];H[f+2|0]=K[g+2|0];H[f+3|0]=K[g+3|0];H[f+4|0]=K[g+4|0];H[f+5|0]=K[g+5|0];H[f+6|0]=K[g+6|0];H[f+7|0]=K[g+7|0];f=f+8|0;g=g+8|0;if((n|0)!=(g|0)){continue}break}l=J[h+120>>2];q=J[h+116>>2];p=J[h+112>>2];}J[j+12>>2]=c;J[j+16>>2]=0;J[j+20>>2]=0;J[j+24>>2]=0;J[j+28>>2]=0;J[j+52>>2]=l;J[j+48>>2]=q;o=j+44|0;J[o>>2]=p;J[j+40>>2]=i;J[j+36>>2]=f;J[j+32>>2]=k;J[h+120>>2]=0;J[h+112>>2]=0;J[h+116>>2]=0;J[5996]=0;i=j+12|0;aa(86,j+56|0,i|0);f=J[5996];J[5996]=0;F:{if((f|0)==1){break F}J[5996]=0;aa(87,j+76|0,i|0);f=J[5996];J[5996]=0;if((f|0)==1){break F}J[5996]=0;f=$(88,i|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break F}J[j+92>>2]=f;g=0;l=0;k=J[j+44>>2];f=k;n=J[j+48>>2];if((f|0)!=(n|0)){while(1){g=J[J[f>>2]+84>>2]+g|0;f=f+8|0;if((n|0)!=(f|0)){continue}break}l=J[J[k>>2]+80>>2];}f=J[j+16>>2];p=J[j+12>>2];q=J[j+20>>2];t=J[j+24>>2];J[5996]=0;f=ka(89,p|0,l|0,g|0,f|0,n-k>>3,(t-q|0)/12|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break F}J[j+96>>2]=f;J[5996]=0;J[h+140>>2]=i;ea(90,h+124|0,i|0,h+140|0);f=J[5996];J[5996]=0;if((f|0)==1){break F}f=K[h+136|0];if(f){Db(J[h+128>>2]);}H[j+100|0]=f^1;J[h+128>>2]=j;J[h+124>>2]=i;J[5996]=0;f=J[h+128>>2];J[h+8>>2]=J[h+124>>2];J[h+12>>2]=f;ea(61,a|0,h+80|0,h+8|0);f=J[5996];J[5996]=0;if((f|0)==1){break u}g=J[h+100>>2];if(g){f=J[h+104>>2];if((g|0)!=(f|0)){while(1){f=f-8|0;i=J[f+4>>2];G:{if(!i){break G}j=J[i+4>>2];J[i+4>>2]=j-1;if(j){break G}Xa[J[J[i>>2]+8>>2]](i);bb(i);}if((f|0)!=(g|0)){continue}break}g=J[h+100>>2];}$a(g);}m=m+1|0;if((r|0)!=(m|0)){continue}break j}break}f=Z()|0;lb(o);a=J[j+32>>2];if(a){J[j+36>>2]=a;$a(a);}rb(j+20|0);break v}f=Z()|0;}lb(h+112|0);$a(j);break t}f=Z()|0;}lb(h+100|0);break h}i=J[h+20>>2];if(i){f=J[h+24>>2];if((i|0)!=(f|0)){while(1){k=f-12|0;j=J[k>>2];if(j){m=f-8|0;f=J[m>>2];g=j;if((f|0)!=(g|0)){while(1){f=f-32|0;g=J[f+4>>2];H:{if(!g){break H}l=J[g+4>>2];J[g+4>>2]=l-1;if(l){break H}Xa[J[J[g>>2]+8>>2]](g);bb(g);}if((f|0)!=(j|0)){continue}break}g=J[k>>2];}J[m>>2]=j;$a(g);}f=k;if((i|0)!=(f|0)){continue}break}i=J[h+20>>2];}$a(i);}s=s+1|0;f=J[h+44>>2];if(s>>>0<(J[h+48>>2]-f|0)/12>>>0){continue}break b}break}J[5996]=0;ga(44,5170,1900,642,2497);a=J[5996];J[5996]=0;if((a|0)!=1){break a}f=Z()|0;}Xc(h+20|0);break g}f=Z()|0;}a=J[h+32>>2];if(a){$a(a);}Wc(h+44|0);break c}f=Z()|0;}Mc(h+56|0);da(f|0);B();}a=J[h+32>>2];if(a){$a(a);f=J[h+44>>2];}if(f){g=f;i=J[h+48>>2];if((i|0)!=(f|0)){while(1){a=i-12|0;b=J[a>>2];if(b){J[i-8>>2]=b;$a(b);}i=a;if((i|0)!=(f|0)){continue}break}g=J[h+44>>2];}J[h+48>>2]=f;$a(g);}a=J[h+68>>2];if(a){J[h+72>>2]=a;$a(a);}a=J[h+56>>2];if(a){J[h+60>>2]=a;$a(a);}Ua=h+144|0;return}B();}function Wm(a,b,c,d,e,f,g,h,i){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;i=i|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0;j=Ua-208|0;Ua=j;a:{b:{c:{d:{e:{f:{g:{h:{i:{j:{k:{l:{m:{n:{o:{p:{q:{r:{s:{t:{u:{v:{w:{x:{y:{z:{k=Gb(a);if(k>>>0>=2147483632){J[5996]=0;ca(2);a=J[5996];J[5996]=0;if((a|0)==1){break z}break a}A:{if(k>>>0<=10){H[j+171|0]=k;n=j+160|0;break A}J[5996]=0;m=(k|15)+1|0;n=$(3,m|0)|0;l=J[5996];J[5996]=0;if((l|0)==1){break z}J[j+160>>2]=n;J[j+164>>2]=k;J[j+168>>2]=m|-2147483648;}H[fb(n,a,k)+k|0]=0;l=J[j+160>>2];a=K[j+171|0];p=a<<24>>24;k=(p|0)<0;o=k?l:j+160|0;r=k?J[j+164>>2]:a;k=o+r|0;n=-1;B:{while(1){if((k|0)==(o|0)){break B}k=k-1|0;m=K[k|0];s=m>>>0<=63;a=m&31;if((m&63)>>>0>=32){m=1<<a;a=0;}else {q=1<<a;m=q-1&1>>>32-a;a=q;}a=a&9728;if(s&(m&1|(a|0)!=0)){continue}break}n=k-o|0;}a=n+1|0;if(a>>>0>r>>>0){J[5996]=0;ca(108);a=J[5996];J[5996]=0;if((a|0)!=1){break a}k=fa(22864)|0;a=Va;break l}C:{if((p|0)<0){J[j+164>>2]=a;break C}H[j+171|0]=a;l=j+160|0;}H[a+l|0]=0;J[j+152>>2]=0;J[j+156>>2]=0;D:{if(H[j+171|0]>=0){J[j+144>>2]=J[j+168>>2];a=J[j+164>>2];J[j+136>>2]=J[j+160>>2];J[j+140>>2]=a;break D}J[5996]=0;ea(40,j+136|0,J[j+160>>2],J[j+164>>2]);a=J[5996];J[5996]=0;if((a|0)==1){break y}}J[5996]=0;a=j+172|0;aa(109,a|0,j+136|0);k=J[5996];J[5996]=0;if((k|0)==1){break x}m=a;a=H[j+183|0];k=(a|0)<0;J[j+104>>2]=k?J[j+172>>2]:m;J[j+108>>2]=k?J[j+176>>2]:a&255;J[5996]=0;a=J[j+108>>2];J[j+8>>2]=J[j+104>>2];J[j+12>>2]=a;aa(140,j+16|0,j+8|0);a=J[5996];J[5996]=0;if((a|0)==1){break w}a=J[j+20>>2];J[j+156>>2]=a;k=J[j+16>>2];J[j+152>>2]=k;if(H[j+183|0]<0){$a(J[j+172>>2]);}if(H[j+147|0]<0){$a(J[j+136>>2]);}E:{if(!k){break E}n=J[k+80>>2];if(!n|M[k+84>>2]>3600){break E}J[5996]=0;m=_(54,3557,1)|0;l=J[5996];J[5996]=0;if((l|0)==1){break u}if(!((n^-1)&m)){break r}}J[5996]=0;f=$(3,32)|0;g=J[5996];J[5996]=0;if((g|0)==1){break t}H[f+16|0]=0;g=K[2571]|K[2572]<<8|(K[2573]<<16|K[2574]<<24);h=K[2567]|K[2568]<<8|(K[2569]<<16|K[2570]<<24);H[f+8|0]=h;H[f+9|0]=h>>>8;H[f+10|0]=h>>>16;H[f+11|0]=h>>>24;H[f+12|0]=g;H[f+13|0]=g>>>8;H[f+14|0]=g>>>16;H[f+15|0]=g>>>24;g=K[2563]|K[2564]<<8|(K[2565]<<16|K[2566]<<24);h=K[2559]|K[2560]<<8|(K[2561]<<16|K[2562]<<24);H[f|0]=h;H[f+1|0]=h>>>8;H[f+2|0]=h>>>16;H[f+3|0]=h>>>24;H[f+4|0]=g;H[f+5|0]=g>>>8;H[f+6|0]=g>>>16;H[f+7|0]=g>>>24;g=((c|0)>=17?17:c)-1|0;H[fb(b,f,g)+g|0]=0;$a(f);J[5996]=0;f=$(3,32)|0;g=J[5996];J[5996]=0;if((g|0)==1){break s}H[f+16|0]=0;c=K[2571]|K[2572]<<8|(K[2573]<<16|K[2574]<<24);b=f;f=K[2567]|K[2568]<<8|(K[2569]<<16|K[2570]<<24);H[b+8|0]=f;H[b+9|0]=f>>>8;H[b+10|0]=f>>>16;H[b+11|0]=f>>>24;H[b+12|0]=c;H[b+13|0]=c>>>8;H[b+14|0]=c>>>16;H[b+15|0]=c>>>24;c=K[2563]|K[2564]<<8|(K[2565]<<16|K[2566]<<24);f=K[2559]|K[2560]<<8|(K[2561]<<16|K[2562]<<24);H[b|0]=f;H[b+1|0]=f>>>8;H[b+2|0]=f>>>16;H[b+3|0]=f>>>24;H[b+4|0]=c;H[b+5|0]=c>>>8;H[b+6|0]=c>>>16;H[b+7|0]=c>>>24;c=((e|0)>=17?17:e)-1|0;H[fb(d,b,c)+c|0]=0;$a(b);break q}k=fa(22864)|0;a=Va;break k}k=fa(22864)|0;a=Va;break m}k=fa(22864)|0;a=Va;break v}k=fa(22864)|0;a=Va;if(H[j+183|0]>=0){break v}$a(J[j+172>>2]);}if(H[j+147|0]>=0){break m}$a(J[j+136>>2]);break m}k=fa(22864)|0;a=Va;break m}k=fa(22864)|0;a=Va;break m}k=fa(22864)|0;a=Va;break m}a=J[k+84>>2];J[5996]=0;n=j+60|0;aa(115,n|0,a|0);a=J[5996];J[5996]=0;F:{G:{H:{I:{J:{K:{L:{M:{N:{O:{P:{Q:{R:{if((a|0)!=1){J[5996]=0;a=ba(99,n|0,0,5546)|0;n=J[5996];J[5996]=0;if((n|0)==1){break R}J[j+80>>2]=J[a+8>>2];n=J[a+4>>2];J[j+72>>2]=J[a>>2];J[j+76>>2]=n;J[a>>2]=0;J[a+4>>2]=0;J[a+8>>2]=0;J[5996]=0;n=_(97,j+72|0,3581)|0;m=J[5996];J[5996]=0;if((m|0)==1){break Q}J[j+96>>2]=J[n+8>>2];m=J[n+4>>2];J[j+88>>2]=J[n>>2];J[j+92>>2]=m;J[n>>2]=0;J[n+4>>2]=0;J[n+8>>2]=0;J[5996]=0;m=j+172|0;aa(141,m|0,j+152|0);l=J[5996];J[5996]=0;if((l|0)==1){break P}J[5996]=0;q=m;m=K[j+183|0];l=m<<24>>24<0;m=ba(100,j+88|0,(l?J[j+172>>2]:q)|0,(l?J[j+176>>2]:m)|0)|0;l=J[5996];J[5996]=0;if((l|0)==1){break O}J[j+112>>2]=J[m+8>>2];l=J[m+4>>2];J[j+104>>2]=J[m>>2];J[j+108>>2]=l;J[m>>2]=0;J[m+4>>2]=0;J[m+8>>2]=0;J[5996]=0;l=_(97,j+104|0,3570)|0;o=J[5996];J[5996]=0;if((o|0)==1){break N}J[j+128>>2]=J[l+8>>2];o=J[l+4>>2];J[j+120>>2]=J[l>>2];J[j+124>>2]=o;J[l>>2]=0;J[l+4>>2]=0;J[l+8>>2]=0;if(H[j+183|0]<0){$a(J[j+172>>2]);}if(H[m+11|0]<0){$a(J[m>>2]);}if(H[n+11|0]<0){$a(J[n>>2]);}if(H[a+11|0]<0){$a(J[a>>2]);}n=J[j+120>>2];a=K[j+131|0];m=a<<24>>24;l=(m|0)<0;q=l?n:j+120|0;a=l?J[j+124>>2]:a;l=c-1|0;a=(a|0)<(l|0)?a:l;H[fb(b,q,a)+a|0]=0;J[5996]=0;l=j+172|0;aa(117,l|0,k|0);a=J[5996];J[5996]=0;if((a|0)==1){break I}J[j+44>>2]=J[j+200>>2];a=J[j+196>>2];J[j+36>>2]=J[j+192>>2];J[j+40>>2]=a;a=J[j+188>>2];J[j+28>>2]=J[j+184>>2];J[j+32>>2]=a;a=J[j+180>>2];J[j+20>>2]=J[j+176>>2];J[j+24>>2]=a;a=J[j+172>>2];J[j+16>>2]=a;J[5996]=0;o=a>>>0<29;p=o?j+20|0:J[j+24>>2];J[j+172>>2]=p;r=j+48|0;ga(102,r|0,l|0,p+(o?a:a-29|0)|0,0);a=J[5996];J[5996]=0;if((a|0)==1){break H}a=J[j+48>>2];l=K[j+59|0];o=l<<24>>24;p=(o|0)<0;q=p?a:r;l=p?J[j+52>>2]:l;p=e-1|0;l=(l|0)<(p|0)?l:p;H[fb(d,q,l)+l|0]=0;if((o|0)<0){$a(a);}if(M[j+16>>2]>=29){$a(J[j+24>>2]);k=J[j+152>>2];}a=J[k+80>>2];if(!a|M[k+84>>2]>3600|(J[k+52>>2]+J[k+44>>2]>>>0>201|M[k+68>>2]>100)){break G}J[5996]=0;l=_(54,2049,1)|0;o=J[5996];J[5996]=0;if((o|0)==1){break p}if((a^-1)&l){break G}a=J[k+80>>2];J[5996]=0;l=_(54,2165,1)|0;o=J[5996];J[5996]=0;if((o|0)==1){break p}if(K[k+88|0]|(a^-1)&l){break G}H[j+176|0]=0;J[j+172>>2]=1702195828;H[j+183|0]=4;a=((i|0)>=5?5:i)-1|0;H[fb(h,j+172|0,a)+a|0]=0;break F}k=fa(22864)|0;a=Va;break m}k=fa(22864)|0;a=Va;break J}k=fa(22864)|0;a=Va;break K}k=fa(22864)|0;a=Va;break L}k=fa(22864)|0;a=Va;break M}k=fa(22864)|0;a=Va;if(H[j+115|0]>=0){break M}$a(J[j+104>>2]);}if(H[j+183|0]>=0){break L}$a(J[j+172>>2]);}if(H[j+99|0]>=0){break K}$a(J[j+88>>2]);}if(H[j+83|0]>=0){break J}$a(J[j+72>>2]);}if(H[j+71|0]>=0){break m}$a(J[j+60>>2]);break m}k=fa(22864)|0;break o}k=fa(22864)|0;a=Va;if(M[j+16>>2]<29){break n}$a(J[j+24>>2]);break n}H[j+176|0]=K[2354];H[j+183|0]=5;H[j+177|0]=0;J[j+172>>2]=K[2350]|K[2351]<<8|(K[2352]<<16|K[2353]<<24);a=((i|0)>=6?6:i)-1|0;H[fb(h,j+172|0,a)+a|0]=0;}J[5996]=0;a=$(118,k|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break p}S:{if(a){H[j+176|0]=0;J[j+172>>2]=1702195828;H[j+183|0]=4;a=((g|0)>=5?5:g)-1|0;H[fb(f,j+172|0,a)+a|0]=0;break S}H[j+176|0]=K[2354];H[j+183|0]=5;H[j+177|0]=0;J[j+172>>2]=K[2350]|K[2351]<<8|(K[2352]<<16|K[2353]<<24);a=((g|0)>=6?6:g)-1|0;H[fb(f,j+172|0,a)+a|0]=0;}if((m|0)<0){$a(n);}a=J[j+156>>2];}T:{if(!a){break T}b=J[a+4>>2];J[a+4>>2]=b-1;if(b){break T}Xa[J[J[a>>2]+8>>2]](a);bb(a);}if(H[j+171|0]>=0){break j}$a(J[j+160>>2]);break j}k=fa(22864)|0;}a=Va;}if((m|0)>=0){break m}$a(n);}mb(j+152|0);}if(H[j+171|0]>=0){break k}$a(J[j+160>>2]);}if((wa(22864)|0)!=(a|0)){break c}a=pa(k|0)|0;a=Xa[J[J[a>>2]+8>>2]](a)|0;J[5996]=0;g=_(30,j+120|0,a|0)|0;a=J[5996];J[5996]=0;if((a|0)==1){break i}J[5996]=0;a=ba(99,g|0,0,5474)|0;f=J[5996];J[5996]=0;if((f|0)==1){break h}J[j+168>>2]=J[a+8>>2];f=J[a+4>>2];J[j+160>>2]=J[a>>2];J[j+164>>2]=f;J[a>>2]=0;J[a+4>>2]=0;J[a+8>>2]=0;J[5996]=0;f=_(97,j+160|0,2611)|0;h=J[5996];J[5996]=0;if((h|0)==1){break g}J[j+24>>2]=J[f+8>>2];g=J[f+4>>2];J[j+16>>2]=J[f>>2];J[j+20>>2]=g;J[f>>2]=0;J[f+4>>2]=0;J[f+8>>2]=0;k=b;b=J[j+16>>2];g=K[j+27|0];h=g<<24>>24;i=(h|0)<0;g=i?J[j+20>>2]:g;c=c-1|0;c=(c|0)>(g|0)?g:c;H[fb(k,i?b:j+16|0,c)+c|0]=0;U:{if((h|0)>=0){break U}$a(b);if(H[f+11|0]>=0){break U}$a(J[f>>2]);}if(H[a+11|0]<0){$a(J[a>>2]);}J[5996]=0;a=_(30,j+16|0,5575)|0;b=J[5996];J[5996]=0;if((b|0)==1){break e}g=d;b=J[a>>2];c=K[a+11|0];d=c<<24>>24;f=(d|0)<0;h=f?b:a;a=f?J[a+4>>2]:c;c=e-1|0;a=(a|0)<(c|0)?a:c;H[fb(g,h,a)+a|0]=0;if((d|0)<0){$a(b);}ra();}Ua=j+208|0;return}k=Z()|0;break d}k=Z()|0;break f}k=Z()|0;if(H[j+171|0]>=0){break f}$a(J[j+160>>2]);}if(H[g+11|0]>=0){break d}$a(J[g>>2]);break d}k=Z()|0;}J[5996]=0;ca(110);a=J[5996];J[5996]=0;if((a|0)==1){break b}}da(k|0);B();}fa(0)|0;Vb();B();}B();}function Xg(a,b,c,d,e){var f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0;h=Ua-160|0;Ua=h;Yc(h+48|0,c,O[a+16>>3],O[a+24>>3],0,0);J[5996]=0;g=$(46,c|0)|0;f=J[5996];J[5996]=0;a:{b:{c:{if((f|0)!=1){J[h+44>>2]=0;J[h+36>>2]=0;J[h+40>>2]=0;H[h+140|0]=0;J[h+136>>2]=h+36;f=J[g+4>>2];j=J[g>>2];m=f-j|0;i=(m|0)/12|0;if((f|0)!=(j|0)){d:{e:{f:{if(i>>>0>=357913942){J[5996]=0;ca(47);a=J[5996];J[5996]=0;if((a|0)==1){break f}break a}J[5996]=0;f=$(3,m|0)|0;j=J[5996];J[5996]=0;if((j|0)==1){break f}J[h+40>>2]=f;J[h+36>>2]=f;J[h+44>>2]=f+P(i,12);i=J[g+4>>2];g=J[g>>2];J[5996]=0;g=ia(48,h+44|0,g|0,i|0,f|0)|0;i=J[5996];J[5996]=0;if((i|0)!=1){break d}g=Z()|0;J[h+40>>2]=f;break e}g=Z()|0;}Nc(h+136|0);break c}J[h+40>>2]=g;}J[h+32>>2]=0;J[h+24>>2]=0;J[h+28>>2]=0;f=J[d+4>>2];g=J[d>>2];J[5996]=0;aa(49,h+24|0,f-g>>2);f=J[5996];J[5996]=0;g:{if((f|0)!=1){g=J[h+40>>2];f=J[h+36>>2];if((g|0)==(f|0)){break b}s=h+112|0;g=f;h:{while(1){i:{J[h+20>>2]=0;J[h+12>>2]=0;J[h+16>>2]=0;i=J[d+4>>2];f=J[d>>2];n=P(r,12);g=n+g|0;if(i-f>>2!=J[g+4>>2]-J[g>>2]>>3){break i}j=0;o=1;l=0;j:{if((f|0)!=(i|0)){k:{l:{m:{while(1){g=l<<3;t=O[g+J[h+60>>2]>>3];u=O[g+J[h+48>>2]>>3];f=J[(l<<2)+f>>2];J[5996]=0;f=qa(50,f|0,+u,+t,b|0)|0;i=J[5996];J[5996]=0;if((i|0)==1){break l}g=g+J[n+J[h+36>>2]>>2]|0;i=J[g>>2];g=J[g+4>>2];m=J[f+4>>2];f=J[f>>2];J[5996]=0;oa(51,h+136|0,f|0,m|0,i|0,g|0);f=J[5996];J[5996]=0;if((f|0)!=1){i=J[h+20>>2];n:{if(i>>>0>j>>>0){J[j+8>>2]=0;J[j>>2]=0;J[j+4>>2]=0;J[j>>2]=J[h+136>>2];J[j+4>>2]=J[h+140>>2];J[j+8>>2]=J[h+144>>2];J[h+16>>2]=j+12;break n}o:{f=J[h+12>>2];m=(j-f|0)/12|0;g=m+1|0;p:{if(g>>>0>=357913942){J[5996]=0;ca(52);break p}i=(i-f|0)/12|0;k=i<<1;g=i>>>0>=178956970?357913941:g>>>0<k>>>0?k:g;if(g>>>0<357913942){break o}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break a}break m}J[5996]=0;k=P(g,12);i=$(3,k|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break m}g=i+P(m,12)|0;J[g>>2]=J[h+136>>2];J[g+4>>2]=J[h+140>>2];J[g+8>>2]=J[h+144>>2];J[h+144>>2]=0;J[h+136>>2]=0;J[h+140>>2]=0;i=i+k|0;m=g+12|0;q:{if((f|0)==(j|0)){J[h+20>>2]=i;J[h+16>>2]=m;J[h+12>>2]=g;break q}while(1){g=g-12|0;j=j-12|0;J[g>>2]=J[j>>2];J[g+4>>2]=J[j+4>>2];J[g+8>>2]=J[j+8>>2];J[j+8>>2]=0;J[j>>2]=0;J[j+4>>2]=0;if((f|0)!=(j|0)){continue}break}J[h+20>>2]=i;f=J[h+16>>2];J[h+16>>2]=m;j=J[h+12>>2];J[h+12>>2]=g;if((f|0)==(j|0)){break q}while(1){k=f-12|0;m=J[k>>2];if(m){p=f-8|0;g=J[p>>2];i=m;if((g|0)!=(i|0)){while(1){g=g-32|0;f=J[g+4>>2];r:{if(!f){break r}i=J[f+4>>2];J[f+4>>2]=i-1;if(i){break r}Xa[J[J[f>>2]+8>>2]](f);bb(f);}if((g|0)!=(m|0)){continue}break}i=J[k>>2];}J[p>>2]=m;$a(i);}f=k;if((j|0)!=(f|0)){continue}break}}if(j){$a(j);}i=J[h+136>>2];if(!i){break n}f=i;g=J[h+140>>2];if((f|0)!=(g|0)){while(1){g=g-32|0;f=J[g+4>>2];s:{if(!f){break s}j=J[f+4>>2];J[f+4>>2]=j-1;if(j){break s}Xa[J[J[f>>2]+8>>2]](f);bb(f);}if((g|0)!=(i|0)){continue}break}f=J[h+136>>2];}J[h+140>>2]=i;$a(f);}j=J[h+16>>2];f=j-12|0;o=P(J[f+4>>2]-J[f>>2]>>5,o);l=l+1|0;f=J[d>>2];if(l>>>0>=J[d+4>>2]-f>>2>>>0){break k}continue}break}g=Z()|0;break h}g=Z()|0;_b(h+136|0);break h}g=Z()|0;break h}if(!o){break j}}m=0;t:{u:{v:{w:{x:{y:{while(1){j=J[h+24>>2];if(J[d+4>>2]!=J[d>>2]){g=0;i=J[h+12>>2];f=m;while(1){k=P(g,12);l=k+i|0;i=J[l>>2];J[(g<<2)+j>>2]=i+((f>>>0)%(J[l+4>>2]-i>>5>>>0)<<5);l=f;i=J[h+12>>2];f=k+i|0;f=(l>>>0)/(J[f+4>>2]-J[f>>2]>>5>>>0)|0;g=g+1|0;if(g>>>0<J[d+4>>2]-J[d>>2]>>2>>>0){continue}break}}J[h+100>>2]=0;J[h+92>>2]=0;J[h+96>>2]=0;g=j;n=J[h+28>>2];if((g|0)!=(n|0)){z:{while(1){A:{k=J[j>>2];g=J[h+96>>2];B:{if((g|0)!=J[h+100>>2]){J[g>>2]=J[k>>2];f=J[k+4>>2];J[g+4>>2]=f;if(f){J[f+4>>2]=J[f+4>>2]+1;}J[h+96>>2]=g+8;break B}C:{i=J[h+92>>2];f=g-i|0;p=f>>3;l=p+1|0;D:{if(l>>>0>=536870912){J[5996]=0;ca(59);break D}q=f>>2;f=f>>>0>=2147483640?536870911:l>>>0<q>>>0?q:l;if(f>>>0<536870912){break C}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break a}g=Z()|0;break t}J[5996]=0;q=f<<3;l=$(3,q|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){break A}f=l+(p<<3)|0;J[f>>2]=J[k>>2];k=J[k+4>>2];J[f+4>>2]=k;if(k){J[k+4>>2]=J[k+4>>2]+1;g=J[h+96>>2];}k=l+q|0;l=f+8|0;E:{if((g|0)==(i|0)){J[h+100>>2]=k;J[h+96>>2]=l;J[h+92>>2]=f;break E}while(1){f=f-8|0;g=g-8|0;J[f>>2]=J[g>>2];J[f+4>>2]=J[g+4>>2];J[g>>2]=0;J[g+4>>2]=0;if((g|0)!=(i|0)){continue}break}J[h+100>>2]=k;g=J[h+96>>2];J[h+96>>2]=l;i=J[h+92>>2];J[h+92>>2]=f;if((g|0)==(i|0)){break E}while(1){g=g-8|0;f=J[g+4>>2];F:{if(!f){break F}k=J[f+4>>2];J[f+4>>2]=k-1;if(k){break F}Xa[J[J[f>>2]+8>>2]](f);bb(f);}if((g|0)!=(i|0)){continue}break}}if(!i){break B}$a(i);}j=j+4|0;if((n|0)!=(j|0)){continue}break z}break}g=Z()|0;break t}j=J[h+28>>2];g=J[h+24>>2];}J[5996]=0;va(63,h+72|0,c|0,g|0,j|0,0);f=J[5996];J[5996]=0;if((f|0)==1){break x}J[5996]=0;j=$(3,104)|0;f=J[5996];J[5996]=0;if((f|0)==1){break x}J[j>>2]=7e3;J[j+4>>2]=0;J[j+8>>2]=0;J[h+112>>2]=0;J[h+104>>2]=0;J[h+108>>2]=0;J[h+116>>2]=J[h+92>>2];f=J[h+100>>2];g=J[h+96>>2];J[h+100>>2]=0;J[h+120>>2]=g;J[h+124>>2]=f;J[h+92>>2]=0;J[h+96>>2]=0;H[h+132|0]=0;J[h+128>>2]=h+104;k=J[e+4>>2];g=J[e>>2];i=k-g|0;f=(i|0)/12|0;G:{if((g|0)!=(k|0)){if(f>>>0>=357913942){J[5996]=0;ca(41);a=J[5996];J[5996]=0;if((a|0)!=1){break a}g=Z()|0;break v}J[5996]=0;i=$(3,i|0)|0;l=J[5996];J[5996]=0;if((l|0)==1){break G}J[h+108>>2]=i;J[h+104>>2]=i;J[h+112>>2]=i+P(f,12);J[h+156>>2]=i;J[h+152>>2]=i;H[h+148|0]=0;J[h+136>>2]=s;J[h+144>>2]=h+156;J[h+140>>2]=h+152;f=i;while(1){H:{if(H[g+11|0]>=0){l=J[g+4>>2];J[f>>2]=J[g>>2];J[f+4>>2]=l;J[f+8>>2]=J[g+8>>2];break H}l=J[g+4>>2];n=J[g>>2];J[5996]=0;ea(40,f|0,n|0,l|0);f=J[5996];J[5996]=0;if((f|0)==1){break w}f=J[h+156>>2];}f=f+12|0;J[h+156>>2]=f;g=g+12|0;if((k|0)!=(g|0)){continue}break}J[h+108>>2]=f;}J[5996]=0;k=xa(60,j+12|0,c|0,h+116|0,h+104|0,0)|0;f=J[5996];J[5996]=0;if((f|0)==1){break y}i=J[h+104>>2];if(i){f=J[h+108>>2];g=i;if((f|0)!=(g|0)){while(1){g=f-12|0;if(H[f-1|0]<0){$a(J[g>>2]);}f=g;if((i|0)!=(f|0)){continue}break}g=J[h+104>>2];}J[h+108>>2]=i;$a(g);}f=J[h+116>>2];if(f){g=J[h+120>>2];if((f|0)!=(g|0)){while(1){g=g-8|0;i=J[g+4>>2];I:{if(!i){break I}l=J[i+4>>2];J[i+4>>2]=l-1;if(l){break I}Xa[J[J[i>>2]+8>>2]](i);bb(i);}if((f|0)!=(g|0)){continue}break}f=J[h+116>>2];}$a(f);}J[h+140>>2]=j;J[h+136>>2]=k;J[5996]=0;f=J[h+140>>2];J[h>>2]=J[h+136>>2];J[h+4>>2]=f;ea(61,a|0,h+72|0,h|0);f=J[5996];J[5996]=0;if((f|0)==1){break x}f=J[h+92>>2];if(f){g=J[h+96>>2];if((f|0)!=(g|0)){while(1){g=g-8|0;i=J[g+4>>2];J:{if(!i){break J}j=J[i+4>>2];J[i+4>>2]=j-1;if(j){break J}Xa[J[J[i>>2]+8>>2]](i);bb(i);}if((f|0)!=(g|0)){continue}break}f=J[h+92>>2];}$a(f);}m=m+1|0;if((o|0)!=(m|0)){continue}break j}break}g=Z()|0;break v}g=Z()|0;rb(h+104|0);break u}g=Z()|0;break t}g=Z()|0;Wg(h+136|0);J[h+108>>2]=i;}Vg(h+128|0);}lb(h+116|0);$a(j);}lb(h+92|0);break h}i=J[h+12>>2];if(i){g=J[h+16>>2];if((i|0)!=(g|0)){while(1){m=g-12|0;j=J[m>>2];if(j){f=j;k=g-8|0;g=J[k>>2];if((f|0)!=(g|0)){while(1){g=g-32|0;f=J[g+4>>2];K:{if(!f){break K}o=J[f+4>>2];J[f+4>>2]=o-1;if(o){break K}Xa[J[J[f>>2]+8>>2]](f);bb(f);}if((g|0)!=(j|0)){continue}break}f=J[m>>2];}J[k>>2]=j;$a(f);}g=m;if((i|0)!=(g|0)){continue}break}i=J[h+12>>2];}$a(i);}r=r+1|0;g=J[h+36>>2];if(r>>>0<(J[h+40>>2]-g|0)/12>>>0){continue}break b}break}J[5996]=0;ga(44,5170,1900,642,2497);a=J[5996];J[5996]=0;if((a|0)!=1){break a}g=Z()|0;}Xc(h+12|0);break g}g=Z()|0;}a=J[h+24>>2];if(a){$a(a);}Wc(h+36|0);break c}g=Z()|0;}Mc(h+48|0);da(g|0);B();}a=J[h+24>>2];if(a){$a(a);g=J[h+36>>2];}if(g){f=g;i=J[h+40>>2];if((f|0)!=(i|0)){while(1){a=i-12|0;b=J[a>>2];if(b){J[i-8>>2]=b;$a(b);}i=a;if((i|0)!=(g|0)){continue}break}f=J[h+36>>2];}J[h+40>>2]=g;$a(f);}a=J[h+60>>2];if(a){J[h- -64>>2]=a;$a(a);}a=J[h+48>>2];if(a){J[h+52>>2]=a;$a(a);}Ua=h+160|0;return}B();}function Ng(a,b,c,d,e,f,g,h,i){var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0,A=0,B=0,C=0,D=0,E=0,F=0,G=0,H=0,I=0,K=0,L=0,N=0,O=0,P=0,Q=0;k=Ua-336|0;Ua=k;m=h;n=i&65535;o=d;p=e&65535;t=(e^i)&-2147483648;r=i>>>16&32767;s=e>>>16&32767;a:{b:{if(r-32767>>>0>4294934529&s-32767>>>0>=4294934530){break b}j=e&2147483647;if(!(!d&(j|0)==2147418112?!(b|c):j>>>0<2147418112)){q=d;t=e|32768;break a}e=i&2147483647;if(!(!h&(e|0)==2147418112?!(f|g):e>>>0<2147418112)){q=h;t=i|32768;b=f;c=g;break a}if(!(b|d|(j^2147418112|c))){if(!(f|h|(e^2147418112|g))){b=0;c=0;t=2147450880;break a}t=t|2147418112;b=0;c=0;break a}if(!(f|h|(e^2147418112|g))){b=0;c=0;break a}if(!(b|d|(c|j))){b=!(f|h|(e|g));q=b?0:q;t=b?2147450880:t;b=0;c=0;break a}if(!(f|h|(e|g))){t=t|2147418112;b=0;c=0;break a}if((j|0)==65535|j>>>0<65535){d=!(o|p);i=d;j=d?b:o;d=d<<6;i=S(i?c:p);d=d+((i|0)==32?S(j)+32|0:i)|0;Cb(k+320|0,b,c,o,p,d-15|0);u=16-d|0;o=J[k+328>>2];p=J[k+332>>2];c=J[k+324>>2];b=J[k+320>>2];}if(e>>>0>65535){break b}d=!(n|m);h=d;i=d?f:m;d=d<<6;h=S(h?g:n);d=d+((h|0)==32?S(i)+32|0:h)|0;Cb(k+304|0,f,g,m,n,d-15|0);u=(d+u|0)-16|0;m=J[k+312>>2];n=J[k+316>>2];f=J[k+304>>2];g=J[k+308>>2];}e=n|65536;z=e;A=m;d=m;j=e<<15|d>>>17;e=d<<15|g>>>17;d=e;i=0-d|0;h=j;j=1963258675-(j+((d|0)!=0)|0)|0;xb(k+288|0,d,h,0,0,i,j,0,0);d=J[k+296>>2];xb(k+272|0,0-d|0,0-(J[k+300>>2]+((d|0)!=0)|0)|0,0,0,i,j,0,0);d=J[k+280>>2];i=d<<1|J[k+276>>2]>>>31;d=J[k+284>>2]<<1|d>>>31;xb(k+256|0,i,d,0,0,e,h,0,0);j=J[k+264>>2];xb(k+240|0,i,d,0,0,0-j|0,0-(J[k+268>>2]+((j|0)!=0)|0)|0,0,0);i=J[k+248>>2];j=i<<1|J[k+244>>2]>>>31;d=J[k+252>>2]<<1|i>>>31;xb(k+224|0,j,d,0,0,e,h,0,0);i=J[k+232>>2];xb(k+208|0,j,d,0,0,0-i|0,0-(J[k+236>>2]+((i|0)!=0)|0)|0,0,0);d=J[k+216>>2];i=d<<1|J[k+212>>2]>>>31;d=J[k+220>>2]<<1|d>>>31;xb(k+192|0,i,d,0,0,e,h,0,0);j=J[k+200>>2];xb(k+176|0,i,d,0,0,0-j|0,0-(J[k+204>>2]+((j|0)!=0)|0)|0,0,0);i=e;e=J[k+184>>2];d=h;m=e<<1|J[k+180>>2]>>>31;h=m-1|0;e=(J[k+188>>2]<<1|e>>>31)-!m|0;xb(k+160|0,i,d,0,0,h,e,0,0);d=h;xb(k+144|0,f<<15,g<<15|f>>>17,0,0,d,e,0,0);w=k+112|0;x=J[k+168>>2];h=J[k+172>>2];m=J[k+160>>2];i=J[k+152>>2];l=m+i|0;n=J[k+164>>2];j=n+J[k+156>>2]|0;j=i>>>0>l>>>0?j+1|0:j;i=j;j=(n|0)==(j|0)&l>>>0<m>>>0|j>>>0<n>>>0;n=j+x|0;j=j>>>0>n>>>0?h+1|0:h;m=!i&l>>>0>1|(i|0)!=0;h=n+m|0;j=m>>>0>h>>>0?j+1|0:j;xb(w,d,e,0,0,0-h|0,0-(((h|0)!=0)+j|0)|0,0,0);xb(k+128|0,1-l|0,0-((l>>>0>1)+i|0)|0,0,0,d,e,0,0);I=(s-r|0)+u|0;e=J[k+116>>2];w=e;d=J[k+112>>2];j=e<<1|d>>>31;i=d<<1;n=j;d=j;h=J[k+140>>2];y=h;e=J[k+136>>2];j=h<<1|e>>>31;l=e<<1|J[k+132>>2]>>>31;h=l+i|0;d=d+j|0;d=h>>>0<l>>>0?d+1|0:d;e=d;d=d-(h>>>0<13927)|0;v=d;x=d;l=0;j=p|65536;K=j;L=o;d=o;j=j<<1|d>>>31;O=d<<1;P=j;E=j;d=co(v,l,j,0);j=Wa;B=d;C=j;u=b<<1;d=c<<1|b>>>31;r=d;j=0;s=j;m=h-13927|0;v=(e|0)==(v|0)&m>>>0<h>>>0|e>>>0>v>>>0;e=(e|0)==(n|0)&h>>>0<i>>>0|e>>>0<n>>>0;d=J[k+120>>2];h=J[k+124>>2]<<1|d>>>31;d=d<<1|w>>>31;l=y>>>31|0;d=l+d|0;j=h;j=d>>>0<l>>>0?j+1|0:j;h=d;d=d+e|0;j=h>>>0>d>>>0?j+1|0:j;e=d;d=d+v|0;l=e>>>0>d>>>0?j+1|0:j;e=d-1|0;D=l-!d|0;y=0;i=co(r,s,D,y);d=i+B|0;h=Wa+C|0;h=d>>>0<i>>>0?h+1|0:h;n=(C|0)==(h|0)&d>>>0<B>>>0|h>>>0<C>>>0;B=e;N=c>>>31|0;G=N|o<<1;v=0;e=co(e,0,G,v);o=e+d|0;j=Wa+h|0;l=0;j=e>>>0>o>>>0?j+1|0:j;p=j;d=(j|0)==(h|0)&d>>>0>o>>>0|h>>>0>j>>>0;e=d;d=d+n|0;l=e>>>0>d>>>0?1:l;e=co(E,s,D,y);d=e+d|0;j=Wa+l|0;n=d;d=d>>>0<e>>>0?j+1|0:j;e=co(E,s,B,v);l=Wa;h=e;e=co(G,v,D,y);i=h+e|0;j=Wa+l|0;j=e>>>0>i>>>0?j+1|0:j;e=j;j=(l|0)==(j|0)&h>>>0>i>>>0|j>>>0<l>>>0;l=n+e|0;d=d+j|0;d=l>>>0<e>>>0?d+1|0:d;n=l;l=d;e=0;d=e+o|0;j=i+p|0;j=d>>>0<e>>>0?j+1|0:j;e=j;h=(j|0)==(p|0)&d>>>0<o>>>0|j>>>0<p>>>0;j=l;i=h;h=h+n|0;j=i>>>0>h>>>0?j+1|0:j;F=h;l=j;w=d;o=d;C=m;d=co(m,0,G,v);i=Wa;h=d;m=co(x,q,r,q);d=d+m|0;j=Wa+i|0;j=d>>>0<m>>>0?j+1|0:j;m=(i|0)==(j|0)&d>>>0<h>>>0|i>>>0>j>>>0;h=j;H=u&-2;i=co(B,v,H,0);p=i+d|0;j=Wa+j|0;j=i>>>0>p>>>0?j+1|0:j;i=j;d=(j|0)==(h|0)&d>>>0>p>>>0|h>>>0>j>>>0;h=0;j=d+m|0;d=(j>>>0<d>>>0?1:h)+e|0;n=j+o|0;d=n>>>0<j>>>0?d+1|0:d;j=l;o=d;d=(d|0)==(e|0)&n>>>0<w>>>0|d>>>0<e>>>0;e=d;d=d+F|0;j=e>>>0>d>>>0?j+1|0:j;Q=d;w=j;d=co(E,s,C,q);F=Wa;E=d;e=co(D,y,H,q);d=d+e|0;j=Wa+F|0;j=d>>>0<e>>>0?j+1|0:j;m=d;l=co(x,q,G,v);e=d+l|0;h=j;d=j+Wa|0;d=e>>>0<l>>>0?d+1|0:d;s=e;j=co(r,q,B,v);e=e+j|0;l=Wa+d|0;l=e>>>0<j>>>0?l+1|0:l;D=(d|0)==(l|0)&e>>>0<s>>>0|d>>>0>l>>>0;j=(h|0)==(F|0)&m>>>0<E>>>0|h>>>0<F>>>0;d=(d|0)==(h|0)&m>>>0>s>>>0|d>>>0<h>>>0;d=d+j|0;d=d+D|0;h=l;s=h+n|0;j=(d|y)+o|0;j=h>>>0>s>>>0?j+1|0:j;m=j;d=(o|0)==(j|0)&n>>>0>s>>>0|j>>>0<o>>>0;j=w;h=d;d=d+Q|0;j=h>>>0>d>>>0?j+1|0:j;y=d;o=j;d=co(x,q,H,q);x=Wa;n=d;h=co(r,q,C,q);d=d+h|0;j=Wa+x|0;j=d>>>0<h>>>0?j+1|0:j;w=0;h=(j|0)==(x|0)&d>>>0<n>>>0|j>>>0<x>>>0;n=j;d=j+p|0;j=(h|w)+i|0;j=d>>>0<n>>>0?j+1|0:j;i=(i|0)==(j|0)&d>>>0<p>>>0|i>>>0>j>>>0;h=j;j=e;p=0;e=p+d|0;l=h+j|0;j=0;l=e>>>0<p>>>0?l+1|0:l;d=(h|0)==(l|0)&d>>>0>e>>>0|h>>>0>l>>>0;e=d;d=d+i|0;j=(e>>>0>d>>>0?1:j)+m|0;l=o;e=d;d=d+s|0;j=e>>>0>d>>>0?j+1|0:j;e=j;h=(m|0)==(j|0)&d>>>0<s>>>0|j>>>0<m>>>0;i=h;h=h+y|0;l=i>>>0>h>>>0?l+1|0:l;i=l;c:{if((l|0)==131071|l>>>0<131071){L=O|N;K=v|P;xb(k+80|0,d,j,h,l,f,g,A,z);l=J[k+84>>2];n=l;j=b<<17;m=0;p=J[k+88>>2];c=m-p|0;b=J[k+80>>2];l=(l|b)!=0;o=c-l|0;p=(j-(J[k+92>>2]+(m>>>0<p>>>0)|0)|0)-(c>>>0<l>>>0)|0;m=0-b|0;n=0-(((b|0)!=0)+n|0)|0;b=I+16382|0;break c}d=(e&1)<<31|d>>>1;e=h<<31|e>>>1;h=(i&1)<<31|h>>>1;i=i>>>1|0;xb(k+96|0,d,e,h,i,f,g,A,z);o=J[k+100>>2];r=o;n=0;u=J[k+104>>2];m=n-u|0;l=J[k+96>>2];p=(o|l)!=0;o=m-p|0;p=((b<<16)-(J[k+108>>2]+(n>>>0<u>>>0)|0)|0)-(m>>>0<p>>>0)|0;m=0-l|0;n=0-(((l|0)!=0)+r|0)|0;u=b;r=c;b=I+16383|0;}if((b|0)>=32767){t=t|2147418112;b=0;c=0;break a}d:{if((b|0)>0){l=p<<1|o>>>31;o=o<<1|n>>>31;p=l;u=h;r=i&65535|b<<16;l=n<<1|m>>>31;i=m<<1;break d}if((b|0)<=-113){b=0;c=0;break a}xc(k- -64|0,d,e,h,i,1-b|0);Cb(k+48|0,u,r,L,K,b+112|0);d=J[k+64>>2];e=J[k+68>>2];u=J[k+72>>2];r=J[k+76>>2];xb(k+32|0,f,g,A,z,d,e,u,r);b=J[k+40>>2];c=J[k+56>>2];l=J[k+36>>2];o=b<<1|l>>>31;m=c-o|0;p=J[k+60>>2]-((J[k+44>>2]<<1|b>>>31)+(c>>>0<o>>>0)|0)|0;b=J[k+32>>2];h=l<<1|b>>>31;j=b<<1;i=J[k+52>>2];c=J[k+48>>2];b=(h|0)==(i|0)&j>>>0>c>>>0|h>>>0>i>>>0;o=m-b|0;p=p-(b>>>0>m>>>0)|0;l=i-((c>>>0<j>>>0)+h|0)|0;i=c-j|0;}b=i;xb(k+16|0,f,g,A,z,3,0,0,0);xb(k,f,g,A,z,5,0,0,0);c=0;h=l+c|0;j=d&1;b=b+j|0;h=i>>>0>b>>>0?h+1|0:h;i=b;g=(h|0)==(g|0)&f>>>0<i>>>0|g>>>0<h>>>0;l=p;b=(c|0)==(h|0)&i>>>0<j>>>0|c>>>0>h>>>0;f=b+o|0;l=b>>>0>f>>>0?l+1|0:l;j=e;b=(l|0)==(z|0);b=b&(f|0)==(A|0)?g:b&f>>>0>A>>>0|l>>>0>z>>>0;c=b;b=b+d|0;j=c>>>0>b>>>0?j+1|0:j;c=j;d=(e|0)==(j|0)&b>>>0<d>>>0|e>>>0>j>>>0;j=r;e=d;d=d+u|0;j=e>>>0>d>>>0?j+1|0:j;g=d;e=J[k+20>>2];n=(e|0)==(h|0)&M[k+16>>2]<i>>>0|e>>>0<h>>>0;e=J[k+28>>2];d=J[k+24>>2];e=j>>>0<2147418112&((d|0)==(f|0)&(e|0)==(l|0)?n:(e|0)==(l|0)&d>>>0<f>>>0|e>>>0<l>>>0);d=c;n=e;e=b+e|0;d=n>>>0>e>>>0?d+1|0:d;b=(c|0)==(d|0)&b>>>0>e>>>0|c>>>0>d>>>0;c=b;b=b+g|0;j=c>>>0>b>>>0?j+1|0:j;g=b;c=J[k+4>>2];h=(c|0)==(h|0)&M[k>>2]<i>>>0|c>>>0<h>>>0;c=J[k+12>>2];b=J[k+8>>2];b=j>>>0<2147418112&((b|0)==(f|0)&(c|0)==(l|0)?h:(c|0)==(l|0)&b>>>0<f>>>0|c>>>0<l>>>0);c=b;b=b+e|0;l=c>>>0>b>>>0?d+1|0:d;c=l;e=(d|0)==(l|0)&b>>>0<e>>>0|d>>>0>l>>>0;d=j;f=e;e=e+g|0;d=f>>>0>e>>>0?d+1|0:d;q=e|q;t=d|t;}J[a>>2]=b;J[a+4>>2]=c;J[a+8>>2]=q;J[a+12>>2]=t;Ua=k+336|0;}function hc(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=+e;f=f|0;var g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0;h=Ua-128|0;Ua=h;Yc(h+56|0,c,O[a+16>>3],O[a+24>>3],e,f);J[5996]=0;f=$(46,c|0)|0;g=J[5996];J[5996]=0;a:{b:{c:{if((g|0)!=1){J[h+52>>2]=0;J[h+44>>2]=0;J[h+48>>2]=0;H[h+84|0]=0;J[h+80>>2]=h+44;g=J[f+4>>2];j=J[f>>2];l=g-j|0;i=(l|0)/12|0;if((g|0)!=(j|0)){d:{e:{f:{if(i>>>0>=357913942){J[5996]=0;ca(47);a=J[5996];J[5996]=0;if((a|0)==1){break f}break a}J[5996]=0;g=$(3,l|0)|0;j=J[5996];J[5996]=0;if((j|0)==1){break f}J[h+48>>2]=g;J[h+44>>2]=g;J[h+52>>2]=g+P(i,12);i=J[f+4>>2];f=J[f>>2];J[5996]=0;f=ia(48,h+52|0,f|0,i|0,g|0)|0;i=J[5996];J[5996]=0;if((i|0)!=1){break d}f=Z()|0;J[h+48>>2]=g;break e}f=Z()|0;}Nc(h+80|0);break c}J[h+48>>2]=f;}J[h+40>>2]=0;J[h+32>>2]=0;J[h+36>>2]=0;f=J[d+4>>2];g=J[d>>2];J[5996]=0;aa(49,h+32|0,f-g>>2);f=J[5996];J[5996]=0;g:{if((f|0)!=1){f=J[h+48>>2];g=J[h+44>>2];if((f|0)==(g|0)){break b}f=g;h:{while(1){i:{J[h+28>>2]=0;J[h+20>>2]=0;J[h+24>>2]=0;i=J[d+4>>2];g=J[d>>2];o=P(r,12);f=o+f|0;if(i-g>>2!=J[f+4>>2]-J[f>>2]>>3){break i}j=0;n=1;m=0;j:{if((g|0)!=(i|0)){k:{l:{m:{while(1){f=m<<3;s=O[f+J[h+68>>2]>>3];t=O[f+J[h+56>>2]>>3];g=J[(m<<2)+g>>2];J[5996]=0;g=qa(50,g|0,+t,+s,b|0)|0;i=J[5996];J[5996]=0;if((i|0)==1){break l}f=f+J[o+J[h+44>>2]>>2]|0;i=J[f>>2];f=J[f+4>>2];l=J[g+4>>2];g=J[g>>2];J[5996]=0;oa(51,h+80|0,g|0,l|0,i|0,f|0);f=J[5996];J[5996]=0;if((f|0)!=1){i=J[h+28>>2];n:{if(i>>>0>j>>>0){J[j+8>>2]=0;J[j>>2]=0;J[j+4>>2]=0;J[j>>2]=J[h+80>>2];J[j+4>>2]=J[h+84>>2];J[j+8>>2]=J[h+88>>2];J[h+24>>2]=j+12;break n}o:{g=J[h+20>>2];l=(j-g|0)/12|0;f=l+1|0;p:{if(f>>>0>=357913942){J[5996]=0;ca(52);break p}i=(i-g|0)/12|0;k=i<<1;f=i>>>0>=178956970?357913941:f>>>0<k>>>0?k:f;if(f>>>0<357913942){break o}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break a}break m}J[5996]=0;k=P(f,12);i=$(3,k|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){break m}f=i+P(l,12)|0;J[f>>2]=J[h+80>>2];J[f+4>>2]=J[h+84>>2];J[f+8>>2]=J[h+88>>2];J[h+88>>2]=0;J[h+80>>2]=0;J[h+84>>2]=0;i=i+k|0;l=f+12|0;q:{if((g|0)==(j|0)){J[h+28>>2]=i;J[h+24>>2]=l;J[h+20>>2]=f;break q}while(1){f=f-12|0;j=j-12|0;J[f>>2]=J[j>>2];J[f+4>>2]=J[j+4>>2];J[f+8>>2]=J[j+8>>2];J[j+8>>2]=0;J[j>>2]=0;J[j+4>>2]=0;if((g|0)!=(j|0)){continue}break}J[h+28>>2]=i;g=J[h+24>>2];J[h+24>>2]=l;j=J[h+20>>2];J[h+20>>2]=f;if((g|0)==(j|0)){break q}while(1){k=g-12|0;l=J[k>>2];if(l){p=g-8|0;f=J[p>>2];i=l;if((f|0)!=(i|0)){while(1){f=f-32|0;g=J[f+4>>2];r:{if(!g){break r}i=J[g+4>>2];J[g+4>>2]=i-1;if(i){break r}Xa[J[J[g>>2]+8>>2]](g);bb(g);}if((f|0)!=(l|0)){continue}break}i=J[k>>2];}J[p>>2]=l;$a(i);}g=k;if((j|0)!=(g|0)){continue}break}}if(j){$a(j);}i=J[h+80>>2];if(!i){break n}f=J[h+84>>2];g=i;if((f|0)!=(g|0)){while(1){f=f-32|0;g=J[f+4>>2];s:{if(!g){break s}j=J[g+4>>2];J[g+4>>2]=j-1;if(j){break s}Xa[J[J[g>>2]+8>>2]](g);bb(g);}if((f|0)!=(i|0)){continue}break}g=J[h+80>>2];}J[h+84>>2]=i;$a(g);}j=J[h+24>>2];f=j-12|0;n=P(J[f+4>>2]-J[f>>2]>>5,n);m=m+1|0;g=J[d>>2];if(m>>>0>=J[d+4>>2]-g>>2>>>0){break k}continue}break}f=Z()|0;break h}f=Z()|0;_b(h+80|0);break h}f=Z()|0;break h}if(!n){break j}}l=0;t:{u:{while(1){j=J[h+32>>2];if(J[d+4>>2]!=J[d>>2]){f=0;i=J[h+20>>2];g=l;while(1){k=P(f,12);m=k+i|0;i=J[m>>2];J[(f<<2)+j>>2]=i+((g>>>0)%(J[m+4>>2]-i>>5>>>0)<<5);m=g;i=J[h+20>>2];g=k+i|0;g=(m>>>0)/(J[g+4>>2]-J[g>>2]>>5>>>0)|0;f=f+1|0;if(f>>>0<J[d+4>>2]-J[d>>2]>>2>>>0){continue}break}}J[h+104>>2]=0;J[h+96>>2]=0;J[h+100>>2]=0;f=j;o=J[h+36>>2];if((f|0)!=(o|0)){v:{while(1){w:{k=J[j>>2];f=J[h+100>>2];x:{if((f|0)!=J[h+104>>2]){J[f>>2]=J[k>>2];g=J[k+4>>2];J[f+4>>2]=g;if(g){J[g+4>>2]=J[g+4>>2]+1;}J[h+100>>2]=f+8;break x}y:{i=J[h+96>>2];g=f-i|0;p=g>>3;m=p+1|0;z:{if(m>>>0>=536870912){J[5996]=0;ca(59);break z}q=g>>2;g=g>>>0>=2147483640?536870911:m>>>0<q>>>0?q:m;if(g>>>0<536870912){break y}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break a}f=Z()|0;break t}J[5996]=0;q=g<<3;m=$(3,q|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break w}g=m+(p<<3)|0;J[g>>2]=J[k>>2];k=J[k+4>>2];J[g+4>>2]=k;if(k){J[k+4>>2]=J[k+4>>2]+1;f=J[h+100>>2];}k=m+q|0;m=g+8|0;A:{if((f|0)==(i|0)){J[h+104>>2]=k;J[h+100>>2]=m;J[h+96>>2]=g;break A}while(1){g=g-8|0;f=f-8|0;J[g>>2]=J[f>>2];J[g+4>>2]=J[f+4>>2];J[f>>2]=0;J[f+4>>2]=0;if((f|0)!=(i|0)){continue}break}J[h+104>>2]=k;f=J[h+100>>2];J[h+100>>2]=m;i=J[h+96>>2];J[h+96>>2]=g;if((f|0)==(i|0)){break A}while(1){f=f-8|0;g=J[f+4>>2];B:{if(!g){break B}k=J[g+4>>2];J[g+4>>2]=k-1;if(k){break B}Xa[J[J[g>>2]+8>>2]](g);bb(g);}if((f|0)!=(i|0)){continue}break}}if(!i){break x}$a(i);}j=j+4|0;if((o|0)!=(j|0)){continue}break v}break}f=Z()|0;break t}j=J[h+36>>2];f=J[h+32>>2];}J[5996]=0;va(63,h+80|0,c|0,f|0,j|0,+e);f=J[5996];J[5996]=0;if((f|0)==1){break u}J[5996]=0;i=$(3,104)|0;f=J[5996];J[5996]=0;if((f|0)==1){break u}J[i>>2]=7e3;J[i+4>>2]=0;J[i+8>>2]=0;J[5996]=0;J[h+116>>2]=J[h+96>>2];f=J[h+104>>2];g=J[h+100>>2];J[h+104>>2]=0;J[h+120>>2]=g;J[h+124>>2]=f;J[h+96>>2]=0;J[h+100>>2]=0;k=ia(64,i+12|0,c|0,h+116|0,0)|0;f=J[5996];J[5996]=0;if((f|0)!=1){g=J[h+116>>2];if(g){f=J[h+120>>2];if((g|0)!=(f|0)){while(1){f=f-8|0;j=J[f+4>>2];C:{if(!j){break C}m=J[j+4>>2];J[j+4>>2]=m-1;if(m){break C}Xa[J[J[j>>2]+8>>2]](j);bb(j);}if((f|0)!=(g|0)){continue}break}g=J[h+116>>2];}$a(g);}J[h+112>>2]=i;J[h+108>>2]=k;J[5996]=0;f=J[h+112>>2];J[h+8>>2]=J[h+108>>2];J[h+12>>2]=f;ea(61,a|0,h+80|0,h+8|0);f=J[5996];J[5996]=0;if((f|0)==1){break u}g=J[h+96>>2];if(g){f=J[h+100>>2];if((g|0)!=(f|0)){while(1){f=f-8|0;i=J[f+4>>2];D:{if(!i){break D}j=J[i+4>>2];J[i+4>>2]=j-1;if(j){break D}Xa[J[J[i>>2]+8>>2]](i);bb(i);}if((f|0)!=(g|0)){continue}break}g=J[h+96>>2];}$a(g);}l=l+1|0;if((n|0)!=(l|0)){continue}break j}break}f=Z()|0;lb(h+116|0);$a(i);break t}f=Z()|0;}lb(h+96|0);break h}i=J[h+20>>2];if(i){f=J[h+24>>2];if((i|0)!=(f|0)){while(1){l=f-12|0;j=J[l>>2];if(j){k=f-8|0;f=J[k>>2];g=j;if((f|0)!=(g|0)){while(1){f=f-32|0;g=J[f+4>>2];E:{if(!g){break E}n=J[g+4>>2];J[g+4>>2]=n-1;if(n){break E}Xa[J[J[g>>2]+8>>2]](g);bb(g);}if((f|0)!=(j|0)){continue}break}g=J[l>>2];}J[k>>2]=j;$a(g);}f=l;if((i|0)!=(f|0)){continue}break}i=J[h+20>>2];}$a(i);}r=r+1|0;f=J[h+44>>2];if(r>>>0<(J[h+48>>2]-f|0)/12>>>0){continue}break b}break}J[5996]=0;ga(44,5170,1900,642,2497);a=J[5996];J[5996]=0;if((a|0)!=1){break a}f=Z()|0;}Xc(h+20|0);break g}f=Z()|0;}a=J[h+32>>2];if(a){$a(a);}Wc(h+44|0);break c}f=Z()|0;}Mc(h+56|0);da(f|0);B();}a=J[h+32>>2];if(a){$a(a);f=J[h+44>>2];}if(f){g=f;i=J[h+48>>2];if((i|0)!=(f|0)){while(1){a=i-12|0;b=J[a>>2];if(b){J[i-8>>2]=b;$a(b);}i=a;if((i|0)!=(f|0)){continue}break}g=J[h+44>>2];}J[h+48>>2]=f;$a(g);}a=J[h+68>>2];if(a){J[h+72>>2]=a;$a(a);}a=J[h+56>>2];if(a){J[h+60>>2]=a;$a(a);}Ua=h+128|0;return}B();}function tk(a,b,c,d,e,f,g,h,i,j,k){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;i=i|0;j=j|0;k=k|0;var l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0;l=Ua-528|0;Ua=l;J[l+520>>2]=k;J[l+524>>2]=b;a:{b:{if(Tb(a,l+524|0)){J[f>>2]=J[f>>2]|4;a=0;break b}J[l+72>>2]=220;t=l+104|0;J[t>>2]=l+112;s=l+72|0;J[t+4>>2]=J[s>>2];o=J[t>>2];J[l+100>>2]=o;J[l+96>>2]=o+400;J[s>>2]=0;J[s+4>>2]=0;J[s+8>>2]=0;p=l+60|0;J[p>>2]=0;J[p+4>>2]=0;J[p+8>>2]=0;m=l+48|0;J[m>>2]=0;J[m+4>>2]=0;J[m+8>>2]=0;n=l+36|0;J[n>>2]=0;J[n+4>>2]=0;J[n+8>>2]=0;q=l+24|0;J[q>>2]=0;J[q+4>>2]=0;J[q+8>>2]=0;b=J[d>>2];J[5996]=0;sa(229,c|0,b|0,l+92|0,l+88|0,l+84|0,s|0,p|0,m|0,n|0,l+20|0);b=J[5996];J[5996]=0;if((b|0)!=1){J[j>>2]=J[i>>2];v=e&512;u=J[l+20>>2];e=0;while(1){c:{d:{e:{f:{g:{if((e|0)==4){break g}J[5996]=0;b=_(202,a|0,l+524|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}if(b){break g}h:{i:{j:{k:{l:{switch(H[(l+92|0)+e|0]){case 1:if((e|0)==3){break c}b=J[a>>2];J[5996]=0;b=$(203,b|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}J[5996]=0;b=ba(230,h|0,1,b|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}if(b){J[5996]=0;aa(231,l+12|0,a|0);b=J[5996];J[5996]=0;if((b|0)!=1){J[5996]=0;aa(232,q|0,J[l+12>>2]);b=J[5996];J[5996]=0;if((b|0)!=1){break k}}break a}J[f>>2]=J[f>>2]|4;a=0;break d;case 4:break h;case 2:break i;case 3:break j;case 0:break l;default:break c}}if((e|0)==3){break c}}while(1){J[5996]=0;b=_(202,a|0,l+524|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}if(b){break c}b=J[a>>2];J[5996]=0;b=$(203,b|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}J[5996]=0;b=ba(230,h|0,1,b|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}if(!b){break c}J[5996]=0;aa(231,l+12|0,a|0);b=J[5996];J[5996]=0;if((b|0)!=1){J[5996]=0;aa(232,q|0,J[l+12>>2]);b=J[5996];J[5996]=0;if((b|0)!=1){continue}}break}break a}m:{b=K[m+11|0];if(!((b&128)>>>7|0?J[m+4>>2]:b&127)){break m}b=J[a>>2];J[5996]=0;b=$(203,b|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}if(J[((K[m+11|0]&128)>>>7|0?J[m>>2]:m)>>2]!=(b|0)){break m}J[5996]=0;$(205,a|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break a}H[g|0]=0;b=K[m+11|0];r=((b&128)>>>7|0?J[m+4>>2]:b&127)>>>0>1?m:r;break c}n:{o:{k=J[n+4>>2];b=K[n+11|0];if((b&128)>>>7|0?k:b&127){b=J[a>>2];J[5996]=0;b=$(203,b|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}if(J[((K[n+11|0]&128)>>>7|0?J[n>>2]:n)>>2]==(b|0)){break o}k=J[n+4>>2];b=K[n+11|0];}c=K[m+11|0];c=(c&128)>>>7|0?J[m+4>>2]:c&127;if(!c|!((b&128)>>>7|0?k:b&127)){break n}J[f>>2]=J[f>>2]|4;a=0;break d}J[5996]=0;$(205,a|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break a}H[g|0]=1;b=K[n+11|0];r=((b&128)>>>7|0?J[n+4>>2]:b&127)>>>0>1?n:r;break c}b=(b&128)>>>7|0?k:b&127;if(!(b|c)){break c}H[g|0]=!b;break c}if(!(r|e>>>0<2|((e|0)==2&K[l+95|0]!=0|v))){r=0;break c}J[l+12>>2]=Mb(p);p:{if(!e|K[(e+l|0)+91|0]>1){break p}q:{while(1){c=qc(p);b=J[l+12>>2];if((c|0)==(b|0)){break q}b=J[b>>2];J[5996]=0;b=ba(230,h|0,1,b|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){if(!b){break q}J[l+12>>2]=J[l+12>>2]+4;continue}break}break a}b=Mb(p);b=J[l+12>>2]-b>>2;c=K[q+11|0];if(b>>>0<=((c&128)>>>7|0?J[q+4>>2]:c&127)>>>0){c=Rd(qc(q),0-b|0);d=qc(q);k=Mb(p);b=Ua-16|0;Ua=b;J[b+8>>2]=k;J[b+12>>2]=c;while(1){k=(c|0)!=(d|0);if(!(!k|J[c>>2]!=J[J[b+8>>2]>>2])){J[b+12>>2]=J[b+12>>2]+4;J[b+8>>2]=J[b+8>>2]+4;c=J[b+12>>2];continue}break}Ua=b+16|0;if(!k){break p}}J[l+8>>2]=Mb(p);J[l+12>>2]=J[l+8>>2];}J[l+8>>2]=J[l+12>>2];r:{while(1){if((qc(p)|0)==J[l+8>>2]){break r}J[5996]=0;b=_(202,a|0,l+524|0)|0;c=J[5996];J[5996]=0;s:{if((c|0)==1){break s}if(b){break r}b=J[a>>2];J[5996]=0;b=$(203,b|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break s}if(J[J[l+8>>2]>>2]!=(b|0)){break r}J[5996]=0;$(205,a|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break a}J[l+8>>2]=J[l+8>>2]+4;continue}break}break a}if(!v){break c}if((qc(p)|0)==J[l+8>>2]){break c}J[f>>2]=J[f>>2]|4;a=0;break d}b=0;w=J[l+84>>2];c=o;t:{while(1){u:{J[5996]=0;d=_(202,a|0,l+524|0)|0;k=J[5996];J[5996]=0;if((k|0)==1){break u}if(d){break t}d=J[a>>2];J[5996]=0;k=$(203,d|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break a}J[5996]=0;d=ba(230,h|0,64,k|0)|0;x=J[5996];J[5996]=0;if((x|0)==1){break a}v:{if(d){d=J[j>>2];if((d|0)==J[l+520>>2]){J[5996]=0;ea(233,i|0,j|0,l+520|0);d=J[5996];J[5996]=0;if((d|0)==1){break a}d=J[j>>2];}J[j>>2]=d+4;J[d>>2]=k;b=b+1|0;break v}d=K[s+11|0];if(!((d&128)>>>7|0?J[s+4>>2]:d&127)|!b|(k|0)!=(w|0)){break t}if(J[l+96>>2]==(o|0)){J[5996]=0;ea(226,t|0,l+100|0,l+96|0);c=J[5996];J[5996]=0;if((c|0)==1){break a}o=J[l+100>>2];}c=o+4|0;J[l+100>>2]=c;J[o>>2]=b;o=c;b=0;}J[5996]=0;$(205,a|0)|0;d=J[5996];J[5996]=0;if((d|0)!=1){continue}}break}break a}if(!b|J[t>>2]==(c|0)){break f}if(J[l+96>>2]==(c|0)){J[5996]=0;ea(226,t|0,l+100|0,l+96|0);c=J[5996];J[5996]=0;if((c|0)==1){break a}c=J[l+100>>2];}o=c+4|0;J[l+100>>2]=o;J[c>>2]=b;break e}J[l+20>>2]=u;w:{if(!r){break w}k=1;while(1){b=K[r+11|0];if(((b&128)>>>7|0?J[r+4>>2]:b&127)>>>0<=k>>>0){break w}J[5996]=0;b=_(202,a|0,l+524|0)|0;c=J[5996];J[5996]=0;x:{if((c|0)==1){break x}y:{if(!b){b=J[a>>2];J[5996]=0;b=$(203,b|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break x}if(J[(k<<2)+((K[r+11|0]&128)>>>7|0?J[r>>2]:r)>>2]==(b|0)){break y}}J[f>>2]=J[f>>2]|4;a=0;break d}J[5996]=0;$(205,a|0)|0;b=J[5996];J[5996]=0;k=k+1|0;if((b|0)!=1){continue}}break}break a}a=1;b=J[t>>2];if((b|0)==(o|0)){break d}a=0;J[l+12>>2]=0;zb(s,b,o,l+12|0);if(J[l+12>>2]){J[f>>2]=J[f>>2]|4;break d}a=1;break d}o=c;}z:{if((u|0)<=0){break z}J[5996]=0;b=_(202,a|0,l+524|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}A:{if(!b){b=J[a>>2];J[5996]=0;b=$(203,b|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}if(J[l+88>>2]==(b|0)){break A}}J[f>>2]=J[f>>2]|4;a=0;break d}J[5996]=0;b=$(205,a|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}while(1){if((u|0)<=0){u=0;break z}J[5996]=0;c=_(202,b|0,l+524|0)|0;d=J[5996];J[5996]=0;B:{if((d|0)==1){break B}C:{if(!c){c=J[b>>2];J[5996]=0;c=$(203,c|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break B}J[5996]=0;c=ba(230,h|0,64,c|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break B}if(c){break C}}J[f>>2]=J[f>>2]|4;a=0;break d}if(J[j>>2]==J[l+520>>2]){J[5996]=0;ea(233,i|0,j|0,l+520|0);c=J[5996];J[5996]=0;if((c|0)==1){break B}}c=J[b>>2];J[5996]=0;c=$(203,c|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break B}d=J[j>>2];J[j>>2]=d+4;J[d>>2]=c;J[5996]=0;$(205,b|0)|0;c=J[5996];J[5996]=0;u=u-1|0;if((c|0)!=1){continue}}break}break a}if(J[j>>2]!=J[i>>2]){break c}J[f>>2]=J[f>>2]|4;a=0;}ab(q);ab(n);ab(m);ab(p);ab(s);db(t);break b}e=e+1|0;continue}}break a}Ua=l+528|0;return a|0}a=Z()|0;ab(q);ab(n);ab(m);ab(p);ab(s);db(t);da(a|0);B();}function Ak(a,b,c,d,e,f,g,h,i,j,k){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;i=i|0;j=j|0;k=k|0;var l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0;l=Ua-528|0;Ua=l;J[l+520>>2]=k;J[l+524>>2]=b;a:{b:{if(Ub(a,l+524|0)){J[f>>2]=J[f>>2]|4;a=0;break b}J[l+76>>2]=220;t=l+104|0;J[t>>2]=l+112;s=l+76|0;J[t+4>>2]=J[s>>2];o=J[t>>2];J[l+100>>2]=o;J[l+96>>2]=o+400;J[s>>2]=0;J[s+4>>2]=0;J[s+8>>2]=0;p=l- -64|0;J[p>>2]=0;J[p+4>>2]=0;J[p+8>>2]=0;m=l+52|0;J[m>>2]=0;J[m+4>>2]=0;J[m+8>>2]=0;n=l+40|0;J[n>>2]=0;J[n+4>>2]=0;J[n+8>>2]=0;q=l+28|0;J[q>>2]=0;J[q+4>>2]=0;J[q+8>>2]=0;b=J[d>>2];J[5996]=0;sa(223,c|0,b|0,l+92|0,l+91|0,l+90|0,s|0,p|0,m|0,n|0,l+24|0);b=J[5996];J[5996]=0;if((b|0)!=1){J[j>>2]=J[i>>2];w=e&512;u=J[l+24>>2];x=K[l+95|0];y=K[l+91|0];z=K[l+90|0];d=o;e=0;while(1){c:{d:{e:{f:{if((e|0)==4){break f}J[5996]=0;b=_(186,a|0,l+524|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}if(b){break f}b=0;g:{h:{i:{j:{k:{switch(H[(l+92|0)+e|0]){case 1:if((e|0)==3){break c}b=J[a>>2];J[5996]=0;b=$(187,b|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}c=J[h+8>>2];if(J[c+((b&255)<<2)>>2]&1&(b|0)>=0){J[5996]=0;aa(224,l+16|0,a|0);b=J[5996];J[5996]=0;if((b|0)!=1){J[5996]=0;aa(98,q|0,H[l+16|0]);b=J[5996];J[5996]=0;if((b|0)!=1){break j}}break a}J[f>>2]=J[f>>2]|4;a=0;break d;case 4:break g;case 2:break h;case 3:break i;case 0:break k;default:break c}}if((e|0)==3){break c}}while(1){J[5996]=0;b=_(186,a|0,l+524|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}if(b){break c}b=J[a>>2];J[5996]=0;b=$(187,b|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}c=J[h+8>>2];if(!(J[c+((b&255)<<2)>>2]&1&(b|0)>=0)){break c}J[5996]=0;aa(224,l+16|0,a|0);b=J[5996];J[5996]=0;if((b|0)!=1){J[5996]=0;aa(98,q|0,H[l+16|0]);b=J[5996];J[5996]=0;if((b|0)!=1){continue}}break}break a}l:{b=K[m+11|0];if(!((b&128)>>>7|0?J[m+4>>2]:b&127)){break l}b=J[a>>2];J[5996]=0;b=$(187,b|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}if(K[((K[m+11|0]&128)>>>7|0?J[m>>2]:m)|0]!=(b&255)){break l}J[5996]=0;$(189,a|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break a}H[g|0]=0;b=K[m+11|0];r=((b&128)>>>7|0?J[m+4>>2]:b&127)>>>0>1?m:r;break c}m:{n:{k=J[n+4>>2];b=K[n+11|0];if((b&128)>>>7|0?k:b&127){b=J[a>>2];J[5996]=0;b=$(187,b|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}if(K[((K[n+11|0]&128)>>>7|0?J[n>>2]:n)|0]==(b&255)){break n}k=J[n+4>>2];b=K[n+11|0];}c=K[m+11|0];c=(c&128)>>>7|0?J[m+4>>2]:c&127;if(!c|!((b&128)>>>7|0?k:b&127)){break m}J[f>>2]=J[f>>2]|4;a=0;break d}J[5996]=0;$(189,a|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break a}H[g|0]=1;b=K[n+11|0];r=((b&128)>>>7|0?J[n+4>>2]:b&127)>>>0>1?n:r;break c}b=(b&128)>>>7|0?k:b&127;if(!(b|c)){break c}H[g|0]=!b;break c}if(!(r|e>>>0<2|((e|0)==2&(x|0)!=0|w))){r=0;break c}J[l+16>>2]=Mb(p);o:{if(!e|K[(e+l|0)+91|0]>1){break o}while(1){p:{c=sc(p);b=J[l+16>>2];if((c|0)==(b|0)){break p}c=J[h+8>>2];b=H[b|0];if((b|0)>=0){b=J[c+((b&255)<<2)>>2]&1;}else {b=0;}if(!b){break p}J[l+16>>2]=J[l+16>>2]+1;continue}break}b=Mb(p);b=J[l+16>>2]-b|0;c=K[q+11|0];if(b>>>0<=((c&128)>>>7|0?J[q+4>>2]:c&127)>>>0){c=Td(sc(q),0-b|0);k=sc(q);v=Mb(p);b=Ua-16|0;Ua=b;J[b+8>>2]=v;J[b+12>>2]=c;while(1){v=(c|0)!=(k|0);if(!(!v|K[c|0]!=K[J[b+8>>2]])){J[b+12>>2]=J[b+12>>2]+1;J[b+8>>2]=J[b+8>>2]+1;c=J[b+12>>2];continue}break}Ua=b+16|0;if(!v){break o}}J[l+12>>2]=Mb(p);J[l+16>>2]=J[l+12>>2];}J[l+12>>2]=J[l+16>>2];q:{while(1){if((sc(p)|0)==J[l+12>>2]){break q}J[5996]=0;b=_(186,a|0,l+524|0)|0;c=J[5996];J[5996]=0;r:{if((c|0)==1){break r}if(b){break q}b=J[a>>2];J[5996]=0;b=$(187,b|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break r}if(K[J[l+12>>2]]!=(b&255)){break q}J[5996]=0;$(189,a|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break a}J[l+12>>2]=J[l+12>>2]+1;continue}break}break a}if(!w){break c}if((sc(p)|0)==J[l+12>>2]){break c}J[f>>2]=J[f>>2]|4;a=0;break d}s:{while(1){t:{J[5996]=0;c=_(186,a|0,l+524|0)|0;k=J[5996];J[5996]=0;if((k|0)==1){break t}if(c){break s}c=J[a>>2];J[5996]=0;k=$(187,c|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}c=J[h+8>>2];u:{if((k|0)>=0?J[c+((k&255)<<2)>>2]&64:0){c=J[j>>2];if((c|0)==J[l+520>>2]){J[5996]=0;ea(225,i|0,j|0,l+520|0);c=J[5996];J[5996]=0;if((c|0)==1){break a}c=J[j>>2];}J[j>>2]=c+1;H[c|0]=k;b=b+1|0;break u}c=K[s+11|0];if(!((c&128)>>>7|0?J[s+4>>2]:c&127)|!b|(k&255)!=(z|0)){break s}if(J[l+96>>2]==(d|0)){J[5996]=0;ea(226,t|0,l+100|0,l+96|0);c=J[5996];J[5996]=0;if((c|0)==1){break a}d=J[l+100>>2];}o=d+4|0;J[l+100>>2]=o;J[d>>2]=b;d=o;b=0;}J[5996]=0;$(189,a|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){continue}}break}break a}if(!(!b|J[t>>2]==(o|0))){if(J[l+96>>2]==(o|0)){J[5996]=0;ea(226,t|0,l+100|0,l+96|0);c=J[5996];J[5996]=0;if((c|0)==1){break a}o=J[l+100>>2];}c=o+4|0;J[l+100>>2]=c;J[o>>2]=b;o=c;}if((u|0)<=0){break e}J[5996]=0;b=_(186,a|0,l+524|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}v:{if(!b){b=J[a>>2];J[5996]=0;b=$(187,b|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}if((b&255)==(y|0)){break v}}J[f>>2]=J[f>>2]|4;a=0;break d}J[5996]=0;b=$(189,a|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}while(1){if((u|0)<=0){u=0;break e}J[5996]=0;c=_(186,b|0,l+524|0)|0;d=J[5996];J[5996]=0;w:{if((d|0)==1){break w}x:{if(!c){c=J[b>>2];J[5996]=0;c=$(187,c|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break w}d=J[h+8>>2];if((c|0)>=0?J[d+((c&255)<<2)>>2]&64:0){break x}}J[f>>2]=J[f>>2]|4;a=0;break d}if(J[j>>2]==J[l+520>>2]){J[5996]=0;ea(225,i|0,j|0,l+520|0);c=J[5996];J[5996]=0;if((c|0)==1){break w}}c=J[b>>2];J[5996]=0;c=$(187,c|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break w}d=J[j>>2];J[j>>2]=d+1;H[d|0]=c;J[5996]=0;$(189,b|0)|0;c=J[5996];J[5996]=0;u=u-1|0;if((c|0)!=1){continue}}break}break a}J[l+24>>2]=u;y:{if(!r){break y}k=1;while(1){b=K[r+11|0];if(((b&128)>>>7|0?J[r+4>>2]:b&127)>>>0<=k>>>0){break y}J[5996]=0;b=_(186,a|0,l+524|0)|0;c=J[5996];J[5996]=0;z:{if((c|0)==1){break z}A:{if(!b){b=J[a>>2];J[5996]=0;b=$(187,b|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break z}if(K[((K[r+11|0]&128)>>>7|0?J[r>>2]:r)+k|0]==(b&255)){break A}}J[f>>2]=J[f>>2]|4;a=0;break d}J[5996]=0;$(189,a|0)|0;b=J[5996];J[5996]=0;k=k+1|0;if((b|0)!=1){continue}}break}break a}a=1;b=J[t>>2];if((b|0)==(o|0)){break d}a=0;J[l+16>>2]=0;zb(s,b,o,l+16|0);if(J[l+16>>2]){J[f>>2]=J[f>>2]|4;break d}a=1;break d}d=o;if(J[j>>2]!=J[i>>2]){break c}J[f>>2]=J[f>>2]|4;a=0;}ab(q);ab(n);ab(m);ab(p);ab(s);db(t);break b}e=e+1|0;continue}}break a}Ua=l+528|0;return a|0}a=Z()|0;ab(q);ab(n);ab(m);ab(p);ab(s);db(t);da(a|0);B();}function An(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0;a:{b:{c:{d:{e:{f:{g:{switch(J[b>>2]){case 0:J[a+12>>2]=1;J[a+16>>2]=0;J[a+4>>2]=0;J[a+8>>2]=0;J[a>>2]=0;return;case 2:J[a+12>>2]=1;J[a+16>>2]=0;break b;case 3:J[a+12>>2]=1;J[a+16>>2]=0;J[a+4>>2]=1;J[a+8>>2]=0;J[a>>2]=3;return;case 4:case 5:J[a+12>>2]=0;J[a+16>>2]=0;J[a+4>>2]=1;J[a+8>>2]=0;J[a>>2]=1;return;case 6:case 7:case 8:case 9:J[a+12>>2]=0;J[a+16>>2]=0;J[a+4>>2]=1;J[a+8>>2]=0;J[a>>2]=4;return;case 17:b=J[b+32>>2];c=J[b+8>>2];h=J[c+52>>2];b=J[b>>2];f=J[b+52>>2];e=K[b+48|0];g=K[c+48|0];c=J[c+44>>2];b=J[b+44>>2];J[a+12>>2]=0;J[a+16>>2]=0;J[a>>2]=b+c;break a;case 18:b=J[b+32>>2];m=J[b+8>>2];h=J[m+52>>2];j=J[b>>2];f=J[j+52>>2];e=K[j+48|0];g=K[m+48|0];i=J[m+60>>2];d=J[j+60>>2];c=K[j+56|0];b=K[m+56|0];J[a>>2]=(J[j+44>>2]+J[m+44>>2]|0)+1;c=(c|0)!=0&(b|0)!=0;b=c&0;J[a+12>>2]=b|c;J[a+16>>2]=c?d+i|0:0;break a;case 19:b=J[b+32>>2];h=J[b>>2];g=J[h+60>>2];f=K[h+56|0]!=0;c=J[b+8>>2];l=f&K[c+48|0]!=0;e=l?g+J[c+52>>2]|0:0;b=(J[h+44>>2]+J[c+44>>2]|0)+1|0;i=J[c+60>>2];d=K[c+56|0];if(!(!K[h+48|0]|!d)){c=i+J[h+52>>2]|0;e=l?c>>>0>e>>>0?c:e:c;l=1;}J[a>>2]=b;J[a+4>>2]=l;J[a+8>>2]=e;e=f&(d|0)!=0;b=e&0;J[a+12>>2]=b|e;J[a+16>>2]=e?g+i|0:0;return;case 21:b=J[b+32>>2];c=J[b>>2];j=J[c+60>>2];m=J[b+8>>2];e=j+J[m+52>>2]|0;h=K[c+56|0];b=K[m+48|0]!=0&(h|0)!=0;g=(J[c+44>>2]+J[m+44>>2]|0)+3|0;d=b?e:0;h:{if(!K[c+48|0]){break h}if(!b){b=J[c+48>>2];f=b&-256;d=J[c+52>>2];break h}b=1;c=J[c+52>>2];d=c>>>0>e>>>0?c:e;}i=J[m+60>>2];e=K[m+56|0];J[a>>2]=g;c=b&255;b=0;J[a+4>>2]=c|(f|b);J[a+8>>2]=d|p;c=(h|0)!=0&(e|0)!=0;b=c?b:0;J[a+12>>2]=b|c;J[a+16>>2]=c?i+j|0:0;return;case 20:b=J[b+32>>2];i=J[b>>2];c=J[b+8>>2];d=J[i+60>>2]+J[c+52>>2]|0;b=K[c+48|0]!=0&K[i+56|0]!=0;e=(J[i+44>>2]+J[c+44>>2]|0)+2|0;c=b?d:0;i:{if(!K[i+48|0]){break i}if(!b){b=J[i+48>>2];f=b&-256;c=J[i+52>>2];break i}b=1;c=J[i+52>>2];c=c>>>0>d>>>0?c:d;}J[a+12>>2]=0;J[a+16>>2]=0;J[a>>2]=e;J[a+4>>2]=f|b&255;J[a+8>>2]=c|p;return;case 22:b=J[b+32>>2];d=J[b>>2];j=J[b+8>>2];f=J[d+44>>2]+J[j+44>>2]|0;j:{if(!K[d+48|0]){b=J[j+48>>2];h=b&-256;e=J[j+52>>2];break j}if(!K[j+48|0]){b=J[d+48>>2];h=b&-256;e=J[d+52>>2];break j}b=1;e=J[d+52>>2];c=J[j+52>>2];e=c>>>0<e>>>0?e:c;}i=f+3|0;k:{if(!K[d+56|0]){f=J[j+56>>2];g=f&-256;c=J[j+60>>2];break k}if(!K[j+56|0]){f=J[d+56>>2];g=f&-256;c=J[d+60>>2];break k}f=1;d=J[d+60>>2];c=J[j+60>>2];c=c>>>0<d>>>0?d:c;}J[a>>2]=i;J[a+12>>2]=f&255|g&-256;d=0;J[a+16>>2]=d|(c|d);J[a+4>>2]=b&255|h&-256;f=0;J[a+8>>2]=f|(e|f);return;case 23:b=J[b+32>>2];h=J[b+16>>2];f=J[b>>2];g=J[f+60>>2];i=K[f+56|0]!=0;d=i&K[h+48|0]!=0;e=d?J[h+52>>2]+g|0:0;b=J[b+8>>2];c=(J[h+44>>2]+(J[f+44>>2]+J[b+44>>2]|0)|0)+3|0;if(!(!K[b+48|0]|!K[f+48|0])){b=J[f+52>>2]+J[b+52>>2]|0;e=d?b>>>0>e>>>0?b:e:b;d=1;}f=J[h+60>>2];b=K[h+56|0];J[a>>2]=c;J[a+4>>2]=d;J[a+8>>2]=e;e=i&(b|0)!=0;b=e&0;J[a+12>>2]=b|e;J[a+16>>2]=e?f+g|0:0;return;case 25:c=J[b+12>>2];b=J[b+8>>2];J[a>>2]=1;b=(c-b|0)/12|0;c=1;J[a+12>>2]=c;J[a+16>>2]=b;J[a+4>>2]=c;J[a+8>>2]=b;return;case 11:case 12:case 16:d=J[J[b+32>>2]>>2];f=J[d+48>>2];e=J[d+52>>2];c=J[d+44>>2];b=J[d+60>>2];J[a+12>>2]=J[d+56>>2];J[a+16>>2]=b;J[a+4>>2]=f;J[a+8>>2]=e;J[a>>2]=c+1;return;case 10:d=J[J[b+32>>2]>>2];f=J[d+48>>2];e=J[d+52>>2];c=J[d+44>>2];b=J[d+60>>2];J[a+12>>2]=J[d+56>>2];J[a+16>>2]=b;J[a+4>>2]=f;J[a+8>>2]=e;J[a>>2]=c+2;return;case 13:b=J[J[b+32>>2]>>2];e=J[b+48>>2];c=J[b+52>>2];b=J[b+44>>2];J[a+12>>2]=1;J[a+16>>2]=0;J[a+4>>2]=e;J[a+8>>2]=c;J[a>>2]=b+3;return;case 15:b=J[J[b+32>>2]>>2];e=J[b+48>>2];c=J[b+52>>2];b=J[b+44>>2];J[a+12>>2]=1;J[a+16>>2]=0;J[a+4>>2]=e;J[a+8>>2]=c;J[a>>2]=b+4;return;case 14:c=J[J[b+32>>2]>>2];d=J[c+44>>2];f=J[c+80>>2];e=Wb(1239,1);b=J[J[b+32>>2]>>2];c=J[b+48>>2];b=J[b+52>>2];J[a+12>>2]=0;J[a+16>>2]=0;J[a+4>>2]=c;J[a+8>>2]=b;J[a>>2]=d+!(e&(f^-1));return;case 24:c=cb(8);J[c>>2]=1;J[c+4>>2]=0;l=c+8|0;n=J[b+32>>2];p=J[b+36>>2];if((n|0)!=(p|0)){while(1){e=J[n>>2];h=J[e+60>>2];g=K[e+56|0];d=J[c+4>>2];i=c;f=K[c|0];m=J[e+44>>2];J[5996]=0;e=$(3,8)|0;c=J[5996];J[5996]=0;l:{m:{if((c|0)!=1){c=d+h|0;d=(f|0)!=0&(g|0)!=0;f=d&0;J[e>>2]=d|f;J[e+4>>2]=d?c:0;g=e+8|0;c=l-i|0;r=c>>3;if(c>>>0>=9){break m}f=g;break l}f=Z()|0;c=i;break e}j=r>>>0<=2?2:r;l=1;f=g;while(1){q=J[n>>2];c=i+(l<<3)|0;k=J[q+60>>2]+J[c+4>>2]|0;d=c-8|0;o=K[d|0]!=0&K[q+48|0]!=0;h=K[c|0]!=0&K[q+56|0]!=0;c=h?k:0;d=o?J[q+52>>2]+J[d+4>>2]|0:0;c=h?o?c>>>0>d>>>0?c:d:k:d;o=h|o;n:{if(f>>>0<g>>>0){J[f>>2]=o;J[f+4>>2]=c;d=f;break n}h=f-e>>3;k=h+1|0;if(k>>>0>=536870912){J[5996]=0;ca(93);a=J[5996];J[5996]=0;if((a|0)==1){break f}break c}g=g-e|0;d=g>>2;k=g>>>0>=2147483640?536870911:d>>>0>k>>>0?d:k;o:{if(!k){g=0;break o}if(k>>>0>=536870912){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break c}break f}J[5996]=0;g=$(3,k<<3)|0;d=J[5996];J[5996]=0;if((d|0)==1){break f}}d=(h<<3)+g|0;J[d>>2]=o;J[d+4>>2]=c;g=(k<<3)+g|0;c=d;p:{if((e|0)!=(f|0)){while(1){f=f-8|0;h=J[f+4>>2];c=c-8|0;J[c>>2]=J[f>>2];J[c+4>>2]=h;if((e|0)!=(f|0)){continue}break}if(!e){break p}}$a(e);}e=c;}f=d+8|0;l=l+1|0;if((j|0)!=(l|0)){continue}break}}d=(i+(r<<3)|0)-8|0;c=J[n>>2];k=K[d|0]!=0&K[c+48|0]!=0;d=k?J[c+52>>2]+J[d+4>>2]|0:0;q:{if(f>>>0<g>>>0){J[f>>2]=k;J[f+4>>2]=d;l=f+8|0;c=e;break q}r:{s:{h=f-e>>3;j=h+1|0;t:{if(j>>>0>=536870912){J[5996]=0;ca(93);break t}g=g-e|0;c=g>>2;c=g>>>0>=2147483640?536870911:c>>>0>j>>>0?c:j;if(!c){c=0;break r}if(c>>>0<536870912){break s}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break c}break f}J[5996]=0;c=$(3,c<<3)|0;g=J[5996];J[5996]=0;if((g|0)==1){break f}}c=(h<<3)+c|0;J[c>>2]=k;J[c+4>>2]=d;l=c+8|0;if((e|0)!=(f|0)){while(1){f=f-8|0;d=J[f+4>>2];c=c-8|0;J[c>>2]=J[f>>2];J[c+4>>2]=d;if((e|0)!=(f|0)){continue}break}if(!e){break q}}$a(e);}s=(m+s|0)+1|0;$a(i);n=n+8|0;if((p|0)!=(n|0)){continue}break}}b=J[b+4>>2];if(b>>>0<=l-c>>3>>>0){b=(b<<3)+c|0;f=J[b>>2];e=J[b+4>>2];b=J[c+4>>2];J[a+12>>2]=J[c>>2];J[a+16>>2]=b;J[a+4>>2]=f;J[a+8>>2]=e;J[a>>2]=s;$a(c);return}J[5996]=0;ga(44,5118,2210,799,1422);a=J[5996];J[5996]=0;if((a|0)!=1){break c}f=Z()|0;if(!c){break d}break e;default:la(2350,2210,803,1422);B();case 1:break g}}J[a+12>>2]=0;J[a+16>>2]=0;break b}f=Z()|0;if(!e){c=i;break e}$a(e);c=i;}$a(c);}da(f|0);B();}B();}J[a+4>>2]=1;J[a+8>>2]=0;J[a>>2]=0;return}e=(e|0)!=0&(g|0)!=0;b=e&0;J[a+4>>2]=b|e;J[a+8>>2]=e?f+h|0:0;}function Em(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0;e=Ua-80|0;Ua=e;d=cb(32);J[e+68>>2]=d;J[e+72>>2]=19;J[e+76>>2]=-2147483616;f=K[5568]|K[5569]<<8|(K[5570]<<16|K[5571]<<24);H[d+15|0]=f;H[d+16|0]=f>>>8;H[d+17|0]=f>>>16;H[d+18|0]=f>>>24;f=K[5565]|K[5566]<<8|(K[5567]<<16|K[5568]<<24);g=K[5561]|K[5562]<<8|(K[5563]<<16|K[5564]<<24);H[d+8|0]=g;H[d+9|0]=g>>>8;H[d+10|0]=g>>>16;H[d+11|0]=g>>>24;H[d+12|0]=f;H[d+13|0]=f>>>8;H[d+14|0]=f>>>16;H[d+15|0]=f>>>24;f=K[5557]|K[5558]<<8|(K[5559]<<16|K[5560]<<24);g=K[5553]|K[5554]<<8|(K[5555]<<16|K[5556]<<24);H[d|0]=g;H[d+1|0]=g>>>8;H[d+2|0]=g>>>16;H[d+3|0]=g>>>24;H[d+4|0]=f;H[d+5|0]=f>>>8;H[d+6|0]=f>>>16;H[d+7|0]=f>>>24;H[d+19|0]=0;a:{b:{c:{d:{e:{f:{g:{h:{i:{j:{k:{l:{m:{n:{o:{p:{d=J[J[b>>2]+80>>2];q:{if(!d){J[5996]=0;_(97,e+68|0,2603)|0;d=J[5996];J[5996]=0;if((d|0)!=1){break q}break b}J[5996]=0;f=_(54,3557,1)|0;g=J[5996];J[5996]=0;if((g|0)==1){break p}if(!((d^-1)&f)){J[5996]=0;aa(98,e+68|0,66);d=J[5996];J[5996]=0;if((d|0)==1){break b}}d=J[J[b>>2]+80>>2];J[5996]=0;f=_(54,2796,1)|0;g=J[5996];J[5996]=0;if((g|0)==1){break o}if(!((d^-1)&f)){J[5996]=0;aa(98,e+68|0,86);d=J[5996];J[5996]=0;if((d|0)==1){break b}}d=J[J[b>>2]+80>>2];J[5996]=0;f=_(54,2777,1)|0;g=J[5996];J[5996]=0;if((g|0)==1){break n}if(!((d^-1)&f)){J[5996]=0;aa(98,e+68|0,87);d=J[5996];J[5996]=0;if((d|0)==1){break b}}d=J[J[b>>2]+80>>2];J[5996]=0;f=_(54,3310,1)|0;g=J[5996];J[5996]=0;if((g|0)==1){break m}if(!((d^-1)&f)){J[5996]=0;aa(98,e+68|0,75);d=J[5996];J[5996]=0;if((d|0)==1){break b}}d=J[J[b>>2]+80>>2];J[5996]=0;f=_(54,1029,1)|0;g=J[5996];J[5996]=0;if((g|0)==1){break l}if(!((d^-1)&f)){J[5996]=0;aa(98,e+68|0,122);d=J[5996];J[5996]=0;if((d|0)==1){break b}}d=J[J[b>>2]+80>>2];J[5996]=0;f=_(54,1959,1)|0;g=J[5996];J[5996]=0;if((g|0)==1){break k}if(!((d^-1)&f)){J[5996]=0;aa(98,e+68|0,111);d=J[5996];J[5996]=0;if((d|0)==1){break b}}d=J[J[b>>2]+80>>2];J[5996]=0;f=_(54,2029,1)|0;g=J[5996];J[5996]=0;if((g|0)==1){break j}if(!((d^-1)&f)){J[5996]=0;aa(98,e+68|0,110);d=J[5996];J[5996]=0;if((d|0)==1){break b}}d=J[J[b>>2]+80>>2];J[5996]=0;f=_(54,2522,1)|0;g=J[5996];J[5996]=0;if((g|0)==1){break i}if(!((d^-1)&f)){J[5996]=0;aa(98,e+68|0,100);d=J[5996];J[5996]=0;if((d|0)==1){break b}}d=J[J[b>>2]+80>>2];J[5996]=0;f=_(54,2315,1)|0;g=J[5996];J[5996]=0;if((g|0)==1){break h}if(!((d^-1)&f)){J[5996]=0;aa(98,e+68|0,102);d=J[5996];J[5996]=0;if((d|0)==1){break b}}d=J[J[b>>2]+80>>2];J[5996]=0;f=_(54,2451,1)|0;g=J[5996];J[5996]=0;if((g|0)==1){break g}if(!((d^-1)&f)){J[5996]=0;aa(98,e+68|0,101);d=J[5996];J[5996]=0;if((d|0)==1){break b}}d=J[J[b>>2]+80>>2];J[5996]=0;f=_(54,2049,1)|0;g=J[5996];J[5996]=0;if((g|0)==1){break f}if(!((d^-1)&f)){J[5996]=0;aa(98,e+68|0,109);d=J[5996];J[5996]=0;if((d|0)==1){break b}}d=J[J[b>>2]+80>>2];J[5996]=0;f=_(54,1295,1)|0;g=J[5996];J[5996]=0;if((g|0)==1){break e}if(!((d^-1)&f)){J[5996]=0;aa(98,e+68|0,117);d=J[5996];J[5996]=0;if((d|0)==1){break b}}d=J[J[b>>2]+80>>2];J[5996]=0;f=_(54,1681,1)|0;g=J[5996];J[5996]=0;if((g|0)==1){break d}if(!((d^-1)&f)){J[5996]=0;aa(98,e+68|0,115);d=J[5996];J[5996]=0;if((d|0)==1){break b}}d=J[J[b>>2]+80>>2];J[5996]=0;f=_(54,2165,1)|0;g=J[5996];J[5996]=0;if((g|0)==1){break c}if((d^-1)&f){break q}J[5996]=0;aa(98,e+68|0,107);d=J[5996];J[5996]=0;if((d|0)==1){break b}}d=J[J[b>>2]+84>>2];J[5996]=0;f=e+44|0;aa(115,f|0,d|0);d=J[5996];J[5996]=0;r:{s:{t:{u:{v:{w:{x:{y:{z:{A:{B:{C:{D:{E:{F:{if((d|0)!=1){J[5996]=0;d=ba(99,f|0,0,5487)|0;f=J[5996];J[5996]=0;if((f|0)==1){break F}J[e- -64>>2]=J[d+8>>2];f=J[d+4>>2];J[e+56>>2]=J[d>>2];J[e+60>>2]=f;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;f=K[e+67|0];g=f<<24>>24<0;f=ba(100,e+68|0,(g?J[e+56>>2]:e+56|0)|0,(g?J[e+60>>2]:f)|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break E}if(H[e+67|0]<0){$a(J[e+56>>2]);}if(H[d+11|0]<0){$a(J[d>>2]);}d=J[b>>2];g=J[d+52>>2];d=J[d+44>>2];J[5996]=0;h=e+32|0;aa(164,h|0,d+g|0);d=J[5996];J[5996]=0;if((d|0)==1){break C}J[5996]=0;d=ba(99,h|0,0,5459)|0;g=J[5996];J[5996]=0;if((g|0)==1){break B}J[e- -64>>2]=J[d+8>>2];g=J[d+4>>2];J[e+56>>2]=J[d>>2];J[e+60>>2]=g;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;h=f;f=K[e+67|0];g=f<<24>>24<0;f=ba(100,h|0,(g?J[e+56>>2]:e+56|0)|0,(g?J[e+60>>2]:f)|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break A}if(H[e+67|0]<0){$a(J[e+56>>2]);}if(H[d+11|0]<0){$a(J[d>>2]);}b=J[J[b>>2]+68>>2];J[5996]=0;d=e+20|0;aa(164,d|0,b+1|0);b=J[5996];J[5996]=0;if((b|0)==1){break y}J[5996]=0;b=ba(99,d|0,0,5524)|0;d=J[5996];J[5996]=0;if((d|0)==1){break x}J[e- -64>>2]=J[b+8>>2];d=J[b+4>>2];J[e+56>>2]=J[b>>2];J[e+60>>2]=d;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;J[5996]=0;h=f;d=K[e+67|0];f=d<<24>>24<0;d=ba(100,h|0,(f?J[e+56>>2]:e+56|0)|0,(f?J[e+60>>2]:d)|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){break w}if(H[e+67|0]<0){$a(J[e+56>>2]);}if(H[b+11|0]<0){$a(J[b>>2]);}J[5996]=0;d=_(97,d|0,3759)|0;b=J[5996];J[5996]=0;if((b|0)==1){break u}J[e+16>>2]=J[d+8>>2];b=J[d+4>>2];J[e+8>>2]=J[d>>2];J[e+12>>2]=b;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;f=J[c>>2];g=J[c+4>>2];b=K[c+11|0];J[5996]=0;h=c;c=b<<24>>24<0;b=ba(100,e+8|0,(c?f:h)|0,(c?g:b)|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break t}J[e- -64>>2]=J[b+8>>2];c=J[b+4>>2];J[e+56>>2]=J[b>>2];J[e+60>>2]=c;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;J[5996]=0;c=_(97,e+56|0,3559)|0;f=J[5996];J[5996]=0;if((f|0)==1){break s}f=J[c+4>>2];J[a>>2]=J[c>>2];J[a+4>>2]=f;J[a+8>>2]=J[c+8>>2];J[c>>2]=0;J[c+4>>2]=0;J[c+8>>2]=0;if(H[b+11|0]<0){$a(J[b>>2]);}if(H[d+11|0]<0){$a(J[d>>2]);}Ua=e+80|0;return}b=Z()|0;break a}b=Z()|0;break D}b=Z()|0;if(H[e+67|0]>=0){break D}$a(J[e+56>>2]);}if(H[e+55|0]>=0){break a}$a(J[e+44>>2]);break a}b=Z()|0;break a}b=Z()|0;break z}b=Z()|0;if(H[e+67|0]>=0){break z}$a(J[e+56>>2]);}if(H[e+43|0]>=0){break a}$a(J[e+32>>2]);break a}b=Z()|0;break a}b=Z()|0;break v}b=Z()|0;if(H[e+67|0]>=0){break v}$a(J[e+56>>2]);}if(H[e+31|0]>=0){break a}$a(J[e+20>>2]);break a}b=Z()|0;break a}b=Z()|0;break r}b=Z()|0;if(H[e+67|0]>=0){break r}$a(J[e+56>>2]);}if(H[e+19|0]>=0){break a}$a(J[e+8>>2]);break a}b=Z()|0;break a}b=Z()|0;break a}b=Z()|0;break a}b=Z()|0;break a}b=Z()|0;break a}b=Z()|0;break a}b=Z()|0;break a}b=Z()|0;break a}b=Z()|0;break a}b=Z()|0;break a}b=Z()|0;break a}b=Z()|0;break a}b=Z()|0;break a}b=Z()|0;break a}b=Z()|0;}if(H[e+79|0]<0){$a(J[e+68>>2]);}da(b|0);B();}function Gg(a,b,c,d,e,f){a=a|0;b=+b;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,w=0,x=0,y=0,z=0;m=Ua-560|0;Ua=m;J[m+44>>2]=0;A(+b);g=v(1)|0;v(0)|0;a:{if((g|0)<0){s=1;x=1222;b=-b;A(+b);g=v(1)|0;v(0)|0;break a}if(e&2048){s=1;x=1225;break a}s=e&1;x=s?1228:1223;z=!s;}b:{if((g&2146435072)==2146435072){g=s+3|0;Bb(a,32,c,g,e&-65537);wb(a,x,s);d=f&32;wb(a,b!=b?d?2020:3082:d?2300:3355,3);Bb(a,32,c,g,e^8192);o=(c|0)<(g|0)?g:c;break b}u=m+16|0;c:{d:{e:{b=Lg(b,m+44|0);b=b+b;if(b!=0){g=J[m+44>>2];J[m+44>>2]=g-1;w=f|32;if((w|0)!=97){break e}break c}w=f|32;if((w|0)==97){break c}j=J[m+44>>2];l=(d|0)<0?6:d;break d}j=g-29|0;J[m+44>>2]=j;b=b*268435456;l=(d|0)<0?6:d;}q=(m+48|0)+((j|0)>=0?288:0)|0;h=q;while(1){if(b<4294967296&b>=0){d=~~b>>>0;}else {d=0;}J[h>>2]=d;h=h+4|0;b=(b-+(d>>>0))*1e9;if(b!=0){continue}break}f:{if((j|0)<=0){d=j;g=h;i=q;break f}i=q;d=j;while(1){k=(d|0)>=29?29:d;g=h-4|0;g:{if(i>>>0>g>>>0){break g}n=0;while(1){d=J[g>>2];o=k&31;y=n;if((k&63)>>>0>=32){n=d<<o;d=0;}else {n=(1<<o)-1&d>>>32-o;d=d<<o;}o=y+d|0;n=n+p|0;n=eo(o,d>>>0>o>>>0?n+1|0:n,1e9);d=co(n,Wa,-1e9,0)+o|0;J[g>>2]=d;g=g-4|0;if(i>>>0<=g>>>0){continue}break}if(!n){break g}i=i-4|0;J[i>>2]=n;}while(1){g=h;if(i>>>0<g>>>0){h=g-4|0;if(!J[h>>2]){continue}}break}d=J[m+44>>2]-k|0;J[m+44>>2]=d;h=g;if((d|0)>0){continue}break}}if((d|0)<0){t=((l+25>>>0)/9|0)+1|0;p=(w|0)==102;while(1){d=0-d|0;o=(d|0)>=9?9:d;h:{if(g>>>0<=i>>>0){h=J[i>>2];break h}n=1e9>>>o|0;k=-1<<o^-1;d=0;h=i;while(1){y=d;d=J[h>>2];J[h>>2]=y+(d>>>o|0);d=P(n,d&k);h=h+4|0;if(h>>>0<g>>>0){continue}break}h=J[i>>2];if(!d){break h}J[g>>2]=d;g=g+4|0;}d=o+J[m+44>>2]|0;J[m+44>>2]=d;i=(!h<<2)+i|0;h=p?q:i;g=g-h>>2>(t|0)?h+(t<<2)|0:g;if((d|0)<0){continue}break}}d=0;i:{if(g>>>0<=i>>>0){break i}d=P(q-i>>2,9);h=10;k=J[i>>2];if(k>>>0<10){break i}while(1){d=d+1|0;h=P(h,10);if(k>>>0>=h>>>0){continue}break}}h=(l-((w|0)!=102?d:0)|0)-((w|0)==103&(l|0)!=0)|0;if((h|0)<(P(g-q>>2,9)-9|0)){n=h+9216|0;k=(n|0)/9|0;j=((((j|0)<0?4:292)+m|0)+(k<<2)|0)-4048|0;h=10;o=n+P(k,-9)|0;if((o|0)<=7){while(1){h=P(h,10);o=o+1|0;if((o|0)!=8){continue}break}}n=J[j>>2];t=(n>>>0)/(h>>>0)|0;p=P(t,h);k=j+4|0;j:{if((n|0)==(p|0)&(k|0)==(g|0)){break j}n=n-p|0;k:{if(!(t&1)){b=9007199254740992;if(!(H[j-4|0]&1)|((h|0)!=1e9|i>>>0>=j>>>0)){break k}}b=9007199254740994;}r=(g|0)==(k|0)?1:1.5;k=h>>>1|0;r=k>>>0>n>>>0?.5:(k|0)==(n|0)?r:1.5;if(!(K[x|0]!=45|z)){r=-r;b=-b;}J[j>>2]=p;if(b+r==b){break j}d=h+p|0;J[j>>2]=d;if(d>>>0>=1e9){while(1){J[j>>2]=0;j=j-4|0;if(j>>>0<i>>>0){i=i-4|0;J[i>>2]=0;}d=J[j>>2]+1|0;J[j>>2]=d;if(d>>>0>999999999){continue}break}}d=P(q-i>>2,9);h=10;k=J[i>>2];if(k>>>0<10){break j}while(1){d=d+1|0;h=P(h,10);if(k>>>0>=h>>>0){continue}break}}h=j+4|0;g=g>>>0>h>>>0?h:g;}while(1){k=g;n=g>>>0<=i>>>0;if(!n){g=g-4|0;if(!J[g>>2]){continue}}break}l:{if((w|0)!=103){j=e&8;break l}h=l?l:1;g=(h|0)>(d|0)&(d|0)>-5;l=(g?d^-1:-1)+h|0;f=(g?-1:-2)+f|0;j=e&8;if(j){break l}g=-9;m:{if(n){break m}j=J[k-4>>2];if(!j){break m}o=10;g=0;if((j>>>0)%10|0){break m}while(1){h=g;g=g+1|0;o=P(o,10);if(!((j>>>0)%(o>>>0)|0)){continue}break}g=h^-1;}h=P(k-q>>2,9);if((f&-33)==70){j=0;g=(g+h|0)-9|0;g=(g|0)>0?g:0;l=(g|0)>(l|0)?l:g;break l}j=0;g=((d+h|0)+g|0)-9|0;g=(g|0)>0?g:0;l=(g|0)>(l|0)?l:g;}o=-1;n=j|l;if(((n?2147483645:2147483646)|0)<(l|0)){break b}p=(((n|0)!=0)+l|0)+1|0;h=f&-33;n:{if((h|0)==70){if((p^2147483647)<(d|0)){break b}g=(d|0)>0?d:0;break n}g=d>>31;g=Jc((g^d)-g|0,0,u);if((u-g|0)<=1){while(1){g=g-1|0;H[g|0]=48;if((u-g|0)<2){continue}break}}t=g-2|0;H[t|0]=f;H[g-1|0]=(d|0)<0?45:43;g=u-t|0;if((g|0)>(p^2147483647)){break b}}d=g+p|0;if((d|0)>(s^2147483647)){break b}p=d+s|0;Bb(a,32,c,p,e);wb(a,x,s);Bb(a,48,c,p,e^65536);o:{p:{q:{if((h|0)==70){f=m+16|0;d=f|8;j=f|9;h=i>>>0>q>>>0?q:i;i=h;while(1){g=Jc(J[i>>2],0,j);r:{if((h|0)!=(i|0)){if(m+16>>>0>=g>>>0){break r}while(1){g=g-1|0;H[g|0]=48;if(m+16>>>0<g>>>0){continue}break}break r}if((g|0)!=(j|0)){break r}H[m+24|0]=48;g=d;}wb(a,g,j-g|0);i=i+4|0;if(q>>>0>=i>>>0){continue}break}if(n){wb(a,4352,1);}if((l|0)<=0|i>>>0>=k>>>0){break q}while(1){g=Jc(J[i>>2],0,j);if(g>>>0>m+16>>>0){while(1){g=g-1|0;H[g|0]=48;if(m+16>>>0<g>>>0){continue}break}}wb(a,g,(l|0)>=9?9:l);g=l-9|0;i=i+4|0;if(k>>>0<=i>>>0){break p}d=(l|0)>9;l=g;if(d){continue}break}break p}s:{if((l|0)<0){break s}q=i>>>0<k>>>0?k:i+4|0;f=m+16|0;d=f|8;k=f|9;h=i;while(1){g=Jc(J[h>>2],0,k);if((k|0)==(g|0)){H[m+24|0]=48;g=d;}t:{if((h|0)!=(i|0)){if(m+16>>>0>=g>>>0){break t}while(1){g=g-1|0;H[g|0]=48;if(m+16>>>0<g>>>0){continue}break}break t}wb(a,g,1);g=g+1|0;if(!(j|l)){break t}wb(a,4352,1);}f=k-g|0;wb(a,g,(f|0)<(l|0)?f:l);l=l-f|0;h=h+4|0;if(q>>>0<=h>>>0){break s}if((l|0)>=0){continue}break}}Bb(a,48,l+18|0,18,0);wb(a,t,u-t|0);break o}g=l;}Bb(a,48,g+9|0,9,0);}Bb(a,32,c,p,e^8192);o=(c|0)<(p|0)?p:c;break b}j=(f<<26>>31&9)+x|0;u:{if(d>>>0>11){break u}g=12-d|0;r=16;while(1){r=r*16;g=g-1|0;if(g){continue}break}if(K[j|0]==45){b=-(r+(-b-r));break u}b=b+r-r;}q=s|2;i=f&32;h=J[m+44>>2];g=h>>31;g=Jc((g^h)-g|0,0,u);if((u|0)==(g|0)){H[m+15|0]=48;g=m+15|0;}l=g-2|0;H[l|0]=f+15;H[g-1|0]=(h|0)<0?45:43;g=e&8;h=m+16|0;while(1){f=h;if(R(b)<2147483648){k=~~b;}else {k=-2147483648;}H[h|0]=i|K[k+7488|0];b=(b-+(k|0))*16;h=f+1|0;if(!(!((d|0)>0|g)&b==0|(h-(m+16|0)|0)!=1)){H[f+1|0]=46;h=f+2|0;}if(b!=0){continue}break}o=-1;g=u-l|0;f=g+q|0;if((2147483645-f|0)<(d|0)){break b}k=f;f=m+16|0;i=h-f|0;d=d?(i-2|0)<(d|0)?d+2|0:i:i;h=k+d|0;Bb(a,32,c,h,e);wb(a,j,q);Bb(a,48,c,h,e^65536);wb(a,f,i);Bb(a,48,d-i|0,0,0);wb(a,l,g);Bb(a,32,c,h,e^8192);o=(c|0)<(h|0)?h:c;}Ua=m+560|0;return o|0}function zn(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0;a:{b:{c:{d:{e:{f:{g:{switch(J[b>>2]){case 1:case 4:case 5:J[a+8>>2]=0;J[a+12>>2]=0;J[a>>2]=1;J[a+4>>2]=0;return;case 2:J[a+8>>2]=1;J[a+12>>2]=1;break a;case 3:J[a+8>>2]=1;J[a+12>>2]=2;J[a>>2]=1;J[a+4>>2]=2;return;case 6:case 7:case 8:case 9:J[a+8>>2]=0;J[a+12>>2]=0;break a;case 23:d=J[b+32>>2];b=J[d+16>>2];c=J[d>>2];e=J[c+76>>2];j=K[c+72|0]!=0;i=j&K[b+64|0]!=0;f=i?J[b+68>>2]+e|0:0;h:{if(!K[c+64|0]){break h}d=J[d+8>>2];if(!K[d+64|0]){break h}c=J[d+68>>2]+J[c+68>>2]|0;f=i?c>>>0>f>>>0?c:f:c;i=1;}d=J[b+76>>2];b=K[b+72|0];J[a>>2]=i;J[a+4>>2]=f;b=j&(b|0)!=0;c=b&0;J[a+8>>2]=b|c;J[a+12>>2]=b?d+e|0:0;return;case 17:b=J[b+32>>2];c=J[b+8>>2];d=J[c+68>>2];b=J[b>>2];f=J[b+68>>2];b=K[b+64|0];e=K[c+64|0];J[a+8>>2]=0;J[a+12>>2]=0;c=d+f|0;b=(b|0)!=0&(e|0)!=0;d=b&0;J[a>>2]=b|d;J[a+4>>2]=b?c:0;return;case 18:c=J[b+32>>2];b=J[c+8>>2];f=J[b+68>>2];d=J[c>>2];e=J[d+68>>2];j=K[d+64|0];g=K[b+64|0];c=J[b+76>>2]+J[d+76>>2]|0;b=K[d+72|0]!=0&K[b+72|0]!=0;d=b&0;J[a+8>>2]=b|d;J[a+12>>2]=b?c:0;b=(j|0)!=0&(g|0)!=0;d=b&0;J[a>>2]=b|d;J[a+4>>2]=b?e+f|0:0;return;case 19:c=J[b+32>>2];b=J[c+8>>2];f=J[b+76>>2];j=K[b+72|0]!=0;d=J[c>>2];i=j&K[d+64|0]!=0;c=i?f+J[d+68>>2]|0:0;e=J[d+76>>2];g=K[d+72|0];if(!(!g|!K[b+64|0])){b=e+J[b+68>>2]|0;c=i?b>>>0>c>>>0?b:c:b;i=1;}J[a>>2]=i;J[a+4>>2]=c;b=j&(g|0)!=0;d=b&0;J[a+8>>2]=b|d;J[a+12>>2]=b?e+f|0:0;return;case 20:b=J[b+32>>2];e=J[b+8>>2];c=J[b>>2];d=J[e+68>>2]+J[c+76>>2]|0;b=K[c+72|0]!=0&K[e+64|0]!=0;e=b?d:0;i:{if(!K[c+64|0]){break i}if(!b){b=J[c+64>>2];f=b&-256;e=J[c+68>>2];break i}b=1;c=J[c+68>>2];e=c>>>0>d>>>0?c:d;}c=e;J[a+8>>2]=0;J[a+12>>2]=0;J[a>>2]=f|b&255;J[a+4>>2]=c|j;return;case 21:b=J[b+32>>2];d=J[b+8>>2];c=J[b>>2];g=J[c+76>>2];e=J[d+68>>2]+g|0;i=K[c+72|0];b=(i|0)!=0&K[d+64|0]!=0;h=b?e:0;j:{if(!K[c+64|0]){break j}if(!b){b=J[c+64>>2];f=b&-256;h=J[c+68>>2];break j}b=1;c=J[c+68>>2];h=c>>>0>e>>>0?c:e;}e=h;c=J[d+76>>2];d=K[d+72|0];h=b&255;b=0;J[a>>2]=h|(f|b);J[a+4>>2]=e|j;e=c+g|0;c=b;b=(i|0)!=0&(d|0)!=0;c=b?c:0;J[a+8>>2]=b|c;J[a+12>>2]=b?e:0;return;case 22:d=J[b+32>>2];b=J[d+8>>2];f=K[b+64|0];c=f?J[b+68>>2]+1|0:0;e=J[d>>2];if(K[e+64|0]){d=J[e+68>>2]+1|0;c=f?c>>>0<d>>>0?d:c:d;f=1;}j=J[b+76>>2]+1|0;b=K[b+72|0];d=b?j:0;if(K[e+72|0]){e=J[e+76>>2]+1|0;d=b?d>>>0<e>>>0?e:d:e;b=1;}j=b;b=0;J[a+8>>2]=j|b;J[a+12>>2]=d;J[a>>2]=b|f;J[a+4>>2]=c;return;case 25:c=J[b+4>>2]+1|0;b=1;J[a+8>>2]=b;J[a+12>>2]=c;break b;case 10:case 11:case 12:case 16:b=J[J[b+32>>2]>>2];c=J[b+68>>2];J[a>>2]=J[b+64>>2];J[a+4>>2]=c;c=J[b+76>>2];J[a+8>>2]=J[b+72>>2];J[a+12>>2]=c;return;case 13:b=J[J[b+32>>2]>>2];c=J[b+68>>2];b=K[b+64|0];J[a+8>>2]=1;J[a+12>>2]=1;j=b;b=!b;J[a>>2]=j|b&0;J[a+4>>2]=b?0:c+1|0;return;case 14:c=J[J[b+32>>2]>>2];b=J[c+64>>2];c=J[c+68>>2];J[a+8>>2]=0;J[a+12>>2]=0;break b;case 15:c=J[J[b+32>>2]>>2];b=J[c+64>>2];c=J[c+68>>2];J[a+8>>2]=1;J[a+12>>2]=1;break b;case 24:c=cb(8);J[c>>2]=1;J[c+4>>2]=0;i=c+8|0;l=J[b+32>>2];o=J[b+36>>2];if((l|0)!=(o|0)){while(1){d=J[l>>2];e=J[d+76>>2];d=K[d+72|0];g=J[c+4>>2];j=c;k=K[c|0];J[5996]=0;f=$(3,8)|0;c=J[5996];J[5996]=0;k:{l:{if((c|0)!=1){c=e+g|0;d=(k|0)!=0&(d|0)!=0;e=d&0;J[f>>2]=d|e;J[f+4>>2]=d?c:0;g=f+8|0;c=i-j|0;k=c>>3;if(c>>>0>=9){break l}d=g;break k}d=Z()|0;c=j;break e}q=k>>>0<=2?2:k;i=1;d=g;while(1){c=J[l>>2];h=j+(i<<3)|0;m=J[c+76>>2]+J[h+4>>2]|0;n=h-8|0;e=K[n|0]!=0&K[c+64|0]!=0;h=K[h|0]!=0&K[c+72|0]!=0;p=h?m:0;c=e?J[c+68>>2]+J[n+4>>2]|0:0;c=h?e?c>>>0<p>>>0?p:c:m:c;m=e|h;m:{if(d>>>0<g>>>0){J[d>>2]=m;J[d+4>>2]=c;e=d;break m}n=d-f>>3;e=n+1|0;if(e>>>0>=536870912){J[5996]=0;ca(93);a=J[5996];J[5996]=0;if((a|0)==1){break f}break c}g=g-f|0;h=g>>2;h=g>>>0>=2147483640?536870911:e>>>0<h>>>0?h:e;n:{if(!h){g=0;break n}if(h>>>0>=536870912){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break c}break f}J[5996]=0;g=$(3,h<<3)|0;e=J[5996];J[5996]=0;if((e|0)==1){break f}}e=(n<<3)+g|0;J[e>>2]=m;J[e+4>>2]=c;g=(h<<3)+g|0;c=e;o:{if((d|0)!=(f|0)){while(1){d=d-8|0;h=J[d+4>>2];c=c-8|0;J[c>>2]=J[d>>2];J[c+4>>2]=h;if((d|0)!=(f|0)){continue}break}if(!f){break o}}$a(f);}f=c;}d=e+8|0;i=i+1|0;if((q|0)!=(i|0)){continue}break}}c=(j+(k<<3)|0)-8|0;e=J[l>>2];i=K[c|0]!=0&K[e+64|0]!=0;e=i?J[e+68>>2]+J[c+4>>2]|0:0;p:{if(d>>>0<g>>>0){J[d>>2]=i;J[d+4>>2]=e;i=d+8|0;c=f;break p}q:{r:{k=d-f>>3;c=k+1|0;s:{if(c>>>0>=536870912){J[5996]=0;ca(93);break s}g=g-f|0;h=g>>2;c=g>>>0>=2147483640?536870911:c>>>0<h>>>0?h:c;if(!c){c=0;break q}if(c>>>0<536870912){break r}J[5996]=0;ca(35);}a=J[5996];J[5996]=0;if((a|0)!=1){break c}break f}J[5996]=0;c=$(3,c<<3)|0;g=J[5996];J[5996]=0;if((g|0)==1){break f}}c=(k<<3)+c|0;J[c>>2]=i;J[c+4>>2]=e;i=c+8|0;if((d|0)!=(f|0)){while(1){d=d-8|0;e=J[d+4>>2];c=c-8|0;J[c>>2]=J[d>>2];J[c+4>>2]=e;if((d|0)!=(f|0)){continue}break}if(!f){break p}}$a(f);}$a(j);l=l+8|0;if((o|0)!=(l|0)){continue}break}}b=J[b+4>>2];if(b>>>0<=i-c>>3>>>0){b=(b<<3)+c|0;d=J[b>>2];b=J[b+4>>2];f=J[c+4>>2];J[a+8>>2]=J[c>>2];J[a+12>>2]=f;J[a>>2]=d;J[a+4>>2]=b;$a(c);return}J[5996]=0;ga(44,5118,2210,850,2317);a=J[5996];J[5996]=0;if((a|0)!=1){break c}d=Z()|0;if(!c){break d}break e;default:la(2350,2210,854,2317);B();case 0:break g}}J[a+8>>2]=1;J[a+12>>2]=0;J[a>>2]=0;J[a+4>>2]=0;return}d=Z()|0;if(!f){c=j;break e}$a(f);c=j;}$a(c);}da(d|0);B();}B();}J[a>>2]=b;J[a+4>>2]=c;return}J[a>>2]=1;J[a+4>>2]=1;}function Kg(a,b,c,d,e){var f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0;h=Ua-80|0;Ua=h;J[h+76>>2]=b;x=h+55|0;q=h+56|0;a:{b:{c:{d:while(1){f=0;e:while(1){k=b;if((p^2147483647)<(f|0)){break c}p=f+p|0;f:{g:{h:{f=b;g=K[f|0];if(g){while(1){i:{b=g&255;j:{if(!b){b=f;break j}if((b|0)!=37){break i}g=f;while(1){if(K[g+1|0]!=37){b=g;break j}f=f+1|0;l=K[g+2|0];b=g+2|0;g=b;if((l|0)==37){continue}break}}f=f-k|0;w=p^2147483647;if((f|0)>(w|0)){break c}if(a){wb(a,k,f);}if(f){continue e}J[h+76>>2]=b;f=b+1|0;o=-1;g=H[b+1|0];if(!(K[b+2|0]!=36|g-48>>>0>=10)){o=g-48|0;r=1;f=b+3|0;}J[h+76>>2]=f;m=0;g=H[f|0];b=g-32|0;k:{if(b>>>0>31){l=f;break k}l=f;b=1<<b;if(!(b&75913)){break k}while(1){l=f+1|0;J[h+76>>2]=l;m=b|m;g=H[f+1|0];b=g-32|0;if(b>>>0>=32){break k}f=l;b=1<<b;if(b&75913){continue}break}}l:{if((g|0)==42){b=H[l+1|0];m:{if(!(K[l+2|0]!=36|b-48>>>0>=10)){b=b-48|0;n:{if(!a){J[(b<<2)+e>>2]=10;n=0;break n}n=J[(b<<3)+d>>2];}b=l+3|0;r=1;break m}if(r){break h}b=l+1|0;if(!a){J[h+76>>2]=b;r=0;n=0;break l}f=J[c>>2];J[c>>2]=f+4;n=J[f>>2];r=0;}J[h+76>>2]=b;if((n|0)>=0){break l}n=0-n|0;m=m|8192;break l}n=Jg(h+76|0);if((n|0)<0){break c}b=J[h+76>>2];}f=0;i=-1;g=0;o:{if(K[b|0]!=46){break o}if(K[b+1|0]==42){g=H[b+2|0];p:{if(!(K[b+3|0]!=36|g-48>>>0>=10)){g=g-48|0;b=b+4|0;q:{if(!a){J[(g<<2)+e>>2]=10;i=0;break q}i=J[(g<<3)+d>>2];}break p}if(r){break h}b=b+2|0;i=0;if(!a){break p}g=J[c>>2];J[c>>2]=g+4;i=J[g>>2];}J[h+76>>2]=b;g=(i^-1)>>>31|0;break o}J[h+76>>2]=b+1;i=Jg(h+76|0);b=J[h+76>>2];g=1;}t=g;while(1){s=f;j=28;u=b;g=H[b|0];if(g-123>>>0<4294967238){break b}b=b+1|0;f=K[(g+P(f,58)|0)+6959|0];if(f-1>>>0<8){continue}break}J[h+76>>2]=b;r:{if((f|0)!=27){if(!f){break b}if((o|0)>=0){if(!a){J[(o<<2)+e>>2]=f;continue d}f=(o<<3)+d|0;g=J[f+4>>2];J[h+64>>2]=J[f>>2];J[h+68>>2]=g;break r}if(!a){break f}Ig(h- -64|0,f,c);break r}if((o|0)>=0){break b}f=0;if(!a){continue e}}j=-1;if(K[a|0]&32){break a}g=m&-65537;m=m&8192?g:m;o=0;v=1212;l=q;s:{t:{u:{v:{w:{x:{y:{z:{A:{B:{C:{D:{E:{F:{G:{H:{f=H[u|0];f=s?(f&15)==3?f&-33:f:f;switch(f-88|0){case 11:break s;case 9:case 13:case 14:case 15:break t;case 27:break y;case 12:case 17:break B;case 23:break C;case 0:case 32:break D;case 24:break E;case 22:break F;case 29:break G;case 1:case 2:case 3:case 4:case 5:case 6:case 7:case 8:case 10:case 16:case 18:case 19:case 20:case 21:case 25:case 26:case 28:case 30:case 31:break g;default:break H}}I:{switch(f-65|0){case 0:case 4:case 5:case 6:break t;case 2:break w;case 1:case 3:break g;default:break I}}if((f|0)==83){break x}break g}g=J[h+64>>2];j=J[h+68>>2];f=1212;break A}f=0;J:{switch(s&255){case 0:J[J[h+64>>2]>>2]=p;continue e;case 1:J[J[h+64>>2]>>2]=p;continue e;case 2:g=J[h+64>>2];J[g>>2]=p;J[g+4>>2]=p>>31;continue e;case 3:I[J[h+64>>2]>>1]=p;continue e;case 4:H[J[h+64>>2]]=p;continue e;case 6:J[J[h+64>>2]>>2]=p;continue e;case 7:break J;default:continue e}}g=J[h+64>>2];J[g>>2]=p;J[g+4>>2]=p>>31;continue e}i=i>>>0<=8?8:i;m=m|8;f=120;}k=q;u=f&32;g=J[h+64>>2];j=J[h+68>>2];if(g|j){while(1){k=k-1|0;H[k|0]=u|K[(g&15)+7488|0];s=!j&g>>>0>15|(j|0)!=0;g=(j&15)<<28|g>>>4;j=j>>>4|0;if(s){continue}break}}if(!(J[h+64>>2]|J[h+68>>2])|!(m&8)){break z}v=(f>>>4|0)+1212|0;o=2;break z}f=q;k=J[h+68>>2];j=k;g=J[h+64>>2];if(k|g){while(1){f=f-1|0;H[f|0]=g&7|48;s=!j&g>>>0>7|(j|0)!=0;g=(j&7)<<29|g>>>3;j=j>>>3|0;if(s){continue}break}}k=f;if(!(m&8)){break z}f=q-f|0;i=(f|0)<(i|0)?i:f+1|0;break z}g=J[h+64>>2];f=J[h+68>>2];j=f;if((f|0)<0){k=0-(f+((g|0)!=0)|0)|0;j=k;g=0-g|0;J[h+64>>2]=g;J[h+68>>2]=k;o=1;f=1212;break A}if(m&2048){o=1;f=1213;break A}o=m&1;f=o?1214:1212;}v=f;k=Jc(g,j,q);}if((i|0)<0?t:0){break c}m=t?m&-65537:m;f=J[h+64>>2];g=J[h+68>>2];if(!(i|(f|g)!=0)){k=q;i=0;break g}f=!(f|g)+(q-k|0)|0;i=(f|0)<(i|0)?i:f;break g}f=J[h+64>>2];k=f?f:4806;f=k;l=i>>>0>=2147483647?2147483647:i;j=Zb(f,0,l);f=j?j-f|0:l;l=f+k|0;if((i|0)>=0){m=g;i=f;break g}m=g;i=f;if(K[l|0]){break c}break g}if(i){g=J[h+64>>2];break v}f=0;Bb(a,32,n,0,m);break u}J[h+12>>2]=0;J[h+8>>2]=J[h+64>>2];g=h+8|0;J[h+64>>2]=g;i=-1;}f=0;K:{while(1){k=J[g>>2];if(!k){break K}k=Mg(h+4|0,k);l=(k|0)<0;if(!(l|k>>>0>i-f>>>0)){g=g+4|0;f=f+k|0;if(i>>>0>f>>>0){continue}break K}break}if(l){break a}}j=61;if((f|0)<0){break b}Bb(a,32,n,f,m);if(!f){f=0;break u}l=0;g=J[h+64>>2];while(1){k=J[g>>2];if(!k){break u}j=h+4|0;k=Mg(j,k);l=k+l|0;if(l>>>0>f>>>0){break u}wb(a,j,k);g=g+4|0;if(f>>>0>l>>>0){continue}break}}Bb(a,32,n,f,m^8192);f=(f|0)<(n|0)?n:f;continue e}if((i|0)<0?t:0){break c}j=61;f=Gg(a,O[h+64>>3],n,i,m,f);if((f|0)>=0){continue e}break b}H[h+55|0]=J[h+64>>2];i=1;k=x;m=g;break g}g=K[f+1|0];f=f+1|0;continue}}j=p;if(a){break a}if(!r){break f}f=1;while(1){a=J[(f<<2)+e>>2];if(a){Ig((f<<3)+d|0,a,c);j=1;f=f+1|0;if((f|0)!=10){continue}break a}break}j=1;if(f>>>0>=10){break a}while(1){if(J[(f<<2)+e>>2]){break h}f=f+1|0;if((f|0)!=10){continue}break}break a}j=28;break b}l=l-k|0;i=(i|0)>(l|0)?i:l;if((i|0)>(o^2147483647)){break c}j=61;g=i+o|0;f=(g|0)<(n|0)?n:g;if((w|0)<(f|0)){break b}Bb(a,32,f,g,m);wb(a,v,o);Bb(a,48,f,g,m^65536);Bb(a,48,i,l,0);wb(a,k,l);Bb(a,32,f,g,m^8192);continue}break}break}j=0;break a}j=61;}J[5732]=j;j=-1;}Ua=h+80|0;return j}function un(a){a=a|0;var b=0,c=0,d=0,e=0,f=0;a:{if(a>>>0<=211){d=J[Eg(7504,7696,a)>>2];break a}if(a>>>0>=4294967292){a=Rb(8);J[5996]=0;d=$(167,a|0)|0;e=J[5996];J[5996]=0;if((e|0)!=1){na(d|0,22716,92);B();}d=Z()|0;ic(a);da(d|0);B();}f=(a>>>0)/210|0;d=P(f,210);e=Eg(7696,7888,a-d|0)-7696>>2;while(1){d=J[(e<<2)+7696>>2]+d|0;a=5;while(1){b:{if((a|0)==47){a=211;while(1){b=(d>>>0)/(a>>>0)|0;if(b>>>0<a>>>0){break a}if((P(a,b)|0)==(d|0)){break b}b=a+10|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+12|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+16|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+18|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+22|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+28|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+30|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+36|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+40|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+42|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+46|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+52|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+58|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+60|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+66|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+70|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+72|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+78|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+82|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+88|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+96|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+100|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+102|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+106|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+108|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+112|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+120|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+126|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+130|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+136|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+138|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+142|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+148|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+150|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+156|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+162|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+166|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+168|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+172|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+178|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+180|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+186|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+190|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+192|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+196|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+198|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}if((P(b,c)|0)==(d|0)){break b}b=a+208|0;c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}a=a+210|0;if((P(b,c)|0)!=(d|0)){continue}break}break b}b=J[(a<<2)+7504>>2];c=(d>>>0)/(b>>>0)|0;if(b>>>0>c>>>0){break a}a=a+1|0;if((P(b,c)|0)!=(d|0)){continue}}break}d=e+1|0;a=(d|0)==48;e=a?0:d;f=a+f|0;d=P(f,210);continue}}return d|0}function ob(a,b,c,d,e,f,g,h,i){var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0,y=0,z=0,A=0,B=0,C=0,D=0,E=0,F=0,G=0,H=0,I=0,K=0,L=0,M=0,N=0,O=0,P=0,Q=0,R=0;k=Ua-96|0;Ua=k;u=i&65535;o=(e^i)&-2147483648;r=e&65535;y=r;Q=i>>>16&32767;R=e>>>16&32767;a:{b:{if(Q-32767>>>0>4294934529&R-32767>>>0>=4294934530){break b}s=e&2147483647;q=s;j=d;if(!(!d&(q|0)==2147418112?!(b|c):q>>>0<2147418112)){m=d;o=e|32768;break a}s=i&2147483647;p=s;e=h;if(!(!e&(p|0)==2147418112?!(f|g):p>>>0<2147418112)){m=e;o=i|32768;b=f;c=g;break a}if(!(b|j|(q^2147418112|c))){if(!(e|f|(g|p))){o=2147450880;b=0;c=0;break a}o=o|2147418112;b=0;c=0;break a}if(!(e|f|(p^2147418112|g))){e=b|j;d=c|q;b=0;c=0;if(!(d|e)){o=2147450880;break a}o=o|2147418112;break a}if(!(b|j|(c|q))){b=0;c=0;break a}if(!(e|f|(g|p))){b=0;c=0;break a}if((q|0)==65535|q>>>0<65535){j=!(d|r);i=j?b:d;s=j<<6;e=S(i)+32|0;i=S(j?c:r);i=s+((i|0)==32?e:i)|0;Cb(k+80|0,b,c,d,r,i-15|0);t=16-i|0;d=J[k+88>>2];y=J[k+92>>2];c=J[k+84>>2];b=J[k+80>>2];}if(p>>>0>65535){break b}i=!(h|u);e=i?f:h;r=i<<6;j=S(e)+32|0;e=S(i?g:u);e=r+((e|0)==32?j:e)|0;Cb(k- -64|0,f,g,h,u,e-15|0);t=(t-e|0)+16|0;h=J[k+72>>2];u=J[k+76>>2];f=J[k+64>>2];g=J[k+68>>2];}e=f;f=g<<15|f>>>17;i=e<<15;e=0;v=i&-32768;K=c;s=co(v,e,c,0);e=Wa;z=e;L=f;q=b;c=co(f,0,b,0);b=c+s|0;f=Wa+e|0;i=b;c=b>>>0<c>>>0?f+1|0:f;f=b;j=0;b=co(q,l,v,l);p=j+b|0;e=Wa+f|0;e=b>>>0>p>>>0?e+1|0:e;r=e;M=(f|0)==(e|0)&j>>>0>p>>>0|e>>>0<f>>>0;N=d;w=co(v,l,d,0);O=Wa;b=co(K,l,L,l);x=b+w|0;j=Wa+O|0;j=b>>>0>x>>>0?j+1|0:j;b=u<<15|h>>>17;A=h<<15|g>>>17;d=co(A,0,q,l);B=d+x|0;f=Wa+j|0;f=d>>>0>B>>>0?f+1|0:f;C=f;d=(c|0)==(z|0)&i>>>0<s>>>0|c>>>0<z>>>0;D=c+B|0;f=d+f|0;f=c>>>0>D>>>0?f+1|0:f;u=f;g=D;c=f;E=y|65536;y=co(v,l,E,n);P=Wa;d=co(N,m,L,l);F=d+y|0;e=Wa+P|0;e=d>>>0>F>>>0?e+1|0:e;i=e;G=b|-2147483648;b=co(G,0,q,l);H=b+F|0;f=Wa+e|0;f=b>>>0>H>>>0?f+1|0:f;d=co(A,m,K,l);b=d+H|0;I=f;f=f+Wa|0;z=b;s=b>>>0<d>>>0?f+1|0:f;f=b;b=0;q=b+g|0;e=c+f|0;e=b>>>0>q>>>0?e+1|0:e;v=e;f=e;b=q+M|0;c=b>>>0<q>>>0?f+1|0:f;t=((R+Q|0)+t|0)-16383|0;d=co(G,m,K,l);h=Wa;e=co(E,m,L,l);g=e+d|0;f=Wa+h|0;f=e>>>0>g>>>0?f+1|0:f;n=(h|0)==(f|0)&d>>>0>g>>>0|f>>>0<h>>>0;h=f;e=co(A,m,N,m);d=e+g|0;f=Wa+f|0;f=d>>>0<e>>>0?f+1|0:f;e=f;g=(f|0)==(h|0)&d>>>0<g>>>0|f>>>0<h>>>0;f=0;h=g;g=g+n|0;f=h>>>0>g>>>0?1:f;h=g;g=co(G,m,E,m);h=h+g|0;f=Wa+f|0;M=h;g=g>>>0>h>>>0?f+1|0:f;h=d;l=e;e=(j|0)==(O|0)&w>>>0>x>>>0|j>>>0<O>>>0;f=0;j=(j|0)==(C|0)&x>>>0>B>>>0|j>>>0>C>>>0;n=j;j=e+j|0;e=(n>>>0>j>>>0?1:f)+l|0;f=g;d=d+j|0;e=d>>>0<j>>>0?e+1|0:e;n=e;w=d;d=(e|0)==(l|0)&d>>>0<h>>>0|e>>>0<l>>>0;e=d;d=d+M|0;f=e>>>0>d>>>0?f+1|0:f;j=d;h=f;e=co(A,m,E,m);l=Wa;g=co(G,m,N,m);d=g+e|0;f=Wa+l|0;f=d>>>0<g>>>0?f+1|0:f;g=f;f=(l|0)==(f|0)&d>>>0<e>>>0|f>>>0<l>>>0;x=g+j|0;e=f+h|0;e=g>>>0>x>>>0?e+1|0:e;l=x;h=e;j=d;e=0;d=e+w|0;f=j+n|0;f=d>>>0<e>>>0?f+1|0:f;e=f;g=(n|0)==(f|0)&d>>>0<w>>>0|f>>>0<n>>>0;f=h;h=g;g=g+l|0;f=h>>>0>g>>>0?f+1|0:f;w=g;g=f;h=d;j=e;e=(s|0)==(I|0)&z>>>0<H>>>0|s>>>0<I>>>0;d=(i|0)==(P|0)&y>>>0>F>>>0|i>>>0<P>>>0;i=(i|0)==(I|0)&F>>>0>H>>>0|i>>>0>I>>>0;d=d+i|0;d=d+e|0;n=s;i=n+h|0;e=d+j|0;f=g;d=i;e=d>>>0<n>>>0?e+1|0:e;h=(e|0)==(j|0)&h>>>0>d>>>0|e>>>0<j>>>0;g=h+w|0;f=h>>>0>g>>>0?f+1|0:f;j=g;g=f;h=d;f=0;i=e;n=(u|0)==(v|0)&q>>>0<D>>>0|u>>>0>v>>>0;l=n;n=n+((u|0)==(C|0)&B>>>0>D>>>0|u>>>0<C>>>0)|0;e=e+(l>>>0>n>>>0?1:f)|0;f=g;d=d+n|0;e=d>>>0<n>>>0?e+1|0:e;h=(e|0)==(i|0)&d>>>0<h>>>0|e>>>0<i>>>0;g=h;h=h+j|0;f=g>>>0>h>>>0?f+1|0:f;i=f;c:{if(f&65536){t=t+1|0;break c}j=r>>>31|0;g=0;f=i<<1|h>>>31;h=h<<1|e>>>31;i=f;f=e<<1|d>>>31;d=d<<1|c>>>31;e=f;f=r<<1|p>>>31;p=p<<1;r=f;f=c<<1|b>>>31;b=b<<1|j;c=f|g;}if((t|0)>=32767){o=o|2147418112;b=0;c=0;break a}d:{if((t|0)<=0){g=1-t|0;if(g>>>0<=127){f=t+127|0;Cb(k+48|0,p,r,b,c,f);Cb(k+32|0,d,e,h,i,f);xc(k+16|0,p,r,b,c,g);xc(k,d,e,h,i,g);p=J[k+32>>2]|J[k+16>>2]|(J[k+48>>2]|J[k+56>>2]|(J[k+52>>2]|J[k+60>>2]))!=0;r=J[k+36>>2]|J[k+20>>2];b=J[k+40>>2]|J[k+24>>2];c=J[k+44>>2]|J[k+28>>2];d=J[k>>2];e=J[k+4>>2];g=J[k+8>>2];f=J[k+12>>2];break d}b=0;c=0;break a}g=h;f=i&65535|t<<16;}m=g|m;o=f|o;if(!(!b&(c|0)==-2147483648?!(p|r):(c|0)>0|(c|0)>=0)){b=d+1|0;j=b?e:e+1|0;e=o;c=j;d=!(j|b);m=d+m|0;o=d>>>0>m>>>0?e+1|0:e;break a}if(b|p|(c^-2147483648|r)){b=d;c=e;break a}f=e;b=d&1;c=b;b=b+d|0;c=c>>>0>b>>>0?f+1|0:f;d=(f|0)==(c|0)&b>>>0<d>>>0|c>>>0<f>>>0;f=o;m=d+m|0;o=m>>>0<d>>>0?f+1|0:f;}J[a>>2]=b;J[a+4>>2]=c;J[a+8>>2]=m;J[a+12>>2]=o;Ua=k+96|0;}function Pg(){var a=0,b=0,c=0;if(K[25020]){return J[6254]}J[6663]=0;J[6662]=22008;J[6662]=14144;J[6662]=12664;a=Ua-16|0;Ua=a;J[6664]=0;J[6665]=0;J[6666]=0;H[26792]=0;J[a+4>>2]=26656;c=J[a+4>>2];b=a+8|0;H[b+4|0]=0;J[b>>2]=c;J[5996]=0;ha(273,26656);c=J[5996];J[5996]=0;a:{if((c|0)!=1){ff(26656,30);H[b+4|0]=1;ef(b);Ua=a+16|0;break a}b=Z()|0;ef(a+8|0);da(b|0);B();}J[5996]=0;b=_(30,26800,3538)|0;a=J[5996];J[5996]=0;b:{c:{d:{if((a|0)!=1){df(26656);J[6581]=0;J[6580]=22008;J[6580]=14144;J[6580]=15252;J[5996]=0;ha(240,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6583]=0;J[6582]=22008;J[6582]=14144;J[6582]=15284;J[5996]=0;ha(241,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6585]=0;J[6584]=22008;J[6584]=14144;H[26348]=0;J[6586]=0;J[6584]=12684;J[6586]=12736;J[5996]=0;ha(242,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6589]=0;J[6588]=22008;J[6588]=14144;J[6588]=14200;J[5996]=0;ha(243,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6591]=0;J[6590]=22008;J[6590]=14144;J[6590]=14348;J[5996]=0;ha(244,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[5996]=0;ca(245);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[5996]=0;ha(246,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6597]=0;J[6596]=22008;J[6596]=14144;J[6596]=14496;J[5996]=0;ha(247,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6599]=0;J[6598]=22008;J[6598]=14144;J[6598]=14728;J[5996]=0;ha(248,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6601]=0;J[6600]=22008;J[6600]=14144;J[6600]=14612;J[5996]=0;ha(249,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6603]=0;J[6602]=22008;J[6602]=14144;J[6602]=14844;J[5996]=0;ha(250,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6605]=0;J[6604]=22008;J[6604]=14144;I[13212]=11310;J[6604]=13816;J[6607]=0;J[6608]=0;J[6609]=0;J[5996]=0;ha(251,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6611]=0;J[6610]=22008;J[6610]=14144;J[6612]=46;J[6613]=44;J[6610]=13856;J[6614]=0;J[6615]=0;J[6616]=0;J[5996]=0;ha(252,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6619]=0;J[6618]=22008;J[6618]=14144;J[6618]=15316;J[5996]=0;ha(253,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6621]=0;J[6620]=22008;J[6620]=14144;J[6620]=15560;J[5996]=0;ha(254,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6623]=0;J[6622]=22008;J[6622]=14144;J[6622]=15772;J[5996]=0;ha(255,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6625]=0;J[6624]=22008;J[6624]=14144;J[6624]=16004;J[5996]=0;ha(256,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6627]=0;J[6626]=22008;J[6626]=14144;J[6626]=16988;J[5996]=0;ha(257,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6629]=0;J[6628]=22008;J[6628]=14144;J[6628]=17136;J[5996]=0;ha(258,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6631]=0;J[6630]=22008;J[6630]=14144;J[6630]=17252;J[5996]=0;ha(259,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6633]=0;J[6632]=22008;J[6632]=14144;J[6632]=17368;J[5996]=0;ha(260,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6635]=0;J[6634]=22008;J[6634]=14144;J[6634]=17484;J[5996]=0;ha(261,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6637]=0;J[6636]=22008;J[6636]=14144;J[6636]=17648;J[5996]=0;ha(262,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6639]=0;J[6638]=22008;J[6638]=14144;J[6638]=17812;J[5996]=0;ha(263,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6641]=0;J[6640]=22008;J[6640]=14144;J[6640]=17976;J[5996]=0;ha(264,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6643]=0;J[6642]=22008;J[6642]=14144;J[6644]=19232;J[6644]=16252;J[6642]=16204;J[5996]=0;ha(265,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6647]=0;J[6646]=22008;J[6646]=14144;J[6648]=19268;J[6648]=16516;J[6646]=16468;J[5996]=0;ha(266,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[5996]=0;ca(267);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[5996]=0;ha(268,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[5996]=0;ca(269);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[5996]=0;ha(270,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6659]=0;J[6658]=22008;J[6658]=14144;J[6658]=18140;J[5996]=0;ha(271,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}J[6661]=0;J[6660]=22008;J[6660]=14144;J[6660]=18260;J[5996]=0;ha(272,26648);a=J[5996];J[5996]=0;if((a|0)==1){break d}break b}a=Z()|0;break c}a=Z()|0;ab(b);}gf(26656);da(a|0);B();}J[6253]=26648;H[25020]=1;J[6254]=25012;return 25012}function Ek(a,b,c,d,e,f,g,h){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;var i=0,j=0,k=0;h=Ua-48|0;Ua=h;J[h+44>>2]=b;J[e>>2]=0;i=J[d+28>>2];J[h>>2]=i;J[i+4>>2]=J[i+4>>2]+1;J[5996]=0;i=$(197,J[h>>2])|0;j=J[5996];J[5996]=0;a:{b:{c:{d:{e:{f:{g:{h:{i:{j:{k:{l:{m:{n:{o:{p:{q:{r:{s:{t:{u:{v:{w:{x:{y:{z:{A:{if((j|0)!=1){j=J[h>>2];k=J[j+4>>2]-1|0;J[j+4>>2]=k;if((k|0)==-1){Xa[J[J[j>>2]+8>>2]](j);}switch(g-65|0){case 54:break j;case 19:break k;case 18:break l;case 17:break m;case 49:break n;case 47:break o;case 45:case 51:break p;case 12:break q;case 44:break r;case 41:break s;case 8:break t;case 7:break u;case 5:break v;case 3:break w;case 35:case 36:break x;case 34:break y;case 1:case 33:case 39:break z;case 0:case 32:break A;case 2:case 4:case 6:case 9:case 10:case 11:case 13:case 14:case 15:case 16:case 20:case 21:case 22:case 25:case 26:case 27:case 28:case 29:case 30:case 31:case 37:case 38:case 40:case 42:case 43:case 46:case 48:case 50:case 52:case 53:break d;case 24:break f;case 56:break g;case 23:break h;case 55:break i;default:break e}}c=Z()|0;a=J[h>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(c|0);B();}Bf(a,f+24|0,h+44|0,c,e,i);break b}Af(a,f+16|0,h+44|0,c,e,i);break b}g=Xa[J[J[a+8>>2]+12>>2]](a+8|0)|0;i=K[g+11|0];j=a;k=b;b=(i&128)>>>7|0;a=b?J[g>>2]:g;J[h+44>>2]=jc(j,k,c,d,e,f,a,a+((b?J[g+4>>2]:i&127)<<2)|0);break b}a=bc(h+44|0,c,e,i,2);b=J[e>>2];B:{if(!(a-1>>>0>30|b&4)){J[f+12>>2]=a;break B}J[e>>2]=b|4;}break b}g=J[3127];J[h+24>>2]=J[3126];J[h+28>>2]=g;g=J[3125];J[h+16>>2]=J[3124];J[h+20>>2]=g;g=J[3123];J[h+8>>2]=J[3122];J[h+12>>2]=g;g=J[3121];J[h>>2]=J[3120];J[h+4>>2]=g;J[h+44>>2]=jc(a,b,c,d,e,f,h,h+32|0);break b}g=J[3135];J[h+24>>2]=J[3134];J[h+28>>2]=g;g=J[3133];J[h+16>>2]=J[3132];J[h+20>>2]=g;g=J[3131];J[h+8>>2]=J[3130];J[h+12>>2]=g;g=J[3129];J[h>>2]=J[3128];J[h+4>>2]=g;J[h+44>>2]=jc(a,b,c,d,e,f,h,h+32|0);break b}a=bc(h+44|0,c,e,i,2);b=J[e>>2];C:{if(!((a|0)>23|b&4)){J[f+8>>2]=a;break C}J[e>>2]=b|4;}break b}a=bc(h+44|0,c,e,i,2);b=J[e>>2];D:{if(!(a-1>>>0>11|b&4)){J[f+8>>2]=a;break D}J[e>>2]=b|4;}break b}a=bc(h+44|0,c,e,i,3);b=J[e>>2];E:{if(!((a|0)>365|b&4)){J[f+28>>2]=a;break E}J[e>>2]=b|4;}break b}b=bc(h+44|0,c,e,i,2)-1|0;a=J[e>>2];F:{if(!(b>>>0>11|a&4)){J[f+16>>2]=b;break F}J[e>>2]=a|4;}break b}a=bc(h+44|0,c,e,i,2);b=J[e>>2];G:{if(!((a|0)>59|b&4)){J[f+4>>2]=a;break G}J[e>>2]=b|4;}break b}a=h+44|0;b=Ua-16|0;Ua=b;J[b+12>>2]=c;while(1){H:{if(Tb(a,b+12|0)){break H}c=J[a>>2];d=J[c+12>>2];I:{if((d|0)==J[c+16>>2]){c=Xa[J[J[c>>2]+36>>2]](c)|0;break I}c=J[d>>2];}if(!(Xa[J[J[i>>2]+12>>2]](i,1,c)|0)){break H}Hc(a);continue}break}if(Tb(a,b+12|0)){J[e>>2]=J[e>>2]|2;}Ua=b+16|0;break b}d=h+44|0;a=Xa[J[J[a+8>>2]+8>>2]](a+8|0)|0;b=K[a+11|0];g=(b&128)>>>7|0?J[a+4>>2]:b&127;b=K[a+23|0];J:{if((g|0)==(0-((b&128)>>>7|0?J[a+16>>2]:b&127)|0)){J[e>>2]=J[e>>2]|4;break J}c=ld(d,c,a,a+24|0,i,e,0);b=J[f+8>>2];if(!((c|0)!=(a|0)|(b|0)!=12)){J[f+8>>2]=0;break J}if(!((c-a|0)!=12|(b|0)>11)){J[f+8>>2]=b+12;}}break b}g=fb(h,12544,44);J[g+44>>2]=jc(a,b,c,d,e,f,g,g+44|0);break b}J[h+16>>2]=J[3152];g=J[3151];J[h+8>>2]=J[3150];J[h+12>>2]=g;g=J[3149];J[h>>2]=J[3148];J[h+4>>2]=g;J[h+44>>2]=jc(a,b,c,d,e,f,h,h+20|0);break b}a=bc(h+44|0,c,e,i,2);b=J[e>>2];K:{if(!((a|0)>60|b&4)){J[f>>2]=a;break K}J[e>>2]=b|4;}break b}g=J[3163];J[h+24>>2]=J[3162];J[h+28>>2]=g;g=J[3161];J[h+16>>2]=J[3160];J[h+20>>2]=g;g=J[3159];J[h+8>>2]=J[3158];J[h+12>>2]=g;g=J[3157];J[h>>2]=J[3156];J[h+4>>2]=g;J[h+44>>2]=jc(a,b,c,d,e,f,h,h+32|0);break b}a=bc(h+44|0,c,e,i,1);b=J[e>>2];L:{if(!((a|0)>6|b&4)){J[f+24>>2]=a;break L}J[e>>2]=b|4;}break b}a=Xa[J[J[a>>2]+20>>2]](a,b,c,d,e,f)|0;break a}g=Xa[J[J[a+8>>2]+24>>2]](a+8|0)|0;i=K[g+11|0];j=a;k=b;b=(i&128)>>>7|0;a=b?J[g>>2]:g;J[h+44>>2]=jc(j,k,c,d,e,f,a,a+((b?J[g+4>>2]:i&127)<<2)|0);break b}zf(f+20|0,h+44|0,c,e,i);break b}a=bc(h+44|0,c,e,i,4);if(!(K[e|0]&4)){J[f+20>>2]=a-1900;}break b}if((g|0)==37){break c}}J[e>>2]=J[e>>2]|4;break b}a=Ua-16|0;Ua=a;J[a+12>>2]=c;b=6;d=h+44|0;g=a+12|0;M:{N:{if(Tb(d,g)){break N}b=4;c=J[d>>2];f=J[c+12>>2];O:{if((f|0)==J[c+16>>2]){c=Xa[J[J[c>>2]+36>>2]](c)|0;break O}c=J[f>>2];}if((Xa[J[J[i>>2]+52>>2]](i,c,0)|0)!=37){break N}b=2;if(!Tb(Hc(d),g)){break M}}J[e>>2]=J[e>>2]|b;}Ua=a+16|0;}a=J[h+44>>2];}Ua=h+48|0;return a|0}function qh(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0,h=0,i=0;g=J[b+4>>2]-J[b>>2]|0;a:{if(g>>>0<=75){c=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);d=c-29|0;e=c>>>0<29?c:d;f=e+1|0;if(f>>>0>(c>>>0<=28?28:K[a+4|0]|K[a+5|0]<<8|(K[a+6|0]<<16|K[a+7|0]<<24))>>>0){Ob(a,f+(f>>>1|0)|0);c=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);d=c-29|0;}h=c>>>0<29;f=e+(h?a+4|0:K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24))|0;eb(f+1|0,f,(h?c:d)-e|0);c=(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))+1|0;H[a|0]=c;H[a+1|0]=c>>>8;H[a+2|0]=c>>>16;H[a+3|0]=c>>>24;H[f|0]=g;break a}if(g>>>0<=255){c=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);d=c-29|0;e=c>>>0<29?c:d;f=e+1|0;if(f>>>0>(c>>>0<=28?28:K[a+4|0]|K[a+5|0]<<8|(K[a+6|0]<<16|K[a+7|0]<<24))>>>0){Ob(a,f+(f>>>1|0)|0);c=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);d=c-29|0;}h=a+4|0;g=c>>>0<29;f=e+(g?h:K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24))|0;eb(f+1|0,f,(g?c:d)-e|0);c=(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))+1|0;H[a|0]=c;H[a+1|0]=c>>>8;H[a+2|0]=c>>>16;H[a+3|0]=c>>>24;H[f|0]=76;g=J[b+4>>2]-J[b>>2]|0;c=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);d=c-29|0;e=c>>>0<29?c:d;f=e+1|0;if(f>>>0>(c>>>0<=28?28:K[a+4|0]|K[a+5|0]<<8|(K[a+6|0]<<16|K[a+7|0]<<24))>>>0){Ob(a,f+(f>>>1|0)|0);c=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);d=c-29|0;}i=h;h=c>>>0<29;f=e+(h?i:K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24))|0;eb(f+1|0,f,(h?c:d)-e|0);c=(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))+1|0;H[a|0]=c;H[a+1|0]=c>>>8;H[a+2|0]=c>>>16;H[a+3|0]=c>>>24;H[f|0]=g;break a}c=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);e=c-29|0;f=c>>>0<29?c:e;d=f+1|0;i=c>>>0<=28?28:K[a+4|0]|K[a+5|0]<<8|(K[a+6|0]<<16|K[a+7|0]<<24);h=a+4|0;if(g>>>0<=65535){if(d>>>0>i>>>0){Ob(a,d+(d>>>1|0)|0);c=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);e=c-29|0;}g=c>>>0<29;d=f+(g?h:K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24))|0;eb(d+1|0,d,(g?c:e)-f|0);c=(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))+1|0;H[a|0]=c;H[a+1|0]=c>>>8;H[a+2|0]=c>>>16;H[a+3|0]=c>>>24;H[d|0]=77;f=J[b+4>>2]-J[b>>2]|0;c=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);d=c-29|0;g=c>>>0<29?c:d;e=g+2|0;if(e>>>0>(c>>>0<=28?28:K[a+4|0]|K[a+5|0]<<8|(K[a+6|0]<<16|K[a+7|0]<<24))>>>0){Ob(a,e+(e>>>1|0)|0);c=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);d=c-29|0;}i=h;h=c>>>0<29;e=g+(h?i:K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24))|0;eb(e+2|0,e,(h?c:d)-g|0);c=(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))+2|0;H[a|0]=c;H[a+1|0]=c>>>8;H[a+2|0]=c>>>16;H[a+3|0]=c>>>24;H[e|0]=f;H[e+1|0]=f>>>8;break a}if(d>>>0>i>>>0){Ob(a,d+(d>>>1|0)|0);c=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);e=c-29|0;}g=c>>>0<29;d=f+(g?h:K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24))|0;eb(d+1|0,d,(g?c:e)-f|0);c=(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))+1|0;H[a|0]=c;H[a+1|0]=c>>>8;H[a+2|0]=c>>>16;H[a+3|0]=c>>>24;H[d|0]=78;f=J[b+4>>2]-J[b>>2]|0;c=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);d=c-29|0;g=c>>>0<29?c:d;e=g+4|0;if(e>>>0>(c>>>0<=28?28:K[a+4|0]|K[a+5|0]<<8|(K[a+6|0]<<16|K[a+7|0]<<24))>>>0){Ob(a,e+(e>>>1|0)|0);c=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);d=c-29|0;}i=h;h=c>>>0<29;e=g+(h?i:K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24))|0;eb(e+4|0,e,(h?c:d)-g|0);c=(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))+4|0;H[a|0]=c;H[a+1|0]=c>>>8;H[a+2|0]=c>>>16;H[a+3|0]=c>>>24;H[e|0]=f;H[e+1|0]=f>>>8;H[e+2|0]=f>>>16;H[e+3|0]=f>>>24;}d=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);e=d-29|0;f=J[b+4>>2];b=J[b>>2];h=f-b|0;g=d>>>0<29?d:e;c=h+g|0;if(c>>>0>(d>>>0<=28?28:K[a+4|0]|K[a+5|0]<<8|(K[a+6|0]<<16|K[a+7|0]<<24))>>>0){Ob(a,c+(c>>>1|0)|0);d=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);e=d-29|0;}i=d>>>0<29;c=g+(i?a+4|0:K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24))|0;eb(h+c|0,c,(i?d:e)-g|0);d=h+(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))|0;H[a|0]=d;H[a+1|0]=d>>>8;H[a+2|0]=d>>>16;H[a+3|0]=d>>>24;b:{if((b|0)==(f|0)){break b}e=f+(b^-1)|0;h=h&7;if(h){d=0;while(1){H[c|0]=K[b|0];b=b+1|0;c=c+1|0;d=d+1|0;if((h|0)!=(d|0)){continue}break}}if(e>>>0<7){break b}while(1){H[c|0]=K[b|0];H[c+1|0]=K[b+1|0];H[c+2|0]=K[b+2|0];H[c+3|0]=K[b+3|0];H[c+4|0]=K[b+4|0];H[c+5|0]=K[b+5|0];H[c+6|0]=K[b+6|0];H[c+7|0]=K[b+7|0];c=c+8|0;b=b+8|0;if((f|0)!=(b|0)){continue}break}}return a|0}function Kk(a,b,c,d,e,f,g,h){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;var i=0,j=0,k=0;h=Ua-16|0;Ua=h;J[h+12>>2]=b;J[e>>2]=0;i=J[d+28>>2];J[h>>2]=i;J[i+4>>2]=J[i+4>>2]+1;J[5996]=0;i=$(170,J[h>>2])|0;j=J[5996];J[5996]=0;a:{b:{c:{d:{e:{f:{g:{h:{i:{j:{k:{l:{m:{n:{o:{p:{q:{r:{s:{t:{u:{v:{w:{x:{y:{z:{A:{if((j|0)!=1){j=J[h>>2];k=J[j+4>>2]-1|0;J[j+4>>2]=k;if((k|0)==-1){Xa[J[J[j>>2]+8>>2]](j);}switch(g-65|0){case 54:break j;case 19:break k;case 18:break l;case 17:break m;case 49:break n;case 47:break o;case 45:case 51:break p;case 12:break q;case 44:break r;case 41:break s;case 8:break t;case 7:break u;case 5:break v;case 3:break w;case 35:case 36:break x;case 34:break y;case 1:case 33:case 39:break z;case 0:case 32:break A;case 2:case 4:case 6:case 9:case 10:case 11:case 13:case 14:case 15:case 16:case 20:case 21:case 22:case 25:case 26:case 27:case 28:case 29:case 30:case 31:case 37:case 38:case 40:case 42:case 43:case 46:case 48:case 50:case 52:case 53:break d;case 24:break f;case 56:break g;case 23:break h;case 55:break i;default:break e}}c=Z()|0;a=J[h>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(c|0);B();}Ef(a,f+24|0,h+12|0,c,e,i);break b}Df(a,f+16|0,h+12|0,c,e,i);break b}g=Xa[J[J[a+8>>2]+12>>2]](a+8|0)|0;i=K[g+11|0];j=a;k=b;b=(i&128)>>>7|0;a=b?J[g>>2]:g;J[h+12>>2]=kc(j,k,c,d,e,f,a,a+(b?J[g+4>>2]:i&127)|0);break b}a=cc(h+12|0,c,e,i,2);b=J[e>>2];B:{if(!(a-1>>>0>30|b&4)){J[f+12>>2]=a;break B}J[e>>2]=b|4;}break b}J[h>>2]=623865125;J[h+4>>2]=2032480100;J[h+12>>2]=kc(a,b,c,d,e,f,h,h+8|0);break b}J[h>>2]=623728933;J[h+4>>2]=1680158061;J[h+12>>2]=kc(a,b,c,d,e,f,h,h+8|0);break b}a=cc(h+12|0,c,e,i,2);b=J[e>>2];C:{if(!((a|0)>23|b&4)){J[f+8>>2]=a;break C}J[e>>2]=b|4;}break b}a=cc(h+12|0,c,e,i,2);b=J[e>>2];D:{if(!(a-1>>>0>11|b&4)){J[f+8>>2]=a;break D}J[e>>2]=b|4;}break b}a=cc(h+12|0,c,e,i,3);b=J[e>>2];E:{if(!((a|0)>365|b&4)){J[f+28>>2]=a;break E}J[e>>2]=b|4;}break b}b=cc(h+12|0,c,e,i,2)-1|0;a=J[e>>2];F:{if(!(b>>>0>11|a&4)){J[f+16>>2]=b;break F}J[e>>2]=a|4;}break b}a=cc(h+12|0,c,e,i,2);b=J[e>>2];G:{if(!((a|0)>59|b&4)){J[f+4>>2]=a;break G}J[e>>2]=b|4;}break b}a=h+12|0;b=Ua-16|0;Ua=b;J[b+12>>2]=c;while(1){H:{if(Ub(a,b+12|0)){break H}c=J[a>>2];d=J[c+12>>2];I:{if((d|0)==J[c+16>>2]){c=Xa[J[J[c>>2]+36>>2]](c)|0;break I}c=K[d|0];}d=J[i+8>>2];c=c<<24>>24;if((c|0)>=0){c=J[d+((c&255)<<2)>>2]&1;}else {c=0;}if(!c){break H}Ic(a);continue}break}if(Ub(a,b+12|0)){J[e>>2]=J[e>>2]|2;}Ua=b+16|0;break b}d=h+12|0;a=Xa[J[J[a+8>>2]+8>>2]](a+8|0)|0;b=K[a+11|0];g=(b&128)>>>7|0?J[a+4>>2]:b&127;b=K[a+23|0];J:{if((g|0)==(0-((b&128)>>>7|0?J[a+16>>2]:b&127)|0)){J[e>>2]=J[e>>2]|4;break J}c=nd(d,c,a,a+24|0,i,e,0);b=J[f+8>>2];if(!((c|0)!=(a|0)|(b|0)!=12)){J[f+8>>2]=0;break J}if(!((c-a|0)!=12|(b|0)>11)){J[f+8>>2]=b+12;}}break b}g=K[12456]|K[12457]<<8|(K[12458]<<16|K[12459]<<24);H[h+7|0]=g;H[h+8|0]=g>>>8;H[h+9|0]=g>>>16;H[h+10|0]=g>>>24;g=K[12453]|K[12454]<<8|(K[12455]<<16|K[12456]<<24);J[h>>2]=K[12449]|K[12450]<<8|(K[12451]<<16|K[12452]<<24);J[h+4>>2]=g;J[h+12>>2]=kc(a,b,c,d,e,f,h,h+11|0);break b}H[h+4|0]=K[12464];J[h>>2]=K[12460]|K[12461]<<8|(K[12462]<<16|K[12463]<<24);J[h+12>>2]=kc(a,b,c,d,e,f,h,h+5|0);break b}a=cc(h+12|0,c,e,i,2);b=J[e>>2];K:{if(!((a|0)>60|b&4)){J[f>>2]=a;break K}J[e>>2]=b|4;}break b}J[h>>2]=624576549;J[h+4>>2]=1394948685;J[h+12>>2]=kc(a,b,c,d,e,f,h,h+8|0);break b}a=cc(h+12|0,c,e,i,1);b=J[e>>2];L:{if(!((a|0)>6|b&4)){J[f+24>>2]=a;break L}J[e>>2]=b|4;}break b}a=Xa[J[J[a>>2]+20>>2]](a,b,c,d,e,f)|0;break a}g=Xa[J[J[a+8>>2]+24>>2]](a+8|0)|0;i=K[g+11|0];j=a;k=b;b=(i&128)>>>7|0;a=b?J[g>>2]:g;J[h+12>>2]=kc(j,k,c,d,e,f,a,a+(b?J[g+4>>2]:i&127)|0);break b}Cf(f+20|0,h+12|0,c,e,i);break b}a=cc(h+12|0,c,e,i,4);if(!(K[e|0]&4)){J[f+20>>2]=a-1900;}break b}if((g|0)==37){break c}}J[e>>2]=J[e>>2]|4;break b}a=Ua-16|0;Ua=a;J[a+12>>2]=c;b=6;d=h+12|0;g=a+12|0;M:{N:{if(Ub(d,g)){break N}b=4;c=J[d>>2];f=J[c+12>>2];O:{if((f|0)==J[c+16>>2]){c=Xa[J[J[c>>2]+36>>2]](c)|0;break O}c=K[f|0];}if((Xa[J[J[i>>2]+36>>2]](i,c<<24>>24,0)|0)!=37){break N}b=2;if(!Ub(Ic(d),g)){break M}}J[e>>2]=J[e>>2]|b;}Ua=a+16|0;}a=J[h+12>>2];}Ua=h+16|0;return a|0}function Yc(a,b,c,d,e,f){var g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0;g=Ua-32|0;Ua=g;if(!K[24588]){J[6145]=0;J[6146]=0;J[6143]=0;J[6144]=0;J[6141]=0;J[6142]=0;H[24588]=1;}i=1-e;a:{b:{c:{d:{e:{f:{switch(b|0){case 10:case 11:case 12:case 16:b=cb(8);J[g+20>>2]=b;f=b+8|0;J[g+28>>2]=f;O[b>>3]=c;J[g+24>>2]=f;J[5996]=0;f=$(3,8)|0;h=J[5996];J[5996]=0;if((h|0)==1){break b}J[g+8>>2]=f;h=f+8|0;J[g+16>>2]=h;O[f>>3]=d;J[5996]=0;J[g+12>>2]=h;ba(65,a|0,g+20|0,g+8|0)|0;a=J[5996];J[5996]=0;if((a|0)!=1){$a(f);$a(b);break e}break c;case 13:case 14:case 15:b=cb(8);J[g+20>>2]=b;f=b+8|0;J[g+28>>2]=f;O[b>>3]=c;J[g+24>>2]=f;J[5996]=0;f=$(3,8)|0;h=J[5996];J[5996]=0;if((h|0)==1){break b}J[g+8>>2]=f;h=f+8|0;J[g+16>>2]=h;J[f>>2]=0;J[f+4>>2]=0;J[5996]=0;J[g+12>>2]=h;ba(65,a|0,g+20|0,g+8|0)|0;a=J[5996];J[5996]=0;if((a|0)!=1){$a(f);$a(b);break e}break c;case 17:case 18:b=cb(16);J[g+20>>2]=b;f=b+16|0;J[g+28>>2]=f;O[b+8>>3]=c;O[b>>3]=c;J[g+24>>2]=f;J[5996]=0;f=$(3,16)|0;h=J[5996];J[5996]=0;if((h|0)==1){break b}J[g+8>>2]=f;h=f+16|0;J[g+16>>2]=h;O[f+8>>3]=d;O[f>>3]=d;J[5996]=0;J[g+12>>2]=h;ba(65,a|0,g+20|0,g+8|0)|0;a=J[5996];J[5996]=0;if((a|0)!=1){$a(f);$a(b);break e}break c;case 19:b=cb(16);J[g+20>>2]=b;f=b+16|0;J[g+28>>2]=f;i=i*c;O[b+8>>3]=i;c=c*e;O[b>>3]=c;J[g+24>>2]=f;J[5996]=0;f=$(3,16)|0;h=J[5996];J[5996]=0;if((h|0)==1){break b}J[g+8>>2]=f;h=f+16|0;J[g+16>>2]=h;O[f+8>>3]=c+d;O[f>>3]=i+d;J[5996]=0;J[g+12>>2]=h;ba(65,a|0,g+20|0,g+8|0)|0;a=J[5996];J[5996]=0;if((a|0)!=1){$a(f);$a(b);break e}break c;case 21:b=cb(16);J[g+20>>2]=b;f=b+16|0;J[g+28>>2]=f;i=i*c;O[b+8>>3]=i;O[b>>3]=c*e;J[g+24>>2]=f;J[5996]=0;f=$(3,16)|0;h=J[5996];J[5996]=0;if((h|0)==1){break b}J[g+8>>2]=f;h=f+16|0;J[g+16>>2]=h;O[f+8>>3]=d;O[f>>3]=i+d;J[5996]=0;J[g+12>>2]=h;ba(65,a|0,g+20|0,g+8|0)|0;a=J[5996];J[5996]=0;if((a|0)!=1){$a(f);$a(b);break e}break c;case 20:b=cb(16);J[g+20>>2]=b;f=b+16|0;J[g+28>>2]=f;d=i*c;O[b+8>>3]=d;O[b>>3]=c*e;J[g+24>>2]=f;J[5996]=0;f=$(3,16)|0;h=J[5996];J[5996]=0;if((h|0)==1){break b}J[g+8>>2]=f;h=f+16|0;J[g+16>>2]=h;J[f+8>>2]=0;J[f+12>>2]=0;O[f>>3]=d;J[5996]=0;J[g+12>>2]=h;ba(65,a|0,g+20|0,g+8|0)|0;a=J[5996];J[5996]=0;if((a|0)!=1){$a(f);$a(b);break e}break c;case 22:b=cb(16);J[g+20>>2]=b;h=b+16|0;J[g+28>>2]=h;O[b+8>>3]=i*c;O[b>>3]=c*e;J[g+24>>2]=h;J[5996]=0;h=$(3,16)|0;j=J[5996];J[5996]=0;if((j|0)==1){break b}J[g+8>>2]=h;j=h+16|0;J[g+16>>2]=j;O[h+8>>3]=(f|0)==1?d:0;O[h>>3]=f?0:d;J[5996]=0;J[g+12>>2]=j;ba(65,a|0,g+20|0,g+8|0)|0;a=J[5996];J[5996]=0;if((a|0)!=1){$a(h);$a(b);break e}a=Z()|0;$a(h);break a;case 23:b=cb(24);J[g+20>>2]=b;f=b+24|0;J[g+28>>2]=f;i=i*c;O[b+16>>3]=i;c=c*e;O[b+8>>3]=c;O[b>>3]=c;J[g+24>>2]=f;J[5996]=0;f=$(3,24)|0;h=J[5996];J[5996]=0;if((h|0)==1){break b}J[g+8>>2]=f;h=f+24|0;J[g+16>>2]=h;O[f+16>>3]=d;J[f+8>>2]=0;J[f+12>>2]=0;O[f>>3]=i+d;J[5996]=0;J[g+12>>2]=h;ba(65,a|0,g+20|0,g+8|0)|0;a=J[5996];J[5996]=0;if((a|0)!=1){$a(f);$a(b);break e}break c;case 24:g:{h:{if(!f){f=0;break h}if(f>>>0>=536870912){break d}e=c*e;h=f<<3;k=cb(h);b=k;l=f&7;if(l){while(1){O[b>>3]=e;b=b+8|0;j=j+1|0;if((l|0)!=(j|0)){continue}break}}m=h+k|0;n=f-1&536870911;if(n>>>0>=7){while(1){O[b+56>>3]=e;O[b+48>>3]=e;O[b+40>>3]=e;O[b+32>>3]=e;O[b+24>>3]=e;O[b+16>>3]=e;O[b+8>>3]=e;O[b>>3]=e;b=b- -64|0;if((m|0)!=(b|0)){continue}break}}j=0;J[5996]=0;h=$(3,h|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break g}c=c*i+d;b=h;if(l){while(1){O[b>>3]=c;b=b+8|0;j=j+1|0;if((l|0)!=(j|0)){continue}break}}f=(f<<3)+h|0;if(n>>>0<7){break h}while(1){O[b+56>>3]=c;O[b+48>>3]=c;O[b+40>>3]=c;O[b+32>>3]=c;O[b+24>>3]=c;O[b+16>>3]=c;O[b+8>>3]=c;O[b>>3]=c;b=b- -64|0;if((f|0)!=(b|0)){continue}break}}J[a+12>>2]=h;J[a+8>>2]=m;J[a+4>>2]=m;J[a>>2]=k;J[a+20>>2]=f;J[a+16>>2]=f;break e}a=Z()|0;$a(k);da(a|0);B();default:la(2350,1900,558,1676);B();case 0:case 1:case 2:case 3:case 4:case 5:case 6:case 7:case 8:case 9:case 25:break f}}Yg(a);}Ua=g+32|0;return}nb();B();}a=Z()|0;$a(f);break a}a=Z()|0;}$a(b);da(a|0);B();}function Zg(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0;a:{b:{e=J[c>>2];d=J[e+80>>2];c:{if(!d|M[e+84>>2]>3600|(J[e+52>>2]+J[e+44>>2]>>>0>201|M[e+68>>2]>100)){break c}J[5996]=0;h=_(54,2049,1)|0;f=J[5996];J[5996]=0;if((f|0)==1){break b}if((d^-1)&h){break c}d=J[e+80>>2];J[5996]=0;h=_(54,2165,1)|0;f=J[5996];J[5996]=0;if((f|0)==1){break b}if(K[e+88|0]|(d^-1)&h){break c}i=O[a+24>>3];k=i==0?0:i*O[b+8>>3];i=O[a+16>>3];g=J[e+84>>2];i=k+((i==0?0:i*O[b>>3])+ +(g>>>0));if(i>1e4){break c}f=J[e+80>>2];d:{e:{d=J[a>>2];h=J[a+4>>2];if((d|0)==(h|0)){break e}e=d;while(1){j=J[e>>2];f:{if(f&(J[j+80>>2]^-1)){break f}k=O[e+24>>3];if(k<i){break c}if(i<k){break f}if(g>>>0<=M[j+84>>2]){break c}}e=e+32|0;if((h|0)!=(e|0)){continue}break}e=f^-1;while(1){j=J[d>>2];g:{if(e&J[j+80>>2]){break g}k=O[d+24>>3];if(k<i){break g}if(g>>>0>=M[j+84>>2]|i<k){break e}}d=d+32|0;if((h|0)!=(d|0)){continue}break}e=h;break d}if((d|0)==(h|0)){e=h;break d}g=d+32|0;h:{if((g|0)==(h|0)){e=d;break h}j=f^-1;e=d;while(1){f=d;d=g;g=J[f+32>>2];i:{j:{if(j&J[g+80>>2]){break j}k=O[f+56>>3];if(k<i){break j}if(M[g+84>>2]<=M[J[c>>2]+84>>2]|i<k){break i}}l=J[f+36>>2];J[d>>2]=0;J[f+36>>2]=0;J[e>>2]=g;g=J[e+4>>2];J[e+4>>2]=l;k:{if(!g){break k}l=J[g+4>>2];J[g+4>>2]=l-1;if(l){break k}Xa[J[J[g>>2]+8>>2]](g);bb(g);}g=J[f+60>>2];J[e+24>>2]=J[f+56>>2];J[e+28>>2]=g;g=J[f+52>>2];J[e+16>>2]=J[f+48>>2];J[e+20>>2]=g;g=J[f+44>>2];J[e+8>>2]=J[f+40>>2];J[e+12>>2]=g;e=e+32|0;}g=d+32|0;if((g|0)!=(h|0)){continue}break}h=J[a+4>>2];}if((e|0)==(h|0)){e=h;break d}d=(h-e|0)+e|0;if((h|0)!=(d|0)){while(1){g=J[d>>2];j=J[d+4>>2];J[d>>2]=0;J[d+4>>2]=0;f=J[e+4>>2];J[e>>2]=g;J[e+4>>2]=j;l:{if(!f){break l}g=J[f+4>>2];J[f+4>>2]=g-1;if(g){break l}Xa[J[J[f>>2]+8>>2]](f);bb(f);}f=J[d+12>>2];J[e+8>>2]=J[d+8>>2];J[e+12>>2]=f;f=J[d+28>>2];J[e+24>>2]=J[d+24>>2];J[e+28>>2]=f;f=J[d+20>>2];J[e+16>>2]=J[d+16>>2];J[e+20>>2]=f;e=e+32|0;d=d+32|0;if((h|0)!=(d|0)){continue}break}h=J[a+4>>2];}if((e|0)!=(h|0)){while(1){h=h-32|0;d=J[h+4>>2];m:{if(!d){break m}f=J[d+4>>2];J[d+4>>2]=f-1;if(f){break m}Xa[J[J[d>>2]+8>>2]](d);bb(d);}if((e|0)!=(h|0)){continue}break}}J[a+4>>2]=e;}f=J[a+8>>2];n:{if(f>>>0>e>>>0){J[e>>2]=J[c>>2];J[e+4>>2]=J[c+4>>2];J[c>>2]=0;J[c+4>>2]=0;d=J[b+12>>2];J[e+16>>2]=J[b+8>>2];J[e+20>>2]=d;d=J[b+4>>2];J[e+8>>2]=J[b>>2];J[e+12>>2]=d;O[e+24>>3]=i;J[a+4>>2]=e+32;break n}h=J[a>>2];g=e-h>>5;d=g+1|0;if(d>>>0>=134217728){J[5996]=0;ca(62);a=J[5996];J[5996]=0;if((a|0)==1){break b}break a}f=f-h|0;j=f>>4;d=f>>>0>=2147483616?134217727:d>>>0<j>>>0?j:d;if(d>>>0>=134217728){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break b}J[5996]=0;j=d<<5;f=$(3,j|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break b}d=f+(g<<5)|0;J[d>>2]=J[c>>2];J[d+4>>2]=J[c+4>>2];J[c>>2]=0;J[c+4>>2]=0;g=J[b+12>>2];J[d+16>>2]=J[b+8>>2];J[d+20>>2]=g;g=J[b+4>>2];J[d+8>>2]=J[b>>2];J[d+12>>2]=g;O[d+24>>3]=i;b=f+j|0;f=d+32|0;o:{if((e|0)!=(h|0)){while(1){d=d-32|0;e=e-32|0;J[d>>2]=J[e>>2];J[d+4>>2]=J[e+4>>2];J[e>>2]=0;J[e+4>>2]=0;g=J[e+28>>2];J[d+24>>2]=J[e+24>>2];J[d+28>>2]=g;g=J[e+20>>2];J[d+16>>2]=J[e+16>>2];J[d+20>>2]=g;g=J[e+12>>2];J[d+8>>2]=J[e+8>>2];J[d+12>>2]=g;if((e|0)!=(h|0)){continue}break}J[a+8>>2]=b;b=J[a+4>>2];J[a+4>>2]=f;e=J[a>>2];J[a>>2]=d;if((b|0)==(e|0)){break o}while(1){b=b-32|0;d=J[b+4>>2];p:{if(!d){break p}h=J[d+4>>2];J[d+4>>2]=h-1;if(h){break p}Xa[J[J[d>>2]+8>>2]](d);bb(d);}if((b|0)!=(e|0)){continue}break}break o}J[a+8>>2]=b;J[a+4>>2]=f;J[a>>2]=d;}if(!e){break n}$a(e);}J[a+32>>2]=J[a+32>>2]+1;}a=J[c+4>>2];q:{if(!a){break q}b=J[a+4>>2];J[a+4>>2]=b-1;if(b){break q}Xa[J[J[a>>2]+8>>2]](a);bb(a);}return}a=Z()|0;mb(c);da(a|0);B();}B();}function eh(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0,i=Q(0),j=0,k=0,l=0,m=0,n=Q(0),o=0;d=P(a,1540483477);d=P(d>>>24^d,1540483477)^1866966612;d=P(d>>>13^d,1540483477);g=d>>>15^d;a:{b:{c:{d:{h=J[b+4>>2];e:{if(!h){break e}e=J[b>>2];k=fo(h);d=h-1&g;f:{if(k>>>0<=1){break f}d=g;if(d>>>0<h>>>0){break f}d=(d>>>0)%(h>>>0)|0;}f=d;d=J[e+(f<<2)>>2];if(!d){break e}d=J[d>>2];if(!d){break e}if(k>>>0<=1){e=h-1|0;while(1){h=J[d+4>>2];g:{if((h|0)!=(g|0)){if((f|0)==(e&h)){break g}break e}if(J[d+8>>2]==(a|0)){break d}}d=J[d>>2];if(d){continue}break}break e}while(1){e=J[d+4>>2];h:{if((e|0)!=(g|0)){if(e>>>0>=h>>>0){e=(e>>>0)%(h>>>0)|0;}if((e|0)==(f|0)){break h}break e}if(J[d+8>>2]==(a|0)){break d}}d=J[d>>2];if(d){continue}break}}m=ze(a,b,c);if(!m){return 0}d=J[b+4>>2];if(!d){break c}e=d-1|0;if(e&d){c=d>>>0<=g>>>0?(g>>>0)%(d>>>0)|0:g;f=J[J[b>>2]+(c<<2)>>2];if(!f){break c}while(1){f=J[f>>2];if(!f){break c}e=J[f+4>>2];if((e|0)!=(g|0)){if(d>>>0<=e>>>0){e=(e>>>0)%(d>>>0)|0;}if((c|0)!=(e|0)){break c}}if(J[f+8>>2]!=(a|0)){continue}break}break b}c=e&g;f=J[J[b>>2]+(c<<2)>>2];if(!f){break c}while(1){f=J[f>>2];if(!f){break c}h=J[f+4>>2];if((h|0)!=(g|0)&(e&h)!=(c|0)){break c}if(J[f+8>>2]!=(a|0)){continue}break}break b}return J[d+12>>2]}h=cb(16);J[h+8>>2]=a;J[h+4>>2]=g;J[h>>2]=0;J[h+12>>2]=m;k=b+8|0;n=Q(J[b+12>>2]+1>>>0);i=N[b+16>>2];i:{if(!(!d|n>Q(i*Q(d>>>0)))){g=c;break i}e=2;c=(d-1&d)!=0|d>>>0<3|d<<1;i=Q(W(Q(n/i)));j:{if(i<Q(4294967296)&i>=Q(0)){a=~~i>>>0;break j}a=0;}a=a>>>0<c>>>0?c:a;k:{if((a|0)==1){break k}if(!(a&a-1)){e=a;break k}J[5996]=0;e=$(43,a|0)|0;a=J[5996];J[5996]=0;if((a|0)==1){break a}d=J[b+4>>2];}l:{m:{if(d>>>0>=e>>>0){if(d>>>0<=e>>>0){break l}c=d>>>0<3;i=Q(W(Q(Q(M[b+12>>2])/N[b+16>>2])));n:{if(i<Q(4294967296)&i>=Q(0)){a=~~i>>>0;break n}a=0;}o:{p:{if(c){break p}if(fo(d)>>>0>1){break p}a=a>>>0<2?a:1<<32-S(a-1|0);break o}J[5996]=0;a=$(43,a|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}}e=a>>>0<e>>>0?e:a;if(e>>>0>=d>>>0){d=J[b+4>>2];break l}if(!e){break m}}if(e>>>0>=1073741824){J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)==1){break a}B();}J[5996]=0;c=$(3,e<<2)|0;a=J[5996];J[5996]=0;if((a|0)==1){break a}a=J[b>>2];J[b>>2]=c;if(a){$a(a);}J[b+4>>2]=e;f=0;d=0;j=e-1|0;if(j>>>0>=3){c=e&-4;while(1){a=d<<2;J[a+J[b>>2]>>2]=0;J[J[b>>2]+(a|4)>>2]=0;J[J[b>>2]+(a|8)>>2]=0;J[J[b>>2]+(a|12)>>2]=0;d=d+4|0;if((c|0)!=(d|0)){continue}break}}a=e&3;if(a){while(1){J[J[b>>2]+(d<<2)>>2]=0;d=d+1|0;f=f+1|0;if((a|0)!=(f|0)){continue}break}}d=J[b+8>>2];if(!d){d=e;break l}c=J[d+4>>2];q:{if(e-1&e){c=c>>>0>=e>>>0?(c>>>0)%(e>>>0)|0:c;J[J[b>>2]+(c<<2)>>2]=k;f=J[d>>2];if(f){break q}d=e;break l}c=c&j;J[J[b>>2]+(c<<2)>>2]=k;a=J[d>>2];if(!a){d=e;break l}while(1){f=j&J[a+4>>2];r:{if((f|0)==(c|0)){d=a;break r}l=f<<2;o=l+J[b>>2]|0;if(J[o>>2]){J[d>>2]=J[a>>2];J[a>>2]=J[J[l+J[b>>2]>>2]>>2];J[J[l+J[b>>2]>>2]>>2]=a;break r}J[o>>2]=d;d=a;c=f;}a=J[d>>2];if(a){continue}break}d=e;break l}while(1){a=J[f+4>>2];if(e>>>0<=a>>>0){a=(a>>>0)%(e>>>0)|0;}s:{if((a|0)==(c|0)){d=f;break s}j=a<<2;l=j+J[b>>2]|0;if(!J[l>>2]){J[l>>2]=d;d=f;c=a;break s}J[d>>2]=J[f>>2];J[f>>2]=J[J[j+J[b>>2]>>2]>>2];J[J[j+J[b>>2]>>2]>>2]=f;}f=J[d>>2];if(f){continue}break}d=e;break l}a=J[b>>2];d=0;J[b>>2]=0;if(a){$a(a);}J[b+4>>2]=0;}a=d-1|0;if(!(a&d)){g=a&g;break i}if(d>>>0>g>>>0){break i}g=(g>>>0)%(d>>>0)|0;}a=J[b>>2]+(g<<2)|0;g=J[a>>2];t:{u:{if(!g){J[h>>2]=J[b+8>>2];J[b+8>>2]=h;J[a>>2]=k;a=J[h>>2];if(!a){break t}g=J[a+4>>2];a=d-1|0;v:{if(!(a&d)){g=a&g;break v}if(d>>>0>g>>>0){break v}g=(g>>>0)%(d>>>0)|0;}g=J[b>>2]+(g<<2)|0;break u}J[h>>2]=J[g>>2];}J[g>>2]=h;}J[b+12>>2]=J[b+12>>2]+1;}return m|0}a=Z()|0;$a(h);da(a|0);B();}function bo(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;var f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0;i=Ua-16|0;Ua=i;J[i+8>>2]=0;J[i+12>>2]=0;k=i+8|0;J[i+4>>2]=k;a:{b:{c:{d:{e:{if((b|0)!=(c|0)){while(1){j=J[b>>2];g=J[j+80>>2];f:{if((g^-1)&d){break f}l=e&g;h=k;g=h;g:{m=J[i+8>>2];f=m;h:{if(!f){break h}while(1){g=f;f=J[g+16>>2];if(f>>>0>l>>>0){h=g;f=J[g>>2];if(f){continue}break h}if(f>>>0>=l>>>0){break g}f=J[g+4>>2];if(f){continue}break}h=g+4|0;}J[5996]=0;f=$(3,56)|0;p=J[5996];J[5996]=0;if((p|0)==1){break e}J[f+16>>2]=l;J[f+24>>2]=j;j=J[b+4>>2];J[f+28>>2]=j;if(j){J[j+4>>2]=J[j+4>>2]+1;}j=J[b+12>>2];J[f+32>>2]=J[b+8>>2];J[f+36>>2]=j;j=J[b+28>>2];J[f+48>>2]=J[b+24>>2];J[f+52>>2]=j;j=J[b+20>>2];J[f+40>>2]=J[b+16>>2];J[f+44>>2]=j;J[f+8>>2]=g;J[f>>2]=0;J[f+4>>2]=0;J[h>>2]=f;g=J[J[i+4>>2]>>2];if(g){J[i+4>>2]=g;f=J[h>>2];}zd(J[i+8>>2],f);J[i+12>>2]=J[i+12>>2]+1;break f}n=O[b+24>>3];o=O[g+48>>3];if(!(n<o)&(M[j+84>>2]<=M[J[g+24>>2]+84>>2]|n>o)){break f}h=J[b+4>>2];if(h){J[h+4>>2]=J[h+4>>2]+1;}J[g+24>>2]=j;f=J[g+28>>2];J[g+28>>2]=h;i:{if(!f){break i}h=J[f+4>>2];J[f+4>>2]=h-1;if(h){break i}Xa[J[J[f>>2]+8>>2]](f);bb(f);}f=J[b+12>>2];J[g+32>>2]=J[b+8>>2];J[g+36>>2]=f;f=J[b+28>>2];J[g+48>>2]=J[b+24>>2];J[g+52>>2]=f;f=J[b+20>>2];J[g+40>>2]=J[b+16>>2];J[g+44>>2]=f;}b=b+32|0;if((c|0)!=(b|0)){continue}break d}}J[a+8>>2]=0;J[a>>2]=0;J[a+4>>2]=0;break a}g=Z()|0;break c}d=J[i+4>>2];J[a+8>>2]=0;J[a>>2]=0;J[a+4>>2]=0;if((d|0)==(k|0)){break a}while(1){j:{g=J[a+4>>2];k:{if((g|0)!=J[a+8>>2]){J[g>>2]=J[d+24>>2];b=J[d+28>>2];J[g+4>>2]=b;if(b){J[b+4>>2]=J[b+4>>2]+1;}b=J[d+52>>2];J[g+24>>2]=J[d+48>>2];J[g+28>>2]=b;b=J[d+44>>2];J[g+16>>2]=J[d+40>>2];J[g+20>>2]=b;b=J[d+36>>2];J[g+8>>2]=J[d+32>>2];J[g+12>>2]=b;J[a+4>>2]=g+32;break k}b=J[a>>2];c=g-b|0;f=c>>5;e=f+1|0;if(e>>>0>=134217728){J[5996]=0;ca(62);b=J[5996];J[5996]=0;if((b|0)==1){break j}break b}h=c>>4;e=c>>>0>=2147483616?134217727:e>>>0<h>>>0?h:e;l:{if(!e){c=0;break l}if(e>>>0>=134217728){J[5996]=0;ca(35);b=J[5996];J[5996]=0;if((b|0)!=1){break b}break j}J[5996]=0;c=$(3,e<<5)|0;h=J[5996];J[5996]=0;if((h|0)==1){break j}}f=(f<<5)+c|0;J[f>>2]=J[d+24>>2];h=J[d+28>>2];J[f+4>>2]=h;if(h){J[h+4>>2]=J[h+4>>2]+1;g=J[a+4>>2];b=J[a>>2];}c=(e<<5)+c|0;e=J[d+36>>2];J[f+8>>2]=J[d+32>>2];J[f+12>>2]=e;e=J[d+52>>2];J[f+24>>2]=J[d+48>>2];J[f+28>>2]=e;e=J[d+44>>2];J[f+16>>2]=J[d+40>>2];J[f+20>>2]=e;e=f+32|0;m:{if((b|0)==(g|0)){J[a+8>>2]=c;J[a+4>>2]=e;J[a>>2]=f;break m}while(1){f=f-32|0;g=g-32|0;J[f>>2]=J[g>>2];J[f+4>>2]=J[g+4>>2];J[g>>2]=0;J[g+4>>2]=0;h=J[g+28>>2];J[f+24>>2]=J[g+24>>2];J[f+28>>2]=h;h=J[g+20>>2];J[f+16>>2]=J[g+16>>2];J[f+20>>2]=h;h=J[g+12>>2];J[f+8>>2]=J[g+8>>2];J[f+12>>2]=h;if((b|0)!=(g|0)){continue}break}J[a+8>>2]=c;g=J[a+4>>2];J[a+4>>2]=e;b=J[a>>2];J[a>>2]=f;if((b|0)==(g|0)){break m}while(1){g=g-32|0;c=J[g+4>>2];n:{if(!c){break n}e=J[c+4>>2];J[c+4>>2]=e-1;if(e){break n}Xa[J[J[c>>2]+8>>2]](c);bb(c);}if((b|0)!=(g|0)){continue}break}}if(!b){break k}$a(b);}f=J[d+4>>2];o:{if(f){while(1){g=f;f=J[g>>2];if(f){continue}break o}}while(1){g=J[d+8>>2];b=J[g>>2]!=(d|0);d=g;if(b){continue}break}}d=g;if((g|0)!=(k|0)){continue}break a}break}g=Z()|0;_b(a);m=J[i+8>>2];}yd(m);da(g|0);B();}B();}yd(J[i+8>>2]);Ua=i+16|0;}function Hb(a,b,c,d,e,f,g,h,i){var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0;j=Ua-112|0;Ua=j;k=i&2147483647;a:{b:{m=e&2147483647;l=2147418112;n=!(b|c);if(!(d|m?m-l>>>0<2147549184:n)){p=k-l|0;if(!h&(p|0)==-2147418112?f|g:(p|0)==-2147418112&(h|0)!=0|p>>>0>2147549184){break b}}if(!(!d&(m|0)==2147418112?n:m>>>0<2147418112)){h=d;i=e|32768;f=b;g=c;break a}if(!(!h&(k|0)==2147418112?!(f|g):k>>>0<2147418112)){i=i|32768;break a}if(!(b|d|(m^2147418112|c))){l=d;d=!(b^f|d^h|(c^g|e^i^-2147483648));h=d?0:l;i=d?2147450880:e;f=d?0:b;g=d?0:c;break a}l=f|h;if(!(l|(k^2147418112|g))){break a}if(!(b|d|(c|m))){if(g|k|l){break a}f=b&f;g=c&g;h=d&h;i=e&i;break a}if(f|h|(g|k)){break b}f=b;g=c;h=d;i=e;break a}l=(k|0)==(m|0);s=l&(d|0)==(h|0)?(c|0)==(g|0)&b>>>0<f>>>0|c>>>0<g>>>0:l&d>>>0<h>>>0|k>>>0>m>>>0;n=s;m=n?f:b;p=n?g:c;l=n?i:e;t=l;n=n?h:d;q=l&65535;d=s?d:h;e=s?e:i;r=e;l=e>>>16&32767;o=t>>>16&32767;if(!o){e=!(n|q);i=e;k=e?m:n;e=e<<6;i=S(i?p:q);e=e+((i|0)==32?S(k)+32|0:i)|0;Cb(j+96|0,m,p,n,q,e-15|0);n=J[j+104>>2];q=J[j+108>>2];p=J[j+100>>2];o=16-e|0;m=J[j+96>>2];}f=s?b:f;g=s?c:g;h=d;i=r&65535;if(!l){b=!(d|i);e=b;k=b?f:d;b=b<<6;e=S(e?g:i);b=b+((e|0)==32?S(k)+32|0:e)|0;Cb(j+80|0,f,g,d,i,b-15|0);l=16-b|0;h=J[j+88>>2];i=J[j+92>>2];g=J[j+84>>2];f=J[j+80>>2];}c=i<<3|h>>>29;b=h<<3|g>>>29;c=c|524288;e=q<<3|n>>>29;q=n<<3|p>>>29;h=e;i=r^t;k=g<<3|f>>>29;d=f<<3;c:{if((l|0)==(o|0)){break c}e=o-l|0;if(e>>>0>127){b=0;c=0;k=0;d=1;break c}Cb(j- -64|0,d,k,b,c,128-e|0);xc(j+48|0,d,k,b,c,e);b=J[j+56>>2];c=J[j+60>>2];k=J[j+52>>2];d=J[j+48>>2]|(J[j+64>>2]|J[j+72>>2]|(J[j+68>>2]|J[j+76>>2]))!=0;}n=d;l=k;r=h|524288;k=p<<3|m>>>29;m=m<<3;d:{if((i|0)<0){f=0;g=0;h=0;i=0;if(!(m^n|b^q|(k^l|c^r))){break a}d=m-n|0;e=k-((m>>>0<n>>>0)+l|0)|0;f=q-b|0;g=(k|0)==(l|0)&m>>>0<n>>>0|k>>>0<l>>>0;h=f-g|0;b=(r-((b>>>0>q>>>0)+c|0)|0)-(f>>>0<g>>>0)|0;i=b;if(b>>>0>524287){break d}c=!(b|h);g=c;i=g?d:h;c=g<<6;g=S(g?e:b);c=c+((g|0)==32?S(i)+32|0:g)|0;f=b;b=c-12|0;Cb(j+32|0,d,e,h,f,b);o=o-b|0;h=J[j+40>>2];i=J[j+44>>2];d=J[j+32>>2];e=J[j+36>>2];break d}k=k+l|0;d=m+n|0;e=d>>>0<m>>>0?k+1|0:k;f=(l|0)==(e|0)&d>>>0<n>>>0|e>>>0<l>>>0;k=c+r|0;b=b+q|0;k=b>>>0<q>>>0?k+1|0:k;h=b+f|0;i=h>>>0<b>>>0?k+1|0:k;if(!(i&1048576)){break d}d=n&1|((e&1)<<31|d>>>1);e=h<<31|e>>>1;o=o+1|0;h=(i&1)<<31|h>>>1;i=i>>>1|0;}c=0;b=t&-2147483648;p=b;if((o|0)>=32767){h=c;i=b|2147418112;f=0;g=0;break a}l=0;e:{if((o|0)>0){l=o;break e}Cb(j+16|0,d,e,h,i,o+127|0);xc(j,d,e,h,i,1-o|0);d=J[j>>2]|(J[j+16>>2]|J[j+24>>2]|(J[j+20>>2]|J[j+28>>2]))!=0;e=J[j+4>>2];h=J[j+8>>2];i=J[j+12>>2];}m=d&7;d=(e&7)<<29|d>>>3;f=(m>>>0>4)+d|0;b=h<<29|e>>>3;g=d>>>0>f>>>0?b+1|0:b;d=(b|0)==(g|0)&d>>>0>f>>>0|b>>>0>g>>>0;b=c|((i&7)<<29|h>>>3);h=d+b|0;i=p|(i>>>3&65535|l<<16);i=b>>>0>h>>>0?i+1|0:i;f:{if((m|0)==4){b=0;k=g+b|0;d=f;c=f&1;f=f+c|0;g=d>>>0>f>>>0?k+1|0:k;b=(b|0)==(g|0)&c>>>0>f>>>0|b>>>0>g>>>0;h=b+h|0;i=b>>>0>h>>>0?i+1|0:i;break f}if(!m){break a}}}J[a>>2]=f;J[a+4>>2]=g;J[a+8>>2]=h;J[a+12>>2]=i;Ua=j+112|0;}function rh(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0;g=Ua-16|0;Ua=g;a:{f=c-!b|0;if(!((!f&b-1>>>0>15|(f|0)!=0)&(b&c)!=-1)){d=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);e=d-29|0;c=(d>>>0<29?d:e)+1|0;if(c>>>0>(d>>>0<=28?28:K[a+4|0]|K[a+5|0]<<8|(K[a+6|0]<<16|K[a+7|0]<<24))>>>0){Ob(a,c+(c>>>1|0)|0);d=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);e=d-29|0;}c=d>>>0<29;H[(c?a+4|0:K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24))+(c?d:e)|0]=b+80;b=(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))+1|0;H[a|0]=b;H[a+1|0]=b>>>8;H[a+2|0]=b>>>16;H[a+3|0]=b>>>24;break a}if(!(b|c)){d=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);e=d-29|0;b=(d>>>0<29?d:e)+1|0;if(b>>>0>(d>>>0<=28?28:K[a+4|0]|K[a+5|0]<<8|(K[a+6|0]<<16|K[a+7|0]<<24))>>>0){Ob(a,b+(b>>>1|0)|0);d=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);e=d-29|0;}b=d>>>0<29;H[(b?a+4|0:K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24))+(b?d:e)|0]=0;b=(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))+1|0;H[a|0]=b;H[a+1|0]=b>>>8;H[a+2|0]=b>>>16;H[a+3|0]=b>>>24;break a}J[g+12>>2]=0;J[g+4>>2]=0;J[g+8>>2]=0;f=b;b=c>>31;f=f^b;h=f-b|0;b=(b^c)-((f>>>0<b>>>0)+b|0)|0;b:{c:{d:{while(1){j=b;k=h;e:{if(d>>>0<e>>>0){H[d|0]=h;f=d;break e}m=d-i|0;h=m+1|0;if((h|0)<0){J[5996]=0;ca(8);a=J[5996];J[5996]=0;if((a|0)==1){break d}break b}f=e-i|0;b=f<<1;n=f>>>0>=1073741823?2147483647:b>>>0>h>>>0?b:h;f:{if(!n){h=0;break f}J[5996]=0;h=$(3,n|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break d}}f=h+m|0;H[f|0]=k;if((d|0)==(i|0)){b=f;}else {o=(i^-1)+d|0;l=0;e=f;b=m&3;if(b){while(1){e=e-1|0;d=d-1|0;H[e|0]=K[d|0];l=l+1|0;if((b|0)!=(l|0)){continue}break}}if(o>>>0>=3){while(1){H[e-1|0]=K[d-1|0];H[e-2|0]=K[d-2|0];H[e-3|0]=K[d-3|0];e=e-4|0;d=d-4|0;H[e|0]=K[d|0];if((d|0)!=(i|0)){continue}break}}d=i;b=h;}i=b;e=h+n|0;if(!d){break e}$a(d);}b=j>>>8|0;h=(j&255)<<24|k>>>8;d=f+1|0;if(!j&k>>>0>=256|j){continue}break}J[g+8>>2]=d;J[g+4>>2]=i;J[g+12>>2]=e;g:{h:{i:{b=H[f|0];if((b|0)<0){h=c>>>24&128;if(d>>>0<e>>>0){H[f+1|0]=h;J[g+8>>2]=f+2;break g}k=d-i|0;f=k+1|0;if((f|0)<0){J[5996]=0;ca(8);a=J[5996];J[5996]=0;if((a|0)!=1){break b}break d}c=e-i|0;b=c<<1;c=c>>>0>=1073741823?2147483647:b>>>0>f>>>0?b:f;j:{if(!c){f=0;break j}J[5996]=0;f=$(3,c|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break d}}e=f+k|0;H[e|0]=h;j=c+f|0;h=e+1|0;if((d|0)==(i|0)){break i}c=(i^-1)+d|0;b=k&3;if(b){l=0;while(1){e=e-1|0;d=d-1|0;H[e|0]=K[d|0];l=l+1|0;if((b|0)!=(l|0)){continue}break}}if(c>>>0>=3){while(1){H[e-1|0]=K[d-1|0];H[e-2|0]=K[d-2|0];H[e-3|0]=K[d-3|0];e=e-4|0;d=d-4|0;H[e|0]=K[d|0];if((d|0)!=(i|0)){continue}break}}J[g+12>>2]=j;J[g+8>>2]=h;J[g+4>>2]=f;if(!i){break g}break h}if((c|0)>0|(c|0)>=0){break g}H[f|0]=b|128;break g}J[g+12>>2]=j;J[g+8>>2]=h;J[g+4>>2]=e;}$a(i);}J[5996]=0;_(19,a|0,g+4|0)|0;b=J[5996];J[5996]=0;if((b|0)!=1){b=J[g+4>>2];if(!b){break a}$a(b);break a}d=Z()|0;a=J[g+4>>2];if(!a){break c}$a(a);da(d|0);B();}d=Z()|0;if(!i){break c}$a(i);}da(d|0);B();}B();}Ua=g+16|0;return a|0}function se(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;var e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0;H[a|0]=0;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;H[a+4|0]=0;H[a+5|0]=0;H[a+6|0]=0;H[a+7|0]=0;H[a+24|0]=0;H[a+25|0]=0;H[a+26|0]=0;H[a+27|0]=0;H[a+28|0]=0;H[a+29|0]=0;H[a+30|0]=0;H[a+31|0]=0;H[a+16|0]=0;H[a+17|0]=0;H[a+18|0]=0;H[a+19|0]=0;H[a+20|0]=0;H[a+21|0]=0;H[a+22|0]=0;H[a+23|0]=0;H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;H[a+12|0]=0;H[a+13|0]=0;H[a+14|0]=0;H[a+15|0]=0;k=K[b+8|0]|K[b+9|0]<<8|(K[b+10|0]<<16|K[b+11|0]<<24);f=-29;g=K[b|0]|K[b+1|0]<<8|(K[b+2|0]<<16|K[b+3|0]<<24);i=g>>>0<29;a:{g=i?g:g-29|0;if(g>>>0>=29){J[5996]=0;aa(124,a|0,g+(g>>>1|0)|0);f=J[5996];J[5996]=0;if((f|0)==1){break a}e=K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24);h=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);f=h-29|0;}j=a+4|0;l=h>>>0<29;e=l?j:e;eb(g+e|0,e,l?h:f);f=g+(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))|0;H[a|0]=f;H[a+1|0]=f>>>8;H[a+2|0]=f>>>16;H[a+3|0]=f>>>24;b:{if(!g){break b}f=i?b+4|0:k;i=g&7;c:{if(!i){b=f;break c}h=0;b=f;while(1){H[e|0]=K[b|0];b=b+1|0;e=e+1|0;h=h+1|0;if((i|0)!=(h|0)){continue}break}}if(g>>>0<8){break b}f=f+g|0;while(1){H[e|0]=K[b|0];H[e+1|0]=K[b+1|0];H[e+2|0]=K[b+2|0];H[e+3|0]=K[b+3|0];H[e+4|0]=K[b+4|0];H[e+5|0]=K[b+5|0];H[e+6|0]=K[b+6|0];H[e+7|0]=K[b+7|0];e=e+8|0;b=b+8|0;if((f|0)!=(b|0)){continue}break}}k=K[c+8|0]|K[c+9|0]<<8|(K[c+10|0]<<16|K[c+11|0]<<24);e=K[c|0]|K[c+1|0]<<8|(K[c+2|0]<<16|K[c+3|0]<<24);h=e>>>0<29;b=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);f=b-29|0;g=h?e:e-29|0;i=b>>>0<29?b:f;e=g+i|0;if(e>>>0>(b>>>0<=28?28:K[a+4|0]|K[a+5|0]<<8|(K[a+6|0]<<16|K[a+7|0]<<24))>>>0){J[5996]=0;aa(124,a|0,e+(e>>>1|0)|0);b=J[5996];J[5996]=0;if((b|0)==1){break a}b=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);f=b-29|0;}e=j;j=b>>>0<29;e=i+(j?e:K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24))|0;eb(g+e|0,e,(j?b:f)-i|0);b=g+(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))|0;H[a|0]=b;H[a+1|0]=b>>>8;H[a+2|0]=b>>>16;H[a+3|0]=b>>>24;d:{if(!g){break d}c=h?c+4|0:k;f=g&7;e:{if(!f){b=c;break e}h=0;b=c;while(1){H[e|0]=K[b|0];b=b+1|0;e=e+1|0;h=h+1|0;if((f|0)!=(h|0)){continue}break}}if(g>>>0<8){break d}c=c+g|0;while(1){H[e|0]=K[b|0];H[e+1|0]=K[b+1|0];H[e+2|0]=K[b+2|0];H[e+3|0]=K[b+3|0];H[e+4|0]=K[b+4|0];H[e+5|0]=K[b+5|0];H[e+6|0]=K[b+6|0];H[e+7|0]=K[b+7|0];e=e+8|0;b=b+8|0;if((c|0)!=(b|0)){continue}break}}b=J[d>>2];J[5996]=0;_(127,a|0,b|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break a}return}b=Z()|0;if((K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))>>>0>=29){$a(K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24));H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;}da(b|0);B();}function Zn(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=+d;e=e|0;var f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0;f=Ua+-64|0;Ua=f;J[f+40>>2]=0;J[f+32>>2]=0;J[f+36>>2]=0;k=J[c>>2];n=J[c+4>>2];a:{b:{if((k|0)==(n|0)){c=k;break b}c:{d:{while(1){e:{j=J[k>>2];g=J[f+36>>2];f:{if((g|0)!=J[f+40>>2]){J[g>>2]=J[j>>2];i=J[j+4>>2];J[g+4>>2]=i;if(i){J[i+4>>2]=J[i+4>>2]+1;}J[f+36>>2]=g+8;break f}g:{i=J[f+32>>2];h=g-i|0;o=h>>3;l=o+1|0;h:{if(l>>>0>=536870912){J[5996]=0;ca(59);a=J[5996];J[5996]=0;if((a|0)==1){break h}break d}m=h>>2;h=h>>>0>=2147483640?536870911:l>>>0<m>>>0?m:l;if(h>>>0<536870912){break g}J[5996]=0;ca(35);a=J[5996];J[5996]=0;if((a|0)!=1){break d}}g=Z()|0;break a}J[5996]=0;m=h<<3;l=$(3,m|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break e}h=l+(o<<3)|0;J[h>>2]=J[j>>2];j=J[j+4>>2];J[h+4>>2]=j;if(j){J[j+4>>2]=J[j+4>>2]+1;g=J[f+36>>2];}j=l+m|0;l=h+8|0;i:{if((g|0)==(i|0)){J[f+40>>2]=j;J[f+36>>2]=l;J[f+32>>2]=h;break i}while(1){h=h-8|0;g=g-8|0;J[h>>2]=J[g>>2];J[h+4>>2]=J[g+4>>2];J[g>>2]=0;J[g+4>>2]=0;if((g|0)!=(i|0)){continue}break}J[f+40>>2]=j;g=J[f+36>>2];J[f+36>>2]=l;i=J[f+32>>2];J[f+32>>2]=h;if((g|0)==(i|0)){break i}while(1){g=g-8|0;h=J[g+4>>2];j:{if(!h){break j}j=J[h+4>>2];J[h+4>>2]=j-1;if(j){break j}Xa[J[J[h>>2]+8>>2]](h);bb(h);}if((g|0)!=(i|0)){continue}break}}if(!i){break f}$a(i);}k=k+4|0;if((n|0)!=(k|0)){continue}break c}break}g=Z()|0;break a}B();}k=J[c+4>>2];c=J[c>>2];}J[5996]=0;va(63,f+16|0,b|0,c|0,k|0,+d);c=J[5996];J[5996]=0;k:{if((c|0)==1){break k}J[5996]=0;c=$(3,104)|0;k=J[5996];J[5996]=0;if((k|0)==1){break k}J[c>>2]=7e3;J[c+4>>2]=0;J[c+8>>2]=0;J[f+52>>2]=J[f+32>>2];k=J[f+40>>2];g=J[f+36>>2];J[f+40>>2]=0;J[f+56>>2]=g;J[f+60>>2]=k;J[f+32>>2]=0;J[f+36>>2]=0;e=J[e>>2];J[5996]=0;k=ia(64,c+12|0,b|0,f+52|0,e|0)|0;b=J[5996];J[5996]=0;if((b|0)!=1){b=J[f+52>>2];if(b){g=J[f+56>>2];if((b|0)!=(g|0)){while(1){g=g-8|0;e=J[g+4>>2];l:{if(!e){break l}i=J[e+4>>2];J[e+4>>2]=i-1;if(i){break l}Xa[J[J[e>>2]+8>>2]](e);bb(e);}if((b|0)!=(g|0)){continue}break}b=J[f+52>>2];}$a(b);}J[f+48>>2]=c;J[f+44>>2]=k;J[5996]=0;b=J[f+48>>2];J[f+8>>2]=J[f+44>>2];J[f+12>>2]=b;ea(61,a|0,f+16|0,f+8|0);a=J[5996];J[5996]=0;if((a|0)==1){break k}a=J[f+32>>2];if(a){g=J[f+36>>2];if((a|0)!=(g|0)){while(1){g=g-8|0;b=J[g+4>>2];m:{if(!b){break m}c=J[b+4>>2];J[b+4>>2]=c-1;if(c){break m}Xa[J[J[b>>2]+8>>2]](b);bb(b);}if((a|0)!=(g|0)){continue}break}a=J[f+32>>2];}$a(a);}Ua=f- -64|0;return}g=Z()|0;lb(f+52|0);$a(c);break a}g=Z()|0;}lb(f+32|0);da(g|0);B();}function $g(a,b,c,d){a=a|0;b=+b;c=+c;d=d|0;var e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0;g=Ua-48|0;Ua=g;a:{b:{if(J[a>>2]==21){j=d+4|0;c:{d:{e:{f:{e=J[d+4>>2];if(!e){break f}f=j;while(1){g:{h:{i:{i=J[e+16>>2];if(i>>>0<a>>>0){break i}if(a>>>0<i>>>0){break h}k=O[e+24>>3];if(k<b){break i}if(b<k){break h}h=O[e+32>>3]<c;i=h?e+4|0:e;f=h?f:e;break g}i=e+4|0;break g}i=e;f=e;}e=J[i>>2];if(e){continue}break}if((f|0)==(j|0)){break f}e=J[f+16>>2];if(e>>>0>a>>>0){break f}j:{if(a>>>0>e>>>0){break j}k=O[f+24>>3];if(k>b){break f}if(b>k){break j}if(O[f+32>>3]>c){break f}}if(O[f+56>>3]!=b){break e}if(O[f- -64>>3]!=c){break d}l=f+40|0;break a}J[g+40>>2]=0;O[g+32>>3]=c;O[g+24>>3]=b;J[g+16>>2]=0;J[g+8>>2]=0;J[g+12>>2]=0;e=J[a+4>>2];if((e|0)==J[a+8>>2]){J[5996]=0;ga(44,4277,1900,673,1984);a=J[5996];J[5996]=0;if((a|0)!=1){break c}break b}e=J[e>>2];J[5996]=0;ea(45,e|0,g+8|0,d|0);e=J[5996];J[5996]=0;if((e|0)==1){break b}k:{l:{e=j;i=J[e>>2];m:{if(!i){break m}while(1){n:{o:{e=i;f=J[e+16>>2];p:{if(f>>>0>a>>>0){break p}if(a>>>0>f>>>0){break n}k=O[e+24>>3];if(k>b){break p}if(b>k){break n}k=O[e+32>>3];if(!(k>c)){break o}}j=e;i=J[e>>2];if(i){continue}break m}if(!(c>k)){break l}}i=J[e+4>>2];if(i){continue}break}j=e+4|0;}J[5996]=0;h=$(3,80)|0;f=J[5996];J[5996]=0;if((f|0)==1){break b}J[h+16>>2]=a;O[h+32>>3]=c;O[h+24>>3]=b;l=h+40|0;J[l>>2]=J[g+8>>2];J[h+44>>2]=J[g+12>>2];J[h+48>>2]=J[g+16>>2];J[g+16>>2]=0;J[g+8>>2]=0;J[g+12>>2]=0;J[h+72>>2]=J[g+40>>2];i=J[g+36>>2];f=h- -64|0;J[f>>2]=J[g+32>>2];J[f+4>>2]=i;f=J[g+28>>2];J[h+56>>2]=J[g+24>>2];J[h+60>>2]=f;J[h+8>>2]=e;J[h>>2]=0;J[h+4>>2]=0;J[j>>2]=h;e=h;f=J[J[d>>2]>>2];if(f){J[d>>2]=f;e=J[j>>2];}zd(J[d+4>>2],e);J[d+8>>2]=J[d+8>>2]+1;j=J[a+4>>2];if(J[a+8>>2]-j>>>0>=5){f=J[h+72>>2];i=1;e=1;while(1){j=J[(e<<2)+j>>2];J[5996]=0;ea(45,j|0,l|0,d|0);j=J[5996];J[5996]=0;if((j|0)==1){break k}j=f;f=J[h+72>>2];i=(j|0)==(f|0)?i:e;e=e+1|0;j=J[a+4>>2];e=(e|0)==J[a+8>>2]-j>>2?1:e;if((i|0)!=(e|0)){continue}break}}d=J[g+8>>2];if(!d){break a}a=J[g+12>>2];if((d|0)!=(a|0)){while(1){a=a-32|0;e=J[a+4>>2];q:{if(!e){break q}f=J[e+4>>2];J[e+4>>2]=f-1;if(f){break q}Xa[J[J[e>>2]+8>>2]](e);bb(e);}if((a|0)!=(d|0)){continue}break}d=J[g+8>>2];}$a(d);break a}J[5996]=0;ga(44,2453,1900,676,1984);a=J[5996];J[5996]=0;if((a|0)!=1){break c}break b}break b}la(1940,1900,668,1984);B();}la(1841,1900,669,1984);}B();}la(3430,1900,664,1984);B();}a=Z()|0;_b(g+8|0);da(a|0);B();}Ua=g+48|0;return l|0}function hk(a,b,c,d,e,f,g,h,i){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;i=i|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0;j=Ua-1056|0;Ua=j;J[j+16>>2]=f;J[j+20>>2]=g;J[j+24>>2]=h;J[j+28>>2]=i;a=j+944|0;J[j+940>>2]=a;l=od(a,100,2304,j+16|0);J[j+48>>2]=184;n=j+520|0;J[n>>2]=0;J[n+4>>2]=J[j+48>>2];J[j+48>>2]=184;m=j+512|0;J[m>>2]=0;J[m+4>>2]=J[j+48>>2];a:{b:{c:{d:{if(l>>>0<100){h=j+528|0;break d}a=jb();J[j>>2]=f;J[j+4>>2]=g;J[j+8>>2]=h;J[j+12>>2]=i;l=dc(j+940|0,a,2304,j);if((l|0)!=-1){a=J[j+940>>2];tb(n,a);tb(m,sb(l<<2));h=J[m>>2];if(h){break d}}J[5996]=0;ca(185);a=J[5996];J[5996]=0;if((a|0)!=1){break a}l=Z()|0;break c}f=J[d+28>>2];J[j+508>>2]=f;J[f+4>>2]=J[f+4>>2]+1;J[5996]=0;r=$(197,J[j+508>>2])|0;f=J[5996];J[5996]=0;e:{f:{g:{h:{i:{j:{if((f|0)==1){break j}J[5996]=0;ga(208,r|0,a|0,a+l|0,h|0);f=J[5996];J[5996]=0;if((f|0)==1){break j}p=(l|0)>0?K[a|0]==45:p;g=j+484|0;J[g>>2]=0;J[g+4>>2]=0;J[g+8>>2]=0;a=j+472|0;J[a>>2]=0;J[a+4>>2]=0;J[a+8>>2]=0;f=j+460|0;J[f>>2]=0;J[f+4>>2]=0;J[f+8>>2]=0;J[5996]=0;sa(238,c|0,p|0,J[j+508>>2],j+504|0,j+500|0,j+496|0,g|0,a|0,f|0,j+456|0);c=J[5996];J[5996]=0;if((c|0)==1){break i}J[j+36>>2]=184;c=j+40|0;J[c>>2]=0;J[c+4>>2]=J[j+36>>2];i=j+48|0;o=J[j+456>>2];k:{if((o|0)<(l|0)){k=K[f+11|0];q=(l-o<<1)+((k&128)>>>7|0?J[f+4>>2]:k&127)|0;k=K[a+11|0];k=(q+((k&128)>>>7|0?J[a+4>>2]:k&127)|0)+1|0;break k}k=K[f+11|0];q=(k&128)>>>7|0?J[f+4>>2]:k&127;k=K[a+11|0];k=(q+((k&128)>>>7|0?J[a+4>>2]:k&127)|0)+2|0;}k=k+o|0;if(k>>>0<101){break h}tb(c,sb(k<<2));i=J[c>>2];if(i){break h}J[5996]=0;ca(185);b=J[5996];J[5996]=0;if((b|0)!=1){break a}break g}l=Z()|0;break e}l=Z()|0;break f}k=J[d+4>>2];J[5996]=0;ta(239,i|0,j+36|0,j+32|0,k|0,h|0,(l<<2)+h|0,r|0,p|0,j+504|0,J[j+500>>2],J[j+496>>2],g|0,a|0,f|0,o|0);h=J[5996];J[5996]=0;if((h|0)!=1){J[5996]=0;d=ka(219,b|0,i|0,J[j+36>>2],J[j+32>>2],d|0,e|0)|0;b=J[5996];J[5996]=0;if((b|0)!=1){break b}}}l=Z()|0;db(c);}ab(f);ab(a);ab(g);}a=J[j+508>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}}db(m);db(n);da(l|0);B();}db(c);ab(f);ab(a);ab(g);a=J[j+508>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}db(m);db(n);Ua=j+1056|0;return d|0}B();}function ih(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0;c=J[a+4>>2];d=J[a+8>>2];if(c>>>0<d>>>0){d=J[b>>2];J[c+12>>2]=0;J[c+4>>2]=0;J[c+8>>2]=0;J[c>>2]=d;J[c+4>>2]=J[b+4>>2];J[c+8>>2]=J[b+8>>2];J[c+12>>2]=J[b+12>>2];J[b+12>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;J[c+24>>2]=0;J[c+16>>2]=0;J[c+20>>2]=0;J[c+16>>2]=J[b+16>>2];J[c+20>>2]=J[b+20>>2];J[c+24>>2]=J[b+24>>2];J[b+24>>2]=0;J[b+16>>2]=0;J[b+20>>2]=0;J[c+36>>2]=0;J[c+28>>2]=0;J[c+32>>2]=0;J[c+28>>2]=J[b+28>>2];J[c+32>>2]=J[b+32>>2];J[c+36>>2]=J[b+36>>2];J[b+36>>2]=0;J[b+28>>2]=0;J[b+32>>2]=0;J[c+48>>2]=0;J[c+40>>2]=0;J[c+44>>2]=0;J[c+40>>2]=J[b+40>>2];J[c+44>>2]=J[b+44>>2];J[c+48>>2]=J[b+48>>2];J[b+48>>2]=0;J[b+40>>2]=0;J[b+44>>2]=0;J[c+52>>2]=J[b+52>>2];J[a+4>>2]=c+56;return}a:{b:{c:{e=J[a>>2];f=(c-e|0)/56|0;c=f+1|0;if(c>>>0<76695845){d=(d-e|0)/56|0;e=d<<1;c=d>>>0>=38347922?76695844:c>>>0<e>>>0?e:c;if(c>>>0>=76695845){break c}d=P(c,56);e=cb(d);c=e+P(f,56)|0;J[c>>2]=J[b>>2];J[c+4>>2]=J[b+4>>2];J[c+8>>2]=J[b+8>>2];J[c+12>>2]=J[b+12>>2];J[b+12>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;J[c+16>>2]=J[b+16>>2];J[c+20>>2]=J[b+20>>2];J[c+24>>2]=J[b+24>>2];J[b+24>>2]=0;J[b+16>>2]=0;J[b+20>>2]=0;J[c+28>>2]=J[b+28>>2];J[c+32>>2]=J[b+32>>2];J[c+36>>2]=J[b+36>>2];J[b+36>>2]=0;J[b+28>>2]=0;J[b+32>>2]=0;J[c+40>>2]=J[b+40>>2];J[c+44>>2]=J[b+44>>2];J[c+48>>2]=J[b+48>>2];J[b+48>>2]=0;J[b+40>>2]=0;J[b+44>>2]=0;J[c+52>>2]=J[b+52>>2];d=d+e|0;e=c+56|0;b=J[a+4>>2];f=J[a>>2];if((b|0)==(f|0)){break b}while(1){c=c-56|0;b=b-56|0;J[c>>2]=J[b>>2];J[c+4>>2]=J[b+4>>2];J[c+8>>2]=J[b+8>>2];J[c+12>>2]=J[b+12>>2];J[b+12>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;J[c+16>>2]=J[b+16>>2];J[c+20>>2]=J[b+20>>2];J[c+24>>2]=J[b+24>>2];J[b+24>>2]=0;J[b+16>>2]=0;J[b+20>>2]=0;J[c+28>>2]=J[b+28>>2];J[c+32>>2]=J[b+32>>2];J[c+36>>2]=J[b+36>>2];J[b+36>>2]=0;J[b+28>>2]=0;J[b+32>>2]=0;J[c+40>>2]=J[b+40>>2];J[c+44>>2]=J[b+44>>2];J[c+48>>2]=J[b+48>>2];J[b+48>>2]=0;J[b+40>>2]=0;J[b+44>>2]=0;J[c+52>>2]=J[b+52>>2];if((b|0)!=(f|0)){continue}break}J[a+8>>2]=d;d=J[a+4>>2];J[a+4>>2]=e;b=J[a>>2];J[a>>2]=c;if((b|0)==(d|0)){break a}while(1){d=Fb(d-56|0);if((d|0)!=(b|0)){continue}break}break a}nb();B();}Pb();B();}J[a+8>>2]=d;J[a+4>>2]=e;J[a>>2]=c;}if(b){$a(b);}}function lk(a,b,c,d,e,f,g,h,i){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;i=i|0;var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0;j=Ua-448|0;Ua=j;J[j+16>>2]=f;J[j+20>>2]=g;J[j+24>>2]=h;J[j+28>>2]=i;a=j+336|0;J[j+332>>2]=a;l=od(a,100,2304,j+16|0);J[j+48>>2]=184;n=j+216|0;J[n>>2]=0;J[n+4>>2]=J[j+48>>2];J[j+48>>2]=184;m=j+208|0;J[m>>2]=0;J[m+4>>2]=J[j+48>>2];a:{b:{c:{d:{if(l>>>0<100){h=j+224|0;break d}a=jb();J[j>>2]=f;J[j+4>>2]=g;J[j+8>>2]=h;J[j+12>>2]=i;l=dc(j+332|0,a,2304,j);if((l|0)!=-1){a=J[j+332>>2];tb(n,a);tb(m,sb(l));h=J[m>>2];if(h){break d}}J[5996]=0;ca(185);a=J[5996];J[5996]=0;if((a|0)!=1){break a}l=Z()|0;break c}f=J[d+28>>2];J[j+204>>2]=f;J[f+4>>2]=J[f+4>>2]+1;J[5996]=0;r=$(170,J[j+204>>2])|0;f=J[5996];J[5996]=0;e:{f:{g:{h:{i:{j:{if((f|0)==1){break j}J[5996]=0;ga(195,r|0,a|0,a+l|0,h|0);f=J[5996];J[5996]=0;if((f|0)==1){break j}p=(l|0)>0?K[a|0]==45:p;g=j+184|0;J[g>>2]=0;J[g+4>>2]=0;J[g+8>>2]=0;a=j+172|0;J[a>>2]=0;J[a+4>>2]=0;J[a+8>>2]=0;f=j+160|0;J[f>>2]=0;J[f+4>>2]=0;J[f+8>>2]=0;J[5996]=0;sa(236,c|0,p|0,J[j+204>>2],j+200|0,j+199|0,j+198|0,g|0,a|0,f|0,j+156|0);c=J[5996];J[5996]=0;if((c|0)==1){break i}J[j+36>>2]=184;c=j+40|0;J[c>>2]=0;J[c+4>>2]=J[j+36>>2];i=j+48|0;o=J[j+156>>2];k:{if((o|0)<(l|0)){k=K[f+11|0];q=(l-o<<1)+((k&128)>>>7|0?J[f+4>>2]:k&127)|0;k=K[a+11|0];k=(q+((k&128)>>>7|0?J[a+4>>2]:k&127)|0)+1|0;break k}k=K[f+11|0];q=(k&128)>>>7|0?J[f+4>>2]:k&127;k=K[a+11|0];k=(q+((k&128)>>>7|0?J[a+4>>2]:k&127)|0)+2|0;}k=k+o|0;if(k>>>0<101){break h}tb(c,sb(k));i=J[c>>2];if(i){break h}J[5996]=0;ca(185);b=J[5996];J[5996]=0;if((b|0)!=1){break a}break g}l=Z()|0;break e}l=Z()|0;break f}k=J[d+4>>2];J[5996]=0;ta(237,i|0,j+36|0,j+32|0,k|0,h|0,h+l|0,r|0,p|0,j+200|0,H[j+199|0],H[j+198|0],g|0,a|0,f|0,o|0);h=J[5996];J[5996]=0;if((h|0)!=1){J[5996]=0;d=ka(213,b|0,i|0,J[j+36>>2],J[j+32>>2],d|0,e|0)|0;b=J[5996];J[5996]=0;if((b|0)!=1){break b}}}l=Z()|0;db(c);}ab(f);ab(a);ab(g);}a=J[j+204>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}}db(m);db(n);da(l|0);B();}db(c);ab(f);ab(a);ab(g);a=J[j+204>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}db(m);db(n);Ua=j+448|0;return d|0}B();}function Vn(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=+e;var f=0,g=0,h=0,i=0,j=0;a:{b:{if(!(b-19>>>0>=7&e!=0)){f=1-e;c:{switch(b|0){case 24:if((c|0)!=(d|0)){while(1){b=J[c>>2];h=h+O[b+8>>3];g=g+O[b+16>>3];c=c+4|0;if((d|0)!=(c|0)){continue}break}}O[a+8>>3]=g;O[a>>3]=(f==0?0:f*g)+(e==0?0:h*e);return;case 3:J[a+8>>2]=0;J[a+12>>2]=1078034432;J[a>>2]=0;J[a+4>>2]=1079689216;return;case 6:case 7:case 8:case 9:J[a+8>>2]=0;J[a+12>>2]=1077968896;J[a>>2]=0;J[a+4>>2]=1077968896;return;case 10:case 11:case 12:case 16:b=J[c>>2];c=J[b+12>>2];J[a>>2]=J[b+8>>2];J[a+4>>2]=c;c=J[b+20>>2];J[a+8>>2]=J[b+16>>2];J[a+12>>2]=c;return;case 13:e=O[J[c>>2]+8>>3];J[a+8>>2]=0;J[a+12>>2]=1072693248;O[a>>3]=e+2;return;case 14:e=O[J[c>>2]+8>>3];J[a+8>>2]=0;J[a+12>>2]=2146435072;O[a>>3]=e;return;case 15:e=O[J[c>>2]+8>>3];J[a+8>>2]=0;J[a+12>>2]=1072693248;O[a>>3]=e;return;case 0:J[a+8>>2]=0;J[a+12>>2]=0;J[a>>2]=0;J[a+4>>2]=2146435072;return;case 17:e=O[J[c+4>>2]+8>>3];f=O[J[c>>2]+8>>3];J[a+8>>2]=0;J[a+12>>2]=2146435072;O[a>>3]=f+e;return;case 18:b=J[c+4>>2];e=O[b+16>>3];c=J[c>>2];f=O[c+16>>3];O[a>>3]=O[c+8>>3]+O[b+8>>3];O[a+8>>3]=f+e;return;case 19:b=J[c>>2];g=O[b+8>>3];c=J[c+4>>2];h=O[c+8>>3];i=O[c+16>>3];j=O[b+16>>3];O[a+8>>3]=i+j;O[a>>3]=(e==0?0:(g+i)*e)+(f==0?0:f*(j+h));return;case 20:case 21:b=J[c+4>>2];g=O[b+8>>3];c=J[c>>2];h=O[c+8>>3];i=O[c+16>>3];O[a+8>>3]=i+O[b+16>>3];O[a>>3]=(e==0?0:h*e)+(f==0?0:f*(i+g));return;case 22:b=J[c>>2];g=O[b+8>>3];c=J[c+4>>2];h=O[c+8>>3];i=O[c+16>>3]+1;j=O[b+16>>3]+2;O[a+8>>3]=i<j?i:j;O[a>>3]=(e==0?0:(g+2)*e)+(f==0?0:f*(h+1));return;case 23:b=J[c+8>>2];g=O[b+8>>3];h=O[J[c+4>>2]+8>>3];c=J[c>>2];i=O[c+8>>3];j=O[c+16>>3];O[a+8>>3]=j+O[b+16>>3];O[a>>3]=(e==0?0:(i+h)*e)+(f==0?0:f*(j+g));return;default:a=Rb(8);J[5996]=0;b=_(91,a|0,1318)|0;c=J[5996];J[5996]=0;if((c|0)!=1){break b}b=Z()|0;ic(a);da(b|0);B();case 1:case 4:case 5:break a;case 2:break c}}J[a+8>>2]=0;J[a+12>>2]=1072693248;J[a>>2]=0;J[a+4>>2]=1079132160;return}la(4323,1900,476,1746);B();}na(b|0,22664,92);B();}J[a+8>>2]=0;J[a+12>>2]=2146435072;J[a>>2]=0;J[a+4>>2]=0;}function Hg(a,b,c,d,e,f,g,h,i){var j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0;j=Ua-128|0;Ua=j;a:{b:{c:{if(!yc(f,g,h,i,0,0,0,0)){break c}k=i&65535;n=i>>>16&32767;d:{e:{if((n|0)!=32767){l=4;if(n){break e}l=f|h|(g|k)?3:2;break d}l=!(f|h|(g|k));}}s=e>>>16|0;o=s&32767;if((o|0)==32767){break c}if(l){break b}}ob(j+16|0,b,c,d,e,f,g,h,i);b=J[j+16>>2];d=J[j+20>>2];e=J[j+24>>2];c=J[j+28>>2];Ng(j,b,d,e,c,b,d,e,c);d=J[j+8>>2];e=J[j+12>>2];h=J[j>>2];i=J[j+4>>2];break a}l=d;p=e&2147483647;k=p;n=h;m=i&2147483647;if((yc(b,c,l,k,f,g,h,m)|0)<=0){if(yc(b,c,l,k,f,g,h,m)){h=b;i=c;break a}ob(j+112|0,b,c,d,e,0,0,0,0);d=J[j+120>>2];e=J[j+124>>2];h=J[j+112>>2];i=J[j+116>>2];break a}q=i>>>16&32767;if(o){i=c;h=b;}else {ob(j+96|0,b,c,l,p,0,0,0,1081540608);l=J[j+104>>2];h=J[j+108>>2];p=h;o=(h>>>16|0)-120|0;i=J[j+100>>2];h=J[j+96>>2];}if(!q){ob(j+80|0,f,g,n,m,0,0,0,1081540608);n=J[j+88>>2];f=J[j+92>>2];m=f;q=(f>>>16|0)-120|0;g=J[j+84>>2];f=J[j+80>>2];}r=n;t=m&65535|65536;p=p&65535|65536;if((o|0)>(q|0)){while(1){m=l-r|0;k=(g|0)==(i|0)&f>>>0>h>>>0|g>>>0>i>>>0;n=m-k|0;k=(p-((l>>>0<r>>>0)+t|0)|0)-(k>>>0>m>>>0)|0;f:{if((k|0)>0|(k|0)>=0){l=h;h=h-f|0;i=i-((f>>>0>l>>>0)+g|0)|0;if(!(h|n|(i|k))){ob(j+32|0,b,c,d,e,0,0,0,0);d=J[j+40>>2];e=J[j+44>>2];h=J[j+32>>2];i=J[j+36>>2];break a}k=k<<1|n>>>31;l=n<<1|i>>>31;break f}k=p<<1|l>>>31;l=l<<1|i>>>31;}p=k;k=i<<1|h>>>31;h=h<<1;i=k;o=o-1|0;if((o|0)>(q|0)){continue}break}o=q;}m=l-r|0;k=(g|0)==(i|0)&f>>>0>h>>>0|g>>>0>i>>>0;n=m-k|0;k=(p-((l>>>0<r>>>0)+t|0)|0)-(k>>>0>m>>>0)|0;m=k;g:{if((k|0)<0){n=l;m=p;break g}l=h;h=h-f|0;i=i-((f>>>0>l>>>0)+g|0)|0;if(h|n|(i|m)){break g}ob(j+48|0,b,c,d,e,0,0,0,0);d=J[j+56>>2];e=J[j+60>>2];h=J[j+48>>2];i=J[j+52>>2];break a}if((m|0)==65535|m>>>0<65535){while(1){b=i>>>31|0;o=o-1|0;p=i<<1|h>>>31;h=h<<1;i=p;c=b;b=m<<1|n>>>31;n=c|n<<1;m=b;if(b>>>0<65536){continue}break}}b=s&32768;if((o|0)<=0){ob(j- -64|0,h,i,n,m&65535|(b|o+120)<<16,0,0,0,1065811968);d=J[j+72>>2];e=J[j+76>>2];h=J[j+64>>2];i=J[j+68>>2];break a}d=n;e=m&65535|(b|o)<<16;}J[a>>2]=h;J[a+4>>2]=i;J[a+8>>2]=d;J[a+12>>2]=e;Ua=j+128|0;}function tn(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0,h=0,i=0;h=J[b>>2];f=K[b+11|0];g=f<<24>>24;c=(g|0)<0;i=J[b+4>>2];d=c?i:f;a:{if((d|0)<9){break a}f=c?h:b;e=f+d|0;while(1){c=f;while(1){c=Zb(c,115,d-8|0);if(!c){break a}if(Eb(c,4824,9)){c=c+1|0;d=e-c|0;if((d|0)>=9){continue}break a}break}if((c|0)==(e|0)){break a}c=c-f|0;if((c|0)==-1){break a}Lc(b,c,9,5034);h=J[b>>2];c=K[b+11|0];g=c<<24>>24;e=(g|0)<0;f=e?h:b;i=J[b+4>>2];d=e?i:c;e=f+d|0;if((d|0)>8){continue}break}}c=g<<24>>24;d=(c|0)<0?i:g&255;b:{if((d|0)<10){break b}f=(c|0)<0?h:b;e=f+d|0;while(1){c=f;while(1){c=Zb(c,104,d-9|0);if(!c){break b}if(Eb(c,4813,10)){c=c+1|0;d=e-c|0;if((d|0)>=10){continue}break b}break}if((c|0)==(e|0)){break b}c=c-f|0;if((c|0)==-1){break b}Lc(b,c,10,4960);h=J[b>>2];c=K[b+11|0];g=c<<24>>24;e=(g|0)<0;f=e?h:b;i=J[b+4>>2];d=e?i:c;e=f+d|0;if((d|0)>9){continue}break}}c=g<<24>>24<0;d=c?i:g&255;c:{if((d|0)<12){break c}f=c?h:b;e=f+d|0;while(1){c=f;while(1){c=Zb(c,114,d-11|0);if(!c){break c}if(Eb(c,4845,12)){c=c+1|0;d=e-c|0;if((d|0)>=12){continue}break c}break}if((c|0)==(e|0)){break c}c=c-f|0;if((c|0)==-1){break c}Lc(b,c,12,4908);h=J[b>>2];c=K[b+11|0];g=c<<24>>24;e=(g|0)<0;f=e?h:b;i=J[b+4>>2];d=e?i:c;e=f+d|0;if((d|0)>11){continue}break}}c=g<<24>>24;d=(c|0)<0?i:g&255;d:{if((d|0)<10){break d}f=(c|0)<0?h:b;e=f+d|0;while(1){c=f;while(1){c=Zb(c,104,d-9|0);if(!c){break d}if(Eb(c,4834,10)){c=c+1|0;d=e-c|0;if((d|0)>=10){continue}break d}break}if((c|0)==(e|0)){break d}c=c-f|0;if((c|0)==-1){break d}Lc(b,c,10,4858);h=J[b>>2];c=K[b+11|0];g=c<<24>>24;e=(g|0)<0;f=e?h:b;i=J[b+4>>2];d=e?i:c;e=f+d|0;if((d|0)>9){continue}break}}c=g<<24>>24<0;d=c?i:g&255;e:{if((d|0)<=0){break e}e=c?h:b;g=e+d|0;while(1){c=e;while(1){c=Zb(c,32,d);if(!c){break e}if(K[c|0]!=32){c=c+1|0;d=g-c|0;if((d|0)>0){continue}break e}break}if((c|0)==(g|0)){break e}c=c-e|0;if((c|0)==-1){break e}Lc(b,c,1,5575);c=K[b+11|0];f=c<<24>>24<0;e=f?J[b>>2]:b;d=f?J[b+4>>2]:c;g=e+d|0;if((d|0)>0){continue}break}}c=J[b+4>>2];J[a>>2]=J[b>>2];J[a+4>>2]=c;J[a+8>>2]=J[b+8>>2];J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;}
function Sk(a,b,c,d,e,f,g){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;var h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0;k=Ua-16|0;Ua=k;l=je(J[g>>2]);o=Zd(J[g>>2]);g=o;Xa[J[J[g>>2]+20>>2]](k+4|0,g);J[f>>2]=d;a:{b:{c:{i=a;g=K[i|0];switch(g-43|0){case 0:case 2:break c;default:break b}}J[5996]=0;g=_(216,l|0,g<<24>>24)|0;i=J[5996];J[5996]=0;if((i|0)==1){break a}i=J[f>>2];J[f>>2]=i+4;J[i>>2]=g;i=a+1|0;}d:{g=i;if(!((c-g|0)<=1|K[g|0]!=48|(K[g+1|0]|32)!=120)){J[5996]=0;i=_(216,l|0,48)|0;h=J[5996];J[5996]=0;if((h|0)==1){break a}h=J[f>>2];J[f>>2]=h+4;J[h>>2]=i;i=H[g+1|0];J[5996]=0;i=_(216,l|0,i|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break a}h=J[f>>2];J[f>>2]=h+4;J[h>>2]=i;i=g+2|0;g=i;while(1){if(c>>>0<=g>>>0){break d}h=H[g|0];jb();if(!(h-48>>>0<10|(h|32)-97>>>0<6)){break d}g=g+1|0;continue}}while(1){if(c>>>0<=g>>>0){break d}h=H[g|0];jb();if(h-48>>>0>=10){break d}g=g+1|0;continue}}e:{h=K[k+15|0];if(!((h&128)>>>7|0?J[k+8>>2]:h&127)){h=J[f>>2];J[5996]=0;ga(208,l|0,i|0,g|0,h|0);h=J[5996];J[5996]=0;if((h|0)==1){break a}J[f>>2]=J[f>>2]+(g-i<<2);break e}vc(i,g);J[5996]=0;q=$(206,o|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break a}h=i;while(1){if(g>>>0<=h>>>0){md((i-a<<2)+d|0,J[f>>2]);break e}j=k+4|0;m=(K[j+11|0]&128)>>>7|0;if(!(H[(m?J[j>>2]:j)+n|0]<=0|H[(m?J[j>>2]:j)+n|0]!=(p|0))){j=J[f>>2];J[f>>2]=j+4;J[j>>2]=q;p=0;j=K[k+15|0];n=(((j&128)>>>7|0?J[k+8>>2]:j&127)-1>>>0>n>>>0)+n|0;}j=H[h|0];J[5996]=0;j=_(216,l|0,j|0)|0;m=J[5996];J[5996]=0;if((m|0)!=1){m=J[f>>2];J[f>>2]=m+4;J[m>>2]=j;h=h+1|0;p=p+1|0;continue}break}break a}f:{while(1){if(c>>>0>g>>>0){i=K[g|0];if((i|0)==46){J[5996]=0;i=$(209,o|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break a}j=J[f>>2];h=j+4|0;J[f>>2]=h;J[j>>2]=i;g=g+1|0;break f}J[5996]=0;i=_(216,l|0,i<<24>>24)|0;h=J[5996];J[5996]=0;if((h|0)==1){break a}h=J[f>>2];J[f>>2]=h+4;J[h>>2]=i;g=g+1|0;continue}break}h=J[f>>2];}J[5996]=0;ga(208,l|0,g|0,c|0,h|0);i=J[5996];J[5996]=0;if((i|0)==1){break a}i=f;f=J[f>>2]+(c-g<<2)|0;J[i>>2]=f;J[e>>2]=(b|0)==(c|0)?f:(b-a<<2)+d|0;ab(k+4|0);Ua=k+16|0;return}a=Z()|0;ab(k+4|0);da(a|0);B();}function bl(a,b,c,d,e,f,g){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;var h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0;k=Ua-16|0;Ua=k;l=le(J[g>>2]);o=de(J[g>>2]);g=o;Xa[J[J[g>>2]+20>>2]](k+4|0,g);J[f>>2]=d;a:{b:{c:{i=a;g=K[i|0];switch(g-43|0){case 0:case 2:break c;default:break b}}J[5996]=0;g=_(171,l|0,g<<24>>24)|0;i=J[5996];J[5996]=0;if((i|0)==1){break a}i=J[f>>2];J[f>>2]=i+1;H[i|0]=g;i=a+1|0;}d:{g=i;if(!((c-g|0)<=1|K[g|0]!=48|(K[g+1|0]|32)!=120)){J[5996]=0;i=_(171,l|0,48)|0;h=J[5996];J[5996]=0;if((h|0)==1){break a}h=J[f>>2];J[f>>2]=h+1;H[h|0]=i;i=H[g+1|0];J[5996]=0;i=_(171,l|0,i|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break a}h=J[f>>2];J[f>>2]=h+1;H[h|0]=i;i=g+2|0;g=i;while(1){if(c>>>0<=g>>>0){break d}h=H[g|0];jb();if(!(h-48>>>0<10|(h|32)-97>>>0<6)){break d}g=g+1|0;continue}}while(1){if(c>>>0<=g>>>0){break d}h=H[g|0];jb();if(h-48>>>0>=10){break d}g=g+1|0;continue}}e:{h=K[k+15|0];if(!((h&128)>>>7|0?J[k+8>>2]:h&127)){h=J[f>>2];J[5996]=0;ga(195,l|0,i|0,g|0,h|0);h=J[5996];J[5996]=0;if((h|0)==1){break a}J[f>>2]=J[f>>2]+(g-i|0);break e}vc(i,g);J[5996]=0;q=$(193,o|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break a}h=i;while(1){if(g>>>0<=h>>>0){vc((i-a|0)+d|0,J[f>>2]);break e}j=k+4|0;m=(K[j+11|0]&128)>>>7|0;if(!(H[(m?J[j>>2]:j)+n|0]<=0|H[(m?J[j>>2]:j)+n|0]!=(p|0))){j=J[f>>2];J[f>>2]=j+1;H[j|0]=q;p=0;j=K[k+15|0];n=(((j&128)>>>7|0?J[k+8>>2]:j&127)-1>>>0>n>>>0)+n|0;}j=H[h|0];J[5996]=0;j=_(171,l|0,j|0)|0;m=J[5996];J[5996]=0;if((m|0)!=1){m=J[f>>2];J[f>>2]=m+1;H[m|0]=j;h=h+1|0;p=p+1|0;continue}break}break a}while(1){f:{if(c>>>0>g>>>0){i=K[g|0];if((i|0)!=46){break f}J[5996]=0;i=$(196,o|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break a}h=J[f>>2];J[f>>2]=h+1;H[h|0]=i;g=g+1|0;}i=J[f>>2];J[5996]=0;ga(195,l|0,g|0,c|0,i|0);i=J[5996];J[5996]=0;if((i|0)==1){break a}i=f;f=J[f>>2]+(c-g|0)|0;J[i>>2]=f;J[e>>2]=(b|0)==(c|0)?f:(b-a|0)+d|0;ab(k+4|0);Ua=k+16|0;return}J[5996]=0;i=_(171,l|0,i<<24>>24)|0;h=J[5996];J[5996]=0;if((h|0)==1){break a}h=J[f>>2];J[f>>2]=h+1;H[h|0]=i;g=g+1|0;continue}}a=Z()|0;ab(k+4|0);da(a|0);B();}function bg(a,b,c,d){var e=0,f=0,g=0,h=0,i=0;e=J[b>>2];a:{b:{c:{d:{e:{f:{g:{h:{i:{j:{k:{l:{if(!d){break l}g=J[d>>2];if(!g){break l}if(!a){d=c;break j}J[d>>2]=0;d=c;break k}m:{if(!J[J[6178]>>2]){if(!a){break m}if(!c){break a}g=c;while(1){d=H[e|0];if(d){J[a>>2]=d&57343;a=a+4|0;e=e+1|0;g=g-1|0;if(g){continue}break a}break}J[a>>2]=0;J[b>>2]=0;return c-g|0}d=c;if(!a){break i}break g}return Gb(e)}f=1;break g}f=0;break h}f=1;}while(1){if(!f){f=K[e|0]>>>3|0;if((f-16|f+(g>>26))>>>0>7){break f}f=e+1|0;n:{if(!(g&33554432)){break n}if((K[f|0]&192)!=128){e=e-1|0;break d}f=e+2|0;if(!(g&524288)){break n}if((K[f|0]&192)!=128){e=e-1|0;break d}f=e+3|0;}e=f;d=d-1|0;f=1;continue}while(1){g=K[e|0];o:{if(e&3|g-1>>>0>126){break o}g=J[e>>2];if((g|g-16843009)&-2139062144){break o}while(1){d=d-4|0;g=J[e+4>>2];e=e+4|0;if(!((g-16843009|g)&-2139062144)){continue}break}}f=g&255;if(f-1>>>0<=126){d=d-1|0;e=e+1|0;continue}break}f=f-194|0;if(f>>>0>50){break e}e=e+1|0;g=J[(f<<2)+12208>>2];f=0;continue}}while(1){if(!f){if(!d){break a}while(1){p:{f=K[e|0];h=f-1|0;q:{r:{if(h>>>0>126){g=f;break r}if(e&3|d>>>0<5){break q}s:{while(1){g=J[e>>2];if((g|g-16843009)&-2139062144){break s}J[a>>2]=g&255;J[a+4>>2]=K[e+1|0];J[a+8>>2]=K[e+2|0];J[a+12>>2]=K[e+3|0];a=a+16|0;e=e+4|0;d=d-4|0;if(d>>>0>4){continue}break}g=K[e|0];}f=g&255;h=f-1|0;}if(h>>>0>126){break p}}J[a>>2]=f;a=a+4|0;e=e+1|0;d=d-1|0;if(d){continue}break a}break}f=f-194|0;if(f>>>0>50){break e}e=e+1|0;g=J[(f<<2)+12208>>2];f=1;continue}i=K[e|0];f=i>>>3|0;if((f-16|f+(g>>26))>>>0>7){break f}t:{u:{h=e+1|0;f=i-128|g<<6;v:{if((f|0)>=0){break v}i=K[h|0]-128|0;if(i>>>0>63){break u}h=e+2|0;f=i|f<<6;if((f|0)>=0){break v}h=K[h|0]-128|0;if(h>>>0>63){break u}f=h|f<<6;h=e+3|0;}e=h;J[a>>2]=f;d=d-1|0;a=a+4|0;break t}J[5732]=25;e=e-1|0;break c}f=0;continue}}e=e-1|0;if(g){break d}g=K[e|0];}if(g&255){break d}if(a){J[a>>2]=0;J[b>>2]=0;}return c-d|0}J[5732]=25;if(!a){break b}}J[b>>2]=e;}return -1}J[b>>2]=e;return c}function ul(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0;a=Ua-256|0;Ua=a;J[a+248>>2]=c;J[a+252>>2]=b;h=a+196|0;J[h>>2]=0;J[h+4>>2]=0;J[h+8>>2]=0;g=a+16|0;b=J[d+28>>2];J[g>>2]=b;J[b+4>>2]=J[b+4>>2]+1;J[5996]=0;b=$(170,J[a+16>>2])|0;c=J[5996];J[5996]=0;a:{b:{if((c|0)==1){break b}J[5996]=0;ga(195,b|0,12416,12442,a+208|0);b=J[5996];J[5996]=0;if((b|0)==1){break b}b=J[a+16>>2];c=J[b+4>>2]-1|0;J[b+4>>2]=c;if((c|0)==-1){Xa[J[J[b>>2]+8>>2]](b);}c=a+184|0;J[c>>2]=0;J[c+4>>2]=0;J[c+8>>2]=0;b=(K[c+11|0]&128)>>>7|0?(J[c+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,c|0,b|0);b=J[5996];J[5996]=0;if((b|0)!=1){c:{b=(K[c+11|0]&128)>>>7|0?J[c>>2]:c;J[a+180>>2]=b;J[a+12>>2]=g;J[a+8>>2]=0;while(1){d:{J[5996]=0;g=_(186,a+252|0,a+248|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break d}d=J[a+180>>2];e:{if(!g){g=d;d=K[c+11|0];d=(d&128)>>>7|0?J[c+4>>2]:d&127;if((g|0)==(d+b|0)){J[5996]=0;aa(172,c|0,d<<1);b=J[5996];J[5996]=0;if((b|0)==1){break c}b=(K[c+11|0]&128)>>>7|0?(J[c+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,c|0,b|0);b=J[5996];J[5996]=0;if((b|0)==1){break c}b=(K[c+11|0]&128)>>>7|0?J[c>>2]:c;J[a+180>>2]=d+b;}J[5996]=0;d=$(187,J[a+252>>2])|0;g=J[5996];J[5996]=0;if((g|0)==1){break d}if(!Dc(d,16,b,a+180|0,a+8|0,0,J[h+4>>2],K[h+11|0],a+16|0,a+12|0,a+208|0)){break e}d=J[a+180>>2];}J[5996]=0;aa(172,c|0,d-b|0);b=J[5996];J[5996]=0;if((b|0)==1){break d}b=J[c>>2];d=(K[c+11|0]&128)>>>7|0;g=jb();J[a>>2]=f;if((Kf(d?b:c,g,a)|0)!=1){J[e>>2]=4;}J[5996]=0;b=_(186,a+252|0,a+248|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break d}if(b){J[e>>2]=J[e>>2]|2;}b=J[a+252>>2];ab(c);ab(h);Ua=a+256|0;return b|0}J[5996]=0;$(189,a+252|0)|0;d=J[5996];J[5996]=0;if((d|0)!=1){continue}}break}}}b=Z()|0;ab(c);break a}b=Z()|0;a=J[a+16>>2];c=J[a+4>>2]-1|0;J[a+4>>2]=c;if((c|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}}ab(h);da(b|0);B();}function jl(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0;a=Ua-320|0;Ua=a;J[a+312>>2]=c;J[a+316>>2]=b;h=a+196|0;J[h>>2]=0;J[h+4>>2]=0;J[h+8>>2]=0;g=a+16|0;b=J[d+28>>2];J[g>>2]=b;J[b+4>>2]=J[b+4>>2]+1;J[5996]=0;b=$(197,J[a+16>>2])|0;c=J[5996];J[5996]=0;a:{b:{if((c|0)==1){break b}J[5996]=0;ga(208,b|0,12416,12442,a+208|0);b=J[5996];J[5996]=0;if((b|0)==1){break b}b=J[a+16>>2];c=J[b+4>>2]-1|0;J[b+4>>2]=c;if((c|0)==-1){Xa[J[J[b>>2]+8>>2]](b);}c=a+184|0;J[c>>2]=0;J[c+4>>2]=0;J[c+8>>2]=0;b=(K[c+11|0]&128)>>>7|0?(J[c+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,c|0,b|0);b=J[5996];J[5996]=0;if((b|0)!=1){c:{b=(K[c+11|0]&128)>>>7|0?J[c>>2]:c;J[a+180>>2]=b;J[a+12>>2]=g;J[a+8>>2]=0;while(1){d:{J[5996]=0;g=_(202,a+316|0,a+312|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break d}d=J[a+180>>2];e:{if(!g){g=d;d=K[c+11|0];d=(d&128)>>>7|0?J[c+4>>2]:d&127;if((g|0)==(d+b|0)){J[5996]=0;aa(172,c|0,d<<1);b=J[5996];J[5996]=0;if((b|0)==1){break c}b=(K[c+11|0]&128)>>>7|0?(J[c+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,c|0,b|0);b=J[5996];J[5996]=0;if((b|0)==1){break c}b=(K[c+11|0]&128)>>>7|0?J[c>>2]:c;J[a+180>>2]=d+b;}J[5996]=0;d=$(203,J[a+316>>2])|0;g=J[5996];J[5996]=0;if((g|0)==1){break d}if(!Cc(d,16,b,a+180|0,a+8|0,0,J[h+4>>2],K[h+11|0],a+16|0,a+12|0,a+208|0)){break e}d=J[a+180>>2];}J[5996]=0;aa(172,c|0,d-b|0);b=J[5996];J[5996]=0;if((b|0)==1){break d}b=J[c>>2];d=(K[c+11|0]&128)>>>7|0;g=jb();J[a>>2]=f;if((Kf(d?b:c,g,a)|0)!=1){J[e>>2]=4;}J[5996]=0;b=_(202,a+316|0,a+312|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break d}if(b){J[e>>2]=J[e>>2]|2;}b=J[a+316>>2];ab(c);ab(h);Ua=a+320|0;return b|0}J[5996]=0;$(205,a+316|0)|0;d=J[5996];J[5996]=0;if((d|0)!=1){continue}}break}}}b=Z()|0;ab(c);break a}b=Z()|0;a=J[a+16>>2];c=J[a+4>>2]-1|0;J[a+4>>2]=c;if((c|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}}ab(h);da(b|0);B();}function ek(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0;a=Ua-480|0;Ua=a;g=J[d+28>>2];J[a+476>>2]=g;J[g+4>>2]=J[g+4>>2]+1;J[5996]=0;o=$(197,J[a+476>>2])|0;g=J[5996];J[5996]=0;a:{b:{c:{d:{e:{if((g|0)!=1){i=K[f+11|0];g=(i&128)>>>7|0;if(g?J[f+4>>2]:i&127){g=J[(g?J[f>>2]:f)>>2];J[5996]=0;i=_(216,o|0,45)|0;k=J[5996];J[5996]=0;if((k|0)==1){break e}p=(g|0)==(i|0);}k=a+452|0;J[k>>2]=0;J[k+4>>2]=0;J[k+8>>2]=0;g=a+440|0;J[g>>2]=0;J[g+4>>2]=0;J[g+8>>2]=0;i=a+428|0;J[i>>2]=0;J[i+4>>2]=0;J[i+8>>2]=0;J[5996]=0;sa(238,c|0,p|0,J[a+476>>2],a+472|0,a+468|0,a+464|0,k|0,g|0,i|0,a+424|0);c=J[5996];J[5996]=0;if((c|0)==1){break d}J[a+4>>2]=184;l=a+8|0;J[l>>2]=0;J[l+4>>2]=J[a+4>>2];c=a+16|0;j=K[f+11|0];j=(j&128)>>>7|0?J[f+4>>2]:j&127;m=J[a+424>>2];f:{if((j|0)>(m|0)){h=K[i+11|0];n=(j-m<<1)+((h&128)>>>7|0?J[i+4>>2]:h&127)|0;h=K[g+11|0];h=(n+((h&128)>>>7|0?J[g+4>>2]:h&127)|0)+1|0;break f}h=K[i+11|0];n=(h&128)>>>7|0?J[i+4>>2]:h&127;h=K[g+11|0];h=(n+((h&128)>>>7|0?J[g+4>>2]:h&127)|0)+2|0;}g:{h=h+m|0;if(h>>>0>=101){tb(l,sb(h<<2));c=J[l>>2];if(!c){break g}j=K[f+11|0];j=(j&128)>>>7|0?J[f+4>>2]:j&127;}h=J[d+4>>2];f=(K[f+11|0]&128)>>>7|0?J[f>>2]:f;J[5996]=0;ta(239,c|0,a+4|0,a|0,h|0,f|0,f+(j<<2)|0,o|0,p|0,a+472|0,J[a+468>>2],J[a+464>>2],k|0,g|0,i|0,m|0);f=J[5996];J[5996]=0;if((f|0)==1){break c}J[5996]=0;d=ka(219,b|0,c|0,J[a+4>>2],J[a>>2],d|0,e|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break c}db(l);ab(i);ab(g);ab(k);b=J[a+476>>2];c=J[b+4>>2]-1|0;J[b+4>>2]=c;if((c|0)==-1){Xa[J[J[b>>2]+8>>2]](b);}Ua=a+480|0;return d|0}J[5996]=0;ca(185);b=J[5996];J[5996]=0;if((b|0)==1){break c}B();}f=Z()|0;break a}f=Z()|0;break a}f=Z()|0;break b}f=Z()|0;db(l);}ab(i);ab(g);ab(k);}a=J[a+476>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(f|0);B();}function ik(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0;a=Ua-176|0;Ua=a;g=J[d+28>>2];J[a+172>>2]=g;J[g+4>>2]=J[g+4>>2]+1;J[5996]=0;o=$(170,J[a+172>>2])|0;g=J[5996];J[5996]=0;a:{b:{c:{d:{e:{if((g|0)!=1){i=K[f+11|0];g=(i&128)>>>7|0;if(g?J[f+4>>2]:i&127){g=K[(g?J[f>>2]:f)|0];J[5996]=0;i=_(171,o|0,45)|0;k=J[5996];J[5996]=0;if((k|0)==1){break e}p=(i&255)==(g|0);}k=a+152|0;J[k>>2]=0;J[k+4>>2]=0;J[k+8>>2]=0;g=a+140|0;J[g>>2]=0;J[g+4>>2]=0;J[g+8>>2]=0;i=a+128|0;J[i>>2]=0;J[i+4>>2]=0;J[i+8>>2]=0;J[5996]=0;sa(236,c|0,p|0,J[a+172>>2],a+168|0,a+167|0,a+166|0,k|0,g|0,i|0,a+124|0);c=J[5996];J[5996]=0;if((c|0)==1){break d}J[a+4>>2]=184;l=a+8|0;J[l>>2]=0;J[l+4>>2]=J[a+4>>2];c=a+16|0;j=K[f+11|0];j=(j&128)>>>7|0?J[f+4>>2]:j&127;m=J[a+124>>2];f:{if((j|0)>(m|0)){h=K[i+11|0];n=(j-m<<1)+((h&128)>>>7|0?J[i+4>>2]:h&127)|0;h=K[g+11|0];h=(n+((h&128)>>>7|0?J[g+4>>2]:h&127)|0)+1|0;break f}h=K[i+11|0];n=(h&128)>>>7|0?J[i+4>>2]:h&127;h=K[g+11|0];h=(n+((h&128)>>>7|0?J[g+4>>2]:h&127)|0)+2|0;}g:{h=h+m|0;if(h>>>0>=101){tb(l,sb(h));c=J[l>>2];if(!c){break g}j=K[f+11|0];j=(j&128)>>>7|0?J[f+4>>2]:j&127;}h=J[d+4>>2];f=(K[f+11|0]&128)>>>7|0?J[f>>2]:f;J[5996]=0;ta(237,c|0,a+4|0,a|0,h|0,f|0,f+j|0,o|0,p|0,a+168|0,H[a+167|0],H[a+166|0],k|0,g|0,i|0,m|0);f=J[5996];J[5996]=0;if((f|0)==1){break c}J[5996]=0;d=ka(213,b|0,c|0,J[a+4>>2],J[a>>2],d|0,e|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break c}db(l);ab(i);ab(g);ab(k);b=J[a+172>>2];c=J[b+4>>2]-1|0;J[b+4>>2]=c;if((c|0)==-1){Xa[J[J[b>>2]+8>>2]](b);}Ua=a+176|0;return d|0}J[5996]=0;ca(185);b=J[5996];J[5996]=0;if((b|0)==1){break c}B();}f=Z()|0;break a}f=Z()|0;break a}f=Z()|0;break b}f=Z()|0;db(l);}ab(i);ab(g);ab(k);}a=J[a+172>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(f|0);B();}function fk(a,b,c,d,e,f,g,h,i,j,k,l,m,n,o){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;i=i|0;j=j|0;k=k|0;l=l|0;m=m|0;n=n|0;o=o|0;var p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0;J[c>>2]=a;v=d&512;w=h<<2;while(1){if((u|0)==4){e=K[n+11|0];if(((e&128)>>>7|0?J[n+4>>2]:e&127)>>>0>1){J[c>>2]=mf(Rd(Mb(n),1),qc(n),J[c>>2]);}d=d&176;if((d|0)!=16){if((d|0)==32){a=J[c>>2];}J[b>>2]=a;}}else {a:{b:{switch(H[i+u|0]){case 0:J[b>>2]=J[c>>2];break a;case 1:J[b>>2]=J[c>>2];h=Xa[J[J[g>>2]+44>>2]](g,32)|0;p=J[c>>2];J[c>>2]=p+4;J[p>>2]=h;break a;case 3:h=K[n+11|0];r=h&127;h=(h&128)>>>7|0;if(!(h?J[n+4>>2]:r)){break a}h=J[(h?J[n>>2]:n)>>2];p=J[c>>2];J[c>>2]=p+4;J[p>>2]=h;break a;case 2:if(!v){break a}h=K[m+11|0];if(!((h&128)>>>7|0?J[m+4>>2]:h&127)){break a}J[c>>2]=mf(Mb(m),qc(m),J[c>>2]);break a;case 4:break b;default:break a}}x=J[c>>2];e=e+w|0;h=e;while(1){c:{if(f>>>0<=h>>>0){break c}if(!(Xa[J[J[g>>2]+12>>2]](g,64,J[h>>2])|0)){break c}h=h+4|0;continue}break}if((o|0)>0){p=J[c>>2];q=o;while(1){if(!(!q|e>>>0>=h>>>0)){q=q-1|0;h=h-4|0;s=J[h>>2];r=p+4|0;J[c>>2]=r;J[p>>2]=s;p=r;continue}break}d:{if(!q){s=0;break d}s=Xa[J[J[g>>2]+44>>2]](g,48)|0;p=J[c>>2];}while(1){r=p+4|0;if((q|0)>0){J[p>>2]=s;q=q-1|0;p=r;continue}break}J[c>>2]=r;J[p>>2]=j;}e:{if((e|0)==(h|0)){p=Xa[J[J[g>>2]+44>>2]](g,48)|0;q=J[c>>2];h=q+4|0;J[c>>2]=h;J[q>>2]=p;break e}s=-1;p=K[l+11|0];r=(p&128)>>>7|0;if(r?J[l+4>>2]:p&127){s=H[(r?J[l>>2]:l)|0];}q=0;t=0;while(1){if((e|0)!=(h|0)){r=J[c>>2];f:{if((q|0)!=(s|0)){p=r;r=q;break f}p=r+4|0;J[c>>2]=p;J[r>>2]=k;r=0;t=t+1|0;s=K[l+11|0];if(t>>>0>=((s&128)>>>7|0?J[l+4>>2]:s&127)>>>0){s=q;break f}s=-1;q=((K[l+11|0]&128)>>>7|0?J[l>>2]:l)+t|0;if(K[q|0]==127){break f}s=H[q|0];}h=h-4|0;q=J[h>>2];J[c>>2]=p+4;J[p>>2]=q;q=r+1|0;continue}break}h=J[c>>2];}md(x,h);}u=u+1|0;continue}break}}function jk(a,b,c,d,e,f,g,h,i,j,k,l,m,n,o){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;i=i|0;j=j|0;k=k|0;l=l|0;m=m|0;n=n|0;o=o|0;var p=0,q=0,r=0,s=0,t=0,u=0,v=0,w=0,x=0;J[c>>2]=a;w=d&512;while(1){if((u|0)==4){e=K[n+11|0];if(((e&128)>>>7|0?J[n+4>>2]:e&127)>>>0>1){J[c>>2]=nf(Td(Mb(n),1),sc(n),J[c>>2]);}d=d&176;if((d|0)!=16){if((d|0)==32){a=J[c>>2];}J[b>>2]=a;}}else {a:{b:{switch(H[i+u|0]){case 0:J[b>>2]=J[c>>2];break a;case 1:J[b>>2]=J[c>>2];p=Xa[J[J[g>>2]+28>>2]](g,32)|0;q=J[c>>2];J[c>>2]=q+1;H[q|0]=p;break a;case 3:p=K[n+11|0];q=(p&128)>>>7|0;if(!(q?J[n+4>>2]:p&127)){break a}p=K[(q?J[n>>2]:n)|0];q=J[c>>2];J[c>>2]=q+1;H[q|0]=p;break a;case 2:if(!w){break a}p=K[m+11|0];if(!((p&128)>>>7|0?J[m+4>>2]:p&127)){break a}J[c>>2]=nf(Mb(m),sc(m),J[c>>2]);break a;case 4:break b;default:break a}}q=J[g+8>>2];x=J[c>>2];e=e+h|0;r=e;while(1){c:{if(f>>>0<=r>>>0){break c}p=H[r|0];if((p|0)>=0){p=(J[q+((p&255)<<2)>>2]&64)!=0;}else {p=0;}if(!p){break c}r=r+1|0;continue}break}p=o;if((p|0)>0){while(1){if(!(!p|e>>>0>=r>>>0)){p=p-1|0;r=r-1|0;q=K[r|0];s=J[c>>2];J[c>>2]=s+1;H[s|0]=q;continue}break}if(p){q=Xa[J[J[g>>2]+28>>2]](g,48)|0;}else {q=0;}while(1){s=J[c>>2];J[c>>2]=s+1;if((p|0)>0){H[s|0]=q;p=p-1|0;continue}break}H[s|0]=j;}d:{if((e|0)==(r|0)){p=Xa[J[J[g>>2]+28>>2]](g,48)|0;q=J[c>>2];J[c>>2]=q+1;H[q|0]=p;break d}q=-1;p=K[l+11|0];s=p&127;p=(p&128)>>>7|0;if(p?J[l+4>>2]:s){q=H[(p?J[l>>2]:l)|0];}p=0;t=0;while(1){if((e|0)==(r|0)){break d}e:{if((p|0)!=(q|0)){s=p;break e}q=J[c>>2];J[c>>2]=q+1;H[q|0]=k;s=0;t=t+1|0;q=K[l+11|0];if(t>>>0>=((q&128)>>>7|0?J[l+4>>2]:q&127)>>>0){q=p;break e}q=-1;p=((K[l+11|0]&128)>>>7|0?J[l>>2]:l)+t|0;if(K[p|0]==127){break e}q=H[p|0];}r=r-1|0;p=K[r|0];v=J[c>>2];J[c>>2]=v+1;H[v|0]=p;p=s+1|0;continue}}vc(x,J[c>>2]);}u=u+1|0;continue}break}}function Fi(a,b,c,d,e,f,g,h){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;var i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0,t=0,u=0,v=0;r=Ua-16|0;Ua=r;j=c;while(1){a:{if((d|0)==(j|0)){j=d;break a}if(!K[j|0]){break a}j=j+1|0;continue}break}J[h>>2]=f;J[e>>2]=c;while(1){b:{c:{d:{if((c|0)==(d|0)|(f|0)==(g|0)){break d}i=J[b+4>>2];J[r+8>>2]=J[b>>2];J[r+12>>2]=i;s=Ua-16|0;Ua=s;v=Xb(s+12|0,J[a+8>>2]);m=j-c|0;k=0;n=0;o=Ua-1040|0;Ua=o;q=J[e>>2];J[o+12>>2]=q;p=f?f:o+16|0;l=f?g-f>>2:256;e:{f:{g:{if(!(!q|!l)){while(1){i=m>>>2|0;if(!(m>>>0>131|i>>>0>=l>>>0)){i=q;break f}t=bg(p,o+12|0,i>>>0<l>>>0?i:l,b);i=J[o+12>>2];if((t|0)==-1){l=0;k=-1;break g}u=(o+16|0)!=(p|0)?t:0;l=l-u|0;p=(u<<2)+p|0;m=i?(m+q|0)-i|0:0;k=k+t|0;if(!i){break g}q=i;if(l){continue}break}break g}i=q;}if(!i){break e}}if(!(!l|!m)){n=i;i=k;while(1){h:{k=qd(p,n,m,b);i:{if(k+2>>>0<=2){j:{switch(k+1|0){case 1:n=0;break i;case 0:break e;default:break j}}J[b>>2]=0;break i}n=k+n|0;J[o+12>>2]=n;i=i+1|0;l=l-1|0;if(l){break h}}k=i;break e}p=p+4|0;m=m-k|0;k=i;if(m){continue}break}break e}n=i;}if(f){J[e>>2]=n;}Ua=o+1040|0;i=J[v>>2];if(i){J[6178]=(i|0)==-1?24816:i;}Ua=s+16|0;k:{l:{m:{n:{if((k|0)==-1){while(1){o:{J[h>>2]=f;if(J[e>>2]==(c|0)){break o}g=1;p:{q:{r:{b=Me(f,c,j-c|0,r+8|0,J[a+8>>2]);switch(b+2|0){case 0:break l;case 2:break p;case 1:break r;default:break q}}J[e>>2]=c;break n}g=b;}c=c+g|0;f=J[h>>2]+4|0;continue}break}J[e>>2]=c;break d}f=J[h>>2]+(k<<2)|0;J[h>>2]=f;if((f|0)==(g|0)){break k}c=J[e>>2];if((d|0)==(j|0)){j=d;continue}if(!Me(f,c,1,b,J[a+8>>2])){break m}}a=2;break c}J[h>>2]=J[h>>2]+4;c=J[e>>2]+1|0;J[e>>2]=c;j=c;while(1){if((d|0)==(j|0)){j=d;break b}if(!K[j|0]){break b}j=j+1|0;continue}}J[e>>2]=c;a=1;break c}c=J[e>>2];}a=(c|0)!=(d|0);}Ua=r+16|0;return a|0}f=J[h>>2];continue}}function nd(a,b,c,d,e,f,g){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;var h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0;j=Ua-128|0;Ua=j;J[j+124>>2]=b;J[j+4>>2]=184;b=j+8|0;J[b>>2]=0;J[b+4>>2]=J[j+4>>2];o=b;k=j+16|0;l=(d-c|0)/12|0;a:{b:{if(l>>>0>=101){k=sb(l);if(!k){J[5996]=0;ca(185);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break b}tb(o,k);}h=k;b=c;c:{d:{while(1){if((b|0)==(d|0)){e:while(1){f:{J[5996]=0;h=_(186,a|0,j+124|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break f}if((!l|h)==1){J[5996]=0;b=_(186,a|0,j+124|0)|0;a=J[5996];J[5996]=0;if((a|0)==1){break c}if(!b){break d}J[f>>2]=J[f>>2]|2;break d}b=J[a>>2];J[5996]=0;p=$(187,b|0)|0;b=J[5996];J[5996]=0;g:{if((b|0)!=1){if(g){break g}J[5996]=0;p=_(188,e|0,p|0)|0;b=J[5996];J[5996]=0;if((b|0)!=1){break g}}break b}i=q+1|0;s=0;h=k;b=c;while(1){if((b|0)==(d|0)){q=i;if(!s){continue e}J[5996]=0;$(189,a|0)|0;b=J[5996];J[5996]=0;if((b|0)!=1){h=k;b=c;if(l+m>>>0<2){continue e}while(1){if((b|0)==(d|0)){continue e}h:{if(K[h|0]!=2){break h}i=K[b+11|0];if((((i&128)>>>7|0?J[b+4>>2]:i&127)|0)==(q|0)){break h}H[h|0]=0;m=m-1|0;}h=h+1|0;b=b+12|0;continue}}break b}i:{if(K[h|0]!=1){break i}r=K[((K[b+11|0]&128)>>>7|0?J[b>>2]:b)+q|0];j:{if(g){break j}J[5996]=0;r=_(188,e|0,r<<24>>24)|0;n=J[5996];J[5996]=0;if((n|0)!=1){break j}break b}k:{if((p&255)==(r&255)){s=1;n=K[b+11|0];if((i|0)!=(((n&128)>>>7|0?J[b+4>>2]:n&127)|0)){break i}H[h|0]=2;m=m+1|0;break k}H[h|0]=0;}l=l-1|0;}h=h+1|0;b=b+12|0;continue}}break}}else {i=K[b+11|0];i=!((i&128)>>>7|0?J[b+4>>2]:i&127);H[h|0]=i?2:1;m=i+m|0;l=l-i|0;h=h+1|0;b=b+12|0;continue}break}break b}l:{m:{while(1){if((c|0)==(d|0)){break m}if(K[k|0]!=2){k=k+1|0;c=c+12|0;continue}break}d=c;break l}J[f>>2]=J[f>>2]|4;}db(o);Ua=j+128|0;return d|0}}a=Z()|0;db(o);da(a|0);}B();}function ld(a,b,c,d,e,f,g){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;var h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0;j=Ua-128|0;Ua=j;J[j+124>>2]=b;J[j+4>>2]=184;b=j+8|0;J[b>>2]=0;J[b+4>>2]=J[j+4>>2];o=b;k=j+16|0;l=(d-c|0)/12|0;a:{b:{if(l>>>0>=101){k=sb(l);if(!k){J[5996]=0;ca(185);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break b}tb(o,k);}h=k;b=c;c:{d:{while(1){if((b|0)==(d|0)){e:while(1){f:{J[5996]=0;h=_(202,a|0,j+124|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break f}if((!l|h)==1){J[5996]=0;b=_(202,a|0,j+124|0)|0;a=J[5996];J[5996]=0;if((a|0)==1){break c}if(!b){break d}J[f>>2]=J[f>>2]|2;break d}b=J[a>>2];J[5996]=0;p=$(203,b|0)|0;b=J[5996];J[5996]=0;g:{if((b|0)!=1){if(g){break g}J[5996]=0;p=_(204,e|0,p|0)|0;b=J[5996];J[5996]=0;if((b|0)!=1){break g}}break b}i=q+1|0;s=0;h=k;b=c;while(1){if((b|0)==(d|0)){q=i;if(!s){continue e}J[5996]=0;$(205,a|0)|0;b=J[5996];J[5996]=0;if((b|0)!=1){h=k;b=c;if(l+m>>>0<2){continue e}while(1){if((b|0)==(d|0)){continue e}h:{if(K[h|0]!=2){break h}i=K[b+11|0];if((((i&128)>>>7|0?J[b+4>>2]:i&127)|0)==(q|0)){break h}H[h|0]=0;m=m-1|0;}h=h+1|0;b=b+12|0;continue}}break b}i:{if(K[h|0]!=1){break i}r=J[(q<<2)+((K[b+11|0]&128)>>>7|0?J[b>>2]:b)>>2];j:{if(g){break j}J[5996]=0;r=_(204,e|0,r|0)|0;n=J[5996];J[5996]=0;if((n|0)!=1){break j}break b}k:{if((p|0)==(r|0)){s=1;n=K[b+11|0];if((i|0)!=(((n&128)>>>7|0?J[b+4>>2]:n&127)|0)){break i}H[h|0]=2;m=m+1|0;break k}H[h|0]=0;}l=l-1|0;}h=h+1|0;b=b+12|0;continue}}break}}else {i=K[b+11|0];i=!((i&128)>>>7|0?J[b+4>>2]:i&127);H[h|0]=i?2:1;m=i+m|0;l=l-i|0;h=h+1|0;b=b+12|0;continue}break}break b}l:{m:{while(1){if((c|0)==(d|0)){break m}if(K[k|0]!=2){k=k+1|0;c=c+12|0;continue}break}d=c;break l}J[f>>2]=J[f>>2]|4;}db(o);Ua=j+128|0;return d|0}}a=Z()|0;db(o);da(a|0);}B();}function uk(a,b,c,d,e,f,g){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;var h=0,i=0,j=0,k=0;h=Ua-624|0;Ua=h;J[h+616>>2]=c;J[h+620>>2]=b;J[h+16>>2]=220;i=h+200|0;J[i>>2]=h+208;a=h+16|0;J[i+4>>2]=J[a>>2];j=h+192|0;b=J[e+28>>2];J[j>>2]=b;J[b+4>>2]=J[b+4>>2]+1;J[5996]=0;k=$(197,J[h+192>>2])|0;b=J[5996];J[5996]=0;a:{b:{c:{d:{e:{f:{g:{if((b|0)!=1){H[h+191|0]=0;b=J[e+4>>2];J[5996]=0;b=ua(228,h+620|0,c|0,d|0,j|0,b|0,f|0,h+191|0,k|0,i|0,h+196|0,h+608|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break d}if(!b){break e}b=K[3833]|K[3834]<<8|(K[3835]<<16|K[3836]<<24);H[h+183|0]=b;H[h+184|0]=b>>>8;H[h+185|0]=b>>>16;H[h+186|0]=b>>>24;b=K[3830]|K[3831]<<8|(K[3832]<<16|K[3833]<<24);J[h+176>>2]=K[3826]|K[3827]<<8|(K[3828]<<16|K[3829]<<24);J[h+180>>2]=b;J[5996]=0;ga(208,k|0,h+176|0,h+186|0,h+128|0);b=J[5996];J[5996]=0;if((b|0)==1){break g}J[h+4>>2]=184;d=h+8|0;J[d>>2]=0;J[d+4>>2]=J[h+4>>2];e=a;b=J[h+196>>2];a=b-J[i>>2]|0;if((a|0)<393){break f}tb(d,sb((a>>2)+2|0));e=J[d>>2];if(e){break f}J[5996]=0;ca(185);a=J[5996];J[5996]=0;if((a|0)==1){break c}break a}c=Z()|0;break b}c=Z()|0;break b}if(K[h+191|0]){H[e|0]=45;e=e+1|0;}c=J[i>>2];while(1){if(b>>>0<=c>>>0){H[e|0]=0;J[h>>2]=g;if((eg(h+16|0,h)|0)!=1){J[5996]=0;ha(222,1730);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break c}db(d);}else {a=h+128|0;H[e|0]=K[(h+176|0)+(Yd(a,a+40|0,J[c>>2])-a>>2)|0];e=e+1|0;c=c+4|0;b=J[h+196>>2];continue}break}}J[5996]=0;a=_(202,h+620|0,h+616|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break d}if(a){J[f>>2]=J[f>>2]|2;}c=J[h+620>>2];a=J[h+192>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}db(i);Ua=h+624|0;return c|0}c=Z()|0;break b}c=Z()|0;db(d);}a=J[h+192>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}db(i);da(c|0);B();}B();}function vj(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0,h=0,i=0,j=0;e=J[a+4>>2];c=J[a>>2];d=e-c>>2;if(d>>>0<b>>>0){h=Ua-32|0;Ua=h;f=b-d|0;a:{if(f>>>0<=J[a+8>>2]-e>>2>>>0){ff(a,f);break a}g=a+16|0;d=h+12|0;c=Ua-16|0;Ua=c;e=f+(e-J[a>>2]>>2)|0;J[c+12>>2]=e;b:{b=_e();if(e>>>0<=b>>>0){e=J[a+8>>2]-J[a>>2]>>2;if(e>>>0<b>>>1>>>0){J[c+8>>2]=e<<1;b=J[(M[c+8>>2]<M[c+12>>2]?c+12|0:c+8|0)>>2];}Ua=c+16|0;break b}nb();B();}i=J[a+4>>2]-J[a>>2]>>2;e=Ua-16|0;Ua=e;J[d+12>>2]=0;J[d+16>>2]=g;c:{if(!b){c=0;b=0;break c}Ze(e+8|0,J[d+16>>2],b);c=J[e+8>>2];b=J[e+12>>2];}J[d>>2]=c;g=c+(i<<2)|0;J[d+8>>2]=g;J[d+4>>2]=g;J[d+12>>2]=c+(b<<2);Ua=e+16|0;c=Ua-16|0;Ua=c;J[c+4>>2]=J[d+8>>2];b=J[d+8>>2];J[c+12>>2]=d+8;J[c+8>>2]=b+(f<<2);b=J[c+4>>2];while(1){if(J[c+8>>2]!=(b|0)){J[b>>2]=0;b=J[c+4>>2]+4|0;J[c+4>>2]=b;continue}break}J[J[c+12>>2]>>2]=J[c+4>>2];Ua=c+16|0;e=Ua-16|0;Ua=e;J[e+12>>2]=J[a+4>>2];J[e+8>>2]=J[a>>2];J[e+4>>2]=J[d+4>>2];f=J[e+12>>2];i=J[e+4>>2];g=Ua-16|0;Ua=g;b=Ua-32|0;Ua=b;c=Ua-16|0;Ua=c;J[c+12>>2]=J[e+8>>2];J[b+24>>2]=f;J[b+28>>2]=J[c+12>>2];Ua=c+16|0;f=J[b+24>>2];j=J[b+28>>2];c=Ua-16|0;Ua=c;J[c+8>>2]=i;J[c+12>>2]=f;while(1){if((f|0)!=(j|0)){J[J[c+8>>2]-4>>2]=J[f-4>>2];J[c+12>>2]=J[c+12>>2]-4;J[c+8>>2]=J[c+8>>2]-4;f=J[c+12>>2];continue}break}J[b+16>>2]=f;J[b+20>>2]=J[c+8>>2];Ua=c+16|0;c=J[b+16>>2];J[b+12>>2]=J[b+20>>2];J[g+8>>2]=c;J[g+12>>2]=J[b+12>>2];Ua=b+32|0;Ua=g+16|0;J[d+4>>2]=J[g+12>>2];b=J[a>>2];J[a>>2]=J[d+4>>2];J[d+4>>2]=b;b=J[a+4>>2];J[a+4>>2]=J[d+8>>2];J[d+8>>2]=b;b=J[a+8>>2];J[a+8>>2]=J[d+12>>2];J[d+12>>2]=b;J[d>>2]=J[d+4>>2];Ua=e+16|0;b=J[d+4>>2];a=J[d+8>>2];while(1){if((a|0)!=(b|0)){a=a-4|0;J[d+8>>2]=a;continue}break}a=J[d>>2];if(a){bf(J[d+16>>2],a);}}Ua=h+32|0;return}if(b>>>0<d>>>0){af(a,c+(b<<2)|0);}}function Bk(a,b,c,d,e,f,g){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;var h=0,i=0,j=0,k=0;h=Ua-272|0;Ua=h;J[h+264>>2]=c;J[h+268>>2]=b;J[h+16>>2]=220;i=h+152|0;J[i>>2]=h+160;a=h+16|0;J[i+4>>2]=J[a>>2];j=h+144|0;b=J[e+28>>2];J[j>>2]=b;J[b+4>>2]=J[b+4>>2]+1;J[5996]=0;k=$(170,J[h+144>>2])|0;b=J[5996];J[5996]=0;a:{b:{c:{d:{e:{f:{g:{if((b|0)!=1){H[h+143|0]=0;b=J[e+4>>2];J[5996]=0;b=ua(221,h+268|0,c|0,d|0,j|0,b|0,f|0,h+143|0,k|0,i|0,h+148|0,h+260|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break d}if(!b){break e}b=K[3833]|K[3834]<<8|(K[3835]<<16|K[3836]<<24);H[h+135|0]=b;H[h+136|0]=b>>>8;H[h+137|0]=b>>>16;H[h+138|0]=b>>>24;b=K[3830]|K[3831]<<8|(K[3832]<<16|K[3833]<<24);J[h+128>>2]=K[3826]|K[3827]<<8|(K[3828]<<16|K[3829]<<24);J[h+132>>2]=b;J[5996]=0;ga(195,k|0,h+128|0,h+138|0,h+118|0);b=J[5996];J[5996]=0;if((b|0)==1){break g}J[h+4>>2]=184;d=h+8|0;J[d>>2]=0;J[d+4>>2]=J[h+4>>2];e=a;b=J[h+148>>2];a=b-J[i>>2]|0;if((a|0)<99){break f}tb(d,sb(a+2|0));e=J[d>>2];if(e){break f}J[5996]=0;ca(185);a=J[5996];J[5996]=0;if((a|0)==1){break c}break a}c=Z()|0;break b}c=Z()|0;break b}if(K[h+143|0]){H[e|0]=45;e=e+1|0;}c=J[i>>2];while(1){if(b>>>0<=c>>>0){H[e|0]=0;J[h>>2]=g;if((eg(h+16|0,h)|0)!=1){J[5996]=0;ha(222,1730);a=J[5996];J[5996]=0;if((a|0)!=1){break a}break c}db(d);}else {a=h+118|0;H[e|0]=K[((be(a,a+10|0,K[c|0])-h|0)+h|0)+10|0];e=e+1|0;c=c+1|0;b=J[h+148>>2];continue}break}}J[5996]=0;a=_(186,h+268|0,h+264|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break d}if(a){J[f>>2]=J[f>>2]|2;}c=J[h+268>>2];a=J[h+144>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}db(i);Ua=h+272|0;return c|0}c=Z()|0;break b}c=Z()|0;db(d);}a=J[h+144>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}db(i);da(c|0);B();}B();}function vl(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0;d=J[d+28>>2];a=Ua-272|0;Ua=a;J[a+264>>2]=c;J[a+268>>2]=b;$d(a+208|0,d,a+224|0,a+223|0,a+222|0);d=a+196|0;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;b=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,b|0);b=J[5996];J[5996]=0;a:{b:{c:{if((b|0)!=1){c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+192>>2]=c;J[a+28>>2]=a+32;J[a+24>>2]=0;H[a+23|0]=1;H[a+22|0]=69;g=K[a+222|0]<<24>>24;i=K[a+223|0]<<24>>24;while(1){d:{J[5996]=0;b=_(186,a+268|0,a+264|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break d}if(b){break c}b=K[d+11|0];b=(b&128)>>>7|0?J[d+4>>2]:b&127;if(J[a+192>>2]==(b+c|0)){J[5996]=0;aa(172,d|0,b<<1);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+192>>2]=b+c;}J[5996]=0;b=$(187,J[a+268>>2])|0;h=J[5996];J[5996]=0;if((h|0)==1){break d}if(_d(b,a+23|0,a+22|0,c,a+192|0,i,g,a+208|0,a+32|0,a+28|0,a+24|0,a+224|0)){break c}J[5996]=0;$(189,a+268|0)|0;b=J[5996];J[5996]=0;if((b|0)!=1){continue}}break}break b}break b}b=K[a+219|0];e:{if(!((b&128)>>>7|0?J[a+212>>2]:b&127)|!K[a+23|0]){break e}b=J[a+28>>2];if((b-(a+32|0)|0)>159){break e}J[a+28>>2]=b+4;J[b>>2]=J[a+24>>2];}Lf(a,c,J[a+192>>2],e);b=J[a>>2];c=J[a+4>>2];g=J[a+12>>2];J[f+8>>2]=J[a+8>>2];J[f+12>>2]=g;J[f>>2]=b;J[f+4>>2]=c;zb(a+208|0,a+32|0,J[a+28>>2],e);J[5996]=0;b=_(186,a+268|0,a+264|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){if(b){J[e>>2]=J[e>>2]|2;}b=J[a+268>>2];ab(d);ab(a+208|0);Ua=a+272|0;break a}}b=Z()|0;ab(d);ab(a+208|0);da(b|0);B();}return b|0}function kl(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0;d=J[d+28>>2];a=Ua-384|0;Ua=a;J[a+376>>2]=c;J[a+380>>2]=b;Xd(a+220|0,d,a+240|0,a+236|0,a+232|0);d=a+208|0;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;b=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,b|0);b=J[5996];J[5996]=0;a:{b:{c:{if((b|0)!=1){c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+204>>2]=c;J[a+28>>2]=a+32;J[a+24>>2]=0;H[a+23|0]=1;H[a+22|0]=69;g=J[a+232>>2];i=J[a+236>>2];while(1){d:{J[5996]=0;b=_(202,a+380|0,a+376|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break d}if(b){break c}b=K[d+11|0];b=(b&128)>>>7|0?J[d+4>>2]:b&127;if(J[a+204>>2]==(b+c|0)){J[5996]=0;aa(172,d|0,b<<1);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+204>>2]=b+c;}J[5996]=0;b=$(203,J[a+380>>2])|0;h=J[5996];J[5996]=0;if((h|0)==1){break d}if(Wd(b,a+23|0,a+22|0,c,a+204|0,i,g,a+220|0,a+32|0,a+28|0,a+24|0,a+240|0)){break c}J[5996]=0;$(205,a+380|0)|0;b=J[5996];J[5996]=0;if((b|0)!=1){continue}}break}break b}break b}b=K[a+231|0];e:{if(!((b&128)>>>7|0?J[a+224>>2]:b&127)|!K[a+23|0]){break e}b=J[a+28>>2];if((b-(a+32|0)|0)>159){break e}J[a+28>>2]=b+4;J[b>>2]=J[a+24>>2];}Lf(a,c,J[a+204>>2],e);b=J[a>>2];c=J[a+4>>2];g=J[a+12>>2];J[f+8>>2]=J[a+8>>2];J[f+12>>2]=g;J[f>>2]=b;J[f+4>>2]=c;zb(a+220|0,a+32|0,J[a+28>>2],e);J[5996]=0;b=_(202,a+380|0,a+376|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){if(b){J[e>>2]=J[e>>2]|2;}b=J[a+380>>2];ab(d);ab(a+220|0);Ua=a+384|0;break a}}b=Z()|0;ab(d);ab(a+220|0);da(b|0);B();}return b|0}function al(a,b,c,d,e,f,g,h){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;var i=0,j=0,k=0,l=0,m=0,n=0;n=b;i=Ua-256|0;Ua=i;J[i+248>>2]=37;J[i+252>>2]=0;b=i+248|0;l=kd(b|1,3264,J[c+4>>2]);j=i+208|0;J[i+204>>2]=j;a=jb();a:{if(l){k=J[c+8>>2];m=i- -64|0;J[m>>2]=g;J[m+4>>2]=h;J[i+56>>2]=e;J[i+60>>2]=f;J[i+48>>2]=k;j=yb(j,30,a,b,i+48|0);break a}J[i+80>>2]=e;J[i+84>>2]=f;J[i+88>>2]=g;J[i+92>>2]=h;j=yb(i+208|0,30,a,i+248|0,i+80|0);}J[i+128>>2]=184;k=i+196|0;J[k>>2]=0;J[k+4>>2]=J[i+128>>2];b=i+208|0;a=b;b:{c:{d:{e:{if((j|0)>=30){a=jb();f:{if(l){j=J[c+8>>2];J[i+16>>2]=g;J[i+20>>2]=h;J[i+8>>2]=e;J[i+12>>2]=f;J[i>>2]=j;j=dc(i+204|0,a,i+248|0,i);break f}J[i+32>>2]=e;J[i+36>>2]=f;J[i+40>>2]=g;J[i+44>>2]=h;j=dc(i+204|0,a,i+248|0,i+32|0);}if((j|0)==-1){J[5996]=0;ca(185);a=J[5996];J[5996]=0;if((a|0)==1){break d}break e}a=J[i+204>>2];tb(k,a);}g=a+j|0;h=Lb(a,g,J[c+4>>2]);J[i+116>>2]=184;e=i+120|0;J[e>>2]=0;J[e+4>>2]=J[i+116>>2];g:{h:{if((i+208|0)==(a|0)){j=i+128|0;break h}j=sb(j<<1);if(!j){J[5996]=0;ca(185);a=J[5996];J[5996]=0;if((a|0)!=1){break e}a=Z()|0;break g}tb(e,j);b=a;}a=i+108|0;f=J[c+28>>2];J[a>>2]=f;J[f+4>>2]=J[f+4>>2]+1;J[5996]=0;ma(212,b|0,h|0,g|0,j|0,i+116|0,i+112|0,a|0);b=J[5996];J[5996]=0;i:{if((b|0)!=1){a=J[a>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}J[5996]=0;a=ka(213,n|0,j|0,J[i+116>>2],J[i+112>>2],c|0,d|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break i}db(e);db(k);Ua=i+256|0;break b}a=Z()|0;b=J[i+108>>2];c=J[b+4>>2]-1|0;J[b+4>>2]=c;if((c|0)==-1){Xa[J[J[b>>2]+8>>2]](b);}break g}a=Z()|0;}db(e);break c}B();}a=Z()|0;}db(k);da(a|0);B();}return a|0}function Rk(a,b,c,d,e,f,g,h){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;var i=0,j=0,k=0,l=0,m=0,n=0;n=b;i=Ua-416|0;Ua=i;J[i+408>>2]=37;J[i+412>>2]=0;b=i+408|0;l=kd(b|1,3264,J[c+4>>2]);j=i+368|0;J[i+364>>2]=j;a=jb();a:{if(l){k=J[c+8>>2];m=i- -64|0;J[m>>2]=g;J[m+4>>2]=h;J[i+56>>2]=e;J[i+60>>2]=f;J[i+48>>2]=k;j=yb(j,30,a,b,i+48|0);break a}J[i+80>>2]=e;J[i+84>>2]=f;J[i+88>>2]=g;J[i+92>>2]=h;j=yb(i+368|0,30,a,i+408|0,i+80|0);}J[i+128>>2]=184;k=i+356|0;J[k>>2]=0;J[k+4>>2]=J[i+128>>2];b=i+368|0;a=b;b:{c:{d:{e:{if((j|0)>=30){a=jb();f:{if(l){j=J[c+8>>2];J[i+16>>2]=g;J[i+20>>2]=h;J[i+8>>2]=e;J[i+12>>2]=f;J[i>>2]=j;j=dc(i+364|0,a,i+408|0,i);break f}J[i+32>>2]=e;J[i+36>>2]=f;J[i+40>>2]=g;J[i+44>>2]=h;j=dc(i+364|0,a,i+408|0,i+32|0);}if((j|0)==-1){J[5996]=0;ca(185);a=J[5996];J[5996]=0;if((a|0)==1){break d}break e}a=J[i+364>>2];tb(k,a);}g=a+j|0;h=Lb(a,g,J[c+4>>2]);J[i+116>>2]=184;e=i+120|0;J[e>>2]=0;J[e+4>>2]=J[i+116>>2];g:{h:{if((i+368|0)==(a|0)){j=i+128|0;break h}j=sb(j<<3);if(!j){J[5996]=0;ca(185);a=J[5996];J[5996]=0;if((a|0)!=1){break e}a=Z()|0;break g}tb(e,j);b=a;}a=i+108|0;f=J[c+28>>2];J[a>>2]=f;J[f+4>>2]=J[f+4>>2]+1;J[5996]=0;ma(218,b|0,h|0,g|0,j|0,i+116|0,i+112|0,a|0);b=J[5996];J[5996]=0;i:{if((b|0)!=1){a=J[a>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}J[5996]=0;a=ka(219,n|0,j|0,J[i+116>>2],J[i+112>>2],c|0,d|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break i}db(e);db(k);Ua=i+416|0;break b}a=Z()|0;b=J[i+108>>2];c=J[b+4>>2]-1|0;J[b+4>>2]=c;if((c|0)==-1){Xa[J[J[b>>2]+8>>2]](b);}break g}a=Z()|0;}db(e);break c}B();}a=Z()|0;}db(k);da(a|0);B();}return a|0}function te(a,b,c){var d=0,e=0,f=0,g=0,h=0,i=0,j=0;H[a|0]=0;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;H[a+4|0]=0;H[a+5|0]=0;H[a+6|0]=0;H[a+7|0]=0;H[a+24|0]=0;H[a+25|0]=0;H[a+26|0]=0;H[a+27|0]=0;H[a+28|0]=0;H[a+29|0]=0;H[a+30|0]=0;H[a+31|0]=0;H[a+16|0]=0;H[a+17|0]=0;H[a+18|0]=0;H[a+19|0]=0;H[a+20|0]=0;H[a+21|0]=0;H[a+22|0]=0;H[a+23|0]=0;H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;H[a+12|0]=0;H[a+13|0]=0;H[a+14|0]=0;H[a+15|0]=0;j=K[b+8|0]|K[b+9|0]<<8|(K[b+10|0]<<16|K[b+11|0]<<24);e=-29;f=K[b|0]|K[b+1|0]<<8|(K[b+2|0]<<16|K[b+3|0]<<24);h=f>>>0<29;a:{f=h?f:f-29|0;if(f>>>0>=29){J[5996]=0;aa(124,a|0,f+(f>>>1|0)|0);e=J[5996];J[5996]=0;if((e|0)==1){break a}d=K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24);g=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);e=g-29|0;}i=g>>>0<29;d=i?a+4|0:d;eb(f+d|0,d,i?g:e);e=f+(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))|0;H[a|0]=e;H[a+1|0]=e>>>8;H[a+2|0]=e>>>16;H[a+3|0]=e>>>24;b:{if(!f){break b}e=h?b+4|0:j;h=f&7;c:{if(!h){b=e;break c}g=0;b=e;while(1){H[d|0]=K[b|0];b=b+1|0;d=d+1|0;g=g+1|0;if((h|0)!=(g|0)){continue}break}}if(f>>>0<8){break b}e=e+f|0;while(1){H[d|0]=K[b|0];H[d+1|0]=K[b+1|0];H[d+2|0]=K[b+2|0];H[d+3|0]=K[b+3|0];H[d+4|0]=K[b+4|0];H[d+5|0]=K[b+5|0];H[d+6|0]=K[b+6|0];H[d+7|0]=K[b+7|0];d=d+8|0;b=b+8|0;if((e|0)!=(b|0)){continue}break}}b=J[c>>2];J[5996]=0;_(127,a|0,b|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break a}return}b=Z()|0;if((K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))>>>0>=29){$a(K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24));H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;}da(b|0);B();}function zl(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0;d=J[d+28>>2];a=Ua-256|0;Ua=a;J[a+248>>2]=c;J[a+252>>2]=b;$d(a+192|0,d,a+208|0,a+207|0,a+206|0);d=a+180|0;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;b=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,b|0);b=J[5996];J[5996]=0;a:{b:{c:{if((b|0)!=1){c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+176>>2]=c;J[a+12>>2]=a+16;J[a+8>>2]=0;H[a+7|0]=1;H[a+6|0]=69;h=K[a+206|0]<<24>>24;i=K[a+207|0]<<24>>24;while(1){d:{J[5996]=0;b=_(186,a+252|0,a+248|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break d}if(b){break c}b=K[d+11|0];b=(b&128)>>>7|0?J[d+4>>2]:b&127;if(J[a+176>>2]==(b+c|0)){J[5996]=0;aa(172,d|0,b<<1);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+176>>2]=b+c;}J[5996]=0;b=$(187,J[a+252>>2])|0;g=J[5996];J[5996]=0;if((g|0)==1){break d}if(_d(b,a+7|0,a+6|0,c,a+176|0,i,h,a+192|0,a+16|0,a+12|0,a+8|0,a+208|0)){break c}J[5996]=0;$(189,a+252|0)|0;b=J[5996];J[5996]=0;if((b|0)!=1){continue}}break}break b}break b}b=K[a+203|0];e:{if(!((b&128)>>>7|0?J[a+196>>2]:b&127)|!K[a+7|0]){break e}b=J[a+12>>2];if((b-(a+16|0)|0)>159){break e}J[a+12>>2]=b+4;J[b>>2]=J[a+8>>2];}N[f>>2]=Of(c,J[a+176>>2],e);zb(a+192|0,a+16|0,J[a+12>>2],e);J[5996]=0;b=_(186,a+252|0,a+248|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){if(b){J[e>>2]=J[e>>2]|2;}b=J[a+252>>2];ab(d);ab(a+192|0);Ua=a+256|0;break a}}b=Z()|0;ab(d);ab(a+192|0);da(b|0);B();}return b|0}function xl(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0;d=J[d+28>>2];a=Ua-256|0;Ua=a;J[a+248>>2]=c;J[a+252>>2]=b;$d(a+192|0,d,a+208|0,a+207|0,a+206|0);d=a+180|0;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;b=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,b|0);b=J[5996];J[5996]=0;a:{b:{c:{if((b|0)!=1){c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+176>>2]=c;J[a+12>>2]=a+16;J[a+8>>2]=0;H[a+7|0]=1;H[a+6|0]=69;h=K[a+206|0]<<24>>24;i=K[a+207|0]<<24>>24;while(1){d:{J[5996]=0;b=_(186,a+252|0,a+248|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break d}if(b){break c}b=K[d+11|0];b=(b&128)>>>7|0?J[d+4>>2]:b&127;if(J[a+176>>2]==(b+c|0)){J[5996]=0;aa(172,d|0,b<<1);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+176>>2]=b+c;}J[5996]=0;b=$(187,J[a+252>>2])|0;g=J[5996];J[5996]=0;if((g|0)==1){break d}if(_d(b,a+7|0,a+6|0,c,a+176|0,i,h,a+192|0,a+16|0,a+12|0,a+8|0,a+208|0)){break c}J[5996]=0;$(189,a+252|0)|0;b=J[5996];J[5996]=0;if((b|0)!=1){continue}}break}break b}break b}b=K[a+203|0];e:{if(!((b&128)>>>7|0?J[a+196>>2]:b&127)|!K[a+7|0]){break e}b=J[a+12>>2];if((b-(a+16|0)|0)>159){break e}J[a+12>>2]=b+4;J[b>>2]=J[a+8>>2];}O[f>>3]=Mf(c,J[a+176>>2],e);zb(a+192|0,a+16|0,J[a+12>>2],e);J[5996]=0;b=_(186,a+252|0,a+248|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){if(b){J[e>>2]=J[e>>2]|2;}b=J[a+252>>2];ab(d);ab(a+192|0);Ua=a+256|0;break a}}b=Z()|0;ab(d);ab(a+192|0);da(b|0);B();}return b|0}function kc(a,b,c,d,e,f,g,h){var i=0,j=0,k=0;i=Ua-16|0;Ua=i;J[i+8>>2]=c;J[i+12>>2]=b;b=J[d+28>>2];J[i+4>>2]=b;J[b+4>>2]=J[b+4>>2]+1;c=0;J[5996]=0;j=$(170,J[i+4>>2])|0;b=J[5996];J[5996]=0;if((b|0)!=1){b=J[i+4>>2];k=J[b+4>>2]-1|0;J[b+4>>2]=k;if((k|0)==-1){Xa[J[J[b>>2]+8>>2]](b);}J[e>>2]=0;a:{while(1){if(c|(g|0)==(h|0)){break a}b:{if(Ub(i+12|0,i+8|0)){break b}c:{if((Xa[J[J[j>>2]+36>>2]](j,H[g|0],0)|0)==37){if((g+1|0)==(h|0)){break b}c=Xa[J[J[j>>2]+36>>2]](j,H[g+1|0],0)|0;d:{if(!((c|0)==69|(c&255)==48)){b=0;k=1;break d}if((g+2|0)==(h|0)){break b}b=c;c=Xa[J[J[j>>2]+36>>2]](j,H[g+2|0],0)|0;k=2;}J[i+12>>2]=Xa[J[J[a>>2]+36>>2]](a,J[i+12>>2],J[i+8>>2],d,e,f,c,b);g=(g+k|0)+1|0;break c}b=J[j+8>>2];c=H[g|0];if((c|0)>=0){c=J[b+((c&255)<<2)>>2]&1;}else {c=0;}if(c){while(1){e:{g=g+1|0;if((h|0)==(g|0)){g=h;break e}c=H[g|0];if((c|0)>=0){c=J[b+((c&255)<<2)>>2]&1;}else {c=0;}if(c){continue}}break}while(1){k=i+12|0;if(Ub(k,i+8|0)){break c}b=J[i+12>>2];c=J[b+12>>2];f:{if((c|0)==J[b+16>>2]){b=Xa[J[J[b>>2]+36>>2]](b)|0;break f}b=K[c|0];}c=J[j+8>>2];b=b<<24>>24;if((b|0)>=0){b=J[c+((b&255)<<2)>>2]&1;}else {b=0;}if(!b){break c}Ic(k);continue}}b=J[i+12>>2];c=J[b+12>>2];g:{if((c|0)==J[b+16>>2]){b=Xa[J[J[b>>2]+36>>2]](b)|0;break g}b=K[c|0];}if((Xa[J[J[j>>2]+12>>2]](j,b<<24>>24)|0)==(Xa[J[J[j>>2]+12>>2]](j,H[g|0])|0)){g=g+1|0;Ic(i+12|0);break c}J[e>>2]=4;}c=J[e>>2];continue}break}J[e>>2]=4;}if(Ub(i+12|0,i+8|0)){J[e>>2]=J[e>>2]|2;}Ua=i+16|0;return J[i+12>>2]}c=Z()|0;a=J[i+4>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(c|0);B();}function ml(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0;d=J[d+28>>2];a=Ua-368|0;Ua=a;J[a+360>>2]=c;J[a+364>>2]=b;Xd(a+204|0,d,a+224|0,a+220|0,a+216|0);d=a+192|0;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;b=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,b|0);b=J[5996];J[5996]=0;a:{b:{c:{if((b|0)!=1){c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+188>>2]=c;J[a+12>>2]=a+16;J[a+8>>2]=0;H[a+7|0]=1;H[a+6|0]=69;h=J[a+216>>2];i=J[a+220>>2];while(1){d:{J[5996]=0;b=_(202,a+364|0,a+360|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break d}if(b){break c}b=K[d+11|0];b=(b&128)>>>7|0?J[d+4>>2]:b&127;if(J[a+188>>2]==(b+c|0)){J[5996]=0;aa(172,d|0,b<<1);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+188>>2]=b+c;}J[5996]=0;b=$(203,J[a+364>>2])|0;g=J[5996];J[5996]=0;if((g|0)==1){break d}if(Wd(b,a+7|0,a+6|0,c,a+188|0,i,h,a+204|0,a+16|0,a+12|0,a+8|0,a+224|0)){break c}J[5996]=0;$(205,a+364|0)|0;b=J[5996];J[5996]=0;if((b|0)!=1){continue}}break}break b}break b}b=K[a+215|0];e:{if(!((b&128)>>>7|0?J[a+208>>2]:b&127)|!K[a+7|0]){break e}b=J[a+12>>2];if((b-(a+16|0)|0)>159){break e}J[a+12>>2]=b+4;J[b>>2]=J[a+8>>2];}N[f>>2]=Of(c,J[a+188>>2],e);zb(a+204|0,a+16|0,J[a+12>>2],e);J[5996]=0;b=_(202,a+364|0,a+360|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){if(b){J[e>>2]=J[e>>2]|2;}b=J[a+364>>2];ab(d);ab(a+204|0);Ua=a+368|0;break a}}b=Z()|0;ab(d);ab(a+204|0);da(b|0);B();}return b|0}function ll(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0;d=J[d+28>>2];a=Ua-368|0;Ua=a;J[a+360>>2]=c;J[a+364>>2]=b;Xd(a+204|0,d,a+224|0,a+220|0,a+216|0);d=a+192|0;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;b=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,b|0);b=J[5996];J[5996]=0;a:{b:{c:{if((b|0)!=1){c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+188>>2]=c;J[a+12>>2]=a+16;J[a+8>>2]=0;H[a+7|0]=1;H[a+6|0]=69;h=J[a+216>>2];i=J[a+220>>2];while(1){d:{J[5996]=0;b=_(202,a+364|0,a+360|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break d}if(b){break c}b=K[d+11|0];b=(b&128)>>>7|0?J[d+4>>2]:b&127;if(J[a+188>>2]==(b+c|0)){J[5996]=0;aa(172,d|0,b<<1);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+188>>2]=b+c;}J[5996]=0;b=$(203,J[a+364>>2])|0;g=J[5996];J[5996]=0;if((g|0)==1){break d}if(Wd(b,a+7|0,a+6|0,c,a+188|0,i,h,a+204|0,a+16|0,a+12|0,a+8|0,a+224|0)){break c}J[5996]=0;$(205,a+364|0)|0;b=J[5996];J[5996]=0;if((b|0)!=1){continue}}break}break b}break b}b=K[a+215|0];e:{if(!((b&128)>>>7|0?J[a+208>>2]:b&127)|!K[a+7|0]){break e}b=J[a+12>>2];if((b-(a+16|0)|0)>159){break e}J[a+12>>2]=b+4;J[b>>2]=J[a+8>>2];}O[f>>3]=Mf(c,J[a+188>>2],e);zb(a+204|0,a+16|0,J[a+12>>2],e);J[5996]=0;b=_(202,a+364|0,a+360|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){if(b){J[e>>2]=J[e>>2]|2;}b=J[a+364>>2];ab(d);ab(a+204|0);Ua=a+368|0;break a}}b=Z()|0;ab(d);ab(a+204|0);da(b|0);B();}return b|0}function _c(a,b){var c=0,d=0,e=0,f=0,g=0,h=0,i=0;a:{b:{if(a-1&a|b>>>0>4294967239){break b}while(1){a=a>>>0<=8?8:a;e=J[5995];f=J[5994];b=b>>>0<=8?8:b+3&-4;c:{if(b>>>0<=127){g=(b>>>3|0)-1|0;break c}d=S(b);g=((b>>>29-d^4)-(d<<2)|0)+110|0;if(b>>>0<=4095){break c}d=((b>>>30-d^2)-(d<<1)|0)+71|0;g=d>>>0>=63?63:d;}c=g&31;if((g&63)>>>0>=32){d=0;c=e>>>c|0;}else {d=e>>>c|0;c=((1<<c)-1&e)<<32-c|f>>>c;}if(c|d){while(1){e=c;d:{if(d|e){f=d-1|0;h=f+1|0;c=f;f=e-1|0;h=(f|0)!=-1?h:c;c=S(d^h);c=(c|0)==32?S(e^f)+32|0:c;f=63-c|0;Wa=0-(c>>>0>63)|0;break d}Wa=0;f=64;}h=f;f=h&31;if((h&63)>>>0>=32){c=0;f=d>>>f|0;}else {c=d>>>f|0;f=((1<<f)-1&d)<<32-f|e>>>f;}d=c;g=g+h|0;h=g<<4;e=J[h+22952>>2];i=h+22944|0;e:{if((e|0)!=(i|0)){c=ye(e,a,b);if(c){break b}c=J[e+4>>2];J[c+8>>2]=J[e+8>>2];J[J[e+8>>2]+4>>2]=c;J[e+8>>2]=i;c=h+22948|0;J[e+4>>2]=J[c>>2];J[c>>2]=e;J[J[e+4>>2]+8>>2]=e;g=g+1|0;c=(d&1)<<31|f>>>1;d=d>>>1|0;break e}c=J[5995];J[5994]=J[5994]&go(-2,-1,g);J[5995]=Wa&c;c=f^1;}if(d|c){continue}break}f=J[5994];e=J[5995];}f:{if(e|f){d=S(e);h=63-((d|0)==32?S(f)+32|0:d)|0;c=h<<4;d=J[c+22952>>2];g:{if(!e&f>>>0<1073741824){break g}g=99;e=c+22944|0;if((e|0)==(d|0)){break g}while(1){if(!g){break g}c=ye(d,a,b);if(c){break b}g=g-1|0;d=J[d+8>>2];if((e|0)!=(d|0)){continue}break}d=e;}if(Cd(b+48|0)){break f}if(!d){break a}e=(h<<4)+22944|0;if((e|0)==(d|0)){break a}while(1){c=ye(d,a,b);if(c){break b}d=J[d+8>>2];if((e|0)!=(d|0)){continue}break}break a}if(!Cd(b+48|0)){break a}}c=0;if(a-1&a){break b}if(b>>>0<=4294967239){continue}break}}return c}return 0}function ql(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0;a=Ua-336|0;Ua=a;J[a+328>>2]=c;J[a+332>>2]=b;b=ec(J[d+4>>2]);i=Sc(J[d+28>>2],a+208|0);Rc(a+196|0,J[d+28>>2],a+324|0);d=a+184|0;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;a:{b:{c:{if((c|0)!=1){c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+180>>2]=c;J[a+12>>2]=a+16;J[a+8>>2]=0;while(1){d:{J[5996]=0;g=_(202,a+332|0,a+328|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break d}if(g){break c}g=K[d+11|0];g=(g&128)>>>7|0?J[d+4>>2]:g&127;if(J[a+180>>2]==(g+c|0)){J[5996]=0;aa(172,d|0,g<<1);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+180>>2]=g+c;}J[5996]=0;g=$(203,J[a+332>>2])|0;h=J[5996];J[5996]=0;if((h|0)==1){break d}if(Cc(g,b,c,a+180|0,a+8|0,J[a+324>>2],J[a+200>>2],K[a+207|0],a+16|0,a+12|0,i)){break c}J[5996]=0;$(205,a+332|0)|0;g=J[5996];J[5996]=0;if((g|0)!=1){continue}}break}break b}break b}g=K[a+207|0];e:{if(!((g&128)>>>7|0?J[a+200>>2]:g&127)){break e}g=J[a+12>>2];if((g-(a+16|0)|0)>159){break e}J[a+12>>2]=g+4;J[g>>2]=J[a+8>>2];}J[f>>2]=Tf(c,J[a+180>>2],e,b);J[f+4>>2]=Wa;zb(a+196|0,a+16|0,J[a+12>>2],e);J[5996]=0;b=_(202,a+332|0,a+328|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){if(b){J[e>>2]=J[e>>2]|2;}b=J[a+332>>2];ab(d);ab(a+196|0);Ua=a+336|0;break a}}b=Z()|0;ab(d);ab(a+196|0);da(b|0);B();}return b|0}function ol(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0;a=Ua-336|0;Ua=a;J[a+328>>2]=c;J[a+332>>2]=b;b=ec(J[d+4>>2]);i=Sc(J[d+28>>2],a+208|0);Rc(a+196|0,J[d+28>>2],a+324|0);d=a+184|0;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;a:{b:{c:{if((c|0)!=1){c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+180>>2]=c;J[a+12>>2]=a+16;J[a+8>>2]=0;while(1){d:{J[5996]=0;g=_(202,a+332|0,a+328|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break d}if(g){break c}g=K[d+11|0];g=(g&128)>>>7|0?J[d+4>>2]:g&127;if(J[a+180>>2]==(g+c|0)){J[5996]=0;aa(172,d|0,g<<1);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+180>>2]=g+c;}J[5996]=0;g=$(203,J[a+332>>2])|0;h=J[5996];J[5996]=0;if((h|0)==1){break d}if(Cc(g,b,c,a+180|0,a+8|0,J[a+324>>2],J[a+200>>2],K[a+207|0],a+16|0,a+12|0,i)){break c}J[5996]=0;$(205,a+332|0)|0;g=J[5996];J[5996]=0;if((g|0)!=1){continue}}break}break b}break b}g=K[a+207|0];e:{if(!((g&128)>>>7|0?J[a+200>>2]:g&127)){break e}g=J[a+12>>2];if((g-(a+16|0)|0)>159){break e}J[a+12>>2]=g+4;J[g>>2]=J[a+8>>2];}J[f>>2]=Pf(c,J[a+180>>2],e,b);J[f+4>>2]=Wa;zb(a+196|0,a+16|0,J[a+12>>2],e);J[5996]=0;b=_(202,a+332|0,a+328|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){if(b){J[e>>2]=J[e>>2]|2;}b=J[a+332>>2];ab(d);ab(a+196|0);Ua=a+336|0;break a}}b=Z()|0;ab(d);ab(a+196|0);da(b|0);B();}return b|0}function sl(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0;a=Ua-336|0;Ua=a;J[a+328>>2]=c;J[a+332>>2]=b;b=ec(J[d+4>>2]);i=Sc(J[d+28>>2],a+208|0);Rc(a+196|0,J[d+28>>2],a+324|0);d=a+184|0;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;a:{b:{c:{if((c|0)!=1){c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+180>>2]=c;J[a+12>>2]=a+16;J[a+8>>2]=0;while(1){d:{J[5996]=0;g=_(202,a+332|0,a+328|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break d}if(g){break c}g=K[d+11|0];g=(g&128)>>>7|0?J[d+4>>2]:g&127;if(J[a+180>>2]==(g+c|0)){J[5996]=0;aa(172,d|0,g<<1);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+180>>2]=g+c;}J[5996]=0;g=$(203,J[a+332>>2])|0;h=J[5996];J[5996]=0;if((h|0)==1){break d}if(Cc(g,b,c,a+180|0,a+8|0,J[a+324>>2],J[a+200>>2],K[a+207|0],a+16|0,a+12|0,i)){break c}J[5996]=0;$(205,a+332|0)|0;g=J[5996];J[5996]=0;if((g|0)!=1){continue}}break}break b}break b}g=K[a+207|0];e:{if(!((g&128)>>>7|0?J[a+200>>2]:g&127)){break e}g=J[a+12>>2];if((g-(a+16|0)|0)>159){break e}J[a+12>>2]=g+4;J[g>>2]=J[a+8>>2];}J[f>>2]=Yf(c,J[a+180>>2],e,b);zb(a+196|0,a+16|0,J[a+12>>2],e);J[5996]=0;b=_(202,a+332|0,a+328|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){if(b){J[e>>2]=J[e>>2]|2;}b=J[a+332>>2];ab(d);ab(a+196|0);Ua=a+336|0;break a}}b=Z()|0;ab(d);ab(a+196|0);da(b|0);B();}return b|0}function pl(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0;a=Ua-336|0;Ua=a;J[a+328>>2]=c;J[a+332>>2]=b;b=ec(J[d+4>>2]);i=Sc(J[d+28>>2],a+208|0);Rc(a+196|0,J[d+28>>2],a+324|0);d=a+184|0;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;a:{b:{c:{if((c|0)!=1){c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+180>>2]=c;J[a+12>>2]=a+16;J[a+8>>2]=0;while(1){d:{J[5996]=0;g=_(202,a+332|0,a+328|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break d}if(g){break c}g=K[d+11|0];g=(g&128)>>>7|0?J[d+4>>2]:g&127;if(J[a+180>>2]==(g+c|0)){J[5996]=0;aa(172,d|0,g<<1);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+180>>2]=g+c;}J[5996]=0;g=$(203,J[a+332>>2])|0;h=J[5996];J[5996]=0;if((h|0)==1){break d}if(Cc(g,b,c,a+180|0,a+8|0,J[a+324>>2],J[a+200>>2],K[a+207|0],a+16|0,a+12|0,i)){break c}J[5996]=0;$(205,a+332|0)|0;g=J[5996];J[5996]=0;if((g|0)!=1){continue}}break}break b}break b}g=K[a+207|0];e:{if(!((g&128)>>>7|0?J[a+200>>2]:g&127)){break e}g=J[a+12>>2];if((g-(a+16|0)|0)>159){break e}J[a+12>>2]=g+4;J[g>>2]=J[a+8>>2];}I[f>>1]=Sf(c,J[a+180>>2],e,b);zb(a+196|0,a+16|0,J[a+12>>2],e);J[5996]=0;b=_(202,a+332|0,a+328|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){if(b){J[e>>2]=J[e>>2]|2;}b=J[a+332>>2];ab(d);ab(a+196|0);Ua=a+336|0;break a}}b=Z()|0;ab(d);ab(a+196|0);da(b|0);B();}return b|0}function Jf(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0;a=Ua-336|0;Ua=a;J[a+328>>2]=c;J[a+332>>2]=b;b=ec(J[d+4>>2]);i=Sc(J[d+28>>2],a+208|0);Rc(a+196|0,J[d+28>>2],a+324|0);d=a+184|0;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;a:{b:{c:{if((c|0)!=1){c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+180>>2]=c;J[a+12>>2]=a+16;J[a+8>>2]=0;while(1){d:{J[5996]=0;g=_(202,a+332|0,a+328|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break d}if(g){break c}g=K[d+11|0];g=(g&128)>>>7|0?J[d+4>>2]:g&127;if(J[a+180>>2]==(g+c|0)){J[5996]=0;aa(172,d|0,g<<1);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+180>>2]=g+c;}J[5996]=0;g=$(203,J[a+332>>2])|0;h=J[5996];J[5996]=0;if((h|0)==1){break d}if(Cc(g,b,c,a+180|0,a+8|0,J[a+324>>2],J[a+200>>2],K[a+207|0],a+16|0,a+12|0,i)){break c}J[5996]=0;$(205,a+332|0)|0;g=J[5996];J[5996]=0;if((g|0)!=1){continue}}break}break b}break b}g=K[a+207|0];e:{if(!((g&128)>>>7|0?J[a+200>>2]:g&127)){break e}g=J[a+12>>2];if((g-(a+16|0)|0)>159){break e}J[a+12>>2]=g+4;J[g>>2]=J[a+8>>2];}J[f>>2]=Qf(c,J[a+180>>2],e,b);zb(a+196|0,a+16|0,J[a+12>>2],e);J[5996]=0;b=_(202,a+332|0,a+328|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){if(b){J[e>>2]=J[e>>2]|2;}b=J[a+332>>2];ab(d);ab(a+196|0);Ua=a+336|0;break a}}b=Z()|0;ab(d);ab(a+196|0);da(b|0);B();}return b|0}function Cl(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0;g=J[d+4>>2];d=J[d+28>>2];a=Ua-224|0;Ua=a;J[a+216>>2]=c;J[a+220>>2]=b;b=ec(g);Tc(a+200|0,d,a+215|0);d=a+188|0;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;a:{b:{c:{if((c|0)!=1){c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+184>>2]=c;J[a+12>>2]=a+16;J[a+8>>2]=0;while(1){d:{J[5996]=0;g=_(186,a+220|0,a+216|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break d}if(g){break c}g=K[d+11|0];g=(g&128)>>>7|0?J[d+4>>2]:g&127;if(J[a+184>>2]==(g+c|0)){J[5996]=0;aa(172,d|0,g<<1);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+184>>2]=g+c;}J[5996]=0;g=$(187,J[a+220>>2])|0;h=J[5996];J[5996]=0;if((h|0)==1){break d}if(Dc(g,b,c,a+184|0,a+8|0,H[a+215|0],J[a+204>>2],K[a+211|0],a+16|0,a+12|0,12416)){break c}J[5996]=0;$(189,a+220|0)|0;g=J[5996];J[5996]=0;if((g|0)!=1){continue}}break}break b}break b}g=K[a+211|0];e:{if(!((g&128)>>>7|0?J[a+204>>2]:g&127)){break e}g=J[a+12>>2];if((g-(a+16|0)|0)>159){break e}J[a+12>>2]=g+4;J[g>>2]=J[a+8>>2];}J[f>>2]=Tf(c,J[a+184>>2],e,b);J[f+4>>2]=Wa;zb(a+200|0,a+16|0,J[a+12>>2],e);J[5996]=0;b=_(186,a+220|0,a+216|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){if(b){J[e>>2]=J[e>>2]|2;}b=J[a+220>>2];ab(d);ab(a+200|0);Ua=a+224|0;break a}}b=Z()|0;ab(d);ab(a+200|0);da(b|0);B();}return b|0}function Al(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0;g=J[d+4>>2];d=J[d+28>>2];a=Ua-224|0;Ua=a;J[a+216>>2]=c;J[a+220>>2]=b;b=ec(g);Tc(a+200|0,d,a+215|0);d=a+188|0;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;a:{b:{c:{if((c|0)!=1){c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+184>>2]=c;J[a+12>>2]=a+16;J[a+8>>2]=0;while(1){d:{J[5996]=0;g=_(186,a+220|0,a+216|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break d}if(g){break c}g=K[d+11|0];g=(g&128)>>>7|0?J[d+4>>2]:g&127;if(J[a+184>>2]==(g+c|0)){J[5996]=0;aa(172,d|0,g<<1);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+184>>2]=g+c;}J[5996]=0;g=$(187,J[a+220>>2])|0;h=J[5996];J[5996]=0;if((h|0)==1){break d}if(Dc(g,b,c,a+184|0,a+8|0,H[a+215|0],J[a+204>>2],K[a+211|0],a+16|0,a+12|0,12416)){break c}J[5996]=0;$(189,a+220|0)|0;g=J[5996];J[5996]=0;if((g|0)!=1){continue}}break}break b}break b}g=K[a+211|0];e:{if(!((g&128)>>>7|0?J[a+204>>2]:g&127)){break e}g=J[a+12>>2];if((g-(a+16|0)|0)>159){break e}J[a+12>>2]=g+4;J[g>>2]=J[a+8>>2];}J[f>>2]=Pf(c,J[a+184>>2],e,b);J[f+4>>2]=Wa;zb(a+200|0,a+16|0,J[a+12>>2],e);J[5996]=0;b=_(186,a+220|0,a+216|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){if(b){J[e>>2]=J[e>>2]|2;}b=J[a+220>>2];ab(d);ab(a+200|0);Ua=a+224|0;break a}}b=Z()|0;ab(d);ab(a+200|0);da(b|0);B();}return b|0}function Vc(a,b){var c=0,d=0,e=0,f=0,g=0,h=0,i=0,j=0,k=0;a:{if(!a){a=_c(8,b);break a}if(!b){$a(a);a=0;break a}b:{if(b>>>0>4294967239){break b}h=b>>>0<=8?8:b+3&-4;b=h+8|0;c:{d:{k=a-4|0;f=k;e=J[f>>2];c=e+f|0;j=J[c>>2];g=j+c|0;e:{f:{if(J[g-4>>2]!=(j|0)){d=b+f|0;if(d+16>>>0<=g>>>0){e=J[c+4>>2];J[e+8>>2]=J[c+8>>2];J[J[c+8>>2]+4>>2]=e;c=g-d|0;J[d>>2]=c;J[(d+(c&-4)|0)-4>>2]=c|1;e=J[d>>2]-8|0;g:{if(e>>>0<=127){c=(e>>>3|0)-1|0;break g}g=S(e);c=((e>>>29-g^4)-(g<<2)|0)+110|0;if(e>>>0<=4095){break g}c=((e>>>30-g^2)-(g<<1)|0)+71|0;c=c>>>0>=63?63:c;}e=c<<4;J[d+4>>2]=e+22944;e=e+22952|0;J[d+8>>2]=J[e>>2];J[e>>2]=d;J[J[d+8>>2]+4>>2]=d;e=J[5995];d=c&31;if((c&63)>>>0>=32){c=1<<d;g=0;}else {g=1<<d;c=g-1&1>>>32-d;}J[5994]=g|J[5994];J[5995]=c|e;J[f>>2]=b;break d}if(d>>>0>g>>>0){break f}b=J[c+4>>2];J[b+8>>2]=J[c+8>>2];J[J[c+8>>2]+4>>2]=b;b=e+j|0;J[f>>2]=b;break d}if(e>>>0>=b+16>>>0){J[f>>2]=b;J[(f+(b&-4)|0)-4>>2]=b;d=b+f|0;b=e-b|0;J[d>>2]=b;J[(d+(b&-4)|0)-4>>2]=b|1;c=J[d>>2]-8|0;h:{if(c>>>0<=127){b=(c>>>3|0)-1|0;break h}f=S(c);b=((c>>>29-f^4)-(f<<2)|0)+110|0;if(c>>>0<=4095){break h}b=((c>>>30-f^2)-(f<<1)|0)+71|0;b=b>>>0>=63?63:b;}c=b<<4;J[d+4>>2]=c+22944;c=c+22952|0;J[d+8>>2]=J[c>>2];J[c>>2]=d;J[J[d+8>>2]+4>>2]=d;c=J[5995];d=b&31;if((b&63)>>>0>=32){b=1<<d;e=0;}else {e=1<<d;b=e-1&1>>>32-d;}J[5994]=e|J[5994];J[5995]=b|c;d=1;break c}d=1;if(b>>>0<=e>>>0){break e}}d=0;}break c}J[(f+(b&-4)|0)-4>>2]=b;d=1;}if(d){break a}b=_c(8,h);if(!b){break b}i=J[k>>2]-8|0;fb(b,a,h>>>0<i>>>0?h:i);$a(a);i=b;}a=i;}return a}function an(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0,h=0,i=0;d=Ua-160|0;Ua=d;J[d+72>>2]=8356;J[d+16>>2]=8336;h=d+72|0;c=h;J[c+20>>2]=0;e=d+20|0;J[c+24>>2]=e;J[c+12>>2]=0;J[c+4>>2]=4098;J[c+8>>2]=6;J[c+16>>2]=!e;Sb(c+32|0,0,40);yg(c+28|0);J[d+72>>2]=8316;J[d+16>>2]=8296;J[d+144>>2]=0;J[d+148>>2]=-1;J[e>>2]=7896;yg(e+4|0);J[e+24>>2]=0;J[e+28>>2]=0;J[e+16>>2]=0;J[e+20>>2]=0;J[e+8>>2]=0;J[e+12>>2]=0;J[e>>2]=7960;J[d+60>>2]=0;J[d+64>>2]=0;J[d+52>>2]=0;J[d+56>>2]=0;J[d+68>>2]=16;c=J[J[d+16>>2]-12>>2];J[5996]=0;g=Ca(129)|0;f=J[5996];J[5996]=0;a:{b:{if((f|0)==1){break b}f=c+(d+16|0)|0;c=J[f+28>>2];J[d+12>>2]=c;J[c+4>>2]=J[c+4>>2]+1;J[5996]=0;ea(130,d+156|0,f|0,g|0);c=J[5996];J[5996]=0;c:{if((c|0)==1){break c}c=J[d+156>>2];i=J[c+4>>2]-1|0;J[c+4>>2]=i;if((i|0)==-1){Xa[J[J[c>>2]+8>>2]](c);}c=J[f+24>>2];if(c){f=J[J[c>>2]+8>>2];J[5996]=0;aa(f|0,c|0,g|0);f=J[5996];J[5996]=0;if((f|0)==1){break c}f=J[c+4>>2];J[d+152>>2]=f;J[f+4>>2]=J[f+4>>2]+1;Og(c+4|0,g);c=J[d+152>>2];g=J[c+4>>2]-1|0;J[c+4>>2]=g;if((g|0)==-1){Xa[J[J[c>>2]+8>>2]](c);}}c=J[d+12>>2];g=J[c+4>>2]-1|0;J[c+4>>2]=g;if((g|0)==-1){Xa[J[J[c>>2]+8>>2]](c);}b=J[b>>2];J[5996]=0;b=_(131,d+16|0,b|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break b}J[5996]=0;aa(132,a|0,e|0);a=J[5996];J[5996]=0;if((a|0)==1){break b}J[b>>2]=8296;J[d+72>>2]=8316;J[e>>2]=7960;if(H[b+47|0]<0){$a(J[d+52>>2]);}Qc(e);Od(h);Ua=d+160|0;return}a=Z()|0;b=J[d+12>>2];e=J[b+4>>2]-1|0;J[b+4>>2]=e;if((e|0)==-1){Xa[J[J[b>>2]+8>>2]](b);}break a}a=Z()|0;}gd(d+16|0);da(a|0);B();}
function Rf(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0;g=J[d+4>>2];d=J[d+28>>2];a=Ua-224|0;Ua=a;J[a+216>>2]=c;J[a+220>>2]=b;b=ec(g);Tc(a+200|0,d,a+215|0);d=a+188|0;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;a:{b:{c:{if((c|0)!=1){c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+184>>2]=c;J[a+12>>2]=a+16;J[a+8>>2]=0;while(1){d:{J[5996]=0;g=_(186,a+220|0,a+216|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break d}if(g){break c}g=K[d+11|0];g=(g&128)>>>7|0?J[d+4>>2]:g&127;if(J[a+184>>2]==(g+c|0)){J[5996]=0;aa(172,d|0,g<<1);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+184>>2]=g+c;}J[5996]=0;g=$(187,J[a+220>>2])|0;h=J[5996];J[5996]=0;if((h|0)==1){break d}if(Dc(g,b,c,a+184|0,a+8|0,H[a+215|0],J[a+204>>2],K[a+211|0],a+16|0,a+12|0,12416)){break c}J[5996]=0;$(189,a+220|0)|0;g=J[5996];J[5996]=0;if((g|0)!=1){continue}}break}break b}break b}g=K[a+211|0];e:{if(!((g&128)>>>7|0?J[a+204>>2]:g&127)){break e}g=J[a+12>>2];if((g-(a+16|0)|0)>159){break e}J[a+12>>2]=g+4;J[g>>2]=J[a+8>>2];}J[f>>2]=Qf(c,J[a+184>>2],e,b);zb(a+200|0,a+16|0,J[a+12>>2],e);J[5996]=0;b=_(186,a+220|0,a+216|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){if(b){J[e>>2]=J[e>>2]|2;}b=J[a+220>>2];ab(d);ab(a+200|0);Ua=a+224|0;break a}}b=Z()|0;ab(d);ab(a+200|0);da(b|0);B();}return b|0}function Dl(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0;g=J[d+4>>2];d=J[d+28>>2];a=Ua-224|0;Ua=a;J[a+216>>2]=c;J[a+220>>2]=b;b=ec(g);Tc(a+200|0,d,a+215|0);d=a+188|0;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;a:{b:{c:{if((c|0)!=1){c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+184>>2]=c;J[a+12>>2]=a+16;J[a+8>>2]=0;while(1){d:{J[5996]=0;g=_(186,a+220|0,a+216|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break d}if(g){break c}g=K[d+11|0];g=(g&128)>>>7|0?J[d+4>>2]:g&127;if(J[a+184>>2]==(g+c|0)){J[5996]=0;aa(172,d|0,g<<1);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+184>>2]=g+c;}J[5996]=0;g=$(187,J[a+220>>2])|0;h=J[5996];J[5996]=0;if((h|0)==1){break d}if(Dc(g,b,c,a+184|0,a+8|0,H[a+215|0],J[a+204>>2],K[a+211|0],a+16|0,a+12|0,12416)){break c}J[5996]=0;$(189,a+220|0)|0;g=J[5996];J[5996]=0;if((g|0)!=1){continue}}break}break b}break b}g=K[a+211|0];e:{if(!((g&128)>>>7|0?J[a+204>>2]:g&127)){break e}g=J[a+12>>2];if((g-(a+16|0)|0)>159){break e}J[a+12>>2]=g+4;J[g>>2]=J[a+8>>2];}J[f>>2]=Yf(c,J[a+184>>2],e,b);zb(a+200|0,a+16|0,J[a+12>>2],e);J[5996]=0;b=_(186,a+220|0,a+216|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){if(b){J[e>>2]=J[e>>2]|2;}b=J[a+220>>2];ab(d);ab(a+200|0);Ua=a+224|0;break a}}b=Z()|0;ab(d);ab(a+200|0);da(b|0);B();}return b|0}function Bl(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0;g=J[d+4>>2];d=J[d+28>>2];a=Ua-224|0;Ua=a;J[a+216>>2]=c;J[a+220>>2]=b;b=ec(g);Tc(a+200|0,d,a+215|0);d=a+188|0;J[d>>2]=0;J[d+4>>2]=0;J[d+8>>2]=0;c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;a:{b:{c:{if((c|0)!=1){c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+184>>2]=c;J[a+12>>2]=a+16;J[a+8>>2]=0;while(1){d:{J[5996]=0;g=_(186,a+220|0,a+216|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break d}if(g){break c}g=K[d+11|0];g=(g&128)>>>7|0?J[d+4>>2]:g&127;if(J[a+184>>2]==(g+c|0)){J[5996]=0;aa(172,d|0,g<<1);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?(J[d+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,d|0,c|0);c=J[5996];J[5996]=0;if((c|0)==1){break b}c=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+184>>2]=g+c;}J[5996]=0;g=$(187,J[a+220>>2])|0;h=J[5996];J[5996]=0;if((h|0)==1){break d}if(Dc(g,b,c,a+184|0,a+8|0,H[a+215|0],J[a+204>>2],K[a+211|0],a+16|0,a+12|0,12416)){break c}J[5996]=0;$(189,a+220|0)|0;g=J[5996];J[5996]=0;if((g|0)!=1){continue}}break}break b}break b}g=K[a+211|0];e:{if(!((g&128)>>>7|0?J[a+204>>2]:g&127)){break e}g=J[a+12>>2];if((g-(a+16|0)|0)>159){break e}J[a+12>>2]=g+4;J[g>>2]=J[a+8>>2];}I[f>>1]=Sf(c,J[a+184>>2],e,b);zb(a+200|0,a+16|0,J[a+12>>2],e);J[5996]=0;b=_(186,a+220|0,a+216|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){if(b){J[e>>2]=J[e>>2]|2;}b=J[a+220>>2];ab(d);ab(a+200|0);Ua=a+224|0;break a}}b=Z()|0;ab(d);ab(a+200|0);da(b|0);B();}return b|0}function jc(a,b,c,d,e,f,g,h){var i=0,j=0,k=0;i=Ua-16|0;Ua=i;J[i+8>>2]=c;J[i+12>>2]=b;b=J[d+28>>2];J[i+4>>2]=b;J[b+4>>2]=J[b+4>>2]+1;b=0;J[5996]=0;j=$(197,J[i+4>>2])|0;c=J[5996];J[5996]=0;if((c|0)!=1){c=J[i+4>>2];k=J[c+4>>2]-1|0;J[c+4>>2]=k;if((k|0)==-1){Xa[J[J[c>>2]+8>>2]](c);}J[e>>2]=0;a:{while(1){if(b|(g|0)==(h|0)){break a}b:{if(Tb(i+12|0,i+8|0)){break b}c:{if((Xa[J[J[j>>2]+52>>2]](j,J[g>>2],0)|0)==37){if((g+4|0)==(h|0)){break b}b=Xa[J[J[j>>2]+52>>2]](j,J[g+4>>2],0)|0;d:{if(!((b|0)==69|(b&255)==48)){c=0;k=1;break d}if((g+8|0)==(h|0)){break b}c=b;b=Xa[J[J[j>>2]+52>>2]](j,J[g+8>>2],0)|0;k=2;}J[i+12>>2]=Xa[J[J[a>>2]+36>>2]](a,J[i+12>>2],J[i+8>>2],d,e,f,b,c);g=((k<<2)+g|0)+4|0;break c}if(Xa[J[J[j>>2]+12>>2]](j,1,J[g>>2])|0){while(1){e:{g=g+4|0;if((h|0)==(g|0)){g=h;break e}if(Xa[J[J[j>>2]+12>>2]](j,1,J[g>>2])|0){continue}}break}while(1){k=i+12|0;if(Tb(k,i+8|0)){break c}b=J[i+12>>2];c=J[b+12>>2];f:{if((c|0)==J[b+16>>2]){b=Xa[J[J[b>>2]+36>>2]](b)|0;break f}b=J[c>>2];}if(!(Xa[J[J[j>>2]+12>>2]](j,1,b)|0)){break c}Hc(k);continue}}b=J[i+12>>2];c=J[b+12>>2];g:{if((c|0)==J[b+16>>2]){b=Xa[J[J[b>>2]+36>>2]](b)|0;break g}b=J[c>>2];}if((Xa[J[J[j>>2]+28>>2]](j,b)|0)==(Xa[J[J[j>>2]+28>>2]](j,J[g>>2])|0)){g=g+4|0;Hc(i+12|0);break c}J[e>>2]=4;}b=J[e>>2];continue}break}J[e>>2]=4;}if(Tb(i+12|0,i+8|0)){J[e>>2]=J[e>>2]|2;}Ua=i+16|0;return J[i+12>>2]}c=Z()|0;a=J[i+4>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(c|0);B();}function cl(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=+e;var f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0;l=b;f=Ua-208|0;Ua=f;J[f+200>>2]=37;J[f+204>>2]=0;b=f+200|0;h=kd(b|1,5575,J[c+4>>2]);g=f+160|0;J[f+156>>2]=g;a=jb();a:{if(h){i=J[c+8>>2];O[f+40>>3]=e;J[f+32>>2]=i;g=yb(g,30,a,b,f+32|0);break a}O[f+48>>3]=e;g=yb(f+160|0,30,a,f+200|0,f+48|0);}J[f+80>>2]=184;i=f+148|0;J[i>>2]=0;J[i+4>>2]=J[f+80>>2];b=f+160|0;a=b;b:{c:{d:{e:{if((g|0)>=30){a=jb();f:{if(h){g=J[c+8>>2];O[f+8>>3]=e;J[f>>2]=g;g=dc(f+156|0,a,f+200|0,f);break f}O[f+16>>3]=e;g=dc(f+156|0,a,f+200|0,f+16|0);}if((g|0)==-1){J[5996]=0;ca(185);a=J[5996];J[5996]=0;if((a|0)==1){break d}break e}a=J[f+156>>2];tb(i,a);}k=a+g|0;m=Lb(a,k,J[c+4>>2]);J[f+68>>2]=184;h=f+72|0;J[h>>2]=0;J[h+4>>2]=J[f+68>>2];g:{h:{if((f+160|0)==(a|0)){g=f+80|0;break h}g=sb(g<<1);if(!g){J[5996]=0;ca(185);a=J[5996];J[5996]=0;if((a|0)!=1){break e}a=Z()|0;break g}tb(h,g);b=a;}a=f+60|0;j=J[c+28>>2];J[a>>2]=j;J[j+4>>2]=J[j+4>>2]+1;J[5996]=0;ma(212,b|0,m|0,k|0,g|0,f+68|0,f- -64|0,a|0);b=J[5996];J[5996]=0;i:{if((b|0)!=1){a=J[a>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}J[5996]=0;a=ka(213,l|0,g|0,J[f+68>>2],J[f+64>>2],c|0,d|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break i}db(h);db(i);Ua=f+208|0;break b}a=Z()|0;b=J[f+60>>2];c=J[b+4>>2]-1|0;J[b+4>>2]=c;if((c|0)==-1){Xa[J[J[b>>2]+8>>2]](b);}break g}a=Z()|0;}db(h);break c}B();}a=Z()|0;}db(i);da(a|0);B();}return a|0}function Tk(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=+e;var f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0;l=b;f=Ua-368|0;Ua=f;J[f+360>>2]=37;J[f+364>>2]=0;b=f+360|0;h=kd(b|1,5575,J[c+4>>2]);g=f+320|0;J[f+316>>2]=g;a=jb();a:{if(h){i=J[c+8>>2];O[f+40>>3]=e;J[f+32>>2]=i;g=yb(g,30,a,b,f+32|0);break a}O[f+48>>3]=e;g=yb(f+320|0,30,a,f+360|0,f+48|0);}J[f+80>>2]=184;i=f+308|0;J[i>>2]=0;J[i+4>>2]=J[f+80>>2];b=f+320|0;a=b;b:{c:{d:{e:{if((g|0)>=30){a=jb();f:{if(h){g=J[c+8>>2];O[f+8>>3]=e;J[f>>2]=g;g=dc(f+316|0,a,f+360|0,f);break f}O[f+16>>3]=e;g=dc(f+316|0,a,f+360|0,f+16|0);}if((g|0)==-1){J[5996]=0;ca(185);a=J[5996];J[5996]=0;if((a|0)==1){break d}break e}a=J[f+316>>2];tb(i,a);}k=a+g|0;m=Lb(a,k,J[c+4>>2]);J[f+68>>2]=184;h=f+72|0;J[h>>2]=0;J[h+4>>2]=J[f+68>>2];g:{h:{if((f+320|0)==(a|0)){g=f+80|0;break h}g=sb(g<<3);if(!g){J[5996]=0;ca(185);a=J[5996];J[5996]=0;if((a|0)!=1){break e}a=Z()|0;break g}tb(h,g);b=a;}a=f+60|0;j=J[c+28>>2];J[a>>2]=j;J[j+4>>2]=J[j+4>>2]+1;J[5996]=0;ma(218,b|0,m|0,k|0,g|0,f+68|0,f- -64|0,a|0);b=J[5996];J[5996]=0;i:{if((b|0)!=1){a=J[a>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}J[5996]=0;a=ka(219,l|0,g|0,J[f+68>>2],J[f+64>>2],c|0,d|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break i}db(h);db(i);Ua=f+368|0;break b}a=Z()|0;b=J[f+60>>2];c=J[b+4>>2]-1|0;J[b+4>>2]=c;if((c|0)==-1){Xa[J[J[b>>2]+8>>2]](b);}break g}a=Z()|0;}db(h);break c}B();}a=Z()|0;}db(i);da(a|0);B();}return a|0}function Gi(a,b,c,d,e,f,g,h){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;var i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0;n=Ua-16|0;Ua=n;b=c;while(1){a:{if((b|0)==(d|0)){b=d;break a}if(!J[b>>2]){break a}b=b+4|0;continue}break}J[h>>2]=f;J[e>>2]=c;while(1){b:{c:{if(!((c|0)==(d|0)|(f|0)==(g|0))){q=1;r=Ua-16|0;Ua=r;s=Xb(r+12|0,J[a+8>>2]);i=f;j=g-i|0;m=0;o=Ua-16|0;Ua=o;k=J[e>>2];p=b-c>>2;d:{if(!k|!p){break d}j=i?j:0;while(1){l=pe(j>>>0<4?o+12|0:i,J[k>>2]);if((l|0)==-1){m=-1;break d}if(i){if(j>>>0<=3){if(j>>>0<l>>>0){break d}fb(i,o+12|0,l);}j=j-l|0;i=i+l|0;}else {i=0;}if(!J[k>>2]){k=0;break d}m=m+l|0;k=k+4|0;p=p-1|0;if(p){continue}break}}if(i){J[e>>2]=k;}Ua=o+16|0;i=J[s>>2];if(i){J[6178]=(i|0)==-1?24816:i;}Ua=r+16|0;e:{f:{g:{h:{switch(m+1|0){case 0:J[h>>2]=f;while(1){i:{if(J[e>>2]==(c|0)){break i}b=Md(f,J[c>>2],J[a+8>>2]);if((b|0)==-1){break i}f=b+J[h>>2]|0;J[h>>2]=f;c=c+4|0;continue}break}J[e>>2]=c;break g;case 1:break c;default:break h}}f=m+J[h>>2]|0;J[h>>2]=f;if((f|0)==(g|0)){break e}if((b|0)==(d|0)){c=J[e>>2];b=d;continue}b=Md(n+12|0,0,J[a+8>>2]);if((b|0)!=-1){break f}}q=2;break c}c=n+12|0;if(g-J[h>>2]>>>0<b>>>0){break c}while(1){if(b){f=K[c|0];i=J[h>>2];J[h>>2]=i+1;H[i|0]=f;b=b-1|0;c=c+1|0;continue}break}c=J[e>>2]+4|0;J[e>>2]=c;b=c;while(1){if((b|0)==(d|0)){b=d;break b}if(!J[b>>2]){break b}b=b+4|0;continue}}c=J[e>>2];}q=(c|0)!=(d|0);}Ua=n+16|0;return q|0}f=J[h>>2];continue}}function Yk(a,b,c,d,e,f,g){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;var h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0;k=Ua-16|0;Ua=k;m=je(J[g>>2]);g=Zd(J[g>>2]);Xa[J[J[g>>2]+20>>2]](k+4|0,g);a:{j=K[k+15|0];b:{if(!((j&128)>>>7|0?J[k+8>>2]:j&127)){J[5996]=0;ga(208,m|0,a|0,c|0,d|0);g=J[5996];J[5996]=0;if((g|0)==1){break b}g=(c-a<<2)+d|0;J[f>>2]=g;break a}J[f>>2]=d;c:{d:{j=a;i=K[j|0];switch(i-43|0){case 0:case 2:break d;default:break c}}J[5996]=0;j=_(216,m|0,i<<24>>24)|0;i=J[5996];J[5996]=0;if((i|0)==1){break b}i=J[f>>2];J[f>>2]=i+4;J[i>>2]=j;j=a+1|0;}if(!(K[j|0]!=48|(c-j|0)<2|(K[j+1|0]|32)!=120)){J[5996]=0;i=_(216,m|0,48)|0;h=J[5996];J[5996]=0;if((h|0)==1){break b}h=J[f>>2];J[f>>2]=h+4;J[h>>2]=i;i=H[j+1|0];J[5996]=0;i=_(216,m|0,i|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break b}h=J[f>>2];J[f>>2]=h+4;J[h>>2]=i;j=j+2|0;}vc(j,c);J[5996]=0;o=$(206,g|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break b}i=0;g=j;while(1){if(c>>>0<=g>>>0){md((j-a<<2)+d|0,J[f>>2]);g=J[f>>2];break a}h=k+4|0;l=(K[h+11|0]&128)>>>7|0;if(!(!K[(l?J[h>>2]:h)+i|0]|H[(l?J[h>>2]:h)+i|0]!=(n|0))){h=J[f>>2];J[f>>2]=h+4;J[h>>2]=o;n=0;l=i;i=K[k+15|0];i=l+(l>>>0<((i&128)>>>7|0?J[k+8>>2]:i&127)-1>>>0)|0;}h=H[g|0];J[5996]=0;h=_(216,m|0,h|0)|0;l=J[5996];J[5996]=0;if((l|0)!=1){l=J[f>>2];J[f>>2]=l+4;J[l>>2]=h;g=g+1|0;n=n+1|0;continue}break}}a=Z()|0;ab(k+4|0);da(a|0);B();}J[e>>2]=(b|0)==(c|0)?g:(b-a<<2)+d|0;ab(k+4|0);Ua=k+16|0;}function gl(a,b,c,d,e,f,g){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;var h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0;k=Ua-16|0;Ua=k;m=le(J[g>>2]);g=de(J[g>>2]);Xa[J[J[g>>2]+20>>2]](k+4|0,g);a:{j=K[k+15|0];b:{if(!((j&128)>>>7|0?J[k+8>>2]:j&127)){J[5996]=0;ga(195,m|0,a|0,c|0,d|0);g=J[5996];J[5996]=0;if((g|0)==1){break b}g=(c-a|0)+d|0;J[f>>2]=g;break a}J[f>>2]=d;c:{d:{j=a;i=K[j|0];switch(i-43|0){case 0:case 2:break d;default:break c}}J[5996]=0;j=_(171,m|0,i<<24>>24)|0;i=J[5996];J[5996]=0;if((i|0)==1){break b}i=J[f>>2];J[f>>2]=i+1;H[i|0]=j;j=a+1|0;}if(!(K[j|0]!=48|(c-j|0)<2|(K[j+1|0]|32)!=120)){J[5996]=0;i=_(171,m|0,48)|0;h=J[5996];J[5996]=0;if((h|0)==1){break b}h=J[f>>2];J[f>>2]=h+1;H[h|0]=i;i=H[j+1|0];J[5996]=0;i=_(171,m|0,i|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break b}h=J[f>>2];J[f>>2]=h+1;H[h|0]=i;j=j+2|0;}vc(j,c);J[5996]=0;o=$(193,g|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break b}i=0;g=j;while(1){if(c>>>0<=g>>>0){vc((j-a|0)+d|0,J[f>>2]);g=J[f>>2];break a}h=k+4|0;l=(K[h+11|0]&128)>>>7|0;if(!(!K[(l?J[h>>2]:h)+i|0]|H[(l?J[h>>2]:h)+i|0]!=(n|0))){h=J[f>>2];J[f>>2]=h+1;H[h|0]=o;n=0;l=i;i=K[k+15|0];i=l+(l>>>0<((i&128)>>>7|0?J[k+8>>2]:i&127)-1>>>0)|0;}h=H[g|0];J[5996]=0;h=_(171,m|0,h|0)|0;l=J[5996];J[5996]=0;if((l|0)!=1){l=J[f>>2];J[f>>2]=l+1;H[l|0]=h;g=g+1|0;n=n+1|0;continue}break}}a=Z()|0;ab(k+4|0);da(a|0);B();}J[e>>2]=(b|0)==(c|0)?g:(b-a|0)+d|0;ab(k+4|0);Ua=k+16|0;}function Pe(a,b,c,d,e,f,g,h){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;var i=0,j=0,k=0;a=Ua-16|0;Ua=a;J[a+12>>2]=c;J[a+8>>2]=f;J[a+12>>2]=c;J[a+8>>2]=f;a:{b:{c:{while(1){c=J[a+12>>2];if(!(c>>>0>=d>>>0|f>>>0>=g>>>0)){i=H[c|0];j=i&255;d:{if((i|0)>=0){I[f>>1]=j;b=c+1|0;break d}b=2;if(i>>>0<4294967234){break b}if(i>>>0<=4294967263){if((d-c|0)<2){break c}i=K[c+1|0];if((i&192)!=128){break a}I[f>>1]=i&63|j<<6&1984;b=c+2|0;break d}if(i>>>0<=4294967279){if((d-c|0)<3){break c}k=K[c+2|0];i=K[c+1|0];e:{f:{if((j|0)!=237){if((j|0)!=224){break f}if((i&224)==160){break e}break a}if((i&224)==128){break e}break a}if((i&192)!=128){break a}}if((k&192)!=128){break a}I[f>>1]=k&63|((i&63)<<6|j<<12);b=c+3|0;break d}if(i>>>0>4294967284){break b}b=1;if((d-c|0)<4){break a}k=K[c+3|0];i=K[c+2|0];c=K[c+1|0];g:{h:{switch(j-240|0){case 0:if((c+112&255)>>>0>=48){break b}break g;case 4:if((c&240)!=128){break b}break g;default:break h}}if((c&192)!=128){break b}}if((i&192)!=128|(k&192)!=128){break b}if((g-f|0)<4){break a}b=2;j=j&7;if((c<<12&196608|j<<18)>>>0>1114111){break a}b=c<<2;I[f>>1]=(i>>>4&3|(b&192|j<<8|b&60))+16320|55296;J[a+8>>2]=f+2;I[f+2>>1]=i<<6&960|k&63|56320;b=J[a+12>>2]+4|0;}J[a+12>>2]=b;f=J[a+8>>2]+2|0;J[a+8>>2]=f;continue}break}b=c>>>0<d>>>0;break a}b=1;break a}b=2;}J[e>>2]=J[a+12>>2];J[h>>2]=J[a+8>>2];Ua=a+16|0;return b|0}function tl(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0;g=Ua-32|0;Ua=g;J[g+28>>2]=b;a:{b:{if(!(H[d+4|0]&1)){J[g>>2]=-1;b=Xa[J[J[a>>2]+16>>2]](a,b,c,d,e,g)|0;c:{switch(J[g>>2]){case 1:H[f|0]=1;break a;case 0:break b;default:break c}}H[f|0]=1;J[e>>2]=4;break a}a=J[d+28>>2];J[g>>2]=a;J[a+4>>2]=J[a+4>>2]+1;J[5996]=0;h=$(197,J[g>>2])|0;a=J[5996];J[5996]=0;d:{e:{f:{g:{if((a|0)!=1){a=J[g>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}a=J[d+28>>2];J[g>>2]=a;J[a+4>>2]=J[a+4>>2]+1;J[5996]=0;b=$(198,J[g>>2])|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}a=J[g>>2];d=J[a+4>>2]-1|0;J[a+4>>2]=d;if((d|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}J[5996]=0;aa(199,g|0,b|0);a=J[5996];J[5996]=0;if((a|0)==1){b=Z()|0;break d}J[5996]=0;aa(200,g|12,b|0);a=J[5996];J[5996]=0;if((a|0)==1){break f}J[5996]=0;d=g+24|0;a=Ba(201,g+28|0,c|0,g|0,d|0,h|0,e|0,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break e}H[f|0]=(a|0)==(g|0);b=J[g+28>>2];while(1){d=ab(d-12|0);if((g|0)!=(d|0)){continue}break}break a}b=Z()|0;a=J[g>>2];c=J[a+4>>2]-1|0;J[a+4>>2]=c;if((c|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}break d}b=Z()|0;a=J[g>>2];c=J[a+4>>2]-1|0;J[a+4>>2]=c;if((c|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}break d}b=Z()|0;ab(g);break d}b=Z()|0;while(1){d=ab(d-12|0);if((g|0)!=(d|0)){continue}break}}da(b|0);B();}H[f|0]=0;}Ua=g+32|0;return b|0}function Hl(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0;g=Ua-32|0;Ua=g;J[g+28>>2]=b;a:{b:{if(!(H[d+4|0]&1)){J[g>>2]=-1;b=Xa[J[J[a>>2]+16>>2]](a,b,c,d,e,g)|0;c:{switch(J[g>>2]){case 1:H[f|0]=1;break a;case 0:break b;default:break c}}H[f|0]=1;J[e>>2]=4;break a}a=J[d+28>>2];J[g>>2]=a;J[a+4>>2]=J[a+4>>2]+1;J[5996]=0;h=$(170,J[g>>2])|0;a=J[5996];J[5996]=0;d:{e:{f:{g:{if((a|0)!=1){a=J[g>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}a=J[d+28>>2];J[g>>2]=a;J[a+4>>2]=J[a+4>>2]+1;J[5996]=0;b=$(180,J[g>>2])|0;a=J[5996];J[5996]=0;if((a|0)==1){break g}a=J[g>>2];d=J[a+4>>2]-1|0;J[a+4>>2]=d;if((d|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}J[5996]=0;aa(181,g|0,b|0);a=J[5996];J[5996]=0;if((a|0)==1){b=Z()|0;break d}J[5996]=0;aa(182,g|12,b|0);a=J[5996];J[5996]=0;if((a|0)==1){break f}J[5996]=0;d=g+24|0;a=Ba(183,g+28|0,c|0,g|0,d|0,h|0,e|0,1)|0;b=J[5996];J[5996]=0;if((b|0)==1){break e}H[f|0]=(a|0)==(g|0);b=J[g+28>>2];while(1){d=ab(d-12|0);if((g|0)!=(d|0)){continue}break}break a}b=Z()|0;a=J[g>>2];c=J[a+4>>2]-1|0;J[a+4>>2]=c;if((c|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}break d}b=Z()|0;a=J[g>>2];c=J[a+4>>2]-1|0;J[a+4>>2]=c;if((c|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}break d}b=Z()|0;ab(g);break d}b=Z()|0;while(1){d=ab(d-12|0);if((g|0)!=(d|0)){continue}break}}da(b|0);B();}H[f|0]=0;}Ua=g+32|0;return b|0}function Lm(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;var e=0,f=0,g=0,h=0,i=0;e=Ua-32|0;Ua=e;a:{b:{c:{if((c|0)<=0){break c}while(1){h=b+g|0;if(K[h|0]!=41){g=g+1|0;if((c|0)!=(g|0)){continue}break c}break}if(g){break b}}H[a+16|0]=0;H[a|0]=0;break a}if(g>>>0<2147483632){d:{if(g>>>0<=10){H[e+31|0]=g;c=e+20|0;break d}f=(g|15)+1|0;c=cb(f);J[e+28>>2]=f|-2147483648;J[e+20>>2]=c;J[e+24>>2]=g;}i=g&7;if(i){f=0;while(1){H[c|0]=K[b|0];c=c+1|0;b=b+1|0;f=f+1|0;if((i|0)!=(f|0)){continue}break}}if(g>>>0>=8){while(1){H[c|0]=K[b|0];H[c+1|0]=K[b+1|0];H[c+2|0]=K[b+2|0];H[c+3|0]=K[b+3|0];H[c+4|0]=K[b+4|0];H[c+5|0]=K[b+5|0];H[c+6|0]=K[b+6|0];H[c+7|0]=K[b+7|0];c=c+8|0;b=b+8|0;if((h|0)!=(b|0)){continue}break}}H[c|0]=0;f=K[e+31|0];c=f<<24>>24;b=(c|0)<0;e:{f:{g:{f=b?J[e+24>>2]:f;h:{if(!f){break h}b=b?J[e+20>>2]:e+20|0;h=f+b|0;while(1){if(H[K[b|0]+5584|0]<0){break h}b=b+1|0;if((h|0)!=(b|0)){continue}break}if(!(f&1)){break g}}H[a+16|0]=0;H[a|0]=0;break f}J[5996]=0;aa(31,e+8|0,e+20|0);b=J[5996];J[5996]=0;if((b|0)==1){break e}c=J[e+12>>2];b=J[e+8>>2];i:{if((c-b|0)==(d|0)){d=J[e+16>>2];H[a+16|0]=1;J[a+12>>2]=g;J[a+8>>2]=d;J[a+4>>2]=c;J[a>>2]=b;break i}H[a+16|0]=0;H[a|0]=0;if(!b){break i}$a(b);}c=K[e+31|0];}if(c<<24>>24>=0){break a}$a(J[e+20>>2]);break a}a=Z()|0;if(H[e+31|0]<0){$a(J[e+20>>2]);}da(a|0);B();}ub();B();}Ua=e+32|0;}function wk(a,b,c,d,e,f,g){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;var h=0,i=0,j=0,k=0;a=Ua-144|0;Ua=a;J[a+136>>2]=c;J[a+140>>2]=b;J[a+20>>2]=220;b=a+24|0;J[b>>2]=a+32;i=a+20|0;J[b+4>>2]=J[i>>2];j=a+16|0;h=J[e+28>>2];J[j>>2]=h;J[h+4>>2]=J[h+4>>2]+1;J[5996]=0;h=$(170,J[a+16>>2])|0;k=J[5996];J[5996]=0;if((k|0)!=1){a:{H[a+15|0]=0;e=J[e+4>>2];J[5996]=0;c=ua(221,a+140|0,c|0,d|0,j|0,e|0,f|0,a+15|0,h|0,b|0,i|0,a+132|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break a}b:{c:{if(!c){break c}d:{if((K[g+11|0]&128)>>>7|0){H[J[g>>2]]=0;J[g+4>>2]=0;break d}H[g|0]=0;H[g+11|0]=0;}if(K[a+15|0]){J[5996]=0;c=_(171,h|0,45)|0;d=J[5996];J[5996]=0;if((d|0)==1){break a}J[5996]=0;aa(98,g|0,c|0);c=J[5996];J[5996]=0;if((c|0)==1){break a}}J[5996]=0;d=_(171,h|0,48)|0;c=J[5996];J[5996]=0;if((c|0)==1){break b}e=J[a+20>>2];h=e-1|0;c=J[b>>2];d=d&255;while(1){if(!((d|0)!=K[c|0]|c>>>0>=h>>>0)){c=c+1|0;continue}break}J[5996]=0;ba(227,g|0,c|0,e|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){break c}break a}J[5996]=0;c=_(186,a+140|0,a+136|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break a}if(c){J[f>>2]=J[f>>2]|2;}e=J[a+140>>2];c=J[a+16>>2];d=J[c+4>>2]-1|0;J[c+4>>2]=d;if((d|0)==-1){Xa[J[J[c>>2]+8>>2]](c);}db(b);Ua=a+144|0;return e|0}}}d=Z()|0;a=J[a+16>>2];c=J[a+4>>2]-1|0;J[a+4>>2]=c;if((c|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}db(b);da(d|0);B();}function ok(a,b,c,d,e,f,g){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;var h=0,i=0,j=0,k=0;a=Ua-448|0;Ua=a;J[a+440>>2]=c;J[a+444>>2]=b;J[a+20>>2]=220;b=a+24|0;J[b>>2]=a+32;i=a+20|0;J[b+4>>2]=J[i>>2];j=a+16|0;h=J[e+28>>2];J[j>>2]=h;J[h+4>>2]=J[h+4>>2]+1;J[5996]=0;h=$(197,J[a+16>>2])|0;k=J[5996];J[5996]=0;if((k|0)!=1){a:{H[a+15|0]=0;e=J[e+4>>2];J[5996]=0;c=ua(228,a+444|0,c|0,d|0,j|0,e|0,f|0,a+15|0,h|0,b|0,i|0,a+432|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break a}b:{c:{if(!c){break c}d:{if((K[g+11|0]&128)>>>7|0){J[J[g>>2]>>2]=0;J[g+4>>2]=0;break d}J[g>>2]=0;H[g+11|0]=0;}if(K[a+15|0]){J[5996]=0;c=_(216,h|0,45)|0;d=J[5996];J[5996]=0;if((d|0)==1){break a}J[5996]=0;aa(232,g|0,c|0);c=J[5996];J[5996]=0;if((c|0)==1){break a}}J[5996]=0;d=_(216,h|0,48)|0;c=J[5996];J[5996]=0;if((c|0)==1){break b}e=J[a+20>>2];h=e-4|0;c=J[b>>2];while(1){if(!((d|0)!=J[c>>2]|c>>>0>=h>>>0)){c=c+4|0;continue}break}J[5996]=0;ba(234,g|0,c|0,e|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){break c}break a}J[5996]=0;c=_(202,a+444|0,a+440|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break a}if(c){J[f>>2]=J[f>>2]|2;}e=J[a+444>>2];c=J[a+16>>2];d=J[c+4>>2]-1|0;J[c+4>>2]=d;if((d|0)==-1){Xa[J[J[c>>2]+8>>2]](c);}db(b);Ua=a+448|0;return e|0}}}d=Z()|0;a=J[a+16>>2];c=J[a+4>>2]-1|0;J[a+4>>2]=c;if((c|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}db(b);da(d|0);B();}
function Hh(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;var f=0,g=0,h=0;if(Jb(a,J[b+8>>2],e)){if(!(J[b+28>>2]==1|J[b+4>>2]!=(c|0))){J[b+28>>2]=d;}return}a:{if(Jb(a,J[b>>2],e)){if(!(J[b+16>>2]!=(c|0)&J[b+20>>2]!=(c|0))){if((d|0)!=1){break a}J[b+32>>2]=1;return}J[b+32>>2]=d;b:{if(J[b+44>>2]==4){break b}f=a+16|0;h=f+(J[a+12>>2]<<3)|0;d=0;c:{d:{while(1){e:{if(f>>>0>=h>>>0){break e}I[b+52>>1]=0;Fd(J[f>>2],J[f+4>>2],b,c,c,1,e);if(K[b+54|0]){break e}f:{if(!K[b+53|0]){break f}if(K[b+52|0]){if(J[b+24>>2]!=1){d=1;g=1;if(K[a+8|0]&2){break f}}J[b+44>>2]=3;return}d=1;if(!(H[a+8|0]&1)){break d}}f=f+8|0;continue}break}if(!d){break c}}J[b+44>>2]=3;if(!g){break b}break a}J[b+44>>2]=4;if(g){break a}}J[b+20>>2]=c;J[b+40>>2]=J[b+40>>2]+1;if(J[b+36>>2]!=1|J[b+24>>2]!=2){break a}H[b+54|0]=1;return}g=J[a+12>>2];cd(J[a+16>>2],J[a+20>>2],b,c,d,e);f=a+24|0;g=(a+16|0)+(g<<3)|0;if(f>>>0>=g>>>0){break a}a=J[a+8>>2];if(!(!(a&2)&J[b+36>>2]!=1)){while(1){if(K[b+54|0]){break a}cd(J[f>>2],J[f+4>>2],b,c,d,e);f=f+8|0;if(g>>>0>f>>>0){continue}break}break a}if(!(a&1)){while(1){if(K[b+54|0]|J[b+36>>2]==1){break a}cd(J[f>>2],J[f+4>>2],b,c,d,e);f=f+8|0;if(g>>>0>f>>>0){continue}break a}}while(1){if(K[b+54|0]|J[b+36>>2]==1&J[b+24>>2]==1){break a}cd(J[f>>2],J[f+4>>2],b,c,d,e);f=f+8|0;if(g>>>0>f>>>0){continue}break}}}function Qe(a,b,c,d,e,f,g,h){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;var i=0;a=Ua-16|0;Ua=a;J[a+12>>2]=c;J[a+8>>2]=f;J[a+12>>2]=c;J[a+8>>2]=f;c=J[a+12>>2];a:{while(1){if(c>>>0>=d>>>0){f=0;break a}b:{b=L[c>>1];c:{if(b>>>0<=127){f=1;c=J[a+8>>2];if((g-c|0)<=0){break a}J[a+8>>2]=c+1;H[c|0]=b;break c}if(b>>>0<=2047){c=J[a+8>>2];if((g-c|0)<2){break b}J[a+8>>2]=c+1;H[c|0]=b>>>6|192;c=J[a+8>>2];J[a+8>>2]=c+1;H[c|0]=b&63|128;break c}if(b>>>0<=55295){c=J[a+8>>2];if((g-c|0)<3){break b}J[a+8>>2]=c+1;H[c|0]=b>>>12|224;c=J[a+8>>2];J[a+8>>2]=c+1;H[c|0]=b>>>6&63|128;c=J[a+8>>2];J[a+8>>2]=c+1;H[c|0]=b&63|128;break c}d:{e:{if(b>>>0<=56319){f=1;if((d-c|0)<4){break a}i=L[c+2>>1];if((i&64512)!=56320){break e}if((g-J[a+8>>2]|0)<4){break a}J[a+12>>2]=c+2;c=J[a+8>>2];J[a+8>>2]=c+1;f=c;c=(b>>>6&15)+1|0;H[f|0]=c>>>2|240;f=J[a+8>>2];J[a+8>>2]=f+1;H[f|0]=c<<4&48|b>>>2&15|128;c=J[a+8>>2];J[a+8>>2]=c+1;H[c|0]=i>>>6&15|b<<4&48|128;b=J[a+8>>2];J[a+8>>2]=b+1;H[b|0]=i&63|128;break c}if(b>>>0>=57344){break d}}f=2;break a}c=J[a+8>>2];if((g-c|0)<3){break b}J[a+8>>2]=c+1;H[c|0]=b>>>12|224;c=J[a+8>>2];J[a+8>>2]=c+1;H[c|0]=b>>>6&63|128;c=J[a+8>>2];J[a+8>>2]=c+1;H[c|0]=b&63|128;}c=J[a+12>>2]+2|0;J[a+12>>2]=c;continue}break}f=1;}J[e>>2]=J[a+12>>2];J[h>>2]=J[a+8>>2];Ua=a+16|0;return f|0}function Yn(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0,h=0;c=J[a+4>>2];d=J[a+8>>2];if(c>>>0<d>>>0){J[c>>2]=J[b>>2];J[c+4>>2]=J[b+4>>2];J[b>>2]=0;J[b+4>>2]=0;e=J[b+28>>2];J[c+24>>2]=J[b+24>>2];J[c+28>>2]=e;e=J[b+20>>2];J[c+16>>2]=J[b+16>>2];J[c+20>>2]=e;e=J[b+12>>2];J[c+8>>2]=J[b+8>>2];J[c+12>>2]=e;J[a+4>>2]=c+32;return}a:{h=J[a>>2];f=c-h>>5;e=f+1|0;if(e>>>0<134217728){d=d-h|0;g=d>>4;g=d>>>0>=2147483616?134217727:e>>>0<g>>>0?g:e;if(g){if(g>>>0>=134217728){break a}e=cb(g<<5);}else {e=0;}d=e+(f<<5)|0;J[d>>2]=J[b>>2];J[d+4>>2]=J[b+4>>2];J[b>>2]=0;J[b+4>>2]=0;f=J[b+28>>2];J[d+24>>2]=J[b+24>>2];J[d+28>>2]=f;f=J[b+20>>2];J[d+16>>2]=J[b+16>>2];J[d+20>>2]=f;f=J[b+12>>2];J[d+8>>2]=J[b+8>>2];J[d+12>>2]=f;f=d+32|0;if((c|0)!=(h|0)){while(1){d=d-32|0;c=c-32|0;J[d>>2]=J[c>>2];J[d+4>>2]=J[c+4>>2];J[c>>2]=0;J[c+4>>2]=0;b=J[c+28>>2];J[d+24>>2]=J[c+24>>2];J[d+28>>2]=b;b=J[c+20>>2];J[d+16>>2]=J[c+16>>2];J[d+20>>2]=b;b=J[c+12>>2];J[d+8>>2]=J[c+8>>2];J[d+12>>2]=b;if((c|0)!=(h|0)){continue}break}}J[a+8>>2]=e+(g<<5);b=J[a>>2];J[a>>2]=d;c=J[a+4>>2];J[a+4>>2]=f;if((b|0)!=(c|0)){while(1){c=c-32|0;a=J[c+4>>2];b:{if(!a){break b}e=J[a+4>>2];J[a+4>>2]=e-1;if(e){break b}Xa[J[J[a>>2]+8>>2]](a);bb(a);}if((b|0)!=(c|0)){continue}break}}if(b){$a(b);}return}nb();B();}Pb();B();}function hg(a,b){var c=0,d=0,e=0,f=0,g=0;a:{b:{c:{d:{e:{f:{c=J[a+4>>2];g:{if((c|0)!=J[a+104>>2]){J[a+4>>2]=c+1;c=K[c|0];break g}c=gb(a);}switch(c-43|0){case 0:case 2:break f;default:break e}}g=(c|0)==45;b=!b;c=J[a+4>>2];h:{if((c|0)!=J[a+104>>2]){J[a+4>>2]=c+1;c=K[c|0];break h}c=gb(a);}d=c-58|0;if(b|d>>>0>4294967285){break d}b=J[a+116>>2];if((b|0)>0|(b|0)>=0){break c}break a}d=c-58|0;}if(d>>>0<=4294967285){if(J[a+116>>2]<0){break a}break b}i:{if(c-48>>>0>=10){break i}while(1){e=(P(e,10)+c|0)-48|0;f=(e|0)<214748364;c=J[a+4>>2];j:{if((c|0)!=J[a+104>>2]){J[a+4>>2]=c+1;c=K[c|0];break j}c=gb(a);}b=c-48|0;if(f&b>>>0<=9){continue}break}f=e>>31;if(b>>>0>=10){break i}while(1){b=c;c=co(e,f,10,0);d=b+c|0;b=Wa;e=d-48|0;d=(c>>>0>d>>>0?b+1|0:b)-(d>>>0<48)|0;f=d;d=e>>>0<2061584302&(d|0)<=21474836|(d|0)<21474836;c=J[a+4>>2];k:{if((c|0)!=J[a+104>>2]){J[a+4>>2]=c+1;c=K[c|0];break k}c=gb(a);}b=c-48|0;if(d&b>>>0<=9){continue}break}if(b>>>0>=10){break i}while(1){b=J[a+4>>2];l:{if((b|0)!=J[a+104>>2]){J[a+4>>2]=b+1;b=K[b|0];break l}b=gb(a);}if(b-48>>>0<10){continue}break}}b=J[a+116>>2];if((b|0)>0|(b|0)>=0){J[a+4>>2]=J[a+4>>2]-1;}Wa=g?0-(((e|0)!=0)+f|0)|0:f;return g?0-e|0:e}J[a+4>>2]=J[a+4>>2]-1;}J[a+4>>2]=J[a+4>>2]-1;}Wa=-2147483648;return 0}function Se(a,b,c,d,e,f,g,h){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;var i=0,j=0,k=0,l=0,m=0;a=Ua-16|0;Ua=a;J[a+12>>2]=c;J[a+8>>2]=f;J[a+12>>2]=c;J[a+8>>2]=f;a:{b:{while(1){b=J[a+12>>2];if(!(d>>>0<=b>>>0|f>>>0>=g>>>0)){i=H[b|0];c=i&255;c:{if((i|0)>=0){i=1;break c}k=2;if(i>>>0<4294967234){break a}if(i>>>0<=4294967263){if((d-b|0)<2){break b}i=K[b+1|0];if((i&192)!=128){break a}c=i&63|c<<6&1984;i=2;break c}if(i>>>0<=4294967279){if((d-b|0)<3){break b}j=K[b+2|0];i=K[b+1|0];d:{e:{if((c|0)!=237){if((c|0)!=224){break e}if((i&224)==160){break d}break a}if((i&224)==128){break d}break a}if((i&192)!=128){break a}}if((j&192)!=128){break a}c=j&63|(c<<12&61440|(i&63)<<6);i=3;break c}if(i>>>0>4294967284){break a}if((d-b|0)<4){break b}l=K[b+3|0];m=K[b+2|0];j=K[b+1|0];f:{g:{switch(c-240|0){case 0:if((j+112&255)>>>0<48){break f}break a;case 4:if((j&240)==128){break f}break a;default:break g}}if((j&192)!=128){break a}}if((m&192)!=128|(l&192)!=128){break a}i=4;c=l&63|(m<<6&4032|(c<<18&1835008|(j&63)<<12));if(c>>>0>1114111){break a}}J[f>>2]=c;J[a+12>>2]=b+i;f=J[a+8>>2]+4|0;J[a+8>>2]=f;continue}break}k=b>>>0<d>>>0;break a}k=1;}J[e>>2]=J[a+12>>2];J[h>>2]=J[a+8>>2];Ua=a+16|0;return k|0}function Wn(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;var f=0,g=0,h=0,i=0,j=0;f=Ua-32|0;Ua=f;J[a+8>>2]=0;J[a+12>>2]=0;J[a+4>>2]=e;J[a>>2]=b;J[a+16>>2]=0;J[a+8>>2]=J[d>>2];J[a+12>>2]=J[d+4>>2];J[a+16>>2]=J[d+8>>2];J[d+8>>2]=0;J[d>>2]=0;J[d+4>>2]=0;J[a+36>>2]=0;J[a+40>>2]=0;J[a+28>>2]=0;J[a+32>>2]=0;J[a+20>>2]=0;J[a+24>>2]=0;J[a+32>>2]=J[c>>2];J[a+36>>2]=J[c+4>>2];J[a+40>>2]=J[c+8>>2];J[c+8>>2]=0;J[c>>2]=0;J[c+4>>2]=0;J[5996]=0;aa(86,a+44|0,a|0);b=J[5996];J[5996]=0;a:{if((b|0)==1){break a}J[5996]=0;aa(87,a- -64|0,a|0);b=J[5996];J[5996]=0;if((b|0)==1){break a}c=0;J[5996]=0;b=$(88,a|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break a}J[a+80>>2]=b;b=J[a+32>>2];e=J[a+36>>2];if((b|0)==(e|0)){d=0;}else {d=b;while(1){c=J[J[d>>2]+84>>2]+c|0;d=d+8|0;if((e|0)!=(d|0)){continue}break}d=J[J[b>>2]+80>>2];}g=J[a+4>>2];h=J[a>>2];i=J[a+8>>2];j=J[a+12>>2];J[5996]=0;b=ka(89,h|0,d|0,c|0,g|0,e-b>>3,(j-i|0)/12|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}J[a+84>>2]=b;J[5996]=0;J[f+28>>2]=a;ea(90,f+12|0,a|0,f+28|0);b=J[5996];J[5996]=0;if((b|0)==1){break a}b=K[f+24|0];if(b){Db(J[f+16>>2]);}H[a+88|0]=b^1;Ua=f+32|0;return a|0}c=Z()|0;lb(a+32|0);b=J[a+20>>2];if(b){J[a+24>>2]=b;$a(b);}rb(a+8|0);da(c|0);B();}function Km(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0;g=Ua-32|0;Ua=g;d=cb(104);J[d>>2]=7e3;J[d+4>>2]=0;J[d+8>>2]=0;e=J[c+8>>2];J[c+8>>2]=0;f=J[c>>2];h=J[c+4>>2];J[c>>2]=0;J[c+4>>2]=0;J[d+16>>2]=0;J[d+20>>2]=0;J[d+12>>2]=b;J[d+24>>2]=0;J[d+28>>2]=0;J[d+52>>2]=0;i=d+44|0;b=i;J[b>>2]=0;J[b+4>>2]=0;J[d+40>>2]=e;J[d+32>>2]=f;J[d+36>>2]=h;J[5996]=0;f=d+12|0;aa(86,d+56|0,f|0);b=J[5996];J[5996]=0;a:{if((b|0)==1){break a}J[5996]=0;aa(87,d+76|0,f|0);b=J[5996];J[5996]=0;if((b|0)==1){break a}b=0;J[5996]=0;c=$(88,f|0)|0;e=J[5996];J[5996]=0;if((e|0)==1){break a}J[d+92>>2]=c;h=J[d+48>>2];e=J[d+44>>2];if((h|0)==(e|0)){c=0;}else {c=e;while(1){b=J[J[c>>2]+84>>2]+b|0;c=c+8|0;if((h|0)!=(c|0)){continue}break}c=J[J[e>>2]+80>>2];}j=J[d+16>>2];k=J[d+12>>2];l=J[d+20>>2];m=J[d+24>>2];J[5996]=0;b=ka(89,k|0,c|0,b|0,j|0,h-e>>3,(m-l|0)/12|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}J[d+96>>2]=b;J[5996]=0;J[g+28>>2]=f;ea(90,g+12|0,f|0,g+28|0);b=J[5996];J[5996]=0;if((b|0)==1){break a}b=K[g+24|0];if(b){Db(J[g+16>>2]);}H[d+100|0]=b^1;J[a+4>>2]=d;J[a>>2]=f;Ua=g+32|0;return}b=Z()|0;lb(i);a=J[d+32>>2];if(a){J[d+36>>2]=a;$a(a);}rb(d+20|0);$a(d);da(b|0);B();}function sh(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0;g=Ua-32|0;Ua=g;h=1;a:{b:{c:{d:{switch(a|0){case 3:h=24;break c;case 4:case 5:J[g+24>>2]=0;J[g+28>>2]=0;J[g+16>>2]=0;J[g+20>>2]=0;J[g+8>>2]=0;J[g+12>>2]=0;J[g>>2]=0;J[g+4>>2]=0;J[5996]=0;a=nc(g,d,0);b=J[5996];J[5996]=0;if((b|0)==1){h=Z()|0;if(M[g>>2]<29){break a}break b}a=J[a>>2];b=a>>>0<29;h=(b?a:a-29|0)+1|0;if(b){break c}$a(J[g+8>>2]);break c;case 6:case 7:h=39;break c;case 8:case 9:h=27;break c;case 25:h=(P(f,34)|d>>>0>16)+(f>>>0>16?4:3)|0;break c;case 17:h=c;break c;case 14:h=!(Wb(1239,1)&(b^-1))+c|0;break c;case 11:case 12:case 16:case 18:case 19:h=c+1|0;break c;case 10:case 20:h=c+2|0;break c;case 13:case 21:case 22:case 23:h=c+3|0;break c;case 15:h=c+4|0;break c;case 24:J[g+24>>2]=0;J[g+28>>2]=0;J[g+16>>2]=0;J[g+20>>2]=0;J[g+8>>2]=0;J[g+12>>2]=0;J[g>>2]=0;J[g+4>>2]=0;J[5996]=0;a=nc(g,d,0);b=J[5996];J[5996]=0;if((b|0)==1){h=Z()|0;if(M[g>>2]>=29){break b}break a}a=J[a>>2];b=a>>>0<29;h=(c+e|0)+(b?a:a-29|0)|0;if(b){break c}$a(J[g+8>>2]);break c;default:la(2350,1859,280,2003);B();case 0:case 1:break c;case 2:break d}}h=34;}Ua=g+32|0;return h|0}$a(J[g+8>>2]);}da(h|0);B();}function cn(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0,i=0,j=0;g=J[a+8>>2];e=J[a+4>>2];if(g-e>>>0>=b>>>0){d=e;h=b&7;if(h){while(1){H[d|0]=K[c|0];d=d+1|0;f=f+1|0;if((h|0)!=(f|0)){continue}break}}b=b+e|0;while(1){H[d|0]=K[c|0];H[d+1|0]=K[c|0];H[d+2|0]=K[c|0];H[d+3|0]=K[c|0];H[d+4|0]=K[c|0];H[d+5|0]=K[c|0];H[d+6|0]=K[c|0];H[d+7|0]=K[c|0];d=d+8|0;if((b|0)!=(d|0)){continue}break}J[a+4>>2]=b;return}d=J[a>>2];f=e-d|0;i=f+b|0;if((i|0)>=0){g=g-d|0;d=g<<1;i=g>>>0>=1073741823?2147483647:d>>>0>i>>>0?d:i;if(i){j=cb(i);}f=f+j|0;d=f;g=b&7;if(g){while(1){H[d|0]=K[c|0];d=d+1|0;h=h+1|0;if((g|0)!=(h|0)){continue}break}}h=b+f|0;while(1){H[d|0]=K[c|0];H[d+1|0]=K[c|0];H[d+2|0]=K[c|0];H[d+3|0]=K[c|0];H[d+4|0]=K[c|0];H[d+5|0]=K[c|0];H[d+6|0]=K[c|0];H[d+7|0]=K[c|0];d=d+8|0;if((h|0)!=(d|0)){continue}break}g=J[a>>2];if((g|0)!=(e|0)){d=(g^-1)+e|0;b=e-g&3;if(b){c=0;while(1){f=f-1|0;e=e-1|0;H[f|0]=K[e|0];c=c+1|0;if((b|0)!=(c|0)){continue}break}}if(d>>>0>=3){while(1){H[f-1|0]=K[e-1|0];H[f-2|0]=K[e-2|0];H[f-3|0]=K[e-3|0];f=f-4|0;e=e-4|0;H[f|0]=K[e|0];if((e|0)!=(g|0)){continue}break}}e=J[a>>2];}J[a+8>>2]=i+j;J[a+4>>2]=h;J[a>>2]=f;if(e){$a(e);}return}nb();B();}function qe(a,b,c,d,e){var f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0;m=Ua-16|0;Ua=m;g=K[a|0];a:{if(!g){f=a;break a}f=a;b:{while(1){i=g<<24>>24;if(!((i|0)==32|i-9>>>0<5)){break b}g=K[f+1|0];f=f+1|0;if(g){continue}break}break a}c:{g=g&255;switch(g-43|0){case 0:case 2:break c;default:break a}}j=(g|0)==45?-1:0;f=f+1|0;}d:{if(!((c|16)!=16|K[f|0]!=48)){o=1;if((K[f+1|0]&223)==88){f=f+2|0;i=16;break d}f=f+1|0;i=c?c:8;break d}i=c?c:10;}c=0;while(1){e:{g=-48;k=H[f|0];f:{if((k-48&255)>>>0<10){break f}g=-87;if((k-97&255)>>>0<26){break f}g=-55;if((k-65&255)>>>0>25){break e}}n=g+k|0;if((n|0)>=(i|0)){break e}xb(m,i,0,0,0,l,h,0,0);g=1;g:{if(J[m+8>>2]|J[m+12>>2]){break g}p=co(l,h,i,0);k=Wa;if((k|0)==-1&(n^-1)>>>0<p>>>0){break g}h=k;l=n+p|0;h=l>>>0<n>>>0?h+1|0:h;o=1;g=c;}f=f+1|0;c=g;continue}break}if(b){J[b>>2]=o?f:a;}h:{i:{j:{if(c){J[5732]=68;a=d&1;j=a?0:j;l=d;h=e;break j}if((e|0)==(h|0)&d>>>0>l>>>0|e>>>0>h>>>0){break i}a=d&1;}if(!(a|j)){J[5732]=68;a=d;d=a-1|0;e=e-!a|0;break h}if((e|0)==(h|0)&d>>>0>=l>>>0|e>>>0>h>>>0){break i}J[5732]=68;break h}a=j^l;d=a-j|0;b=j>>31;e=(b^h)-((a>>>0<j>>>0)+b|0)|0;}Ua=m+16|0;Wa=e;return d}function gk(a,b,c,d,e,f,g,h,i,j){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;i=i|0;j=j|0;var k=0;k=Ua-16|0;Ua=k;a:{if(a){a=qf(c);b:{if(b){b=k+4|0;Xa[J[J[a>>2]+44>>2]](b,a);c=J[k+4>>2];H[d|0]=c;H[d+1|0]=c>>>8;H[d+2|0]=c>>>16;H[d+3|0]=c>>>24;Xa[J[J[a>>2]+32>>2]](b,a);break b}b=k+4|0;Xa[J[J[a>>2]+40>>2]](b,a);c=J[k+4>>2];H[d|0]=c;H[d+1|0]=c>>>8;H[d+2|0]=c>>>16;H[d+3|0]=c>>>24;Xa[J[J[a>>2]+28>>2]](b,a);}ac(i,b);ab(b);J[e>>2]=Xa[J[J[a>>2]+12>>2]](a);J[f>>2]=Xa[J[J[a>>2]+16>>2]](a);b=k+4|0;Xa[J[J[a>>2]+20>>2]](b,a);Ab(g,b);ab(b);Xa[J[J[a>>2]+24>>2]](b,a);ac(h,b);ab(b);a=Xa[J[J[a>>2]+36>>2]](a)|0;break a}a=pf(c);c:{if(b){b=k+4|0;Xa[J[J[a>>2]+44>>2]](b,a);c=J[k+4>>2];H[d|0]=c;H[d+1|0]=c>>>8;H[d+2|0]=c>>>16;H[d+3|0]=c>>>24;Xa[J[J[a>>2]+32>>2]](b,a);break c}b=k+4|0;Xa[J[J[a>>2]+40>>2]](b,a);c=J[k+4>>2];H[d|0]=c;H[d+1|0]=c>>>8;H[d+2|0]=c>>>16;H[d+3|0]=c>>>24;Xa[J[J[a>>2]+28>>2]](b,a);}ac(i,b);ab(b);J[e>>2]=Xa[J[J[a>>2]+12>>2]](a);J[f>>2]=Xa[J[J[a>>2]+16>>2]](a);b=k+4|0;Xa[J[J[a>>2]+20>>2]](b,a);Ab(g,b);ab(b);Xa[J[J[a>>2]+24>>2]](b,a);ac(h,b);ab(b);a=Xa[J[J[a>>2]+36>>2]](a)|0;}J[j>>2]=a;Ua=k+16|0;}function fb(a,b,c){var d=0,e=0,f=0;if(c>>>0>=512){Ia(a|0,b|0,c|0);return a}e=a+c|0;a:{if(!((a^b)&3)){b:{if(!(a&3)){c=a;break b}if(!c){c=a;break b}c=a;while(1){H[c|0]=K[b|0];b=b+1|0;c=c+1|0;if(!(c&3)){break b}if(c>>>0<e>>>0){continue}break}}d=e&-4;c:{if(d>>>0<64){break c}f=d+-64|0;if(f>>>0<c>>>0){break c}while(1){J[c>>2]=J[b>>2];J[c+4>>2]=J[b+4>>2];J[c+8>>2]=J[b+8>>2];J[c+12>>2]=J[b+12>>2];J[c+16>>2]=J[b+16>>2];J[c+20>>2]=J[b+20>>2];J[c+24>>2]=J[b+24>>2];J[c+28>>2]=J[b+28>>2];J[c+32>>2]=J[b+32>>2];J[c+36>>2]=J[b+36>>2];J[c+40>>2]=J[b+40>>2];J[c+44>>2]=J[b+44>>2];J[c+48>>2]=J[b+48>>2];J[c+52>>2]=J[b+52>>2];J[c+56>>2]=J[b+56>>2];J[c+60>>2]=J[b+60>>2];b=b- -64|0;c=c- -64|0;if(f>>>0>=c>>>0){continue}break}}if(c>>>0>=d>>>0){break a}while(1){J[c>>2]=J[b>>2];b=b+4|0;c=c+4|0;if(d>>>0>c>>>0){continue}break}break a}if(e>>>0<4){c=a;break a}d=e-4|0;if(d>>>0<a>>>0){c=a;break a}c=a;while(1){H[c|0]=K[b|0];H[c+1|0]=K[b+1|0];H[c+2|0]=K[b+2|0];H[c+3|0]=K[b+3|0];b=b+4|0;c=c+4|0;if(d>>>0>=c>>>0){continue}break}}if(c>>>0<e>>>0){while(1){H[c|0]=K[b|0];b=b+1|0;c=c+1|0;if((e|0)!=(c|0)){continue}break}}return a}function kk(a,b,c,d,e,f,g,h,i,j){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;i=i|0;j=j|0;var k=0;k=Ua-16|0;Ua=k;a:{if(a){a=tf(c);b:{if(b){b=k+4|0;Xa[J[J[a>>2]+44>>2]](b,a);c=J[k+4>>2];H[d|0]=c;H[d+1|0]=c>>>8;H[d+2|0]=c>>>16;H[d+3|0]=c>>>24;Xa[J[J[a>>2]+32>>2]](b,a);break b}b=k+4|0;Xa[J[J[a>>2]+40>>2]](b,a);c=J[k+4>>2];H[d|0]=c;H[d+1|0]=c>>>8;H[d+2|0]=c>>>16;H[d+3|0]=c>>>24;Xa[J[J[a>>2]+28>>2]](b,a);}Ab(i,b);ab(b);H[e|0]=Xa[J[J[a>>2]+12>>2]](a);H[f|0]=Xa[J[J[a>>2]+16>>2]](a);b=k+4|0;Xa[J[J[a>>2]+20>>2]](b,a);Ab(g,b);ab(b);Xa[J[J[a>>2]+24>>2]](b,a);Ab(h,b);ab(b);a=Xa[J[J[a>>2]+36>>2]](a)|0;break a}a=sf(c);c:{if(b){b=k+4|0;Xa[J[J[a>>2]+44>>2]](b,a);c=J[k+4>>2];H[d|0]=c;H[d+1|0]=c>>>8;H[d+2|0]=c>>>16;H[d+3|0]=c>>>24;Xa[J[J[a>>2]+32>>2]](b,a);break c}b=k+4|0;Xa[J[J[a>>2]+40>>2]](b,a);c=J[k+4>>2];H[d|0]=c;H[d+1|0]=c>>>8;H[d+2|0]=c>>>16;H[d+3|0]=c>>>24;Xa[J[J[a>>2]+28>>2]](b,a);}Ab(i,b);ab(b);H[e|0]=Xa[J[J[a>>2]+12>>2]](a);H[f|0]=Xa[J[J[a>>2]+16>>2]](a);b=k+4|0;Xa[J[J[a>>2]+20>>2]](b,a);Ab(g,b);ab(b);Xa[J[J[a>>2]+24>>2]](b,a);Ab(h,b);ab(b);a=Xa[J[J[a>>2]+36>>2]](a)|0;}J[j>>2]=a;Ua=k+16|0;}function Un(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;var e=0,f=0,g=0,h=0,i=0,j=0,k=0;e=Ua-32|0;Ua=e;J[a+8>>2]=0;J[a+12>>2]=0;J[a+4>>2]=d;J[a>>2]=b;J[a+16>>2]=0;J[a+20>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;g=a+32|0;b=g;J[b>>2]=0;J[b+4>>2]=0;J[a+40>>2]=0;J[b>>2]=J[c>>2];J[a+36>>2]=J[c+4>>2];J[a+40>>2]=J[c+8>>2];J[c+8>>2]=0;J[c>>2]=0;J[c+4>>2]=0;J[5996]=0;aa(86,a+44|0,a|0);b=J[5996];J[5996]=0;a:{if((b|0)==1){break a}J[5996]=0;aa(87,a- -64|0,a|0);b=J[5996];J[5996]=0;if((b|0)==1){break a}d=0;J[5996]=0;b=$(88,a|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}J[a+80>>2]=b;f=J[a+36>>2];b=J[a+32>>2];if((f|0)==(b|0)){c=0;}else {c=b;while(1){d=J[J[c>>2]+84>>2]+d|0;c=c+8|0;if((f|0)!=(c|0)){continue}break}c=J[J[b>>2]+80>>2];}h=J[a+4>>2];i=J[a>>2];j=J[a+8>>2];k=J[a+12>>2];J[5996]=0;b=ka(89,i|0,c|0,d|0,h|0,f-b>>3,(k-j|0)/12|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}J[a+84>>2]=b;J[5996]=0;J[e+28>>2]=a;ea(90,e+12|0,a|0,e+28|0);b=J[5996];J[5996]=0;if((b|0)==1){break a}b=K[e+24|0];if(b){Db(J[e+16>>2]);}H[a+88|0]=b^1;Ua=e+32|0;return a|0}c=Z()|0;lb(g);b=J[a+20>>2];if(b){J[a+24>>2]=b;$a(b);}rb(a+8|0);da(c|0);B();}function Dm(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;var e=0,f=0,g=0,h=0,i=0,j=0;e=Ua-32|0;Ua=e;J[a+8>>2]=0;J[a+12>>2]=0;J[a+4>>2]=d;J[a>>2]=b;J[a+16>>2]=0;J[a+8>>2]=J[c>>2];J[a+12>>2]=J[c+4>>2];J[a+16>>2]=J[c+8>>2];J[c+8>>2]=0;J[c>>2]=0;J[c+4>>2]=0;J[a+36>>2]=0;J[a+40>>2]=0;J[a+28>>2]=0;J[a+32>>2]=0;J[a+20>>2]=0;J[a+24>>2]=0;J[5996]=0;aa(86,a+44|0,a|0);b=J[5996];J[5996]=0;a:{if((b|0)==1){break a}J[5996]=0;aa(87,a- -64|0,a|0);b=J[5996];J[5996]=0;if((b|0)==1){break a}d=0;J[5996]=0;b=$(88,a|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}J[a+80>>2]=b;f=J[a+36>>2];b=J[a+32>>2];if((f|0)==(b|0)){c=0;}else {c=b;while(1){d=J[J[c>>2]+84>>2]+d|0;c=c+8|0;if((f|0)!=(c|0)){continue}break}c=J[J[b>>2]+80>>2];}g=J[a+4>>2];h=J[a>>2];i=J[a+8>>2];j=J[a+12>>2];J[5996]=0;b=ka(89,h|0,c|0,d|0,g|0,f-b>>3,(j-i|0)/12|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}J[a+84>>2]=b;J[5996]=0;J[e+28>>2]=a;ea(90,e+12|0,a|0,e+28|0);b=J[5996];J[5996]=0;if((b|0)==1){break a}b=K[e+24|0];if(b){Db(J[e+16>>2]);}H[a+88|0]=b^1;Ua=e+32|0;return a|0}c=Z()|0;lb(a+32|0);b=J[a+20>>2];if(b){J[a+24>>2]=b;$a(b);}rb(a+8|0);da(c|0);B();}function Zm(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0,h=0;d=Ua-16|0;Ua=d;J[5996]=0;f=_(134,d+8|0,a|0)|0;c=J[5996];J[5996]=0;a:{b:{c:{if((c|0)!=1){d:{if(!K[f|0]){break d}c=J[(J[J[a>>2]-12>>2]+a|0)+28>>2];J[d+4>>2]=c;J[c+4>>2]=J[c+4>>2]+1;J[5996]=0;h=$(135,J[d+4>>2])|0;c=J[5996];J[5996]=0;e:{f:{if((c|0)!=1){c=J[d+4>>2];e=J[c+4>>2]-1|0;J[c+4>>2]=e;if((e|0)==-1){Xa[J[J[c>>2]+8>>2]](c);}J[d+4>>2]=J[(J[J[a>>2]-12>>2]+a|0)+24>>2];c=J[J[a>>2]-12>>2];J[5996]=0;c=a+c|0;e=$(136,c|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break f}g=J[d+4>>2];J[5996]=0;b=xa(137,h|0,g|0,c|0,e|0,b|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break f}if(b){break d}b=J[J[a>>2]-12>>2];J[5996]=0;aa(138,a+b|0,5);b=J[5996];J[5996]=0;if((b|0)!=1){break d}b=fa(0)|0;break e}b=fa(0)|0;c=J[d+4>>2];e=J[c+4>>2]-1|0;J[c+4>>2]=e;if((e|0)==-1){Xa[J[J[c>>2]+8>>2]](c);}break e}b=fa(0)|0;}ud(f);break c}ud(f);break b}b=fa(0)|0;}pa(b|0)|0;b=J[J[a>>2]-12>>2];J[5996]=0;ha(139,a+b|0);b=J[5996];J[5996]=0;if((b|0)==1){break a}ra();}Ua=d+16|0;return a|0}a=Z()|0;J[5996]=0;ca(110);b=J[5996];J[5996]=0;if((b|0)!=1){da(a|0);B();}fa(0)|0;Vb();B();}function Sn(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0,i=0,j=0;e=Ua-32|0;Ua=e;J[a+8>>2]=0;J[a>>2]=0;J[a+4>>2]=0;H[e+4|0]=0;J[e>>2]=a;a:{b:{c:{d:{e:{if(c){if(c>>>0>=357913942){J[5996]=0;ca(47);a=J[5996];J[5996]=0;if((a|0)==1){break c}break a}J[5996]=0;c=P(c,12);g=$(3,c|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break c}J[a+4>>2]=g;J[a>>2]=g;J[a+8>>2]=c+g;J[e+24>>2]=g;H[e+20|0]=0;J[e+8>>2]=a+8;j=b+c|0;J[e+16>>2]=e+28;J[e+12>>2]=e+24;d=g;while(1){J[d+8>>2]=0;J[d>>2]=0;J[d+4>>2]=0;f=J[b>>2];c=J[b+4>>2];if((f|0)!=(c|0)){h=c-f|0;if((h|0)<0){J[5996]=0;J[e+28>>2]=d;ca(84);b=J[5996];J[5996]=0;if((b|0)!=1){break a}c=Z()|0;b=J[d>>2];if(!b){break d}J[d+4>>2]=b;$a(b);break d}J[5996]=0;c=$(3,h|0)|0;i=J[5996];J[5996]=0;if((i|0)==1){break e}J[d+4>>2]=c;J[d>>2]=c;J[d+8>>2]=(h&-8)+c;h=J[b+4>>2];if((h|0)!=(f|0)){while(1){i=J[f+4>>2];J[c>>2]=J[f>>2];J[c+4>>2]=i;c=c+8|0;f=f+8|0;if((h|0)!=(f|0)){continue}break}}J[d+4>>2]=c;}d=d+12|0;b=b+12|0;if((j|0)!=(b|0)){continue}break}J[a+4>>2]=d;}Ua=e+32|0;return a|0}c=Z()|0;J[e+28>>2]=d;}Ed(e+8|0);J[a+4>>2]=g;break b}c=Z()|0;}Nc(e);da(c|0);B();}B();}function pk(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0,h=0,i=0,j=0,k=0;h=J[b>>2];c=H[b+11|0];J[a+8>>2]=0;J[a>>2]=0;J[a+4>>2]=0;b=(c|0)<0?h:b;c=0;h=0;a:{b:{c:{while(1){e=d;g=c;f=h;while(1){c=K[b|0];if(!(c-9>>>0>=5&(c|0)!=32)){b=b+1|0;continue}d=K[c+5584|0];if((d|0)==255){break a}c=K[K[b+1|0]+5584|0];if((c|0)==255){break a}b=b+2|0;h=c|d<<4;if((e|0)!=(f|0)){H[f|0]=h;f=f+1|0;J[a+4>>2]=f;continue}break}i=f-g|0;d=i+1|0;if((d|0)<0){J[5996]=0;ca(8);b=J[5996];J[5996]=0;if((b|0)==1){break c}B();}d:{c=i<<1;d=i>>>0>=1073741823?2147483647:c>>>0>d>>>0?c:d;e:{if(!d){j=0;break e}J[5996]=0;j=$(3,d|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break d}}c=i+j|0;H[c|0]=h;d=d+j|0;h=c+1|0;if((e|0)!=(g|0)){k=(g^-1)+f|0;f=0;i=i&3;if(i){while(1){c=c-1|0;e=e-1|0;H[c|0]=K[e|0];f=f+1|0;if((i|0)!=(f|0)){continue}break}}if(k>>>0>=3){while(1){H[c-1|0]=K[e-1|0];H[c-2|0]=K[e-2|0];H[c-3|0]=K[e-3|0];c=c-4|0;e=e-4|0;H[c|0]=K[e|0];if((e|0)!=(g|0)){continue}break}}c=j;}J[a+8>>2]=d;J[a+4>>2]=h;J[a>>2]=c;if(!g){continue}$a(g);continue}break}b=Z()|0;break b}b=Z()|0;g=J[a>>2];}if(g){J[a+4>>2]=g;$a(g);}da(b|0);B();}}function vd(a,b,c,d,e,f){var g=0,h=0;H[a|0]=0;H[a+1|0]=0;H[a+2|0]=0;H[a+3|0]=0;H[a+4|0]=0;H[a+5|0]=0;H[a+6|0]=0;H[a+7|0]=0;H[a+24|0]=0;H[a+25|0]=0;H[a+26|0]=0;H[a+27|0]=0;H[a+28|0]=0;H[a+29|0]=0;H[a+30|0]=0;H[a+31|0]=0;H[a+16|0]=0;H[a+17|0]=0;H[a+18|0]=0;H[a+19|0]=0;H[a+20|0]=0;H[a+21|0]=0;H[a+22|0]=0;H[a+23|0]=0;H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;H[a+12|0]=0;H[a+13|0]=0;H[a+14|0]=0;H[a+15|0]=0;J[5996]=0;g=_(127,a|0,130)|0;h=J[5996];J[5996]=0;if((h|0)!=1){a:{b=J[b>>2];J[5996]=0;b=nc(g,b,b>>31);g=J[5996];J[5996]=0;if((g|0)==1){break a}c=J[c>>2];J[5996]=0;b=_(127,b|0,c|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}c=J[d>>2];J[5996]=0;b=_(127,b|0,c|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}J[5996]=0;b=_(19,b|0,e|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}c=J[f>>2];J[5996]=0;_(127,b|0,c|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break a}return}}b=Z()|0;if((K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))>>>0>=29){$a(K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24));H[a+8|0]=0;H[a+9|0]=0;H[a+10|0]=0;H[a+11|0]=0;}da(b|0);B();}function _d(a,b,c,d,e,f,g,h,i,j,k,l){a:{b:{if((a|0)==(f|0)){if(!K[b|0]){break a}f=0;H[b|0]=0;a=J[e>>2];J[e>>2]=a+1;H[a|0]=46;a=K[h+11|0];if(!((a&128)>>>7|0?J[h+4>>2]:a&127)){break b}a=J[j>>2];if((a-i|0)>159){break b}b=J[k>>2];J[j>>2]=a+4;J[a>>2]=b;return 0}c:{if((a|0)!=(g|0)){break c}f=K[h+11|0];if(!((f&128)>>>7|0?J[h+4>>2]:f&127)){break c}if(!K[b|0]){break a}f=0;a=J[j>>2];if((a-i|0)>159){break b}b=J[k>>2];J[j>>2]=a+4;J[a>>2]=b;J[k>>2]=0;return 0}f=-1;g=be(l,l+32|0,a)-l|0;if((g|0)>31){break b}a=H[g+12416|0];d:{e:{switch((g&-2)-22|0){case 2:b=J[e>>2];if((b|0)!=(d|0)){d=H[b-1|0];c=H[c|0];if(((d-97>>>0<26?d&95:d)|0)!=((c-97>>>0<26?c&95:c)|0)){break b}}J[e>>2]=b+1;H[b|0]=a;return 0;case 0:H[c|0]=80;break d;default:break e}}d=a-97>>>0<26?a&95:a;if((d|0)!=H[c|0]){break d}H[c|0]=d-65>>>0<26?d|32:d;if(!K[b|0]){break d}H[b|0]=0;b=K[h+11|0];if(!((b&128)>>>7|0?J[h+4>>2]:b&127)){break d}b=J[j>>2];if((b-i|0)>159){break d}c=J[k>>2];J[j>>2]=b+4;J[b>>2]=c;}b=J[e>>2];J[e>>2]=b+1;H[b|0]=a;f=0;if((g|0)>21){break b}J[k>>2]=J[k>>2]+1;}return f}return -1}function Wd(a,b,c,d,e,f,g,h,i,j,k,l){a:{b:{if((a|0)==(f|0)){if(!K[b|0]){break a}f=0;H[b|0]=0;a=J[e>>2];J[e>>2]=a+1;H[a|0]=46;a=K[h+11|0];if(!((a&128)>>>7|0?J[h+4>>2]:a&127)){break b}a=J[j>>2];if((a-i|0)>159){break b}b=J[k>>2];J[j>>2]=a+4;J[a>>2]=b;return 0}c:{if((a|0)!=(g|0)){break c}f=K[h+11|0];if(!((f&128)>>>7|0?J[h+4>>2]:f&127)){break c}if(!K[b|0]){break a}f=0;a=J[j>>2];if((a-i|0)>159){break b}b=J[k>>2];J[j>>2]=a+4;J[a>>2]=b;J[k>>2]=0;return 0}f=-1;g=Yd(l,l+128|0,a)-l|0;if((g|0)>124){break b}a=H[(g>>2)+12416|0];d:{e:{l=g&-5;if((l|0)!=88){if((l|0)!=96){break e}b=J[e>>2];if((b|0)!=(d|0)){d=H[b-1|0];c=H[c|0];if(((d-97>>>0<26?d&95:d)|0)!=((c-97>>>0<26?c&95:c)|0)){break b}}J[e>>2]=b+1;H[b|0]=a;return 0}H[c|0]=80;break d}d=a-97>>>0<26?a&95:a;if((d|0)!=H[c|0]){break d}H[c|0]=d-65>>>0<26?d|32:d;if(!K[b|0]){break d}H[b|0]=0;b=K[h+11|0];if(!((b&128)>>>7|0?J[h+4>>2]:b&127)){break d}b=J[j>>2];if((b-i|0)>159){break d}c=J[k>>2];J[j>>2]=b+4;J[b>>2]=c;}b=J[e>>2];J[e>>2]=b+1;H[b|0]=a;f=0;if((g|0)>84){break b}J[k>>2]=J[k>>2]+1;}return f}return -1}function Tl(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0,p=0,q=0,r=0,s=0;if((c|0)!=1){l=Ua-1040|0;Ua=l;q=J[5732];a:{b:{g=l+16|0;n=L[((c>>>0<=153?c:0)<<1)+21296>>1]+19372|0;b=J[J[6178]+20>>2];if(b){h=J[b+4>>2];d=J[b>>2];f=J[d>>2]+1794895138|0;i=wc(J[d+8>>2],f);j=wc(J[d+12>>2],f);e=wc(J[d+16>>2],f);c:{if(h>>>2>>>0<=i>>>0){break c}b=h-(i<<2)|0;if((e|j)&3|(b>>>0<=j>>>0|b>>>0<=e>>>0)){break c}r=e>>>2|0;s=j>>>2|0;while(1){m=i>>>1|0;j=m+o|0;e=j<<1;b=d+(e+s<<2)|0;p=wc(J[b>>2],f);b=wc(J[b+4>>2],f);if(b>>>0>=h>>>0|h-b>>>0<=p>>>0|K[d+(b+p|0)|0]){break c}b=Uc(n,b+d|0);if(!b){b=d+(e+r<<2)|0;e=wc(J[b>>2],f);b=wc(J[b+4>>2],f);if(b>>>0>=h>>>0|e>>>0>=h-b>>>0){break c}k=K[d+(b+e|0)|0]?0:b+d|0;break c}if((i|0)==1){break c}b=(b|0)<0;i=b?m:i-m|0;o=b?o:j;continue}}}k=k?k:n;b=Gb(k);d:{if(b>>>0>=1024){fb(g,k,1023);H[g+1023|0]=0;b=68;break d}fb(g,k,b+1|0);b=0;}if(b){if((b|0)!=28){break b}b=5575;}else {b=g;}break a}za();B();}if(!K[b|0]){J[l>>2]=c;od(g,1024,2507,l);b=g;}J[5732]=q;Kb(a,b);Ua=l+1040|0;return}Kb(a,1694);}function li(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;var e=0,f=0,g=0,h=0,i=0,j=0;f=Ua-32|0;Ua=f;J[f+28>>2]=d;J[f+24>>2]=d;H[f+20|0]=0;J[f+8>>2]=a;J[f+16>>2]=f+28;J[f+12>>2]=f+24;a:{b:{c:{if((b|0)!=(c|0)){while(1){J[d+8>>2]=0;J[d>>2]=0;J[d+4>>2]=0;a=J[b+4>>2];e=J[b>>2];if((a|0)!=(e|0)){e=a-e|0;if((e|0)<0){J[5996]=0;ca(8);a=J[5996];J[5996]=0;if((a|0)==1){break c}B();}J[5996]=0;a=$(3,e|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break b}J[d+4>>2]=a;J[d>>2]=a;J[d+8>>2]=a+e;e=J[b>>2];g=J[b+4>>2];d:{if((e|0)==(g|0)){break d}j=(e^-1)+g|0;h=0;i=g-e&7;if(i){while(1){H[a|0]=K[e|0];a=a+1|0;e=e+1|0;h=h+1|0;if((i|0)!=(h|0)){continue}break}}if(j>>>0<7){break d}while(1){H[a|0]=K[e|0];H[a+1|0]=K[e+1|0];H[a+2|0]=K[e+2|0];H[a+3|0]=K[e+3|0];H[a+4|0]=K[e+4|0];H[a+5|0]=K[e+5|0];H[a+6|0]=K[e+6|0];H[a+7|0]=K[e+7|0];a=a+8|0;e=e+8|0;if((g|0)!=(e|0)){continue}break}}J[d+4>>2]=a;}d=d+12|0;J[f+28>>2]=d;b=b+12|0;if((c|0)!=(b|0)){continue}break}}Ua=f+32|0;return d|0}a=Z()|0;b=J[d>>2];if(!b){break a}J[d+4>>2]=b;$a(b);break a}a=Z()|0;}Ed(f+8|0);da(a|0);B();}function zd(a,b){var c=0,d=0,e=0;c=(a|0)==(b|0);H[b+12|0]=c;a:{if(c){break a}while(1){d=J[b+8>>2];if(K[d+12|0]){break a}b:{c=J[d+8>>2];e=J[c>>2];if((e|0)==(d|0)){e=J[c+4>>2];if(!(!e|K[e+12|0])){break b}c:{if(J[d>>2]==(b|0)){b=d;break c}b=J[d+4>>2];a=J[b>>2];J[d+4>>2]=a;if(a){J[a+8>>2]=d;c=J[d+8>>2];}J[b+8>>2]=c;a=J[d+8>>2];J[((J[a>>2]!=(d|0))<<2)+a>>2]=b;J[b>>2]=d;J[d+8>>2]=b;c=J[b+8>>2];d=J[c>>2];}H[b+12|0]=1;H[c+12|0]=0;a=J[d+4>>2];J[c>>2]=a;if(a){J[a+8>>2]=c;}J[d+8>>2]=J[c+8>>2];a=J[c+8>>2];J[((J[a>>2]!=(c|0))<<2)+a>>2]=d;J[d+4>>2]=c;J[c+8>>2]=d;return}if(!(K[e+12|0]|!e)){break b}d:{if(J[d>>2]!=(b|0)){b=d;break d}a=J[b+4>>2];J[d>>2]=a;if(a){J[a+8>>2]=d;c=J[d+8>>2];}J[b+8>>2]=c;a=J[d+8>>2];J[((J[a>>2]!=(d|0))<<2)+a>>2]=b;J[b+4>>2]=d;J[d+8>>2]=b;c=J[b+8>>2];}H[b+12|0]=1;H[c+12|0]=0;a=J[c+4>>2];b=J[a>>2];J[c+4>>2]=b;if(b){J[b+8>>2]=c;}J[a+8>>2]=J[c+8>>2];b=J[c+8>>2];J[((J[b>>2]!=(c|0))<<2)+b>>2]=a;J[a>>2]=c;J[c+8>>2]=a;break a}H[d+12|0]=1;H[c+12|0]=(a|0)==(c|0);H[e+12|0]=1;b=c;if((a|0)!=(c|0)){continue}break}}}function Tn(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0;J[a+8>>2]=0;J[a>>2]=0;J[a+4>>2]=0;a:{b:{c:{d:{e:{f:{e=J[b+4>>2];d=J[b>>2];if((e|0)!=(d|0)){d=e-d|0;if((d|0)<0){J[5996]=0;ca(85);b=J[5996];J[5996]=0;if((b|0)==1){break f}break a}J[5996]=0;e=$(3,d|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){break f}J[a+4>>2]=e;J[a>>2]=e;J[a+8>>2]=(d&-8)+e;d=J[b>>2];b=J[b+4>>2];if((d|0)!=(b|0)){while(1){O[e>>3]=O[d>>3];e=e+8|0;d=d+8|0;if((b|0)!=(d|0)){continue}break}}J[a+4>>2]=e;}J[a+12>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;b=J[c+4>>2];e=J[c>>2];if((b|0)==(e|0)){break d}b=b-e|0;if((b|0)>=0){break e}J[5996]=0;ca(85);b=J[5996];J[5996]=0;if((b|0)!=1){break a}break c}d=Z()|0;break b}J[5996]=0;e=$(3,b|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break c}J[a+16>>2]=e;J[a+12>>2]=e;J[a+20>>2]=(b&-8)+e;d=J[c>>2];b=J[c+4>>2];if((d|0)!=(b|0)){while(1){O[e>>3]=O[d>>3];e=e+8|0;d=d+8|0;if((b|0)!=(d|0)){continue}break}}J[a+16>>2]=e;}return a|0}d=Z()|0;b=J[a+12>>2];if(!b){break b}J[a+16>>2]=b;$a(b);}b=J[a>>2];if(b){J[a+4>>2]=b;$a(b);}da(d|0);B();}B();}function Tm(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0,i=0,j=0,k=0;d=Ua-32|0;Ua=d;J[a+8>>2]=0;J[a+12>>2]=0;J[a+4>>2]=c;J[a>>2]=b;J[a+16>>2]=0;J[a+20>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;f=a+32|0;J[f>>2]=0;J[f+4>>2]=0;J[a+40>>2]=0;J[5996]=0;aa(86,a+44|0,a|0);b=J[5996];J[5996]=0;a:{if((b|0)==1){break a}J[5996]=0;aa(87,a- -64|0,a|0);b=J[5996];J[5996]=0;if((b|0)==1){break a}b=0;J[5996]=0;c=$(88,a|0)|0;e=J[5996];J[5996]=0;if((e|0)==1){break a}J[a+80>>2]=c;e=J[a+32>>2];g=J[a+36>>2];if((e|0)==(g|0)){c=0;}else {c=e;while(1){b=J[J[c>>2]+84>>2]+b|0;c=c+8|0;if((g|0)!=(c|0)){continue}break}c=J[J[e>>2]+80>>2];}h=J[a+4>>2];i=J[a>>2];j=J[a+8>>2];k=J[a+12>>2];J[5996]=0;b=ka(89,i|0,c|0,b|0,h|0,g-e>>3,(k-j|0)/12|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}J[a+84>>2]=b;J[5996]=0;J[d+28>>2]=a;ea(90,d+12|0,a|0,d+28|0);b=J[5996];J[5996]=0;if((b|0)==1){break a}b=K[d+24|0];if(b){Db(J[d+16>>2]);}H[a+88|0]=b^1;Ua=d+32|0;return a|0}c=Z()|0;lb(f);b=J[a+20>>2];if(b){J[a+24>>2]=b;$a(b);}rb(a+8|0);da(c|0);B();}
function Pn(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0,i=0;J[a+8>>2]=0;J[a>>2]=0;J[a+4>>2]=0;a:{if(c){if(c>>>0>=536870912){J[5996]=0;ca(84);b=J[5996];J[5996]=0;if((b|0)==1){break a}B();}J[5996]=0;e=c<<3;d=$(3,e|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){break a}J[a+4>>2]=d;J[a>>2]=d;J[a+8>>2]=d+e;f=c-1&536870911;g=c&7;b:{if(!g){c=b;break b}c=b;while(1){i=J[c+4>>2];J[d>>2]=J[c>>2];J[d+4>>2]=i;d=d+8|0;c=c+8|0;h=h+1|0;if((h|0)!=(g|0)){continue}break}}if(f>>>0>=7){b=b+e|0;while(1){e=J[c+4>>2];J[d>>2]=J[c>>2];J[d+4>>2]=e;e=J[c+12>>2];J[d+8>>2]=J[c+8>>2];J[d+12>>2]=e;e=J[c+20>>2];J[d+16>>2]=J[c+16>>2];J[d+20>>2]=e;e=J[c+28>>2];J[d+24>>2]=J[c+24>>2];J[d+28>>2]=e;e=J[c+36>>2];J[d+32>>2]=J[c+32>>2];J[d+36>>2]=e;e=J[c+44>>2];J[d+40>>2]=J[c+40>>2];J[d+44>>2]=e;e=J[c+52>>2];J[d+48>>2]=J[c+48>>2];J[d+52>>2]=e;e=J[c+60>>2];J[d+56>>2]=J[c+56>>2];J[d+60>>2]=e;d=d- -64|0;c=c- -64|0;if((b|0)!=(c|0)){continue}break}}J[a+4>>2]=d;}return a|0}c=Z()|0;b=J[a>>2];if(b){J[a+4>>2]=b;$a(b);}da(c|0);B();}function ge(a,b,c,d){var e=0,f=0,g=0,h=0;g=Ua-32|0;Ua=g;e=d&2147483647;h=e;f=e-1006698496|0;e=e-1140785152|0;a:{if((f|0)==(e|0)&0|e>>>0>f>>>0){e=c<<4|b>>>28;c=d<<4|c>>>28;b=b&268435455;if((b|0)==134217728&(a|0)!=0|b>>>0>134217728){f=c+1073741824|0;e=e+1|0;f=e?f:f+1|0;break a}f=c+1073741824|0;if(a|(b|0)!=134217728){break a}a=e&1;e=a+e|0;f=a>>>0>e>>>0?f+1|0:f;break a}if(!(!c&(h|0)==2147418112?!(a|b):h>>>0<2147418112)){a=d<<4|c>>>28;e=c<<4|b>>>28;f=a&524287|2146959360;break a}e=0;f=2146435072;if(h>>>0>1140785151){break a}f=0;h=h>>>16|0;if(h>>>0<15249){break a}e=d&65535|65536;Cb(g+16|0,a,b,c,e,h-15233|0);xc(g,a,b,c,e,15361-h|0);b=J[g+8>>2];e=b<<4;b=J[g+12>>2]<<4|b>>>28;c=J[g>>2];f=J[g+4>>2];h=f;e=f>>>28|e;f=b;a=h&268435455;b=c|(J[g+16>>2]|J[g+24>>2]|(J[g+20>>2]|J[g+28>>2]))!=0;if((a|0)==134217728&(b|0)!=0|a>>>0>134217728){e=e+1|0;f=e?f:f+1|0;break a}if(b|(a|0)!=134217728){break a}a=e;e=e+(e&1)|0;f=a>>>0>e>>>0?f+1|0:f;}Ua=g+32|0;x(0,e|0);x(1,d&-2147483648|f);return +z()}function pn(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0,h=0,i=0,j=0;d=Ua-16|0;Ua=d;J[d+12>>2]=b;if(b>>>0<2147483632){a:{if(((K[a+11|0]&128)>>>7|0?(J[a+8>>2]&2147483647)-1|0:10)>>>0>=b>>>0){break a}b=K[a+11|0];J[d+8>>2]=(b&128)>>>7|0?J[a+4>>2]:b&127;c=J[(M[d+12>>2]<M[d+8>>2]?d+8|0:d+12|0)>>2];if(c>>>0>=11){c=c+16&-16;e=c-1|0;c=(e|0)==11?c:e;}else {c=10;}if((c|0)==(((K[a+11|0]&128)>>>7|0?(J[a+8>>2]&2147483647)-1|0:10)|0)){break a}f=Ua-16|0;Ua=f;b=K[a+11|0];e=(b&128)>>>7|0?J[a+4>>2]:b&127;b:{c:{i=c>>>0<11;d:{if(i){j=1;g=c+1|0;h=J[a>>2];c=e;b=a;break d}g=c+1|0;e:{if(c>>>0>((K[a+11|0]&128)>>>7|0?(J[a+8>>2]&2147483647)-1|0:10)>>>0){Gc(f+8|0,g);break e}J[5996]=0;aa(289,f+8|0,g|0);b=J[5996];J[5996]=0;if((b|0)==1){break c}}g=J[f+12>>2];c=K[a+11|0];b=(c&128)>>>7|0;j=b;h=b?J[a>>2]:a;c=b?J[a+4>>2]:c&127;b=J[f+8>>2];}b=gc(b,h,c+1|0);if(j){$a(h);}if(!i){J[a+8>>2]=g|-2147483648;J[a+4>>2]=e;J[a>>2]=b;break b}H[a+11|0]=e;break b}pa(fa(0)|0)|0;ra();}Ua=f+16|0;}Ua=d+16|0;return}ub();B();}function Cd(a){var b=0,c=0,d=0,e=0,f=0;d=J[5726];c=a+7&-8;b=d+c|0;a:{b:{if(b>>>0<=d>>>0?c:0){break b}if(b>>>0>Ya()<<16>>>0){if(!(Qa(b|0)|0)){break b}}J[5726]=b;break a}J[5732]=48;d=-1;}if((d|0)!=-1){a=a+d|0;J[a-4>>2]=16;c=a-16|0;J[c>>2]=16;b=J[5992];if(b){f=J[b+8>>2];}else {f=0;}c:{d:{if((f|0)==(d|0)){e=d-(J[d-4>>2]&-2)|0;f=J[e-4>>2];J[b+8>>2]=a;a=e-(f&-2)|0;b=-16;if(!(H[(a+J[a>>2]|0)-4|0]&1)){break d}b=J[a+4>>2];J[b+8>>2]=J[a+8>>2];J[J[a+8>>2]+4>>2]=b;b=c-a|0;J[a>>2]=b;break c}J[d+12>>2]=16;J[d>>2]=16;J[d+8>>2]=a;J[d+4>>2]=b;J[5992]=d;b=16;}a=b+d|0;b=c-a|0;J[a>>2]=b;}J[((b&-4)+a|0)-4>>2]=b|1;c=J[a>>2]-8|0;e:{if(c>>>0<=127){b=(c>>>3|0)-1|0;break e}e=S(c);b=((c>>>29-e^4)-(e<<2)|0)+110|0;if(c>>>0<=4095){break e}b=((c>>>30-e^2)-(e<<1)|0)+71|0;b=b>>>0>=63?63:b;}c=b<<4;J[a+4>>2]=c+22944;c=c+22952|0;J[a+8>>2]=J[c>>2];J[c>>2]=a;J[J[a+8>>2]+4>>2]=a;c=J[5994];e=J[5995];a=b&31;if((b&63)>>>0>=32){b=1<<a;f=0;}else {f=1<<a;b=f-1&1>>>32-a;}J[5994]=f|c;J[5995]=b|e;}return (d|0)!=-1}function Te(a,b,c,d,e,f,g,h){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;a=Ua-16|0;Ua=a;J[a+12>>2]=c;J[a+8>>2]=f;J[a+12>>2]=c;J[a+8>>2]=f;b=J[a+12>>2];a:{while(1){if(b>>>0>=d>>>0){c=0;break a}c=2;b=J[b>>2];if(b>>>0>1114111|(b&-2048)==55296){break a}b:{c:{if(b>>>0<=127){c=1;f=J[a+8>>2];if((g-f|0)<=0){break a}J[a+8>>2]=f+1;H[f|0]=b;break c}if(b>>>0<=2047){c=J[a+8>>2];if((g-c|0)<2){break b}J[a+8>>2]=c+1;H[c|0]=b>>>6|192;c=J[a+8>>2];J[a+8>>2]=c+1;H[c|0]=b&63|128;break c}c=J[a+8>>2];f=g-c|0;if(b>>>0<=65535){if((f|0)<3){break b}J[a+8>>2]=c+1;H[c|0]=b>>>12|224;c=J[a+8>>2];J[a+8>>2]=c+1;H[c|0]=b>>>6&63|128;c=J[a+8>>2];J[a+8>>2]=c+1;H[c|0]=b&63|128;break c}if((f|0)<4){break b}J[a+8>>2]=c+1;H[c|0]=b>>>18|240;c=J[a+8>>2];J[a+8>>2]=c+1;H[c|0]=b>>>12&63|128;c=J[a+8>>2];J[a+8>>2]=c+1;H[c|0]=b>>>6&63|128;c=J[a+8>>2];J[a+8>>2]=c+1;H[c|0]=b&63|128;}b=J[a+12>>2]+4|0;J[a+12>>2]=b;continue}break}c=1;}J[e>>2]=J[a+12>>2];J[h>>2]=J[a+8>>2];Ua=a+16|0;return c|0}function Yg(a){var b=0,c=0,d=0,e=0,f=0;J[a+8>>2]=0;J[a>>2]=0;J[a+4>>2]=0;a:{b:{c:{d:{e:{f:{c=J[6141];d=J[6142];if((c|0)!=(d|0)){e=d-c|0;if((e|0)<0){J[5996]=0;ca(85);b=J[5996];J[5996]=0;if((b|0)==1){break f}break a}J[5996]=0;b=$(3,e|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){break f}J[a+4>>2]=b;J[a>>2]=b;J[a+8>>2]=(e&-8)+b;while(1){O[b>>3]=O[c>>3];b=b+8|0;c=c+8|0;if((d|0)!=(c|0)){continue}break}J[a+4>>2]=b;}J[a+12>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;c=J[6144];d=J[6145];if((c|0)==(d|0)){break d}e=d-c|0;if((e|0)>=0){break e}J[5996]=0;ca(85);b=J[5996];J[5996]=0;if((b|0)!=1){break a}break c}c=Z()|0;break b}J[5996]=0;b=$(3,e|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){break c}J[a+16>>2]=b;J[a+12>>2]=b;J[a+20>>2]=(e&-8)+b;while(1){O[b>>3]=O[c>>3];b=b+8|0;c=c+8|0;if((d|0)!=(c|0)){continue}break}J[a+16>>2]=b;}return a}c=Z()|0;b=J[a+12>>2];if(!b){break b}J[a+16>>2]=b;$a(b);}b=J[a>>2];if(b){J[a+4>>2]=b;$a(b);}da(c|0);B();}B();}function Oe(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;var f=0,g=0,h=0,i=0;b=0;a=c;while(1){a:{if(a>>>0>=d>>>0|b>>>0>=e>>>0){break a}g=1;f=H[a|0];b:{if((f|0)>=0){break b}if(f>>>0<4294967234){break a}if(f>>>0<=4294967263){if((d-a|0)<2){break a}g=2;if((K[a+1|0]&192)!=128){break a}break b}g=f&255;if(f>>>0<=4294967279){if((d-a|0)<3){break a}h=K[a+2|0];f=K[a+1|0];c:{d:{if((g|0)!=237){if((g|0)!=224){break d}if((f&224)==160){break c}break a}if((f&224)!=128){break a}break c}if((f&192)!=128){break a}}g=3;if((h&192)!=128){break a}break b}if((d-a|0)<4|f>>>0>4294967284|e-b>>>0<2){break a}h=K[a+3|0];i=K[a+2|0];f=K[a+1|0];e:{f:{switch(g-240|0){case 0:if((f+112&255)>>>0>=48){break a}break e;case 4:if((f&240)!=128){break a}break e;default:break f}}if((f&192)!=128){break a}}if((i&192)!=128|(h&192)!=128|(g<<18&1835008|(f&48)<<12)>>>0>1114111){break a}b=b+1|0;g=4;}b=b+1|0;a=a+g|0;continue}break}return a-c|0}function gh(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0,h=0;c=Ua-32|0;Ua=c;J[a+8>>2]=0;J[a>>2]=0;J[a+4>>2]=0;H[c+4|0]=0;J[c>>2]=a;e=J[b+4>>2];f=J[b>>2];g=e-f|0;d=(g|0)/12|0;a:{b:{c:{if((e|0)!=(f|0)){if(d>>>0>=357913942){J[5996]=0;ca(41);a=J[5996];J[5996]=0;if((a|0)==1){break c}B();}J[5996]=0;e=$(3,g|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){break c}J[a+4>>2]=e;J[a>>2]=e;J[a+8>>2]=e+P(d,12);f=J[b+4>>2];b=J[b>>2];J[c+28>>2]=e;J[c+24>>2]=e;H[c+20|0]=0;J[c+8>>2]=a+8;J[c+16>>2]=c+28;J[c+12>>2]=c+24;d:{if((b|0)==(f|0)){d=e;break d}d=e;while(1){e:{if(H[b+11|0]>=0){g=J[b+4>>2];J[d>>2]=J[b>>2];J[d+4>>2]=g;J[d+8>>2]=J[b+8>>2];break e}g=J[b+4>>2];h=J[b>>2];J[5996]=0;ea(40,d|0,h|0,g|0);d=J[5996];J[5996]=0;if((d|0)==1){break b}d=J[c+28>>2];}d=d+12|0;J[c+28>>2]=d;b=b+12|0;if((f|0)!=(b|0)){continue}break}}J[a+4>>2]=d;}Ua=c+32|0;return a|0}b=Z()|0;break a}b=Z()|0;Wg(c+8|0);J[a+4>>2]=e;}Vg(c);da(b|0);B();}function Re(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;var f=0,g=0,h=0,i=0;a=c;while(1){a:{if(e>>>0<=i>>>0|a>>>0>=d>>>0){break a}f=1;b=H[a|0];b:{if((b|0)>=0){break b}if(b>>>0<4294967234){break a}if(b>>>0<=4294967263){if((d-a|0)<2){break a}f=2;if((K[a+1|0]&192)!=128){break a}break b}g=b&255;if(b>>>0<=4294967279){if((d-a|0)<3){break a}h=K[a+2|0];b=K[a+1|0];c:{d:{if((g|0)!=237){if((g|0)!=224){break d}if((b&224)==160){break c}break a}if((b&224)!=128){break a}break c}if((b&192)!=128){break a}}f=3;if((h&192)!=128){break a}break b}if((d-a|0)<4|b>>>0>4294967284){break a}f=K[a+3|0];h=K[a+2|0];b=K[a+1|0];e:{f:{switch(g-240|0){case 0:if((b+112&255)>>>0>=48){break a}break e;case 4:if((b&240)!=128){break a}break e;default:break f}}if((b&192)!=128){break a}}if((h&192)!=128|(f&192)!=128){break a}f=4;if((g<<18&1835008|(b&48)<<12)>>>0>1114111){break a}}i=i+1|0;a=a+f|0;continue}break}return a-c|0}function zk(a,b,c,d,e,f,g,h,i,j){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;i=i|0;j=j|0;var k=0,l=0;k=Ua-16|0;Ua=k;l=j;a:{if(a){j=k+4|0;a=tf(b);Xa[J[J[a>>2]+44>>2]](j,a);b=J[k+4>>2];H[c|0]=b;H[c+1|0]=b>>>8;H[c+2|0]=b>>>16;H[c+3|0]=b>>>24;Xa[J[J[a>>2]+32>>2]](j,a);Ab(i,j);ab(j);Xa[J[J[a>>2]+28>>2]](j,a);Ab(h,j);ab(j);H[d|0]=Xa[J[J[a>>2]+12>>2]](a);H[e|0]=Xa[J[J[a>>2]+16>>2]](a);Xa[J[J[a>>2]+20>>2]](j,a);Ab(f,j);ab(j);Xa[J[J[a>>2]+24>>2]](j,a);Ab(g,j);ab(k+4|0);a=Xa[J[J[a>>2]+36>>2]](a)|0;break a}j=k+4|0;a=sf(b);Xa[J[J[a>>2]+44>>2]](j,a);b=J[k+4>>2];H[c|0]=b;H[c+1|0]=b>>>8;H[c+2|0]=b>>>16;H[c+3|0]=b>>>24;Xa[J[J[a>>2]+32>>2]](j,a);Ab(i,j);ab(j);Xa[J[J[a>>2]+28>>2]](j,a);Ab(h,j);ab(j);H[d|0]=Xa[J[J[a>>2]+12>>2]](a);H[e|0]=Xa[J[J[a>>2]+16>>2]](a);Xa[J[J[a>>2]+20>>2]](j,a);Ab(f,j);ab(j);Xa[J[J[a>>2]+24>>2]](j,a);Ab(g,j);ab(k+4|0);a=Xa[J[J[a>>2]+36>>2]](a)|0;}J[l>>2]=a;Ua=k+16|0;}function sk(a,b,c,d,e,f,g,h,i,j){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;h=h|0;i=i|0;j=j|0;var k=0,l=0;k=Ua-16|0;Ua=k;l=j;a:{if(a){j=k+4|0;a=qf(b);Xa[J[J[a>>2]+44>>2]](j,a);b=J[k+4>>2];H[c|0]=b;H[c+1|0]=b>>>8;H[c+2|0]=b>>>16;H[c+3|0]=b>>>24;Xa[J[J[a>>2]+32>>2]](j,a);ac(i,j);ab(j);Xa[J[J[a>>2]+28>>2]](j,a);ac(h,j);ab(j);J[d>>2]=Xa[J[J[a>>2]+12>>2]](a);J[e>>2]=Xa[J[J[a>>2]+16>>2]](a);Xa[J[J[a>>2]+20>>2]](j,a);Ab(f,j);ab(j);Xa[J[J[a>>2]+24>>2]](j,a);ac(g,j);ab(j);a=Xa[J[J[a>>2]+36>>2]](a)|0;break a}j=k+4|0;a=pf(b);Xa[J[J[a>>2]+44>>2]](j,a);b=J[k+4>>2];H[c|0]=b;H[c+1|0]=b>>>8;H[c+2|0]=b>>>16;H[c+3|0]=b>>>24;Xa[J[J[a>>2]+32>>2]](j,a);ac(i,j);ab(j);Xa[J[J[a>>2]+28>>2]](j,a);ac(h,j);ab(j);J[d>>2]=Xa[J[J[a>>2]+12>>2]](a);J[e>>2]=Xa[J[J[a>>2]+16>>2]](a);Xa[J[J[a>>2]+20>>2]](j,a);Ab(f,j);ab(j);Xa[J[J[a>>2]+24>>2]](j,a);ac(g,j);ab(j);a=Xa[J[J[a>>2]+36>>2]](a)|0;}J[l>>2]=a;Ua=k+16|0;}function gb(a){var b=0,c=0,d=0,e=0,f=0,g=0,h=0,i=0,j=0;b=J[a+112>>2];d=J[a+116>>2];g=(b|d)!=0;h=b;c=J[a+4>>2];f=J[a+44>>2];b=c-f|0;i=b;e=b+J[a+120>>2]|0;b=J[a+124>>2]+(b>>31)|0;a:{b=e>>>0<i>>>0?b+1|0:b;if(!(((b|0)>=(d|0)&e>>>0>=h>>>0|(b|0)>(d|0))&g)){d=Ua-16|0;Ua=d;c=-1;b:{if(Dg(a)){break b}if((Xa[J[a+32>>2]](a,d+15|0,1)|0)!=1){break b}c=K[d+15|0];}Ua=d+16|0;d=c;if((c|0)>=0){break a}f=J[a+44>>2];c=J[a+4>>2];}J[a+112>>2]=-1;J[a+116>>2]=-1;J[a+104>>2]=c;d=f-c|0;c=d+e|0;b=(d>>31)+b|0;J[a+120>>2]=c;J[a+124>>2]=c>>>0<d>>>0?b+1|0:b;return -1}c=e+1|0;b=c?b:b+1|0;h=J[a+4>>2];f=J[a+8>>2];e=J[a+116>>2];i=e;g=J[a+112>>2];c:{if(!(e|g)){break c}e=g-c|0;g=i-(b+(c>>>0>g>>>0)|0)|0;j=f-h|0;i=j>>31;if((g|0)>=(i|0)&e>>>0>=j>>>0|(g|0)>(i|0)){break c}f=e+h|0;}J[a+104>>2]=f;e=J[a+44>>2];f=e-h|0;c=f+c|0;b=(f>>31)+b|0;J[a+120>>2]=c;J[a+124>>2]=c>>>0<f>>>0?b+1|0:b;if(e>>>0>=h>>>0){H[h-1|0]=d;}return d}function $l(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0,h=0,i=0,j=0,k=0;g=Ua-16|0;Ua=g;a:{b:{if((b|0)!=-1){i=J[a+8>>2];j=J[a+12>>2];d=J[a+24>>2];if((d|0)==J[a+28>>2]){e=-1;if(!(K[a+48|0]&16)){break a}k=J[a+44>>2];h=J[a+20>>2];J[5996]=0;c=a+32|0;aa(98,c|0,0);f=J[5996];J[5996]=0;if((f|0)==1){break b}f=(K[c+11|0]&128)>>>7|0?(J[c+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,c|0,f|0);f=J[5996];J[5996]=0;if((f|0)==1){break b}c=(K[c+11|0]&128)>>>7|0?J[c>>2]:c;e=K[a+43|0];J[a+28>>2]=c+((e&128)>>>7|0?J[a+36>>2]:e&127);J[a+20>>2]=c;J[a+24>>2]=c;J[a+24>>2]=J[a+24>>2]+(d-h|0);J[a+44>>2]=J[a+20>>2]+(k-h|0);d=J[a+24>>2];}J[g+12>>2]=d+1;d=a+44|0;c=g+12|0;c=J[(M[c>>2]<M[d>>2]?d:c)>>2];J[a+44>>2]=c;if(K[a+48|0]&8){d=a+32|0;d=(K[d+11|0]&128)>>>7|0?J[d>>2]:d;J[a+16>>2]=c;J[a+12>>2]=d+(j-i|0);J[a+8>>2]=d;}e=vg(a,b<<24>>24);break a}e=(b|0)!=-1?b:0;break a}pa(fa(0)|0)|0;ra();}Ua=g+16|0;return e|0}function Vf(a,b,c,d){var e=0,f=0,g=0,h=0;g=Ua-32|0;Ua=g;f=d&2147483647;e=f-1065418752|0;h=f-1082064896|0;a:{if((e|0)==(h|0)&0|e>>>0<h>>>0){f=(d&33554431)<<7|c>>>25;e=0;h=e;c=c&33554431;if(!(!e&(c|0)==16777216?!(a|b):!e&c>>>0<16777216)){e=f+1073741825|0;break a}e=f+1073741824|0;if(c^16777216|a|(b|h)){break a}e=(f&1)+e|0;break a}if(!(!c&(f|0)==2147418112?!(a|b):f>>>0<2147418112)){e=((d&33554431)<<7|c>>>25)&4194303|2143289344;break a}e=2139095040;if(f>>>0>1082064895){break a}e=0;f=f>>>16|0;if(f>>>0<16145){break a}e=d&65535|65536;Cb(g+16|0,a,b,c,e,f-16129|0);xc(g,a,b,c,e,16257-f|0);a=J[g+8>>2];e=(J[g+12>>2]&33554431)<<7|a>>>25;c=J[g>>2]|(J[g+16>>2]|J[g+24>>2]|(J[g+20>>2]|J[g+28>>2]))!=0;f=J[g+4>>2];b=0;a=a&33554431;if(!(!b&(a|0)==16777216?!(c|f):!b&a>>>0<16777216)){e=e+1|0;break a}if(a^16777216|c|(b|f)){break a}e=(e&1)+e|0;}Ua=g+32|0;return x(2,d&-2147483648|e),C()}function eo(a,b,c){var d=0,e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0;a:{b:{c:{d:{e:{f:{g:{h:{i:{j:{if(b){if(!c){break j}break i}Wa=0;a=(a>>>0)/(c>>>0)|0;break a}if(!a){break h}break g}if(!(c-1&c)){break f}f=(S(c)+33|0)-S(b)|0;g=0-f|0;break d}Wa=0;a=(b>>>0)/0|0;break a}d=32-S(b)|0;if(d>>>0<31){break e}break c}if((c|0)==1){break b}f=c?31-S(c-1^c)|0:32;c=f&31;if((f&63)>>>0>=32){a=b>>>c|0;}else {d=b>>>c|0;a=((1<<c)-1&b)<<32-c|a>>>c;}Wa=d;break a}f=d+1|0;g=63-d|0;}d=f&63;e=d&31;if(d>>>0>=32){d=0;h=b>>>e|0;}else {d=b>>>e|0;h=((1<<e)-1&b)<<32-e|a>>>e;}g=g&63;e=g&31;if(g>>>0>=32){b=a<<e;a=0;}else {b=(1<<e)-1&a>>>32-e|b<<e;a=a<<e;}if(f){g=c-1|0;l=(g|0)==-1?-1:0;while(1){i=d<<1|h>>>31;d=h<<1|b>>>31;e=l-(i+(d>>>0>g>>>0)|0)>>31;j=c&e;h=d-j|0;d=i-(d>>>0<j>>>0)|0;b=b<<1|a>>>31;a=k|a<<1;k=e&1;f=f-1|0;if(f){continue}break}}Wa=b<<1|a>>>31;a=k|a<<1;break a}a=0;b=0;}Wa=b;}return a}function _l(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0;i=J[b+44>>2];j=J[b+24>>2];if(i>>>0<j>>>0){J[b+44>>2]=j;i=j;}m=-1;n=-1;o=f&24;a:{if(!o|(e|0)==1&(o|0)==24){break a}if(i){g=b+32|0;g=i-((K[g+11|0]&128)>>>7|0?J[g>>2]:g)|0;k=g>>31;}b:{c:{switch(e|0){case 1:if(f&8){h=J[b+12>>2]-J[b+8>>2]|0;l=h>>31;break b}h=j-J[b+20>>2]|0;l=h>>31;break b;case 0:break b;case 2:break c;default:break a}}h=g;l=k;}d=d+l|0;c=c+h|0;d=h>>>0>c>>>0?d+1|0:d;if((d|0)<0|(c>>>0>g>>>0&(d|0)>=(k|0)|(d|0)>(k|0))){break a}e=f&8;d:{if(!(c|d)){break d}if(J[b+12>>2]?0:e){break a}if(!(f&16)){break d}if(!j){break a}}if(e){e=J[b+8>>2];J[b+16>>2]=i;J[b+12>>2]=c+e;J[b+8>>2]=e;}if(f&16){e=J[b+20>>2];J[b+28>>2]=J[b+28>>2];J[b+20>>2]=e;J[b+24>>2]=e;J[b+24>>2]=c+J[b+24>>2];}m=c;n=d;}J[a+8>>2]=m;J[a+12>>2]=n;J[a>>2]=0;J[a+4>>2]=0;}function fe(a){var b=0,c=0,d=0,e=0,f=0,g=0,h=0,i=0,j=0;b=a;a:{if(b&3){while(1){c=K[b|0];if(!c|(c|0)==61){break a}b=b+1|0;if(b&3){continue}break}}c=J[b>>2];b:{if((c^-1)&c-16843009&-2139062144){break b}while(1){if((c^1027423549)-16843009&(c^-1)&-2139062144){break b}c=J[b+4>>2];b=b+4|0;if(!(c-16843009&(c^-1)&-2139062144)){continue}break}}while(1){c=b;d=K[b|0];if((d|0)!=61){b=b+1|0;if(d){continue}}break}b=c;}if((a|0)==(b|0)){return 0}f=b-a|0;c:{if(K[f+a|0]){break c}e=J[6181];if(!e){break c}b=J[e>>2];if(!b){break c}while(1){d:{g=a;c=b;h=f;d=0;e:{if(!f){break e}d=K[a|0];if(d){f:{while(1){i=K[c|0];if((i|0)!=(d|0)|!i){break f}h=h-1|0;if(!h){break f}c=c+1|0;d=K[g+1|0];g=g+1|0;if(d){continue}break}d=0;}}else {d=0;}d=d-K[c|0]|0;}if(!d){b=b+f|0;if(K[b|0]==61){break d}}b=J[e+4>>2];e=e+4|0;if(b){continue}break c}break}j=b+1|0;}return j}function Ob(a,b){a=a|0;b=b|0;var c=0,d=0;d=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);a:{b:{c:{if(b>>>0<=28){if(d>>>0<29){break c}b=K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24);fb(a+4|0,b,d-29|0);$a(b);b=(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))-29|0;H[a|0]=b;H[a+1|0]=b>>>8;H[a+2|0]=b>>>16;H[a+3|0]=b>>>24;return}if(d>>>0>=29){c=Vc(K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24),b);H[a+8|0]=c;H[a+9|0]=c>>>8;H[a+10|0]=c>>>16;H[a+11|0]=c>>>24;if(!c){break b}H[a+4|0]=b;H[a+5|0]=b>>>8;H[a+6|0]=b>>>16;H[a+7|0]=b>>>24;return}c=sb(b);if(!c){break a}c=fb(c,a+4|0,d);H[a+4|0]=b;H[a+5|0]=b>>>8;H[a+6|0]=b>>>16;H[a+7|0]=b>>>24;H[a+8|0]=c;H[a+9|0]=c>>>8;H[a+10|0]=c>>>16;H[a+11|0]=c>>>24;b=d+29|0;H[a|0]=b;H[a+1|0]=b>>>8;H[a+2|0]=b>>>16;H[a+3|0]=b>>>24;}return}la(1370,2249,181,1040);B();}la(1357,2249,185,1040);B();}function pd(a,b,c,d){var e=0,f=0,g=0;e=Ua-160|0;Ua=e;f=b?a:e+158|0;J[e+148>>2]=f;a=-1;g=b-1|0;J[e+152>>2]=b>>>0>=g>>>0?g:0;e=Sb(e,0,144);J[e+76>>2]=-1;J[e+36>>2]=179;J[e+80>>2]=-1;J[e+44>>2]=e+159;J[e+84>>2]=e+148;a:{if((b|0)<0){J[5732]=61;break a}H[f|0]=0;f=0;b=Ua-208|0;Ua=b;J[b+204>>2]=d;a=b+160|0;Sb(a,0,40);J[b+200>>2]=J[b+204>>2];b:{if((Kg(0,c,b+200|0,b+80|0,a)|0)<0){a=-1;break b}d=J[e+76>>2]>=0;g=J[e>>2];J[e>>2]=g&-33;c:{d:{e:{if(!J[e+48>>2]){J[e+48>>2]=80;J[e+28>>2]=0;J[e+16>>2]=0;J[e+20>>2]=0;f=J[e+44>>2];J[e+44>>2]=b;break e}if(J[e+16>>2]){break d}}a=-1;if(Be(e)){break c}}a=Kg(e,c,b+200|0,b+80|0,b+160|0);}c=a;if(f){Xa[J[e+36>>2]](e,0,0)|0;J[e+48>>2]=0;J[e+44>>2]=f;J[e+28>>2]=0;a=J[e+20>>2];J[e+16>>2]=0;J[e+20>>2]=0;c=a?c:-1;}a=J[e>>2];J[e>>2]=a|g&32;a=a&32?-1:c;if(!d){break b}}Ua=b+208|0;}Ua=e+160|0;return a}function Ck(a,b,c,d,e,f,g){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;var h=0,i=0;h=Ua-416|0;Ua=h;d=h+416|0;J[h+12>>2]=d;c=Ua-144|0;Ua=c;J[c+28>>2]=c+132;i=c+32|0;yf(J[a+8>>2],i,c+28|0,e,f,g);J[c+16>>2]=0;J[c+20>>2]=0;J[c+12>>2]=i;e=h+16|0;g=J[h+12>>2]-e>>2;f=Ua-16|0;Ua=f;a=Xb(f+12|0,J[a+8>>2]);g=bg(e,c+12|0,g,c+16|0);a=J[a>>2];if(a){J[6178]=(a|0)==-1?24816:a;}Ua=f+16|0;if((g|0)==-1){xf(2472);B();}J[h+12>>2]=e+(g<<2);Ua=c+144|0;c=Ua-16|0;Ua=c;a=Ua-32|0;Ua=a;f=Ua-16|0;Ua=f;J[f+12>>2]=J[h+12>>2];J[a+24>>2]=e;J[a+28>>2]=J[f+12>>2];Ua=f+16|0;f=J[a+24>>2];g=J[a+28>>2];e=Ua-16|0;Ua=e;J[e+12>>2]=b;while(1){if((f|0)!=(g|0)){pg(e+12|0,J[f>>2]);f=f+4|0;continue}break}J[a+16>>2]=g;J[a+20>>2]=J[e+12>>2];Ua=e+16|0;b=J[a+16>>2];J[a+12>>2]=J[a+20>>2];J[c+8>>2]=b;J[c+12>>2]=J[a+12>>2];Ua=a+32|0;Ua=c+16|0;Ua=d;return J[c+12>>2]}function Kc(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0,h=0,i=0;e=Ua-32|0;Ua=e;h=e+32|0;c=h;d=e+21|0;f=c-d|0;a:{if((f|0)<=9){i=f;f=P(32-S(b|1)|0,1233)>>>12|0;g=61;if((i|0)<((M[(f<<2)+21616>>2]<=b>>>0)+f|0)){break a}}b:{if(b>>>0<=999999){if(b>>>0<=9999){if(b>>>0<=99){if(b>>>0<=9){H[d|0]=b+48;c=d+1|0;break b}c=uc(d,b);break b}if(b>>>0<=999){c=(b>>>0)/100|0;H[d|0]=c+48;c=uc(d+1|0,P(c,-100)+b|0);break b}c=Kd(d,b);break b}if(b>>>0<=99999){c=(b>>>0)/1e4|0;H[d|0]=c+48;c=Kd(d+1|0,P(c,-1e4)+b|0);break b}c=Jd(d,b);break b}if(b>>>0<=99999999){if(b>>>0<=9999999){c=(b>>>0)/1e6|0;H[d|0]=c+48;c=Jd(d+1|0,P(c,-1e6)+b|0);break b}c=Id(d,b);break b}if(b>>>0<=999999999){c=(b>>>0)/1e8|0;H[d|0]=c+48;c=Id(d+1|0,P(c,-1e8)+b|0);break b}c=(b>>>0)/1e8|0;c=Id(uc(d,c),P(c,-1e8)+b|0);}g=0;}J[e+16>>2]=g;J[e+12>>2]=c;td(a,d,J[e+12>>2]);Ua=h;}function ye(a,b,c){var d=0,e=0,f=0,g=0;d=a+4|0;e=(d+b|0)-1&0-b;b=J[a>>2];if(e+c>>>0<=(b+a|0)-4>>>0){f=J[a+4>>2];J[f+8>>2]=J[a+8>>2];J[J[a+8>>2]+4>>2]=f;if((d|0)!=(e|0)){e=e-d|0;f=a-(J[a-4>>2]&-2)|0;d=e+J[f>>2]|0;J[f>>2]=d;J[(f+(d&-4)|0)-4>>2]=d;a=a+e|0;b=b-e|0;J[a>>2]=b;}a:{if(c+24>>>0<=b>>>0){d=(a+c|0)+8|0;b=(b-c|0)-8|0;J[d>>2]=b;J[(d+(b&-4)|0)-4>>2]=b|1;e=J[d>>2]-8|0;b:{if(e>>>0<=127){b=(e>>>3|0)-1|0;break b}f=S(e);b=((e>>>29-f^4)-(f<<2)|0)+110|0;if(e>>>0<=4095){break b}b=((e>>>30-f^2)-(f<<1)|0)+71|0;b=b>>>0>=63?63:b;}e=b<<4;J[d+4>>2]=e+22944;e=e+22952|0;J[d+8>>2]=J[e>>2];J[e>>2]=d;J[J[d+8>>2]+4>>2]=d;e=J[5994];f=J[5995];d=b&31;if((b&63)>>>0>=32){b=1<<d;d=0;}else {g=1<<d;b=g-1&1>>>32-d;d=g;}J[5994]=d|e;J[5995]=b|f;b=c+8|0;J[a>>2]=b;J[((b&-4)+a|0)-4>>2]=b;break a}J[(a+b|0)-4>>2]=b;}a=a+4|0;}else {a=0;}return a}function $m(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0;a:{b:{c:{d=J[a>>2];f=(J[a+4>>2]-d|0)/12|0;c=f+1|0;if(c>>>0<357913942){d=(J[a+8>>2]-d|0)/12|0;g=d<<1;d=d>>>0>=178956970?357913941:c>>>0<g>>>0?g:c;if(d){if(d>>>0>=357913942){break c}e=cb(P(d,12));}c=P(f,12)+e|0;f=J[b+4>>2];J[c>>2]=J[b>>2];J[c+4>>2]=f;J[c+8>>2]=J[b+8>>2];J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;e=P(d,12)+e|0;d=c+12|0;b=J[a+4>>2];f=J[a>>2];if((b|0)==(f|0)){break b}while(1){b=b-12|0;g=J[b+4>>2];c=c-12|0;J[c>>2]=J[b>>2];J[c+4>>2]=g;J[c+8>>2]=J[b+8>>2];J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;if((b|0)!=(f|0)){continue}break}J[a+8>>2]=e;e=J[a+4>>2];J[a+4>>2]=d;b=J[a>>2];J[a>>2]=c;if((b|0)==(e|0)){break a}while(1){a=e-12|0;if(H[e-1|0]<0){$a(J[a>>2]);}e=a;if((a|0)!=(b|0)){continue}break}break a}nb();B();}Pb();B();}J[a+8>>2]=e;J[a+4>>2]=d;J[a>>2]=c;}if(b){$a(b);}}function Lf(a,b,c,d){var e=0,f=0,g=0,h=0,i=0,j=0,k=0,l=0,m=0,n=0,o=0;i=Ua-32|0;Ua=i;a:{b:{c:{if((b|0)!=(c|0)){k=J[5732];J[5732]=0;g=Ua-16|0;Ua=g;jb();f=Ua-16|0;Ua=f;e=Ua-16|0;Ua=e;ee(e,b,i+28|0,2);h=J[e>>2];j=J[e+4>>2];b=f;f=J[e+12>>2];J[b+8>>2]=J[e+8>>2];J[b+12>>2]=f;J[b>>2]=h;J[b+4>>2]=j;Ua=e+16|0;f=J[b>>2];h=J[b+4>>2];e=g;g=J[b+12>>2];J[e+8>>2]=J[b+8>>2];J[e+12>>2]=g;J[e>>2]=f;J[e+4>>2]=h;Ua=b+16|0;g=J[e>>2];f=J[e+4>>2];h=J[e+12>>2];b=i;J[b+16>>2]=J[e+8>>2];J[b+20>>2]=h;J[b+8>>2]=g;J[b+12>>2]=f;Ua=e+16|0;e=J[b+16>>2];f=J[b+20>>2];g=J[b+8>>2];h=J[b+12>>2];j=J[5732];if(!j){break c}if(J[b+28>>2]!=(c|0)){break b}l=g;m=h;n=e;o=f;if((j|0)!=68){break a}break b}J[d>>2]=4;break a}J[5732]=k;if(J[i+28>>2]==(c|0)){break a}}J[d>>2]=4;g=l;h=m;e=n;f=o;}J[a>>2]=g;J[a+4>>2]=h;J[a+8>>2]=e;J[a+12>>2]=f;Ua=i+32|0;}function Gm(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;var e=0,f=0,g=0,h=0,i=0,j=0;f=J[a+4>>2];e=J[a+8>>2];if(f>>>0<e>>>0){J[f>>2]=b;J[f+8>>2]=c;J[f+12>>2]=c>>31;b=J[d+4>>2];J[f+16>>2]=J[d>>2];J[f+20>>2]=b;J[a+4>>2]=f+24;return}a:{i=J[a>>2];g=(f-i|0)/24|0;h=g+1|0;if(h>>>0<178956971){j=P(g,24);e=(e-i|0)/24|0;g=e<<1;g=e>>>0>=89478485?178956970:g>>>0>h>>>0?g:h;if(g){if(g>>>0>=178956971){break a}h=cb(P(g,24));}else {h=0;}e=j+h|0;J[e+8>>2]=c;J[e+12>>2]=c>>31;J[e>>2]=b;b=J[d+4>>2];J[e+16>>2]=J[d>>2];J[e+20>>2]=b;b=e+24|0;if((f|0)!=(i|0)){while(1){f=f-24|0;d=J[f+4>>2];e=e-24|0;J[e>>2]=J[f>>2];J[e+4>>2]=d;c=J[f+20>>2];J[e+16>>2]=J[f+16>>2];J[e+20>>2]=c;c=J[f+12>>2];J[e+8>>2]=J[f+8>>2];J[e+12>>2]=c;if((f|0)!=(i|0)){continue}break}f=J[a>>2];}J[a+8>>2]=h+P(g,24);J[a+4>>2]=b;J[a>>2]=e;if(f){$a(f);}return}nb();B();}Pb();B();}function _k(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;var f=0;f=Ua-32|0;Ua=f;J[f+28>>2]=b;a:{if(!(H[c+4|0]&1)){c=Xa[J[J[a>>2]+24>>2]](a,b,c,d,e)|0;break a}d=f+16|0;a=J[c+28>>2];J[d>>2]=a;J[a+4>>2]=J[a+4>>2]+1;J[5996]=0;a=$(198,J[f+16>>2])|0;b=J[5996];J[5996]=0;b:{if((b|0)!=1){b=J[d>>2];c=J[b+4>>2]-1|0;J[b+4>>2]=c;if((c|0)==-1){Xa[J[J[b>>2]+8>>2]](b);}c:{if(e){Xa[J[J[a>>2]+24>>2]](d,a);break c}Xa[J[J[a>>2]+28>>2]](f+16|0,a);}J[f+12>>2]=Mb(f+16|0);while(1){a=f+16|0;c=qc(a);b=J[f+12>>2];if((c|0)==(b|0)){c=J[f+28>>2];ab(a);break a}a=J[b>>2];J[5996]=0;_(214,f+28|0,a|0)|0;a=J[5996];J[5996]=0;if((a|0)!=1){J[f+12>>2]=J[f+12>>2]+4;continue}break}c=Z()|0;ab(f+16|0);break b}c=Z()|0;a=J[f+16>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}}da(c|0);B();}Ua=f+32|0;return c|0}function il(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;var f=0;f=Ua-32|0;Ua=f;J[f+28>>2]=b;a:{if(!(H[c+4|0]&1)){c=Xa[J[J[a>>2]+24>>2]](a,b,c,d,e)|0;break a}d=f+16|0;a=J[c+28>>2];J[d>>2]=a;J[a+4>>2]=J[a+4>>2]+1;J[5996]=0;a=$(180,J[f+16>>2])|0;b=J[5996];J[5996]=0;b:{if((b|0)!=1){b=J[d>>2];c=J[b+4>>2]-1|0;J[b+4>>2]=c;if((c|0)==-1){Xa[J[J[b>>2]+8>>2]](b);}c:{if(e){Xa[J[J[a>>2]+24>>2]](d,a);break c}Xa[J[J[a>>2]+28>>2]](f+16|0,a);}J[f+12>>2]=Mb(f+16|0);while(1){a=f+16|0;c=sc(a);b=J[f+12>>2];if((c|0)==(b|0)){c=J[f+28>>2];ab(a);break a}a=H[b|0];J[5996]=0;_(210,f+28|0,a|0)|0;a=J[5996];J[5996]=0;if((a|0)!=1){J[f+12>>2]=J[f+12>>2]+1;continue}break}c=Z()|0;ab(f+16|0);break b}c=Z()|0;a=J[f+16>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}}da(c|0);B();}Ua=f+32|0;return c|0}function $n(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;var e=0,f=0,g=0,h=0;f=Ua-32|0;Ua=f;J[f+24>>2]=d;H[f+20|0]=0;J[f+8>>2]=a;J[f+16>>2]=f+28;J[f+12>>2]=f+24;a:{b:{c:{if((b|0)!=(c|0)){while(1){J[d+8>>2]=0;J[d>>2]=0;J[d+4>>2]=0;a=J[b+4>>2];e=J[b>>2];if((a|0)!=(e|0)){e=a-e|0;if((e|0)<0){J[5996]=0;J[f+28>>2]=d;ca(84);a=J[5996];J[5996]=0;if((a|0)==1){break c}B();}J[5996]=0;a=$(3,e|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break b}J[d+4>>2]=a;J[d>>2]=a;J[d+8>>2]=(e&-8)+a;e=J[b>>2];g=J[b+4>>2];if((e|0)!=(g|0)){while(1){h=J[e+4>>2];J[a>>2]=J[e>>2];J[a+4>>2]=h;a=a+8|0;e=e+8|0;if((g|0)!=(e|0)){continue}break}}J[d+4>>2]=a;}d=d+12|0;b=b+12|0;if((c|0)!=(b|0)){continue}break}}Ua=f+32|0;return d|0}a=Z()|0;b=J[d>>2];if(!b){break a}J[d+4>>2]=b;$a(b);break a}a=Z()|0;J[f+28>>2]=d;}Ed(f+8|0);da(a|0);B();}function Fm(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;var e=0,f=0,g=0,h=0,i=0,j=0;f=J[a+4>>2];e=J[a+8>>2];if(f>>>0<e>>>0){J[f>>2]=b;J[f+16>>2]=d;J[f+20>>2]=d>>31;J[f+8>>2]=c;J[f+12>>2]=c>>31;J[a+4>>2]=f+24;return}a:{i=J[a>>2];g=(f-i|0)/24|0;h=g+1|0;if(h>>>0<178956971){j=P(g,24);e=(e-i|0)/24|0;g=e<<1;g=e>>>0>=89478485?178956970:g>>>0>h>>>0?g:h;if(g){if(g>>>0>=178956971){break a}h=cb(P(g,24));}else {h=0;}e=j+h|0;J[e+16>>2]=d;J[e+20>>2]=d>>31;J[e+8>>2]=c;J[e+12>>2]=c>>31;J[e>>2]=b;b=e+24|0;if((f|0)!=(i|0)){while(1){f=f-24|0;d=J[f+4>>2];e=e-24|0;J[e>>2]=J[f>>2];J[e+4>>2]=d;c=J[f+20>>2];J[e+16>>2]=J[f+16>>2];J[e+20>>2]=c;c=J[f+12>>2];J[e+8>>2]=J[f+8>>2];J[e+12>>2]=c;if((f|0)!=(i|0)){continue}break}f=J[a>>2];}J[a+8>>2]=h+P(g,24);J[a+4>>2]=b;J[a>>2]=e;if(f){$a(f);}return}nb();B();}Pb();B();}function Lc(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;var e=0,f=0,g=0,h=0,i=0,j=0;g=d;d=Gb(d);i=Ua-16|0;Ua=i;J[i+12>>2]=c;c=a;a:{b:{e=J[a+4>>2];a=K[a+11|0];h=(a&128)>>>7|0?e:a&127;if(h>>>0>=b>>>0){e=h-b|0;J[i+8>>2]=e;a=i+8|0;f=i+12|0;a=J[(M[a>>2]<M[f>>2]?a:f)>>2];f=(K[c+11|0]&128)>>>7|0?(J[c+8>>2]&2147483647)-1|0:10;if(a+(f-h|0)>>>0>=d>>>0){f=(K[c+11|0]&128)>>>7|0?J[c>>2]:c;c:{if((a|0)==(d|0)){a=d;break c}if((a|0)==(e|0)){a=e;break c}j=e-a|0;e=b+f|0;if(a>>>0>d>>>0){break b}d:{if(f+h>>>0<=g>>>0|e>>>0>=g>>>0){break d}if(a+e>>>0<=g>>>0){g=(d-a|0)+g|0;break d}oc(e,g,a);e=d-a|0;g=d+g|0;b=a+b|0;a=0;d=e;}e=b+f|0;oc(e+d|0,a+e|0,j);}oc(b+f|0,g,d);sd(c,f,(d+h|0)-a|0);break a}dd(c,f,d+h-(a+f)|0,h,b,a,d,g);break a}ue();B();}b=oc(e,g,d);oc(b+d|0,a+b|0,j);sd(c,f,(d+h|0)-a|0);}Ua=i+16|0;}function nk(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0,i=0;h=Ua-16|0;Ua=h;d=K[a+11|0];e=d&127;d=(d&128)>>>7|0;e=d?J[a+4>>2]:e;d=d?(J[a+8>>2]&2147483647)-1|0:1;f=c-b>>2;a:{if(!f){break a}b:{g=K[a+11|0];i=(g&128)>>>7|0?J[a>>2]:a;if(i>>>0<=b>>>0){g=(((g&128)>>>7|0?J[a+4>>2]:g&127)<<2)+i>>>0>=b>>>0;}else {g=0;}if(!g){if(d-e>>>0<f>>>0){of(a,d,(e-d|0)+f|0,e,e);}d=(e<<2)+((K[a+11|0]&128)>>>7|0?J[a>>2]:a)|0;while(1){if((b|0)==(c|0)){break b}J[d>>2]=J[b>>2];b=b+4|0;d=d+4|0;continue}}b=ag(h+4|0,b,c);c=K[b+11|0];f=(c&128)>>>7|0;d=f?J[b>>2]:b;e=J[b+4>>2];J[5996]=0;ba(235,a|0,d|0,(f?e:c&127)|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){ab(b);break a}a=Z()|0;ab(b);da(a|0);B();}J[d>>2]=0;b=e+f|0;c:{if((K[a+11|0]&128)>>>7|0){J[a+4>>2]=b;break c}H[a+11|0]=b;}}Ua=h+16|0;return a|0}function rm(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0;J[a+4>>2]=b;H[a|0]=0;b=J[J[b>>2]-12>>2]+b|0;if(!J[b+16>>2]){b=J[b+72>>2];if(b){a:{e=Ua-16|0;Ua=e;b:{c:{if(!J[(b+J[J[b>>2]-12>>2]|0)+24>>2]){break c}J[5996]=0;d=_(134,e+8|0,b|0)|0;c=J[5996];J[5996]=0;d:{if((c|0)!=1){e:{if(!K[d|0]){break e}c=J[(b+J[J[b>>2]-12>>2]|0)+24>>2];J[5996]=0;c=$(169,c|0)|0;f=J[5996];J[5996]=0;if((f|0)!=1){if((c|0)!=-1){break e}c=J[J[b>>2]-12>>2];J[5996]=0;aa(138,b+c|0,1);c=J[5996];J[5996]=0;if((c|0)!=1){break e}}c=fa(0)|0;ud(d);break d}ud(d);break c}c=fa(0)|0;}pa(c|0)|0;d=J[J[b>>2]-12>>2];J[5996]=0;ha(139,b+d|0);b=J[5996];J[5996]=0;if((b|0)==1){break b}ra();}Ua=e+16|0;break a}a=Z()|0;J[5996]=0;ca(110);b=J[5996];J[5996]=0;if((b|0)!=1){da(a|0);B();}fa(0)|0;Vb();B();}}H[a|0]=1;}return a|0}function Pm(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0;d=Ua-32|0;Ua=d;a:{b:{c:{if((c|0)<=0){break c}while(1){if(K[b+e|0]!=41){e=e+1|0;if((c|0)!=(e|0)){continue}break c}break}if(e){break b}}H[a+16|0]=0;H[a|0]=0;break a}d:{e:{if(e>>>0<=17){if(e>>>0>10){break e}H[d+31|0]=e;c=d+20|0;break d}H[a+16|0]=0;H[a|0]=0;break a}f=(e|15)+1|0;c=cb(f);J[d+28>>2]=f|-2147483648;J[d+20>>2]=c;J[d+24>>2]=e;}H[fb(c,b,e)+e|0]=0;c=J[d+20>>2];b=K[d+27|0]|K[d+28|0]<<8|(K[d+29|0]<<16|K[d+30|0]<<24);H[d+15|0]=b;H[d+16|0]=b>>>8;H[d+17|0]=b>>>16;H[d+18|0]=b>>>24;J[d+12>>2]=J[d+24>>2];f=K[d+31|0];J[a>>2]=c;J[a+4>>2]=J[d+12>>2];b=K[d+15|0]|K[d+16|0]<<8|(K[d+17|0]<<16|K[d+18|0]<<24);H[a+7|0]=b;H[a+8|0]=b>>>8;H[a+9|0]=b>>>16;H[a+10|0]=b>>>24;H[a+16|0]=1;J[a+12>>2]=e;H[a+11|0]=f;}Ua=d+32|0;}function Ac(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0,j=0;g=Ua-16|0;Ua=g;a:{b:{c:{if(!a){break c}i=J[e+12>>2];j=c-b|0;if((j|0)>0){if((Xa[J[J[a>>2]+48>>2]](a,b,j)|0)!=(j|0)){break c}}b=d-b|0;b=(b|0)<(i|0)?i-b|0:0;if((b|0)>0){if(b>>>0>=2147483632){break a}d:{if(b>>>0>=11){i=(b|15)+1|0;h=cb(i);J[g+12>>2]=i|-2147483648;J[g+4>>2]=h;J[g+8>>2]=b;break d}H[g+15|0]=b;h=g+4|0;}H[Sb(h,f,b)+b|0]=0;f=J[J[a>>2]+48>>2];J[5996]=0;f=ba(f|0,a|0,(H[g+15|0]<0?J[g+4>>2]:g+4|0)|0,b|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break b}if(H[g+15|0]<0){$a(J[g+4>>2]);}h=0;if((b|0)!=(f|0)){break c}}b=d-c|0;if((b|0)>0){if((Xa[J[J[a>>2]+48>>2]](a,c,b)|0)!=(b|0)){break c}}J[e+12>>2]=0;h=a;}Ua=g+16|0;return h|0}a=Z()|0;if(H[g+15|0]<0){$a(J[g+4>>2]);}da(a|0);B();}ub();B();}function vk(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0,i=0;h=Ua-16|0;Ua=h;d=K[a+11|0];e=d&127;d=(d&128)>>>7|0;e=d?J[a+4>>2]:e;d=d?(J[a+8>>2]&2147483647)-1|0:10;f=c-b|0;a:{if(!f){break a}b:{g=K[a+11|0];i=(g&128)>>>7|0?J[a>>2]:a;if(i>>>0<=b>>>0){g=((g&128)>>>7|0?J[a+4>>2]:g&127)+i>>>0>=b>>>0;}else {g=0;}if(!g){if(d-e>>>0<f>>>0){Sd(a,d,(e-d|0)+f|0,e,e);}d=e+((K[a+11|0]&128)>>>7|0?J[a>>2]:a)|0;while(1){if((b|0)==(c|0)){break b}H[d|0]=K[b|0];b=b+1|0;d=d+1|0;continue}}b=td(h+4|0,b,c);c=K[b+11|0];f=(c&128)>>>7|0;d=f?J[b>>2]:b;e=J[b+4>>2];J[5996]=0;ba(100,a|0,d|0,(f?e:c&127)|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){ab(b);break a}a=Z()|0;ab(b);da(a|0);B();}H[d|0]=0;b=e+f|0;c:{if((K[a+11|0]&128)>>>7|0){J[a+4>>2]=b;break c}H[a+11|0]=b;}}Ua=h+16|0;return a|0}function eb(a,b,c){var d=0,e=0;a:{if((a|0)==(b|0)){break a}e=a+c|0;if(b-e>>>0<=0-(c<<1)>>>0){return fb(a,b,c)}d=(a^b)&3;b:{c:{if(a>>>0<b>>>0){if(d){d=a;break b}if(!(a&3)){d=a;break c}d=a;while(1){if(!c){break a}H[d|0]=K[b|0];b=b+1|0;c=c-1|0;d=d+1|0;if(d&3){continue}break}break c}d:{if(d){break d}if(e&3){while(1){if(!c){break a}c=c-1|0;d=c+a|0;H[d|0]=K[b+c|0];if(d&3){continue}break}}if(c>>>0<=3){break d}while(1){c=c-4|0;J[c+a>>2]=J[b+c>>2];if(c>>>0>3){continue}break}}if(!c){break a}while(1){c=c-1|0;H[c+a|0]=K[b+c|0];if(c){continue}break}break a}if(c>>>0<=3){break b}while(1){J[d>>2]=J[b>>2];b=b+4|0;d=d+4|0;c=c-4|0;if(c>>>0>3){continue}break}}if(!c){break a}while(1){H[d|0]=K[b|0];d=d+1|0;b=b+1|0;c=c-1|0;if(c){continue}break}}return a}function Ig(a,b,c){a:{switch(b-9|0){case 0:b=J[c>>2];J[c>>2]=b+4;J[a>>2]=J[b>>2];return;case 6:b=J[c>>2];J[c>>2]=b+4;b=I[b>>1];J[a>>2]=b;J[a+4>>2]=b>>31;return;case 7:b=J[c>>2];J[c>>2]=b+4;J[a>>2]=L[b>>1];J[a+4>>2]=0;return;case 8:b=J[c>>2];J[c>>2]=b+4;b=H[b|0];J[a>>2]=b;J[a+4>>2]=b>>31;return;case 9:b=J[c>>2];J[c>>2]=b+4;J[a>>2]=K[b|0];J[a+4>>2]=0;return;case 16:b=J[c>>2]+7&-8;J[c>>2]=b+8;O[a>>3]=O[b>>3];return;case 17:Fg(a,c);default:return;case 1:case 4:case 14:b=J[c>>2];J[c>>2]=b+4;b=J[b>>2];J[a>>2]=b;J[a+4>>2]=b>>31;return;case 2:case 5:case 11:case 15:b=J[c>>2];J[c>>2]=b+4;J[a>>2]=J[b>>2];J[a+4>>2]=0;return;case 3:case 10:case 12:case 13:break a}}b=J[c>>2]+7&-8;J[c>>2]=b+8;c=J[b+4>>2];J[a>>2]=J[b>>2];J[a+4>>2]=c;}function Xd(a,b,c,d,e){var f=0,g=0;f=Ua-16|0;Ua=f;J[f+12>>2]=b;J[b+4>>2]=J[b+4>>2]+1;J[5996]=0;b=$(197,J[f+12>>2])|0;g=J[5996];J[5996]=0;a:{if((g|0)==1){break a}J[5996]=0;ga(208,b|0,12416,12448,c|0);b=J[5996];J[5996]=0;if((b|0)==1){break a}J[5996]=0;b=$(198,J[f+12>>2])|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}J[5996]=0;c=$(209,b|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break a}J[d>>2]=c;J[5996]=0;c=$(206,b|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break a}J[e>>2]=c;J[5996]=0;aa(207,a|0,b|0);a=J[5996];J[5996]=0;if((a|0)==1){break a}a=J[f+12>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}Ua=f+16|0;return}c=Z()|0;a=J[f+12>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(c|0);B();}function $d(a,b,c,d,e){var f=0,g=0;f=Ua-16|0;Ua=f;J[f+12>>2]=b;J[b+4>>2]=J[b+4>>2]+1;J[5996]=0;b=$(170,J[f+12>>2])|0;g=J[5996];J[5996]=0;a:{if((g|0)==1){break a}J[5996]=0;ga(195,b|0,12416,12448,c|0);b=J[5996];J[5996]=0;if((b|0)==1){break a}J[5996]=0;b=$(180,J[f+12>>2])|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}J[5996]=0;c=$(196,b|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break a}H[d|0]=c;J[5996]=0;c=$(193,b|0)|0;d=J[5996];J[5996]=0;if((d|0)==1){break a}H[e|0]=c;J[5996]=0;aa(194,a|0,b|0);a=J[5996];J[5996]=0;if((a|0)==1){break a}a=J[f+12>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}Ua=f+16|0;return}c=Z()|0;a=J[f+12>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(c|0);B();}function jg(a,b,c,d,e,f){var g=0;g=Ua-80|0;Ua=g;a:{if((f|0)>=16384){ob(g+32|0,b,c,d,e,0,0,0,2147352576);d=J[g+40>>2];e=J[g+44>>2];b=J[g+32>>2];c=J[g+36>>2];if(f>>>0<32767){f=f-16383|0;break a}ob(g+16|0,b,c,d,e,0,0,0,2147352576);f=((f|0)>=49149?49149:f)-32766|0;d=J[g+24>>2];e=J[g+28>>2];b=J[g+16>>2];c=J[g+20>>2];break a}if((f|0)>-16383){break a}ob(g- -64|0,b,c,d,e,0,0,0,7471104);d=J[g+72>>2];e=J[g+76>>2];b=J[g+64>>2];c=J[g+68>>2];if(f>>>0>4294934644){f=f+16269|0;break a}ob(g+48|0,b,c,d,e,0,0,0,7471104);f=((f|0)<=-48920?-48920:f)+32538|0;d=J[g+56>>2];e=J[g+60>>2];b=J[g+48>>2];c=J[g+52>>2];}ob(g,b,c,d,e,0,0,0,f+16383<<16);b=J[g+12>>2];J[a+8>>2]=J[g+8>>2];J[a+12>>2]=b;b=J[g+4>>2];J[a>>2]=J[g>>2];J[a+4>>2]=b;Ua=g+80|0;}function cc(a,b,c,d,e){var f=0,g=0,h=0,i=0;h=Ua-16|0;Ua=h;J[h+12>>2]=b;b=0;g=6;a:{b:{if(Ub(a,h+12|0)){break b}g=J[a>>2];f=J[g+12>>2];c:{if((f|0)==J[g+16>>2]){f=Xa[J[J[g>>2]+36>>2]](g)|0;break c}f=K[f|0];}g=4;i=J[d+8>>2];f=f<<24>>24;if((f|0)>=0){i=(J[((f&255)<<2)+i>>2]&64)!=0;}else {i=0;}if(!i){break b}b=Xa[J[J[d>>2]+36>>2]](d,f,0)|0;while(1){d:{b=b-48|0;f=Ic(a);if(Ub(f,h+12|0)|(e|0)<2){break d}g=J[f>>2];f=J[g+12>>2];e:{if((f|0)==J[g+16>>2]){g=Xa[J[J[g>>2]+36>>2]](g)|0;break e}g=K[f|0];}f=J[d+8>>2];g=g<<24>>24;if((g|0)>=0){f=(J[f+((g&255)<<2)>>2]&64)!=0;}else {f=0;}if(!f){break a}e=e-1|0;b=(Xa[J[J[d>>2]+36>>2]](d,g,0)|0)+P(b,10)|0;continue}break}g=2;if(!Ub(f,h+12|0)){break a}}J[c>>2]=J[c>>2]|g;}Ua=h+16|0;return b}function Wb(a,b){a=a|0;b=b|0;var c=0,d=0;a:{if((b|0)>0){d=a+b|0;while(1){b=1;b:{c:{switch(K[a|0]-66|0){default:a=Rb(8);J[5996]=0;b=_(17,a|0,2064)|0;c=J[5996];J[5996]=0;if((c|0)==1){break a}na(b|0,22528,7);B();case 20:b=2;break b;case 9:b=4;break b;case 21:b=8;break b;case 56:b=16;break b;case 45:b=32;break b;case 44:b=64;break b;case 34:b=128;break b;case 51:b=256;break b;case 35:b=512;break b;case 36:b=1024;break b;case 49:b=2048;break b;case 43:b=4096;break b;case 54:b=8192;break b;case 37:b=16384;break b;case 38:b=32768;break b;case 39:b=65536;break b;case 40:b=131072;break b;case 0:break b;case 41:break c}}b=262144;}c=b|c;a=a+1|0;if(d>>>0>a>>>0){continue}break}}return c|0}b=Z()|0;ic(a);da(b|0);B();}function Sb(a,b,c){var d=0,e=0,f=0,g=0;a:{if(!c){break a}H[a|0]=b;d=a+c|0;H[d-1|0]=b;if(c>>>0<3){break a}H[a+2|0]=b;H[a+1|0]=b;H[d-3|0]=b;H[d-2|0]=b;if(c>>>0<7){break a}H[a+3|0]=b;H[d-4|0]=b;if(c>>>0<9){break a}d=0-a&3;e=d+a|0;b=P(b&255,16843009);J[e>>2]=b;d=c-d&-4;c=d+e|0;J[c-4>>2]=b;if(d>>>0<9){break a}J[e+8>>2]=b;J[e+4>>2]=b;J[c-8>>2]=b;J[c-12>>2]=b;if(d>>>0<25){break a}J[e+24>>2]=b;J[e+20>>2]=b;J[e+16>>2]=b;J[e+12>>2]=b;J[c-16>>2]=b;J[c-20>>2]=b;J[c-24>>2]=b;J[c-28>>2]=b;g=e&4|24;c=d-g|0;if(c>>>0<32){break a}d=co(b,0,1,1);f=Wa;b=e+g|0;while(1){J[b+24>>2]=d;J[b+28>>2]=f;J[b+16>>2]=d;J[b+20>>2]=f;J[b+8>>2]=d;J[b+12>>2]=f;J[b>>2]=d;J[b+4>>2]=f;b=b+32|0;c=c-32|0;if(c>>>0>31){continue}break}}return a}function re(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0,h=0;a:{d=J[a+4>>2];h=J[a>>2];f=d-h>>3;c=f+1|0;if(c>>>0<536870912){g=J[a+8>>2]-h|0;e=g>>2;c=g>>>0>=2147483640?536870911:c>>>0<e>>>0?e:c;if(c>>>0>=536870912){break a}e=f<<3;f=c<<3;g=cb(f);c=e+g|0;J[c>>2]=J[b>>2];J[c+4>>2]=J[b+4>>2];J[b>>2]=0;J[b+4>>2]=0;e=c+8|0;if((d|0)!=(h|0)){while(1){c=c-8|0;d=d-8|0;J[c>>2]=J[d>>2];J[c+4>>2]=J[d+4>>2];J[d>>2]=0;J[d+4>>2]=0;if((d|0)!=(h|0)){continue}break}}J[a+8>>2]=f+g;b=J[a>>2];J[a>>2]=c;d=J[a+4>>2];J[a+4>>2]=e;if((b|0)!=(d|0)){while(1){d=d-8|0;a=J[d+4>>2];b:{if(!a){break b}c=J[a+4>>2];J[a+4>>2]=c-1;if(c){break b}Xa[J[J[a>>2]+8>>2]](a);bb(a);}if((b|0)!=(d|0)){continue}break}}if(b){$a(b);}return}nb();B();}Pb();B();}function jb(){var a=0,b=0,c=0,d=0;if(K[25008]){return J[6251]}c=Ua-32|0;Ua=c;a:{b:{while(1){b=c+8|0;d=dg(a,1<<a&2147483647?3538:5575);J[b+(a<<2)>>2]=d;if((d|0)==-1){break b}a=a+1|0;if((a|0)!=6){continue}break}a=19324;if(!Eb(b,19324,24)){break a}a=19348;if(!Eb(b,19348,24)){break a}a=0;if(!K[24780]){while(1){J[(a<<2)+24732>>2]=dg(a,5575);a=a+1|0;if((a|0)!=6){continue}break}H[24780]=1;J[6189]=J[6183];}a=24732;b=c+8|0;if(!Eb(b,24732,24)){break a}a=24756;if(!Eb(b,24756,24)){break a}a=sb(24);if(!a){break b}b=J[c+12>>2];J[a>>2]=J[c+8>>2];J[a+4>>2]=b;b=J[c+28>>2];J[a+16>>2]=J[c+24>>2];J[a+20>>2]=b;b=J[c+20>>2];J[a+8>>2]=J[c+16>>2];J[a+12>>2]=b;break a}a=0;}Ua=c+32|0;H[25008]=1;J[6251]=a;return a}function Fe(a,b){var c=0,d=0,e=0,f=0;c=Ua+-64|0;Ua=c;d=J[a>>2];e=J[d-4>>2];f=J[d-8>>2];J[c+28>>2]=0;J[c+32>>2]=0;J[c+36>>2]=0;J[c+40>>2]=0;J[c+44>>2]=0;J[c+48>>2]=0;J[c+52>>2]=0;J[c+56>>2]=0;d=0;H[c+59|0]=0;H[c+60|0]=0;H[c+61|0]=0;H[c+62|0]=0;J[c+20>>2]=0;J[c+24>>2]=0;J[c+16>>2]=22056;J[c+12>>2]=a;J[c+8>>2]=b;a=a+f|0;a:{if(Jb(e,b,0)){J[c+56>>2]=1;Xa[J[J[e>>2]+20>>2]](e,c+8|0,a,a,1,0);d=J[c+32>>2]==1?a:0;break a}Xa[J[J[e>>2]+24>>2]](e,c+8|0,a,1,0);b:{switch(J[c+44>>2]){case 0:d=J[c+48>>2]==1?J[c+36>>2]==1?J[c+40>>2]==1?J[c+28>>2]:0:0:0;break a;case 1:break b;default:break a}}if(J[c+32>>2]!=1){if(J[c+48>>2]|J[c+36>>2]!=1|J[c+40>>2]!=1){break a}}d=J[c+24>>2];}Ua=c- -64|0;return d}function dg(a,b){var c=0,d=0,e=0;a:{if(K[b|0]){break a}b=fe(3169);if(K[b|0]?b:0){break a}b=fe(P(a,12)+9024|0);if(K[b|0]?b:0){break a}b=fe(3321);if(K[b|0]?b:0){break a}b=3845;}b:{while(1){d=K[b+c|0];if(!(!d|(d|0)==47)){d=23;c=c+1|0;if((c|0)!=23){continue}break b}break}d=c;}e=3845;c:{d:{c=K[b|0];e:{f:{if(!(K[b+d|0]|(c|0)==46)){e=b;if((c|0)!=67){break f}}if(!K[e+1|0]){break e}}if(!Uc(e,3845)){break e}if(Uc(e,2764)){break d}}if(!a){c=9096;if(K[e+1|0]==46){break c}}return 0}c=J[6182];if(c){while(1){if(!Uc(e,c+8|0)){break c}c=J[c+32>>2];if(c){continue}break}}c=sb(36);if(c){J[c+4>>2]=20;J[c>>2]=8992;b=c+8|0;fb(b,e,d);H[b+d|0]=0;J[c+32>>2]=J[6182];J[6182]=c;}c=a|c?c:9096;}return c}function $a(a){var b=0,c=0,d=0,e=0,f=0;if(a){b=a-4|0;e=J[b>>2];d=e;c=b;f=J[a-8>>2];a=f&-2;if((a|0)!=(f|0)){c=b-a|0;f=J[c+4>>2];J[f+8>>2]=J[c+8>>2];J[J[c+8>>2]+4>>2]=f;d=a+d|0;}a=b+e|0;b=J[a>>2];if((b|0)!=J[(a+b|0)-4>>2]){e=J[a+4>>2];J[e+8>>2]=J[a+8>>2];J[J[a+8>>2]+4>>2]=e;d=b+d|0;}J[c>>2]=d;J[((d&-4)+c|0)-4>>2]=d|1;b=J[c>>2]-8|0;a:{if(b>>>0<=127){a=(b>>>3|0)-1|0;break a}d=S(b);a=((b>>>29-d^4)-(d<<2)|0)+110|0;if(b>>>0<=4095){break a}a=((b>>>30-d^2)-(d<<1)|0)+71|0;a=a>>>0>=63?63:a;}b=a<<4;J[c+4>>2]=b+22944;b=b+22952|0;J[c+8>>2]=J[b>>2];J[b>>2]=c;J[J[c+8>>2]+4>>2]=c;b=J[5994];d=J[5995];c=a&31;if((a&63)>>>0>=32){a=1<<c;e=0;}else {e=1<<c;a=e-1&1>>>32-c;}J[5994]=e|b;J[5995]=a|d;}}function Dc(a,b,c,d,e,f,g,h,i,j,k){var l=0,m=0,n=0;a:{b:{l=J[d>>2];if((l|0)!=(c|0)){break b}m=43;n=a&255;if((n|0)!=K[k+24|0]){m=45;if(K[k+25|0]!=(n|0)){break b}}J[d>>2]=c+1;H[c|0]=m;break a}c:{if(!(!((h&128)>>>7|0?g:h&127)|(a|0)!=(f|0))){h=0;a=J[j>>2];if((a-i|0)>159){break c}b=J[e>>2];J[j>>2]=a+4;J[a>>2]=b;break a}h=-1;a=be(k,k+26|0,a)-k|0;if((a|0)>23){break c}d:{e:{switch(b-8|0){case 0:case 2:if((a|0)<(b|0)){break d}break c;case 1:break d;default:break e}}if((b|0)!=16|(a|0)<22){break d}if((c|0)==(l|0)|(l-c|0)>2|K[l-1|0]!=48){break c}J[e>>2]=0;J[d>>2]=l+1;H[l|0]=K[a+12416|0];return 0}J[d>>2]=l+1;H[l|0]=K[a+12416|0];J[e>>2]=J[e>>2]+1;h=0;}return h}J[e>>2]=0;return 0}function Cc(a,b,c,d,e,f,g,h,i,j,k){var l=0,m=0;a:{b:{l=J[d>>2];if((l|0)!=(c|0)){break b}m=43;if(J[k+96>>2]!=(a|0)){m=45;if(J[k+100>>2]!=(a|0)){break b}}J[d>>2]=c+1;H[c|0]=m;break a}c:{if(!(!((h&128)>>>7|0?g:h&127)|(a|0)!=(f|0))){h=0;a=J[j>>2];if((a-i|0)>159){break c}b=J[e>>2];J[j>>2]=a+4;J[a>>2]=b;break a}h=-1;f=Yd(k,k+104|0,a)-k|0;if((f|0)>92){break c}a=f>>2;d:{e:{switch(b-8|0){case 0:case 2:if((a|0)<(b|0)){break d}break c;case 1:break d;default:break e}}if((b|0)!=16|(f|0)<88){break d}if((c|0)==(l|0)|(l-c|0)>2|K[l-1|0]!=48){break c}J[e>>2]=0;J[d>>2]=l+1;H[l|0]=K[a+12416|0];return 0}J[d>>2]=l+1;H[l|0]=K[a+12416|0];J[e>>2]=J[e>>2]+1;h=0;}return h}J[e>>2]=0;return 0}function mc(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0,h=0;c=J[a+4>>2];d=J[a+8>>2];if(c>>>0<d>>>0){J[c>>2]=b;J[a+4>>2]=c+4;return}a:{b:{c:{g=J[a>>2];e=c-g>>2;f=e+1|0;if(f>>>0<1073741824){h=e<<2;d=d-g|0;e=d>>1;d=d>>>0>=2147483644?1073741823:f>>>0<e>>>0?e:f;if(d){if(d>>>0>=1073741824){break c}e=cb(d<<2);}else {e=0;}f=h+e|0;J[f>>2]=b;b=e+(d<<2)|0;d=f+4|0;if((c|0)==(g|0)){break b}while(1){c=c-4|0;e=J[c>>2];J[c>>2]=0;f=f-4|0;J[f>>2]=e;if((c|0)!=(g|0)){continue}break}J[a+8>>2]=b;b=J[a+4>>2];J[a+4>>2]=d;c=J[a>>2];J[a>>2]=f;if((b|0)==(c|0)){break a}while(1){b=b-4|0;ad(b);if((b|0)!=(c|0)){continue}break}break a}nb();B();}Pb();B();}J[a+8>>2]=b;J[a+4>>2]=d;J[a>>2]=f;}if(c){$a(c);}}function bn(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0;a:{if(b>>>0>=256){a=Rb(8);J[5996]=0;b=_(91,a|0,2394)|0;c=J[5996];J[5996]=0;if((c|0)!=1){break a}b=Z()|0;ic(a);da(b|0);B();}c=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);e=c-29|0;f=c>>>0<29?c:e;d=f+1|0;if(d>>>0>(c>>>0<=28?28:K[a+4|0]|K[a+5|0]<<8|(K[a+6|0]<<16|K[a+7|0]<<24))>>>0){Ob(a,d+(d>>>1|0)|0);c=K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24);e=c-29|0;}g=c>>>0<29;d=(g?a+4|0:K[a+8|0]|K[a+9|0]<<8|(K[a+10|0]<<16|K[a+11|0]<<24))+f|0;eb(d+1|0,d,(g?c:e)-f|0);c=(K[a|0]|K[a+1|0]<<8|(K[a+2|0]<<16|K[a+3|0]<<24))+1|0;H[a|0]=c;H[a+1|0]=c>>>8;H[a+2|0]=c>>>16;H[a+3|0]=c>>>24;H[d|0]=b;return a|0}na(b|0,22664,92);B();}function Wk(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0;g=Ua-256|0;Ua=g;J[g+248>>2]=37;J[g+252>>2]=0;h=g+248|0;rc(h|1,2055,1,J[c+4>>2]);i=jb();J[g>>2]=e;J[g+4>>2]=f;a=g+224|0;e=yb(a,24,i,h,g)+a|0;i=Lb(a,e,J[c+4>>2]);f=g+20|0;h=J[c+28>>2];J[f>>2]=h;J[h+4>>2]=J[h+4>>2]+1;J[5996]=0;h=g+32|0;ma(215,a|0,i|0,e|0,h|0,g+28|0,g+24|0,f|0);a=J[5996];J[5996]=0;a:{if((a|0)!=1){a=J[f>>2];e=J[a+4>>2]-1|0;J[a+4>>2]=e;if((e|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}a=Bc(b,h,J[g+28>>2],J[g+24>>2],c,d);Ua=g+256|0;break a}c=Z()|0;a=J[g+20>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(c|0);B();}return a|0}function Uk(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0;g=Ua-256|0;Ua=g;J[g+248>>2]=37;J[g+252>>2]=0;h=g+248|0;rc(h|1,2055,0,J[c+4>>2]);i=jb();J[g>>2]=e;J[g+4>>2]=f;a=g+224|0;e=yb(a,24,i,h,g)+a|0;i=Lb(a,e,J[c+4>>2]);f=g+20|0;h=J[c+28>>2];J[f>>2]=h;J[h+4>>2]=J[h+4>>2]+1;J[5996]=0;h=g+32|0;ma(215,a|0,i|0,e|0,h|0,g+28|0,g+24|0,f|0);a=J[5996];J[5996]=0;a:{if((a|0)!=1){a=J[f>>2];e=J[a+4>>2]-1|0;J[a+4>>2]=e;if((e|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}a=Bc(b,h,J[g+28>>2],J[g+24>>2],c,d);Ua=g+256|0;break a}c=Z()|0;a=J[g+20>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(c|0);B();}return a|0}function fl(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0;g=Ua-112|0;Ua=g;J[g+104>>2]=37;J[g+108>>2]=0;h=g+104|0;rc(h|1,2055,1,J[c+4>>2]);i=jb();J[g>>2]=e;J[g+4>>2]=f;a=g+80|0;e=yb(a,24,i,h,g)+a|0;i=Lb(a,e,J[c+4>>2]);f=g+20|0;h=J[c+28>>2];J[f>>2]=h;J[h+4>>2]=J[h+4>>2]+1;J[5996]=0;h=g+32|0;ma(211,a|0,i|0,e|0,h|0,g+28|0,g+24|0,f|0);a=J[5996];J[5996]=0;a:{if((a|0)!=1){a=J[f>>2];e=J[a+4>>2]-1|0;J[a+4>>2]=e;if((e|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}a=Ac(b,h,J[g+28>>2],J[g+24>>2],c,d);Ua=g+112|0;break a}c=Z()|0;a=J[g+20>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(c|0);B();}return a|0}function dl(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0;g=Ua-112|0;Ua=g;J[g+104>>2]=37;J[g+108>>2]=0;h=g+104|0;rc(h|1,2055,0,J[c+4>>2]);i=jb();J[g>>2]=e;J[g+4>>2]=f;a=g+80|0;e=yb(a,24,i,h,g)+a|0;i=Lb(a,e,J[c+4>>2]);f=g+20|0;h=J[c+28>>2];J[f>>2]=h;J[h+4>>2]=J[h+4>>2]+1;J[5996]=0;h=g+32|0;ma(211,a|0,i|0,e|0,h|0,g+28|0,g+24|0,f|0);a=J[5996];J[5996]=0;a:{if((a|0)!=1){a=J[f>>2];e=J[a+4>>2]-1|0;J[a+4>>2]=e;if((e|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}a=Ac(b,h,J[g+28>>2],J[g+24>>2],c,d);Ua=g+112|0;break a}c=Z()|0;a=J[g+20>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(c|0);B();}return a|0}function Fh(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0,j=0,k=0,l=0;if(Jb(a,J[b+8>>2],f)){Hd(b,c,d,e);return}g=K[b+53|0];i=J[a+12>>2];H[b+53|0]=0;h=K[b+52|0];H[b+52|0]=0;Fd(J[a+16>>2],J[a+20>>2],b,c,d,e,f);j=K[b+52|0];h=(h|j)!=0;k=K[b+53|0];l=(g|k)!=0;g=a+24|0;i=(a+16|0)+(i<<3)|0;a:{if(g>>>0>=i>>>0){break a}while(1){if(K[b+54|0]){break a}b:{if(j){if(J[b+24>>2]==1){break a}if(K[a+8|0]&2){break b}break a}if(!k){break b}if(!(H[a+8|0]&1)){break a}}I[b+52>>1]=0;Fd(J[g>>2],J[g+4>>2],b,c,d,e,f);k=K[b+53|0];l=(k|l)!=0;j=K[b+52|0];h=(j|h)!=0;g=g+8|0;if(i>>>0>g>>>0){continue}break}}H[b+53|0]=l;H[b+52|0]=h;}function bc(a,b,c,d,e){var f=0,g=0,h=0,i=0;h=Ua-16|0;Ua=h;J[h+12>>2]=b;b=0;g=6;a:{b:{if(Tb(a,h+12|0)){break b}g=4;f=J[a>>2];i=J[f+12>>2];c:{if((i|0)==J[f+16>>2]){f=Xa[J[J[f>>2]+36>>2]](f)|0;break c}f=J[i>>2];}if(!(Xa[J[J[d>>2]+12>>2]](d,64,f)|0)){break b}b=Xa[J[J[d>>2]+52>>2]](d,f,0)|0;while(1){d:{b=b-48|0;f=Hc(a);if(Tb(f,h+12|0)|(e|0)<2){break d}g=J[f>>2];f=J[g+12>>2];e:{if((f|0)==J[g+16>>2]){g=Xa[J[J[g>>2]+36>>2]](g)|0;break e}g=J[f>>2];}if(!(Xa[J[J[d>>2]+12>>2]](d,64,g)|0)){break a}e=e-1|0;b=(Xa[J[J[d>>2]+52>>2]](d,g,0)|0)+P(b,10)|0;continue}break}g=2;if(!Tb(f,h+12|0)){break a}}J[c>>2]=J[c>>2]|g;}Ua=h+16|0;return b}function qd(a,b,c,d){var e=0,f=0,g=0,h=0;g=d?d:24720;d=J[g>>2];a:{b:{c:{if(!b){if(d){break c}return 0}e=-2;if(!c){break b}d:{if(d){e=c;break d}d=K[b|0];f=d<<24>>24;if((f|0)>=0){if(a){J[a>>2]=d;}return (f|0)!=0}if(!J[J[6178]>>2]){e=1;if(!a){break b}J[a>>2]=f&57343;return 1}d=d-194|0;if(d>>>0>50){break c}d=J[(d<<2)+12208>>2];e=c-1|0;if(!e){break a}b=b+1|0;}f=K[b|0];h=f>>>3|0;if((h-16|(d>>26)+h)>>>0>7){break c}while(1){e=e-1|0;d=f-128|d<<6;if((d|0)>=0){J[g>>2]=0;if(a){J[a>>2]=d;}return c-e|0}if(!e){break a}b=b+1|0;f=K[b|0];if((f&192)==128){continue}break}}J[g>>2]=0;J[5732]=25;e=-1;}return e}J[g>>2]=d;return -2}function Zk(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;var f=0,g=0,h=0,i=0;f=Ua-144|0;Ua=f;J[f+136>>2]=37;J[f+140>>2]=0;h=f+136|0;rc(h|1,2096,1,J[c+4>>2]);g=jb();J[f>>2]=e;a=f+123|0;e=yb(a,13,g,h,f)+a|0;i=Lb(a,e,J[c+4>>2]);h=f+4|0;g=J[c+28>>2];J[h>>2]=g;J[g+4>>2]=J[g+4>>2]+1;J[5996]=0;g=f+16|0;ma(215,a|0,i|0,e|0,g|0,f+12|0,f+8|0,h|0);a=J[5996];J[5996]=0;a:{if((a|0)!=1){a=J[h>>2];e=J[a+4>>2]-1|0;J[a+4>>2]=e;if((e|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}a=Bc(b,g,J[f+12>>2],J[f+8>>2],c,d);Ua=f+144|0;break a}c=Z()|0;a=J[f+4>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(c|0);B();}return a|0}function Vk(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;var f=0,g=0,h=0,i=0;f=Ua-144|0;Ua=f;J[f+136>>2]=37;J[f+140>>2]=0;h=f+136|0;rc(h|1,2096,0,J[c+4>>2]);g=jb();J[f>>2]=e;a=f+123|0;e=yb(a,13,g,h,f)+a|0;i=Lb(a,e,J[c+4>>2]);h=f+4|0;g=J[c+28>>2];J[h>>2]=g;J[g+4>>2]=J[g+4>>2]+1;J[5996]=0;g=f+16|0;ma(215,a|0,i|0,e|0,g|0,f+12|0,f+8|0,h|0);a=J[5996];J[5996]=0;a:{if((a|0)!=1){a=J[h>>2];e=J[a+4>>2]-1|0;J[a+4>>2]=e;if((e|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}a=Bc(b,g,J[f+12>>2],J[f+8>>2],c,d);Ua=f+144|0;break a}c=Z()|0;a=J[f+4>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(c|0);B();}return a|0}function hl(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;var f=0,g=0,h=0,i=0;f=Ua+-64|0;Ua=f;J[f+56>>2]=37;J[f+60>>2]=0;h=f+56|0;rc(h|1,2096,1,J[c+4>>2]);g=jb();J[f>>2]=e;a=f+43|0;e=yb(a,13,g,h,f)+a|0;i=Lb(a,e,J[c+4>>2]);h=f+4|0;g=J[c+28>>2];J[h>>2]=g;J[g+4>>2]=J[g+4>>2]+1;J[5996]=0;g=f+16|0;ma(211,a|0,i|0,e|0,g|0,f+12|0,f+8|0,h|0);a=J[5996];J[5996]=0;a:{if((a|0)!=1){a=J[h>>2];e=J[a+4>>2]-1|0;J[a+4>>2]=e;if((e|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}a=Ac(b,g,J[f+12>>2],J[f+8>>2],c,d);Ua=f- -64|0;break a}c=Z()|0;a=J[f+4>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(c|0);B();}return a|0}function el(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;var f=0,g=0,h=0,i=0;f=Ua+-64|0;Ua=f;J[f+56>>2]=37;J[f+60>>2]=0;h=f+56|0;rc(h|1,2096,0,J[c+4>>2]);g=jb();J[f>>2]=e;a=f+43|0;e=yb(a,13,g,h,f)+a|0;i=Lb(a,e,J[c+4>>2]);h=f+4|0;g=J[c+28>>2];J[h>>2]=g;J[g+4>>2]=J[g+4>>2]+1;J[5996]=0;g=f+16|0;ma(211,a|0,i|0,e|0,g|0,f+12|0,f+8|0,h|0);a=J[5996];J[5996]=0;a:{if((a|0)!=1){a=J[h>>2];e=J[a+4>>2]-1|0;J[a+4>>2]=e;if((e|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}a=Ac(b,g,J[f+12>>2],J[f+8>>2],c,d);Ua=f- -64|0;break a}c=Z()|0;a=J[f+4>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(c|0);B();}return a|0}function jh(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0;J[c>>2]=0;a:{e=J[b+4>>2];if(!e){break a}f=J[b>>2];b:{c:{d:{while(1){d=H[f|0];g=J[c>>2];if((d|0)<((g?48:49)|0)|(d|0)>57){break d}d=(d+P(g,10)|0)-48|0;J[c>>2]=d>>>0>=65535?65535:d;e=e-1|0;J[b+4>>2]=e;f=f+1|0;J[b>>2]=f;if(e){continue}break}if(J[c>>2]){break c}break a}if(!g){break a}if((d|0)==64){break b}}J[a>>2]=0;J[a+4>>2]=0;J[a+48>>2]=0;J[a+52>>2]=0;J[a+40>>2]=0;J[a+44>>2]=0;J[a+32>>2]=0;J[a+36>>2]=0;J[a+24>>2]=0;J[a+28>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;J[a+8>>2]=0;J[a+12>>2]=0;return}J[b+4>>2]=e-1;J[b>>2]=f+1;Bd(a,b);return}J[c>>2]=1;Bd(a,b);}function yc(a,b,c,d,e,f,g,h){var i=0,j=0,k=0,l=0;j=1;i=d&2147483647;l=i;k=(i|0)==2147418112;a:{if(k&!c?a|b:k&(c|0)!=0|i>>>0>2147418112){break a}i=h&2147483647;if(!g&(i|0)==2147418112?e|f:(i|0)==2147418112&(g|0)!=0|i>>>0>2147418112){break a}if(!(a|e|(c|g)|(b|f|(i|l)))){return 0}j=d&h;if((j|0)>0|(j|0)>=0){j=-1;if((c|0)==(g|0)&(d|0)==(h|0)?(b|0)==(f|0)&a>>>0<e>>>0|b>>>0<f>>>0:c>>>0<g>>>0&(d|0)<=(h|0)|(d|0)<(h|0)){break a}return (a^e|c^g|(b^f|d^h))!=0}j=-1;if((c|0)==(g|0)&(d|0)==(h|0)?(b|0)==(f|0)&a>>>0>e>>>0|b>>>0>f>>>0:c>>>0>g>>>0&(d|0)>=(h|0)|(d|0)>(h|0)){break a}j=(a^e|c^g|(b^f|d^h))!=0;}return j}function _n(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0,h=0;a:{b:{e=J[a+4>>2];f=J[a>>2];d=e-f>>2;c:{if(d>>>0<b>>>0){g=b-d|0;c=J[a+8>>2];if(g>>>0<=c-e>>2>>>0){b=a;a=g<<2;J[b+4>>2]=Sb(e,0,a)+a;return}if(b>>>0>=1073741824){break b}c=c-f|0;h=c>>1;b=c>>>0>=2147483644?1073741823:b>>>0<h>>>0?h:b;if(b>>>0>=1073741824){break a}c=b<<2;b=cb(c);c=b+c|0;b=b+(d<<2)|0;d=g<<2;d=Sb(b,0,d)+d|0;if((e|0)!=(f|0)){while(1){b=b-4|0;e=e-4|0;J[b>>2]=J[e>>2];if((e|0)!=(f|0)){continue}break}}J[a+8>>2]=c;J[a+4>>2]=d;J[a>>2]=b;if(!f){break c}$a(f);return}if(b>>>0>=d>>>0){break c}J[a+4>>2]=(b<<2)+f;}return}nb();B();}Pb();B();}function Qk(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;var f=0,g=0,h=0,i=0,j=0,k=0;a=Ua-208|0;Ua=a;g=jb();J[a>>2]=e;e=a+176|0;h=yb(e,20,g,1937,a);g=h+e|0;i=Lb(e,g,J[c+4>>2]);f=J[c+28>>2];J[a+12>>2]=f;J[f+4>>2]=J[f+4>>2]+1;J[5996]=0;j=$(197,J[a+12>>2])|0;f=J[5996];J[5996]=0;if((f|0)!=1){f=J[a+12>>2];k=J[f+4>>2]-1|0;J[f+4>>2]=k;if((k|0)==-1){Xa[J[J[f>>2]+8>>2]](f);}f=e;e=a+16|0;Xa[J[J[j>>2]+48>>2]](j,f,g,e)|0;f=b;b=e+(h<<2)|0;b=Bc(f,e,(g|0)==(i|0)?b:(a+(i-a<<2)|0)-688|0,b,c,d);Ua=a+208|0;return b|0}c=Z()|0;a=J[a+12>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(c|0);B();}function Rb(a){var b=0,c=0,d=0,e=0,f=0;f=a+27&-4;a=f;b=a>>>0<=1?1:a;a=b;c=a+15&-16;a=a>>>0<c>>>0?c:a;if(a&15){a=0;}else {a=_c(16,a);}if(!a){a:{c=0;a=J[6705];if(!a){J[6705]=26844;I[13423]=125;I[13422]=128;a=J[6705];}d=(b+3>>>2|0)+1|0;while(1){b=0;b:{if(!a|(a|0)==27344){a=b;}else {b=L[a+2>>1];e=d+(b>>>0>d>>>0?b-d&3:0)|0;if(e>>>0<b>>>0){c=b-e|0;I[a+2>>1]=c;a=((c&65535)<<2)+a|0;I[a+2>>1]=e;I[a>>1]=0;a=a+4|0;break a}if(b>>>0<d>>>0){break b}b=L[a>>1];c:{if(!c){J[6705]=(b<<2)+26832;break c}I[c>>1]=b;}I[a>>1]=0;a=a+4|0;}break a}c=a;a=(L[a>>1]<<2)+26832|0;continue}}}if(!a){Vb();B();}return Sb(a,0,f)+24|0}function $k(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;var f=0,g=0,h=0,i=0,j=0,k=0;a=Ua-96|0;Ua=a;g=jb();J[a>>2]=e;e=a- -64|0;h=yb(e,20,g,1937,a);g=h+e|0;i=Lb(e,g,J[c+4>>2]);f=J[c+28>>2];J[a+12>>2]=f;J[f+4>>2]=J[f+4>>2]+1;J[5996]=0;j=$(170,J[a+12>>2])|0;f=J[5996];J[5996]=0;if((f|0)!=1){f=J[a+12>>2];k=J[f+4>>2]-1|0;J[f+4>>2]=k;if((k|0)==-1){Xa[J[J[f>>2]+8>>2]](f);}f=e;e=a+16|0;Xa[J[J[j>>2]+32>>2]](j,f,g,e)|0;f=b;b=e+h|0;b=Ac(f,e,(g|0)==(i|0)?b:(a+(i-a|0)|0)-48|0,b,c,d);Ua=a+96|0;return b|0}c=Z()|0;a=J[a+12>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(c|0);B();}function He(a,b,c,d,e,f,g,h){var i=0,j=0,k=0,l=0;i=Ua-16|0;Ua=i;if(1073741806-b>>>0>=c>>>0){k=(K[a+11|0]&128)>>>7|0?J[a>>2]:a;j=i+4|0;l=j;if(b>>>0<=536870886){J[i+12>>2]=b<<1;J[i+4>>2]=b+c;c=J[(M[j>>2]<M[i+12>>2]?i+12|0:j)>>2];if(c>>>0>=2){j=c+4&-4;c=j-1|0;c=(c|0)==2?j:c;}else {c=1;}c=c+1|0;}else {c=1073741807;}Fc(l,c);c=J[i+4>>2];if(e){tc(c,k,e);}if(g){tc(c+(e<<2)|0,h,g);}j=e+f|0;h=d-j|0;if((d|0)!=(j|0)){d=e<<2;tc((d+c|0)+(g<<2)|0,(d+k|0)+(f<<2)|0,h);}if((b|0)!=1){$a(k);}J[a>>2]=c;J[a+8>>2]=J[i+8>>2]|-2147483648;b=a;a=h+(e+g|0)|0;J[b+4>>2]=a;J[c+(a<<2)>>2]=0;Ua=i+16|0;return}ub();B();}function Ad(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0;e=c-b|0;if(e>>>0<2147483632){a:{if(e>>>0<=10){H[a+11|0]=e;d=a;break a}f=(e|15)+1|0;d=cb(f);J[a+8>>2]=f|-2147483648;J[a>>2]=d;J[a+4>>2]=e;}b:{if((b|0)==(c|0)){break b}f=(b^-1)+c|0;g=e&7;if(g){e=0;while(1){H[d|0]=K[b|0];d=d+1|0;b=b+1|0;e=e+1|0;if((g|0)!=(e|0)){continue}break}}if(f>>>0<7){break b}while(1){H[d|0]=K[b|0];H[d+1|0]=K[b+1|0];H[d+2|0]=K[b+2|0];H[d+3|0]=K[b+3|0];H[d+4|0]=K[b+4|0];H[d+5|0]=K[b+5|0];H[d+6|0]=K[b+6|0];H[d+7|0]=K[b+7|0];d=d+8|0;b=b+8|0;if((c|0)!=(b|0)){continue}break}}H[d|0]=0;return a|0}ub();B();}function $c(a,b,c,d){var e=0,f=0,g=0;e=Ua-32|0;Ua=e;if(c>>>0<2147483632){a:{b:{if(c>>>0>=11){g=(c|15)+1|0;f=cb(g);J[e+16>>2]=g|-2147483648;J[e+8>>2]=f;J[e+12>>2]=c;break b}H[e+19|0]=c;f=e+8|0;if(!c){break a}}f=fb(f,b,c)+c|0;}H[f|0]=0;J[5996]=0;aa(31,e+20|0,e+8|0);b=J[5996];J[5996]=0;c:{d:{if((b|0)!=1){if(H[e+19|0]<0){$a(J[e+8>>2]);}c=J[e+24>>2];b=J[e+20>>2];if((c-b|0)!=(d|0)){break d}J[a+4>>2]=c;J[a>>2]=b;J[a+8>>2]=J[e+28>>2];break c}a=Z()|0;if(H[e+19|0]<0){$a(J[e+8>>2]);}da(a|0);B();}J[a+8>>2]=0;J[a>>2]=0;J[a+4>>2]=0;if(!b){break c}$a(b);}Ua=e+32|0;return}ub();B();}function dd(a,b,c,d,e,f,g,h){var i=0,j=0,k=0,l=0;i=Ua-16|0;Ua=i;if(2147483630-b>>>0>=c>>>0){k=(K[a+11|0]&128)>>>7|0?J[a>>2]:a;j=i+4|0;l=j;if(b>>>0<=1073741798){J[i+12>>2]=b<<1;J[i+4>>2]=b+c;c=J[(M[j>>2]<M[i+12>>2]?i+12|0:j)>>2];if(c>>>0>=11){j=c+16&-16;c=j-1|0;c=(c|0)==11?j:c;}else {c=10;}c=c+1|0;}else {c=2147483631;}Gc(l,c);c=J[i+4>>2];if(e){gc(c,k,e);}if(g){gc(c+e|0,h,g);}j=e+f|0;h=d-j|0;if((d|0)!=(j|0)){gc((c+e|0)+g|0,(e+k|0)+f|0,h);}if((b|0)!=10){$a(k);}J[a>>2]=c;J[a+8>>2]=J[i+8>>2]|-2147483648;b=a;a=h+(e+g|0)|0;J[b+4>>2]=a;H[a+c|0]=0;Ua=i+16|0;return}ub();B();}function Bc(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0,j=0;i=Ua-16|0;Ua=i;a:{b:{if(!a){break b}g=J[e+12>>2];h=c-b|0;if((h|0)>0){h=h>>>2|0;if((Xa[J[J[a>>2]+48>>2]](a,b,h)|0)!=(h|0)){break b}}b=d-b>>2;g=(b|0)<(g|0)?g-b|0:0;if((g|0)>0){b=Hf(i+4|0,g,f);f=(K[b+11|0]&128)>>>7|0?J[b>>2]:b;J[5996]=0;f=ba(217,a|0,f|0,g|0)|0;h=J[5996];J[5996]=0;if((h|0)==1){break a}ab(b);if((f|0)!=(g|0)){break b}}b=d-c|0;if((b|0)>0){b=b>>>2|0;if((Xa[J[J[a>>2]+48>>2]](a,c,b)|0)!=(b|0)){break b}}J[e+12>>2]=0;j=a;}Ua=i+16|0;return j|0}a=Z()|0;ab(b);da(a|0);B();}function Mh(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;if(Jb(a,J[b+8>>2],e)){if(!(J[b+28>>2]==1|J[b+4>>2]!=(c|0))){J[b+28>>2]=d;}return}a:{if(Jb(a,J[b>>2],e)){if(!(J[b+16>>2]!=(c|0)&J[b+20>>2]!=(c|0))){if((d|0)!=1){break a}J[b+32>>2]=1;return}J[b+32>>2]=d;b:{if(J[b+44>>2]==4){break b}I[b+52>>1]=0;a=J[a+8>>2];Xa[J[J[a>>2]+20>>2]](a,b,c,c,1,e);if(K[b+53|0]){J[b+44>>2]=3;if(!K[b+52|0]){break b}break a}J[b+44>>2]=4;}J[b+20>>2]=c;J[b+40>>2]=J[b+40>>2]+1;if(J[b+36>>2]!=1|J[b+24>>2]!=2){break a}H[b+54|0]=1;return}a=J[a+8>>2];Xa[J[J[a>>2]+24>>2]](a,b,c,d,e);}}function pb(a){var b=0,c=0,d=0,e=0;e=Ua-16|0;Ua=e;b=e+4|0;J[b+4>>2]=190;J[b>>2]=a;J[b+8>>2]=0;d=Ua-16|0;Ua=d;if(J[a>>2]!=-1){a:{c=d+12|0;J[c>>2]=b;b=d+8|0;J[b>>2]=c;while(1){c=J[a>>2];if((c|0)==1){continue}break}b:{if(!c){J[a>>2]=1;b=J[J[b>>2]>>2];J[5996]=0;ha(191,b|0);b=J[5996];J[5996]=0;if((b|0)==1){break b}J[a>>2]=-1;}break a}pa(fa(0)|0)|0;J[a>>2]=0;J[5996]=0;ca(192);a=J[5996];J[5996]=0;c:{if((a|0)==1){a=Z()|0;J[5996]=0;ca(110);b=J[5996];J[5996]=0;if((b|0)!=1){break c}fa(0)|0;Vb();}B();}da(a|0);B();}}Ua=d+16|0;Ua=e+16|0;return J[a+4>>2]-1|0}function Dk(a,b,c,d,e,f,g){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;g=g|0;c=Ua-128|0;Ua=c;J[c+12>>2]=c+116;d=J[a+8>>2];a=c+16|0;yf(d,a,c+12|0,e,f,g);e=Ua-16|0;Ua=e;g=Ua-32|0;Ua=g;d=Ua-16|0;Ua=d;J[d+12>>2]=J[c+12>>2];J[g+24>>2]=a;J[g+28>>2]=J[d+12>>2];Ua=d+16|0;f=J[g+24>>2];a=J[g+28>>2];d=Ua-16|0;Ua=d;J[d+12>>2]=b;while(1){if((a|0)!=(f|0)){sg(d+12|0,H[f|0]);f=f+1|0;continue}break}J[g+16>>2]=a;J[g+20>>2]=J[d+12>>2];Ua=d+16|0;a=J[g+16>>2];J[g+12>>2]=J[g+20>>2];J[e+8>>2]=a;J[e+12>>2]=J[g+12>>2];Ua=g+32|0;Ua=e+16|0;Ua=c+128|0;return J[e+12>>2]}function Sm(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;var e=0,f=0;J[a+8>>2]=0;J[a>>2]=0;J[a+4>>2]=0;J[5996]=0;e=$(3,24)|0;f=J[5996];J[5996]=0;a:{if((f|0)!=1){J[a+4>>2]=e;J[a>>2]=e;f=e+24|0;J[a+8>>2]=f;J[e>>2]=J[b>>2];J[e+4>>2]=J[b+4>>2];J[b>>2]=0;J[b+4>>2]=0;J[a+4>>2]=e+8;J[e+8>>2]=J[c>>2];J[e+12>>2]=J[c+4>>2];J[c>>2]=0;J[c+4>>2]=0;b=e+16|0;J[a+4>>2]=b;if(b>>>0<f>>>0){J[b>>2]=J[d>>2];J[b+4>>2]=J[d+4>>2];J[d>>2]=0;J[d+4>>2]=0;J[a+4>>2]=b+8;break a}J[5996]=0;aa(162,a|0,d|0);b=J[5996];J[5996]=0;if((b|0)!=1){break a}}b=Z()|0;lb(a);da(b|0);B();}}function Rm(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0;d=Ua-16|0;Ua=d;J[a>>2]=7e3;J[a+4>>2]=0;J[a+8>>2]=0;J[d+4>>2]=J[c>>2];J[d+8>>2]=J[c+4>>2];J[d+12>>2]=J[c+8>>2];J[c+8>>2]=0;J[c>>2]=0;J[c+4>>2]=0;J[5996]=0;ia(64,a+12|0,b|0,d+4|0,0)|0;b=J[5996];J[5996]=0;if((b|0)!=1){b=J[d+4>>2];if(b){c=J[d+8>>2];if((b|0)!=(c|0)){while(1){c=c-8|0;e=J[c+4>>2];a:{if(!e){break a}f=J[e+4>>2];J[e+4>>2]=f-1;if(f){break a}Xa[J[J[e>>2]+8>>2]](e);bb(e);}if((b|0)!=(c|0)){continue}break}b=J[d+4>>2];}$a(b);}Ua=d+16|0;return a|0}a=Z()|0;lb(d+4|0);da(a|0);B();}function fc(a,b){var c=0,d=0,e=0,f=0,g=0,h=0,i=0;e=Ua-16|0;Ua=e;A(+b);g=v(1)|0;c=v(0)|0;d=g&2147483647;f=d+-1048576|0;a:{if((f|0)==2145386495|f>>>0<2145386495){h=c<<28;c=(d&15)<<28|c>>>4;d=(d>>>4|0)+1006632960|0;break a}if((d|0)==2146435072|d>>>0>2146435072){h=c<<28;c=(g&15)<<28|c>>>4;d=g>>>4|2147418112;break a}if(!(c|d)){c=0;d=0;break a}f=c;c=d?S(d):S(c)+32|0;Cb(e,f,d,0,0,c+49|0);i=J[e>>2];h=J[e+4>>2];f=15372-c<<16;c=J[e+8>>2];d=f|J[e+12>>2]^65536;}J[a>>2]=i;J[a+4>>2]=h;J[a+8>>2]=c;J[a+12>>2]=g&-2147483648|d;Ua=e+16|0;}function hm(a){a=a|0;var b=0,c=0,d=0,e=0;c=J[a+76>>2];if((c|0)==-1){c=a;d=Ua-16|0;Ua=d;a=J[a+28>>2];J[d+12>>2]=a;J[a+4>>2]=J[a+4>>2]+1;J[5996]=0;a=$(170,J[d+12>>2])|0;b=J[5996];J[5996]=0;a:{b:{if((b|0)==1){break b}J[5996]=0;a=_(171,a|0,32)|0;b=J[5996];J[5996]=0;if((b|0)==1){break b}b=J[d+12>>2];e=J[b+4>>2]-1|0;J[b+4>>2]=e;if((e|0)==-1){Xa[J[J[b>>2]+8>>2]](b);}Ua=d+16|0;break a}b=Z()|0;a=J[d+12>>2];c=J[a+4>>2]-1|0;J[a+4>>2]=c;if((c|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(b|0);B();}b=c;c=a;J[b+76>>2]=a;}return c<<24>>24}function ch(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0,h=0;c=J[a+4>>2];d=J[a+8>>2];if(c>>>0<d>>>0){J[c>>2]=b;J[a+4>>2]=c+4;return}a:{g=J[a>>2];e=c-g>>2;f=e+1|0;if(f>>>0<1073741824){h=e<<2;d=d-g|0;e=d>>1;e=d>>>0>=2147483644?1073741823:e>>>0>f>>>0?e:f;if(e){if(e>>>0>=1073741824){break a}f=cb(e<<2);}else {f=0;}d=h+f|0;J[d>>2]=b;b=d+4|0;if((c|0)!=(g|0)){while(1){d=d-4|0;c=c-4|0;J[d>>2]=J[c>>2];if((c|0)!=(g|0)){continue}break}c=J[a>>2];}J[a+8>>2]=f+(e<<2);J[a+4>>2]=b;J[a>>2]=d;if(c){$a(c);}return}nb();B();}Pb();B();}function Hm(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;var f=0,g=0;e=Ua-16|0;Ua=e;f=cb(104);J[f>>2]=7e3;J[f+4>>2]=0;J[f+8>>2]=0;J[e+4>>2]=J[c>>2];J[e+8>>2]=J[c+4>>2];J[e+12>>2]=J[c+8>>2];J[c+8>>2]=0;J[c>>2]=0;J[c+4>>2]=0;J[5996]=0;g=ia(163,f+12|0,b|0,e+4|0,d|0)|0;b=J[5996];J[5996]=0;if((b|0)!=1){d=J[e+4>>2];if(d){b=J[e+8>>2];if((d|0)!=(b|0)){while(1){c=b-12|0;if(H[b-1|0]<0){$a(J[c>>2]);}b=c;if((c|0)!=(d|0)){continue}break}}$a(d);}J[a+4>>2]=f;J[a>>2]=g;Ua=e+16|0;return}a=Z()|0;rb(e+4|0);$a(f);da(a|0);B();}function Yf(a,b,c,d){var e=0,f=0,g=0;e=Ua-16|0;Ua=e;a:{b:{if((a|0)!=(b|0)){c:{g=J[5732];J[5732]=0;jb();a=Uf(a,e+12|0,d);d=Wa;d:{f=J[5732];if(f){if(J[e+12>>2]!=(b|0)){break c}if((f|0)!=68){break d}J[c>>2]=4;b=2147483647;if(!!a&(d|0)>=0|(d|0)>0){break a}break b}J[5732]=g;if(J[e+12>>2]==(b|0)){break d}break c}if((d|0)<0&a>>>0<=2147483647|(d|0)<-1){J[c>>2]=4;break b}if((d|0)>=0&a>>>0>=2147483648|(d|0)>0){J[c>>2]=4;b=2147483647;break a}b=a;break a}}J[c>>2]=4;b=0;break a}b=-2147483648;}Ua=e+16|0;return b}function xc(a,b,c,d,e,f){var g=0,h=0,i=0,j=0;a:{if(f&64){c=f+-64|0;b=c&31;if((c&63)>>>0>=32){c=0;b=e>>>b|0;}else {c=e>>>b|0;b=((1<<b)-1&e)<<32-b|d>>>b;}d=0;e=0;break a}if(!f){break a}i=d;h=64-f|0;g=h&31;if((h&63)>>>0>=32){h=d<<g;j=0;}else {h=(1<<g)-1&i>>>32-g|e<<g;j=i<<g;}i=b;b=f&31;if((f&63)>>>0>=32){g=0;b=c>>>b|0;}else {g=c>>>b|0;b=((1<<b)-1&c)<<32-b|i>>>b;}b=j|b;c=g|h;g=d;d=f&31;if((f&63)>>>0>=32){h=0;d=e>>>d|0;}else {h=e>>>d|0;d=((1<<d)-1&e)<<32-d|g>>>d;}e=h;}J[a>>2]=b;J[a+4>>2]=c;J[a+8>>2]=d;J[a+12>>2]=e;}function Rc(a,b,c){var d=0,e=0,f=0;d=Ua-16|0;Ua=d;J[d+12>>2]=b;J[b+4>>2]=J[b+4>>2]+1;J[5996]=0;b=$(198,J[d+12>>2])|0;e=J[5996];J[5996]=0;a:{if((e|0)==1){break a}J[5996]=0;e=$(206,b|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){break a}J[c>>2]=e;J[5996]=0;aa(207,a|0,b|0);a=J[5996];J[5996]=0;if((a|0)==1){break a}a=J[d+12>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}Ua=d+16|0;return}c=Z()|0;a=J[d+12>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(c|0);B();}function Tc(a,b,c){var d=0,e=0,f=0;d=Ua-16|0;Ua=d;J[d+12>>2]=b;J[b+4>>2]=J[b+4>>2]+1;J[5996]=0;b=$(180,J[d+12>>2])|0;e=J[5996];J[5996]=0;a:{if((e|0)==1){break a}J[5996]=0;e=$(193,b|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){break a}H[c|0]=e;J[5996]=0;aa(194,a|0,b|0);a=J[5996];J[5996]=0;if((a|0)==1){break a}a=J[d+12>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}Ua=d+16|0;return}c=Z()|0;a=J[d+12>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(c|0);B();}function sj(a){if(K[25148]){return J[6286]}if(!K[26240]){H[26240]=1;}ib(25952,18680);ib(25964,18712);ib(25976,18748);ib(25988,18772);ib(26e3,18796);ib(26012,18812);ib(26024,18832);ib(26036,18852);ib(26048,18880);ib(26060,18920);ib(26072,18952);ib(26084,18988);ib(26096,19024);ib(26108,19040);ib(26120,19056);ib(26132,19072);ib(26144,18796);ib(26156,19088);ib(26168,19104);ib(26180,19120);ib(26192,19136);ib(26204,19152);ib(26216,19168);ib(26228,19184);H[25148]=1;J[6286]=25952;return 25952}function qn(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0;f=Gb(c);a:{d=K[a+11|0];e=d&127;d=(d&128)>>>7|0;e=d?J[a+4>>2]:e;if(e>>>0>=b>>>0){d=d?(J[a+8>>2]&2147483647)-1|0:10;if(d-e>>>0>=f>>>0){if(!f){break a}d=(K[a+11|0]&128)>>>7|0?J[a>>2]:a;h=d+b|0;if((b|0)!=(e|0)){g=b+d|0;oc(g+f|0,g,e-b|0);c=(c>>>0>=g>>>0?d+e>>>0>c>>>0?f:0:0)+c|0;}oc(h,c,f);b=f+e|0;b:{if((K[a+11|0]&128)>>>7|0){J[a+4>>2]=b;break b}H[a+11|0]=b;}H[b+d|0]=0;break a}dd(a,d,(f+e|0)-d|0,e,b,0,f,c);break a}ue();B();}return a|0}function hh(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0,h=0;e=J[a+4>>2];if((e|0)!=J[a+8>>2]){J[e>>2]=b;J[a+4>>2]=e+4;return}a:{g=J[a>>2];f=e-g|0;c=f>>2;d=c+1|0;if(d>>>0<1073741824){h=c<<2;c=f>>1;c=f>>>0>=2147483644?1073741823:c>>>0>d>>>0?c:d;if(c){if(c>>>0>=1073741824){break a}f=cb(c<<2);}else {f=0;}d=h+f|0;J[d>>2]=b;b=d+4|0;if((e|0)!=(g|0)){while(1){d=d-4|0;e=e-4|0;J[d>>2]=J[e>>2];if((e|0)!=(g|0)){continue}break}}J[a+8>>2]=f+(c<<2);J[a+4>>2]=b;J[a>>2]=d;if(g){$a(g);}return}nb();B();}Pb();B();}function Cb(a,b,c,d,e,f){var g=0,h=0,i=0;a:{if(f&64){e=f+-64|0;f=b;d=e&31;if((e&63)>>>0>=32){e=f<<d;d=0;}else {e=(1<<d)-1&f>>>32-d|c<<d;d=f<<d;}b=0;c=0;break a}if(!f){break a}h=d;g=f&31;if((f&63)>>>0>=32){i=d<<g;h=0;}else {i=(1<<g)-1&h>>>32-g|e<<g;h=h<<g;}g=b;e=64-f|0;d=e&31;if((e&63)>>>0>=32){e=0;d=c>>>d|0;}else {e=c>>>d|0;d=((1<<d)-1&c)<<32-d|g>>>d;}d=h|d;e=e|i;h=b;g=f&31;if((f&63)>>>0>=32){i=b<<g;b=0;}else {i=(1<<g)-1&h>>>32-g|c<<g;b=h<<g;}c=i;}J[a>>2]=b;J[a+4>>2]=c;J[a+8>>2]=d;J[a+12>>2]=e;}function vn(a){a=a|0;var b=0,c=0,d=0,e=0;e=J[a+44>>2];if(e){b=e;d=J[a+48>>2];if((b|0)!=(d|0)){while(1){d=d-8|0;c=J[d+4>>2];a:{if(!c){break a}b=J[c+4>>2];J[c+4>>2]=b-1;if(b){break a}Xa[J[J[c>>2]+8>>2]](c);bb(c);}if((d|0)!=(e|0)){continue}break}b=J[a+44>>2];}J[a+48>>2]=e;$a(b);}b=J[a+32>>2];if(b){J[a+36>>2]=b;$a(b);}c=J[a+20>>2];if(c){d=c;b=J[a+24>>2];if((c|0)!=(b|0)){while(1){e=b-12|0;if(H[b-1|0]<0){$a(J[e>>2]);}b=e;if((c|0)!=(b|0)){continue}break}d=J[a+20>>2];}J[a+24>>2]=c;$a(d);}}function Sl(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;var e=0,f=0;e=Ua-16|0;Ua=e;J[e>>2]=c;J[e+4>>2]=d;J[e+8>>2]=c;J[e+12>>2]=d;c=Ua-32|0;Ua=c;b=Kb(c+8|0,b);J[5996]=0;d=c+20|0;ea(174,d|0,e|0,b|0);f=J[5996];J[5996]=0;a:{b:{c:{if((f|0)!=1){J[5996]=0;a=_(175,a|0,d|0)|0;f=J[5996];J[5996]=0;if((f|0)==1){break c}ab(d);ab(b);J[a>>2]=21872;b=J[e+4>>2];J[a+8>>2]=J[e>>2];J[a+12>>2]=b;Ua=c+32|0;break a}a=Z()|0;break b}a=Z()|0;ab(c+20|0);}ab(b);da(a|0);B();}J[a>>2]=8492;Ua=e+16|0;return a|0}function ej(a){if(K[25140]){return J[6284]}if(!K[25936]){H[25936]=1;}hb(25648,1065);hb(25660,1056);hb(25672,2204);hb(25684,2058);hb(25696,1135);hb(25708,2381);hb(25720,1073);hb(25732,1297);hb(25744,1802);hb(25756,1785);hb(25768,1793);hb(25780,1812);hb(25792,2024);hb(25804,2543);hb(25816,1837);hb(25828,1683);hb(25840,1135);hb(25852,1965);hb(25864,2051);hb(25876,2280);hb(25888,1924);hb(25900,1386);hb(25912,1263);hb(25924,2539);H[25140]=1;J[6284]=25648;return 25648}function of(a,b,c,d,e){var f=0,g=0,h=0,i=0;f=Ua-16|0;Ua=f;if(1073741807-b>>>0>=c>>>0){h=(K[a+11|0]&128)>>>7|0?J[a>>2]:a;g=f+4|0;i=g;if(b>>>0<=536870886){J[f+12>>2]=b<<1;J[f+4>>2]=b+c;c=J[(M[g>>2]<M[f+12>>2]?f+12|0:g)>>2];if(c>>>0>=2){g=c+4&-4;c=g-1|0;c=(c|0)==2?g:c;}else {c=1;}c=c+1|0;}else {c=1073741807;}Fc(i,c);c=J[f+4>>2];if(e){tc(c,h,e);}if((d|0)!=(e|0)){g=e<<2;tc(g+c|0,g+h|0,d-e|0);}if((b|0)!=1){$a(h);}J[a>>2]=c;J[a+8>>2]=J[f+8>>2]|-2147483648;Ua=f+16|0;return}ub();B();}function Sd(a,b,c,d,e){var f=0,g=0,h=0,i=0;f=Ua-16|0;Ua=f;if(2147483631-b>>>0>=c>>>0){h=(K[a+11|0]&128)>>>7|0?J[a>>2]:a;g=f+4|0;i=g;if(b>>>0<=1073741798){J[f+12>>2]=b<<1;J[f+4>>2]=b+c;c=J[(M[g>>2]<M[f+12>>2]?f+12|0:g)>>2];if(c>>>0>=11){g=c+16&-16;c=g-1|0;c=(c|0)==11?g:c;}else {c=10;}c=c+1|0;}else {c=2147483631;}Gc(i,c);c=J[f+4>>2];if(e){gc(c,h,e);}if((d|0)!=(e|0)){gc(c+e|0,e+h|0,d-e|0);}if((b|0)!=10){$a(h);}J[a>>2]=c;J[a+8>>2]=J[f+8>>2]|-2147483648;Ua=f+16|0;return}ub();B();}function wm(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0;e=Ua-16|0;Ua=e;while(1){a:{if((c|0)<=(f|0)){break a}g=J[a+16>>2];d=J[a+12>>2];b:{if(g>>>0>d>>>0){J[e+12>>2]=2147483647;J[e+8>>2]=g-d;J[e+4>>2]=c-f;h=b;g=d;b=e+4|0;d=e+8|0;b=J[b>>2]<J[d>>2]?b:d;d=e+12|0;d=J[(J[b>>2]<J[d>>2]?b:d)>>2];b=gc(h,g,d);J[a+12>>2]=J[a+12>>2]+d;b=b+d|0;break b}d=Xa[J[J[a>>2]+40>>2]](a)|0;if((d|0)==-1){break a}H[b|0]=d<<24>>24;d=1;b=b+1|0;}f=d+f|0;continue}break}Ua=e+16|0;return f|0}function fh(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;var e=0,f=0,g=0;e=cb(56);a:{b:{if((c|0)==(d|0)){d=0;c=0;break b}f=d-c|0;if((f|0)<0){J[5996]=0;ca(8);a=J[5996];J[5996]=0;if((a|0)==1){break a}B();}J[5996]=0;d=$(3,f|0)|0;g=J[5996];J[5996]=0;if((g|0)==1){break a}c=fb(d,c,f)+f|0;}J[e+4>>2]=0;J[e+8>>2]=0;J[e>>2]=b;J[e+40>>2]=0;J[e+44>>2]=0;J[e+28>>2]=d;J[e+12>>2]=0;J[e+16>>2]=0;J[e+20>>2]=0;J[e+24>>2]=0;J[e+36>>2]=c;J[e+32>>2]=c;mc(a,e);return e|0}a=Z()|0;$a(e);da(a|0);B();}function Zb(a,b,c){var d=0,e=0;d=(c|0)!=0;a:{b:{c:{if(!(a&3)|!c){break c}e=b&255;while(1){if((e|0)==K[a|0]){break b}c=c-1|0;d=(c|0)!=0;a=a+1|0;if(!(a&3)){break c}if(c){continue}break}}if(!d){break a}d=b&255;if(!((d|0)==K[a|0]|c>>>0<4)){d=P(d,16843009);while(1){e=d^J[a>>2];if((e^-1)&e-16843009&-2139062144){break b}a=a+4|0;c=c-4|0;if(c>>>0>3){continue}break}}if(!c){break a}}b=b&255;while(1){if((b|0)==K[a|0]){return a}a=a+1|0;c=c-1|0;if(c){continue}break}}return 0}function ck(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0;a:{if(!((K[f+11|0]&128)>>>7|0)){b=J[f+4>>2];J[a>>2]=J[f>>2];J[a+4>>2]=b;J[a+8>>2]=J[f+8>>2];break a}e=J[f>>2];d=Ua-16|0;Ua=d;b:{c:{c=J[f+4>>2];d:{if(c>>>0<2){H[a+11|0]=c;break d}if(c>>>0>=1073741808){break c}g=d+8|0;if(c>>>0>=2){f=c+4&-4;b=f-1|0;b=(b|0)==2?f:b;}else {b=1;}Fc(g,b+1|0);b=J[d+8>>2];J[a>>2]=b;J[a+8>>2]=J[d+12>>2]|-2147483648;J[a+4>>2]=c;a=b;}tc(a,e,c+1|0);Ua=d+16|0;break b}ub();B();}}}function ud(a){var b=0,c=0;a:{b=J[a+4>>2];b=J[J[b>>2]-12>>2]+b|0;b:{if(!J[b+24>>2]|J[b+16>>2]|!(K[b+5|0]&32)){break b}if((Ha()|0)>0){break b}b=J[a+4>>2];b=J[(J[J[b>>2]-12>>2]+b|0)+24>>2];J[5996]=0;b=$(169,b|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){if((b|0)!=-1){break b}a=J[a+4>>2];b=J[J[a>>2]-12>>2];J[5996]=0;aa(138,a+b|0,1);a=J[5996];J[5996]=0;if((a|0)!=1){break b}}pa(fa(0)|0)|0;J[5996]=0;ca(110);a=J[5996];J[5996]=0;if((a|0)==1){break a}}return}fa(0)|0;Vb();B();}function Sf(a,b,c,d){var e=0,f=0,g=0,h=0;e=Ua-16|0;Ua=e;a:{b:{if((a|0)!=(b|0)){c:{d:{f=K[a|0];if((f|0)!=45){break d}a=a+1|0;if((b|0)!=(a|0)){break d}break c}h=J[5732];J[5732]=0;jb();a=ae(a,e+12|0,d);d=Wa;g=J[5732];e:{if(g){if(J[e+12>>2]!=(b|0)){break c}if(!d&a>>>0>65535|d|(g|0)==68){break e}break b}J[5732]=h;if(J[e+12>>2]!=(b|0)){break c}if(!d&a>>>0<65536){break b}}J[c>>2]=4;a=65535;break a}}J[c>>2]=4;a=0;break a}a=(f|0)==45?0-a|0:a;}Ua=e+16|0;return a&65535}function jf(a,b){var c=0,d=0,e=0,f=0,g=0,h=0;e=Ua-16|0;Ua=e;d=K[a+11|0];c=d<<24>>24;c=(c|0)<0;d=c?J[a+4>>2]:d;a:{if(!d){break a}c=c?J[a>>2]:a;g=K[c|0];f=g-9&255;if(f>>>0<24&(8388639>>>f&1)){break a}f=K[(c+d|0)-1|0]-9|0;if((f&255)>>>0<24&(8388639>>>f&1)){break a}if((g|0)==45|(Gb(c)|0)!=(d|0)){break a}J[e+12>>2]=0;J[5732]=0;a=hf(H[a+11|0]<0?J[a>>2]:a,e+12|0,10);c=Wa;if(b){J[b>>2]=a;J[b+4>>2]=c;}a=J[e+12>>2];if(!a|K[a|0]){break a}h=!J[5732];}Ua=e+16|0;return h}function Pf(a,b,c,d){var e=0,f=0,g=0,h=0;e=Ua-16|0;Ua=e;a:{if((a|0)!=(b|0)){b:{c:{f=K[a|0];if((f|0)!=45){break c}a=a+1|0;if((b|0)!=(a|0)){break c}break b}h=J[5732];J[5732]=0;jb();a=ae(a,e+12|0,d);d=Wa;d:{g=J[5732];if(g){if(J[e+12>>2]!=(b|0)){break b}if((g|0)!=68){break d}J[c>>2]=4;a=-1;b=-1;break a}J[5732]=h;if(J[e+12>>2]==(b|0)){break d}break b}b=a;c=(f|0)==45;a=c?0-b|0:b;b=c?0-(d+((b|0)!=0)|0)|0:d;break a}}J[c>>2]=4;a=0;b=0;}Ua=e+16|0;Wa=b;return a}function Qm(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0;d=Ua-16|0;Ua=d;J[a>>2]=7e3;J[a+4>>2]=0;J[a+8>>2]=0;J[d+4>>2]=J[c>>2];J[d+8>>2]=J[c+4>>2];J[d+12>>2]=J[c+8>>2];J[c+8>>2]=0;J[c>>2]=0;J[c+4>>2]=0;J[5996]=0;ia(163,a+12|0,b|0,d+4|0,0)|0;b=J[5996];J[5996]=0;if((b|0)!=1){e=J[d+4>>2];if(e){b=J[d+8>>2];if((e|0)!=(b|0)){while(1){c=b-12|0;if(H[b-1|0]<0){$a(J[c>>2]);}b=c;if((c|0)!=(e|0)){continue}break}}$a(e);}Ua=d+16|0;return a|0}a=Z()|0;rb(d+4|0);da(a|0);B();}function hd(a,b){var c=0,d=0,e=0,f=0,g=0;e=Ua-16|0;Ua=e;d=K[a+11|0];c=d<<24>>24;c=(c|0)<0;d=c?J[a+4>>2]:d;a:{if(!d){break a}c=c?J[a>>2]:a;f=K[c|0]-9&255;if(f>>>0<24&(8388639>>>f&1)){break a}f=K[(c+d|0)-1|0]-9|0;if((f&255)>>>0<24&(8388639>>>f&1)){break a}if((Gb(c)|0)!=(d|0)){break a}J[e+12>>2]=0;J[5732]=0;a=kf(H[a+11|0]<0?J[a>>2]:a,e+12|0,10);c=Wa;if(b){J[b>>2]=a;J[b+4>>2]=c;}a=J[e+12>>2];if(!a|K[a|0]){break a}g=!J[5732];}Ua=e+16|0;return g}function Sc(a,b){var c=0,d=0;c=Ua-16|0;Ua=c;J[c+12>>2]=a;J[a+4>>2]=J[a+4>>2]+1;J[5996]=0;a=$(197,J[c+12>>2])|0;d=J[5996];J[5996]=0;a:{b:{if((d|0)==1){break b}J[5996]=0;ga(208,a|0,12416,12442,b|0);a=J[5996];J[5996]=0;if((a|0)==1){break b}a=J[c+12>>2];d=J[a+4>>2]-1|0;J[a+4>>2]=d;if((d|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}Ua=c+16|0;break a}d=Z()|0;a=J[c+12>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(d|0);B();}return b}function Nk(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0;g=Ua-16|0;Ua=g;J[g+12>>2]=b;b=J[d+28>>2];J[g+8>>2]=b;J[b+4>>2]=J[b+4>>2]+1;J[5996]=0;h=$(170,J[g+8>>2])|0;b=J[5996];J[5996]=0;if((b|0)!=1){b=J[g+8>>2];d=J[b+4>>2]-1|0;J[b+4>>2]=d;if((d|0)==-1){Xa[J[J[b>>2]+8>>2]](b);}Ef(a,f+24|0,g+12|0,c,e,h);Ua=g+16|0;return J[g+12>>2]}c=Z()|0;a=J[g+8>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(c|0);B();}function Mk(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0;g=Ua-16|0;Ua=g;J[g+12>>2]=b;b=J[d+28>>2];J[g+8>>2]=b;J[b+4>>2]=J[b+4>>2]+1;J[5996]=0;h=$(170,J[g+8>>2])|0;b=J[5996];J[5996]=0;if((b|0)!=1){b=J[g+8>>2];d=J[b+4>>2]-1|0;J[b+4>>2]=d;if((d|0)==-1){Xa[J[J[b>>2]+8>>2]](b);}Df(a,f+16|0,g+12|0,c,e,h);Ua=g+16|0;return J[g+12>>2]}c=Z()|0;a=J[g+8>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(c|0);B();}function Hk(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0;g=Ua-16|0;Ua=g;J[g+12>>2]=b;b=J[d+28>>2];J[g+8>>2]=b;J[b+4>>2]=J[b+4>>2]+1;J[5996]=0;h=$(197,J[g+8>>2])|0;b=J[5996];J[5996]=0;if((b|0)!=1){b=J[g+8>>2];d=J[b+4>>2]-1|0;J[b+4>>2]=d;if((d|0)==-1){Xa[J[J[b>>2]+8>>2]](b);}Bf(a,f+24|0,g+12|0,c,e,h);Ua=g+16|0;return J[g+12>>2]}c=Z()|0;a=J[g+8>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(c|0);B();}function Gk(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0;g=Ua-16|0;Ua=g;J[g+12>>2]=b;b=J[d+28>>2];J[g+8>>2]=b;J[b+4>>2]=J[b+4>>2]+1;J[5996]=0;h=$(197,J[g+8>>2])|0;b=J[5996];J[5996]=0;if((b|0)!=1){b=J[g+8>>2];d=J[b+4>>2]-1|0;J[b+4>>2]=d;if((d|0)==-1){Xa[J[J[b>>2]+8>>2]](b);}Af(a,f+16|0,g+12|0,c,e,h);Ua=g+16|0;return J[g+12>>2]}c=Z()|0;a=J[g+8>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(c|0);B();}function pe(a,b){a:{if(a){if(b>>>0<=127){break a}b:{if(!J[J[6178]>>2]){if((b&-128)==57216){break a}break b}if(b>>>0<=2047){H[a+1|0]=b&63|128;H[a|0]=b>>>6|192;return 2}if(!((b&-8192)!=57344&b>>>0>=55296)){H[a+2|0]=b&63|128;H[a|0]=b>>>12|224;H[a+1|0]=b>>>6&63|128;return 3}if(b-65536>>>0<=1048575){H[a+3|0]=b&63|128;H[a|0]=b>>>18|240;H[a+2|0]=b>>>6&63|128;H[a+1|0]=b>>>12&63|128;return 4}}J[5732]=25;a=-1;}else {a=1;}return a}H[a|0]=b;return 1}function ic(a){var b=0,c=0,d=0,e=0,f=0;a:{d=a-24|0;a=d;if(a>>>0<27344&a>>>0>=26832){b=a-4|0;e=J[6705];c=e;while(1){a=c;if(!(!a|(a|0)==27344)){c=L[a+2>>1];if((a+(c<<2)|0)==(b|0)){I[a+2>>1]=c+L[d-2>>1];break a}if((a|0)==((L[b+2>>1]<<2)+b|0)){c=d-2|0;I[c>>1]=L[a+2>>1]+L[c>>1];if(!f){J[6705]=b;I[b>>1]=L[a>>1];break a}I[f>>1]=b-26832>>>2;break a}else {c=(L[a>>1]<<2)+26832|0;f=a;continue}}break}I[b>>1]=e-26832>>>2;J[6705]=b;break a}$a(d);}}function Rg(a,b,c,d,e){var f=0,g=0,h=0;h=-1;g=d&2147483647;f=(g|0)==2147418112;a:{if(f&!c?a|b:f&(c|0)!=0|g>>>0>2147418112){break a}f=e&2147483647;if(((f|0)==2147418112&0|f>>>0>2147418112)&(f|0)!=2147418112){break a}if(!(a|c|(f|g|b))){return 0}f=d&e;if((f|0)>0|(f|0)>=0){if(((c|0)!=0|(d|0)!=(e|0))&(d|0)<(e|0)){break a}return (a|c|(d^e|b))!=0}if(!c&(d|0)==(e|0)?a|b:(c|0)!=0&(d|0)>=(e|0)|(d|0)>(e|0)){break a}h=(a|c|(d^e|b))!=0;}return h}function Lk(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0;a=Ua-16|0;Ua=a;J[a+12>>2]=b;b=J[d+28>>2];J[a+8>>2]=b;J[b+4>>2]=J[b+4>>2]+1;J[5996]=0;g=$(170,J[a+8>>2])|0;b=J[5996];J[5996]=0;if((b|0)!=1){b=J[a+8>>2];d=J[b+4>>2]-1|0;J[b+4>>2]=d;if((d|0)==-1){Xa[J[J[b>>2]+8>>2]](b);}Cf(f+20|0,a+12|0,c,e,g);Ua=a+16|0;return J[a+12>>2]}c=Z()|0;a=J[a+8>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(c|0);B();}function Fk(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0;a=Ua-16|0;Ua=a;J[a+12>>2]=b;b=J[d+28>>2];J[a+8>>2]=b;J[b+4>>2]=J[b+4>>2]+1;J[5996]=0;g=$(197,J[a+8>>2])|0;b=J[5996];J[5996]=0;if((b|0)!=1){b=J[a+8>>2];d=J[b+4>>2]-1|0;J[b+4>>2]=d;if((d|0)==-1){Xa[J[J[b>>2]+8>>2]](b);}zf(f+20|0,a+12|0,c,e,g);Ua=a+16|0;return J[a+12>>2]}c=Z()|0;a=J[a+8>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(c|0);B();}function Hf(a,b,c){var d=0,e=0,f=0,g=0,h=0;e=a;g=Ua-16|0;Ua=g;a:{if(b>>>0<1073741808){b:{if(b>>>0<2){H[e+11|0]=b;break b}h=g+8|0;if(b>>>0>=2){f=b+4&-4;d=f-1|0;d=(d|0)==2?f:d;}else {d=1;}Fc(h,d+1|0);d=J[g+8>>2];J[e>>2]=d;J[e+8>>2]=J[g+12>>2]|-2147483648;J[e+4>>2]=b;e=d;}f=Ua-16|0;Ua=f;J[f+12>>2]=c;c=e;d=b;while(1){if(d){J[c>>2]=J[f+12>>2];d=d-1|0;c=c+4|0;continue}break}Ua=f+16|0;J[(b<<2)+e>>2]=0;Ua=g+16|0;break a}ub();B();}return a}function xb(a,b,c,d,e,f,g,h,i){var j=0,k=0,l=0,m=0;i=co(b,c,h,i);h=Wa;e=co(d,e,f,g);i=e+i|0;d=Wa+h|0;h=e>>>0>i>>>0?d+1|0:d;j=g;e=0;k=c;d=0;c=co(g,e,c,d);g=c+i|0;i=Wa+h|0;l=g;c=c>>>0>g>>>0?i+1|0:i;g=co(f,0,b,0);h=Wa;i=0;d=co(f,i,k,d);h=h+d|0;f=Wa+i|0;f=d>>>0>h>>>0?f+1|0:f;i=f+l|0;d=c;f=f>>>0>i>>>0?d+1|0:d;b=co(b,m,j,e)+h|0;e=Wa;e=b>>>0<h>>>0?e+1|0:e;h=e+i|0;i=f;J[a+8>>2]=h;J[a+12>>2]=e>>>0>h>>>0?i+1|0:i;J[a>>2]=g;J[a+4>>2]=b;}function Qf(a,b,c,d){var e=0,f=0,g=0,h=0;e=Ua-16|0;Ua=e;a:{b:{if((a|0)!=(b|0)){c:{d:{f=K[a|0];if((f|0)!=45){break d}a=a+1|0;if((b|0)!=(a|0)){break d}break c}h=J[5732];J[5732]=0;jb();a=ae(a,e+12|0,d);d=Wa;g=J[5732];e:{if(g){if(J[e+12>>2]!=(b|0)){break c}if((g|0)==68|d){break e}break b}J[5732]=h;if(J[e+12>>2]!=(b|0)){break c}if(!d){break b}}J[c>>2]=4;a=-1;break a}}J[c>>2]=4;a=0;break a}a=(f|0)==45?0-a|0:a;}Ua=e+16|0;return a}function ie(a,b){a=a|0;b=b|0;var c=0,d=0;a:{d=K[a+11|0];c=d&127;d=(d&128)>>>7|0;c=d?J[a+4>>2]:c;if(c>>>0<b>>>0){b=b-c|0;if(b){d=d?(J[a+8>>2]&2147483647)-1|0:10;c=K[a+11|0];c=(c&128)>>>7|0?J[a+4>>2]:c&127;if(d-c>>>0<b>>>0){Sd(a,d,c+(b-d|0)|0,c,c);}d=(K[a+11|0]&128)>>>7|0?J[a>>2]:a;og(d+c|0,b,0);b=b+c|0;b:{if((K[a+11|0]&128)>>>7|0){J[a+4>>2]=b;break b}H[a+11|0]=b;}H[b+d|0]=0;}break a}sd(a,(K[a+11|0]&128)>>>7|0?J[a>>2]:a,b);}}function qb(a,b,c){var d=0,e=0,f=0;e=Ua-16|0;Ua=e;J[b+4>>2]=J[b+4>>2]+1;f=a+8|0;d=e+12|0;J[d>>2]=b;a:{b=J[a+12>>2];a=J[a+8>>2];if(b-a>>2>>>0<=c>>>0){J[5996]=0;aa(274,f|0,c+1|0);a=J[5996];J[5996]=0;if((a|0)==1){break a}a=J[f>>2];}a=(c<<2)+a|0;b=J[a>>2];if(b){a=J[b+4>>2]-1|0;J[b+4>>2]=a;if((a|0)==-1){Xa[J[J[b>>2]+8>>2]](b);}a=J[f>>2]+(c<<2)|0;}b=J[d>>2];J[d>>2]=0;J[a>>2]=b;$e(d);Ua=e+16|0;return}a=Z()|0;$e(d);da(a|0);B();}function Im(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0;d=Ua-16|0;Ua=d;a:{b:{c:{if((b|0)!=(c|0)){c=c-b|0;if((c|0)<18){break c}}H[a+12|0]=0;H[a|0]=0;break b}if(c>>>0>=2147483632){break a}d:{if(c>>>0<=10){H[d+15|0]=c;e=d+4|0;break d}f=(c|15)+1|0;e=cb(f);J[d+12>>2]=f|-2147483648;J[d+4>>2]=e;J[d+8>>2]=c;}H[fb(e,b,c)+c|0]=0;J[a+8>>2]=J[d+12>>2];b=J[d+8>>2];J[a>>2]=J[d+4>>2];J[a+4>>2]=b;H[a+12|0]=1;}Ua=d+16|0;return}ub();B();}function Uh(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0;e=Ua-16|0;Ua=e;a:{f=J[b>>2];if(f){d=K[c+11|0];if((d&128)>>>7|0?J[c+4>>2]:d&127){Sg(c,5570);f=J[b>>2];}d=e+4|0;b=J[b+4>>2];Xa[J[J[b>>2]+24>>2]](d,b,f);J[5996]=0;_(290,c|0,d|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break a}ab(d);}b=J[c+4>>2];J[a>>2]=J[c>>2];J[a+4>>2]=b;J[a+8>>2]=J[c+8>>2];J[c>>2]=0;J[c+4>>2]=0;J[c+8>>2]=0;Ua=e+16|0;return}a=Z()|0;ab(e+4|0);da(a|0);B();}function Fb(a){var b=0,c=0,d=0;b=J[a+40>>2];if(b){J[a+44>>2]=b;$a(b);}c=J[a+28>>2];if(c){d=c;b=J[a+32>>2];if((c|0)!=(b|0)){while(1){d=b-12|0;if(H[b-1|0]<0){$a(J[d>>2]);}b=d;if((c|0)!=(b|0)){continue}break}d=J[a+28>>2];}J[a+32>>2]=c;$a(d);}b=J[a+16>>2];if(b){J[a+20>>2]=b;$a(b);}c=J[a+4>>2];if(c){b=c;d=J[a+8>>2];if((b|0)!=(d|0)){while(1){d=Fb(d-56|0);if((c|0)!=(d|0)){continue}break}b=J[a+4>>2];}J[a+8>>2]=c;$a(b);}return a}function zb(a,b,c,d){var e=0,f=0,g=0;a:{e=K[a+11|0];if(!((e&128)>>>7|0?J[a+4>>2]:e&127)|(c-b|0)<5){break a}md(b,c);e=c-4|0;c=(K[a+11|0]&128)>>>7|0?J[a>>2]:a;f=J[a+4>>2];a=K[a+11|0];g=c+((a&128)>>>7|0?f:a&127)|0;b:{while(1){c:{a=H[c|0];f=a-127|0;if(b>>>0>=e>>>0){break c}if((a|0)!=J[b>>2]&(f&255)>>>0>=130){break b}b=b+4|0;c=((g-c|0)>1)+c|0;continue}break}if((f&255)>>>0<130|a>>>0>J[e>>2]-1>>>0){break a}}J[d>>2]=4;}}function Xc(a){var b=0,c=0,d=0,e=0,f=0,g=0,h=0;d=J[a>>2];if(d){b=d;c=J[a+4>>2];if((b|0)!=(c|0)){while(1){f=c-12|0;e=J[f>>2];if(e){b=e;g=c-8|0;c=J[g>>2];if((b|0)!=(c|0)){while(1){c=c-32|0;b=J[c+4>>2];a:{if(!b){break a}h=J[b+4>>2];J[b+4>>2]=h-1;if(h){break a}Xa[J[J[b>>2]+8>>2]](b);bb(b);}if((c|0)!=(e|0)){continue}break}b=J[f>>2];}J[g>>2]=e;$a(b);}c=f;if((d|0)!=(c|0)){continue}break}b=J[a>>2];}J[a+4>>2]=d;$a(b);}}function td(a,b,c){var d=0,e=0,f=0,g=0,h=0,i=0;d=a;g=Ua-16|0;Ua=g;a:{f=c-b|0;if(f>>>0<2147483632){b:{if(f>>>0<11){H[d+11|0]=f;break b}i=g+8|0;if(f>>>0>=11){h=f+16&-16;e=h-1|0;e=(e|0)==11?h:e;}else {e=10;}Gc(i,e+1|0);e=J[g+8>>2];J[d>>2]=e;J[d+8>>2]=J[g+12>>2]|-2147483648;J[d+4>>2]=f;d=e;}while(1){if((b|0)!=(c|0)){H[d|0]=K[b|0];d=d+1|0;b=b+1|0;continue}break}H[d|0]=0;Ua=g+16|0;break a}ub();B();}return a}function ag(a,b,c){var d=0,e=0,f=0,g=0,h=0,i=0;d=a;g=Ua-16|0;Ua=g;a:{f=c-b>>2;if(f>>>0<1073741808){b:{if(f>>>0<2){H[d+11|0]=f;break b}i=g+8|0;if(f>>>0>=2){h=f+4&-4;e=h-1|0;e=(e|0)==2?h:e;}else {e=1;}Fc(i,e+1|0);e=J[g+8>>2];J[d>>2]=e;J[d+8>>2]=J[g+12>>2]|-2147483648;J[d+4>>2]=f;d=e;}while(1){if((b|0)!=(c|0)){J[d>>2]=J[b>>2];d=d+4|0;b=b+4|0;continue}break}J[d>>2]=0;Ua=g+16|0;break a}ub();B();}return a}
function pm(a,b){a=a|0;b=b|0;var c=0,d=0;b=!J[a+24>>2]|(J[a+16>>2]|b);J[a+16>>2]=b;if(b&J[a+20>>2]){a=Ua-16|0;Ua=a;c=Rb(16);a=a+8|0;b=Ua-16|0;Ua=b;if(!K[24716]){H[24716]=1;}J[b+12>>2]=22908;J[b+8>>2]=1;d=J[b+12>>2];J[a>>2]=J[b+8>>2];J[a+4>>2]=d;Ua=b+16|0;b=J[a+4>>2];a=J[a>>2];J[5996]=0;a=Ja(176,c|0,1821,a|0,b|0)|0;b=J[5996];J[5996]=0;if((b|0)!=1){na(a|0,8576,177);B();}a=Z()|0;ic(c);da(a|0);B();}}function tm(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0;e=Ua-16|0;Ua=e;while(1){a:{if((c|0)<=(f|0)){break a}d=J[a+24>>2];g=J[a+28>>2];if(d>>>0>=g>>>0){if((Xa[J[J[a>>2]+52>>2]](a,K[b|0])|0)==-1){break a}f=f+1|0;b=b+1|0;}else {J[e+12>>2]=g-d;J[e+8>>2]=c-f;h=d;d=e+8|0;g=e+12|0;d=J[(J[d>>2]<J[g>>2]?d:g)>>2];gc(h,b,d);J[a+24>>2]=d+J[a+24>>2];f=d+f|0;b=b+d|0;}continue}break}Ua=e+16|0;return f|0}function go(a,b,c){var d=0,e=0,f=0,g=0;g=c&63;f=g;e=f&31;if(f>>>0>=32){f=-1>>>e|0;}else {d=-1>>>e|0;f=d|(1<<e)-1<<32-e;}f=f&a;d=b&d;e=g&31;if(g>>>0>=32){d=f<<e;g=0;}else {d=(1<<e)-1&f>>>32-e|d<<e;g=f<<e;}f=d;e=0-c&63;d=e&31;if(e>>>0>=32){d=-1<<d;c=0;}else {c=-1<<d;d=c|(1<<d)-1&-1>>>32-d;}a=c&a;b=b&d;d=e&31;if(e>>>0>=32){c=0;a=b>>>d|0;}else {c=b>>>d|0;a=((1<<d)-1&b)<<32-d|a>>>d;}a=a|g;Wa=c|f;return a}
function wb(a,b,c){var d=0,e=0,f=0;if(!(K[a|0]&32)){a:{d=b;b=a;a=J[b+16>>2];b:{if(!a){if(Be(b)){break b}a=J[b+16>>2];}f=J[b+20>>2];if(a-f>>>0<c>>>0){Xa[J[b+36>>2]](b,d,c)|0;break a}c:{if(J[b+80>>2]<0){break c}a=c;while(1){e=a;if(!a){break c}a=a-1|0;if(K[d+a|0]!=10){continue}break}if(Xa[J[b+36>>2]](b,d,e)>>>0<e>>>0){break b}d=d+e|0;c=c-e|0;f=J[b+20>>2];}fb(f,d,c);J[b+20>>2]=J[b+20>>2]+c;}}}}function am(a,b){a=a|0;b=b|0;var c=0,d=0,e=0,f=0,g=0;e=J[a+44>>2];c=J[a+24>>2];if(e>>>0<c>>>0){J[a+44>>2]=c;e=c;}c=J[a+12>>2];f=J[a+8>>2];d=-1;a:{if(c>>>0<=f>>>0){break a}if((b|0)==-1){J[a+16>>2]=e;J[a+12>>2]=c-1;J[a+8>>2]=f;return ((b|0)!=-1?b:0)|0}g=b<<24>>24;if(!(J[a+48>>2]&16)){d=-1;if((g|0)!=H[c-1|0]){break a}}J[a+16>>2]=e;J[a+12>>2]=c-1;J[a+8>>2]=f;H[J[a+12>>2]]=g;d=b;}return d|0}function Yh(a,b,c){a=a|0;b=b|0;c=+c;var d=0,e=0,f=0;f=Ua-16|0;Ua=f;d=K[b+11|0];d=(d&128)>>>7|0?J[b+4>>2]:d&127;while(1){a:{e=(K[b+11|0]&128)>>>7|0?J[b>>2]:b;O[f>>3]=c;e=od(e,d+1|0,2314,f);b:{if((e|0)>=0){if(d>>>0>=e>>>0){break a}d=e;break b}d=d<<1|1;}ie(b,d);continue}break}ie(b,e);d=J[b+4>>2];J[a>>2]=J[b>>2];J[a+4>>2]=d;J[a+8>>2]=J[b+8>>2];J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;Ua=f+16|0;}function Of(a,b,c){var d=0,e=0,f=Q(0),g=0,h=Q(0);e=Ua-16|0;Ua=e;a:{b:{c:{if((a|0)!=(b|0)){g=J[5732];J[5732]=0;jb();d=Ua-16|0;Ua=d;ee(d,a,e+12|0,0);f=Vf(J[d>>2],J[d+4>>2],J[d+8>>2],J[d+12>>2]);Ua=d+16|0;a=J[5732];if(!a){break c}if(J[e+12>>2]!=(b|0)){break b}h=f;if((a|0)!=68){break a}break b}J[c>>2]=4;break a}J[5732]=g;if(J[e+12>>2]==(b|0)){break a}}J[c>>2]=4;f=h;}Ua=e+16|0;return f}function Mf(a,b,c){var d=0,e=0,f=0,g=0,h=0;e=Ua-16|0;Ua=e;a:{b:{c:{if((a|0)!=(b|0)){g=J[5732];J[5732]=0;jb();d=Ua-16|0;Ua=d;ee(d,a,e+12|0,1);f=ge(J[d>>2],J[d+4>>2],J[d+8>>2],J[d+12>>2]);Ua=d+16|0;a=J[5732];if(!a){break c}if(J[e+12>>2]!=(b|0)){break b}h=f;if((a|0)!=68){break a}break b}J[c>>2]=4;break a}J[5732]=g;if(J[e+12>>2]==(b|0)){break a}}J[c>>2]=4;f=h;}Ua=e+16|0;return f}function mk(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0;e=(K[a+11|0]&128)>>>7|0?(J[a+8>>2]&2147483647)-1|0:1;f=K[a+11|0];d=f;d=(d&128)>>>7|0?J[a+4>>2]:d&127;a:{if(e-d>>>0>=c>>>0){if(!c){break a}e=(f&128)>>>7|0?J[a>>2]:a;tc(e+(d<<2)|0,b,c);b=c+d|0;b:{if((K[a+11|0]&128)>>>7|0){J[a+4>>2]=b;break b}H[a+11|0]=b;}J[e+(b<<2)>>2]=0;return a|0}He(a,e,d+(c-e|0)|0,d,d,0,c,b);}return a|0}function Bi(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;var f=0,g=0,h=0,i=0,j=0,k=0;while(1){a:{if((c|0)==(d|0)|e>>>0<=i>>>0){break a}g=1;h=Ua-16|0;Ua=h;f=Xb(h+12|0,J[a+8>>2]);j=qd(0,c,d-c|0,b?b:24840);f=J[f>>2];if(f){J[6178]=(f|0)==-1?24816:f;}Ua=h+16|0;b:{switch(j+2|0){default:g=j;break;case 0:case 1:break a;case 2:break b}}i=i+1|0;k=g+k|0;c=c+g|0;continue}break}return k|0}function kd(a,b,c){var d=0,e=0;if(c&2048){H[a|0]=43;a=a+1|0;}if(c&1024){H[a|0]=35;a=a+1|0;}d=c&260;if((d|0)!=260){H[a|0]=46;H[a+1|0]=42;a=a+2|0;}c=c&16384;while(1){e=K[b|0];if(e){H[a|0]=e;a=a+1|0;b=b+1|0;continue}break}a:{b:{if((d|0)!=256){if((d|0)!=4){break b}b=c?70:102;break a}b=c?69:101;break a}b=c?65:97;if((d|0)==260){break a}b=c?71:103;}H[a|0]=b;return (d|0)!=260}function ve(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0;e=(K[a+11|0]&128)>>>7|0?(J[a+8>>2]&2147483647)-1|0:10;f=K[a+11|0];d=f;d=(d&128)>>>7|0?J[a+4>>2]:d&127;a:{if(e-d>>>0>=c>>>0){if(!c){break a}e=(f&128)>>>7|0?J[a>>2]:a;gc(e+d|0,b,c);b=c+d|0;b:{if((K[a+11|0]&128)>>>7|0){J[a+4>>2]=b;break b}H[a+11|0]=b;}H[b+e|0]=0;return a|0}dd(a,e,d+(c-e|0)|0,d,d,0,c,b);}return a|0}function Gh(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;if(Jb(a,J[b+8>>2],e)){if(!(J[b+28>>2]==1|J[b+4>>2]!=(c|0))){J[b+28>>2]=d;}return}a:{if(!Jb(a,J[b>>2],e)){break a}if(!(J[b+16>>2]!=(c|0)&J[b+20>>2]!=(c|0))){if((d|0)!=1){break a}J[b+32>>2]=1;return}J[b+20>>2]=c;J[b+32>>2]=d;J[b+40>>2]=J[b+40>>2]+1;if(!(J[b+36>>2]!=1|J[b+24>>2]!=2)){H[b+54|0]=1;}J[b+44>>2]=4;}}function uf(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0,i=0;e=Ua-16|0;Ua=e;i=J[b>>2];d=J[c>>2];g=J[a>>2];f=d-g|0;f=f>>>0>=2147483647?-1:(d|0)==(g|0)?4:f<<1;d=J[a+4>>2]==220;h=Vc(d?0:g,f);if(h){if(!d){J[a>>2]=0;}J[e+4>>2]=184;d=e+8|0;J[d>>2]=h;J[d+4>>2]=J[e+4>>2];a=rf(a,d);db(d);J[b>>2]=J[a>>2]+(i-g|0);J[c>>2]=J[a>>2]+(f&-4);Ua=e+16|0;return}ce();B();}function gn(a,b){a=a|0;b=+b;var c=0,d=0,e=0;d=Ua-16|0;Ua=d;c=d+4|0;J[c>>2]=0;J[c+4>>2]=0;J[c+8>>2]=0;e=(K[c+11|0]&128)>>>7|0?(J[c+8>>2]&2147483647)-1|0:10;J[5996]=0;aa(172,c|0,e|0);e=J[5996];J[5996]=0;if((e|0)==1){a=Z()|0;ab(c);da(a|0);B();}J[5996]=0;Pa(120,a|0,c|0,+b);a=J[5996];J[5996]=0;if((a|0)!=1){ab(c);Ua=d+16|0;return}a=Z()|0;ab(d+4|0);da(a|0);B();}function fd(a,b){var c=0,d=0,e=0,f=0,g=0,h=0;c=K[a+11|0];d=c<<24>>24;f=J[b+4>>2];d=(d|0)<0;c=d?J[a+4>>2]:c;a:{if(f>>>0<c+2>>>0){break a}e=J[b>>2];g=c+e|0;if(K[g|0]!=40|K[(e+f|0)-1|0]!=41){break a}if(c){a=d?J[a>>2]:a;d=c+a|0;while(1){if(K[a|0]!=K[e|0]){return 0}e=e+1|0;a=a+1|0;if((d|0)!=(a|0)){continue}break}}J[b+4>>2]=(f-c|0)-2;h=1;J[b>>2]=g+1;}return h}function Tf(a,b,c,d){var e=0,f=0,g=0;e=Ua-16|0;Ua=e;a:{if((a|0)!=(b|0)){g=J[5732];J[5732]=0;jb();a=Uf(a,e+12|0,d);d=Wa;f=J[5732];b:{if(f){if(J[e+12>>2]!=(b|0)){break b}if((f|0)!=68){break a}J[c>>2]=4;b=!!a&(d|0)>=0|(d|0)>0;a=b?-1:0;d=b?2147483647:-2147483648;break a}J[5732]=g;if(J[e+12>>2]==(b|0)){break a}}}J[c>>2]=4;a=0;d=0;}Ua=e+16|0;Wa=d;return a}function Ol(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0;e=J[a+84>>2];f=J[e>>2];d=J[e+4>>2];h=J[a+28>>2];g=J[a+20>>2]-h|0;g=d>>>0<g>>>0?d:g;if(g){fb(f,h,g);f=g+J[e>>2]|0;J[e>>2]=f;d=J[e+4>>2]-g|0;J[e+4>>2]=d;}d=c>>>0>d>>>0?d:c;if(d){fb(f,b,d);f=d+J[e>>2]|0;J[e>>2]=f;J[e+4>>2]=J[e+4>>2]-d;}H[f|0]=0;b=J[a+44>>2];J[a+28>>2]=b;J[a+20>>2]=b;return c|0}function xk(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0,h=0,i=0;e=Ua-16|0;Ua=e;i=J[b>>2];g=J[a>>2];d=J[c>>2]-g|0;f=d<<1;f=d>>>0>=2147483647?-1:f>>>0<=1?1:f;d=J[a+4>>2]==220;h=Vc(d?0:g,f);if(h){if(!d){J[a>>2]=0;}J[e+4>>2]=184;d=e+8|0;J[d>>2]=h;J[d+4>>2]=J[e+4>>2];a=rf(a,d);db(d);J[b>>2]=J[a>>2]+(i-g|0);J[c>>2]=f+J[a>>2];Ua=e+16|0;return}ce();B();}function Hd(a,b,c,d){H[a+53|0]=1;a:{if(J[a+4>>2]!=(c|0)){break a}H[a+52|0]=1;c=J[a+16>>2];b:{if(!c){J[a+36>>2]=1;J[a+24>>2]=d;J[a+16>>2]=b;if((d|0)!=1){break a}if(J[a+48>>2]==1){break b}break a}if((b|0)==(c|0)){c=J[a+24>>2];if((c|0)==2){J[a+24>>2]=d;c=d;}if(J[a+48>>2]!=1){break a}if((c|0)==1){break b}break a}J[a+36>>2]=J[a+36>>2]+1;}H[a+54|0]=1;}}function Oh(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0;d=Ua+-64|0;Ua=d;e=1;a:{if(Jb(a,b,0)){break a}e=0;if(!b){break a}b=Fe(b,22104);e=0;if(!b){break a}Sb(d+12|0,0,52);J[d+56>>2]=1;J[d+20>>2]=-1;J[d+16>>2]=a;J[d+8>>2]=b;Xa[J[J[b>>2]+28>>2]](b,d+8|0,J[c>>2],1);a=J[d+32>>2];if((a|0)==1){J[c>>2]=J[d+24>>2];}e=(a|0)==1;}a=e;Ua=d- -64|0;return a|0}function tj(a){if(K[25132]){return J[6282]}if(!K[25640]){H[25640]=1;}ib(25472,18340);ib(25484,18368);ib(25496,18396);ib(25508,18428);ib(25520,18468);ib(25532,18504);ib(25544,18532);ib(25556,18568);ib(25568,18584);ib(25580,18600);ib(25592,18616);ib(25604,18632);ib(25616,18648);ib(25628,18664);H[25132]=1;J[6282]=25472;return 25472}function en(a){a=a|0;var b=0,c=0;b=J[a+80>>2];a:{if(!b|M[a+84>>2]>3600){break a}if(Wb(3557,1)&(b^-1)){break a}b=J[a+80>>2];if(!b|M[a+84>>2]>3600|(J[a+52>>2]+J[a+44>>2]>>>0>201|M[a+68>>2]>100)){break a}if(Wb(2049,1)&(b^-1)){break a}b=J[a+80>>2];if(Wb(2165,1)&(b^-1)|K[a+88|0]){break a}a=J[a+80>>2];c=!(Wb(1681,1)&(a^-1));}return c|0}function Zc(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0,g=0;e=Ua-16|0;Ua=e;a:{b:{if(c>>>0<11){H[a+11|0]=c;break b}if(c>>>0>=2147483632){break a}g=e+8|0;if(c>>>0>=11){f=c+16&-16;d=f-1|0;d=(d|0)==11?f:d;}else {d=10;}Gc(g,d+1|0);d=J[e+8>>2];J[a>>2]=d;J[a+8>>2]=J[e+12>>2]|-2147483648;J[a+4>>2]=c;a=d;}gc(a,b,c+1|0);Ua=e+16|0;return}ub();B();}function ib(a,b){var c=0,d=0,e=0,f=0;a:{c=We(b);d=(K[a+11|0]&128)>>>7|0?(J[a+8>>2]&2147483647)-1|0:1;if(c>>>0<=d>>>0){b=eb((K[a+11|0]&128)>>>7|0?J[a>>2]:a,b,c<<2);b:{if((K[a+11|0]&128)>>>7|0){J[a+4>>2]=c;break b}H[a+11|0]=c;}J[(c<<2)+b>>2]=0;break a}e=a;f=J[a+4>>2];a=K[a+11|0];a=(a&128)>>>7|0?f:a&127;He(e,d,c-d|0,a,0,a,c,b);}}function Qg(a){var b=0,c=0,d=0;d=J[a>>2];if(d){b=J[a+4>>2];c=d;if((b|0)!=(c|0)){while(1){c=b-32|0;if((K[c|0]|K[c+1|0]<<8|(K[c+2|0]<<16|K[c+3|0]<<24))>>>0>=29){b=b-24|0;$a(K[b|0]|K[b+1|0]<<8|(K[b+2|0]<<16|K[b+3|0]<<24));H[b|0]=0;H[b+1|0]=0;H[b+2|0]=0;H[b+3|0]=0;}b=c;if((d|0)!=(b|0)){continue}break}c=J[a>>2];}J[a+4>>2]=d;$a(c);}}function Ei(a,b){a=a|0;b=b|0;var c=0,d=0;H[a+6|0]=0;I[a+4>>1]=0;J[a>>2]=1;c=J[b>>2];d=J[b+4>>2];J[a+20>>2]=0;J[a+12>>2]=0;J[a+16>>2]=0;J[a+8>>2]=(d-c|0)+1;c=cb(12);J[a+16>>2]=c;J[a+12>>2]=c;d=c+12|0;J[a+20>>2]=d;J[c>>2]=J[b>>2];J[c+4>>2]=J[b+4>>2];J[c+8>>2]=J[b+8>>2];J[b+8>>2]=0;J[b>>2]=0;J[b+4>>2]=0;J[a+16>>2]=d;return a|0}function dh(a,b,c,d){a=a|0;b=b|0;c=c|0;d=+d;var e=0,f=0,g=0,h=0;e=cb(56);f=J[c+8>>2];J[c+8>>2]=0;g=J[c>>2];h=J[c+4>>2];J[c>>2]=0;J[c+4>>2]=0;J[e+16>>2]=0;J[e+20>>2]=0;J[e+12>>2]=f;J[e+4>>2]=g;J[e+8>>2]=h;J[e>>2]=b;J[e+24>>2]=0;J[e+28>>2]=0;J[e+32>>2]=0;J[e+36>>2]=0;J[e+40>>2]=0;J[e+44>>2]=0;O[e+48>>3]=d;mc(a,e);return e|0}function fj(a){if(K[25124]){return J[6280]}if(!K[25464]){H[25464]=1;}hb(25296,1114);hb(25308,1121);hb(25320,1087);hb(25332,1095);hb(25344,1078);hb(25356,1128);hb(25368,1105);hb(25380,1961);hb(25392,1999);hb(25404,2336);hb(25416,2493);hb(25428,1273);hb(25440,2172);hb(25452,1390);H[25124]=1;J[6280]=25296;return 25296}function zc(a,b){var c=0,d=0,e=0,f=0,g=0;a:{c=We(b);e=Ua-16|0;Ua=e;if(c>>>0<1073741808){b:{if(c>>>0<2){H[a+11|0]=c;break b}g=e+8|0;if(c>>>0>=2){f=c+4&-4;d=f-1|0;d=(d|0)==2?f:d;}else {d=1;}Fc(g,d+1|0);d=J[e+8>>2];J[a>>2]=d;J[a+8>>2]=J[e+12>>2]|-2147483648;J[a+4>>2]=c;a=d;}J[tc(a,b,c)+(c<<2)>>2]=0;Ua=e+16|0;break a}ub();B();}}function dc(a,b,c,d){var e=0,f=0,g=0,h=0;e=Ua-16|0;Ua=e;J[e+12>>2]=d;g=Xb(e+8|0,b);b=Ua-16|0;Ua=b;d=J[e+12>>2];J[b+12>>2]=d;J[b+8>>2]=d;f=-1;d=pd(0,0,c,d);a:{if((d|0)<0){break a}h=a;d=d+1|0;a=sb(d);J[h>>2]=a;if(!a){break a}f=pd(a,d,c,J[b+12>>2]);}Ua=b+16|0;a=J[g>>2];if(a){J[6178]=(a|0)==-1?24816:a;}Ua=e+16|0;return f}function bh(a,b,c){a=a|0;b=b|0;c=+c;var d=0,e=0,f=0,g=0;d=cb(56);e=J[b+8>>2];J[b+8>>2]=0;f=J[b>>2];g=J[b+4>>2];J[b>>2]=0;J[b+4>>2]=0;J[d+16>>2]=0;J[d+20>>2]=0;J[d+12>>2]=e;J[d+4>>2]=f;J[d+8>>2]=g;J[d>>2]=12;J[d+24>>2]=0;J[d+28>>2]=0;J[d+32>>2]=0;J[d+36>>2]=0;J[d+40>>2]=0;J[d+44>>2]=0;O[d+48>>3]=c;mc(a,d);return d|0}function xe(a){var b=0;b=1;a:{if((a|0)>=1024){b=898846567431158e293;if(a>>>0<2047){a=a-1023|0;break a}b=Infinity;a=((a|0)>=3069?3069:a)-2046|0;break a}if((a|0)>-1023){break a}b=2004168360008973e-307;if(a>>>0>4294965304){a=a+969|0;break a}b=0;a=((a|0)<=-2960?-2960:a)+1938|0;}x(0,0);x(1,a+1023<<20);return b*+z()}function xd(a){var b=0,c=0,d=0,e=0;if(a){xd(J[a>>2]);xd(J[a+4>>2]);d=J[a+40>>2];if(d){b=d;c=J[a+44>>2];if((b|0)!=(c|0)){while(1){c=c-32|0;b=J[c+4>>2];a:{if(!b){break a}e=J[b+4>>2];J[b+4>>2]=e-1;if(e){break a}Xa[J[J[b>>2]+8>>2]](b);bb(b);}if((d|0)!=(c|0)){continue}break}b=J[a+40>>2];}J[a+44>>2]=d;$a(b);}$a(a);}}function Di(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;var f=0,g=0;g=Ua-16|0;Ua=g;J[e>>2]=c;b=g+12|0;a=Md(b,0,J[a+8>>2]);f=2;a:{if(a+1>>>0<2){break a}c=a-1|0;f=1;if(c>>>0>d-J[e>>2]>>>0){break a}while(1){if(c){a=K[b|0];d=J[e>>2];J[e>>2]=d+1;H[d|0]=a;c=c-1|0;b=b+1|0;continue}break}f=0;}a=f;Ua=g+16|0;return a|0}function rc(a,b,c,d){var e=0;a:{if(!(d&2048)){break a}e=d&74;if(!c|((e|0)==8|(e|0)==64)){break a}H[a|0]=43;a=a+1|0;}if(d&512){H[a|0]=35;a=a+1|0;}while(1){e=K[b|0];if(e){H[a|0]=e;a=a+1|0;b=b+1|0;continue}break}e=d&74;b=111;b:{if((e|0)==64){break b}b=d&16384?88:120;if((e|0)==8){break b}b=c?100:117;}H[a|0]=b;}function ad(a){var b=0,c=0,d=0;b=J[a>>2];J[a>>2]=0;if(b){a=J[b+28>>2];if(a){J[b+32>>2]=a;$a(a);}c=J[b+16>>2];if(c){a=c;d=J[b+20>>2];if((a|0)!=(d|0)){while(1){a=d-12|0;if(H[d-1|0]<0){$a(J[a>>2]);}d=a;if((a|0)!=(c|0)){continue}break}a=J[b+16>>2];}J[b+20>>2]=c;$a(a);}a=J[b+4>>2];if(a){J[b+8>>2]=a;$a(a);}$a(b);}}function Om(a,b){a=a|0;b=b|0;var c=0,d=0,e=0;J[a+8>>2]=0;J[a>>2]=0;J[a+4>>2]=0;J[5996]=0;c=$(3,12)|0;d=J[5996];J[5996]=0;if((d|0)!=1){J[a>>2]=c;d=c+12|0;J[a+8>>2]=d;J[c+8>>2]=J[b+8>>2];e=J[b+4>>2];J[c>>2]=J[b>>2];J[c+4>>2]=e;J[b>>2]=0;J[b+4>>2]=0;J[b+8>>2]=0;J[a+4>>2]=d;return}b=Z()|0;rb(a);da(b|0);B();}function Jk(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0;g=Ua-32|0;Ua=g;h=J[3163];J[g+24>>2]=J[3162];J[g+28>>2]=h;h=J[3161];J[g+16>>2]=J[3160];J[g+20>>2]=h;h=J[3159];J[g+8>>2]=J[3158];J[g+12>>2]=h;h=J[3157];J[g>>2]=J[3156];J[g+4>>2]=h;h=a;a=g+32|0;b=jc(h,b,c,d,e,f,g,a);Ua=a;return b|0}function bm(a){a=a|0;var b=0,c=0,d=0,e=0;c=J[a+44>>2];b=J[a+24>>2];if(c>>>0<b>>>0){J[a+44>>2]=b;c=b;}d=-1;a:{if(!(K[a+48|0]&8)){break a}b=J[a+16>>2];if(b>>>0<c>>>0){e=J[a+8>>2];b=J[a+12>>2];J[a+16>>2]=c;J[a+12>>2]=b;J[a+8>>2]=e;b=J[a+16>>2];}a=J[a+12>>2];if(a>>>0>=b>>>0){break a}d=K[a|0];}return d|0}function Ih(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;var e=0,f=0;if(Jb(a,J[b+8>>2],0)){Gd(b,c,d);return}f=J[a+12>>2];Ee(J[a+16>>2],J[a+20>>2],b,c,d);e=a+24|0;a=(a+16|0)+(f<<3)|0;a:{if(e>>>0>=a>>>0){break a}while(1){Ee(J[e>>2],J[e+4>>2],b,c,d);if(K[b+54|0]){break a}e=e+8|0;if(a>>>0>e>>>0){continue}break}}}function vb(a,b){var c=0,d=0,e=0,f=0,g=0;d=K[a+11|0];c=d<<24>>24;c=(c|0)<0;e=c?J[a+4>>2]:d;f=J[b+4>>2];if(e>>>0<=f>>>0){d=J[b>>2];if(e){a=c?J[a>>2]:a;g=e+a|0;c=d;while(1){if(K[a|0]!=K[c|0]){return 0}c=c+1|0;a=a+1|0;if((g|0)!=(a|0)){continue}break}}J[b+4>>2]=f-e;J[b>>2]=d+e;a=1;}else {a=0;}return a}function Gb(a){var b=0,c=0,d=0;a:{b:{b=a;if(!(b&3)){break b}if(!K[b|0]){return 0}while(1){b=b+1|0;if(!(b&3)){break b}if(K[b|0]){continue}break}break a}while(1){c=b;b=b+4|0;d=J[c>>2];if(!((d^-1)&d-16843009&-2139062144)){continue}break}while(1){b=c;c=b+1|0;if(K[b|0]){continue}break}}return b-a|0}function Eb(a,b,c){var d=0,e=0;a:{b:{if(c>>>0>=4){if((a|b)&3){break b}while(1){if(J[a>>2]!=J[b>>2]){break b}b=b+4|0;a=a+4|0;c=c-4|0;if(c>>>0>3){continue}break}}if(!c){break a}}while(1){d=K[a|0];e=K[b|0];if((d|0)==(e|0)){b=b+1|0;a=a+1|0;c=c-1|0;if(c){continue}break a}break}return d-e|0}return 0}function qk(a,b){a=a|0;b=b|0;var c=0,d=0;a:{b:{c=K[a+11|0];c:{d:{if((c&128)>>>7|0){d=(J[a+8>>2]&2147483647)-1|0;c=J[a+4>>2];if((d|0)==(c|0)){break d}break c}d=1;c=c&127;if((c|0)!=1){break b}}of(a,d,1,d,d);c=d;}d=J[a>>2];J[a+4>>2]=c+1;break a}H[a+11|0]=c+1;d=a;}a=(c<<2)+d|0;J[a>>2]=b;J[a+4>>2]=0;}function Nb(a,b){var c=0,d=0,e=0,f=0;d=Ua-16|0;Ua=d;a:{if(!b){b=0;break a}c=b>>31;e=(c^b)-c|0;c=S(e);Cb(d,e,0,0,0,c+81|0);e=0+J[d+8>>2]|0;c=(J[d+12>>2]^65536)+(16414-c<<16)|0;c=e>>>0<f>>>0?c+1|0:c;f=b&-2147483648|c;c=J[d+4>>2];b=J[d>>2];}J[a>>2]=b;J[a+4>>2]=c;J[a+8>>2]=e;J[a+12>>2]=f;Ua=d+16|0;}function mf(a,b,c){var d=0,e=0,f=0;e=Ua-16|0;Ua=e;d=Ua-32|0;Ua=d;f=Ua-16|0;Ua=f;J[f+12>>2]=b;J[d+24>>2]=a;J[d+28>>2]=J[f+12>>2];Ua=f+16|0;me(d+16|0,J[d+24>>2],J[d+28>>2],c);a=Rd(a,J[d+16>>2]-a>>2);J[d+12>>2]=J[d+20>>2];J[e+8>>2]=a;J[e+12>>2]=J[d+12>>2];Ua=d+32|0;Ua=e+16|0;return J[e+12>>2]}function ee(a,b,c,d){var e=0,f=0,g=0,h=0;e=Ua-160|0;Ua=e;J[e+60>>2]=b;J[e+20>>2]=b;J[e+24>>2]=-1;f=e+16|0;Yb(f,0,0);ig(e,f,d,1);d=J[e+8>>2];g=J[e+12>>2];f=J[e>>2];h=J[e+4>>2];if(c){J[c>>2]=((J[e+20>>2]+J[e+136>>2]|0)-J[e+60>>2]|0)+b;}J[a+8>>2]=d;J[a+12>>2]=g;J[a>>2]=f;J[a+4>>2]=h;Ua=e+160|0;}function nf(a,b,c){var d=0,e=0,f=0;e=Ua-16|0;Ua=e;d=Ua-32|0;Ua=d;f=Ua-16|0;Ua=f;J[f+12>>2]=b;J[d+24>>2]=a;J[d+28>>2]=J[f+12>>2];Ua=f+16|0;me(d+16|0,J[d+24>>2],J[d+28>>2],c);a=Td(a,J[d+16>>2]-a|0);J[d+12>>2]=J[d+20>>2];J[e+8>>2]=a;J[e+12>>2]=J[d+12>>2];Ua=d+32|0;Ua=e+16|0;return J[e+12>>2]}function rn(a,b){a=a|0;b=b|0;var c=0,d=0;a:{b:{c=K[a+11|0];c:{d:{if((c&128)>>>7|0){d=(J[a+8>>2]&2147483647)-1|0;c=J[a+4>>2];if((d|0)==(c|0)){break d}break c}d=10;c=c&127;if((c|0)!=10){break b}}Sd(a,d,1,d,d);c=d;}d=J[a>>2];J[a+4>>2]=c+1;break a}H[a+11|0]=c+1;d=a;}a=d+c|0;H[a|0]=b;H[a+1|0]=0;}function Nl(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;var f=0,g=0,h=0;h=(e-d|0)+b|0;a:{while(1){if((d|0)!=(e|0)){a=-1;if((b|0)==(c|0)){break a}f=H[b|0];g=H[d|0];if((f|0)<(g|0)){break a}if((f|0)>(g|0)){return 1}else {d=d+1|0;b=b+1|0;continue}}break}a=(c|0)!=(h|0);}return a|0}function oe(a){a=a|0;var b=0,c=0;J[a>>2]=8512;J[5996]=0;aa(133,a|0,0);b=J[5996];J[5996]=0;if((b|0)!=1){b=J[a+28>>2];c=J[b+4>>2]-1|0;J[b+4>>2]=c;if((c|0)==-1){Xa[J[J[b>>2]+8>>2]](b);}$a(J[a+32>>2]);$a(J[a+36>>2]);$a(J[a+48>>2]);$a(J[a+60>>2]);return a|0}fa(0)|0;Vb();B();}function Jc(a,b,c){var d=0,e=0,f=0;a:{if(!b){d=a;break a}while(1){d=eo(a,b,10);e=Wa;a=co(d,e,246,0)+a|0;c=c-1|0;H[c|0]=a|48;f=b>>>0>9;a=d;b=e;if(f){continue}break}}if(d){while(1){c=c-1|0;a=(d>>>0)/10|0;H[c|0]=P(a,246)+d|48;b=d>>>0>9;d=a;if(b){continue}break}}return c}function _b(a){var b=0,c=0,d=0,e=0;d=J[a>>2];if(d){b=d;c=J[a+4>>2];if((b|0)!=(c|0)){while(1){c=c-32|0;b=J[c+4>>2];a:{if(!b){break a}e=J[b+4>>2];J[b+4>>2]=e-1;if(e){break a}Xa[J[J[b>>2]+8>>2]](b);bb(b);}if((d|0)!=(c|0)){continue}break}b=J[a>>2];}J[a+4>>2]=d;$a(b);}}function lb(a){var b=0,c=0,d=0,e=0;d=J[a>>2];if(d){b=d;c=J[a+4>>2];if((b|0)!=(c|0)){while(1){c=c-8|0;b=J[c+4>>2];a:{if(!b){break a}e=J[b+4>>2];J[b+4>>2]=e-1;if(e){break a}Xa[J[J[b>>2]+8>>2]](b);bb(b);}if((d|0)!=(c|0)){continue}break}b=J[a>>2];}J[a+4>>2]=d;$a(b);}}function Ec(a,b){var c=0,d=0,e=0,f=0;c=Ua-16|0;Ua=c;a:{if(!b){b=0;break a}d=b;b=S(b);Cb(c,d,0,0,0,b+81|0);d=0+J[c+8>>2]|0;b=(J[c+12>>2]^65536)+(16414-b<<16)|0;f=e>>>0>d>>>0?b+1|0:b;e=J[c+4>>2];b=J[c>>2];}J[a>>2]=b;J[a+4>>2]=e;J[a+8>>2]=d;J[a+12>>2]=f;Ua=c+16|0;}function _m(a,b,c){a=a|0;b=b|0;c=c|0;var d=0;d=J[b+28>>2];J[a>>2]=d;J[d+4>>2]=J[d+4>>2]+1;Og(b+28|0,c);J[5996]=0;aa(133,b|0,1);b=J[5996];J[5996]=0;if((b|0)==1){c=Z()|0;a=J[a>>2];b=J[a+4>>2]-1|0;J[a+4>>2]=b;if((b|0)==-1){Xa[J[J[a>>2]+8>>2]](a);}da(c|0);B();}}function Kl(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;var f=0,g=0;a:{while(1){if((d|0)!=(e|0)){a=-1;if((b|0)==(c|0)){break a}f=J[b>>2];g=J[d>>2];if((f|0)<(g|0)){break a}if((f|0)>(g|0)){return 1}else {d=d+4|0;b=b+4|0;continue}}break}a=(b|0)!=(c|0);}return a|0}function Ie(a){a=a|0;var b=0,c=0,d=0,e=0;J[a>>2]=12664;e=a+8|0;while(1){b=J[a+8>>2];if(J[a+12>>2]-b>>2>>>0>c>>>0){b=J[b+(c<<2)>>2];if(b){d=J[b+4>>2]-1|0;J[b+4>>2]=d;if((d|0)==-1){Xa[J[J[b>>2]+8>>2]](b);}}c=c+1|0;continue}break}ab(a+152|0);gf(e);return a|0}function Nc(a){var b=0,c=0,d=0,e=0,f=0;a:{if(K[a+4|0]){break a}e=J[a>>2];b=J[e>>2];if(!b){break a}c=b;d=J[e+4>>2];if((d|0)!=(b|0)){while(1){c=d-12|0;f=J[c>>2];if(f){J[d-8>>2]=f;$a(f);}d=c;if((d|0)!=(b|0)){continue}break}c=J[J[a>>2]>>2];}J[e+4>>2]=b;$a(c);}}function Dg(a){var b=0,c=0;b=J[a+72>>2];J[a+72>>2]=b-1|b;if(J[a+20>>2]!=J[a+28>>2]){Xa[J[a+36>>2]](a,0,0)|0;}J[a+28>>2]=0;J[a+16>>2]=0;J[a+20>>2]=0;b=J[a>>2];if(b&4){J[a>>2]=b|32;return -1}c=J[a+44>>2]+J[a+48>>2]|0;J[a+8>>2]=c;J[a+4>>2]=c;return b<<27>>31}function hb(a,b){var c=0,d=0,e=0,f=0;a:{c=Gb(b);d=(K[a+11|0]&128)>>>7|0?(J[a+8>>2]&2147483647)-1|0:10;if(c>>>0<=d>>>0){sd(a,oc((K[a+11|0]&128)>>>7|0?J[a>>2]:a,b,c),c);break a}e=a;f=J[a+4>>2];a=K[a+11|0];a=(a&128)>>>7|0?f:a&127;dd(e,d,c-d|0,a,0,a,c,b);}}function Yb(a,b,c){var d=0,e=0,f=0,g=0;J[a+112>>2]=b;J[a+116>>2]=c;e=J[a+4>>2];d=J[a+44>>2]-e|0;J[a+120>>2]=d;J[a+124>>2]=d>>31;d=J[a+8>>2];a:{if(!(b|c)){break a}f=d-e|0;g=f>>31;if((c|0)>=(g|0)&b>>>0>=f>>>0|(c|0)>(g|0)){break a}d=b+e|0;}J[a+104>>2]=d;}function Lg(a,b){var c=0,d=0,e=0;A(+a);d=v(1)|0;e=v(0)|0;c=d>>>20&2047;if((c|0)!=2047){if(!c){if(a==0){c=0;}else {a=Lg(a*0x10000000000000000,b);c=J[b>>2]+-64|0;}J[b>>2]=c;return a}J[b>>2]=c-1022;x(0,e|0);x(1,d&-2146435073|1071644672);a=+z();}return a}function Vg(a){var b=0,c=0,d=0,e=0;a:{if(K[a+4|0]){break a}e=J[a>>2];b=J[e>>2];if(!b){break a}c=b;d=J[e+4>>2];if((b|0)!=(d|0)){while(1){c=d-12|0;if(H[d-1|0]<0){$a(J[c>>2]);}d=c;if((d|0)!=(b|0)){continue}break}c=J[J[a>>2]>>2];}J[e+4>>2]=b;$a(c);}}function ff(a,b){var c=0,d=0;d=Ua-16|0;Ua=d;c=d+4|0;J[c>>2]=a;a=J[a+4>>2];J[c+4>>2]=a;J[c+8>>2]=a+(b<<2);b=J[c+4>>2];a=J[c+8>>2];while(1){if((a|0)==(b|0)){J[J[c>>2]+4>>2]=J[c+4>>2];Ua=d+16|0;}else {J[b>>2]=0;b=b+4|0;J[c+4>>2]=b;continue}break}}function Vb(){var a=0;a=J[5728];J[5996]=0;ca(a|0);a=J[5996];J[5996]=0;a:{if((a|0)!=1){J[5996]=0;ca(291);a=J[5996];J[5996]=0;if((a|0)!=1){break a}}pa(fa(0)|0)|0;J[5996]=0;ca(291);a=J[5996];J[5996]=0;if((a|0)!=1){break a}fa(0)|0;Vb();}B();}function Ag(a,b,c,d){var e=0,f=0;e=Ua-32|0;Ua=e;f=Ua-16|0;Ua=f;J[f+12>>2]=c;J[e+24>>2]=b;J[e+28>>2]=J[f+12>>2];Ua=f+16|0;me(e+16|0,J[e+24>>2],J[e+28>>2],d);b=J[e+16>>2];J[e+12>>2]=J[e+20>>2];J[a>>2]=b;J[a+4>>2]=J[e+12>>2];Ua=e+32|0;}function Lb(a,b,c){c=c&176;if((c|0)==32){return b}a:{if((c|0)!=16){break a}b:{c:{c=K[a|0];switch(c-43|0){case 0:case 2:break c;default:break b}}return a+1|0}if((c|0)!=48|(b-a|0)<2|(K[a+1|0]|32)!=120){break a}a=a+2|0;}return a}function Ym(a,b){a=a|0;b=b|0;var c=0,d=0;c=J[b+48>>2];if(c&16){d=J[b+44>>2];c=J[b+24>>2];if(d>>>0<c>>>0){J[b+44>>2]=c;d=c;}td(a,J[b+20>>2],d);return}if(c&8){td(a,J[b+8>>2],J[b+16>>2]);return}J[a>>2]=0;J[a+4>>2]=0;J[a+8>>2]=0;}function Rh(a){a=a|0;var b=0,c=0;a:{b:{if(!a){break b}b=a-24|0;c=J[b>>2]-1|0;J[b>>2]=c;if(K[b+13|0]|c){break b}b=J[b+8>>2];if(b){J[5996]=0;$(b|0,a|0)|0;b=J[5996];J[5996]=0;if((b|0)==1){break a}}ic(a);}return}fa(0)|0;Vb();B();}function Eg(a,b,c){var d=0,e=0,f=0,g=0;d=Ua-16|0;Ua=d;b=b-a>>2;while(1){if(b){J[d+12>>2]=a;e=b>>>1|0;J[d+12>>2]=J[d+12>>2]+(e<<2);f=J[d+12>>2];g=M[f>>2]<c>>>0;a=g?f+4|0:a;b=g?(e^-1)+b|0:e;continue}break}Ua=d+16|0;return a}function Kb(a,b){a=a|0;b=b|0;var c=0,d=0,e=0;c=Gb(b);if(c>>>0<2147483632){a:{if(c>>>0<=10){H[a+11|0]=c;d=a;break a}e=(c|15)+1|0;d=cb(e);J[a+8>>2]=e|-2147483648;J[a>>2]=d;J[a+4>>2]=c;}H[eb(d,b,c)+c|0]=0;return a|0}ub();B();}function Ik(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0,j=0;g=Xa[J[J[a+8>>2]+20>>2]](a+8|0)|0;h=K[g+11|0];i=a;j=b;b=(h&128)>>>7|0;a=b?J[g>>2]:g;return jc(i,j,c,d,e,f,a,a+((b?J[g+4>>2]:h&127)<<2)|0)|0}function Qn(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0;a:{if(K[b|0]!=47){while(1){if((c|0)==(d|0)){e=Wb(b,c);b=0;break a}d=d+1|0;if(K[d+b|0]!=47){continue}break}}e=Wb(b,d);b=Wb((b+d|0)+1|0,(d^-1)+c|0);}J[a+4>>2]=b;J[a>>2]=e;}function co(a,b,c,d){var e=0,f=0,g=0,h=0,i=0,j=0;e=c>>>16|0;f=a>>>16|0;j=P(e,f);g=c&65535;h=a&65535;i=P(g,h);f=(i>>>16|0)+P(f,g)|0;e=(f&65535)+P(e,h)|0;Wa=(P(b,c)+j|0)+P(a,d)+(f>>>16)+(e>>>16)|0;return i&65535|e<<16}function Ok(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0,i=0,j=0;g=Xa[J[J[a+8>>2]+20>>2]](a+8|0)|0;h=K[g+11|0];i=a;j=b;b=(h&128)>>>7|0;a=b?J[g>>2]:g;return kc(i,j,c,d,e,f,a,a+(b?J[g+4>>2]:h&127)|0)|0}function pg(a,b){a=a|0;b=b|0;var c=0,d=0;c=J[a>>2];a:{if(!c){break a}d=J[c+24>>2];b:{if((d|0)==J[c+28>>2]){b=Xa[J[J[c>>2]+52>>2]](c,b)|0;break b}J[c+24>>2]=d+4;J[d>>2]=b;}if((b|0)!=-1){break a}J[a>>2]=0;}return a|0}function Jm(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;d=cb(104);J[d>>2]=7e3;J[d+4>>2]=0;J[d+8>>2]=0;J[5996]=0;b=ba(143,d+12|0,b|0,c|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){a=Z()|0;$a(d);da(a|0);B();}J[a+4>>2]=d;J[a>>2]=b;}function Gd(a,b,c){var d=0;d=J[a+16>>2];if(!d){J[a+36>>2]=1;J[a+24>>2]=c;J[a+16>>2]=b;return}a:{if((b|0)==(d|0)){if(J[a+24>>2]!=2){break a}J[a+24>>2]=c;return}H[a+54|0]=1;J[a+24>>2]=2;J[a+36>>2]=J[a+36>>2]+1;}}function Ed(a){var b=0,c=0,d=0;a:{if(K[a+12|0]){break a}b=J[J[a+8>>2]>>2];d=J[J[a+4>>2]>>2];if((b|0)==(d|0)){break a}while(1){a=b-12|0;c=J[a>>2];if(c){J[b-8>>2]=c;$a(c);}b=a;if((a|0)!=(d|0)){continue}break}}}function yf(a,b,c,d,e,f){var g=0;g=Ua-16|0;Ua=g;H[g+15|0]=0;H[g+14|0]=f;H[g+13|0]=e;H[g+12|0]=37;if(f){e=K[g+13|0];H[g+13|0]=K[g+14|0];H[g+14|0]=e;}J[c>>2]=(Ta(b|0,J[c>>2]-b|0,g+12|0,d|0,a|0)|0)+b;Ua=g+16|0;}function Bb(a,b,c,d,e){var f=0;f=Ua-256|0;Ua=f;if(!(e&73728|(c|0)<=(d|0))){d=c-d|0;c=d>>>0<256;Sb(f,b&255,c?d:256);if(!c){while(1){wb(a,f,256);d=d-256|0;if(d>>>0>255){continue}break}}wb(a,f,d);}Ua=f+256|0;}function Wc(a){var b=0,c=0,d=0,e=0;b=J[a>>2];if(b){c=b;d=J[a+4>>2];if((d|0)!=(b|0)){while(1){c=d-12|0;e=J[c>>2];if(e){J[d-8>>2]=e;$a(e);}d=c;if((d|0)!=(b|0)){continue}break}c=J[a>>2];}J[a+4>>2]=b;$a(c);}}function wi(a){a=a|0;var b=0,c=0,d=0;a=J[6019];if(a){b=a;c=J[6020];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6019];}J[6020]=a;$a(b);}}function ph(a){a=a|0;var b=0,c=0,d=0;a=J[6049];if(a){b=a;c=J[6050];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6049];}J[6050]=a;$a(b);}}function on(a){a=a|0;var b=0,c=0,d=0;a=J[6151];if(a){b=a;c=J[6152];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6151];}J[6152]=a;$a(b);}}function oh(a){a=a|0;var b=0,c=0,d=0;a=J[6055];if(a){b=a;c=J[6056];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6055];}J[6056]=a;$a(b);}}function nn(a){a=a|0;var b=0,c=0,d=0;a=J[6157];if(a){b=a;c=J[6158];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6157];}J[6158]=a;$a(b);}}function nh(a){a=a|0;var b=0,c=0,d=0;a=J[6061];if(a){b=a;c=J[6062];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6061];}J[6062]=a;$a(b);}}function mn(a){a=a|0;var b=0,c=0,d=0;a=J[6163];if(a){b=a;c=J[6164];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6163];}J[6164]=a;$a(b);}}function mh(a){a=a|0;var b=0,c=0,d=0;a=J[6067];if(a){b=a;c=J[6068];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6067];}J[6068]=a;$a(b);}}function ln(a){a=a|0;var b=0,c=0,d=0;a=J[6169];if(a){b=a;c=J[6170];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6169];}J[6170]=a;$a(b);}}function lh(a){a=a|0;var b=0,c=0,d=0;a=J[6073];if(a){b=a;c=J[6074];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6073];}J[6074]=a;$a(b);}}function kn(a){a=a|0;var b=0,c=0,d=0;a=J[6175];if(a){b=a;c=J[6176];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6175];}J[6176]=a;$a(b);}}function ci(a){a=a|0;var b=0,c=0,d=0;a=J[6025];if(a){b=a;c=J[6026];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6025];}J[6026]=a;$a(b);}}function Zh(a){a=a|0;var b=0,c=0,d=0;a=J[6031];if(a){b=a;c=J[6032];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6031];}J[6032]=a;$a(b);}}function Rn(a){a=a|0;var b=0,c=0,d=0;a=J[6081];if(a){b=a;c=J[6082];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6081];}J[6082]=a;$a(b);}}function Qh(a){a=a|0;var b=0,c=0,d=0;a=J[6037];if(a){b=a;c=J[6038];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6037];}J[6038]=a;$a(b);}}function On(a){a=a|0;var b=0,c=0,d=0;a=J[6085];if(a){b=a;c=J[6086];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6085];}J[6086]=a;$a(b);}}function Nn(a){a=a|0;var b=0,c=0,d=0;a=J[6089];if(a){b=a;c=J[6090];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6089];}J[6090]=a;$a(b);}}function Mn(a){a=a|0;var b=0,c=0,d=0;a=J[6093];if(a){b=a;c=J[6094];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6093];}J[6094]=a;$a(b);}}function Ln(a){a=a|0;var b=0,c=0,d=0;a=J[6097];if(a){b=a;c=J[6098];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6097];}J[6098]=a;$a(b);}}function Kn(a){a=a|0;var b=0,c=0,d=0;a=J[6101];if(a){b=a;c=J[6102];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6101];}J[6102]=a;$a(b);}}function Jn(a){a=a|0;var b=0,c=0,d=0;a=J[6105];if(a){b=a;c=J[6106];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6105];}J[6106]=a;$a(b);}}function Jh(a){a=a|0;var b=0,c=0,d=0;a=J[6043];if(a){b=a;c=J[6044];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6043];}J[6044]=a;$a(b);}}function Jg(a){var b=0,c=0,d=0;d=J[a>>2];while(1){c=H[d|0];if(c-48>>>0<10){d=d+1|0;J[a>>2]=d;if(b>>>0<=214748364){c=c-48|0;b=P(b,10);b=(c|0)>(b^2147483647)?-1:c+b|0;}else {b=-1;}continue}break}return b}function In(a){a=a|0;var b=0,c=0,d=0;a=J[6109];if(a){b=a;c=J[6110];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6109];}J[6110]=a;$a(b);}}function Hn(a){a=a|0;var b=0,c=0,d=0;a=J[6113];if(a){b=a;c=J[6114];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6113];}J[6114]=a;$a(b);}}function Gn(a){a=a|0;var b=0,c=0,d=0;a=J[6117];if(a){b=a;c=J[6118];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6117];}J[6118]=a;$a(b);}}function Fn(a){a=a|0;var b=0,c=0,d=0;a=J[6121];if(a){b=a;c=J[6122];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6121];}J[6122]=a;$a(b);}}function En(a){a=a|0;var b=0,c=0,d=0;a=J[6125];if(a){b=a;c=J[6126];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6125];}J[6126]=a;$a(b);}}function Dn(a){a=a|0;var b=0,c=0,d=0;a=J[6129];if(a){b=a;c=J[6130];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6129];}J[6130]=a;$a(b);}}function Cn(a){a=a|0;var b=0,c=0,d=0;a=J[6133];if(a){b=a;c=J[6134];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6133];}J[6134]=a;$a(b);}}function Bn(a){a=a|0;var b=0,c=0,d=0;a=J[6137];if(a){b=a;c=J[6138];if((c|0)!=(a|0)){while(1){b=c-12|0;d=J[b>>2];if(d){J[c-8>>2]=d;$a(d);}c=b;if((c|0)!=(a|0)){continue}break}b=J[6137];}J[6138]=a;$a(b);}}function Wg(a){var b=0,c=0;a:{if(K[a+12|0]){break a}b=J[J[a+8>>2]>>2];c=J[J[a+4>>2]>>2];if((b|0)==(c|0)){break a}while(1){a=b-12|0;if(H[b-1|0]<0){$a(J[a>>2]);}b=a;if((a|0)!=(c|0)){continue}break}}}function lc(a,b){var c=0;b=pb(b);c=J[a+12>>2];a=J[a+8>>2];if(b>>>0<c-a>>2>>>0){c=J[a+(b<<2)>>2]!=0;}else {c=0;}if(!c){a=Rb(4);J[a>>2]=22892;J[a>>2]=22736;na(a|0,22872,173);B();}return J[a+(b<<2)>>2]}function rb(a){var b=0,c=0,d=0;b=J[a>>2];if(b){c=b;d=J[a+4>>2];if((b|0)!=(d|0)){while(1){c=d-12|0;if(H[d-1|0]<0){$a(J[c>>2]);}d=c;if((d|0)!=(b|0)){continue}break}c=J[a>>2];}J[a+4>>2]=b;$a(c);}}function gg(a,b,c,d){a:{if(!a){break a}b:{switch(b+2|0){case 0:H[a|0]=c;return;case 1:I[a>>1]=c;return;case 2:case 3:J[a>>2]=c;return;case 5:break b;default:break a}}J[a>>2]=c;J[a+4>>2]=d;}}function Be(a){var b=0;b=J[a+72>>2];J[a+72>>2]=b-1|b;b=J[a>>2];if(b&8){J[a>>2]=b|32;return -1}J[a+4>>2]=0;J[a+8>>2]=0;b=J[a+44>>2];J[a+28>>2]=b;J[a+20>>2]=b;J[a+16>>2]=b+J[a+48>>2];return 0}function he(a,b,c,d,e,f,g,h,i){var j=0;j=Ua-16|0;Ua=j;Hb(j,b,c,d,e,f,g,h,i^-2147483648);d=J[j>>2];c=J[j+4>>2];b=J[j+12>>2];J[a+8>>2]=J[j+8>>2];J[a+12>>2]=b;J[a>>2]=d;J[a+4>>2]=c;Ua=j+16|0;}function cb(a){a=a|0;var b=0;a=a>>>0<=1?1:a;a:{while(1){b=sb(a);if(b){break a}b=J[6704];if(b){Xa[b|0]();continue}break}a=Rb(4);J[a>>2]=22892;J[a>>2]=22364;na(a|0,22448,5);B();}return b|0}function Uc(a,b){var c=0,d=0;c=K[a|0];d=K[b|0];a:{if(!c|(c|0)!=(d|0)){break a}while(1){d=K[b+1|0];c=K[a+1|0];if(!c){break a}b=b+1|0;a=a+1|0;if((c|0)==(d|0)){continue}break}}return c-d|0}function Pl(a,b,c){a=a|0;b=b|0;c=c|0;var d=0,e=0,f=0;e=J[a+84>>2];d=c+256|0;f=Zb(e,0,d);d=f?f-e|0:d;c=c>>>0>d>>>0?d:c;fb(b,e,c);b=e+d|0;J[a+84>>2]=b;J[a+8>>2]=b;J[a+4>>2]=c+e;return c|0}function Pk(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;var g=0,h=0;g=Ua-16|0;Ua=g;J[g+8>>2]=624576549;J[g+12>>2]=1394948685;h=a;a=g+16|0;b=kc(h,b,c,d,e,f,g+8|0,a);Ua=a;return b|0}function xi(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;while(1){if((b|0)!=(c|0)){a=J[b>>2];if(a>>>0<=127){a=J[(a<<2)+12736>>2];}else {a=0;}J[d>>2]=a;d=d+4|0;b=b+4|0;continue}break}return c|0}function rg(a){var b=0,c=0;b=J[a>>2];if(b){c=J[b+12>>2];a:{if((c|0)==J[b+16>>2]){b=Xa[J[J[b>>2]+36>>2]](b)|0;break a}b=J[c>>2];}if((b|0)!=-1){return !J[a>>2]}J[a>>2]=0;}return 1}function xg(a){var b=0,c=0;b=J[a>>2];if(b){c=J[b+12>>2];a:{if((c|0)==J[b+16>>2]){b=Xa[J[J[b>>2]+36>>2]](b)|0;break a}b=K[c|0];}if((b|0)!=-1){return !J[a>>2]}J[a>>2]=0;}return 1}function Ug(a){var b=0,c=0,d=0;b=J[a>>2];if(b){d=b;c=J[a+4>>2];if((c|0)!=(b|0)){while(1){c=c-12|0;Db(J[c+4>>2]);if((b|0)!=(c|0)){continue}break}d=J[a>>2];}J[a+4>>2]=b;$a(d);}}function Nm(a,b,c){a=a|0;b=b|0;c=c|0;var d=0;d=cb(104);J[5996]=0;b=ba(147,d|0,b|0,c|0)|0;c=J[5996];J[5996]=0;if((c|0)==1){a=Z()|0;$a(d);da(a|0);B();}J[a+4>>2]=b;J[a>>2]=b+12;}function ui(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;a:{while(1){if((c|0)==(d|0)){break a}a=J[c>>2];if(!(a>>>0>127|!(J[(a<<2)+12736>>2]&b))){c=c+4|0;continue}break}d=c;}return d|0}function vi(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;while(1){a:{if((c|0)!=(d|0)){a=J[c>>2];if(a>>>0>127|!(J[(a<<2)+12736>>2]&b)){break a}}else {c=d;}return c|0}c=c+4|0;continue}}function Ci(a){a=a|0;var b=0,c=0;b=Ua-16|0;Ua=b;c=J[Xb(b+12|0,J[a+8>>2])>>2];if(c){J[6178]=(c|0)==-1?24816:c;}Ua=b+16|0;a=J[a+8>>2];if(!a){return 1}return (Le(a)|0)==1|0}function yd(a){var b=0,c=0;if(a){yd(J[a>>2]);yd(J[a+4>>2]);b=J[a+28>>2];a:{if(!b){break a}c=J[b+4>>2];J[b+4>>2]=c-1;if(c){break a}Xa[J[J[b>>2]+8>>2]](b);bb(b);}$a(a);}}function wj(a){a=a|0;var b=0,c=0;b=Ua-16|0;Ua=b;if(_e()>>>0<=29){nb();B();}Ze(b+8|0,a+16|0,30);c=J[b+8>>2];J[a+4>>2]=c;J[a>>2]=c;J[a+8>>2]=(J[b+12>>2]<<2)+c;Ua=b+16|0;}function Oc(a){var b=0,c=0,d=0;b=J[a>>2];if(b){d=b;c=J[a+4>>2];if((c|0)!=(b|0)){while(1){c=Fb(c-56|0);if((c|0)!=(b|0)){continue}break}d=J[a>>2];}J[a+4>>2]=b;$a(d);}}function kh(a){a=a|0;var b=0,c=0;a=J[6076];if(a){c=a;b=J[6077];if((b|0)!=(a|0)){while(1){b=b-4|0;ad(b);if((a|0)!=(b|0)){continue}break}c=J[6076];}J[6077]=a;$a(c);}}function Dh(a,b,c){a=a|0;b=b|0;c=c|0;var d=0;d=Ua-16|0;Ua=d;J[d+12>>2]=J[c>>2];a=Xa[J[J[a>>2]+16>>2]](a,b,d+12|0)|0;if(a){J[c>>2]=J[d+12>>2];}Ua=d+16|0;return a|0}function Il(a,b,c){a=a|0;b=b|0;c=c|0;var d=0;a=0;while(1){if((b|0)!=(c|0)){a=J[b>>2]+(a<<4)|0;d=a&-268435456;a=(d|d>>>24)^a;b=b+4|0;continue}break}return a|0}function zi(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;a=Ua-16|0;Ua=a;J[a+12>>2]=e;J[a+8>>2]=d-c;Ua=a+16|0;b=a+8|0;a=a+12|0;return J[(M[b>>2]<M[a>>2]?b:a)>>2]}function yb(a,b,c,d,e){var f=0;f=Ua-16|0;Ua=f;J[f+12>>2]=e;c=Xb(f+8|0,c);b=pd(a,b,d,J[f+12>>2]);a=J[c>>2];if(a){J[6178]=(a|0)==-1?24816:a;}Ua=f+16|0;return b}function Ll(a,b,c){a=a|0;b=b|0;c=c|0;var d=0;a=0;while(1){if((b|0)!=(c|0)){a=H[b|0]+(a<<4)|0;d=a&-268435456;a=(d|d>>>24)^a;b=b+1|0;continue}break}return a|0}function Mm(a,b){a=a|0;b=b|0;var c=0;c=J[a+4>>2];if(c>>>0<M[a+8>>2]){J[c>>2]=J[b>>2];J[c+4>>2]=J[b+4>>2];J[b>>2]=0;J[b+4>>2]=0;J[a+4>>2]=c+8;return}re(a,b);}function tb(a,b){var c=0;c=J[a>>2];J[a>>2]=b;a:{if(c){a=J[a+4>>2];J[5996]=0;ha(a|0,c|0);a=J[5996];J[5996]=0;if((a|0)==1){break a}}return}fa(0)|0;Vb();B();}function Kf(a,b,c){var d=0;d=Ua-16|0;Ua=d;J[d+12>>2]=c;b=Xb(d+8|0,b);c=fg(a,1937,J[d+12>>2]);a=J[b>>2];if(a){J[6178]=(a|0)==-1?24816:a;}Ua=d+16|0;return c}function si(a,b,c){a=a|0;b=b|0;c=c|0;while(1){if((b|0)!=(c|0)){a=J[b>>2];if(a>>>0<=127){a=J[(a<<2)+11184>>2];}J[b>>2]=a;b=b+4|0;continue}break}return c|0}function yg(a){var b=0,c=0;J[5996]=0;c=Ca(168)|0;b=J[5996];J[5996]=0;if((b|0)!=1){b=a;a=J[c>>2];J[b>>2]=a;J[a+4>>2]=J[a+4>>2]+1;return}fa(0)|0;Vb();B();}function qi(a,b,c){a=a|0;b=b|0;c=c|0;while(1){if((b|0)!=(c|0)){a=J[b>>2];if(a>>>0<=127){a=J[(a<<2)+9648>>2];}J[b>>2]=a;b=b+4|0;continue}break}return c|0}function qc(a){var b=0,c=0,d=0;b=Ua-16|0;Ua=b;c=K[a+11|0];d=(c&128)>>>7|0;J[b+12>>2]=(d?J[a>>2]:a)+((d?J[a+4>>2]:c&127)<<2);Ua=b+16|0;return J[b+12>>2]}function ni(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;while(1){if((b|0)!=(c|0)){a=J[b>>2];H[e|0]=a>>>0<128?a:d;e=e+1|0;b=b+4|0;continue}break}return c|0}function Og(a,b){var c=0,d=0;c=J[b>>2];J[c+4>>2]=J[c+4>>2]+1;c=J[a>>2];d=J[c+4>>2]-1|0;J[c+4>>2]=d;if((d|0)==-1){Xa[J[J[c>>2]+8>>2]](c);}J[a>>2]=J[b>>2];}function md(a,b){var c=0;a:{if((a|0)==(b|0)){break a}while(1){b=b-4|0;if(b>>>0<=a>>>0){break a}c=J[a>>2];J[a>>2]=J[b>>2];J[b>>2]=c;a=a+4|0;continue}}}function Wh(a,b,c){a=a|0;b=b|0;c=c|0;var d=0;d=Ua-16|0;Ua=d;Xa[J[J[a>>2]+12>>2]](d+8|0,a,b);Ua=d+16|0;return J[d+12>>2]==J[c+4>>2]&J[d+8>>2]==J[c>>2]}function xf(a){a=a|0;var b=0,c=0;b=Rb(8);J[5996]=0;a=_(91,b|0,a|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){na(a|0,22664,92);B();}a=Z()|0;ic(b);da(a|0);B();}function ji(a,b,c){a=a|0;b=b|0;c=c|0;while(1){if((b|0)!=(c|0)){a=H[b|0];if((a|0)>=0){a=J[(a<<2)+11184>>2];}H[b|0]=a;b=b+1|0;continue}break}return c|0}function ei(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;while(1){if((b|0)!=(c|0)){a=H[b|0];H[e|0]=(a|0)<0?d:a;e=e+1|0;b=b+1|0;continue}break}return c|0}function hi(a,b,c){a=a|0;b=b|0;c=c|0;while(1){if((b|0)!=(c|0)){a=H[b|0];if((a|0)>=0){a=J[(a<<2)+9648>>2];}H[b|0]=a;b=b+1|0;continue}break}return c|0}function Nh(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;if(Jb(a,J[b+8>>2],f)){Hd(b,c,d,e);return}a=J[a+8>>2];Xa[J[J[a>>2]+20>>2]](a,b,c,d,e,f);}function sc(a){var b=0,c=0,d=0;b=Ua-16|0;Ua=b;c=K[a+11|0];d=(c&128)>>>7|0;J[b+12>>2]=(d?J[a>>2]:a)+(d?J[a+4>>2]:c&127);Ua=b+16|0;return J[b+12>>2]}function Rl(a,b){a=a|0;b=b|0;var c=0,d=0;c=J[a+40>>2];while(1){if(c){c=c-1|0;d=c<<2;Xa[J[J[a+32>>2]+d>>2]](b,a,J[d+J[a+36>>2]>>2]);continue}break}}function vc(a,b){var c=0;a:{if((a|0)==(b|0)){break a}while(1){b=b-1|0;if(b>>>0<=a>>>0){break a}c=K[a|0];H[a|0]=K[b|0];H[b|0]=c;a=a+1|0;continue}}}function _e(){var a=0,b=0;a=Ua-16|0;Ua=a;J[a+12>>2]=1073741823;J[a+8>>2]=2147483647;Ua=a+16|0;b=a+8|0;a=a+12|0;return J[(M[b>>2]<M[a>>2]?b:a)>>2]}function ac(a,b){var c=0;if((K[a+11|0]&128)>>>7|0){$a(J[a>>2]);}c=J[b+4>>2];J[a>>2]=J[b>>2];J[a+4>>2]=c;J[a+8>>2]=J[b+8>>2];H[b+11|0]=0;J[b>>2]=0;}function Ud(a,b){var c=0;if(!((K[b+11|0]&128)>>>7|0)){c=J[b+4>>2];J[a>>2]=J[b>>2];J[a+4>>2]=c;J[a+8>>2]=J[b+8>>2];return}Zc(a,J[b>>2],J[b+4>>2]);}function rj(a){if(K[25164]){return J[6290]}if(!K[26312]){H[26312]=1;}ib(26288,19200);ib(26300,19212);H[25164]=1;J[6290]=26288;return 26288}function Ab(a,b){var c=0;if((K[a+11|0]&128)>>>7|0){$a(J[a>>2]);}c=J[b+4>>2];J[a>>2]=J[b>>2];J[a+4>>2]=c;J[a+8>>2]=J[b+8>>2];H[b+11|0]=0;H[b|0]=0;}function ue(){var a=0,b=0,c=0;b=Rb(8);J[5996]=0;a=$(125,b|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){na(a|0,22632,7);B();}a=Z()|0;ic(b);da(a|0);B();}function dj(a){if(K[25156]){return J[6288]}if(!K[26280]){H[26280]=1;}hb(26256,3116);hb(26268,3113);H[25156]=1;J[6288]=26256;return 26256}function If(a){var b=0,c=0;b=Rb(8);J[5996]=0;a=_(6,b|0,a|0)|0;c=J[5996];J[5996]=0;if((c|0)!=1){na(a|0,22580,7);B();}a=Z()|0;ic(b);da(a|0);B();}function Ee(a,b,c,d,e){var f=0;f=c;c=0;a:{if(!d){break a}c=b>>8;if(!(b&1)){break a}c=J[c+J[d>>2]>>2];}Xa[J[J[a>>2]+28>>2]](a,f,c+d|0,b&2?e:2);}function og(a,b,c){var d=0;d=Ua-16|0;Ua=d;H[d+15|0]=c;c=a;while(1){if(b){H[c|0]=K[d+15|0];b=b-1|0;c=c+1|0;continue}break}Ua=d+16|0;return a}function Le(a){var b=0,c=0;b=Ua-16|0;Ua=b;a=Xb(b+12|0,a);c=J[J[6178]>>2]?4:1;a=J[a>>2];if(a){J[6178]=(a|0)==-1?24816:a;}Ua=b+16|0;return c}function vg(a,b){var c=0;c=J[a+24>>2];if((c|0)==J[a+28>>2]){return Xa[J[J[a>>2]+52>>2]](a,b&255)|0}J[a+24>>2]=c+1;H[c|0]=b;return b&255}function Me(a,b,c,d,e){var f=0;f=Ua-16|0;Ua=f;e=Xb(f+12|0,e);b=qd(a,b,c,d);a=J[e>>2];if(a){J[6178]=(a|0)==-1?24816:a;}Ua=f+16|0;return b}function Qc(a){a=a|0;var b=0,c=0;J[a>>2]=7896;b=J[a+4>>2];c=J[b+4>>2]-1|0;J[b+4>>2]=c;if((c|0)==-1){Xa[J[J[b>>2]+8>>2]](b);}return a|0}function bi(){var a=0;if(K[25032]){return J[6257]}a=J[Pg()>>2];J[6256]=a;J[a+4>>2]=J[a+4>>2]+1;H[25032]=1;J[6257]=25024;return 25024}function nl(a){if(H[24063]<0){$a(J[6013]);}if(H[24051]<0){$a(J[6010]);}if(H[24039]<0){$a(J[6007]);}if(H[24027]<0){$a(J[6004]);}}function mm(a){a=a|0;var b=0;b=J[a+12>>2];a:{if((b|0)==J[a+16>>2]){a=Xa[J[J[a>>2]+36>>2]](a)|0;break a}a=K[b|0];}return a<<24>>24}function pi(a,b,c,d){b=b|0;c=c|0;d=d|0;while(1){if((b|0)!=(c|0)){J[d>>2]=H[b|0];d=d+4|0;b=b+1|0;continue}break}return c|0}function gd(a){a=a|0;J[a+56>>2]=8316;J[a>>2]=8296;J[a+4>>2]=7960;if(H[a+47|0]<0){$a(J[a+36>>2]);}Qc(a+4|0);Od(a+56|0);return a|0}function Md(a,b,c){var d=0;d=Ua-16|0;Ua=d;c=Xb(d+12|0,c);b=pe(a,b);a=J[c>>2];if(a){J[6178]=(a|0)==-1?24816:a;}Ua=d+16|0;return b}function Df(a,b,c,d,e,f){a=Xa[J[J[a+8>>2]+4>>2]](a+8|0)|0;a=nd(c,d,a,a+288|0,f,e,0)-a|0;if((a|0)<=287){J[b>>2]=((a|0)/12|0)%12;}}function Af(a,b,c,d,e,f){a=Xa[J[J[a+8>>2]+4>>2]](a+8|0)|0;a=ld(c,d,a,a+288|0,f,e,0)-a|0;if((a|0)<=287){J[b>>2]=((a|0)/12|0)%12;}}function um(a){a=a|0;var b=0;if((Xa[J[J[a>>2]+36>>2]](a)|0)!=-1){b=a;a=J[a+12>>2];J[b+12>>2]=a+1;a=K[a|0];}else {a=-1;}return a|0}function gi(a,b,c,d){b=b|0;c=c|0;d=d|0;while(1){if((b|0)!=(c|0)){H[d|0]=K[b|0];d=d+1|0;b=b+1|0;continue}break}return c|0}function bb(a){var b=0,c=0;a:{b=a+8|0;if(J[b>>2]){c=b;b=J[b>>2]-1|0;J[c>>2]=b;if((b|0)!=-1){break a}}Xa[J[J[a>>2]+16>>2]](a);}}function Lh(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;if(Jb(a,J[b+8>>2],0)){Gd(b,c,d);return}a=J[a+8>>2];Xa[J[J[a>>2]+28>>2]](a,b,c,d);}function Ef(a,b,c,d,e,f){a=Xa[J[J[a+8>>2]>>2]](a+8|0)|0;a=nd(c,d,a,a+168|0,f,e,0)-a|0;if((a|0)<=167){J[b>>2]=((a|0)/12|0)%7;}}function Bf(a,b,c,d,e,f){a=Xa[J[J[a+8>>2]>>2]](a+8|0)|0;a=ld(c,d,a,a+168|0,f,e,0)-a|0;if((a|0)<=167){J[b>>2]=((a|0)/12|0)%7;}}function me(a,b,c,d){var e=0,f=0;e=Ua-16|0;Ua=e;f=b;b=c-b|0;J[e+12>>2]=eb(d,f,b)+b;J[a>>2]=c;J[a+4>>2]=J[e+12>>2];Ua=e+16|0;}function dm(a){a=a|0;var b=0;b=J[a+12>>2];a:{if((b|0)==J[a+16>>2]){a=Xa[J[J[a>>2]+36>>2]](a)|0;break a}a=J[b>>2];}return a|0}function mb(a){var b=0;a:{a=J[a+4>>2];if(!a){break a}b=J[a+4>>2];J[a+4>>2]=b-1;if(b){break a}Xa[J[J[a>>2]+8>>2]](a);bb(a);}}function Th(a,b){a=a|0;b=b|0;var c=0;c=a;a=K[b+11|0];return ve(c,(a&128)>>>7|0?J[b>>2]:b,(a&128)>>>7|0?J[b+4>>2]:a&127)|0}function qg(a){var b=0;b=J[a+12>>2];if((b|0)==J[a+16>>2]){return Xa[J[J[a>>2]+40>>2]](a)|0}J[a+12>>2]=b+4;return J[b>>2]}function wg(a){var b=0;b=J[a+12>>2];if((b|0)==J[a+16>>2]){return Xa[J[J[a>>2]+40>>2]](a)|0}J[a+12>>2]=b+1;return K[b|0]}function Fg(a,b){a=a|0;b=b|0;var c=0;c=b;b=J[b>>2]+7&-8;J[c>>2]=b+16;O[a>>3]=ge(J[b>>2],J[b+4>>2],J[b+8>>2],J[b+12>>2]);}function be(a,b,c){c=c&255;while(1){a:{if((a|0)!=(b|0)){if((c|0)!=K[a|0]){break a}}else {a=b;}return a}a=a+1|0;continue}}function El(a){a=a|0;var b=0,c=0;c=J[a+4>>2];b=J[a>>2];a=J[a+8>>2];b=b+(a>>1)|0;if(a&1){c=J[c+J[b>>2]>>2];}Xa[c|0](b);}function kg(a,b,c,d,e,f,g,h,i){J[a>>2]=b;J[a+4>>2]=c;J[a+8>>2]=d;J[a+12>>2]=e&65535|(i>>>16&32768|e>>>16&32767)<<16;}function $e(a){var b=0;b=J[a>>2];J[a>>2]=0;if(b){a=J[b+4>>2]-1|0;J[b+4>>2]=a;if((a|0)==-1){Xa[J[J[b>>2]+8>>2]](b);}}}function sg(a,b){a=a|0;b=b|0;var c=0;c=J[a>>2];a:{if(!c){break a}if((vg(c,b)|0)!=-1){break a}J[a>>2]=0;}return a|0}function Yd(a,b,c){while(1){a:{if((a|0)!=(b|0)){if(J[a>>2]!=(c|0)){break a}}else {a=b;}return a}a=a+4|0;continue}}function Fd(a,b,c,d,e,f,g){var h=0;h=b>>8;if(b&1){h=J[J[e>>2]+h>>2];}Xa[J[J[a>>2]+20>>2]](a,c,d,e+h|0,b&2?f:2,g);}function Ld(a,b){var c=0,d=0;c=Gb(b);d=cb(c+13|0);J[d+8>>2]=0;J[d+4>>2]=c;J[d>>2]=c;J[a>>2]=fb(d+12|0,b,c+1|0);}function Jb(a,b,c){if(!c){return J[a+4>>2]==J[b+4>>2]}if((a|0)==(b|0)){return 1}return !Uc(J[a+4>>2],J[b+4>>2])}function zm(a,b,c,d,e,f){a=a|0;J[a+8>>2]=-1;J[a+12>>2]=-1;J[a>>2]=0;J[a+4>>2]=0;}function _h(a,b){a=a|0;b=b|0;J[a>>2]=22892;J[a>>2]=22500;Ld(a+4|0,(K[b+11|0]&128)>>>7|0?J[b>>2]:b);return a|0}function Rd(a,b){var c=0;c=Ua-16|0;Ua=c;J[c+12>>2]=a;a=c+12|0;J[a>>2]=J[a>>2]+(b<<2);Ua=c+16|0;return J[a>>2]}function zf(a,b,c,d,e){b=bc(b,c,d,e,4);if(!(K[d|0]&4)){J[a>>2]=((b|0)<69?b+2e3|0:(b|0)<100?b+1900|0:b)-1900;}}function Mb(a){var b=0;b=Ua-16|0;Ua=b;J[b+12>>2]=(K[a+11|0]&128)>>>7|0?J[a>>2]:a;Ua=b+16|0;return J[b+12>>2]}function Cf(a,b,c,d,e){b=cc(b,c,d,e,4);if(!(K[d|0]&4)){J[a>>2]=((b|0)<69?b+2e3|0:(b|0)<100?b+1900|0:b)-1900;}}function cd(a,b,c,d,e,f){var g=0;g=b>>8;if(b&1){g=J[J[d>>2]+g>>2];}Xa[J[J[a>>2]+24>>2]](a,c,d+g|0,b&2?e:2,f);}function Ke(a,b,c,d,e,f,g,h){c=c|0;e=e|0;f=f|0;h=h|0;J[e>>2]=c;J[h>>2]=f;return 3}function Td(a,b){var c=0;c=Ua-16|0;Ua=c;J[c+12>>2]=a;a=c+12|0;J[a>>2]=J[a>>2]+b;Ua=c+16|0;return J[a>>2]}function ec(a){a:{a=a&74;if(a){if((a|0)==64){return 8}if((a|0)!=8){break a}return 16}return 0}return 10}function Xb(a,b){var c=0;c=J[6178];if(b){J[6178]=(b|0)==-1?24816:b;}J[a>>2]=(c|0)==24816?-1:c;return a}function Ze(a,b,c){a:{if(!(K[b+120|0]|c>>>0>30)){H[b+120|0]=1;break a}b=$f(c);}J[a+4>>2]=c;J[a>>2]=b;}function af(a,b){J[a+4>>2];J[a+4>>2]=b;}function Yl(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;Xa[J[J[b>>2]+16>>2]](a,b,J[c+8>>2],J[c+12>>2],0,d);}function Mc(a){var b=0;b=J[a+12>>2];if(b){J[a+16>>2]=b;$a(b);}b=J[a>>2];if(b){J[a+4>>2]=b;$a(b);}}function Eh(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;if(Jb(a,J[b+8>>2],f)){Hd(b,c,d,e);}}function xm(a,b,c,d){a=a|0;J[a+8>>2]=-1;J[a+12>>2]=-1;J[a>>2]=0;J[a+4>>2]=0;}function gm(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;return Xa[J[J[a>>2]+24>>2]](a,b,c,d,e)|0}function We(a){var b=0,c=0;c=a;while(1){b=c;c=b+4|0;if(J[b>>2]){continue}break}return b-a>>2}function Je(a){a=a|0;var b=0;J[a>>2]=12684;b=J[a+8>>2];if(!(!b|!K[a+12|0])){$a(b);}return a|0}function Pb(){var a=0;a=Rb(4);J[a>>2]=22892;J[a>>2]=22364;J[a>>2]=22384;na(a|0,22460,5);B();}function sd(a,b,c){a:{if((K[a+11|0]&128)>>>7|0){J[a+4>>2]=c;break a}H[a+11|0]=c;}H[b+c|0]=0;}function od(a,b,c,d){var e=0;e=Ua-16|0;Ua=e;J[e+12>>2]=d;a=pd(a,b,c,d);Ua=e+16|0;return a}function eg(a,b){var c=0;c=Ua-16|0;Ua=c;J[c+12>>2]=b;a=fg(a,2310,b);Ua=c+16|0;return a}function cg(a){if((a|0)!=0&(a|0)!=19324&(a|0)!=19348&(a|0)!=24732&(a|0)!=24756){$a(a);}}function yk(a,b){a=a|0;b=b|0;var c=0;c=wg(J[b>>2]);J[a+4>>2]=J[b>>2];H[a|0]=c<<24>>24;}function wf(a,b){a=a|0;b=b|0;b=Ua-16|0;Ua=b;H[a+11|0]=1;H[og(a,1,45)+1|0]=0;Ua=b+16|0;}function tc(a,b,c){var d=0;d=Ua-16|0;Ua=d;Ag(d+8|0,b,(c<<2)+b|0,a);Ua=d+16|0;return a}function ki(a,b){b=b|0;if((b|0)>=0){b=J[((b&255)<<2)+11184>>2];}return b<<24>>24}function Xn(a){a=a|0;a=J[6144];if(a){J[6145]=a;$a(a);}a=J[6141];if(a){J[6142]=a;$a(a);}}function Bg(a,b,c){var d=0;d=Ua-16|0;Ua=d;Ag(d+8|0,a,b,c);Ua=d+16|0;return J[d+12>>2]}function ii(a,b){b=b|0;if((b|0)>=0){b=J[((b&255)<<2)+9648>>2];}return b<<24>>24}function De(a){var b=0;a=J[a>>2]-12|0;b=J[a+8>>2]-1|0;J[a+8>>2]=b;if((b|0)<0){$a(a);}}function Db(a){if(a){Db(J[a>>2]);Db(J[a+4>>2]);if(H[a+27|0]<0){$a(J[a+16>>2]);}$a(a);}}function ij(a){a=a|0;a=26312;while(1){a=ab(a-12|0);if((a|0)!=26288){continue}break}}function hj(a){a=a|0;a=26240;while(1){a=ab(a-12|0);if((a|0)!=25952){continue}break}}function gj(a){a=a|0;a=25640;while(1){a=ab(a-12|0);if((a|0)!=25472){continue}break}}function Wi(a){a=a|0;a=26280;while(1){a=ab(a-12|0);if((a|0)!=26256){continue}break}}function Vi(a){a=a|0;a=25936;while(1){a=ab(a-12|0);if((a|0)!=25648){continue}break}}function Ui(a){a=a|0;a=25464;while(1){a=ab(a-12|0);if((a|0)!=25296){continue}break}}function Ne(a){a=a|0;J[a>>2]=13768;if(J[a+8>>2]!=(jb()|0)){cg(J[a+8>>2]);}return a|0}function rf(a,b){var c=0;c=J[b>>2];J[b>>2]=0;tb(a,c);J[a+4>>2]=J[b+4>>2];return a}function cf(a){var b=0;b=J[a>>2];if(J[b>>2]){df(b);a=J[a>>2];bf(a+16|0,J[a>>2]);}}function yi(a,b,c){b=b|0;c=c|0;return (J[(c<<2)+12736>>2]&b)!=0&c>>>0<=127}function uh(a,b,c,d,e,f){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;f=f|0;Xa[a|0](b,c,d,e,f);}function Vd(a,b){a=a|0;b=b|0;J[a>>2]=22892;J[a>>2]=22480;Ld(a+4|0,b);return a|0}function Tg(a,b){a=a|0;b=b|0;J[a>>2]=22892;J[a>>2]=22500;Ld(a+4|0,b);return a|0}function th(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;return Xa[a|0](b,c,d,e)|0}function rk(a,b){a=a|0;b=b|0;var c=0;c=qg(J[b>>2]);J[a+4>>2]=J[b>>2];J[a>>2]=c;}function Yj(){J[6593]=0;J[6592]=22008;J[6592]=14144;J[6592]=13768;J[6594]=jb();}function fo(a){var b=0;while(1){if(a){a=a-1&a;b=b+1|0;continue}break}return b}function ce(){var a=0;a=Rb(4);J[a>>2]=22892;J[a>>2]=22364;na(a|0,22448,5);B();}function Kh(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;if(Jb(a,J[b+8>>2],0)){Gd(b,c,d);}}function Kd(a,b){var c=0;c=a;a=(b>>>0)/100|0;return uc(uc(c,a),P(a,-100)+b|0)}function Jd(a,b){var c=0;c=a;a=(b>>>0)/1e4|0;return Kd(uc(c,a),P(a,-1e4)+b|0)}function Id(a,b){var c=0;c=a;a=(b>>>0)/1e6|0;return Jd(uc(c,a),P(a,-1e6)+b|0)}function yl(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;Xa[J[J[a>>2]+32>>2]](a,b,c,d)|0;}function rl(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;Xa[J[J[a>>2]+48>>2]](a,b,c,d)|0;}function gf(a){var b=0,c=0;b=Ua-16|0;Ua=b;c=b+12|0;J[c>>2]=a;cf(c);Ua=b+16|0;}function ti(a,b){b=b|0;if(b>>>0<=127){b=J[(b<<2)+11184>>2];}return b|0}function Vh(a,b,c){a=a|0;b=b|0;c=c|0;return J[b+4>>2]==(a|0)&J[b>>2]==(c|0)}function Cj(){J[6651]=0;J[6650]=22008;J[6650]=14144;Ye(26608);J[6650]=16704;}function Aj(){J[6655]=0;J[6654]=22008;J[6654]=14144;Ye(26624);J[6654]=16860;}function ri(a,b){b=b|0;if(b>>>0<=127){b=J[(b<<2)+9648>>2];}return b|0}function qj(a){if(!K[25260]){zc(25248,14004);H[25260]=1;}return 25248}function oj(a){if(!K[25292]){zc(25280,14088);H[25292]=1;}return 25280}function mj(a){if(!K[25196]){zc(25184,13932);H[25196]=1;}return 25184}function kj(a){if(!K[25228]){zc(25216,13968);H[25228]=1;}return 25216}function fm(a,b,c){a=a|0;b=b|0;c=c|0;return Xa[J[J[a>>2]+12>>2]](a,b,c)|0}function cm(a,b,c){a=a|0;b=b|0;c=c|0;return Xa[J[J[a>>2]+48>>2]](a,b,c)|0}function cj(a){if(!K[25244]){Kb(25232,2743);H[25244]=1;}return 25232}function aj(a){if(!K[25276]){Kb(25264,1928);H[25276]=1;}return 25264}function _i(a){if(!K[25180]){Kb(25168,1139);H[25180]=1;}return 25168}function Yi(a){if(!K[25212]){Kb(25200,2881);H[25212]=1;}return 25200}function Pc(a,b,c,d,e){c=c|0;e=e|0;J[e>>2]=c;return 3}function pc(a,b){a=a|0;H[a|0]=2;H[a+1|0]=3;H[a+2|0]=0;H[a+3|0]=4;}function om(a){a=a|0;J[a+16>>2]=J[a+16>>2]|1;if(H[a+20|0]&1){Aa();B();}}function vh(a,b,c,d,e){a=a|0;b=b|0;c=c|0;d=d|0;e=e|0;Xa[a|0](b,c,d,e);}function Pd(a){a=a|0;if(J[a+8>>2]!=(jb()|0)){cg(J[a+8>>2]);}return a|0}function xh(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;return Xa[a|0](b,c,d)|0}function dk(a,b,c,d,e,f){a=a|0;f=f|0;Ud(a,f);}function wc(a,b){return b?a<<24|(a&65280)<<8|(a>>>8&65280|a>>>24):a}function oi(a,b,c){b=b|0;c=c|0;return (b>>>0<128?b:c)<<24>>24}function Gc(a,b){a=a|0;b=b|0;var c=0;c=cb(b);J[a+4>>2]=b;J[a>>2]=c;}function bf(a,b){a:{if((a|0)==(b|0)){H[b+120|0]=0;break a}$a(b);}}function Fl(a){a=a|0;var b=0;b=J[6259]+1|0;J[6259]=b;J[a+4>>2]=b;}function tg(a,b){a=a|0;b=b|0;return Xa[J[J[a>>2]+28>>2]](a,b)|0}function Xk(a,b){a=a|0;b=b|0;a=Vd(a,b);J[a>>2]=22548;return a|0}function Vl(a,b){a=a|0;b=b|0;return Xa[J[J[a>>2]+44>>2]](a,b)|0}function Gl(a,b){a=a|0;b=b|0;return Xa[J[J[a>>2]+12>>2]](a,b)|0}function Ch(a){a=a|0;if(!a){return 0}return (Fe(a,22200)|0)!=0|0}function Ai(a){a=a|0;a=J[a+8>>2];if(!a){return 1}return Le(a)|0}function em(a,b){a=a|0;b=b|0;if(!J[5996]){J[5996]=a;J[5997]=b;}}function Qb(a,b){a=a|0;J[a>>2]=0;J[a+4>>2]=0;J[a+8>>2]=0;}function ab(a){if((K[a+11|0]&128)>>>7|0){$a(J[a>>2]);}return a}function $f(a){if(a>>>0>=1073741824){Pb();B();}return cb(a<<2)}function wh(a,b,c,d){a=a|0;b=b|0;c=c|0;d=d|0;Xa[a|0](b,c,d);}function ng(a){a=a|0;J[a>>2]=7960;ab(a+32|0);return Qc(a)|0}function fi(a,b,c){b=b|0;c=c|0;return ((b|0)<0?c:b)|0}function Xm(a){a=a|0;a=Vd(a,2284);J[a>>2]=22600;return a|0}function Xh(a,b,c){a=a|0;b=b|0;c=c|0;J[a+4>>2]=b;J[a>>2]=c;}function Cm(a){a=a|0;a=Tg(a,1241);J[a>>2]=22684;return a|0}function Ve(a){a=a|0;J[a>>2]=13856;ab(a+16|0);return a|0}function Ue(a){a=a|0;J[a>>2]=13816;ab(a+12|0);return a|0}function we(a){a=a|0;J[a>>2]=22500;De(a+4|0);return a|0}function Ff(a){a=a|0;J[a>>2]=22480;De(a+4|0);return a|0}function uc(a,b){b=(b<<1)+21664|0;return Bg(b,b+2|0,a)}function lm(a){a=a|0;return ke(J[J[a>>2]-12>>2]+a|0)|0}function _f(a,b){a=a|0;b=b|0;Xa[J[J[b>>2]+24>>2]](a,b);}function Zf(a,b){a=a|0;b=b|0;Xa[J[J[b>>2]+28>>2]](a,b);}function Xl(a){a=a|0;return gd(J[J[a>>2]-12>>2]+a|0)|0}function Wf(a,b){a=a|0;b=b|0;Xa[J[J[b>>2]+20>>2]](a,b);}function Sh(a){a=a|0;if(a){a=a-24|0;J[a>>2]=J[a>>2]+1;}}function Ml(a,b,c,d){a=a|0;c=c|0;d=d|0;Ad(a,c,d);}function Jl(a,b,c,d){a=a|0;c=c|0;d=d|0;ag(a,c,d);}function Fc(a,b){var c=0;c=$f(b);J[a+4>>2]=b;J[a>>2]=c;}function qm(a){a=a|0;return Xa[J[J[a>>2]+24>>2]](a)|0}function kf(a,b,c){a=qe(a,b,c,0,-2147483648);return a}function Xf(a){a=a|0;return Xa[J[J[a>>2]+16>>2]](a)|0}function Nf(a){a=a|0;return Xa[J[J[a>>2]+12>>2]](a)|0}function Sg(a,b){a=a|0;b=b|0;return ve(a,b,Gb(b))|0}function Ub(a,b){a=a|0;b=b|0;return xg(a)^xg(b)^1}function Tb(a,b){a=a|0;b=b|0;return rg(a)^rg(b)^1}function wl(a){if(H[24003]<0){$a(J[5998]);}}function jm(a){a=a|0;a=Ua-a&-16;Ua=a;return a|0}function Mg(a,b){if(!a){return 0}return pe(a,b)}function Am(a,b,c){a=a|0;return a|0}function nc(a,b,c){return Oa(18,a|0,b|0,c|0)|0}function lf(a,b,c){return -1}function km(a){a=a|0;ug(J[J[a>>2]-12>>2]+a|0);}function hf(a,b,c){a=qe(a,b,c,-1,-1);return a}function Wl(a){a=a|0;mg(J[J[a>>2]-12>>2]+a|0);}function bd(a,b,c,d){Ma(152,a|0,b|0,c|0,d|0);}function Ae(a,b,c,d){La(155,a|0,b|0,c|0,d|0);}function wn(a){a=a|0;J[a>>2]=7e3;return a|0}function uj(a){a=a|0;Xa[J[J[a>>2]+4>>2]](a);}function Ic(a){a=a|0;wg(J[a>>2]);return a|0}function Hc(a){a=a|0;qg(J[a>>2]);return a|0}function zj(a){a=a|0;qb(a,26616,pb(24916));}function yj(a){a=a|0;qb(a,26632,pb(24988));}function xj(a){a=a|0;qb(a,26640,pb(24996));}function bk(a){a=a|0;qb(a,26320,pb(24844));}function ak(a){a=a|0;qb(a,26328,pb(24852));}function _j(a){a=a|0;qb(a,26352,pb(25040));}function Zj(a){a=a|0;qb(a,26360,pb(25056));}function Xj(a){a=a|0;qb(a,26368,pb(25064));}function Wj(a){a=a|0;qb(a,26384,pb(25072));}function Vj(a){a=a|0;qb(a,26392,pb(25088));}function Uj(a){a=a|0;qb(a,26400,pb(25080));}function Tj(a){a=a|0;qb(a,26408,pb(25096));}function Sj(a){a=a|0;qb(a,26416,pb(25104));}function Rj(a){a=a|0;qb(a,26440,pb(25112));}function Qj(a){a=a|0;qb(a,26472,pb(24860));}function Pj(a){a=a|0;qb(a,26480,pb(24868));}function Oj(a){a=a|0;qb(a,26488,pb(24876));}function Nj(a){a=a|0;qb(a,26496,pb(24884));}function Mj(a){a=a|0;qb(a,26504,pb(24924));}function Lj(a){a=a|0;qb(a,26512,pb(24932));}function Kj(a){a=a|0;qb(a,26520,pb(24940));}function Jj(a){a=a|0;qb(a,26528,pb(24948));}function Ij(a){a=a|0;qb(a,26536,pb(24956));}function Hj(a){a=a|0;qb(a,26544,pb(24964));}function Gj(a){a=a|0;qb(a,26552,pb(24972));}function Fj(a){a=a|0;qb(a,26560,pb(24980));}function Ej(a){a=a|0;qb(a,26568,pb(24892));}function Dj(a){a=a|0;qb(a,26584,pb(24900));}function Bj(a){a=a|0;qb(a,26600,pb(24908));}function $j(a){a=a|0;qb(a,26336,pb(25048));}function le(a){a=a|0;return lc(a,25048)|0}function ke(a){a=a|0;Od(a+4|0);return a|0}function je(a){a=a|0;return lc(a,25040)|0}function im(a){a=a|0;return lc(a,24876)|0}function gc(a,b,c){Bg(b,b+c|0,a);return a}function de(a){a=a|0;return lc(a,25104)|0}function Zd(a){a=a|0;return lc(a,25112)|0}function Qi(a,b){a=a|0;b=b|0;Ud(a,b+16|0);}function Ki(a,b){a=a|0;b=b|0;Ud(a,b+12|0);}function Pi(a,b){a=a|0;zc(a,13888);}function Oi(a,b){a=a|0;zc(a,13908);}function zg(a,b){b=b|0;return b|0}function vf(a,b){a=a|0;Hf(a,1,45);}function ae(a,b,c){a=hf(a,b,c);return a}function Uf(a,b,c){a=kf(a,b,c);return a}function Ji(a,b){a=a|0;Kb(a,2331);}function Ii(a,b){a=a|0;Kb(a,2350);}function id(a){return 2147483647}function Ri(a){a=a|0;return J[a+12>>2]}function sb(a){a=a|0;return _c(8,a)|0}function Si(a){a=a|0;return J[a+8>>2]}function Cg(a,b){return -1}function Ce(a){a=a|0;return J[a+4>>2]}function $h(a,b){return 0}function Mi(a){a=a|0;return H[a+8|0]}function Li(a){a=a|0;return H[a+9|0]}function oc(a,b,c){return eb(a,b,c)}function lg(a){a=a|0;return we(a)|0}function ef(a){if(!K[a+4|0]){cf(a);}}function tf(a){return lc(a,24932)}function sf(a){return lc(a,24924)}function qf(a){return lc(a,24948)}function pf(a){return lc(a,24940)}function zh(a){return 2176}function yh(a){return 1304}function Ul(a){return 2042}function Bh(a){return 1969}function Ah(a){return 2524}function jd(a){return 127}function $b(a){a=a|0;return a|0}function ug(a){a=a|0;$a(ke(a));}function rd(a){a=a|0;$a(we(a));}function pj(a){ab(25248);}function nj(a){ab(25280);}function mi(a){a=a|0;$a(Je(a));}function mg(a){a=a|0;$a(gd(a));}function lj(a){ab(25184);}function jj(a){ab(25216);}function di(a){a=a|0;$a(Ie(a));}function bj(a){ab(25232);}function Zl(a){a=a|0;$a(ng(a));}function Zi(a){ab(25168);}function Xi(a){ab(25200);}function Xe(a){a=a|0;$a(Pd(a));}function Ti(a){a=a|0;$a(Ve(a));}function Ql(a){a=a|0;$a(oe(a));}function Ni(a){a=a|0;$a(Ue(a));}function Hi(a){a=a|0;$a(Ne(a));}function Dd(a){a=a|0;$a(Ff(a));}function Bm(a){a=a|0;$a(Qc(a));}function $i(a){ab(25264);}function vm(a){return -1}function ed(a){return 4}function Qd(a){return 1}function Ib(a){return 0}function Gf(a){return 2}function ne(a,b){}function df(a){af(a,J[a>>2]);}function Ye(a){J[a>>2]=jb();}function ub(){If(2284);B();}function nb(){If(1687);B();}function kb(a){a=a|0;$a(a);}function ym(a){a=a|0;Va=a;}function sm(){return Ua|0}function nm(a){a=a|0;Ua=a;}function ai(a){B();}function db(a){tb(a,0);}function Ph(){Ge();B();}function Ge(){za();B();}function Od(a){oe(a);}function Nd(a){}
// EMSCRIPTEN_END_FUNCS
e=K;p();var Xa=c([null,wl,ub,cb,nl,$b,Xk,Ff,nb,Ei,wi,nb,li,ci,Zh,Qh,Jh,Vd,rh,qh,ph,oh,nh,mh,lh,kh,jh,ih,hh,Bd,Kb,pk,gh,mc,nb,Pb,fh,eh,dh,ch,Zc,nb,bh,un,la,_g,ao,nb,$n,_n,$g,bo,nb,Zn,Wb,Yn,nb,hc,Xn,nb,Wn,Zg,nb,Vn,Un,Tn,Sn,Rn,Qn,Pn,On,Nn,Mn,Ln,Kn,Jn,In,Hn,Gn,Fn,En,Dn,Cn,Bn,nb,nb,An,zn,yn,sh,xn,Tg,we,nb,nb,nb,nb,Sg,rn,qn,ve,pn,sn,on,nn,mn,ln,kn,ue,tn,ra,ze,nb,hn,Lc,Kc,gn,fn,en,an,Yh,nb,dn,nb,Ob,Xm,cn,bn,se,Pg,_m,Zm,Ym,Rl,rm,im,hm,gm,pm,om,Vm,Um,nb,Tm,re,Sm,Rm,Qm,Pm,Om,Nm,Mm,Lm,Km,Ad,Jm,Im,$m,Hm,Gm,Fm,Em,re,Dm,Kc,Gg,Fg,Cm,bi,qm,le,tg,ie,$b,Uh,_h,Sl,lg,Pl,Ol,de,_f,Zf,nd,kb,ce,Ub,mm,Gl,Ic,Fl,El,Aa,Xf,Wf,yl,Nf,je,Zd,_f,Zf,ld,Tb,dm,tg,Hc,Xf,Wf,rl,Nf,sg,gl,bl,Ac,pg,Yk,Vl,cm,Sk,Bc,Nd,Ak,xf,zk,yk,xk,uf,vk,tk,sk,fm,rk,qk,uf,nk,mk,kk,jk,gk,fk,bk,ak,$j,_j,Zj,Yj,Xj,Wj,Vj,Uj,Tj,Sj,Rj,Qj,Pj,Oj,Nj,Mj,Lj,Kj,Jj,Ij,Hj,Gj,Fj,Ej,Dj,Cj,Bj,Aj,zj,yj,xj,wj,vj,gj,hj,ij,pj,nj,lj,jj,Ui,Vi,Wi,bj,$i,Zi,Xi,Gc,Th,Ge,wn,kb,vn,$h,kb,Qc,Bm,ne,Am,zm,xm,Ib,Ib,wm,vm,um,Cg,tm,Cg,ng,Zl,_l,Yl,bm,am,$l,gd,mg,Xl,Wl,ke,ug,lm,km,$b,kb,Ul,Xh,Wh,Vh,Tl,rd,Ce,oe,Ql,Ie,di,uj,Je,mi,ki,ji,ii,hi,zg,gi,fi,ei,Ne,Hi,Gi,Fi,Di,Ci,Ib,Bi,Ai,Ue,Ni,Mi,Li,Ki,Ji,Ii,Ve,Ti,Si,Ri,Qi,Pi,Oi,$b,kb,kb,yi,xi,vi,ui,ti,si,ri,qi,zg,pi,oi,ni,kb,Ke,Ke,Pc,Qd,Qd,zi,Qd,kb,Qe,Pe,Pc,Ib,Ib,Oe,ed,kb,Qe,Pe,Pc,Ib,Ib,Oe,ed,kb,Te,Se,Pc,Ib,Ib,Re,ed,kb,Te,Se,Pc,Ib,Ib,Re,ed,$b,kb,Nl,Ml,Ll,$b,kb,Kl,Jl,Il,kb,Hl,Dl,Cl,Bl,Rf,Rf,Al,zl,xl,vl,ul,kb,tl,sl,ql,pl,Jf,Jf,ol,ml,ll,kl,jl,kb,il,hl,fl,el,dl,cl,al,$k,kb,_k,Zk,Wk,Vk,Uk,Tk,Rk,Qk,$b,kb,Gf,Pk,Ok,Nk,Mk,Lk,Kk,fj,ej,dj,cj,aj,_i,Yi,$b,kb,Gf,Jk,Ik,Hk,Gk,Fk,Ek,tj,sj,rj,qj,oj,mj,kj,Pd,Xe,Dk,Pd,Xe,Ck,kb,jd,jd,Qb,Qb,Qb,wf,Ib,pc,pc,kb,jd,jd,Qb,Qb,Qb,wf,Ib,pc,pc,kb,id,id,Qb,Qb,Qb,vf,Ib,pc,pc,kb,id,id,Qb,Qb,Qb,vf,Ib,pc,pc,kb,Bk,wk,kb,uk,ok,kb,lk,ik,kb,hk,ek,kb,lf,dk,ne,kb,lf,ck,ne,lg,rd,$b,ai,Ph,za,$b,kb,Nd,Nd,Oh,Fh,Hh,Ih,kb,Ah,kb,zh,Dd,Ce,rd,Dd,Dd,rd,kb,yh,kb,Eh,Gh,Kh,kb,Nh,Mh,Lh,kb,Bh]);function Ya(){return G.byteLength/65536|0}return {"Y":ah,"Z":kb,"_":sb,"$":jn,"aa":Wm,"ba":Xa,"ca":em,"da":ym,"ea":sm,"fa":nm,"ga":jm,"ha":Rh,"ia":Sh,"ja":Dh,"ka":Ch,"la":xh,"ma":wh,"na":vh,"oa":vh,"pa":uh,"qa":th}}return Za(_a)}
// EMSCRIPTEN_END_ASM


)(info);},instantiate:function(binary,info){return {then:function(ok){var module=new WebAssembly.Module(binary);ok({"instance":new WebAssembly.Instance(module,info)});}}},RuntimeError:Error};wasmBinary=[];if(typeof WebAssembly!="object"){abort("no native wasm support detected");}var wasmMemory;var ABORT=false;function assert(condition,text){if(!condition){abort(text);}}var HEAP8,HEAPU8,HEAP32,HEAPU32;function updateMemoryViews(){var b=wasmMemory.buffer;Module["HEAP8"]=HEAP8=new Int8Array(b);Module["HEAP16"]=new Int16Array(b);Module["HEAPU8"]=HEAPU8=new Uint8Array(b);Module["HEAPU16"]=new Uint16Array(b);Module["HEAP32"]=HEAP32=new Int32Array(b);Module["HEAPU32"]=HEAPU32=new Uint32Array(b);Module["HEAPF32"]=new Float32Array(b);Module["HEAPF64"]=new Float64Array(b);}var INITIAL_MEMORY=Module["INITIAL_MEMORY"]||16777216;assert(INITIAL_MEMORY>=65536,"INITIAL_MEMORY should be larger than STACK_SIZE, was "+INITIAL_MEMORY+"! (STACK_SIZE="+65536+")");if(Module["wasmMemory"]){wasmMemory=Module["wasmMemory"];}else {wasmMemory=new WebAssembly.Memory({"initial":INITIAL_MEMORY/65536,"maximum":INITIAL_MEMORY/65536});}updateMemoryViews();INITIAL_MEMORY=wasmMemory.buffer.byteLength;var wasmTable;var __ATPRERUN__=[];var __ATINIT__=[];var __ATPOSTRUN__=[];function preRun(){if(Module["preRun"]){if(typeof Module["preRun"]=="function")Module["preRun"]=[Module["preRun"]];while(Module["preRun"].length){addOnPreRun(Module["preRun"].shift());}}callRuntimeCallbacks(__ATPRERUN__);}function initRuntime(){callRuntimeCallbacks(__ATINIT__);}function postRun(){if(Module["postRun"]){if(typeof Module["postRun"]=="function")Module["postRun"]=[Module["postRun"]];while(Module["postRun"].length){addOnPostRun(Module["postRun"].shift());}}callRuntimeCallbacks(__ATPOSTRUN__);}function addOnPreRun(cb){__ATPRERUN__.unshift(cb);}function addOnInit(cb){__ATINIT__.unshift(cb);}function addOnPostRun(cb){__ATPOSTRUN__.unshift(cb);}var runDependencies=0;var dependenciesFulfilled=null;function addRunDependency(id){runDependencies++;if(Module["monitorRunDependencies"]){Module["monitorRunDependencies"](runDependencies);}}function removeRunDependency(id){runDependencies--;if(Module["monitorRunDependencies"]){Module["monitorRunDependencies"](runDependencies);}if(runDependencies==0){if(dependenciesFulfilled){var callback=dependenciesFulfilled;dependenciesFulfilled=null;callback();}}}function abort(what){if(Module["onAbort"]){Module["onAbort"](what);}what="Aborted("+what+")";err(what);ABORT=true;what+=". Build with -sASSERTIONS for more info.";var e=new WebAssembly.RuntimeError(what);readyPromiseReject(e);throw e}var dataURIPrefix="data:application/octet-stream;base64,";function isDataURI(filename){return filename.startsWith(dataURIPrefix)}var wasmBinaryFile;wasmBinaryFile="bindings.wasm";if(!isDataURI(wasmBinaryFile)){wasmBinaryFile=locateFile(wasmBinaryFile);}function getBinarySync(file){if(file==wasmBinaryFile&&wasmBinary){return new Uint8Array(wasmBinary)}var binary=tryParseAsDataURI(file);if(binary){return binary}throw "sync fetching of the wasm failed: you can preload it to Module['wasmBinary'] manually, or emcc.py will do that for you when generating HTML (but not JS)"}function instantiateSync(file,info){var module;var binary=getBinarySync(file);module=new WebAssembly.Module(binary);var instance=new WebAssembly.Instance(module,info);return [instance,module]}function createWasm(){var info={"a":wasmImports};function receiveInstance(instance,module){var exports=instance.exports;wasmExports=exports;wasmTable=wasmExports["ba"];addOnInit(wasmExports["Y"]);removeRunDependency();return exports}addRunDependency();if(Module["instantiateWasm"]){try{return Module["instantiateWasm"](info,receiveInstance)}catch(e){err(`Module.instantiateWasm callback failed with error: ${e}`);readyPromiseReject(e);}}var result=instantiateSync(wasmBinaryFile,info);return receiveInstance(result[0])}var callRuntimeCallbacks=callbacks=>{while(callbacks.length>0){callbacks.shift()(Module);}};var UTF8Decoder=typeof TextDecoder!="undefined"?new TextDecoder("utf8"):undefined;var UTF8ArrayToString=(heapOrArray,idx,maxBytesToRead)=>{var endIdx=idx+maxBytesToRead;var endPtr=idx;while(heapOrArray[endPtr]&&!(endPtr>=endIdx))++endPtr;if(endPtr-idx>16&&heapOrArray.buffer&&UTF8Decoder){return UTF8Decoder.decode(heapOrArray.subarray(idx,endPtr))}var str="";while(idx<endPtr){var u0=heapOrArray[idx++];if(!(u0&128)){str+=String.fromCharCode(u0);continue}var u1=heapOrArray[idx++]&63;if((u0&224)==192){str+=String.fromCharCode((u0&31)<<6|u1);continue}var u2=heapOrArray[idx++]&63;if((u0&240)==224){u0=(u0&15)<<12|u1<<6|u2;}else {u0=(u0&7)<<18|u1<<12|u2<<6|heapOrArray[idx++]&63;}if(u0<65536){str+=String.fromCharCode(u0);}else {var ch=u0-65536;str+=String.fromCharCode(55296|ch>>10,56320|ch&1023);}}return str};var UTF8ToString=(ptr,maxBytesToRead)=>ptr?UTF8ArrayToString(HEAPU8,ptr,maxBytesToRead):"";var ___assert_fail=(condition,filename,line,func)=>{abort(`Assertion failed: ${UTF8ToString(condition)}, at: `+[filename?UTF8ToString(filename):"unknown filename",line,func?UTF8ToString(func):"unknown function"]);};var exceptionCaught=[];var uncaughtExceptionCount=0;var ___cxa_begin_catch=ptr=>{var info=new ExceptionInfo(ptr);if(!info.get_caught()){info.set_caught(true);uncaughtExceptionCount--;}info.set_rethrown(false);exceptionCaught.push(info);___cxa_increment_exception_refcount(info.excPtr);return info.get_exception_ptr()};var exceptionLast=0;var ___cxa_end_catch=()=>{_setThrew(0,0);var info=exceptionCaught.pop();___cxa_decrement_exception_refcount(info.excPtr);exceptionLast=0;};function ExceptionInfo(excPtr){this.excPtr=excPtr;this.ptr=excPtr-24;this.set_type=function(type){HEAPU32[this.ptr+4>>2]=type;};this.get_type=function(){return HEAPU32[this.ptr+4>>2]};this.set_destructor=function(destructor){HEAPU32[this.ptr+8>>2]=destructor;};this.get_destructor=function(){return HEAPU32[this.ptr+8>>2]};this.set_caught=function(caught){caught=caught?1:0;HEAP8[this.ptr+12>>0]=caught;};this.get_caught=function(){return HEAP8[this.ptr+12>>0]!=0};this.set_rethrown=function(rethrown){rethrown=rethrown?1:0;HEAP8[this.ptr+13>>0]=rethrown;};this.get_rethrown=function(){return HEAP8[this.ptr+13>>0]!=0};this.init=function(type,destructor){this.set_adjusted_ptr(0);this.set_type(type);this.set_destructor(destructor);};this.set_adjusted_ptr=function(adjustedPtr){HEAPU32[this.ptr+16>>2]=adjustedPtr;};this.get_adjusted_ptr=function(){return HEAPU32[this.ptr+16>>2]};this.get_exception_ptr=function(){var isPointer=___cxa_is_pointer_type(this.get_type());if(isPointer){return HEAPU32[this.excPtr>>2]}var adjusted=this.get_adjusted_ptr();if(adjusted!==0)return adjusted;return this.excPtr};}var ___resumeException=ptr=>{if(!exceptionLast){exceptionLast=ptr;}throw exceptionLast};var findMatchingCatch=args=>{var thrown=exceptionLast;if(!thrown){setTempRet0(0);return 0}var info=new ExceptionInfo(thrown);info.set_adjusted_ptr(thrown);var thrownType=info.get_type();if(!thrownType){setTempRet0(0);return thrown}for(var arg in args){var caughtType=args[arg];if(caughtType===0||caughtType===thrownType){break}var adjusted_ptr_addr=info.ptr+16;if(___cxa_can_catch(caughtType,thrownType,adjusted_ptr_addr)){setTempRet0(caughtType);return thrown}}setTempRet0(thrownType);return thrown};var ___cxa_find_matching_catch_2=()=>findMatchingCatch([]);var ___cxa_find_matching_catch_3=arg0=>findMatchingCatch([arg0]);var ___cxa_find_matching_catch_4=(arg0,arg1)=>findMatchingCatch([arg0,arg1]);var ___cxa_rethrow=()=>{var info=exceptionCaught.pop();if(!info){abort("no exception to throw");}var ptr=info.excPtr;if(!info.get_rethrown()){exceptionCaught.push(info);info.set_rethrown(true);info.set_caught(false);uncaughtExceptionCount++;}exceptionLast=ptr;throw exceptionLast};var ___cxa_throw=(ptr,type,destructor)=>{var info=new ExceptionInfo(ptr);info.init(type,destructor);exceptionLast=ptr;uncaughtExceptionCount++;throw exceptionLast};var ___cxa_uncaught_exceptions=()=>uncaughtExceptionCount;var _abort=()=>{abort("");};var _emscripten_memcpy_big=(dest,src,num)=>HEAPU8.copyWithin(dest,src,src+num);var abortOnCannotGrowMemory=requestedSize=>{abort("OOM");};var _emscripten_resize_heap=requestedSize=>{HEAPU8.length;abortOnCannotGrowMemory();};var ENV={};var getExecutableName=()=>thisProgram||"./this.program";var getEnvStrings=()=>{if(!getEnvStrings.strings){var lang=(typeof navigator=="object"&&navigator.languages&&navigator.languages[0]||"C").replace("-","_")+".UTF-8";var env={"USER":"web_user","LOGNAME":"web_user","PATH":"/","PWD":"/","HOME":"/home/web_user","LANG":lang,"_":getExecutableName()};for(var x in ENV){if(ENV[x]===undefined)delete env[x];else env[x]=ENV[x];}var strings=[];for(var x in env){strings.push(`${x}=${env[x]}`);}getEnvStrings.strings=strings;}return getEnvStrings.strings};var stringToAscii=(str,buffer)=>{for(var i=0;i<str.length;++i){HEAP8[buffer++>>0]=str.charCodeAt(i);}HEAP8[buffer>>0]=0;};var _environ_get=(__environ,environ_buf)=>{var bufSize=0;getEnvStrings().forEach((string,i)=>{var ptr=environ_buf+bufSize;HEAPU32[__environ+i*4>>2]=ptr;stringToAscii(string,ptr);bufSize+=string.length+1;});return 0};var _environ_sizes_get=(penviron_count,penviron_buf_size)=>{var strings=getEnvStrings();HEAPU32[penviron_count>>2]=strings.length;var bufSize=0;strings.forEach(string=>bufSize+=string.length+1);HEAPU32[penviron_buf_size>>2]=bufSize;return 0};var _llvm_eh_typeid_for=type=>type;var isLeapYear=year=>year%4===0&&(year%100!==0||year%400===0);var arraySum=(array,index)=>{var sum=0;for(var i=0;i<=index;sum+=array[i++]){}return sum};var MONTH_DAYS_LEAP=[31,29,31,30,31,30,31,31,30,31,30,31];var MONTH_DAYS_REGULAR=[31,28,31,30,31,30,31,31,30,31,30,31];var addDays=(date,days)=>{var newDate=new Date(date.getTime());while(days>0){var leap=isLeapYear(newDate.getFullYear());var currentMonth=newDate.getMonth();var daysInCurrentMonth=(leap?MONTH_DAYS_LEAP:MONTH_DAYS_REGULAR)[currentMonth];if(days>daysInCurrentMonth-newDate.getDate()){days-=daysInCurrentMonth-newDate.getDate()+1;newDate.setDate(1);if(currentMonth<11){newDate.setMonth(currentMonth+1);}else {newDate.setMonth(0);newDate.setFullYear(newDate.getFullYear()+1);}}else {newDate.setDate(newDate.getDate()+days);return newDate}}return newDate};var lengthBytesUTF8=str=>{var len=0;for(var i=0;i<str.length;++i){var c=str.charCodeAt(i);if(c<=127){len++;}else if(c<=2047){len+=2;}else if(c>=55296&&c<=57343){len+=4;++i;}else {len+=3;}}return len};var stringToUTF8Array=(str,heap,outIdx,maxBytesToWrite)=>{if(!(maxBytesToWrite>0))return 0;var startIdx=outIdx;var endIdx=outIdx+maxBytesToWrite-1;for(var i=0;i<str.length;++i){var u=str.charCodeAt(i);if(u>=55296&&u<=57343){var u1=str.charCodeAt(++i);u=65536+((u&1023)<<10)|u1&1023;}if(u<=127){if(outIdx>=endIdx)break;heap[outIdx++]=u;}else if(u<=2047){if(outIdx+1>=endIdx)break;heap[outIdx++]=192|u>>6;heap[outIdx++]=128|u&63;}else if(u<=65535){if(outIdx+2>=endIdx)break;heap[outIdx++]=224|u>>12;heap[outIdx++]=128|u>>6&63;heap[outIdx++]=128|u&63;}else {if(outIdx+3>=endIdx)break;heap[outIdx++]=240|u>>18;heap[outIdx++]=128|u>>12&63;heap[outIdx++]=128|u>>6&63;heap[outIdx++]=128|u&63;}}heap[outIdx]=0;return outIdx-startIdx};function intArrayFromString(stringy,dontAddNull,length){var len=length>0?length:lengthBytesUTF8(stringy)+1;var u8array=new Array(len);var numBytesWritten=stringToUTF8Array(stringy,u8array,0,u8array.length);if(dontAddNull)u8array.length=numBytesWritten;return u8array}var writeArrayToMemory=(array,buffer)=>{HEAP8.set(array,buffer);};var _strftime=(s,maxsize,format,tm)=>{var tm_zone=HEAPU32[tm+40>>2];var date={tm_sec:HEAP32[tm>>2],tm_min:HEAP32[tm+4>>2],tm_hour:HEAP32[tm+8>>2],tm_mday:HEAP32[tm+12>>2],tm_mon:HEAP32[tm+16>>2],tm_year:HEAP32[tm+20>>2],tm_wday:HEAP32[tm+24>>2],tm_yday:HEAP32[tm+28>>2],tm_isdst:HEAP32[tm+32>>2],tm_gmtoff:HEAP32[tm+36>>2],tm_zone:tm_zone?UTF8ToString(tm_zone):""};var pattern=UTF8ToString(format);var EXPANSION_RULES_1={"%c":"%a %b %d %H:%M:%S %Y","%D":"%m/%d/%y","%F":"%Y-%m-%d","%h":"%b","%r":"%I:%M:%S %p","%R":"%H:%M","%T":"%H:%M:%S","%x":"%m/%d/%y","%X":"%H:%M:%S","%Ec":"%c","%EC":"%C","%Ex":"%m/%d/%y","%EX":"%H:%M:%S","%Ey":"%y","%EY":"%Y","%Od":"%d","%Oe":"%e","%OH":"%H","%OI":"%I","%Om":"%m","%OM":"%M","%OS":"%S","%Ou":"%u","%OU":"%U","%OV":"%V","%Ow":"%w","%OW":"%W","%Oy":"%y"};for(var rule in EXPANSION_RULES_1){pattern=pattern.replace(new RegExp(rule,"g"),EXPANSION_RULES_1[rule]);}var WEEKDAYS=["Sunday","Monday","Tuesday","Wednesday","Thursday","Friday","Saturday"];var MONTHS=["January","February","March","April","May","June","July","August","September","October","November","December"];function leadingSomething(value,digits,character){var str=typeof value=="number"?value.toString():value||"";while(str.length<digits){str=character[0]+str;}return str}function leadingNulls(value,digits){return leadingSomething(value,digits,"0")}function compareByDay(date1,date2){function sgn(value){return value<0?-1:value>0?1:0}var compare;if((compare=sgn(date1.getFullYear()-date2.getFullYear()))===0){if((compare=sgn(date1.getMonth()-date2.getMonth()))===0){compare=sgn(date1.getDate()-date2.getDate());}}return compare}function getFirstWeekStartDate(janFourth){switch(janFourth.getDay()){case 0:return new Date(janFourth.getFullYear()-1,11,29);case 1:return janFourth;case 2:return new Date(janFourth.getFullYear(),0,3);case 3:return new Date(janFourth.getFullYear(),0,2);case 4:return new Date(janFourth.getFullYear(),0,1);case 5:return new Date(janFourth.getFullYear()-1,11,31);case 6:return new Date(janFourth.getFullYear()-1,11,30)}}function getWeekBasedYear(date){var thisDate=addDays(new Date(date.tm_year+1900,0,1),date.tm_yday);var janFourthThisYear=new Date(thisDate.getFullYear(),0,4);var janFourthNextYear=new Date(thisDate.getFullYear()+1,0,4);var firstWeekStartThisYear=getFirstWeekStartDate(janFourthThisYear);var firstWeekStartNextYear=getFirstWeekStartDate(janFourthNextYear);if(compareByDay(firstWeekStartThisYear,thisDate)<=0){if(compareByDay(firstWeekStartNextYear,thisDate)<=0){return thisDate.getFullYear()+1}return thisDate.getFullYear()}return thisDate.getFullYear()-1}var EXPANSION_RULES_2={"%a":date=>WEEKDAYS[date.tm_wday].substring(0,3),"%A":date=>WEEKDAYS[date.tm_wday],"%b":date=>MONTHS[date.tm_mon].substring(0,3),"%B":date=>MONTHS[date.tm_mon],"%C":date=>{var year=date.tm_year+1900;return leadingNulls(year/100|0,2)},"%d":date=>leadingNulls(date.tm_mday,2),"%e":date=>leadingSomething(date.tm_mday,2," "),"%g":date=>getWeekBasedYear(date).toString().substring(2),"%G":date=>getWeekBasedYear(date),"%H":date=>leadingNulls(date.tm_hour,2),"%I":date=>{var twelveHour=date.tm_hour;if(twelveHour==0)twelveHour=12;else if(twelveHour>12)twelveHour-=12;return leadingNulls(twelveHour,2)},"%j":date=>leadingNulls(date.tm_mday+arraySum(isLeapYear(date.tm_year+1900)?MONTH_DAYS_LEAP:MONTH_DAYS_REGULAR,date.tm_mon-1),3),"%m":date=>leadingNulls(date.tm_mon+1,2),"%M":date=>leadingNulls(date.tm_min,2),"%n":()=>"\n","%p":date=>{if(date.tm_hour>=0&&date.tm_hour<12){return "AM"}return "PM"},"%S":date=>leadingNulls(date.tm_sec,2),"%t":()=>"\t","%u":date=>date.tm_wday||7,"%U":date=>{var days=date.tm_yday+7-date.tm_wday;return leadingNulls(Math.floor(days/7),2)},"%V":date=>{var val=Math.floor((date.tm_yday+7-(date.tm_wday+6)%7)/7);if((date.tm_wday+371-date.tm_yday-2)%7<=2){val++;}if(!val){val=52;var dec31=(date.tm_wday+7-date.tm_yday-1)%7;if(dec31==4||dec31==5&&isLeapYear(date.tm_year%400-1)){val++;}}else if(val==53){var jan1=(date.tm_wday+371-date.tm_yday)%7;if(jan1!=4&&(jan1!=3||!isLeapYear(date.tm_year)))val=1;}return leadingNulls(val,2)},"%w":date=>date.tm_wday,"%W":date=>{var days=date.tm_yday+7-(date.tm_wday+6)%7;return leadingNulls(Math.floor(days/7),2)},"%y":date=>(date.tm_year+1900).toString().substring(2),"%Y":date=>date.tm_year+1900,"%z":date=>{var off=date.tm_gmtoff;var ahead=off>=0;off=Math.abs(off)/60;off=off/60*100+off%60;return (ahead?"+":"-")+String("0000"+off).slice(-4)},"%Z":date=>date.tm_zone,"%%":()=>"%"};pattern=pattern.replace(/%%/g,"\0\0");for(var rule in EXPANSION_RULES_2){if(pattern.includes(rule)){pattern=pattern.replace(new RegExp(rule,"g"),EXPANSION_RULES_2[rule](date));}}pattern=pattern.replace(/\0\0/g,"%");var bytes=intArrayFromString(pattern,false);if(bytes.length>maxsize){return 0}writeArrayToMemory(bytes,s);return bytes.length-1};var _strftime_l=(s,maxsize,format,tm,loc)=>_strftime(s,maxsize,format,tm);var wasmTableMirror=[];var getWasmTableEntry=funcPtr=>{var func=wasmTableMirror[funcPtr];if(!func){if(funcPtr>=wasmTableMirror.length)wasmTableMirror.length=funcPtr+1;wasmTableMirror[funcPtr]=func=wasmTable.get(funcPtr);}return func};var getCFunc=ident=>{var func=Module["_"+ident];return func};var stringToUTF8=(str,outPtr,maxBytesToWrite)=>stringToUTF8Array(str,HEAPU8,outPtr,maxBytesToWrite);var stringToUTF8OnStack=str=>{var size=lengthBytesUTF8(str)+1;var ret=stackAlloc(size);stringToUTF8(str,ret,size);return ret};var ccall=(ident,returnType,argTypes,args,opts)=>{var toC={"string":str=>{var ret=0;if(str!==null&&str!==undefined&&str!==0){ret=stringToUTF8OnStack(str);}return ret},"array":arr=>{var ret=stackAlloc(arr.length);writeArrayToMemory(arr,ret);return ret}};function convertReturnValue(ret){if(returnType==="string"){return UTF8ToString(ret)}if(returnType==="boolean")return Boolean(ret);return ret}var func=getCFunc(ident);var cArgs=[];var stack=0;if(args){for(var i=0;i<args.length;i++){var converter=toC[argTypes[i]];if(converter){if(stack===0)stack=stackSave();cArgs[i]=converter(args[i]);}else {cArgs[i]=args[i];}}}var ret=func.apply(null,cArgs);function onDone(ret){if(stack!==0)stackRestore(stack);return convertReturnValue(ret)}ret=onDone(ret);return ret};var cwrap=(ident,returnType,argTypes,opts)=>{var numericArgs=!argTypes||argTypes.every(type=>type==="number"||type==="boolean");var numericRet=returnType!=="string";if(numericRet&&numericArgs&&!opts){return getCFunc(ident)}return function(){return ccall(ident,returnType,argTypes,arguments)}};var wasmImports={p:___assert_fail,t:___cxa_begin_catch,v:___cxa_end_catch,b:___cxa_find_matching_catch_2,j:___cxa_find_matching_catch_3,V:___cxa_find_matching_catch_4,E:___cxa_rethrow,r:___cxa_throw,L:___cxa_uncaught_exceptions,h:___resumeException,D:_abort,M:_emscripten_memcpy_big,U:_emscripten_resize_heap,J:_environ_get,K:_environ_sizes_get,G:invoke_i,d:invoke_ii,u:invoke_iiddi,c:invoke_iii,W:invoke_iiid,f:invoke_iiii,C:invoke_iiiid,m:invoke_iiiii,B:invoke_iiiiii,o:invoke_iiiiiii,F:invoke_iiiiiiii,y:invoke_iiiiiiiiiiii,N:invoke_iiij,S:invoke_iij,g:invoke_v,l:invoke_vi,H:invoke_vid,e:invoke_vii,T:invoke_viid,i:invoke_viii,I:invoke_viiidi,k:invoke_viiii,z:invoke_viiiid,n:invoke_viiiidi,s:invoke_viiiii,q:invoke_viiiiiii,w:invoke_viiiiiiiiii,x:invoke_viiiiiiiiiiiiiii,O:invoke_viiij,P:invoke_viij,R:invoke_vij,Q:invoke_viji,A:_llvm_eh_typeid_for,a:wasmMemory,X:_strftime_l};var wasmExports=createWasm();wasmExports["Y"];wasmExports["__errno_location"];wasmExports["__cxa_free_exception"];Module["_free"]=wasmExports["Z"];Module["_malloc"]=wasmExports["_"];Module["_miniscript_compile"]=wasmExports["$"];Module["_miniscript_analyze"]=wasmExports["aa"];var _setThrew=wasmExports["ca"];var setTempRet0=wasmExports["da"];var stackSave=wasmExports["ea"];var stackRestore=wasmExports["fa"];var stackAlloc=wasmExports["ga"];var ___cxa_decrement_exception_refcount=wasmExports["ha"];var ___cxa_increment_exception_refcount=wasmExports["ia"];var ___cxa_can_catch=wasmExports["ja"];var ___cxa_is_pointer_type=wasmExports["ka"];var dynCall_iij=Module["dynCall_iij"]=wasmExports["la"];var dynCall_vij=Module["dynCall_vij"]=wasmExports["ma"];var dynCall_viji=Module["dynCall_viji"]=wasmExports["na"];var dynCall_viij=Module["dynCall_viij"]=wasmExports["oa"];var dynCall_viiij=Module["dynCall_viiij"]=wasmExports["pa"];var dynCall_iiij=Module["dynCall_iiij"]=wasmExports["qa"];function invoke_v(index){var sp=stackSave();try{getWasmTableEntry(index)();}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_ii(index,a1){var sp=stackSave();try{return getWasmTableEntry(index)(a1)}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_iii(index,a1,a2){var sp=stackSave();try{return getWasmTableEntry(index)(a1,a2)}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_iiii(index,a1,a2,a3){var sp=stackSave();try{return getWasmTableEntry(index)(a1,a2,a3)}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_iiiii(index,a1,a2,a3,a4){var sp=stackSave();try{return getWasmTableEntry(index)(a1,a2,a3,a4)}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_viii(index,a1,a2,a3){var sp=stackSave();try{getWasmTableEntry(index)(a1,a2,a3);}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_vii(index,a1,a2){var sp=stackSave();try{getWasmTableEntry(index)(a1,a2);}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_iiiid(index,a1,a2,a3,a4){var sp=stackSave();try{return getWasmTableEntry(index)(a1,a2,a3,a4)}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_iiid(index,a1,a2,a3){var sp=stackSave();try{return getWasmTableEntry(index)(a1,a2,a3)}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_viiii(index,a1,a2,a3,a4){var sp=stackSave();try{getWasmTableEntry(index)(a1,a2,a3,a4);}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_iiddi(index,a1,a2,a3,a4){var sp=stackSave();try{return getWasmTableEntry(index)(a1,a2,a3,a4)}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_viiiii(index,a1,a2,a3,a4,a5){var sp=stackSave();try{getWasmTableEntry(index)(a1,a2,a3,a4,a5);}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_viiidi(index,a1,a2,a3,a4,a5){var sp=stackSave();try{getWasmTableEntry(index)(a1,a2,a3,a4,a5);}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_viiiidi(index,a1,a2,a3,a4,a5,a6){var sp=stackSave();try{getWasmTableEntry(index)(a1,a2,a3,a4,a5,a6);}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_iiiiii(index,a1,a2,a3,a4,a5){var sp=stackSave();try{return getWasmTableEntry(index)(a1,a2,a3,a4,a5)}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_viiiid(index,a1,a2,a3,a4,a5){var sp=stackSave();try{getWasmTableEntry(index)(a1,a2,a3,a4,a5);}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_iiiiiii(index,a1,a2,a3,a4,a5,a6){var sp=stackSave();try{return getWasmTableEntry(index)(a1,a2,a3,a4,a5,a6)}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_vid(index,a1,a2){var sp=stackSave();try{getWasmTableEntry(index)(a1,a2);}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_i(index){var sp=stackSave();try{return getWasmTableEntry(index)()}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_vi(index,a1){var sp=stackSave();try{getWasmTableEntry(index)(a1);}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_iiiiiiii(index,a1,a2,a3,a4,a5,a6,a7){var sp=stackSave();try{return getWasmTableEntry(index)(a1,a2,a3,a4,a5,a6,a7)}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_viiiiiii(index,a1,a2,a3,a4,a5,a6,a7){var sp=stackSave();try{getWasmTableEntry(index)(a1,a2,a3,a4,a5,a6,a7);}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_iiiiiiiiiiii(index,a1,a2,a3,a4,a5,a6,a7,a8,a9,a10,a11){var sp=stackSave();try{return getWasmTableEntry(index)(a1,a2,a3,a4,a5,a6,a7,a8,a9,a10,a11)}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_viiiiiiiiii(index,a1,a2,a3,a4,a5,a6,a7,a8,a9,a10){var sp=stackSave();try{getWasmTableEntry(index)(a1,a2,a3,a4,a5,a6,a7,a8,a9,a10);}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_viiiiiiiiiiiiiii(index,a1,a2,a3,a4,a5,a6,a7,a8,a9,a10,a11,a12,a13,a14,a15){var sp=stackSave();try{getWasmTableEntry(index)(a1,a2,a3,a4,a5,a6,a7,a8,a9,a10,a11,a12,a13,a14,a15);}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_viid(index,a1,a2,a3){var sp=stackSave();try{getWasmTableEntry(index)(a1,a2,a3);}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_iij(index,a1,a2,a3){var sp=stackSave();try{return dynCall_iij(index,a1,a2,a3)}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_vij(index,a1,a2,a3){var sp=stackSave();try{dynCall_vij(index,a1,a2,a3);}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_viji(index,a1,a2,a3,a4){var sp=stackSave();try{dynCall_viji(index,a1,a2,a3,a4);}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_viij(index,a1,a2,a3,a4){var sp=stackSave();try{dynCall_viij(index,a1,a2,a3,a4);}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_viiij(index,a1,a2,a3,a4,a5){var sp=stackSave();try{dynCall_viiij(index,a1,a2,a3,a4,a5);}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function invoke_iiij(index,a1,a2,a3,a4){var sp=stackSave();try{return dynCall_iiij(index,a1,a2,a3,a4)}catch(e){stackRestore(sp);if(e!==e+0)throw e;_setThrew(1,0);}}function intArrayFromBase64(s){try{var decoded=atob(s);var bytes=new Uint8Array(decoded.length);for(var i=0;i<decoded.length;++i){bytes[i]=decoded.charCodeAt(i);}return bytes}catch(_){throw new Error("Converting base64 string to bytes failed.")}}function tryParseAsDataURI(filename){if(!isDataURI(filename)){return}return intArrayFromBase64(filename.slice(dataURIPrefix.length))}Module["cwrap"]=cwrap;Module["UTF8ToString"]=UTF8ToString;var calledRun;dependenciesFulfilled=function runCaller(){if(!calledRun)run();if(!calledRun)dependenciesFulfilled=runCaller;};function run(){if(runDependencies>0){return}preRun();if(runDependencies>0){return}function doRun(){if(calledRun)return;calledRun=true;Module["calledRun"]=true;if(ABORT)return;initRuntime();readyPromiseResolve(Module);if(Module["onRuntimeInitialized"])Module["onRuntimeInitialized"]();postRun();}if(Module["setStatus"]){Module["setStatus"]("Running...");setTimeout(function(){setTimeout(function(){Module["setStatus"]("");},1);doRun();},1);}else {doRun();}}if(Module["preInit"]){if(typeof Module["preInit"]=="function")Module["preInit"]=[Module["preInit"]];while(Module["preInit"].length>0){Module["preInit"].pop()();}}run();


  return moduleArg
}

);
})();
module.exports = Module;
});

// Initial author: Pieter Wuille ( https://github.com/sipa/miniscript/blob/master/index.html)
const Module = bindings();

const em_miniscript_compile = Module.cwrap('miniscript_compile', 'none', [
  'string',
  'number',
  'number',
  'number',
  'number',
  'number',
  'number'
]);
const em_miniscript_analyze = Module.cwrap('miniscript_analyze', 'none', [
  'string',
  'number',
  'number',
  'number',
  'number'
]);

const cleanAsm = asm =>
  asm
    .trim()
    .replace(/\n/g, ' ')
    .replace(/ +(?= )/g, '');

/**
 * @typedef {Object} CompilePolicyResult
 * @property {string} miniscript - The compiled miniscript expression.
 * @property {string} asm - The compiled miniscript as Bitcoin asm code.
 * @property {boolean} issane - Whether the miniscript is sane at the top level.
 * @property {boolean} issanesublevel - Whether the miniscript is sane at the sublevel.
 */

/**
 * @typedef {Object} CompileMiniscriptResult
 * @property {string} asm - The Bitcoin asm code of the compiled miniscript expression.
 * @property {boolean} issane - Whether the miniscript is sane at the top level.
 * @property {boolean} issanesublevel - Whether the miniscript is sane at the sublevel.
 */


/**
 * Compiles a miniscript policy into a miniscript expression (if possible).
 * @Function
 *
 * @param {string} policy - The miniscript policy to compile.
 * @returns {CompilePolicyResult}
 */
const compilePolicy = policy => {
  const miniscript = Module._malloc(10000);
  const cost = Module._malloc(500);
  const asm = Module._malloc(100000);
  const issane = Module._malloc(10);
  const issanesublevel = Module._malloc(10);
  em_miniscript_compile(
    policy,
    miniscript,
    10000,
    cost,
    500,
    asm,
    100000,
    issane,
    10,
    issanesublevel,
    10
  );
  const result = {
    miniscript: Module.UTF8ToString(miniscript),
    asm: cleanAsm(Module.UTF8ToString(asm)),
    issane: Module.UTF8ToString(issane) === 'true' ? true : false,
    issanesublevel:
      Module.UTF8ToString(issanesublevel) === 'true' ? true : false
  };
  Module._free(miniscript);
  Module._free(cost);
  Module._free(asm);
  Module._free(issane);
  Module._free(issanesublevel);

  return result;
};

/**
 * Compiles a miniscript expression and returns its asm code.
 * @Function
 *
 * @param {string} miniscript - A miniscript expression.
 * @returns {CompileMiniscriptResult}
 */
const compileMiniscript = miniscript => {
  const analysis = Module._malloc(50000);
  const asm = Module._malloc(100000);
  const issane = Module._malloc(10);
  const issanesublevel = Module._malloc(10);
  em_miniscript_analyze(
    miniscript,
    analysis,
    50000,
    asm,
    100000,
    issane,
    10,
    issanesublevel,
    10
  );
  const result_asm = Module.UTF8ToString(asm);
  const result_issane = Module.UTF8ToString(issane);
  const result_issanesublebel = Module.UTF8ToString(issanesublevel);
  Module._free(analysis);
  Module._free(asm);
  Module._free(issane);
  Module._free(issanesublevel);

  return {
    asm: cleanAsm(result_asm),
    issane: result_issane === 'true' ? true : false,
    issanesublevel: result_issanesublebel === 'true' ? true : false
  };
};

// Copyright (c) 2022 Jose-Luis Landabaso - https://bitcoinerlab.com

/**
 * @typedef {Object} Solution
 * @property {number} nSequence - the maximum nSequence time of all the sat() or dsat() expressions in the solution.
 * @property {number} nLockTime - the maximum nLockTime of all the sat() or dsat() expressions in the solution.
 * @property {string} asm - the resulting witness after combining all the sat() or dsat() expressions.
 */

/**
 * @typedef {Object} Satisfactions
 * @property {Solution[]} sats - An array of {@link Solution} objects representing the sat() expressions.
 * @property {Solution[]} dsats - An array of {@link Solution} objects representing the dsat() expressions.
 * @see {@link Solution}
 */

/**
 * @typedef {Object} SatisfierResult
 * @property {Solution[]} nonMalleableSats - An array of {@link Solution} objects representing the non-malleable sat() expressions.
 * @property {Solution[]} malleableSats - An array of {@link Solution} objects representing the malleable sat() expressions.
 * @property {Solution[]} unknownSats - An array of {@link Solution} objects representing the sat() expressions that contain some of the `unknown` pieces of information.
 * @see {@link Solution}
 */


/**
 * Computes the weight units (WU) of a witness.
 * @param {string} asm - the witness to compute the WU of.
 * @returns {number} the weight units (WU) of the witness.
 */
function witnessWU(asm) {
  // Split the witness string into an array of substrings
  // a miniscipt witness is either, <sig..., <sha256..., <hash256...,
  // <ripemd160..., <hash160...,  <... (for pubkeys) 0 or 1
  const substrings = asm.split(' ');

  // Initialize the sum to 0
  let wu = 0;

  // Iterate over the array of substrings
  for (const substring of substrings) {
    if (substring === '') ;
    // Check if the substring starts with "<sig"
    else if (substring.startsWith('<sig')) {
      //https://en.bitcoin.it/wiki/BIP_0137
      //Signatures are either 73, 72, or 71 bytes long
      //Also https://bitcoin.stackexchange.com/a/77192/89665
      wu += 74; //73 + push op code
    }
    //
    // preimages:
    else if (
      substring.startsWith('<sha256_preimage(') ||
      substring.startsWith('<hash256_preimage(') ||
      substring.startsWith('<ripemd160_preimage(') ||
      substring.startsWith('<hash160_preimage(') ||
      substring.startsWith('<random_preimage()>')
    ) {
      //any preimage is checked against <32> with SIZE <32> EQUALVERIFY:
      //See the miniscript Translation table:
      //https://bitcoin.sipa.be/miniscript/
      //SIZE <32> EQUALVERIFY {SHA256,RIPEMD,160,HASH160,HASH256} <h> EQUAL
      wu += 33; //32 + push op code
    }
    // Pub keys
    else if (substring.startsWith('<')) {
      //https://en.bitcoin.it/wiki/BIP_0137
      //Compressed public keys are 33 bytes
      wu += 34; //33 + push op code
    } else if (substring === '1' || substring === '0') {
      wu += 1;
    } else {
      throw new Error(`Invalid witness substring ${substring}`);
    }
  }

  // Return the final wu
  return wu;
}

/**
 * Performs a malleability analysis on an array of sat() solutions.
 * @param {Solution[]} sats - the array of sat() solutions to perform the analysis on.
 * @returns {Object} An object with two keys:
 *   - `nonMalleableSats`: an array of {@link Solution} objects representing the non-malleable sat() expressions.
 *   - `malleableSats`: an array of {@link Solution} objects representing the malleable sat() expressions.
 * @see {@link Solution}
 */
function malleabilityAnalysis(sats) {
  sats = sats
    .map(sat => {
      //Deep copy the objects so that this function is pure
      //(does not mutate sats)
      sat = { ...sat };
      //Extract the signatures used in this witness as an array
      sat.signatures = sat.asm.split(' ').filter(op => {
        return op.startsWith('<sig');
      });
      //A non-zero solution without a signature is malleable, and a solution
      //without signature is unacceptable anyway
      if (sat.signatures.length === 0) {
        sat.dontuse = true;
      }
      //<random_preimage()> is a dissatisfaction for a preimage. It is
      //interchangable for any 32 bytes random number and thus, it is malleable.
      if (sat.asm.includes('<random_preimage()>')) {
        sat.dontuse = true;
      }
      return sat;
    })
    // Sort sats by weight unit in ascending order
    .sort((a, b) => witnessWU(a.asm) - witnessWU(b.asm));

  for (const sat of sats) {
    //For the same nLockTime and nSequence, check if otherSat signatures are a
    //subset of sat. If this is the case then sat cannot be used.
    //"For the same nLockTime and nSequence" condition is set because signatures
    //change for each tuple of (nLockTime, nSequence):
    for (const otherSat of sats) {
      if (
        otherSat !== sat &&
        //for the same nLockTime and nSequence
        otherSat.nLockTime === sat.nLockTime &&
        otherSat.nSequence === sat.nSequence &&
        //is otherSat.signatures equal or a subset of sat.signatures?
        otherSat.signatures.every(sig => sat.signatures.includes(sig))
      ) {
        //sat is for example <sig(key1)> <sig(key2)> and otherSat is <sig(key1)>
        //otherSat cannot be used because an attacker could use it to create
        //<sig(key1)
        sat.dontuse = true;
      }
    }
  }

  const nonMalleableSats = sats.filter(sat => !sat.dontuse);
  const malleableSats = sats.filter(sat => sat.dontuse);

  //Clean the objects before returning them
  for (const sats of [nonMalleableSats, malleableSats]) {
    sats.forEach(sat => {
      delete sat.signatures;
      delete sat.dontuse;
    });
  }

  return { nonMalleableSats, malleableSats };
}

/**
 * Determines whether the specified argument of the given miniscript expression
 * is a scalar.
 * @param {string} functionName - the name of the function to check.
 * @param {number} argPosition - the position of the argument to check,
 * starting from 0.
 * @returns {boolean} `true` if the specified argument of the given function is
 * a scalar; `false` otherwise.
 */
function isScalarArg(functionName, argPosition) {
  switch (functionName) {
    case 'pk_k':
    case 'pk_h':
    case 'older':
    case 'after':
    case 'sha256':
    case 'ripemd160':
    case 'hash256':
    case 'hash160':
    case 'multi':
      return true;
    case 'thresh':
      if (argPosition === 0) return true;
  }
  return false;
}

/**
 * Evaluates a miniscript expression and returns its satisfactions.
 *
 * This function is called recursively to find subexpressions
 * within subexpressions until all the arguments of a subexpression are
 * scalars.
 *
 * @param {string} miniscript - A miniscript expression.
 *
 * @returns {Satisfactions} - The satisfactions.
 */
const evaluate = miniscript => {
  if (typeof miniscript !== 'string')
    throw new Error('Invalid expression: ' + miniscript);
  //convert wrappers like this "sln:" into "s:l:n:"
  while (miniscript.match(/^[a-z]{2,}:/)) {
    miniscript = miniscript.replace(/^[a-z]{2,}:/, match =>
      match.replace(match[0], match[0] + ':')
    );
  }
  //https://bitcoin.sipa.be/miniscript/
  //The pk, pkh, and and_n fragments and t:, l:, and u: wrappers are syntactic
  //sugar for other miniscripts:
  miniscript = miniscript
    .replace(/^pk\(/, 'c:pk_k(')
    .replace(/^pkh\(/, 'c:pk_h(')
    .replace(/^and_n\(.*\)/, match =>
      match.replace('and_n', 'andor').replace(/\)$/, ',0)')
    )
    .replace(/^t:(.*)/, match => match.replace('t:', 'and_v(') + ',1)')
    .replace(/^l:(.*)/, match => match.replace('l:', 'or_i(0,') + ')')
    .replace(/^u:(.*)/, match => match.replace('u:', 'or_i(') + ',0)');
  const reFunctionName = String.raw`([^\(:]*)`;
  const matchFunctionName = miniscript.match(reFunctionName);
  if (!matchFunctionName) throw new Error('Invalid expression: ' + miniscript);
  const functionName = matchFunctionName[0];
  if (typeof satisfactionsMaker[functionName] !== 'function')
    throw new Error(functionName + ' not implemented');

  let args;
  if (miniscript[functionName.length] === '(')
    args = miniscript.substring(functionName.length + 1, miniscript.length - 1);
  else if (miniscript[functionName.length] === ':')
    args = miniscript.substring(functionName.length + 1);

  //the function arguments for satisfactionsMaker[functionName]:
  //They will be called like using ES6 spread operator:
  //satisfactionsMaker[functionName](...satisfactionMakerArgs)
  const satisfactionMakerArgs = [];
  if (args) {
    let lastCommaPosition = -1;
    let argLevel = 0; //argLevel tracks nested parenthesis
    let argPosition = 0; //argPosition tracks argument order within functionName
    for (let i = 0; i < args.length; i++) {
      args[i] === '(' && argLevel++;
      args[i] === ')' && argLevel--;
      if (argLevel === 0) {
        let arg;
        if (args[i] === ',') {
          arg = args.substring(lastCommaPosition + 1, i);
          lastCommaPosition = i;
        } else if (i === args.length - 1) {
          arg = args.substring(lastCommaPosition + 1);
        }
        if (arg) {
          if (isScalarArg(functionName, argPosition)) {
            //This is the case when arg is a scalar value that will be directly
            //passed to a satisfier maker function (arg is not a nested
            //miniscript expression).
            //That is, arg is one of these: a key or a hash or an nLockTime,
            //nSequence, or the k (number of keys) in thresh/multi.
            satisfactionMakerArgs.push(arg);
          } else {
            //arg is a miniscript expression that has to be further evaluated:
            satisfactionMakerArgs.push(evaluate(arg));
          }
          argPosition++;
        }
      }
    }
  }

  return satisfactionsMaker[functionName](...satisfactionMakerArgs);
};

/**
 * Obtains the satisfactions of a miniscript.
 * @Function
 *
 * @param {string} miniscript - A miniscript expression.
 * @param {object} [options]
 * @param {string[]} [options.unknowns] - An array with the pieces of information that
 * cannot be used to construct solutions because they are unknown.
 *
 * For example, if a honest user cannot sign with `key`, doesn't know the
 * preimage of `H` and the preimage of `e946029032eae1752e181bebab65de15cf0b93aaac4ee0ffdcfccb683c874d43` then `unknown` must be:
 * ```
 * [
 *   '<sig(key)>',
 *   '<ripemd160_preimage(H)>',
 *   '<sha256_preimage(e946029032eae1752e181bebab65de15cf0b93aaac4ee0ffdcfccb683c874d43)>'
 * ]
 * ```
 *
 * Note that an expression will allways be NOT sane if it is NOT sane at the the
 * miniscript top level as per {@link https://bitcoin.sipa.be/miniscript/
 * Wuille's algorithm}.
 *
 * For example; if a miniscript is NOT sane at the top level because it can be
 * satisfyed using only preimages, then setting preimages as unknowns will not
 * change this behaviour.
 *
 * The reason for this limitation is that the satisfier uses an unmodified
 * version of Wuille's algorithm as an initial precondition before finding
 * solutions. If the miniscript is sane, then unknowns can be set to produce
 * more possible solutions, including preimages, as described above.
 *
 * @param {string[]} [options.knowns] - An array
 * with the only pieces of information that can be used to build satisfactions.
 * This is the complimentary to unknowns. Only `knowns` or `unknowns` must be
 * passed.
 *
 * If neither knowns and unknowns is passed then it is assumed that there are
 * no unknowns, in other words, that all pieces of information are known.
 *
 * @returns {SatisfierResult}
 *
 * @see {@link Solution}
 */
const satisfier = (miniscript, options = {}) => {
  let { unknowns, knowns } = options;
  const { issane, issanesublevel } = compileMiniscript(miniscript);

  if (!issane) {
    throw new Error(`Miniscript ${miniscript} is not sane.`);
  }

  if (typeof unknowns === 'undefined' && typeof knowns === 'undefined') {
    unknowns = [];
  } else if (typeof unknowns !== 'undefined' && typeof knowns !== 'undefined') {
    throw new Error(`Cannot pass both knowns and unknowns`);
  } else if (
    (knowns && !Array.isArray(knowns)) ||
    (unknowns && !Array.isArray(unknowns))
  ) {
    throw new Error(`Incorrect types for unknowns / knowns`);
  }

  const knownSats = [];
  const unknownSats = [];
  const sats = evaluate(miniscript).sats || [];
  sats.map(sat => {
    if (typeof sat.nSequence === 'undefined') delete sat.nSequence;
    if (typeof sat.nLockTime === 'undefined') delete sat.nLockTime;
    //Clean format: 1 consecutive spaces at most, no leading & trailing spaces
    sat.asm = sat.asm.replace(/  +/g, ' ').trim();

    if (unknowns) {
      if (unknowns.some(unknown => sat.asm.includes(unknown))) {
        unknownSats.push(sat);
      } else {
        knownSats.push(sat);
      }
    } else {
      const delKnowns = knowns.reduce(
        (acc, known) => acc.replace(known, ''),
        sat.asm
      );
      if (
        delKnowns.match(
          /<sig\(|<sha256_preimage\(|<hash256_preimage\(|<ripemd160_preimage\(|<hash160_preimage\(/
        )
      ) {
        //Even thought all known pieces of information are removed, there are
        //still other pieces of info needed. Thus, this sat is unkown.
        unknownSats.push(sat);
      } else {
        knownSats.push(sat);
      }
    }
  });

  return { ...malleabilityAnalysis(knownSats), unknownSats };
};

exports.compileMiniscript = compileMiniscript;
exports.compilePolicy = compilePolicy;
exports.satisfier = satisfier;
