"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.openIssueCreationLink = exports.maybePrintIssueCreationPrompt = exports.shouldRecommendIssue = exports.getPackageVCSDetails = void 0;
const chalk_1 = __importDefault(require("chalk"));
const open_1 = __importDefault(require("open"));
const querystring_1 = require("querystring");
const path_1 = require("./path");
const repoSpecifier = /^([\w.-]+)\/([\w.-]+)$/;
const githubURL = /github.com(:|\/)([\w.-]+\/[\w.-]+?)(.git|\/.*)?$/;
function parseRepoString(repository) {
    if (repository.startsWith("github:")) {
        repository = repository.replace(/^github:/, "");
    }
    const urlMatch = repository.match(githubURL);
    if (urlMatch) {
        repository = urlMatch[2];
    }
    const specMatch = repository.match(repoSpecifier);
    if (!specMatch) {
        return null;
    }
    const [, org, repo] = specMatch;
    return { org, repo, provider: "GitHub" };
}
function getPackageVCSDetails(packageDetails) {
    const repository = require(path_1.resolve(path_1.join(packageDetails.path, "package.json")))
        .repository;
    if (!repository) {
        return null;
    }
    if (typeof repository === "string") {
        return parseRepoString(repository);
    }
    else if (typeof repository === "object" &&
        typeof repository.url === "string") {
        return parseRepoString(repository.url);
    }
}
exports.getPackageVCSDetails = getPackageVCSDetails;
function shouldRecommendIssue(vcsDetails) {
    if (!vcsDetails) {
        return true;
    }
    const { repo, org } = vcsDetails;
    if (repo === "DefinitelyTyped" && org === "DefinitelyTyped") {
        return false;
    }
    return true;
}
exports.shouldRecommendIssue = shouldRecommendIssue;
function maybePrintIssueCreationPrompt(vcs, packageDetails, packageManager) {
    if (vcs) {
        console.log(`💡 ${chalk_1.default.bold(packageDetails.name)} is on ${vcs.provider}! To draft an issue based on your patch run

    ${packageManager === "yarn" ? "yarn" : "npx"} patch-package ${packageDetails.pathSpecifier} --create-issue
`);
    }
}
exports.maybePrintIssueCreationPrompt = maybePrintIssueCreationPrompt;
function openIssueCreationLink({ packageDetails, patchFileContents, packageVersion, }) {
    const vcs = getPackageVCSDetails(packageDetails);
    if (!vcs) {
        console.log(`Error: Couldn't find VCS details for ${packageDetails.pathSpecifier}`);
        process.exit(1);
    }
    // trim off trailing newline since we add an extra one in the markdown block
    if (patchFileContents.endsWith("\n")) {
        patchFileContents = patchFileContents.slice(0, -1);
    }
    open_1.default(`https://github.com/${vcs.org}/${vcs.repo}/issues/new?${querystring_1.stringify({
        title: "",
        body: `Hi! 👋 
      
Firstly, thanks for your work on this project! 🙂

Today I used [patch-package](https://github.com/ds300/patch-package) to patch \`${packageDetails.name}@${packageVersion}\` for the project I'm working on.

<!-- 🔺️🔺️🔺️ PLEASE REPLACE THIS BLOCK with a description of your problem, and any other relevant context 🔺️🔺️🔺️ -->

Here is the diff that solved my problem:

\`\`\`diff
${patchFileContents}
\`\`\`

<em>This issue body was [partially generated by patch-package](https://github.com/ds300/patch-package/issues/296).</em>
`,
    })}`);
}
exports.openIssueCreationLink = openIssueCreationLink;
//# sourceMappingURL=data:application/json;base64,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