"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.decodeScriptSignature = void 0;
// Taken from https://github.com/bitcoinjs/bitcoinjs-lib/blob/5d2ff1c61165932e2814d5f37630e6720168561c/ts_src/script_signature.ts#L29
function decodeScriptSignature(buffer) {
    const hashType = buffer.readUInt8(buffer.length - 1);
    const hashTypeMod = hashType & ~0x80;
    if (hashTypeMod <= 0 || hashTypeMod >= 4)
        throw new Error('Invalid hashType ' + hashType);
    const decoded = decode2(buffer.slice(0, -1));
    const r = fromDER(decoded.r);
    const s = fromDER(decoded.s);
    const signature = Buffer.concat([r, s], 64);
    return { signature, hashType };
}
exports.decodeScriptSignature = decodeScriptSignature;
function fromDER(x) {
    if (x[0] === 0x00)
        x = x.slice(1);
    const buffer = Buffer.alloc(32, 0);
    const bstart = Math.max(0, 32 - x.length);
    x.copy(buffer, bstart);
    return buffer;
}
function decode2(buffer) {
    if (buffer.length < 8)
        throw new Error('DER sequence length is too short');
    if (buffer.length > 72)
        throw new Error('DER sequence length is too long');
    if (buffer[0] !== 0x30)
        throw new Error('Expected DER sequence');
    if (buffer[1] !== buffer.length - 2)
        throw new Error('DER sequence length is invalid');
    if (buffer[2] !== 0x02)
        throw new Error('Expected DER integer');
    const lenR = buffer[3];
    if (lenR === 0)
        throw new Error('R length is zero');
    if (5 + lenR >= buffer.length)
        throw new Error('R length is too long');
    if (buffer[4 + lenR] !== 0x02)
        throw new Error('Expected DER integer (2)');
    const lenS = buffer[5 + lenR];
    if (lenS === 0)
        throw new Error('S length is zero');
    if (6 + lenR + lenS !== buffer.length)
        throw new Error('S length is invalid');
    if (buffer[4] & 0x80)
        throw new Error('R value is negative');
    if (lenR > 1 && buffer[4] === 0x00 && !(buffer[5] & 0x80))
        throw new Error('R value excessively padded');
    if (buffer[lenR + 6] & 0x80)
        throw new Error('S value is negative');
    if (lenS > 1 && buffer[lenR + 6] === 0x00 && !(buffer[lenR + 7] & 0x80))
        throw new Error('S value excessively padded');
    // non-BIP66 - extract R, S values
    return {
        r: buffer.slice(4, 4 + lenR),
        s: buffer.slice(6 + lenR),
    };
}
//# sourceMappingURL=DecodeScriptSignature.js.map