"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.scheduleAction = exports.ScheduleActionDeadlineError = exports.SCHEDULE_ACTION_DEADLINE_ERROR_MESSAGE = exports.ScheduleActionTimeoutError = exports.SCHEDULE_ACTION_TIMEOUT_ERROR_MESSAGE = exports.RejectWhenAbortedError = exports.SCHEDULE_ACTION_ABORTED_ERROR_MESSAGE = void 0;
exports.SCHEDULE_ACTION_ABORTED_ERROR_MESSAGE = 'Aborted by signal';
class RejectWhenAbortedError extends Error {
    constructor() {
        super(exports.SCHEDULE_ACTION_ABORTED_ERROR_MESSAGE);
    }
}
exports.RejectWhenAbortedError = RejectWhenAbortedError;
const isArray = (attempts) => Array.isArray(attempts);
const resolveAfterMs = (ms, clear) => new Promise((resolve, reject) => {
    const errorSignal = new RejectWhenAbortedError();
    if (clear.aborted)
        return reject(errorSignal);
    if (ms === undefined)
        return resolve();
    let timeout;
    const onClear = () => {
        clearTimeout(timeout);
        clear.removeEventListener('abort', onClear);
        reject(errorSignal);
    };
    timeout = setTimeout(() => {
        clear.removeEventListener('abort', onClear);
        resolve();
    }, ms);
    clear.addEventListener('abort', onClear);
});
const rejectAfterMs = (ms, reason, clear) => new Promise((_, reject) => {
    const errorSignal = new RejectWhenAbortedError();
    if (clear.aborted)
        return reject(errorSignal);
    let timeout;
    const onClear = () => {
        clearTimeout(timeout);
        clear.removeEventListener('abort', onClear);
        reject(errorSignal);
    };
    timeout = setTimeout(() => {
        clear.removeEventListener('abort', onClear);
        reject(reason);
    }, ms);
    clear.addEventListener('abort', onClear);
});
const maybeRejectAfterMs = (ms, reason, clear) => ms === undefined ? [] : [rejectAfterMs(ms, reason, clear)];
const rejectWhenAborted = (signal, clear) => new Promise((_, reject) => {
    const errorSignal = new RejectWhenAbortedError();
    if (clear.aborted)
        return reject(errorSignal);
    if (signal?.aborted)
        return reject(errorSignal);
    const onAbort = () => reject(errorSignal);
    signal?.addEventListener('abort', onAbort);
    const onClear = () => {
        signal?.removeEventListener('abort', onAbort);
        clear.removeEventListener('abort', onClear);
        reject(errorSignal);
    };
    clear.addEventListener('abort', onClear);
});
const resolveAction = async (action, clear) => {
    const aborter = new AbortController();
    if (clear.aborted)
        aborter.abort();
    const onClear = () => {
        clear.removeEventListener('abort', onClear);
        aborter.abort();
    };
    clear.addEventListener('abort', onClear);
    try {
        return await new Promise(resolve => resolve(action(aborter.signal)));
    }
    finally {
        if (!clear.aborted)
            clear.removeEventListener('abort', onClear);
    }
};
const attemptLoop = async (attempts, attempt, failure, clear) => {
    for (let a = 0; a < attempts - 1; a++) {
        if (clear.aborted)
            break;
        const aborter = new AbortController();
        const onClear = () => aborter.abort();
        clear.addEventListener('abort', onClear);
        try {
            return await attempt(a, aborter.signal);
        }
        catch (error) {
            onClear();
            await failure(a, error);
        }
        finally {
            clear.removeEventListener('abort', onClear);
        }
    }
    return clear.aborted
        ? Promise.reject(new RejectWhenAbortedError())
        : attempt(attempts - 1, clear);
};
exports.SCHEDULE_ACTION_TIMEOUT_ERROR_MESSAGE = 'Aborted by timeout';
class ScheduleActionTimeoutError extends Error {
    constructor() {
        super(exports.SCHEDULE_ACTION_TIMEOUT_ERROR_MESSAGE);
    }
}
exports.ScheduleActionTimeoutError = ScheduleActionTimeoutError;
exports.SCHEDULE_ACTION_DEADLINE_ERROR_MESSAGE = 'Aborted by deadline';
class ScheduleActionDeadlineError extends Error {
    constructor() {
        super(exports.SCHEDULE_ACTION_DEADLINE_ERROR_MESSAGE);
    }
}
exports.ScheduleActionDeadlineError = ScheduleActionDeadlineError;
const scheduleAction = async (action, params) => {
    const { signal, delay, attempts, timeout, deadline, gap, attemptFailureHandler } = params;
    const deadlineMs = deadline && deadline - Date.now();
    const attemptCount = isArray(attempts)
        ? attempts.length
        : (attempts ?? (deadline ? Infinity : 1));
    const clearAborter = new AbortController();
    const clear = clearAborter.signal;
    const getParams = isArray(attempts)
        ? (attempt) => attempts[attempt]
        : () => ({ timeout, gap });
    const errorDeadline = new ScheduleActionDeadlineError();
    const errorTimeout = new ScheduleActionTimeoutError();
    const graceful = params.graceful && signal;
    const actionAborter = new AbortController();
    if (graceful) {
        if (signal.aborted) {
            actionAborter.abort();
        }
        else {
            const onAbort = () => {
                signal.removeEventListener('abort', onAbort);
                clear.removeEventListener('abort', onAbort);
                actionAborter.abort();
            };
            signal.addEventListener('abort', onAbort);
            clear.addEventListener('abort', onAbort);
        }
    }
    try {
        return await Promise.race([
            ...(graceful ? [] : [rejectWhenAborted(signal, clear)]),
            ...maybeRejectAfterMs(deadlineMs, errorDeadline, clear),
            resolveAfterMs(delay, clear).then(() => attemptLoop(attemptCount, (attempt, abort) => Promise.race([
                ...maybeRejectAfterMs(getParams(attempt).timeout, errorTimeout, clear),
                resolveAction(action, abort),
            ]), (attempt, error) => {
                const errorHandlerResult = attemptFailureHandler?.(error);
                return errorHandlerResult
                    ? Promise.reject(errorHandlerResult)
                    : resolveAfterMs(getParams(attempt).gap ?? 0, clear);
            }, graceful ? actionAborter.signal : clear)),
        ]);
    }
    finally {
        clearAborter.abort();
    }
};
exports.scheduleAction = scheduleAction;
//# sourceMappingURL=scheduleAction.js.map